package com.andaily.hb.service.operation.job;

import com.andaily.hb.domain.log.FrequencyMonitorLog;
import com.andaily.hb.domain.network.NetworkInstance;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.*;
import java.net.DatagramPacket;
import java.net.DatagramSocket;
import java.net.InetAddress;
import java.net.Socket;
import java.nio.charset.StandardCharsets;

/**
 * 2023/12/28 11:34
 * <p>
 * 每一次 network 监控日志生成操作类
 *
 * @author Shengzhao Li
 * @since 3.0.0
 */
public class NetworkFrequencyMonitorLogGenerator {


    private static final Logger LOG = LoggerFactory.getLogger(NetworkFrequencyMonitorLogGenerator.class);

    private final NetworkInstance instance;

    public NetworkFrequencyMonitorLogGenerator(NetworkInstance instance) {
        this.instance = instance;
    }


    /**
     * 生成监控日志.
     * 先通过 Socket 发送请求 并根据响应情况 记录日志
     *
     * @return FrequencyMonitorLog instance
     */
    public FrequencyMonitorLog generate() {

        NetworkInstance.Type type = this.instance.type();
        FrequencyMonitorLog monitorLog;
        if (type.isUdp()) {
            monitorLog = generateUdpLog();
        } else {
            monitorLog = generateTcpLog();
        }

        monitorLog.networkInstance(instance);
        return monitorLog;
    }

    /**
     * TCP flow
     *
     * @return FrequencyMonitorLog
     */
    private FrequencyMonitorLog generateTcpLog() {
        FrequencyMonitorLog monitorLog = new FrequencyMonitorLog();

        final long start = System.currentTimeMillis();
        try {
            try (Socket socket = new Socket(instance.serverAddress(), instance.serverPort())) {
                socket.setSoTimeout(instance.maxConnectionSeconds() * 1000);
                //发送内容
                String sendContent = instance.sendContent();
                if (StringUtils.isNotBlank(sendContent)) {
                    OutputStream out = socket.getOutputStream();
                    out.write(sendContent.getBytes(StandardCharsets.UTF_8));
                    out.flush();
                }
                //接收响应
                InputStream in = socket.getInputStream();
                byte[] buffer = new byte[1024];
                int bytesRead = in.read(buffer);
                String response = null;
                if (bytesRead != -1) {
                    response = new String(buffer, 0, bytesRead);
                } else {
                    if (LOG.isDebugEnabled()) {
                        LOG.debug("Null or empty response from socket: {}", socket);
                    }
                }

                monitorLog.normal(true).remark("Response: " + response).costTime(costTime(start))
                        .responseSize(response != null ? response.length() : 0);

            }
        } catch (Exception e) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("Socket(TCP) error, serverAddress: {}, port: {}", instance.serverAddress(), instance.serverPort(), e);
            }
            monitorLog.costTime(costTime(start)).normal(false)
                    .remark(e.getClass().getSimpleName() + ": " + e.getMessage());
        }

        return monitorLog;
    }


    /**
     * UDP flow
     *
     * @return FrequencyMonitorLog
     */
    private FrequencyMonitorLog generateUdpLog() {
        FrequencyMonitorLog monitorLog = new FrequencyMonitorLog();

        final long start = System.currentTimeMillis();
        try {
            InetAddress address = InetAddress.getByName(instance.serverAddress());
            try (DatagramSocket socket = new DatagramSocket()) {
                socket.setSoTimeout(instance.maxConnectionSeconds() * 1000);
                //发送内容
                String sendContent = instance.sendContent();
                if (StringUtils.isNotBlank(sendContent)) {
                    DatagramPacket packet = new DatagramPacket(sendContent.getBytes(StandardCharsets.UTF_8), sendContent.length(), address, instance.serverPort());
                    socket.send(packet);
                }
                //接收响应
                DatagramPacket receivePacket = new DatagramPacket(new byte[1024], 1024);
                socket.receive(receivePacket);
                String response = new String(receivePacket.getData(), 0, receivePacket.getLength());

                monitorLog.normal(true).remark("Response: " + response).costTime(costTime(start))
                        .responseSize(response.length());

            }
        } catch (Exception e) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("Socket(UDP) error, serverAddress: {}, port: {}", instance.serverAddress(), instance.serverPort(), e);
            }
            monitorLog.costTime(costTime(start)).normal(false)
                    .remark(e.getClass().getSimpleName() + ": " + e.getMessage());
        }

        return monitorLog;
    }


    private long costTime(long start) {
        return System.currentTimeMillis() - start;
    }

}
