package com.andaily.hb.service.operation.job;

import com.andaily.hb.domain.shared.Application;
import com.andaily.hb.domain.shared.BeanProvider;
import com.andaily.hb.infrastructure.DateUtils;
import com.andaily.hb.infrastructure.jpa.FrequencyMonitorLogRepository;
import com.andaily.hb.infrastructure.jpa.MonitoringReminderLogRepository;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Date;

/**
 * 2017/1/21
 * <p/>
 * 自动清理过期的监控日志
 *
 * @author Shengzhao Li
 */
public class AutoMonitorLogCleaner {

    private static final Logger LOG = LoggerFactory.getLogger(AutoMonitorLogCleaner.class);


    private final transient MonitoringReminderLogRepository logRepository = BeanProvider.getBean(MonitoringReminderLogRepository.class);

    /**
     * @since 3.0.0
     */
    private final transient FrequencyMonitorLogRepository frequencyMonitorLogRepository = BeanProvider.getBean(FrequencyMonitorLogRepository.class);


    public AutoMonitorLogCleaner() {
    }


    /**
     * 处理 自动清理 监控日志的逻辑
     *
     * @return 清理成功的数据数量
     */
    public long clean() {

        long cleanedAmount = 0;
        long start = System.currentTimeMillis();

        Date date = getCleanedTime();

        /*
         * 处理MonitoringReminderLog
         * */

        cleanedAmount += cleanMonitoringReminderLogs(date);

        /*
         * 处理 FrequencyMonitorLog
         * */
        cleanedAmount += cleanFrequencyMonitorLogs(date);

        LOG.debug("HB finished auto clean monitor logs, cost time: {}, cleanedAmount: {}", (System.currentTimeMillis() - start), cleanedAmount);
        return cleanedAmount;
    }

    private Date getCleanedTime() {
        final int cleanFrequency = Application.systemSetting().cleanMonitorLogFrequency();
        //获取具体的时间点, 之前的清除
        Date date = DateUtils.plusDays(DateUtils.now(), -cleanFrequency);
        LOG.debug("HB will clean the monitor logs before the date: {},frequency: {}", date, cleanFrequency);
        return date;
    }

    private long cleanMonitoringReminderLogs(Date date) {
        long reminderLogAmount = logRepository.amountOfMonitoringReminderLogsBeforeDate(date.getTime());
        if (reminderLogAmount > 0) {
            int row = logRepository.deleteMonitoringReminderLogsBeforeDate(date);
            LOG.debug("Cleaned {} MonitoringReminderLogs the createTime before: {} -> row: {}", reminderLogAmount, date, row);
        }
        return reminderLogAmount;
    }

    private long cleanFrequencyMonitorLogs(Date date) {
        long monitorAmount = frequencyMonitorLogRepository.amountOfFrequencyMonitorLogsBeforeDate(date);
        if (monitorAmount > 0) {
            int row = frequencyMonitorLogRepository.deleteFrequencyMonitorLogsBeforeDate(date);
            LOG.debug("Cleaned {} FrequencyMonitorLog the createTime before: {} -> row: {}", monitorAmount, date, row);
        }
        return monitorAmount;
    }

}
