package com.andaily.hb.service.operation.instance;

import com.andaily.hb.domain.dto.application.InstanceStatisticsDto;
import com.andaily.hb.domain.log.FrequencyMonitorLog;
import com.andaily.hb.infrastructure.DateUtils;
import com.andaily.hb.infrastructure.jpa.FrequencyMonitorLogRepository;
import com.andaily.hb.infrastructure.jpa.MonitoringReminderLogRepository;
import com.andaily.hb.infrastructure.jpa.NetworkInstanceRepository;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.util.Date;

import static com.andaily.hb.infrastructure.DateUtils.now;

/**
 * @author Shengzhao Li
 */
@Component
public class InstanceStatisticsDtoLoader {


    private static final String EMPTY_TEXT = "---";

    @Autowired
    private MonitoringReminderLogRepository logRepository;

    /**
     * @since 3.0.0
     */
    @Autowired
    private FrequencyMonitorLogRepository monitorLogRepository;

    /**
     * @since 3.0.0
     */
    @Autowired
    private NetworkInstanceRepository networkInstanceRepository;


    private String guid;

    /**
     * 标识是否是网络实例
     *
     * @since 3.0.0
     */
    private boolean networkInstance;

    public InstanceStatisticsDtoLoader() {
    }

    public InstanceStatisticsDto load(String guid) {
        this.guid = guid;
        //判断是否是网络实例
        int count = networkInstanceRepository.countByGuid(guid);
        this.networkInstance = count > 0;

        InstanceStatisticsDto statisticsDto = new InstanceStatisticsDto(guid);
        statisticsDto.setNetworkInstance(this.networkInstance);

        final Date lastNotNormalTime = lastNotNormalTime(statisticsDto);
        normalConnection(statisticsDto, lastNotNormalTime);

        normalAmount(statisticsDto);
        notNormalAmount(statisticsDto);

        interruptTime(statisticsDto);
        sendReminderTime(statisticsDto);

        return statisticsDto;
    }

    private void sendReminderTime(InstanceStatisticsDto statisticsDto) {
        int amount = networkInstance ? logRepository.totalNetworkMonitoringReminderLogs(guid) : logRepository.totalMonitoringReminderLogs(guid);
        statisticsDto.setSendReminderTime(amount / 2);
    }

    private void interruptTime(InstanceStatisticsDto statisticsDto) {
        int times = networkInstance ? logRepository.amountOfNetworkMonitoringReminderLogs(guid, false) :
                logRepository.amountOfMonitoringReminderLogs(guid, false);
        statisticsDto.setInterruptTime(times);
    }

    /**
     * Format:  0y 2M 1d 12h 23m 23s
     */
    private void normalConnection(InstanceStatisticsDto statisticsDto, Date lastNotNormalTime) {
        Date firstNormalAfterLastNotNormalTime;
        if (lastNotNormalTime != null) {
            firstNormalAfterLastNotNormalTime = networkInstance ?
                    monitorLogRepository.findFirstNetworkMonitoringLogTimeAfterSpecifyTime(guid, lastNotNormalTime) :
                    monitorLogRepository.findFirstMonitoringLogTimeAfterSpecifyTime(guid, lastNotNormalTime);
        } else {
            firstNormalAfterLastNotNormalTime = networkInstance ? monitorLogRepository.findFirstNetworkMonitoringLogTime(guid, true) :
                    monitorLogRepository.findFirstMonitoringLogTime(guid, true);
        }

        if (firstNormalAfterLastNotNormalTime == null) {
            statisticsDto.setNormalConnection(EMPTY_TEXT);
        } else {
            final long seconds = DateUtils.periodAsSeconds(firstNormalAfterLastNotNormalTime, now());
            statisticsDto.setNormalConnection(DateUtils.secondsToText(seconds));
        }
    }

    private void notNormalAmount(InstanceStatisticsDto statisticsDto) {
        int notNormalAmount = networkInstance ? monitorLogRepository.amountOfNetworkInstanceMonitorLogs(guid, false) :
                monitorLogRepository.amountOfInstanceMonitorLogs(guid, false);
        statisticsDto.setNotNormalAmount(notNormalAmount);
    }

    private void normalAmount(InstanceStatisticsDto statisticsDto) {
        int normalAmount = networkInstance ? monitorLogRepository.amountOfNetworkInstanceMonitorLogs(guid, true) :
                monitorLogRepository.amountOfInstanceMonitorLogs(guid, true);
        statisticsDto.setNormalAmount(normalAmount);
    }

    private Date lastNotNormalTime(InstanceStatisticsDto statisticsDto) {
        FrequencyMonitorLog lastNotNormalLog = networkInstance ? monitorLogRepository.findNetworkLastNotNormalFrequencyMonitorLog(guid, now()) :
                monitorLogRepository.findLastNotNormalFrequencyMonitorLog(guid, now());
        Date lastNotNormalTime = null;
        if (lastNotNormalLog != null) {
            lastNotNormalTime = lastNotNormalLog.createTime();
            statisticsDto.setLastNotNormalTime(DateUtils.toDateText(lastNotNormalTime, DateUtils.DEFAULT_DATE_TIME_FORMAT));
        } else {
            statisticsDto.setLastNotNormalTime(EMPTY_TEXT);
        }
        return lastNotNormalTime;
    }
}