package com.andaily.hb.service.operation.instance;

import com.andaily.hb.domain.application.ApplicationInstance;
import com.andaily.hb.domain.application.ApplicationInstanceWeixinUser;
import com.andaily.hb.domain.application.InstanceMonitorURLParameter;
import com.andaily.hb.domain.dto.application.ApplicationInstanceFormDto;
import com.andaily.hb.domain.dto.application.InstanceMonitorURLParameterDto;
import com.andaily.hb.domain.shared.BeanProvider;
import com.andaily.hb.domain.shared.security.SecurityUtils;
import com.andaily.hb.domain.user.WeixinUser;
import com.andaily.hb.infrastructure.jpa.ApplicationInstanceRepositoryJpa;
import com.andaily.hb.infrastructure.jpa.ApplicationInstanceWeixinUserRepository;
import com.andaily.hb.infrastructure.jpa.InstanceMonitorURLParameterRepository;
import com.andaily.hb.infrastructure.jpa.UserRepositoryJpa;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.ArrayList;
import java.util.List;

/**
 * 15-1-4
 *
 * @author Shengzhao Li
 */
public class ApplicationInstanceFormDtoPersister {

    private static final Logger LOGGER = LoggerFactory.getLogger(ApplicationInstanceFormDtoPersister.class);

    private final transient ApplicationInstanceRepositoryJpa instanceRepository = BeanProvider.getBean(ApplicationInstanceRepositoryJpa.class);
    private final transient UserRepositoryJpa userRepository = BeanProvider.getBean(UserRepositoryJpa.class);

    /**
     * @since 3.0.0
     */
    private final transient ApplicationInstanceWeixinUserRepository weixinUserRepository = BeanProvider.getBean(ApplicationInstanceWeixinUserRepository.class);

    /**
     * @since 3.0.0
     */
    private final transient InstanceMonitorURLParameterRepository urlParameterRepository = BeanProvider.getBean(InstanceMonitorURLParameterRepository.class);


    private final ApplicationInstanceFormDto formDto;

    public ApplicationInstanceFormDtoPersister(ApplicationInstanceFormDto formDto) {
        this.formDto = formDto;
    }

    public void persist() {
        if (formDto.isNewly()) {
            createInstance();
        } else {
            updateInstance();
        }
    }

    private void updateInstance() {
        ApplicationInstance instance = instanceRepository.findByGuid(formDto.getGuid());
        if (instance.enabled()) {
            throw new IllegalStateException("Only Disabled ApplicationInstance support edit");
        }
        updateWeixinUsers(instance, false);
        formDto.updateDomain(instance);
        setRequestParams(instance);
        LOGGER.debug("<{}> Update ApplicationInstance [{}]", SecurityUtils.currentUsername(), instance);
    }

    private void createInstance() {
        ApplicationInstance instance = formDto.updateDomain(new ApplicationInstance());
        instance.creator(SecurityUtils.currentUser());

        setRequestParams(instance);

        instanceRepository.save(instance);
        updateWeixinUsers(instance, true);
        LOGGER.debug("<{}> Create ApplicationInstance [{}]", SecurityUtils.currentUsername(), instance);
    }

    private void updateWeixinUsers(ApplicationInstance instance, boolean newly) {
        if (!newly) {
            //clean old
            final int amount = instanceRepository.deleteApplicationInstanceWeixinUsers(instance);
            LOGGER.debug("<{}> delete old ApplicationInstanceWeixinUsers: {}", SecurityUtils.currentUsername(), amount);
        }
        final List<String> weixinUserGuids = formDto.getWeixinUserGuids();
        if (weixinUserGuids == null || weixinUserGuids.isEmpty()) {
            return;
        }
        List<ApplicationInstanceWeixinUser> newWeixinUsers = new ArrayList<>();
        for (String weixinUserGuid : weixinUserGuids) {
            WeixinUser weixinUser = userRepository.findWeixinUserByGuid(weixinUserGuid);
            newWeixinUsers.add(new ApplicationInstanceWeixinUser(instance, weixinUser));
        }
        weixinUserRepository.saveAll(newWeixinUsers);
        LOGGER.debug("<{}> Create newWeixinUsers: {}", SecurityUtils.currentUsername(), newWeixinUsers);
    }

    private void setRequestParams(ApplicationInstance instance) {
        final List<InstanceMonitorURLParameter> urlParameters = instance.instanceURL().urlParameters();
        if (!instance.isNewly()) {
            //remove old
            urlParameterRepository.deleteAll(urlParameters);
            urlParameters.clear();
        }

        final List<InstanceMonitorURLParameterDto> urlParameterDtos = formDto.getUrlParameters();
        for (InstanceMonitorURLParameterDto urlParameterDto : urlParameterDtos) {
            if (urlParameterDto.available()) {
                urlParameters.add(urlParameterDto.newDomain());
            } else {
                LOGGER.debug("Ignore URL parameter[{}], because key is empty", urlParameterDto);
            }
        }

    }
}
