package com.andaily.hb.service.operation.instance;

import com.andaily.hb.domain.application.ApplicationInstance;
import com.andaily.hb.domain.shared.BeanProvider;
import com.andaily.hb.domain.shared.security.SecurityUtils;
import com.andaily.hb.infrastructure.jpa.ApplicationInstanceRepositoryJpa;
import com.andaily.hb.infrastructure.scheduler.DynamicJob;
import com.andaily.hb.infrastructure.scheduler.JobParamManager;
import com.andaily.hb.service.operation.job.MonitoringInstanceJob;
import org.apache.commons.lang.StringUtils;
import org.quartz.SchedulerException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import static com.andaily.hb.infrastructure.scheduler.DynamicSchedulerFactory.existJob;
import static com.andaily.hb.infrastructure.scheduler.DynamicSchedulerFactory.registerJob;
import static com.andaily.hb.infrastructure.scheduler.DynamicSchedulerFactory.resumeJob;

/**
 * @author Shengzhao Li
 */
public class ApplicationInstanceEnabler {

    private static final Logger LOGGER = LoggerFactory.getLogger(ApplicationInstanceEnabler.class);

    private final transient ApplicationInstanceRepositoryJpa instanceRepository = BeanProvider.getBean(ApplicationInstanceRepositoryJpa.class);
    private final String guid;

    public ApplicationInstanceEnabler(String guid) {
        this.guid = guid;
    }

    public boolean enable() {
        final ApplicationInstance instance = instanceRepository.findByGuid(guid);
        if (instance.enabled()) {
            LOGGER.debug("<{}> Instance[guid={}] already enabled, ignore it", username(), instance.guid());
            return false;
        }

        final boolean addSuccessful = startupMonitoringJob(instance);
        if (!addSuccessful) {
            LOGGER.debug("<{}> NOTE: Add MonitoringJob[jobName={}] failed", username(), instance.jobName());
            return false;
        }

        //update
        instance.enabled(true);
        LOGGER.debug("<{}> Update ApplicationInstance[guid={}] enabled=true,jobName={}", username(), instance.guid(), instance.jobName());

        return true;
    }

    private boolean startupMonitoringJob(ApplicationInstance instance) {
        final String jobName = getAndSetJobName(instance);

        DynamicJob job = new DynamicJob(jobName)
                .cronExpression(instance.frequency().getCronExpression())
                .target(MonitoringInstanceJob.class)
                .addJobData(MonitoringInstanceJob.INSTANCE_GUID, instance.guid());

        return executeStartup(instance, job);
    }

    private boolean executeStartup(ApplicationInstance instance, DynamicJob job) {
        boolean result = false;
        try {
            if (existJob(job)) {
                result = resumeJob(job);
                LOGGER.debug("<{}> Resume  [{}] by ApplicationInstance[guid={},instanceName={}] result: {}", username(), job, instance.guid(), instance.instanceName(), result);
            } else {
                result = registerJob(job);
                LOGGER.debug("<{}> Register  [{}] by ApplicationInstance[guid={},instanceName={}] result: {}", username(), job, instance.guid(), instance.instanceName(), result);
            }
        } catch (SchedulerException e) {
            LOGGER.error("<{}> Register [" + job + "] failed", username(), e);
        }
        return result;
    }

    private String getAndSetJobName(ApplicationInstance instance) {
        String jobName = instance.jobName();
        if (StringUtils.isEmpty(jobName)) {
            jobName = JobParamManager.generateMonitoringInstanceJobName(instance.guid());
            instance.jobName(jobName);
            //update to DB
            instanceRepository.save(instance);
        }
        return jobName;
    }

    private String username() {
        return SecurityUtils.currentUsername();
    }
}