package com.andaily.hb.service.operation;

import com.andaily.hb.domain.application.ApplicationInstance;
import com.andaily.hb.domain.dto.IndexDto;
import com.andaily.hb.domain.dto.IndexInstanceDto;
import com.andaily.hb.domain.log.FrequencyMonitorLog;
import com.andaily.hb.domain.network.NetworkInstance;
import com.andaily.hb.domain.shared.security.SecurityUtils;
import com.andaily.hb.infrastructure.DateUtils;
import com.andaily.hb.infrastructure.jpa.ApplicationInstanceRepositoryJpa;
import com.andaily.hb.infrastructure.jpa.FrequencyMonitorLogRepository;
import com.andaily.hb.infrastructure.jpa.NetworkInstanceRepository;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.stereotype.Component;

import java.util.Date;
import java.util.List;

/**
 * 首页加载
 *
 * @author Shengzhao Li
 */
@Component
public class IndexDtoLoader {

    @Autowired
    private FrequencyMonitorLogRepository logRepository;

    /**
     * @since 3.0.0
     */
    @Autowired
    private ApplicationInstanceRepositoryJpa instanceRepository;

    /**
     * @since 3.0.0
     */
    @Autowired
    private NetworkInstanceRepository networkInstanceRepository;


    public IndexDtoLoader() {
    }

    public IndexDto load(IndexDto indexDto) {
        final List<IndexInstanceDto> instanceDtos = indexDto.getInstanceDtos();
        if (indexDto.isNetworkInstance()) {
            List<NetworkInstance> instanceList = loadNetworkInstances(indexDto);
            for (NetworkInstance instance : instanceList) {
                instanceDtos.add(generateIndexInstanceDto(instance, indexDto));
            }
        } else {
            final List<ApplicationInstance> instances = loadInstances(indexDto);
            for (ApplicationInstance instance : instances) {
                instanceDtos.add(generateIndexInstanceDto(instance, indexDto));
            }
        }
        return indexDto;
    }

    private List<ApplicationInstance> loadInstances(IndexDto indexDto) {
        final boolean enabled = indexDto.isEnabled();
        if (enabled) {
            if (SecurityUtils.currentUser() == null) {
                return instanceRepository.findHadLogPublicInstancesByEnabled(true);
            } else {
                return instanceRepository.findHadLogInstancesByEnabled(true);
            }

        } else {
            return instanceRepository.findHadLogInstances();
        }
//        return enabled ? instanceRepository.findHadLogInstancesByEnabled(true) : instanceRepository.findHadLogInstances();
    }

    /**
     * @since 3.0.0
     */
    private List<NetworkInstance> loadNetworkInstances(IndexDto indexDto) {
        final boolean enabled = indexDto.isEnabled();
        if (enabled) {
            if (SecurityUtils.currentUser() == null) {
                return networkInstanceRepository.findHadLogPublicInstancesByEnabled(true);
            } else {
                return networkInstanceRepository.findHadLogInstancesByEnabled(true);
            }

        } else {
            return networkInstanceRepository.findHadLogInstances();
        }
    }

    private IndexInstanceDto generateIndexInstanceDto(ApplicationInstance instance, IndexDto indexDto) {
        IndexInstanceDto indexInstanceDto = new IndexInstanceDto(instance);
        PageRequest pageRequest = PageRequest.ofSize(indexDto.getMaxResult());
        List<FrequencyMonitorLog> monitorLogs = logRepository.findLatestFrequencyMonitorLogs(instance, pageRequest);

        MonitoringChartDataGenerator chartDataGenerator = new MonitoringChartDataGenerator(monitorLogs);
        indexInstanceDto.setCategoryData(chartDataGenerator.generateCategoryData());
        indexInstanceDto.setSeriesData(chartDataGenerator.generateSeriesData());

        lastLogDate(indexInstanceDto, monitorLogs);
        return indexInstanceDto;
    }

    /**
     * @since 3.0.0
     */
    private IndexInstanceDto generateIndexInstanceDto(NetworkInstance instance, IndexDto indexDto) {
        IndexInstanceDto indexInstanceDto = new IndexInstanceDto(instance);
        PageRequest pageRequest = PageRequest.ofSize(indexDto.getMaxResult());
        List<FrequencyMonitorLog> monitorLogs = logRepository.findLatestFrequencyMonitorLogs(instance, pageRequest);

        MonitoringChartDataGenerator chartDataGenerator = new MonitoringChartDataGenerator(monitorLogs);
        indexInstanceDto.setCategoryData(chartDataGenerator.generateCategoryData());
        indexInstanceDto.setSeriesData(chartDataGenerator.generateSeriesData());

        lastLogDate(indexInstanceDto, monitorLogs);
        return indexInstanceDto;
    }

    private void lastLogDate(IndexInstanceDto indexInstanceDto, List<FrequencyMonitorLog> monitorLogs) {
        final Date time = monitorLogs.isEmpty() ? DateUtils.now() : monitorLogs.get(0).createTime();
        indexInstanceDto.setLastLogDate(DateUtils.toDateText(time, DateUtils.DEFAULT_DATE_TIME_FORMAT));
    }
}