package com.andaily.hb.service.operation;

import com.andaily.hb.domain.application.ApplicationInstance;
import com.andaily.hb.domain.dto.CommonsInstanceDto;
import com.andaily.hb.domain.dto.log.FrequencyMonitorLogDto;
import com.andaily.hb.domain.dto.log.FrequencyMonitorLogListDto;
import com.andaily.hb.domain.log.FrequencyMonitorLog;
import com.andaily.hb.domain.network.NetworkInstance;
import com.andaily.hb.domain.shared.Application;
import com.andaily.hb.domain.shared.security.SecurityUtils;
import com.andaily.hb.infrastructure.jpa.ApplicationInstanceRepositoryJpa;
import com.andaily.hb.infrastructure.jpa.FrequencyMonitorLogRepository;
import com.andaily.hb.infrastructure.jpa.NetworkInstanceRepository;
import jakarta.persistence.criteria.Order;
import jakarta.persistence.criteria.Predicate;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.jpa.domain.Specification;
import org.springframework.stereotype.Component;

import java.util.ArrayList;
import java.util.List;

/**
 * 15-2-13
 *
 * @author Shengzhao Li
 */
@Component
public class FrequencyMonitorLogListDtoLoader {

    @Autowired
    private FrequencyMonitorLogRepository logRepository;


    /**
     * @since 3.0.0
     */
    @Autowired
    private ApplicationInstanceRepositoryJpa instanceRepositoryJpa;

    /**
     * @since 3.0.0
     */
    @Autowired
    private NetworkInstanceRepository networkInstanceRepository;


    public FrequencyMonitorLogListDtoLoader() {
    }

    public FrequencyMonitorLogListDto load(FrequencyMonitorLogListDto listDto) {

        loadInstanceDtos(listDto);
        listDto.setCleanMonitorLogFrequency(Application.systemSetting().cleanMonitorLogFrequency());

        //page 从0开始
        PageRequest page = PageRequest.of(listDto.getPageNumber(), listDto.getPerPageSize());
        Page<FrequencyMonitorLog> aPage = logRepository.findAll((Specification<FrequencyMonitorLog>) (root, query, criteriaBuilder) -> {
            List<Predicate> predicates = new ArrayList<>();
            //查询条件   see FrequencyMonitorLogListQueryHelper
            if (SecurityUtils.currentUser() == null) {
                predicates.add(criteriaBuilder.equal(root.get(listDto.isNetworkInstance() ? "networkInstance" : "instance").get("privateInstance"), false));
            }
            final int normal = listDto.getNormal();
            if (normal == 1 || normal == 2) {
                predicates.add(criteriaBuilder.equal(root.get("normal"), normal == 1));
            }
            final String instanceGuid = listDto.getInstanceGuid();
            if (StringUtils.isNotEmpty(instanceGuid)) {
                predicates.add(criteriaBuilder.equal(root.get(listDto.isNetworkInstance() ? "networkInstance" : "instance").get("guid"), instanceGuid));
            } else {
                //限制范围
                predicates.add(criteriaBuilder.isNotNull(root.get(listDto.isNetworkInstance() ? "networkInstance" : "instance")));
            }
            String groupId = listDto.getGroupId();
            if (StringUtils.isNotEmpty(groupId)) {
                predicates.add(criteriaBuilder.like(root.get(listDto.isNetworkInstance() ? "networkInstance" : "instance").get("groupId"), "%" + groupId + "%"));
            }

            predicates.add(criteriaBuilder.equal(root.get("archived"), false));
            //sort
            final FrequencyMonitorLogListDto.OrderItem orderItem = listDto.getOrderItem();
            Order order = criteriaBuilder.desc(root.get(orderItem.getField()));
            return query.where(predicates.toArray(new Predicate[0])).orderBy(order).getRestriction();
        }, page);

        List<FrequencyMonitorLogDto> dtos = FrequencyMonitorLogDto.toDtos(aPage.getContent());
        return listDto.load(aPage.getTotalElements(), dtos);
    }

    private void loadInstanceDtos(FrequencyMonitorLogListDto listDto) {
        List<CommonsInstanceDto> instanceDtos = listDto.getInstanceDtos();
        if (listDto.isNetworkInstance()) {
            List<NetworkInstance> networkInstances;
            if (SecurityUtils.currentUser() == null) {
                networkInstances = networkInstanceRepository.findHadLogPublicInstances();
            } else {
                networkInstances = networkInstanceRepository.findHadLogInstances();
            }
            for (NetworkInstance networkInstance : networkInstances) {
                instanceDtos.add(new CommonsInstanceDto(networkInstance));
            }
        } else {
            List<ApplicationInstance> instances;
            if (SecurityUtils.currentUser() == null) {
                instances = instanceRepositoryJpa.findHadLogPublicInstances();
            } else {
                instances = instanceRepositoryJpa.findHadLogInstances();
            }

            for (ApplicationInstance instance : instances) {
                instanceDtos.add(new CommonsInstanceDto(instance));
            }
        }

    }
}
