package com.andaily.hb.service.impl;

import com.andaily.hb.domain.dto.user.*;
import com.andaily.hb.domain.shared.security.AndailyUserDetails;
import com.andaily.hb.domain.shared.security.SecurityUtils;
import com.andaily.hb.domain.user.User;
import com.andaily.hb.domain.user.WeixinUser;
import com.andaily.hb.infrastructure.MatchUtils;
import com.andaily.hb.infrastructure.jpa.UserRepositoryJpa;
import com.andaily.hb.infrastructure.jpa.WeixinUserRepository;
import com.andaily.hb.service.UserService;
import com.andaily.hb.service.operation.user.RegisterUserHandler;
import com.andaily.hb.service.operation.user.SystemSettingDtoLoader;
import com.andaily.hb.service.operation.user.SystemSettingUpdater;
import com.andaily.hb.service.operation.user.UserFormDtoPersister;
import jakarta.persistence.criteria.Order;
import jakarta.persistence.criteria.Predicate;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.jpa.domain.Specification;
import org.springframework.security.core.userdetails.UserDetails;
import org.springframework.security.core.userdetails.UsernameNotFoundException;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

/**
 * @author Shengzhao Li
 */
@Service("userService")
public class UserServiceImpl implements UserService {

    @Autowired
    private UserRepositoryJpa userRepository;

    /**
     * @since 3.0.0
     */
    @Autowired
    private WeixinUserRepository weixinUserRepository;


    @Transactional(readOnly = true)
    @Override
    public UserDetails loadUserByUsername(String username) throws UsernameNotFoundException {
        User user = userRepository.findByUsername(username);
        if (user == null) {
            throw new UsernameNotFoundException("Not found any user for username[" + username + "]");
        }
        return new AndailyUserDetails(user);
    }


    //    @Transactional(readOnly = true)
    @Override
    public UserListDto loadUserListDto(UserListDto listDto) {
        final Map<String, Object> map = listDto.queryMap();
        PageRequest page = PageRequest.of(listDto.getPageNumber(), listDto.getPerPageSize());
        Page<User> userPage = userRepository.findAll((Specification<User>) (root, query, criteriaBuilder) -> {
            List<Predicate> predicates = new ArrayList<>();
            //查询条件
            final String username = (String) map.get("username");
            if (StringUtils.isNotEmpty(username)) {
                predicates.add(criteriaBuilder.like(root.get("username"), "%" + username + "%"));
            }
            predicates.add(criteriaBuilder.equal(root.get("archived"), false));
            //sort
            Order order = criteriaBuilder.desc(root.get("createTime"));
            return query.where(predicates.toArray(new Predicate[0])).orderBy(order).getRestriction();
        }, page);

        return listDto.load(userPage.getTotalElements(), UserDto.toDtos(userPage.getContent()));
    }


    //    @Transactional(readOnly = true)
    @Override
    public UserFormDto loadUserFormDto(String guid) {
        if (MatchUtils.isCreate(guid)) {
            UserFormDto formDto = new UserFormDto();
            formDto.setGuid(guid);
            return formDto;
        } else {
            return new UserFormDto(userRepository.findByGuid(guid));
        }
    }


    //    @Transactional(readOnly = true)
    @Override
    public boolean isExistUsername(String username) {
        User user = userRepository.findByUsernameIgnoreArchived(username);
        return user != null;
    }


    @Transactional
    @Override
    public void persistUserFormDto(UserFormDto formDto) {
        UserFormDtoPersister persister = new UserFormDtoPersister(formDto);
        persister.persist();
    }


    @Transactional
    @Override
    public void deleteUser(String guid) {
        User user = userRepository.findByGuid(guid);
        user.deleteMe();
    }


    @Transactional
    @Override
    public ResetUserPasswordDto resetUserPassword(String guid) {
        User user = userRepository.findByGuid(guid);
        final String newPassword = user.resetPassword();
        return new ResetUserPasswordDto(user, newPassword);
    }

    @Transactional
    @Override
    public void updateUserProfile(UserProfileDto profileDto) {
        User user = userRepository.findByGuid(SecurityUtils.currentUser().guid());
        user.updatePassword(profileDto.getPassword());
    }


    //    @Transactional(readOnly = true)
    @Override
    public SystemSettingDto loadSystemSettingDto() {
        SystemSettingDtoLoader systemSettingDtoLoader = new SystemSettingDtoLoader();
        return systemSettingDtoLoader.load();
    }

    @Transactional
    @Override
    public void updateSystemSetting(SystemSettingDto settingDto) {
        SystemSettingUpdater updater = new SystemSettingUpdater(settingDto);
        updater.update();
    }

    @Transactional
    @Override
    public void registerUser(UserRegisterDto formDto) {
        RegisterUserHandler registerUserHandler = new RegisterUserHandler(formDto);
        registerUserHandler.handle();
    }


    //    @Transactional(readOnly = true)
    @Override
    public WeixinUserPaginated loadWeixinUserPaginated(WeixinUserPaginated listDto) {
        final Map<String, Object> map = listDto.queryMap();
        //page 从0开始
        PageRequest page = PageRequest.of(listDto.getPageNumber(), listDto.getPerPageSize());
        Page<WeixinUser> userPage = weixinUserRepository.findAll((Specification<WeixinUser>) (root, query, criteriaBuilder) -> {
            List<Predicate> predicates = new ArrayList<>();
            //查询条件
            final String nickName = (String) map.get("nickName");
            if (StringUtils.isNotEmpty(nickName)) {
                predicates.add(criteriaBuilder.like(root.get("nickName"), "%" + nickName + "%"));
            }
            predicates.add(criteriaBuilder.equal(root.get("archived"), false));
            //sort
            Order order = criteriaBuilder.desc(root.get("createTime"));
            return query.where(predicates.toArray(new Predicate[0])).orderBy(order).getRestriction();
        }, page);

        return listDto.load(userPage.getTotalElements(), WeixinUserDto.toDtos(userPage.getContent()));
    }
}