package com.andaily.hb.service.impl;

import com.andaily.hb.domain.dto.network.NetworkInstanceDto;
import com.andaily.hb.domain.dto.network.NetworkInstanceFormDto;
import com.andaily.hb.domain.dto.network.NetworkInstanceListDto;
import com.andaily.hb.domain.network.NetworkInstance;
import com.andaily.hb.domain.shared.Application;
import com.andaily.hb.domain.shared.security.SecurityUtils;
import com.andaily.hb.domain.user.User;
import com.andaily.hb.infrastructure.jpa.NetworkInstanceRepository;
import com.andaily.hb.service.NetworkInstanceService;
import com.andaily.hb.service.operation.network.MonitoringNetworkInstanceKiller;
import com.andaily.hb.service.operation.network.NetworkInstanceDeleter;
import com.andaily.hb.service.operation.network.NetworkInstanceEnabler;
import com.andaily.hb.service.operation.network.NetworkInstanceFormPersister;
import jakarta.persistence.criteria.Order;
import jakarta.persistence.criteria.Predicate;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.jpa.domain.Specification;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

/**
 * 2023/12/13 20:25
 *
 * @author Shengzhao Li
 * @since 3.0.0
 */
@Service
public class NetworkInstanceServiceImpl implements NetworkInstanceService {

    @Autowired
    private NetworkInstanceRepository instanceRepository;


    @Autowired
    private NetworkInstanceEnabler networkInstanceEnabler;

    @Autowired
    private MonitoringNetworkInstanceKiller networkInstanceKiller;


    @Autowired
    private NetworkInstanceDeleter networkInstanceDeleter;

    @Autowired
    private NetworkInstanceFormPersister networkInstanceFormPersister;


    /**
     * {@inheritDoc}
     */
    @Override
    public NetworkInstanceListDto loadNetworkInstanceListDto(NetworkInstanceListDto listDto) {

        final Map<String, Object> map = listDto.queryMap();
        //page 从0开始
        PageRequest page = PageRequest.of(listDto.getPageNumber(), listDto.getPerPageSize());
        Page<NetworkInstance> aPage = instanceRepository.findAll((Specification<NetworkInstance>) (root, query, criteriaBuilder) -> {
            List<Predicate> predicates = new ArrayList<>();
            final String enabled = (String) map.get("enabled");
            if (StringUtils.isNotEmpty(enabled)) {
                predicates.add(criteriaBuilder.equal(root.get("enabled"), "true".equalsIgnoreCase(enabled)));
            }
            final String email = (String) map.get("email");
            if (StringUtils.isNotEmpty(email)) {
                predicates.add(criteriaBuilder.like(root.get("email"), "%" + email + "%"));
            }
            final String groupId = listDto.getGroupId();
            if (StringUtils.isNotEmpty(groupId)) {
                predicates.add(criteriaBuilder.like(root.get("groupId"), "%" + groupId + "%"));
            }
            if (SecurityUtils.currentUser() == null) {
                predicates.add(criteriaBuilder.equal(root.get("privateInstance"), false));
            }
            final User user = SecurityUtils.currentUser();
            if (user != null && user.registerUser()) {
                predicates.add(criteriaBuilder.equal(root.get("creator"), user));
            }
            final String instanceName = (String) map.get("instanceName");
            if (StringUtils.isNotEmpty(instanceName)) {
                predicates.add(criteriaBuilder.like(root.get("instanceName"), "%" + instanceName + "%"));
            }

            predicates.add(criteriaBuilder.equal(root.get("archived"), false));
            //sort
            Order order = criteriaBuilder.desc(root.get("createTime"));
            return query.where(predicates.toArray(new Predicate[0])).orderBy(order).getRestriction();
        }, page);

        List<NetworkInstanceDto> dtos = NetworkInstanceDto.toDtos(aPage.getContent());
        listDto.load(aPage.getTotalElements(), dtos);

        return listDto;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public boolean enableNetworkInstance(String guid) {
        return networkInstanceEnabler.enable(guid);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public boolean stopMonitoringNetworkInstance(String guid) {
        return networkInstanceKiller.kill(guid);
    }


    /**
     * {@inheritDoc}
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public boolean deleteNetworkInstance(String guid) {
        return networkInstanceDeleter.delete(guid);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public NetworkInstanceFormDto loadNetworkInstanceFormDto(String guid) {
        NetworkInstanceFormDto formDto;

        if (StringUtils.isNotEmpty(guid)) {
            NetworkInstance instance = instanceRepository.findByGuid(guid);
            if (instance.enabled()) {
                throw new IllegalStateException("Only Disabled ApplicationInstance support edit");
            }
            formDto = new NetworkInstanceFormDto(instance);
        } else {
            formDto = new NetworkInstanceFormDto();
        }

        //设置系统的 proxy是否启用
        formDto.setProxyEnabled(Application.systemSetting().proxyEnabled());
        return formDto;
    }


    /**
     * {@inheritDoc}
     */
    @Override
    public void persistNetworkInstance(NetworkInstanceFormDto formDto) {
        networkInstanceFormPersister.persist(formDto);
    }

}
