package com.andaily.hb.service.impl;

import com.andaily.hb.domain.dto.notify.EmailConfigFormDto;
import com.andaily.hb.domain.dto.notify.EmailConfigTestResult;
import com.andaily.hb.domain.notify.EmailConfig;
import com.andaily.hb.domain.shared.security.SecurityUtils;
import com.andaily.hb.infrastructure.AesUtils;
import com.andaily.hb.infrastructure.jpa.EmailConfigRepository;
import com.andaily.hb.infrastructure.mail.JavaMailSenderHolder;
import com.andaily.hb.infrastructure.mail.MailTransmitter;
import com.andaily.hb.service.EmailConfigService;
import jakarta.mail.internet.MimeMessage;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.mail.javamail.JavaMailSenderImpl;
import org.springframework.mail.javamail.MimeMessageHelper;
import org.springframework.stereotype.Service;

import java.util.Date;
import java.util.Properties;

import static com.andaily.hb.domain.shared.Application.ENCODING;

/**
 * 2023/12/4 14:09
 *
 * @author Shengzhao Li
 * @since 3.0.0
 */
@Service
public class EmailConfigServiceImpl implements EmailConfigService {

    private static final Logger LOG = LoggerFactory.getLogger(EmailConfigServiceImpl.class);

    @Autowired
    private EmailConfigRepository configRepository;


    @Autowired
    private JavaMailSenderHolder mailSenderHolder;

    @Autowired
    private MailTransmitter mailTransmitter;


    /**
     * {@inheritDoc}
     */
    @Override
    public EmailConfigFormDto loadEmailConfigFormDto() {
        EmailConfig config = configRepository.findEmailConfig();
        if (config == null) {
            return new EmailConfigFormDto();
        } else {
            return new EmailConfigFormDto(config);
        }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void saveEmailConfig(EmailConfigFormDto formDto) {
        EmailConfig config = configRepository.findEmailConfig();
        if (config == null) {
            //新建
            config = new EmailConfig();
        }

        config.smtpAuth(formDto.isSmtpAuth())
                .smtpPort(formDto.getSmtpPort())
                .sendEmail(formDto.getSendEmail())
                .enabled(formDto.isEnabled())
                .developEnvironment(formDto.isDevelopEnvironment())
                .developEmail(formDto.getDevelopEmail())
                .hostName(formDto.getHostName())
                .sendMailUseThread(formDto.isSendMailUseThread());

        String password = formDto.getPassword();
        if (StringUtils.isNotBlank(password)) {
            config.password(AesUtils.encrypt(password));
        }

        configRepository.save(config);

        mailSenderHolder.update(config);
        mailTransmitter.updateMailProperties(config);
        if (LOG.isDebugEnabled()) {
            LOG.debug("<{}> Update EmailConfig [{}]", SecurityUtils.currentUsername(), config);
        }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public EmailConfigTestResult testEmailConfig(EmailConfigFormDto formDto) {

        final JavaMailSenderImpl javaMailSender = new JavaMailSenderImpl();

        javaMailSender.setUsername(formDto.getSendEmail());
        javaMailSender.setPassword(formDto.getPassword());

        javaMailSender.setHost(formDto.getHostName());
        javaMailSender.setPort(formDto.getSmtpPort());
        javaMailSender.setDefaultEncoding(ENCODING);

        Properties properties = new Properties();
        properties.setProperty("mail.smtp.auth", "true");
        if (formDto.isSmtpAuth()) {
            properties.setProperty("mail.smtp.socketFactory.class", "javax.net.ssl.SSLSocketFactory");
        }
        javaMailSender.setJavaMailProperties(properties);

        try {
            MimeMessage message = javaMailSender.createMimeMessage();
            MimeMessageHelper helper = new MimeMessageHelper(message, true);

            helper.setFrom(formDto.getSendEmail());

            helper.setTo(formDto.getDevelopEmail());
            helper.setSentDate(new Date());
            helper.setSubject("Test Email from HeartBeat ");
            helper.setText("Just test email config correct, please ignored. <br/> <strong>HeartBeat</strong>", true);

            javaMailSender.send(message);
            return new EmailConfigTestResult(true, "Success");
        } catch (Exception e) {
            LOG.error("Send Test-Email error", e);
            return new EmailConfigTestResult(false, e.getMessage());
        }


    }
}
