package com.andaily.hb.infrastructure.mail;

import com.andaily.hb.domain.notify.EmailConfig;
import com.andaily.hb.infrastructure.jpa.EmailConfigRepository;
import jakarta.mail.MessagingException;
import jakarta.mail.internet.InternetAddress;
import jakarta.mail.internet.MimeMessage;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.io.Resource;
import org.springframework.mail.javamail.JavaMailSender;
import org.springframework.mail.javamail.MimeMessageHelper;
import org.springframework.stereotype.Component;
import org.springframework.util.Assert;

import java.util.Arrays;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;

/**
 * 邮件发送器
 *
 * @author Shengzhao Li
 */
@Component
public class MailTransmitter extends Thread implements InitializingBean {

    private static final Logger LOG = LoggerFactory.getLogger(MailTransmitter.class);


    protected static MailProperties mailProperties = new MailProperties();

    /**
     * @since 3.0.0
     */
    @Autowired
    private EmailConfigRepository configRepository;

    @Autowired
    private JavaMailSenderHolder mailSenderHolder;

    /**
     * mail fields
     */
    protected String from;
    protected String[] to;
    protected String subject;
    protected String content;
    protected Date sendDate;

    protected String[] cc;

    protected Map<String, Resource> attachments = new HashMap<>();
    protected Map<String, Resource> inlineResources = new HashMap<>();

    protected transient MailTransmitResult sendResult;

    /**
     * For spring container use it
     */
    public MailTransmitter() {
    }

    public MailTransmitter(String subject, String content, String... to) {
        this.to = to;
        this.subject = subject;
        this.content = content;
        //default value
        this.from = mailProperties.defaultFromAddress();
        this.sendDate = new Date();
    }

    public MailTransmitter from(String from) {
        this.from = from;
        return this;
    }

    @SuppressWarnings("unchecked")
    public <T extends MailTransmitter> T cc(String... cc) {
        this.cc = cc;
        return (T) this;
    }

    public MailTransmitter sendDate(Date sendDate) {
        this.sendDate = sendDate;
        return this;
    }

    public MailTransmitter addAttachment(String attachName, Resource resource) {
        this.attachments.put(attachName, resource);
        return this;
    }

    public MailTransmitter addInlineResource(String inlineName, Resource inlineResource) {
        this.inlineResources.put(inlineName, inlineResource);
        return this;
    }

    //For subclass override it
    protected JavaMailSender retrieveMailSender() {
        return mailProperties.defaultMailSender();
    }

    /**
     * transmit.
     * Note: If use new thread send mail, will get result is NULL.
     *
     * @return MailTransmitResult or null
     */
    public MailTransmitResult transmit() {
        if (!mailProperties.enableMailService()) {
            if (LOG.isWarnEnabled()) {
                LOG.warn("Mail service is not enabled, so mail will not be sent. checking Mail-Config ");
            }
            return null;
        }

        if (mailProperties.sendMailUseThread()) {
            this.start();
        } else {
            this.run();
        }
        return sendResult;
    }

    @Override
    public void run() {
        try {
            long start = System.currentTimeMillis();

            final JavaMailSender mailSender = retrieveMailSender();
            if (mailSender == null) {
                if (LOG.isWarnEnabled()) {
                    LOG.warn("****************\nJavaMailSender is null, please config Email firstly.\n****************");
                }
                return;
            }
            MimeMessage message = mailSender.createMimeMessage();
            MimeMessageHelper helper = new MimeMessageHelper(message, true);

            helper.setFrom(from);
            setCC(helper);
            setTo(helper);

            helper.setSentDate(sendDate);
            helper.setSubject(subject);
            helper.setText(content, true);

            //attachments
            addAttachments(helper);
            //inline resource.
            addInlineResources(helper);
            mailSender.send(message);
            if (LOG.isDebugEnabled()) {
                LOG.debug(">>>>> Send email to [{}] finished.", Arrays.toString(to));
            }
            sendResult = new MailTransmitResult().success(true).costTime(System.currentTimeMillis() - start);
        } catch (Exception e) {
            LOG.error("Send Email error", e);
            sendResult = new MailTransmitResult().success(false).exception(e);
        }
    }

    private void setTo(MimeMessageHelper helper) throws MessagingException {
        if (mailProperties.developEnvironment()) {
            LOG.warn("NOTE: Current is Test Environment, The email will be send to developer [{}]", mailProperties.developEmailAddress());
            InternetAddress[] addresses = InternetAddress.parse(mailProperties.developEmailAddress());
            helper.setTo(addresses);
        } else {
            helper.setTo(to);
        }
    }

    //ignore development environment
    private void setCC(MimeMessageHelper helper) throws MessagingException {
        if (!mailProperties.developEnvironment() && cc != null && cc.length > 0) {
            helper.setCc(cc);
        }
    }

    private void addAttachments(MimeMessageHelper helper) throws MessagingException {
        for (String name : attachments.keySet()) {
            helper.addAttachment(name, attachments.get(name));
        }
    }

    private void addInlineResources(MimeMessageHelper helper) throws MessagingException {
        for (String name : inlineResources.keySet()) {
            helper.addInline(name, inlineResources.get(name));
        }
    }


    @Override
    public void afterPropertiesSet() throws Exception {
        Assert.notNull(this.configRepository, "configRepository is null");

        EmailConfig emailConfig = configRepository.findEmailConfig();
        if (emailConfig == null) {
            if (LOG.isWarnEnabled()) {
                LOG.warn("****************\nEmailConfig not found, please config Email firstly.\n****************");
            }
            return;
        }
        updateMailProperties(emailConfig);

        JavaMailSender javaMailSender = mailSenderHolder.mailSender();
        if (javaMailSender != null) {
            mailProperties.defaultMailSender(javaMailSender);
        } else {
            mailProperties.defaultMailSender(mailSenderHolder.create(emailConfig));
        }

    }

    /**
     * 更新邮件配置
     *
     * @param emailConfig EmailConfig
     * @since 3.0.0
     */
    public void updateMailProperties(EmailConfig emailConfig) {
        if (emailConfig != null) {
            mailProperties.enableMailService(emailConfig.enabled());
            mailProperties.defaultFromAddress(emailConfig.sendEmail());
            mailProperties.developEnvironment(emailConfig.developEnvironment());
            mailProperties.developEmailAddress(emailConfig.developEmail());
            mailProperties.sendMailUseThread(emailConfig.sendMailUseThread());

            mailProperties.enableMailService(true);
        } else {
            mailProperties.enableMailService(false);
            if (LOG.isWarnEnabled()) {
                LOG.warn("No EmailConfig found, so mail service initial failed, Please check email configuration.");
            }
        }
    }


    /**
     * 内部类，用于保存邮件发送的相关属性
     *
     * @since 3.0.0
     */
    private static class MailProperties {

        JavaMailSender defaultMailSender;

        String defaultFromAddress;
        boolean developEnvironment;
        String developEmailAddress;

        boolean sendMailUseThread;

        /**
         * 是否启用了邮件服务
         *
         * @since 3.0.0
         */
        boolean enableMailService;

        public MailProperties() {
        }

        public JavaMailSender defaultMailSender() {
            return defaultMailSender;
        }

        public MailProperties defaultMailSender(JavaMailSender defaultMailSender) {
            this.defaultMailSender = defaultMailSender;
            return this;
        }

        public String defaultFromAddress() {
            return defaultFromAddress;
        }

        public MailProperties defaultFromAddress(String defaultFromAddress) {
            this.defaultFromAddress = defaultFromAddress;
            return this;
        }

        public boolean developEnvironment() {
            return developEnvironment;
        }

        public MailProperties developEnvironment(boolean developEnvironment) {
            this.developEnvironment = developEnvironment;
            return this;
        }

        public String developEmailAddress() {
            return developEmailAddress;
        }

        public MailProperties developEmailAddress(String developEmailAddress) {
            this.developEmailAddress = developEmailAddress;
            return this;
        }

        public boolean sendMailUseThread() {
            return sendMailUseThread;
        }

        public MailProperties sendMailUseThread(boolean sendMailUseThread) {
            this.sendMailUseThread = sendMailUseThread;
            return this;
        }

        public boolean enableMailService() {
            return enableMailService;
        }

        public MailProperties enableMailService(boolean enableMailService) {
            this.enableMailService = enableMailService;
            return this;
        }
    }
}