package com.andaily.hb.infrastructure.mail;

import com.andaily.hb.domain.notify.EmailConfig;
import com.andaily.hb.infrastructure.AesUtils;
import com.andaily.hb.infrastructure.jpa.EmailConfigRepository;
import org.slf4j.Logger;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.mail.javamail.JavaMailSender;
import org.springframework.mail.javamail.JavaMailSenderImpl;
import org.springframework.stereotype.Component;

import java.util.Properties;

import static com.andaily.hb.domain.shared.Application.ENCODING;

/**
 * 2023/12/4 10:19
 * <p>
 * Mail sender holder.
 *
 * @author Shengzhao Li
 * @since 3.0.0
 */
@Component
public class JavaMailSenderHolder implements InitializingBean {

    private static final Logger LOG = org.slf4j.LoggerFactory.getLogger(JavaMailSenderHolder.class);


    @Autowired
    private EmailConfigRepository configRepository;

    private JavaMailSenderImpl mailSender;


    public JavaMailSenderHolder() {
    }


    public JavaMailSender mailSender() {
        return mailSender;
    }

    /**
     * 更新邮件配置
     */
    public JavaMailSender update(EmailConfig emailConfig) {
        if (this.mailSender == null) {
            return create(emailConfig);
        } else {
            this.setFields(emailConfig, this.mailSender);
        }

        return this.mailSender;
    }

    protected JavaMailSenderImpl create(EmailConfig emailConfig) {

        final JavaMailSenderImpl javaMailSender = new JavaMailSenderImpl();
        setFields(emailConfig, javaMailSender);

        if (this.mailSender == null) {
            this.mailSender = javaMailSender;
        }
        return javaMailSender;
    }

    private void setFields(EmailConfig emailConfig, JavaMailSenderImpl javaMailSender) {
        javaMailSender.setUsername(emailConfig.sendEmail());
        javaMailSender.setPassword(AesUtils.decrypt(emailConfig.password()));

        javaMailSender.setHost(emailConfig.hostName());
        javaMailSender.setPort(emailConfig.smtpPort());
        javaMailSender.setDefaultEncoding(ENCODING);

        Properties properties = new Properties();
        properties.setProperty("mail.smtp.auth", "true");
        if (emailConfig.smtpAuth()) {
            properties.setProperty("mail.smtp.socketFactory.class", "javax.net.ssl.SSLSocketFactory");
        }
        javaMailSender.setJavaMailProperties(properties);
    }

    @Override
    public void afterPropertiesSet() throws Exception {
        //load EmailConfig
        if (this.mailSender == null) {
            final EmailConfig emailConfig = configRepository.findEmailConfig();
            if (emailConfig != null) {
                this.mailSender = create(emailConfig);
            } else {
                if (LOG.isWarnEnabled()) {
                    LOG.warn("EmailConfig not found, please config Email firstly.");
                }
            }
        } else {
            LOG.info("JavaMailSender: {} already initialized.", this.mailSender);
        }

    }
}
