package com.andaily.hb.infrastructure.jpa;

import com.andaily.hb.domain.application.ApplicationInstance;
import com.andaily.hb.domain.log.FrequencyMonitorLog;
import com.andaily.hb.domain.network.NetworkInstance;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.jpa.repository.JpaRepository;
import org.springframework.data.jpa.repository.JpaSpecificationExecutor;
import org.springframework.data.jpa.repository.Modifying;
import org.springframework.data.jpa.repository.Query;

import java.util.Date;
import java.util.List;

/**
 * 2023/11/21 22:51
 *
 * @author Shengzhao Li
 * @since 3.0.0
 */
public interface FrequencyMonitorLogRepository extends JpaRepository<FrequencyMonitorLog, Integer>, JpaSpecificationExecutor<FrequencyMonitorLog> {


    /**
     * Find last logs by current log
     */
    @Query("select ml from FrequencyMonitorLog ml where ml.archived = false and ml.instance = :instance " +
            " and ml.createTime < :createTime and ml.id < :currId order by ml.createTime desc ")
    List<FrequencyMonitorLog> findLastLogsByCurrentLog(ApplicationInstance instance, int currId, Date createTime, PageRequest pageRequest);

    /**
     * Find last logs by current log (network)
     *
     * @since 3.0.0
     */
    @Query("select ml from FrequencyMonitorLog ml where ml.archived = false and ml.networkInstance = :instance " +
            " and ml.createTime < :createTime and ml.id < :currId order by ml.createTime desc ")
    List<FrequencyMonitorLog> findNetworkLastLogsByCurrentLog(NetworkInstance instance, int currId, Date createTime, PageRequest pageRequest);

    /**
     * Find frequency monitor logs
     */
    @Query("select ml from FrequencyMonitorLog ml where ml.archived = false and ml.instance.guid = :instanceGuid " +
            " and ml.createTime > :minCreateTime order by ml.createTime desc ")
    List<FrequencyMonitorLog> findFrequencyMonitorLogs(String instanceGuid, Date minCreateTime);

    /**
     * Find frequency monitor logs (network)
     *
     * @since 3.0.0
     */
    @Query("select ml from FrequencyMonitorLog ml where ml.archived = false and ml.networkInstance.guid = :instanceGuid " +
            " and ml.createTime > :minCreateTime order by ml.createTime desc ")
    List<FrequencyMonitorLog> findNetworkFrequencyMonitorLogs(String instanceGuid, Date minCreateTime);

    /**
     * Find latest frequency monitor logs
     */
    @Query("select ml from FrequencyMonitorLog ml where ml.archived = false and ml.instance = :instance order by ml.createTime desc ")
    List<FrequencyMonitorLog> findLatestFrequencyMonitorLogs(ApplicationInstance instance, PageRequest pageRequest);

    /**
     * Find latest frequency monitor logs  (network)
     *
     * @since 3.0.0
     */
    @Query("select ml from FrequencyMonitorLog ml where ml.archived = false and ml.networkInstance = :instance order by ml.createTime desc ")
    List<FrequencyMonitorLog> findLatestFrequencyMonitorLogs(NetworkInstance instance, PageRequest pageRequest);

    /**
     * Find first monitoring log createTime after specify time
     */
    @Query(" select ml.createTime from  FrequencyMonitorLog ml where ml.archived = false " +
            " and ml.instance.guid = :instanceGuid and ml.createTime > :time order by ml.createTime asc limit 1 ")
    Date findFirstMonitoringLogTimeAfterSpecifyTime(String instanceGuid, Date time);

    /**
     * Find first monitoring log createTime after specify time  (network)
     */
    @Query(" select ml.createTime from  FrequencyMonitorLog ml where ml.archived = false " +
            " and ml.networkInstance.guid = :instanceGuid and ml.createTime > :time order by ml.createTime asc limit 1 ")
    Date findFirstNetworkMonitoringLogTimeAfterSpecifyTime(String instanceGuid, Date time);

    /**
     * Find first monitoring log createTime
     */
    @Query(" select ml.createTime from  FrequencyMonitorLog ml where ml.archived = false " +
            " and ml.instance.guid = :instanceGuid and ml.normal = :normal order by ml.createTime asc limit 1 ")
    Date findFirstMonitoringLogTime(String instanceGuid, boolean normal);

    /**
     * Find first monitoring log createTime (network)
     */
    @Query(" select ml.createTime from  FrequencyMonitorLog ml where ml.archived = false " +
            " and ml.networkInstance.guid = :instanceGuid and ml.normal = :normal order by ml.createTime asc limit 1 ")
    Date findFirstNetworkMonitoringLogTime(String instanceGuid, boolean normal);

    /**
     * Find last not normal frequency monitor log
     */
    @Query(" select ml from FrequencyMonitorLog ml where ml.archived = false and ml.instance.guid = :instanceGuid and ml.normal = false " +
            " and ml.createTime < :createTime order by ml.createTime desc limit 1 ")
    FrequencyMonitorLog findLastNotNormalFrequencyMonitorLog(String instanceGuid, Date createTime);

    /**
     * Find last not normal frequency monitor log (network)
     */
    @Query(" select ml from FrequencyMonitorLog ml where ml.archived = false and ml.networkInstance.guid = :instanceGuid and ml.normal = false " +
            " and ml.createTime < :createTime order by ml.createTime desc limit 1 ")
    FrequencyMonitorLog findNetworkLastNotNormalFrequencyMonitorLog(String instanceGuid, Date createTime);

    /**
     * Amount of monitoring reminder logs
     */
    @Query(" select count(ml.guid) from  FrequencyMonitorLog ml where ml.archived = false and ml.instance.guid = :instanceGuid and ml.normal = :normal ")
    int amountOfInstanceMonitorLogs(String instanceGuid, boolean normal);

    /**
     * Amount of monitoring reminder logs (network)
     */
    @Query(" select count(ml.guid) from  FrequencyMonitorLog ml where ml.archived = false and ml.networkInstance.guid = :instanceGuid and ml.normal = :normal ")
    int amountOfNetworkInstanceMonitorLogs(String instanceGuid, boolean normal);

    /**
     * Amount of frequency monitor logs before date
     */
    @Query(" select count(ml.guid) from  FrequencyMonitorLog ml where ml.archived = false and ml.createTime < :date ")
    long amountOfFrequencyMonitorLogsBeforeDate(Date date);

    /**
     * Delete frequency monitor logs before date
     */
    @Modifying
    @Query(" delete from FrequencyMonitorLog ml where ml.archived = false and ml.createTime < :date ")
    int deleteFrequencyMonitorLogsBeforeDate(Date date);

    /**
     * Delete network instance logs
     *
     * @since 3.0.0
     */
    @Modifying
    @Query("delete from FrequencyMonitorLog ml where ml.networkInstance.guid = :instanceGuid")
    int deleteNetworkInstanceLogs(String instanceGuid);


}
