package com.andaily.hb.infrastructure.hibernate.queryhelper.impl;

import com.andaily.hb.domain.application.ApplicationInstance;
import com.andaily.hb.domain.shared.security.SecurityUtils;
import com.andaily.hb.infrastructure.MatchUtils;
import com.andaily.hb.infrastructure.hibernate.queryhelper.AbstractQueryHelper;
import com.andaily.hb.infrastructure.hibernate.queryhelper.Filter;
import com.andaily.hb.infrastructure.hibernate.queryhelper.ParameterFilter;
import com.andaily.hb.infrastructure.hibernate.queryhelper.SortCriterionFilter;
import jakarta.persistence.EntityManager;
import jakarta.persistence.Query;


import java.util.Map;

/**
 * 15-1-3
 *
 * @author Shengzhao Li
 */
public class HBSearchInstancesQueryHelper extends AbstractQueryHelper<ApplicationInstance> {

    private final Map<String, Object> map;

    public HBSearchInstancesQueryHelper(EntityManager entityManager, Map<String, Object> map) {
        super(entityManager);
        this.map = map;

        addSearchFilter();
        addPrivateInstanceFilter();
        addSortFilter();
    }

    private void addPrivateInstanceFilter() {
        addFilter(new Filter() {
            @Override
            public String getSubHql() {
                if (SecurityUtils.currentUser() == null) {
                    return " and ai.privateInstance = false ";
                } else {
                    return "";
                }
            }
        });
    }

    private void addSortFilter() {
        addSortCriterionFilter(new SortCriterionFilter() {
            @Override
            public String getSubHql() {
                return " ai.createTime desc ";
            }
        });
    }

    private void addSearchFilter() {
        final String key = getKey();
        final boolean positiveNumber = MatchUtils.isPositiveNumber(key);

        addFilter(new ParameterFilter() {
            @Override
            public void setParameter(Query query) {
                query.setParameter("likeKey", "%" + key + "%");
                if (positiveNumber) {
                    query.setParameter("key", Integer.valueOf(key));
                }
            }

            @Override
            public String getSubHql() {
                return " and (ai.instanceName like :likeKey " +
                        " or ai.instanceURL.monitorUrl like :likeKey " +
                        " or ai.email like :likeKey " +
                        (positiveNumber ? " or ai.maxConnectionSeconds = :key " : "") + ")";
            }
        });
    }

    private String getKey() {
        return (String) map.get("key");
    }

    @Override
    public int getStartPosition() {
        return (Integer) map.get("startIndex");
    }

    @Override
    public int getItemsAmountPerPage() {
        return (Integer) map.get("perPageSize");
    }

    @Override
    public String getAmountHql() {
        return " select count(ai.id) from ApplicationInstance ai where ai.archived = false ";
    }

    @Override
    public String getResultHql() {
        return " from ApplicationInstance ai where ai.archived = false ";
    }
}
