package com.andaily.hb.domain.user;

import com.andaily.hb.domain.AbstractDomain;
import com.andaily.hb.domain.shared.BeanProvider;
import com.andaily.hb.domain.shared.security.SecurityUtils;
import com.andaily.hb.infrastructure.PasswordHandler;
import com.andaily.hb.infrastructure.jpa.UserRepositoryJpa;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import jakarta.persistence.*;

import java.io.Serial;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

/**
 * @author Shengzhao Li
 */
@Entity
@Table(name = "user_")
public class User extends AbstractDomain {

    @Serial
    private static final long serialVersionUID = -7873396364481790308L;

    private static final Logger LOG = LoggerFactory.getLogger(User.class);

    private final transient UserRepositoryJpa userRepository = BeanProvider.getBean(UserRepositoryJpa.class);

    @Column(name = "username", unique = true)
    private String username;

    @Column(name = "password")
    private String password;

    @Column(name = "phone")
    private String phone;

    @Column(name = "email")
    private String email;

    /**
     * Default user is initial when create database, do not delete
     */
    @Column(name = "default_user")
    private boolean defaultUser = false;

    @Column(name = "last_login_time")
    private Date lastLoginTime;

    /**
     * Register is null; otherwise is current logged user
     * */
    @ManyToOne
    @JoinColumn(name = "creator_id")
    private User creator;


    public User() {
    }

    public User(String username, String password, String phone, String email) {
        this.username = username;
        this.password = password;
        this.phone = phone;
        this.email = email;
    }


    public boolean registerUser() {
        return !defaultUser && creator == null;
    }

    public User creator() {
        return creator;
    }

    public User creator(User creator) {
        this.creator = creator;
        return this;
    }

    public List<Privilege> privileges() {
        List<UserPrivilege> userPrivileges = userRepository.findUserPrivileges(this);
        List<Privilege> privileges = new ArrayList<>(userPrivileges.size());
        for (UserPrivilege userPrivilege : userPrivileges) {
            privileges.add(userPrivilege.privilege());
        }
        return privileges;
    }

    public boolean defaultUser() {
        return defaultUser;
    }

    public String username() {
        return username;
    }

    public String password() {
        return password;
    }

    public String phone() {
        return phone;
    }

    public String email() {
        return email;
    }

    @Override
    public String toString() {
        final StringBuilder sb = new StringBuilder();
        sb.append("{username='").append(username).append('\'');
        sb.append(", phone='").append(phone).append('\'');
        sb.append(", id='").append(id).append('\'');
        sb.append(", guid='").append(guid).append('\'');
        sb.append(", defaultUser='").append(defaultUser).append('\'');
        sb.append(", email='").append(email).append('\'');
        sb.append('}');
        return sb.toString();
    }

    public User email(String email) {
        this.email = email;
        return this;
    }

    public User phone(String phone) {
        this.phone = phone;
        return this;
    }


    public User username(String username) {
        this.username = username;
        return this;
    }

    public String resetPassword() {
        String newOriginalPass = PasswordHandler.randomPassword();
        this.password = PasswordHandler.encryptPassword(newOriginalPass);
        LOG.debug("<{}> reset User [username={},guid={}] password", SecurityUtils.currentUsername(), username, guid);
        return newOriginalPass;
    }


    public Date lastLoginTime() {
        return lastLoginTime;
    }

    public void lastLoginTime(Date lastLoginTime) {
        this.lastLoginTime = lastLoginTime;
    }

    public User deleteMe() {
        this.archived(true);
        LOG.debug("<{}> delete User: {} [Logic delete]", SecurityUtils.currentUsername(), this);
        return this;
    }

    public User updatePassword(String newPassword) {
        this.password = PasswordHandler.encryptPassword(newPassword);
        LOG.debug("<{}> update user[{}] password", SecurityUtils.currentUsername(), this);
        return this;
    }
}