package com.andaily.hb.domain.shared.paginated;

import com.andaily.hb.domain.shared.security.SecurityUtils;
import org.springframework.data.domain.Page;


import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * @author Shengzhao Li
 */
public class DefaultPaginated<T> implements Paginated<T> {

    public static final int DEFAULT_PER_PAGE_SIZE = 10;

    protected List<T> list = new ArrayList<>();

    protected int perPageSize;
    protected long totalSize = 0L;

    /**
     * 从0开始,  v3.0.0
     */
    protected int pageNumber = 0;

    protected String sortName;
    protected PaginatedSort sort;

    protected String sortCriterion;

    private String searchId;


    public DefaultPaginated() {
        this(DEFAULT_PER_PAGE_SIZE);
    }

    public DefaultPaginated(int perPageSize) {
        this.perPageSize = perPageSize;
    }

    @Override
    public List<T> getList() {
        return list;
    }

    public Map<String, Object> queryMap() {
        Map<String, Object> map = new HashMap<>();
        map.put("user", SecurityUtils.currentUser());
        map.put("perPageSize", getPerPageSize());
        map.put("startIndex", getStartIndex());
        return map;
    }

    @Override
    public long getTotalPages() {
        if (totalSize % perPageSize == 0) {
            return totalSize / perPageSize;
        } else {
            return (totalSize / perPageSize) + 1;
        }
    }

    @Override
    public int getPageNumber() {
        return pageNumber;
    }

    /**
     * 前端展示的页码
     *
     * @since 3.0.0
     */
    public long getShowPageNumber() {
        if (this.getTotalPages() == 0) {
            return 0;
        }
        if (this.pageNumber == this.getTotalPages()) {
            return this.getTotalPages();
        }
        return this.pageNumber + 1;
    }

    /**
     * 最后一页的页码
     *
     * @return last page
     * @since 3.0.0
     */
    public long getLastPageNumber() {
        long totalPages = getTotalPages();
        if (totalPages > 0) {
            return totalPages - 1;
        } else {
            return totalPages;
        }
    }

    public void setSortCriterion(String sortCriterion) {
        this.sortCriterion = sortCriterion;
    }


    public String getSearchId() {
        return searchId;
    }

    @Override
    public int getPerPageSize() {
        return perPageSize;
    }

    @Override
    public long getTotalSize() {
        return totalSize;
    }

    @Override
    public String getSortName() {
        return sortName;
    }

    @Override
    public PaginatedSort getSort() {
        return sort;
    }

    public int getStartIndex() {
        return getPageNumber() * getPerPageSize();
    }

    @SuppressWarnings("unchecked")
    public <K extends Paginated> K load(PaginatedLoader<T> paginatedLoader) {
        if (this.totalSize == 0) {
            this.totalSize = paginatedLoader.loadTotalSize();
        }
        this.list = paginatedLoader.loadList();
        afterLoad();
        return (K) this;
    }

    /**
     * load by Page
     *
     * @since 3.0.0
     */
    @SuppressWarnings("unchecked")
    public <K extends Paginated> K load(Page<T> page) {
        if (this.totalSize == 0) {
            this.totalSize = page.getTotalElements();
        }
        this.list = page.getContent();
        afterLoad();
        return (K) this;
    }

    /**
     * load by data
     *
     * @since 3.0.0
     */
    @SuppressWarnings("unchecked")
    public <K extends Paginated> K load(long totalSize, List<T> content) {
        if (this.totalSize == 0) {
            this.totalSize = totalSize;
        }
        this.list = content;
        afterLoad();
        return (K) this;
    }

    public void afterLoad() {
        // Callback after load data.
    }

    public void setPerPageSize(int perPageSize) {
        this.perPageSize = perPageSize;
    }

    public void setTotalSize(int totalSize) {
        this.totalSize = totalSize;
    }

    public void setPageNumber(int pageNumber) {
        this.pageNumber = pageNumber;
    }

    public void setSortName(String sortName) {
        this.sortName = sortName;
    }

    public void setSort(PaginatedSort sort) {
        this.sort = sort;
    }

    public boolean isHasNext() {
        return getStartIndex() + this.perPageSize < totalSize;
    }

    public boolean isHasPrevious() {
        return getStartIndex() != 0;
    }


    public void setSearchId(String searchId) {
        this.searchId = searchId;
    }
}