package com.andaily.hb.domain.log.reminder;

import com.andaily.hb.domain.AbstractInstance;
import com.andaily.hb.domain.application.ApplicationInstance;
import com.andaily.hb.domain.log.FrequencyMonitorLog;
import com.andaily.hb.domain.log.MonitoringReminderLog;
import com.andaily.hb.infrastructure.STRender;
import com.andaily.hb.infrastructure.mail.MailTransmitter;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Arrays;
import java.util.Map;

/**
 * @author Shengzhao Li
 */
public class EmailPerMonitoringReminderSender extends AbstractPerMonitoringReminderSender {


    private static final Logger LOGGER = LoggerFactory.getLogger(EmailPerMonitoringReminderSender.class);


    private static final String CHANGE_NORMAL_EMAIL_SUBJECT = "Connect [%s] restore normally  -- HeartBeat";
    private static final String CHANGE_NORMAL_EMAIL_CONTENT_TEMPLATE = "templates/reminder/instance_change_normal_content.html";

    private static final String CHANGE_NOT_NORMAL_EMAIL_SUBJECT = "Unable to connect [%s]  -- HeartBeat";
    private static final String CHANGE_NOT_NORMAL_EMAIL_CONTENT_TEMPLATE = "templates/reminder/instance_change_not_normal_content.html";


    @Override
    public boolean support(FrequencyMonitorLog monitorLog) {
        return StringUtils.isNotEmpty(getEmail(monitorLog));
    }

    @Override
    public void send(MonitoringReminderLog reminderLog, FrequencyMonitorLog monitorLog) {
        reminderLog.receiveEmail(getEmail(monitorLog));
        final boolean normal = monitorLog.normal();

        String emailContent;
        if (normal) {
            emailContent = sendChangeNormalMail(monitorLog);
        } else {
            emailContent = sendChangeUnNormalMail(monitorLog);
        }

        reminderLog.emailContent(emailContent);
    }

    /**
     * @since 3.0.0
     */
    private String getEmail(FrequencyMonitorLog monitorLog) {
        ApplicationInstance instance = monitorLog.instance();
        if (instance != null) {
            return instance.email();
        } else {
            return monitorLog.networkInstance().email();
        }
    }

    /**
     * Subject: Unable to connect [instance_name]  -- HeartBeat
     */
    private String sendChangeUnNormalMail(FrequencyMonitorLog monitorLog) {
        final AbstractInstance instance = getInstance(monitorLog);

        String subject = String.format(CHANGE_NOT_NORMAL_EMAIL_SUBJECT, instance.instanceName());
        final String[] to = instance.emailAsArray();

        Map<String, Object> model = contentModel(monitorLog);
        STRender stRender = new STRender(CHANGE_NOT_NORMAL_EMAIL_CONTENT_TEMPLATE, model);
        final String content = stRender.render();

        sendEmail(to, subject, content);
        LOGGER.debug("Sent Change-Not-Normal Email of Instance[guid={},name={}], FrequencyMonitorLog is [{}]", instance.guid(), instance.instanceName(), monitorLog);

        return content;
    }

    /**
     * Subject: Connect [instance_name] restore normally  -- HeartBeat
     */
    private String sendChangeNormalMail(FrequencyMonitorLog monitorLog) {
        final AbstractInstance instance = getInstance(monitorLog);

        String subject = String.format(CHANGE_NORMAL_EMAIL_SUBJECT, instance.instanceName());
        final String[] to = instance.emailAsArray();

        Map<String, Object> model = contentModel(monitorLog);
        STRender stRender = new STRender(CHANGE_NORMAL_EMAIL_CONTENT_TEMPLATE, model);
        final String content = stRender.render();

        sendEmail(to, subject, content);
        LOGGER.debug("Sent Change-Normal Email of Instance[guid={},name={}], FrequencyMonitorLog is [{}]", instance.guid(), instance.instanceName(), monitorLog);

        return content;
    }

    /**
     * @since 3.0.0
     */
    private AbstractInstance getInstance(FrequencyMonitorLog monitorLog) {
        ApplicationInstance instance = monitorLog.instance();
        return instance != null ? instance : monitorLog.networkInstance();
    }


    private void sendEmail(String to[], String subject, String content) {
        MailTransmitter transmitter = new MailTransmitter(subject, content, to);
        transmitter.transmit();
        LOGGER.debug("Send MonitoringReminder-Email to {}", Arrays.toString(to));
    }

}