package com.andaily.hb.domain.log.reminder;

import com.andaily.hb.domain.application.ApplicationInstance;
import com.andaily.hb.domain.log.FrequencyMonitorLog;
import com.andaily.hb.domain.log.MonitoringReminderLog;
import com.andaily.hb.domain.network.NetworkInstance;
import com.andaily.hb.domain.notify.DingtalkRobotConfig;
import com.andaily.hb.domain.shared.Application;
import com.andaily.hb.domain.shared.BeanProvider;
import com.andaily.hb.infrastructure.DateUtils;
import com.andaily.hb.infrastructure.HttpClientPostHandler;
import com.andaily.hb.infrastructure.jpa.DingtalkRobotConfigRepository;
import com.google.common.collect.ImmutableMap;
import net.sf.json.JSONObject;
import org.apache.commons.lang3.StringUtils;
import org.apache.http.entity.StringEntity;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;

import static com.andaily.hb.domain.shared.Application.ENCODING;

/**
 * 2024/1/9 11:26
 * <p>
 * 钉钉机器人提醒
 *
 * @author Shengzhao Li
 * @since 3.0.0
 */
public class DingtalkRobotPerMonitoringReminderSender extends AbstractPerMonitoringReminderSender {

    private static final Logger LOG = LoggerFactory.getLogger(DingtalkRobotPerMonitoringReminderSender.class);


    private final transient DingtalkRobotConfigRepository configRepository = BeanProvider.getBean(DingtalkRobotConfigRepository.class);


    private DingtalkRobotConfig robotConfig;


    @Override
    public boolean support(FrequencyMonitorLog monitorLog) {
        robotConfig = configRepository.findDingtalkRobotConfig();
        return robotConfig != null && robotConfig.enabled();
    }

    @Override
    public void send(MonitoringReminderLog reminderLog, FrequencyMonitorLog monitorLog) {

        if (robotConfig == null) {
            robotConfig = configRepository.findDingtalkRobotConfig();
        }

        String fullContent = getContent(reminderLog, monitorLog);

        HttpClientPostHandler postHandler = new HttpClientPostHandler(robotConfig.robotWebhook());
        postHandler.maxConnectionSeconds(10);
        postHandler.contentType("application/json");
        // see https://developers.dingtalk.com/document/app/custom-robot-access/title-zob-eyu-qse
        JSONObject json = new JSONObject()
                .accumulate("msgtype", "text")
                .accumulate("text", ImmutableMap.of("content", fullContent));
        if (robotConfig.atAll()) {
            json.accumulate("at", ImmutableMap.of("isAtAll", true));
        } else {
            String phones = robotConfig.phones();
            if (StringUtils.isNotBlank(phones)) {
                json.accumulate("at", ImmutableMap.of("atMobiles", Arrays.asList(phones.split(","))));
            }
        }

        postHandler.httpEntity(new StringEntity(json.toString(), ENCODING));
        String response = postHandler.handleAsString();
        if (LOG.isDebugEnabled()) {
            LOG.debug("Dingtalk Robot Server response: {}", response);
        }
    }

    private String getContent(MonitoringReminderLog reminderLog, FrequencyMonitorLog monitorLog) {
        Map<String, String> model = new HashMap<>();
        ApplicationInstance instance = monitorLog.instance();

        if (instance != null) {
            model.put("$instanceName$", instance.instanceName());
            model.put("$monitorUri$", instance.monitorUrl());
            model.put("$link$", instanceUrl(instance.guid()));
        } else {
            NetworkInstance networkInstance = monitorLog.networkInstance();
            model.put("$instanceName$", networkInstance.instanceName());
            model.put("$monitorUri$", networkInstance.serverAddress() + ":" + networkInstance.serverPort());
            model.put("$link$", instanceUrl(networkInstance.guid()));
        }
        model.put("$time$", DateUtils.toDateText(monitorLog.createTime(), DateUtils.DEFAULT_DATE_TIME_FORMAT));
        model.put("$costTime$", String.valueOf(monitorLog.costTime()));
        model.put("$host$", Application.host());

        final boolean normal = monitorLog.normal();
        if (normal) {
            model.put("$status$", "恢复正常");
        } else {
            model.put("$status$", "连接异常");
        }

        String notifyContent = robotConfig.notifyContent();
        return StringUtils.replaceEach(notifyContent, model.keySet().toArray(new String[0]), model.values().toArray(new String[0]));
    }
}
