package com.andaily.hb.domain.application;

import com.andaily.hb.domain.AbstractInstance;
import com.andaily.hb.domain.shared.BeanProvider;
import com.andaily.hb.domain.user.User;
import com.andaily.hb.domain.user.WeixinUser;
import com.andaily.hb.infrastructure.jpa.UserRepositoryJpa;
import jakarta.persistence.*;


import java.io.Serial;
import java.util.List;

/**
 * A server instance that is need monitoring
 * 定义一个需要心跳监测的应用实例
 *
 * @author Shengzhao Li
 */
@Entity
@Table(name = "application_instance")
public class ApplicationInstance extends AbstractInstance {

    @Serial
    private static final long serialVersionUID = 1826152029135090793L;


    private transient UserRepositoryJpa userRepository = BeanProvider.getBean(UserRepositoryJpa.class);


    @Embedded
    private ApplicationInstanceURL instanceURL = new ApplicationInstanceURL();


    /**
     * 若出现测试正常或不正常时提醒的邮件地址
     * 若有多个请用英文分号(;)分隔
     */
    @Column(name = "email")
    private String email;


    /**
     * Schedule中的任务名称,
     * 当启用该监听任务时, 将会有唯一对应的jobName
     */
    @Column(name = "job_name")
    private String jobName;


    /**
     * REST请求时可指定
     *
     * @since 2.0.1
     */
    @Column(name = "request_body")
    private String requestBody;

    /**
     * 实例进行连接的方式，
     * 默认 HTTP
     *
     * @since 2.0.1
     * @deprecated use NetworkInstance replaced from v3.0.0
     */
    @Column(name = "connect_type")
    @Enumerated(value = EnumType.STRING)
    private InstanceConnectType connectType = InstanceConnectType.HTTP;


    /**
     * Default
     */
    public ApplicationInstance() {
        super();
    }


    public ApplicationInstance useProxy(boolean useProxy) {
        this.useProxy = useProxy;
        return this;
    }

    public String requestBody() {
        return requestBody;
    }

    public ApplicationInstance requestBody(String requestBody) {
        this.requestBody = requestBody;
        return this;
    }

    public InstanceConnectType connectType() {
        return connectType;
    }

    public ApplicationInstance connectType(InstanceConnectType connectType) {
        this.connectType = connectType;
        return this;
    }


    public ApplicationInstance continueFailedTimes(int continueFailedTimes) {
        this.continueFailedTimes = continueFailedTimes;
        return this;
    }


    public ApplicationInstance privateInstance(boolean privateInstance) {
        this.privateInstance = privateInstance;
        return this;
    }

    public List<WeixinUser> weixinUsers() {
        return userRepository.findWeixinUsersByInstanceGuid(this.guid);
    }

    public ApplicationInstanceURL instanceURL() {
        return instanceURL;
    }

    public ApplicationInstance addMonitorURLParameter(InstanceMonitorURLParameter urlParameter) {
        instanceURL.urlParameters().add(urlParameter);
        return this;
    }


    public ApplicationInstance creator(User creator) {
        this.creator = creator;
        return this;
    }

    public MonitorUrlRequestMethod requestMethod() {
        return instanceURL.requestMethod();
    }

    public ApplicationInstance requestMethod(MonitorUrlRequestMethod requestMethod) {
        instanceURL.requestMethod(requestMethod);
        return this;
    }


    public ApplicationInstance instanceName(String instanceName) {
        this.instanceName = instanceName;
        return this;
    }

    public String monitorUrl() {
        return instanceURL.monitorUrl();
    }

    public ApplicationInstance monitorUrl(String monitorUrl) {
        instanceURL.monitorUrl(monitorUrl);
        return this;
    }


    public ApplicationInstance maxConnectionSeconds(int maxConnectionSeconds) {
        this.maxConnectionSeconds = maxConnectionSeconds;
        return this;
    }


    public ApplicationInstance enabled(boolean enabled) {
        this.enabled = enabled;
        return this;
    }

    /**
     * @since 3.0.0
     */
    public ApplicationInstance groupId(String groupId) {
        this.groupId = groupId;
        return this;
    }


    public ApplicationInstance frequency(HeartBeatFrequency frequency) {
        this.frequency = frequency;
        return this;
    }

    public String email() {
        return email;
    }

    @Override
    public String[] emailAsArray() {
        return email.contains(";") ? email.split(";") : new String[]{email};
    }

    @Override
    public String monitorUri() {
        return this.monitorUrl();
    }

    public ApplicationInstance email(String email) {
        this.email = email;
        return this;
    }

    public String jobName() {
        return jobName;
    }

    public ApplicationInstance jobName(String jobName) {
        this.jobName = jobName;
        return this;
    }


    public ApplicationInstance remark(String remark) {
        this.remark = remark;
        return this;
    }

    @Override
    public String toString() {
        return "{" +
                "instanceName='" + instanceName + '\'' +
                ", instanceURL=" + instanceURL +
                ", maxConnectionSeconds=" + maxConnectionSeconds +
                ", enabled=" + enabled +
                ", frequency=" + frequency +
                ", email='" + email + '\'' +
                ", connectType='" + connectType + '\'' +
                ", remark='" + remark + '\'' +
                ", privateInstance=" + privateInstance +
                ", continueFailedTimes=" + continueFailedTimes +
                ", id=" + id +
                '}';
    }
}