package com.andaily.hb.domain;

import com.andaily.hb.domain.application.HeartBeatFrequency;
import com.andaily.hb.domain.user.User;
import jakarta.persistence.*;

import java.io.Serial;

/**
 * 2023/12/12 20:30
 * <p>
 * 监控实例的抽象类
 * 公共属性
 *
 * @author Shengzhao Li
 * @since 3.0.0
 */
@MappedSuperclass
public abstract class AbstractInstance extends AbstractDomain {
    @Serial
    private static final long serialVersionUID = -6766567830913404190L;


    /**
     * 实例名称
     */
    @Column(name = "instance_name")
    protected String instanceName;


    /**
     * 连接时超时的时间
     * 0,表示无超时
     */
    @Column(name = "max_connection_seconds")
    protected int maxConnectionSeconds;

    /**
     * enabled or disabled
     * 是否启用
     */
    @Column(name = "enabled")
    protected boolean enabled;

    /**
     * 心跳检测频率, 默认30秒
     */
    @Column(name = "frequency")
    @Enumerated(value = EnumType.STRING)
    protected HeartBeatFrequency frequency = HeartBeatFrequency.THIRTY;


    /**
     * 备注信息
     */
    @Column(name = "remark")
    protected String remark;


    /**
     * The instance creator (owner)
     */
    @ManyToOne
    @JoinColumn(name = "creator_id")
    protected User creator;


    /**
     * 是否为私有应用, 私有应用只有自己登录后才能查看,
     * 别人不能看见, 默认为false,公开
     * <p/>
     * From Version 0.5
     */
    @Column(name = "private_instance")
    protected boolean privateInstance;


    /**
     * 当连续连接失败 指定的次数后才发送提醒.
     * 这用于处理有时连接在检测链接状态的时候，不要发现一次链接故障的时候就马上发邮件通知，
     * 这个过程可能因为网络故障（如常见的：丢包、dns故障）而出现问题
     * <p/>
     * 默认为连续2次
     */
    @Column(name = "continue_failed_times")
    protected int continueFailedTimes = 2;


    /**
     * 如果系统开启了代理proxy,
     * 允许每个实例开启/关闭 是否使用proxy
     *
     * @since 2.0.1
     */
    @Column(name = "use_proxy")
    protected boolean useProxy;


    /**
     * 分组标识
     *
     * @since 3.0.0
     */
    @Column(name = "group_id")
    protected String groupId;


    public AbstractInstance() {
    }


    /**
     * 分组标识
     *
     * @since 3.0.0
     */
    public String groupId() {
        return groupId;
    }

    public String instanceName() {
        return instanceName;
    }


    public int maxConnectionSeconds() {
        return maxConnectionSeconds;
    }


    public boolean enabled() {
        return enabled;
    }


    public HeartBeatFrequency frequency() {
        return frequency;
    }


    public String remark() {
        return remark;
    }


    public User creator() {
        return creator;
    }


    public boolean privateInstance() {
        return privateInstance;
    }


    public int continueFailedTimes() {
        return continueFailedTimes;
    }


    public boolean useProxy() {
        return useProxy;
    }

    /**
     * email array
     *
     * @return array
     * @since 3.0.0
     */
    public abstract String[] emailAsArray();

    /**
     * 各自的监控uri
     *
     * @return uri
     */
    public abstract String monitorUri();
}
