package com.andaily.hb.configuration;

import com.andaily.hb.domain.shared.security.SecurityUtils;
import com.andaily.hb.service.UserService;
import com.andaily.hb.web.context.HBAuthenticationSuccessHandler;
import com.andaily.hb.web.context.SpringSecurityHolder;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.http.HttpMethod;
import org.springframework.security.authentication.AuthenticationProvider;
import org.springframework.security.authentication.dao.DaoAuthenticationProvider;
import org.springframework.security.config.annotation.web.builders.HttpSecurity;
import org.springframework.security.config.annotation.web.configuration.WebSecurityCustomizer;
import org.springframework.security.crypto.bcrypt.BCryptPasswordEncoder;
import org.springframework.security.crypto.password.PasswordEncoder;
import org.springframework.security.web.SecurityFilterChain;

/**
 * 2018/2/1
 * <p>
 * Security
 * <p>
 * replace security.xml
 *
 * @author Shengzhao Li
 */
@Configuration
public class SecurityConfiguration {


    /**
     * 需要调试时 可把此配置参数换为 true
     *
     * @since 3.0.0
     */
    @Value("${hb.spring.web.security.debug:false}")
    private boolean springWebSecurityDebug;

    @Autowired
    private UserService userService;


    @Autowired
    private HBAuthenticationSuccessHandler authenticationSuccessHandler;


    /**
     * 扩展默认的 Web安全配置项
     * <p>
     * defaultSecurityFilterChain
     *
     * @throws Exception e
     */
    @Bean
    public SecurityFilterChain defaultSecurityFilterChain(HttpSecurity http) throws Exception {

        http.csrf(csrfConfigurer -> {
            csrfConfigurer.ignoringRequestMatchers("/public/**");
        });

        http.authorizeHttpRequests(matcherRegistry -> {
            matcherRegistry.requestMatchers(
                            "/favicon.ico*", "/public/**", "/js/**", "/images/**", "/flat_ui/**", "*.html", "*.jpg", "*.png").permitAll()
                    .requestMatchers(HttpMethod.GET, "/login.hb*").anonymous()

                    .requestMatchers("/instance/instance_form.hb*").hasAnyRole("CREATE_EDIT_INSTANCE")
                    .requestMatchers("/instance/delete.hb*").hasAnyRole("DELETE_INSTANCE")
                    .requestMatchers("/instance/enable.hb*").hasAnyRole("START_STOP_INSTANCE")
                    .requestMatchers("/instance/stop.hb*").hasAnyRole("START_STOP_INSTANCE")
                    // v3.0.0 added network instance
                    .requestMatchers("/network_instance/instance_form.hb*").hasAnyRole("CREATE_EDIT_INSTANCE")
                    .requestMatchers("/network_instance/delete.hb*").hasAnyRole("DELETE_INSTANCE")
                    .requestMatchers("/network_instance/enable.hb*").hasAnyRole("START_STOP_INSTANCE")
                    .requestMatchers("/network_instance/stop.hb*").hasAnyRole("START_STOP_INSTANCE")

                    .requestMatchers("/user/**").hasAnyRole("USER_MANAGEMENT")
                    .requestMatchers("/system/setting.hb*").hasAnyRole("SYSTEM_SETTING")
                    .requestMatchers("/system/email/**").hasAnyRole("SYSTEM_SETTING")
                    .requestMatchers("/system/dingtalk_robot/**").hasAnyRole("SYSTEM_SETTING")

                    .requestMatchers("/user_profile.hb*").hasAnyRole("DEFAULT")

                    // anyRequest() 放最后
                    .anyRequest().permitAll();
        });

        http.formLogin(formLoginConfigurer -> {
            formLoginConfigurer.loginPage("/login.hb")
                    .loginProcessingUrl("/login")
                    .failureUrl("/login.hb?failed=true")
                    .successHandler(authenticationSuccessHandler)
//                    .defaultSuccessUrl("/")
                    .usernameParameter("username")
                    .passwordParameter("password");

        });

        http.exceptionHandling(exceptionHandlingConfigurer -> {
            exceptionHandlingConfigurer.accessDeniedPage("/login.hb?access_denied=true");
        });

        http.logout(logoutConfigurer -> {
            logoutConfigurer.logoutUrl("/logout")
                    .deleteCookies("JSESSIONID")
                    .logoutSuccessUrl("/index.hb");
        });

        return http.build();
    }


    /**
     * 安全配置自定义扩展
     *
     * @return WebSecurityCustomizer
     */
    @Bean
    public WebSecurityCustomizer webSecurityCustomizer() {
        return web -> web.debug(this.springWebSecurityDebug);
    }


//    @Override
//    public void configure(WebSecurity web) throws Exception {
//        //Ignore, public
//        web.ignoring().antMatchers("/public/**", "/static/**");
//    }
//
//
//    @Override
//    protected void configure(HttpSecurity http) throws Exception {
////        http.csrf().disable()
//        http.authorizeRequests()
//                .antMatchers("/public/**").permitAll()
//                .antMatchers("/static/**").permitAll()
//                .antMatchers("/login.hb*").permitAll()
//
//                .antMatchers("/instance/instance_form.hb*").hasAnyRole("CREATE_EDIT_INSTANCE")
//                .antMatchers("/instance/delete.hb*").hasAnyRole("DELETE_INSTANCE")
//
//                .antMatchers("/instance/enable.hb*").hasAnyRole("START_STOP_INSTANCE")
//                .antMatchers("/instance/stop.hb*").hasAnyRole("START_STOP_INSTANCE")
//
//                .antMatchers("/user/**").hasAnyRole("USER_MANAGEMENT")
//                .antMatchers("/system/setting.hb*").hasAnyRole("SYSTEM_SETTING")
//
//                .antMatchers("/user_profile.hb*").hasAnyRole("DEFAULT")
//
//                .antMatchers(HttpMethod.GET, "/login.hb*").anonymous()
//                .anyRequest().permitAll()
//                .and()
//                .formLogin()
//                .loginPage("/login.hb")
//                .loginProcessingUrl("/login")
//                .failureUrl("/login.hb?error=1")
//                .successHandler(authenticationSuccessHandler)
//                .usernameParameter("username")
//                .passwordParameter("password")
//                .permitAll()
//                .and()
//                .logout()
//                .logoutUrl("/logout")
//                .deleteCookies("JSESSIONID")
//                .logoutSuccessUrl("/index.hb")
//                .permitAll()
//                .and()
//                .exceptionHandling();
//
//        http.authenticationProvider(authenticationProvider());
//    }


    @Bean
    public AuthenticationProvider authenticationProvider() {
        DaoAuthenticationProvider daoAuthenticationProvider = new DaoAuthenticationProvider();
        daoAuthenticationProvider.setUserDetailsService(userService);
        daoAuthenticationProvider.setPasswordEncoder(passwordEncoder());
        return daoAuthenticationProvider;
    }


//    @Bean
//    public Md5PasswordEncoder passwordEncoder() {
//        return new Md5PasswordEncoder();
//    }


    /**
     * BCrypt  加密
     *
     * @return PasswordEncoder
     * @since 3.0.0
     */
    @Bean
    public PasswordEncoder passwordEncoder() {
        return new BCryptPasswordEncoder(BCryptPasswordEncoder.BCryptVersion.$2A, 11);
    }

    /**
     * SecurityUtils
     */
    @Bean
    public SecurityUtils securityUtils() {
        SecurityUtils securityUtils = new SecurityUtils();
        securityUtils.setSecurityHolder(new SpringSecurityHolder());
        return securityUtils;
    }


}
