## # Template Engine 🎴
## 
## Provides templates render with nimja library
## 
## ## Usage ❔
## 
## .. code-block:: nim
##    templateFolder("/public/templates")
## 
##    proc renderIndex(name: string): string =
##      renderTemplate("index.html")
##    
##    serve "127.0.0.1", 5000:
##      get "/":
##        return renderIndex("Ethosa")
## 

import
  nimja,
  std/macros,
  std/macrocache,
  std/os

export
  nimja


const templatesFolder = CacheTable"HappyXTemplateFolder"


macro templateFolder*(f: string) =
  ## Specifies templates folder
  templatesFolder["f"] = f


macro renderTemplate*(name: static[string]): untyped =
  ## Renders template from file
  let folder = getScriptDir() / $templatesFolder["f"] / name
  newCall("compileTemplateFile", newLit(folder))


proc render*(model: auto, htmlTemplate: static[string]): string =
  ## This proc renders `htmlTemplate` file in `templates` folder.
  ## 
  ## .. Note::
  ##    See also `templateFolder procedure <#templateFolder.m,string>`_.
  ##    This macro setup template folder
  ##
  ## ## Example
  ## 
  ## `main.nim`:
  ## 
  ## .. code-block:: nim
  ##    type
  ##      Cat = object
  ##        name: string
  ##    
  ##    var cat = Cat(name: "Baron")
  ##    cat.render("profile.html")
  ## 
  ## `templates/profile.html`:
  ## 
  ## .. code-block:: html
  ##    <!DOCTYPE html><html>
  ##    <head>
  ##      <title>{{ model.name }}</title>
  ##    </head>
  ##    <body></body>
  ##    </html>
  renderTemplate(htmlTemplate)
