/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.hw.openai.entity.chat;

import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.annotation.JsonProperty;

import jakarta.validation.constraints.NotNull;
import lombok.AllArgsConstructor;
import lombok.Builder;
import lombok.Data;
import lombok.NoArgsConstructor;

import java.io.Serializable;
import java.util.List;

/**
 * Message
 *
 * @author HamaWhite
 */
@Data
@Builder
@NoArgsConstructor
@AllArgsConstructor
@JsonInclude(JsonInclude.Include.NON_NULL)
public class ChatMessage implements Serializable {

    /**
     * The role of the author of this message. One of system, user, or assistant.
     */
    @NotNull
    private ChatMessageRole role;

    /**
     * The contents of the message.
     * content should always exist in the call, even if it is null
     */
    @JsonInclude()
    private String content;

    /**
     * The name of the author of this message. May contain a-z, A-Z, 0-9, and underscores,
     * with a maximum length of 64 characters.
     */
    private String name;

    /**
     * The name and arguments of a function that should be called, as generated by the model.
     */
    @JsonProperty("tool_calls")
    private List<ToolCall> toolCalls;

    public ChatMessage(String content) {
        this.content = content;
    }

    public ChatMessage(ChatMessageRole role, String content) {
        this.role = role;
        this.content = content;
    }

    public ChatMessage(ChatMessageRole role, String content, String name) {
        this.role = role;
        this.content = content;
        this.name = name;
    }

    public ChatMessage(ChatMessage source) {
        this.role = source.role;
        this.content = source.content;
        this.name = source.name;
        this.toolCalls = source.toolCalls;
    }

    public static ChatMessage of(String role, String content) {
        return new ChatMessage(ChatMessageRole.fromValue(role), content);
    }

    public static ChatMessage of(String content) {
        return new ChatMessage(ChatMessageRole.USER, content);
    }

    public static ChatMessage ofSystem(String content) {
        return new ChatMessage(ChatMessageRole.SYSTEM, content);
    }

    public static ChatMessage ofAssistant(String content) {
        return new ChatMessage(ChatMessageRole.ASSISTANT, content);
    }

    public static ChatMessage ofFunction(String content, String name) {
        return new ChatMessage(ChatMessageRole.FUNCTION, content, name);
    }
}
