/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.hw.langchain.chat.models.base;

import com.hw.langchain.llms.base.BaseLLM;
import com.hw.langchain.schema.AsyncLLMResult;
import com.hw.langchain.schema.Generation;
import com.hw.langchain.schema.LLMResult;

import lombok.experimental.SuperBuilder;
import reactor.core.publisher.Flux;

import java.util.List;

/**
 * Base LLM abstract class.
 * The purpose of this class is to expose a simpler interface for working
 * with LLMs, rather than expect the user to implement the full innerGenerate method.
 *
 * @author HamaWhite
 */
@SuperBuilder
public abstract class LLM extends BaseLLM {

    /**
     * Run the LLM on the given prompt and input.
     *
     * @param prompt The prompt to pass into the model.
     * @param stop   list of stop words to use when generating.
     * @return The string generated by the model.
     */
    public abstract String innerCall(String prompt, List<String> stop);

    @Override
    protected LLMResult innerGenerate(List<String> prompts, List<String> stop) {
        List<List<Generation>> generations = prompts.stream().map(prompt -> {
            String text = innerCall(prompt, stop);
            return List.of(new Generation(text));
        }).toList();

        return new LLMResult(generations);
    }

    @Override
    protected Flux<AsyncLLMResult> asyncInnerGenerate(List<String> prompts, List<String> stop) {
        throw new UnsupportedOperationException("not supported yet.");
    }
}
