---
layout: post
title: Рекомендации по формам входа в систему
subhead: Используя кроссплатформенные функции браузеров, создавайте безопасные и простые в применении формы входа в систему с поддержкой специальных возможностей.
authors:
  - samdutton
scheduled: true
date: 2020-06-29
updated: 2021-09-27
description: Используя кроссплатформенные функции браузеров, создавайте безопасные и простые в применении формы входа в систему с поддержкой специальных возможностей.
hero: image/admin/pErOjllBUXhnj68qOhfr.jpg
alt: Человек, держащий телефон.
tags:
  - forms
  - identity
  - layout
  - mobile
  - security
  - ux
codelabs:
  - codelab-sign-in-form-best-practices
---

{% YouTube 'alGcULGtiv8' %}

Если нужно, чтобы пользователи выполняли вход в систему на вашем сайте, очень важно правильно разработать дизайн формы входа. Последнее особенно важно для пользователей, у которая плохая связь, которые пользуются сайтом с мобильных устройств, спешат или находятся в стрессе. Плохо разработанные формы входа в систему увеличивают показатель отказов от использования сайта. Каждый отказ может означать потерянного и недовольного пользователя, а не просто упущенную возможность входа в систему.

{% Aside 'codelab' %} Если вы хотите изучить эти рекомендации с помощью практического руководства, ознакомьтесь с codelab «[Рекомендации по формам входа в систему](/codelab-sign-in-form-best-practices/)». {% endAside %}

Вот пример простой формы входа в систему, в дизайне которой реализованы все рекомендации:

{% Glitch { id: 'sign-in-form', path: 'index.html', height: 480 } %}

## Контрольный список

- [Используйте понятные элементы HTML](#meaningful-html): `<form>`, `<input>`, `<label>` и `<button>`.
- [Для каждого поля ввода данных создайте соответствующую подпись `<label>`](#label).
- Для [доступа к встроенным функциям браузера](#element-attributes) используйте атрибуты элементов: `type`, `name`, `autocomplete` и `required`.
- Присвойте атрибутам `name` и `id` стабильные значения, которые не будут изменяться между загрузками страницы или развертываниями веб-сайта.
- Поместите форму входа в систему в [отдельный элемент <form>](#form).
- [Примите все меры для успешной отправки формы](#submission).
- Для ввода пароля в форме регистрации и для ввода нового пароля в форме сброса пароля используйте параметры [`autocomplete="new-password"`](#new-password) и [`id="new-password"`](#new-password).
- Для ввода пароля при входе в систему используйте параметры [`autocomplete="current-password"`](#current-password) и [`id="current-password"`](#current-password).
- Реализуйте функцию «[Показать пароль](#show-password)».
- Для ввода пароля [используйте элементы `aria-label` и `aria-describedby`](#accessible-password-inputs).
- [Не дублируйте вводимые данные](#no-double-inputs).
- Разрабатывайте формы таким образом, чтобы [мобильная клавиатура не закрывала собой поля ввода данных и кнопки](#keyboard-obstruction).
- Проверьте, удобно ли работать с формами на мобильных устройствах: используйте [хорошо читаемый текст](#size-text-correctly) и убедитесь, что поля ввода данных и кнопки [достаточно велики, чтобы использовать их в качестве сенсорных элементов управления](#tap-targets).
- [Используйте фирменные оформление и стиль](#general-guidelines) на страницах регистрации и входа в систему.
- [Выполните тестирование как в «полевых», так и в лабораторных условиях](#analytics): добавьте в потоки регистрации и входа в систему средства аналитики страниц, аналитики взаимодействия и ориентированного на пользователей измерения производительности.
- [Выполните тестирование в разных браузерах и на разных устройствах](#devices): поведение формы может значительно отличаться на разных платформах.

{% Aside %} В этой статье изложены рекомендации по работе с интерфейсной частью. В ней не рассказывается, как создавать серверные службы для проверки подлинности пользователей, хранения их учетных данных или управления учетными записями. В статье [12 рекомендаций по управлению учетными записями пользователей, авторизацией и паролями](https://cloud.google.com/blog/products/gcp/12-best-practices-for-user-account) изложены основные принципы организации серверной части. Если ваши пользователи находятся в разных частях мира, выполните локализацию содержимого сайта, а также сторонних служб по работе с удостоверениями.

Кроме того, существует два указанных ниже относительно новых API, которые не рассмотрены в этой статье и с помощью которых можно улучшить процесс входа в систему.

- [**WebOTP**](/web-otp/): используется для доставки одноразовых секретных кодов или PIN-кодов с помощью SMS на мобильные телефоны. Благодаря этому пользователи могут выбрать номер телефона в качестве идентификатора (не нужно будет вводить адрес электронной почты). Кроме того, можно будет использовать двухэтапную проверку для входа в систему и одноразовые коды для подтверждения платежей.
- [**Управление учетными данными**](https://developer.chrome.com/blog/credential-management-api/): позволяет разработчикам программным способом хранить и получать учетные данные с паролями и федеративные учетные данные. {% endAside %}

## Используйте понятный HTML {: #meaningful-html }

Используйте элементы, созданные для форм: `<form>`, `<label>` и `<button>`. Они позволяют использовать встроенные функции браузера, лучше поддерживать специальные возможности и делают разметку понятной.

### Используйте элемент `<form>` {: #form }

У вас может возникнуть соблазн вложить поля ввода данных в элементы `<div>` и отправлять входные данные исключительно с помощью JavaScript. В общем случае лучше использовать старый простой элемент [`<form>`](https://developer.mozilla.org/docs/Web/HTML/Element/form). Благодаря этому ваш сайт будет доступен для программ чтения с экрана и других устройств, обеспечивающих специальные возможности, вы сможете использовать ряд встроенных функций браузера, вам будет проще создавать базовые функциональные средства для входа в систему для старых браузеров, и это все будет работать даже в случае сбоя JavaScript.

{% Aside 'gotchas' %} Одна из распространенных ошибок — вложить всю веб-страницу в одну форму; это может привести к проблемам, связанным с диспетчерами паролей и функциями автозаполнения в браузерах. Используйте отдельный элемент <form> для каждого компонента пользовательского интерфейса, для работы которого требуется форма. Например, если на одной странице имеется функция входа в систему и функция поиска, следует использовать два элемента формы. {% endAside %}

### Используйте элемент `<label>` {: #label }

Создать подпись для поля ввода данных можно с помощью элемента [`<label>`](https://developer.mozilla.org/docs/Web/HTML/Element/label).

```html
<label for="email">Адрес электронной почты</label>
<input id="email" …>
```

Вот две причины, по которым нужно делать это.

- При касании подписи или щелчку по ней фокус будет перемещен на поле ввода данных, с которым связана подпись. Свяжите подпись с полем ввода данных, используя атрибут `for` подписи с параметром `name` или `id` поля ввода данных.
- Программы чтения с экрана воспроизводят текст подписи, когда фокус перемещается на подпись или на поле ввода данных, связанное с подписью.

Не используйте заполнители в качестве подписей, связанных с полями ввода данных. Люди склонны забывать, что было указано в поле ввода данных, как только они начинают вводить текст, особенно если они отвлеклись («Что нужно было ввести? Адрес электронной почты, номер телефона или идентификатор учетной записи?»). Есть много других потенциальных проблем с заполнителями: если вы до сих пор не согласны с этим, см. статьи «[Не используйте атрибут заполнителя](https://www.smashingmagazine.com/2018/06/placeholder-attribute/)» и «[Заполнители в полях формы вредны](https://www.nngroup.com/articles/form-design-placeholders/)».

Вероятно, самый лучший вариант — размещать подписи над полями ввода данных. Это позволяет создать единообразный дизайн для мобильных устройств и ПК и (согласно [исследованиям ИИ Google](https://ai.googleblog.com/2014/07/simple-is-better-making-your-web-forms.html)) ускоряет «сканирование» элементов управления пользователями. Подписи и поля ввода данных будут иметь полную ширину, и вам не придется ее корректировать, чтобы подписи и поля ввода данных соответствовали тексту подписей.

<figure>{% Img src="image/admin/k0ioJa9CqnMI8vyAvQPS.png", alt="Снимок экрана, на котором показано положение подписи для поля ввода данных в форме на мобильном устройстве: рядом с полем ввода данных и над полем ввода данных.", width="500", height="253" %} <figcaption>Если подпись и поле ввода данных находятся в одной строке, их ширина ограничена.</figcaption></figure>

Откройте Glitch [label-position](https://label-position.glitch.me) на мобильном устройстве и убедитесь в этом сами.

### Используйте элемент `<button>` {: #button }

Используйте элемент [`<button>`](https://developer.mozilla.org/docs/Web/HTML/Element/button) для кнопок. Элементы кнопок поддерживают специальные возможности и встроенные функции отправки форм. Кроме того, к ним можно легко применять стили. Нет смысла использовать в качестве кнопки элемент `<div>` или какой-нибудь другой элемент.

Убедитесь, что на кнопке отправки написано, что она делает. Примеры: **Создать учетную запись** или **Войти**, но не **Отправить** или **Начать**.

### Что нужно сделать для успешной отправки формы {: #submission }

Сделайте так, чтобы диспетчеры паролей «понимали», что форма отправлена. Для этого можно воспользоваться двумя указанными ниже способами.

- Перейти на другую страницу.
- Имитировать переход с помощью метода `History.pushState()` или `History.replaceState()` и удалить форму ввода пароля.

Выполнив запрос `XMLHttpRequest` или `fetch`, убедитесь, что в ответе сообщается об успешном входе в систему и выполняется обработка входа в систему путем извлечения формы из DOM, а также имеется информация об успешном выполнении для пользователя.

Рассмотрите возможность отключения кнопки **Войти** после того как пользователь нажмет ее. [Многие пользователи нажимают кнопки несколько раз](https://baymard.com/blog/users-double-click-online) даже на быстро реагирующих сайтах. Такие действия замедляют взаимодействие и увеличивают нагрузку на сервер.

И наоборот, не отключайте функцию отправки формы, ожидающую ввод данных пользователем. Например, не отключайте кнопку **Войти**, если пользователь не ввел PIN-код клиента. Пользователь может что-то пропустить в форме, а затем попытаться несколько раз нажать (отключенную) кнопку **Войти** и подумать, что она не работает. Если необходимо отключить функцию отправки формы, объясните пользователю, каких данных не хватает, когда он нажимает отключенную кнопку.

{% Aside 'caution' %} По умолчанию в форме используются кнопки типа `submit`. Если нужно добавить еще одну кнопку в форму (например, для команды **Показать пароль**), добавьте элемент `type="button"`. В противном случае при нажатии кнопки будет отправлена форма.

Когда фокус находится на любом поле формы, нажатие клавиши `ВВОД` приводит к щелчку по первой кнопке `submit` в форме. Если добавить в форму какую-либо кнопку перед кнопкой **Отправить** и не указать ее тип, эта кнопка будет иметь тип, используемый по умолчанию для кнопок в форме (`submit`), и получит событие щелчка перед отправкой формы. В качестве примера см. нашу [демонстрацию](https://enter-button.glitch.me/): заполните форму и нажмите клавишу `ВВОД`. {% endAside %}

### Не дублируйте поля ввода данных {: #no-double-inputs }

На некоторых сайтах пользователям необходимо дважды вводить адрес электронной почты или пароль. Это позволяет уменьшить количество ошибок для небольшого числа пользователей, но при этом *всем* пользователям нужно выполнять дополнительную работу, что [увеличивает показатель отказов](https://uxmovement.com/forms/why-the-confirm-password-field-must-die/). Кроме того, нет смысла просить пользователя дважды вводить данные, когда браузеры автоматически вводят адреса электронной почты или предлагают надежные пароли. Лучше разрешить пользователям подтверждать свой адрес электронной почты (вам все равно нужно это сделать) и при необходимости упростить для них процедуру сброса пароля.

## Максимально полно используйте возможности, предоставляемые атрибутами элементов {: #element-attributes }

Вот где действительно происходит волшебство! В браузерах есть несколько полезных встроенных функций, для которых используются атрибуты полей ввода данных.

## Скрывайте пароли, но позволяйте пользователям просматривать их при необходимости {: #show-password }

Поля ввода паролей должны иметь тип `type="password"`, чтобы можно было скрыть текст пароля и сообщить браузеру, что то или иное поле ввода данных предназначено для паролей. (Обратите внимание, что в браузерах используются [различные методы](#autofill) определения ролей полей ввода данных и принятия решений о том, предлагать или не предлагать пользователям сохранять пароли.)

Добавьте значок или кнопку **Показать пароль**, чтобы пользователь мог проверить введенный текст. Кроме того, обязательно добавьте ссылку **Забыли пароль?**. См. раздел «[Включение функции отображения пароля](#password-display)».

<figure>{% Img src="image/admin/58suVe0HnSLaJvNjKY53.png", alt="Форма входа в систему Google, на которой отображен значок Show password (Показать пароль).", width="300", height="107" %} <figcaption>Поле ввода пароля на форме входа в систему Google со значком <strong>Show password</strong> (Показать пароль) и ссылкой <strong>Forgot password</strong> (Забыли пароль?).</figcaption></figure>

## Предоставьте пользователям мобильных устройств подходящую клавиатуру {: #mobile-keyboards }

Используя элемент `<input type="email">`, можно предоставить пользователям мобильных устройств подходящую клавиатуру и включить базовую встроенную функцию проверки адреса электронной почты в браузере. При этом не потребуется использовать JavaScript.

Если вам нужно использовать номер телефона вместо адреса электронной почты, элемент `<input type="tel">` позволяет активировать клавиатуру для ввода номера телефона на мобильном телефоне. При необходимости также можно использовать атрибут `inputmode`: элемент `inputmode="numeric"` идеально подойдет для ввода числовых PIN-кодов. Подробные сведения см. в статье «[Все, что вы хотели знать о режиме ввода](https://css-tricks.com/everything-you-ever-wanted-to-know-about-inputmode/)».

{% Aside 'caution' %} При использовании атрибута `type="number"` в поле ввода данных отображаются стрелки вверх или вниз для изменения числа, поэтому не применяйте его для чисел, которые не предполагается увеличивать или уменьшать, например идентификаторов и номеров счетов. {% endAside %}

### Клавиатура на экране мобильного устройства не должна закрывать собой кнопку «**Войти**» {: #keyboard-obstruction }

К сожалению, если вы не будете аккуратны, клавиатура на экране мобильного устройства может закрыть собой форму или (что еще хуже) кнопку **Войти**. В этом случае пользователи могут прекратить пытаться войти в систему, прежде чем осознают, что произошло.

<figure>{% Img src="image/admin/rLo5sW9LBpTcJU7KNnb7.png", alt="Два снимка экрана с формой входа в систему на телефоне с ОС Android: на одном их них показано, как клавиатура телефона закрывает собой кнопку Submit (Отправить).", width="400", height="360" %} <figcaption>Кнопка <b>Sign in</b> (Войти) видна на одном экране, но не видна на другом.</figcaption></figure>

По возможности избегайте этого, отображая поля ввода адреса электронной почты, телефона и пароля, а также кнопку **Войти** в верхней части страницы входа в систему. Остальной контент разместите под этими элементами.

<figure>{% Img src="image/admin/0OebKiAP4sTgaXbcbvYx.png", alt="Снимок экрана, на котором показана форма входа в систему на телефоне с ОС Android: кнопка Sign in (Войти) не закрыта клавиатурой телефона.", width="200", height="342" %} <figcaption>Клавиатура не закрывает собой кнопку <b>Sign in</b> (Войти).</figcaption></figure>

#### Выполните тестирование на разных устройствах {: #devices }

Вам потребуется выполнить тестирование на ряде устройств, используемых вашей целевой аудиторией, и внести соответствующие корректировки. С помощью BrowserStack можно [бесплатно тестировать проекты с открытым исходным кодом](https://www.browserstack.com/open-source) на различных реальных устройствах и браузерах.

<figure>{% Img src="image/admin/jToMlWgjS3J2WKmjs1hx.png", alt="Снимки экрана с формой входа в систему на iPhone 7, iPhone 8 и iPhone 11. На iPhone 7 и iPhone 8 кнопка Sign in (Войти) закрыта клавиатурой телефона, а на iPhone 11 — нет", width="800", height="522" %} <figcaption>Кнопка <b>Sign in</b> (Войти) закрыта на iPhone 7 и iPhone 8, но не на iPhone 11.</figcaption></figure>

#### Попробуйте использовать две страницы {: #two-pages }

Некоторые сайты (в том числе Amazon и eBay) избегают этой проблемы, запрашивая адрес электронной почты или номер телефона и пароль на двух страницах. Кроме того, этот подход упрощает взаимодействие с пользователем: пользователю требуется выполнять только по одной задаче за раз.

<figure>{% Img src="image/admin/CxpObjYZMs0MMFo66f4P.png", alt="Снимок экрана с формой входа в систему на веб-сайте Amazon: поля ввода адреса электронной почты или номера телефона и пароля расположены на двух отдельных страницах.", width="400", height="385" %} <figcaption>Двухэтапный вход в систему: ввод адреса электронной почты или номера телефона, а затем пароля.</figcaption></figure>

В идеальном случае следует реализовать этот подход с помощью одного элемента <form>. С помощью JavaScript можно сначала отобразить только поле ввода адреса электронной почты, затем скрыть его и отобразить поле ввода пароля. Если необходимо заставить пользователя перейти на новую страницу между вводом адреса электронной почты и вводом пароля, в форме на второй странице должен быть скрытый элемент поля ввода данных со значением адреса электронной почты, чтобы диспетчеры паролей могли сохранять правильные значения. Пример кода см. в статье «[Стили форм ввода паролей, которые понимает Chromium](https://www.chromium.org/developers/design-documents/form-styles-that-chromium-understands)».

### Избавьте пользователей от необходимости повторно вводить данные {: #autofill }

Вы можете помочь браузерам правильно хранить данные и автоматически заполнять поля ввода данных, чтобы пользователям не приходилось вводить адреса электронной почты и пароли. Это особенно важно для мобильных устройств и критически важно для полей ввода адресов электронной почты, для которых наблюдается [большое количество отказов](https://www.formisimo.com/blog/conversion-rate-increases-57-with-form-analytics-case-study/).

Этот процесс включает две указанные ниже части.

1. Благодаря атрибутам `autocomplete`, `name`, `id` и `type` браузер «понимает» роль полей ввода данных. Это позволяет сохранять данные и впоследствии использовать их для автозаполнения полей. Чтобы можно было хранить данные для автозаполнения, в современных браузерах также требуется, чтобы поля ввода данных имели стабильные значения атрибутов `name` или `id` (а не создаваемые случайным образом при каждой загрузке страницы или развертывании сайта) и присутствовали в форме <form> наряду с кнопкой `submit`.

2. Атрибут `autocomplete` помогает браузерам правильно автоматически заполнять поля ввода данных ранее сохраненными данными.

Для ввода адресов электронной почты используйте параметр `autocomplete="username"`, так как в современных браузерах диспетчеры паролей распознают параметр `username` даже в тех случаях, когда следует использовать параметр `type="email"`. Кроме того, вам, возможно, потребуется использовать параметры `id="email"` и `name="email"`.

Для полей ввода паролей используйте соответствующие значения атрибутов `autocomplete` и `id`, чтобы браузеры могли различать новые и текущие пароли.

### Используйте атрибуты `autocomplete="new-password"` и `id="new-password"` для нового пароля {: #new-password }

- Используйте атрибуты `autocomplete="new-password"` и `id="new-password"` для полей ввода паролей в форме регистрации или новых паролей в форме изменения пароля.

### Используйте атрибуты `autocomplete="current-password"` и `id="current-password"` для текущего пароля {: #current-password }

- Используйте атрибуты `autocomplete="current-password"` и `id="current-password"` для поля ввода пароля в форме входа в систему или поля ввода старого пароля пользователя в форме изменения пароля. Таким образом вы сообщите браузеру, что нужно использовать текущий пароль, который он сохранил для данного сайта.

Для формы регистрации:

```html
<input type="password" autocomplete="new-password" id="new-password" …>
```

Для формы входа в систему:

```html
<input type="password" autocomplete="current-password" id="current-password" …>
```

{% Aside %} В некоторых браузерах, например в Chrome, может использоваться собственный диспетчер паролей для автоматического заполнения полей в процессе входа в систему для пользователей, уже посещавших тот или иной сайт. Чтобы эти функции работали, у браузера должна быть возможность определять случаи, когда пользователь меняет свой пароль.

В частности, после настройки нового пароля необходимо очистить форму для изменения пароля пользователя или скрыть ее со страницы. Если форма для изменения пароля пользователя остается заполненной на странице после изменения пароля, браузер не сможет записать обновленные сведения. {% endAside %}

### Поддержка диспетчеров паролей {: #password-managers }

В разных браузерах функции автозаполнения адреса электронной почты и предложения пароля работают по-разному, но результаты во многом одинаковы. Например, в Safari 11 и более поздних версий на настольных компьютерах отображается диспетчер паролей, а затем используется биометрическая проверка подлинности (проверка отпечатка пальца или распознавание лица), если она доступна.

<figure>{% Img src="image/admin/UjBRRYaLbX9bh3LDFcAM.png", alt="Снимки экрана трех этапов процесса входа в систему в браузере Safari на настольном компьютере: диспетчер паролей, биометрическая проверка подлинности, автозаполнение.", width="800", height="234" %} <figcaption>Вход в систему с помощью функции автозаполнения — не нужно вводить текст.</figcaption></figure>

Chrome для настольных компьютеров предлагает варианты адресов электронной почты, отображает диспетчер паролей и автоматически вводит пароль.

<figure>{% Img src="image/admin/mDm1cstWZB9jJDzMmzgE.png", alt="Снимки экрана четырех этапов процесса входа в систему в Chrome на настольном компьютере: ввод адреса электронной почты, предложение адреса электронной почты, диспетчер паролей, автозаполнение при выборе.", width="800", height="232" %} <figcaption>Автозаполнение в процессе входа в систему в Chrome 84.</figcaption></figure>

Системы работы с паролями и автозаполнения в браузерах не просты. Алгоритмы прогнозирования, хранения и отображения значений не стандартизированы и зависят от конкретной платформы. Например, [Хидде де Фрис](https://hiddedevries.nl/en/blog/2018-01-13-making-password-managers-play-ball-with-your-login-form) (Hidde de Vries) говорит следующее: «В Firefox эвристические функции диспетчера паролей дополнены [системой рецептов](https://bugzilla.mozilla.org/show_bug.cgi?id=1119454)».

В статье «[Автозаполнение: что должны знать, но не знают веб-разработчики](https://cloudfour.com/thinks/autofill-what-web-devs-should-know-but-dont)» содержится намного больше информации об использовании атрибутов `name` и `autocomplete`. В [спецификации HTML](https://html.spec.whatwg.org/multipage/form-control-infrastructure.html#inappropriate-for-the-control) перечислены все 59 возможных значений.

{% Aside %} Вы можете помочь диспетчерам паролей, используя разные значения атрибутов `name` и `id` в формах регистрации и формах входа в систему для самого элемента `form`, а также для любых элементов `input`, `select` и `textarea` {% endAside %}

### Предоставьте браузеру возможность предлагать надежные пароли {: #password-suggestions }

В современных браузерах для принятия решений о том, когда отображать пользовательский интерфейс диспетчера паролей и предлагать надежный пароль, используются эвристические функции.

Вот как это работает в Safari для настольных компьютеров.

<figure>{% Img src="image/admin/B1DlZK0CllVjrOUbb5xB.png", alt="Снимок экрана с диспетчером паролей Firefox на настольном компьютере.", width="800", height="229" %} <figcaption>Процесс предложения пароля в Safari.</figcaption></figure>

(Функция предложения надежных уникальных паролей доступна в Safari, начиная с версии 12.0.)

Благодаря встроенным генераторам паролей в браузерах пользователям и разработчикам не нужно продумывать, что такое «надежный пароль». Так как браузеры могут безопасно хранить пароли и при необходимости автоматически вводить их, пользователям не нужно запоминать или вводить пароли. Если поощрять пользователей применять встроенные генераторы паролей в браузерах, они с большей вероятностью будут использовать уникальные надежные пароли на вашем сайте и с меньшей вероятностью будут повторно использовать пароли, которые, возможно, скомпрометированы на других сайтах.

{% Aside %} Недостаток такого подхода заключается в том, что нет возможности обмениваться паролями между платформами. Например, пользователь может использовать Safari на iPhone и Chrome на ноутбуке с ОС Windows. {% endAside %}

### Сделайте так, чтобы пользователи не пропускали поля ввода данных {: #required-fields }

Добавляйте атрибут `required` в поля ввода адреса электронной почты и пароля. Современные браузеры автоматически запрашивают недостающие данные и перемещают фокус на них. В этом случае не потребуется использовать JavaScript.

<figure>{% Img src="image/admin/n5Nr290upVmQGvlc263U.png", alt="Снимок экрана с Firefox для настольного компьютера и Chrome для ОС Android, на котором показана подсказка Please fill out this field (Заполните это поле) для незаполненного поля.", width="600", height="392" %} <figcaption> Подсказка и фокус на незаполненном поле в Firefox (версии 76) для настольного компьютера и Chrome (версии 83) для ОС Android. </figcaption></figure>

## Дизайн для пальцев и больших пальцев {: #mobile-design }

По умолчанию размеры всех элементов ввода данных и кнопок в браузерах слишком малы (особенно на мобильных устройствах). Это может показаться очевидным, но такая проблема широко распространена в формах входа в систему на многих сайтах.

### Убедитесь, что поля ввода данных и кнопки имеют достаточно большой размер {: #tap-targets }

Размеры и отступы, используемые по умолчанию для полей ввода данных и кнопок, слишком малы на настольных компьютерах и еще меньше на мобильных устройствах.

<figure>{% Img src="image/tcFciHGuF3MxnTr1y5ue01OGLBn2/lJNO6w2dOyp4cYKl5b3y.png", alt="Снимок экрана с формой без стиля в Chrome для настольных компьютеров и Chrome для ОС Android.", width="800", height="434" %}</figure>

Согласно [руководству по специальным возможностям в ОС Android](https://support.google.com/accessibility/android/answer/7101858?hl=en-GB) рекомендуемый целевой размер объектов на сенсорных экранах равен 7–10 мм. В рекомендациях компании Apple по интерфейсу рекомендуется размер 48 x 48 пикселей, а организация W3C рекомендует использовать элементы с размером [не менее 44 x 44 пикселя CSS](https://www.w3.org/WAI/WCAG21/Understanding/target-size.html). Исходя из этого, добавьте отступ размером приблизительно 15 пикселей (как минимум) для элементов ввода данных и кнопок для мобильных устройств и приблизительно 10 пикселей для настольных компьютеров. Попробуйте использовать такие элементы управления на реальном мобильном устройстве и с помощью реальных пальцев или большого пальца. Вам должно быть удобно нажимать каждое поле ввода данных и каждую кнопку.

С помощью аудита Lighthouse [Tap targets are not sized appropriately](https://developer.chrome.com/docs/lighthouse/seo/http-status-code/) можно автоматизировать процесс обнаружения слишком маленьких элементов ввода данных.

#### Дизайн для больших пальцев {: #design-for-thumbs }

Если выполнить поиск по ключевым словам [touch target](https://www.google.com/search?q=touch+target) (цель касания), отобразится множество изображений указательных пальцев. Тем не менее в реальной жизни для взаимодействия с телефоном многие люди используют большие пальцы. Большие пальцы имеют больший размер, чем указательные, и управлять с их помощью устройствами менее удобно. Это еще одна причина создавать элементы управления правильного размера для сенсорных экранов.

### Сделайте шрифт текста достаточно большим {: #size-text-correctly }

Как в случае с размерами и отступами, в браузерах для элементов ввода данных и кнопок по умолчанию используется слишком малый шрифт (особенно на мобильных устройствах).

<figure>{% Img src="image/admin/EeIsqWhLbot15p4SYpo2.png", alt="Снимок экрана с формой без стиля в Chrome на настольном компьютере и в ОС Android.", width="800", height="494" %} <figcaption>Стиль, используемый по умолчанию, на настольном компьютере и на мобильном устройстве: шрифт вводимого текста слишком мал, и для многих пользователей он неразборчив.</figcaption></figure>

В браузерах на разных платформах используются разные размеры шрифтов, поэтому сложно указать конкретный размер шрифта, который будет хорошо работать везде. Небольшой обзор популярных веб-сайтов показывает, что размер используемых на них шрифтов на экранах настольных компьютеров составляет 13–16 пикселей. Этот физический размер — хороший минимум для отображения текста на мобильных устройствах.

Это означает, что для мобильных устройств следует использовать больший размер шрифта в пикселях: шрифт размером `16 пикселей` в Chrome для настольных компьютеров вполне разборчив, но даже при хорошем зрении трудно читать шрифт размером `16 пикселей` в Chrome для ОС Android. Можно настроить разные размеры шрифтов в пикселях для разных размеров областей просмотра с помощью [медиа-запросов](https://developers.google.com/web/fundamentals/design-and-ux/responsive#apply_media_queries_based_on_viewport_size). Шрифт размером `20 пикселей` практически идеально подходит для мобильных устройств, но вам следует проверить его на друзьях или коллегах с плохим зрением.

С помощью аудита Lighthouse [Document doesn't use legible font sizes](https://developer.chrome.com/docs/lighthouse/seo/font-size/) можно автоматизировать процесс обнаружения слишком мелкого текста.

### Оставьте достаточно места между полями ввода данных {: #size-margins-correctly }

Оставьте достаточно места вокруг элементов управления, чтобы полями ввода данных было удобно пользоваться на сенсорных экранах. Другими словами, размер свободного места должен быть примерно равен ширине пальца.

### Убедитесь, что поля ввода данных хорошо видны {: #visible-inputs }

Стиль границ полей ввода данных, используемый по умолчанию, делает их трудноразличимыми. Они почти невидимы на некоторых платформах, например в Chrome для ОС Android.

Помимо отступов, добавьте границу: на белом фоне хорошим стандартным правилом считается использовать цвет `#ccc` или более темный.

<figure>{% Img src="image/admin/OgDJ5V2N7imHXSBkN4pr.png", alt="Снимок экрана с формой, к которой применен стиль, в Chrome для ОС Android.", width="250", height="525" %} <figcaption>Разборчивый текст, видимые границы поля ввода данных, правильные отступы и поля.</figcaption></figure>

### Предупреждайте пользователей о вводе недопустимых значений с помощью встроенных функций браузера {: #built-in-validation }

В браузерах имеются встроенные функции для базовой проверки содержимого полей ввода данных с атрибутом `type` в формах. Браузеры предупреждают пользователя, когда он пытается отправить форму с недопустимым значением, и перемещают фокус на поле ввода данных, в котором возникла проблема.

<figure>{% Img src="image/admin/Phf9m5J66lIX9x5brzOL.png", alt="Снимок экрана с формой входа в систему в Chrome на настольном компьютере, на котором отображается подсказка браузера, а фокус находится на недопустимом значении адреса электронной почты.", width="300", height="290" %} <figcaption>Базовая встроенная функция проверки в браузере.</figcaption></figure>

С помощью селектора CSS `:invalid` можно выделять недопустимые данные. Используйте конструкцию `:not(:placeholder-shown)`, чтобы не допустить выбор полей ввода данных без содержимого.

```css
input[type=email]:not(:placeholder-shown):invalid {
  color: red;
  outline-color: red;
}
```

Попробуйте использовать разные способы выделения полей ввода данных с недопустимыми значениями.

## При необходимости используйте JavaScript {: #javascript }

### Включение и выключение отображения пароля {: #password-display }

Вам следует добавить значок или кнопку **Показать пароль**, чтобы пользователь мог проверить вводимый текст. Если у пользователя нет возможности просмотреть введенный текст, это [снижает удобство использования](https://www.nngroup.com/articles/stop-password-masking/) сайта. В настоящее время нет встроенного способа сделать это, хотя [есть планы по его реализации](https://twitter.com/sw12/status/1251191795377156099). Вместо этого вам потребуется использовать JavaScript.

<figure><img src="./show-password-google.png" width="350" alt="Форма входа в систему на сайте Google, на которой отображается значок Show password (Показать пароль)."> <figcaption>Форма входа в систему на сайте Google: со значком <strong>Show password</strong> (Показать пароль) и ссылкой <strong>Forgot password</strong> (Забыли пароль?).</figcaption></figure>

В приведенном ниже коде для добавления функции **Показать пароль** используется текстовая кнопка.

HTML:

```html/2
<section>
  <label for="password">Пароль</label>
  <button id="toggle-password" type="button" aria-label="Показать пароль в виде простого текста. Предупреждение: в этом случае ваш пароль будет отображен на экране.">Показать пароль</button>
  <input id="password" name="password" type="password" autocomplete="current-password" required>
</section>
```

С помощью этого кода CSS можно сделать так, чтобы кнопка выглядела как простой текст:

```css
button#toggle-password {
  background: none;
  border: none;
  cursor: pointer;
  /* Медиа-запрос не показан здесь. */
  font-size: var(--mobile-font-size);
  font-weight: 300;
  padding: 0;
  /* Отображение в верхней правой части контейнера */
  position: absolute;
  top: 0;
  right: 0;
}
```

И JavaScript для отображения пароля:

```js
const passwordInput = document.getElementById('password');
const togglePasswordButton = document.getElementById('toggle-password');

togglePasswordButton.addEventListener('click', togglePassword);

function togglePassword() {
  if (passwordInput.type === 'password') {
    passwordInput.type = 'text';
    togglePasswordButton.textContent = 'Скрыть пароль';
    togglePasswordButton.setAttribute('aria-label',
      'Скрытие пароля.');
  } else {
    passwordInput.type = 'password';
    togglePasswordButton.textContent = 'Показать пароль';
    togglePasswordButton.setAttribute('aria-label',
      'Показать пароль в виде простого текста. ' +
      'Предупреждение: если нажать эту кнопку, ваш пароль будет отображен на экране.');
  }
}
```

Вот конечный результат:

<figure>{% Img src="image/admin/x4NP9JMf1KI8PapQ9JFh.png", alt="Снимки экрана с формой входа в систему с текстовой кнопкой Show password (Показать пароль) в Safari на компьютере Mac и на iPhone 7.", width="800", height="468" %} <figcaption>Форма входа в систему с текстовой кнопкой <strong>Show password</strong> (Показать пароль) в Safari на компьютере Mac и на iPhone 7.</figcaption></figure>

### Включите поддержку специальных возможностей для полей ввода данных {: #accessible-password-inputs }

С помощью атрибута `aria-describedby` можно указать правила работы с паролями, присвоив ему значение идентификатора элемента, в котором описаны ограничения. Программы чтения с экрана произносят текст подписи, тип поля ввода данных (пароль), а затем описание.

```html
<input type="password" aria-describedby="password-constraints" …>
<div id="password-constraints">Не менее восьми знаков, включая буквы, цифры и специальные символы.</div>
```

Когда вы добавляете функцию **Показать пароль**, не забудьте включить в код атрибут `aria-label`, чтобы предупредить пользователя о том, что пароль будет отображен. В противном случае пользователь может случайно раскрыть свой пароль посторонним лицам.

```html/1-2
<button id="toggle-password"
        aria-label="Показать пароль в виде простого текста.
                    Предупреждение: в результате этого действия ваш пароль будет отображен на экране.">
  Показать пароль
</button>
```

Посмотреть обе функции ARIA в действии можно в следующем Glitch:

{% Glitch { id: 'sign-in-form', path: 'index.html', height: 480 } %}

В статье «[Создание форм с поддержкой специальных возможностей](https://webaim.org/techniques/forms/)» есть дополнительные советы по использованию специальных возможностей в формах.

### Выполняйте проверку в режиме реального времени и перед отправкой {: #validation }

Для элементов и атрибутов форм HTML имеются встроенные функции базовой проверки. Тем не менее следует выполнять более надежную проверку с использованием JavaScript, когда пользователи вводят данные и когда они пытаются отправить форму.

{% Aside 'warning' %} Благодаря проверке на стороне клиента пользователи будут вводить правильные данные, а вы предотвратите ненужную нагрузку на сервер. Тем не менее следует всегда проверять и отбраковывать данные на сервере. {% endAside %}

В [действии 5](https://glitch.com/edit/#!/sign-in-form-codelab-5) codelab, посвященного форме входа в систему, используется [Constraint Validation API](https://html.spec.whatwg.org/multipage/forms.html#constraints) ([широко поддерживаемый](https://caniuse.com/#feat=constraint-validation)), который позволяет добавлять пользовательскую проверку с помощью встроенного пользовательского интерфейса браузера для перемещения фокуса на нужный элемент и отображения подсказок.

См. дополнительные сведения в статье «[Использование JavaScript для сложной проверки в режиме реального времени](https://developers.google.com/web/fundamentals/design-and-ux/input/forms#use_javascript_for_more_complex_real-time_validation)».

### Аналитика и RUM {: #analytics }

Выражение «невозможно улучшить то, что не удается измерить» особенно справедливо для форм регистрации и входа в систему. Вам нужно поставить цели, измерить успех, улучшить сайт, а затем повторить этот цикл еще раз.

[Дисконтное тестирование удобства использования](https://www.nngroup.com/articles/discount-usability-20-years/) может быть полезно для проверки изменений, но вам потребуются реальные данные, чтобы понимать, насколько пользователям удобно использовать ваши формы регистрации и входа в систему.

- **Аналитика страницы**: количество просмотров страниц регистрации и входа в систему, показатели отказов и количество выходов.
- **Аналитика взаимодействия**: [воронки целей](https://support.google.com/analytics/answer/6180923?hl=en) (в каких местах пользователи прерывают процесс регистрации или входа в систему?) и [события](https://developers.google.com/analytics/devguides/collection/gtagjs/events) (какие действия выполняют пользователи при взаимодействии с формами?).
- **Производительность веб-сайта**: [метрики, ориентированные на пользователей](/user-centric-performance-metrics) (медленно ли работают ваши формы регистрации и входа в систему? Если да, то в чем причина?).

Кроме того, прежде чем выпускать изменения для всех пользователей, вам, возможно, потребуется реализовать тестирование A/B, чтобы опробовать различные подходы к процессам регистрации и входа в систему, а также поэтапное развертывание для проверки изменений на подмножестве пользователей.

## Общие рекомендации {: #general-guidelines }

Хорошо продуманный пользовательский интерфейс может уменьшить количество отказов пользователей от использования формы входа в систему.

- Не заставляйте пользователей искать функцию входа в систему. Разместите ссылку на форму входа в систему в верхней части страницы, используя понятные формулировки, например **Войти**, **Создать учетную запись** или **Зарегистрироваться**.
- Разместите фокус на нужных элементах. В формах регистрации не следует отвлекать пользователей предложениями и другими возможностями сайта.
- Сделайте процесс регистрации как можно проще. Собирайте другие данные пользователей (например, адреса или данные кредитных карт) только тогда, когда пользователи понимают явные преимущества от предоставления этих данных.
- Прежде чем пользователи начнут заполнять форму регистрации, объясните, в чем состоит ценностное предложение. Какие преимущества они получат, выполнив вход в систему? Предоставьте пользователям конкретные стимулы для завершения регистрации.
- Если возможно, разрешите пользователям идентифицировать себя по номеру мобильного телефона, а не по адресу электронной почты, так как некоторые пользователи могут не использовать электронную почту.
- Упростите процедуру сброса пароля для пользователей и сделайте так, чтобы ссылка **Забыли пароль?** была очевидна.
- Ссылка на документы с условиями обслуживания и политикой конфиденциальности: с самого начала объясните пользователям, как вы защищаете их данные.
- Добавьте логотип и название вашей компании или организации на страницы регистрации и входа в систему и убедитесь, что язык, шрифты и стили соответствуют остальной части вашего сайта. По некоторым формам трудно догадаться, что они относятся к тому или иному сайту, особенно если их URL-адреса существенно отличаются от URL-адреса сайта.

## Продолжайте учиться {: #resources }

- [Создание потрясающих форм](/learn/forms/)
- [Рекомендации по разработке форм для мобильных устройств](https://www.smashingmagazine.com/2018/08/best-practices-for-mobile-form-design/)
- [Функциональные элементы управления в формах](/more-capable-form-controls)
- [Создание форм с поддержкой специальных возможностей](https://webaim.org/techniques/forms/)
- [Оптимизация процесса входа в систему с помощью Credential Management API](https://developer.chrome.com/blog/credential-management-api/)
- [Проверка номеров телефонов в Интернете с помощью WebOTP API](/web-otp/)

Фотография [Меган Ширек](https://unsplash.com/photos/_XFObcM_7KU) (Meghan Schiereck) с [Unsplash](https://unsplash.com) .
