---
layout: post
title: First Input Delay (FID)
authors:
  - philipwalton
date: 2019-11-07
updated: 2022-07-18
description: この投稿では、First Input Delay（FID）メトリックを紹介し、説明しますそれを測定する方法
tags:
  - performance
  - metrics
---

{% Aside %}First Input Delay (初回入力までの遅延時間、FID) は、[読み込みの応答性](/user-centric-performance-metrics/#types-of-metrics)を測定するための重要なユーザーを中心とした指標です。これは、応答のないページを操作する場合のユーザー体験を数値化したものであり、FID が短ければ短いほど、そのページがユーザーにとって[使いやすい](/user-centric-performance-metrics/#questions)ものであることが保証されます。{% endAside %}

良い第一印象を与えることがいかに重要かは誰もが知っています。新しい人と出会うとき、そしてウェブ上で体験を構築するときも重要です。

Web 上では、第一印象の良し悪しがお客様がロイヤル ユーザーとなってくれるか、それとも離脱して二度と戻ってこないかの分かれ道となります。そして、一体何が良い印象としてお客様の中に残るのか、またこのサイトがユーザー対してどのような印象を与えているのかをいかに測定するかが課題となります。

そして Web 上での第一印象には、様々な見方が存在します。サイトのデザインや視覚的な魅力を重視する視点もあれば、スピードや応答性などを重視する視点も存在します。

Web API を使用してサイトのデザインがどの程度ユーザーに好まれているかを測定するのは難しいですが、そのスピードや応答性については測定が可能です。

ユーザーにとっての第一印象であるサイトの読み込みの速さは、[First Contentful Paint (視覚コンテンツの初期表示時間、FCP)](/fcp/) を使用して測定が可能です。しかし、サイトが画面にピクセルを描画するスピードは、ユーザーが体験する物語の一部に過ぎません。ユーザーがそのピクセルを操作しようとしたときにサイトがいかに早く反応するかという点も、同様に重要なのです。

First Input Delay (FID) 指標は、サイトのインタラクティブ性と応答性に関するユーザーの第一印象を測定する場合に役立ちます。

## FIDとは何ですか？

FID は、ユーザーが最初にページを操作したとき (リンクをクリックしたり、ボタンをタップしたり、JavaScript を使用して実装されたカスタム コントロールを使用したりしたとき) から、その操作に応じてブラウザーが実際にイベント ハンドラーの処理を開始するまでの時間を測定します。

<picture>
  <source srcset="{{ "image/tcFciHGuF3MxnTr1y5ue01OGLBn2/eXyvkqRHQZ5iG38Axh1Z.svg" | imgix }}" media="(min-width: 640px)" width="400", height="100">
  {% Img src="image/tcFciHGuF3MxnTr1y5ue01OGLBn2/Se4TiXIdp8jtLJVScWed.svg", alt="良いfid値は2.5秒、悪い値は4.0秒を超え、その間の値は改善が必要", width="400", height="300" %}
</picture>

### FID における良いスコアとは？

良好なユーザー体験を提供するために、サイトは First Input Delay が **100 ミリ秒**以下になるように努力する必要があります。ほぼすべてのユーザーに対してこの目標値を確実に達成するためには、モバイル デバイスとデスクトップ デバイスに分けた上で、総ページロード数の 75 パーセンタイルをしきい値として設定します。

{% Aside %}この推奨事項の根拠となる調査および方法論に関する詳細については、「[Core Web Vitals の指標のしきい値の定義](/defining-core-web-vitals-thresholds/)」を参照してください。{% endAside %}

## FID の詳細

イベントに応答するコードを書く開発者としては、イベントが発生すればコードはすぐに実行されるものだと思い込みがちです。しかしながらユーザーとしては、スマートフォンで Web ページを読み込み、操作をしようとしているのに何も反応がなくてイライラした経験がきっとあるはずです。

一般的に入力遅延 (別名: 入力レイテンシー) は、ブラウザーのメイン スレッドが何か他の作業で忙しく、ユーザーの操作に (まだ) 応答できない場合に発生します。よくある理由としては、アプリが読み込んでいる大きな JavaScript ファイルの解析と実行でブラウザーがビジー状態になっていることが挙げられます。この間、ブラウザーはイベント リスナーを実行することができなくなります。なぜなら、ブラウザーが読み込んでいる JavaScript がブラウザーに対して別の作業を指示している可能性があるからです。

{% Aside 'gotchas' %}FID は、イベント処理の "遅延" のみを測定します。イベントの処理時間そのものや、イベント ハンドラーの実行後にブラウザーが UI を更新するまでの時間については測定を行いません。こういった時間もユーザー エクスペリエンスに影響を及ぼすものではありますが、FID の一部に含めてしまうと、イベントの非同期処理へと開発者を駆り立ててしまう可能性があります。これにより指標は改善されますが、ユーザー体験は悪化してしまう可能性があります。詳細については、後述する「[なぜ入力遅延のみを考慮するのですか？](#why-only-consider-the-input-delay)」を参照してください。{% endAside %}

典型的な Web ページの読み込みについて、次のタイムラインを例に考えてみましょう。

{% Img src="image/admin/9tm3f6pwlHMqNKuFvaP0.svg", alt="ページ読み込みの追跡例", width="800", height="260", linkTo=true %}

上の図は、ページがリソース (主に CSS や JS ファイル) に対していくつかのネットワーク リクエストを行い、それらのリソースのダウンロードが完了した後、メイン スレッドで処理を行っている様子を示しています。

その結果として、メイン スレッドが一時的にビジー状態となる期間 (ベージュ色の[タスク](https://html.spec.whatwg.org/multipage/webappapis.html#concept-task)ブロックで示されている部分) が現れます。

最初に発生する長い入力遅延は、一般的に [First Contentful Paint (FCP)](/fcp/) と [Time to Interactive (操作可能になるまでの時間、TTI)](/tti/) の間に発生します。これは、ページがコンテンツのレンダリングを一部完了しているものの、まだ確実に操作可能になっていないために発生します。この現象を解説するために、FCP と TTI をタイムラインに追加してみましょう。

{% Img src="image/admin/24Y3T5sWNuZD9fKhkuER.svg", alt="FCPとTTI を追加したページ読み込みの追跡例", width="800", height="340", linkTo=true %}

FCP と TTI の間に、かなり長い期間に渡って (3 つの[長く時間がかかっているタスク](/custom-metrics/#long-tasks-api)を含む) 処理が発生していることにお気付きかもしれません。ユーザーがその期間内にページの操作を試みると (たとえばリンクをクリックするなど)、クリックを受け取ってからメイン スレッドが応答するまでの時間に遅延が生じます。

それでは、ユーザーが最も長く時間がかかっているタスクの発生初期に近いタイミングでページの操作を試みた場合に一体何が起こるかについて考えてみましょう。

{% Img src="image/admin/krOoeuQ4TWCbt9t6v5Wf.svg", alt="FCP、TTI、FID を追加したページ読み込みの追跡例", width="800", height="380", linkTo=true %}

ブラウザーがタスクを実行している最中に入力が発生したため、ブラウザーが入力内容に応答するためには、このタスクが完了するまで待機しなければなりません。この待機時間が、このページでのユーザーの FID 値となります。

この例では、たまたまメイン スレッドが最もビジーな時間帯の初期にユーザーがページにアクセスしました。もしもユーザーがもう少し早く (アイドル期間内に) ページにアクセスしていれば、ブラウザーはすぐに反応できたはずです。このような入力遅延のばらつきは、指標のレポートで FID 値の分布を確認することの重要性を示しています。この点については、後述する「FID データの分析とレポート」のセクションで詳しく説明します。

### インタラクションにイベントリスナーがない場合はどうなりますか？

FID は、ブラウザーが入力イベントを受け取ってからメイン スレッドが次のアイドル期間に入るまでの差分を測定します。つまり、**イベント リスナーが登録されていない場合でも** FID は測定されるのです。その理由は、ユーザーによるインタラクションの多くはイベント リスナーを必要としませんが、実行のためにはメイン スレッドがアイドル状態であることは*必須である*からです。

たとえば、以下の HTML 要素はすべて、ユーザーによるインタラクションに応答する前に、メイン スレッドで実行中のタスクの完了を待つ必要があります。

- テキスト フィールド、チェックボックス、ラジオ ボタン ( `<input>`、`<textarea>`)
- ドロップダウンの選択 (`<select>`)
- リンク (`<a>`)

### なぜ最初の入力のみを考慮するのですか？

遅延の発生がユーザー エクスペリエンスの低下につながるのはどのような入力であっても同様ですが、Google では以下の理由から初回入力の遅延の測定を主に推奨しています。

- 初回入力の遅延は、サイトの応答性に関するユーザーの第一印象となります。第一印象は、サイトの品質や信頼性についての全体的な印象を形成する上で非常に重要な要素となります。
- 今日の Web で見られるインタラクティブ性に関する問題の中でも、最も重大なものはページの読み込み時に発生しています。そのため Google では、サイトでのユーザーによる初回のインタラクションの改善に注力することが、Web 全体におけるインタラクティブ性の改善に最も大きな効果をもたらすことになると考えています。
- 最初に発生する長い入力遅延に対するサイト側での修正方法として推奨されているソリューション (コードの分割、JavaScript の事前読み込みの削減など) は、ページの読み込み後に発生する入力に時間のかかるタイプの遅延を修正するためのソリューションとは必ずしも一致しません。これらの指標を分離することにより、より具体的なパフォーマンス ガイドラインを Web 開発者に提供できるようになります。

### 何が初回入力としてカウントされますか？

FID は、読み込み時のページの応答性を測定するための指標です。そのため、個別のアクション (クリック、タップ、キー押下など) による入力イベントにのみ焦点を当てています。

スクロールやズームなどといったその他のインタラクションは連続的な動作であり、パフォーマンス上の制限事項が全く異なります (また、別のスレッドで実行することによってブラウザー側でレイテンシーを隠すことも可能です)。

言い換えるならば、FID は [RAIL パフォーマンス モデル](/rail/)における R (応答性) に焦点を当てているのに対し、スクロールやズームは A (アニメーション) により強い関連性を持っており、それらのパフォーマンス品質は個別に評価される必要があります。

### ユーザーがサイトを操作したことがない場合はどうなりますか？

すべてのユーザーが訪問するサイトで必ず操作を行うわけではありません。また、すべての操作が FID に関連しているわけでもありません (前のセクションで説明したとおりです)。さらには、ユーザーによって初回操作が悪いタイミング (メイン スレッドが長期間ビジー状態になっている時) に行われる場合もあれば、良いタイミング (メイン スレッドが完全にアイドル状態になっている時) に行われる場合もあります。

つまり、FID 値がないユーザーもいれば、FID 値が低いユーザーや FID 値が高いユーザーもいるのです。

FID をどのように追跡し、レポートし、分析するかについては、おそらくあなたが慣れ親しんできた他の指標とはかなり方法が異なるはずです。次のセクションでは、これらを実施する場合に考えられる最善の方法について説明します。

### なぜ入力遅延のみを考慮するのですか？

先に述べたように、FID はイベント処理の "遅延" のみを測定します。イベントの処理時間そのものや、イベント ハンドラーの実行後にブラウザーが UI を更新するまでの時間については測定を行いません。

こういった時間もユーザーにとって重要であり、ユーザー体験に*かなりの*影響を及ぼすものですが、この指標には含まれません。これは、これらの時間を指標に含めることが実際にはユーザー体験の質を落としてしまう回避策の実装へと開発者を駆り立ててしまうことを避けることが目的です。この場合の回避策には、イベント ハンドラーのロジックを、イベントに関連するタスクから分離するために (`setTimeout()` や `requestAnimationFrame()` を介した) 非同期コールバックでラップすることなどが挙げられます。結果的に指標のスコアは改善されますが、ユーザーが実際に感じる応答時間は長くなってしまいます。

FID はイベントのレイテンシーの "遅延" 部分のみを測定するものですが、イベントのライフサイクルのより幅広い部分を追跡したいという開発者の方は、[Event Timing API](https://wicg.github.io/event-timing/) を使用することでそれらを実行することができます。詳細については、「[カスタム指標](/custom-metrics/#event-timing-api)」に記載されているガイドを参照してください。

## FID の測定方法

FID の測定には実際のユーザーによるページの操作が必要となるため、実際のユーザー環境でしか測定できません。FID は、以下のツールを使用して測定することができます。

{% Aside %}FID の測定には実際のユーザーが必要となるため、ラボ環境では測定できません。しかしながら、[Total Blocking Time (合計ブロック時間、TBT)](/tbt/) 指標はラボ環境でも測定可能です。実際のユーザー環境での FID との相関性も高く、インタラクティブ性に影響を与える問題の把握も可能です。ラボ環境での TBT の改善による最適化は、実際のユーザーに対する FID の改善にも繋がるはずです。{% endAside %}

### フィールド測定を実施するためのツール

- [Chrome User Experience Report](https://developer.chrome.com/docs/crux/)
- [PageSpeed Insights](https://pagespeed.web.dev/)
- [Search Console (Core Web Vitals Report)](https://support.google.com/webmasters/answer/9205520)
- [`web-vitals` JavaScript ライブラリ](https://github.com/GoogleChrome/web-vitals)

### JavaScript を使用して FID を測定する

{% BrowserCompat 'api.PerformanceEventTiming' %}

JavaScript を使用した FID の測定には、[Event Timing API](https://wicg.github.io/event-timing) を使用することができます。以下の例では、[`first-input`](https://developer.mozilla.org/docs/Web/API/PerformanceObserver) エントリをリッスンしてコンソールにログを記録する [`PerformanceObserver`](https://wicg.github.io/event-timing/#sec-performance-event-timing) の作成方法を示しています。

```js
new PerformanceObserver((entryList) => {
  for (const entry of entryList.getEntries()) {
    const delay = entry.processingStart - entry.startTime;
    console.log('FID candidate:', delay, entry);
  }
}).observe({type: 'first-input', buffered: true});
```

{% Aside 'warning' %}このコードは `first-input` エントリをコンソールにログとして記録して遅延を計算する方法を示していますが、JavaScript を使用して FID を測定する方法はより複雑です。詳細については、以下を参照してください。{% endAside %}

上記の例では、`first-input` エントリの遅延値をエントリの `startTime` と `processingStart` タイムスタンプ間の差分を取得することによって測定しています。ほとんどの場合においてこれが FID 値となりますが、すべての `first-input` エントリが FID の測定に有効であるとは限りません。

次のセクションでは、API がレポートする内容と、指標の計算方法の違いについて説明します。

#### 指標と API の違い

- API はバックグラウンドのタブで読み込まれているページに対して `first-input` エントリをディスパッチしますが、FID を計算する場合にはそういったページを無視する必要があります。
- また、API は初回入力が発生する前にページがバックグラウンドに移行したとしても `first-input` エントリをディスパッチしますが、FID を計算する場合にはこういったエントリも無視する必要があります (入力が考慮されるのは、ページがずっとフォアグラウンドにあった場合のみです)。
- API では、ページが [Back/Forward Cache](/bfcache/#impact-on-core-web-vitals) から復元された場合の `first-input` エントリはレポートされませんが、これらはユーザーにとって別々のページ訪問となるため、こういったケースにおいても FID は測定される必要があります。
- API では iframe 内で発生した入力はレポートされませんが、FID を適切に測定するためにはこれらの入力も考慮に入れる必要があります。サブフレームが集約のために API を使用してその親フレームに `first-input` エントリをレポートすることができます。

こういった微妙な違いをすべて記憶していなくても、[`web-vitals` JavaScript ライブラリ](https://github.com/GoogleChrome/web-vitals)を使用して FID を測定すれば、これらの違いを (可能な限り) 処理してくれます。

```js
import {onFID} from 'web-vitals';

// 実行可能となった時点ですぐに FID の測定やログ記録を実行します。
onFID(console.log);
```

JavaScript を使用して FID を測定する方法に関する詳細な例については、[`onFID()` のソース コード](https://github.com/GoogleChrome/web-vitals/blob/main/src/onFID.ts)を参照してください。

{% Aside %}場合によっては (クロスオリジン iframe など)、JavaScript を使用して FID を測定することはできません。詳細については、`web-vitals` ライブラリの「[limitations](https://github.com/GoogleChrome/web-vitals#limitations) (制限事項)」セクションを参照してください。{% endAside %}

### FID データの分析とレポート

FID 値にはばらつきが予想されるため、FID をレポートする際には値の分布を確認し、パーセンタイルが高いものに注目することが重要です。

すべての Core Web Vitals でしきい値となる[パーセンタイルの選択](/defining-core-web-vitals-thresholds/#choice-of-percentile)は 75 ですが、FID については特に 95 から 99 パーセンタイルを確認することを強くお勧めします。なぜならこれらのパーセンタイルは、ユーザーがあなたのサイトで経験した第一印象の中でも特に悪いものに対応しているからです。これらを確認することによって、最も改善が必要な領域を見つけることができます。

これは、デバイスのカテゴリや種類ごとにレポートをセグメント化した場合でも同様です。たとえば、デスクトップとモバイルで別々のレポートを実行している場合、デスクトップで最も気をつけなければならない FID 値はデスクトップ ユーザーのうち 95 から 99 パーセンタイルに含まれるユーザーの値であり、モバイルで最も気をつけなければならない FID 値はモバイル ユーザーのうち 95 から 99 パーセンタイルに含まれるユーザーの値となります。

## FID の改善方法

特定のサイトについて FID の改善方法を把握するには、Lighthouse でパフォーマンス監査を実行し、そこで推奨される具体的な [Opportunities](https://developer.chrome.com/docs/lighthouse/performance/#opportunities) (改善機会) に注目します。

FID はユーザー環境で測定される指標ですが (Lighthouse はラボ環境で使用するメトリック ツールです)、FID を改善するためのガイダンスは、ラボ環境での指標である [Total Blocking Time (TBT)](/tbt/) を改善するためのガイダンスと同じです。

FID の改善方法の詳細については、「[Optimize FID](/optimize-fid/) (FID を最適化する)」を参照してください。FID の改善にもつながる個別のパフォーマンス改善手法に関するその他のガイダンスについては、以下を参照してください。

- [サードパーティ製コードの影響を減らす](https://developer.chrome.com/docs/lighthouse/performance/third-party-summary/)
- [JavaScript の実行にかかる時間を短縮する](https://developer.chrome.com/docs/lighthouse/performance/bootup-time/)
- [メイン スレッドの作業を最小限に抑える](https://developer.chrome.com/docs/lighthouse/performance/mainthread-work-breakdown/)
- [リクエスト数を少なく、転送サイズを小さく維持する](https://developer.chrome.com/docs/lighthouse/performance/resource-summary/)<br>

{% include 'content/metrics/metrics-changelog.njk' %}
