import 'package:flutter/material.dart';
import 'package:stream_chat_flutter/stream_chat_flutter.dart';

/// A tile that is used to display the error indicator when
/// loading more items fails.
class StreamScrollViewLoadMoreError extends StatelessWidget {
  /// Creates a new instance of [StreamScrollViewLoadMoreError.list].
  const StreamScrollViewLoadMoreError.list({
    super.key,
    this.error,
    this.errorStyle,
    this.errorIcon,
    this.backgroundColor,
    required this.onTap,
    this.padding = const EdgeInsets.all(16),
    this.mainAxisSize = MainAxisSize.max,
    this.mainAxisAlignment = MainAxisAlignment.spaceBetween,
    this.crossAxisAlignment = CrossAxisAlignment.center,
  }) : _isList = true;

  /// Creates a new instance of [StreamScrollViewLoadMoreError.grid].
  const StreamScrollViewLoadMoreError.grid({
    super.key,
    this.error,
    this.errorStyle,
    this.errorIcon,
    this.backgroundColor,
    required this.onTap,
    this.padding = const EdgeInsets.all(16),
    this.mainAxisSize = MainAxisSize.max,
    this.mainAxisAlignment = MainAxisAlignment.spaceEvenly,
    this.crossAxisAlignment = CrossAxisAlignment.center,
  }) : _isList = false;

  /// The error message to display.
  final Widget? error;

  /// The style of the error message.
  final TextStyle? errorStyle;

  /// The icon to display next to the message.
  final Widget? errorIcon;

  /// The background color of the error message.
  final Color? backgroundColor;

  /// The callback to invoke when the user taps on the error indicator.
  final GestureTapCallback onTap;

  /// The amount of space by which to inset the child.
  final EdgeInsetsGeometry padding;

  /// The main axis size of the error view.
  final MainAxisSize mainAxisSize;

  /// The main axis alignment of the error view.
  final MainAxisAlignment mainAxisAlignment;

  /// The cross axis alignment of the error view.
  final CrossAxisAlignment crossAxisAlignment;

  final bool _isList;

  @override
  Widget build(BuildContext context) {
    final theme = StreamChatTheme.of(context);

    final errorText = AnimatedDefaultTextStyle(
      style: errorStyle ?? theme.textTheme.body.copyWith(color: Colors.white),
      duration: kThemeChangeDuration,
      child: error ?? const SizedBox(),
    );

    final errorIcon = AnimatedSwitcher(
      duration: kThemeChangeDuration,
      child: this.errorIcon ?? StreamSvgIcon.retry(color: Colors.white),
    );

    final backgroundColor = this.backgroundColor ??
        // ignore: deprecated_member_use
        theme.colorTheme.textLowEmphasis.withOpacity(0.9);

    final children = [errorText, errorIcon];

    return InkWell(
      onTap: onTap,
      child: ColoredBox(
        color: backgroundColor,
        child: Padding(
          padding: padding,
          child: _isList
              ? Row(
                  mainAxisSize: mainAxisSize,
                  mainAxisAlignment: mainAxisAlignment,
                  crossAxisAlignment: crossAxisAlignment,
                  children: children,
                )
              : Column(
                  mainAxisSize: mainAxisSize,
                  mainAxisAlignment: mainAxisAlignment,
                  crossAxisAlignment: crossAxisAlignment,
                  children: children,
                ),
        ),
      ),
    );
  }
}
