import 'package:analyzer/dart/element/element.dart';
import 'package:brick_build/generators.dart';
import 'package:build/build.dart';
import 'package:source_gen/source_gen.dart';

/// Output serializing code for all models with the `@_Annotation` annotation.
/// This generator is the annotation that discovers all models in the domain
/// and most commonly invokes sub generators.
///
/// For example, all `@ConnectOfflineFirstWithRest` models may be discovered and passed to
/// subsequent generators.
abstract class AnnotationSuperGenerator<_Annotation> extends GeneratorForAnnotation<_Annotation> {
  String get superAdapterName;

  const AnnotationSuperGenerator() : super();

  /// Given an [element] and an [annotation], scaffold all
  /// (de)serializing generators per provider.
  ///
  /// For example, from `SqliteSerdes(element, annotation).generators`.
  List<SerdesGenerator> buildGenerators(Element element, ConstantReader annotation);

  /// Combine relevant serdes generators for output to a single file: the adapter.
  String generateAdapter(Element element, ConstantReader annotation, BuildStep? buildStep) {
    final generators = buildGenerators(element, annotation);

    final adapterGenerator = AdapterGenerator(
      superAdapterName: superAdapterName,
      className: element.name!,
      generators: generators,
    );

    return adapterGenerator.generate();
  }

  /// Outputs any sub generators with the exception of the adapter.
  /// The adapter _should_ include code generated by this method.
  @override
  String generateForAnnotatedElement(element, annotation, buildStep) {
    final generators = buildGenerators(element, annotation);

    return generators.fold<List<String>>(<String>[], (acc, generator) {
      acc.add(generator.generate());
      return acc;
    }).join('\n');
  }
}
