# 综合实践

## 前端应用如何做国际化？

::: details

1. 使用国际化库：这是最常用的方法之一，可以通过引用第三方库来管理不同语言环境的资源文件。比如：

- React：可以使用 react-intl 或 react-i18next。
- Vue：可以使用 vue-i18n。
- Angular：可以使用@ngx-translate/core。

2. 浏览器 API：利用浏览器内置的国际化 API，如 Intl 对象，来格式化日期、时间、货币等。

3. 自建国际化框架：根据项目的具体需求，自定义国际化实现。这通常包括：

- 创建资源文件：为每种语言创建一个资源文件，用于存储翻译字符串。
- 语言选择功能：允许用户选择偏好的语言。
- 加载对应资源文件：根据用户的语言偏好，动态加载对应的资源文件并在界面上显示相应的文本。

4. 服务端支持：有些情况下，前端应用可能需要服务端的支持来实现国际化，如动态提供不同语言的数据内容。

5. URL 路由：在 URL 中包含语言参数，来确定显示哪种语言的内容。例如，/en/about 显示英文版“关于”页面，而 /zh/about 显示中文版。

6. 浏览器语言检测：通过检测浏览器的 navigator.language 属性来自动选择最合适的语言版本。

在实际应用中，根据项目的大小、复杂度以及特定需求，可以选择一种或多种方案结合使用，以达到最佳的国际化效果。

:::

## 请求失败会弹出一个 toast , 如何保证批量请求失败， 只弹出一个 toast

::: details

设置全局标志位：定义一个全局变量（如 isToastShown）来表示是否已经弹出过 toast。在请求失败的处理逻辑中，首先检查该标志位。如果尚未弹出 toast，则进行弹出操作，并设置标志位为 true；如果标志位已经为 true，则直接忽略后续的弹出操作。
使用防抖或节流函数：防抖（debounce）或节流（throttle）函数可以限制某个函数在一定时间内的执行次数。将弹出 toast 的操作封装在防抖或节流函数中，确保在短时间内的多个请求失败时，不会频繁弹出 toast。
集中处理错误：将所有请求的错误集中处理，而不是在每个请求的 catch 块中直接弹出 toast。例如，把所有请求的 Promise 添加到一个数组中，然后使用 Promise.all() 或其他类似方法来统一处理这些 Promise 的结果。如果所有请求都失败了，再弹出一个 toast。

代码示例

```js
let isToastShown = false; // 全局标志位

function makeRequests() {
  const requests = [fetchPost(), fetchComments()]; // 多个请求的 Promise

  Promise.all(requests)
  .then(() => {
    // 所有请求成功的处理逻辑
  })
  .catch(errors => {
    if (!isToastShown) { // 检查标志位
      notify(errors[0]); // 弹出 toast
      isToastShown = true; // 设置标志位为 true
    }
  });
}

function fetchJSON(url, input) {
  return fetch(url, input)
    .then(res => {
      if (res.ok) {
        return res.json();
      }
      const err = new HttpError(res);
      if (!isToastShown) { // 检查标志位
        notify(err); // 弹出 toast
        is toastShown = true; // 设置标志位为 true
      }
      throw err;
    });
}
```

:::

## 微前端架构一般是如何做 JavaScript 隔离的？

::: details

1. 使用沙箱技术：

iframe：最直接的隔离方式是将子应用运行在 iframe 中。这种方式提供了良好的隔离性，因为 iframe 内部有自己独立的执行环境，包括 JavaScript 运行环境和 DOM 环境。但 iframe 的使用可能会导致性能问题，且父子通信复杂。
JavaScript Sandboxing：通过创建一个独立的 JavaScript 执行环境，比如使用 Web Workers，或者更高级的沙箱库（如 Google 的 Caja），以在主页环境隔离执行 JavaScript 代码。

2. 命名空间和模块化：

命名空间：通过命名空间（Namespace）封装每个子应用的代码，确保全局变量和函数不会与其他应用冲突。
模块化：利用 ES Modules 或 CommonJS 等模块化标准，使代码封装在模块中运行，通过 import/export 管理依赖，减少全局变量的使用，从而实现隔离。

3. 状态管理隔离：

虽然主要关注 JavaScript 代码的隔离，但在单页应用中，子应用间状态管理（如使用 Redux、Vuex 等状态管理库）也可能导致隔离问题。可以为每个子应用创建独立的状态树，只通过明确定义的接口来共享必要的状态信息。

4. 使用微前端框架或库：

模块联邦（Module Federation）：Webpack 的模块联邦功能允许不同的前端应用共享 JavaScript 模块，同时保持应用间的隔离。它可以动态地加载另一个应用导出的模块，而不需要将它们打包进单个文件里。
专门的微前端框架：如 Single-SPA、Qiankun 等，这些框架提供了一套完整的解决方案，用于管理微前端应用的加载、卸载以及相互隔离，部分内部采用了类似沙箱的技术实现隔离。

5. 服务端渲染（SSR）隔离：

通过服务端渲染各个微前端应用，再将渲染好的静态 HTML 集成到主应用中。这样，每个子应用的 JavaScript 在客户端激活（Hydration）之前是隔离的。SSR 可以减少初次加载时间，同时具备部分隔离性，尤其是在初次加载阶段。

实施 JavaScript 隔离时，需要根据具体项目需求、技术栈和团队的熟练度来选取合适的隔离策略，以确保子应用之间的高度独立性和可维护性。

:::

## Qiankun 是如何做 JS 隔离的？

::: details

Qiankun 是一个基于 Single-SPA 的微前端实现库，它提供了比较完善的 JS 隔离能力，确保微前端应用间的独立运行，避免了全局变量污染、样式冲突等问题。Qiankun 实现 JS 隔离的主要机制包括：

1. JS 沙箱

Qiankun 使用 JS 沙箱技术为每个子应用创建一个独立的运行环境。沙箱有以下两种类型：

- 快照沙箱（Snapshot Sandbox）：在子应用启动时，快照并记录当前全局环境的状态，然后在子应用卸载时，恢复全局环境到启动前的状态。这种方式不会对全局对象进行真正的隔离，而是通过记录和恢复的方式避免全局环境被污染。

- Proxy 沙箱：通过 Proxy 对象创建一个全新的全局对象代理，子应用的所有全局变量修改操作都将在这个代理对象上进行，从而不会影响到真实的全局对象。这种方式提供了更为彻底的隔离效果，是 Qiankun 中推荐的沙箱隔离方式。

2. 动态执行 JS 代码

Qiankun 通过动态执行 JS 代码的方式加载子应用，避免了脚本直接在全局环境下执行可能导致的变量污染。具体来说，它可以动态获取子应用的 JS 资源，然后在沙箱环境中运行这些代码，确保代码执行的全局变量不会泄露到主应用的全局环境中。

3. 生命周期隔离

Qiankun 给每个子应用定义了一套生命周期钩子，如 bootstrap、mount、unmount 等，确保在应用加载、激活和卸载的过程中正确管理和隔离资源。通过在 unmount 生命周期钩子中正确清理子应用创建的全局监听器、定时器等，进一步保证了不同子应用间的独立性和隔离性。

4. 样式隔离

虽然主要针对 JS 隔离，Qiankun 也提供了样式隔离机制，通过动态添加和移除样式标签，保证子应用样式的独立性，避免不同子应用间的样式冲突。

通过以上机制，Qiankun 能够有效实现微前端架构中子应用的 JS 隔离，加强了应用间的独立性和安全性，使得不同子应用可以无缝集成在一起，同时又能够保持各自的运行环境独立不受影响。

:::

## 应用如何做应用灰度发布？

::: details

应用的灰度发布是将新版本逐步推出给有限的用户群体，以在完全发布之前监控其性能和搜集用户反馈的过程。这可以确保新版本的稳健性，减少因新版本可能引起的问题对所有用户的影响。以下是实现应用灰度发布的几种常见方法：

1. 基于 HTTP 头或 Cookie 的路由

通过识别用户的 HTTP 请求头（如 User-Agent）或特定的 Cookie，决定用户请求被路由到新版本还是旧版本的应用。这种方法通常需要负载均衡器或网关支持特定路由规则。

2. 使用服务网格（Service Mesh）

服务网格如 Istio 提供了复杂的流量管理能力，可以在微服务架构中实现灰度发布。通过定义路由规则，Istio 可以将特定比例或特定条件的流量导向新版本服务。

3. 功能开关（Feature Toggles）

功能开关允许开发者在代码中嵌入开关，根据配置动态激活或关闭某些功能。这样，新版本的功能可以被隐藏，直到你决定通过更改配置为特定用户群体开放。

4. DNS 路由

通过 DNS 管理，将部分用户的请求解析到部署了新版本应用的服务器上。这种方法简单，但切换和回退可能不如其他方法灵活。

5. CDN 切换

对于前端应用或静态资源，可以通过 CDN 配置，将部分用户的请求路由到包含新版本资源的 CDN 上。通过调整 CDN 的缓存规则控制版本切换。

6. A/B 测试平台

将灰度发布作为 A/B 测试的一部分，使用专门的 A/B 测试平台来控制哪些用户看到新版本。这种方法不仅可以实现灰度发布，还能搜集用户反馈和使用情况数据。

7. 容器编排和管理

在支持容器编排（如 Kubernetes）的环境中，可以通过部署新版本的 Pod 副本，并逐步增加新版本副本的数量，同时减少旧版本副本的数量实现灰度发布。

在实施灰度发布时，应该配合监控和日志记录工具，以便快速识别并解决新版本可能引入的问题。同时，在决定完全推出新版本之前，逐渐增加访问新版本的用户比例，确保在所有阶段都能够保持应用的稳定性和高性能。

:::

## 如何清理源码里面没有被应用的代码， 主要是 JS、TS、CSS 代码

清理源码中未被应用的 JavaScript (JS)、TypeScript (TS) 和 CSS 代码的关键在于合理利用工具和策略，来识别和移除这些废弃的代码。下面是一份指南，帮助你高效完成这一任务：

### 对于 JavaScript 和 TypeScript

::: details

1. 使用 ESLint

- 初始化 ESLint：如果你还没有使用 ESLint，可以通过 npx eslint --init 命令来初始化配置。
- 配置规则：确保在.eslintrc 配置文件中启用了 no-unused-vars 规则，以识别未使用的变量和函数。
- 使用 ESLint 的 --fix 选项: 虽然 ESLint 主要用于识别问题，但它的 --fix 选项可以自动修复一些问题，包括删除未使用的变量等。不过，这种方式相对保守，无法删除大块的未使用代码

2. 使用 TypeScript 编译器选项

对于 TypeScript 项目，可以在 tsconfig.json 文件中启用 noUnusedLocals 和 noUnusedParameters 选项，以识别未使用的本地变量和函数参数。

3. 利用 Webpack 的 Tree Shaking

- 确保在生产模式下使用 Webpack，它自带 Tree Shaking 功能，可以去除死代码（未被使用的代码）。
- 使用 ES6 模块语法（即 import 和 export），因为 Tree Shaking 仅支持静态导入。

:::

### 对于 CSS

::: details

1. 使用 PurgeCSS

PurgeCSS 分析你的内容和 CSS 文件，去除不匹配的选择器。非常适用于清楚在 HTML 或 JS 文件中未引用的 CSS 代码。
可以通过 Webpack、Gulp 或 PostCSS 等多种方式与 PurgeCSS 集成。

使用 PurgeCSS 时，配置你的内容文件路径（如 HTML 或 JSX 文件），它会扫描这些文件以确定哪些 CSS 选择器被使用

2. 使用 Codemods

Codemods 是 Facebook 提出的一种工具，允许你对代码库进行大规模的自动化重构。通过编写特定的脚本，你可以自定义删除或修改未被调用的代码的逻辑。例如，使用 jscodeshift 工具可以配合具体规则进行代码修改。

:::

注意事项

- 测试：自动删除代码后，务必执行完整的测试套件，确认改动不会影响现有功能。
- 版本控制：在进行删除操作之前，确保代码已经提交到版本控制系统，以便必要时可以恢复。
- 逐步执行：尤其是在较大或复杂的项目中，建议分步骤、逐渐移除未使用的代码，每次删除后都进行测试和评估。

使用这些策略和工具可以帮助自动化清理未使用的代码，但是请注意，完全自动化的过程可能会有风险，依然需要人工审核和测试以确保代码的质量和应用的稳定性。

## 一般是怎么做代码重构的

::: details

在前端项目中进行代码重构，一般可以遵循以下步骤：

1. 明确重构目标

确定需要解决的问题，例如提高代码的可读性、可维护性、性能，或者去除重复代码等。

2. 代码分析

- 对现有代码进行全面的审查和理解，包括代码结构、逻辑流程、函数和模块之间的关系等。
- 可以使用工具如 ESLint 检查代码风格和潜在问题，使用性能分析工具如 Chrome DevTools 的 Performance 面板来检测性能瓶颈。

3. 制定重构计划

- 根据分析结果，确定重构的步骤和顺序。
- 将大型的重构任务分解为较小的、可管理的子任务。

4. 重写代码结构

- 对模块和组件进行合理的拆分和组织，使代码结构更加清晰。
- 例如，将功能相关的代码提取到单独的函数或模块中，提高代码的内聚性和复用性。

5. 优化函数和方法

- 检查函数的长度和复杂性，对过长或过于复杂的函数进行分解。
- 去除不必要的参数传递和全局变量的使用。

6. 处理数据结构

评估数据的存储和使用方式，选择更合适的数据结构（如从数组切换到对象，或者使用 Map、Set 等）来提高数据操作的效率。

7. 优化性能

例如，减少不必要的计算、优化 DOM 操作、合理使用缓存等。

8. 测试和验证

对重构后的代码进行全面的单元测试、集成测试和端到端测试，确保功能的正确性和稳定性。

9. 代码审查

邀请团队成员对重构后的代码进行审查，获取反馈和建议，进一步优化代码。

10. 文档更新

对重构后的代码功能、接口和使用方法进行文档更新，方便其他开发人员理解和使用。

以一个简单的前端项目为例，假设有一个处理用户数据展示的模块，最初的代码可能是所有功能都写在一个大型的函数中，并且数据存储在全局变量中。 重构时：

- 将数据处理、数据获取和数据展示的功能分别提取到不同的函数中。
- 将数据从全局变量改为使用模块内部的私有变量或通过参数传递。
- 对数据处理函数进行优化，去除重复的代码逻辑。
- 为新的函数和模块添加必要的注释和文档说明。

通过这样的重构过程，可以使前端项目的代码质量得到显著提升，为后续的开发和维护提供更好的基础。

:::

## 深度SEO优化的方式有哪些（前端技术层面）

1. 网站结构优化：标签、语义化，更好地让搜索引擎读取
2. 网站速度优化：压缩、CDN、缓存、预加载、懒加载等
3. 页面渲染优化：SPA、SSR等，快速获取到完整页面
4. URL优化，尽可能短，描述性的URL或关键词优化URL结构
5. 链接优化：在网站内部设置相关性强的连接，使页面之间相互连接
6. Schema标记：使用结构化数据标记来表示网页内容
7. XML网站地图：创建和提交XML网站地图，提供网站结构和页面信息
8. Robots.txt文件：通过该文件指示搜索引擎哪些页面可以被爬取和索引，哪些页面不可访问
9. 使用HTTPS协议，更安全
10. 移动友好，响应式网站
11. 无障碍

## 参考

https://juejin.cn/post/7390413094445481984
