<?php /** @noinspection UnknownInspectionInspection */

/** @noinspection PhpComposerExtensionStubsInspection */

namespace eftec\DocumentStoreOne;

use DateTime;
use eftec\DocumentStoreOne\services\DocumentStoreOneCsv;
use eftec\DocumentStoreOne\services\DocumentStoreOneIgBinary;
use eftec\DocumentStoreOne\services\DocumentStoreOneJsonArray;
use eftec\DocumentStoreOne\services\DocumentStoreOneMsgPack;
use eftec\DocumentStoreOne\services\DocumentStoreOneJsonObj;
use eftec\DocumentStoreOne\services\DocumentStoreOneNone;
use eftec\DocumentStoreOne\services\DocumentStoreOnePHP;
use eftec\DocumentStoreOne\services\DocumentStoreOnePHPArray;
use eftec\DocumentStoreOne\services\IDocumentStoreOneSrv;
use Exception;
use Redis;
use RedisException;
use ReflectionObject;
use RuntimeException;

/**
 * Class DocumentStoreOne
 *
 * @version 1.27 2024-07-19
 * @author  Jorge Castro Castillo jcastro@eftec.cl
 * @link    https://github.com/EFTEC/DocumentStoreOne
 * @license LGPLv3 or commercial
 */
class DocumentStoreOne
{
    public const VERSION = '1.27';
    public const DSO_AUTO = 'auto';
    public const DSO_FOLDER = 'folder';
    public const DSO_APCU = 'apcu';
    public const DSO_REDIS = 'redis';
    public const DSO_NONE = 'none';
    /**
     * @var string It is used to append a value without updating the whole file.
     *             This value must be enough complex to avoid collisions.
     *             But it is also must be utf-8 compatible.
     *             10 x 8 bits = 80 bits (1,208,925,819,614,629,174,706,176 chances of colision)
     */
    public $separatorAppend = "~§¶¤¶¤¶§~\n";
    /** @var mixed $currentId The "ID" of the current document */
    public $currentId;
    /** @var string root folder of the database */
    public $database;
    /** @var string collection (subfolder) of the database */
    public $collection;
    /** @var int Maximium duration of the lock (in seconds). By default, it's 2 minutes */
    public $maxLockTime = 120;
    /** @var int Default number of retries. By default, it tries 100x0.1sec=10 seconds */
    public $defaultNumRetry = 100;
    /** @var int Interval (in microseconds) between retries. 100000 means 0.1 seconds */
    public $intervalBetweenRetry = 100000;
    /** @var string Default extension (with dot) of the document */
    public $docExt = ".dson";
    /** @var string=['php','php_array','json_object','json_array','csv','igbinary','msgpack','none'][$i] */
    public $serializeStrategy = 'php';
    /** @var IDocumentStoreOneSrv */
    public $srvSerialize;
    /** @var bool if true then it will never lock or unlock the document. It is useful for a read only base */
    public $neverLock = false;
    /** @var int DocumentStoreOne::DSO_* */
    public $strategy = self::DSO_FOLDER;
    public $regionDecimal = '.';
    public $regionDateTime = 'Y-m-d H:i:s';
    public $regionDate = 'Y-m-d';
    /** @var array=[]['auto','int','decimal','datetime','string][$i] */
    public $schemas;
    protected $tabular = false;
    /** @var Redis */
    public $redis;
    /** @var string=['','md5','sha1','sha256','sha512'][$i] Indicates if the key is encrypted or not when it's stored (the file name). Empty means, no encryption. You could use md5,sha1,sha256,.. */
    public $keyEncryption = '';
    /**
     * @var int nodeId It is the identifier of the node.<br>
     * It must be between 0..1023<br>
     * If the value is -1, then it randomizes its value each call.
     */
    public $nodeId = 1;
    protected $autoSerialize;
    public $throwable = true;
    public $latestError = '';
    /** @var DocumentStoreOne */
    protected static $instance;
    /**
     * @var string= 'sequencephp' (snowfake generation),'nextsequence' (generated by a document) or the name of the
     *      field.
     */
    protected $objectIndex;

    /**
     * DocumentStoreOne constructor.<br>
     * **Example:**
     * ```
     * $obj=new DocumentStoreOne(__DIR__."/base",'collectionFolder'); // using __DIR__
     * $obj=new DocumentStoreOne("/path/p2",'collectionFolder'); // using absolute path
     * $obj=new DocumentStoreOne("c:\path\p2",'collectionFolder'); // using absolute path (Windows)
     * $obj=new DocumentStoreOne("path/p2",'collectionFolder'); // using relative path (relative to the first PHP)
     * ```
     *
     * @param string      $database          root folder of the database
     * @param string      $collection        collection (subfolder) of the database. If the namr of the collection is
     *                                       empty then it uses the root folder.
     * @param string      $strategy          =['auto','folder','apcu','redis','none'][$i] The strategy is
     *                                       only used to lock/unlock purposes.
     * @param string      $server            Used for 'redis' (localhost:6379)
     * @param bool|string $serializeStrategy =['php','php_array','json_object','json_array','csv','igbinary','msgpack','none'][$i]
     *                                       It sets the strategy of serialization<br> If false then the value
     *                                       (inserted) is auto serialized<br>
     * @param string      $keyEncryption     =['','md5','sha1','sha256','sha512'][$i] it uses to encrypt the name of
     *                                       the keys
     *                                       (filename)
     *
     * @throws Exception
     * @example $flatcon=new DocumentStoreOne(__DIR__."/base",'collectionFolder');
     */
    public function __construct(
        string $database,
        string $collection = '',
        string $strategy = 'auto',
        string $server = "",
               $serializeStrategy = false,
        string $keyEncryption = ''
    )
    {
        if (self::isRelativePath($database)) {
            $database = dirname($_SERVER['SCRIPT_FILENAME']) . DIRECTORY_SEPARATOR . $database;
        }
        $this->database = $database;
        $this->collection = $collection;
        if ($serializeStrategy === false) {
            $this->autoSerialize = false;
            $this->srvSerialize = new DocumentStoreOneNone($this);
        } else if ($serializeStrategy === true) {
            // it is for old compatibility
            $this->srvSerialize = new DocumentStoreOneNone($this);
            $this->autoSerialize = true;
        } else {
            $this->autoSerialize(true, $serializeStrategy);
        }
        $this->keyEncryption = $keyEncryption;
        $this->setStrategy($strategy, $server);
        if (!is_dir($this->getPath())) {
            $backup = $this->collection;
            try {
                $this->createCollection("", false); // create base
                $this->createCollection($backup, false); // create collection and return previous configuration.
            } catch (Exception $ex) {
                $this->throwError("the folder is incorrect or I'm not unable to read  it: "
                    . $this->getPath() .
                    '. ' . $ex->getMessage());
            }
        }
        if (self::$instance === null) {
            self::$instance = $this;
        }
    }

    /**
     * It returns the first instance created DocumentStoreOne or throw an error if the instance is not set.<br>
     * **Example:**
     * ```
     * $doc=DocumentStoreOne::instance();
     * ```
     * @param bool $throwIfNull if true and the instance is not set, then it throws an exception<br>
     *                          if false and the instance is not set, then it returns null
     * @return DocumentStoreOne|null
     */
    public static function instance(bool $throwIfNull = true): ?DocumentStoreOne
    {
        if (self::$instance === null && $throwIfNull) {
            throw new RuntimeException('instance not created for DocumentStoreOne');
        }
        return self::$instance;
    }

    /**
     * It is a utility function that returns true if the path supplied is absolute or relative<br>
     * **Example:**
     * ```
     * $isRel=DocumentStoreOne::isRelativePath('somepath/path');
     * ```
     * @param string $path
     * @return bool
     */
    public static function isRelativePath(string $path): bool
    {
        $l = strlen($path);
        if ($l > 0 && $path[0] === '/') { // *nix absolute path
            return false;
        }
        if ($l > 2 && $path[1] === ':') { // windows absolute path
            return false;
        }
        return true;
    }

    /**
     * It returns true if it is using a tabular value<br>
     * **Example:**
     * ```
     * $isTabular=$this->isTabular();
     * ```
     * @return bool
     */
    public function isTabular(): bool
    {
        return $this->tabular;
    }

    /**
     * It sets the strategy to lock and unlock the folders<br>
     * **Example:**
     * ```
     * $this->setStrategy('folder');
     * $this->setStrategy('redis','127.0.0.1:6379');
     * ```
     *
     * @param string|int $strategy =['auto','folder','apcu','redis','none'][$i]
     * @param string     $server
     *
     * @throws Exception
     * @noinspection ClassConstantCanBeUsedInspection
     */
    public function setStrategy($strategy, string $server = ""): void
    {
        if ($strategy === self::DSO_AUTO) {
            if (function_exists('apcu_add')) {
                $this->strategy = self::DSO_APCU;
            } else {
                $this->strategy = self::DSO_FOLDER;
            }
            $strategy = $this->strategy;
        } else {
            $this->strategy = $strategy;
        }
        switch ($strategy) {
            case self::DSO_NONE:
            case self::DSO_FOLDER:
                break;
            case self::DSO_APCU:
                if (!function_exists("apcu_add")) {
                    $this->throwError('APCU is not defined');
                    return;
                }
                break;
            case self::DSO_REDIS:
                /** @noinspection ClassConstantCanBeUsedInspection */
                if (!class_exists("\Redis")) {
                    $this->throwError('Redis is not defined');
                    return;
                }
                if (function_exists('cache')) {
                    /** @noinspection PhpUndefinedFunctionInspection */
                    $this->redis = cache();// inject using the cache function (if any).
                } else {
                    $this->redis = new Redis();
                    $host = explode(':', $server);
                    $r = @$this->redis->pconnect($host[0], $host[1]??6379, 30); // 30 seconds timeout
                    if (!$r) {
                        $this->throwError('Unable to open Redis, check the server and connections.');
                        return;
                    }
                }
                break;
            default:
                $this->throwError("Strategy not defined");
        }
        $this->resetChain();
    }

    /**
     * It gets the current path.<br>
     * **Example:**
     * ```
     * $path=$this->getPath();
     * ```
     *
     * @return string
     */
    protected function getPath(): string
    {
        return $this->database . "/" . $this->collection;
    }

    /**
     * Util function to fix the cast of an object using an empty object.<br>
     * **Example:**
     * ```
     * utilCache::fixCast($objectRightButEmpty,$objectBadCast);
     * ```
     *
     * @param object|array $destination Object may be empty with the right cast.
     * @param object|array $source      Object with the wrong cast.
     *
     * @return void
     */
    public static function fixCast(&$destination, $source): void
    {
        if (is_array($source)) {
            $getClass = get_class($destination[0]);
            $array = [];
            foreach ($source as $sourceItem) {
                $obj = new $getClass();
                self::fixCast($obj, $sourceItem);
                $array[] = $obj;
            }
            $destination = $array;
        } else {
            $sourceReflection = new ReflectionObject($source);
            $sourceProperties = $sourceReflection->getProperties();
            foreach ($sourceProperties as $sourceProperty) {
                $name = $sourceProperty->getName();
                if (is_object(@$destination->{$name})) {
                    if (get_class(@$destination->{$name}) === "DateTime") {
                        // source->name is a stdclass, not a DateTime, so we could read the value with the field date
                        try {
                            $destination->{$name} = new DateTime($source->$name->date);
                        } catch (Exception $e) {
                            $destination->{$name} = null;
                        }
                    } else {
                        self::fixCast($destination->{$name}, $source->$name);
                    }
                } else {
                    $destination->{$name} = $source->$name;
                }
            }
        }
    }

    /**
     * Set if we need to lock/unlock every time we want to read/write a value
     *
     * @param bool $neverLock if its true then the register is never locked. It is fast but it's not concurrency-safe
     */
    public function setNeverLock(bool $neverLock = true): void
    {
        $this->neverLock = $neverLock;
    }

    /**
     * It appends a value to an existing document. It creates an array of values if it doesn't exist.<br>
     * **Example:**
     * ```
     * $this->appendvalue('20',[1,2,3]); // document "20" = [[1,2,3]]
     * $this->appendvalue('20',[3,4,5]); // document "20" = [[1,2,3],[3,4,5]]
     * ```
     *
     * @param string $id       "ID" of the document.
     * @param mixed  $addValue This value could be serialized.
     * @param int    $tries    number of tries. The default value is -1 (it uses the default value $defaultNumRetry)
     *
     * @return bool It returns false if it fails to lock the document or if it's unable to read the document. Otherwise
     *              it returns true
     * @throws Exception
     */
    public function appendValue(string $id, $addValue, int $tries = -1): bool
    {
        $this->currentId = $id;
        $file = $this->filename($id);
        if ($this->lock($file, $tries)) {
            try {
                $result = $this->srvSerialize->appendValue($file, $id, $addValue, $tries);
                $this->unlock($file, $tries);
                $this->resetChain();
                return $result;
            } catch (Exception $ex) {
                $this->unlock($file, $tries);
                $this->throwError($ex);
            }
        }
        $this->throwError("unable to lock file [$this->currentId]");
        return false; // unable to lock
    }

    /**
     * Used internally
     * @param     $file
     * @param     $id
     * @param     $addValue
     * @param int $tries
     * @return bool|resource
     */
    public function appendValueDecorator($file, $id, $addValue, int $tries = -1)
    {
        if ($this->ifExist($id, null)) {
            $fp = @fopen($file, 'rb+');
        } else {
            $this->unlock($file);
            $this->resetChain();
            return $this->insert($id, [$addValue], $tries);
        }
        if ($fp === false) {
            $this->unlock($file);
            $this->throwError(error_get_last());
            return false; // file exists but I am unable to open it.
        }
        $this->resetChain();
        return $fp;
    }

    /**
     * For internal use.<br>
     * It is used when the service is unable to insert an information without modify the whole file<br>
     * So it modifies the whole file
     * @param $filePath
     * @param $addValue
     * @return bool
     */
    public function appendValueRaw($filePath, $addValue): bool
    {
        $fp = @fopen($filePath, 'ab');
        if ($fp === false) {
            $this->unlock($filePath);
            $this->throwError(error_get_last());
            return false; // file exists but I am unable to open it.
        }
        $addValue = $this->serialize($addValue);
        if ($this->tabular) {
            $addValue = $this->separatorAppend . $addValue;
        }
        $r = @fwrite($fp, $addValue);
        @fclose($fp);
        $this->unlock($filePath);
        if ($r === false) {
            $this->throwError(error_get_last());
        }
        $this->resetChain();
        return ($r !== false);
    }

    /**
     * Used internally.
     * @param     $id
     * @param     $addValue
     * @param int $tries
     * @return bool
     * @noinspection PhpUnusedParameterInspection
     * @throws RedisException
     */
    public function appendValueRaw2($id, $addValue, int $tries = -1): bool
    {
        // it is called pre-locked, so we won't lock or unlock
        $content = $this->get($id, 0);
        if ($content === false) {
            $this->throwError('Append: unable to read file', true);
            return false;  // unable to read
        }
        if (!is_array($content)) {
            $this->throwError('Append: content is not an array, unable to insert', true);
            return false; // not able to add
        }
        $content[] = $addValue;
        $this->resetChain();
        return $this->update($id, $content, 0); // it is called pre-locked, so we won't lock or unlock
    }

    /**
     * Convert "ID" to a full filename. If keyencryption then the name is encrypted.<br>
     * **Example:**
     * ```
     * $fullfilename=$this->filename('doc20'); // "/folder/somefolder/doc20.dson"
     * ```
     *
     * @param string $id
     *
     * @return string full filename
     */
    public function filename(string $id): string
    {
        $file = $this->keyEncryption ? hash($this->keyEncryption, $id) : $id;
        return $this->getPath() . "/" . $file . $this->docExt;
    }

    /**
     * It locks a file depending on the current strategy.
     *
     * @param     $filepath
     * @param int $maxRetry the number of times that it will retry to lock the file<br>>
     *                      -1 means it uses the default value.<br>
     *                      0 means it will not try to lock<br>
     * @return bool returns false if it fails
     */
    protected function lock($filepath, int $maxRetry = -1): bool
    {
        if ($this->neverLock || $maxRetry === 0) {
            return true;
        }
        $maxRetry = ($maxRetry === -1) ? $this->defaultNumRetry : $maxRetry;
        try {
            switch ($this->strategy) {
                case self::DSO_APCU:
                    $try = 0;
                    while (@apcu_add("documentstoreone." . $filepath, 1, $this->maxLockTime) === false && $try < $maxRetry) {
                        $try++;
                        usleep($this->intervalBetweenRetry);
                    }
                    return ($try < $maxRetry);
                case self::DSO_REDIS:
                    $try = 0;
                    while (@$this->redis->set("documentstoreone." . $filepath, 1, ['NX', 'EX' => $this->maxLockTime]) !==
                        true
                        && $try < $maxRetry) {
                        $try++;
                        usleep($this->intervalBetweenRetry);
                    }
                    return ($try < $maxRetry);
                case self::DSO_FOLDER:
                    clearstatcache();
                    $lockname = $filepath . ".lock";
                    $life = @filectime($lockname); // $life=false is ok (the file doesn't exist)
                    $try = 0;
                    while (!@mkdir($lockname, 0755) && is_dir($lockname) && $try < $maxRetry) {
                        $try++;
                        if ($life && (time() - $life) > $this->maxLockTime) {
                            rmdir($lockname);
                            $life = false;
                        }
                        usleep($this->intervalBetweenRetry);
                    }
                    return ($try < $maxRetry);
                case self::DSO_NONE:
                    return true;
            }
        } catch (Exception $ex) {
        }
        return false;
    }

    /**
     * Return if the document exists. It doesn't check until the document is fully unlocked unless $tries=null<br>
     * **Example:**
     * ```
     * $found=$this->ifExist('doc20');
     * ```
     * @param string   $id    "ID" of the document.
     * @param int|null $tries number of tries.<br>
     *                        The default value is -1 (it uses the default value $defaultNumRetry)<br>
     *                        If $tries=null then it never checks the lock.
     *
     * @return bool True if the information was read, otherwise false.
     */
    public function ifExist(string $id, ?int $tries = -1): bool
    {
        $file = $this->filename($id);
        if ($tries !== null) {
            if ($this->lock($file, $tries)) {
                $exist = file_exists($file);
                $this->unlock($file);
                return $exist;
            }
        } else {
            return file_exists($file);
        }
        return false;
    }

    /**
     * Unlocks a document
     *
     * @param string $filePath full file path/key of the document to unlock.
     * @param int    $tries
     * @return bool returns true if the unlocks works. Otherwise, it returns false.
     */
    public function unlock(string $filePath, int $tries = -1): bool
    {
        if ($this->neverLock || $tries === 0) {
            return true;
        }
        try {
            switch ($this->strategy) {
                case self::DSO_APCU:
                    return apcu_delete("documentstoreone." . $filePath);
                case self::DSO_REDIS:
                    return ($this->redis->del("documentstoreone." . $filePath) > 0);
                case self::DSO_NONE:
                    return true;
            }
            $unlockname = $filePath . ".lock";
            $tries = $tries === -1 ? $this->defaultNumRetry : $tries;
            $try = 0;
            // retry to delete the unlockname folder. If fails then it tries it again.
            while (!@rmdir($unlockname) && $try < $tries) {
                $try++;
                if (!is_dir($unlockname)) { // is not directory or it was already deleted.
                    break;
                }
                usleep($this->intervalBetweenRetry);
            }
            return ($try < $this->defaultNumRetry);
        } catch (Exception $ex) {
            return false;
        }
    }

    /**
     * Add a document.<br>
     * **Example**
     * ```
     * $ok=$this->insert("doc20",$values);
     * ```
     *
     * @param string       $id       "ID" of the document.
     * @param string|array $document The document
     * @param int          $tries    number of tries. The default value is -1 (it uses the default value
     *                               $defaultNumRetry)
     *
     * @return bool True if the information was added, otherwise false
     */
    public function insert(string $id, $document, int $tries = -1): bool
    {
        $file = $this->filename($id);
        $this->currentId = $id;
        if ($this->lock($file, $tries)) {
            if ((!is_array($document) || !isset($document[0])) && $this->tabular) {
                $write = false;
                $errorLast = 'Data is not tabular';
            } else if (!file_exists($file)) {
                if ($this->autoSerialize) {
                    $this->srvSerialize->insert($id, $document, $tries);
                    // note: for the csv, it includes the header (if the csv uses headers)
                    $write = @file_put_contents($file, $this->serialize($document), LOCK_EX);
                } else {
                    $write = @file_put_contents($file, $document, LOCK_EX);
                }
                $errorLast = @error_get_last();
            } else {
                $write = false;
                $errorLast = "File does exist [$this->currentId]. Unable to override file";
            }
            if ($write === false) {
                @$this->unlock($file);
                $this->throwError($errorLast);
                return false;
            }
            $this->unlock($file);
            $this->resetChain();
            return true;
        }
        $this->unlock($file);
        $this->throwError("Unable to lock file [$this->currentId]");
        return false;
    }

    /**
     * @param mixed $document The document content to serialize.
     * @param bool  $special  used by serialize_php_array() and serializeCSV()
     *
     * @return string
     */
    public function serialize($document, bool $special = false): string
    {
        return $this->srvSerialize->serialize($document, $special);
    }


    //region csv

    /**
     * Returns true if the input is a table (a 2-dimensional array)
     * @param mixed $table
     * @return bool
     */
    public function isTable($table): bool
    {
        return isset($table[0]) && is_array($table[0]);
    }

    public function getType($input): string
    {
        // array
        if (is_array($input)) {
            return 'array';
        }
        // int?
        if ($this->isInt($input)) {
            return 'int';
        }
        // decimal value?
        if ($this->regionDecimal !== '.') {
            $inputD = str_replace($this->regionDecimal, '.', $input);
        } else {
            $inputD = $input;
        }
        if (is_numeric($inputD)) {
            return 'decimal';
        }
        // datetime?
        $inputD = ($input instanceof DateTime) ? $input : DateTime::createFromFormat($this->regionDateTime, $input);
        if ($inputD instanceof DateTime) {
            return 'datetime';
        }
        // date
        $inputD = DateTime::createFromFormat($this->regionDate, $input);
        if ($inputD instanceof DateTime) {
            return 'date';
        }
        return is_string($input) ? 'string' : 'oobject';
    }

    public function isInt($input): bool
    {
        if ($input === null || $input === '') {
            return false;
        }
        if (is_int($input)) {
            return true;
        }
        if (is_float($input) || is_object($input)) {
            return false;
        }
        if ($input[0] === '-') {
            return ctype_digit(substr($input, 1));
        }
        return ctype_digit($input);
    }

    public function convertTypeBack($input, $type)
    {
        return $this->srvSerialize->convertTypeBack($input, $type);
    }

    public function convertType($input, $type)
    {
        return $this->srvSerialize->convertType($input, $type);
    }

    //endregion
    public static function serialize_php_array($document, $special = false): ?string
    {
        if ($special) {
            // for append
            return var_export($document, true);
        }
        return "<?php /** @generated */\nreturn " . var_export($document, true) . ';';
    }

    /**
     * It sets the default object field used for index.
     *
     * @param string $indexField
     */
    public function setObjectIndex(string $indexField): void
    {
        $this->objectIndex = $indexField;
    }

    /**
     * It inserts (or update) an associative array or an object into the document store<br>
     * Example:<br>
     * $obj=['id'=>1,'name'=>'john'];<br>
     * $doc->insertOrUpdateObject($obj,'id');<br>
     *
     * @param array|object $object     The object (or associative array) to store
     * @param string|null  $indexField =['sequencephp','nextsequence'][$i]<br>
     *                                 if null then it uses the default index field defined by setObjectIndex()<br>
     *                                 if sequencephp then it uses snowflake for generate a new sequence<br>
     *                                 if nextsequence then it uses a document sequence<br>
     *
     * @return string the index value
     * @throws Exception
     * @throws Exception
     * @see DocumentStoreOne::setObjectIndex
     */
    public function insertOrUpdateObject($object, ?string $indexField = null)
    {
        return $this->insertObject($object, $indexField, 'insertorupdate');
    }

    /**
     * It inserts an associative array or an object into the document store
     *
     * @param array|object $object     The object to store
     * @param string|null  $indexField =['sequencephp','nextsequence'][$i]<br>
     *                                 if null then it uses the default index field defined by setObjectIndex()<br>
     *                                 if sequencephp then it uses snowflake for generate a new sequence<br>
     *                                 if nextsequence then it uses a document sequence<br>
     *                                 if the value is tabular, then it uses the first row to find the id<br>
     *
     * @param string       $operation  =['insert','insertorupdate'][$i]
     *
     * @return string the index value
     * @throws Exception
     * @see DocumentStoreOne::setObjectIndex
     */
    public function insertObject($object, ?string $indexField = null, string $operation = 'insert')
    {
        if ($indexField === null) {
            $indexField = $this->objectIndex;
        }
        switch ($indexField) {
            case 'sequencephp':
                $idx = $this->getSequencePHP();
                break;
            case 'nextsequence':
                $idx = $this->getNextSequence();
                break;
            default:
                $idx = $indexField;
                if ($this->tabular && isset($object[0])) {
                    $idx = is_object($object[0]) ? $object[0]->{$idx} : $object[0][$idx];
                } else {
                    $idx = is_object($object) ? $object->{$idx} : $object[$idx];
                }
        }
        try {
            switch ($operation) {
                case 'insert':
                    $this->insert($idx, $object);
                    break;
                case 'insertorupdate':
                    $this->insertOrUpdate($idx, $object);
                    break;
                default:
                    trigger_error('insertObject: operation not defined');
            }
            $this->resetChain();
            return $idx;
        } catch (Exception $ex) {
            $this->throwError($ex->getMessage());
            return false;
        }
    }

    public function throwError($msg, $append = false): void
    {
        if (is_array($msg)) {
            $msg = "{$msg['message']} file:{$msg['file']}[{$msg['line']}] type:{$msg['type']}";
        }
        if ($this->throwable) {
            $this->resetChain();
            throw @new RuntimeException($msg);
        }
        $this->resetChain();
        if ($append) {
            $this->latestError .= $msg;
        } else {
            $this->latestError = $msg;
        }
    }

    public function lastError(): string
    {
        return $this->latestError;
    }

    public function resetError(): void
    {
        $this->latestError = '';
    }

    protected function resetChain(): void
    {
        $this->throwable = true;
    }

    /**
     * <p>This function returns a unique sequence<p>
     * It ensures a collision free number only if we don't do more than one operation
     * per 0.0001 second However,it also adds a pseudo random number (0-4095)
     * so the chances of collision is 1/4095 (per two operations executed every 0.0001 second).<br>
     * It is based on Twitter's Snowflake number.
     *
     * @return string (it returns a 64bit integer).
     * @throws Exception
     */
    public function getSequencePHP(): string
    {
        $ms = microtime(true); // we use this number as a random number generator (we use the decimals)
        //$ms=1000;
        $timestamp = round($ms * 1000);
        $rand = (int)(fmod($ms, 1) * 1000000) % 4096; // 4096= 2^12 It is the millionth of seconds
        if ($this->nodeId === -1) {
            $number = random_int(0, 1023); // a 10bit number.
            $calc = (($timestamp - 1459440000000) << 22) + ($number << 12) + $rand;
        } else {
            $calc = (($timestamp - 1459440000000) << 22) + ($this->nodeId << 12) + $rand;
        }
        usleep(1);
        return '' . $calc;
    }

    /**
     * It gets the next sequence. If the sequence doesn't exist, it generates a new one with 1.<br>
     * You could peek a sequence with get('genseq_*name*')<br>
     * If the sequence is corrupt then it's resetted.<br>
     *
     * @param string $name              Name of the sequence.
     * @param int    $tries             number of tries. It uses the default value defined in $defaultNumRetry (-1)
     * @param int    $init              The initial value of the sequence (if it's created)
     * @param int    $interval          The interval between each sequence. It could be negative.
     * @param int    $reserveAdditional Reserve an additional number of sequence. It's useful when you want to
     *                                  generates many sequences at once.
     *
     * @return bool|int It returns false if it fails to lock the sequence or if it's unable to read thr sequence.
     *                  Otherwise, it returns the sequence
     */
    public function getNextSequence(string $name = "seq", int $tries = -1, int $init = 1, int $interval = 1, int $reserveAdditional = 0)
    {
        $id = "genseq_" . $name;
        $file = $this->filename($id) . ".seq";
        if ($this->lock($file, $tries)) {
            if (file_exists($file)) {
                $read = @file_get_contents($file);
                if ($read === false) {
                    $this->unlock($file);
                    $this->throwError(error_get_last());
                    return false; // file exists but I am unable to read it.
                }
                $read = (is_numeric($read)) ? ($read + $interval)
                    : $init; // if the value stored is numeric, then we add one, otherwise, it starts with 1
            } else {
                $read = $init;
            }
            $write = @file_put_contents($file, $read + $reserveAdditional, LOCK_EX);
            $this->unlock($file);
            if ($write === false) {
                $this->throwError(error_get_last());
            }
            $this->resetChain();
            return ($write === false) ? false : $read;
        }
        $this->resetChain();
        return false; // unable to lock
    }

    /**
     * Add or update a whole document.
     *
     * @param string       $id       "ID" of the document.
     * @param string|array $document The document
     * @param int          $tries    number of tries. The default value is -1, it uses the default value
     *                               $defaultNumRetry.
     *
     * @return bool True if the information was added, otherwise false
     */
    public function insertOrUpdate(string $id, $document, int $tries = -1): bool
    {
        $file = $this->filename($id);
        $this->currentId = $id;
        if ($this->lock($file, $tries)) {
            if ($this->autoSerialize) {
                $this->srvSerialize->insert($id, $document, $tries);
                $write = @file_put_contents($file, $this->serialize($document), LOCK_EX);
            } else {
                $write = @file_put_contents($file, $document, LOCK_EX);
            }
            $this->unlock($file);
            if ($write === false) {
                $this->throwError(error_get_last());
            }
            $this->resetChain();
            return ($write !== false);
        }
        $this->throwError("Unable to lock file [$this->currentId]");
        return false;
    }

    /**
     * Update a whole document
     *
     * @param string       $id       "ID" of the document.
     * @param string|array $document The document
     * @param int          $tries    number of tries. The default value is -1 (it uses the default value
     *                               $defaultNumRetry)
     *
     * @return bool True if the information was added, otherwise false
     */
    public function update(string $id, $document, int $tries = -1): bool
    {
        $file = $this->filename($id);
        $this->currentId = $id;
        if ($this->lock($file, $tries)) {
            if (file_exists($file)) {
                if ($this->autoSerialize) {
                    $write = @file_put_contents($file, $this->serialize($document), LOCK_EX);
                } else {
                    $write = @file_put_contents($file, $document, LOCK_EX);
                }
            } else {
                $write = false;
            }
            $this->unlock($file, $tries);
            if ($write === false) {
                $this->throwError(error_get_last());
            }
            $this->resetChain();
            return ($write !== false);
        }
        $this->throwError("Unable to lock file [$this->currentId]");
        return false;
    }

    /**
     * Set the current collection. It also could create the collection.
     *
     * @param string $collection
     * @param bool   $createIfNotExist if true then it checks if the collection (folder) exists, if not then it's
     *                                 created
     *
     * @return DocumentStoreOne
     */
    public function collection(string $collection, bool $createIfNotExist = false): DocumentStoreOne
    {
        $this->collection = $collection;
        if ($createIfNotExist && !$this->isCollection($collection)) {
            $this->createCollection($collection);
        }
        return $this;
    }

    /**
     * Check a collection
     *
     * @param string $collection
     *
     * @return bool It returns false if it's not a collection (a valid folder)
     */
    public function isCollection(string $collection): bool
    {
        $this->collection = $collection;
        return is_dir($this->getPath());
    }

    /**
     * Creates a collection
     *
     * @param string $collection
     * @param bool   $throwifFail
     * @return bool true if the operation is right, false if it fails.
     */
    public function createCollection(string $collection, bool $throwifFail = true): bool
    {
        $oldCollection = $this->collection;
        $this->collection = $collection;
        $r = @mkdir($this->getPath(), 0755);
        if ($r === false && $throwifFail) {
            $this->throwError('error create collection :' . @error_get_last());
        }
        $this->collection = $oldCollection;
        $this->resetChain();
        return $r;
    }

    /**
     * It deletes a collection inside a database,including its content.
     *
     * @param string  $collection
     * @param boolean $throwOnError
     * @param bool    $includeContent if true (default), then it deletes the content.
     * @return void
     */
    public function deleteCollection(string $collection, bool $throwOnError = false, bool $includeContent = true): void
    {
        $oldCollection = $this->collection;
        $this->collection = $collection;
        try {
            $r = false;
            if ($includeContent) {
                array_map('unlink', glob($this->getPath() . "/*.*"));
            }
            $r = rmdir($this->getPath());
        } catch (Exception $ex) {
            if ($throwOnError) {
                $this->throwError($ex->getMessage());
            }
        }
        if ($r === false && $throwOnError) {
            $this->throwError('error create collection :' . @error_get_last());
        }
        $this->collection = $oldCollection;
        $this->resetChain();
    }

    /**
     * It sets if we want to auto serialize the information, and we set how it is serialized
     *      php = it serializes using serialize() function
     *      php_array = it serializes using include()/var_export() function. The result could be cached using OpCache
     *      json_object = it is serialized using json (as object)
     *      json_array = it is serialized using json (as array)
     *      none = it is not serialized. Information must be serialized/de-serialized manually
     *      php_array = it is serialized as a php_array
     *
     * @param bool      $value
     * @param string    $strategy =['auto','php','php_array','json_object','json_array','csv','igbinary','msgpack','none'][$i]
     * @param bool|null $tabular  indicates if the value is tabular (it has rows and columns) or not.<br>
     *                            Note: if null then it takes the default value (csv is tabular, while others
     *                            don't<br>
     */
    public function autoSerialize(bool $value = true, string $strategy = 'auto', ?bool $tabular = null): void
    {
        $this->autoSerialize = $value;
        if ($value === false) {
            $strategy = 'none';
        }
        $this->serializeStrategy = $strategy;
        if ($this->serializeStrategy === 'auto') {
            if (function_exists('igbinary_serialize')) {
                $this->serializeStrategy = 'igbinary';
            } elseif (function_exists('msgpack_pack')) {
                $this->serializeStrategy = 'msgpack';
            } else {
                $this->serializeStrategy = 'php';
            }
        }
        switch ($this->serializeStrategy) {
            case 'csv':
                $this->srvSerialize = new DocumentStoreOneCsv($this);
                break;
            case 'igbinary':
                $this->srvSerialize = new DocumentStoreOneIgBinary($this);
                break;
            case 'msgpack':
                $this->srvSerialize = new DocumentStoreOneMsgPack($this);
                break;
            case 'php':
                $this->srvSerialize = new DocumentStoreOnePHP($this);
                break;
            case 'php_array':
                $this->srvSerialize = new DocumentStoreOnePHPArray($this);
                break;
            case 'json_object':
                $this->srvSerialize = new DocumentStoreOneJsonObj($this);
                break;
            case 'json_array':
                $this->srvSerialize = new DocumentStoreOneJsonArray($this);
                break;
            default:
                $this->srvSerialize = new DocumentStoreOneNone($this);
                break;
        }
        if ($tabular === null) {
            $this->tabular = $this->srvSerialize->defaultTabular();
        }
    }

    /**
     * It sets the style of the csv.
     *
     * @param string $separator    For example ",", "\t", etc.
     * @param string $enclosure    The enclosure of a string value.
     * @param string $lineEnd
     * @param string $header       If the csv contains or not a header
     * @param string $prefixColumn If the csv or the data does not contain name, then it uses this value to create a
     *                             column name<br>
     *                             **Example:**  <br>
     *                             'col': columns 'col0','col1',etc.<br>
     *                             '': columns 0,1
     *
     * @return void
     */
    public function csvStyle(string $separator = ',', string $enclosure = '"', string $lineEnd = "\n"
        ,                           $header = true, string $prefixColumn = ''): void
    {
        if ($this->srvSerialize instanceof DocumentStoreOneCsv) {
            $this->srvSerialize->csvSeparator = $separator;
            $this->srvSerialize->csvText = $enclosure;
            $this->srvSerialize->csvLineEnd = $lineEnd;
            $this->srvSerialize->csvHeader = $header;
            $this->srvSerialize->csvPrefixColumn = $prefixColumn;
        }
    }

    /**
     * It sets the regional style
     *
     * @param string $regionalDecimal
     * @param string $regionalDate
     * @param string $regionalDateTime
     * @return void
     */
    public function regionalStyle(string $regionalDecimal = '.', string $regionalDate = 'Y-m-d'
        , string                         $regionalDateTime = 'Y-m-d H:i:s'): void
    {
        $this->regionDecimal = $regionalDecimal;
        $this->regionDate = $regionalDate;
        $this->regionDateTime = $regionalDateTime;
    }

    /**
     * List all the Ids in a collection (or returns the documents if $returnOnlyIndex is false)
     *
     * @param string $mask see http://php.net/manual/en/function.glob.php
     *
     * @param bool   $returnOnlyIndex
     *                     If false then it returns each document.
     *                     If returns (default) then it returns indexes.
     *
     * @return array
     * @throws RedisException
     */
    public function select(string $mask = "*", bool $returnOnlyIndex = true): array
    {
        $list = glob($this->database . "/" . $this->collection . "/" . $mask . $this->docExt);
        foreach ($list as $key => $fileId) {
            $list[$key] = basename($fileId, $this->docExt);
        }
        if ($returnOnlyIndex) {
            return $list;
        }
        $listDoc = [];
        foreach ($list as $fileId) {
            $listDoc[] = $this->get($fileId);
        }
        return $listDoc;
    }

    /**
     * Get a document from the datastore<br>
     * **Example:**  <br>
     * ```
     * $data=$this->get('rows',-1,false); // it returns a value or false
     * $data=$this->get('rows',-1,"not null"); // it returns a value or "not null" if not found
     * ```
     *
     * @param string $id      "ID" of the document.
     * @param int    $tries   number of tries.<br>
     *                        The default value is -1 (it uses the default value $defaultNumRetry)<br>
     *                        0 means it will not try to lock<br>
     * @param mixed  $default Default value (if the value is not found)
     *
     * @return mixed The object if the information was read, otherwise false (or default value).
     * @throws RedisException
     * @throws RedisException
     */
    public function get(string $id, int $tries = -1, $default = false)
    {
        $file = $this->filename($id);
        $this->currentId = $id;
        if ($this->lock($file, $tries)) {
            if ($this->serializeStrategy === 'php_array') {
                $json = @include $file;
                $this->unlock($file);
            } else {
                $json = @file_get_contents($file);
                $this->unlock($file, $tries);
                if ($json !== false) {
                    if (strpos($json, $this->separatorAppend) === false) {
                        if ($this->autoSerialize) {
                            $json = $this->deserialize($json);
                        }
                    } else {
                        $arr = explode($this->separatorAppend, $json);
                        if (count($arr) > 0 && $arr[0] === '') {
                            unset($arr[0]);
                        }
                        $json = [];
                        foreach ($arr as $item) {
                            $json[] = $this->deserialize($item);
                        }
                    }
                }
            }
            if ($json === false) {
                $this->throwError(error_get_last());
            }
            $this->resetChain();
            return ($json === false) ? $default : $json;
        }
        $this->resetChain();
        return $default;
    }

    /**
     * Set if you don't want to throw an exception when error.<br>
     * This value is reset every time the value is read or an exception is throw.<br>
     * However, the error message lastError() is still stored.<br>
     * **Example:**  <br>
     * ```
     * $this->noThrowOnError()->get('id'); // it will not throw an exception if "ID" document does not exist.
     * ```
     * @param bool $throw if false (default), then it doesn't throw an exception on error.
     * @return $this
     */
    public function noThrowOnError(bool $throw = false): DocumentStoreOne
    {
        $this->throwable = $throw;
        return $this;
    }

    protected function deserialize($document)
    {
        return $this->srvSerialize->deserialize($document);
    }

    /**
     * It gets the timestamp of a document or false in case of error.<br>
     * **Example**
     * ```
     * $timeStamp=$this->getTimeStamp("doc20");
     * ```
     *
     * @param string $id                   "ID" of the document.
     * @param bool   $returnAsAbsoluteTime if true then it returns the age (how many seconds are elapsed)<br>
     *                                     if false then it returns the regular timestamp of the time.
     *
     * @return false|int
     */
    public function getTimeStamp(string $id, bool $returnAsAbsoluteTime = false)
    {
        $this->currentId = $id;
        $file = $this->filename($id);
        try {
            $rt = @filemtime($file);
        } catch (Exception $ex) {
            $rt = false;
        }
        if ($returnAsAbsoluteTime) {
            if ($rt === false) {
                return false;
            }
            return time() - $rt;
        }
        return $rt;
    }

    /**
     * It sets the modification time of a document<br>
     * **Example**
     * ```
     * $timeStamp=$this->setTimeStamp("doc20",12323232); // set the absolute timestemp
     * $timeStamp=$this->setTimeStamp("doc20",100,false); // set the relative timestamp (relative current time)
     * ```
     * @param string $id                   "ID" of the document
     * @param int    $ttl                  the interval (in seconds).
     * @param bool   $setTTLAsAbsoluteTime if true then it sets the age ($ttl+time())<br>
     *                                     if false then it sets the regular timestamp of the time.
     * @return bool
     */
    public function setTimeStamp(string $id, int $ttl, bool $setTTLAsAbsoluteTime = true): bool
    {
        $time = $setTTLAsAbsoluteTime ? $ttl : time() + $ttl;
        try {
            return @touch($this->filename($id), $time);
        } catch (Exception $ex) {
            return false;
        }
    }

    /**
     * It gets a values from the datastore filtered by a condition<br>
     * **Example:**
     * ```
     * $data=$this->getFiltered('rows',-1,false,['type'=>'busy']); // it returns values [0=>...,1=>...]
     * $data=$this->getFiltered('rows',-1,false,['type'=>'busy'],false); // it returns values [2=>...,4=>..]
     * ```
     *
     * @param string $id        "ID" of the document.
     * @param int    $tries     number of tries. The default value is -1 (it uses the default value $defaultNumRetry)
     * @param mixed  $default   default value (if the value is not found)
     * @param array  $condition An associative array with the conditions.
     * @param bool   $reindex   If true then the result is reindexed (starting from zero).
     *
     * @return array
     * @noinspection TypeUnsafeComparisonInspection
     * @throws RedisException
     */
    public function getFiltered(string $id, int $tries = -1, $default = false, array $condition = [], bool $reindex = true): array
    {
        $rows = $this->get($id, $tries, $default);
        $result = [];
        foreach ($rows as $k => $v) {
            $fail = false;
            foreach ($condition as $k2 => $v2) {
                if (is_object($v)) {
                    if (!isset($v->{$k2}) || $v->{$k2} != $v2) {
                        $fail = true;
                        break;
                    }
                } elseif (!isset($v[$k2]) || $v[$k2] != $v2) {
                    $fail = true;
                    break;
                }
            }
            if (!$fail) {
                if ($reindex) {
                    $result[] = $v;
                } else {
                    $result[$k] = $v;
                }
            }
        }
        return $result;
    }

    /**
     * Delete a document.<br>
     * **Example:**
     * ```
     * $isDeleted=$this->delete('doc20');
     * ```
     *
     * @param string $id    "ID" of the document
     * @param int    $tries number of tries. The default value is -1 (it uses the default value $defaultNumRetry)
     *
     * @return bool if it's unable to unlock or the document doesn't exist.
     */
    public function delete(string $id, int $tries = -1, $throwOnError = false): bool
    {
        $file = $this->filename($id);
        if ($this->lock($file, $tries)) {
            $r = @unlink($file);
            $this->unlock($file);
            if ($r === false && $throwOnError) {
                $this->throwError(error_get_last());
            }
            $this->resetChain();
            return $r;
        }
        $this->resetChain();
        return false;
    }

    /**
     * Copy a document. If the destination exists then it's replaced.<br>
     * **Example:**
     * ```
     * $isCopied=$this->copy('doc20','doc20copy');
     * ```
     * @param string $idOrigin
     * @param string $idDestination
     * @param int    $tries number of tries. The default value is -1 (it uses the default value $defaultNumRetry)
     *
     * @return bool true if the operation is correct, otherwise it returns false (unable to lock / unable to copy)
     */
    public function copy(string $idOrigin, string $idDestination, int $tries = -1): bool
    {
        $fileOrigin = $this->filename($idOrigin);
        $fileDestination = $this->filename($idDestination);
        if ($this->lock($fileOrigin, $tries)) {
            if ($this->lock($fileDestination, $tries)) {
                $r = @copy($fileOrigin, $fileDestination);
                $this->unlock($fileOrigin);
                $this->unlock($fileDestination);
                if ($r === false) {
                    $this->throwError(error_get_last());
                }
                $this->resetChain();
                return $r;
            }
            $this->unlock($fileOrigin);
            $this->throwError("Unable to lock file [$idDestination]");
            return false;
        }
        $this->throwError("Unable to lock file [$idOrigin]");
        return false;
    }

    /**
     * Rename a document. If the destination exists then it's not renamed<br>
     * **Example:**
     * ```
     * $isRenamed=$this->rename('doc20','doc20newName');
     * ```
     *
     * @param string $idOrigin
     * @param string $idDestination
     * @param int    $tries number of tries. The default value is -1 (it uses the default value $defaultNumRetry)
     *
     * @return bool true if the operation is correct, otherwise it returns false
     *              (unable to lock / unable to rename)
     */
    public function rename(string $idOrigin, string $idDestination, int $tries = -1): bool
    {
        $fileOrigin = $this->filename($idOrigin);
        $fileDestination = $this->filename($idDestination);
        if ($this->lock($fileOrigin, $tries)) {
            if ($this->lock($fileDestination, $tries)) {
                $r = @rename($fileOrigin, $fileDestination);
                $this->unlock($fileOrigin);
                $this->unlock($fileDestination);
                if ($r === false) {
                    $this->throwError(error_get_last());
                }
                $this->resetChain();
                return $r;
            }
            $this->unlock($fileOrigin);
            $this->resetChain();
            return false;
        }
        $this->resetChain();
        return false;
    }
}

