package de.cxp.ocs.controller;

import java.util.concurrent.ExecutionException;

import org.slf4j.MDC;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;

import de.cxp.ocs.api.indexer.ImportSession;
import de.cxp.ocs.indexer.AbstractIndexer;
import de.cxp.ocs.model.index.BulkImportData;
import lombok.extern.slf4j.Slf4j;

@RestController
@RequestMapping(path = "/indexer-api/v1/full")
@Slf4j
/**
 * Controller that implements the FullIndexationService
 */
// unfortunately it's not possible to use 'implements FullIndexationService'
// properly, because the http-code statuses need ResponseEntity as return type
// in Spring Boot.
public class FullIndexationController {

	@Autowired
	private IndexerCache indexerManager;

	@GetMapping("/start/{indexName}")
	public ResponseEntity<?> startImport(@PathVariable("indexName") String indexName, @RequestParam("locale") String locale) {
		if (indexName == null || indexName.isEmpty()) return ResponseEntity.status(HttpStatus.BAD_REQUEST).build();
		if (locale == null || locale.isEmpty()) return ResponseEntity.status(HttpStatus.BAD_REQUEST).build();
		MDC.put("index", indexName);
		try {
			return ResponseEntity.ok(indexerManager.getIndexer(indexName).startImport(indexName, locale));
		}
		catch (IllegalArgumentException argEx) {
			return ResponseEntity.status(HttpStatus.BAD_REQUEST).body(argEx.getMessage());
		}
		catch (IllegalStateException ise) {
			return ResponseEntity.status(HttpStatus.CONFLICT).body(ise.getMessage());
		}
		catch (ExecutionException e) {
			return ResponseEntity.status(HttpStatus.INTERNAL_SERVER_ERROR).build();
		}
		finally {
			MDC.remove("index");
		}
	}

	/**
	 * Add products to import into current session.
	 * 
	 * @param data
	 *        {@link BulkImportData} that contains the ImportSession that was
	 *        created at the start of the import plus one or more documents to
	 *        be indexed
	 * @throws Exception
	 */
	@PostMapping("/add")
	public ResponseEntity<Integer> add(@RequestBody BulkImportData data) throws Exception {
		if (data.session == null) return ResponseEntity.status(HttpStatus.BAD_REQUEST).body(0);
		if (data.documents == null || data.documents.length == 0) {
			return ResponseEntity.status(HttpStatus.BAD_REQUEST).body(0);
		}

		MDC.put("index", data.session.finalIndexName);
		try {
			AbstractIndexer indexer = indexerManager.getIndexer(data.getSession().getFinalIndexName());
			if (!indexer.isImportRunning(data.session.temporaryIndexName)) {
				log.warn("Tried to add documents int an index that is not expecting bulk imports: {}", data.session.temporaryIndexName);
				return ResponseEntity.status(HttpStatus.BAD_REQUEST).body(0);
			}
			int successCount = indexer.add(data);
			return ResponseEntity.ok().body(successCount);
		}
		finally {
			MDC.remove("index");
		}
	}

	@PostMapping("/done")
	public ResponseEntity<Boolean> done(@RequestBody ImportSession session) throws Exception {
		AbstractIndexer indexer = indexerManager.getIndexer(session.getFinalIndexName());
		if (!indexer.isImportRunning(session.temporaryIndexName)) {
			log.warn("Called 'done' for an index that is not expecting bulk imports: {}", session.temporaryIndexName);
			return ResponseEntity.status(HttpStatus.BAD_REQUEST).build();
		}
		MDC.put("index", session.finalIndexName);
		try {
			boolean ok = indexer.done(session);
			return ok ? ResponseEntity.ok(true) : ResponseEntity.status(HttpStatus.BAD_REQUEST).body(false);
		}
		catch (IllegalArgumentException iae) {
			return ResponseEntity.status(HttpStatus.BAD_REQUEST).body(false);
		}
		finally {
			MDC.remove("index");
		}
	}

	@PostMapping("/cancel")
	public ResponseEntity<Void> cancel(@RequestBody ImportSession session) {
		MDC.put("index", session.finalIndexName);
		try {
			AbstractIndexer indexer = indexerManager.getIndexer(session.getFinalIndexName());
			if (!indexer.isImportRunning(session.temporaryIndexName)) {
				log.warn("Called 'cancel' for an index that is not expecting bulk imports: {}", session.temporaryIndexName);
				return ResponseEntity.status(HttpStatus.BAD_REQUEST).build();
			}
			indexer.cancel(session);
			return ResponseEntity.accepted().build();
		}
		catch (ExecutionException e) {
			log.error("exception while canceling import: ", e);
			return ResponseEntity.status(HttpStatus.INTERNAL_SERVER_ERROR).build();
		}
		finally {
			MDC.remove("index");
		}
	}

}
