/*jshint browser:true, -W100:true, evil:true */

(function(window, undefined) {

var E= '', SP= ' ', EA= [], N= 'number', S= 'string', O= 'object',
  A= 'array', B= 'boolean', R= 'regexp', F= 'function', BN = '\n';

function makeClass() { // from: Resig, TODO: make work with strict
  return function(args) {
    if (this instanceof arguments.callee) {
      if (typeof this.init == "function") {
        this.init.apply(this, args && args.callee ? args : arguments);
      }
    } else return new arguments.callee(arguments);
  };
}

// Returns true if the object is of type 'type', otherwise false
function is(obj, type) {
  return get(obj) === type;
}

// Throws TypeError if not the correct type, else returns true
function ok(obj, type) {
  if (get(obj) != type) {
    throw TypeError('Expected ' + (type ? type.toUpperCase() : type + E) +
      ", but received " + (obj ? get(obj).toUpperCase() : obj + E));
  }
  return true;
}

function isNum(n) {
  return !isNaN(parseFloat(n)) && isFinite(n);
}

function okeys(obj) {
  var keys = []; // replaces Object.keys();
  for (var k in obj) keys.push(k);
  return keys;
}

function err() {
  var msg = "[RiTa] " + arguments[0];
  for (var i = 1; i < arguments.length; i++)
    msg += '\n' + arguments[i];
  throw Error(msg);
}

function warn() {
  if (RiTa.SILENT || !console) return;
  if (arguments && arguments.length) {
    console.warn("[WARN] " + arguments[0]);
    for (var i = 1; i < arguments.length; i++)
      console.warn(arguments[i]);
  }
}

function log() {
  if (RiTa.SILENT || !console) return;
  console.log.apply(console, arguments);
}

function strOk(str) {
  return (typeof str === S && str.length > 0);
}

function trim(str) {
  if (!strOk(str)) return str;
  // from: http://blog.stevenlevithan.com/archives/faster-trim-javascript
  return str.replace(/^\s\s*/, '').replace(/\s\s*$/, '');
}

function last(word) { // last char of string
  if (!word || !word.length) return E;
  return word.charAt(word.length - 1);
}

function extend(l1, l2) { // python extend
  for (var i = 0; i < l2.length; i++)
    l1.push(l2[i]);
}

function endsWith(str, ending) {
  if (!is(str, S)) return false;
  return new RegExp(ending + '$').test(str);
}

function startsWith(str, starting) {
  if (!is(str, S)) return false;
  return new RegExp('^' + starting).test(str);
}

function equalsIgnoreCase(str1, str2) {
  return (is(str1, S) && is(str2, S)) ?
    (str1.toLowerCase() === str2.toLowerCase()) : false;
}

// Returns true if NodeJS is the current environment
function isNode() {
  return (typeof module != 'undefined' && module.exports);
}

function shuffle(oldArray) { // shuffle array
  var newArray = oldArray.slice(),
    len = newArray.length,
    i = len;
  while (i--) {
    var p = parseInt(Math.random() * len),
      t = newArray[i];
    newArray[i] = newArray[p];
    newArray[p] = t;
  }
  return newArray;
}

function inArray(array, val) {
  return (!array) ? false : array.indexOf(val) > -1;
}

function escapeRegExp(string) {
  return string.replace(/([.*+?^=!:${}()|\[\]\/\\])/g, "\\$1");
}

// From: http://javascriptweblog.wordpress.com/2011/08/08/fixing-the-javascript-typeof-operator/
function get(obj) {
  if (typeof obj != 'undefined') // else return undef
    return ({}).toString.call(obj).match(/\s([a-zA-Z]+)/)[1].toLowerCase();
}

'use strict';

var RiLexicon = makeClass(); // stub

RiLexicon.enabled = false;

RiLexicon.prototype.init = function() {
    throw Error('RiLexicon is not available -- ' +
      'if needed, make sure to include rilexicon.js');
};

var RiTa = {

  VERSION: '1.1.40',

  LEXICON: null, // static RiLexicon instance

  /* For tokenization, Can't -> Can not, etc. */
  SPLIT_CONTRACTIONS: false,

  JAVA: 1, JS: 2, NODE: 3,

  _FEATURES: [ 'tokens', 'stresses', 'phonemes', 'syllables', 'pos', 'text' ],

  // For Conjugator =================================

  FIRST_PERSON: 1,
  SECOND_PERSON: 2,
  THIRD_PERSON: 3,
  PAST_TENSE: 4,
  PRESENT_TENSE: 5,
  FUTURE_TENSE: 6,
  SINGULAR: 7,
  PLURAL: 8,
  NORMAL: 9,
  FEATURE_DELIM: ':',
  STRESSED: '1',
  UNSTRESSED: '0',
  PHONEME_BOUNDARY: '-',
  WORD_BOUNDARY: " ",
  SYLLABLE_BOUNDARY: "/",
  SENTENCE_BOUNDARY: "|",
  VOWELS: "aeiou",
  ABBREVIATIONS: ["Adm.", "Capt.", "Cmdr.", "Col.", "Dr.", "Gen.", "Gov.", "Lt.", "Maj.", "Messrs.", "Mr.", "Mrs.", "Ms.", "Prof.", "Rep.", "Reps.", "Rev.", "Sen.", "Sens.", "Sgt.", "Sr.", "St.", "a.k.a.", "c.f.", "i.e.", "e.g.", "vs.", "v.", "Jan.", "Feb.", "Mar.", "Apr.", "Mar.", "Jun.", "Jul.", "Aug.", "Sept.", "Oct.", "Nov.", "Dec."],
  ALL_PHONES: ['aa','ae','ah','ao','aw','ay','b','ch','d','dh','eh','er','ey','f','g','hh','ih','iy','jh', 'k','l', 'm','n','ng','ow','oy','p','r','s','sh','t','th','uh', 'uw','v','w','y','z','zh'],

  /* The infinitive verb form  - 'to eat an apple' */
  INFINITIVE: 1,

  /* Gerund form of a verb  - 'eating an apple' */
  GERUND: 2,

  /* The imperative verb form - 'eat an apple!' */
  IMPERATIVE: 3,

  /* Bare infinitive verb form - 'eat an apple' */
  BARE_INFINITIVE: 4,

  /* The subjunctive verb form */
  SUBJUNCTIVE: 5,

  /* Set to true to disable all console output */
  SILENT: false,

  /* Stemmer type: Lancaster */
  LANCASTER: "Lancaster",

  /* Stemmer type: Porter */
  PORTER: "Porter",

  /* Stemmer type: Pling */
  PLING: "Pling",

  DATA_LOADED: 'DataLoaded',
  INTERNAL: 'Internal',
  UNKNOWN: 'Unknown',

  NON_BREAKING_SPACE: '<sp/>',
	PARAGRAPH_BREAK: '<p/>',  //   regex: /<p\/?>/g;
	LINE_BREAK: '<br/>',

  STOP_WORDS: [ ".", ",", "the",
    "and", "a", "of", "\"", "in", "i", ":", "you", "is", "to",
    "that", ")", "(", "it", "for", "on", "!", "have", "with", "?",
    "this", "be", "...", "not", "are", "as", "was", "but", "or", "from",
    "my", "at", "if", "they", "your", "all", "he", "by", "one",
    "me", "what", "so", "can", "will", "do", "an", "about", "we", "just",
    "would", "there", "no", "like", "out", "his", "has", "up", "more", "who",
    "when", "don't", "some", "had", "them", "any", "their", "it's", "only",
    ";", "which", "i'm", "been", "other", "were", "how", "then", "now",
    "her", "than", "she", "well", "also", "us", "very", "because",
    "am", "here", "could", "even", "him", "into", "our", "much",
    "too", "did", "should", "over", "want", "these", "may", "where", "most",
    "many", "those", "does", "why", "please", "off", "going", "its", "i've",
    "down", "that's", "can't", "you're", "didn't", "another", "around",
    "must",  "few", "doesn't", "every", "yes", "each", "maybe",
    "i'll", "away", "doing", "oh", "else", "isn't", "he's", "there's", "hi",
    "won't", "ok", "they're", "yeah", "mine", "we're", "what's", "shall",
    "she's", "hello", "okay", "here's", "-", "less"
  ],

  // Start functions =================================

  _lexicon: function() {

    if (!RiTa.LEXICON) {
      RiTa.LEXICON = new RiLexicon();
    }
    return RiTa.LEXICON;
  },

  untokenize: function(arr, delim, adjustPunctuationSpacing) {

    delim = delim || SP;
    adjustPunctuationSpacing = adjustPunctuationSpacing || 1;

    var dbug = 0;
    var punct = /^[,\.\;\:\?\!\)""“”’‘`']+$/;
    var quotes = /^[\(""“”’‘`']+$/;

    if (adjustPunctuationSpacing) {

      var newStr = arr[0] || E;
      var inMiddleOfSentence = false;
      var quotationStarted;
      var quotationJustFinished = false;

      if (arr[0])
        quotationStarted = quotes.test(arr[0]);
      else
        quotationStarted = false;

      for (var i = 1; i < arr.length; i++) {

        if (arr[i]) {

          var thisPunct = punct.test(arr[i]);
          var lastPunct = punct.test(arr[i - 1]);
          var thisQuote = quotes.test(arr[i]);
          var lastQuote = quotes.test(arr[i -1]);
          var thisComma = arr[i].match(/,/);
          var lastComma = arr[i - 1].match(/,/);

          if (dbug) {
            console.log(i+") CHECK: "+arr[i]+" "+arr[i-1]+ " "+thisPunct+" "+lastPunct + " " +thisQuote);
          }

          if (quotationStarted && thisQuote) {
            // skip adding delim and mark qutation as ended
            quotationJustFinished = true;
            quotationStarted = false;
          } else if (quotationJustFinished) {
            newStr += delim;
            quotationJustFinished = false;
          } else if (lastQuote && thisComma) {
            inMiddleOfSentence = true;
          } else if (inMiddleOfSentence && lastComma) {
            newStr += delim;
            inMiddleofSentence = false;
          } else if (i != arr.length - 1 && thisPunct && lastPunct) {
            if (dbug) console.log(i + ") HIT1: " + arr[i]);
            newStr += delim;
          } else if (!thisPunct && !lastQuote) {
            if (dbug) console.log(i+") HIT2: "+arr[i]+" "+arr[i-1]+ " "+thisPunct+" "+lastQuote);
            newStr += delim;
          } else {
            if (dbug) console.log(i + ") MISS: " + arr[i]);
          }
          newStr += arr[i];
        }
      }
      return newStr.trim();//.replace(//);
    }

    return arr.join(delim);
    //var punct = /^[,\.\;\:\?\!\)"“”’‘`']+$/;
  },

  random: function() {
    var currentRandom = Math.random();
    if (!arguments.length) return currentRandom;
    return (arguments.length === 1) ? currentRandom * arguments[0] :
      currentRandom * (arguments[1] - arguments[0]) + arguments[0];
  },

  randomItem: function(arr) {
    return arr[Math.floor(Math.random() * arr.length)];
  },

  distance: function(x1, y1, x2, y2) {
    var dx = x1 - x2, dy = y1 - y2;
    return Math.sqrt(dx * dx + dy * dy);
  },

  _tagForPENN: function(words) {
    if (!words || !words.length) return EA;
    var arr = is(words, S) ? RiTa.tokenize(words) : words;
    return PosTagger.tag(arr);
  },

  _tagForWordNet: function(words) {

    var pos, posArr = RiTa._tagForPENN(words);
    if (words && posArr.length) {

      for (var i = 0; i < posArr.length; i++) {
        pos = posArr[i];
        posArr[i] = '-'; // default=other
        if (PosTagger.isNoun(pos)) posArr[i] = 'n';
        else if (PosTagger.isVerb(pos)) posArr[i] = 'v';
        else if (PosTagger.isAdverb(pos)) posArr[i] = 'r';
        else if (PosTagger.isAdj(pos)) posArr[i] = 'a';
      }
      return posArr;
    }
    return EA;
  },

  getPosTags: function(words, useWordNetTags) {
    return (useWordNetTags) ? RiTa._tagForWordNet(words) : RiTa._tagForPENN(words);
  },

  getPosTagsInline: function(words, delimiter) {

    if (!words || !words.length) return E;

    delimiter = delimiter || '/';
    words = is(words, S) ? RiTa.tokenize(words) : words;

    var sb = E,
      tags = RiTa.getPosTags(words);
    for (var i = 0; i < words.length; i++) {

      sb += (words[i]);
      if (!RiTa.isPunctuation(words[i])) {
        sb += delimiter + tags[i];
      }
      sb += SP;
    }

    return sb.trim();
  },

  getPresentParticiple: function(verb) {
    return Conjugator().getPresentParticiple(verb);
  },

  getPastParticiple: function(verb) {
    return Conjugator().getPastParticiple(verb);
  },

  // TODO: 2 examples
  concordance: function(text, options) {
    return Concorder(text, options).concordance();
  },

  // TODO: 2 examples (add cache)
  kwic: function(text, word, options) {
    wordCount = (options && options.wordCount) || 4;
    return Concorder(text, options).kwic(word, wordCount);
  },

  // TODO: 2 examples
  conjugate: function(verb, args) {
    return Conjugator().conjugate(verb, args);
  },

  upperCaseFirst: function(s) {
    return s.charAt(0).toUpperCase() + s.substring(1);
  },

  pluralize: function(word) {

    if (!strOk(word)) return E;

    var i, rule, rules = PLURAL_RULES;

    if (inArray(MODALS, word.toLowerCase())) {
      return word;
    }

    for (i = 0; i < rules.length; i++) {
      rule = rules[i];
      if (rule.applies(word.toLowerCase())) {
        return rule.fire(word);
      }
    }

    return DEFAULT_PLURAL_RULE.fire(word);
  },

  singularize: function(word) {

    if (!strOk(word)) return E;

    var i, rule, rules = SINGULAR_RULES;
    if (inArray(MODALS, word.toLowerCase())) {
      return word;
    }
    i = rules.length;
    while (i--) {
      rule = rules[i];
      if (rule.applies(word.toLowerCase())) {
        return rule.fire(word);
      }
    }
    ////////////////////////////////////////

    return this.stem(word, 'Pling');
  },

  trim: function(str) {
    return trim(str); // delegate to private
  },

  tokenize: function(words, regex) {

    if (!is(words,S)) return [];

    if (regex) return words.split(regex);

    words = trim(words);

    words = words.replace(/([\\?!\"\\.,;:@#$%&])/g, " $1 ");
    words = words.replace(/\\.\\.\\./g, " ... ");
    words = words.replace(/\\s+/g, SP);
    words = words.replace(/,([^0-9])/g, " , $1");
    words = words.replace(/([^.])([.])([\])}>\"']*)\\s*$/g, "$1 $2$3 ");
    words = words.replace(/([\[\](){}<>])/g, " $1 ");
    words = words.replace(/--/g, " -- ");
    words = words.replace(/$/g, SP);
    words = words.replace(/^/g, SP);
    words = words.replace(/([^'])' /g, "$1 ' ");
    words = words.replace(/'([SMD]) /g, " '$1 ");

    if (RiTa.SPLIT_CONTRACTIONS) {

      words = words.replace(/([Cc])an't/g, "$1an not");
      words = words.replace(/([Dd])idn't/g, "$1id not");
      words = words.replace(/([CcWw])ouldn't/g, "$1ould not");
      words = words.replace(/([Ss])houldn't/g, "$1hould not");
      words = words.replace(/ ([Ii])t's/g, " $1t is");
      words = words.replace(/n't /g, " not ");
      words = words.replace(/'ve /g, " have ");
      words = words.replace(/'re /g, " are ");
    }

    // "Nicole I. Kidman" gets tokenized as "Nicole I . Kidman"
    words = words.replace(/ ([A-Z]) \\./g, " $1. ");
    words = words.replace(/\\s+/g, SP);
    words = words.replace(/^\\s+/g, E);

    return trim(words).split(/\s+/);
  },

  splitSentences: function(text, regex) {

    var arr = text.match(/(\S.+?[.!?])(?=\s+|$)/g);
    return (text.length && arr && arr.length) ? arr : [text];
  },

  isAbbreviation: function(input, caseSensitive) {

    caseSensitive = caseSensitive || false;
    input = caseSensitive ? input : RiTa._titleCase(input);
    return inArray(this.ABBREVIATIONS, input);
  },

  _loadStringsNode: function(url, callback) {

    var data = '', isUrl = /.+?:\/\/.+/.test(url), me = this;

    function processResponse(data) {
      data = data.toString('utf-8').trim();
      var lines = data.split(/(\r\n|\n)/gm);
      me.fireDataLoaded(url, callback, lines);
    }
    //log("Using Node for: "+url +" isUrl="+isUrl);

    if (isUrl) {

      var httpcb = function(response) {
        response.on('data', function(chunk) {
          data += chunk;
        });
        response.on('error', function(e) {
          throw e;
        });
        response.on('end', function() {
          processResponse(data);
        });
      };

      var req = require('http').request(url, httpcb);
      req.on('socket', function(socket) { // shouldnt be needed

        socket.setTimeout(5000); // ?
        socket.on('timeout', function() {
          req.abort();
          throw Error("[RiTa] loadString timed-out and aborted request");
        });
      });
      req.end();

    } else {

      // try with node file-system
      var rq = require('fs');
      rq.readFile(url, function(e, data) {
        if (e || !data) {
          err("[Node] Error reading file: " + url + BN + e);
          throw e;
        }
        processResponse(data);
      });
    }
    //return data;
  },

  _loadStringNode: function(url, callback, linebreakChars) {

    var data = '', lb = linebreakChars || BN,
      isUrl = /.+?:\/\/.+/.test(url), me = this;

    //log("Using Node for: "+url +" isUrl="+isUrl);

    if (isUrl) {

      var httpcb = function(response) {
        response.on('data', function(chunk) {
          data += chunk;
        });
        response.on('error', function(e) {
          throw e;
        });
        response.on('end', function() {
          data = data.toString('utf-8').replace(/[\r\n]+/g, lb).trim();
          me.fireDataLoaded(url, callback, data);
        });
      };

      var req = require('http').request(url, httpcb);
      req.on('socket', function(socket) { // shouldnt be needed

        socket.setTimeout(5000); // ?
        socket.on('timeout', function() {
          req.abort();
          throw Error("[RiTa] loadString timed-out and aborted request");
        });
      });
      req.end();

    } else {

      // try with node file-system
      var rq = require('fs');
      rq.readFile(url, function(e, data) {
        if (e || !data) {
          err("[Node] Error reading file: " + url + BN + e);
          throw e;
        }
        data = data.toString('utf-8').replace(/[\r\n]+/g, lb).trim();
        me.fireDataLoaded(url, callback, data);
      });
    }
    //return data;
  },

  loadString: function(url, callback, linebreakChars) {

    ok(url, S);

    if (isNode()) {
      return this._loadStringNode.apply(this, arguments);
    }

    var me = this, res = '', lb = linebreakChars || BN,
      req = new XMLHttpRequest();

    req.addEventListener('error', function () {
      console.error('[RiTa] loadStrings() unable to load ' + url);
    });

    req.open('GET', url, true);
    req.onreadystatechange = function () {
      if (req.readyState === 4) {
        if (req.status === 200) {
          var ret = [];
          // remove blank lines
          var arr = req.responseText.match(/[^\r\n]+/g);
          for (var k in arr) {
            ret[k] = arr[k];
          }
          ret = ret.join(BN);
          me.fireDataLoaded(url, callback, ret);
          //callback(res);
        } else {
          console.error('[RiTa] loadString() unable to load: ' + url);
        }
      }
    };
    req.send(null);
    return res;
  },

  // TODO: update 'return' value in docs (for preload())
  loadStrings: function(url, callback, linebreakChars) {

    ok(url, S);

    if (isNode()) {
      return this._loadStringsNode.apply(this, arguments);
    }

    var me = this, res = '', lb = linebreakChars || BN,
      req = new XMLHttpRequest();

    req.addEventListener('error', function () {
      console.error('[RiTa] loadStrings() unable to load ' + url);
    });

    req.open('GET', url, true);
    req.onreadystatechange = function () {
      if (req.readyState === 4) {
        if (req.status === 200) {

          // remove blank lines
          var arr = req.responseText.match(/[^\r\n]+/g);
          var ret = [];
          for (var k in arr) {
            ret[k] = arr[k];
          }
          me.fireDataLoaded(url, callback, ret);
          //callback(res);
        } else {
          console.error('[RiTa] loadString() unable to load: ' + url);
        }
      }
    };
    req.send(null);
    return res;
  },

  fireDataLoaded: function(url, callback, data) {

    //log('fireDataLoaded: '+url, callback, data);
    return (callback) ? callback(data, url) :
      RiTaEvent({
        name: 'RiTaLoader',
        urls: is(url, S) ? [ url ] : url
      }, RiTa.DATA_LOADED, data)._fire();
  },

  isQuestion: function(sentence) {

    var sentenceArr = RiTa.tokenize(sentence);

    for (var i = 0; i < QUESTION_STARTS.length; i++) {
  console.log(sentenceArr[0], QUESTION_STARTS[i]);
      if (equalsIgnoreCase(sentenceArr[0], QUESTION_STARTS[i]))
        return true;
    }
    return false;
  },

  isSentenceEnd: function(currentWord, nextWord) {

    if (!is(currentWord, S) || !is(nextWord, S))
      return false;

    var cw = currentWord.charAt(0),
      cWL = currentWord.length;

    // token is a mid-sentence abbreviation (mainly, titles) --> middle of sent
    if (RiTa.isAbbreviation(currentWord))
      return false;

    if (cWL > 1 && cw.indexOf("`'\"([{<") != -1 && RiTa.isAbbreviation(currentWord.substring(1)))
      return false;

    if (cWL > 2 && ((currentWord.charAt(0) == '\'' && currentWord.charAt(1) == '\'') ||
      (currentWord.charAt(0) == '`' && currentWord.charAt(1) == '`')) &&
      RiTa.isAbbreviation(currentWord.substring(2)))
    {
      return false;
    }

    var nTL = nextWord.length,
      currentToken0 = currentWord.charAt(cWL - 1),
      currentToken1 = (cWL > 1) ? currentWord.charAt(cWL - 2) : SP,
      currentToken2 = (cWL > 2) ? currentWord.charAt(cWL - 3) : SP,
      nextToken0 = nextWord.charAt(0),
      nextToken1 = (nTL > 1) ? nextWord.charAt(1) : SP,
      nextToken2 = (nTL > 2) ? nextWord.charAt(2) : SP;

    // nextToken does not begin with an upper case,
    // [`'"([{<] + upper case, `` + upper case, or < -> middle of sent.
    if (!(nextToken0 == nextToken0.toUpperCase() ||
        (nextToken1 == nextToken1.toUpperCase() && nextToken0.indexOf("`'\"([{<") != -1) ||
        (nextToken2 == nextToken2.toUpperCase() && ((nextToken0 == '`' && nextToken1 == '`') ||
          (nextToken0 == '\'' && nextToken1 == '\''))) ||
        nextWord == "_" || nextToken0 == '<'))
      return false;

    // ends with ?, !, [!?.]["'}>)], or [?!.]'' -> end of sentence
    if (currentToken0 == '?' || currentToken0 == '!' ||
      (currentToken1.indexOf("?!.") != -1 && currentToken0.indexOf("\"'}>)") != -1) ||
      (currentToken2.indexOf("?!.") != -1 && currentToken1 == '\'' && currentToken0 == '\''))
      return true;

    // last char not "." -> middle of sentence
    if (currentToken0 != '.') return false;

    // Note: wont handle Q. / A. at start of sentence, as in a news wire
    //if (startOfSentence && (currentWord.equalsIgnoreCase("Q.")
    //|| currentWord.equalsIgnoreCase("A.")))return true;

    // single upper-case alpha + "." -> middle of sentence
    if (cWL == 2 && currentToken1 == currentToken1.toUpperCase())
      return false;

    // double initial (X.Y.) -> middle of sentence << added for ACE
    if (cWL == 4 && currentToken2 == '.' && (currentToken1 == currentToken1.toUpperCase() && currentWord.charAt(0) == currentWord.charAt(0).toUpperCase()))
      return false;

    return true;
  },

  isW_Question: function(sentence) {
    var sentenceArr = RiTa.tokenize((sentence));
    for (var i = 0; i < W_QUESTION_STARTS.length; i++)
      if (equalsIgnoreCase(sentenceArr[0], W_QUESTION_STARTS[i]))
        return true;
    return false;
  },

  unescapeHTML: function(input) {

    if (!input || !input.length) return input;

    var answer = input.replace(/&lt;/g, "<").replace(/&gt;/g, ">")
      .replace(/&amp;/g, "&").replace(/&quot;/g, "\"");

    String.fromCharCodePoint = function() { // uggh
      var codeunits = [];
      for (var i = 0; i < arguments.length; i++) {
        var c = arguments[i];
        if (arguments[i] < 0x10000) {
          codeunits.push(arguments[i]);
        } else if (arguments[i] < 0x110000) {
          c -= 0x10000;
          codeunits.push((c >> 10 & 0x3FF) + 0xD800);
          codeunits.push((c & 0x3FF) + 0xDC00);
        }
      }
      return String.fromCharCode.apply(String, codeunits);
    };

    answer = answer.replace(/&#(\d+);/g, function(_, n) {
      return String.fromCharCodePoint(parseInt(n, 10));
    }).replace(/&#x([0-9a-f]+);/gi, function(_, n) {
      return String.fromCharCodePoint(parseInt(n, 16));
    });

    return answer;
  },

  randomOrdering: function(num) {

    var o = [];
    if (num) {
      for (var z = 0; z < num; z++) o.push(z);
      // Array shuffle, from http://jsfromhell.com/array/shuffle
      for (var j, x, i = o.length; i; j = parseInt(Math.random() * i),
        x = o[--i], o[i] = o[j], o[j] = x) { /* no-op */ }
    }
    return o;
  },

  // Trims punctuation from each side of token
  //   (doesnt trim whitespace or internal punctuation).
  trimPunctuation: function(text) {

    if (!is(text,S)) return text;

    var s = '[�`~\"\/' + "\\'_\\-[\\]{}()*+!?%&.,\\\\^$|#@<>|+=;:]";
    var regex = new RegExp("^" + s + "+|" + s + "+$", 'g');

    return (text === E) ? E : text.replace(regex,E);
  },

  stripPunctuation: function(text) {
    return (text === E) ? E : text.replace(PUNCTUATION_CLASS,E);
  },

  isPunctuation: function(text) {
    if (!text || !text.length) return false;
    return ONLY_PUNCT.test(text);
  },

  hasPunctuation: function(text) {
    if (!text || !text.length) return false;
    return ONLY_PUNCT.test(text);
  },

  env: function() {
    return isNode() ? RiTa.NODE : RiTa.JS;
  },

  chomp: function(txt) {
    return txt.replace(/\s+$|^\s+/g,E);
  },

  getPhonemes: function(words) {
    return RiString(words).analyze().get(RiTa.PHONEMES);
  },

  getStresses: function(words) {
    return RiString(words).analyze().get(RiTa.STRESSES);
  },

  getSyllables: function(words) {
    return RiString(words).analyze().get(RiTa.SYLLABLES);
  },

  getWordCount: function(words) {
    return RiTa.tokenize(words).length;
  },

  stem: function(word, type) {

    type = type || 'Pling';

    if (type != 'Lancaster' && type != 'Porter' && type != 'Pling')
      err('Bad stemmer type: ' + type);

    var stemImpl = RiTa['stem_' + type];

    if (word.indexOf(SP) < 0) return stemImpl(word);

    // remove non-words && white-space chars
    word = word.replace(/[^\w]/g, SP).replace(/\s+/g, SP);

    var res = [],
      words = word.split(SP);

    for (var i = 0; i < words.length; i++) {

      res.push(stemImpl(words[i]));
    }
    return res.join(SP);
  },

  // Converts 'input' to Titlecase (1st letter upper, rest lower)
  _titleCase: function(input) {

    if (!input || !input.length) return input;
    return input.substring(0, 1).toUpperCase() + input.substring(1);
  },

  /*
   * Takes pair of strings or string-arrays and returns the min-edit distance
   * @param normalized based on max-length if 3rd (optional) parameter is true (default=f).
   */
  minEditDistance: function(a, b, adjusted) {

    var func = adjusted ? MinEditDist.computeAdjusted : MinEditDist.computeRaw;
    return func.call(MinEditDist, a, b);
  },

  _makeClass: function() { // By John Resig (MIT Licensed)

    return function(args) {
      if (this instanceof arguments.callee) {
        if (typeof this.init == "function") {
          this.init.apply(this, args && args.callee ? args : arguments);
        }
      } else {
        return new arguments.callee(arguments);
      }
    };
  }
}; // end RiTa object

// set additional properties/functions on RiTa
for (var i = 0; i < RiTa._FEATURES.length; i++)
  RiTa[RiTa._FEATURES[i].toUpperCase()] = RiTa._FEATURES[i];

// ////////////////////////////////////////////////////////////
// RiMarkov
// ////////////////////////////////////////////////////////////

var RiMarkov = makeClass();

RiMarkov.MAX_GENERATION_ATTEMPTS = 1000;
RiMarkov._SSRE = /"?[A-Z][a-z"',;`-]*/;
RiMarkov._SSDLM = 'D=l1m_';

RiMarkov.prototype = {

  /*
   * Construct a Markov chain (or n-gram) model and set its n-Factor
   */
  init: function(nFactor, recognizeSentences, allowDuplicates) {

    var a = this._initArgs.apply(this, arguments);

    ok(a[0], N);

    this._n = a[0];
    this.pathTrace = [];
    this.sentenceList = [];
    this.sentenceStarts = [];
    this.minSentenceLength = 6;
    this.maxSentenceLength = 35;
    this.maxDuplicatesToSkip = 10000;
    this.root = new TextNode(null, 'ROOT');
    this.isSentenceAware = (a.length > 1 && !a[1]) ? false : true;
    this.allowDuplicates = (a.length > 2 && !a[2]) ? false : true;
    this.printIgnoredText = false;
  },

  _initArgs: function() {

    var a = arguments, t = get(a[0]);
    // recurse, ignore 'this'
    if (a.length && (t === O || t === 'global' || t === 'window'))
      return this._initArgs.apply(this, Array.prototype.slice.call(a, 1));
    return a;
  },

  getProbability: function(data) {

    if (!this.root) err("Model not initd: null root!");

    var tn = is(data, S) ? this.root.lookup(data) : this._findNode(data);

    return (tn) ? tn.probability() : 0;

  },

  getProbabilities: function(path) {

    if (is(path, S)) path = [path];

    if (path.length > this._n) {

      path = path.slice(Math.max(0, path.length - (this._n - 1)), path.length);
    }

    var probs = {},
      tn = this._findNode(path);

    if (!tn) return {};

    var nexts = tn.childNodes();
    for (var i = 0; i < nexts.length; i++) {
      var node = nexts[i];
      if (node) {
        probs[node.token] = node.probability();
      }
    }

    return probs;
  },

  getCompletions: function(pre, post) {

    var tn, result = [],
      node, atest, nexts;

    if (post) { // fill the center

      if (pre.length + post.length > this._n) {

        err('Sum of pre.length && post.length must be < N, was ' + (pre.length + post.length));
      }

      tn = this._findNode(pre);
      if (!tn) return null;

      nexts = tn.childNodes();
      for (var i = 0; i < nexts.length; i++) {

        node = nexts[i];

        atest = pre.slice(0); // clone

        atest.push(node.token);

        for (var j = 0; j < post.length; j++)
          atest.push(post[j]);

        if (this._findNode(atest)) result.push(node.token);
      }
      return result;
    } else { // fill the end

      var hash = this.getProbabilities(pre);
      var keys = okeys(hash);
      return keys.sort(function(a, b) {

        return hash[b] - hash[a];
      });
    }
  },

  generateUntil: function(regex, minLength, maxLength) {

    minLength = minLength || 1;
    maxLength = maxLength || Number.MAX_VALUE;

    var mn, tokens, tries = 0,
      maxTries = 999;

    OUT: while (++tries < maxTries) {

      // generate the min number of tokens
      tokens = this.generateTokens(minLength);

      // keep adding one and checking until we pass the max
      while (tokens.length < maxLength) {

        mn = this._nextNodeForArr(tokens);

        if (!mn || !mn.token)
          continue OUT; // fail, restart

        tokens.push(mn.token);

        // check against our regex
        if (mn.token.search(regex) > -1)
          return tokens;
      }
    }

    // uh-oh, we failed
    if (tries >= maxTries)
      err(BN+"RiMarkov failed to complete after " + tries + " attempts." +
      "You may need to add more text to your model..."+BN);

    return tokens;

  },

  generateTokens: function(targetNumber) {

    var tries = 0,
      maxTries = 500,
      tokens = [];

    OUT: while (++tries < maxTries) {

      var mn = this.root.selectChild(null, true);
      if (!mn || !mn.token) continue OUT;
      tokens.push(mn);

      while (tokens.length < targetNumber) {

        mn = this._nextNodeForArr(tokens);
        if (!mn || !mn.token) { // hit the end
          tokens = []; // start over
          continue OUT;
        }

        tokens.push(mn);
      }

      break;
    }

    // uh-oh, looks like we failed...
    if (tokens.length < targetNumber) {

      this._onGenerationIncomplete(tries, tokens.length);
    }

    var res = [];
    for (var i = 0; i < tokens.length; i++) {
      res[i] = tokens[i].token;
    }

    return res;
  },

  sentenceAware: function() {

    if (arguments.length > 0)
      throw Error('sentenceAware() takes no arguments, instead ' +
        'use the constructor RiMarkov(n, recognizeSentences);');
    return this.isSentenceAware;
  },

  getN: function() {
    return this._n;
  },

  print: function() {

    if (console) console.log(this.root.asTree(false));
    return this;
  },

  ready: function(url) {
    return this.size() > 0;
  },

  loadFrom: function(url, multiplier, regex, callback) {

    var me = this;

    is(url, S) || ok(url, A);

    if (is(multiplier, F)) {
      callback = multiplier;
      multiplier = undefined;
    }
    else if (is(regex, F)) {
      callback = regex;
      regex = undefined;
    }

    RiTa.loadStrings(url, function(data) {

      data = data.join(BN);
      me.loadText(data, multiplier, regex);
      callback && is(callback, F) && (callback(data));
    });
  },

  loadText: function(text, multiplier, regex) {

    //log("loadText: "+text.length + " "+this.isSentenceAware);

    ok(text, S);

    multiplier = multiplier || 1;

    if (multiplier < 1 || multiplier != Math.floor(multiplier)) // TODO: really?
      err('Multiplier must be an positive integer, found: ' + multiplier);

    var result = !this.isSentenceAware ?
      this.loadTokens(RiTa.tokenize(text, regex), multiplier) :
      this._loadSentences(RiTa.splitSentences(text), multiplier);

    return result;
  },

  loadTokens: function(tokens, multiplier) {

    multiplier = multiplier || 1;

    if (multiplier < 1 || multiplier != Math.floor(multiplier))
      err('multiplier must be an positive integer, found: ' + multiplier);

    this.root.count += tokens.length; // here?

    for (var toAdd, k = 0; k < tokens.length; k++) {
      toAdd = [];

      for (var j = 0; j < this._n; j++) {
        if ((k + j) < tokens.length) toAdd[j] = (tokens[k + j]) ? tokens[k + j] : null;
        else toAdd[j] = null;
      }

      // hack to deal with multiplier...
      for (var l = 0; l < multiplier; l++) {

        var node = this.root;
        for (var i = 0; i < toAdd.length; i++) {
          if (node.token) {
            node = node.addChild(toAdd[i], 1);
          }
        }
      }
    }

    return this;
  },

  generateSentences: function(num) {

    if (!this.isSentenceAware) {
      err('generateSentences() can only be called when the model is ' +
        'in \'sentence-aware\' mode, otherwise use generateTokens()');
    }

    var mn = this._getSentenceStart(),
      s = mn.token + SP,
      result = [],
      tries = 0,
      totalTries = 0,
      wordsInSentence = 1;

    while (result.length < num) {

      if (wordsInSentence >= this.maxSentenceLength) {

        //console.log("MarkovModel.generateSentences().reject:: too long!");

        mn = this._getSentenceStart();
        s = mn.token + SP;
        wordsInSentence = 1;
      }

      if (mn.isLeaf()) {
        mn = this._tracePathFromRoot(mn);
        continue;
      }

      mn = this._nextNodeForNode(mn);

      if (mn.isSentenceStart) {

        if (wordsInSentence >= this.minSentenceLength) {

          var candidate = RiTa.untokenize(s.trim().split(/\s+/));

          // deals with incorrect output: "word( next" -> "word (next"
          candidate = candidate.replace(/(.)\( /, "$1 ("); // added DCH: 5/2/14

          if (this._validateSentence(candidate)) {

            // got one, store and reset the counters
            if (result.indexOf(candidate) < 0)
              result.push(candidate);

            //log(result.length+" RESULTS SO FAR");
            totalTries += tries;
            tries = 0;
          }
        }
        mn = this._getSentenceStart();
        s = mn.token + SP;
        wordsInSentence = 1;
        continue;
      }

      // add the next word
      wordsInSentence++;
      s += mn.token + SP;

      // check if its time to give up
      if (++tries >= RiMarkov.MAX_GENERATION_ATTEMPTS) {

        this._onGenerationIncomplete(totalTries += tries, result.length);
        break; // give-up
      }
    }

    return result;
  },

  _validateSentence: function(sent) {

    var tokens = RiTa.tokenize(sent),
      first = tokens[0],
      last = tokens[tokens.length - 1];

    if (!first.match(/[A-Z]\S*/)) {
      if (this.printIgnoredText)
        log("Skipping: bad first char in '" + sent + "'");
      return false;
    }

    if (!last.match(/[!?.]/)) {
      if (this.printIgnoredText)
        log("Bad last token: '" + last + "' in: " + sent);
      return false;
    }

    if (!this.allowDuplicates) {
      if (!this.isSentenceAware) {
        err('Invalid state: allowDuplicates must be' +
          ' true when not generating sentences');
      }

      if (this.sentenceList.indexOf(sent) > -1) {
        if (++this.skippedDups == this.maxDuplicatesToSkip) {
          warn('Hit skip-maximum (RiMarkov.maxDuplicatesToSkip=' + this.maxDuplicatesToSkip +
            ') after skipping ' + this.maxDuplicatesToSkip + ', now allowing duplicates!');
          this.allowDuplicates = true;
        }

        if (this.printIgnoredText)
          log('Ignoring duplicate: ' + sent);

        return false;
      }
    }

    var words = sent.split(/\s+/);
    if (RiTa.isAbbreviation(words[words.length - 1])) {
      //log("BAD SENTENCE: "+sent);
      return false;
    }

    return true;
  },

  _tracePathFromRoot: function(node) {

    node.pathFromRoot(this.pathTrace);

    this.pathTrace.pop(); // ignore the first element
    var mn = this.root;
    while (this.pathTrace.length) {
      var search = this.pathTrace.pop();
      mn = mn.lookup(search);
    }

    return mn;
  },

  _nextNodeForArr: function(previousTokens) {

    // Follow the seed path down the tree
    var firstLookupIdx = Math.max(0, previousTokens.length - (this._n - 1)),
      node = this.root.lookup(previousTokens[firstLookupIdx++]);

    for (var i = firstLookupIdx; i < previousTokens.length; i++) {

      if (node) node = node.lookup(previousTokens[i]);
    }

    // Now select the next node
    return node ? node.selectChild(null, true) : null;
  },

  _nextNodeForNode: function(current) {

    var attempts = 0,
      selector, pTotal = 0,
      nodes = current.childNodes(),
      maxProbMisses = 1000;

    while (true) {

      pTotal = 0;
      selector = Math.random();
      for (var i = 0, j = nodes.length; i < j; i++) {

        pTotal += nodes[i].probability();
        if (current.isRoot() && (this.isSentenceAware && !nodes[i].isSentenceStart)) {
          continue;
        }
        if (selector < pTotal) {
          return nodes[i];
        }
      }

      attempts++;
      warn('Prob. miss (#\'+attempts+\') in RiMarkov.nextNode().' +
        ' Make sure there are a sufficient\n       # of sentences' +
        ' in the model that are longer than \'minSentenceLength\'');

      if (attempts == maxProbMisses) // should never happen
        err('PROB. MISS' + current + ' total=' + pTotal + ' selector=' + selector);
    }
  },

  _clean: function(sentence) {

    return RiTa.trim(sentence.replace(/\s+/, SP));
  },

  _onGenerationIncomplete: function(tries, successes) {

    warn(BN+'RiMarkov failed to complete after ' + tries +
      ' tries and ' + successes + ' successful generations.' +
      ' You may need to add more text to the model...'+BN);
  },

  // Loads a sentence[] into the model; each element must be a single sentence
  _loadSentences: function(sentences, multiplier) {

    ok(sentences, A);

    multiplier = multiplier || 1;
    multiplier = Math.min(multiplier, 1);

    // log("_loadSentences("+sentences.length+", multiplier="+multiplier+" "+this.allowDuplicates+")");

    var i, j, tokens, sentence, allWords = [];

    // do the cleaning/splitting first ---------------------

    for (i = 0; i < sentences.length; i++) {

      sentence = this._clean(sentences[i]);

      // do we need this?
      if (!this.allowDuplicates) this.sentenceList.push(sentence);

      tokens = RiTa.tokenize(sentence);

      if (!this._validSentenceStart(tokens[0])) {

        if (this.printIgnoredText)
          warn("Skipping (bad sentence start): " + tokens);
        continue;
      }

      //log("Added sentence start] " + tokens);

      allWords.push(RiMarkov._SSDLM + tokens[0]); // bad hack for sentence-starts

      for (j = 1; j < tokens.length; j++)
        allWords.push(tokens[j]);
    }

    // ------------------------------------------------

    var toAdd, words = allWords,
      nFactor = this.getN();

    for (i = 0; i < words.length; i++) {

      toAdd = [];
      for (j = 0; j < nFactor; j++) {
        if ((i + j) < words.length)
          toAdd[j] = words[i + j];
      }

      // hack to deal with multiplier...
      for (j = 0; j < multiplier; j++)
        this._addSentenceSequence(toAdd);
    }

    this.root.count += words.length;

    return this;
  },

  size: function() {

    return this.root.count;
  },

  _validSentenceStart: function(word) {

    return (!this.isSentenceAware || word && word.match(RiMarkov._SSRE));
  },

  _addSentenceSequence: function(toAdd) {

    var node = this.root;

    for (var i = 0; i < toAdd.length; i++) {

      if (!toAdd[i]) continue;

      if (node.token) {

        var add = toAdd[i];

        if (startsWith(add, RiMarkov._SSDLM)) {

          add = add.substring(RiMarkov._SSDLM.length);
          var parent = node;

          node = node.addChild(add, 1);
          node.isSentenceStart = true;

          if (parent.isRoot()) {
            this.sentenceStarts.push(node.token);
          }

        } else
          node = node.addChild(add, 1);
      }
    }
  },

  _getSentenceStart: function() {

    if (!this.isSentenceAware) {
      err("getSentenceStart() can only " +
        "be called when the model is in 'sentence-aware' mode...");
    }

    if (!this.sentenceStarts || !this.sentenceStarts.length)
      err('No sentence starts found! genSen=' + this.isSentenceAware);

    return this.root.lookup(RiTa.randomItem(this.sentenceStarts));
  },

  _findNode: function(path) {

    if (!path || !is(path, A) || !path.length)
      return null;

    var nFactor = this._n;
    var numNodes = Math.min(path.length, nFactor - 1);
    var firstLookupIdx = Math.max(0, path.length - (nFactor - 1));
    var node = this.root.lookup(path[firstLookupIdx++]);

    if (!node) return null;

    var idx = 0; // found at least one good node
    var nodes = [];
    nodes[idx++] = node;
    for (var i = firstLookupIdx; i < path.length; i++) {
      node = node.lookup(path[i]);
      if (!node) return null;
      nodes[idx++] = node;
    }

    return nodes ? nodes[nodes.length - 1] : null;
  }
};

var RiWordNet = function() { // stub
    throw Error("RiWordNet is not yet implemented in JavaScript!");
};

////////////////////////////////////////////////////////////////
// RiString
////////////////////////////////////////////////////////////////

var RiString = makeClass();

RiString.phones = {

  consonants: ['b', 'ch', 'd', 'dh', 'f', 'g', 'hh', 'jh', 'k', 'l', 'm',
    'n', 'ng', 'p', 'r', 's', 'sh', 't', 'th', 'v', 'w', 'y', 'z', 'zh'
  ],

  vowels: ['aa', 'ae', 'ah', 'ao', 'aw', 'ax', 'ay', 'eh', 'er', 'ey', 'ih',
    'iy', 'ow', 'oy', 'uh', 'uw'
  ],

  onsets: ['p', 't', 'k', 'b', 'd', 'g', 'f', 'v', 'th', 'dh', 's', 'z',
    'sh', 'ch', 'jh', 'm', 'n', 'r', 'l', 'hh', 'w', 'y', 'p r', 't r',
    'k r', 'b r', 'd r', 'g r', 'f r', 'th r', 'sh r', 'p l', 'k l', 'b l',
    'g l', 'f l', 's l', 't w', 'k w', 'd w', 's w', 's p', 's t', 's k',
    's f', 's m', 's n', 'g w', 'sh w', 's p r', 's p l', 's t r', 's k r',
    's k w', 's k l', 'th w', 'zh', 'p y', 'k y', 'b y', 'f y', 'hh y',
    'v y', 'th y', 'm y', 's p y', 's k y', 'g y', 'hh w', ''
  ],

  digits: ['z-ih-r-ow', 'w-ah-n', 't-uw', 'th-r-iy', 'f-ao-r', 'f-ay-v',
    's-ih-k-s', 's-eh1-v-ax-n', 'ey-t', 'n-ih-n'
  ]
};

RiString._syllabify = function(input) { // adapted from FreeTTS

  var dbug, None, internuclei = [],
    syllables = [], // returned data structure.
    sylls = ((typeof(input) == 'string') ? input.split('-') : input);

  for (var i = 0; i < sylls.length; i++) {

    var phoneme = sylls[i].trim(),
      stress = None;

    if (!phoneme.length) continue;

    if (isNum(last(phoneme))) {

      stress = parseInt(last(phoneme));
      phoneme = phoneme.substring(0, phoneme.length - 1);
    }

    if (dbug) log(i + ")" + phoneme + ' stress=' + stress + ' inter=' + internuclei.join(':'));

    if (inArray(RiString.phones.vowels, phoneme)) {

      // Split the consonants seen since the last nucleus into coda and onset.
      var coda = None,
        onset = None;

      // Make the largest onset we can. The 'split' variable marks the break point.
      for (var split = 0; split < internuclei.length + 1; split++) {

        coda = internuclei.slice(0, split);
        onset = internuclei.slice(split, internuclei.length);

        if (dbug) log('  ' + split + ') onset=' + onset.join(':') +
          '  coda=' + coda.join(':') + '  inter=' + internuclei.join(':'));

        // If we are looking at a valid onset, or if we're at the start of the word
        // (in which case an invalid onset is better than a coda that doesn't follow
        // a nucleus), or if we've gone through all of the onsets and we didn't find
        // any that are valid, then split the nonvowels we've seen at this location.
        var bool = inArray(RiString.phones.onsets, onset.join(" "));
        if (bool || syllables.length === 0 || onset.length === 0) {
          if (dbug) log('  break ' + phoneme);
          break;
        }
      }

      //if (dbug)log('  onset='+join(',',onset)+'  coda='+join(',',coda));

      // Tack the coda onto the coda of the last syllable. Can't do it if this
      // is the first syllable.
      if (syllables.length > 0) {

        //syllables[syllables.length-1][3] = syllables[syllables.length-1][3] || [];
        //log('  len='+syllables[syllables.length-1][3].length);
        extend(syllables[syllables.length - 1][3], coda);

        if (dbug) log('  tack: ' + coda + ' -> len=' +
          syllables[syllables.length - 1][3].length + " [" +
          syllables[syllables.length - 1][3] + "]");
      }

      // Make a new syllable out of the onset and nucleus.

      var toPush = [
        [stress], onset, [phoneme],
        []
      ];
      syllables.push(toPush);

      // At this point we've processed the internuclei list.
      internuclei = [];
    }
    else if (!inArray(RiString.phones.consonants, phoneme) && phoneme != " ") {
      throw Error('Invalid phoneme: ' + phoneme);
    }
    else { // a consonant

      //log('inter.push: '+phoneme);
      internuclei.push(phoneme);
    }
  }


  // Done looping through phonemes. We may have consonants left at the end.
  // We may have even not found a nucleus.
  if (internuclei.length > 0) {

    if (syllables.length === 0) {

      syllables.push([
        [None], internuclei, [],
        []
      ]);
    } else {

      extend(syllables[syllables.length - 1][3], internuclei);
    }
  }

  return RiString._stringify(syllables);
};

/*
 * Takes a syllabification and turns it into a string of phonemes,
 * delimited with dashes, with spaces between syllables
 */
RiString._stringify = function(syllables) {

  var i, j, ret = [];
  for (i = 0; i < syllables.length; i++) {

    var syl = syllables[i];
    var stress = syl[0][0];
    var onset = syl[1];
    var nucleus = syl[2];
    var coda = syl[3];

    if (stress !== undefined && nucleus.length) // dch
      nucleus[0] += (E + stress);

    var data = [];
    for (j = 0; j < onset.length; j++)
      data.push(onset[j]);

    for (j = 0; j < nucleus.length; j++)
      data.push(nucleus[j]);

    for (j = 0; j < coda.length; j++)
      data.push(coda[j]);

    ret.push(data.join('-'));
  }

  return ret.join(SP);
};

// ////////////////////////////////////////////////////////////
// Member functions
// ////////////////////////////////////////////////////////////

RiString.prototype = {

  init: function(text) {

    if (is(text, N)) {

      text = String.fromCharCode(text);
    }

    text = text || '';

    // convenience fields, in case we use this object for rendering
    this.x = 0;
    this.y = 0;
    this.z = 0;

    this._text = text;
    this._features = undefined;
  },

  copy: function() {

    var rs = RiString(this._text),
      feats = this.features();

    if (feats) {
      rs._features = {};

      for (var prop in feats) {

        rs._features[prop] = feats[prop];
      }
    }
    return rs;
  },

  features: function() {

    if (!this._features) {
      this.analyze();
    }
    return this._features;
  },

  _initFeatureMap: function() {

    if (!this._features) {

      this._features = {};

    } else {

      delete this._features.tokens;
      delete this._features.stresses;
      delete this._features.phonemes;
      delete this._features.syllables;
      delete this._features.pos;
      delete this._features.text;
    }

    //this._features.mutable = "true";
    this._features.text = this.text();
  },


  analyze: function() {

    var phonemes = E, syllables = E, stresses = E, slash = '/',
      delim = '-', lex, stressyls, phones, lts, ltsPhones, useRaw,
      words = RiTa.tokenize(this._text);

    if (!this._features) this._initFeatureMap();

    this._features.tokens = words.join(SP);
    this._features.pos = RiTa.getPosTags(this._text).join(SP);

    if (RiLexicon.enabled) {

      lex = RiTa._lexicon();
      for (var i = 0, l = words.length; i < l; i++) {

        useRaw = false;

        phones = lex && lex._getRawPhones(words[i]);

        if (!phones) {

          if (words[i].match(/[a-zA-Z]+/))
            log("[RiTa] Used LTS-rules for '" + words[i] + "'");

          lts = lex._letterToSound();

          ltsPhones = lts.getPhones(words[i]);

          if (ltsPhones && ltsPhones.length > 0) {

            phones = RiString._syllabify(ltsPhones);

          } else {
            phones = words[i];
            useRaw = true;
          }
        }

        phonemes += phones.replace(/[0-2]/g, E).replace(/ /g, delim) + SP;
        syllables += phones.replace(/ /g, slash).replace(/1/g, E) + SP;

        if (!useRaw) {
          stressyls = phones.split(SP);
          for (var j = 0; j < stressyls.length; j++) {

            if (!stressyls[j].length) continue;

            stresses += (stressyls[j].indexOf(RiTa.STRESSED) > -1) ?
              RiTa.STRESSED : RiTa.UNSTRESSED;

            if (j < stressyls.length - 1) stresses += slash;
          }
        } else {

          stresses += words[i];
        }

        if (!endsWith(stresses, SP)) stresses += SP;
      }

      stresses = stresses.trim();
      phonemes = phonemes.trim().replace(/\\s+/, SP);
      syllables = syllables.trim().replace(/\\s+/, SP);
    }

    this._features.stresses = stresses;
    this._features.phonemes = phonemes;
    this._features.syllables = syllables;

    return this;
  },

  get: function(featureName) {

    this._features || this.analyze();
    var s = this._features[featureName];

    //console.log(featureName,tracked.indexOf(featureName));
    if (!s && (RiTa._FEATURES.indexOf(featureName) > -1) &&
      (!this._features.hasOwnProperty(featureName)))
    {
      this.analyze();
      s = this._features[featureName];
    }
    return s;
  },

  set: function(featureName, featureValue) {

    this._features || (this._features = {});
    this._features[featureName] = featureValue;
    //log(this._features);
    return this;
  },

  endsWith: function(substr) {

    return endsWith(this._text, substr);
  },

  equals: function(arg) {

    return (typeof arg === S) ? arg === this._text : arg.text() === this._text;
  },

  equalsIgnoreCase: function(arg) {

    if (typeof arg === S) {

      return arg.toLowerCase() === this._text.toLowerCase();
    } else {

      return arg.text().toLowerCase() === this._text.toLowerCase();
    }
  },

  text: function(theText) {

    if (arguments.length > 0) {
      this._text = theText;
      this._initFeatureMap();
      return this;
    }
    return this._text;
  },

  pos: function() {

    var words = RiTa.tokenize(this._text);
    var tags = PosTagger.tag(words);

    for (var i = 0, l = tags.length; i < l; i++) {
      if (!strOk(tags[i]))
        err("RiString: can't parse pos for:" + words[i]);
    }

    return tags;
  },

  posAt: function(index) {

    var tags = this.pos();

    if (!tags || !tags.length)
      return E;

    return tags[ Math.min(index < 0 ?
      tags.length + index : index, tags.length - 1) ];
  },

  wordAt: function(index) {

    var words = this.words();
    if (index < 0 || index >= words.length)
      return E;
    return words[index];
  },

  wordCount: function() {

    return this._text.length ? this.words().length : 0;
  },

  words: function() {

    return RiTa.tokenize(this._text);
  },

  indexOf: function(searchstring, start) {

    return this._text.indexOf(searchstring, start);
  },

  lastIndexOf: function(searchstring, start) {

    return this._text.lastIndexOf(searchstring, start);
  },

  length: function() {

    return this._text.length;
  },

  match: function(regex) {

    return this._text.match(regex) || [];
  },

  slice: function(begin, end) {

    return this._text.slice(begin, end) || E;
  },

  insertChar: function(idx, toInsert) {

    var s = this.text();

    if (idx > s.length || idx < -s.length) {

      warn("RiString.insertChar: bad index=" + idx);
      return this;
    }

    idx = idx < 0 ? idx += s.length : idx;
    var beg = s.substring(0, idx);
    var end = s.substring(idx);

    if (toInsert) beg += toInsert;

    return this.text(beg + end);
  },

  removeChar: function(idx) {

    var s = this.text();

    if (idx > s.length || idx < -s.length) {
      warn("RiString.removeChar: bad index=" + idx);
      return this;
    }
    idx = idx < 0 ? idx += s.length : idx;

    this.text(this._text.substring(0, idx).concat(this._text.substring(idx + 1)));
    return this;
  },

  replaceChar: function(idx, replaceWith) {

    var s = this.text();

    if (idx > s.length || idx < -s.length) {
      warn("RiString.replaceChar: bad index=" + idx);
      return this;
    }
    idx = idx < 0 ? idx += s.length : idx;

    s = this.text();
    var beg = s.substring(0, idx),
      end = s.substring(idx + 1);

    if (replaceWith)
      beg += replaceWith;

    return this.text(beg + end);
  },

  replaceFirst: function(regex, replaceWith) {

    // strip out global if we have it
    if (regex && !is(regex, S) && regex.global) {

      var flags = '';
      regex.ignoreCase && (flags += 'i');
      regex.multiline && (flags += 'm');
      regex.sticky && (flags += 'y');
      regex = new RegExp(regex.source, flags);
    }

    this._text = this._text.replace(regex, replaceWith);
    return this;
  },

  replaceAll: function(pattern, replacement) {

    var flags = 'g';

    if (pattern && (replacement || replacement === '')) {

      if (!is(pattern, S) && pattern.source) {

        pattern.ignoreCase && (flags += 'i');
        pattern.multiline && (flags += 'm');
        pattern.sticky && (flags += 'y');

        pattern = pattern.source;
      } else {

        pattern = escapeRegExp(pattern);
      }

      //console.log("RE: /"+pattern+"/"+flags);

      this._text = this._text.replace(new RegExp(pattern, flags), replacement);
    }

    return this;
  },

  removeWord: function(wordIdx) {

    return this.replaceWord(wordIdx, E);
  },

  insertWord: function(wordIdx, newWord) {

    var words = this.words(); //  tokenize

    if (wordIdx < 0) wordIdx += words.length;

    // log("insertWord("+ newWord+', '+wordIdx+") -> words["+wordIdx+"] = " + words[wordIdx]);

    if (newWord && newWord.length >= 0 && wordIdx >= 0 && wordIdx < words.length) {

      words[wordIdx] = newWord + SP + words[wordIdx];

      this.text(RiTa.untokenize(words));
    }

    return this;
  },

  toCharArray: function() {

    return this._text.split(RiTa.E);
  },

  replaceWord: function(wordIdx, newWord) {

    //log("replaceWord: "+wordIdx+", '"+newWord+"'");

    var words = this.words(); //  tokenize

    if (wordIdx < 0) wordIdx += words.length;

    if ((newWord || newWord === E) && wordIdx >= 0 && wordIdx < words.length) {

      words[wordIdx] = newWord;

      this.text(RiTa.untokenize(words));
    }

    return this;
  },

  split: function(separator, limit) {

    var parts = this._text.split(separator, limit);
    var rs = [];
    for (var i = 0; i < parts.length; i++) {
      if (parts[i])
        rs.push(new RiString(parts[i]));
    }
    return rs;
  },

  startsWith: function(substr) {

    return this.indexOf(substr) === 0;
  },

  substr: function(start, length) {

    return this._text.substr(start, length);
  },

  substring: function(from, to) {

    return this._text.substring(from, to);
  },

  toLowerCase: function() {

    return this.text(this._text.toLowerCase());
  },

  toString: function() {

    return '[' + this._text + ']';
  },

  toUpperCase: function() {

    return this.text(this._text.toUpperCase());
  },

  trim: function() {

    return this.text(trim(this._text));
  },

  charAt: function(index) {

    return this._text.charAt(index);
  },

  concat: function(riString) {

    return this._text.concat(riString.text());
  }
};

// ////////////////////////////////////////////////////////////
// RiGrammar
// ////////////////////////////////////////////////////////////

var RiGrammar = makeClass();

RiGrammar.START_RULE = "<start>";
RiGrammar.PROB_PATT = /(.*[^\s])\s*\[([0-9.]+)\](.*)/;
RiGrammar.EXEC_PATT = /(.*?)(`[^`]+?\(.*?\);?`)(.*)/;
RiGrammar.STRIP_TICKS = /`([^`]*)`/g;
RiGrammar.OR_PATT = /\s*\|\s*/;

RiGrammar.prototype = {

  init: function(grammar) {

    this._rules = {};
    this.execDisabled = false;

    if (grammar) {

      // a String or Object, but make sure its not a PApplet (e.g., this)
      if (is(grammar, S) || (is(grammar, O) && typeof grammar.loadStrings !== F))
        this.load(grammar);
    }
  },

  ready: function(url) {

    return (okeys(this._rules).length > 0);
  },

  loadFrom: function(url, callback) {

    RiTa.loadStrings(url, function(data) {

      data = data.join(BN);
      this.load(data);
      is(callback, F) && (callback(data));

    }.bind(this));
  },

  load: function(grammar) {

    this.reset();

    if (is(grammar, S)) {

      if (typeof YAML != 'undefined') { // found a yaml-parser, so try it first

        try {
          //console.log('trying YAML');
          grammar = YAML.parse(grammar);

        } catch (e) {

          warn('YAML parsing failed, trying JSON');
        }
      }

      if (!is(grammar, O)) { // we failed with our yaml-parser, so try JSON
        try {

          //console.log('trying JSON');
          grammar = JSON.parse(grammar);

        } catch (e) {

          var ex = e;
        }
      }
    }

    if (ex || !is(grammar, O)) {

      if (typeof YAML != 'undefined') {
        err('Grammar appears to be invalid JSON/YAML, please check it!' +
          ' (http://jsonlint.com/ or http://yamllint.com/)', grammar);
      }
      else {
        var isnode = RiTa.env() === RiTa.NODE, verb =  isnode ? 'require'
          : 'include', syntax = isnode ? "YAML = require('yamljs')" :
            '<script src="yaml.min.js"></script>';

        err('Grammar appears to be invalid JSON, please check it at ' +
          'http://jsonlint.com/. If you are using YAML, be sure to '  +
          verb + ' yamljs (https://github.com/jeremyfa/yaml.js), e.g. ' +
          syntax, grammar);
      }

      return;
    }

    for (var rule in grammar) {
      this.addRule(rule, grammar[rule]);
    }

    return this;
  },

  addRule: function(name, theRule, weight) {
    var dbug = false;

    weight = weight || 1.0; // default

    if (dbug) log('addRule: "' + name + '" : "' + theRule +
      '"       [' + get(theRule) + ']');

    var ruleset = theRule;
    if (!is(theRule, A))
      ruleset = theRule.split(RiGrammar.OR_PATT);

    for (var i = 0; i < ruleset.length; i++) {

      var rule = ruleset[i];
      var prob = weight;
      var m = RiGrammar.PROB_PATT.exec(rule);

      if (m) // found weighting
      {
        if (dbug) {
          log("Found weight for " + rule);
          for (i = 0; i < m.length; i++)
            log("  " + i + ") '" + m[i] + "'");
        }

        rule = m[1] + m[3];
        prob = m[2];

        if (dbug) log("weight=" + prob + " rule='" + rule + "'");
      }

      if (this.hasRule(name)) {

        if (dbug) log("rule exists");
        var temp = this._rules[name];
        temp[rule] = prob;
      } else {

        var temp2 = {};
        temp2[rule] = prob;
        this._rules[name] = temp2;

        if (dbug) log("added rule: " + name);
      }
    }

    return this;
  },

  removeRule: function(name) {

    delete this._rules[name];
    return this;
  },

  _copy: function() {

    var tmp = RiGrammar();
    for (var name in this._rules) {
      tmp._rules[name] = this._rules[name];
    }
    return tmp;
  },

  reset: function() {

    this._rules = {};
    return this;

  },

  doRule: function(pre) {

    var cnt = 0,
      name = E,
      rules = this._rules[pre];

    if (!rules) return null;

    for (name in rules) cnt++;

    if (!cnt) return null;

    return (cnt == 1) ? name : this._getStochasticRule(rules);
  },


  getGrammar: function() {

    var s = E;
    for (var name in this._rules) {
      s += (name + BN);
      var choices = this._rules[name];
      for (var p in choices) {
        s += ("  '" + p + "' [" + choices[p] + ']' + BN);
      }
    }
    return RiTa.chomp(s);
  },

  print: function() {

    if (console) {
      var ln = "\n------------------------------";
      console.log(ln + BN + this.getGrammar() + ln);
    }
    return this;

  },

  hasRule: function(name) {

    return (typeof this._rules[name] !== 'undefined');
  },

  expandWith: function(literal, symbol) { // TODO: finish

    var name, gr = this._copy(),
      match = false;

    for (name in gr._rules) {

      if (name === symbol) {

        var obj = {};
        obj[literal] = 1.0;
        gr._rules[name] = obj;
        match = true;
      }
    }

    if (!match)
      err("Rule '" + symbol + "' not found in grammar");

    // TODO: tmp, awful hack, write this correctly
    var tries, maxTries = 1000;
    for (tries = 0; tries < maxTries; tries++) {
      var s = gr.expand();
      if (s.indexOf(literal) > -1)
        return s;
    }
    err("RiGrammar failed to complete after " + tries + " tries" + BN);
  },

  expand: function(context) {

    return this.expandFrom(RiGrammar.START_RULE, context);
  },

  expandFrom: function(rule, context) {

    if (!okeys(this._rules).length)
      err("(RiGrammar) No grammar rules found!");

    if (!this.hasRule(rule))
      err("Rule not found: " + rule + BN + "Rules:" + BN + JSON.stringify(this._rules));

    var parts, theCall, callResult, tries = 0,
      maxIterations = 1000;
    while (++tries < maxIterations) {
      var next = this._expandRule(rule);

      if (next && next.length) { // matched a rule

        rule = next;
        continue;
      }

      if (this.execDisabled) break; // return

      // finished rules, check for back-ticked exec calls

      parts = RiGrammar.EXEC_PATT.exec(rule);

      if (!parts || !parts.length) break; // return, no evals

      if (parts.length > 2) {

        theCall = parts[2];

        if (this._countTicks(theCall) != 2) {

          warn("Unable to parse recursive exec: " + theCall + "...");
          return null;
        }

        callResult = this._handleExec(theCall, context);

        if (!callResult) {

          if (0) log("[WARN] (RiGrammar.expandFrom) Unexpected" +
            " state: eval(" + theCall + ") :: returning '" + rule + "'");

          break; // return
        }

        rule = parts[1] + callResult;

        (parts.length > 3) && (rule += parts[3]);
      }
    }

    if (tries >= maxIterations)
      log("[WARN] max number of iterations reached: " + maxIterations);

    return RiTa.unescapeHTML(rule);
  },

  _countTicks: function(theCall) {

    var count = 0;
    for (var i = 0; i < theCall.length; i++) {
      if (theCall.charAt(i) == '`')
        count++;
    }
    return count;
  },

  openEditor: function() {

    warn("Editor not yet implemented in JavaScript");
    return this;
  },

  _handleExec: function(input, context) { // TODO: list all cases and rewrite

    if (!input || !input.length) return null;

    // strip backticks and eval
    var res, exec = input.replace(RiGrammar.STRIP_TICKS, '$1');

    try {
      // TODO: See issue #9 [https://github.com/dhowe/RiTaJS/issues?state=open]
      // if (typeof module != 'undefined' && module.exports) // for node
      // 		return require("vm").runInThisContext(exec,context);

      res = eval(exec); // try in global context

      return res ? res + E : null;

    } catch (e) {

      // try with the PApplet context
      // TODO: clean this up
      // TODO: do we need to explicitly check window[funName] in the browser?
      var parts = exec.split('(');
      if (parts && parts.length == 2) {

        var funName = parts[0];
        var argStr = parts[1].replace(/\)/,E);
        var g = context;
        if (!g && typeof window != 'undefined') g = window;
        if (g && g[funName] && is(g[funName], F)) {

          var args = argStr.split(',');
          //log("calling "+funName + "("+argStr+");");
          res = g[funName].apply(g, args);
          return res ? res + E : null;
        }
      }

      warn("RiGrammar failed parsing: " + input + BN + " -> " + e.message);
      return null;
    }
  },

  _expandRule: function(prod) {

    var entry, idx, pre, expanded, post, dbug = 0;

    if (dbug) log("_expandRule(" + prod + ")");

    for (var name in this._rules) {

      entry = this._rules[name];

      if (dbug) log("  name=" + name + "  entry=" + entry + "  prod=" + prod + "  idx=" + idx);

      idx = prod.indexOf(name);

      if (idx >= 0) { // got a match, split into 3 parts

        pre = prod.substring(0, idx) || E;
        expanded = this.doRule(name) || E;
        post = prod.substring(idx + name.length) || E;

        if (dbug) log("  pre=" + pre + "  expanded=" + expanded +
          "  post=" + post + "  result=" + pre + expanded + post);

        return pre + expanded + post;
      }
    }

    return null; // no rules matched
  },

  _getStochasticRule: function(temp) { // map

    var name, dbug = false;

    if (dbug) log("_getStochasticRule(" + temp + ")");

    var p = Math.random();
    var result, total = 0;
    for (name in temp) {
      total += parseFloat(temp[name]);
    }

    if (dbug) log("total=" + total + "p=" + p);

    for (name in temp) {
      if (dbug) log("  name=" + name);
      var amt = temp[name] / total;

      if (dbug) log("amt=" + amt);

      if (p < amt) {
        result = name;
        if (dbug) log("hit!=" + name);
        break;
      } else {
        p -= amt;
      }
    }
    return result;
  }


}; // end RiGrammar

var RiTaEvent = makeClass();

RiTaEvent._callbacksDisabled = false;
RiTaEvent.ID = 0;

RiTaEvent.prototype = {

  init: function(source, eventType, data) {

    is(source, O) || ok(source, S);

    this._id = ++RiTaEvent.ID;
    this._data = data;
    this._source = source;
    this._type = eventType || RiTa.UNKNOWN;
  },

  toString: function() {

    var s = 'RiTaEvent[#' + this._id + ' type=' +
      '(' + this._type + ') source=' + this._source.toString();

    s += !this._data ? s += ' data=null' :
      (' data-length=' + this._data.toString().length);

    return s + ']';
  },

  isType: function(t) {

    return this._type === t;
  },

  _fire: function(callback) {

    callback = callback || window.onRiTaEvent;

    if (callback && is(callback, F)) {
      try {

        callback(this); // first arg ??
        return this;

      } catch (err) {

        RiTaEvent._callbacksDisabled = true;
        var msg = "RiTaEvent: error calling '" + callback + "': " + err;
        is(callback, S) && (msg += " Callback must be a function in JS!");
        warn(msg);
        throw err;
      }
    }
  }
};


/////////////////////////////////////////////////////////////////////////
//StringTokenizer
/////////////////////////////////////////////////////////////////////////

var StringTokenizer = makeClass();

StringTokenizer.prototype = {

  init: function(str, delim) {

    this.idx = 0;
    this.text = str;
    this.delim = delim || " ";
    this.tokens = str.split(delim);
  },

  nextToken: function() {

    return (this.idx < this.tokens.length) ? this.tokens[this.idx++] : null;
  }
};

////////////////////////// PRIVATE CLASSES ///////////////////////////////

// ////////////////////////////////////////////////////////////
// TextNode
// ////////////////////////////////////////////////////////////

var TextNode = makeClass();

TextNode.prototype = {

  init: function(parent, token) {
    this.count = 0;
    this.children = {};
    this.parent = parent;
    this.token = token;
  },

  pathFromRoot: function(result) {
    var mn = this;
    while (true) {
      if (mn.isRoot()) break;
      result.push(mn.token);
      mn = mn.parent;
    }
  },

  selectChild: function(regex, probabalisticSelect) {
    var ps = probabalisticSelect || true;
    return this.children ? this._select(this.childNodes(regex), ps) : null;
  },

  _select: function(arr, probabalisticSelect) {
    if (!arr) throw TypeError("bad arg to '_select()'");
    probabalisticSelect = probabalisticSelect || false;
    return (probabalisticSelect ? this._probabalisticSelect(arr) : arr[Math.floor((Math.random() * arr.length))]);
  },

  _probabalisticSelect: function(arr) {

    if (!arr) throw TypeError("bad arg to '_probabalisticSelect()'");

    //log("RiTa.probabalisticSelect("+c+", size="+c.size()+")");
    if (!arr.length) return null;
    if (arr.length == 1) return arr[0];

    // select from multiple options based on frequency
    var pTotal = 0,
      selector = Math.random();
    for (var i = 0; i < arr.length; i++) {

      pTotal += arr[i].probability();
      if (selector < pTotal)
        return arr[i];
    }
    err("Invalid State in RiTa.probabalisticSelect()");
  },

  addChild: function(newToken, initialCount) {

    initialCount = initialCount || 1;

    var node = this.children[newToken];

    //  add first instance of this token
    if (!node) {

      node = new TextNode(this, newToken);
      node.count = initialCount;
      this.children[newToken] = node;
    } else {

      node.count++;
    }

    return node;
  },

  asTree: function(sort) {
    var s = this.token + " ";
    if (!this.isRoot())
      s += "(" + this.count + ")->";
    s += "{";
    if (!this.isLeaf())
      return this.childrenToString(this, s, 1, sort);
    return s + "}";
  },

  isRoot: function() {
    return !this.parent;
  },

  isLeaf: function() {
    return this.childCount() === 0;
  },

  probability: function() {
    //log('probability: '+ this.count+'/'+this.siblingCount());
    return this.count / this.siblingCount();
  },

  childNodes: function(regex) {
    if (!this.children) return EA;
    regex = is(regex, S) ? new RegExp(regex) : regex;
    var res = [];
    for (var k in this.children) {
      var nd = this.children[k];
      if (!regex || (nd && nd.token && nd.token.search(regex) > -1)) {
        res.push(nd);
      }
    }
    return res;
  },

  siblingCount: function() {
    if (!this.parent) err("Illegal siblingCount on ROOT!");
    return this.parent.childCount();
  },

  childCount: function() {
    if (!this.children) return 0;
    var sum = 0;
    for (var k in this.children) {
      if (k && this.children[k])
        sum += this.children[k].count;
    }
    return sum;
  },

  // takes node or string, returns node
  lookup: function(obj) {
    if (!obj) return null;
    obj = (typeof obj != S && obj.token) ? obj.token : obj;
    return obj ? this.children[obj] : null;
  },

  childrenToString: function(textNode, str, depth, sort) {

    var i, j, k, mn = textNode,
      l = [],
      node = null,
      indent = BN;

    sort = sort || false;

    for (k in textNode.children) {
      l.push(textNode.children[k]);
    }

    if (!l.length) return str;

    if (sort) l.sort();

    for (j = 0; j < depth; j++)
      indent += "  ";

    for (i = 0; i < l.length; i++) {

      node = l[i];

      if (!node) break;

      var tok = node.token;
      if (tok) {
        (tok == BN) && (tok = "\\n");
        (tok == "\r") && (tok = "\\r");
        (tok == "\t") && (tok = "\\t");
        (tok == "\r\n") && (tok = "\\r\\n");
      }

      str += indent + "'" + tok + "'";

      if (!node.count)
        err("ILLEGAL FREQ: " + node.count + " -> " + mn.token + "," + node.token);

      if (!node.isRoot())
        str += " [" + node.count + ",p=" + //formatter.format
        (node.probability().toFixed(3)) + "]->{";

      if (node.children)
        str = this.childrenToString(node, str, depth + 1, sort);
      else
        str += "}";
    }

    indent = BN;
    for (j = 0; j < depth - 1; j++)
      indent += "  ";

    return str + indent + "}";
  },

  toString: function() {
    return '[ ' + this.token + " (" + this.count + '/' + this.probability().toFixed(3) + '%)]';
  }
};

// ////////////////////////////////////////////////////////////
// Conjugator
// ////////////////////////////////////////////////////////////

var Conjugator = makeClass();

Conjugator.prototype = {

  init: function() {

    this.perfect = this.progressive = this.passive = this.interrogative = false;
    this.tense = RiTa.PRESENT_TENSE;
    this.person = RiTa.FIRST_PERSON;
    this.number = RiTa.SINGULAR;
    this.form = RiTa.NORMAL;
  },

  // !@# TODO: add handling of past tense modals.
  conjugate: function(theVerb, args) {

    var v, s, actualModal, conjs = [],
      frontVG, verbForm;

    if (!theVerb || !theVerb.length) return E;

    if (!args) return theVerb;

    // ------------------ handle arguments ------------------

    v = theVerb.toLowerCase();
    if (v === "am" || v === "are" || v === "is" || v === "was" || v === "were") {
      v = "be";
    }
    frontVG = v;

    args.number && (this.number = args.number);
    args.person && (this.person = args.person);
    args.tense && (this.tense = args.tense);
    args.form && (this.form = args.form);
    args.passive && (this.passive = args.passive);
    args.progressive && (this.progressive = args.progressive);
    args.interrogative && (this.interrogative = args.interrogative);
    args.perfect && (this.perfect = args.perfect);

    // ----------------------- start ---------------------------
    if (this.form == RiTa.INFINITIVE) {
      actualModal = "to";
    }

    if (this.tense == RiTa.FUTURE_TENSE) {
      actualModal = "will";
    }

    if (this.passive) {
      conjs.push(this.getPastParticiple(frontVG));
      frontVG = "be";
    }

    if (this.progressive) {
      conjs.push(this.getPresentParticiple(frontVG));
      frontVG = "be";
    }

    if (this.perfect) {
      conjs.push(this.getPastParticiple(frontVG));
      frontVG = "have";
    }

    if (actualModal) {

      // log("push: "+frontVG);
      conjs.push(frontVG);
      frontVG = null;
    }

    // Now inflect frontVG (if it exists) and push it on restVG
    if (frontVG) {

      if (this.form === RiTa.GERUND) { // gerund - use ING form

        var pp = this.getPresentParticiple(frontVG);

        // !@# not yet implemented! ??? WHAT?
        conjs.push(pp);
      } else if (this.interrogative && frontVG != "be" && conjs.length < 1) {

        conjs.push(frontVG);
      } else {

        verbForm = this.getVerbForm(frontVG, this.tense, this.person, this.number);
        conjs.push(verbForm);
      }
    }

    // add modal, and we're done
    actualModal && conjs.push(actualModal);

    s = E;
    for (var i = 0; i < conjs.length; i++) {

      s = conjs[i] + " " + s;
    }

    // !@# test this
    endsWith(s, "peted") && err("Unexpected output: " + this.toString());

    return trim(s);
  },

  checkRules: function(ruleSet, theVerb) {

    var res, name = ruleSet.name,
      rules = ruleSet.rules,
      defRule = ruleSet.defaultRule;

    if (!rules) err("no rule: " + ruleSet.name + ' of ' + theVerb);

    if (inArray(MODALS, theVerb)) return theVerb;

    for (var i = 0; i < rules.length; i++) {

      //log("checkRules2("+name+").fire("+i+")="+rules[i].regex);
      if (rules[i].applies(theVerb)) {

        var got = rules[i].fire(theVerb);

        //log("HIT("+name+").fire("+i+")="+rules[i].regex+"_returns: "+got);
        return got;
      }
    }
    //log("NO HIT!");

    if (ruleSet.doubling && inArray(VERB_CONS_DOUBLING, theVerb)) {

      //log("doDoubling!");
      theVerb = this.doubleFinalConsonant(theVerb);
    }

    res = defRule.fire(theVerb);

    //log("checkRules("+name+").returns: "+res);

    return res;
  },

  doubleFinalConsonant: function(word) {
    var letter = word.charAt(word.length - 1);
    return word + letter;
  },

  getPast: function(theVerb, pers, numb) {

    if (theVerb.toLowerCase() == "be") {

      switch (numb) {

        case RiTa.SINGULAR:

          switch (pers) {

            case RiTa.FIRST_PERSON:
              break;

            case RiTa.THIRD_PERSON:
              return "was";

            case RiTa.SECOND_PERSON:
              return "were";

          }
          break;

        case RiTa.PLURAL:

          return "were";
      }
    }

    var got = this.checkRules(PAST_TENSE_RULESET, theVerb);

    return got;
  },

  getPresent: function(theVerb, person, number) {

    person = person || this.person;
    number = number || this.number;

    if ((person == RiTa.THIRD_PERSON) && (number == RiTa.SINGULAR)) {

      return this.checkRules(PRESENT_TENSE_RULESET, theVerb);
    } else if (theVerb == "be") {

      if (number == RiTa.SINGULAR) {

        switch (person) {

          case RiTa.FIRST_PERSON:
            return "am";

          case RiTa.SECOND_PERSON:
            return "are";

          case RiTa.THIRD_PERSON:
            return "is";

            // default: ???
        }

      } else {
        return "are";
      }
    }
    return theVerb;
  },

  getPresentParticiple: function(theVerb) {

    return strOk(theVerb) ? this.checkRules(PRESENT_PARTICIPLE_RULESET, theVerb) : E;
  },

  getPastParticiple: function(theVerb) {

    var res = strOk(theVerb) ? this.checkRules(PAST_PARTICIPLE_RULESET, theVerb) : E;
    return res;
  },

  getVerbForm: function(theVerb, tense, person, number) {

    switch (tense) {

      case RiTa.PRESENT_TENSE:
        return this.getPresent(theVerb, person, number);

      case RiTa.PAST_TENSE:
        return this.getPast(theVerb, person, number);

      default:
        return theVerb;
    }
  },

  toString: function() {
    return "  ---------------------" + BN + "  Passive = " + this.passive +
      BN + "  Perfect = " + this.perfect + BN + "  Progressive = " +
      this.progressive + BN + "  ---------------------" + BN + "  Number = " +
      this.number + BN + "  Person = " + this.person + BN + "  Tense = " +
      this.tense + BN + "  ---------------------" + BN;
  }
};

// ////////////////////////////////////////////////////////////
// PosTagger  (singleton)
// ////////////////////////////////////////////////////////////
var PosTagger = {

  // Penn Pos types ------------------------------ (40+UKNOWN)

  UNKNOWN: ['???', 'unknown'],
  N: ['n', 'NOUN_KEY'],
  V: ['v', 'VERB_KEY'],
  R: ['r', 'ADVERB_KEY'],
  A: ['a', 'ADJECTIVE_KEY'],
  CC: ['cc', 'Coordinating conjunction'],
  CD: ['cd', 'Cardinal number'],
  DT: ['dt', 'Determiner'],
  EX: ['ex', 'Existential there'],
  FW: ['fw', 'Foreign word'],
  IN: ['in', 'Preposition or subordinating conjunction'],
  JJ: ['jj', 'Adjective'],
  JJR: ['jjr', 'Adjective, comparative'],
  JJS: ['jjs', 'Adjective, superlative'],
  LS: ['ls', 'List item marker'],
  MD: ['md', 'Modal'],
  NN: ['nn', 'Noun, singular or mass'],
  NNS: ['nns', 'Noun, plural'],
  NNP: ['nnp', 'Proper noun, singular'],
  NNPS: ['nnps', 'Proper noun, plural'],
  PDT: ['pdt', 'Predeterminer'],
  POS: ['pos', 'Possessive ending'],
  PRP: ['prp', 'Personal pronoun'],
  PRP$: ['prp$', 'Possessive pronoun (prolog version PRP-S)'],
  RB: ['rb', 'Adverb'],
  RBR: ['rbr', 'Adverb, comparative'],
  RBS: ['rbs', 'Adverb, superlative'],
  RP: ['rp', 'Particle'],
  SYM: ['sym', 'Symbol'],
  TO: ['to', 'to'],
  UH: ['uh', 'Interjection'],
  VB: ['vb', 'Verb, base form'],
  VBD: ['vbd', 'Verb, past tense'],
  VBG: ['vbg', 'Verb, gerund or present participle'],
  VBN: ['vbn', 'Verb, past participle'],
  VBP: ['vbp', 'Verb, non-3rd person singular present'],
  VBZ: ['vbz', 'Verb, 3rd person singular present'],
  WDT: ['wdt', 'Wh-determiner'],
  WP: ['wp', 'Wh-pronoun'],
  WP$: ['wp$', 'Possessive wh-pronoun (prolog version WP-S)'],
  WRB: ['wrb', 'Wh-adverb'],

  TAGS: ['cc', 'cd', 'dt', 'ex', 'fw', 'in', 'jj',
    'jjr', 'jjs', 'ls', 'md', 'nn', 'nns', 'nnp',
    'nnps', 'pdt', 'pos', 'prp', 'prp$', 'rb',
    'rbr', 'rbs', 'rp', 'sym', 'to',
    'uh', 'vb', 'vbd', 'vbg', 'vbn', 'vbp', 'vbz', 'wdt',
    'wp', 'wp$', 'wrb', 'unknown'
  ],

  NOUNS: ['nn', 'nns', 'nnp', 'nnps'],
  VERBS: ['vb', 'vbd', 'vbg', 'vbn', 'vbp', 'vbz'],
  ADJ: ['jj', 'jjr', 'jjs'],
  ADV: ['rb', 'rbr', 'rbs', 'rp'],
  NOLEX_WARNED: false,

  isVerb: function(tag) {
    return inArray(this.VERBS, tag);
  },

  isNoun: function(tag) {
    return inArray(this.NOUNS, tag);
  },

  isAdverb: function(tag) {
    return inArray(this.ADV, tag);
  },

  isAdj: function(tag) {
    return inArray(this.ADJ, tag);
  },

  isTag: function(tag) {
    return inArray(this.TAGS, tag);
  },

  hasTag: function(choices, tag) {
    ok(choices, A);
    var choiceStr = choices.join();
    return (choiceStr.indexOf(tag) > -1);
  },

  // Returns an array of parts-of-speech from the Penn tagset,
  // each corresponding to one word of input
  tag: function(words) {

    var result = [], choices2d = [], lex;

    if (RiLexicon.enabled) {
      lex = RiTa._lexicon();
    }
    else if (!RiTa.SILENT && !this.NOLEX_WARNED){
      this.NOLEX_WARNED = true;
      console.warn('No RiLexicon found: ' +
        'part-of-speech tagging will be inaccurate!');
    }

    words = is(words, A) ? words : [words];

    for (var i = 0, l = words.length; i < l; i++) {

      /*if (!words[i]) {
				choices2d[i] = [];
				continue;
			}*/

      if (words[i].length < 1) {

        result.push(E);
        continue;
      }

      if (words[i].length == 1) {

        result.push(this._handleSingleLetter(words[i]));
        continue;
      }

      var data = lex && lex._getPosArr(words[i]);

      if (!data || !data.length) {

        choices2d[i] = [];
        result.push((endsWith(words[i], 's') ? 'nns' : 'nn'));

      } else {

        result.push(data[0]);
        choices2d[i] = data;
      }
    }

    // Adjust pos according to transformation rules
    return this._applyContext(words, result, choices2d);
  },

  _handleSingleLetter: function(c) {

    var result = c;

    if (c === 'a' || c === 'A')
      result = "dt";
    else if (c === 'I')
      result = "prp";
    else if (c >= '0' && c <= '9')
      result = "cd";

    //System.out.println("handleSingleLetter("+word+") :: "+result);

    return result;
  },

  // Applies a customized subset of the Brill transformations
  _applyContext: function(words, result, choices) {

    //log("_applyContext("+words+","+result+","+choices+")");

    var sW = startsWith, eW = endsWith,
      PRINT_CUSTOM_TAGS = (0 && !RiTa.SILENT);

    // Apply transformations
    for (var i = 0, l = words.length; i < l; i++) {

      // transform 1: DT, {VBD | VBP | VB} --> DT, NN
      if (i > 0 && (result[i - 1] == "dt")) {

        if (sW(result[i], "vb")) {
          if (PRINT_CUSTOM_TAGS) {
            log("PosTagger: changing verb to noun: " + words[i]);
          }
          result[i] = "nn";
        }

        // transform 1: DT, {RB | RBR | RBS} --> DT, {JJ |
        // JJR | JJS}
        else if (sW(result[i], "rb")) {

          if (PRINT_CUSTOM_TAGS)
            log("PosTagger: custom tagged '" + words[i] + "', " + result[i]);
          result[i] = (result[i].length > 2) ? "jj" + result[i].charAt(2) : "jj";
          if (PRINT_CUSTOM_TAGS) {
            log(" -> " + result[i]);
          }
        }
      }

      // transform 2: convert a noun to a number (cd) if it is
      // all digits and/or a decimal "."
      if (sW(result[i], "n") && !choices[i]) {
        if (isNum(words[i])) {
          result[i] = "cd";
        } // mods: dch (add choice check above) <---- ? >
      }

      // transform 3: convert a noun to a past participle if
      // words[i] ends with "ed"
      if (sW(result[i], "n") && eW(words[i], "ed")) {
        result[i] = "vbn";
      }

      // transform 4: convert any type to adverb if it ends in "ly";
      if (eW(words[i], "ly")) {
        result[i] = "rb";
      }

      // transform 5: convert a common noun (NN or NNS) to a
      // adjective if it ends with "al", special-case for mammal
      if (sW(result[i], "nn") && eW(words[i], "al") && words[i] != 'mammal') {
        result[i] = "jj";
      }

      // transform 6: convert a noun to a verb if the
      // preceeding word is "would"
      if (i > 0 && sW(result[i], "nn") && equalsIgnoreCase(words[i - 1], "would")) {
        result[i] = "vb";
      }

      // transform 7: if a word has been categorized as a
      // common noun and it ends with "s", then set its type to plural common noun (NNS)
      if ((result[i] == "nn") && words[i].match(/^.*[^s]s$/)) {
        if (!NULL_PLURALS.applies(words[i]))
          result[i] = "nns";
      }

      // transform 8: convert a common noun to a present
      // participle verb (i.e., a gerund)
      if (sW(result[i], "nn") && eW(words[i], "ing")) {
        // DH: fixed here -- add check on choices for any verb: eg. // 'morning'
        if (this.hasTag(choices[i], "vb")) {
          result[i] = "vbg";
        } else if (PRINT_CUSTOM_TAGS) {
          log("[RiTa] PosTagger tagged '" + words[i] + "' as " + result[i]);
        }
      }

      // transform 9(dch): convert plural nouns (which are also 3sg-verbs) to
      // 3sg-verbs when following a singular noun (the dog dances, Dave dances, he dances)
      if (i > 0 && result[i] == "nns" && this.hasTag(choices[i], "vbz") && result[i - 1].match(/^(nn|prp|nnp)$/)) {
        result[i] = "vbz";
      }

      // transform 10(dch): convert common nouns to proper
      // nouns when they start w' a capital and (?are not a
      // sentence start?)
      if ( /*i > 0 && */ sW(result[i], "nn") && (words[i].charAt(0) == words[i].charAt(0).toUpperCase())) {
        result[i] = eW(result[i], "s") ? "nnps" : "nnp";
      }

      // DISABLED: transform 10(dch): convert plural nouns (which are
      // also 3sg-verbs) to 3sg-verbs when followed by adverb
      /*if (i < result.length - 1 && result[i] == "nns" && sW(result[i + 1], "rb")
					&& this.hasTag(choices[i], "vbz")) {
				result[i] = "vbz";
			}*/
    }

    return result;
  }
}; // end PosTagger

// Stemming demo/comparison - http://text-processing.com/demo/stem/

/*
 *  Porter stemmer in Javascript: from https://github.com/kristopolous/Porter-Stemmer
 *  Ported from Porter, 1980, An algorithm for suffix stripping, Program, Vol. 14,
 *  no. 3, pp 130-137, see also http:www.tartarus.org/~martin/PorterStemmer
 *
 *  Porter is default Stemmer: for Lancaster or Pling, include the specific file
 */
RiTa.stem_Porter = (function() {

  var step2list = {
      'ational': 'ate',
      'tional': 'tion',
      'enci': 'ence',
      'anci': 'ance',
      'izer': 'ize',
      'bli': 'ble',
      'alli': 'al',
      'entli': 'ent',
      'eli': 'e',
      'ousli': 'ous',
      'ization': 'ize',
      'ation': 'ate',
      'ator': 'ate',
      'alism': 'al',
      'iveness': 'ive',
      'fulness': 'ful',
      'ousness': 'ous',
      'aliti': 'al',
      'iviti': 'ive',
      'biliti': 'ble',
      'logi': 'log'
    },

    step3list = {
      'icate': 'ic',
      'ative': '',
      'alize': 'al',
      'iciti': 'ic',
      'ical': 'ic',
      'ful': '',
      'ness': ''
    },

    c = '[^aeiou]', // consonant
    v = '[aeiouy]', // vowel
    C = c + '[^aeiouy]*', // consonant sequence
    V = v + '[aeiou]*', // vowel sequence

    mgr0 = '^(' + C + ')?' + V + C, // [C]VC... is m>0
    meq1 = '^(' + C + ')?' + V + C + '(' + V + ')?$', // [C]VC[V] is m=1
    mgr1 = '^(' + C + ')?' + V + C + V + C, // [C]VCVC... is m>1
    s_v = '^(' + C + ')?' + v; // vowel in stem

  return function(w) {

    var fp, stem, suffix, firstch, re, re2, re3, re4, origword = w;

    if (w.length < 3) {
      return w;
    }

    firstch = w.substr(0, 1);
    if (firstch == "y") {
      w = firstch + w.substr(1);
    }

    // Step 1a
    re = /^(.+?)(ss|i)es$/;
    re2 = /^(.+?)([^s])s$/;

    if (re.test(w)) {
      w = w.replace(re, "$1$2");
    } else if (re2.test(w)) {
      w = w.replace(re2, "$1$2");
    }

    // Step 1b
    re = /^(.+?)eed$/;
    re2 = /^(.+?)(ed|ing)$/;
    if (re.test(w)) {
      fp = re.exec(w);
      re = new RegExp(mgr0);
      if (re.test(fp[1])) {
        re = /.$/;
        w = w.replace(re,E);
      }
    } else if (re2.test(w)) {
      fp = re2.exec(w);
      stem = fp[1];
      re2 = new RegExp(s_v);
      if (re2.test(stem)) {
        w = stem;
        re2 = /(at|bl|iz)$/;
        re3 = new RegExp("([^aeiouylsz])\\1$");
        re4 = new RegExp("^" + C + v + "[^aeiouwxy]$");
        if (re2.test(w)) {
          w = w + "e";
        } else if (re3.test(w)) {
          re = /.$/;
          w = w.replace(re,E);
        } else if (re4.test(w)) {
          w = w + "e";
        }
      }
    }

    // Step 1c
    re = /^(.+?)y$/;
    if (re.test(w)) {
      fp = re.exec(w);
      stem = fp[1];
      re = new RegExp(s_v);
      if (re.test(stem)) w = stem + "i";
    }

    // Step 2
    re = /^(.+?)(ational|tional|enci|anci|izer|bli|alli|entli|eli|ousli|ization|ation|ator|alism|iveness|fulness|ousness|aliti|iviti|biliti|logi)$/;
    if (re.test(w)) {
      fp = re.exec(w);
      stem = fp[1];
      suffix = fp[2];
      re = new RegExp(mgr0);
      if (re.test(stem)) {
        w = stem + step2list[suffix];
      }
    }

    // Step 3
    re = /^(.+?)(icate|ative|alize|iciti|ical|ful|ness)$/;
    if (re.test(w)) {
      fp = re.exec(w);
      stem = fp[1];
      suffix = fp[2];
      re = new RegExp(mgr0);
      if (re.test(stem)) {
        w = stem + step3list[suffix];
      }
    }

    // Step 4
    re = /^(.+?)(al|ance|ence|er|ic|able|ible|ant|ement|ment|ent|ou|ism|ate|iti|ous|ive|ize)$/;
    re2 = /^(.+?)(s|t)(ion)$/;
    if (re.test(w)) {
      fp = re.exec(w);
      stem = fp[1];
      re = new RegExp(mgr1);
      if (re.test(stem)) {
        w = stem;
      }
    } else if (re2.test(w)) {
      fp = re2.exec(w);
      stem = fp[1] + fp[2];
      re2 = new RegExp(mgr1);
      if (re2.test(stem)) {
        w = stem;
      }
    }

    // Step 5
    re = /^(.+?)e$/;
    if (re.test(w)) {
      fp = re.exec(w);
      stem = fp[1];
      re = new RegExp(mgr1);
      re2 = new RegExp(meq1);
      re3 = new RegExp("^" + C + v + "[^aeiouwxy]$");
      if (re.test(stem) || (re2.test(stem) && !(re3.test(stem)))) {
        w = stem;
      }
    }

    re = /ll$/;
    re2 = new RegExp(mgr1);
    if (re.test(w) && re2.test(w)) {
      re = /.$/;
      w = w.replace(re,E);
    }

    // and turn initial Y back to y

    (firstch == "y") && (w = firstch.toLowerCase() + w.substr(1));

    return w;
  };

})();

/* From the PlingStemmer impl in the Java Tools package (see http://mpii.de/yago-naga/javatools). */
RiTa.stem_Pling = (function() {

  /* Words that are both singular and plural */
  var categorySP = ['acoustics', 'aestetics', 'aquatics', 'basics', 'ceramics', 'classics', 'cosmetics', 'dermatoglyphics', 'dialectics', 'deer', 'dynamics', 'esthetics', 'ethics', 'harmonics', 'heroics', 'isometrics', 'mechanics', 'metrics', 'statistics', 'optic', 'people', 'physics', 'polemics', 'propaedeutics', 'pyrotechnics', 'quadratics', 'quarters', 'statistics', 'tactics', 'tropics'];

  /* Words that end in '-se' in their plural forms (like 'nurse' etc.) */
  var categorySE_SES = ['nurses', 'cruises'];

  /* Words that do not have a distinct plural form (like 'atlas' etc.) */
  var category00 = ['alias', 'asbestos', 'atlas', 'barracks', 'bathos', 'bias', 'breeches', 'britches', 'canvas', 'chaos', 'clippers', 'contretemps', 'corps', 'cosmos', 'crossroads', 'diabetes', 'ethos', 'gallows', 'gas', 'graffiti', 'headquarters', 'herpes', 'high-jinks', 'innings', 'jackanapes', 'lens', 'means', 'measles', 'mews', 'mumps', 'news', 'pathos', 'pincers', 'pliers', 'proceedings', 'rabies', 'rhinoceros', 'sassafras', 'scissors', 'series', 'shears', 'species', 'tuna'];

  /* Words that change from '-um' to '-a' (like 'curriculum' etc.), listed in their plural forms */
  var categoryUM_A = ['addenda', 'agenda', 'aquaria', 'bacteria', 'candelabra', 'compendia', 'consortia', 'crania', 'curricula', 'data', 'desiderata', 'dicta', 'emporia', 'enconia', 'errata', 'extrema', 'gymnasia', 'honoraria', 'interregna', 'lustra', 'maxima', 'media', 'memoranda', 'millenia', 'minima', 'momenta', 'optima', 'ova', 'phyla', 'quanta', 'rostra', 'spectra', 'specula', 'stadia', 'strata', 'symposia', 'trapezia', 'ultimata', 'vacua', 'vela'];

  /* Words that change from '-on' to '-a' (like 'phenomenon' etc.), listed in their plural forms */
  var categoryON_A = ['aphelia', 'asyndeta', 'automata', 'criteria', 'hyperbata', 'noumena', 'organa', 'perihelia', 'phenomena', 'prolegomena'];

  /* Words that change from '-o' to '-i' (like 'libretto' etc.), listed in their plural forms */
  var categoryO_I = ['alti', 'bassi', 'canti', 'contralti', 'crescendi', 'libretti', 'soli', 'soprani', 'tempi', 'virtuosi'];

  /*  Words that change from '-us' to '-i' (like 'fungus' etc.), listed in their plural forms		 */
  var categoryUS_I = ['alumni', 'bacilli', 'cacti', 'foci', 'fungi', 'genii', 'hippopotami', 'incubi', 'nimbi', 'nuclei', 'nucleoli', 'octopi', 'radii', 'stimuli', 'styli', 'succubi', 'syllabi', 'termini', 'tori', 'umbilici', 'uteri'];

  /* Words that change from '-ix' to '-ices' (like 'appendix' etc.), listed in their plural forms */
  var categoryIX_ICES = ['appendices', 'cervices'];

  /* Words that change from '-is' to '-es' (like 'axis' etc.), listed in their plural forms, plus everybody ending in theses */
  var categoryIS_ES = ['analyses', 'axes', 'bases', 'crises', 'diagnoses', 'ellipses', 'em_PHASEs', 'neuroses', 'oases', 'paralyses', 'synopses'];

  /* Words that change from '-oe' to '-oes' (like 'toe' etc.), listed in their plural forms*/
  var categoryOE_OES = ['aloes', 'backhoes', 'beroes', 'canoes', 'chigoes', 'cohoes', 'does', 'felloes', 'floes', 'foes', 'gumshoes', 'hammertoes', 'hoes', 'hoopoes', 'horseshoes', 'leucothoes', 'mahoes', 'mistletoes', 'oboes', 'overshoes', 'pahoehoes', 'pekoes', 'roes', 'shoes', 'sloes', 'snowshoes', 'throes', 'tic-tac-toes', 'tick-tack-toes', 'ticktacktoes', 'tiptoes', 'tit-tat-toes', 'toes', 'toetoes', 'tuckahoes', 'woes'];

  /* Words that change from '-ex' to '-ices' (like 'index' etc.), listed in their plural forms*/
  var categoryEX_ICES = ['apices', 'codices', 'cortices', 'indices', 'latices', 'murices', 'pontifices', 'silices', 'simplices', 'vertices', 'vortices'];

  /* Words that change from '-u' to '-us' (like 'emu' etc.), listed in their plural forms*/
  var categoryU_US = [ 'menus', 'gurus', 'apercus', 'barbus', 'cornus', 'ecrus', 'emus', 'fondus', 'gnus', 'iglus', 'mus', 'nandus', 'napus', 'poilus', 'quipus', 'snafus', 'tabus', 'tamandus', 'tatus', 'timucus', 'tiramisus', 'tofus', 'tutus'];

  /* Words that change from '-sse' to '-sses' (like 'finesse' etc.), listed in their plural forms,plus those ending in mousse*/
  var categorySSE_SSES = ['bouillabaisses', 'coulisses', 'crevasses', 'crosses', 'cuisses', 'demitasses', 'ecrevisses', 'fesses', 'finesses', 'fosses', 'impasses', 'lacrosses', 'largesses', 'masses', 'noblesses', 'palliasses', 'pelisses', 'politesses', 'posses', 'tasses', 'wrasses'];

  /* Words that change from '-che' to '-ches' (like 'brioche' etc.), listed in their plural forms*/
  var categoryCHE_CHES = ['adrenarches', 'attaches', 'avalanches', 'barouches', 'brioches', 'caches', 'caleches', 'caroches', 'cartouches', 'cliches', 'cloches', 'creches', 'demarches', 'douches', 'gouaches', 'guilloches', 'headaches', 'heartaches', 'huaraches', 'menarches', 'microfiches', 'moustaches', 'mustaches', 'niches', 'panaches', 'panoches', 'pastiches', 'penuches', 'pinches', 'postiches', 'psyches', 'quiches', 'schottisches', 'seiches', 'soutaches', 'synecdoches', 'thelarches', 'troches'];

  /* Words that end with '-ics' and do not exist as nouns without the 's' (like 'aerobics' etc.)*/
  var categoryICS = ['aerobatics', 'aerobics', 'aerodynamics', 'aeromechanics', 'aeronautics', 'alphanumerics', 'animatronics', 'apologetics', 'architectonics', 'astrodynamics', 'astronautics', 'astrophysics', 'athletics', 'atmospherics', 'autogenics', 'avionics', 'ballistics', 'bibliotics', 'bioethics', 'biometrics', 'bionics', 'bionomics', 'biophysics', 'biosystematics', 'cacogenics', 'calisthenics', 'callisthenics', 'catoptrics', 'civics', 'cladistics', 'cryogenics', 'cryonics', 'cryptanalytics', 'cybernetics', 'cytoarchitectonics', 'cytogenetics', 'diagnostics', 'dietetics', 'dramatics', 'dysgenics', 'econometrics', 'economics', 'electromagnetics', 'electronics', 'electrostatics', 'endodontics', 'enterics', 'ergonomics', 'eugenics', 'eurhythmics', 'eurythmics', 'exodontics', 'fibreoptics', 'futuristics', 'genetics', 'genomics', 'geographics', 'geophysics', 'geopolitics', 'geriatrics', 'glyptics', 'graphics', 'gymnastics', 'hermeneutics', 'histrionics', 'homiletics', 'hydraulics', 'hydrodynamics', 'hydrokinetics', 'hydroponics', 'hydrostatics', 'hygienics', 'informatics', 'kinematics', 'kinesthetics', 'kinetics', 'lexicostatistics', 'linguistics', 'lithoglyptics', 'liturgics', 'logistics', 'macrobiotics', 'macroeconomics', 'magnetics', 'magnetohydrodynamics', 'mathematics', 'metamathematics', 'metaphysics', 'microeconomics', 'microelectronics', 'mnemonics', 'morphophonemics', 'neuroethics', 'neurolinguistics', 'nucleonics', 'numismatics', 'obstetrics', 'onomastics', 'orthodontics', 'orthopaedics', 'orthopedics', 'orthoptics', 'paediatrics', 'patristics', 'patristics', 'pedagogics', 'pediatrics', 'periodontics', 'pharmaceutics', 'pharmacogenetics', 'pharmacokinetics', 'phonemics', 'phonetics', 'phonics', 'photomechanics', 'physiatrics', 'pneumatics', 'poetics', 'politics', 'pragmatics', 'prosthetics', 'prosthodontics', 'proteomics', 'proxemics', 'psycholinguistics', 'psychometrics', 'psychonomics', 'psychophysics', 'psychotherapeutics', 'robotics', 'semantics', 'semiotics', 'semitropics', 'sociolinguistics', 'stemmatics', 'strategics', 'subtropics', 'systematics', 'tectonics', 'telerobotics', 'therapeutics', 'thermionics', 'thermodynamics', 'thermostatics'];

  /* Words that change from '-ie' to '-ies' (like 'auntie' etc.), listed in their plural forms*/
  var categoryIE_IES = ['aeries', 'anomies', 'aunties', 'baddies', 'beanies', 'birdies', 'boccies', 'bogies', 'bolshies', 'bombies', 'bonhomies', 'bonxies', 'booboisies', 'boogies', 'boogie-woogies', 'bookies', 'booties', 'bosies', 'bourgeoisies', 'brasseries', 'brassies', 'brownies', 'budgies', 'byrnies', 'caddies', 'calories', 'camaraderies', 'capercaillies', 'capercailzies', 'cassies', 'catties', 'causeries', 'charcuteries', 'chinoiseries', 'collies', 'commies', 'cookies', 'coolies', 'coonties', 'cooties', 'corries', 'coteries', 'cowpies', 'cowries', 'cozies', 'crappies', 'crossties', 'curies', 'dachsies', 'darkies', 'dassies', 'dearies', 'dickies', 'dies', 'dixies', 'doggies', 'dogies', 'dominies', 'dovekies', 'eyries', 'faeries', 'falsies', 'floozies', 'folies', 'foodies', 'freebies', 'gaucheries', 'gendarmeries', 'genies', 'ghillies', 'gillies', 'goalies', 'goonies', 'grannies', 'grotesqueries', 'groupies', 'hankies', 'hippies', 'hoagies', 'honkies', 'hymies', 'indies', 'junkies', 'kelpies', 'kilocalories', 'knobkerries', 'koppies', 'kylies', 'laddies', 'lassies', 'lies', 'lingeries', 'magpies', 'magpies', 'marqueteries', 'mashies', 'mealies', 'meanies', 'menageries', 'millicuries', 'mollies', 'facts1', 'moxies', 'neckties', 'newbies', 'nighties', 'nookies', 'oldies', 'organdies', 'panties', 'parqueteries', 'passementeries', 'patisseries', 'pies', 'pinkies', 'pixies', 'porkpies', 'potpies', 'prairies', 'preemies', 'premies', 'punkies', 'pyxies', 'quickies', 'ramies', 'reveries', 'rookies', 'rotisseries', 'scrapies', 'sharpies', 'smoothies', 'softies', 'stoolies', 'stymies', 'swaggies', 'sweeties', 'talkies', 'techies', 'ties', 'tooshies', 'toughies', 'townies', 'veggies', 'walkie-talkies', 'wedgies', 'weenies', 'weirdies', 'yardies', 'yuppies', 'zombies'];

  /* Maps irregular Germanic English plural nouns to their singular form */
  var categoryIRR = [ 'blondes', 'blonde', 'beefs', 'beef', 'beeves', 'beef', 'brethren', 'brother', 'busses', 'bus', 'cattle', 'cattlebeast', 'children', 'child', 'corpora', 'corpus', 'ephemerides', 'ephemeris', 'genera', 'genus', 'genies', 'genie', 'genii', 'genie', 'kine', 'cow', 'lice', 'louse', 'mice', 'mouse', 'mongooses', 'mongoose', 'monies', 'money', 'mythoi', 'mythos', 'octopodes', 'octopus',  'oxen', 'ox', 'people', 'person', 'soliloquies', 'soliloquy', 'taxis', 'taxi', 'throes', 'throes', 'trilbys', 'trilby', 'innings', 'inning', 'alibis', 'alibi', 'skis', 'ski' ];

  /* Tells whether a noun is plural. */
  function isPlural(s) {
    return s !== stem(s);
  }

  /*
	 * Tells whether a word form is singular. Note that a word can be both
	 * plural and singular
	 */
  function isSingular(s) {
    return (categorySP._arrayContains(s.toLowerCase()) || !isPlural(s));
  }

  /*
	 * Tells whether a word form is the singular form of one word and at the
	 * same time the plural form of another.
	 */
  function isSingularAndPlural(s) {
    return (categorySP._arrayContains(s.toLowerCase()));
  }

  /*
	 * Cuts a suffix from a string (that is the number of chars given by the
	 * suffix)
	 */
  function cut(s, suffix) {
    return (s.substring(0, s.length - suffix.length));
  }

  /* Returns true if a word is probably not Latin */
  function noLatin(s) {
    return (s.indexOf('h') > 0 || s.indexOf('j') > 0 || s.indexOf('k') > 0 || s.indexOf('w') > 0 || s.indexOf('y') > 0 || s.indexOf('z') > 0 || s.indexOf("ou") > 0 || s.indexOf("sh") > 0 || s.indexOf("ch") > 0 || s._endsWith("aus"));
  }

  /* Returns true if a word is probably Greek */
  function greek(s) {
    return (s.indexOf("ph") > 0 || s.indexOf('y') > 0 && s._endsWith("nges"));
  }

  function stem(s) {

    // Handle irregular ones
    var irreg = categoryIRR[s];

    if (irreg) return (irreg);

    // -on to -a
    if (categoryON_A._arrayContains(s))
      return (cut(s, "a") + "on");

    // -um to -a
    if (categoryUM_A._arrayContains(s))
      return (cut(s, "a") + "um");

    // -x to -ices
    if (categoryIX_ICES._arrayContains(s))
      return (cut(s, "ices") + "ix");

    // -o to -i
    if (categoryO_I._arrayContains(s))
      return (cut(s, "i") + "o");

    // -se to ses
    if (categorySE_SES._arrayContains(s))
      return (cut(s, "s"));

    // -is to -es
    if (categoryIS_ES._arrayContains(s) || s._endsWith("theses"))
      return (cut(s, "es") + "is");

    // -us to -i
    if (categoryUS_I._arrayContains(s))
      return (cut(s, "i") + "us");

    //Wrong plural
    if (s._endsWith("uses") && (categoryUS_I._arrayContains(cut(s, "uses") + "i") || s === ("genuses") || s === ("corpuses")))
      return (cut(s, "es"));

    // -ex to -ices
    if (categoryEX_ICES._arrayContains(s))
      return (cut(s, "ices") + "ex");

    // Words that do not inflect in the plural
    if (s._endsWith("ois") || s._endsWith("itis") || category00._arrayContains(s) || categoryICS._arrayContains(s))
      return (s);

    // -en to -ina
    // No other common words end in -ina
    if (s._endsWith("ina"))
      return (cut(s, "en"));

    // -a to -ae
    // No other common words end in -ae
    if (s._endsWith("ae"))
      return (cut(s, "e"));

    // -a to -ata
    // No other common words end in -ata
    if (s._endsWith("ata"))
      return (cut(s, "ta"));

    // trix to -trices
    // No common word ends with -trice(s)
    if (s._endsWith("trices"))
      return (cut(s, "trices") + "trix");

    // -us to -us
    //No other common word ends in -us, except for false plurals of French words
    //Catch words that are not latin or known to end in -u
    if (s._endsWith("us") && !s._endsWith("eaus") && !s._endsWith("ieus") && !noLatin(s) && !categoryU_US._arrayContains(s))
      return (s);

    // -tooth to -teeth
    // -goose to -geese
    // -foot to -feet
    // -zoon to -zoa
    //No other common words end with the indicated suffixes
    if (s._endsWith("teeth"))
      return (cut(s, "teeth") + "tooth");
    if (s._endsWith("geese"))
      return (cut(s, "geese") + "goose");
    if (s._endsWith("feet"))
      return (cut(s, "feet") + "foot");
    if (s._endsWith("zoa"))
      return (cut(s, "zoa") + "zoon");

    // -men to -man
    // -firemen to -fireman
    if(s._endsWith("men")) return (cut(s,"men")+"man");

    // -martinis to -martini
    // -bikinis to -bikini
    if(s._endsWith("inis")) return (cut(s,"inis")+"ini");

    // -children to -child
    // -schoolchildren to -schoolchild
    if(s._endsWith("children")) return (cut(s,"ren"));

    // -eau to -eaux
    //No other common words end in eaux
    if (s._endsWith("eaux"))
      return (cut(s, "x"));

    // -ieu to -ieux
    //No other common words end in ieux
    if (s._endsWith("ieux"))
      return (cut(s, "x"));

    // -nx to -nges
    // Pay attention not to kill words ending in -nge with plural -nges
    // Take only Greek words (works fine, only a handfull of exceptions)
    if (s._endsWith("nges") && greek(s))
      return (cut(s, "nges") + "nx");

    // -[sc]h to -[sc]hes
    //No other common word ends with "shes", "ches" or "she(s)"
    //Quite a lot end with "che(s)", filter them out
    if (s._endsWith("shes") || s._endsWith("ches") && !categoryCHE_CHES._arrayContains(s))
      return (cut(s, "es"));

    // -ss to -sses
    // No other common singular word ends with "sses"
    // Filter out those ending in "sse(s)"
    if (s._endsWith("sses") && !categorySSE_SSES._arrayContains(s) && !s._endsWith("mousses"))
      return (cut(s, "es"));

    // -x to -xes
    // No other common word ends with "xe(s)" except for "axe"
    if (s._endsWith("xes") && s !== "axes")
      return (cut(s, "es"));

    // -[nlw]ife to -[nlw]ives
    //No other common word ends with "[nlw]ive(s)" except for olive
    if (s._endsWith("nives") || s._endsWith("lives") && !s._endsWith("olives") || s._endsWith("wives"))
      return (cut(s, "ves") + "fe");

    // -[aeo]lf to -ves  exceptions: valve, solve
    // -[^d]eaf to -ves  exceptions: heave, weave
    // -arf to -ves      no exception
    if (s._endsWith("alves") && !s._endsWith("valves") || s._endsWith("olves") && !s._endsWith("solves") || s._endsWith("eaves") && !s._endsWith("heaves") && !s._endsWith("weaves") || s._endsWith("arves"))
      return (cut(s, "ves") + "f");

    // -y to -ies
    // -ies is very uncommon as a singular suffix
    // but -ie is quite common, filter them out
    if (s._endsWith("ies") && !categoryIE_IES._arrayContains(s))
      return (cut(s, "ies") + "y");

    // -o to -oes
    // Some words end with -oe, so don't kill the "e"
    if (s._endsWith("oes") && !categoryOE_OES._arrayContains(s))
      return (cut(s, "es"));

    // -s to -ses
    // -z to -zes
    // no words end with "-ses" or "-zes" in singular
    if (s._endsWith("ses") || s._endsWith("zes"))
      return (cut(s, "es"));

    // - to -s
    if (s._endsWith("s") && !s._endsWith("ss") && !s._endsWith("is"))
      return (cut(s, "s"));

    return (s);
  }

  return function(token) {

    return stem(token.toLowerCase());
  };

})();

Array.prototype._arrayContains = function(ele) {
  return (Array.prototype.indexOf(ele) > -1);
};

String.prototype._endsWith = function(suffix) {
  return this.indexOf(suffix, this.length - suffix.length) !== -1;
};

/*
 * Minimum-Edit-Distance (or Levenshtein distance) is a measure of the similarity
 * between two strings, the source string and the target string (t). The distance
 * is the number of deletions, insertions, or substitutions required to transform
 * the source into the target / avg_string_length<p>
 *
 * Adapted from Michael Gilleland's algorithm
 */
var MinEditDist = {

  _min3: function(a, b, c) {

    var min = a;
    if (b < min) min = b;
    if (c < min) min = c;
    return min;
  },

  /* Computes min-edit-distance between 2 string arrays where each array element either matches or does not */
  _computeRawArray: function(srcArr, trgArr) {

    //log((srcArr)+" "+(trgArr));

    var matrix = []; // matrix
    var sI; // ith element of s
    var tJ; // jth element of t
    var cost; // cost
    var i, j, sl, tl;

    // Step 1 ----------------------------------------------

    if (!srcArr.length) return trgArr.length;

    if (!trgArr.length) return srcArr.length;

    // Step 2 ----------------------------------------------

    for (i = 0, sl = srcArr.length; i <= sl; i++) {

      matrix[i] = [];
      matrix[i][0] = i;
    }

    for (j = 0, tl = trgArr.length; j <= tl; j++)
      matrix[0][j] = j;

    // Step 3 ----------------------------------------------

    for (i = 1, sl = srcArr.length; i <= sl; i++) {

      sI = srcArr[i - 1];

      // Step 4 --------------------------------------------

      for (j = 1, tl = trgArr.length; j <= tl; j++) {

        tJ = trgArr[j - 1];

        // Step 5 ------------------------------------------

        cost = (sI === tJ) ? 0 : 1;

        // Step 6 ------------------------------------------

        matrix[i][j] = this._min3(
          matrix[i - 1][j] + 1,
          matrix[i][j - 1] + 1,
          matrix[i - 1][j - 1] + cost);
      }
    }

    // Step 7 ----------------------------------------------

    return matrix[srcArr.length][trgArr.length];
  },


  /*
   * Compute min-edit-distance between 2 strings (or 2 arrays)
   */
  computeRaw: function(source, target) {

    //log('computeRaw: '+arguments.length+ " "+Type.get(source));

    if (is(source, A)) return this._computeRawArray(source, target);

    if (!source.length && !target.length) return 0;

    var i, j, matrix = []; // matrix
    var cost; // cost
    var sI; // ith character of s
    var tJ; // jth character of t

    // Step 1 ----------------------------------------------

    var sourceLength = source.length;
    var targetLength = target.length;

    if (!sourceLength) return targetLength;

    if (!targetLength) return sourceLength;

    // Step 2 ----------------------------------------------

    for (i = 0; i <= sourceLength; i++) {
      matrix[i] = [];
      matrix[i][0] = i;
    }

    for (j = 0; j <= targetLength; j++)
      matrix[0][j] = j;

    // Step 3 ----------------------------------------------

    for (i = 1; i <= sourceLength; i++) {
      sI = source.charAt(i - 1);

      // Step 4 --------------------------------------------

      for (j = 1; j <= targetLength; j++) {
        tJ = target.charAt(j - 1);

        // Step 5 ------------------------------------------

        cost = (sI == tJ) ? 0 : 1;

        // Step 6 ------------------------------------------
        matrix[i][j] = this._min3(
          matrix[i - 1][j] + 1,
          matrix[i][j - 1] + 1,
          matrix[i - 1][j - 1] + cost);
      }
    }

    // Step 7 ----------------------------------------------

    return matrix[sourceLength][targetLength];

  },

  /*
   * Compute min-edit-distance between 2 strings (or 2 arrays of strings)
   * divided by the max of their lengths.
   */
  computeAdjusted: function(source, target) {

    var st = get(source), tt = get(source);

    if (st === tt) {
      if (tt === S) {
        if (!source.length && !target.length) return 0;
        //log(this.computeRaw(source, target)+'/'+(source.length + target.length));
        return this.computeRaw(source, target) / Math.max(source.length, target.length);
      } else if (tt === A) {
        if (!source.length && !target.length) return 0;
        //log(_computeRawArray(source, target)+'/'+(source.length + target.length));
        return this._computeRawArray(source, target) / Math.max(source.length, target.length);
      }
    }
    err('Unexpected args: ' + source + "/" + target);
  }
};

////////////////////////////////////////////////////////////////
//////// Concorder
////////////////////////////////////////////////////////////////

var Concorder = makeClass();

Concorder.prototype = {

  init: function(text, options) {

    this.model = null;
    this.wordsToIgnore = [];
    this.ignoreCase = false;
    this.ignoreStopWords = false;
    this.ignorePunctuation = false;

    if (options) {
      options.ignoreCase && (this.ignoreCase = true);
      options.ignoreStopWords && (this.ignoreStopWords = true);
      options.ignorePunctuation && (this.ignorePunctuation = true);
      options.wordsToIgnore && (this.wordsToIgnore = options.wordsToIgnore);
    }

    if (this.ignoreStopWords)
      this.wordsToIgnore = this.wordsToIgnore.concat(RiTa.STOP_WORDS);

    this.words = is(text, A) ? text : RiTa.tokenize(text);
  },

  count: function(word) {

    var value = this.lookup(word);
    return value === null ? 0 : value.count;
  },

  concordance: function() {

    if (!this.model) this.build();

    var result = {};
    for (var name in this.model)
      result[name] = this.model[name].indexes.length;

    // TODO: need to sort by value here
    return result;
  },

  kwic: function(word, numWords) {

    var value = this.lookup(word), result = [];
    if (value) {
      var idxs = value.indexes;
      for (var i = 0; i < idxs.length; i++) {
          var sub = this.words.slice(Math.max(0,idxs[i] - numWords),
            Math.min(this.words.length, idxs[i] + numWords+1));

          if (i < 1 || (idxs[i] - idxs[i - 1]) > numWords)
            result.push(RiTa.untokenize(sub));
      }
    }
    return result;
  },

  /////////////////// helpers ////////////////////////////

  build: function() {

    if (!this.words) throw Error('No text in model');

    this.model = {};
    var ts = +new Date();
    for (var j = 0; j < this.words.length; j++) {

      var word = this.words[j];
      if (this.ignorable(word)) continue;
      var lookup = this.lookup(word);

      // The typeof check below fixes a strange bug in Firefox: #XYZ
      // where the string 'watch' comes back from lookup as a function
      // TODO: resolve in a better way
      if (!lookup || typeof lookup !== 'object') {

         lookup = { word: word, key: this.compareKey(word), indexes: [] };
         this.model[lookup.key] = lookup;
      }
      lookup.indexes.push(j);
    }
  },

  ignorable: function(key) {

    if (this.ignorePunctuation && RiTa.isPunctuation(key))
      return true;

    for (var i = 0; i < this.wordsToIgnore.length; i++) {
      var word = this.wordsToIgnore[i];
      if ((this.ignoreCase && key.toUpperCase() === word.toUpperCase()) || key===word)
          return true;
    }
    return false;
  },

  compareKey: function(word) {
    return this.ignoreCase ? word.toLowerCase() : word;
  },

  lookup: function(word) {
    var key = this.compareKey(word);
    if (!this.model) this.build();
    return this.model[key];
  }
};

//////////////////////////////////////////////////////////////////
//////// RE
////////////////////////////////////////////////////////////////

var RE = makeClass();

RE.prototype = {

  init: function(regex, offset, suffix) {
    this.regex = new RegExp(regex);
    this.offset = offset;
    this.suffix = suffix || '';
  },

  applies: function(word) {
    return this.regex.test(trim(word));
  },

  fire: function(word) {
    return this.truncate(trim(word)) + this.suffix;
  },

  analyze: function(word) {
    return ((this.suffix != E) && endsWith(word, this.suffix)) ? true : false;
  },

  truncate: function(word) {
    return (this.offset === 0) ? word : word.substr(0, word.length - this.offset);
  }
};

////////////////////////////////// End Classes ///////////////////////////////////

var QUESTION_STARTS =   [ "Was", "What", "When", "Where", "Which", "Why", "Who", "Will", "Would",
                          "How", "If", "Who", "Is", "Could", "Might", "Does", "Are", "Have" ];

var W_QUESTION_STARTS = [ "Was", "What", "When", "Where", "Which", "Why", "Who", "Will", "Would" ];

var PUNCTUATION_CLASS = /[�`~\"\/'_\-[\]{}()*+!?%&.,\\^$|#@<>|+=;:]/g; // TODO: add smart-quotes

var ONLY_PUNCT = /^[^0-9A-Za-z\s]*$/,
  DEFAULT_PLURAL_RULE = RE("^((\\w+)(-\\w+)*)(\\s((\\w+)(-\\w+)*))*$", 0, "s"),
  ALL_PUNCT = /^[-[\]{}()*+!?%&.,\\^$|#@<>|+=;:]+$/g;

var NULL_PLURALS = RE( // these don't change for plural/singular
  "^(bantu|bengalese|bengali|beninese|boche|bonsai|digitalis|mess|" + "burmese|chinese|colossus|congolese|discus|emphasis|gabonese|guyanese|japanese|javanese|" + "lebanese|maltese|olympics|portuguese|senegalese|siamese|singhalese|innings|" + "sinhalese|sioux|sudanese|swiss|taiwanese|togolese|vietnamese|aircraft|" + "anopheles|apparatus|asparagus|barracks|bellows|bison|bluefish|bob|bourgeois|" + "bream|brill|butterfingers|cargo|carp|catfish|chassis|clothes|chub|cod|codfish|" + "coley|contretemps|corps|crawfish|crayfish|crossroads|cuttlefish|dace|deer|dice|" + "dogfish|doings|dory|downstairs|eldest|earnings|economics|electronics|finnan|" + "firstborn|fish|flatfish|flounder|fowl|fry|fries|works|globefish|goldfish|golf|" + "grand|grief|gudgeon|gulden|haddock|hake|halibut|headquarters|herring|hertz|horsepower|" + "goods|hovercraft|hundredweight|ironworks|jackanapes|kilohertz|kurus|kwacha|ling|lungfish|" + "mackerel|means|megahertz|moorfowl|moorgame|mullet|nepalese|offspring|pampas|parr|pants|" + "patois|pekinese|penn'orth|perch|pickerel|pike|pince-nez|plaice|precis|quid|rand|" + "rendezvous|revers|roach|roux|salmon|samurai|series|seychelles|seychellois|shad|" + "sheep|shellfish|smelt|spacecraft|species|starfish|stockfish|sunfish|superficies|" + "sweepstakes|swordfish|tench|tennis|[a-z]+osis|[a-z]+itis|[a-z]+ness|" + "tobacco|tope|triceps|trout|tuna|tunafish|tunny|turbot|trousers|" + "undersigned|veg|waterfowl|waterworks|waxworks|whiting|wildfowl|woodworm|" + "yen|aries|pisces|forceps|lieder|jeans|physics|mathematics|news|odds|politics|remains|" + "surroundings|thanks|statistics|goods|aids|wildlife)$", 0);

var SINGULAR_RULES = [
  NULL_PLURALS,
  RE("^(oxen|buses|octopuses)$", 2),
  RE("^(toes|taxis|menus|gurus)$", 1),
  RE("(men|women)$", 2, "an"),
  RE("^[lm]ice$", 3, "ouse"),
  RE("^children", 3),
  RE("^(appendices|indices|matrices)", 3, "x"),
  RE("^(stimuli|alumni)$", 1, "us"),
  RE("^(data)$", 1, "um"),
  RE("^(memoranda|bacteria|curricula|minima|" + "maxima|referenda|spectra|phenomena|criteria)$", 1, "um"),
  RE("monies", 3, "ey"),
  RE("people", 4, "rson"),
  RE("^meninges|phalanges$", 3, "x"),
  RE("schemata$", 2, "s"),
  RE("^corpora$", 3, "us"),
  RE("^(curi|formul|vertebr|larv|uln|alumn|signor|alg)ae$", 1),
  RE("^apices|cortices$", 4, "ex"),
  RE("^teeth$", 4, "ooth"),
  RE("^feet$", 3, "oot"),
  RE("femora", 3, "ur"),
  RE("geese", 4, "oose"),
  RE("crises", 2, "is"),
];

var C = "[bcdfghjklmnpqrstvwxyz]",
  VL = "[lraeiou]";

var PLURAL_RULES = [
    NULL_PLURALS,
    RE("^(piano|photo|solo|ego|tobacco|cargo|golf|grief)$", 0, "s"),
    RE("^(wildlife)$", 0, "s"),
    RE(C + "o$", 0, "es"),
    RE(C + "y$", 1, "ies"),
    RE("^ox$", 0, "en"),
    RE("^(stimulus|alumnus)$", 2, "i"),
    RE("^corpus$", 2, "ora"),
    RE("(xis|sis)$", 2, "es"),
    RE("([zsx]|ch|sh)$", 0, "es"),
    RE(VL + "fe$", 2, "ves"),
    RE(VL + "f$", 1, "ves"),
    RE("(eu|eau)$", 0, "x"),

    RE("(man|woman)$", 2, "en"),
    RE("money$", 2, "ies"),
    RE("person$", 4, "ople"),
    RE("motif$", 0, "s"),
    RE("^meninx|phalanx$", 1, "ges"),

    RE("schema$", 0, "ta"),
    RE("^bus$", 0, "ses"),
    RE("child$", 0, "ren"),
    RE("^(curi|formul|vertebr|larv|uln|alumn|signor|alg)a$", 0, "e"),
    RE("^(maharaj|raj|myn|mull)a$", 0, "hs"),
    RE("^aide-de-camp$", 8, "s-de-camp"),
    RE("^apex|cortex$", 2, "ices"),
    RE("^weltanschauung$", 0, "en"),
    RE("^lied$", 0, "er"),
    RE("^tooth$", 4, "eeth"),
    RE("^[lm]ouse$", 4, "ice"),
    RE("^foot$", 3, "eet"),
    RE("femur", 2, "ora"),
    RE("goose", 4, "eese"),
    RE("(human|german|roman)$", 0, "s"),
    RE("^(monarch|loch|stomach)$", 0, "s"),
    RE("^(taxi|chief|proof|ref|relief|roof|belief)$", 0, "s"),
    RE("^(co|no)$", 0, "'s"),
    RE("^blond$", 0, "es"),

    // Latin stems
    RE("^(memorandum|bacterium|curriculum|minimum|" + "maximum|referendum|spectrum|phenomenon|criterion)$", 2, "a"),
    RE("^(appendix|index|matrix)", 2, "ices")
  ],

  ANY_STEM = "^((\\w+)(-\\w+)*)(\\s((\\w+)(-\\w+)*))*$",
  CONS = "[bcdfghjklmnpqrstvwxyz]",
  VERBAL_PREFIX = "((be|with|pre|un|over|re|mis|under|out|up|fore|for|counter|co|sub)(-?))",
  AUXILIARIES = ["do", "have", "be"],
  MODALS = ["shall", "would", "may", "might", "ought", "should"],
  SYMBOLS = ["!", "?", "$", "%", "*", "+", "-", "="],

  ING_FORM_RULES = [
    RE(CONS + "ie$", 2, "ying", 1),
    RE("[^ie]e$", 1, "ing", 1),
    RE("^bog-down$", 5, "ging-down", 0),
    RE("^chivy$", 1, "vying", 0),
    RE("^trek$", 1, "cking", 0),
    RE("^bring$", 0, "ing", 0),
    RE("^be$", 0, "ing", 0),
    RE("^age$", 1, "ing", 0),
    RE("(ibe)$", 1, "ing", 0)
  ],

  PAST_PARTICIPLE_RULES = [

    RE(CONS + "y$", 1, "ied", 1),
    RE("^" + VERBAL_PREFIX + "?(bring)$", 3, "ought", 0),
    RE("^" + VERBAL_PREFIX + "?(take|rise|strew|blow|draw|drive|know|give|" + "arise|gnaw|grave|grow|hew|know|mow|see|sew|throw|prove|saw|quartersaw|" + "partake|sake|shake|shew|show|shrive|sightsee|strew|strive)$",
      0, "n", 0),
    RE("^" + VERBAL_PREFIX + "?[gd]o$", 0, "ne", 1),
    RE("^(beat|eat|be|fall)$", 0, "en", 0),
    RE("^(have)$", 2, "d", 0),
    RE("^" + VERBAL_PREFIX + "?bid$", 0, "den", 0),
    RE("^" + VERBAL_PREFIX + "?[lps]ay$", 1, "id", 1),
    RE("^behave$", 0, "d", 0),
    RE("^" + VERBAL_PREFIX + "?have$", 2, "d", 1),
    RE("(sink|slink|drink|shrink|stink)$", 3, "unk", 0),
    RE("(([sfc][twlp]?r?|w?r)ing|hang)$", 3, "ung", 0),
    RE("^" + VERBAL_PREFIX + "?(shear|swear|bear|wear|tear)$", 3, "orn", 0),
    RE("^" + VERBAL_PREFIX + "?(bend|spend|send|lend)$", 1, "t", 0),
    RE("^" + VERBAL_PREFIX + "?(weep|sleep|sweep|creep|keep$)$", 2, "pt", 0),
    RE("^" + VERBAL_PREFIX + "?(sell|tell)$", 3, "old", 0),
    RE("^(outfight|beseech)$", 4, "ought", 0),
    RE("^bethink$", 3, "ought", 0),
    RE("^buy$", 2, "ought", 0),
    RE("^aby$", 1, "ought", 0),
    RE("^tarmac", 0, "ked", 0),
    RE("^abide$", 3, "ode", 0),
    RE("^" + VERBAL_PREFIX + "?(speak|(a?)wake|break)$", 3, "oken", 0),
    RE("^backbite$", 1, "ten", 0),
    RE("^backslide$", 1, "den", 0),
    RE("^become$", 3, "ame", 0),
    RE("^begird$", 3, "irt", 0),
    RE("^outlie$", 2, "ay", 0),
    RE("^rebind$", 3, "ound", 0),
    RE("^relay$", 2, "aid", 0),
    RE("^shit$", 3, "hat", 0),
    RE("^bereave$", 4, "eft", 0),
    RE("^foreswear$", 3, "ore", 0),
    RE("^overfly$", 1, "own", 0),
    RE("^beget$", 2, "otten", 0),
    RE("^begin$", 3, "gun", 0),
    RE("^bestride$", 1, "den", 0),
    RE("^bite$", 1, "ten", 0),
    RE("^bleed$", 4, "led", 0),
    RE("^bog-down$", 5, "ged-down", 0),
    RE("^bind$", 3, "ound", 0),
    RE("^(.*)feed$", 4, "fed", 0),
    RE("^breed$", 4, "red", 0),
    RE("^brei", 0, "d", 0),
    RE("^bring$", 3, "ought", 0),
    RE("^build$", 1, "t", 0),
    RE("^come", 0),
    RE("^catch$", 3, "ught", 0),
    RE("^chivy$", 1, "vied", 0),
    RE("^choose$", 3, "sen", 0),
    RE("^cleave$", 4, "oven", 0),
    RE("^crossbreed$", 4, "red", 0),
    RE("^deal", 0, "t", 0),
    RE("^dow$", 1, "ught", 0),
    RE("^dream", 0, "t", 0),
    RE("^dig$", 3, "dug", 0),
    RE("^dwell$", 2, "lt", 0),
    RE("^enwind$", 3, "ound", 0),
    RE("^feel$", 3, "elt", 0),
    RE("^flee$", 2, "ed", 0),
    RE("^floodlight$", 5, "lit", 0),
    RE("^fly$", 1, "own", 0),
    RE("^forbear$", 3, "orne", 0),
    RE("^forerun$", 3, "ran", 0),
    RE("^forget$", 2, "otten", 0),
    RE("^fight$", 4, "ought", 0),
    RE("^find$", 3, "ound", 0),
    RE("^freeze$", 4, "ozen", 0),
    RE("^gainsay$", 2, "aid", 0),
    RE("^gin$", 3, "gan", 0),
    RE("^gen-up$", 3, "ned-up", 0),
    RE("^ghostwrite$", 1, "ten", 0),
    RE("^get$", 2, "otten", 0),
    RE("^grind$", 3, "ound", 0),
    RE("^hacksaw", 0, "n", 0),
    RE("^hear", 0, "d", 0),
    RE("^hold$", 3, "eld", 0),
    RE("^hide$", 1, "den", 0),
    RE("^honey$", 2, "ied", 0),
    RE("^inbreed$", 4, "red", 0),
    RE("^indwell$", 3, "elt", 0),
    RE("^interbreed$", 4, "red", 0),
    RE("^interweave$", 4, "oven", 0),
    RE("^inweave$", 4, "oven", 0),
    RE("^ken$", 2, "ent", 0),
    RE("^kneel$", 3, "elt", 0),
    RE("^lie$", 2, "ain", 0),
    RE("^leap$", 0, "t", 0),
    RE("^learn$", 0, "t", 0),
    RE("^lead$", 4, "led", 0),
    RE("^leave$", 4, "eft", 0),
    RE("^light$", 5, "lit", 0),
    RE("^lose$", 3, "ost", 0),
    RE("^make$", 3, "ade", 0),
    RE("^mean", 0, "t", 0),
    RE("^meet$", 4, "met", 0),
    RE("^misbecome$", 3, "ame", 0),
    RE("^misdeal$", 2, "alt", 0),
    RE("^mishear$", 1, "d", 0),
    RE("^mislead$", 4, "led", 0),
    RE("^misunderstand$", 3, "ood", 0),
    RE("^outbreed$", 4, "red", 0),
    RE("^outrun$", 3, "ran", 0),
    RE("^outride$", 1, "den", 0),
    RE("^outshine$", 3, "one", 0),
    RE("^outshoot$", 4, "hot", 0),
    RE("^outstand$", 3, "ood", 0),
    RE("^outthink$", 3, "ought", 0),
    RE("^outgo$", 2, "went", 0),
    RE("^overbear$", 3, "orne", 0),
    RE("^overbuild$", 3, "ilt", 0),
    RE("^overcome$", 3, "ame", 0),
    RE("^overfly$", 2, "lew", 0),
    RE("^overhear$", 2, "ard", 0),
    RE("^overlie$", 2, "ain", 0),
    RE("^overrun$", 3, "ran", 0),
    RE("^override$", 1, "den", 0),
    RE("^overshoot$", 4, "hot", 0),
    RE("^overwind$", 3, "ound", 0),
    RE("^overwrite$", 1, "ten", 0),
    RE("^plead$", 2, "d", 0),
    //RE("^run$", 3, "ran", 0), //DH
    //RE("^rerun$", 3, "run", 0),
    RE("^rebuild$", 3, "ilt", 0),
    RE("^red$", 3, "red", 0),
    RE("^redo$", 1, "one", 0),
    RE("^remake$", 3, "ade", 0),
    RE("^resit$", 3, "sat", 0),
    RE("^rethink$", 3, "ought", 0),
    RE("^rewind$", 3, "ound", 0),
    RE("^rewrite$", 1, "ten", 0),
    RE("^ride$", 1, "den", 0),
    RE("^reeve$", 4, "ove", 0),
    RE("^sit$", 3, "sat", 0),
    RE("^shoe$", 3, "hod", 0),
    RE("^shine$", 3, "one", 0),
    RE("^shoot$", 4, "hot", 0),
    RE("^ski$", 1, "i'd", 0),
    RE("^slide$", 1, "den", 0),
    RE("^smite$", 1, "ten", 0),
    RE("^seek$", 3, "ought", 0),
    RE("^spit$", 3, "pat", 0),
    RE("^speed$", 4, "ped", 0),
    RE("^spellbind$", 3, "ound", 0),
    RE("^spoil$", 2, "ilt", 0),
    RE("^spotlight$", 5, "lit", 0),
    RE("^spin$", 3, "pun", 0),
    RE("^steal$", 3, "olen", 0),
    RE("^stand$", 3, "ood", 0),
    RE("^stave$", 3, "ove", 0),
    RE("^stride$", 1, "den", 0),
    RE("^strike$", 3, "uck", 0),
    RE("^stick$", 3, "uck", 0),
    RE("^swell$", 3, "ollen", 0),
    RE("^swim$", 3, "wum", 0),
    RE("^teach$", 4, "aught", 0),
    RE("^think$", 3, "ought", 0),
    RE("^tread$", 3, "odden", 0),
    RE("^typewrite$", 1, "ten", 0),
    RE("^unbind$", 3, "ound", 0),
    RE("^underbuy$", 2, "ought", 0),
    RE("^undergird$", 3, "irt", 0),
    RE("^undergo$", 1, "one", 0),
    RE("^underlie$", 2, "ain", 0),
    RE("^undershoot$", 4, "hot", 0),
    RE("^understand$", 3, "ood", 0),
    RE("^unfreeze$", 4, "ozen", 0),
    RE("^unlearn", 0, "t", 0),
    RE("^unmake$", 3, "ade", 0),
    RE("^unreeve$", 4, "ove", 0),
    RE("^unstick$", 3, "uck", 0),
    RE("^unteach$", 4, "aught", 0),
    RE("^unthink$", 3, "ought", 0),
    RE("^untread$", 3, "odden", 0),
    RE("^unwind$", 3, "ound", 0),
    RE("^upbuild$", 1, "t", 0),
    RE("^uphold$", 3, "eld", 0),
    RE("^upheave$", 4, "ove", 0),
    RE("^waylay$", 2, "ain", 0),
    RE("^whipsaw$", 2, "awn", 0),
    RE("^withhold$", 3, "eld", 0),
    RE("^withstand$", 3, "ood", 0),
    RE("^win$", 3, "won", 0),
    RE("^wind$", 3, "ound", 0),
    RE("^weave$", 4, "oven", 0),
    RE("^write$", 1, "ten", 0),
    RE("^trek$", 1, "cked", 0),
    RE("^ko$", 1, "o'd", 0),
    RE("^win$", 2, "on", 0),

    RE("e$", 0, "d", 1),

    // Null past forms
    RE("^" + VERBAL_PREFIX + "?(cast|thrust|typeset|cut|bid|upset|wet|bet|cut|hit|hurt|inset|let|cost|burst|beat|beset|set|upset|hit|offset|put|quit|" + "wed|typeset|wed|spread|split|slit|read|run|rerun|shut|shed)$", 0)
  ],

  PAST_TENSE_RULES = [
    RE("^(reduce)$", 0, "d", 0),
    RE("e$", 0, "d", 1),
    RE("^" + VERBAL_PREFIX + "?[pls]ay$", 1, "id", 1),
    RE(CONS + "y$", 1, "ied", 1),
    RE("^(fling|cling|hang)$", 3, "ung", 0),
    RE("(([sfc][twlp]?r?|w?r)ing)$", 3, "ang", 1),
    RE("^" + VERBAL_PREFIX + "?(bend|spend|send|lend|spend)$", 1, "t", 0),
    RE("^" + VERBAL_PREFIX + "?lie$", 2, "ay", 0),
    RE("^" + VERBAL_PREFIX + "?(weep|sleep|sweep|creep|keep)$", 2, "pt",
      0),
    RE("^" + VERBAL_PREFIX + "?(sell|tell)$", 3, "old", 0),
    RE("^" + VERBAL_PREFIX + "?do$", 1, "id", 0),
    RE("^" + VERBAL_PREFIX + "?dig$", 2, "ug", 0),
    RE("^behave$", 0, "d", 0),
    RE("^(have)$", 2, "d", 0),
    RE("(sink|drink)$", 3, "ank", 0),
    RE("^swing$", 3, "ung", 0),
    RE("^be$", 2, "was", 0),
    RE("^outfight$", 4, "ought", 0),
    RE("^tarmac", 0, "ked", 0),
    RE("^abide$", 3, "ode", 0),
    RE("^aby$", 1, "ought", 0),
    RE("^become$", 3, "ame", 0),
    RE("^begird$", 3, "irt", 0),
    RE("^outlie$", 2, "ay", 0),
    RE("^rebind$", 3, "ound", 0),
    RE("^shit$", 3, "hat", 0),
    RE("^bereave$", 4, "eft", 0),
    RE("^foreswear$", 3, "ore", 0),
    RE("^bename$", 3, "empt", 0),
    RE("^beseech$", 4, "ought", 0),
    RE("^bethink$", 3, "ought", 0),
    RE("^bleed$", 4, "led", 0),
    RE("^bog-down$", 5, "ged-down", 0),
    RE("^buy$", 2, "ought", 0),
    RE("^bind$", 3, "ound", 0),
    RE("^(.*)feed$", 4, "fed", 0),
    RE("^breed$", 4, "red", 0),
    RE("^brei$", 2, "eid", 0),
    RE("^bring$", 3, "ought", 0),
    RE("^build$", 3, "ilt", 0),
    RE("^come$", 3, "ame", 0),
    RE("^catch$", 3, "ught", 0),
    RE("^clothe$", 5, "lad", 0),
    RE("^crossbreed$", 4, "red", 0),
    RE("^deal$", 2, "alt", 0),
    RE("^dow$", 1, "ught", 0),
    RE("^dream$", 2, "amt", 0),
    RE("^dwell$", 3, "elt", 0),
    RE("^enwind$", 3, "ound", 0),
    RE("^feel$", 3, "elt", 0),
    RE("^flee$", 3, "led", 0),
    RE("^floodlight$", 5, "lit", 0),
    RE("^arise$", 3, "ose", 0),
    RE("^eat$", 3, "ate", 0),
    RE("^backbite$", 4, "bit", 0),
    RE("^backslide$", 4, "lid", 0),
    RE("^befall$", 3, "ell", 0),
    RE("^begin$", 3, "gan", 0),
    RE("^beget$", 3, "got", 0),
    RE("^behold$", 3, "eld", 0),
    RE("^bespeak$", 3, "oke", 0),
    RE("^bestride$", 3, "ode", 0),
    RE("^betake$", 3, "ook", 0),
    RE("^bite$", 4, "bit", 0),
    RE("^blow$", 3, "lew", 0),
    RE("^bear$", 3, "ore", 0),
    RE("^break$", 3, "oke", 0),
    RE("^choose$", 4, "ose", 0),
    RE("^cleave$", 4, "ove", 0),
    RE("^countersink$", 3, "ank", 0),
    RE("^drink$", 3, "ank", 0),
    RE("^draw$", 3, "rew", 0),
    RE("^drive$", 3, "ove", 0),
    RE("^fall$", 3, "ell", 0),
    RE("^fly$", 2, "lew", 0),
    RE("^flyblow$", 3, "lew", 0),
    RE("^forbid$", 2, "ade", 0),
    RE("^forbear$", 3, "ore", 0),
    RE("^foreknow$", 3, "new", 0),
    RE("^foresee$", 3, "saw", 0),
    RE("^forespeak$", 3, "oke", 0),
    RE("^forego$", 2, "went", 0),
    RE("^forgive$", 3, "ave", 0),
    RE("^forget$", 3, "got", 0),
    RE("^forsake$", 3, "ook", 0),
    RE("^forspeak$", 3, "oke", 0),
    RE("^forswear$", 3, "ore", 0),
    RE("^forgo$", 2, "went", 0),
    RE("^fight$", 4, "ought", 0),
    RE("^find$", 3, "ound", 0),
    RE("^freeze$", 4, "oze", 0),
    RE("^give$", 3, "ave", 0),
    RE("^geld$", 3, "elt", 0),
    RE("^gen-up$", 3, "ned-up", 0),
    RE("^ghostwrite$", 3, "ote", 0),
    RE("^get$", 3, "got", 0),
    RE("^grow$", 3, "rew", 0),
    RE("^grind$", 3, "ound", 0),
    RE("^hear$", 2, "ard", 0),
    RE("^hold$", 3, "eld", 0),
    RE("^hide$", 4, "hid", 0),
    RE("^honey$", 2, "ied", 0),
    RE("^inbreed$", 4, "red", 0),
    RE("^indwell$", 3, "elt", 0),
    RE("^interbreed$", 4, "red", 0),
    RE("^interweave$", 4, "ove", 0),
    RE("^inweave$", 4, "ove", 0),
    RE("^ken$", 2, "ent", 0),
    RE("^kneel$", 3, "elt", 0),
    RE("^^know$$", 3, "new", 0),
    RE("^leap$", 2, "apt", 0),
    RE("^learn$", 2, "rnt", 0),
    RE("^lead$", 4, "led", 0),
    RE("^leave$", 4, "eft", 0),
    RE("^light$", 5, "lit", 0),
    RE("^lose$", 3, "ost", 0),
    RE("^make$", 3, "ade", 0),
    RE("^mean$", 2, "ant", 0),
    RE("^meet$", 4, "met", 0),
    RE("^misbecome$", 3, "ame", 0),
    RE("^misdeal$", 2, "alt", 0),
    RE("^misgive$", 3, "ave", 0),
    RE("^mishear$", 2, "ard", 0),
    RE("^mislead$", 4, "led", 0),
    RE("^mistake$", 3, "ook", 0),
    RE("^misunderstand$", 3, "ood", 0),
    RE("^outbreed$", 4, "red", 0),
    RE("^outgrow$", 3, "rew", 0),
    RE("^outride$", 3, "ode", 0),
    RE("^outshine$", 3, "one", 0),
    RE("^outshoot$", 4, "hot", 0),
    RE("^outstand$", 3, "ood", 0),
    RE("^outthink$", 3, "ought", 0),
    RE("^outgo$", 2, "went", 0),
    RE("^outwear$", 3, "ore", 0),
    RE("^overblow$", 3, "lew", 0),
    RE("^overbear$", 3, "ore", 0),
    RE("^overbuild$", 3, "ilt", 0),
    RE("^overcome$", 3, "ame", 0),
    RE("^overdraw$", 3, "rew", 0),
    RE("^overdrive$", 3, "ove", 0),
    RE("^overfly$", 2, "lew", 0),
    RE("^overgrow$", 3, "rew", 0),
    RE("^overhear$", 2, "ard", 0),
    RE("^overpass$", 3, "ast", 0),
    RE("^override$", 3, "ode", 0),
    RE("^oversee$", 3, "saw", 0),
    RE("^overshoot$", 4, "hot", 0),
    RE("^overthrow$", 3, "rew", 0),
    RE("^overtake$", 3, "ook", 0),
    RE("^overwind$", 3, "ound", 0),
    RE("^overwrite$", 3, "ote", 0),
    RE("^partake$", 3, "ook", 0),
    RE("^" + VERBAL_PREFIX + "?run$", 2, "an", 0),
    RE("^ring$", 3, "ang", 0),
    RE("^rebuild$", 3, "ilt", 0),
    RE("^red", 0),
    RE("^reave$", 4, "eft", 0),
    RE("^remake$", 3, "ade", 0),
    RE("^resit$", 3, "sat", 0),
    RE("^rethink$", 3, "ought", 0),
    RE("^retake$", 3, "ook", 0),
    RE("^rewind$", 3, "ound", 0),
    RE("^rewrite$", 3, "ote", 0),
    RE("^ride$", 3, "ode", 0),
    RE("^rise$", 3, "ose", 0),
    RE("^reeve$", 4, "ove", 0),
    RE("^sing$", 3, "ang", 0),
    RE("^sink$", 3, "ank", 0),
    RE("^sit$", 3, "sat", 0),
    RE("^see$", 3, "saw", 0),
    RE("^shoe$", 3, "hod", 0),
    RE("^shine$", 3, "one", 0),
    RE("^shake$", 3, "ook", 0),
    RE("^shoot$", 4, "hot", 0),
    RE("^shrink$", 3, "ank", 0),
    RE("^shrive$", 3, "ove", 0),
    RE("^sightsee$", 3, "saw", 0),
    RE("^ski$", 1, "i'd", 0),
    RE("^skydive$", 3, "ove", 0),
    RE("^slay$", 3, "lew", 0),
    RE("^slide$", 4, "lid", 0),
    RE("^slink$", 3, "unk", 0),
    RE("^smite$", 4, "mit", 0),
    RE("^seek$", 3, "ought", 0),
    RE("^spit$", 3, "pat", 0),
    RE("^speed$", 4, "ped", 0),
    RE("^spellbind$", 3, "ound", 0),
    RE("^spoil$", 2, "ilt", 0),
    RE("^speak$", 3, "oke", 0),
    RE("^spotlight$", 5, "lit", 0),
    RE("^spring$", 3, "ang", 0),
    RE("^spin$", 3, "pun", 0),
    RE("^stink$", 3, "ank", 0),
    RE("^steal$", 3, "ole", 0),
    RE("^stand$", 3, "ood", 0),
    RE("^stave$", 3, "ove", 0),
    RE("^stride$", 3, "ode", 0),
    RE("^strive$", 3, "ove", 0),
    RE("^strike$", 3, "uck", 0),
    RE("^stick$", 3, "uck", 0),
    RE("^swim$", 3, "wam", 0),
    RE("^swear$", 3, "ore", 0),
    RE("^teach$", 4, "aught", 0),
    RE("^think$", 3, "ought", 0),
    RE("^throw$", 3, "rew", 0),
    RE("^take$", 3, "ook", 0),
    RE("^tear$", 3, "ore", 0),
    RE("^transship$", 4, "hip", 0),
    RE("^tread$", 4, "rod", 0),
    RE("^typewrite$", 3, "ote", 0),
    RE("^unbind$", 3, "ound", 0),
    RE("^unclothe$", 5, "lad", 0),
    RE("^underbuy$", 2, "ought", 0),
    RE("^undergird$", 3, "irt", 0),
    RE("^undershoot$", 4, "hot", 0),
    RE("^understand$", 3, "ood", 0),
    RE("^undertake$", 3, "ook", 0),
    RE("^undergo$", 2, "went", 0),
    RE("^underwrite$", 3, "ote", 0),
    RE("^unfreeze$", 4, "oze", 0),
    RE("^unlearn$", 2, "rnt", 0),
    RE("^unmake$", 3, "ade", 0),
    RE("^unreeve$", 4, "ove", 0),
    RE("^unspeak$", 3, "oke", 0),
    RE("^unstick$", 3, "uck", 0),
    RE("^unswear$", 3, "ore", 0),
    RE("^unteach$", 4, "aught", 0),
    RE("^unthink$", 3, "ought", 0),
    RE("^untread$", 4, "rod", 0),
    RE("^unwind$", 3, "ound", 0),
    RE("^upbuild$", 3, "ilt", 0),
    RE("^uphold$", 3, "eld", 0),
    RE("^upheave$", 4, "ove", 0),
    RE("^uprise$", 3, "ose", 0),
    RE("^upspring$", 3, "ang", 0),
    RE("^go$", 2, "went", 0),
    RE("^wiredraw$", 3, "rew", 0),
    RE("^withdraw$", 3, "rew", 0),
    RE("^withhold$", 3, "eld", 0),
    RE("^withstand$", 3, "ood", 0),
    RE("^wake$", 3, "oke", 0),
    RE("^win$", 3, "won", 0),
    RE("^wear$", 3, "ore", 0),
    RE("^wind$", 3, "ound", 0),
    RE("^weave$", 4, "ove", 0),
    RE("^write$", 3, "ote", 0),
    RE("^trek$", 1, "cked", 0),
    RE("^ko$", 1, "o'd", 0),
    RE("^bid", 2, "ade", 0),
    RE("^win$", 2, "on", 0),
    RE("^swim", 2, "am", 0),

    // Null past forms
    RE("^" + VERBAL_PREFIX + "?(cast|thrust|typeset|cut|bid|upset|wet|bet|cut|hit|hurt|inset|" + "let|cost|burst|beat|beset|set|upset|offset|put|quit|wed|typeset|" + "wed|spread|split|slit|read|run|shut|shed|lay)$", 0)
  ],

  PRESENT_TENSE_RULES = [
    RE("^aby$", 0, "es", 0),
    RE("^bog-down$", 5, "s-down", 0),
    RE("^chivy$", 1, "vies", 0),
    RE("^gen-up$", 3, "s-up", 0),
    RE("^prologue$", 3, "gs", 0),
    RE("^picknic$", 0, "ks", 0),
    //RE("^swim$", 0, "s", 0),
    RE("^ko$", 0, "'s", 0),
    RE("[osz]$", 0, "es", 1),
    RE("^have$", 2, "s", 0),
    RE(CONS + "y$", 1, "ies", 1),
    RE("^be$", 2, "is"),
    RE("([zsx]|ch|sh)$", 0, "es", 1)
  ],

  VERB_CONS_DOUBLING = ["abat", "abet", "abhor", "abut", "accur", "acquit", "adlib",
    "admit", "aerobat", "aerosol", "agendaset", "allot", "alot", "anagram",
    "annul", "appal", "apparel", "armbar", "aver", "babysit", "airdrop", "appal",
    "blackleg", "bobsled", "bur", "chum", "confab", "counterplot", "curet", "dib",
    "backdrop", "backfil", "backflip", "backlog", "backpedal", "backslap",
    "backstab", "bag", "balfun", "ballot", "ban", "bar", "barbel", "bareleg",
    "barrel", "bat", "bayonet", "becom", "bed", "bedevil", "bedwet", "beenhop",
    "befit", "befog", "beg", "beget", "begin", "bejewel", "bemedal", "benefit",
    "benum", "beset", "besot", "bestir", "bet", "betassel", "bevel", "bewig",
    "bib", "bid", "billet", "bin", "bip", "bit", "bitmap", "blab", "blag", "blam",
    "blan", "blat", "bles", "blim", "blip", "blob", "bloodlet", "blot", "blub",
    "blur", "bob", "bodypop", "bog", "booby-trap", "boobytrap", "booksel",
    "bootleg", "bop", "bot", "bowel", "bracket", "brag", "brig", "brim", "bud",
    "buffet", "bug", "bullshit", "bum", "bun", "bus", "but", "cab", "cabal", "cam",
    "can", "cancel", "cap", "caracol", "caravan", "carburet", "carnap", "carol",
    "carpetbag", "castanet", "cat", "catcal", "catnap", "cavil", "chan", "chanel",
    "channel", "chap", "char", "chargecap", "chat", "chin", "chip", "chir",
    "chirrup", "chisel", "chop", "chug", "chur", "clam", "clap", "clearcut",
    "clip", "clodhop", "clog", "clop", "closet", "clot", "club", "co-occur",
    "co-program", "co-refer", "co-run", "co-star", "cob", "cobweb", "cod", "coif",
    "com", "combat", "comit", "commit", "compel", "con", "concur", "confer",
    "confiscat", "control", "cop", "coquet", "coral", "corbel", "corral", "cosset",
    "cotransmit", "councel", "council", "counsel", "court-martial", "crab", "cram",
    "crap", "crib", "crop", "crossleg", "cub", "cudgel", "cum", "cun", "cup",
    "cut", "dab", "dag", "dam", "dan", "dap", "daysit", "de-control", "de-gazet",
    "de-hul", "de-instal", "de-mob", "de-program", "de-rig", "de-skil", "deadpan",
    "debag", "debar", "log", "decommit", "decontrol", "defer", "defog", "deg",
    "degas", "deinstal", "demit", "demob", "demur", "den", "denet", "depig",
    "depip", "depit", "der", "deskil", "deter", "devil", "diagram", "dial", "dig",
    "dim", "din", "dip", "disbar", "disbud", "discomfit", "disembed", "disembowel",
    "dishevel", "disinter", "dispel", "disprefer", "distil", "dog", "dognap",
    "don", "doorstep", "dot", "dowel", "drag", "drat", "driftnet", "distil",
    "egotrip", "enrol", "enthral", "extol", "fulfil", "gaffe", "golliwog", "idyl",
    "inspan", "drip", "drivel", "drop", "drub", "drug", "drum", "dub", "duel",
    "dun", "dybbuk", "earwig", "eavesdrop", "ecolabel", "eitherspigot",
    "electroblot", "embed", "emit", "empanel", "enamel", "endlabel", "endtrim",
    "enrol", "enthral", "entrammel", "entrap", "enwrap", "equal", "equip", "estop",
    "exaggerat", "excel", "expel", "extol", "fag", "fan", "farewel", "fat",
    "featherbed", "feget", "fet", "fib", "fig", "fin", "fingerspel", "fingertip",
    "fit", "flab", "flag", "flap", "flip", "flit", "flog", "flop", "fob", "focus",
    "fog", "footbal", "footslog", "fop", "forbid", "forget", "format",
    "fortunetel", "fot", "foxtrot", "frag", "freefal", "fret", "frig", "frip",
    "frog", "frug", "fuel", "fufil", "fulfil", "fullyfit", "fun", "funnel", "fur",
    "furpul", "gab", "gad", "gag", "gam", "gambol", "gap", "garot", "garrot",
    "gas", "gat", "gel", "gen", "get", "giftwrap", "gig", "gimbal", "gin", "glam",
    "glenden", "glendin", "globetrot", "glug", "glut", "gob", "goldpan", "goostep",
    "gossip", "grab", "gravel", "grid", "grin", "grip", "grit", "groundhop",
    "grovel", "grub", "gum", "gun", "gunrun", "gut", "gyp", "haircut", "ham",
    "han", "handbag", "handicap", "handknit", "handset", "hap", "hareleg", "hat",
    "headbut", "hedgehop", "hem", "hen", "hiccup", "highwal", "hip", "hit",
    "hobnob", "hog", "hop", "horsewhip", "hostel", "hot", "hotdog", "hovel", "hug",
    "hum", "humbug", "hup", "hushkit", "hut", "illfit", "imbed", "immunblot",
    "immunoblot", "impannel", "impel", "imperil", "incur", "infer", "infil",
    "inflam", "initial", "input", "inset", "instil", "inter", "interbed",
    "intercrop", "intercut", "interfer", "instal", "instil", "intermit", "japan",
    "jug", "kris", "manumit", "mishit", "mousse", "mud", "interwar", "jab", "jag",
    "jam", "jar", "jawdrop", "jet", "jetlag", "jewel", "jib", "jig", "jitterbug",
    "job", "jog", "jog-trot", "jot", "jut", "ken", "kennel", "kid", "kidnap",
    "kip", "kissogram", "kit", "knap", "kneecap", "knit", "knob", "knot", "kor",
    "label", "lag", "lam", "lap", "lavel", "leafcut", "leapfrog", "leg", "lem",
    "lep", "let", "level", "libel", "lid", "lig", "lip", "lob", "log", "lok",
    "lollop", "longleg", "lop", "lowbal", "lug", "mackerel", "mahom", "man", "map",
    "mar", "marshal", "marvel", "mat", "matchwin", "metal", "micro-program",
    "microplan", "microprogram", "milksop", "mis-cal", "mis-club", "mis-spel",
    "miscal", "mishit", "mislabel", "mit", "mob", "mod", "model", "mohmam",
    "monogram", "mop", "mothbal", "mug", "multilevel", "mum", "nab", "nag", "nan",
    "nap", "net", "nightclub", "nightsit", "nip", "nod", "nonplus", "norkop",
    "nostril", "not", "nut", "nutmeg", "occur", "ocur", "offput", "offset", "omit",
    "ommit", "onlap", "out-general", "out-gun", "out-jab", "out-plan", "out-pol",
    "out-pul", "out-put", "out-run", "out-sel", "outbid", "outcrop", "outfit",
    "outgas", "outgun", "outhit", "outjab", "outpol", "output", "outrun",
    "outship", "outshop", "outsin", "outstrip", "outswel", "outspan", "overcrop",
    "pettifog", "photostat", "pouf", "preset", "prim", "pug", "ret", "rosin",
    "outwit", "over-commit", "over-control", "over-fil", "over-fit", "over-lap",
    "over-model", "over-pedal", "over-pet", "over-run", "over-sel", "over-step",
    "over-tip", "over-top", "overbid", "overcal", "overcommit", "overcontrol",
    "overcrap", "overdub", "overfil", "overhat", "overhit", "overlap", "overman",
    "overplot", "overrun", "overshop", "overstep", "overtip", "overtop", "overwet",
    "overwil", "pad", "paintbal", "pan", "panel", "paperclip", "par", "parallel",
    "parcel", "partiescal", "pat", "patrol", "pedal", "peewit", "peg", "pen",
    "pencil", "pep", "permit", "pet", "petal", "photoset", "phototypeset", "phut",
    "picket", "pig", "pilot", "pin", "pinbal", "pip", "pipefit", "pipet", "pit",
    "plan", "plit", "plod", "plop", "plot", "plug", "plumet", "plummet", "pod",
    "policyset", "polyfil", "ponytrek", "pop", "pot", "pram", "prebag",
    "predistil", "predril", "prefer", "prefil", "preinstal", "prep", "preplan",
    "preprogram", "prizewin", "prod", "profer", "prog", "program", "prop",
    "propel", "pub", "pummel", "pun", "pup", "pushfit", "put", "quarel", "quarrel",
    "quickskim", "quickstep", "quickwit", "quip", "quit", "quivertip", "quiz",
    "rabbit", "rabit", "radiolabel", "rag", "ram", "ramrod", "rap", "rat",
    "ratecap", "ravel", "re-admit", "re-cal", "re-cap", "re-channel", "re-dig",
    "re-dril", "re-emit", "re-fil", "re-fit", "re-flag", "re-format", "re-fret",
    "re-hab", "re-instal", "re-inter", "re-lap", "re-let", "re-map", "re-metal",
    "re-model", "re-pastel", "re-plan", "re-plot", "re-plug", "re-pot",
    "re-program", "re-refer", "re-rig", "re-rol", "re-run", "re-sel", "re-set",
    "re-skin", "re-stal", "re-submit", "re-tel", "re-top", "re-transmit",
    "re-trim", "re-wrap", "readmit", "reallot", "rebel", "rebid", "rebin", "rebut",
    "recap", "rechannel", "recommit", "recrop", "recur", "recut", "red", "redril",
    "refer", "refit", "reformat", "refret", "refuel", "reget", "regret", "reinter",
    "rejig", "rekit", "reknot", "relabel", "relet", "rem", "remap", "remetal",
    "remit", "remodel", "reoccur", "rep", "repel", "repin", "replan", "replot",
    "repol", "repot", "reprogram", "rerun", "reset", "resignal", "resit", "reskil",
    "resubmit", "retransfer", "retransmit", "retro-fit", "retrofit", "rev",
    "revel", "revet", "rewrap", "rib", "richochet", "ricochet", "rid", "rig",
    "rim", "ringlet", "rip", "rit", "rival", "rivet", "roadrun", "rob", "rocket",
    "rod", "roset", "rot", "rowel", "rub", "run", "runnel", "rut", "sab", "sad",
    "sag", "sandbag", "sap", "scab", "scalpel", "scam", "scan", "scar", "scat",
    "schlep", "scrag", "scram", "shall", "sled", "smut", "stet", "sulfuret",
    "trepan", "unrip", "unstop", "whir", "whop", "wig", "scrap", "scrat", "scrub",
    "scrum", "scud", "scum", "scur", "semi-control", "semi-skil", "semi-skim",
    "semiskil", "sentinel", "set", "shag", "sham", "shed", "shim", "shin", "ship",
    "shir", "shit", "shlap", "shop", "shopfit", "shortfal", "shot", "shovel",
    "shred", "shrinkwrap", "shrivel", "shrug", "shun", "shut", "side-step",
    "sideslip", "sidestep", "signal", "sin", "sinbin", "sip", "sit", "skid",
    "skim", "skin", "skip", "skir", "skrag", "slab", "slag", "slam", "slap",
    "slim", "slip", "slit", "slob", "slog", "slop", "slot", "slowclap", "slug",
    "slum", "slur", "smit", "snag", "snap", "snip", "snivel", "snog", "snorkel",
    "snowcem", "snub", "snug", "sob", "sod", "softpedal", "son", "sop", "spam",
    "span", "spar", "spat", "spiderweb", "spin", "spiral", "spit", "splat",
    "split", "spot", "sprag", "spraygun", "sprig", "springtip", "spud", "spur",
    "squat", "squirrel", "stab", "stag", "star", "stem", "sten", "stencil", "step",
    "stir", "stop", "storytel", "strap", "strim", "strip", "strop", "strug",
    "strum", "strut", "stub", "stud", "stun", "sub", "subcrop", "sublet", "submit",
    "subset", "suedetrim", "sum", "summit", "sun", "suntan", "sup", "super-chil",
    "superad", "swab", "swag", "swan", "swap", "swat", "swig", "swim", "swivel",
    "swot", "tab", "tag", "tan", "tansfer", "tap", "tar", "tassel", "tat", "tefer",
    "teleshop", "tendril", "terschel", "th'strip", "thermal", "thermostat", "thin",
    "throb", "thrum", "thud", "thug", "tightlip", "tin", "tinsel", "tip", "tittup",
    "toecap", "tog", "tom", "tomorrow", "top", "tot", "total", "towel", "traget",
    "trainspot", "tram", "trammel", "transfer", "tranship", "transit", "transmit",
    "transship", "trap", "travel", "trek", "trendset", "trim", "trip", "tripod",
    "trod", "trog", "trot", "trousseaushop", "trowel", "trup", "tub", "tug",
    "tunnel", "tup", "tut", "twat", "twig", "twin", "twit", "typeset", "tyset",
    "un-man", "unban", "unbar", "unbob", "uncap", "unclip", "uncompel", "undam",
    "under-bil", "under-cut", "under-fit", "under-pin", "under-skil", "underbid",
    "undercut", "underlet", "underman", "underpin", "unfit", "unfulfil", "unknot",
    "unlip", "unlywil", "unman", "unpad", "unpeg", "unpin", "unplug", "unravel",
    "unrol", "unscrol", "unsnap", "unstal", "unstep", "unstir", "untap", "unwrap",
    "unzip", "up", "upset", "upskil", "upwel", "ven", "verbal", "vet", "victual",
    "vignet", "wad", "wag", "wainscot", "wan", "war", "water-log", "waterfal",
    "waterfil", "waterlog", "weasel", "web", "wed", "wet", "wham", "whet", "whip",
    "whir", "whiteskin", "whiz", "whup", "wildcat", "win", "windmil", "wit",
    "woodchop", "woodcut", "wor", "worship", "wrap", "wiretap", "yen", "yak",
    "yap", "yarnspin", "yip", "yodel", "zag", "zap", "zig", "zig-zag", "zigzag",
    "zip", "ztrip", "hand-bag", "hocus", "hocus-pocus"
  ],

  PAST_PARTICIPLE_RULESET = {
    name: "PAST_PARTICIPLE",
    defaultRule: RE(ANY_STEM, 0, "ed", 2),
    rules: PAST_PARTICIPLE_RULES,
    doubling: true
  },

  PRESENT_PARTICIPLE_RULESET = {
    name: "ING_FORM",
    defaultRule: RE(ANY_STEM, 0, "ing", 2),
    rules: ING_FORM_RULES,
    doubling: true
  },

  PAST_TENSE_RULESET = {
    name: "PAST_TENSE",
    defaultRule: RE(ANY_STEM, 0, "ed", 2),
    rules: PAST_TENSE_RULES,
    doubling: true
  },

  PRESENT_TENSE_RULESET = {
    name: "PRESENT_TENSE",
    defaultRule: RE(ANY_STEM, 0, "s", 2),
    rules: PRESENT_TENSE_RULES,
    doubling: false
  };

// ///////////////////////////// End Functions ////////////////////////////////////

if (!RiTa.SILENT && !isNode() && console)
  console.log('[INFO] RiTaJS.version [' + RiTa.VERSION + ']');

_RiTa_LTS=[
'TOTAL 13100',
'INDEX 0 a',
'STATE 4 r 2 1',
'STATE 6 0 4 3',
'STATE 3 e 6 5',
'STATE 4 u 8 7',
'STATE 4 y 10 9',
'STATE 3 w 12 11',
'STATE 5 t 14 13',
'STATE 3 e 16 15',
'STATE 3 e 18 17',
'STATE 4 # 20 19',
'STATE 3 w 22 21',
'STATE 1 0 24 23',
'STATE 5 e 26 25',
'STATE 5 n 28 27',
'STATE 2 h 29 28',
'STATE 5 e 31 30',
'STATE 2 r 33 32',
'STATE 5 e 35 34',
'STATE 5 t 28 36',
'STATE 3 e 38 37',
'STATE 3 e 40 39',
'STATE 2 c 42 41',
'STATE 2 a 43 42',
'STATE 5 y 45 44',
'STATE 5 r 47 46',
'STATE 5 d 49 48',
'PHONE eh1',
'STATE 5 m 51 50',
'PHONE epsilon',
'PHONE aa1',
'STATE 4 i 53 52',
'STATE 4 g 55 54',
'STATE 4 n 57 56',
'STATE 1 # 59 58',
'STATE 5 g 61 60',
'STATE 1 0 63 62',
'STATE 1 u 29 64',
'STATE 3 o 28 65',
'STATE 4 u 67 66',
'STATE 3 a 29 68',
'STATE 2 l 68 69',
'STATE 1 # 42 70',
'PHONE ey1',
'STATE 1 l 72 71',
'STATE 5 i 74 73',
'STATE 3 n 26 75',
'STATE 5 a 77 76',
'STATE 2 # 79 78',
'STATE 5 i 26 80',
'STATE 1 0 82 81',
'STATE 2 w 28 83',
'STATE 2 r 29 84',
'STATE 5 i 86 85',
'STATE 5 r 88 87',
'STATE 6 # 90 89',
'STATE 1 0 92 91',
'STATE 4 b 94 93',
'STATE 1 # 96 95',
'STATE 4 k 98 97',
'STATE 5 t 100 99',
'STATE 1 0 102 101',
'STATE 2 # 104 103',
'STATE 1 # 63 105',
'PHONE aw1',
'STATE 6 e 107 106',
'STATE 4 w 109 108',
'STATE 4 d 28 110',
'STATE 1 o 111 107',
'PHONE ah',
'STATE 2 y 68 112',
'STATE 2 p 42 113',
'STATE 1 t 72 42',
'PHONE ey',
'STATE 5 e 115 114',
'STATE 6 # 29 116',
'STATE 2 c 26 117',
'STATE 5 i 119 118',
'STATE 3 p 26 120',
'STATE 6 o 100 28',
'STATE 3 n 26 121',
'STATE 1 0 82 122',
'STATE 1 # 82 123',
'PHONE ao1',
'STATE 1 s 125 124',
'STATE 1 # 28 126',
'STATE 4 y 128 127',
'STATE 6 o 130 129',
'STATE 5 n 132 131',
'STATE 3 z 133 26',
'STATE 3 w 135 134',
'STATE 4 t 137 136',
'STATE 1 # 139 138',
'STATE 3 m 100 140',
'STATE 1 i 142 141',
'STATE 1 # 28 68',
'STATE 1 c 28 143',
'STATE 5 d 100 144',
'STATE 4 t 146 145',
'STATE 1 b 42 28',
'STATE 6 n 148 147',
'PHONE ae1',
'STATE 1 # 150 149',
'STATE 3 c 152 151',
'STATE 2 n 154 153',
'STATE 3 l 100 82',
'PHONE aw',
'STATE 1 o 111 155',
'PHONE ow1',
'STATE 1 # 157 156',
'STATE 3 l 82 158',
'STATE 4 n 160 159',
'PHONE ow',
'STATE 1 c 28 68',
'STATE 1 l 162 161',
'STATE 5 r 164 163',
'STATE 6 l 166 165',
'STATE 6 z 28 167',
'STATE 2 e 26 168',
'STATE 5 e 170 169',
'STATE 6 s 172 171',
'STATE 6 # 29 173',
'STATE 3 h 175 174',
'STATE 5 # 28 176',
'STATE 1 r 82 28',
'STATE 5 l 178 177',
'STATE 6 # 179 126',
'PHONE ih1',
'STATE 2 # 181 180',
'STATE 5 a 183 182',
'STATE 6 # 29 184',
'STATE 4 t 186 185',
'STATE 5 l 188 187',
'STATE 3 t 190 189',
'PHONE ay',
'STATE 6 d 192 191',
'STATE 4 t 194 193',
'STATE 3 i 196 195',
'STATE 3 u 198 197',
'STATE 3 i 28 199',
'STATE 2 s 42 200',
'STATE 6 r 42 201',
'STATE 1 t 203 202',
'STATE 2 n 42 204',
'STATE 1 r 68 205',
'STATE 2 d 100 28',
'STATE 5 o 28 206',
'STATE 1 c 42 207',
'STATE 5 e 28 208',
'STATE 4 g 42 28',
'STATE 5 l 210 209',
'STATE 3 h 82 211',
'STATE 2 # 213 212',
'STATE 6 a 82 214',
'STATE 1 # 82 215',
'PHONE ao',
'STATE 2 b 217 216',
'STATE 4 n 219 218',
'STATE 3 a 29 220',
'STATE 1 e 154 82',
'STATE 2 n 222 221',
'STATE 1 c 28 223',
'STATE 3 d 42 224',
'STATE 3 d 225 72',
'STATE 5 t 227 226',
'STATE 2 q 82 228',
'STATE 6 t 28 229',
'STATE 2 c 28 230',
'STATE 6 o 232 231',
'STATE 3 u 26 233',
'STATE 5 o 235 234',
'STATE 6 e 28 236',
'STATE 6 # 29 237',
'STATE 2 # 238 28',
'STATE 6 s 240 239',
'STATE 6 y 26 241',
'STATE 6 e 100 242',
'STATE 2 e 28 243',
'STATE 2 b 245 244',
'STATE 2 p 28 246',
'STATE 2 h 247 28',
'STATE 1 # 249 248',
'STATE 3 w 251 250',
'STATE 5 o 253 252',
'STATE 6 s 230 254',
'STATE 6 z 256 255',
'STATE 4 n 100 257',
'STATE 1 0 100 258',
'STATE 6 # 260 259',
'STATE 3 m 261 42',
'STATE 1 a 263 262',
'STATE 2 s 42 264',
'STATE 6 s 266 265',
'STATE 4 t 268 267',
'STATE 4 l 68 42',
'STATE 2 t 29 82',
'STATE 3 o 28 269',
'STATE 4 l 29 100',
'STATE 3 t 42 270',
'STATE 2 t 42 271',
'STATE 6 n 68 272',
'STATE 3 r 42 273',
'STATE 6 # 42 274',
'STATE 1 # 28 275',
'STATE 4 t 276 28',
'STATE 4 s 28 277',
'STATE 2 c 68 278',
'STATE 4 n 68 279',
'STATE 1 t 28 280',
'STATE 5 s 68 281',
'STATE 6 # 283 282',
'STATE 3 n 107 284',
'STATE 2 c 82 285',
'STATE 5 t 82 286',
'STATE 5 n 82 287',
'STATE 5 s 29 288',
'STATE 2 e 154 289',
'STATE 1 e 111 107',
'STATE 5 c 107 290',
'STATE 4 l 292 291',
'STATE 3 m 68 293',
'STATE 4 u 63 294',
'STATE 1 a 28 295',
'STATE 1 i 68 28',
'STATE 1 # 28 296',
'STATE 2 t 42 297',
'STATE 2 i 72 42',
'STATE 3 o 28 298',
'STATE 2 q 82 299',
'STATE 1 r 26 300',
'STATE 1 c 302 301',
'PHONE aa',
'STATE 6 a 26 303',
'STATE 3 s 29 304',
'STATE 3 r 26 305',
'STATE 5 y 307 306',
'STATE 6 l 309 308',
'STATE 2 # 311 310',
'STATE 3 v 313 312',
'STATE 3 p 28 26',
'STATE 6 b 100 314',
'STATE 3 h 28 29',
'STATE 6 i 316 315',
'STATE 6 o 26 100',
'STATE 1 # 82 317',
'STATE 5 c 28 318',
'STATE 5 d 126 28',
'STATE 1 # 319 28',
'PHONE ih',
'STATE 5 o 321 320',
'STATE 3 o 323 322',
'STATE 4 w 325 324',
'STATE 4 l 327 326',
'STATE 5 r 328 42',
'STATE 2 # 42 329',
'STATE 2 a 331 330',
'STATE 6 n 333 332',
'STATE 1 0 100 68',
'STATE 1 0 335 334',
'STATE 3 n 337 336',
'STATE 5 o 28 338',
'STATE 5 a 29 42',
'STATE 2 e 72 42',
'STATE 6 g 29 339',
'STATE 3 r 42 340',
'STATE 6 e 42 341',
'STATE 4 k 343 342',
'STATE 4 t 345 344',
'STATE 3 o 28 346',
'STATE 3 o 28 42',
'STATE 2 a 348 347',
'STATE 2 i 350 349',
'STATE 1 a 42 68',
'STATE 6 r 352 351',
'STATE 3 l 100 68',
'STATE 2 # 354 353',
'STATE 2 m 42 355',
'STATE 5 r 100 68',
'STATE 4 l 28 68',
'STATE 2 g 68 356',
'STATE 4 l 28 357',
'STATE 5 h 28 358',
'STATE 4 p 68 359',
'STATE 5 s 361 360',
'STATE 5 d 107 362',
'STATE 3 s 82 363',
'STATE 6 e 365 364',
'STATE 5 d 82 366',
'STATE 5 r 368 367',
'STATE 5 l 29 82',
'STATE 2 o 154 369',
'STATE 6 o 107 370',
'STATE 4 s 372 371',
'STATE 3 h 373 68',
'STATE 3 i 68 374',
'STATE 4 i 28 375',
'STATE 4 k 28 376',
'STATE 1 o 68 377',
'STATE 1 c 42 378',
'STATE 1 # 380 379',
'STATE 1 # 29 381',
'STATE 1 e 383 382',
'STATE 6 # 385 384',
'STATE 3 s 29 230',
'STATE 6 c 387 386',
'STATE 3 c 26 388',
'STATE 2 i 26 389',
'STATE 5 u 391 390',
'STATE 6 # 26 392',
'STATE 6 # 29 393',
'STATE 3 p 28 394',
'STATE 6 n 28 230',
'STATE 3 m 230 395',
'STATE 6 t 26 396',
'STATE 6 e 28 26',
'STATE 2 # 398 397',
'STATE 3 m 400 399',
'STATE 3 m 26 401',
'STATE 2 h 82 402',
'STATE 5 s 404 403',
'STATE 6 e 28 126',
'STATE 5 l 406 405',
'STATE 6 # 29 407',
'STATE 4 w 409 408',
'STATE 2 j 29 28',
'STATE 4 l 411 410',
'STATE 5 a 413 412',
'STATE 4 g 100 414',
'STATE 6 o 288 82',
'STATE 6 e 26 42',
'STATE 6 # 42 415',
'STATE 3 r 42 416',
'STATE 6 # 29 68',
'STATE 6 a 418 417',
'STATE 4 t 420 419',
'STATE 2 t 100 421',
'STATE 4 d 42 422',
'STATE 3 u 42 423',
'STATE 2 i 42 424',
'STATE 6 a 426 425',
'STATE 2 a 26 427',
'STATE 3 g 68 42',
'STATE 1 u 68 428',
'STATE 4 y 42 429',
'STATE 3 m 42 430',
'STATE 4 w 82 431',
'STATE 3 u 42 432',
'STATE 4 w 82 433',
'STATE 1 0 42 434',
'STATE 1 l 29 435',
'STATE 3 n 437 436',
'STATE 3 d 42 438',
'STATE 6 o 42 439',
'STATE 2 l 247 440',
'STATE 6 n 68 42',
'STATE 6 n 442 441',
'STATE 1 a 444 443',
'STATE 5 s 68 28',
'STATE 5 t 445 28',
'STATE 1 h 28 68',
'STATE 4 f 68 446',
'STATE 2 s 28 447',
'STATE 6 t 82 448',
'STATE 3 n 82 449',
'STATE 6 t 450 82',
'STATE 5 s 452 451',
'STATE 5 b 63 453',
'STATE 6 t 455 454',
'STATE 5 l 457 456',
'STATE 3 l 82 458',
'STATE 1 0 460 459',
'STATE 1 # 111 107',
'STATE 4 u 105 461',
'STATE 3 a 230 462',
'STATE 2 t 463 68',
'STATE 1 n 465 464',
'STATE 4 o 28 466',
'STATE 4 t 28 467',
'STATE 2 r 28 468',
'STATE 1 e 42 469',
'STATE 6 e 471 470',
'STATE 5 a 473 472',
'STATE 3 p 29 474',
'STATE 2 c 476 475',
'STATE 2 r 100 26',
'STATE 3 v 29 477',
'STATE 3 c 26 478',
'STATE 1 i 26 479',
'STATE 2 a 28 100',
'STATE 3 l 29 26',
'STATE 2 s 26 480',
'STATE 3 h 482 481',
'STATE 6 s 230 28',
'STATE 3 m 100 26',
'STATE 6 s 230 483',
'STATE 3 c 26 484',
'STATE 6 l 230 485',
'STATE 6 b 100 486',
'STATE 6 m 26 28',
'STATE 3 n 230 487',
'STATE 3 g 100 488',
'STATE 6 o 230 29',
'STATE 3 p 29 100',
'STATE 5 t 28 29',
'STATE 5 r 28 489',
'STATE 6 # 126 28',
'STATE 1 0 491 490',
'STATE 4 b 493 492',
'STATE 4 t 495 494',
'STATE 4 l 497 496',
'STATE 5 a 68 498',
'STATE 5 a 500 499',
'STATE 5 a 502 501',
'STATE 6 r 68 503',
'STATE 3 k 29 68',
'STATE 5 t 505 504',
'STATE 1 # 42 72',
'STATE 3 p 42 28',
'STATE 6 v 507 506',
'STATE 2 # 509 508',
'STATE 3 o 28 510',
'STATE 1 0 512 511',
'STATE 3 n 29 513',
'STATE 4 v 42 514',
'STATE 1 i 515 42',
'STATE 1 e 100 516',
'STATE 5 s 518 517',
'STATE 3 m 42 519',
'STATE 3 m 42 520',
'STATE 1 0 42 521',
'STATE 2 # 523 522',
'STATE 2 # 525 524',
'STATE 4 l 527 526',
'STATE 1 t 529 528',
'STATE 4 x 100 530',
'STATE 2 i 532 531',
'STATE 4 l 29 533',
'STATE 3 r 535 534',
'STATE 2 e 42 536',
'STATE 1 c 42 537',
'STATE 2 s 539 538',
'STATE 3 l 72 540',
'STATE 3 h 100 541',
'STATE 3 h 42 100',
'STATE 4 d 28 542',
'STATE 4 t 68 28',
'STATE 1 b 28 100',
'STATE 5 l 100 543',
'STATE 6 h 105 544',
'STATE 3 h 63 545',
'STATE 3 b 63 546',
'STATE 3 f 82 107',
'STATE 5 t 547 82',
'STATE 2 k 63 82',
'STATE 2 k 63 548',
'STATE 5 r 82 549',
'STATE 5 s 82 498',
'STATE 3 g 551 550',
'STATE 6 i 82 552',
'STATE 6 e 105 82',
'STATE 2 c 82 553',
'STATE 6 u 82 554',
'STATE 4 i 556 555',
'STATE 3 v 558 557',
'STATE 1 n 29 82',
'STATE 2 s 560 559',
'STATE 3 h 562 561',
'STATE 3 i 68 563',
'STATE 1 e 68 564',
'STATE 1 a 68 565',
'STATE 1 t 42 566',
'STATE 3 l 568 567',
'STATE 3 b 569 29',
'STATE 5 o 571 570',
'STATE 6 # 29 572',
'STATE 3 c 29 573',
'STATE 3 h 100 574',
'STATE 6 i 100 575',
'STATE 3 p 26 576',
'STATE 3 f 26 577',
'STATE 3 l 579 578',
'STATE 2 u 26 580',
'STATE 5 k 29 581',
'STATE 6 o 582 29',
'STATE 6 u 28 583',
'STATE 3 k 26 100',
'STATE 6 # 26 584',
'STATE 6 e 26 585',
'STATE 6 c 230 586',
'STATE 6 # 29 587',
'STATE 2 l 589 588',
'STATE 3 o 591 590',
'STATE 4 l 593 592',
'STATE 6 y 595 594',
'STATE 6 i 100 596',
'STATE 6 s 598 597',
'STATE 6 r 600 599',
'STATE 2 t 602 601',
'STATE 2 s 604 603',
'STATE 5 c 29 82',
'STATE 5 o 606 605',
'STATE 6 # 608 607',
'STATE 5 o 610 609',
'STATE 6 # 29 611',
'STATE 5 h 82 612',
'STATE 4 c 614 613',
'STATE 4 s 42 29',
'STATE 6 e 616 615',
'STATE 4 t 618 617',
'STATE 4 z 29 619',
'STATE 3 r 42 620',
'STATE 4 g 622 621',
'STATE 3 u 623 42',
'STATE 3 l 230 42',
'STATE 4 z 100 624',
'STATE 2 # 230 29',
'STATE 2 r 625 42',
'STATE 1 d 100 42',
'STATE 5 a 28 626',
'STATE 6 t 29 627',
'STATE 1 # 42 628',
'STATE 3 l 629 42',
'STATE 6 t 68 630',
'STATE 1 # 632 631',
'STATE 6 k 634 633',
'STATE 1 # 42 635',
'STATE 3 t 42 636',
'STATE 4 n 638 637',
'STATE 1 0 640 639',
'STATE 3 g 642 641',
'STATE 3 m 68 42',
'STATE 1 0 644 643',
'STATE 3 a 29 645',
'STATE 4 l 29 42',
'STATE 4 c 29 646',
'STATE 3 l 42 647',
'STATE 2 a 68 648',
'STATE 1 i 68 649',
'STATE 3 m 68 650',
'STATE 1 e 652 651',
'STATE 3 s 331 42',
'STATE 3 r 29 653',
'STATE 3 p 100 42',
'STATE 6 o 28 654',
'STATE 6 g 68 655',
'STATE 2 e 657 656',
'STATE 3 r 63 82',
'STATE 5 s 659 658',
'STATE 2 t 63 82',
'STATE 3 r 82 63',
'STATE 5 b 82 660',
'STATE 3 s 662 661',
'STATE 5 c 63 663',
'STATE 6 t 82 664',
'STATE 1 o 154 665',
'STATE 6 m 230 82',
'STATE 4 o 667 666',
'STATE 2 a 29 28',
'STATE 3 j 230 68',
'STATE 2 i 230 68',
'STATE 3 h 669 668',
'STATE 3 s 68 670',
'STATE 3 g 672 671',
'PHONE ae',
'STATE 3 u 29 673',
'STATE 4 m 28 674',
'STATE 1 e 28 68',
'STATE 2 o 42 675',
'STATE 3 p 677 676',
'STATE 2 u 28 678',
'STATE 1 e 230 29',
'STATE 2 s 680 679',
'STATE 3 h 26 29',
'STATE 3 h 682 681',
'STATE 2 g 28 683',
'STATE 6 # 100 684',
'STATE 6 e 26 100',
'STATE 1 # 686 685',
'STATE 1 # 26 687',
'STATE 6 e 26 688',
'STATE 6 t 26 689',
'STATE 3 i 28 690',
'STATE 6 s 29 691',
'STATE 5 b 29 692',
'STATE 3 m 28 693',
'STATE 3 l 28 694',
'STATE 6 c 100 695',
'STATE 3 c 100 696',
'STATE 6 e 698 697',
'STATE 1 n 28 699',
'STATE 1 c 126 700',
'STATE 4 n 702 701',
'STATE 5 t 704 703',
'STATE 4 e 28 705',
'STATE 5 a 707 706',
'STATE 4 l 709 708',
'STATE 3 c 28 68',
'STATE 1 0 42 68',
'STATE 4 g 711 710',
'STATE 4 g 230 712',
'STATE 2 e 68 713',
'STATE 1 d 42 714',
'STATE 4 a 28 715',
'STATE 5 s 717 716',
'STATE 6 # 719 718',
'STATE 3 t 82 720',
'STATE 4 a 28 721',
'STATE 6 # 29 722',
'STATE 3 v 724 723',
'STATE 4 n 100 29',
'STATE 5 m 726 725',
'STATE 6 # 29 727',
'STATE 3 b 100 728',
'STATE 3 p 729 82',
'STATE 4 a 28 730',
'STATE 5 h 82 100',
'STATE 6 c 732 731',
'STATE 4 n 734 733',
'STATE 4 s 42 735',
'STATE 2 r 68 736',
'STATE 3 t 738 737',
'STATE 3 m 740 739',
'STATE 4 w 82 741',
'STATE 2 i 100 742',
'PHONE w-ey1',
'STATE 1 o 42 743',
'STATE 3 m 42 100',
'STATE 6 s 42 744',
'STATE 3 m 42 745',
'STATE 3 r 42 28',
'STATE 2 p 42 746',
'STATE 1 o 42 747',
'STATE 1 0 749 748',
'STATE 6 r 751 750',
'STATE 3 v 753 752',
'STATE 4 c 29 754',
'STATE 6 n 42 755',
'STATE 3 l 42 756',
'STATE 4 x 100 757',
'STATE 2 0 100 758',
'STATE 2 o 29 759',
'STATE 2 # 42 230',
'STATE 2 c 42 760',
'STATE 2 e 68 42',
'STATE 4 l 68 761',
'STATE 3 m 100 762',
'STATE 1 # 42 763',
'STATE 4 d 42 740',
'STATE 3 i 765 764',
'STATE 2 u 68 766',
'STATE 1 t 68 42',
'STATE 1 t 42 767',
'STATE 2 e 247 768',
'STATE 3 g 42 247',
'STATE 3 n 68 247',
'STATE 5 h 28 769',
'STATE 6 u 68 28',
'STATE 2 t 105 770',
'STATE 6 e 771 82',
'STATE 3 h 107 772',
'STATE 3 h 63 105',
'STATE 6 i 63 773',
'STATE 6 # 82 774',
'STATE 5 t 154 82',
'STATE 5 d 111 775',
'STATE 6 e 82 776',
'STATE 2 a 154 777',
'STATE 4 e 628 778',
'STATE 2 a 28 105',
'STATE 3 w 780 779',
'STATE 1 l 562 781',
'STATE 3 t 100 782',
'STATE 2 a 100 68',
'STATE 2 i 68 562',
'STATE 3 h 100 783',
'STATE 1 # 28 784',
'STATE 3 l 786 785',
'STATE 5 k 788 787',
'STATE 5 a 790 789',
'STATE 2 c 28 791',
'STATE 3 h 29 792',
'STATE 3 c 793 29',
'STATE 2 g 26 28',
'STATE 6 c 26 28',
'STATE 6 y 29 794',
'STATE 2 a 795 29',
'STATE 1 e 26 796',
'STATE 3 h 26 797',
'STATE 3 m 26 798',
'STATE 6 u 26 799',
'STATE 1 # 26 28',
'STATE 1 a 28 800',
'STATE 5 c 802 801',
'STATE 5 p 230 29',
'STATE 3 l 100 803',
'STATE 3 p 804 26',
'STATE 3 c 230 805',
'STATE 3 t 230 806',
'STATE 3 d 100 807',
'STATE 3 f 26 808',
'STATE 2 t 28 809',
'STATE 1 # 126 28',
'STATE 5 a 811 810',
'STATE 5 g 813 812',
'STATE 4 n 68 28',
'STATE 2 c 28 100',
'STATE 4 n 815 814',
'STATE 5 t 817 816',
'STATE 6 r 68 818',
'STATE 4 h 820 819',
'STATE 6 # 822 821',
'STATE 2 a 824 823',
'STATE 6 n 100 68',
'STATE 1 o 100 825',
'STATE 2 u 68 826',
'STATE 2 t 42 827',
'STATE 3 i 829 828',
'STATE 5 a 230 830',
'STATE 6 o 100 831',
'STATE 5 t 82 832',
'STATE 5 a 29 833',
'STATE 3 m 82 834',
'STATE 4 e 836 835',
'STATE 6 r 838 837',
'STATE 6 n 840 839',
'STATE 4 c 72 841',
'STATE 5 d 843 842',
'STATE 6 a 100 844',
'STATE 6 s 68 845',
'STATE 6 s 29 846',
'STATE 5 l 230 82',
'STATE 5 g 100 847',
'STATE 6 t 849 848',
'STATE 4 t 100 850',
'STATE 1 0 852 851',
'STATE 1 0 853 68',
'STATE 4 l 68 230',
'STATE 2 e 68 854',
'STATE 4 n 856 855',
'STATE 4 l 858 857',
'STATE 3 d 42 859',
'STATE 4 n 42 29',
'STATE 4 y 42 860',
'STATE 1 0 100 247',
'STATE 4 c 42 861',
'STATE 6 n 42 862',
'STATE 3 n 42 863',
'STATE 6 e 42 68',
'STATE 6 i 42 864',
'STATE 3 o 28 865',
'STATE 4 m 867 866',
'STATE 2 s 869 868',
'STATE 3 o 28 870',
'STATE 6 l 872 871',
'STATE 4 l 873 562',
'STATE 4 s 29 100',
'STATE 3 b 42 68',
'STATE 3 b 42 100',
'STATE 4 j 68 874',
'STATE 2 e 68 875',
'STATE 1 # 68 876',
'STATE 3 l 878 877',
'STATE 4 h 68 879',
'STATE 4 l 100 880',
'STATE 4 c 882 881',
'STATE 2 e 42 883',
'STATE 2 c 42 884',
'STATE 1 t 68 885',
'STATE 1 d 42 886',
'STATE 3 p 42 887',
'STATE 1 u 28 888',
'STATE 1 r 29 889',
'STATE 3 l 105 82',
'STATE 3 r 63 771',
'STATE 5 s 63 890',
'STATE 5 m 63 891',
'STATE 5 t 82 63',
'STATE 6 # 82 892',
'STATE 2 r 154 82',
'STATE 4 k 894 893',
'STATE 3 g 68 895',
'STATE 2 o 68 896',
'STATE 1 u 68 897',
'STATE 3 c 100 68',
'STATE 3 w 29 898',
'STATE 4 f 28 899',
'STATE 1 u 901 900',
'STATE 1 i 42 902',
'STATE 5 a 904 903',
'STATE 3 m 29 905',
'STATE 2 p 28 906',
'STATE 6 g 26 28',
'STATE 1 o 28 907',
'STATE 3 u 29 908',
'STATE 6 e 29 909',
'STATE 1 n 29 910',
'STATE 3 v 28 230',
'STATE 6 s 29 28',
'STATE 3 l 26 911',
'STATE 2 s 26 29',
'STATE 6 l 26 912',
'STATE 2 a 28 913',
'STATE 6 a 915 914',
'STATE 3 m 917 916',
'STATE 6 t 100 918',
'STATE 6 n 28 26',
'STATE 6 m 28 919',
'STATE 6 h 100 920',
'STATE 3 p 26 921',
'STATE 3 c 230 922',
'STATE 6 n 126 923',
'STATE 4 l 925 924',
'STATE 6 # 927 926',
'STATE 6 # 929 928',
'STATE 6 e 931 930',
'STATE 4 a 28 932',
'STATE 5 y 26 933',
'STATE 5 d 935 934',
'STATE 6 e 82 936',
'STATE 6 n 230 937',
'STATE 4 e 939 938',
'STATE 6 # 940 29',
'STATE 6 s 82 941',
'STATE 3 b 82 942',
'STATE 1 0 944 943',
'STATE 1 p 68 945',
'STATE 4 p 68 946',
'STATE 2 o 68 947',
'STATE 2 i 949 948',
'STATE 4 e 951 950',
'STATE 2 d 953 952',
'STATE 5 o 29 954',
'STATE 6 l 562 955',
'STATE 3 n 68 956',
'STATE 5 l 82 68',
'STATE 5 a 29 957',
'STATE 4 h 959 958',
'STATE 3 m 28 960',
'STATE 6 s 962 961',
'STATE 4 n 100 963',
'STATE 6 s 965 964',
'STATE 3 p 230 966',
'STATE 4 n 562 100',
'STATE 5 v 968 967',
'STATE 6 a 230 969',
'STATE 3 c 29 970',
'STATE 6 m 230 971',
'STATE 3 t 230 972',
'STATE 6 # 29 973',
'STATE 1 0 975 974',
'STATE 2 q 29 976',
'STATE 4 s 756 977',
'STATE 1 # 979 978',
'STATE 4 v 42 980',
'STATE 3 d 100 562',
'STATE 1 c 68 981',
'STATE 2 o 29 982',
'STATE 3 m 42 983',
'STATE 2 s 29 42',
'STATE 1 r 100 29',
'STATE 3 f 230 984',
'STATE 4 k 42 985',
'STATE 2 e 42 986',
'STATE 3 b 28 987',
'STATE 3 c 42 988',
'STATE 2 t 42 989',
'STATE 4 i 28 990',
'STATE 4 d 100 991',
'STATE 6 l 68 992',
'STATE 3 r 994 993',
'STATE 3 p 42 995',
'STATE 3 x 100 996',
'STATE 3 m 998 997',
'STATE 3 l 1000 999',
'STATE 6 r 68 100',
'STATE 3 r 42 1001',
'STATE 3 r 42 1002',
'STATE 2 n 68 29',
'STATE 2 i 529 42',
'STATE 1 m 42 1003',
'STATE 4 m 1005 1004',
'STATE 2 # 42 68',
'STATE 3 d 42 1006',
'STATE 2 p 42 1007',
'STATE 1 l 72 1008',
'STATE 1 e 68 42',
'STATE 1 d 68 42',
'STATE 3 n 42 1009',
'STATE 3 r 1011 1010',
'STATE 2 n 1012 28',
'STATE 5 d 82 1013',
'STATE 6 e 63 82',
'STATE 6 u 29 1014',
'STATE 6 k 82 1015',
'STATE 4 h 1017 1016',
'STATE 3 i 562 1018',
'STATE 3 f 100 1019',
'STATE 2 e 68 230',
'STATE 1 e 68 1020',
'STATE 2 o 68 1021',
'STATE 4 l 28 1022',
'STATE 2 r 42 1023',
'STATE 2 r 28 42',
'STATE 1 a 72 42',
'STATE 5 o 1025 1024',
'STATE 6 # 29 1026',
'STATE 2 m 29 230',
'STATE 5 o 28 1027',
'STATE 6 o 29 1028',
'STATE 3 l 29 1029',
'STATE 6 i 230 29',
'STATE 2 i 230 1030',
'STATE 6 d 26 28',
'STATE 6 t 1032 1031',
'STATE 1 u 28 1033',
'STATE 6 o 1035 1034',
'STATE 5 j 29 1036',
'STATE 3 s 230 909',
'STATE 6 e 230 1037',
'STATE 3 g 230 1038',
'STATE 6 z 28 1039',
'STATE 3 s 26 1040',
'STATE 6 u 230 1041',
'STATE 3 b 100 29',
'STATE 6 # 1042 126',
'STATE 4 a 28 1043',
'STATE 5 s 1045 1044',
'STATE 6 b 1047 1046',
'STATE 4 h 230 29',
'STATE 5 c 1049 1048',
'STATE 5 a 1051 1050',
'STATE 6 # 100 1052',
'STATE 3 l 100 1053',
'STATE 4 m 1055 1054',
'STATE 5 t 1057 1056',
'STATE 5 m 1059 1058',
'STATE 6 e 82 1060',
'STATE 6 a 82 1061',
'STATE 6 m 100 68',
'STATE 4 w 82 1062',
'STATE 2 c 68 1063',
'STATE 2 r 230 1064',
'STATE 2 h 82 1065',
'STATE 3 f 82 1066',
'STATE 4 b 1068 1067',
'STATE 6 i 1070 1069',
'STATE 6 t 230 1071',
'STATE 4 d 29 1072',
'STATE 1 o 68 1073',
'STATE 3 l 1075 1074',
'STATE 1 p 68 1076',
'STATE 3 w 1078 1077',
'STATE 3 r 28 1079',
'STATE 2 c 82 1080',
'STATE 6 n 1082 1081',
'STATE 4 c 1084 1083',
'STATE 6 c 562 1085',
'STATE 6 n 68 1086',
'STATE 3 c 1088 1087',
'STATE 4 n 1090 1089',
'STATE 6 a 1092 1091',
'STATE 3 h 28 1093',
'STATE 6 o 1095 1094',
'STATE 4 n 1097 1096',
'STATE 3 f 42 1098',
'STATE 4 t 1100 1099',
'STATE 4 n 100 1101',
'STATE 4 v 68 1102',
'STATE 6 e 1104 1103',
'STATE 3 m 29 1105',
'STATE 3 v 230 1106',
'STATE 3 b 29 1107',
'STATE 3 s 68 1108',
'STATE 3 s 100 1109',
'STATE 4 n 29 1110',
'STATE 6 u 42 1111',
'STATE 6 s 1113 1112',
'STATE 4 d 68 1114',
'STATE 2 t 68 1115',
'STATE 4 t 42 1116',
'STATE 4 z 42 193',
'STATE 4 l 1118 1117',
'STATE 2 i 1120 1119',
'STATE 2 s 29 1121',
'STATE 2 l 42 1122',
'STATE 4 n 29 100',
'STATE 1 # 1124 1123',
'STATE 2 b 29 1125',
'STATE 3 n 42 1126',
'STATE 1 0 1127 42',
'STATE 2 r 42 68',
'STATE 6 z 29 1128',
'STATE 4 n 100 1129',
'STATE 6 n 68 873',
'STATE 6 n 1131 1130',
'STATE 6 l 1133 1132',
'STATE 4 d 100 1134',
'STATE 2 d 42 1135',
'STATE 6 t 1137 1136',
'STATE 6 y 42 1138',
'STATE 3 h 1140 1139',
'STATE 4 b 42 68',
'STATE 2 # 1142 1141',
'STATE 1 0 42 1143',
'STATE 1 c 68 42',
'STATE 4 d 42 1144',
'STATE 1 # 42 68',
'STATE 4 l 1146 1145',
'STATE 2 s 42 1147',
'STATE 3 g 1148 42',
'STATE 3 g 42 1149',
'STATE 6 l 68 1150',
'STATE 2 t 42 1151',
'STATE 4 l 68 28',
'STATE 5 r 82 1152',
'STATE 6 i 82 1153',
'STATE 6 l 82 1154',
'STATE 4 m 1156 1155',
'STATE 2 r 1157 68',
'STATE 2 c 1159 1158',
'STATE 3 l 68 1160',
'STATE 2 t 68 1161',
'STATE 2 a 68 1162',
'STATE 2 r 68 28',
'STATE 3 s 28 1163',
'STATE 6 # 1165 1164',
'STATE 6 # 1166 28',
'STATE 3 h 1168 1167',
'STATE 1 e 28 1169',
'STATE 5 # 28 1170',
'STATE 3 i 28 1171',
'STATE 1 s 29 1172',
'STATE 1 m 29 1173',
'STATE 1 a 28 26',
'STATE 3 t 28 1174',
'STATE 5 q 1176 1175',
'STATE 5 t 1178 1177',
'STATE 3 f 29 1179',
'STATE 6 i 29 1180',
'STATE 6 c 28 1181',
'STATE 2 # 1182 100',
'STATE 3 k 26 1183',
'STATE 3 c 1185 1184',
'STATE 5 y 126 28',
'STATE 4 w 1187 1186',
'STATE 3 w 1189 1188',
'STATE 6 k 29 68',
'STATE 3 b 711 1190',
'STATE 3 l 100 42',
'STATE 6 s 1192 1191',
'STATE 6 e 1194 1193',
'STATE 5 k 100 1195',
'STATE 2 u 29 1196',
'STATE 6 i 42 1197',
'STATE 3 r 1198 42',
'STATE 5 r 1200 1199',
'STATE 5 a 1202 1201',
'STATE 5 n 1204 1203',
'STATE 6 o 100 1205',
'STATE 5 u 68 1206',
'STATE 6 o 29 230',
'STATE 6 r 82 29',
'STATE 6 o 100 1207',
'STATE 4 o 28 1208',
'STATE 1 a 133 42',
'STATE 2 n 29 1209',
'STATE 2 s 1211 1210',
'STATE 3 c 82 1212',
'STATE 3 h 1214 1213',
'STATE 1 b 42 1215',
'STATE 6 r 1217 1216',
'STATE 4 l 230 68',
'STATE 4 m 68 1218',
'STATE 4 n 29 68',
'STATE 2 a 68 937',
'STATE 1 n 42 1219',
'STATE 1 g 72 42',
'STATE 3 c 42 1220',
'STATE 2 a 1222 1221',
'STATE 5 g 100 1223',
'STATE 2 s 42 28',
'STATE 2 g 1225 1224',
'STATE 4 g 68 1226',
'STATE 4 m 68 100',
'STATE 3 h 100 1227',
'STATE 5 k 100 1228',
'STATE 6 a 100 1229',
'STATE 3 h 100 1230',
'STATE 5 l 1231 29',
'STATE 5 p 100 1232',
'STATE 3 o 28 1233',
'STATE 5 g 1235 1234',
'STATE 6 e 29 1236',
'STATE 5 r 28 29',
'STATE 5 n 28 1237',
'STATE 6 n 1239 1238',
'STATE 4 b 562 1240',
'STATE 3 l 68 29',
'STATE 3 j 68 562',
'STATE 4 b 1242 1241',
'STATE 4 n 1244 1243',
'STATE 3 d 42 1245',
'STATE 4 t 68 29',
'STATE 4 h 68 1246',
'STATE 6 # 82 1247',
'STATE 5 t 82 1248',
'STATE 6 o 29 1249',
'STATE 6 o 1251 1250',
'STATE 6 e 230 1252',
'STATE 6 g 100 1253',
'STATE 6 y 68 1254',
'STATE 5 m 82 1255',
'STATE 1 # 1257 1256',
'STATE 6 k 608 1258',
'STATE 4 o 28 1259',
'STATE 3 b 68 1260',
'STATE 4 n 100 1261',
'STATE 3 v 230 68',
'STATE 2 # 1262 100',
'STATE 2 # 230 42',
'STATE 3 l 42 1263',
'STATE 3 t 42 68',
'STATE 2 g 1265 1264',
'STATE 1 # 29 1266',
'STATE 2 # 1268 1267',
'STATE 3 t 29 1269',
'STATE 4 s 29 42',
'STATE 5 c 42 1270',
'STATE 3 d 42 1271',
'STATE 6 o 1273 1272',
'STATE 4 b 1275 1274',
'STATE 4 t 1277 1276',
'STATE 2 p 230 1278',
'STATE 2 g 42 1279',
'STATE 4 v 1280 100',
'STATE 3 h 42 1281',
'STATE 2 f 42 1282',
'STATE 6 i 68 1283',
'STATE 4 n 68 1284',
'STATE 4 t 68 1285',
'STATE 4 n 1287 1286',
'STATE 4 z 42 100',
'STATE 4 k 42 1288',
'STATE 3 t 100 42',
'STATE 3 l 42 271',
'STATE 4 y 42 1289',
'STATE 3 g 42 1290',
'STATE 1 e 42 1291',
'STATE 3 f 42 1096',
'STATE 2 n 72 42',
'STATE 3 c 42 68',
'STATE 1 i 247 1292',
'STATE 1 o 247 331',
'STATE 2 a 29 82',
'STATE 3 f 1294 1293',
'STATE 6 s 82 1295',
'STATE 4 a 28 1296',
'STATE 2 g 100 1297',
'STATE 1 a 68 29',
'STATE 3 y 562 1298',
'STATE 3 h 68 562',
'STATE 3 n 68 1299',
'STATE 2 o 230 1300',
'STATE 2 i 29 1301',
'STATE 2 s 42 72',
'STATE 5 # 1303 1302',
'STATE 3 y 29 1304',
'STATE 2 i 230 29',
'STATE 1 d 28 1305',
'STATE 1 s 100 28',
'STATE 6 # 28 1306',
'STATE 1 h 28 1307',
'STATE 2 a 1308 29',
'STATE 2 e 230 1309',
'STATE 3 u 230 1310',
'STATE 1 e 26 28',
'STATE 6 i 1312 1311',
'STATE 3 m 29 230',
'STATE 5 d 1314 1313',
'STATE 3 m 230 1315',
'STATE 5 t 230 1316',
'STATE 6 h 29 400',
'STATE 6 m 28 1317',
'STATE 6 n 1319 1318',
'STATE 6 l 230 1320',
'STATE 3 f 29 1321',
'STATE 6 a 230 100',
'STATE 5 t 1323 1322',
'STATE 6 k 29 82',
'STATE 6 # 1325 1324',
'STATE 5 k 1327 1326',
'STATE 6 y 68 1328',
'STATE 3 w 1330 1329',
'STATE 5 t 68 1331',
'STATE 6 y 68 1332',
'STATE 2 f 230 1333',
'STATE 5 d 1335 1334',
'STATE 1 t 29 1336',
'STATE 6 l 100 1337',
'STATE 1 a 42 1338',
'STATE 4 w 1340 1339',
'STATE 6 o 100 1341',
'STATE 5 u 68 1342',
'STATE 6 n 68 1343',
'STATE 6 l 100 1344',
'STATE 6 o 68 100',
'STATE 6 a 562 100',
'STATE 5 p 100 1345',
'STATE 6 i 29 100',
'STATE 4 g 1347 1346',
'STATE 3 d 29 230',
'STATE 3 w 82 1348',
'STATE 1 n 82 1349',
'STATE 2 e 1351 1350',
'STATE 4 v 1353 1352',
'STATE 2 s 100 1354',
'STATE 1 o 100 68',
'STATE 6 u 68 1355',
'STATE 4 m 100 1226',
'STATE 6 g 68 1356',
'STATE 3 i 42 1357',
'STATE 3 t 42 1358',
'STATE 3 u 1360 1359',
'STATE 3 v 29 1361',
'STATE 5 k 82 29',
'STATE 4 n 1363 1362',
'STATE 6 a 68 1364',
'STATE 4 n 100 68',
'STATE 3 r 100 29',
'STATE 5 t 100 1365',
'STATE 6 m 562 1366',
'STATE 6 e 29 1367',
'STATE 3 h 100 29',
'STATE 5 l 100 82',
'STATE 4 o 1369 1368',
'STATE 3 v 1371 1370',
'STATE 6 e 1373 1372',
'STATE 5 l 29 1374',
'STATE 5 t 28 1375',
'STATE 4 k 68 1376',
'STATE 4 x 100 1377',
'STATE 4 h 29 68',
'STATE 3 l 230 1378',
'STATE 3 l 42 100',
'STATE 4 k 1380 1379',
'STATE 3 m 100 1381',
'STATE 3 f 42 1382',
'STATE 4 p 68 1383',
'STATE 5 u 1385 1384',
'STATE 5 k 82 1386',
'STATE 3 s 1388 1387',
'STATE 6 e 1389 82',
'STATE 3 b 230 29',
'STATE 6 o 100 1390',
'STATE 6 r 100 1391',
'STATE 3 c 68 1392',
'STATE 5 r 29 1393',
'STATE 6 s 1395 1394',
'STATE 2 q 29 1396',
'STATE 6 m 1398 1397',
'STATE 4 n 100 1399',
'STATE 4 l 1401 1400',
'STATE 4 l 1403 1402',
'STATE 4 t 42 1404',
'STATE 3 n 42 68',
'STATE 4 l 1405 42',
'STATE 4 c 42 100',
'STATE 1 o 42 100',
'STATE 4 m 1407 1406',
'STATE 3 m 1409 1408',
'STATE 3 x 100 1410',
'STATE 1 t 42 1411',
'STATE 3 r 42 1412',
'STATE 4 m 1414 1413',
'STATE 2 n 100 42',
'STATE 6 l 230 1415',
'STATE 6 r 100 1416',
'STATE 3 i 68 1417',
'STATE 6 a 562 1418',
'STATE 4 m 68 42',
'STATE 2 t 42 1419',
'STATE 2 t 100 42',
'STATE 6 n 100 42',
'STATE 4 t 42 1420',
'STATE 6 y 1422 1421',
'STATE 4 m 100 1423',
'STATE 4 n 68 100',
'STATE 3 m 1425 1424',
'STATE 3 p 100 68',
'STATE 4 d 42 1426',
'STATE 1 r 68 42',
'STATE 4 k 42 1427',
'STATE 3 g 29 1428',
'STATE 3 l 29 1429',
'STATE 6 h 82 1430',
'STATE 5 s 82 63',
'STATE 3 p 82 63',
'STATE 4 z 230 1431',
'STATE 3 h 671 1432',
'STATE 1 t 230 1433',
'STATE 3 y 68 1434',
'STATE 2 g 68 1435',
'STATE 2 e 68 1436',
'STATE 6 s 1438 1437',
'STATE 3 g 28 1439',
'STATE 3 c 29 1440',
'STATE 1 m 29 1441',
'STATE 5 # 28 29',
'STATE 2 l 28 1442',
'STATE 5 # 29 28',
'STATE 1 t 29 1443',
'STATE 1 a 28 1444',
'STATE 6 e 1445 29',
'STATE 5 t 29 1446',
'STATE 3 k 29 1447',
'STATE 3 c 230 29',
'STATE 3 c 230 1251',
'STATE 3 k 29 1448',
'STATE 3 b 1450 1449',
'STATE 3 p 28 1451',
'STATE 3 m 28 26',
'STATE 3 h 28 1452',
'STATE 3 b 1453 29',
'STATE 3 b 1455 1454',
'STATE 4 t 1457 1456',
'STATE 5 d 1459 1458',
'STATE 5 d 68 1460',
'STATE 5 d 1461 68',
'STATE 6 # 29 82',
'STATE 1 r 68 1462',
'STATE 5 a 1464 1463',
'STATE 5 d 29 82',
'STATE 5 d 1466 1465',
'STATE 6 i 1468 1467',
'STATE 3 h 100 1469',
'STATE 3 m 68 1470',
'STATE 3 l 1472 1471',
'STATE 3 t 100 1473',
'STATE 6 u 100 1474',
'STATE 2 t 42 72',
'STATE 5 a 1476 1475',
'STATE 5 a 68 29',
'STATE 6 i 1478 1477',
'STATE 6 r 100 1479',
'STATE 6 z 100 1480',
'STATE 5 g 1482 1481',
'STATE 6 i 1484 1483',
'STATE 4 c 68 1485',
'STATE 3 t 562 1486',
'STATE 3 c 82 1487',
'STATE 1 y 68 29',
'STATE 3 d 68 1488',
'STATE 3 r 82 1489',
'STATE 3 r 1491 1490',
'STATE 6 r 42 230',
'STATE 2 c 28 68',
'STATE 6 l 1493 1492',
'STATE 6 m 68 1070',
'STATE 2 o 1495 1494',
'STATE 1 t 1496 42',
'STATE 2 u 1498 1497',
'STATE 6 r 230 1499',
'STATE 5 o 331 1500',
'STATE 2 v 68 1501',
'STATE 5 c 230 29',
'STATE 4 c 68 1502',
'STATE 6 e 42 100',
'STATE 6 p 562 1503',
'STATE 6 a 100 1504',
'STATE 6 e 1506 1505',
'STATE 6 i 1507 230',
'STATE 3 i 68 1508',
'STATE 5 d 100 1509',
'STATE 6 i 1511 1510',
'STATE 3 t 100 1512',
'STATE 5 r 29 1513',
'STATE 3 b 28 1514',
'STATE 3 c 1516 1515',
'STATE 3 l 1518 1517',
'STATE 4 p 230 1519',
'STATE 6 l 1521 1520',
'STATE 3 n 230 1522',
'STATE 3 c 68 1523',
'STATE 3 c 100 1524',
'STATE 3 b 42 1525',
'STATE 3 m 1527 1526',
'STATE 3 v 562 1528',
'STATE 3 b 1530 1529',
'STATE 6 e 100 1531',
'STATE 6 a 100 1532',
'STATE 3 b 230 82',
'STATE 6 i 82 100',
'STATE 3 m 100 1533',
'STATE 3 k 100 1534',
'STATE 4 h 29 1535',
'STATE 4 b 1537 1536',
'STATE 4 l 68 1538',
'STATE 3 t 1540 1539',
'STATE 6 f 100 1541',
'STATE 3 m 100 1542',
'STATE 4 l 100 1543',
'STATE 4 c 100 1544',
'STATE 1 0 68 1545',
'STATE 3 r 100 1546',
'STATE 1 0 68 100',
'STATE 4 m 42 1547',
'STATE 2 a 29 42',
'STATE 3 r 1549 1548',
'STATE 1 o 100 1550',
'STATE 3 j 68 1551',
'STATE 4 c 100 1552',
'STATE 2 c 1554 1553',
'STATE 6 h 42 1555',
'STATE 6 e 28 42',
'STATE 6 r 1557 1556',
'STATE 6 n 68 1558',
'STATE 6 x 68 1559',
'STATE 6 l 100 68',
'STATE 6 t 100 1560',
'STATE 2 a 68 29',
'STATE 2 c 42 1561',
'STATE 4 m 42 1562',
'STATE 4 p 42 1563',
'STATE 3 r 100 42',
'STATE 4 c 100 68',
'STATE 3 p 68 1564',
'STATE 4 d 100 68',
'STATE 1 # 42 1565',
'STATE 3 c 42 1566',
'STATE 3 v 42 1567',
'STATE 1 d 68 1568',
'STATE 5 t 1570 1569',
'STATE 4 x 1572 1571',
'STATE 3 r 100 1573',
'STATE 1 b 68 1574',
'STATE 3 a 29 1575',
'STATE 2 a 1576 68',
'STATE 3 y 68 100',
'STATE 6 z 28 1577',
'STATE 3 y 29 1578',
'STATE 2 s 1580 1579',
'STATE 5 z 230 1581',
'STATE 3 c 28 1582',
'STATE 5 d 29 1583',
'STATE 1 c 29 1584',
'STATE 3 p 26 1585',
'STATE 5 m 1587 1586',
'STATE 5 p 230 1588',
'STATE 3 g 29 1589',
'STATE 3 p 29 1590',
'STATE 3 c 28 26',
'STATE 6 n 100 28',
'STATE 6 o 230 26',
'STATE 3 m 28 1591',
'STATE 6 a 100 29',
'STATE 4 e 1593 1592',
'STATE 5 h 1595 1594',
'STATE 2 a 1597 1596',
'STATE 6 # 68 1598',
'STATE 5 y 68 1599',
'STATE 2 e 68 1600',
'STATE 5 t 82 1601',
'STATE 1 e 1603 1602',
'STATE 2 d 82 1604',
'STATE 5 s 1606 1605',
'STATE 3 m 100 1607',
'STATE 3 b 100 1608',
'STATE 1 r 100 1609',
'STATE 2 r 29 1610',
'STATE 1 f 100 68',
'STATE 1 f 100 1611',
'STATE 5 y 1613 1612',
'STATE 3 h 100 1614',
'STATE 2 l 68 1615',
'STATE 3 i 100 1616',
'STATE 6 o 100 29',
'STATE 5 u 1618 1617',
'STATE 6 l 230 1619',
'STATE 6 e 1621 1620',
'STATE 4 d 230 100',
'STATE 5 m 100 1622',
'STATE 6 l 68 1623',
'STATE 6 r 100 1624',
'STATE 6 e 100 1625',
'STATE 5 k 100 1626',
'STATE 5 b 230 1627',
'STATE 4 d 100 1628',
'STATE 6 i 29 68',
'STATE 1 p 68 1629',
'STATE 2 o 68 1630',
'STATE 3 w 68 82',
'STATE 4 n 1631 68',
'STATE 2 t 68 1632',
'STATE 4 v 100 1633',
'STATE 4 p 68 100',
'STATE 3 r 1634 1120',
'STATE 3 r 68 1289',
'STATE 3 g 42 68',
'STATE 5 a 1636 1635',
'STATE 3 n 68 100',
'STATE 4 d 29 1637',
'STATE 6 o 100 1638',
'STATE 4 s 100 1101',
'STATE 5 n 68 29',
'STATE 6 f 562 1639',
'STATE 6 s 29 1640',
'STATE 5 y 1642 1641',
'STATE 5 l 1644 1643',
'STATE 5 l 230 72',
'STATE 5 t 1646 1645',
'STATE 6 l 100 1647',
'STATE 6 a 1511 100',
'STATE 3 m 230 100',
'STATE 3 l 42 1648',
'STATE 6 i 29 1649',
'STATE 6 e 28 1650',
'STATE 3 i 29 1651',
'STATE 4 n 68 1493',
'STATE 4 n 100 1652',
'STATE 4 m 230 1653',
'STATE 3 c 68 529',
'STATE 6 r 1655 1654',
'STATE 3 m 100 1656',
'STATE 6 r 68 230',
'STATE 3 f 68 1657',
'STATE 3 n 230 1511',
'STATE 3 l 230 1658',
'STATE 5 t 1660 1659',
'STATE 6 i 100 1661',
'STATE 6 t 68 100',
'STATE 5 l 1663 1662',
'STATE 5 l 100 42',
'STATE 3 g 100 1185',
'STATE 6 i 100 230',
'STATE 6 n 68 100',
'STATE 6 d 100 1664',
'STATE 5 l 29 1665',
'STATE 2 q 29 1666',
'STATE 2 s 42 68',
'STATE 3 s 100 1667',
'STATE 6 k 29 1668',
'STATE 4 b 42 1669',
'STATE 3 w 230 1670',
'STATE 2 # 230 100',
'STATE 3 n 29 1671',
'STATE 2 a 68 1672',
'STATE 2 c 68 100',
'STATE 1 d 100 1673',
'STATE 3 m 100 1674',
'STATE 2 o 1676 1675',
'STATE 2 g 42 1677',
'STATE 1 e 100 68',
'STATE 4 n 42 1678',
'STATE 4 l 100 230',
'STATE 2 g 68 1679',
'STATE 4 v 42 100',
'STATE 2 c 42 1680',
'STATE 2 s 42 1681',
'STATE 4 t 273 1682',
'STATE 6 r 68 1683',
'STATE 4 v 1685 1684',
'STATE 3 n 100 1686',
'STATE 4 d 68 1687',
'STATE 2 t 1688 42',
'STATE 3 h 1690 1689',
'STATE 3 c 68 1691',
'STATE 1 e 42 1692',
'STATE 2 o 42 1693',
'STATE 2 e 42 1694',
'STATE 3 d 247 1695',
'STATE 3 t 63 1696',
'STATE 6 z 82 63',
'STATE 4 v 230 1697',
'STATE 3 n 562 1698',
'STATE 2 t 68 1699',
'STATE 2 r 68 1700',
'STATE 3 u 230 1701',
'STATE 1 r 562 68',
'STATE 2 a 1703 1702',
'STATE 3 z 28 1704',
'STATE 3 a 29 1705',
'STATE 3 s 28 29',
'STATE 3 a 29 1706',
'STATE 2 a 28 230',
'STATE 1 m 28 29',
'STATE 3 t 29 1707',
'STATE 2 s 26 1708',
'STATE 3 f 29 1709',
'STATE 3 p 29 1710',
'STATE 3 s 29 1711',
'STATE 3 t 29 1712',
'STATE 5 l 29 1713',
'STATE 6 n 28 1714',
'STATE 2 a 1716 1715',
'STATE 6 # 42 28',
'STATE 5 k 100 1717',
'STATE 4 c 1719 1718',
'STATE 4 c 1721 1720',
'STATE 1 h 68 100',
'STATE 2 a 29 1722',
'STATE 5 t 68 1723',
'STATE 3 n 29 1724',
'STATE 5 k 82 100',
'STATE 2 e 154 68',
'STATE 2 n 82 154',
'STATE 2 a 1726 1725',
'STATE 6 l 1728 1727',
'STATE 6 k 100 1729',
'STATE 6 r 230 1730',
'STATE 5 n 68 1731',
'STATE 2 s 100 1732',
'STATE 3 i 29 1733',
'STATE 2 d 1735 1734',
'STATE 3 i 68 1736',
'STATE 2 l 42 68',
'STATE 3 r 1738 1737',
'STATE 1 g 68 1739',
'STATE 1 i 29 100',
'STATE 5 h 1741 1740',
'STATE 4 b 68 1742',
'STATE 4 g 100 1743',
'STATE 6 a 68 1744',
'STATE 4 g 68 29',
'STATE 5 b 1745 100',
'STATE 6 s 68 1746',
'STATE 5 a 100 1747',
'STATE 6 o 562 100',
'STATE 6 e 1749 1748',
'STATE 5 v 230 100',
'STATE 4 t 100 1750',
'STATE 6 o 1752 1751',
'STATE 2 r 68 1753',
'STATE 6 v 100 1754',
'STATE 6 n 68 29',
'STATE 4 d 68 1755',
'STATE 2 e 42 68',
'STATE 5 o 1757 1756',
'STATE 6 # 608 1758',
'STATE 5 t 29 1759',
'STATE 4 p 100 1760',
'STATE 6 g 562 1761',
'STATE 6 i 100 68',
'STATE 5 u 1763 1762',
'STATE 6 # 42 1764',
'STATE 3 c 1766 1765',
'STATE 4 c 68 1767',
'STATE 5 u 1769 1768',
'STATE 3 s 1770 100',
'STATE 6 i 100 1771',
'STATE 3 d 42 625',
'STATE 5 u 100 1772',
'STATE 3 k 28 1773',
'STATE 3 z 68 1774',
'STATE 4 g 230 1775',
'STATE 4 f 100 230',
'STATE 3 c 1777 1776',
'STATE 4 d 42 1778',
'STATE 4 m 68 1779',
'STATE 3 p 100 1780',
'STATE 4 t 42 1781',
'STATE 5 k 1783 1782',
'STATE 3 s 82 1784',
'STATE 6 r 562 1785',
'STATE 3 f 82 1786',
'STATE 3 c 82 1787',
'STATE 6 m 100 1788',
'STATE 5 d 29 1789',
'STATE 6 m 1791 1790',
'STATE 4 k 29 1792',
'STATE 6 l 1794 1793',
'STATE 4 t 68 100',
'STATE 3 d 1796 1795',
'STATE 4 t 100 1797',
'STATE 3 h 100 1798',
'STATE 1 0 100 1799',
'STATE 4 p 42 100',
'STATE 4 d 512 1800',
'STATE 3 m 230 68',
'STATE 1 p 42 1801',
'STATE 3 r 1803 1802',
'STATE 2 h 68 1804',
'STATE 5 g 1806 1805',
'STATE 6 k 1808 1807',
'STATE 4 p 42 1809',
'STATE 2 a 68 1810',
'STATE 6 t 68 1811',
'STATE 6 n 230 100',
'STATE 3 c 68 1812',
'STATE 2 d 68 1813',
'STATE 4 v 100 42',
'STATE 6 e 100 1814',
'STATE 4 b 100 1815',
'STATE 3 g 68 1816',
'STATE 4 m 42 1817',
'STATE 1 o 42 1818',
'STATE 3 c 42 29',
'STATE 2 a 247 1819',
'STATE 3 h 773 1820',
'STATE 3 a 29 1821',
'STATE 3 r 562 100',
'STATE 3 d 68 1822',
'STATE 2 s 68 1823',
'STATE 2 t 1825 1824',
'STATE 6 l 28 1826',
'STATE 6 a 29 1827',
'STATE 1 r 29 28',
'STATE 3 n 1829 1828',
'STATE 5 d 1831 1830',
'STATE 2 l 29 1832',
'STATE 2 e 28 1833',
'STATE 3 l 29 1834',
'STATE 2 # 1314 230',
'STATE 3 d 29 1835',
'STATE 3 p 29 1836',
'STATE 3 v 230 1837',
'STATE 3 g 28 1838',
'STATE 5 y 1840 1839',
'STATE 4 o 28 1841',
'STATE 4 b 29 1842',
'STATE 6 # 68 29',
'STATE 1 e 1844 1843',
'STATE 6 e 100 1845',
'STATE 2 m 1846 100',
'STATE 6 a 29 100',
'STATE 5 v 29 1847',
'STATE 6 i 29 1059',
'STATE 2 i 1849 1848',
'STATE 6 n 1851 1850',
'STATE 2 c 1853 1852',
'STATE 5 t 1854 100',
'STATE 2 x 100 1855',
'STATE 6 n 100 1856',
'STATE 1 a 100 29',
'STATE 1 e 100 1857',
'STATE 1 n 100 1858',
'STATE 1 n 68 1859',
'STATE 3 v 100 68',
'STATE 5 t 1861 1860',
'STATE 3 i 68 1862',
'STATE 1 e 562 1863',
'STATE 1 c 68 1864',
'STATE 4 s 1866 1865',
'STATE 4 s 100 1867',
'STATE 6 l 68 1868',
'STATE 6 i 68 1869',
'STATE 6 u 230 68',
'STATE 6 i 562 1205',
'STATE 6 r 100 1870',
'STATE 5 c 1871 100',
'STATE 6 a 1873 1872',
'STATE 5 v 230 1874',
'STATE 4 p 68 1875',
'STATE 6 u 68 1876',
'STATE 2 e 29 1877',
'STATE 3 h 1879 1878',
'STATE 1 r 230 68',
'STATE 4 b 68 1880',
'STATE 3 a 29 1881',
'STATE 6 # 29 1882',
'STATE 4 h 28 1883',
'STATE 2 q 100 1884',
'STATE 4 c 100 1885',
'STATE 6 i 562 100',
'STATE 6 a 1887 1886',
'STATE 6 # 29 1888',
'STATE 4 t 29 1889',
'STATE 5 u 1891 1890',
'STATE 5 t 230 1892',
'STATE 4 b 42 1893',
'STATE 5 y 100 1894',
'STATE 6 s 42 1895',
'STATE 6 o 230 1185',
'STATE 5 n 562 1896',
'STATE 3 m 100 1897',
'STATE 6 # 28 42',
'STATE 6 b 100 1898',
'STATE 4 t 68 1899',
'STATE 4 j 68 1900',
'STATE 4 p 68 1901',
'STATE 3 c 230 1902',
'STATE 3 c 1904 1903',
'STATE 3 h 100 1905',
'STATE 3 c 42 1906',
'STATE 5 l 1908 1907',
'STATE 6 o 100 1909',
'STATE 6 o 29 1910',
'STATE 6 u 562 100',
'STATE 3 s 42 100',
'STATE 3 v 100 1911',
'STATE 3 m 68 1685',
'STATE 5 a 29 1912',
'STATE 6 l 1914 1913',
'STATE 2 a 68 28',
'STATE 2 r 68 1915',
'STATE 2 d 68 1916',
'STATE 3 r 1917 68',
'STATE 3 t 100 1918',
'STATE 6 d 42 68',
'STATE 3 l 100 1554',
'STATE 2 o 230 1919',
'STATE 4 v 100 29',
'STATE 4 l 68 1920',
'STATE 4 f 230 1921',
'STATE 3 k 1923 1922',
'STATE 4 b 68 42',
'STATE 3 r 42 1924',
'STATE 6 e 42 1925',
'STATE 3 d 42 28',
'STATE 6 l 1927 1926',
'STATE 4 c 68 1928',
'STATE 2 a 68 1929',
'STATE 3 h 68 100',
'STATE 4 l 68 1930',
'STATE 4 b 68 1931',
'STATE 2 b 100 29',
'STATE 4 c 1047 1932',
'STATE 6 r 100 1281',
'STATE 3 r 100 1933',
'STATE 4 s 68 42',
'STATE 2 p 42 1934',
'STATE 1 r 247 1935',
'STATE 3 b 63 1936',
'STATE 3 c 100 1937',
'STATE 2 l 68 1938',
'STATE 3 l 68 1939',
'STATE 3 p 100 1940',
'STATE 3 r 100 68',
'STATE 6 t 1942 1941',
'STATE 6 i 28 29',
'STATE 2 l 1944 1943',
'STATE 1 o 29 28',
'STATE 5 s 28 29',
'STATE 3 i 28 1945',
'STATE 6 e 29 1946',
'STATE 3 h 28 1947',
'STATE 3 t 29 1948',
'STATE 3 j 29 1949',
'STATE 2 # 1950 29',
'STATE 3 c 29 1951',
'STATE 3 l 100 230',
'STATE 3 w 1953 1952',
'STATE 4 c 68 1954',
'STATE 6 i 1956 1955',
'STATE 4 g 100 1957',
'STATE 2 r 230 1958',
'STATE 2 n 230 1959',
'STATE 6 a 1961 1960',
'STATE 6 # 562 100',
'STATE 1 d 100 1962',
'STATE 6 g 100 1963',
'STATE 3 y 230 68',
'STATE 6 k 230 1964',
'STATE 3 t 230 68',
'STATE 1 r 100 1965',
'STATE 1 r 68 100',
'STATE 1 i 100 68',
'STATE 3 m 68 1966',
'STATE 6 h 100 68',
'STATE 1 o 100 1967',
'STATE 3 r 29 1968',
'STATE 1 r 68 1969',
'STATE 5 s 68 100',
'STATE 2 p 100 1970',
'STATE 2 e 100 1971',
'STATE 1 s 68 1972',
'STATE 2 o 68 1973',
'STATE 6 r 1975 1974',
'STATE 5 s 68 1976',
'STATE 6 i 68 1977',
'STATE 6 n 68 1978',
'STATE 4 b 68 1979',
'STATE 6 d 230 100',
'STATE 6 h 100 1980',
'STATE 5 g 562 100',
'STATE 5 v 100 1981',
'STATE 5 g 100 1982',
'STATE 4 s 68 1983',
'STATE 6 a 1985 1984',
'STATE 1 0 100 1986',
'STATE 3 g 68 1987',
'STATE 2 s 68 1988',
'STATE 4 m 68 1989',
'STATE 5 y 1991 1990',
'STATE 2 o 68 1992',
'STATE 3 p 230 1993',
'STATE 2 g 29 1994',
'STATE 4 s 1996 1995',
'STATE 5 r 1998 1997',
'STATE 4 j 2000 1999',
'STATE 3 m 2002 2001',
'STATE 3 b 42 2003',
'STATE 5 h 2005 2004',
'STATE 4 g 42 100',
'STATE 4 p 230 100',
'STATE 3 l 42 1674',
'STATE 3 h 100 2006',
'STATE 6 e 230 100',
'STATE 5 v 562 2007',
'STATE 6 # 100 29',
'STATE 4 n 2009 2008',
'STATE 4 m 68 2010',
'STATE 6 t 2012 2011',
'STATE 4 s 1870 68',
'STATE 3 h 100 2013',
'STATE 4 b 230 2014',
'STATE 4 v 100 230',
'STATE 3 k 100 2015',
'STATE 3 h 42 2016',
'STATE 5 s 2018 2017',
'STATE 6 a 100 2019',
'STATE 3 t 82 2020',
'STATE 6 h 100 2021',
'STATE 3 g 100 2022',
'STATE 4 f 29 2023',
'STATE 2 a 68 2024',
'STATE 3 f 100 68',
'STATE 4 t 68 2025',
'STATE 6 r 68 2026',
'STATE 4 z 68 100',
'STATE 4 s 2028 2027',
'STATE 3 d 100 2029',
'STATE 1 o 68 2030',
'STATE 2 a 230 68',
'STATE 4 s 42 2031',
'STATE 4 l 68 100',
'STATE 2 s 42 2032',
'STATE 2 # 2034 2033',
'STATE 4 d 68 2035',
'STATE 4 n 2037 2036',
'STATE 4 n 68 29',
'STATE 3 s 42 2038',
'STATE 4 c 68 2039',
'STATE 2 c 68 2040',
'STATE 4 x 100 2041',
'STATE 3 s 68 2042',
'STATE 3 s 42 2043',
'STATE 1 a 2045 2044',
'STATE 5 b 63 2046',
'STATE 3 t 2048 2047',
'STATE 3 n 68 2049',
'STATE 1 p 68 2050',
'STATE 3 b 68 2051',
'STATE 2 c 29 2052',
'STATE 3 h 2053 29',
'STATE 2 h 28 2054',
'STATE 1 a 230 28',
'STATE 3 k 28 2055',
'STATE 3 b 29 2056',
'STATE 6 n 28 2057',
'STATE 3 k 29 2058',
'STATE 3 t 29 2059',
'STATE 3 c 29 2060',
'STATE 5 b 29 2061',
'STATE 5 u 2063 2062',
'STATE 2 o 68 2064',
'STATE 6 # 2065 68',
'STATE 4 s 2067 2066',
'STATE 1 p 42 2068',
'STATE 2 m 100 2069',
'STATE 1 i 29 2070',
'STATE 2 r 29 230',
'STATE 3 c 100 2071',
'STATE 1 d 68 100',
'STATE 6 a 2073 2072',
'STATE 4 f 230 2074',
'STATE 3 k 230 2075',
'STATE 5 d 2077 2076',
'STATE 2 t 100 68',
'STATE 3 l 68 100',
'STATE 3 l 2078 100',
'STATE 3 d 68 2079',
'STATE 2 u 68 2080',
'STATE 1 o 68 1986',
'STATE 2 b 100 68',
'STATE 1 s 100 1854',
'STATE 4 h 29 2081',
'STATE 4 g 68 2082',
'STATE 5 y 42 2083',
'STATE 4 c 100 2084',
'STATE 4 c 68 2085',
'STATE 6 t 100 2086',
'STATE 6 e 100 230',
'STATE 5 b 100 2087',
'STATE 5 b 100 562',
'STATE 4 f 68 1285',
'STATE 6 e 2089 2088',
'STATE 1 a 100 1403',
'STATE 2 i 68 100',
'STATE 3 w 82 2090',
'STATE 2 c 68 82',
'STATE 4 n 68 2091',
'STATE 6 e 2093 2092',
'STATE 6 # 42 100',
'STATE 4 v 2095 2094',
'STATE 3 m 68 2096',
'STATE 6 # 29 230',
'STATE 3 r 68 2097',
'STATE 3 n 230 100',
'STATE 3 m 2099 2098',
'STATE 4 t 2101 2100',
'STATE 3 m 2103 2102',
'PHONE ay1',
'STATE 4 k 68 2104',
'STATE 4 t 68 1917',
'STATE 4 d 42 100',
'STATE 5 r 1786 2105',
'STATE 3 f 29 2106',
'STATE 6 o 2108 2107',
'STATE 5 c 100 2109',
'STATE 3 n 2028 2110',
'STATE 3 j 68 100',
'STATE 4 p 230 2111',
'STATE 4 c 2113 2112',
'STATE 4 b 29 1838',
'STATE 4 h 29 2114',
'STATE 4 s 68 230',
'STATE 3 b 68 100',
'STATE 3 m 68 2115',
'STATE 5 z 2117 2116',
'STATE 3 f 82 2020',
'STATE 6 u 2119 2118',
'STATE 3 b 82 100',
'STATE 3 b 100 82',
'STATE 3 t 82 100',
'STATE 5 h 29 2120',
'STATE 4 v 230 2121',
'STATE 3 t 1285 2122',
'STATE 2 s 230 100',
'STATE 3 f 68 2123',
'STATE 6 l 230 68',
'STATE 2 # 2125 2124',
'STATE 2 a 68 2126',
'STATE 3 s 2128 2127',
'STATE 2 b 42 2129',
'STATE 1 # 42 2130',
'STATE 6 l 42 2131',
'STATE 6 m 42 2132',
'STATE 2 n 68 2133',
'STATE 2 m 100 230',
'STATE 4 v 230 2134',
'STATE 6 r 29 68',
'STATE 2 a 2136 2135',
'STATE 3 c 2138 2137',
'STATE 3 n 42 2139',
'STATE 3 r 42 2140',
'STATE 2 i 247 68',
'STATE 2 n 247 2141',
'STATE 6 c 63 2142',
'STATE 2 a 2144 2143',
'STATE 4 d 68 2145',
'STATE 2 i 68 2146',
'STATE 1 h 68 2147',
'STATE 1 r 68 2148',
'STATE 3 n 2150 2149',
'STATE 1 e 230 2151',
'STATE 3 y 29 2152',
'STATE 1 r 2154 2153',
'STATE 3 h 2156 2155',
'STATE 3 t 28 29',
'STATE 3 m 2157 29',
'STATE 3 f 29 2158',
'STATE 5 g 230 29',
'STATE 2 # 29 2159',
'STATE 1 s 2161 2160',
'STATE 6 l 2163 2162',
'STATE 5 h 29 2164',
'STATE 4 d 42 68',
'STATE 6 e 68 2165',
'STATE 6 # 2167 2166',
'STATE 4 s 29 2168',
'STATE 4 s 68 100',
'STATE 2 h 29 2169',
'STATE 2 c 100 2170',
'STATE 5 u 1735 2171',
'STATE 5 c 230 100',
'STATE 1 s 29 2172',
'STATE 6 s 29 68',
'STATE 2 i 100 2173',
'STATE 3 l 2175 2174',
'STATE 6 h 68 29',
'STATE 1 m 2176 68',
'STATE 3 n 68 2177',
'STATE 4 p 2179 2178',
'STATE 4 b 68 2180',
'STATE 6 h 100 2181',
'STATE 6 a 2183 2182',
'STATE 6 a 29 2184',
'STATE 6 p 68 2185',
'STATE 5 c 100 562',
'STATE 6 i 100 2186',
'STATE 1 0 100 2187',
'STATE 3 n 68 2188',
'STATE 4 l 68 1493',
'STATE 5 r 2190 2189',
'STATE 2 i 68 2191',
'STATE 6 s 42 2192',
'STATE 2 f 42 68',
'STATE 2 g 2194 2193',
'STATE 4 n 2195 68',
'STATE 6 u 2197 2196',
'STATE 4 c 2199 2198',
'STATE 4 v 100 2200',
'STATE 3 m 42 2201',
'STATE 3 c 2203 2202',
'STATE 4 c 2205 2204',
'STATE 3 y 29 2206',
'STATE 4 s 2208 2207',
'STATE 4 c 2210 2209',
'STATE 3 m 2212 2211',
'STATE 5 k 100 2213',
'STATE 5 m 562 2214',
'STATE 6 t 100 2215',
'STATE 3 m 42 2216',
'STATE 3 y 29 2217',
'STATE 3 m 68 230',
'STATE 4 z 2219 2218',
'STATE 4 g 42 230',
'STATE 5 h 562 100',
'STATE 3 s 82 230',
'STATE 3 b 2221 2220',
'STATE 3 h 68 2222',
'STATE 4 m 29 2223',
'STATE 2 o 68 2224',
'STATE 1 e 68 2225',
'STATE 3 m 68 2226',
'STATE 3 c 100 2227',
'STATE 4 n 100 2228',
'STATE 4 n 100 2229',
'STATE 4 b 100 2230',
'STATE 4 v 230 68',
'STATE 4 m 68 193',
'STATE 6 i 28 42',
'STATE 3 r 42 2231',
'STATE 6 g 100 2232',
'STATE 2 a 68 2233',
'STATE 1 r 29 2234',
'STATE 4 n 100 2235',
'STATE 3 m 29 68',
'STATE 3 t 2237 2236',
'STATE 6 g 100 2238',
'STATE 3 f 42 100',
'STATE 4 v 42 2239',
'STATE 2 c 247 2240',
'STATE 3 k 82 2241',
'STATE 1 a 2243 2242',
'STATE 3 b 230 1423',
'STATE 4 g 100 2244',
'STATE 1 i 68 2245',
'STATE 1 k 68 2246',
'STATE 2 u 68 2247',
'STATE 1 a 2249 2248',
'STATE 5 d 909 230',
'STATE 2 c 230 2250',
'STATE 3 f 230 2251',
'STATE 1 b 29 2252',
'STATE 2 e 29 28',
'STATE 2 c 29 230',
'STATE 1 o 230 2253',
'STATE 5 s 230 29',
'STATE 5 s 29 2254',
'STATE 5 n 230 29',
'STATE 6 # 2256 2255',
'STATE 3 i 247 2257',
'STATE 4 t 2259 2258',
'STATE 4 c 100 2260',
'STATE 4 t 29 154',
'STATE 4 g 2262 2261',
'STATE 6 e 100 2263',
'STATE 5 h 247 100',
'STATE 3 r 29 100',
'STATE 6 # 905 29',
'STATE 6 i 100 2264',
'STATE 2 n 100 2265',
'STATE 4 d 2267 2266',
'STATE 2 o 2269 2268',
'STATE 2 g 562 2270',
'STATE 2 e 68 2271',
'STATE 2 b 68 100',
'STATE 2 i 68 782',
'STATE 5 t 2273 2272',
'STATE 5 p 2274 100',
'STATE 5 c 68 2275',
'STATE 6 o 68 2276',
'STATE 4 d 68 2277',
'STATE 4 t 100 68',
'STATE 4 q 100 1532',
'STATE 6 d 100 2278',
'STATE 6 g 100 82',
'STATE 3 v 68 2279',
'STATE 1 a 68 2280',
'STATE 3 h 2282 2281',
'STATE 4 g 42 2283',
'STATE 5 l 2285 2284',
'STATE 4 t 2287 2286',
'STATE 4 n 2289 2288',
'STATE 4 n 230 100',
'STATE 5 d 68 29',
'STATE 6 o 2291 2290',
'STATE 5 q 2293 2292',
'STATE 5 h 100 2294',
'STATE 5 h 2296 2295',
'STATE 3 l 230 2297',
'STATE 3 p 1204 100',
'STATE 5 k 2299 2298',
'STATE 4 m 230 2300',
'STATE 4 s 100 2301',
'STATE 5 c 100 2302',
'STATE 3 k 29 2303',
'STATE 3 m 2304 100',
'STATE 5 s 100 2305',
'STATE 4 t 2306 100',
'STATE 3 b 29 100',
'STATE 3 l 100 2307',
'STATE 5 c 2308 100',
'STATE 3 p 230 100',
'STATE 6 a 100 2309',
'STATE 6 d 230 2310',
'STATE 4 s 100 68',
'STATE 6 g 100 2311',
'STATE 3 s 2313 2312',
'STATE 3 n 230 68',
'STATE 6 s 82 2314',
'STATE 6 o 68 2315',
'STATE 3 g 230 100',
'STATE 6 o 82 2316',
'STATE 6 r 100 2317',
'STATE 2 u 68 2318',
'STATE 4 m 2320 2319',
'STATE 4 g 100 2321',
'STATE 3 c 100 2322',
'STATE 4 c 68 596',
'STATE 4 l 100 2323',
'STATE 5 t 42 2324',
'STATE 2 c 100 2325',
'STATE 2 o 68 2326',
'STATE 2 t 68 2327',
'STATE 3 l 2287 29',
'STATE 3 l 2329 2328',
'STATE 6 r 2330 100',
'STATE 4 m 100 2331',
'STATE 3 t 42 2332',
'STATE 6 # 247 68',
'STATE 3 l 2334 2333',
'STATE 4 f 68 2335',
'STATE 4 d 29 2336',
'STATE 4 t 100 562',
'STATE 2 s 562 2337',
'STATE 1 r 68 2338',
'STATE 3 d 68 2339',
'STATE 6 h 29 2340',
'STATE 2 n 28 29',
'STATE 1 r 230 29',
'STATE 3 b 29 2341',
'STATE 1 h 28 2342',
'STATE 2 n 29 2343',
'STATE 3 m 29 2344',
'STATE 4 p 2346 2345',
'STATE 4 h 68 2347',
'STATE 2 q 29 2348',
'STATE 6 a 29 2349',
'STATE 2 n 100 2350',
'STATE 1 n 100 230',
'STATE 4 z 29 2351',
'STATE 1 p 68 2352',
'STATE 6 o 29 1722',
'STATE 2 s 2354 2353',
'STATE 3 v 68 2355',
'STATE 1 m 2357 2356',
'STATE 3 r 68 230',
'STATE 6 r 68 2358',
'STATE 1 i 29 2359',
'STATE 6 t 68 2360',
'STATE 6 i 100 2361',
'STATE 6 e 2363 2362',
'STATE 4 t 2364 100',
'STATE 6 o 68 2365',
'STATE 4 p 68 2366',
'STATE 5 c 2365 2367',
'STATE 6 e 100 2368',
'STATE 4 p 68 2369',
'STATE 2 c 100 82',
'STATE 3 t 68 82',
'STATE 3 y 2371 2370',
'STATE 5 u 29 2372',
'STATE 4 h 29 100',
'STATE 3 h 2374 2373',
'STATE 4 b 42 2375',
'STATE 3 r 2377 2376',
'STATE 2 p 68 100',
'STATE 4 c 42 2378',
'STATE 2 b 100 2379',
'STATE 6 i 2381 2380',
'STATE 5 k 2299 2382',
'STATE 5 s 100 2383',
'STATE 3 p 230 2384',
'STATE 4 z 230 2385',
'STATE 5 k 100 68',
'STATE 6 i 247 68',
'STATE 3 g 230 2386',
'STATE 5 h 2388 2387',
'STATE 4 c 100 29',
'STATE 5 c 562 2389',
'STATE 4 m 29 2390',
'STATE 5 h 100 68',
'STATE 3 n 100 2391',
'STATE 4 c 68 2392',
'STATE 3 h 42 2393',
'STATE 3 m 100 2394',
'STATE 3 c 2396 2395',
'STATE 6 i 230 100',
'STATE 6 e 562 2397',
'STATE 4 v 230 2398',
'STATE 6 d 100 2399',
'STATE 3 m 68 2400',
'STATE 4 m 100 230',
'STATE 6 m 82 2401',
'STATE 6 i 100 82',
'STATE 6 e 82 29',
'STATE 1 m 68 100',
'STATE 4 m 29 2402',
'STATE 4 l 2404 2403',
'STATE 3 c 68 2405',
'STATE 2 g 100 2406',
'STATE 4 v 100 273',
'STATE 3 l 2408 2407',
'STATE 6 i 42 2409',
'STATE 6 a 2411 2410',
'STATE 4 b 230 68',
'STATE 2 e 68 2412',
'STATE 4 f 42 2413',
'STATE 6 r 2330 2414',
'STATE 4 v 230 42',
'STATE 6 b 42 2415',
'STATE 1 h 42 2416',
'STATE 5 s 82 2417',
'STATE 5 d 82 63',
'STATE 4 b 100 2418',
'STATE 4 t 562 2419',
'STATE 1 a 562 68',
'STATE 2 o 562 68',
'STATE 1 l 68 2420',
'STATE 3 i 29 2421',
'STATE 3 m 2423 2422',
'STATE 2 r 2425 2424',
'STATE 1 i 230 2426',
'STATE 3 b 2427 29',
'STATE 5 r 2429 2428',
'STATE 6 i 100 2430',
'STATE 5 z 1669 2431',
'STATE 3 r 100 2432',
'STATE 6 # 29 558',
'STATE 3 r 68 2433',
'STATE 1 p 68 2434',
'STATE 6 a 29 230',
'STATE 2 d 100 2435',
'STATE 3 t 562 100',
'STATE 6 u 100 2436',
'STATE 4 z 100 68',
'STATE 2 c 100 896',
'STATE 3 l 2438 2437',
'STATE 6 o 29 2439',
'STATE 1 a 100 2440',
'STATE 6 e 1854 29',
'STATE 4 c 2442 2441',
'STATE 5 c 562 2443',
'STATE 6 i 100 2444',
'STATE 6 e 68 100',
'STATE 4 t 68 2445',
'STATE 6 a 100 2446',
'STATE 6 l 562 100',
'STATE 4 h 68 2447',
'STATE 2 o 1825 2448',
'STATE 5 t 68 100',
'STATE 2 b 29 2449',
'STATE 5 c 2451 2450',
'STATE 5 g 42 2452',
'STATE 3 h 100 2453',
'STATE 3 l 100 2454',
'STATE 6 n 100 2455',
'STATE 6 n 42 2456',
'STATE 2 s 100 273',
'STATE 6 r 2458 2457',
'STATE 5 t 2460 2459',
'STATE 5 h 100 2461',
'STATE 3 t 230 2462',
'STATE 4 c 100 562',
'STATE 6 r 562 2463',
'STATE 4 p 100 2464',
'STATE 5 r 2466 2465',
'STATE 4 s 100 2467',
'STATE 5 h 100 2468',
'STATE 4 t 100 2469',
'STATE 6 n 68 2470',
'STATE 4 g 100 2471',
'STATE 3 t 42 100',
'STATE 3 g 100 756',
'STATE 6 a 100 2472',
'STATE 5 c 1980 100',
'STATE 6 o 2473 100',
'STATE 6 v 230 2474',
'STATE 4 p 68 2475',
'STATE 3 b 68 2476',
'STATE 3 h 2477 100',
'STATE 4 n 2478 1425',
'STATE 6 g 100 2479',
'STATE 6 g 68 2480',
'STATE 2 # 68 2481',
'STATE 4 n 100 256',
'STATE 3 f 42 2482',
'STATE 4 m 100 68',
'STATE 6 o 42 28',
'STATE 6 y 2484 2483',
'STATE 3 r 42 68',
'STATE 2 s 68 2485',
'STATE 6 n 2487 2486',
'STATE 4 t 42 68',
'STATE 4 v 42 2488',
'STATE 1 c 42 2489',
'STATE 3 m 82 2490',
'STATE 2 i 562 2491',
'STATE 3 i 100 2492',
'STATE 2 e 68 2493',
'STATE 1 r 29 2494',
'STATE 1 l 28 2495',
'STATE 1 a 29 2496',
'STATE 1 o 28 2497',
'STATE 3 g 28 29',
'STATE 2 r 230 29',
'STATE 5 b 230 29',
'STATE 5 h 2499 2498',
'STATE 3 p 42 2500',
'STATE 5 s 100 2501',
'STATE 4 s 2503 2502',
'STATE 3 h 100 2504',
'STATE 3 l 42 68',
'STATE 4 k 562 2505',
'STATE 1 e 100 2506',
'STATE 6 i 100 2507',
'STATE 5 n 1207 2508',
'STATE 6 e 2509 100',
'STATE 5 n 100 29',
'STATE 3 s 29 2510',
'STATE 4 z 230 2511',
'STATE 5 k 2513 2512',
'STATE 5 v 100 2514',
'STATE 6 e 68 2515',
'STATE 4 d 68 100',
'STATE 6 u 68 2516',
'STATE 4 d 100 2517',
'STATE 4 h 2519 2518',
'STATE 4 t 100 2520',
'STATE 2 o 68 2521',
'STATE 2 f 230 100',
'STATE 5 b 42 2069',
'STATE 2 b 100 2522',
'STATE 2 s 100 2523',
'STATE 2 g 42 29',
'STATE 6 s 29 2524',
'STATE 5 z 2526 2525',
'STATE 5 t 100 2527',
'STATE 5 h 100 2528',
'STATE 4 s 100 2529',
'STATE 4 b 68 2530',
'STATE 3 h 100 2213',
'STATE 6 u 2532 2531',
'STATE 3 f 100 2533',
'STATE 4 b 100 2534',
'STATE 4 g 42 2535',
'STATE 4 t 100 2536',
'STATE 5 t 2538 2537',
'STATE 5 n 100 2539',
'STATE 4 b 1416 2540',
'STATE 5 z 230 100',
'STATE 3 b 100 2541',
'STATE 5 h 100 562',
'STATE 4 g 230 2542',
'STATE 3 j 68 2543',
'STATE 4 v 2545 2544',
'STATE 6 o 100 2315',
'STATE 2 a 68 2546',
'STATE 4 p 100 2547',
'STATE 6 l 68 2548',
'STATE 6 r 100 68',
'STATE 4 v 100 2549',
'STATE 4 n 68 2550',
'STATE 3 l 68 42',
'STATE 3 h 42 2551',
'STATE 4 b 42 2552',
'STATE 3 g 42 2553',
'STATE 4 s 42 2554',
'STATE 2 l 68 2555',
'STATE 5 p 82 63',
'STATE 3 u 562 2556',
'STATE 4 c 562 100',
'STATE 3 k 68 2557',
'STATE 1 b 230 2558',
'STATE 1 u 29 2559',
'STATE 2 a 29 2560',
'STATE 3 u 29 2561',
'STATE 6 e 2563 2562',
'STATE 3 t 100 2564',
'STATE 6 e 2566 2565',
'STATE 6 s 100 2567',
'STATE 5 p 2569 2568',
'STATE 5 s 2570 100',
'STATE 4 s 100 29',
'STATE 4 d 29 2571',
'STATE 2 t 100 2572',
'STATE 3 c 29 2573',
'STATE 3 s 29 2574',
'STATE 2 p 100 29',
'STATE 1 d 100 2575',
'STATE 5 j 68 2576',
'STATE 6 i 100 2577',
'STATE 6 n 562 100',
'STATE 4 f 68 2578',
'STATE 6 a 68 100',
'STATE 6 i 2579 100',
'STATE 6 n 230 2580',
'STATE 2 i 2582 2581',
'STATE 5 l 29 100',
'STATE 5 p 1625 2583',
'STATE 5 g 2585 2584',
'STATE 2 s 1674 2586',
'STATE 4 p 100 2587',
'STATE 4 v 68 100',
'STATE 5 h 2588 100',
'STATE 4 c 68 100',
'STATE 5 b 100 2589',
'STATE 5 s 2591 2590',
'STATE 4 t 2592 100',
'STATE 4 s 100 2593',
'STATE 6 o 2073 100',
'STATE 5 s 230 100',
'STATE 4 c 100 2594',
'STATE 3 h 100 2595',
'STATE 3 l 230 100',
'STATE 4 p 562 100',
'STATE 5 s 100 2596',
'STATE 4 s 230 100',
'STATE 4 g 100 230',
'STATE 3 r 230 2597',
'STATE 6 u 100 2598',
'STATE 4 f 29 2599',
'STATE 3 r 29 2600',
'STATE 4 b 230 2601',
'STATE 3 n 100 230',
'STATE 2 i 68 29',
'STATE 4 c 1416 2602',
'STATE 6 p 100 2603',
'STATE 4 m 42 2604',
'STATE 6 n 2605 68',
'STATE 1 a 68 42',
'STATE 6 c 100 2606',
'STATE 4 d 42 2607',
'STATE 6 n 100 1353',
'STATE 2 r 68 2608',
'STATE 3 h 100 2609',
'STATE 2 h 68 2610',
'STATE 2 n 230 2611',
'STATE 1 k 28 2612',
'STATE 1 d 29 230',
'STATE 3 t 28 2613',
'STATE 6 o 2615 2614',
'STATE 4 g 2617 2616',
'STATE 3 f 29 2618',
'STATE 6 i 2620 2619',
'STATE 2 i 68 2621',
'STATE 5 h 68 2622',
'STATE 4 p 100 2623',
'STATE 3 k 2624 100',
'STATE 2 g 100 2625',
'STATE 5 u 68 2626',
'STATE 3 r 68 2627',
'STATE 6 o 2317 68',
'STATE 1 u 68 2628',
'STATE 1 m 100 2629',
'STATE 4 t 100 2630',
'STATE 6 o 68 2631',
'STATE 5 n 100 2632',
'STATE 5 p 100 68',
'STATE 6 r 100 2216',
'STATE 6 i 2634 2633',
'STATE 5 c 100 2635',
'STATE 2 s 100 2636',
'STATE 5 u 1891 100',
'STATE 4 g 100 2637',
'STATE 2 c 42 100',
'STATE 3 h 100 68',
'STATE 4 c 2638 100',
'STATE 4 t 100 2639',
'STATE 5 l 100 2640',
'STATE 4 s 100 562',
'STATE 3 b 230 100',
'STATE 3 k 29 2641',
'STATE 4 m 100 2642',
'STATE 5 z 230 2643',
'STATE 5 r 230 100',
'STATE 3 l 230 2644',
'STATE 6 i 1996 100',
'STATE 3 l 100 2645',
'STATE 4 g 100 2646',
'STATE 3 t 100 230',
'STATE 3 p 68 2647',
'STATE 3 v 100 2648',
'STATE 4 p 230 42',
'STATE 4 t 100 2649',
'STATE 3 p 2651 2650',
'STATE 3 p 100 1825',
'STATE 1 e 42 2652',
'STATE 2 t 100 2653',
'STATE 1 e 68 2654',
'STATE 5 s 29 2655',
'STATE 3 t 28 2656',
'STATE 1 a 28 2657',
'STATE 6 h 100 2658',
'STATE 5 n 230 2659',
'STATE 3 m 68 2660',
'STATE 1 r 100 42',
'STATE 4 t 2662 2661',
'STATE 3 l 29 2663',
'STATE 3 i 68 100',
'STATE 3 s 68 42',
'STATE 3 s 68 100',
'STATE 3 v 562 2664',
'STATE 2 n 562 100',
'STATE 1 a 68 100',
'STATE 5 h 68 2665',
'STATE 1 o 100 2666',
'STATE 6 e 2668 2667',
'STATE 6 r 2670 2669',
'STATE 5 f 2672 2671',
'STATE 6 u 68 100',
'STATE 4 d 2673 100',
'STATE 6 o 2675 2674',
'STATE 3 r 2676 100',
'STATE 3 n 1423 2677',
'STATE 5 t 2678 100',
'STATE 3 r 42 100',
'STATE 3 b 29 2679',
'STATE 4 f 100 2680',
'STATE 3 c 2073 2681',
'STATE 5 b 2683 2682',
'STATE 6 i 100 2684',
'STATE 5 l 100 2685',
'STATE 4 c 100 2686',
'STATE 6 m 68 2687',
'STATE 6 y 2689 2688',
'STATE 4 t 100 2690',
'STATE 6 b 100 230',
'STATE 2 t 68 2691',
'STATE 3 n 2693 2692',
'STATE 6 r 68 2694',
'STATE 2 n 68 42',
'STATE 1 d 68 2695',
'STATE 1 m 100 2696',
'STATE 6 o 29 2697',
'STATE 1 o 29 2698',
'STATE 1 i 28 2699',
'STATE 5 k 2701 2700',
'STATE 5 z 29 2702',
'STATE 3 s 68 2703',
'STATE 2 h 100 2704',
'STATE 2 m 68 2705',
'STATE 3 r 28 68',
'STATE 2 t 100 2706',
'STATE 6 # 68 2707',
'STATE 6 # 100 68',
'STATE 5 t 2709 2708',
'STATE 5 t 29 100',
'STATE 3 h 100 2710',
'STATE 3 x 100 29',
'STATE 5 s 2712 2711',
'STATE 6 o 68 2713',
'STATE 5 d 68 100',
'STATE 6 a 2715 2714',
'STATE 3 l 2717 2716',
'STATE 4 n 2719 2718',
'STATE 4 n 29 230',
'STATE 4 n 100 562',
'STATE 6 t 29 100',
'STATE 4 m 100 2299',
'STATE 3 z 230 2720',
'STATE 3 c 100 2721',
'STATE 3 r 68 100',
'STATE 3 m 100 2722',
'STATE 4 m 100 2723',
'STATE 6 l 230 2724',
'STATE 6 u 68 2725',
'STATE 3 t 100 2726',
'STATE 3 m 29 100',
'STATE 3 c 230 2727',
'STATE 3 v 68 2728',
'STATE 4 t 42 2729',
'STATE 6 r 42 68',
'STATE 4 l 42 230',
'STATE 3 v 68 2730',
'STATE 3 c 68 2731',
'STATE 3 c 29 2732',
'STATE 1 a 29 2733',
'STATE 3 r 29 2734',
'STATE 3 l 100 2735',
'STATE 6 s 100 2736',
'STATE 4 s 29 1860',
'STATE 3 n 562 2737',
'STATE 6 i 2739 2738',
'STATE 3 n 68 2740',
'STATE 5 h 2742 2741',
'STATE 6 o 68 1722',
'STATE 5 z 29 2743',
'STATE 1 n 68 2744',
'STATE 6 i 100 2745',
'STATE 6 u 562 2746',
'STATE 4 b 2747 100',
'STATE 6 i 68 2748',
'STATE 3 v 68 2749',
'STATE 5 c 2194 2750',
'STATE 5 k 100 2751',
'STATE 5 b 562 100',
'STATE 2 g 100 2752',
'STATE 5 c 562 100',
'STATE 3 v 100 2753',
'STATE 3 t 2754 100',
'STATE 6 o 68 29',
'STATE 5 m 100 2755',
'STATE 4 t 100 2756',
'STATE 4 s 100 2757',
'STATE 6 h 29 2758',
'STATE 4 k 68 2759',
'STATE 4 l 68 2760',
'STATE 4 s 42 2761',
'STATE 2 r 2763 2762',
'STATE 3 s 68 2764',
'STATE 2 e 230 29',
'STATE 3 i 28 2765',
'STATE 2 t 28 2766',
'STATE 4 c 29 2767',
'STATE 2 e 100 2768',
'STATE 2 o 2216 2769',
'STATE 4 c 2771 2770',
'STATE 3 h 29 68',
'STATE 2 o 100 2772',
'STATE 2 l 2774 2773',
'STATE 4 t 1287 68',
'STATE 5 j 29 2295',
'STATE 1 o 230 2775',
'STATE 1 s 100 2776',
'STATE 5 d 2778 2777',
'STATE 6 t 68 1204',
'STATE 6 l 68 100',
'STATE 5 u 2779 100',
'STATE 4 g 100 2780',
'STATE 5 n 230 2781',
'STATE 5 h 42 100',
'STATE 5 c 2782 100',
'STATE 5 p 100 562',
'STATE 3 z 230 2783',
'STATE 4 m 100 2784',
'STATE 6 w 68 2785',
'STATE 4 v 29 2786',
'STATE 2 # 2788 2787',
'STATE 4 c 42 2789',
'STATE 4 v 42 2790',
'STATE 2 u 562 2791',
'STATE 1 e 100 1425',
'STATE 3 t 68 2792',
'STATE 1 m 28 2793',
'STATE 2 i 28 2794',
'STATE 1 i 100 2795',
'STATE 2 u 29 2796',
'STATE 3 h 2798 2797',
'STATE 6 e 100 68',
'STATE 2 e 29 2799',
'STATE 2 e 100 68',
'STATE 5 k 2801 2800',
'STATE 4 m 68 2802',
'STATE 2 r 100 2803',
'STATE 6 o 100 2804',
'STATE 6 a 2806 2805',
'STATE 4 b 100 2807',
'STATE 2 s 2601 100',
'STATE 4 m 100 2808',
'STATE 2 f 100 2809',
'STATE 3 f 100 230',
'STATE 3 p 2811 2810',
'STATE 3 c 100 2812',
'STATE 4 d 100 2813',
'STATE 6 k 29 2814',
'STATE 4 v 230 2815',
'STATE 6 r 68 2816',
'STATE 4 v 68 230',
'STATE 3 d 42 2817',
'STATE 2 c 100 2818',
'STATE 2 a 68 2819',
'STATE 2 o 29 2820',
'STATE 3 b 28 2821',
'STATE 5 p 100 2822',
'STATE 4 s 29 2823',
'STATE 4 s 2824 100',
'STATE 5 m 100 68',
'STATE 1 t 29 68',
'STATE 1 o 100 2825',
'STATE 1 o 68 2826',
'STATE 5 s 562 68',
'STATE 2 n 100 2827',
'STATE 1 l 100 2828',
'STATE 5 k 562 2829',
'STATE 5 v 68 100',
'STATE 6 i 68 100',
'STATE 4 n 100 2830',
'STATE 5 h 29 100',
'STATE 5 t 2832 2831',
'STATE 5 s 230 2538',
'STATE 3 p 100 230',
'STATE 3 s 100 2748',
'STATE 4 z 68 2833',
'STATE 4 b 68 2834',
'STATE 4 n 68 1799',
'STATE 6 a 42 2835',
'STATE 3 b 100 2836',
'STATE 1 a 68 2837',
'STATE 2 a 29 2838',
'STATE 3 n 28 2839',
'STATE 2 s 100 2840',
'STATE 1 a 100 2841',
'STATE 3 p 100 2625',
'STATE 2 s 100 2842',
'STATE 3 s 68 2843',
'STATE 6 a 100 2844',
'STATE 3 m 2770 100',
'STATE 6 i 562 2845',
'STATE 2 f 100 2846',
'STATE 3 k 100 2847',
'STATE 4 t 230 100',
'STATE 3 l 68 2848',
'STATE 4 g 68 100',
'STATE 6 r 2850 2849',
'STATE 3 p 100 2851',
'STATE 2 o 2852 68',
'STATE 2 i 29 28',
'STATE 2 s 28 2853',
'STATE 1 m 100 2854',
'STATE 2 c 100 29',
'STATE 3 c 100 2855',
'STATE 1 a 68 1853',
'STATE 2 s 100 2856',
'STATE 5 v 100 2857',
'STATE 5 u 100 2858',
'STATE 5 g 100 2859',
'STATE 4 h 68 2860',
'STATE 4 n 100 1552',
'STATE 4 m 42 2861',
'STATE 2 g 100 2862',
'STATE 1 d 29 68',
'STATE 3 g 28 2863',
'STATE 5 m 2865 2864',
'STATE 2 g 68 2866',
'STATE 3 r 100 2867',
'STATE 5 n 100 2868',
'STATE 3 l 100 1227',
'STATE 3 r 100 2869',
'STATE 6 m 230 100',
'STATE 4 d 100 2870',
'STATE 2 e 2183 2871',
'STATE 2 c 28 240',
'STATE 5 b 100 2872',
'STATE 6 a 100 562',
'STATE 4 m 68 2873',
'STATE 2 a 100 2874',
'STATE 4 b 100 2875',
'STATE 5 c 2876 100',
'STATE 3 g 100 42',
'STATE 4 p 100 2877',
'STATE 2 l 2879 2878',
'STATE 3 r 100 2880',
'STATE 6 i 100 29',
'STATE 5 m 562 68',
'STATE 4 c 100 230',
'STATE 3 i 100 2881',
'STATE 5 s 2883 2882',
'STATE 1 a 230 68',
'STATE 5 s 100 68',
'STATE 1 c 68 2884',
'STATE 3 p 230 2885',
'STATE 1 a 68 2886',
'STATE 3 r 2183 2887',
'STATE 4 g 100 2888',
'STATE 4 s 2889 29',
'STATE 4 d 562 2890',
'STATE 4 d 100 2891',
'STATE 2 o 29 100',
'STATE 4 t 100 2892',
'STATE 5 c 100 68',
'STATE 3 m 562 2893',
'STATE 2 s 562 100',
'INDEX 2894 b',
'STATE 4 b 2896 2895',
'STATE 3 m 2898 2897',
'STATE 1 c 2900 2899',
'STATE 4 t 2902 2901',
'STATE 4 # 2899 2903',
'PHONE epsilon',
'STATE 2 r 2904 2899',
'PHONE b',
'STATE 1 d 2899 2905',
'STATE 4 s 2899 2906',
'STATE 3 a 2899 2901',
'STATE 2 d 2899 2901',
'STATE 2 o 2908 2907',
'STATE 1 l 2910 2909',
'STATE 4 a 2901 2911',
'STATE 2 u 2912 2901',
'STATE 2 i 2913 2901',
'STATE 1 b 2899 2914',
'STATE 1 d 2899 2901',
'STATE 4 e 2899 2901',
'STATE 1 c 2913 2901',
'INDEX 2915 c',
'STATE 4 k 2917 2916',
'STATE 4 h 2919 2918',
'STATE 5 i 2921 2920',
'STATE 4 e 2923 2922',
'STATE 3 s 2925 2924',
'PHONE epsilon',
'STATE 6 # 2926 2920',
'STATE 4 i 2928 2927',
'STATE 3 s 2920 2929',
'STATE 3 t 2931 2930',
'STATE 2 t 2931 2932',
'STATE 1 # 2920 2933',
'STATE 4 c 2935 2934',
'STATE 3 s 2937 2936',
'STATE 3 m 2939 2938',
'STATE 2 0 2941 2940',
'PHONE ch',
'STATE 5 o 2943 2942',
'STATE 1 o 2945 2944',
'STATE 4 y 2947 2946',
'STATE 5 e 2949 2948',
'STATE 5 a 2951 2950',
'STATE 5 a 2953 2952',
'STATE 3 x 2920 2954',
'PHONE k',
'STATE 3 n 2956 2955',
'STATE 5 r 2939 2957',
'STATE 5 i 2959 2958',
'STATE 6 o 2939 2920',
'STATE 1 a 2961 2960',
'PHONE t-s',
'STATE 4 z 2963 2962',
'STATE 5 r 2939 2964',
'STATE 5 i 2965 2920',
'STATE 3 a 2939 2966',
'STATE 3 c 2968 2967',
'STATE 6 l 2970 2969',
'STATE 1 p 2931 2920',
'STATE 1 0 2920 2931',
'STATE 5 k 2972 2971',
'STATE 2 v 2931 2973',
'STATE 5 o 2975 2974',
'STATE 5 l 2939 2976',
'STATE 6 d 2978 2977',
'STATE 1 0 2920 2939',
'STATE 3 i 2945 2979',
'STATE 2 l 2920 2945',
'STATE 4 g 2920 2980',
'STATE 2 s 2920 2931',
'PHONE s',
'STATE 6 # 2920 2981',
'STATE 2 s 2939 2920',
'STATE 6 u 2983 2982',
'STATE 5 # 2931 2984',
'STATE 3 i 2970 2985',
'PHONE sh',
'STATE 6 n 2987 2986',
'STATE 3 i 2931 2988',
'STATE 2 e 2990 2989',
'STATE 6 l 2992 2991',
'STATE 6 r 2939 2993',
'STATE 5 a 2995 2994',
'STATE 1 0 2997 2996',
'STATE 5 a 2920 2939',
'STATE 1 r 2945 2998',
'STATE 4 q 2920 2999',
'STATE 6 d 2939 3000',
'STATE 3 x 2920 3001',
'STATE 5 o 2970 2964',
'STATE 5 o 2931 3002',
'STATE 6 t 3004 3003',
'STATE 2 0 3006 3005',
'STATE 5 a 2970 2964',
'STATE 1 # 2931 3007',
'STATE 5 e 3009 3008',
'STATE 3 i 2939 3010',
'STATE 5 e 2931 3011',
'STATE 2 i 2931 2970',
'STATE 6 # 2931 2939',
'STATE 6 v 2931 3012',
'STATE 6 r 2931 3013',
'STATE 5 a 3014 2920',
'STATE 6 m 3015 2920',
'STATE 2 r 2945 3016',
'STATE 3 k 2920 3017',
'STATE 6 a 2920 3018',
'STATE 6 n 3020 3019',
'STATE 1 # 2964 3021',
'STATE 3 c 2931 3022',
'STATE 1 p 2970 2964',
'STATE 5 o 2970 3023',
'STATE 5 n 2964 3024',
'STATE 1 o 2931 3025',
'STATE 3 e 3027 3026',
'STATE 6 r 3029 3028',
'STATE 3 c 2939 3030',
'STATE 1 s 2939 3031',
'STATE 6 r 3033 3032',
'STATE 6 u 2970 2931',
'STATE 6 r 3034 2920',
'STATE 5 e 2939 2920',
'STATE 3 a 2945 3035',
'STATE 3 n 3037 3036',
'STATE 3 a 3038 2920',
'STATE 3 m 2939 3039',
'STATE 2 f 2970 3040',
'STATE 1 v 2964 2931',
'STATE 1 0 2931 3041',
'STATE 6 c 3043 3042',
'STATE 6 s 2964 3044',
'STATE 1 a 2931 3045',
'STATE 2 b 3047 3046',
'STATE 2 i 3049 3048',
'STATE 6 n 3051 3050',
'STATE 3 a 2939 3052',
'STATE 3 u 2939 3053',
'STATE 5 # 2931 3054',
'STATE 6 a 3056 3055',
'STATE 5 o 2939 2931',
'STATE 1 d 2931 2939',
'STATE 3 o 2945 2920',
'STATE 2 u 3058 3057',
'STATE 1 r 3059 2939',
'STATE 1 # 2939 2920',
'STATE 5 e 3061 3060',
'STATE 5 n 2964 3062',
'STATE 3 n 2931 3063',
'STATE 3 n 3065 3064',
'STATE 2 a 2931 2964',
'STATE 5 m 2964 3066',
'STATE 3 a 2964 2931',
'STATE 3 c 2939 3067',
'STATE 5 # 2939 3068',
'STATE 5 a 3070 3069',
'STATE 5 # 2931 2939',
'STATE 6 l 3072 3071',
'STATE 3 r 2931 2939',
'STATE 1 r 2931 3073',
'STATE 3 r 3074 2931',
'STATE 6 s 2931 3075',
'STATE 6 m 3076 2931',
'STATE 5 i 2939 2931',
'STATE 4 # 2939 3077',
'STATE 4 l 2920 2939',
'STATE 5 i 2964 2939',
'STATE 5 # 3079 3078',
'STATE 6 # 2939 3080',
'STATE 3 n 2931 2964',
'STATE 6 n 2970 3081',
'STATE 5 l 3083 3082',
'STATE 5 n 3084 2964',
'STATE 6 u 2931 3085',
'STATE 2 o 3087 3086',
'STATE 1 # 3088 2931',
'STATE 5 i 3090 3089',
'STATE 2 m 2939 2931',
'STATE 6 t 3092 3091',
'STATE 2 r 2970 3093',
'STATE 3 e 2939 3094',
'STATE 5 a 2931 3095',
'STATE 1 r 2931 3096',
'STATE 5 e 2939 2931',
'STATE 1 n 3097 2939',
'STATE 6 g 2964 3098',
'STATE 3 i 2931 3099',
'STATE 3 n 2964 3100',
'STATE 3 r 2964 3101',
'STATE 1 a 3103 3102',
'STATE 6 e 2964 3104',
'STATE 1 v 2931 2964',
'STATE 5 p 2964 3105',
'STATE 2 u 3107 3106',
'STATE 3 r 3108 2931',
'STATE 6 e 2939 3109',
'STATE 1 c 3110 2939',
'STATE 6 n 2931 2939',
'STATE 6 m 3112 3111',
'STATE 2 a 2939 3113',
'STATE 3 a 2931 2939',
'STATE 1 # 2931 3114',
'STATE 1 k 2931 3115',
'STATE 6 n 2931 3116',
'STATE 4 t 3117 2939',
'STATE 5 n 3119 3118',
'STATE 3 a 2931 2964',
'STATE 3 r 2939 3120',
'STATE 3 a 2931 2970',
'STATE 5 s 3122 3121',
'STATE 3 i 2964 3099',
'STATE 3 a 2964 3123',
'STATE 6 t 2964 3124',
'STATE 5 i 3126 3125',
'STATE 5 a 3127 2931',
'STATE 5 a 2931 3056',
'STATE 6 n 2939 3128',
'STATE 2 h 2931 2939',
'STATE 2 s 2939 3129',
'STATE 3 o 2939 3130',
'STATE 1 # 2970 3131',
'STATE 3 r 3133 3132',
'STATE 5 e 2931 2939',
'STATE 2 i 2931 3134',
'STATE 3 i 3135 2939',
'STATE 3 i 2964 3136',
'STATE 2 m 2939 3137',
'STATE 2 0 2931 3138',
'STATE 6 # 2964 3139',
'STATE 6 t 2920 2964',
'STATE 6 y 2964 3140',
'STATE 5 r 3141 2964',
'STATE 5 a 3143 3142',
'STATE 3 a 3145 3144',
'STATE 1 p 2931 2970',
'STATE 6 a 2939 3146',
'STATE 6 d 2931 3147',
'STATE 1 # 2931 2939',
'STATE 3 o 2970 2931',
'STATE 2 o 2931 2939',
'STATE 2 a 2931 2939',
'STATE 5 a 3148 2931',
'STATE 2 d 2920 2939',
'STATE 5 c 3150 3149',
'STATE 6 s 2931 3151',
'STATE 6 s 2964 3152',
'STATE 6 o 2964 3153',
'STATE 1 m 3155 3154',
'STATE 6 e 2964 3156',
'STATE 6 w 3158 3157',
'STATE 6 e 2939 3159',
'STATE 3 r 2939 3160',
'STATE 6 e 2931 3161',
'STATE 3 u 2939 2931',
'STATE 6 v 2931 3162',
'STATE 6 r 2931 2939',
'STATE 5 o 3164 3163',
'STATE 1 0 2964 2931',
'STATE 1 m 2931 3165',
'STATE 3 a 2970 2964',
'STATE 3 i 2964 3166',
'STATE 2 a 2964 3167',
'STATE 6 l 2964 2931',
'STATE 6 a 2964 3168',
'STATE 3 o 3170 3169',
'STATE 3 a 2920 2931',
'STATE 1 0 2939 3171',
'STATE 2 s 2939 3172',
'STATE 2 m 2970 3173',
'STATE 1 l 2931 3174',
'STATE 5 u 2964 3175',
'STATE 3 o 2964 3176',
'STATE 6 o 2931 3177',
'STATE 2 e 2964 3178',
'STATE 6 l 3179 2964',
'STATE 6 v 2964 2931',
'STATE 2 r 3181 3180',
'STATE 5 u 2931 2939',
'STATE 3 r 3183 3182',
'STATE 2 a 2931 3184',
'STATE 1 s 2970 2993',
'STATE 6 c 2931 3185',
'STATE 5 k 2931 3186',
'STATE 6 # 3187 2931',
'STATE 3 l 2931 3188',
'STATE 2 d 2964 3189',
'STATE 3 i 2964 3190',
'STATE 2 l 3192 3191',
'STATE 1 # 2931 3193',
'STATE 6 u 2970 3194',
'STATE 1 # 2970 2939',
'STATE 2 m 2931 3195',
'STATE 3 u 3197 3196',
'STATE 5 m 3199 3198',
'STATE 3 a 2964 3200',
'STATE 6 a 3201 2964',
'STATE 5 r 3203 3202',
'STATE 1 p 2964 2931',
'STATE 2 y 2920 3204',
'STATE 5 # 2939 3205',
'STATE 3 a 3049 2939',
'STATE 3 u 3207 3206',
'STATE 3 o 3208 2931',
'STATE 2 o 3210 3209',
'STATE 6 # 2970 2931',
'STATE 3 e 2964 3211',
'STATE 1 0 2931 2964',
'STATE 1 c 2931 2964',
'STATE 1 l 2964 2931',
'STATE 3 e 2964 3212',
'STATE 2 r 2964 3213',
'STATE 5 # 3215 3214',
'STATE 5 l 2939 3216',
'STATE 3 a 3218 3217',
'STATE 6 r 2939 2931',
'STATE 2 r 2939 2931',
'STATE 1 c 2939 3219',
'STATE 1 # 2939 2931',
'STATE 1 # 3220 2964',
'STATE 2 l 2964 3221',
'STATE 1 # 3045 2964',
'STATE 3 y 2939 3222',
'STATE 1 # 2931 3223',
'STATE 6 e 2939 3224',
'STATE 3 i 2931 3225',
'STATE 1 # 3207 2931',
'STATE 2 r 3227 3226',
'STATE 6 a 2931 2964',
'STATE 1 s 2964 3228',
'STATE 6 l 2939 3229',
'STATE 1 t 2931 3230',
'STATE 5 t 2931 2939',
'STATE 3 o 2939 2931',
'STATE 1 a 2931 3231',
'STATE 3 i 2931 3232',
'STATE 1 f 2964 3233',
'STATE 5 u 2931 3234',
'STATE 1 a 2920 3235',
'STATE 2 a 2931 3236',
'STATE 3 o 2931 2939',
'STATE 6 e 2964 3237',
'STATE 5 o 3239 3238',
'STATE 1 o 2939 3240',
'STATE 1 # 2931 3197',
'STATE 1 t 2964 3241',
'STATE 6 i 3051 3242',
'STATE 6 # 2931 3243',
'STATE 1 u 2939 3244',
'STATE 3 a 3245 2964',
'STATE 2 n 2939 3246',
'STATE 1 0 2939 2931',
'STATE 3 a 2939 3247',
'STATE 2 f 2964 3248',
'STATE 1 # 3250 3249',
'STATE 1 r 2939 3251',
'STATE 5 t 2964 3252',
'STATE 5 m 2931 2939',
'STATE 5 y 2931 3253',
'STATE 3 i 2931 3254',
'STATE 1 # 2964 3255',
'STATE 2 m 2931 3256',
'STATE 2 a 2931 3257',
'STATE 1 g 2964 3258',
'STATE 6 a 2931 2939',
'STATE 2 i 2931 2939',
'STATE 1 r 2964 3259',
'STATE 5 # 2964 2931',
'INDEX 3260 d',
'STATE 4 # 3262 3261',
'STATE 4 d 3264 3263',
'STATE 3 e 3266 3265',
'STATE 4 g 3268 3267',
'STATE 2 g 3270 3269',
'PHONE d',
'STATE 2 k 3272 3271',
'STATE 4 t 3274 3273',
'STATE 5 e 3276 3275',
'STATE 3 o 3276 3277',
'STATE 3 o 3265 3276',
'STATE 2 h 3279 3278',
'PHONE t',
'STATE 4 j 3276 3280',
'STATE 5 # 3282 3281',
'STATE 5 r 3265 3283',
'PHONE epsilon',
'STATE 2 b 3276 3284',
'STATE 2 p 3272 3285',
'STATE 1 t 3265 3286',
'STATE 3 d 3288 3287',
'STATE 1 # 3265 3289',
'STATE 3 u 3265 3270',
'STATE 5 a 3265 3290',
'STATE 3 e 3292 3291',
'STATE 2 c 3272 3293',
'STATE 1 g 3265 3272',
'STATE 4 u 3295 3294',
'STATE 2 e 3297 3296',
'STATE 5 h 3265 3298',
'STATE 5 o 3265 3276',
'STATE 5 y 3276 3299',
'STATE 5 y 3265 3300',
'STATE 2 s 3302 3301',
'STATE 4 z 3303 3265',
'STATE 2 0 3265 3304',
'STATE 4 e 3306 3305',
'STATE 4 y 3276 3307',
'STATE 1 o 3265 3308',
'STATE 5 i 3276 3309',
'STATE 6 r 3276 3310',
'STATE 2 f 3272 3311',
'STATE 1 s 3272 3312',
'STATE 6 # 3265 3313',
'STATE 5 a 3315 3314',
'STATE 1 # 3265 3316',
'STATE 1 b 3265 3317',
'STATE 6 # 3265 3318',
'STATE 5 k 3265 3276',
'STATE 2 m 3276 3319',
'STATE 2 p 3265 3276',
'STATE 2 x 3272 3265',
'STATE 1 r 3272 3320',
'STATE 6 k 3265 3321',
'STATE 5 l 3323 3322',
'STATE 1 g 3321 3324',
'STATE 1 w 3265 3325',
'STATE 1 l 3265 3326',
'STATE 1 p 3276 3327',
'STATE 6 d 3276 3328',
'STATE 1 a 3272 3329',
'PHONE jh',
'STATE 3 e 3330 3265',
'STATE 3 o 3321 3331',
'STATE 3 i 3321 3265',
'STATE 6 d 3265 3332',
'STATE 5 d 3265 3333',
'STATE 5 r 3265 3334',
'STATE 2 w 3265 3335',
'STATE 1 n 3272 3336',
'STATE 1 # 3265 3337',
'STATE 2 h 3321 3338',
'STATE 2 a 3340 3339',
'STATE 1 m 3265 3341',
'STATE 4 i 3265 3342',
'STATE 1 s 3344 3343',
'STATE 1 i 3265 3345',
'STATE 1 0 3321 3346',
'STATE 6 t 3265 3347',
'STATE 1 r 3348 3265',
'STATE 6 # 3265 3349',
'STATE 1 s 3265 3350',
'STATE 5 # 3276 3265',
'STATE 2 p 3276 3351',
'STATE 2 h 3265 3276',
'STATE 1 o 3265 3352',
'STATE 5 c 3265 3321',
'STATE 2 i 3265 3353',
'STATE 2 i 3276 3265',
'STATE 1 h 3276 3265',
'STATE 6 # 3265 3354',
'STATE 1 c 3276 3355',
'STATE 1 u 3265 3272',
'STATE 3 n 3321 3265',
'STATE 2 a 3276 3356',
'STATE 2 l 3276 3357',
'STATE 5 r 3276 3265',
'STATE 2 f 3276 3358',
'STATE 3 i 3360 3359',
'STATE 2 r 3276 3361',
'STATE 2 r 3265 3276',
'STATE 5 e 3362 3276',
'STATE 1 # 3276 3265',
'INDEX 3363 e',
'STATE 6 0 3365 3364',
'STATE 4 r 3367 3366',
'STATE 4 # 3369 3368',
'STATE 4 a 3371 3370',
'STATE 5 r 3373 3372',
'STATE 4 r 3375 3374',
'STATE 3 e 3377 3376',
'STATE 3 e 3379 3378',
'STATE 5 r 3381 3380',
'STATE 5 i 3383 3382',
'STATE 3 b 3385 3384',
'STATE 4 n 3387 3386',
'STATE 3 e 3389 3388',
'STATE 3 n 3391 3390',
'STATE 2 r 3393 3392',
'STATE 4 e 3395 3394',
'STATE 2 r 3397 3396',
'STATE 5 u 3399 3398',
'STATE 3 w 3385 3400',
'STATE 3 e 3389 3401',
'STATE 6 n 3403 3402',
'STATE 1 0 3405 3404',
'PHONE eh1',
'STATE 3 e 3407 3406',
'STATE 3 e 3393 3408',
'STATE 2 e 3409 3399',
'PHONE ih1',
'STATE 3 r 3411 3410',
'STATE 2 o 3413 3412',
'STATE 2 p 3393 3414',
'PHONE iy1',
'STATE 4 i 3416 3415',
'STATE 3 r 3417 3399',
'STATE 2 f 3418 3393',
'STATE 1 # 3420 3419',
'STATE 5 d 3422 3421',
'PHONE epsilon',
'STATE 3 b 3424 3423',
'STATE 5 o 3426 3425',
'STATE 6 a 3428 3427',
'STATE 3 e 3389 3399',
'STATE 3 h 3385 3429',
'STATE 3 p 3399 3430',
'STATE 4 l 3432 3431',
'STATE 1 # 3393 3433',
'STATE 3 m 3435 3434',
'STATE 3 w 3437 3436',
'STATE 3 k 3439 3438',
'STATE 2 d 3441 3440',
'STATE 2 i 3399 3442',
'STATE 1 i 3444 3443',
'STATE 2 n 3393 3445',
'STATE 1 0 3447 3446',
'STATE 5 r 3449 3448',
'STATE 1 0 3451 3450',
'STATE 1 # 3393 3452',
'STATE 1 p 3453 3393',
'STATE 5 a 3455 3454',
'STATE 6 h 3457 3456',
'STATE 3 h 3385 3458',
'STATE 1 u 3444 3459',
'STATE 6 d 3399 3385',
'STATE 5 e 3461 3460',
'STATE 1 0 3463 3462',
'STATE 6 o 3465 3464',
'STATE 3 l 3399 3466',
'STATE 2 o 3399 3467',
'STATE 6 a 3468 3385',
'STATE 4 k 3470 3469',
'STATE 3 i 3472 3471',
'STATE 2 r 3393 3473',
'STATE 2 g 3475 3474',
'STATE 1 a 3455 3476',
'STATE 3 y 3477 3399',
'PHONE y-uw1',
'STATE 2 n 3479 3478',
'STATE 2 t 3444 3480',
'STATE 2 b 3399 3481',
'STATE 1 n 3399 3455',
'STATE 2 a 3399 3482',
'STATE 1 t 3399 3483',
'PHONE iy',
'STATE 1 d 3393 3484',
'STATE 3 i 3486 3485',
'STATE 3 r 3488 3487',
'STATE 5 g 3490 3489',
'STATE 6 a 3385 3491',
'STATE 2 p 3492 3399',
'STATE 6 a 3444 3493',
'STATE 4 d 3444 3393',
'STATE 4 m 3385 3494',
'STATE 4 n 3389 3495',
'PHONE ah',
'STATE 1 0 3497 3496',
'STATE 5 t 3498 3393',
'STATE 3 t 3500 3499',
'STATE 6 r 3393 3399',
'STATE 5 a 3502 3501',
'STATE 6 d 3399 3503',
'STATE 6 # 3505 3504',
'STATE 3 a 3385 3506',
'STATE 6 e 3508 3507',
'STATE 3 t 3389 3509',
'STATE 2 a 3389 3510',
'STATE 6 y 3385 3511',
'STATE 3 s 3385 3512',
'STATE 3 t 3514 3513',
'STATE 1 # 3385 3515',
'STATE 3 t 3517 3516',
'STATE 1 # 3399 3518',
'STATE 4 d 3393 3519',
'STATE 3 i 3521 3520',
'STATE 3 r 3522 3455',
'STATE 2 l 3522 3523',
'STATE 1 m 3399 3524',
'STATE 3 h 3399 3525',
'STATE 3 t 3526 3399',
'STATE 2 z 3444 3527',
'STATE 1 i 3529 3528',
'STATE 2 e 3399 3530',
'STATE 1 h 3399 3531',
'STATE 2 s 3533 3532',
'STATE 5 t 3535 3534',
'STATE 5 t 3537 3536',
'STATE 4 w 3539 3538',
'STATE 4 v 3541 3540',
'STATE 3 r 3543 3542',
'STATE 6 h 3545 3544',
'STATE 6 o 3385 3546',
'STATE 5 m 3444 3399',
'STATE 6 t 3444 3399',
'PHONE ih',
'STATE 6 # 3393 3547',
'STATE 5 s 3549 3548',
'STATE 3 r 3551 3550',
'STATE 2 # 3385 3552',
'STATE 3 d 3385 3553',
'STATE 1 # 3385 3554',
'STATE 5 u 3556 3555',
'STATE 6 # 3558 3557',
'STATE 1 0 3560 3559',
'STATE 1 # 3546 3399',
'STATE 3 i 3399 3385',
'STATE 6 u 3399 3561',
'STATE 3 d 3563 3562',
'STATE 1 0 3385 3564',
'STATE 1 i 3385 3565',
'STATE 2 i 3389 3566',
'STATE 1 m 3385 3567',
'STATE 3 t 3399 3568',
'STATE 4 t 3570 3569',
'STATE 4 d 3455 3571',
'STATE 3 t 3573 3572',
'STATE 3 a 3399 3574',
'STATE 2 s 3455 3575',
'STATE 1 a 3576 3399',
'STATE 1 s 3393 3577',
'STATE 1 # 3578 3455',
'STATE 2 t 3399 3579',
'PHONE eh',
'STATE 1 o 3455 3580',
'PHONE ey1',
'STATE 2 e 3582 3581',
'STATE 1 a 3444 3583',
'STATE 2 d 3444 3584',
'STATE 1 d 3399 3585',
'STATE 2 o 3444 3399',
'STATE 2 y 3399 3586',
'STATE 1 b 3399 3587',
'STATE 1 e 3444 3588',
'STATE 1 s 3444 3393',
'STATE 5 s 3590 3589',
'STATE 6 # 3592 3591',
'STATE 4 n 3594 3593',
'STATE 4 s 3596 3595',
'STATE 4 u 3598 3597',
'STATE 3 n 3600 3599',
'STATE 4 u 3602 3601',
'STATE 5 i 3604 3603',
'STATE 3 v 3606 3605',
'STATE 5 n 3608 3607',
'STATE 6 n 3609 3399',
'STATE 3 l 3399 3610',
'STATE 3 h 3385 3399',
'STATE 5 e 3611 3393',
'STATE 5 b 3613 3612',
'STATE 6 u 3385 3614',
'STATE 3 h 3616 3615',
'STATE 5 g 3618 3617',
'STATE 3 h 3393 3385',
'STATE 3 r 3620 3619',
'STATE 1 m 3385 3621',
'STATE 2 e 3623 3622',
'STATE 1 0 3624 3399',
'STATE 1 0 3626 3625',
'STATE 3 i 3399 3627',
'STATE 6 # 3629 3628',
'STATE 6 g 3631 3630',
'STATE 3 p 3399 3632',
'STATE 6 z 3399 3633',
'STATE 1 0 3634 3399',
'STATE 2 x 3389 3399',
'STATE 3 p 3389 3385',
'STATE 2 m 3389 3635',
'STATE 6 o 3637 3636',
'STATE 3 f 3399 3385',
'STATE 4 w 3639 3638',
'STATE 3 k 3455 3640',
'STATE 4 s 3642 3641',
'STATE 3 c 3644 3643',
'STATE 2 a 3522 3645',
'STATE 3 u 3647 3646',
'STATE 2 t 3455 3648',
'STATE 2 n 3455 3399',
'STATE 2 s 3393 3649',
'STATE 3 o 3399 3650',
'STATE 2 r 3455 3399',
'STATE 2 r 3522 3651',
'STATE 3 b 3399 3652',
'STATE 3 s 3399 3653',
'STATE 1 o 3444 3654',
'STATE 2 m 3444 3399',
'STATE 2 u 3399 3655',
'STATE 1 e 3399 3656',
'STATE 1 d 3399 3657',
'STATE 1 u 3444 3658',
'STATE 4 n 3660 3659',
'STATE 6 # 3662 3661',
'STATE 4 n 3664 3663',
'STATE 4 n 3666 3665',
'STATE 4 t 3668 3667',
'STATE 5 c 3670 3669',
'STATE 1 o 3672 3671',
'STATE 6 # 3455 3399',
'STATE 3 d 3674 3673',
'STATE 3 d 3676 3675',
'STATE 5 e 3678 3677',
'PHONE uw1',
'STATE 4 n 3680 3679',
'STATE 6 i 3444 3681',
'STATE 5 a 3444 3682',
'STATE 6 e 3444 3683',
'STATE 2 t 3493 3684',
'STATE 5 t 3399 3685',
'STATE 5 m 3686 3399',
'STATE 1 0 3687 3399',
'STATE 2 # 3524 3399',
'STATE 3 h 3399 3688',
'STATE 4 l 3455 3393',
'STATE 2 b 3690 3689',
'STATE 2 a 3399 3691',
'STATE 6 a 3385 3692',
'STATE 5 n 3694 3693',
'STATE 6 t 3385 3695',
'STATE 5 c 3444 3696',
'STATE 6 a 3399 3393',
'STATE 1 0 3697 3393',
'STATE 1 0 3699 3698',
'STATE 6 # 3385 3399',
'STATE 6 # 3399 3700',
'STATE 3 w 3600 3701',
'STATE 6 p 3494 3702',
'STATE 3 h 3703 3399',
'STATE 2 # 3705 3704',
'STATE 3 h 3399 3706',
'STATE 3 h 3708 3707',
'STATE 3 i 3710 3709',
'STATE 6 c 3494 3711',
'STATE 3 d 3444 3399',
'STATE 3 s 3522 3712',
'STATE 3 i 3399 3713',
'STATE 6 v 3399 3714',
'STATE 1 0 3385 3389',
'STATE 1 a 3399 3715',
'STATE 2 n 3385 3399',
'STATE 4 s 3717 3716',
'STATE 2 v 3437 3718',
'STATE 3 u 3720 3719',
'STATE 4 e 3399 3721',
'STATE 2 n 3722 3399',
'STATE 3 z 3522 3723',
'STATE 2 i 3522 3724',
'STATE 1 o 3522 3725',
'STATE 1 e 3455 3726',
'STATE 1 # 3399 3455',
'STATE 2 r 3728 3727',
'STATE 2 l 3393 3729',
'STATE 2 o 3455 3730',
'STATE 1 e 3522 3731',
'STATE 2 b 3399 3732',
'STATE 3 w 3600 3399',
'STATE 1 e 3444 3399',
'STATE 1 t 3399 3733',
'STATE 1 y 3399 3734',
'STATE 1 e 3399 3735',
'STATE 2 f 3444 3736',
'STATE 4 l 3738 3737',
'STATE 5 e 3740 3739',
'STATE 6 i 3385 3741',
'STATE 4 y 3399 3742',
'STATE 4 s 3744 3743',
'STATE 6 s 3746 3745',
'STATE 4 t 3748 3747',
'STATE 1 # 3385 3749',
'STATE 5 l 3751 3750',
'STATE 1 # 3753 3752',
'STATE 1 f 3385 3754',
'STATE 1 e 3455 3755',
'STATE 4 n 3757 3756',
'STATE 6 a 3455 3385',
'STATE 4 x 3759 3758',
'STATE 4 m 3761 3760',
'STATE 3 t 3762 3399',
'STATE 5 t 3762 3399',
'STATE 3 b 3455 3763',
'STATE 6 l 3764 3399',
'STATE 4 y 3524 3765',
'STATE 6 w 3455 3766',
'STATE 5 s 3399 3762',
'STATE 6 l 3385 3767',
'STATE 6 v 3494 3444',
'STATE 3 d 3399 3768',
'STATE 6 l 3399 3524',
'STATE 6 b 3393 3399',
'STATE 6 t 3444 3769',
'STATE 2 0 3524 3770',
'STATE 5 m 3393 3771',
'STATE 3 r 3399 3393',
'STATE 3 g 3399 3772',
'STATE 2 b 3385 3773',
'STATE 5 t 3393 3774',
'STATE 6 d 3444 3393',
'STATE 5 v 3385 3393',
'STATE 5 l 3776 3775',
'STATE 3 b 3393 3777',
'STATE 2 b 3385 3778',
'STATE 6 i 3385 3393',
'STATE 3 z 3780 3779',
'STATE 3 u 3762 3399',
'STATE 3 p 3399 3385',
'STATE 6 p 3385 3781',
'STATE 6 s 3494 3522',
'STATE 3 d 3494 3782',
'STATE 2 e 3399 3385',
'STATE 6 o 3385 3783',
'STATE 1 # 3785 3784',
'STATE 2 w 3385 3786',
'STATE 2 r 3399 3522',
'STATE 3 p 3399 3787',
'STATE 3 v 3399 3788',
'STATE 1 a 3790 3789',
'STATE 6 s 3399 3385',
'STATE 1 # 3792 3791',
'STATE 3 d 3794 3793',
'STATE 3 s 3455 3795',
'STATE 1 # 3600 3796',
'STATE 3 l 3455 3797',
'STATE 2 q 3524 3399',
'STATE 4 y 3399 3798',
'STATE 1 e 3522 3799',
'STATE 3 s 3522 3800',
'STATE 2 e 3522 3801',
'STATE 2 i 3522 3385',
'STATE 1 i 3455 3802',
'STATE 2 h 3455 3385',
'STATE 1 e 3455 3385',
'STATE 4 s 3803 3393',
'STATE 3 r 3385 3455',
'STATE 2 s 3522 3804',
'STATE 3 j 3399 3805',
'STATE 1 s 3806 3399',
'STATE 2 u 3399 3807',
'STATE 1 m 3399 3808',
'STATE 1 n 3444 3809',
'STATE 4 w 3811 3810',
'STATE 5 y 3813 3812',
'STATE 5 d 3815 3814',
'STATE 6 # 3817 3816',
'STATE 1 # 3819 3818',
'STATE 4 w 3600 3820',
'STATE 4 u 3822 3821',
'STATE 6 o 3824 3823',
'STATE 6 l 3455 3825',
'STATE 3 m 3455 3826',
'STATE 4 s 3828 3827',
'STATE 3 u 3399 3829',
'STATE 3 m 3455 3830',
'STATE 4 w 3832 3831',
'STATE 6 o 3385 3833',
'STATE 2 p 3399 3834',
'STATE 2 d 3835 3399',
'STATE 5 n 3385 3836',
'STATE 6 e 3455 3399',
'STATE 4 t 3837 3399',
'STATE 6 i 3455 3838',
'STATE 4 y 3840 3839',
'STATE 2 # 3385 3841',
'STATE 4 l 3843 3842',
'STATE 6 g 3385 3844',
'PHONE oy1',
'STATE 3 d 3600 3845',
'STATE 3 j 3600 3385',
'STATE 5 a 3847 3846',
'STATE 5 a 3455 3848',
'STATE 5 e 3494 3849',
'STATE 5 j 3524 3850',
'STATE 6 v 3393 3851',
'STATE 3 r 3852 3524',
'STATE 3 w 3854 3853',
'STATE 3 c 3399 3855',
'STATE 6 # 3856 3393',
'STATE 6 u 3858 3857',
'STATE 6 a 3444 3393',
'STATE 6 i 3393 3859',
'STATE 6 o 3385 3860',
'STATE 2 t 3385 3861',
'STATE 3 a 3385 3862',
'STATE 1 0 3389 3399',
'STATE 1 # 3385 3399',
'STATE 6 r 3399 3863',
'STATE 3 r 3393 3864',
'STATE 2 t 3399 3389',
'STATE 2 t 3385 3522',
'STATE 1 l 3399 3389',
'STATE 6 # 3389 3865',
'STATE 3 z 3389 3866',
'STATE 6 d 3399 3867',
'STATE 2 r 3385 3399',
'STATE 2 n 3399 3868',
'STATE 2 s 3385 3399',
'STATE 4 d 3870 3869',
'STATE 4 d 3455 3871',
'STATE 3 c 3455 3872',
'STATE 3 h 3874 3873',
'STATE 3 i 3876 3875',
'STATE 4 m 3455 3877',
'STATE 1 a 3399 3522',
'STATE 3 l 3879 3878',
'STATE 2 a 3522 3880',
'STATE 2 p 3455 3881',
'STATE 2 n 3393 3882',
'STATE 2 h 3522 3883',
'STATE 3 i 3399 3884',
'STATE 2 a 3444 3399',
'STATE 2 r 3399 3885',
'STATE 1 o 3399 3886',
'STATE 1 a 3444 3887',
'STATE 4 o 3889 3888',
'STATE 1 # 3891 3890',
'STATE 5 l 3893 3892',
'STATE 6 # 3399 3455',
'STATE 5 b 3455 3894',
'STATE 6 o 3896 3895',
'STATE 6 s 3399 3897',
'STATE 2 y 3393 3898',
'STATE 4 w 3900 3899',
'STATE 4 s 3902 3901',
'STATE 1 # 3904 3903',
'STATE 4 o 3444 3905',
'STATE 3 r 3762 3399',
'STATE 3 n 3455 3906',
'STATE 2 i 3399 3907',
'STATE 6 h 3730 3908',
'STATE 1 r 3385 3909',
'STATE 4 c 3385 3910',
'STATE 1 # 3385 3911',
'STATE 1 # 3385 3912',
'STATE 3 s 3914 3913',
'STATE 1 a 3576 3915',
'STATE 5 s 3917 3916',
'STATE 4 l 3790 3399',
'STATE 1 o 3455 3918',
'STATE 5 z 3399 3919',
'STATE 1 a 3920 3399',
'STATE 6 # 3399 3385',
'STATE 1 e 3455 3921',
'STATE 2 # 3923 3922',
'STATE 5 e 3399 3924',
'STATE 5 a 3494 3925',
'STATE 4 v 3927 3926',
'STATE 5 i 3929 3928',
'STATE 5 i 3931 3930',
'STATE 3 l 3933 3932',
'STATE 4 j 3494 3934',
'STATE 6 l 3936 3935',
'STATE 5 e 3393 3937',
'STATE 5 o 3494 3385',
'STATE 6 o 3399 3938',
'STATE 6 f 3393 3939',
'STATE 2 f 3524 3399',
'STATE 6 i 3941 3940',
'STATE 5 t 3385 3393',
'STATE 6 l 3393 3399',
'STATE 1 # 3393 3444',
'STATE 6 o 3943 3942',
'STATE 5 s 3385 3393',
'STATE 6 l 3444 3393',
'STATE 3 l 3944 3393',
'STATE 1 s 3385 3945',
'STATE 1 l 3947 3946',
'STATE 3 p 3399 3948',
'STATE 1 # 3399 3949',
'STATE 3 j 3385 3950',
'STATE 2 # 3568 3494',
'STATE 6 s 3952 3951',
'STATE 1 d 3399 3953',
'STATE 4 y 3955 3954',
'STATE 3 w 3957 3956',
'STATE 4 z 3522 3958',
'STATE 3 g 3455 3959',
'STATE 3 r 3961 3960',
'PHONE y-uw',
'STATE 2 o 3963 3962',
'STATE 2 l 3399 3455',
'STATE 2 t 3399 3964',
'STATE 1 z 3522 3965',
'STATE 2 a 3522 3966',
'STATE 2 u 3522 3385',
'STATE 3 p 3968 3967',
'STATE 2 t 3393 3444',
'STATE 1 w 3522 3969',
'STATE 2 i 3399 3970',
'STATE 2 n 3399 3971',
'STATE 1 r 3399 3972',
'STATE 2 b 3533 3973',
'STATE 6 # 3975 3974',
'STATE 3 g 3399 3976',
'STATE 6 # 3600 3977',
'STATE 5 e 3979 3978',
'STATE 5 d 3981 3980',
'STATE 6 # 3983 3982',
'STATE 6 # 3985 3984',
'STATE 3 t 3385 3986',
'STATE 3 u 3385 3987',
'STATE 6 d 3455 3988',
'STATE 2 l 3393 3989',
'STATE 4 s 3991 3990',
'STATE 2 s 3385 3992',
'STATE 3 o 3399 3993',
'STATE 3 r 3385 3994',
'STATE 3 n 3455 3995',
'STATE 4 n 3455 3996',
'STATE 3 o 3399 3997',
'STATE 2 n 3385 3998',
'STATE 3 l 3399 3999',
'STATE 1 # 3385 4000',
'STATE 3 c 3455 4001',
'STATE 1 # 3385 4002',
'STATE 3 u 3385 4003',
'STATE 3 g 4005 4004',
'STATE 3 v 3385 4006',
'STATE 1 r 3385 3455',
'STATE 5 c 3385 3399',
'STATE 2 v 4007 3399',
'STATE 2 n 3385 3437',
'STATE 2 t 3455 4008',
'STATE 5 e 3399 3455',
'STATE 5 a 3455 3399',
'STATE 2 t 3399 4009',
'STATE 4 n 4011 4010',
'STATE 4 o 4013 4012',
'STATE 3 k 4015 4014',
'STATE 5 i 4017 4016',
'STATE 4 c 4019 4018',
'STATE 6 a 3393 4020',
'STATE 5 u 4022 4021',
'STATE 6 c 3385 4023',
'STATE 6 r 4025 4024',
'STATE 6 l 3444 3385',
'STATE 3 s 3600 4026',
'STATE 5 a 4027 3600',
'STATE 5 e 4029 4028',
'STATE 6 i 3455 4030',
'STATE 4 c 3494 4031',
'STATE 6 i 3522 4032',
'STATE 6 u 3399 4033',
'STATE 6 s 3393 4034',
'STATE 6 t 4036 4035',
'STATE 2 c 3444 4037',
'STATE 5 f 4039 4038',
'STATE 5 l 3385 3393',
'STATE 6 e 3393 3385',
'STATE 2 d 3385 4040',
'STATE 5 y 4042 4041',
'STATE 5 t 4043 3399',
'STATE 6 l 4045 4044',
'STATE 6 r 3399 4046',
'STATE 6 a 3399 4047',
'STATE 2 p 3385 4048',
'STATE 1 0 3385 3399',
'STATE 3 f 3399 4049',
'STATE 4 e 3399 4050',
'STATE 3 k 4052 4051',
'STATE 1 # 3399 4053',
'STATE 2 e 3600 3399',
'STATE 4 a 3444 4054',
'STATE 3 h 4056 4055',
'STATE 1 e 3600 4027',
'STATE 2 b 4027 3600',
'STATE 2 c 4058 4057',
'STATE 3 n 3522 3455',
'STATE 2 s 4060 4059',
'STATE 1 p 3494 4061',
'STATE 2 o 3522 3494',
'STATE 3 g 3455 4062',
'STATE 2 a 3455 4063',
'STATE 1 r 3455 4064',
'STATE 1 a 3399 4065',
'STATE 1 o 3399 4066',
'STATE 1 c 3444 4067',
'STATE 1 i 3444 4068',
'STATE 1 # 4070 4069',
'STATE 5 e 4072 4071',
'STATE 1 # 4074 4073',
'STATE 2 c 3600 4075',
'STATE 3 r 3600 4076',
'STATE 6 l 3600 3399',
'STATE 6 # 4078 4077',
'STATE 3 f 4080 4079',
'STATE 3 o 3399 4081',
'STATE 3 w 4083 4082',
'STATE 1 # 4085 4084',
'STATE 5 y 3455 4086',
'STATE 2 l 4088 4087',
'STATE 3 l 3385 3455',
'STATE 3 g 4090 4089',
'STATE 2 i 3393 4091',
'STATE 6 k 3385 4092',
'STATE 6 l 4094 4093',
'STATE 3 l 3385 4095',
'STATE 4 u 3399 4096',
'STATE 6 a 3522 4097',
'STATE 4 n 3455 4098',
'STATE 3 r 3385 4099',
'STATE 3 u 4101 4100',
'STATE 1 # 4103 4102',
'STATE 2 o 3399 4104',
'STATE 2 i 3385 4105',
'STATE 1 i 3385 3455',
'STATE 4 d 3522 4106',
'STATE 1 d 3385 4107',
'STATE 3 k 4109 4108',
'STATE 2 d 3455 4110',
'STATE 3 r 3455 4111',
'STATE 5 e 3399 3437',
'STATE 1 q 3455 4112',
'STATE 2 c 4113 3596',
'STATE 6 c 4115 4114',
'STATE 5 g 4117 4116',
'STATE 5 t 4119 4118',
'STATE 5 r 3399 4120',
'STATE 5 r 3385 3524',
'STATE 6 e 3524 3393',
'STATE 6 c 3385 4121',
'STATE 6 s 3494 3385',
'STATE 4 g 4123 4122',
'STATE 5 e 3455 4124',
'STATE 6 r 3385 4125',
'STATE 6 o 4127 4126',
'STATE 6 c 3522 3455',
'STATE 6 s 3393 3455',
'STATE 5 e 3389 4128',
'STATE 5 e 3444 4129',
'STATE 3 j 3600 3437',
'PHONE uw',
'STATE 4 c 4131 4130',
'STATE 6 r 4133 4132',
'STATE 6 r 4135 4134',
'STATE 4 g 3494 3455',
'STATE 6 u 3389 3385',
'STATE 3 s 4136 3399',
'STATE 6 c 3444 3399',
'STATE 1 t 3385 4137',
'STATE 3 r 3444 4138',
'STATE 3 d 3393 4139',
'STATE 3 l 3393 4140',
'STATE 3 l 3393 3385',
'STATE 1 t 3385 3393',
'STATE 1 0 4142 4141',
'STATE 6 l 3385 3399',
'STATE 6 y 3399 4143',
'STATE 6 t 3385 4144',
'STATE 3 g 3522 3385',
'STATE 2 r 3389 3399',
'STATE 6 s 4146 4145',
'STATE 6 # 3385 4147',
'STATE 6 e 3399 3385',
'STATE 4 a 4149 4148',
'STATE 3 v 4151 4150',
'STATE 2 c 3399 4152',
'STATE 3 m 3399 4153',
'STATE 4 o 3444 4154',
'STATE 3 z 3455 4155',
'STATE 2 t 3399 3455',
'STATE 3 p 3455 4156',
'STATE 1 o 3524 3455',
'STATE 2 a 3385 4157',
'STATE 4 p 3385 3522',
'STATE 3 r 4159 4158',
'STATE 3 h 3455 4160',
'STATE 1 a 3455 3385',
'STATE 1 i 3522 3455',
'STATE 3 a 3399 4161',
'STATE 2 g 3399 3444',
'STATE 1 z 3444 4162',
'STATE 1 r 3882 4163',
'STATE 5 i 4165 4164',
'STATE 3 o 3399 4166',
'STATE 4 u 4168 4167',
'STATE 4 k 3494 4169',
'STATE 6 l 3393 4170',
'STATE 3 d 3444 4171',
'STATE 6 c 3399 4172',
'STATE 2 s 3600 4173',
'STATE 2 n 4175 4174',
'STATE 3 a 3399 4176',
'STATE 3 h 3385 4177',
'STATE 6 t 3522 4178',
'STATE 3 u 4049 4179',
'STATE 1 # 3385 4180',
'STATE 2 o 3522 4181',
'STATE 5 c 4183 4182',
'STATE 3 u 4185 4184',
'STATE 5 a 4187 4186',
'STATE 1 l 3455 4188',
'STATE 1 a 3385 3522',
'STATE 1 # 4190 4189',
'STATE 6 r 3385 3455',
'STATE 3 c 3393 4191',
'STATE 6 e 4193 4192',
'STATE 3 n 3455 4194',
'STATE 3 l 3455 3399',
'STATE 3 j 3385 4195',
'STATE 3 u 3399 4196',
'STATE 6 n 3399 3385',
'STATE 3 l 3455 4197',
'STATE 2 s 3385 4198',
'STATE 4 l 4200 4199',
'STATE 2 q 3385 4201',
'STATE 6 a 4203 4202',
'STATE 2 p 3385 4204',
'STATE 3 r 3385 4205',
'STATE 1 d 3385 4206',
'STATE 1 o 3522 4207',
'STATE 3 w 3385 4208',
'STATE 3 v 3455 4209',
'STATE 1 o 3455 3522',
'STATE 2 g 3522 3455',
'STATE 3 g 3455 4210',
'STATE 6 r 3455 4211',
'STATE 1 s 3455 3399',
'STATE 4 m 4213 4212',
'STATE 4 l 3455 4214',
'STATE 6 e 4216 4215',
'STATE 6 l 3389 4217',
'STATE 6 # 4219 4218',
'STATE 3 o 3399 4220',
'STATE 3 l 4222 4221',
'STATE 6 m 3494 4223',
'STATE 5 p 4225 4224',
'STATE 6 o 3385 3494',
'STATE 5 i 4227 4226',
'STATE 6 n 3385 4228',
'STATE 6 y 3455 4229',
'STATE 5 m 3522 3385',
'STATE 5 a 3389 4230',
'STATE 5 a 3385 3389',
'STATE 4 g 4232 4231',
'STATE 5 u 3494 4233',
'STATE 4 s 4235 4234',
'STATE 4 f 3385 4236',
'STATE 4 m 3393 4237',
'STATE 4 m 3444 3455',
'STATE 5 n 3524 3399',
'STATE 1 l 4239 4238',
'STATE 5 n 3399 3385',
'STATE 3 p 3393 4240',
'STATE 6 a 3393 4241',
'STATE 5 t 4243 4242',
'STATE 5 d 4245 4244',
'STATE 6 i 3399 3385',
'STATE 3 s 3399 4246',
'STATE 3 d 3385 4247',
'STATE 3 t 3399 3385',
'STATE 3 t 3792 4248',
'STATE 4 o 3444 4249',
'STATE 3 y 3399 4250',
'STATE 1 # 4251 3399',
'STATE 1 u 3524 3399',
'STATE 1 a 4252 3399',
'STATE 3 y 3399 4253',
'STATE 4 e 3399 4254',
'STATE 3 x 3455 4255',
'STATE 3 g 3455 4256',
'STATE 4 c 3385 4257',
'STATE 1 b 3522 4258',
'STATE 2 o 3494 3522',
'STATE 1 u 3455 4259',
'STATE 3 l 4260 3399',
'STATE 1 s 3444 4261',
'STATE 1 t 3393 4262',
'STATE 4 y 3399 4263',
'STATE 6 a 4265 4264',
'STATE 2 p 4267 4266',
'STATE 4 c 4269 4268',
'STATE 5 m 3393 4270',
'STATE 3 o 3399 4271',
'STATE 3 k 3399 4272',
'STATE 5 r 3393 4273',
'STATE 6 l 4274 3399',
'STATE 3 l 3600 3399',
'STATE 6 s 4276 4275',
'STATE 6 a 3455 4277',
'STATE 3 u 3399 4278',
'STATE 6 # 3522 4279',
'STATE 1 r 3522 3385',
'STATE 2 n 4281 4280',
'STATE 3 b 4283 4282',
'STATE 2 e 3522 3385',
'STATE 3 o 4285 4284',
'STATE 6 e 4287 4286',
'STATE 3 o 3399 4288',
'STATE 2 q 3385 3399',
'STATE 5 o 3385 4289',
'STATE 2 e 3385 4290',
'STATE 1 b 3522 4291',
'STATE 3 m 3385 4292',
'STATE 6 l 3385 4293',
'STATE 1 o 3393 4294',
'STATE 4 n 4296 4295',
'STATE 4 n 4298 4297',
'STATE 6 n 3455 4299',
'STATE 3 z 3385 3600',
'STATE 2 s 3385 4300',
'STATE 4 s 4302 4301',
'STATE 4 s 3385 3455',
'STATE 6 i 3385 4303',
'STATE 3 f 3522 4304',
'STATE 2 h 3399 4305',
'STATE 1 d 3385 4306',
'STATE 2 o 3399 4307',
'STATE 2 c 3385 4308',
'STATE 1 a 3385 3399',
'STATE 6 r 4310 4309',
'STATE 4 l 4311 3385',
'STATE 3 v 3455 4312',
'STATE 3 m 3389 4313',
'STATE 2 r 3385 4314',
'STATE 5 a 3399 3579',
'STATE 4 q 4316 4315',
'STATE 5 b 4318 4317',
'STATE 4 v 3494 4319',
'STATE 5 e 3385 4320',
'STATE 5 t 3385 4321',
'STATE 6 r 3455 4322',
'STATE 5 u 4324 4323',
'STATE 5 e 3393 4325',
'STATE 6 a 4327 4326',
'STATE 5 l 3444 4328',
'STATE 6 e 3522 4329',
'STATE 6 l 3494 4330',
'STATE 4 p 4332 4331',
'STATE 6 e 3385 3389',
'STATE 6 m 3393 4333',
'STATE 6 m 3385 3455',
'STATE 5 e 3494 4334',
'STATE 6 t 3455 4335',
'STATE 5 o 4336 3385',
'STATE 4 q 4338 4337',
'STATE 5 r 3494 4339',
'STATE 5 i 3455 4340',
'STATE 4 c 3455 4341',
'STATE 6 n 3455 4342',
'STATE 4 s 3385 4343',
'STATE 4 l 4345 4344',
'STATE 3 c 4347 4346',
'STATE 3 n 3444 3393',
'STATE 2 t 3393 4348',
'STATE 6 t 3943 3393',
'STATE 2 a 4349 3399',
'STATE 3 b 4350 3399',
'STATE 3 p 3399 4351',
'STATE 6 a 3385 3399',
'STATE 3 c 3399 4352',
'STATE 3 s 3385 4353',
'STATE 6 l 4355 4354',
'STATE 3 i 3399 4356',
'STATE 1 a 3444 4357',
'STATE 3 r 3524 3399',
'STATE 2 n 3393 3399',
'STATE 3 f 3399 4358',
'STATE 4 y 3399 4359',
'STATE 3 i 3399 4360',
'STATE 3 c 3455 4361',
'STATE 4 x 3385 4362',
'STATE 3 p 3494 4363',
'STATE 1 a 4365 4364',
'STATE 2 a 4366 3399',
'STATE 1 n 3444 4367',
'STATE 1 l 3444 4368',
'STATE 6 n 4370 4369',
'STATE 6 c 4372 4371',
'STATE 2 s 3385 4373',
'STATE 3 u 4375 4374',
'STATE 4 u 3399 4376',
'STATE 5 h 4378 4377',
'STATE 5 k 4379 3385',
'STATE 5 x 4380 3399',
'STATE 4 s 4382 4381',
'STATE 5 l 3444 4383',
'STATE 5 d 3393 4384',
'STATE 3 n 3455 3399',
'STATE 5 i 4386 4385',
'STATE 5 e 3399 4387',
'STATE 3 g 4389 4388',
'STATE 5 e 4391 4390',
'STATE 6 e 3385 3455',
'STATE 6 e 4393 4392',
'STATE 6 e 3455 4394',
'STATE 2 r 4396 4395',
'STATE 1 a 3522 3385',
'STATE 5 z 4398 4397',
'STATE 1 b 3393 3399',
'STATE 6 y 3455 4399',
'STATE 1 o 3385 3455',
'STATE 2 e 3455 4400',
'STATE 1 # 3385 4401',
'STATE 3 l 3524 4402',
'STATE 2 e 3385 4403',
'STATE 1 e 3455 4404',
'STATE 3 h 3522 3459',
'STATE 1 a 3393 4405',
'STATE 4 l 3455 4406',
'STATE 1 i 4408 4407',
'STATE 4 l 3455 3385',
'STATE 1 l 3455 4409',
'STATE 1 n 3455 4410',
'STATE 4 t 3385 4411',
'STATE 4 u 3444 4412',
'STATE 3 s 3385 4413',
'STATE 2 d 3385 4414',
'STATE 6 i 3455 3385',
'STATE 6 e 3385 3399',
'STATE 2 i 3399 4415',
'STATE 2 i 3522 4359',
'STATE 2 q 3385 4416',
'STATE 2 n 4418 4417',
'STATE 3 c 4419 3455',
'STATE 3 f 3385 3522',
'STATE 3 r 4421 4420',
'STATE 2 b 3522 4422',
'STATE 2 n 4424 4423',
'STATE 4 v 4426 4425',
'STATE 6 a 3393 3494',
'STATE 5 a 4428 4427',
'STATE 6 l 3385 3522',
'STATE 4 d 3385 4429',
'STATE 5 u 3455 4430',
'STATE 5 d 3522 4431',
'STATE 6 a 3522 3385',
'STATE 5 i 4433 4432',
'STATE 4 g 4435 4434',
'STATE 5 a 4437 4436',
'STATE 3 b 3385 4438',
'STATE 3 t 3385 4439',
'STATE 6 a 3385 3393',
'STATE 6 o 3393 4440',
'STATE 5 p 4442 4441',
'STATE 4 f 3455 4443',
'STATE 5 r 4445 4444',
'STATE 5 h 3385 4446',
'STATE 6 l 4448 4447',
'STATE 6 r 4450 4449',
'STATE 6 n 3385 3389',
'STATE 4 f 4452 4451',
'STATE 6 i 3385 3494',
'STATE 6 o 3393 3385',
'STATE 6 n 3444 4453',
'STATE 6 a 4455 4454',
'STATE 6 m 3455 3444',
'STATE 4 p 3385 3444',
'STATE 6 # 3393 4456',
'STATE 6 t 3385 3455',
'STATE 3 h 3393 4457',
'STATE 1 # 3399 3393',
'STATE 5 l 3393 4458',
'STATE 1 c 3399 4459',
'STATE 6 a 3385 4460',
'STATE 6 a 4461 3399',
'STATE 3 g 3399 4097',
'STATE 3 c 3385 4462',
'STATE 3 p 4464 4463',
'STATE 3 p 3385 3399',
'STATE 4 i 3524 4465',
'STATE 2 d 3444 4466',
'STATE 3 b 4464 4467',
'STATE 2 a 3385 3455',
'STATE 2 e 3653 4468',
'STATE 1 b 3455 4469',
'STATE 4 z 3522 3385',
'STATE 1 h 3494 4470',
'STATE 1 # 4359 4471',
'STATE 3 c 3385 3455',
'STATE 1 d 3399 4472',
'STATE 1 p 3444 3399',
'STATE 1 y 3444 4473',
'STATE 5 k 4475 4474',
'STATE 4 m 4477 4476',
'STATE 6 o 4479 4478',
'STATE 4 t 3385 4480',
'STATE 2 e 4481 3393',
'STATE 4 u 3399 4482',
'STATE 2 q 4483 3399',
'STATE 4 v 4485 4484',
'STATE 5 y 4487 4486',
'STATE 4 s 3385 4488',
'STATE 3 b 4490 4489',
'PHONE ow1',
'STATE 4 t 4492 4491',
'STATE 2 a 4494 4493',
'STATE 5 w 3399 4495',
'STATE 2 c 3393 3882',
'STATE 6 n 4497 4496',
'STATE 6 a 4499 4498',
'STATE 3 h 3385 4500',
'STATE 5 e 3455 4501',
'STATE 6 c 3385 4502',
'STATE 5 a 4503 3385',
'STATE 2 e 3455 4504',
'STATE 2 s 4506 4505',
'STATE 3 c 3455 4507',
'STATE 6 y 3455 4508',
'STATE 2 a 4510 4509',
'STATE 3 d 3385 4511',
'STATE 5 i 4513 4512',
'STATE 6 o 3385 4514',
'STATE 2 e 3455 4515',
'STATE 3 c 3455 4516',
'STATE 5 z 3522 3385',
'STATE 3 r 3385 4517',
'STATE 3 s 3455 4518',
'STATE 1 a 3455 4519',
'STATE 2 a 3393 4520',
'STATE 6 o 3455 3399',
'STATE 6 u 3385 3455',
'STATE 2 s 3385 3455',
'STATE 2 n 3385 4521',
'STATE 2 t 4523 4522',
'STATE 3 r 3385 4297',
'STATE 3 k 3455 4524',
'STATE 3 c 3385 4525',
'STATE 1 v 4527 4526',
'STATE 3 v 4529 4528',
'STATE 3 r 3385 3399',
'STATE 2 t 3385 4530',
'STATE 6 o 3455 3385',
'STATE 1 o 3455 3385',
'STATE 3 f 3385 3455',
'STATE 1 e 3389 4531',
'STATE 3 r 3522 4532',
'STATE 1 a 3455 4533',
'STATE 3 t 3385 3455',
'STATE 4 c 4535 4534',
'STATE 6 r 3385 4536',
'STATE 6 l 4538 4537',
'STATE 6 n 3385 3393',
'STATE 4 p 3385 3494',
'STATE 5 i 3385 4539',
'STATE 5 v 3522 3385',
'STATE 5 a 4541 4540',
'STATE 6 o 3393 4542',
'STATE 4 c 4544 4543',
'STATE 3 b 3494 4545',
'STATE 5 o 4546 3385',
'STATE 4 n 3393 4547',
'STATE 6 u 4548 3385',
'STATE 3 p 3385 4549',
'STATE 5 n 4322 3385',
'STATE 5 h 4551 4550',
'STATE 6 a 3494 4552',
'STATE 6 r 4554 4553',
'STATE 5 a 3455 4555',
'STATE 6 e 3455 4556',
'STATE 5 k 3385 4557',
'STATE 5 a 3385 3494',
'STATE 5 a 3494 3385',
'STATE 5 g 3522 4558',
'STATE 5 o 3455 3522',
'STATE 5 u 4560 4559',
'STATE 5 r 3455 4561',
'STATE 6 a 3444 4562',
'STATE 4 b 3455 4563',
'STATE 4 p 3455 3444',
'STATE 6 b 3393 4564',
'STATE 5 k 4566 4565',
'STATE 1 # 3393 4567',
'STATE 3 l 3399 4568',
'STATE 6 o 4569 3399',
'STATE 5 s 3399 4570',
'STATE 6 t 3385 4571',
'STATE 1 0 3385 4572',
'STATE 1 e 3385 3399',
'STATE 4 z 4574 4573',
'STATE 1 c 3399 4575',
'STATE 2 s 3399 4576',
'STATE 3 j 3455 4577',
'STATE 3 r 4579 4578',
'STATE 2 a 3494 4580',
'STATE 3 k 3455 4581',
'STATE 1 n 3444 4582',
'STATE 2 h 4584 4583',
'STATE 2 e 4586 4585',
'STATE 6 e 4587 3385',
'STATE 5 e 4589 4588',
'STATE 2 p 3455 4590',
'STATE 4 d 4592 4591',
'STATE 2 i 3444 4593',
'STATE 4 v 3494 4594',
'STATE 4 d 3393 3455',
'STATE 4 y 4596 4595',
'STATE 4 b 3455 4597',
'STATE 6 m 3455 4598',
'STATE 5 i 3393 3494',
'STATE 3 o 3399 4599',
'STATE 3 l 3399 4600',
'STATE 1 # 3393 4601',
'STATE 2 l 3494 4602',
'STATE 2 e 3522 4603',
'STATE 4 y 3399 4604',
'STATE 3 l 3393 4605',
'STATE 1 a 3524 4606',
'STATE 3 n 3524 3393',
'STATE 3 c 3399 4607',
'STATE 3 o 4609 4608',
'STATE 2 i 3399 4610',
'STATE 2 i 4612 4611',
'STATE 2 o 3385 3393',
'STATE 1 # 3399 4613',
'STATE 5 a 3455 4614',
'STATE 5 o 3522 4615',
'STATE 2 r 3385 4616',
'STATE 2 a 3455 4001',
'STATE 6 s 3385 4617',
'STATE 1 # 3385 4618',
'STATE 3 b 3455 4619',
'STATE 3 c 3455 3385',
'STATE 1 i 3455 4620',
'STATE 3 v 3455 3385',
'STATE 3 r 4622 4621',
'STATE 5 n 4624 4623',
'STATE 6 n 3455 4625',
'STATE 1 l 3522 4626',
'STATE 3 r 3385 4627',
'STATE 2 o 3455 4628',
'STATE 3 h 3455 3393',
'STATE 2 r 3385 4629',
'STATE 6 r 4631 4630',
'STATE 1 e 3393 4632',
'STATE 3 p 3385 4420',
'STATE 3 c 4178 4633',
'STATE 1 s 3385 3455',
'STATE 4 l 4635 4634',
'STATE 3 r 4636 3455',
'STATE 2 s 4638 4637',
'STATE 4 t 3385 3494',
'STATE 1 f 3455 4639',
'STATE 2 l 3385 3455',
'STATE 1 s 3455 4640',
'STATE 2 e 3455 4641',
'STATE 2 l 3522 4642',
'STATE 1 r 4643 3455',
'STATE 4 g 4645 4644',
'STATE 5 o 3494 4646',
'STATE 5 e 4648 4647',
'STATE 5 e 3385 4649',
'STATE 5 p 3522 4650',
'STATE 5 c 4652 4651',
'STATE 5 e 4654 4653',
'STATE 3 m 4656 4655',
'STATE 6 a 4658 4657',
'STATE 4 q 4660 4659',
'STATE 6 r 3494 3385',
'PHONE ey',
'STATE 4 n 3524 3385',
'STATE 3 l 3393 4661',
'STATE 4 n 4662 3385',
'STATE 3 c 3522 4663',
'STATE 6 a 3385 4664',
'STATE 6 i 3494 4665',
'STATE 6 r 3494 4338',
'STATE 5 t 4667 4666',
'STATE 4 t 3455 4668',
'STATE 5 o 3455 4669',
'STATE 6 i 3455 3444',
'STATE 6 d 3494 4670',
'STATE 5 a 3385 4671',
'STATE 5 o 4673 4672',
'STATE 6 s 3393 4674',
'STATE 6 s 3455 4675',
'STATE 5 y 3444 4676',
'STATE 4 p 3455 4677',
'STATE 6 p 3393 4678',
'STATE 5 v 4680 4679',
'STATE 3 p 3393 4681',
'STATE 5 t 3444 3393',
'STATE 3 v 4245 3399',
'STATE 2 m 3399 3385',
'STATE 5 v 3399 4682',
'STATE 6 n 4683 3385',
'STATE 3 m 3385 4684',
'STATE 4 m 4686 4685',
'STATE 1 # 3385 4687',
'STATE 2 a 3393 4688',
'STATE 3 r 4689 3399',
'STATE 3 r 4691 4690',
'STATE 2 r 4693 4692',
'STATE 2 r 3455 3385',
'STATE 1 o 3494 4694',
'STATE 1 k 3455 4695',
'STATE 1 i 3444 3399',
'STATE 1 o 3444 4696',
'STATE 1 s 3444 4697',
'STATE 2 n 4699 4698',
'STATE 3 l 4701 4700',
'STATE 3 n 4703 4702',
'STATE 4 c 4705 4704',
'STATE 2 p 3455 4706',
'STATE 2 c 4708 4707',
'STATE 2 i 3399 4709',
'STATE 6 t 3728 4710',
'STATE 1 s 3385 3393',
'STATE 3 n 3455 4711',
'STATE 3 y 3399 4712',
'STATE 5 e 3399 3524',
'STATE 4 s 4545 3399',
'STATE 5 i 4714 4713',
'STATE 3 u 4716 4715',
'STATE 1 s 3399 3455',
'STATE 2 a 3455 3522',
'STATE 1 # 3385 4717',
'STATE 2 r 3522 3385',
'STATE 3 u 3399 3393',
'STATE 2 c 3393 4718',
'STATE 1 u 3524 3393',
'STATE 2 c 4720 4719',
'STATE 6 p 3385 4721',
'STATE 5 e 3455 3399',
'STATE 3 v 3399 4722',
'STATE 6 o 3393 4723',
'STATE 1 l 3399 4724',
'STATE 3 n 3455 3385',
'STATE 6 e 3455 4725',
'STATE 6 s 3455 4726',
'STATE 2 a 3385 4727',
'STATE 6 y 3455 3385',
'STATE 6 i 3385 4728',
'STATE 1 # 3385 4729',
'STATE 1 k 3455 4730',
'STATE 3 t 3524 3455',
'STATE 1 u 3455 4731',
'STATE 5 k 4733 4732',
'STATE 6 e 3475 3385',
'STATE 6 c 3385 4734',
'STATE 6 i 3385 4735',
'STATE 1 e 3455 4736',
'STATE 5 o 3455 4737',
'STATE 2 h 3522 4738',
'STATE 3 o 3399 4739',
'STATE 1 i 3455 3444',
'STATE 3 l 3393 4740',
'STATE 3 r 3385 4741',
'STATE 4 m 3455 4742',
'STATE 3 t 3455 4743',
'STATE 2 t 3455 4744',
'STATE 6 s 4746 4745',
'STATE 3 t 4747 3385',
'STATE 1 h 3399 4748',
'STATE 3 v 3385 4749',
'STATE 2 o 3455 3385',
'STATE 1 o 3522 4750',
'STATE 3 d 3455 3385',
'STATE 5 a 4297 4751',
'STATE 5 g 3385 4752',
'STATE 5 l 3494 4753',
'STATE 5 i 4755 4754',
'STATE 6 n 3393 3385',
'STATE 5 o 3389 4756',
'STATE 5 u 3385 4757',
'STATE 5 t 4759 4758',
'STATE 6 y 3455 3522',
'STATE 5 o 4761 4760',
'STATE 6 r 4763 4762',
'STATE 4 l 4765 4764',
'STATE 6 r 3455 4766',
'STATE 6 u 3393 4767',
'STATE 4 d 3393 4768',
'STATE 6 c 3455 4769',
'STATE 3 s 3393 4435',
'STATE 4 d 3524 4770',
'STATE 3 v 3522 3385',
'STATE 4 s 3385 4771',
'STATE 6 o 4773 4772',
'STATE 6 a 3494 3522',
'STATE 4 s 4775 4774',
'STATE 6 a 3444 3385',
'STATE 4 s 3455 4776',
'STATE 5 u 3385 3455',
'STATE 5 l 3494 4777',
'STATE 5 l 3385 4778',
'STATE 5 r 4780 4779',
'STATE 6 s 3393 4781',
'STATE 6 b 3444 4782',
'STATE 5 i 4784 4783',
'STATE 6 u 4786 4785',
'STATE 6 e 3455 4787',
'STATE 4 p 4789 4788',
'STATE 2 c 4791 4790',
'STATE 1 e 3385 3393',
'STATE 3 r 3393 4792',
'STATE 3 v 3399 4793',
'STATE 3 b 3385 3389',
'STATE 3 h 3385 4794',
'STATE 4 u 3399 4795',
'STATE 1 # 3385 3455',
'STATE 1 i 3522 4796',
'STATE 1 # 3393 4797',
'STATE 2 f 3455 4798',
'STATE 2 l 3399 4799',
'STATE 2 a 3399 4800',
'STATE 1 e 3455 4801',
'STATE 3 n 3385 3522',
'STATE 2 i 3494 4802',
'STATE 2 u 4803 3455',
'STATE 1 s 4804 3393',
'STATE 1 c 3444 3393',
'STATE 4 t 4806 4805',
'STATE 5 r 3455 4807',
'STATE 5 e 4809 4808',
'STATE 6 i 3385 3455',
'STATE 3 d 3494 3385',
'STATE 1 e 3494 3522',
'STATE 4 t 4811 4810',
'STATE 2 o 3385 4812',
'STATE 4 s 3444 3455',
'STATE 2 e 4813 3399',
'STATE 3 r 3455 3399',
'STATE 2 a 4815 4814',
'STATE 1 n 3399 4816',
'STATE 4 m 3385 4817',
'STATE 4 x 4819 4818',
'STATE 4 c 4821 4820',
'STATE 6 e 3522 4822',
'STATE 5 z 4824 4823',
'STATE 5 a 3385 3399',
'STATE 1 e 3385 4825',
'STATE 3 r 3444 3393',
'STATE 6 t 3399 4826',
'STATE 1 n 3399 3444',
'STATE 2 s 4686 4827',
'STATE 1 # 3385 4828',
'STATE 1 f 3399 4829',
'STATE 6 n 3399 4830',
'STATE 6 n 3455 4831',
'STATE 5 i 3522 4832',
'STATE 2 n 3385 3455',
'STATE 1 a 4833 3385',
'STATE 3 v 4803 3385',
'STATE 2 o 3455 4834',
'STATE 1 o 3455 4835',
'STATE 5 g 4837 4836',
'STATE 1 o 3455 4838',
'STATE 6 e 3393 4839',
'STATE 3 l 3522 3385',
'STATE 1 n 3455 4840',
'STATE 2 a 4701 4841',
'STATE 2 a 4842 3385',
'STATE 1 o 3455 4843',
'STATE 3 r 3393 3444',
'STATE 3 s 3385 4844',
'STATE 4 t 4846 4845',
'STATE 3 p 3385 3455',
'STATE 2 p 3385 4847',
'STATE 1 t 3455 3385',
'STATE 4 t 3455 3385',
'STATE 6 e 3522 3385',
'STATE 6 r 4849 4848',
'STATE 1 a 4851 4850',
'STATE 2 p 3522 4852',
'STATE 4 s 4854 4853',
'STATE 5 l 3385 4855',
'STATE 5 c 3494 3385',
'STATE 6 n 3385 4856',
'STATE 6 l 3393 3385',
'STATE 6 o 3389 4857',
'STATE 5 i 3522 3385',
'STATE 5 d 4859 4858',
'STATE 6 a 3522 4860',
'STATE 5 r 4862 4861',
'STATE 4 g 4274 4863',
'STATE 3 t 4865 4864',
'STATE 3 g 3385 4866',
'STATE 3 p 3385 4867',
'STATE 6 n 3385 4868',
'STATE 4 g 3385 4869',
'STATE 3 b 4871 4870',
'STATE 4 n 3393 3385',
'STATE 3 s 3385 4872',
'STATE 3 m 3385 4873',
'STATE 4 t 3385 3522',
'STATE 6 r 4773 4874',
'STATE 5 t 3494 3385',
'STATE 5 u 4876 4875',
'STATE 5 e 3522 4877',
'STATE 5 a 3455 3385',
'STATE 5 r 3494 4878',
'STATE 5 o 3385 4879',
'STATE 4 d 4881 4880',
'STATE 6 o 4883 4882',
'STATE 4 s 4885 4884',
'STATE 4 d 3455 4886',
'STATE 5 l 3455 4887',
'STATE 6 n 3455 3444',
'STATE 6 v 3494 4888',
'STATE 5 o 3494 3455',
'STATE 6 m 3455 4889',
'STATE 6 c 3455 4890',
'STATE 6 c 3393 3444',
'STATE 2 r 4892 4891',
'STATE 5 t 3444 4893',
'STATE 3 n 3393 4894',
'STATE 3 h 3399 4895',
'STATE 1 # 3385 4896',
'STATE 3 o 3399 4897',
'STATE 3 u 3385 3522',
'STATE 3 n 3444 4898',
'STATE 2 d 3455 3399',
'STATE 1 i 3399 4899',
'STATE 2 d 3524 3399',
'STATE 1 i 3385 4900',
'STATE 2 u 3494 4901',
'STATE 1 r 3455 3385',
'STATE 2 l 3393 3444',
'STATE 4 d 4903 4902',
'STATE 5 e 4905 4904',
'STATE 3 t 4907 4906',
'STATE 4 u 3399 4908',
'STATE 4 v 3494 4909',
'STATE 3 t 4911 4910',
'STATE 5 a 3522 3399',
'STATE 5 o 4913 4912',
'STATE 5 a 3399 3455',
'STATE 6 l 4915 4914',
'STATE 3 t 3455 3399',
'STATE 2 m 3385 4916',
'STATE 3 m 3385 4917',
'STATE 5 e 4919 4918',
'STATE 3 n 3494 4776',
'STATE 3 r 4920 3385',
'STATE 5 e 3455 4921',
'STATE 6 c 3455 4922',
'STATE 4 d 4924 4923',
'STATE 1 # 3385 3522',
'STATE 2 h 3385 4925',
'STATE 5 n 3444 4926',
'STATE 6 h 3385 4927',
'STATE 5 a 4928 3455',
'STATE 6 c 4930 4929',
'STATE 3 d 3385 3399',
'STATE 1 i 3385 4931',
'STATE 1 e 3522 3455',
'STATE 6 a 3385 3455',
'STATE 2 e 3455 4932',
'STATE 1 e 3455 3524',
'STATE 5 u 4934 4933',
'STATE 6 t 3385 4935',
'STATE 6 e 3385 4936',
'STATE 6 a 3393 4937',
'STATE 2 i 3455 4938',
'STATE 5 a 3455 4939',
'STATE 3 m 3385 4686',
'STATE 3 v 3455 3399',
'STATE 3 l 3385 4420',
'STATE 4 d 4930 4343',
'STATE 3 s 3385 3455',
'STATE 1 n 3455 4940',
'STATE 1 c 3522 4941',
'STATE 1 e 3385 4942',
'STATE 1 l 4944 4943',
'STATE 2 c 3385 4945',
'STATE 1 e 3522 4946',
'STATE 4 l 4948 4947',
'STATE 6 a 4950 4949',
'STATE 6 e 3385 4951',
'STATE 5 a 3494 4952',
'STATE 6 s 3385 4953',
'STATE 5 o 3455 4954',
'STATE 6 a 3522 4955',
'STATE 6 h 3522 4956',
'STATE 3 o 3399 4957',
'STATE 4 t 4959 4958',
'STATE 4 c 4961 4960',
'STATE 4 v 4963 4962',
'STATE 4 l 3385 4964',
'STATE 3 p 4966 4965',
'STATE 4 c 3494 4967',
'STATE 3 v 3522 4968',
'STATE 6 n 3385 4969',
'STATE 4 v 3385 4970',
'STATE 6 e 3455 4971',
'STATE 3 l 3522 4972',
'STATE 4 t 3385 3524',
'STATE 5 t 3494 4973',
'STATE 5 a 4975 4974',
'STATE 4 b 3385 3455',
'STATE 6 e 3455 4976',
'STATE 5 a 3494 4977',
'STATE 5 e 3385 4978',
'STATE 4 m 4881 4979',
'STATE 5 i 4980 3385',
'STATE 4 p 4981 3444',
'STATE 4 t 3385 4982',
'STATE 4 b 3444 4983',
'STATE 6 u 3444 4984',
'STATE 6 m 3455 4985',
'STATE 6 r 3455 4986',
'STATE 6 r 3455 4987',
'STATE 4 l 4989 4988',
'STATE 4 c 3444 4990',
'STATE 2 l 3399 4991',
'STATE 5 l 3393 3444',
'STATE 1 a 3399 4992',
'STATE 6 # 3399 3393',
'STATE 3 g 3399 4993',
'STATE 6 c 3399 3385',
'STATE 3 u 3399 4994',
'STATE 2 r 3393 4995',
'STATE 2 n 3399 4996',
'STATE 2 m 3455 4997',
'STATE 1 r 3494 4998',
'STATE 5 c 3385 4999',
'STATE 5 g 5001 5000',
'STATE 6 r 3455 5002',
'STATE 3 m 3455 5003',
'STATE 4 b 4274 5004',
'STATE 5 p 3385 5005',
'STATE 3 n 5007 5006',
'STATE 6 r 3444 5008',
'STATE 2 e 3455 5009',
'STATE 5 a 3522 3455',
'STATE 5 h 3455 3399',
'STATE 1 c 3385 3455',
'STATE 4 f 5011 5010',
'STATE 2 e 3385 4274',
'STATE 3 p 3393 5012',
'STATE 3 d 3385 5013',
'STATE 2 a 5015 5014',
'STATE 4 s 3399 5016',
'STATE 6 r 5018 5017',
'STATE 6 n 3393 5019',
'STATE 4 s 3455 5020',
'STATE 4 j 3524 5021',
'STATE 5 i 3385 5022',
'STATE 3 n 3385 5023',
'STATE 1 g 3399 3444',
'STATE 6 d 3455 5024',
'STATE 2 o 3399 3455',
'STATE 1 # 5026 5025',
'STATE 3 r 3455 3385',
'STATE 1 e 3385 5027',
'STATE 1 r 3455 5028',
'STATE 5 o 3455 5029',
'STATE 3 t 3385 4727',
'STATE 1 e 3494 5030',
'STATE 2 o 3385 5027',
'STATE 2 u 3455 5031',
'STATE 1 i 3455 3385',
'STATE 2 u 3522 5032',
'STATE 2 g 3385 5033',
'STATE 2 l 3399 5034',
'STATE 2 c 3455 5035',
'STATE 2 m 3522 5036',
'STATE 3 m 3385 3522',
'STATE 6 i 4579 5037',
'STATE 3 h 3522 5038',
'STATE 5 f 5037 5039',
'STATE 5 i 5041 5040',
'STATE 5 c 5043 5042',
'STATE 5 t 3455 5044',
'STATE 5 e 3385 3393',
'STATE 6 l 3494 3385',
'STATE 6 g 3385 5045',
'STATE 5 a 3522 5046',
'STATE 6 o 3522 3385',
'STATE 6 i 3522 5047',
'STATE 6 a 5049 5048',
'STATE 4 h 5051 5050',
'STATE 3 b 3455 5052',
'STATE 3 h 5054 5053',
'STATE 6 m 3494 3385',
'STATE 4 g 5056 5055',
'STATE 6 n 3385 5057',
'STATE 4 n 3385 3455',
'STATE 4 d 3385 5058',
'STATE 4 t 3393 3385',
'STATE 6 l 5060 5059',
'STATE 6 g 3455 5061',
'STATE 6 m 3385 5062',
'STATE 4 t 5064 5063',
'STATE 6 n 3385 5065',
'STATE 6 s 3455 5066',
'STATE 6 h 3494 5067',
'STATE 5 e 5069 5068',
'STATE 6 t 3455 5070',
'STATE 5 i 5071 4776',
'STATE 6 n 3444 3385',
'STATE 5 b 3522 3385',
'STATE 5 i 5073 5072',
'STATE 6 s 3444 3385',
'STATE 6 e 3385 3444',
'STATE 4 p 3444 3385',
'STATE 4 p 3455 5074',
'STATE 6 l 3444 3455',
'STATE 6 i 3444 5075',
'STATE 5 u 3455 3385',
'STATE 6 i 5076 3385',
'STATE 4 m 3385 5077',
'STATE 6 n 3455 3385',
'STATE 6 t 3393 3444',
'STATE 3 n 3399 5078',
'STATE 6 s 3385 3393',
'STATE 3 b 3399 5079',
'STATE 4 c 5081 5080',
'STATE 1 n 3444 5082',
'STATE 2 a 5083 3399',
'STATE 2 a 3385 5084',
'STATE 3 b 3494 3522',
'STATE 6 t 5086 5085',
'STATE 6 y 3455 5087',
'STATE 1 o 3494 3385',
'STATE 3 k 3455 5088',
'STATE 3 k 3455 5089',
'STATE 3 d 5091 5090',
'STATE 1 o 3522 3455',
'STATE 1 r 5093 5092',
'STATE 6 t 3494 5094',
'STATE 6 d 3393 3455',
'STATE 4 s 5096 5095',
'STATE 3 s 3385 5097',
'STATE 2 e 3455 5098',
'STATE 3 c 3393 3455',
'STATE 3 l 3455 3393',
'STATE 2 i 5100 5099',
'STATE 5 o 3399 5101',
'STATE 6 s 3393 5102',
'STATE 6 n 5104 5103',
'STATE 4 p 3444 5105',
'STATE 6 u 3494 3444',
'STATE 4 c 3455 5106',
'STATE 5 a 5108 5107',
'STATE 5 o 3524 5109',
'STATE 2 c 3385 3494',
'STATE 5 e 5111 5110',
'STATE 6 n 5113 5112',
'STATE 3 m 3522 5114',
'STATE 6 o 3385 3455',
'STATE 3 s 3455 5115',
'STATE 3 u 3399 5116',
'STATE 6 e 3455 5117',
'STATE 2 l 3455 5118',
'STATE 5 h 3455 3385',
'STATE 2 d 3385 3455',
'STATE 2 q 3385 5119',
'STATE 2 e 3455 3399',
'STATE 2 a 5121 5120',
'STATE 6 e 3455 3385',
'STATE 1 u 3455 5122',
'STATE 6 l 3385 5123',
'STATE 5 u 3455 5124',
'STATE 6 a 3455 5125',
'STATE 6 o 3385 5126',
'STATE 6 h 3385 5127',
'STATE 5 c 3385 5128',
'STATE 5 m 3385 5129',
'STATE 6 o 5131 5130',
'STATE 6 r 3522 4955',
'STATE 4 n 5133 5132',
'STATE 3 s 5134 3385',
'STATE 4 d 5136 5135',
'STATE 6 a 3524 3385',
'STATE 6 o 3385 5137',
'STATE 6 s 5139 5138',
'STATE 6 r 3385 5140',
'STATE 6 o 3393 5141',
'STATE 3 b 3494 3385',
'STATE 6 l 3385 5142',
'STATE 3 f 3393 5143',
'STATE 4 p 3385 5144',
'STATE 4 g 3393 5145',
'STATE 3 b 3455 4045',
'STATE 6 c 3385 5146',
'STATE 4 g 5064 5147',
'STATE 6 t 3455 3385',
'STATE 6 s 3385 5148',
'STATE 6 d 3455 3385',
'STATE 5 c 3494 5149',
'STATE 5 o 5151 5150',
'STATE 4 t 3455 5152',
'STATE 4 t 3455 5153',
'STATE 6 g 3455 3385',
'STATE 4 s 5155 5154',
'STATE 4 s 5157 5156',
'STATE 4 m 3455 5158',
'STATE 4 p 5160 5159',
'STATE 5 t 3385 3494',
'STATE 6 p 3393 5161',
'STATE 1 r 3393 5162',
'STATE 3 f 3399 5163',
'STATE 2 r 3385 5164',
'STATE 2 a 3494 4824',
'STATE 2 o 3444 5165',
'STATE 3 l 5166 3399',
'STATE 2 i 3963 5167',
'STATE 4 g 5169 5168',
'STATE 5 h 5171 5170',
'STATE 1 s 3385 5172',
'STATE 6 e 5174 5173',
'STATE 2 p 5176 5175',
'STATE 1 u 5178 5177',
'STATE 1 a 3455 4701',
'STATE 3 g 3455 5179',
'STATE 4 m 3385 5180',
'STATE 5 r 3455 4776',
'STATE 4 b 3399 5181',
'STATE 3 l 3399 3455',
'STATE 3 r 5183 5182',
'STATE 6 n 3399 5096',
'STATE 4 q 5185 5184',
'STATE 3 n 5186 5113',
'STATE 3 r 3522 5187',
'STATE 6 r 5189 5188',
'STATE 4 p 5191 5190',
'STATE 4 s 3385 3444',
'STATE 4 f 3385 5192',
'STATE 4 d 3393 4992',
'STATE 5 o 5194 5193',
'STATE 4 m 5196 5195',
'STATE 3 r 3385 5197',
'STATE 5 o 3455 5198',
'STATE 3 c 3385 5199',
'STATE 2 a 3455 3579',
'STATE 3 c 3399 3455',
'STATE 3 r 3455 5200',
'STATE 2 n 3455 5201',
'STATE 5 h 3455 5202',
'STATE 3 s 3455 5203',
'STATE 6 t 3385 5204',
'STATE 3 t 3385 5205',
'STATE 6 a 5207 5206',
'STATE 6 i 5209 5208',
'STATE 1 r 3455 3522',
'STATE 5 h 3385 5210',
'STATE 5 o 3455 5211',
'STATE 6 m 3455 5212',
'STATE 5 s 3385 5213',
'STATE 6 o 3385 3522',
'STATE 5 p 3522 3385',
'STATE 5 i 5215 5214',
'STATE 5 l 5217 5216',
'STATE 5 f 3522 5218',
'STATE 6 u 5219 3385',
'STATE 3 m 5221 5220',
'STATE 5 s 3522 3385',
'STATE 4 n 3385 5222',
'STATE 3 p 4545 4701',
'STATE 6 u 3522 3385',
'STATE 6 l 3385 5223',
'STATE 4 l 3522 3393',
'STATE 6 t 3385 3393',
'STATE 3 b 5225 5224',
'STATE 3 l 3385 3494',
'STATE 4 v 3385 5226',
'STATE 6 v 5228 5227',
'STATE 4 f 3455 5229',
'STATE 6 p 3385 5230',
'STATE 3 v 3385 5231',
'STATE 6 t 3455 5232',
'STATE 6 n 3494 3385',
'STATE 4 n 3385 5233',
'STATE 4 n 3455 5234',
'STATE 4 n 3389 4876',
'STATE 4 b 3455 3385',
'STATE 4 o 3444 5235',
'STATE 6 o 3455 5236',
'STATE 4 w 3444 5237',
'STATE 6 s 3455 5238',
'STATE 4 d 3455 3444',
'STATE 4 s 4090 3455',
'STATE 6 t 3385 3444',
'STATE 6 s 3393 5239',
'STATE 2 i 3399 5240',
'STATE 5 n 3385 3399',
'STATE 2 p 3385 5241',
'STATE 3 h 3393 3444',
'STATE 1 r 3522 3399',
'STATE 3 d 3385 5242',
'STATE 4 v 5244 5243',
'STATE 5 a 5246 5245',
'STATE 3 s 5248 5247',
'STATE 2 b 3494 5249',
'STATE 6 w 3399 5250',
'STATE 1 e 5252 5251',
'STATE 5 h 3385 5253',
'STATE 6 r 5255 5254',
'STATE 3 l 3393 3455',
'STATE 4 f 3399 5256',
'STATE 6 r 3455 5257',
'STATE 1 v 3494 5258',
'STATE 5 o 3385 3455',
'STATE 4 h 3399 5259',
'STATE 6 n 5261 5260',
'STATE 2 t 3385 4727',
'STATE 5 a 5263 5262',
'STATE 2 f 3393 3494',
'STATE 4 f 3455 3385',
'STATE 3 l 4545 5264',
'STATE 4 v 5266 5265',
'STATE 2 f 3385 5267',
'STATE 6 d 5269 5268',
'STATE 5 a 3444 3385',
'STATE 4 s 3455 3444',
'STATE 5 i 3385 5270',
'STATE 3 r 3385 5271',
'STATE 4 b 3393 5272',
'STATE 3 n 3455 5273',
'STATE 2 a 3524 5271',
'STATE 5 k 3385 5274',
'STATE 2 r 3385 5275',
'STATE 6 n 5276 3399',
'STATE 2 p 3455 5277',
'STATE 2 o 3455 5278',
'STATE 6 a 3524 3455',
'STATE 6 s 3385 3455',
'STATE 3 d 3455 5279',
'STATE 2 o 5280 3385',
'STATE 3 t 3455 5281',
'STATE 3 m 3385 5282',
'STATE 1 v 3455 3522',
'STATE 4 d 3385 5283',
'STATE 5 l 3385 5284',
'STATE 6 z 3389 5285',
'STATE 5 t 3385 5286',
'STATE 6 t 3385 5287',
'STATE 6 n 3385 3522',
'STATE 5 r 3522 5288',
'STATE 6 a 3455 3522',
'STATE 5 j 3522 5289',
'STATE 3 s 3494 3385',
'STATE 3 z 3385 5290',
'STATE 6 e 3385 5291',
'STATE 3 b 3455 5292',
'STATE 4 m 5294 5293',
'STATE 4 h 3393 5295',
'STATE 4 n 3385 5296',
'STATE 4 m 3385 5297',
'STATE 4 m 5299 5298',
'STATE 4 h 3455 4545',
'STATE 4 v 4545 3455',
'STATE 4 t 5301 5300',
'STATE 6 l 3522 5302',
'STATE 4 n 3455 3385',
'STATE 4 d 3385 5303',
'STATE 6 n 5304 4746',
'STATE 5 l 5306 5305',
'STATE 6 i 5308 5307',
'STATE 4 t 5310 5309',
'STATE 6 g 3385 3455',
'STATE 6 n 5311 3444',
'STATE 1 o 3399 5312',
'STATE 1 n 5314 5313',
'STATE 1 a 3455 5315',
'STATE 4 m 5317 5316',
'STATE 5 a 4545 5318',
'STATE 3 r 5320 5319',
'STATE 6 l 3393 3494',
'STATE 2 g 3455 5321',
'STATE 1 e 3455 3494',
'STATE 2 k 3494 3385',
'STATE 6 s 3393 5322',
'STATE 5 c 3385 5323',
'STATE 5 r 3455 3385',
'STATE 3 t 3522 3399',
'STATE 6 d 3455 3399',
'STATE 2 a 3455 3399',
'STATE 5 u 3455 5324',
'STATE 4 c 3385 3444',
'STATE 4 g 5326 5325',
'STATE 2 c 3455 5327',
'STATE 4 s 5329 5328',
'STATE 4 t 3455 5330',
'STATE 2 o 5332 5331',
'STATE 3 a 3399 5333',
'STATE 3 n 3455 5334',
'STATE 4 d 4648 3385',
'STATE 6 n 5335 3393',
'STATE 4 v 3385 5336',
'STATE 5 e 3393 5337',
'STATE 4 j 3385 3444',
'STATE 5 r 3385 5338',
'STATE 1 # 3393 3524',
'STATE 1 a 3385 5339',
'STATE 3 d 3385 3393',
'STATE 6 a 3455 5340',
'STATE 2 c 3455 5341',
'STATE 2 e 3455 3385',
'STATE 1 t 3455 5342',
'STATE 1 h 3455 5343',
'STATE 2 e 4745 3385',
'STATE 1 p 3385 5344',
'STATE 2 u 3385 5345',
'STATE 1 p 3455 3385',
'STATE 5 i 5347 5346',
'STATE 5 e 3385 5348',
'STATE 6 s 3455 3385',
'STATE 6 i 5128 3385',
'STATE 6 h 3385 5349',
'STATE 5 h 3522 5350',
'STATE 5 r 3522 3385',
'STATE 5 d 4662 3385',
'STATE 5 s 3385 5351',
'STATE 4 l 3385 5352',
'STATE 4 l 5354 5353',
'STATE 6 n 3385 5334',
'STATE 6 d 4545 5355',
'STATE 6 a 3455 5356',
'STATE 4 t 5357 3385',
'STATE 4 d 5359 5358',
'STATE 6 n 3393 5360',
'STATE 6 l 3385 5232',
'STATE 6 l 3385 3455',
'STATE 3 s 3385 5361',
'STATE 6 a 3455 5362',
'STATE 4 t 3385 3455',
'STATE 5 y 3444 3385',
'STATE 6 i 3385 3444',
'STATE 5 p 3455 5363',
'STATE 5 c 3455 3385',
'STATE 4 l 3455 3444',
'STATE 6 r 3455 3385',
'STATE 4 g 3444 3385',
'STATE 1 # 5365 5364',
'STATE 1 # 5367 5366',
'STATE 4 x 3385 3455',
'STATE 1 o 4727 3385',
'STATE 4 x 3385 5368',
'STATE 5 a 5370 5369',
'STATE 3 s 3455 5371',
'STATE 5 o 5373 5372',
'STATE 2 g 3522 3385',
'STATE 1 t 3399 5374',
'STATE 5 r 5376 5375',
'STATE 5 r 5378 5377',
'STATE 3 n 3455 5379',
'STATE 4 s 3385 5380',
'STATE 1 d 3385 3455',
'STATE 4 d 3455 5381',
'STATE 2 l 3399 5382',
'STATE 1 a 3393 3455',
'STATE 3 h 3455 5383',
'STATE 5 u 5385 5384',
'STATE 6 n 3455 5386',
'STATE 2 e 3455 5387',
'STATE 3 m 3385 3455',
'STATE 3 l 3385 3393',
'STATE 3 a 3385 5388',
'STATE 6 a 3385 5389',
'STATE 1 # 3385 5390',
'STATE 4 t 5392 5391',
'STATE 5 a 4686 5393',
'STATE 3 t 3399 5394',
'STATE 3 m 3455 5395',
'STATE 1 c 3455 5396',
'STATE 6 i 3522 3385',
'STATE 3 s 3455 3385',
'STATE 4 t 3385 5397',
'STATE 6 t 3385 5398',
'STATE 5 y 3385 5399',
'STATE 6 e 3385 5128',
'STATE 6 u 3522 5400',
'STATE 6 i 3385 5401',
'STATE 4 c 3944 5402',
'STATE 6 n 4964 5403',
'STATE 3 b 3455 5404',
'STATE 4 l 5406 5405',
'STATE 4 d 3455 3385',
'STATE 3 h 3385 5407',
'STATE 4 b 5345 5408',
'STATE 3 s 3455 3393',
'STATE 3 l 3455 3385',
'STATE 3 h 3385 5409',
'STATE 6 e 3385 5410',
'STATE 6 r 3444 5411',
'STATE 2 u 3444 5412',
'STATE 3 l 5414 5413',
'STATE 2 s 3494 5415',
'STATE 3 r 3385 5416',
'STATE 4 q 5418 5417',
'STATE 5 e 5420 5419',
'STATE 6 k 3399 5421',
'STATE 3 r 5423 5422',
'STATE 3 l 5425 5424',
'STATE 6 r 3455 5426',
'STATE 2 o 3399 5427',
'STATE 3 c 3393 5428',
'STATE 2 a 3393 3385',
'STATE 2 o 3399 5429',
'STATE 6 i 5334 5430',
'STATE 6 e 3399 5431',
'STATE 1 l 3524 5432',
'STATE 4 p 3399 5433',
'STATE 1 o 3385 5434',
'STATE 4 s 3522 5435',
'STATE 5 i 5437 5436',
'STATE 4 c 5276 4416',
'STATE 5 i 3393 3524',
'STATE 3 r 5439 5438',
'STATE 4 g 3393 5440',
'STATE 6 e 3385 5441',
'STATE 1 c 3385 5442',
'STATE 3 r 3393 5443',
'STATE 1 # 3393 3385',
'STATE 3 c 3455 5444',
'STATE 3 d 3455 5445',
'STATE 3 h 3455 5446',
'STATE 3 d 3455 5447',
'STATE 5 o 3385 5448',
'STATE 6 s 3389 3385',
'STATE 5 v 3385 5449',
'STATE 5 v 3522 5450',
'STATE 5 d 3522 3385',
'STATE 6 e 3385 5451',
'STATE 3 g 3385 5452',
'STATE 6 d 3385 5453',
'STATE 6 z 3385 5454',
'STATE 6 c 3455 5455',
'STATE 3 v 3385 3393',
'STATE 6 s 3393 5456',
'STATE 4 l 5458 5457',
'STATE 4 b 5460 5459',
'STATE 6 a 3444 5461',
'STATE 5 t 4697 5462',
'STATE 2 g 3399 3393',
'STATE 6 o 3399 3393',
'STATE 3 w 3385 5463',
'STATE 2 a 3522 3385',
'STATE 1 i 5465 5464',
'STATE 1 a 3494 5466',
'STATE 5 b 5468 5467',
'STATE 3 r 3393 5469',
'STATE 1 a 3455 5470',
'STATE 6 r 3385 4042',
'STATE 1 i 3385 3522',
'STATE 3 u 3399 5471',
'STATE 5 r 3522 5472',
'STATE 2 a 3399 3455',
'STATE 2 t 5474 5473',
'STATE 3 p 3393 5475',
'STATE 1 r 3455 5476',
'STATE 6 y 3455 3399',
'STATE 1 o 3385 5477',
'STATE 3 s 3455 5478',
'STATE 2 o 3494 3455',
'STATE 4 x 3385 5479',
'STATE 2 n 3494 3385',
'STATE 5 o 5481 5480',
'STATE 6 e 5483 5482',
'STATE 4 h 3385 5484',
'STATE 2 c 3393 3385',
'STATE 3 r 3393 3399',
'STATE 4 s 3385 5485',
'STATE 1 s 3385 3522',
'STATE 2 a 3385 4499',
'STATE 3 k 3455 5486',
'STATE 6 r 3455 3647',
'STATE 3 z 3455 5487',
'STATE 3 h 3455 5488',
'STATE 5 l 3385 5489',
'STATE 6 r 5491 5490',
'STATE 6 i 3385 5134',
'STATE 3 n 3393 3385',
'STATE 6 p 3385 5492',
'STATE 6 n 4311 3385',
'STATE 3 w 3385 5493',
'STATE 3 m 3455 5494',
'STATE 6 r 5496 5495',
'STATE 6 e 3393 5497',
'STATE 3 f 3455 5498',
'STATE 4 t 3385 3444',
'STATE 5 i 3455 3385',
'STATE 6 h 3444 3385',
'STATE 3 l 3393 5499',
'STATE 1 a 5501 5500',
'STATE 4 j 3399 5502',
'STATE 3 r 5504 5503',
'STATE 6 e 3393 3455',
'STATE 5 u 3455 5505',
'STATE 2 s 3385 4833',
'STATE 3 h 3393 5506',
'STATE 6 r 5426 3455',
'STATE 6 e 3494 5507',
'STATE 2 l 3455 3385',
'STATE 4 m 3455 5508',
'STATE 3 h 3455 3385',
'STATE 5 n 3455 5509',
'STATE 6 a 3455 5510',
'STATE 1 e 3385 5511',
'STATE 4 c 3385 5512',
'STATE 2 p 3455 5513',
'STATE 2 u 3455 5514',
'STATE 3 r 3385 5515',
'STATE 3 h 5516 3385',
'STATE 2 c 3385 3393',
'STATE 3 l 3385 5517',
'STATE 5 a 3452 3385',
'STATE 5 v 3385 5518',
'STATE 1 u 3455 5519',
'STATE 3 p 3385 5520',
'STATE 4 h 3385 5521',
'STATE 6 o 3385 5522',
'STATE 5 d 3385 3522',
'STATE 6 y 3385 5523',
'STATE 6 s 5525 5524',
'STATE 6 b 3385 5526',
'STATE 3 l 3385 5527',
'STATE 4 g 3522 3385',
'STATE 3 m 3385 5528',
'STATE 6 n 3455 5529',
'STATE 6 # 5530 3393',
'STATE 2 t 3385 5531',
'STATE 4 v 3522 3385',
'STATE 1 e 5533 5532',
'STATE 4 b 3455 5534',
'STATE 4 p 5535 3455',
'STATE 3 m 3385 5536',
'STATE 6 r 3399 3455',
'STATE 5 r 3399 5537',
'STATE 4 g 3455 3399',
'STATE 5 a 3455 5538',
'STATE 5 o 3399 5539',
'STATE 3 c 5540 3385',
'STATE 1 h 3399 5304',
'STATE 3 l 5035 5541',
'STATE 6 a 3385 5542',
'STATE 4 m 3455 3385',
'STATE 4 m 3385 3393',
'STATE 4 f 3385 5543',
'STATE 6 i 3385 5544',
'STATE 3 n 4913 5545',
'STATE 5 f 3455 5546',
'STATE 5 p 3385 5547',
'STATE 5 g 3522 3385',
'STATE 4 s 3385 5548',
'STATE 6 a 3455 5549',
'STATE 4 c 3455 5356',
'STATE 3 c 3522 4989',
'STATE 6 d 3455 5550',
'STATE 4 p 3522 5551',
'STATE 3 m 3522 3385',
'STATE 3 r 3444 5552',
'STATE 4 x 5554 5553',
'STATE 5 q 3385 5555',
'STATE 2 r 4279 5556',
'STATE 2 d 3399 5557',
'STATE 5 r 3385 3455',
'STATE 5 o 3399 3385',
'STATE 5 u 3399 5558',
'STATE 2 a 3399 5559',
'STATE 2 i 3399 5560',
'STATE 6 o 3385 3399',
'STATE 2 s 3455 5561',
'STATE 4 s 5562 3385',
'STATE 3 h 3385 5563',
'STATE 1 i 3455 5564',
'STATE 2 t 3385 5565',
'STATE 6 n 3455 5566',
'STATE 4 b 3385 5567',
'STATE 4 d 5310 5568',
'STATE 4 b 3385 5569',
'STATE 6 c 3455 5570',
'STATE 3 g 3385 5571',
'STATE 1 a 3444 3393',
'STATE 4 v 4359 3455',
'STATE 2 i 3522 5572',
'STATE 2 h 5574 5573',
'STATE 2 f 3385 5575',
'STATE 4 h 3455 5576',
'STATE 6 r 3399 3385',
'STATE 3 l 3455 5577',
'STATE 5 h 3399 3385',
'STATE 2 g 3455 5578',
'STATE 2 c 5579 3385',
'STATE 4 p 3455 3385',
'STATE 2 e 3455 5580',
'STATE 3 r 3455 5581',
'STATE 3 k 3455 5582',
'STATE 5 s 3385 5583',
'STATE 3 l 3385 4424',
'STATE 3 g 5064 5584',
'STATE 4 g 3385 5585',
'STATE 4 d 3385 5586',
'STATE 2 u 3455 5587',
'STATE 3 h 5589 5588',
'STATE 3 o 3399 5590',
'STATE 2 m 3385 3399',
'STATE 6 s 3399 3455',
'STATE 5 e 4090 3455',
'STATE 2 e 3455 5591',
'STATE 5 c 3522 3385',
'STATE 2 o 3385 3455',
'STATE 1 o 3455 5592',
'STATE 2 r 3455 5593',
'STATE 5 e 3393 3385',
'STATE 4 f 3455 5594',
'STATE 4 t 3385 5595',
'STATE 3 f 3385 5596',
'STATE 2 a 3385 5597',
'STATE 4 c 5599 5598',
'STATE 4 u 3399 4523',
'STATE 3 a 3399 3385',
'STATE 2 n 3455 3385',
'STATE 1 e 3455 5600',
'STATE 2 u 3455 4701',
'STATE 3 c 3455 5601',
'STATE 3 t 3385 5602',
'STATE 6 n 5603 3385',
'STATE 2 o 3522 5276',
'STATE 5 h 5605 5604',
'STATE 6 l 3455 5606',
'STATE 3 k 3455 3385',
'STATE 4 m 3385 5037',
'STATE 3 s 3385 5607',
'STATE 4 n 3987 3393',
'STATE 3 d 5609 5608',
'STATE 1 j 3455 5610',
'STATE 5 e 3494 5611',
'STATE 4 v 3385 5612',
'STATE 4 h 3399 5613',
'STATE 2 i 3399 5614',
'STATE 3 r 3399 5615',
'STATE 6 e 5616 4912',
'STATE 4 n 3385 5617',
'STATE 6 s 3399 5618',
'STATE 1 r 3455 5619',
'STATE 6 o 3399 5620',
'STATE 5 h 3385 3494',
'STATE 3 b 3385 3522',
'STATE 5 n 3399 5621',
'STATE 4 b 3399 3455',
'STATE 2 a 3385 5622',
'STATE 5 u 3399 5623',
'STATE 6 i 3399 3494',
'STATE 1 f 3399 5624',
'STATE 3 r 5626 5625',
'STATE 2 i 5628 5627',
'STATE 2 t 3455 5629',
'STATE 4 p 5631 5630',
'STATE 5 o 3399 5632',
'STATE 2 r 3455 5633',
'STATE 3 g 3455 5634',
'STATE 5 p 3385 3399',
'STATE 6 r 3399 5635',
'STATE 6 r 3455 3399',
'STATE 3 a 3399 5636',
'STATE 5 e 3455 5637',
'STATE 1 a 5639 5638',
'STATE 3 n 5640 3399',
'STATE 6 a 3399 5641',
'STATE 4 b 3455 3399',
'STATE 4 s 3399 3455',
'STATE 4 f 3399 5642',
'STATE 3 l 3399 5643',
'STATE 1 n 3399 5644',
'STATE 5 e 5645 3399',
'STATE 6 r 3399 5646',
'STATE 4 k 3399 3455',
'INDEX 5647 f',
'STATE 4 f 5649 5648',
'PHONE f',
'PHONE epsilon',
'INDEX 5650 g',
'STATE 3 n 5652 5651',
'STATE 4 h 5654 5653',
'STATE 4 # 5656 5655',
'STATE 4 g 5658 5657',
'STATE 5 t 5656 5659',
'STATE 4 s 5656 5660',
'PHONE epsilon',
'STATE 4 e 5662 5661',
'STATE 2 s 5664 5663',
'STATE 3 u 5666 5665',
'STATE 4 e 5668 5667',
'STATE 4 i 5670 5669',
'STATE 5 r 5672 5671',
'STATE 1 # 5656 5673',
'STATE 6 s 5674 5656',
'STATE 3 i 5676 5675',
'STATE 5 # 5656 5677',
'STATE 4 i 5679 5678',
'STATE 6 # 5681 5680',
'STATE 4 y 5683 5682',
'STATE 2 l 5685 5684',
'STATE 3 d 5687 5686',
'STATE 6 # 5689 5688',
'STATE 5 l 5691 5690',
'PHONE g',
'STATE 1 0 5674 5692',
'STATE 5 a 5674 5693',
'STATE 1 t 5695 5694',
'STATE 4 t 5697 5696',
'STATE 6 g 5699 5698',
'STATE 5 # 5687 5700',
'STATE 5 l 5674 5701',
'STATE 4 n 5703 5702',
'STATE 3 g 5674 5704',
'STATE 5 a 5687 5705',
'STATE 5 n 5706 5687',
'STATE 3 g 5708 5707',
'PHONE jh',
'STATE 6 a 5687 5709',
'STATE 3 d 5687 5710',
'STATE 2 b 5674 5656',
'STATE 1 t 5674 5656',
'STATE 1 b 5674 5711',
'STATE 5 e 5674 5656',
'STATE 1 l 5713 5712',
'PHONE f',
'STATE 4 y 5687 5714',
'STATE 2 e 5715 5656',
'STATE 5 l 5674 5716',
'STATE 2 i 5656 5717',
'STATE 5 r 5719 5718',
'STATE 5 s 5687 5720',
'STATE 3 d 5722 5721',
'STATE 5 # 5656 5723',
'STATE 2 l 5687 5724',
'STATE 2 0 5726 5725',
'STATE 6 g 5687 5674',
'STATE 5 # 5687 5727',
'STATE 1 s 5687 5674',
'STATE 3 g 5674 5728',
'STATE 1 c 5687 5729',
'STATE 3 e 5674 5730',
'STATE 2 o 5732 5731',
'STATE 2 a 5733 5656',
'STATE 5 y 5735 5734',
'STATE 6 e 5656 5736',
'STATE 6 e 5687 5737',
'STATE 2 o 5656 5687',
'STATE 5 l 5739 5738',
'STATE 1 f 5674 5740',
'STATE 1 s 5742 5741',
'STATE 3 g 5744 5743',
'STATE 4 r 5674 5745',
'STATE 2 0 5656 5746',
'STATE 1 # 5674 5747',
'STATE 3 g 5749 5748',
'STATE 5 l 5674 5750',
'STATE 5 l 5752 5751',
'STATE 3 d 5687 5753',
'STATE 3 a 5755 5754',
'STATE 2 a 5674 5756',
'STATE 2 a 5656 5757',
'STATE 1 c 5695 5656',
'STATE 5 l 5736 5695',
'STATE 4 h 5759 5758',
'STATE 4 l 5656 5674',
'PHONE k',
'STATE 6 l 5687 5760',
'STATE 6 b 5656 5761',
'STATE 2 a 5762 5674',
'STATE 1 d 5687 5763',
'STATE 2 i 5765 5764',
'STATE 2 i 5687 5656',
'STATE 4 t 5766 5674',
'STATE 1 r 5735 5674',
'STATE 6 n 5687 5767',
'STATE 6 d 5656 5768',
'STATE 6 e 5674 5687',
'STATE 3 c 5674 5769',
'STATE 5 o 5687 5770',
'STATE 5 o 5687 5771',
'STATE 5 i 5674 5772',
'STATE 6 l 5774 5773',
'STATE 6 t 5674 5775',
'STATE 1 m 5687 5674',
'STATE 2 r 5674 5776',
'STATE 5 # 5656 5777',
'STATE 5 e 5656 5674',
'STATE 4 m 5656 5778',
'STATE 5 i 5674 5656',
'STATE 1 # 5687 5779',
'STATE 1 l 5674 5780',
'STATE 6 o 5674 5781',
'STATE 6 s 5783 5782',
'STATE 1 w 5656 5784',
'STATE 1 h 5687 5656',
'STATE 3 a 5674 5785',
'STATE 5 e 5687 5786',
'STATE 6 t 5674 5787',
'STATE 5 v 5674 5788',
'STATE 5 # 5687 5789',
'STATE 5 u 5687 5790',
'STATE 5 t 5792 5791',
'STATE 2 0 5793 5674',
'STATE 3 a 5687 5674',
'STATE 6 s 5795 5794',
'STATE 1 s 5674 5796',
'STATE 1 o 5656 5674',
'STATE 4 b 5656 5797',
'STATE 1 r 5687 5798',
'STATE 6 t 5687 5799',
'STATE 1 v 5687 5800',
'STATE 1 l 5674 5801',
'STATE 2 i 5656 5802',
'STATE 2 o 5656 5803',
'STATE 1 # 5674 5656',
'STATE 4 a 5674 5804',
'STATE 3 i 5806 5805',
'STATE 3 o 5674 5807',
'STATE 5 n 5674 5808',
'STATE 5 n 5810 5809',
'STATE 2 h 5812 5811',
'STATE 1 0 5674 5813',
'STATE 6 a 5687 5674',
'STATE 2 e 5674 5814',
'STATE 3 a 5815 5674',
'STATE 1 h 5674 5816',
'STATE 4 d 5656 5817',
'STATE 5 n 5674 5818',
'STATE 2 o 5687 5819',
'STATE 1 # 5820 5674',
'STATE 1 # 5674 5821',
'STATE 2 e 5687 5656',
'STATE 5 d 5687 5822',
'STATE 4 o 5674 5687',
'STATE 6 # 5757 5674',
'STATE 5 m 5656 5823',
'STATE 6 # 5825 5824',
'STATE 1 r 5687 5674',
'STATE 5 v 5674 5826',
'STATE 6 n 5687 5827',
'STATE 6 s 5829 5828',
'STATE 1 n 5674 5830',
'STATE 2 v 5687 5831',
'STATE 6 g 5674 5832',
'STATE 1 # 5674 5687',
'STATE 1 # 5674 5833',
'STATE 4 f 5656 5834',
'STATE 2 i 5674 5835',
'STATE 6 i 5674 5836',
'STATE 6 i 5674 5687',
'STATE 6 t 5674 5837',
'STATE 2 e 5687 5838',
'STATE 5 s 5656 5839',
'STATE 6 l 5841 5840',
'STATE 2 r 5687 5842',
'STATE 5 f 5674 5843',
'STATE 6 t 5687 5844',
'STATE 6 h 5674 5845',
'STATE 1 j 5674 5846',
'STATE 5 m 5674 5847',
'STATE 6 t 5848 5674',
'STATE 6 l 5674 5849',
'STATE 2 e 5674 5687',
'STATE 4 w 5656 5850',
'STATE 5 e 5656 5687',
'STATE 6 u 5687 5851',
'STATE 2 i 5687 5674',
'STATE 1 r 5687 5852',
'STATE 1 0 5674 5853',
'STATE 2 b 5674 5854',
'STATE 1 c 5674 5855',
'STATE 2 o 5855 5856',
'STATE 6 d 5674 5857',
'STATE 6 g 5674 5687',
'STATE 5 u 5859 5858',
'STATE 1 # 5674 5860',
'STATE 5 n 5674 5687',
'STATE 3 r 5674 5687',
'STATE 6 b 5674 5861',
'STATE 4 p 5656 5862',
'STATE 2 a 5687 5863',
'STATE 1 l 5656 5864',
'STATE 5 a 5674 5865',
'STATE 2 h 5674 5866',
'STATE 5 e 5674 5687',
'STATE 1 # 5687 5867',
'STATE 6 s 5674 5868',
'STATE 1 b 5848 5869',
'PHONE zh',
'STATE 5 m 5687 5870',
'STATE 6 y 5687 5871',
'STATE 5 u 5873 5872',
'STATE 1 # 5674 5874',
'STATE 5 r 5875 5687',
'STATE 6 f 5674 5876',
'STATE 2 s 5674 5687',
'STATE 5 s 5674 5877',
'STATE 5 s 5687 5878',
'STATE 2 b 5674 5879',
'STATE 5 n 5880 5674',
'STATE 2 n 5687 5881',
'STATE 4 o 5674 5882',
'STATE 2 a 5656 5883',
'STATE 2 i 5674 5687',
'STATE 2 a 5656 5674',
'STATE 2 m 5674 5884',
'STATE 1 r 5674 5885',
'STATE 5 r 5674 5886',
'STATE 6 r 5888 5887',
'STATE 1 0 5687 5674',
'STATE 2 0 5890 5889',
'STATE 4 a 5892 5891',
'STATE 6 i 5656 5674',
'STATE 6 n 5894 5893',
'STATE 2 e 5674 5895',
'STATE 6 g 5674 5896',
'STATE 3 a 5898 5897',
'STATE 5 o 5687 5899',
'STATE 3 i 5687 5900',
'STATE 6 m 5687 5901',
'STATE 4 r 5674 5902',
'STATE 1 h 5656 5674',
'STATE 6 r 5656 5903',
'STATE 5 i 5656 5674',
'STATE 3 a 5674 5687',
'STATE 6 m 5674 5904',
'STATE 6 t 5687 5905',
'STATE 5 n 5815 5687',
'STATE 5 a 5674 5906',
'STATE 6 i 5687 5674',
'STATE 6 r 5687 5907',
'STATE 4 l 5909 5908',
'STATE 6 # 5656 5674',
'STATE 5 d 5674 5910',
'STATE 5 o 5687 5911',
'STATE 5 e 5674 5912',
'STATE 6 i 5687 5913',
'STATE 5 e 5656 5914',
'STATE 2 e 5757 5915',
'STATE 5 b 5917 5916',
'STATE 5 n 5919 5918',
'STATE 5 h 5674 5687',
'STATE 6 o 5687 5674',
'STATE 4 u 5674 5656',
'STATE 5 e 5921 5920',
'STATE 6 l 5687 5922',
'STATE 6 b 5674 5687',
'STATE 6 # 5687 5923',
'STATE 6 # 5924 5874',
'STATE 6 s 5674 5925',
'STATE 1 l 5656 5674',
'STATE 5 e 5674 5926',
'STATE 5 a 5687 5927',
'STATE 3 o 5687 5928',
'STATE 6 n 5674 5929',
'STATE 6 a 5674 5930',
'STATE 6 i 5687 5931',
'STATE 3 r 5674 5932',
'STATE 6 m 5674 5933',
'STATE 6 t 5674 5687',
'STATE 3 l 5674 5934',
'STATE 1 # 5674 5935',
'STATE 5 a 5656 5894',
'STATE 6 a 5674 5936',
'STATE 3 i 5687 5674',
'STATE 5 m 5687 5937',
'STATE 1 0 5938 5674',
'STATE 5 b 5687 5674',
'INDEX 5939 h',
'STATE 3 # 5941 5940',
'STATE 3 c 5943 5942',
'STATE 4 a 5945 5944',
'STATE 3 s 5947 5946',
'STATE 4 o 5948 5947',
'STATE 4 o 5945 5949',
'PHONE hh',
'STATE 3 t 5951 5950',
'PHONE epsilon',
'STATE 1 i 5945 5947',
'STATE 4 i 5945 5952',
'STATE 3 p 5954 5953',
'STATE 4 o 5955 5947',
'STATE 4 u 5945 5956',
'STATE 3 g 5958 5957',
'STATE 1 # 5947 5959',
'STATE 1 0 5947 5960',
'STATE 4 y 5945 5961',
'STATE 1 0 5963 5962',
'STATE 1 i 5965 5964',
'STATE 1 h 5966 5947',
'STATE 1 g 5945 5947',
'STATE 4 e 5945 5967',
'STATE 4 l 5947 5968',
'STATE 4 o 5970 5969',
'STATE 4 o 5972 5971',
'STATE 4 o 5945 5973',
'STATE 4 e 5947 5945',
'STATE 4 r 5945 5974',
'STATE 4 # 5947 5975',
'STATE 3 o 5977 5976',
'STATE 3 r 5947 5978',
'STATE 4 a 5980 5979',
'STATE 1 0 5947 5945',
'STATE 4 a 5945 5947',
'STATE 4 l 5945 5947',
'STATE 4 n 5947 5981',
'STATE 3 a 5973 5947',
'STATE 4 a 5945 5982',
'STATE 3 w 5945 5947',
'STATE 4 u 5945 5947',
'STATE 1 0 5947 5983',
'STATE 4 r 5947 5984',
'STATE 4 l 5947 5985',
'STATE 1 n 5945 5947',
'STATE 4 m 5947 5986',
'STATE 4 m 5947 5945',
'STATE 4 s 5947 5987',
'STATE 3 w 5983 5988',
'STATE 3 x 5990 5989',
'STATE 4 t 5947 5991',
'STATE 4 i 5947 5992',
'STATE 4 d 5947 5993',
'STATE 4 a 5947 5945',
'STATE 4 o 5995 5994',
'STATE 3 n 5945 5996',
'STATE 1 s 5945 5997',
'STATE 4 y 5999 5998',
'STATE 1 c 5945 6000',
'STATE 1 h 5992 6001',
'STATE 3 e 5945 6002',
'STATE 3 e 5945 6003',
'STATE 3 z 6005 6004',
'STATE 3 a 5945 5947',
'STATE 3 d 5945 6006',
'STATE 3 d 6008 6007',
'STATE 4 a 5945 6009',
'STATE 3 n 5945 6010',
'STATE 1 u 5945 6011',
'STATE 4 e 5945 6012',
'STATE 4 e 5945 5947',
'STATE 1 o 6013 5945',
'STATE 1 e 5945 6014',
'STATE 4 u 5945 6015',
'STATE 3 r 5947 5945',
'STATE 4 a 6017 6016',
'STATE 1 e 5945 5947',
'STATE 1 # 6019 6018',
'STATE 1 # 5945 6020',
'STATE 1 k 5947 6021',
'STATE 4 i 5945 6022',
'STATE 1 b 5945 6023',
'STATE 3 o 5945 6024',
'STATE 4 e 5945 5979',
'STATE 1 v 5945 6025',
'STATE 1 p 5945 6026',
'STATE 1 m 5947 6027',
'STATE 1 t 5945 6028',
'STATE 3 i 5945 6029',
'STATE 1 i 5945 6030',
'STATE 1 f 5945 6031',
'STATE 1 l 5945 6032',
'STATE 1 d 5945 6033',
'STATE 1 o 5945 6034',
'STATE 3 o 5945 6035',
'STATE 1 g 5945 6036',
'STATE 3 a 5945 6037',
'STATE 3 i 6039 6038',
'STATE 3 m 5945 6040',
'STATE 1 c 5945 6041',
'STATE 1 n 5947 5945',
'STATE 1 r 6042 5945',
'STATE 1 n 5945 6043',
'STATE 3 u 5947 5945',
'STATE 3 e 5945 6044',
'STATE 1 w 5945 6045',
'STATE 1 r 5945 6046',
'STATE 1 a 5945 6047',
'STATE 3 r 5945 6048',
'STATE 1 s 5945 6049',
'STATE 1 m 5945 6002',
'INDEX 6050 i',
'STATE 5 g 6052 6051',
'STATE 4 # 6054 6053',
'STATE 6 # 6056 6055',
'STATE 4 o 6058 6057',
'STATE 3 a 6060 6059',
'STATE 2 # 6062 6061',
'STATE 4 n 6064 6063',
'STATE 4 e 6066 6065',
'STATE 3 t 6068 6067',
'STATE 3 e 6070 6069',
'STATE 2 h 6072 6071',
'STATE 4 n 6074 6073',
'STATE 4 e 6076 6075',
'STATE 4 e 6076 6077',
'STATE 2 # 6077 6078',
'STATE 4 a 6080 6079',
'STATE 2 # 6082 6081',
'STATE 3 s 6084 6083',
'STATE 5 n 6086 6085',
'PHONE iy',
'STATE 2 m 6072 6086',
'STATE 2 k 6072 6087',
'PHONE ay1',
'STATE 4 e 6076 6088',
'STATE 3 s 6090 6089',
'STATE 4 r 6086 6091',
'PHONE iy1',
'PHONE ih1',
'STATE 3 r 6093 6092',
'STATE 3 a 6095 6094',
'STATE 3 c 6097 6096',
'STATE 5 r 6099 6098',
'STATE 5 r 6101 6100',
'STATE 2 # 6103 6102',
'STATE 5 n 6086 6069',
'STATE 5 u 6086 6069',
'PHONE epsilon',
'STATE 2 d 6086 6072',
'STATE 6 e 6105 6104',
'STATE 2 0 6077 6106',
'STATE 6 e 6108 6107',
'STATE 4 a 6110 6109',
'STATE 3 w 6111 6107',
'STATE 2 d 6077 6112',
'STATE 5 e 6114 6113',
'STATE 4 c 6116 6115',
'STATE 3 t 6118 6117',
'STATE 5 t 6069 6119',
'STATE 5 # 6069 6120',
'STATE 3 e 6122 6121',
'STATE 5 w 6086 6123',
'STATE 3 h 6072 6124',
'STATE 5 n 6126 6125',
'STATE 3 g 6086 6127',
'STATE 4 o 6129 6128',
'STATE 2 r 6107 6130',
'STATE 3 e 6072 6131',
'PHONE ih',
'PHONE ah',
'STATE 4 o 6132 6077',
'STATE 3 d 6072 6069',
'STATE 2 o 6107 6133',
'STATE 2 e 6107 6134',
'STATE 4 c 6136 6135',
'STATE 4 v 6138 6137',
'STATE 2 z 6077 6139',
'STATE 5 # 6107 6086',
'STATE 2 # 6141 6140',
'STATE 5 t 6069 6142',
'STATE 6 o 6069 6143',
'STATE 5 s 6145 6144',
'STATE 6 # 6147 6146',
'STATE 2 m 6132 6072',
'STATE 3 d 6149 6148',
'STATE 3 p 6077 6150',
'STATE 5 u 6152 6151',
'STATE 2 l 6154 6153',
'STATE 3 f 6069 6155',
'STATE 4 a 6069 6156',
'STATE 2 o 6132 6069',
'STATE 4 d 6158 6157',
'STATE 2 s 6160 6159',
'PHONE ay',
'STATE 2 a 6107 6161',
'STATE 2 a 6107 6162',
'STATE 4 r 6164 6163',
'STATE 2 # 6166 6165',
'STATE 3 o 6168 6167',
'STATE 3 t 6107 6169',
'STATE 5 a 6171 6170',
'STATE 3 s 6173 6172',
'STATE 3 g 6086 6174',
'STATE 6 # 6086 6175',
'STATE 5 l 6086 6176',
'STATE 5 d 6178 6177',
'STATE 3 f 6072 6179',
'STATE 6 s 6181 6180',
'STATE 3 a 6132 6182',
'STATE 6 e 6076 6183',
'STATE 5 t 6185 6184',
'STATE 3 s 6069 6186',
'STATE 5 r 6188 6187',
'STATE 3 c 6086 6189',
'STATE 3 n 6154 6190',
'PHONE y',
'STATE 5 l 6192 6191',
'STATE 6 r 6086 6077',
'STATE 4 g 6077 6072',
'STATE 3 r 6193 6107',
'STATE 3 r 6195 6194',
'STATE 3 w 6077 6196',
'STATE 2 e 6107 6077',
'STATE 2 u 6107 6197',
'STATE 3 e 6199 6198',
'STATE 6 n 6201 6200',
'STATE 5 # 6107 6202',
'STATE 5 r 6072 6203',
'STATE 6 # 6205 6204',
'STATE 4 r 6206 6086',
'STATE 3 s 6107 6207',
'STATE 5 o 6209 6208',
'STATE 4 n 6086 6210',
'STATE 3 l 6212 6211',
'STATE 5 k 6069 6213',
'STATE 3 d 6072 6214',
'STATE 2 s 6216 6215',
'STATE 5 n 6086 6217',
'STATE 5 w 6219 6218',
'STATE 3 f 6072 6220',
'STATE 6 # 6222 6221',
'STATE 6 i 6077 6223',
'STATE 2 i 6072 6224',
'STATE 3 f 6072 6225',
'STATE 5 s 6184 6226',
'STATE 6 # 6072 6076',
'STATE 6 z 6076 6227',
'STATE 6 a 6076 6228',
'STATE 2 0 6072 6229',
'STATE 3 r 6231 6230',
'STATE 3 g 6086 6069',
'STATE 2 0 6132 6232',
'STATE 3 b 6072 6233',
'STATE 6 e 6072 6234',
'STATE 2 b 6077 6235',
'STATE 6 s 6107 6236',
'STATE 6 t 6107 6237',
'STATE 6 u 6077 6238',
'STATE 2 o 6107 6239',
'STATE 3 o 6241 6240',
'STATE 4 g 6243 6242',
'STATE 3 o 6245 6244',
'STATE 5 a 6107 6246',
'STATE 5 a 6248 6247',
'STATE 5 k 6077 6249',
'STATE 2 # 6251 6250',
'STATE 4 n 6253 6252',
'PHONE aa1',
'STATE 3 e 6076 6254',
'STATE 4 v 6086 6255',
'STATE 4 n 6086 6256',
'STATE 2 t 6072 6086',
'STATE 3 g 6258 6257',
'STATE 6 e 6260 6259',
'STATE 6 i 6069 6261',
'STATE 5 n 6263 6262',
'STATE 2 n 6086 6264',
'STATE 5 # 6069 6265',
'STATE 6 # 6069 6086',
'STATE 5 n 6267 6266',
'STATE 3 v 6086 6268',
'STATE 6 # 6270 6269',
'STATE 3 r 6076 6271',
'STATE 3 l 6273 6272',
'STATE 6 e 6275 6274',
'STATE 3 l 6154 6276',
'STATE 3 s 6069 6277',
'STATE 3 b 6279 6278',
'STATE 6 e 6076 6072',
'STATE 6 n 6077 6280',
'STATE 5 # 6282 6281',
'STATE 2 a 6154 6283',
'STATE 2 p 6132 6069',
'STATE 3 l 6285 6284',
'STATE 6 t 6069 6286',
'STATE 6 i 6132 6287',
'STATE 2 t 6077 6288',
'STATE 2 k 6077 6289',
'STATE 2 e 6107 6290',
'STATE 6 i 6077 6291',
'STATE 2 r 6107 6292',
'STATE 2 # 6294 6293',
'STATE 4 s 6296 6295',
'STATE 5 # 6298 6297',
'STATE 5 h 6300 6299',
'STATE 5 a 6302 6301',
'STATE 5 # 6206 6086',
'STATE 2 # 6304 6303',
'STATE 6 # 6306 6305',
'STATE 6 t 6308 6307',
'STATE 3 e 6072 6309',
'STATE 3 e 6311 6310',
'STATE 4 r 6313 6312',
'STATE 4 c 6315 6314',
'STATE 3 l 6317 6316',
'STATE 3 d 6319 6318',
'STATE 4 s 6320 6086',
'STATE 6 r 6086 6321',
'STATE 3 o 6154 6322',
'STATE 2 r 6086 6323',
'STATE 2 g 6069 6324',
'STATE 5 m 6086 6069',
'STATE 5 s 6069 6325',
'STATE 3 h 6072 6326',
'STATE 6 # 6076 6069',
'STATE 6 s 6086 6327',
'STATE 6 i 6086 6069',
'STATE 5 t 6329 6328',
'STATE 2 f 6086 6330',
'STATE 6 i 6108 6069',
'STATE 2 f 6076 6069',
'STATE 2 p 6072 6331',
'STATE 6 t 6069 6332',
'STATE 2 i 6069 6333',
'STATE 2 l 6069 6072',
'STATE 2 n 6069 6077',
'STATE 2 e 6077 6334',
'STATE 2 o 6069 6335',
'STATE 2 r 6069 6336',
'STATE 5 n 6338 6337',
'STATE 5 n 6108 6076',
'STATE 3 w 6077 6339',
'STATE 3 a 6072 6340',
'STATE 3 h 6341 6069',
'STATE 3 n 6154 6342',
'STATE 3 r 6069 6343',
'STATE 2 g 6069 6154',
'STATE 5 n 6072 6344',
'STATE 3 b 6132 6072',
'STATE 2 d 6107 6345',
'STATE 6 u 6077 6346',
'STATE 2 a 6107 6347',
'STATE 3 l 6077 6348',
'STATE 2 i 6107 6345',
'STATE 2 0 6350 6349',
'STATE 4 g 6352 6351',
'STATE 5 c 6108 6086',
'STATE 5 # 6206 6353',
'STATE 2 r 6355 6354',
'STATE 2 h 6357 6356',
'STATE 5 n 6359 6358',
'STATE 6 # 6086 6360',
'STATE 5 o 6362 6361',
'STATE 3 e 6086 6363',
'STATE 2 0 6072 6364',
'STATE 5 o 6107 6086',
'STATE 3 e 6072 6365',
'STATE 5 o 6367 6366',
'STATE 6 l 6369 6368',
'STATE 3 r 6107 6370',
'STATE 5 h 6372 6371',
'STATE 4 z 6374 6373',
'STATE 6 d 6086 6375',
'STATE 6 s 6377 6376',
'STATE 3 d 6086 6378',
'STATE 4 l 6380 6379',
'STATE 2 # 6072 6381',
'STATE 3 w 6072 6382',
'STATE 2 o 6076 6383',
'STATE 2 o 6385 6384',
'STATE 2 # 6386 6072',
'STATE 6 # 6387 6086',
'STATE 6 n 6086 6388',
'STATE 3 n 6390 6389',
'STATE 6 # 6069 6391',
'STATE 2 p 6072 6392',
'STATE 6 # 6086 6393',
'STATE 3 v 6072 6394',
'STATE 5 # 6086 6395',
'STATE 6 l 6069 6396',
'STATE 3 u 6072 6397',
'STATE 3 c 6399 6398',
'STATE 2 r 6069 6400',
'STATE 6 c 6069 6401',
'STATE 2 c 6072 6069',
'STATE 2 i 6069 6077',
'STATE 3 r 6069 6077',
'STATE 3 r 6069 6402',
'STATE 3 l 6076 6403',
'STATE 3 v 6069 6076',
'STATE 3 b 6077 6404',
'STATE 5 t 6069 6405',
'STATE 2 c 6069 6072',
'STATE 6 a 6154 6069',
'STATE 3 p 6069 6406',
'STATE 5 c 6072 6407',
'STATE 2 h 6107 6077',
'STATE 6 a 6107 6408',
'STATE 2 r 6107 6345',
'STATE 6 l 6077 6409',
'STATE 5 a 6411 6410',
'STATE 4 n 6077 6412',
'STATE 5 a 6414 6413',
'STATE 5 h 6072 6415',
'STATE 2 b 6086 6416',
'STATE 6 # 6418 6417',
'STATE 4 n 6420 6419',
'STATE 2 t 6422 6421',
'STATE 4 s 6132 6423',
'STATE 2 w 6425 6424',
'STATE 6 i 6107 6426',
'STATE 2 h 6072 6086',
'STATE 5 r 6428 6427',
'STATE 3 e 6086 6429',
'STATE 3 p 6431 6430',
'STATE 2 n 6072 6432',
'STATE 5 i 6434 6433',
'STATE 3 e 6436 6435',
'STATE 3 r 6076 6437',
'STATE 6 # 6107 6438',
'STATE 3 g 6107 6439',
'STATE 2 e 6107 6440',
'STATE 3 b 6072 6441',
'STATE 3 m 6442 6077',
'STATE 6 s 6444 6443',
'STATE 6 n 6108 6445',
'STATE 4 t 6446 6072',
'STATE 3 e 6072 6447',
'STATE 4 g 6132 6072',
'STATE 6 l 6107 6072',
'STATE 4 r 6072 6448',
'STATE 2 # 6072 6449',
'STATE 3 r 6451 6450',
'STATE 2 # 6072 6452',
'STATE 2 a 6076 6072',
'STATE 3 n 6454 6453',
'STATE 6 r 6069 6107',
'STATE 6 s 6132 6455',
'STATE 5 e 6086 6077',
'STATE 4 d 6072 6086',
'STATE 3 a 6154 6456',
'STATE 6 u 6069 6457',
'STATE 5 n 6069 6458',
'STATE 2 e 6460 6459',
'STATE 5 # 6462 6461',
'STATE 6 # 6072 6463',
'STATE 2 i 6086 6069',
'STATE 5 u 6465 6464',
'STATE 2 p 6072 6466',
'STATE 3 t 6108 6467',
'STATE 2 i 6108 6468',
'STATE 2 a 6069 6469',
'STATE 6 o 6069 6076',
'STATE 3 z 6069 6470',
'STATE 5 t 6472 6471',
'STATE 3 g 6077 6473',
'STATE 2 i 6475 6474',
'STATE 2 s 6086 6476',
'STATE 5 t 6072 6477',
'STATE 6 o 6077 6478',
'STATE 3 t 6479 6107',
'STATE 5 y 6481 6480',
'STATE 6 # 6483 6482',
'STATE 4 m 6077 6484',
'STATE 5 o 6486 6485',
'STATE 6 # 6076 6487',
'STATE 5 r 6489 6488',
'STATE 2 a 6077 6086',
'STATE 4 n 6491 6490',
'STATE 4 s 6493 6492',
'STATE 6 u 6108 6494',
'STATE 6 # 6076 6495',
'STATE 4 n 6497 6496',
'STATE 4 n 6132 6072',
'STATE 4 t 6132 6072',
'STATE 5 # 6076 6072',
'STATE 5 # 6132 6072',
'STATE 2 r 6108 6086',
'STATE 5 i 6499 6498',
'STATE 2 0 6501 6500',
'STATE 6 # 6077 6502',
'STATE 6 # 6077 6503',
'STATE 2 # 6072 6504',
'STATE 5 o 6086 6072',
'STATE 5 k 6506 6505',
'STATE 6 s 6108 6507',
'STATE 5 i 6076 6508',
'STATE 5 h 6509 6076',
'STATE 3 n 6076 6510',
'STATE 2 n 6069 6511',
'STATE 2 s 6107 6512',
'STATE 3 d 6513 6108',
'STATE 5 t 6515 6514',
'STATE 6 e 6107 6077',
'STATE 2 0 6517 6516',
'STATE 4 n 6519 6518',
'STATE 6 d 6072 6520',
'STATE 6 r 6521 6072',
'STATE 4 b 6523 6522',
'STATE 3 e 6525 6524',
'STATE 3 t 6108 6526',
'STATE 2 e 6108 6527',
'STATE 2 o 6077 6072',
'STATE 2 a 6076 6528',
'STATE 2 l 6107 6529',
'STATE 2 u 6107 6072',
'STATE 6 r 6132 6072',
'STATE 5 h 6072 6530',
'STATE 2 e 6531 6069',
'STATE 2 e 6069 6086',
'STATE 5 r 6154 6532',
'STATE 5 # 6069 6533',
'STATE 6 s 6086 6069',
'STATE 2 o 6069 6086',
'STATE 3 w 6072 6534',
'STATE 2 c 6536 6535',
'STATE 6 # 6086 6069',
'STATE 6 y 6072 6537',
'STATE 3 n 6154 6538',
'STATE 2 s 6072 6069',
'STATE 2 o 6069 6539',
'STATE 2 l 6069 6540',
'STATE 3 s 6076 6541',
'STATE 6 z 6076 6542',
'STATE 3 v 6077 6543',
'STATE 3 c 6462 6544',
'STATE 6 a 6086 6069',
'STATE 3 g 6086 6545',
'STATE 3 r 6069 6546',
'STATE 6 t 6107 6547',
'STATE 6 s 6107 6077',
'STATE 5 h 6549 6548',
'STATE 4 n 6550 6108',
'STATE 6 t 6552 6551',
'STATE 4 k 6107 6553',
'STATE 5 o 6072 6554',
'STATE 4 u 6189 6555',
'STATE 6 # 6076 6556',
'STATE 4 s 6077 6557',
'STATE 6 # 6559 6558',
'STATE 6 e 6132 6072',
'STATE 5 t 6122 6560',
'STATE 6 t 6108 6072',
'STATE 2 h 6072 6561',
'STATE 5 t 6108 6562',
'STATE 6 # 6563 6072',
'STATE 5 f 6077 6564',
'STATE 2 f 6072 6565',
'STATE 2 r 6086 6566',
'STATE 5 # 6568 6567',
'STATE 3 p 6077 6569',
'STATE 3 m 6077 6086',
'STATE 6 i 6077 6107',
'STATE 6 l 6107 6570',
'STATE 2 d 6086 6571',
'STATE 6 t 6086 6072',
'STATE 5 t 6573 6572',
'STATE 2 e 6107 6574',
'STATE 6 z 6108 6575',
'STATE 5 s 6107 6576',
'STATE 2 r 6132 6072',
'STATE 3 t 6107 6076',
'STATE 3 r 6578 6577',
'STATE 3 d 6580 6579',
'STATE 2 n 6108 6581',
'STATE 6 # 6076 6582',
'STATE 6 o 6107 6077',
'STATE 4 g 6584 6583',
'STATE 4 n 6077 6585',
'STATE 4 g 6107 6586',
'STATE 3 s 6588 6587',
'STATE 6 s 6072 6589',
'STATE 2 r 6077 6072',
'STATE 3 t 6072 6590',
'STATE 3 f 6072 6591',
'STATE 3 u 6593 6592',
'STATE 4 d 6072 6076',
'STATE 2 a 6108 6594',
'STATE 2 s 6596 6595',
'STATE 3 t 6076 6597',
'STATE 6 s 6599 6598',
'STATE 5 s 6601 6600',
'STATE 6 # 6602 6069',
'STATE 6 s 6154 6603',
'STATE 6 # 6069 6072',
'STATE 5 l 6069 6604',
'STATE 2 a 6606 6605',
'STATE 6 # 6108 6076',
'STATE 3 t 6069 6607',
'STATE 2 b 6072 6608',
'STATE 2 l 6069 6609',
'STATE 2 a 6611 6610',
'STATE 6 h 6613 6612',
'STATE 3 v 6069 6614',
'STATE 3 f 6077 6076',
'STATE 3 v 6072 6069',
'STATE 3 z 6069 6615',
'STATE 6 i 6072 6110',
'STATE 3 h 6077 6616',
'STATE 5 o 6618 6617',
'STATE 4 g 6072 6619',
'STATE 6 # 6072 6077',
'STATE 4 g 6621 6620',
'STATE 4 v 6623 6622',
'STATE 4 l 6624 6076',
'STATE 5 a 6626 6625',
'STATE 5 i 6628 6627',
'STATE 4 k 6107 6629',
'STATE 4 y 6076 6630',
'STATE 5 u 6632 6631',
'STATE 5 s 6077 6076',
'STATE 2 c 6076 6633',
'STATE 4 n 6076 6634',
'STATE 5 s 6072 6077',
'STATE 4 s 6072 6076',
'STATE 5 i 6072 6635',
'STATE 2 b 6132 6636',
'STATE 2 b 6132 6072',
'STATE 5 u 6638 6637',
'STATE 3 e 6077 6639',
'STATE 2 0 6072 6640',
'STATE 2 # 6641 6072',
'STATE 2 # 6643 6642',
'STATE 6 i 6645 6644',
'STATE 2 n 6077 6646',
'STATE 2 r 6107 6647',
'STATE 6 d 6108 6648',
'STATE 5 t 6077 6649',
'STATE 3 h 6107 6650',
'STATE 2 f 6107 6108',
'STATE 2 c 6652 6651',
'STATE 2 a 6108 6107',
'STATE 2 u 6107 6108',
'STATE 5 c 6654 6653',
'STATE 6 n 6656 6655',
'STATE 6 n 6658 6657',
'STATE 4 d 6660 6659',
'STATE 4 l 6662 6661',
'STATE 3 m 6108 6663',
'STATE 2 u 6086 6069',
'STATE 6 r 6072 6077',
'STATE 6 r 6665 6664',
'STATE 6 r 6077 6666',
'STATE 4 t 6668 6667',
'STATE 4 s 6086 6072',
'STATE 3 s 6108 6669',
'STATE 2 n 6108 6670',
'STATE 3 t 6108 6072',
'STATE 2 s 6072 6671',
'STATE 6 # 6072 6672',
'STATE 3 l 6077 6072',
'STATE 3 f 6674 6673',
'STATE 6 # 6675 6072',
'STATE 5 n 6069 6154',
'STATE 6 i 6069 6676',
'STATE 3 l 6072 6069',
'STATE 6 # 6678 6677',
'STATE 3 n 6154 6679',
'STATE 6 t 6069 6680',
'STATE 3 l 6069 6681',
'STATE 2 n 6069 6682',
'STATE 2 t 6069 6683',
'STATE 3 n 6069 6684',
'STATE 6 b 6069 6076',
'STATE 3 p 6069 6076',
'STATE 6 r 6076 6685',
'STATE 3 c 6687 6686',
'STATE 6 i 6077 6688',
'STATE 4 u 6690 6689',
'STATE 6 # 6076 6691',
'STATE 4 s 6693 6692',
'STATE 4 v 6695 6694',
'STATE 6 n 6107 6696',
'STATE 2 c 6697 6108',
'STATE 3 r 6072 6698',
'STATE 3 u 6076 6699',
'STATE 5 l 6701 6700',
'STATE 6 l 6108 6702',
'STATE 5 d 6704 6703',
'STATE 6 n 6706 6705',
'STATE 4 v 6107 6707',
'STATE 6 t 6709 6708',
'STATE 6 e 6077 6710',
'STATE 6 r 6077 6069',
'STATE 4 s 6072 6711',
'STATE 5 y 6108 6712',
'STATE 5 v 6077 6713',
'STATE 4 l 6076 6714',
'STATE 3 e 6716 6715',
'STATE 6 s 6072 6077',
'STATE 3 m 6077 6717',
'STATE 6 c 6072 6718',
'STATE 3 h 6077 6107',
'STATE 2 e 6072 6719',
'STATE 6 d 6132 6720',
'STATE 5 o 6722 6721',
'STATE 5 h 6076 6107',
'STATE 6 s 6723 6077',
'STATE 6 s 6725 6724',
'STATE 6 n 6727 6726',
'STATE 2 s 6729 6728',
'STATE 2 a 6107 6730',
'STATE 3 m 6731 6107',
'STATE 3 t 6108 6107',
'STATE 3 n 6077 6732',
'STATE 6 i 6069 6077',
'STATE 6 d 6734 6733',
'STATE 2 e 6108 6735',
'STATE 2 s 6737 6736',
'STATE 3 l 6108 6738',
'STATE 4 r 6107 6739',
'STATE 6 o 6072 6132',
'STATE 4 c 6596 6740',
'STATE 2 o 6076 6072',
'STATE 3 g 6108 6741',
'STATE 3 l 6743 6742',
'STATE 3 n 6132 6744',
'STATE 6 l 6072 6077',
'STATE 4 s 6746 6745',
'STATE 2 i 6108 6747',
'STATE 3 r 6108 6748',
'STATE 3 v 6108 6749',
'STATE 2 o 6076 6750',
'STATE 2 0 6072 6751',
'STATE 2 t 6753 6752',
'STATE 2 i 6072 6069',
'STATE 3 r 6076 6069',
'STATE 2 l 6754 6069',
'STATE 5 v 6076 6755',
'STATE 5 f 6076 6756',
'STATE 6 # 6069 6076',
'STATE 6 # 6069 6757',
'STATE 6 t 6069 6758',
'STATE 2 e 6069 6759',
'STATE 3 v 6069 6760',
'STATE 3 v 6154 6069',
'STATE 3 p 6072 6076',
'STATE 3 x 6086 6069',
'STATE 6 a 6069 6086',
'STATE 2 m 6107 6761',
'STATE 6 n 6763 6762',
'STATE 6 # 6069 6764',
'STATE 4 k 6107 6765',
'STATE 2 s 6077 6766',
'STATE 3 f 6077 6767',
'STATE 6 s 6769 6768',
'STATE 3 v 6072 6770',
'STATE 3 r 6108 6580',
'STATE 4 n 6108 6072',
'STATE 2 o 6108 6771',
'STATE 2 a 6077 6076',
'STATE 4 g 6773 6772',
'STATE 4 l 6077 6774',
'STATE 6 n 6776 6775',
'STATE 5 u 6778 6777',
'STATE 4 d 6077 6779',
'STATE 6 # 6076 6780',
'STATE 4 s 6077 6781',
'STATE 4 s 6077 6782',
'STATE 6 r 6784 6783',
'STATE 4 l 6132 6077',
'STATE 5 a 6786 6785',
'STATE 2 f 6108 6787',
'STATE 4 t 6076 6788',
'STATE 6 e 6790 6789',
'STATE 2 r 6076 6563',
'STATE 5 y 6072 6086',
'STATE 2 w 6077 6086',
'STATE 2 a 6086 6791',
'STATE 6 l 6107 6792',
'STATE 3 h 6107 6793',
'STATE 3 g 6086 6794',
'STATE 6 l 6796 6795',
'STATE 2 0 6072 6797',
'STATE 3 d 6107 6077',
'STATE 2 i 6107 6798',
'STATE 2 d 6107 6799',
'STATE 2 e 6801 6800',
'STATE 3 d 6108 6072',
'STATE 3 w 6803 6802',
'STATE 3 t 6077 6107',
'STATE 6 b 6077 6804',
'STATE 2 e 6108 6107',
'STATE 5 i 6077 6805',
'STATE 2 w 6072 6806',
'STATE 4 t 6808 6807',
'STATE 2 w 6072 6809',
'STATE 3 r 6077 6107',
'STATE 6 r 6072 6107',
'STATE 2 e 6107 6108',
'STATE 4 k 6072 6810',
'STATE 4 f 6108 6811',
'STATE 3 l 6072 6812',
'STATE 4 p 6072 6813',
'STATE 6 a 6077 6814',
'STATE 4 t 6077 6815',
'STATE 4 g 6107 6072',
'STATE 2 a 6108 6816',
'STATE 2 e 6072 6817',
'STATE 2 o 6108 6509',
'STATE 2 r 6077 6818',
'STATE 3 v 6072 6819',
'STATE 6 a 6107 6820',
'STATE 3 e 6086 6821',
'STATE 5 # 6069 6822',
'STATE 6 # 6824 6823',
'STATE 6 a 6076 6825',
'STATE 2 t 6076 6826',
'STATE 3 p 6076 6827',
'STATE 6 c 6069 6828',
'STATE 2 f 6069 6829',
'STATE 2 p 6069 6830',
'STATE 3 d 6107 6831',
'STATE 4 k 6833 6832',
'STATE 4 z 6072 6834',
'STATE 6 p 6072 6265',
'STATE 6 s 6836 6835',
'STATE 6 o 6108 6837',
'STATE 2 s 6077 6838',
'STATE 4 k 6840 6839',
'STATE 2 o 6108 6076',
'STATE 3 r 6072 6841',
'STATE 2 c 6108 6107',
'STATE 4 s 6842 6077',
'STATE 6 o 6077 6107',
'STATE 6 e 6072 6843',
'STATE 4 s 6077 6844',
'STATE 4 v 6072 6107',
'STATE 3 v 6846 6845',
'STATE 4 l 6108 6847',
'STATE 4 l 6849 6848',
'STATE 4 v 6723 6850',
'STATE 4 v 6723 6851',
'STATE 3 s 6853 6852',
'STATE 3 l 6077 6854',
'STATE 3 b 6132 6855',
'STATE 5 n 6857 6856',
'STATE 3 g 6077 6858',
'STATE 2 v 6086 6859',
'STATE 2 n 6076 6860',
'STATE 5 c 6077 6861',
'STATE 5 t 6077 6086',
'STATE 3 h 6086 6077',
'STATE 2 # 6077 6862',
'STATE 2 0 6107 6072',
'STATE 3 m 6864 6863',
'STATE 5 h 6866 6865',
'STATE 2 r 6077 6867',
'STATE 3 t 6069 6868',
'STATE 3 r 6077 6869',
'STATE 2 n 6107 6077',
'STATE 6 a 6077 6870',
'STATE 3 n 6108 6077',
'STATE 2 l 6107 6871',
'STATE 5 k 6872 6107',
'STATE 2 l 6107 6873',
'STATE 3 v 6077 6874',
'STATE 6 r 6876 6875',
'STATE 4 s 6072 6877',
'STATE 2 i 6108 6878',
'STATE 3 r 6880 6879',
'STATE 6 n 6077 6881',
'STATE 4 s 6072 6882',
'STATE 3 b 6072 6883',
'STATE 4 l 6885 6884',
'STATE 4 c 6072 6886',
'STATE 3 m 6077 6853',
'STATE 3 m 6072 6887',
'STATE 3 s 6072 6509',
'STATE 2 i 6077 6108',
'STATE 3 b 6072 6888',
'STATE 6 t 6077 6889',
'STATE 3 h 6069 6890',
'STATE 5 n 6069 6891',
'STATE 6 t 6154 6069',
'STATE 5 m 6154 6069',
'STATE 5 b 6076 6892',
'STATE 2 s 6076 6675',
'STATE 2 o 6069 6076',
'STATE 6 n 6069 6679',
'STATE 3 r 6072 6069',
'STATE 3 d 6069 6893',
'STATE 3 n 6107 6894',
'STATE 4 g 6896 6895',
'STATE 5 i 6076 6107',
'STATE 4 m 6077 6897',
'STATE 4 g 6899 6898',
'STATE 4 t 6076 6900',
'STATE 6 e 6902 6901',
'STATE 6 # 6904 6903',
'STATE 6 l 6906 6905',
'STATE 6 w 6069 6907',
'STATE 3 n 6107 6908',
'STATE 6 i 6909 6077',
'STATE 6 a 6077 6072',
'STATE 6 r 6077 6072',
'STATE 5 r 6911 6910',
'STATE 4 b 6072 6912',
'STATE 6 s 6914 6913',
'STATE 3 f 6072 6915',
'STATE 3 w 6072 6077',
'STATE 4 p 6069 6077',
'STATE 4 m 6916 6072',
'STATE 6 r 6918 6917',
'STATE 4 m 6077 6072',
'STATE 3 d 6920 6919',
'STATE 4 l 6108 6077',
'STATE 3 d 6922 6921',
'STATE 3 s 6924 6923',
'STATE 3 l 6077 6925',
'STATE 6 h 6086 6926',
'STATE 5 a 6076 6927',
'STATE 5 s 6077 6928',
'STATE 6 # 6077 6929',
'STATE 3 v 6077 6107',
'STATE 6 b 6107 6077',
'STATE 6 e 6931 6930',
'STATE 2 c 6933 6932',
'STATE 3 t 6077 6934',
'STATE 3 n 6925 6935',
'STATE 3 t 6077 6936',
'STATE 3 l 6077 6937',
'STATE 2 r 6107 6938',
'STATE 2 d 6107 6939',
'STATE 6 n 6940 6731',
'STATE 5 a 6942 6941',
'STATE 4 r 6944 6943',
'STATE 4 k 6107 6945',
'STATE 2 r 6108 6946',
'STATE 2 e 6108 6072',
'STATE 4 d 6948 6947',
'STATE 2 b 6077 6072',
'STATE 4 c 6072 6949',
'STATE 2 i 6950 6072',
'STATE 3 f 6072 6951',
'STATE 3 w 6072 6952',
'STATE 3 s 6072 6953',
'STATE 4 f 6072 6954',
'STATE 3 v 6072 6955',
'STATE 3 d 6957 6956',
'STATE 2 d 6072 6958',
'STATE 5 # 6069 6959',
'STATE 6 # 6072 6960',
'STATE 5 f 6076 6961',
'STATE 2 e 6069 6962',
'STATE 6 l 6107 6963',
'STATE 5 d 6965 6964',
'STATE 5 # 6967 6966',
'STATE 4 g 6077 6968',
'STATE 4 v 6970 6969',
'STATE 2 n 6077 6971',
'STATE 3 h 6108 6972',
'STATE 3 r 6077 6973',
'STATE 3 r 6077 6108',
'STATE 3 r 6975 6974',
'STATE 3 k 6107 6976',
'STATE 6 b 6978 6977',
'STATE 3 r 6108 6979',
'STATE 3 r 6108 6107',
'STATE 6 r 6069 6980',
'STATE 5 h 6107 6077',
'STATE 5 l 6982 6981',
'STATE 4 s 6077 6983',
'STATE 6 a 6985 6984',
'STATE 3 s 6077 6986',
'STATE 4 s 6077 6072',
'STATE 3 m 6988 6987',
'STATE 3 d 6108 6077',
'STATE 3 b 6108 6989',
'STATE 4 m 6132 6990',
'STATE 3 m 6992 6991',
'STATE 4 m 6077 6108',
'STATE 5 l 6107 6077',
'STATE 6 t 6077 6107',
'STATE 6 i 6077 6993',
'STATE 6 a 6077 6994',
'STATE 6 l 6069 6107',
'STATE 2 h 6072 6995',
'STATE 4 l 6072 6086',
'STATE 5 t 6077 6072',
'STATE 2 a 6086 6077',
'STATE 5 y 6108 6996',
'STATE 3 t 6108 6997',
'STATE 6 o 6108 6998',
'STATE 6 e 6069 6077',
'STATE 3 d 6077 6108',
'STATE 2 p 6108 6999',
'STATE 2 t 6077 7000',
'STATE 3 f 6077 7001',
'STATE 2 h 6909 6107',
'STATE 2 t 6077 6107',
'STATE 3 f 6107 7002',
'STATE 6 n 6077 7003',
'STATE 3 r 6107 7004',
'STATE 6 k 7006 7005',
'STATE 3 d 6086 7007',
'STATE 4 t 7009 7008',
'STATE 3 t 6108 7010',
'STATE 2 l 6107 6108',
'STATE 2 i 6132 6108',
'STATE 4 t 6072 6589',
'STATE 3 c 6072 6108',
'STATE 2 r 6076 7011',
'STATE 3 s 7013 7012',
'STATE 6 y 6072 7014',
'STATE 4 k 6072 6988',
'STATE 3 w 6072 7015',
'STATE 2 l 6072 7016',
'STATE 2 n 6072 6076',
'STATE 6 r 7018 7017',
'STATE 2 b 6069 7019',
'STATE 6 s 6072 6069',
'STATE 2 y 6076 7020',
'STATE 2 o 6069 7021',
'STATE 2 n 6077 7022',
'STATE 5 i 7024 7023',
'STATE 6 a 7026 7025',
'STATE 5 n 7028 7027',
'STATE 2 e 6107 7029',
'STATE 5 i 7031 7030',
'STATE 6 r 7033 7032',
'STATE 2 r 6072 6077',
'STATE 2 g 6107 7034',
'STATE 4 n 6076 6108',
'STATE 2 e 6108 6077',
'STATE 6 m 6107 7035',
'STATE 6 e 6107 7036',
'STATE 2 a 6107 7037',
'STATE 6 n 7039 7038',
'STATE 2 s 6086 7040',
'STATE 2 s 6108 7041',
'STATE 2 l 6107 6077',
'STATE 4 k 6442 7042',
'STATE 4 l 6077 7043',
'STATE 3 c 6077 7044',
'STATE 5 c 7045 6077',
'STATE 4 n 6077 7046',
'STATE 6 l 6107 7047',
'STATE 6 o 6077 7048',
'STATE 4 n 6072 6077',
'STATE 3 d 6988 7049',
'STATE 3 m 6072 6108',
'STATE 6 l 7051 7050',
'STATE 6 s 6072 7052',
'STATE 3 m 6077 6069',
'STATE 6 i 6107 6072',
'STATE 5 z 6072 7053',
'STATE 3 m 6076 7054',
'STATE 2 o 6108 6107',
'STATE 3 m 6107 7055',
'STATE 6 n 7056 6107',
'STATE 2 q 6077 7057',
'STATE 6 p 6077 7058',
'STATE 2 e 6107 7059',
'STATE 6 r 6072 7060',
'STATE 6 r 6107 7061',
'STATE 6 z 6076 7062',
'STATE 4 c 6107 6077',
'STATE 6 m 6072 7063',
'STATE 2 a 6108 7064',
'STATE 3 l 7066 7065',
'STATE 2 a 6108 7067',
'STATE 3 d 6069 7068',
'STATE 4 s 6077 7069',
'STATE 4 d 6072 6077',
'STATE 3 f 6072 6916',
'STATE 2 e 6132 6072',
'STATE 3 f 6072 7070',
'STATE 6 n 7072 7071',
'STATE 2 # 6077 7073',
'STATE 2 0 7074 6069',
'STATE 5 c 6076 7075',
'STATE 2 i 6069 7076',
'STATE 2 a 6107 7077',
'STATE 4 x 6107 7078',
'STATE 6 # 6076 7079',
'STATE 2 s 6077 7080',
'STATE 2 r 6108 7081',
'STATE 5 r 6108 7082',
'STATE 6 o 7084 7083',
'STATE 2 n 6107 7085',
'STATE 5 u 6108 6077',
'STATE 3 u 7087 7086',
'STATE 4 s 7089 7088',
'STATE 4 s 6072 7090',
'STATE 6 t 6108 6907',
'STATE 3 d 6107 7091',
'STATE 6 m 6077 6107',
'STATE 2 o 6107 7092',
'STATE 2 l 7094 7093',
'STATE 2 a 7096 7095',
'STATE 4 t 6108 7097',
'STATE 2 a 6108 7098',
'STATE 3 d 7100 7099',
'STATE 6 e 7102 7101',
'STATE 3 m 6077 7103',
'STATE 6 e 6077 6069',
'STATE 5 l 6069 6077',
'STATE 4 q 6501 7104',
'STATE 6 # 6072 7105',
'STATE 3 p 6072 7106',
'STATE 3 f 7108 7107',
'STATE 3 v 6072 7109',
'STATE 4 h 6108 7110',
'STATE 4 d 6072 7111',
'STATE 3 r 6107 6773',
'STATE 2 e 6107 7112',
'STATE 2 i 6108 6107',
'STATE 3 l 6077 7113',
'STATE 6 o 6077 7114',
'STATE 3 l 6107 6108',
'STATE 3 d 6107 7115',
'STATE 3 p 6107 6069',
'STATE 6 m 7117 7116',
'STATE 2 e 6072 6107',
'STATE 3 r 6072 7118',
'STATE 3 u 6086 6132',
'STATE 2 l 6077 6076',
'STATE 4 d 6072 7119',
'STATE 3 h 6069 7120',
'STATE 6 l 7122 7121',
'STATE 2 e 6076 7123',
'STATE 2 # 6072 7124',
'STATE 3 g 6077 7125',
'STATE 2 e 6077 7126',
'STATE 5 n 6069 7127',
'STATE 3 p 6076 7128',
'STATE 2 n 6069 7129',
'STATE 3 t 6107 7130',
'STATE 4 n 7132 7131',
'STATE 6 e 7134 7133',
'STATE 3 u 6077 7135',
'STATE 3 l 6076 6077',
'STATE 5 s 7137 7136',
'STATE 6 a 6161 7138',
'STATE 3 s 6069 6077',
'STATE 2 r 6107 7139',
'STATE 4 v 7141 7140',
'STATE 2 g 6072 7142',
'STATE 6 u 6108 7143',
'STATE 2 p 6077 6108',
'STATE 4 m 6076 6108',
'STATE 6 e 6107 7144',
'STATE 2 i 6107 7145',
'STATE 4 y 6108 7146',
'STATE 3 l 6108 6069',
'STATE 3 r 6108 7147',
'STATE 4 t 6108 6069',
'STATE 3 m 6108 6072',
'STATE 3 m 6108 7148',
'STATE 4 z 7150 7149',
'STATE 4 f 7151 6077',
'STATE 3 d 7152 6077',
'STATE 4 n 6077 7153',
'STATE 6 i 7155 7154',
'STATE 4 s 6077 7156',
'STATE 6 l 6077 7157',
'STATE 3 k 7159 7158',
'STATE 3 c 6077 7160',
'STATE 6 n 6108 6077',
'STATE 4 v 6072 7161',
'STATE 4 l 6069 6077',
'STATE 5 i 6076 7162',
'STATE 6 s 6107 7163',
'STATE 6 i 6077 7164',
'STATE 6 t 6077 6072',
'STATE 3 p 6107 6077',
'STATE 6 e 6108 7165',
'STATE 4 d 6108 7166',
'STATE 4 s 6072 7167',
'STATE 2 p 6072 7168',
'STATE 2 a 6069 7169',
'STATE 3 v 6077 7170',
'STATE 3 f 6108 6077',
'STATE 3 m 6108 7171',
'STATE 6 d 6072 6077',
'STATE 3 r 6077 6072',
'STATE 3 r 6072 6077',
'STATE 6 o 6069 6072',
'STATE 2 n 6076 7172',
'STATE 2 s 7173 6069',
'STATE 2 l 6107 7174',
'STATE 3 u 7176 7175',
'STATE 5 k 6077 7177',
'STATE 6 z 6108 7178',
'STATE 4 v 6107 7179',
'STATE 4 d 6077 7180',
'STATE 5 m 6077 7181',
'STATE 6 # 6077 6107',
'STATE 2 n 6077 7182',
'STATE 2 i 6107 7183',
'STATE 2 c 6072 7184',
'STATE 3 g 6077 6072',
'STATE 4 s 6086 6108',
'STATE 4 m 7186 7185',
'STATE 6 i 6107 7187',
'STATE 3 n 6077 7188',
'STATE 2 c 7190 7189',
'STATE 2 n 6108 7191',
'STATE 3 g 6108 7192',
'STATE 5 h 7194 7193',
'STATE 3 p 6069 6077',
'STATE 6 u 6108 6077',
'STATE 4 s 6077 6108',
'STATE 4 s 6077 7195',
'STATE 6 a 6132 7196',
'STATE 4 b 6072 6077',
'STATE 6 r 6077 7197',
'STATE 6 a 6077 7198',
'STATE 6 n 6077 7199',
'STATE 4 l 6077 6108',
'STATE 4 t 7201 7200',
'STATE 3 f 6072 6077',
'STATE 5 o 6108 7202',
'STATE 2 o 6107 6077',
'STATE 2 k 6077 7203',
'STATE 2 a 6108 7204',
'STATE 4 c 6108 6072',
'STATE 2 s 6072 7205',
'STATE 4 n 6072 7206',
'STATE 3 p 6069 7207',
'STATE 3 p 6077 7208',
'STATE 3 u 6072 7209',
'STATE 2 m 6076 7210',
'STATE 3 h 6069 6072',
'STATE 3 v 6107 7211',
'STATE 5 m 7213 7212',
'STATE 2 q 7215 7214',
'STATE 6 k 6077 7216',
'STATE 6 o 7218 7217',
'STATE 4 f 6108 7219',
'STATE 2 a 6077 7220',
'STATE 6 b 6108 7221',
'STATE 3 s 6072 7222',
'STATE 2 l 6107 7223',
'STATE 4 f 6069 7224',
'STATE 3 u 7226 7225',
'STATE 6 t 6069 6108',
'STATE 3 l 6077 6107',
'STATE 3 t 6077 7227',
'STATE 3 l 6108 7228',
'STATE 3 l 6077 6108',
'STATE 2 e 6108 7229',
'STATE 3 n 6108 7230',
'STATE 5 y 6550 7231',
'STATE 3 n 6069 7232',
'STATE 3 t 6072 7233',
'STATE 4 t 6072 7234',
'STATE 6 c 6108 6077',
'STATE 3 k 6077 7235',
'STATE 6 t 6077 7236',
'STATE 4 j 6072 7237',
'STATE 3 t 6072 6069',
'STATE 6 n 6072 7238',
'STATE 2 c 6077 7239',
'STATE 4 n 7241 7240',
'STATE 3 w 6072 7242',
'STATE 3 f 6072 7243',
'STATE 3 r 6076 7201',
'STATE 4 m 6077 7244',
'STATE 2 c 6076 7245',
'STATE 3 h 6076 7246',
'STATE 2 r 6107 7247',
'STATE 3 h 7249 7248',
'STATE 6 a 7250 6077',
'STATE 4 t 7252 7251',
'STATE 5 t 6077 7253',
'STATE 2 u 7255 7254',
'STATE 6 t 7257 7256',
'STATE 4 t 6077 7258',
'STATE 4 t 6108 7259',
'STATE 3 w 6077 7260',
'STATE 6 o 6077 7261',
'STATE 3 l 6072 6077',
'STATE 2 h 6107 7262',
'STATE 2 w 6072 7263',
'STATE 3 h 7265 7264',
'STATE 4 n 6108 6086',
'STATE 2 e 6077 7266',
'STATE 4 s 6077 7267',
'STATE 4 d 6108 7268',
'STATE 4 p 6108 7269',
'STATE 6 a 6077 7270',
'STATE 4 s 6077 7271',
'STATE 3 r 6072 7272',
'STATE 4 b 6108 6077',
'STATE 3 h 6077 7273',
'STATE 3 w 6077 7274',
'STATE 3 w 6072 7275',
'STATE 5 a 6072 7276',
'STATE 2 s 6077 6442',
'STATE 2 i 7096 7277',
'STATE 6 l 7279 7278',
'STATE 4 c 6108 7280',
'STATE 3 h 6108 7166',
'STATE 3 c 6077 7281',
'STATE 2 r 6076 7282',
'STATE 2 h 6076 7283',
'STATE 2 p 6107 7284',
'STATE 4 q 7286 7285',
'STATE 5 u 6108 7287',
'STATE 2 a 6108 6077',
'STATE 2 o 6069 7288',
'STATE 2 c 6108 6086',
'STATE 4 p 6077 7289',
'STATE 3 u 7291 7290',
'STATE 3 n 6077 7292',
'STATE 6 a 7294 7293',
'STATE 3 b 6077 7295',
'STATE 4 s 6077 7296',
'STATE 3 c 6077 7297',
'STATE 6 # 6072 7298',
'STATE 3 f 6077 7299',
'STATE 3 s 6107 7300',
'STATE 4 s 6072 7301',
'STATE 2 t 6072 7302',
'STATE 4 z 6072 6077',
'STATE 3 r 7303 6107',
'STATE 3 m 6108 7304',
'STATE 2 p 6108 7305',
'STATE 2 i 6072 7306',
'STATE 6 o 6077 7307',
'STATE 3 s 6072 7308',
'STATE 3 s 6072 7155',
'STATE 6 b 6077 7309',
'STATE 4 n 6069 7310',
'STATE 3 s 6072 7311',
'STATE 6 a 6072 7312',
'STATE 3 u 7314 7313',
'STATE 6 t 6108 7315',
'STATE 3 t 6108 6069',
'STATE 3 u 6072 7316',
'STATE 4 n 7318 7317',
'STATE 3 s 6076 7319',
'STATE 6 d 7321 7320',
'STATE 3 f 6077 6107',
'STATE 3 g 7323 7322',
'STATE 6 i 6077 6076',
'STATE 4 d 6108 7324',
'STATE 5 # 6108 7325',
'STATE 5 l 6077 7326',
'STATE 5 c 7328 7327',
'STATE 2 q 6077 7329',
'STATE 5 # 7330 6108',
'STATE 4 v 7332 7331',
'STATE 4 f 6108 7333',
'STATE 3 t 6077 7334',
'STATE 2 n 6069 7335',
'STATE 3 t 6108 6855',
'STATE 6 s 6072 7336',
'STATE 5 u 6076 6077',
'STATE 3 t 6107 7337',
'STATE 3 f 6072 7338',
'STATE 2 o 7340 7339',
'STATE 2 u 6107 6077',
'STATE 3 n 6108 7341',
'STATE 2 u 6108 7342',
'STATE 4 t 6072 6108',
'STATE 5 v 6077 7343',
'STATE 6 d 6108 6077',
'STATE 3 l 6077 7344',
'STATE 3 l 6077 7345',
'STATE 6 n 6108 7346',
'STATE 4 f 6072 7347',
'STATE 2 p 6072 7348',
'STATE 2 g 6072 6069',
'STATE 2 r 6108 6077',
'STATE 4 b 7350 7349',
'STATE 3 h 6072 7351',
'STATE 6 t 7353 7352',
'STATE 2 i 6108 7354',
'STATE 3 r 6076 7355',
'STATE 2 l 6076 7356',
'STATE 6 i 7358 7357',
'STATE 2 c 6077 7359',
'STATE 2 w 6077 7360',
'STATE 2 g 6077 7361',
'STATE 5 # 6077 7362',
'STATE 2 a 7364 7363',
'STATE 2 i 6108 7365',
'STATE 5 s 6108 7366',
'STATE 3 b 6077 7367',
'STATE 4 b 6108 7368',
'STATE 3 t 6107 7369',
'STATE 3 b 6076 7370',
'STATE 2 a 6077 7371',
'STATE 2 s 6077 7372',
'STATE 3 m 6072 7373',
'STATE 2 d 6107 7374',
'STATE 2 a 6108 7375',
'STATE 4 b 6108 7376',
'STATE 4 l 6072 6108',
'STATE 3 r 7378 7377',
'STATE 2 i 6108 7379',
'STATE 3 m 7380 6077',
'STATE 3 v 6077 7381',
'STATE 6 f 6077 7382',
'STATE 3 r 6072 6108',
'STATE 4 t 7384 7383',
'STATE 3 m 6069 7385',
'STATE 3 h 6077 7386',
'STATE 2 e 6077 6108',
'STATE 4 d 7388 7387',
'STATE 3 m 6072 7161',
'STATE 3 n 6072 6077',
'STATE 3 p 6072 7389',
'STATE 6 s 6076 6069',
'STATE 2 d 6076 7390',
'STATE 2 i 7392 7391',
'STATE 4 z 6076 7393',
'STATE 6 i 6077 7394',
'STATE 4 s 7395 6077',
'STATE 5 s 6108 6790',
'STATE 4 t 6108 6077',
'STATE 2 p 6077 7396',
'STATE 3 t 6108 7397',
'STATE 2 e 6077 7398',
'STATE 5 # 6108 6077',
'STATE 3 l 6077 7399',
'STATE 4 f 7401 7400',
'STATE 6 l 6077 7402',
'STATE 3 u 6076 7403',
'STATE 4 v 6077 7404',
'STATE 2 o 6077 7405',
'STATE 2 e 6077 7406',
'STATE 3 w 6077 6107',
'STATE 2 i 6108 7407',
'STATE 4 p 6077 7408',
'STATE 3 t 6108 7409',
'STATE 6 r 6108 7410',
'STATE 3 t 6108 7411',
'STATE 4 t 7412 6077',
'STATE 6 i 6072 6077',
'STATE 6 s 6108 6077',
'STATE 4 l 6072 6076',
'STATE 5 h 6076 6072',
'STATE 4 k 6072 7413',
'STATE 4 f 7415 7414',
'STATE 4 c 7416 6077',
'STATE 3 r 6108 6077',
'STATE 3 e 6076 7417',
'STATE 2 g 6076 7418',
'STATE 2 a 7420 7419',
'STATE 6 a 7422 7421',
'STATE 5 t 6077 7423',
'STATE 2 r 7424 6108',
'STATE 2 t 6077 7425',
'STATE 3 x 6108 7426',
'STATE 6 o 6108 7427',
'STATE 6 i 6077 7428',
'STATE 3 p 7429 6077',
'STATE 6 v 7431 7430',
'STATE 2 n 6077 7432',
'STATE 3 d 6107 6108',
'STATE 2 i 6077 7433',
'STATE 2 q 6108 7434',
'STATE 2 e 6076 7435',
'STATE 6 i 6072 7436',
'STATE 2 l 6108 7437',
'STATE 6 l 6108 7438',
'STATE 2 u 6069 7439',
'STATE 6 c 6077 6108',
'STATE 4 n 6108 7440',
'STATE 6 u 6069 6077',
'STATE 2 n 6108 7441',
'STATE 2 n 6077 7442',
'STATE 2 o 6077 6108',
'STATE 3 d 6077 6072',
'STATE 2 t 6072 7443',
'STATE 2 t 6076 7444',
'STATE 6 e 7446 7445',
'STATE 5 t 7448 7447',
'STATE 5 c 6077 7449',
'STATE 4 t 6107 7450',
'STATE 3 c 7452 7451',
'STATE 5 l 6077 6108',
'STATE 2 c 6077 7089',
'STATE 6 o 7454 7453',
'STATE 3 r 6077 7455',
'STATE 6 h 6077 7456',
'PHONE ae1',
'STATE 6 s 7458 7457',
'STATE 4 t 6108 6072',
'STATE 6 c 7459 6108',
'STATE 3 d 6076 6077',
'STATE 3 m 6077 7460',
'STATE 4 l 6077 6076',
'STATE 3 l 7462 7461',
'STATE 3 d 6108 7463',
'STATE 4 n 7465 7464',
'STATE 6 v 6108 7466',
'STATE 3 n 6108 7467',
'STATE 4 t 7469 7468',
'STATE 4 m 6077 7470',
'STATE 2 p 6076 7471',
'STATE 2 f 6069 6076',
'STATE 5 z 7473 7472',
'STATE 4 b 7475 7474',
'STATE 4 s 7477 7476',
'STATE 3 t 6077 7478',
'STATE 5 t 7480 7479',
'STATE 4 d 6077 7159',
'STATE 2 l 7452 7481',
'STATE 4 p 6108 6077',
'STATE 2 g 6077 7482',
'STATE 2 e 6108 7483',
'STATE 3 m 6077 7484',
'STATE 6 t 6077 7485',
'STATE 4 t 7487 7486',
'STATE 4 t 7489 7488',
'STATE 3 c 7491 7490',
'STATE 4 m 6077 7492',
'STATE 2 g 6072 7493',
'STATE 2 b 6072 6077',
'STATE 2 r 6108 7494',
'STATE 6 m 6077 7495',
'STATE 6 i 6108 7496',
'STATE 6 d 6077 7497',
'STATE 3 l 6108 7096',
'STATE 6 a 6108 7498',
'STATE 6 c 6108 7499',
'STATE 4 p 6108 7500',
'STATE 2 n 6076 6072',
'STATE 5 u 7502 7501',
'STATE 6 o 6076 6077',
'STATE 3 t 7504 7503',
'STATE 5 l 6108 6077',
'STATE 5 # 7506 7505',
'STATE 5 # 7508 7507',
'STATE 6 s 6108 7509',
'STATE 4 s 7511 7510',
'STATE 3 n 6108 7512',
'STATE 4 t 7514 7513',
'STATE 2 e 7516 7515',
'STATE 5 s 7190 6077',
'STATE 3 b 6077 6916',
'STATE 3 r 6077 7517',
'STATE 4 h 6069 7518',
'STATE 2 e 6077 7410',
'STATE 4 n 7520 7519',
'STATE 2 e 6076 6072',
'STATE 2 g 6077 6108',
'STATE 2 a 6077 6108',
'STATE 4 p 6077 7521',
'STATE 3 r 6077 7522',
'STATE 4 k 6072 7523',
'STATE 2 a 6108 7524',
'STATE 6 v 6108 7250',
'STATE 3 v 6077 7525',
'STATE 2 o 6076 7526',
'STATE 2 s 6072 6108',
'STATE 2 r 6072 7527',
'STATE 2 o 7529 7528',
'STATE 4 t 6108 7530',
'STATE 3 v 6077 7531',
'STATE 2 s 6077 7532',
'STATE 4 b 6108 7533',
'STATE 4 d 6108 7534',
'STATE 3 r 7536 7535',
'STATE 3 x 6108 6077',
'STATE 3 v 6077 7537',
'STATE 6 e 6108 7538',
'STATE 3 n 6069 6108',
'STATE 3 v 6077 6108',
'STATE 3 m 6077 7539',
'STATE 5 r 6076 6077',
'STATE 3 l 6077 7540',
'STATE 3 m 6077 7541',
'STATE 2 n 6077 6108',
'STATE 3 r 6077 7542',
'STATE 4 l 6108 7543',
'STATE 2 d 6077 7544',
'STATE 3 l 6108 7545',
'STATE 2 n 6077 7546',
'STATE 3 l 6072 7547',
'STATE 3 p 6108 7548',
'STATE 2 r 6108 7549',
'STATE 4 s 6077 7550',
'STATE 2 l 6108 7551',
'STATE 6 h 7553 7552',
'STATE 5 t 7555 7554',
'STATE 4 f 6108 7556',
'STATE 5 l 7558 7557',
'STATE 4 t 6072 7559',
'STATE 4 d 6108 7560',
'STATE 4 l 6108 7561',
'STATE 3 t 6108 7562',
'STATE 5 s 6077 6108',
'STATE 3 r 7563 6108',
'STATE 4 b 6108 7564',
'STATE 4 s 7565 6077',
'STATE 3 k 7567 7566',
'STATE 6 u 6077 7568',
'STATE 4 d 6077 7569',
'STATE 4 m 6108 6069',
'STATE 3 f 6077 6818',
'STATE 2 e 7122 6077',
'STATE 6 r 6077 7570',
'STATE 2 o 6108 7571',
'STATE 6 n 6108 7572',
'STATE 4 n 6077 7573',
'STATE 2 s 6077 7574',
'STATE 3 l 6072 7575',
'STATE 5 r 7577 7576',
'STATE 5 c 6077 7536',
'STATE 3 s 6108 7578',
'STATE 4 s 6108 6077',
'STATE 6 s 6108 7579',
'STATE 5 r 6108 7580',
'STATE 4 l 6077 7581',
'STATE 4 l 6077 7582',
'STATE 6 d 6108 7583',
'STATE 4 t 6108 6076',
'STATE 6 a 6077 6107',
'STATE 6 o 6077 6108',
'STATE 5 s 6108 7584',
'STATE 2 c 6077 7585',
'STATE 5 # 7587 7586',
'STATE 2 c 7366 6077',
'STATE 3 d 6077 7588',
'STATE 6 l 7590 7589',
'STATE 6 e 6077 6072',
'STATE 3 n 6072 7591',
'STATE 3 t 6069 7592',
'STATE 2 s 6108 7593',
'STATE 3 r 7595 7594',
'STATE 4 n 6077 6072',
'STATE 2 e 7597 7596',
'STATE 6 o 7598 6108',
'STATE 4 d 6108 7599',
'STATE 2 t 6077 7600',
'STATE 2 l 6108 7601',
'STATE 3 r 7013 7602',
'STATE 5 l 6108 6069',
'STATE 3 c 6108 7603',
'STATE 6 # 6077 7604',
'STATE 5 k 6077 6902',
'STATE 5 f 6108 7605',
'STATE 3 t 7607 7606',
'STATE 3 n 6077 7608',
'STATE 6 b 6077 7609',
'STATE 3 s 6077 6069',
'STATE 4 l 6072 7610',
'STATE 3 r 6108 7611',
'STATE 3 h 6077 6108',
'STATE 2 l 6108 7612',
'STATE 4 c 6108 7613',
'STATE 4 d 7615 7614',
'STATE 3 l 7617 7616',
'STATE 4 p 6077 6108',
'STATE 5 c 6108 7618',
'STATE 2 s 6077 6108',
'STATE 2 u 6108 7619',
'STATE 4 p 6108 6132',
'STATE 4 t 7621 7620',
'STATE 5 p 6077 6108',
'STATE 6 e 6077 7622',
'STATE 3 d 6077 7623',
'STATE 2 t 6077 7624',
'STATE 3 l 6077 7625',
'STATE 3 h 6077 7626',
'STATE 3 c 6072 7627',
'STATE 6 w 6077 6069',
'STATE 3 h 6108 7628',
'STATE 6 t 6108 6077',
'STATE 4 s 7630 7629',
'STATE 5 t 6077 7631',
'STATE 6 r 6108 7632',
'STATE 4 s 7633 6077',
'STATE 3 f 6077 7634',
'STATE 4 m 6077 7635',
'STATE 3 x 6108 7636',
'STATE 5 c 6077 7637',
'STATE 5 s 6077 7638',
'STATE 3 p 6077 7639',
'STATE 2 s 6077 7517',
'STATE 6 e 6108 6077',
'STATE 4 l 7410 7640',
'STATE 3 r 6072 7641',
'STATE 2 e 6108 7166',
'STATE 3 s 7643 7642',
'STATE 3 r 7645 7644',
'STATE 3 b 6077 7646',
'STATE 5 # 7648 7647',
'STATE 6 # 6108 7649',
'STATE 3 t 6077 7650',
'STATE 2 o 7652 7651',
'STATE 6 # 7475 7653',
'STATE 5 u 6077 7654',
'STATE 5 n 6077 7655',
'STATE 2 i 6902 7656',
'STATE 2 n 6077 7657',
'STATE 4 d 6108 6072',
'STATE 2 r 7659 7658',
'STATE 2 r 6077 6108',
'STATE 5 # 7661 7660',
'STATE 2 h 6077 7662',
'STATE 2 r 6108 7663',
'STATE 3 p 6077 7664',
'STATE 4 l 6108 7665',
'STATE 5 # 6077 6108',
'STATE 3 l 7555 7666',
'STATE 2 r 7388 7667',
'STATE 5 t 6077 6108',
'STATE 3 m 6077 7668',
'STATE 5 z 6077 6108',
'STATE 3 s 6077 7669',
'STATE 2 y 6108 7670',
'STATE 3 l 6077 7671',
'STATE 2 c 6077 7672',
'STATE 3 r 7652 7673',
'STATE 3 t 7675 7674',
'STATE 3 b 6077 7676',
'STATE 5 c 6076 6077',
'STATE 2 u 6108 7677',
'STATE 3 d 7563 7678',
'STATE 3 s 6108 7152',
'STATE 3 d 6077 7679',
'STATE 3 n 7555 6077',
'STATE 6 l 6077 7159',
'STATE 3 c 6077 7680',
'STATE 2 r 6077 7681',
'STATE 6 c 6077 7682',
'STATE 2 u 7684 7683',
'STATE 3 t 6076 7685',
'STATE 3 n 6108 7686',
'STATE 6 # 6108 7687',
'STATE 2 l 6077 7688',
'STATE 2 g 6077 7600',
'STATE 3 n 6108 7689',
'STATE 3 m 6108 7690',
'STATE 2 n 6077 7691',
'STATE 2 l 6077 7692',
'STATE 3 m 6077 6108',
'STATE 2 l 7452 7693',
'STATE 5 # 7695 7694',
'STATE 3 n 6108 7696',
'STATE 6 o 7698 7697',
'STATE 6 s 6108 6069',
'STATE 2 y 6108 7699',
'STATE 5 t 7701 7700',
'STATE 4 t 6077 7702',
'STATE 6 a 6108 6077',
'STATE 3 b 7704 7703',
'STATE 4 l 7706 7705',
'STATE 6 # 6108 6077',
'STATE 4 t 6077 6108',
'STATE 4 t 7707 6077',
'STATE 3 w 6077 7708',
'STATE 5 c 6077 6108',
'STATE 2 r 6077 7709',
'STATE 3 r 7424 7710',
'STATE 3 t 6108 7711',
'STATE 3 b 6108 6077',
'STATE 3 n 6077 7712',
'STATE 2 o 6108 7713',
'STATE 3 t 7714 6077',
'STATE 3 c 7716 7715',
'STATE 3 m 6108 6077',
'STATE 3 s 6077 7717',
'STATE 3 d 6077 7718',
'STATE 6 a 6108 7719',
'STATE 3 s 6077 7720',
'STATE 2 n 7721 6077',
'STATE 2 b 6108 6077',
'STATE 2 n 6108 6077',
'STATE 3 k 6077 7722',
'STATE 2 s 6108 7424',
'STATE 3 k 6077 7723',
'STATE 3 t 7725 7724',
'STATE 3 v 6108 6077',
'STATE 3 x 6077 7726',
'STATE 3 g 6077 6108',
'STATE 2 b 6108 7727',
'STATE 3 p 6077 7728',
'STATE 2 u 6077 7729',
'STATE 2 n 6077 7730',
'STATE 4 s 6902 6077',
'STATE 5 l 7732 7731',
'STATE 6 s 6108 7733',
'STATE 3 n 6077 7734',
'STATE 2 t 6077 6108',
'STATE 3 t 6077 7735',
'STATE 3 b 6077 7736',
'STATE 3 l 6077 7737',
'STATE 3 s 6077 7190',
'STATE 5 t 6108 6077',
'STATE 2 s 7738 6077',
'STATE 3 f 6108 7739',
'STATE 3 t 7691 6077',
'STATE 2 n 7740 6934',
'STATE 6 r 6108 6077',
'INDEX 7741 j',
'STATE 3 # 7743 7742',
'STATE 4 a 7745 7744',
'PHONE jh',
'STATE 3 n 7743 7746',
'STATE 3 o 7748 7747',
'STATE 3 d 7743 7749',
'STATE 3 a 7743 7750',
'PHONE y',
'STATE 4 i 7743 7751',
'STATE 3 e 7748 7752',
'STATE 4 o 7754 7753',
'STATE 3 u 7748 7743',
'STATE 4 e 7743 7755',
'STATE 3 s 7757 7756',
'STATE 4 u 7743 7758',
'STATE 3 b 7748 7743',
'PHONE epsilon',
'STATE 3 o 7757 7759',
'STATE 4 k 7757 7760',
'STATE 4 d 7757 7761',
'STATE 3 e 7757 7743',
'INDEX 7762 k',
'STATE 4 n 7764 7763',
'STATE 4 k 7766 7765',
'STATE 1 0 7766 7767',
'PHONE k',
'PHONE epsilon',
'STATE 1 n 7766 7765',
'INDEX 7768 l',
'STATE 4 l 7770 7769',
'STATE 4 e 7772 7771',
'STATE 5 # 7774 7773',
'STATE 6 g 7776 7775',
'STATE 3 b 7778 7777',
'STATE 6 # 7774 7779',
'PHONE epsilon',
'STATE 4 k 7781 7780',
'STATE 4 i 7783 7782',
'STATE 5 # 7785 7784',
'STATE 5 # 7787 7786',
'STATE 2 t 7774 7788',
'STATE 3 l 7790 7789',
'STATE 3 a 7792 7791',
'PHONE l',
'STATE 3 l 7782 7793',
'STATE 3 t 7795 7794',
'STATE 3 l 7782 7796',
'STATE 2 # 7782 7797',
'PHONE ah-l',
'STATE 6 k 7774 7798',
'STATE 4 m 7800 7799',
'STATE 6 0 7782 7801',
'STATE 3 o 7802 7782',
'STATE 2 w 7774 7803',
'STATE 3 b 7787 7804',
'STATE 6 # 7806 7805',
'STATE 5 y 7782 7807',
'STATE 3 a 7782 7808',
'STATE 5 s 7787 7809',
'STATE 3 a 7774 7810',
'STATE 4 # 7812 7811',
'STATE 3 a 7814 7813',
'STATE 4 u 7815 7782',
'STATE 2 f 7774 7816',
'STATE 2 t 7774 7817',
'STATE 3 g 7787 7818',
'STATE 3 d 7820 7819',
'STATE 5 y 7782 7821',
'STATE 6 s 7782 7822',
'STATE 3 i 7782 7823',
'STATE 5 d 7787 7824',
'STATE 6 l 7825 7774',
'STATE 3 u 7827 7826',
'STATE 3 d 7787 7828',
'STATE 3 o 7829 7782',
'STATE 5 # 7774 7830',
'STATE 2 e 7831 7782',
'STATE 5 # 7774 7782',
'STATE 5 # 7774 7832',
'STATE 3 t 7834 7833',
'STATE 3 g 7836 7835',
'STATE 6 s 7782 7837',
'STATE 3 l 7782 7838',
'STATE 5 r 7840 7839',
'STATE 3 o 7782 7841',
'STATE 5 y 7782 7842',
'STATE 2 c 7782 7774',
'STATE 4 f 7844 7843',
'STATE 5 # 7845 7782',
'STATE 3 t 7787 7846',
'STATE 2 c 7774 7782',
'STATE 5 a 7782 7847',
'STATE 5 m 7782 7848',
'STATE 5 a 7782 7849',
'STATE 2 z 7787 7850',
'STATE 2 r 7782 7851',
'STATE 3 p 7853 7852',
'STATE 2 # 7782 7854',
'STATE 2 d 7787 7855',
'STATE 3 i 7782 7856',
'STATE 6 g 7782 7857',
'STATE 2 i 7787 7858',
'STATE 3 e 7782 7859',
'STATE 2 a 7787 7860',
'STATE 4 v 7862 7861',
'STATE 2 h 7863 7782',
'STATE 4 t 7864 7782',
'STATE 3 g 7787 7865',
'STATE 2 c 7774 7866',
'STATE 5 l 7867 7782',
'STATE 5 o 7782 7868',
'STATE 3 p 7787 7869',
'STATE 2 s 7782 7787',
'STATE 3 k 7871 7870',
'STATE 2 p 7873 7872',
'STATE 5 c 7782 7874',
'STATE 6 a 7787 7875',
'STATE 3 a 7782 7876',
'STATE 5 t 7782 7877',
'STATE 2 s 7782 7878',
'STATE 3 y 7782 7879',
'STATE 2 u 7787 7880',
'STATE 4 n 7829 7782',
'STATE 3 a 7881 7782',
'STATE 3 a 7774 7782',
'STATE 2 a 7774 7782',
'STATE 3 a 7782 7882',
'STATE 2 b 7774 7881',
'PHONE y',
'STATE 2 f 7774 7883',
'STATE 3 d 7885 7884',
'STATE 3 z 7787 7782',
'STATE 2 # 7782 7886',
'STATE 5 c 7787 7782',
'STATE 5 m 7782 7787',
'STATE 6 s 7782 7887',
'STATE 2 n 7787 7782',
'STATE 3 e 7782 7888',
'STATE 2 n 7787 7889',
'STATE 2 u 7787 7782',
'STATE 3 u 7782 7890',
'STATE 6 s 7782 7891',
'STATE 2 h 7774 7782',
'STATE 3 e 7782 7892',
'STATE 2 h 7774 7893',
'STATE 3 f 7787 7894',
'STATE 2 d 7787 7782',
'STATE 6 s 7782 7895',
'STATE 6 o 7787 7896',
'STATE 3 o 7782 7897',
'STATE 5 s 7787 7898',
'STATE 3 r 7900 7899',
'STATE 5 t 7782 7901',
'STATE 3 i 7782 7902',
'STATE 2 b 7774 7782',
'STATE 3 s 7782 7903',
'STATE 5 b 7787 7782',
'STATE 6 a 7787 7904',
'STATE 5 s 7906 7905',
'STATE 5 d 7787 7907',
'STATE 3 w 7782 7908',
'STATE 2 a 7782 7787',
'STATE 2 m 7782 7909',
'STATE 2 c 7911 7910',
'STATE 3 c 7912 7782',
'STATE 6 e 7787 7913',
'STATE 5 d 7915 7914',
'STATE 3 y 7782 7916',
'STATE 5 m 7787 7917',
'STATE 3 s 7918 7787',
'STATE 5 m 7782 7919',
'STATE 3 u 7782 7920',
'STATE 3 h 7787 7782',
'STATE 2 # 7782 7787',
'STATE 2 n 7782 7787',
'STATE 5 r 7921 7782',
'STATE 3 u 7782 7922',
'STATE 3 u 7782 7923',
'STATE 5 b 7787 7924',
'STATE 2 i 7782 7787',
'STATE 2 b 7782 7925',
'STATE 3 o 7782 7926',
'STATE 3 s 7851 7927',
'STATE 3 w 7782 7928',
'STATE 3 r 7782 7929',
'STATE 6 e 7787 7930',
'STATE 5 r 7787 7782',
'STATE 3 h 7782 7931',
'STATE 3 g 7933 7932',
'STATE 3 r 7782 7934',
'STATE 3 w 7782 7787',
'STATE 2 t 7787 7782',
'STATE 3 r 7782 7935',
'STATE 3 d 7937 7936',
'STATE 2 a 7782 7938',
'STATE 2 n 7787 7939',
'STATE 3 y 7782 7929',
'STATE 2 c 7911 7940',
'STATE 2 n 7782 7941',
'STATE 2 e 7782 7942',
'STATE 3 d 7787 7943',
'STATE 3 f 7945 7944',
'STATE 2 d 7782 7787',
'STATE 2 n 7787 7946',
'STATE 3 k 7787 7947',
'STATE 3 p 7948 7782',
'STATE 2 f 7782 7787',
'STATE 2 i 7787 7782',
'STATE 3 g 7787 7949',
'STATE 2 p 7782 7787',
'STATE 2 f 7787 7950',
'STATE 2 z 7787 7951',
'STATE 3 p 7787 7952',
'STATE 3 c 7787 7782',
'INDEX 7953 m',
'STATE 4 m 7955 7954',
'STATE 4 c 7957 7956',
'PHONE epsilon',
'STATE 3 s 7959 7958',
'STATE 2 0 7961 7960',
'STATE 4 l 7960 7962',
'STATE 4 # 7964 7963',
'PHONE m',
'STATE 5 e 7966 7965',
'STATE 3 h 7967 7960',
'STATE 4 s 7964 7968',
'PHONE ah-m',
'STATE 5 i 7970 7969',
'STATE 6 l 7970 7971',
'STATE 2 t 7972 7960',
'STATE 5 r 7973 7960',
'PHONE m-ah',
'PHONE m-ae1',
'STATE 6 n 7970 7969',
'STATE 6 0 7964 7960',
'STATE 4 e 7974 7960',
'STATE 2 e 7960 7964',
'INDEX 7975 n',
'STATE 4 g 7977 7976',
'STATE 4 n 7979 7978',
'STATE 5 # 7981 7980',
'STATE 4 k 7983 7982',
'STATE 2 m 7985 7984',
'STATE 5 e 7987 7986',
'PHONE ng',
'STATE 4 c 7989 7988',
'STATE 6 m 7991 7990',
'STATE 6 s 7985 7992',
'PHONE epsilon',
'STATE 5 i 7994 7993',
'STATE 6 r 7996 7995',
'STATE 4 q 7998 7997',
'STATE 5 t 7981 7999',
'STATE 5 n 8001 8000',
'PHONE n',
'STATE 5 i 7985 8002',
'STATE 5 r 8004 8003',
'STATE 2 # 7991 8005',
'STATE 6 # 7991 8006',
'STATE 2 s 7991 8007',
'STATE 4 x 7981 7991',
'STATE 2 # 7991 8008',
'STATE 5 k 7981 8009',
'STATE 6 y 7981 8010',
'STATE 2 # 7991 7981',
'STATE 2 y 7985 8011',
'STATE 6 g 7991 8012',
'STATE 6 a 7991 8013',
'STATE 6 b 7991 8014',
'STATE 6 s 7991 8015',
'STATE 2 r 7981 8016',
'STATE 6 e 7981 7991',
'STATE 5 e 7991 8017',
'STATE 5 i 7981 8018',
'STATE 2 # 7985 8019',
'STATE 5 y 7991 7981',
'STATE 6 o 7991 7981',
'STATE 6 a 7991 8020',
'STATE 2 l 8022 8021',
'STATE 2 h 7981 8023',
'STATE 5 # 7981 8024',
'STATE 6 p 7991 7981',
'STATE 2 i 7985 8025',
'STATE 2 m 7981 8026',
'STATE 6 l 8028 8027',
'STATE 6 n 7981 8029',
'STATE 2 n 7981 8030',
'STATE 5 i 7991 8031',
'STATE 2 b 8032 7985',
'STATE 2 l 7981 8033',
'STATE 6 o 7991 8034',
'STATE 2 v 7991 7981',
'STATE 6 l 7981 7991',
'STATE 2 f 7981 8035',
'STATE 5 y 7991 8036',
'STATE 6 # 8037 7985',
'STATE 2 b 7981 8038',
'STATE 6 a 7991 8039',
'STATE 2 w 7981 8040',
'STATE 5 h 8042 8041',
'STATE 5 e 7991 7985',
'STATE 2 v 7981 8043',
'STATE 2 t 7991 8044',
'STATE 2 g 7991 7981',
'STATE 6 # 8046 8045',
'STATE 6 e 7991 8047',
'STATE 6 n 8048 7991',
'STATE 2 r 7991 8049',
'STATE 6 l 8051 8050',
'STATE 5 a 7981 8052',
'STATE 2 s 7981 7991',
'STATE 2 r 7981 7991',
'STATE 6 m 7981 8053',
'STATE 2 d 7981 8054',
'STATE 5 a 7991 7981',
'STATE 2 i 7991 8055',
'STATE 6 d 7991 8056',
'STATE 6 s 7991 8057',
'STATE 2 r 7991 7981',
'STATE 2 # 7991 8058',
'STATE 2 r 8059 7991',
'STATE 6 n 7981 7991',
'STATE 6 i 7991 8060',
'STATE 5 o 7981 7991',
'INDEX 8061 o',
'STATE 4 r 8063 8062',
'STATE 4 # 8065 8064',
'STATE 1 0 8067 8066',
'STATE 3 o 8069 8068',
'STATE 3 o 8071 8070',
'STATE 1 # 8073 8072',
'STATE 3 w 8075 8074',
'STATE 4 o 8077 8076',
'STATE 4 d 8079 8078',
'STATE 3 a 8081 8080',
'PHONE uw1',
'STATE 5 # 8083 8082',
'STATE 3 o 8085 8084',
'STATE 3 c 8087 8086',
'PHONE epsilon',
'STATE 4 u 8089 8088',
'STATE 3 c 8090 8075',
'STATE 4 k 8092 8091',
'STATE 2 l 8094 8093',
'STATE 3 g 8096 8095',
'PHONE aw1',
'STATE 3 w 8098 8097',
'STATE 3 c 8100 8099',
'STATE 2 s 8100 8101',
'STATE 2 d 8100 8102',
'STATE 5 r 8104 8103',
'STATE 5 r 8105 8100',
'STATE 4 n 8107 8106',
'STATE 5 s 8109 8108',
'STATE 5 r 8096 8075',
'STATE 2 f 8111 8110',
'STATE 1 s 8113 8112',
'STATE 2 f 8071 8114',
'PHONE ah1',
'STATE 3 d 8096 8115',
'PHONE ow',
'STATE 5 s 8117 8116',
'STATE 2 a 8075 8118',
'STATE 3 o 8120 8119',
'PHONE ao1',
'STATE 2 f 8122 8121',
'STATE 2 c 8100 8120',
'STATE 5 e 8124 8123',
'STATE 6 o 8126 8125',
'STATE 6 i 8100 8127',
'STATE 4 w 8129 8128',
'STATE 5 # 8131 8130',
'STATE 5 r 8133 8132',
'STATE 6 e 8135 8134',
'STATE 2 w 8137 8136',
'STATE 4 t 8120 8071',
'STATE 1 b 8138 8120',
'STATE 5 # 8120 8071',
'STATE 2 w 8120 8139',
'STATE 3 n 8096 8140',
'STATE 5 e 8142 8141',
'STATE 6 # 8075 8143',
'STATE 2 s 8144 8075',
'STATE 3 t 8075 8145',
'PHONE uh1',
'STATE 3 m 8075 8146',
'STATE 3 i 8147 8100',
'STATE 5 a 8149 8148',
'STATE 3 n 8100 8150',
'STATE 6 y 8100 8151',
'STATE 3 b 8152 8100',
'STATE 6 o 8075 8153',
'STATE 4 i 8155 8154',
'STATE 6 k 8100 8156',
'STATE 3 i 8158 8157',
'STATE 3 i 8160 8159',
'STATE 5 g 8162 8161',
'STATE 3 h 8164 8163',
'STATE 1 0 8166 8165',
'STATE 3 h 8081 8167',
'STATE 2 c 8071 8168',
'STATE 4 l 8120 8071',
'STATE 6 # 8169 8120',
'STATE 2 g 8120 8170',
'STATE 2 l 8096 8171',
'STATE 5 f 8173 8172',
'STATE 3 m 8175 8174',
'STATE 6 h 8075 8176',
'STATE 5 t 8075 8100',
'STATE 2 a 8177 8075',
'STATE 2 p 8100 8178',
'PHONE ao',
'STATE 5 o 8180 8179',
'STATE 6 i 8075 8181',
'STATE 6 # 8100 8182',
'STATE 6 e 8183 8100',
'PHONE aa1',
'STATE 6 a 8075 8184',
'STATE 4 y 8186 8185',
'STATE 5 s 8188 8187',
'STATE 5 # 8190 8189',
'STATE 5 e 8192 8191',
'STATE 2 t 8160 8193',
'STATE 3 s 8195 8194',
'PHONE ah',
'STATE 5 t 8197 8196',
'STATE 2 o 8096 8198',
'STATE 3 t 8120 8199',
'STATE 1 0 8081 8200',
'STATE 1 # 8201 8160',
'STATE 6 s 8075 8202',
'STATE 1 # 8081 8203',
'STATE 1 # 8071 8204',
'PHONE uh',
'STATE 2 h 8120 8113',
'STATE 3 t 8205 8096',
'STATE 5 t 8207 8206',
'STATE 3 d 8147 8075',
'STATE 6 d 8209 8208',
'STATE 6 # 8211 8210',
'STATE 6 t 8212 8100',
'STATE 3 d 8100 8075',
'STATE 5 o 8075 8213',
'STATE 6 g 8215 8214',
'STATE 3 b 8075 8216',
'STATE 3 m 8218 8217',
'STATE 6 a 8100 8219',
'STATE 3 f 8100 8220',
'STATE 6 u 8075 8100',
'STATE 5 e 8222 8221',
'STATE 5 o 8224 8223',
'STATE 6 g 8226 8225',
'STATE 2 # 8228 8227',
'STATE 5 n 8230 8229',
'STATE 1 # 8232 8231',
'STATE 5 g 8234 8233',
'STATE 6 # 8236 8235',
'STATE 5 e 8238 8237',
'STATE 3 t 8160 8239',
'STATE 1 a 8240 8160',
'STATE 5 n 8242 8241',
'STATE 2 0 8081 8243',
'STATE 2 # 8245 8244',
'STATE 6 t 8247 8246',
'PHONE aw',
'STATE 2 t 8081 8075',
'STATE 3 c 8075 8248',
'STATE 1 0 8081 8249',
'STATE 5 h 8250 8071',
'STATE 1 r 8096 8251',
'STATE 5 m 8253 8252',
'STATE 3 f 8255 8254',
'STATE 6 t 8075 8256',
'STATE 2 a 8075 8257',
'STATE 6 d 8075 8147',
'STATE 1 r 8147 8258',
'STATE 2 n 8100 8147',
'STATE 3 h 8100 8259',
'STATE 6 o 8261 8260',
'STATE 2 # 8100 8075',
'STATE 3 d 8100 8262',
'STATE 3 d 8147 8263',
'STATE 6 n 8147 8264',
'STATE 6 h 8100 8265',
'STATE 3 t 8100 8266',
'STATE 2 # 8268 8267',
'STATE 4 v 8270 8269',
'STATE 6 m 8096 8271',
'STATE 3 t 8096 8272',
'STATE 5 r 8274 8273',
'STATE 3 d 8071 8275',
'STATE 6 # 8277 8276',
'STATE 3 l 8278 8272',
'STATE 2 # 8280 8279',
'STATE 3 d 8081 8281',
'STATE 3 k 8283 8282',
'STATE 3 l 8275 8284',
'STATE 3 c 8286 8285',
'STATE 3 c 8288 8287',
'STATE 1 0 8290 8289',
'STATE 3 g 8100 8291',
'STATE 6 c 8152 8292',
'STATE 6 # 8275 8293',
'STATE 2 g 8160 8294',
'STATE 2 i 8295 8160',
'STATE 5 l 8297 8296',
'STATE 3 y 8075 8298',
'STATE 6 # 8081 8299',
'STATE 1 # 8301 8300',
'STATE 3 d 8303 8302',
'STATE 6 # 8305 8304',
'STATE 3 c 8100 8147',
'STATE 3 h 8081 8306',
'STATE 3 r 8081 8075',
'STATE 4 t 8071 8075',
'STATE 1 h 8275 8096',
'STATE 3 f 8308 8307',
'STATE 6 a 8309 8100',
'STATE 6 u 8075 8310',
'STATE 6 # 8075 8311',
'STATE 6 l 8147 8312',
'STATE 2 o 8075 8313',
'STATE 2 r 8100 8147',
'STATE 5 g 8100 8314',
'STATE 5 t 8316 8315',
'STATE 3 f 8318 8317',
'STATE 6 n 8075 8319',
'STATE 6 # 8100 8320',
'STATE 6 l 8100 8075',
'STATE 6 y 8100 8321',
'STATE 3 b 8147 8100',
'STATE 5 f 8323 8322',
'STATE 4 m 8325 8324',
'STATE 6 # 8327 8326',
'STATE 2 0 8329 8328',
'STATE 1 0 8272 8330',
'PHONE oy1',
'STATE 5 a 8332 8331',
'STATE 1 0 8272 8277',
'PHONE ow1',
'STATE 3 l 8272 8333',
'PHONE w',
'PHONE oy',
'STATE 1 # 8335 8334',
'STATE 6 l 8337 8336',
'STATE 3 t 8081 8338',
'STATE 3 n 8096 8339',
'STATE 1 a 8096 8200',
'STATE 3 r 8275 8081',
'STATE 1 0 8341 8340',
'STATE 6 # 8343 8342',
'STATE 1 0 8345 8344',
'STATE 6 e 8160 8346',
'STATE 1 # 8348 8347',
'STATE 6 y 8350 8349',
'STATE 3 d 8351 8275',
'STATE 1 # 8160 8352',
'STATE 6 t 8096 8160',
'STATE 2 n 8354 8353',
'PHONE aa',
'STATE 5 d 8356 8355',
'STATE 2 o 8160 8357',
'STATE 1 l 8075 8358',
'STATE 3 m 8360 8359',
'STATE 2 r 8362 8361',
'STATE 2 t 8100 8363',
'STATE 3 c 8100 8364',
'STATE 6 h 8275 8075',
'STATE 3 j 8075 8365',
'STATE 2 e 8169 8366',
'STATE 6 t 8081 8075',
'STATE 5 a 8368 8367',
'STATE 6 # 8075 8369',
'STATE 1 a 8100 8370',
'STATE 1 r 8100 8371',
'STATE 2 m 8075 8100',
'STATE 2 l 8100 8372',
'STATE 3 l 8100 8075',
'STATE 3 l 8100 8373',
'STATE 3 f 8375 8374',
'STATE 6 r 8147 8376',
'STATE 3 t 8147 8100',
'STATE 5 g 8147 8100',
'STATE 3 m 8075 8100',
'STATE 2 # 8378 8377',
'STATE 3 s 8100 8379',
'STATE 4 a 8381 8380',
'STATE 4 f 8383 8382',
'STATE 4 e 8385 8384',
'STATE 3 c 8387 8386',
'STATE 1 0 8389 8388',
'STATE 4 m 8390 8275',
'STATE 3 r 8392 8391',
'STATE 6 r 8275 8094',
'STATE 3 u 8272 8393',
'STATE 5 c 8395 8394',
'STATE 6 # 8275 8096',
'STATE 3 a 8081 8396',
'STATE 6 t 8160 8397',
'STATE 2 s 8275 8398',
'STATE 3 h 8081 8399',
'STATE 5 a 8200 8400',
'STATE 1 # 8402 8401',
'STATE 3 h 8404 8403',
'STATE 6 c 8406 8405',
'STATE 3 n 8408 8407',
'STATE 5 a 8152 8409',
'STATE 5 i 8275 8152',
'STATE 1 # 8411 8410',
'STATE 6 o 8413 8412',
'STATE 6 r 8152 8160',
'STATE 6 s 8415 8414',
'STATE 2 l 8075 8416',
'STATE 2 # 8418 8417',
'STATE 3 h 8094 8419',
'STATE 2 r 8275 8094',
'STATE 2 s 8160 8420',
'STATE 3 m 8160 8421',
'STATE 3 d 8160 8422',
'STATE 5 c 8424 8423',
'STATE 3 b 8075 8425',
'STATE 6 d 8275 8426',
'STATE 6 # 8075 8427',
'STATE 6 s 8081 8428',
'STATE 1 0 8081 8160',
'STATE 2 t 8100 8429',
'STATE 1 u 8096 8200',
'STATE 3 r 8081 8430',
'STATE 3 r 8075 8431',
'STATE 6 a 8075 8432',
'STATE 1 0 8081 8433',
'STATE 6 z 8075 8434',
'STATE 6 # 8100 8435',
'STATE 5 c 8100 8436',
'STATE 2 r 8100 8437',
'STATE 2 p 8100 8438',
'STATE 3 f 8100 8439',
'STATE 5 t 8100 8440',
'STATE 6 z 8075 8441',
'STATE 5 g 8443 8442',
'STATE 6 e 8444 8100',
'STATE 6 t 8100 8445',
'STATE 3 l 8100 8446',
'STATE 3 k 8100 8447',
'STATE 5 # 8449 8448',
'STATE 5 r 8100 8450',
'STATE 3 k 8147 8451',
'STATE 6 # 8453 8452',
'STATE 4 a 8455 8454',
'STATE 5 r 8100 8456',
'STATE 5 a 8458 8457',
'STATE 5 f 8094 8459',
'STATE 4 m 8461 8460',
'STATE 4 m 8463 8462',
'STATE 3 s 8160 8464',
'STATE 3 m 8071 8465',
'STATE 2 p 8467 8466',
'STATE 2 e 8469 8468',
'STATE 6 c 8275 8470',
'STATE 6 e 8272 8471',
'STATE 1 o 8160 8472',
'STATE 6 c 8160 8473',
'STATE 3 n 8275 8474',
'STATE 3 c 8081 8475',
'STATE 5 e 8152 8081',
'STATE 6 e 8275 8476',
'STATE 3 r 8081 8275',
'STATE 3 d 8096 8477',
'STATE 2 c 8096 8275',
'STATE 5 i 8479 8478',
'STATE 5 i 8152 8096',
'STATE 5 t 8481 8480',
'STATE 5 t 8295 8482',
'STATE 1 0 8484 8483',
'STATE 6 # 8486 8485',
'STATE 2 s 8094 8100',
'STATE 6 a 8152 8487',
'STATE 3 m 8152 8295',
'STATE 6 l 8096 8488',
'STATE 1 i 8096 8275',
'STATE 2 e 8152 8275',
'PHONE w-ah1',
'STATE 3 n 8152 8489',
'STATE 3 m 8094 8275',
'STATE 6 # 8490 8160',
'STATE 1 # 8152 8491',
'STATE 1 i 8160 8492',
'STATE 6 t 8494 8493',
'STATE 3 t 8075 8495',
'STATE 3 r 8081 8496',
'STATE 6 t 8275 8497',
'STATE 2 # 8081 8498',
'STATE 1 # 8081 8499',
'STATE 3 l 8200 8500',
'STATE 2 s 8075 8081',
'STATE 3 t 8075 8501',
'STATE 6 i 8075 8502',
'STATE 2 a 8100 8075',
'STATE 3 u 8120 8503',
'STATE 1 a 8505 8504',
'STATE 6 a 8100 8506',
'STATE 3 f 8075 8100',
'STATE 3 t 8100 8507',
'STATE 3 c 8100 8508',
'STATE 5 n 8100 8509',
'STATE 6 n 8100 8510',
'STATE 5 b 8443 8100',
'STATE 6 i 8075 8100',
'STATE 2 # 8100 8147',
'STATE 6 n 8147 8100',
'STATE 3 f 8100 8511',
'STATE 6 t 8147 8512',
'STATE 5 a 8514 8513',
'STATE 4 s 8516 8515',
'STATE 2 b 8518 8517',
'STATE 3 w 8169 8519',
'STATE 6 s 8100 8520',
'STATE 3 l 8147 8521',
'STATE 5 a 8523 8522',
'STATE 5 r 8100 8524',
'STATE 6 r 8100 8525',
'STATE 5 o 8527 8526',
'STATE 3 w 8120 8528',
'STATE 5 p 8530 8529',
'STATE 4 k 8275 8531',
'STATE 3 w 8533 8532',
'STATE 4 s 8535 8534',
'STATE 3 h 8275 8536',
'STATE 3 c 8094 8275',
'STATE 3 n 8538 8537',
'STATE 1 # 8275 8539',
'STATE 1 # 8540 8071',
'STATE 5 # 8541 8272',
'STATE 5 # 8542 8278',
'STATE 1 h 8096 8543',
'STATE 6 h 8096 8275',
'STATE 6 e 8277 8096',
'STATE 2 k 8545 8544',
'STATE 2 p 8081 8546',
'STATE 3 d 8081 8547',
'STATE 3 r 8275 8548',
'STATE 2 g 8275 8549',
'STATE 2 f 8551 8550',
'STATE 6 a 8275 8552',
'STATE 5 a 8554 8553',
'STATE 3 m 8556 8555',
'STATE 5 a 8152 8557',
'STATE 2 e 8559 8558',
'STATE 6 e 8561 8560',
'STATE 3 l 8100 8562',
'STATE 3 r 8100 8563',
'STATE 6 i 8096 8564',
'STATE 3 p 8275 8565',
'STATE 3 d 8152 8566',
'STATE 5 s 8160 8275',
'STATE 2 r 8568 8567',
'STATE 1 o 8160 8569',
'STATE 5 f 8571 8570',
'STATE 5 e 8075 8081',
'STATE 6 h 8081 8572',
'STATE 3 l 8081 8573',
'STATE 3 f 8081 8574',
'STATE 3 r 8081 8575',
'STATE 6 e 8573 8576',
'STATE 2 b 8200 8577',
'STATE 6 h 8579 8578',
'STATE 3 b 8581 8580',
'STATE 5 o 8583 8582',
'STATE 6 t 8075 8584',
'STATE 3 t 8096 8100',
'STATE 5 g 8075 8585',
'STATE 1 e 8586 8100',
'STATE 2 n 8075 8587',
'STATE 6 # 8100 8588',
'STATE 3 k 8100 8589',
'STATE 3 b 8100 8590',
'STATE 3 m 8592 8591',
'STATE 4 e 8594 8593',
'STATE 6 # 8596 8595',
'STATE 4 e 8598 8597',
'STATE 1 o 8600 8599',
'STATE 3 r 8602 8601',
'STATE 5 d 8100 8275',
'STATE 4 l 8604 8603',
'STATE 1 # 8606 8605',
'STATE 1 # 8100 8607',
'STATE 4 l 8609 8608',
'STATE 6 # 8275 8610',
'STATE 6 u 8096 8275',
'STATE 5 w 8152 8275',
'STATE 5 i 8152 8611',
'STATE 6 l 8160 8612',
'STATE 6 l 8160 8613',
'STATE 5 m 8615 8614',
'STATE 6 a 8160 8616',
'STATE 4 x 8152 8617',
'STATE 6 t 8619 8618',
'PHONE ih1',
'STATE 6 r 8621 8620',
'STATE 3 l 8152 8622',
'STATE 3 s 8094 8623',
'STATE 3 p 8152 8624',
'STATE 6 l 8152 8625',
'STATE 2 g 8275 8160',
'STATE 6 n 8275 8071',
'STATE 1 r 8278 8626',
'STATE 3 r 8278 8272',
'STATE 5 e 8277 8627',
'STATE 3 p 8081 8628',
'STATE 1 c 8152 8275',
'STATE 5 d 8081 8629',
'STATE 3 p 8081 8630',
'STATE 1 0 8081 8275',
'STATE 2 d 8632 8631',
'STATE 1 # 8634 8633',
'STATE 5 t 8635 8152',
'STATE 6 # 8275 8636',
'STATE 5 i 8638 8637',
'STATE 6 # 8275 8639',
'STATE 6 i 8295 8152',
'STATE 6 e 8152 8640',
'STATE 5 v 8295 8641',
'STATE 5 i 8275 8642',
'STATE 1 # 8152 8160',
'STATE 5 n 8644 8643',
'STATE 5 c 8160 8645',
'STATE 3 r 8100 8152',
'STATE 1 a 8100 8646',
'STATE 3 l 8100 8647',
'STATE 6 y 8649 8648',
'STATE 3 m 8152 8275',
'STATE 2 a 8651 8650',
'STATE 1 a 8652 8160',
'STATE 1 a 8160 8653',
'STATE 1 0 8654 8075',
'STATE 6 f 8275 8160',
'STATE 6 e 8075 8275',
'STATE 1 0 8081 8075',
'STATE 3 c 8081 8655',
'STATE 1 e 8081 8656',
'STATE 3 s 8081 8657',
'STATE 1 t 8075 8658',
'STATE 6 e 8081 8075',
'STATE 3 l 8081 8659',
'STATE 1 # 8075 8660',
'STATE 6 n 8662 8661',
'STATE 6 a 8664 8663',
'STATE 6 u 8075 8665',
'STATE 3 m 8666 8075',
'STATE 6 s 8075 8100',
'STATE 2 x 8100 8667',
'STATE 6 # 8100 8668',
'STATE 2 a 8670 8669',
'STATE 6 a 8317 8671',
'STATE 6 n 8147 8672',
'STATE 6 n 8674 8673',
'STATE 6 l 8075 8147',
'STATE 4 s 8676 8675',
'STATE 3 h 8678 8677',
'STATE 6 t 8680 8679',
'STATE 3 e 8152 8681',
'STATE 4 m 8683 8682',
'STATE 3 l 8275 8684',
'STATE 1 # 8096 8685',
'STATE 3 l 8160 8096',
'STATE 5 d 8275 8686',
'STATE 6 h 8275 8687',
'STATE 3 h 8275 8152',
'STATE 6 i 8275 8096',
'STATE 1 0 8689 8688',
'STATE 2 r 8100 8690',
'STATE 3 h 8692 8691',
'STATE 5 i 8694 8693',
'STATE 5 l 8696 8695',
'STATE 3 k 8698 8697',
'STATE 5 b 8699 8152',
'STATE 3 k 8096 8700',
'STATE 3 h 8275 8701',
'STATE 5 b 8703 8702',
'STATE 6 o 8152 8160',
'STATE 6 i 8160 8704',
'STATE 6 d 8706 8705',
'STATE 6 s 8464 8707',
'STATE 1 # 8708 8152',
'STATE 4 x 8152 8709',
'STATE 2 # 8710 8152',
'STATE 6 n 8275 8711',
'STATE 3 c 8713 8712',
'STATE 3 s 8275 8714',
'STATE 6 r 8275 8096',
'STATE 2 n 8278 8715',
'STATE 6 e 8272 8716',
'STATE 2 r 8718 8717',
'STATE 3 l 8720 8719',
'STATE 5 r 8100 8721',
'STATE 2 f 8275 8722',
'STATE 3 l 8275 8096',
'STATE 5 a 8724 8723',
'STATE 5 o 8726 8725',
'STATE 6 e 8094 8727',
'STATE 6 z 8729 8728',
'STATE 5 y 8275 8730',
'STATE 6 n 8732 8731',
'STATE 3 m 8152 8733',
'STATE 6 a 8096 8152',
'STATE 5 r 8295 8734',
'STATE 2 b 8152 8735',
'STATE 5 r 8152 8736',
'STATE 6 i 8160 8152',
'STATE 5 v 8160 8737',
'STATE 2 e 8100 8738',
'STATE 6 u 8094 8739',
'STATE 1 a 8293 8160',
'STATE 1 m 8275 8160',
'STATE 1 d 8160 8740',
'STATE 1 e 8160 8741',
'STATE 3 d 8100 8160',
'STATE 3 n 8160 8295',
'STATE 5 e 8152 8742',
'STATE 3 s 8075 8743',
'STATE 1 a 8081 8744',
'STATE 3 b 8075 8745',
'STATE 3 d 8075 8081',
'STATE 3 b 8081 8100',
'STATE 1 r 8147 8746',
'STATE 1 0 8747 8100',
'STATE 1 0 8100 8120',
'STATE 5 p 8749 8748',
'STATE 5 i 8100 8750',
'STATE 3 b 8075 8751',
'STATE 1 m 8075 8752',
'STATE 3 p 8100 8147',
'STATE 2 s 8100 8753',
'STATE 3 n 8100 8754',
'STATE 5 e 8100 8075',
'STATE 3 m 8756 8755',
'STATE 3 h 8147 8100',
'STATE 6 c 8147 8757',
'STATE 3 l 8147 8100',
'STATE 4 l 8759 8758',
'STATE 5 i 8761 8760',
'STATE 1 # 8763 8762',
'STATE 2 s 8071 8764',
'STATE 3 w 8120 8765',
'STATE 1 h 8075 8766',
'STATE 3 s 8275 8767',
'STATE 1 # 8769 8768',
'STATE 3 c 8770 8559',
'STATE 1 i 8275 8771',
'STATE 1 i 8096 8772',
'STATE 1 # 8275 8773',
'STATE 1 e 8096 8404',
'STATE 1 i 8160 8100',
'STATE 6 e 8160 8644',
'STATE 2 s 8152 8774',
'STATE 2 c 8100 8775',
'STATE 2 t 8100 8776',
'STATE 5 o 8778 8777',
'STATE 4 v 8780 8779',
'STATE 5 o 8782 8781',
'STATE 3 c 8784 8783',
'STATE 3 l 8786 8785',
'STATE 6 k 8275 8787',
'STATE 6 a 8789 8788',
'STATE 6 g 8096 8275',
'STATE 6 # 8275 8790',
'STATE 6 n 8160 8791',
'STATE 6 e 8275 8160',
'STATE 6 r 8152 8792',
'STATE 6 s 8794 8793',
'STATE 4 p 8160 8795',
'STATE 3 r 8797 8796',
'STATE 2 g 8160 8152',
'STATE 4 d 8799 8798',
'STATE 3 m 8801 8800',
'STATE 6 t 8160 8275',
'STATE 3 p 8152 8802',
'STATE 6 d 8160 8094',
'STATE 3 c 8804 8803',
'STATE 1 e 8272 8805',
'STATE 5 d 8806 8272',
'STATE 2 l 8808 8807',
'STATE 3 r 8096 8275',
'STATE 2 b 8081 8809',
'STATE 6 r 8081 8275',
'STATE 3 n 8081 8810',
'STATE 2 k 8200 8811',
'STATE 5 t 8813 8812',
'STATE 6 # 8275 8814',
'STATE 2 l 8075 8815',
'STATE 2 p 8160 8816',
'STATE 6 a 8160 8094',
'STATE 6 o 8275 8817',
'STATE 3 r 8075 8160',
'STATE 3 w 8819 8818',
'STATE 6 e 8275 8820',
'STATE 3 b 8096 8275',
'STATE 6 g 8152 8821',
'STATE 5 b 8152 8822',
'STATE 5 t 8824 8823',
'STATE 5 d 8826 8825',
'STATE 5 t 8160 8827',
'STATE 3 s 8100 8147',
'STATE 3 m 8094 8100',
'STATE 3 p 8152 8828',
'STATE 3 n 8160 8829',
'STATE 3 b 8075 8830',
'STATE 6 s 8081 8075',
'STATE 1 # 8081 8831',
'STATE 3 c 8075 8832',
'STATE 6 e 8075 8100',
'STATE 6 g 8169 8075',
'STATE 5 d 8834 8833',
'STATE 2 t 8147 8835',
'STATE 5 g 8100 8836',
'STATE 6 # 8100 8837',
'STATE 2 e 8100 8075',
'STATE 2 a 8100 8313',
'STATE 2 d 8100 8075',
'STATE 6 e 8839 8838',
'STATE 5 i 8840 8100',
'STATE 3 f 8100 8841',
'STATE 1 # 8843 8842',
'STATE 6 g 8845 8844',
'STATE 5 k 8847 8846',
'STATE 6 t 8849 8848',
'STATE 1 0 8275 8850',
'STATE 5 r 8100 8851',
'STATE 1 s 8275 8852',
'STATE 1 # 8854 8853',
'STATE 3 n 8160 8855',
'STATE 4 l 8856 8275',
'STATE 4 v 8858 8857',
'STATE 4 l 8160 8152',
'STATE 1 n 8152 8859',
'STATE 3 h 8275 8096',
'STATE 3 i 8861 8860',
'STATE 6 s 8275 8862',
'STATE 3 r 8152 8160',
'STATE 3 t 8147 8863',
'STATE 1 e 8100 8147',
'STATE 4 h 8865 8864',
'STATE 6 # 8275 8866',
'STATE 4 x 8152 8867',
'STATE 3 m 8071 8868',
'STATE 5 i 8870 8869',
'STATE 6 r 8094 8871',
'STATE 3 r 8873 8872',
'STATE 6 a 8160 8874',
'STATE 3 b 8876 8875',
'STATE 4 c 8275 8877',
'STATE 6 l 8160 8878',
'STATE 3 d 8160 8879',
'STATE 3 b 8295 8152',
'STATE 3 t 8881 8880',
'STATE 6 t 8275 8152',
'STATE 6 o 8160 8882',
'STATE 6 r 8884 8883',
'STATE 4 c 8152 8885',
'STATE 2 i 8160 8886',
'STATE 2 g 8152 8887',
'STATE 1 # 8275 8096',
'STATE 6 n 8889 8888',
'STATE 6 l 8891 8890',
'STATE 3 t 8152 8892',
'STATE 4 d 8152 8275',
'STATE 3 d 8893 8275',
'STATE 1 0 8895 8894',
'STATE 6 r 8094 8896',
'STATE 2 l 8897 8272',
'STATE 3 v 8272 8898',
'STATE 3 d 8900 8899',
'STATE 3 l 8901 8275',
'STATE 3 r 8902 8081',
'STATE 5 a 8904 8903',
'STATE 2 t 8906 8905',
'STATE 3 p 8152 8907',
'STATE 6 e 8566 8908',
'STATE 2 c 8152 8909',
'STATE 2 i 8075 8910',
'STATE 3 r 8275 8160',
'STATE 6 u 8275 8911',
'STATE 6 p 8160 8912',
'STATE 6 e 8094 8152',
'STATE 3 m 8152 8913',
'STATE 3 b 8152 8914',
'STATE 6 u 8295 8915',
'STATE 2 c 8152 8916',
'STATE 6 e 8160 8917',
'STATE 5 q 8152 8918',
'STATE 6 u 8160 8919',
'STATE 5 f 8160 8920',
'STATE 2 t 8922 8921',
'STATE 3 r 8160 8923',
'STATE 5 k 8081 8924',
'STATE 6 c 8081 8925',
'STATE 3 r 8075 8926',
'STATE 3 t 8928 8927',
'STATE 6 e 8100 8929',
'STATE 3 c 8100 8930',
'STATE 1 r 8100 8837',
'STATE 2 o 8075 8100',
'STATE 6 l 8931 8100',
'STATE 5 d 8100 8932',
'STATE 6 s 8100 8147',
'STATE 6 l 8147 8100',
'STATE 6 # 8934 8933',
'STATE 5 o 8936 8935',
'STATE 5 d 8938 8937',
'STATE 1 0 8152 8939',
'STATE 1 0 8941 8940',
'STATE 6 i 8081 8942',
'STATE 6 s 8275 8943',
'STATE 3 p 8944 8152',
'STATE 2 h 8275 8945',
'STATE 5 b 8275 8946',
'STATE 5 n 8160 8096',
'STATE 1 0 8948 8947',
'STATE 4 p 8950 8949',
'STATE 1 a 8160 8951',
'STATE 1 r 8275 8952',
'STATE 4 l 8954 8953',
'STATE 2 o 8152 8295',
'STATE 1 r 8152 8955',
'STATE 3 g 8096 8956',
'STATE 2 r 8096 8275',
'STATE 5 k 8275 8957',
'STATE 1 o 8147 8958',
'STATE 4 s 8960 8959',
'STATE 5 r 8100 8961',
'STATE 4 x 8152 8962',
'STATE 6 t 8152 8963',
'STATE 6 t 8160 8964',
'STATE 5 y 8966 8965',
'STATE 6 n 8968 8967',
'STATE 3 c 8893 8969',
'STATE 6 u 8160 8970',
'STATE 6 e 8275 8971',
'STATE 6 e 8160 8972',
'STATE 4 s 8974 8973',
'STATE 4 l 8275 8975',
'STATE 6 t 8096 8976',
'STATE 4 l 8160 8977',
'STATE 3 t 8071 8152',
'STATE 6 i 8096 8978',
'STATE 6 s 8096 8152',
'STATE 6 e 8160 8979',
'STATE 1 # 8981 8980',
'STATE 2 e 8160 8982',
'STATE 4 f 8160 8983',
'STATE 3 r 8160 8275',
'STATE 1 h 8152 8984',
'STATE 4 b 8986 8985',
'STATE 4 z 8100 8987',
'STATE 6 s 8566 8988',
'STATE 2 # 8152 8096',
'STATE 4 p 8160 8989',
'STATE 6 n 8152 8160',
'STATE 6 r 8991 8990',
'STATE 3 b 8275 8992',
'STATE 6 n 8094 8275',
'STATE 3 r 8272 8278',
'STATE 2 e 8272 8993',
'STATE 1 t 8275 8994',
'STATE 5 s 8096 8995',
'STATE 1 a 8996 8096',
'STATE 5 l 8081 8997',
'STATE 3 l 8275 8998',
'STATE 6 r 8275 8081',
'STATE 2 o 8096 8999',
'STATE 3 l 8096 8275',
'STATE 5 s 9001 9000',
'STATE 1 r 9003 9002',
'STATE 3 s 8160 9004',
'STATE 5 y 8275 9005',
'STATE 6 n 9007 9006',
'STATE 2 # 9009 9008',
'STATE 6 c 8152 9010',
'STATE 6 s 8275 9011',
'STATE 6 i 8295 9012',
'STATE 5 v 8160 9013',
'STATE 1 u 8160 9014',
'STATE 5 j 8152 9015',
'STATE 6 r 8152 8644',
'STATE 5 s 8160 9016',
'STATE 2 e 9018 9017',
'STATE 3 r 9019 8160',
'STATE 1 m 8160 9020',
'STATE 5 p 8075 9021',
'STATE 2 n 8081 9022',
'STATE 6 h 8075 8081',
'STATE 6 c 8100 9023',
'STATE 2 a 8100 9024',
'STATE 3 c 9025 8100',
'STATE 3 m 8100 8075',
'STATE 5 i 8075 8100',
'STATE 5 g 8100 9026',
'STATE 5 t 9028 9027',
'STATE 5 o 8275 9029',
'STATE 2 s 9031 9030',
'STATE 4 t 8275 9032',
'STATE 6 r 8094 9033',
'STATE 3 n 9035 9034',
'STATE 2 p 8152 9036',
'STATE 5 s 9038 9037',
'STATE 6 r 8152 9039',
'STATE 6 e 8152 9040',
'STATE 6 # 8275 9041',
'STATE 2 e 8152 9042',
'STATE 1 o 8275 9043',
'STATE 3 r 8275 9044',
'STATE 4 v 9046 9045',
'STATE 4 h 8295 9047',
'STATE 6 n 9049 9048',
'STATE 2 p 8152 8075',
'STATE 2 e 9051 9050',
'STATE 2 n 9052 8275',
'STATE 4 x 9054 9053',
'STATE 2 t 8295 9055',
'STATE 2 i 8295 9056',
'STATE 2 t 8096 9057',
'STATE 6 # 8275 9058',
'STATE 2 t 8147 9059',
'STATE 4 f 9061 9060',
'STATE 5 s 9063 9062',
'STATE 3 j 8152 9064',
'STATE 3 m 8275 9065',
'STATE 4 t 8275 9066',
'STATE 3 l 8094 9067',
'STATE 3 w 9069 9068',
'STATE 3 p 8152 8275',
'STATE 6 c 9071 9070',
'STATE 3 m 8096 8275',
'STATE 3 m 8160 9072',
'STATE 3 t 9074 9073',
'STATE 6 i 8152 8275',
'STATE 6 o 8160 9075',
'STATE 3 c 9077 9076',
'STATE 6 l 8275 8096',
'STATE 4 h 8152 9078',
'STATE 4 v 8094 8096',
'STATE 4 v 8160 8275',
'STATE 3 r 8096 9079',
'STATE 6 u 8160 9080',
'STATE 3 m 9082 9081',
'STATE 4 s 8152 9083',
'STATE 2 n 8152 9084',
'STATE 2 a 8160 9085',
'STATE 3 t 8160 9086',
'STATE 6 y 8275 9087',
'STATE 2 # 8275 8096',
'STATE 4 t 8275 9088',
'STATE 3 r 9090 9089',
'STATE 3 r 8152 8275',
'STATE 6 n 8275 9091',
'STATE 2 n 8094 9092',
'STATE 3 l 8094 9093',
'STATE 2 a 8272 9094',
'STATE 2 n 9096 9095',
'STATE 2 a 8096 8081',
'STATE 5 e 8096 9097',
'STATE 2 g 8275 9098',
'STATE 3 r 8275 9099',
'STATE 3 s 8096 9100',
'STATE 3 m 9102 9101',
'STATE 6 # 9104 9103',
'STATE 6 o 8160 8152',
'STATE 2 e 8295 8152',
'STATE 2 i 8096 9105',
'STATE 5 a 9107 9106',
'STATE 6 t 8160 9108',
'STATE 3 s 8160 9109',
'STATE 6 o 8295 9110',
'STATE 3 h 8152 9111',
'STATE 6 g 8152 9112',
'STATE 6 l 8152 9113',
'STATE 5 p 9115 9114',
'STATE 6 c 8152 9116',
'STATE 2 s 8160 8152',
'STATE 5 l 8152 9117',
'STATE 5 n 8160 8152',
'STATE 2 u 9119 9118',
'STATE 1 r 8295 9120',
'STATE 1 e 8295 9056',
'STATE 1 r 8152 9121',
'STATE 6 # 8075 9122',
'STATE 3 m 8081 9123',
'STATE 6 i 8100 9124',
'STATE 2 c 8075 9125',
'STATE 2 c 8100 9126',
'STATE 5 s 8100 9127',
'STATE 4 m 9129 9128',
'STATE 1 0 9131 9130',
'STATE 5 i 8275 9132',
'STATE 5 k 8152 9133',
'STATE 5 i 8275 9134',
'STATE 4 p 9136 9135',
'STATE 6 # 9138 9137',
'STATE 3 r 8160 9139',
'STATE 6 # 8096 8160',
'STATE 3 e 9141 9140',
'STATE 1 # 9143 9142',
'STATE 2 c 8100 9144',
'STATE 5 h 8152 9145',
'STATE 1 # 8100 8275',
'STATE 6 a 8275 9146',
'STATE 2 m 8152 8160',
'STATE 5 s 8096 9147',
'STATE 3 l 8275 9148',
'STATE 6 u 8160 9149',
'STATE 6 l 8071 8096',
'STATE 4 c 8096 9150',
'STATE 2 t 8152 9151',
'STATE 3 i 8096 8275',
'STATE 2 h 8096 9152',
'STATE 4 c 8275 8152',
'STATE 1 e 8275 8096',
'STATE 4 h 8096 9153',
'STATE 3 b 8152 8295',
'STATE 1 e 8295 9154',
'STATE 1 a 8160 8295',
'STATE 2 m 8275 8096',
'STATE 2 e 8096 9155',
'STATE 3 r 8147 9156',
'STATE 4 j 8272 9157',
'STATE 3 s 8100 9158',
'STATE 3 k 9160 9159',
'STATE 3 p 9162 9161',
'STATE 5 n 9163 8275',
'STATE 3 l 8275 9164',
'STATE 6 c 9166 9165',
'STATE 6 n 8096 9167',
'STATE 5 u 9169 9168',
'STATE 5 f 8120 8275',
'STATE 6 t 9171 9170',
'STATE 3 s 8160 8152',
'STATE 3 k 8160 9172',
'STATE 6 s 8275 8152',
'STATE 6 e 8152 8275',
'STATE 6 u 8160 8152',
'STATE 3 h 8275 9173',
'STATE 6 b 8096 8275',
'STATE 6 r 8152 9174',
'STATE 3 k 8096 9175',
'STATE 6 l 8160 8152',
'STATE 4 g 8160 9176',
'STATE 4 d 8152 9177',
'STATE 6 c 9179 9178',
'STATE 4 g 8100 9180',
'STATE 1 # 9182 9181',
'STATE 1 # 8096 9183',
'STATE 6 t 9185 9184',
'STATE 4 p 8275 9186',
'STATE 2 # 9187 8152',
'STATE 6 n 8275 8152',
'STATE 6 s 8275 9188',
'STATE 1 # 8094 8275',
'STATE 6 r 8094 9189',
'STATE 3 l 8272 8278',
'STATE 6 a 8081 9190',
'STATE 6 r 8096 8275',
'STATE 5 a 9191 8096',
'STATE 2 c 8275 8081',
'STATE 3 b 9193 9192',
'STATE 2 u 8096 9194',
'STATE 5 y 8160 9195',
'STATE 5 y 9196 8160',
'STATE 3 m 8160 8152',
'STATE 3 r 8160 9197',
'STATE 1 a 9198 8160',
'STATE 2 b 8152 9199',
'STATE 3 r 8275 8152',
'STATE 2 a 8160 9200',
'STATE 1 # 8275 9201',
'STATE 5 o 8096 8152',
'STATE 5 n 9203 9202',
'STATE 6 # 8275 9187',
'STATE 6 t 9204 8152',
'STATE 5 s 9206 9205',
'STATE 6 r 8152 8295',
'STATE 5 d 9075 9207',
'STATE 6 c 8152 9208',
'STATE 3 c 9210 9209',
'STATE 3 l 8160 8295',
'STATE 3 l 8160 9211',
'STATE 3 g 8160 9212',
'STATE 3 l 8075 9213',
'STATE 2 o 8081 9214',
'STATE 3 e 8075 9215',
'STATE 1 m 8075 9216',
'STATE 1 r 8100 8147',
'STATE 5 k 8100 9217',
'STATE 4 x 8152 9218',
'STATE 3 c 9220 9219',
'STATE 6 e 9222 9221',
'STATE 4 b 8160 9223',
'STATE 5 y 9225 9224',
'STATE 5 p 8152 9226',
'STATE 5 h 9227 8152',
'STATE 6 # 8275 9228',
'STATE 6 l 8096 9229',
'STATE 5 v 8152 9230',
'STATE 3 a 8081 9231',
'STATE 1 a 8275 9232',
'STATE 2 a 8152 9233',
'STATE 2 h 8160 9234',
'STATE 6 # 9236 9235',
'STATE 2 f 8100 9237',
'STATE 2 g 9239 9238',
'STATE 5 c 9240 8152',
'STATE 1 m 8096 9241',
'STATE 3 r 8096 9242',
'STATE 2 s 8275 8096',
'STATE 6 l 9244 9243',
'STATE 4 k 8096 9245',
'STATE 6 b 8152 9246',
'STATE 2 s 8096 9247',
'STATE 4 z 8096 9248',
'STATE 1 t 8152 9249',
'STATE 5 t 8275 9250',
'STATE 3 n 8147 9251',
'STATE 5 u 9253 9252',
'STATE 5 f 9255 9254',
'STATE 3 y 8275 9256',
'STATE 6 i 8152 9257',
'STATE 6 a 9259 9258',
'STATE 6 e 8160 8152',
'STATE 6 e 8275 9260',
'STATE 6 m 8275 9261',
'STATE 3 l 8160 9262',
'STATE 3 m 8096 8152',
'STATE 6 c 8152 8275',
'STATE 5 v 8152 9263',
'STATE 6 n 8152 9264',
'STATE 3 p 8096 9265',
'STATE 3 p 8160 9266',
'STATE 6 g 8152 9090',
'STATE 6 c 8152 9267',
'STATE 6 n 8152 8275',
'STATE 3 d 8096 8275',
'STATE 3 r 9269 9268',
'STATE 6 n 8152 8096',
'STATE 6 l 8160 9270',
'STATE 2 b 8096 8160',
'STATE 3 c 8160 9271',
'STATE 3 s 8160 8275',
'STATE 4 l 8275 9272',
'STATE 2 e 8160 9273',
'STATE 4 t 9275 9274',
'STATE 4 l 8096 8160',
'STATE 2 # 9276 8152',
'STATE 3 b 8152 8275',
'STATE 2 e 8094 9277',
'STATE 3 d 8094 8275',
'STATE 2 f 8081 9278',
'STATE 6 n 8081 8096',
'STATE 3 s 8275 9279',
'STATE 6 r 8081 9280',
'STATE 2 r 8096 9281',
'STATE 6 # 9283 9282',
'STATE 2 i 8275 8160',
'STATE 3 m 8160 9284',
'STATE 3 r 8096 8160',
'STATE 2 s 8152 9285',
'STATE 2 e 8160 9286',
'STATE 3 t 8275 8160',
'STATE 5 k 9227 9287',
'STATE 3 t 8094 8152',
'STATE 3 d 8096 8152',
'STATE 6 e 8295 8152',
'STATE 6 t 8295 8152',
'STATE 1 # 8152 9288',
'STATE 6 r 9290 9289',
'STATE 2 l 8160 9291',
'STATE 2 i 8152 8160',
'STATE 3 d 8160 9292',
'STATE 3 l 8152 8160',
'STATE 6 l 8075 9293',
'STATE 3 b 8081 9294',
'STATE 5 y 9296 9295',
'STATE 2 r 8100 9297',
'STATE 5 l 8100 9298',
'STATE 2 p 9300 9299',
'STATE 5 o 8160 9301',
'STATE 2 c 9303 9302',
'STATE 4 c 8152 9304',
'STATE 1 a 8160 9305',
'STATE 4 t 8152 9306',
'STATE 4 c 9308 9307',
'STATE 3 b 8152 9309',
'STATE 3 e 9311 9310',
'STATE 3 m 8094 8152',
'STATE 3 h 8275 9312',
'STATE 2 p 8160 8152',
'STATE 6 t 9314 9313',
'STATE 5 s 8160 9315',
'STATE 2 m 8275 9316',
'STATE 3 m 8152 9317',
'STATE 1 i 8160 8152',
'STATE 2 g 8152 9318',
'STATE 5 h 9320 9319',
'STATE 2 p 9322 9321',
'STATE 1 # 9324 9323',
'STATE 3 l 8100 8275',
'STATE 6 i 8152 8100',
'STATE 3 p 8275 9325',
'STATE 1 i 8275 8096',
'STATE 6 s 9327 9326',
'STATE 4 s 8275 9328',
'STATE 4 s 8096 9329',
'STATE 3 n 8152 9330',
'STATE 1 0 8096 9331',
'STATE 4 g 9333 9332',
'STATE 3 n 8147 9334',
'STATE 5 # 8275 9335',
'STATE 2 a 8100 9336',
'STATE 5 h 9338 9337',
'STATE 6 l 8152 9339',
'STATE 3 h 8152 9340',
'STATE 6 # 8100 9341',
'STATE 3 p 9343 9342',
'STATE 6 a 8275 9344',
'STATE 3 f 8152 9345',
'STATE 3 c 8152 8160',
'STATE 6 # 8152 8275',
'STATE 3 n 9347 9346',
'STATE 6 # 8275 9348',
'STATE 5 c 9350 9349',
'STATE 6 m 8152 8160',
'STATE 6 d 8152 9351',
'STATE 3 s 8152 8096',
'STATE 6 s 9353 9352',
'STATE 6 c 8096 9354',
'STATE 2 p 8160 9355',
'STATE 3 i 8293 9356',
'STATE 4 p 9358 9357',
'STATE 3 l 8275 9359',
'STATE 3 n 8160 9360',
'STATE 4 h 8096 9361',
'STATE 6 l 8096 8275',
'STATE 3 c 8275 9362',
'STATE 3 h 8094 9363',
'STATE 6 n 9365 9364',
'STATE 3 t 8720 9366',
'STATE 5 s 8081 8275',
'STATE 2 s 8906 8096',
'STATE 2 c 8152 9367',
'STATE 5 o 8275 8152',
'STATE 2 r 8160 9368',
'STATE 3 r 8152 9369',
'STATE 6 e 8160 9370',
'STATE 5 d 9372 9371',
'STATE 5 s 9373 8160',
'STATE 6 p 8160 9374',
'STATE 5 t 8152 8160',
'STATE 2 d 8160 9375',
'STATE 3 r 8160 9376',
'STATE 5 v 8075 8578',
'STATE 2 m 8081 9377',
'STATE 6 h 8075 9378',
'STATE 1 c 8100 8075',
'STATE 1 a 8147 9379',
'STATE 5 n 8100 9380',
'STATE 6 c 9382 9381',
'STATE 3 e 8075 9383',
'STATE 5 b 9385 9384',
'STATE 6 n 8094 9386',
'STATE 6 a 8094 8152',
'STATE 2 h 8152 9387',
'STATE 4 t 9388 8152',
'STATE 6 e 8152 9389',
'STATE 3 r 9391 9390',
'STATE 3 d 8160 9392',
'STATE 3 l 8160 9393',
'STATE 5 i 9395 9394',
'STATE 5 l 8096 8160',
'STATE 6 t 8160 9396',
'STATE 5 t 8275 9397',
'STATE 5 s 8275 9398',
'STATE 3 h 9400 9399',
'STATE 3 p 8096 9401',
'STATE 2 i 8152 9402',
'STATE 6 o 9404 9403',
'STATE 5 o 8275 9405',
'STATE 2 n 8152 8094',
'STATE 3 r 8152 9406',
'STATE 5 c 8096 8152',
'STATE 6 e 8160 9407',
'STATE 6 # 8100 9408',
'STATE 6 o 8275 9409',
'STATE 1 m 9411 9410',
'STATE 4 l 8160 9412',
'STATE 2 i 8160 9413',
'STATE 4 p 8096 9414',
'STATE 3 t 8275 9415',
'STATE 4 b 8096 9416',
'STATE 4 f 9418 9417',
'STATE 3 d 8100 9419',
'STATE 1 r 8147 9420',
'STATE 3 c 8275 8096',
'STATE 2 i 8100 9421',
'STATE 5 r 9423 9422',
'STATE 4 t 9425 9424',
'STATE 3 d 8152 9426',
'STATE 3 l 8152 8275',
'STATE 3 h 8100 9427',
'STATE 6 h 8100 9428',
'STATE 6 u 8152 9429',
'STATE 5 t 8152 9002',
'STATE 6 e 8152 9430',
'STATE 6 n 8275 9431',
'STATE 4 v 8152 8275',
'STATE 6 e 9433 9432',
'STATE 5 f 9434 8275',
'STATE 6 a 8295 8275',
'STATE 6 a 8275 9435',
'STATE 6 l 9437 9436',
'STATE 3 p 8160 8275',
'STATE 4 s 8160 9438',
'STATE 2 t 8275 9439',
'STATE 4 g 8160 9440',
'STATE 3 h 9442 9441',
'STATE 2 p 8152 8275',
'STATE 4 d 8096 8275',
'STATE 6 d 8160 8096',
'STATE 6 e 8152 9443',
'STATE 4 g 8152 9444',
'STATE 3 l 8094 8275',
'STATE 5 l 8081 9445',
'STATE 2 c 8081 8275',
'STATE 3 m 8275 8081',
'STATE 6 s 8275 9446',
'STATE 2 l 8160 9447',
'STATE 3 l 8152 9448',
'STATE 6 s 9449 8152',
'STATE 3 s 8100 8152',
'STATE 6 l 8152 9450',
'STATE 1 i 8152 8160',
'STATE 6 t 8152 9451',
'STATE 2 h 8160 9452',
'STATE 3 n 8152 8295',
'STATE 1 r 8081 9453',
'STATE 1 e 9455 9454',
'STATE 1 h 8075 9456',
'STATE 3 s 8100 9457',
'STATE 5 i 9459 9458',
'STATE 4 g 8152 9460',
'STATE 5 r 9462 9461',
'STATE 1 0 9464 9463',
'STATE 1 c 8094 9465',
'STATE 1 e 9467 9466',
'STATE 6 o 8152 9468',
'STATE 3 l 8152 8100',
'STATE 6 o 8295 8152',
'STATE 4 t 9470 9469',
'STATE 5 h 8147 9471',
'STATE 2 l 9259 9472',
'STATE 4 m 8160 9473',
'STATE 5 u 9475 9474',
'STATE 6 s 9477 9476',
'STATE 3 r 9479 9478',
'STATE 6 a 9481 9480',
'STATE 3 p 9482 8160',
'STATE 5 l 8816 9483',
'STATE 2 c 8152 9484',
'STATE 6 t 8096 9485',
'STATE 3 t 8152 9486',
'STATE 5 o 9212 9487',
'STATE 5 c 8160 9488',
'STATE 5 t 9490 9489',
'STATE 5 t 8603 9491',
'STATE 2 o 8152 9492',
'STATE 2 b 8152 8100',
'STATE 1 # 8275 9493',
'STATE 1 s 9495 9494',
'STATE 2 o 8160 8152',
'STATE 3 t 8096 8275',
'STATE 1 i 8160 8275',
'STATE 6 l 8152 9496',
'STATE 3 l 8275 9497',
'STATE 4 p 9499 9498',
'STATE 3 t 8152 9500',
'STATE 3 h 8160 8147',
'STATE 2 a 8147 9501',
'STATE 3 h 8147 9259',
'STATE 1 a 8100 8147',
'STATE 4 g 9503 9502',
'STATE 3 r 8295 9504',
'STATE 6 i 9506 9505',
'STATE 3 r 8100 9507',
'STATE 6 n 8096 9508',
'STATE 6 e 8152 9509',
'STATE 3 c 9511 9510',
'STATE 6 i 8100 8275',
'STATE 3 r 9513 9512',
'STATE 6 t 8816 9514',
'STATE 3 r 9516 9515',
'STATE 4 c 8275 9517',
'STATE 3 g 8152 8275',
'STATE 6 v 8152 9518',
'STATE 3 r 9520 9519',
'STATE 4 t 8275 9521',
'STATE 2 c 8275 9522',
'STATE 6 n 8160 8096',
'STATE 6 e 8096 9523',
'STATE 4 d 8152 9524',
'STATE 4 b 8275 8152',
'STATE 4 c 9526 9525',
'STATE 3 h 8152 8275',
'STATE 1 r 8081 9527',
'STATE 2 e 8152 9528',
'STATE 3 h 8152 9212',
'STATE 5 d 8160 8152',
'STATE 1 a 8152 8774',
'STATE 6 r 8152 9529',
'STATE 6 h 8152 9530',
'STATE 1 e 9531 8160',
'STATE 1 i 8081 8200',
'STATE 6 o 8100 9532',
'STATE 5 i 9534 9533',
'STATE 1 n 8075 8100',
'STATE 5 i 8100 9535',
'STATE 5 h 9537 9536',
'STATE 4 g 9539 9538',
'STATE 4 t 8152 9540',
'STATE 4 v 8295 9541',
'STATE 1 p 8275 9542',
'STATE 6 c 8152 9543',
'STATE 5 i 8096 8152',
'STATE 2 a 8100 8152',
'STATE 5 b 8160 9544',
'STATE 6 u 8160 8094',
'STATE 2 a 8152 8275',
'STATE 4 m 9546 9545',
'STATE 2 c 8100 9547',
'STATE 4 f 8100 9548',
'STATE 3 n 8160 9549',
'STATE 4 d 8160 9550',
'STATE 5 h 9552 9551',
'STATE 6 l 8152 9553',
'STATE 4 h 8096 9554',
'STATE 4 m 8152 8160',
'STATE 3 l 8160 9555',
'STATE 2 p 9556 8275',
'STATE 6 c 9558 9557',
'STATE 5 l 9107 8275',
'STATE 2 o 8152 9559',
'STATE 5 t 8275 9560',
'STATE 5 m 8275 9561',
'STATE 6 # 9562 8275',
'STATE 1 o 8152 9563',
'STATE 6 h 8160 9564',
'STATE 2 c 8160 9565',
'PHONE ih',
'STATE 2 n 8275 9566',
'STATE 5 o 8096 9567',
'STATE 3 l 8100 9568',
'STATE 3 l 8275 9569',
'STATE 4 t 9571 9570',
'STATE 4 t 8160 8152',
'STATE 6 h 8152 9572',
'STATE 4 c 8152 9573',
'STATE 1 # 9574 8160',
'STATE 1 y 8160 8096',
'STATE 2 s 9576 9575',
'STATE 3 r 8160 9577',
'STATE 5 y 9579 9578',
'STATE 3 d 9581 9580',
'STATE 3 c 8275 9582',
'STATE 3 k 8152 9583',
'STATE 4 p 8160 8152',
'STATE 3 m 9585 9584',
'STATE 4 b 9587 9586',
'STATE 3 g 8152 9588',
'STATE 5 q 8152 9589',
'STATE 6 a 8100 9590',
'STATE 6 # 8100 9591',
'STATE 6 i 8100 8152',
'STATE 3 p 9593 9592',
'STATE 6 f 9595 9594',
'STATE 4 b 8152 9596',
'STATE 3 r 8275 9187',
'STATE 3 s 8275 9597',
'STATE 6 t 8096 9598',
'STATE 4 b 8152 8275',
'STATE 4 c 8275 8160',
'STATE 4 b 8096 9599',
'STATE 6 n 9601 9600',
'STATE 1 # 9602 8275',
'STATE 6 m 9604 9603',
'STATE 6 a 8275 8100',
'STATE 6 r 8632 9605',
'STATE 5 o 9607 9606',
'STATE 3 r 8295 8152',
'STATE 6 a 9609 9608',
'STATE 3 n 8152 8160',
'STATE 1 h 8075 9610',
'STATE 6 # 8075 8100',
'STATE 3 r 8075 8100',
'STATE 5 m 8100 8317',
'STATE 5 r 9612 9611',
'STATE 4 t 9614 9613',
'STATE 6 a 8275 9615',
'STATE 3 l 8160 8275',
'STATE 4 p 8152 9616',
'STATE 4 d 8160 9617',
'STATE 1 e 8275 8160',
'STATE 2 e 8152 9618',
'STATE 2 n 9620 9619',
'STATE 4 k 8275 9621',
'STATE 5 s 8160 9622',
'STATE 3 c 9624 9623',
'STATE 4 p 8152 9625',
'STATE 2 r 8160 9626',
'STATE 4 p 8160 8275',
'STATE 5 r 9628 9627',
'STATE 4 t 9630 9629',
'STATE 6 c 8160 9631',
'STATE 4 x 8152 9632',
'STATE 2 s 8096 9633',
'STATE 4 v 8160 8096',
'STATE 1 0 9635 9634',
'STATE 3 t 8275 8152',
'STATE 2 a 8096 8160',
'STATE 3 p 9636 8275',
'STATE 1 e 8096 9637',
'STATE 3 f 8275 9638',
'STATE 2 b 8160 9639',
'STATE 3 r 9641 9640',
'STATE 5 t 8100 8152',
'STATE 3 m 8275 9642',
'STATE 5 y 8096 8100',
'STATE 6 # 9643 8971',
'STATE 6 n 8160 8816',
'STATE 6 b 8160 9644',
'STATE 1 i 8096 8152',
'STATE 4 l 8152 8096',
'STATE 6 s 8275 9645',
'STATE 4 l 8160 8096',
'STATE 3 p 8152 9646',
'STATE 3 h 8152 9647',
'STATE 2 r 8295 8152',
'STATE 5 l 9649 9648',
'STATE 4 p 8152 9650',
'STATE 5 l 9651 8152',
'STATE 6 i 8100 9652',
'STATE 6 e 8152 9653',
'STATE 4 c 9655 9654',
'STATE 3 n 8094 9656',
'STATE 6 e 8094 8100',
'STATE 6 s 8275 9657',
'STATE 3 r 8152 8096',
'STATE 3 m 8152 8100',
'STATE 6 o 9659 9658',
'STATE 5 t 9661 9660',
'STATE 3 b 8100 9662',
'STATE 3 k 8160 9663',
'STATE 4 p 8152 8160',
'STATE 6 l 9665 9664',
'STATE 3 c 8275 8152',
'STATE 4 s 8152 8096',
'STATE 6 e 8275 9444',
'STATE 4 d 9667 9666',
'STATE 2 r 8096 9668',
'STATE 3 l 8275 9669',
'STATE 2 r 8275 9107',
'STATE 3 r 8275 9670',
'STATE 3 m 8160 9671',
'STATE 4 l 8160 8275',
'STATE 3 r 8081 9672',
'STATE 6 a 9674 9673',
'STATE 6 m 8152 8346',
'STATE 6 o 9676 9675',
'STATE 5 t 8160 8295',
'STATE 2 a 8075 9677',
'STATE 4 h 9679 9678',
'STATE 6 a 9681 9680',
'STATE 1 0 8152 9682',
'STATE 6 e 9684 9683',
'STATE 4 t 9686 9685',
'STATE 1 e 8152 9687',
'STATE 5 o 8160 9688',
'STATE 5 i 9690 9689',
'STATE 6 a 8152 8160',
'STATE 1 o 8160 9691',
'STATE 4 h 9693 9692',
'STATE 5 b 8160 8152',
'STATE 5 s 9694 8160',
'STATE 1 a 8152 9695',
'STATE 5 s 8160 8152',
'STATE 2 b 8160 9696',
'STATE 4 h 8275 9697',
'STATE 4 t 8096 9698',
'STATE 3 i 8096 9699',
'STATE 3 r 9585 9700',
'STATE 6 s 8275 9701',
'STATE 3 i 8160 9702',
'STATE 4 d 8275 8160',
'STATE 6 d 8152 9703',
'STATE 5 l 8152 9704',
'STATE 5 i 8160 8275',
'STATE 1 c 8096 8275',
'STATE 3 h 8275 9705',
'STATE 1 e 8152 9706',
'STATE 3 g 8096 9707',
'STATE 2 e 8096 9708',
'STATE 3 p 8275 8100',
'STATE 3 r 8160 8152',
'STATE 4 s 8160 9709',
'STATE 6 r 8096 9710',
'STATE 2 a 8160 9711',
'STATE 4 t 8152 8160',
'STATE 5 c 9713 9712',
'STATE 4 b 9651 8152',
'STATE 4 b 9714 8275',
'STATE 6 e 8275 8152',
'STATE 5 g 8100 8152',
'STATE 6 i 8152 9715',
'STATE 3 s 8152 9716',
'STATE 6 e 8152 9717',
'STATE 6 e 9718 8152',
'STATE 4 g 8275 9719',
'STATE 5 u 8275 9720',
'STATE 3 r 8152 9721',
'STATE 5 m 9722 8152',
'STATE 6 u 8295 9723',
'STATE 3 g 8152 9724',
'STATE 6 o 8160 9725',
'STATE 3 k 9727 9726',
'STATE 3 m 8275 8152',
'STATE 4 k 8160 9728',
'STATE 6 r 8096 8160',
'STATE 2 e 8160 9729',
'STATE 3 r 8275 8771',
'STATE 2 s 8275 8160',
'STATE 2 # 9731 9730',
'STATE 6 # 8275 9732',
'STATE 5 m 8160 9733',
'STATE 3 d 8152 8160',
'STATE 5 f 8160 9734',
'STATE 5 f 8160 9735',
'STATE 6 s 8931 9736',
'STATE 5 o 9738 9737',
'STATE 5 o 8160 8275',
'STATE 1 h 9489 9739',
'STATE 2 c 8096 9740',
'STATE 4 c 9742 9741',
'STATE 6 a 8160 8152',
'STATE 2 y 8152 9743',
'STATE 6 t 9745 9744',
'STATE 3 m 8275 9746',
'STATE 4 b 9747 8160',
'STATE 4 t 8152 9748',
'STATE 3 r 8152 9162',
'STATE 3 t 8160 9749',
'STATE 6 e 8152 9234',
'STATE 2 a 8295 9750',
'STATE 3 s 8160 8096',
'STATE 3 i 8160 9751',
'STATE 2 s 8160 9752',
'STATE 2 s 8152 9753',
'STATE 2 p 9755 9754',
'STATE 2 b 8152 9756',
'STATE 2 e 8160 9757',
'STATE 2 c 8275 8152',
'STATE 4 q 8160 8096',
'STATE 3 r 9759 9758',
'STATE 5 l 9761 9760',
'STATE 5 i 9763 9762',
'STATE 2 e 8275 8415',
'STATE 2 u 8152 9764',
'STATE 6 r 8275 9765',
'STATE 2 u 8096 9766',
'STATE 1 l 8275 9767',
'STATE 3 r 8275 8096',
'STATE 1 b 8295 9768',
'STATE 4 z 8152 9769',
'STATE 4 c 9770 8152',
'STATE 6 # 8275 8160',
'STATE 6 a 9071 8160',
'STATE 4 p 8275 8152',
'STATE 6 a 8275 8152',
'STATE 3 b 8152 8094',
'STATE 4 c 8096 9771',
'STATE 3 t 8096 9772',
'STATE 3 m 8152 8126',
'STATE 6 o 8100 8295',
'STATE 6 e 8100 9773',
'STATE 3 m 8100 9513',
'STATE 3 c 8275 9774',
'STATE 4 d 9776 9775',
'STATE 6 n 8160 8275',
'STATE 6 r 8275 9777',
'STATE 1 o 8160 9778',
'STATE 6 a 8152 9779',
'STATE 3 t 8160 8275',
'STATE 5 e 8275 8096',
'STATE 3 s 8160 9780',
'STATE 6 i 8160 9781',
'STATE 5 s 8160 9782',
'STATE 3 i 8100 9783',
'STATE 5 y 9785 9784',
'STATE 4 p 9787 9786',
'STATE 1 a 8160 9788',
'STATE 2 l 8152 9789',
'STATE 1 p 8096 9790',
'STATE 1 u 8096 9791',
'STATE 2 d 8094 9792',
'STATE 6 e 9794 9793',
'STATE 4 v 8160 9795',
'STATE 3 v 8275 8160',
'STATE 1 0 8160 8275',
'STATE 6 t 8152 8275',
'STATE 2 o 8160 9796',
'STATE 3 l 8160 9797',
'STATE 3 g 8160 8152',
'STATE 2 i 8295 8152',
'STATE 1 s 8152 9798',
'STATE 3 i 8275 9799',
'STATE 5 l 9800 8152',
'STATE 4 p 8160 9801',
'STATE 2 p 8152 9802',
'STATE 2 a 8152 9167',
'STATE 6 c 8152 9803',
'STATE 6 k 8275 9804',
'STATE 3 r 8275 9805',
'STATE 5 y 8096 8275',
'STATE 6 v 8096 8152',
'STATE 1 r 8152 9411',
'STATE 6 s 8275 9806',
'STATE 2 t 8160 8275',
'STATE 4 h 8160 9807',
'STATE 1 c 8152 9808',
'STATE 6 o 8152 9809',
'STATE 6 i 8275 9810',
'STATE 4 q 8096 8275',
'STATE 5 c 8096 9811',
'STATE 6 l 8100 8152',
'STATE 4 b 8275 9812',
'STATE 4 p 8275 9813',
'STATE 6 n 8096 8160',
'STATE 4 l 9815 9814',
'STATE 2 s 8160 9816',
'STATE 6 s 8152 8275',
'STATE 5 h 8160 9817',
'STATE 5 c 8160 9818',
'STATE 5 t 8160 8152',
'STATE 1 d 8100 9819',
'STATE 5 l 9821 9820',
'STATE 4 d 8096 9822',
'STATE 1 0 9824 9823',
'STATE 6 u 8152 9825',
'STATE 6 o 9827 9826',
'STATE 1 o 8152 9828',
'STATE 3 r 8160 9829',
'STATE 3 r 9830 8703',
'STATE 1 0 8094 8160',
'STATE 4 v 8096 9831',
'STATE 1 t 8152 9832',
'STATE 4 c 8152 9833',
'STATE 2 p 8160 9834',
'STATE 1 e 8160 8152',
'STATE 1 n 8152 9835',
'STATE 5 y 8100 9836',
'STATE 4 c 8096 8152',
'STATE 6 e 8160 9837',
'STATE 6 e 8152 9838',
'STATE 2 p 9839 8275',
'STATE 5 p 8275 9840',
'STATE 1 m 8152 9841',
'STATE 1 e 8160 9842',
'STATE 2 s 8160 9843',
'STATE 3 i 8160 9844',
'STATE 5 z 9651 8152',
'STATE 6 a 8152 8275',
'STATE 3 m 9846 9845',
'STATE 4 k 9848 9847',
'STATE 4 c 8275 9849',
'STATE 6 n 8275 9850',
'STATE 6 n 8096 9851',
'STATE 4 t 9439 9852',
'STATE 2 r 8160 9853',
'STATE 6 l 8152 8160',
'STATE 1 o 9855 9854',
'STATE 4 b 9857 9856',
'STATE 3 r 9859 9858',
'STATE 4 t 8160 9860',
'STATE 1 o 8152 9861',
'STATE 4 d 9863 9862',
'STATE 3 n 8152 9864',
'STATE 1 0 8152 9865',
'STATE 1 i 8096 9866',
'STATE 3 n 9868 9867',
'STATE 2 s 8152 8160',
'STATE 6 e 9869 8275',
'STATE 3 m 9871 9870',
'STATE 4 d 8152 9872',
'STATE 4 d 8152 8096',
'STATE 3 d 8160 8096',
'STATE 3 l 8152 9873',
'STATE 4 m 9389 9874',
'STATE 4 g 8275 8160',
'STATE 2 b 8152 9875',
'STATE 4 f 8152 9876',
'STATE 2 o 8160 9877',
'STATE 3 c 8160 9878',
'STATE 2 i 8096 9879',
'STATE 3 c 8160 9880',
'STATE 2 d 8152 9881',
'STATE 6 i 9882 8152',
'STATE 5 t 8152 9651',
'STATE 6 s 8275 9883',
'STATE 3 s 8152 8275',
'STATE 3 s 8096 9884',
'STATE 6 b 8275 9885',
'STATE 6 k 8275 8152',
'STATE 1 i 8160 9886',
'STATE 3 t 8160 8152',
'STATE 5 n 9888 9887',
'STATE 6 # 8100 8075',
'STATE 5 u 9890 9889',
'STATE 6 e 8160 9891',
'STATE 6 e 9893 9892',
'STATE 4 p 8160 8096',
'STATE 4 g 8160 9894',
'STATE 1 r 8160 9895',
'STATE 6 n 8096 8275',
'STATE 6 r 8275 8152',
'STATE 2 h 8160 9896',
'STATE 1 e 8096 8160',
'STATE 1 u 8096 8160',
'STATE 1 a 8152 9897',
'STATE 2 a 8152 8160',
'STATE 1 e 8160 8096',
'STATE 4 k 8275 9898',
'STATE 2 o 8096 8275',
'STATE 1 0 8275 8096',
'STATE 2 y 8152 9899',
'STATE 5 l 9162 8152',
'STATE 4 c 8152 8275',
'STATE 6 n 8152 9900',
'STATE 6 s 9902 9901',
'STATE 1 o 8160 9903',
'STATE 1 i 8096 9904',
'STATE 1 c 9906 9905',
'STATE 3 s 8152 9907',
'STATE 5 h 8275 8152',
'STATE 6 l 8718 9908',
'STATE 3 t 8160 9909',
'STATE 3 m 8096 9910',
'STATE 4 c 8096 9911',
'STATE 6 # 8100 9912',
'STATE 2 t 8075 9913',
'STATE 1 0 9915 9914',
'STATE 4 g 9917 9916',
'STATE 5 s 9918 8152',
'STATE 1 0 8644 9919',
'STATE 4 g 8275 9920',
'STATE 4 c 8160 8096',
'STATE 1 l 8275 9921',
'STATE 6 l 9797 8160',
'STATE 3 l 8160 9922',
'STATE 2 e 8275 9923',
'STATE 1 e 8152 9259',
'STATE 4 v 8160 8152',
'STATE 3 t 8096 9924',
'STATE 2 m 8160 9925',
'STATE 2 s 8152 9444',
'STATE 6 a 8100 9926',
'STATE 6 c 8096 9927',
'STATE 2 t 8096 8275',
'STATE 3 b 8295 9928',
'STATE 6 r 8275 8774',
'STATE 3 j 8096 8275',
'STATE 6 m 8096 8275',
'STATE 2 i 8160 8293',
'STATE 2 r 8075 9929',
'STATE 1 n 8100 9930',
'STATE 2 e 9932 9931',
'STATE 6 r 8160 9933',
'STATE 1 0 8096 9934',
'STATE 3 l 8100 8152',
'STATE 6 t 8160 8152',
'STATE 6 i 8275 8160',
'STATE 3 n 8275 8160',
'STATE 4 c 8096 9935',
'STATE 3 m 9937 9936',
'STATE 1 i 8096 9938',
'STATE 5 o 8160 9939',
'STATE 2 a 9941 9940',
'STATE 2 o 8152 9942',
'STATE 3 r 9943 8096',
'STATE 2 g 8160 9944',
'STATE 2 b 8100 9945',
'STATE 2 n 8100 9946',
'STATE 6 o 9521 9947',
'STATE 1 r 8160 8152',
'STATE 5 c 9683 8152',
'STATE 3 p 8152 9948',
'STATE 6 s 8096 9949',
'STATE 3 t 9951 9950',
'STATE 4 c 8160 8152',
'STATE 6 o 9953 9952',
'STATE 6 z 8160 9954',
'STATE 2 e 8152 8160',
'STATE 3 b 8152 8160',
'STATE 3 c 8152 8275',
'STATE 2 t 8096 9955',
'STATE 2 e 8160 9956',
'STATE 2 t 8100 9957',
'STATE 1 l 8100 9958',
'STATE 4 f 8100 9959',
'STATE 4 c 9918 9198',
'STATE 6 r 8160 9960',
'STATE 4 g 8160 9643',
'STATE 4 c 8160 9961',
'STATE 2 t 8160 9962',
'STATE 4 b 8160 8275',
'STATE 1 c 8160 9963',
'STATE 6 n 9964 8096',
'STATE 2 h 8160 9965',
'STATE 6 n 8100 9966',
'STATE 1 i 8075 9967',
'STATE 6 s 8152 9968',
'STATE 1 m 8160 9969',
'STATE 1 m 8152 8160',
'STATE 3 t 8096 9970',
'STATE 3 v 8152 9971',
'STATE 2 c 8096 8160',
'STATE 2 r 8160 9972',
'STATE 1 m 8100 8075',
'STATE 2 y 8075 9973',
'STATE 1 a 9975 9974',
'STATE 2 e 8160 9976',
'STATE 1 e 8160 9977',
'STATE 6 n 9979 9978',
'STATE 4 d 8152 9980',
'STATE 1 a 8075 9981',
'STATE 3 n 8152 9982',
'STATE 2 l 8152 9983',
'STATE 4 b 8096 9984',
'STATE 1 0 8096 9985',
'STATE 2 n 8160 9986',
'STATE 3 r 8275 9987',
'STATE 2 l 8160 9988',
'STATE 3 h 8100 8075',
'STATE 2 o 9989 8152',
'STATE 6 i 8275 8152',
'STATE 4 d 9198 9990',
'STATE 6 n 8160 9991',
'STATE 2 p 8096 8275',
'STATE 3 i 8160 9992',
'STATE 1 p 8160 9993',
'STATE 5 n 8275 8152',
'STATE 3 r 8160 9994',
'STATE 2 h 8096 8160',
'STATE 1 # 8160 8096',
'STATE 3 r 8160 9995',
'STATE 3 t 8096 8160',
'STATE 1 g 8160 9996',
'STATE 3 l 8152 9997',
'STATE 3 n 8295 8160',
'INDEX 9998 p',
'STATE 4 p 10000 9999',
'STATE 4 h 10002 10001',
'PHONE epsilon',
'STATE 4 f 10004 10003',
'STATE 2 # 10006 10005',
'STATE 4 s 10008 10007',
'STATE 2 0 10000 10009',
'STATE 2 h 10011 10010',
'STATE 3 u 10009 10010',
'STATE 3 p 10009 10012',
'STATE 2 0 10000 10013',
'PHONE p',
'PHONE f',
'STATE 3 e 10009 10010',
'STATE 4 t 10004 10014',
'STATE 2 o 10015 10009',
'STATE 4 b 10017 10016',
'STATE 3 r 10000 10009',
'STATE 4 n 10004 10009',
'STATE 2 a 10000 10009',
'INDEX 10018 q',
'PHONE k',
'INDEX 10019 r',
'STATE 3 e 10021 10020',
'STATE 4 r 10023 10022',
'STATE 1 # 10025 10024',
'STATE 1 0 10027 10026',
'STATE 2 c 10029 10028',
'STATE 4 # 10031 10030',
'STATE 4 r 10033 10032',
'STATE 3 u 10035 10034',
'STATE 3 u 10037 10036',
'STATE 1 0 10029 10038',
'PHONE epsilon',
'STATE 4 r 10040 10039',
'STATE 2 e 10042 10041',
'STATE 4 i 10044 10043',
'STATE 2 p 10046 10045',
'STATE 3 o 10048 10047',
'STATE 1 # 10050 10049',
'STATE 3 i 10052 10051',
'STATE 4 i 10029 10053',
'STATE 2 b 10029 10054',
'STATE 4 i 10056 10055',
'STATE 2 v 10058 10057',
'STATE 2 i 10058 10059',
'PHONE r',
'STATE 4 e 10061 10060',
'STATE 2 p 10063 10062',
'STATE 2 t 10029 10064',
'STATE 5 i 10058 10065',
'STATE 3 i 10067 10066',
'STATE 2 w 10069 10068',
'STATE 2 o 10071 10070',
'STATE 4 i 10073 10072',
'STATE 3 a 10075 10074',
'STATE 4 o 10058 10076',
'STATE 4 a 10042 10077',
'STATE 3 a 10079 10078',
'STATE 2 i 10081 10080',
'STATE 5 n 10083 10082',
'STATE 1 n 10085 10084',
'PHONE er',
'STATE 2 u 10058 10086',
'STATE 4 o 10088 10087',
'STATE 2 p 10058 10089',
'STATE 5 v 10058 10090',
'STATE 5 o 10042 10091',
'STATE 2 g 10042 10092',
'STATE 5 e 10058 10093',
'STATE 3 r 10095 10094',
'STATE 2 a 10042 10096',
'STATE 1 # 10098 10097',
'STATE 1 # 10093 10099',
'STATE 2 a 10101 10100',
'STATE 1 j 10093 10102',
'STATE 4 o 10042 10103',
'STATE 2 k 10058 10042',
'STATE 4 z 10105 10104',
'STATE 4 o 10107 10106',
'STATE 5 i 10093 10108',
'STATE 4 b 10058 10109',
'STATE 2 w 10029 10110',
'STATE 2 g 10029 10111',
'STATE 5 # 10113 10112',
'STATE 4 s 10114 10042',
'STATE 5 z 10058 10115',
'STATE 2 e 10042 10116',
'STATE 2 h 10117 10029',
'STATE 5 e 10058 10029',
'STATE 2 a 10042 10058',
'STATE 4 a 10119 10118',
'STATE 2 p 10121 10120',
'STATE 2 t 10058 10122',
'STATE 5 s 10124 10123',
'STATE 5 s 10042 10058',
'STATE 2 m 10029 10125',
'PHONE er1',
'STATE 3 a 10127 10126',
'STATE 2 u 10129 10128',
'STATE 4 e 10131 10130',
'STATE 4 # 10133 10132',
'STATE 4 o 10135 10134',
'STATE 4 k 10093 10136',
'STATE 4 e 10138 10137',
'STATE 1 t 10058 10042',
'STATE 5 s 10140 10139',
'STATE 4 a 10042 10141',
'STATE 3 o 10142 10042',
'STATE 3 g 10093 10143',
'STATE 5 s 10144 10042',
'STATE 5 u 10058 10042',
'STATE 4 a 10042 10145',
'STATE 4 e 10029 10146',
'STATE 2 d 10148 10147',
'STATE 1 # 10150 10149',
'STATE 1 0 10152 10151',
'STATE 4 a 10154 10153',
'STATE 5 # 10058 10042',
'STATE 5 e 10156 10155',
'STATE 1 t 10058 10157',
'STATE 1 s 10042 10029',
'STATE 2 p 10159 10158',
'STATE 5 # 10042 10160',
'STATE 5 u 10058 10161',
'STATE 5 n 10042 10058',
'STATE 5 s 10042 10162',
'STATE 2 d 10042 10163',
'STATE 2 d 10058 10042',
'STATE 2 h 10029 10164',
'STATE 5 # 10166 10165',
'STATE 2 e 10168 10167',
'STATE 1 c 10170 10169',
'STATE 1 s 10058 10171',
'STATE 4 a 10173 10172',
'STATE 2 m 10042 10174',
'STATE 4 s 10176 10175',
'STATE 2 c 10042 10177',
'STATE 4 a 10179 10178',
'STATE 2 b 10180 10042',
'STATE 4 t 10182 10181',
'STATE 4 a 10184 10183',
'STATE 2 t 10058 10185',
'STATE 1 c 10187 10186',
'STATE 1 n 10093 10042',
'STATE 4 e 10042 10188',
'STATE 5 g 10144 10042',
'STATE 5 y 10058 10042',
'STATE 4 i 10058 10042',
'STATE 4 i 10042 10189',
'STATE 5 u 10058 10093',
'STATE 3 i 10190 10029',
'STATE 5 a 10042 10029',
'STATE 5 a 10029 10191',
'STATE 2 f 10029 10192',
'STATE 5 i 10194 10193',
'STATE 4 a 10042 10195',
'STATE 4 o 10042 10196',
'STATE 2 h 10058 10042',
'STATE 5 # 10042 10197',
'STATE 2 p 10042 10058',
'STATE 1 n 10058 10198',
'STATE 4 u 10042 10199',
'STATE 4 u 10058 10200',
'STATE 2 d 10042 10201',
'STATE 2 f 10058 10042',
'STATE 2 j 10042 10202',
'STATE 5 d 10058 10042',
'STATE 2 c 10029 10203',
'STATE 4 z 10058 10204',
'STATE 4 e 10205 10042',
'STATE 1 # 10207 10206',
'STATE 4 n 10209 10208',
'STATE 1 # 10211 10210',
'STATE 2 a 10042 10212',
'STATE 5 c 10058 10213',
'STATE 4 o 10215 10214',
'STATE 5 # 10042 10216',
'STATE 5 c 10058 10217',
'STATE 4 d 10219 10218',
'STATE 5 # 10221 10220',
'STATE 2 o 10042 10222',
'STATE 2 c 10224 10223',
'STATE 5 i 10058 10042',
'STATE 5 w 10058 10093',
'STATE 1 s 10042 10225',
'STATE 1 s 10058 10226',
'STATE 4 g 10228 10227',
'STATE 5 # 10042 10229',
'STATE 2 s 10231 10230',
'STATE 1 f 10042 10232',
'STATE 5 g 10093 10233',
'STATE 2 s 10235 10234',
'STATE 4 e 10042 10093',
'STATE 5 e 10029 10236',
'STATE 2 m 10029 10237',
'STATE 2 t 10029 10238',
'STATE 2 e 10042 10239',
'STATE 4 s 10241 10240',
'STATE 4 o 10042 10242',
'STATE 4 e 10042 10243',
'STATE 5 o 10042 10244',
'STATE 2 f 10058 10245',
'STATE 4 y 10042 10246',
'STATE 4 l 10093 10247',
'STATE 5 s 10042 10248',
'STATE 2 d 10042 10249',
'STATE 2 j 10029 10250',
'STATE 3 y 10252 10251',
'STATE 3 y 10042 10253',
'STATE 4 # 10255 10254',
'STATE 4 o 10257 10256',
'STATE 5 h 10093 10258',
'STATE 5 # 10259 10093',
'STATE 2 a 10261 10260',
'STATE 2 a 10262 10042',
'STATE 4 i 10042 10263',
'STATE 1 d 10029 10264',
'STATE 4 i 10266 10265',
'STATE 5 n 10267 10042',
'STATE 1 d 10058 10268',
'STATE 2 o 10042 10269',
'STATE 4 a 10271 10270',
'STATE 2 f 10272 10042',
'STATE 5 h 10058 10042',
'STATE 2 o 10042 10058',
'STATE 2 t 10058 10273',
'STATE 4 t 10042 10274',
'STATE 4 s 10275 10042',
'STATE 1 r 10093 10276',
'STATE 1 t 10093 10277',
'STATE 4 i 10279 10278',
'STATE 2 b 10058 10093',
'STATE 5 l 10058 10280',
'STATE 1 i 10029 10281',
'STATE 1 n 10042 10282',
'STATE 5 e 10042 10283',
'STATE 5 e 10284 10042',
'STATE 4 y 10042 10285',
'STATE 4 v 10058 10286',
'STATE 1 # 10042 10029',
'STATE 1 c 10029 10287',
'STATE 5 # 10042 10288',
'STATE 2 d 10058 10289',
'STATE 2 s 10093 10290',
'STATE 1 o 10058 10093',
'STATE 4 u 10042 10291',
'STATE 2 e 10042 10292',
'STATE 5 a 10294 10293',
'STATE 2 h 10058 10295',
'STATE 5 a 10297 10296',
'STATE 4 v 10058 10298',
'STATE 2 p 10058 10299',
'STATE 2 b 10121 10042',
'STATE 2 k 10029 10300',
'STATE 4 o 10042 10301',
'STATE 4 n 10093 10302',
'STATE 3 c 10058 10303',
'STATE 4 d 10305 10304',
'STATE 2 l 10058 10306',
'STATE 4 a 10308 10307',
'STATE 5 # 10042 10309',
'STATE 4 l 10093 10310',
'STATE 1 h 10058 10093',
'STATE 1 f 10312 10311',
'STATE 1 b 10042 10313',
'STATE 4 o 10042 10314',
'STATE 5 # 10042 10315',
'STATE 5 l 10042 10316',
'STATE 4 # 10042 10317',
'STATE 2 p 10042 10318',
'STATE 2 v 10042 10319',
'STATE 5 t 10319 10320',
'STATE 5 d 10058 10321',
'STATE 5 z 10058 10322',
'STATE 5 # 10154 10323',
'STATE 5 # 10058 10324',
'STATE 1 a 10325 10058',
'STATE 5 r 10042 10326',
'STATE 5 e 10058 10042',
'STATE 5 # 10327 10093',
'STATE 1 e 10093 10328',
'STATE 4 y 10330 10329',
'STATE 1 i 10029 10331',
'STATE 5 t 10333 10332',
'STATE 2 e 10093 10334',
'STATE 1 s 10042 10058',
'STATE 5 o 10042 10335',
'STATE 4 s 10042 10093',
'STATE 4 u 10042 10336',
'STATE 5 e 10093 10337',
'STATE 2 h 10029 10338',
'STATE 2 j 10029 10339',
'STATE 5 e 10341 10340',
'STATE 4 m 10343 10342',
'STATE 5 c 10042 10093',
'STATE 4 s 10058 10344',
'STATE 1 0 10042 10345',
'STATE 2 l 10058 10042',
'STATE 2 i 10042 10058',
'STATE 5 o 10347 10346',
'STATE 4 d 10042 10348',
'STATE 4 f 10058 10349',
'STATE 5 l 10042 10350',
'STATE 5 e 10352 10351',
'STATE 4 a 10354 10353',
'STATE 4 t 10093 10355',
'STATE 1 f 10058 10356',
'STATE 5 t 10358 10357',
'STATE 5 # 10360 10359',
'STATE 2 g 10058 10361',
'STATE 4 i 10363 10362',
'STATE 2 h 10324 10364',
'STATE 2 p 10366 10365',
'STATE 1 l 10368 10367',
'STATE 1 t 10370 10369',
'STATE 5 d 10093 10371',
'STATE 1 e 10058 10372',
'STATE 5 v 10058 10373',
'STATE 5 c 10058 10374',
'STATE 5 t 10093 10375',
'STATE 2 e 10042 10376',
'STATE 5 # 10042 10377',
'STATE 1 # 10042 10058',
'STATE 2 h 10042 10378',
'STATE 2 f 10380 10379',
'STATE 4 o 10382 10381',
'STATE 5 t 10384 10383',
'STATE 5 s 10058 10042',
'STATE 2 d 10042 10058',
'STATE 4 n 10042 10385',
'STATE 4 d 10093 10058',
'STATE 1 l 10058 10093',
'STATE 4 o 10387 10386',
'STATE 2 b 10042 10058',
'STATE 2 c 10042 10388',
'STATE 2 c 10029 10389',
'STATE 2 t 10058 10029',
'STATE 2 j 10058 10390',
'STATE 1 p 10042 10391',
'STATE 5 u 10393 10392',
'STATE 4 m 10058 10394',
'STATE 5 e 10029 10395',
'STATE 5 i 10029 10396',
'STATE 2 h 10398 10397',
'STATE 2 o 10042 10399',
'STATE 4 e 10042 10400',
'STATE 1 e 10093 10401',
'STATE 2 b 10058 10402',
'STATE 2 d 10058 10403',
'STATE 5 u 10405 10404',
'STATE 2 v 10058 10406',
'STATE 2 d 10093 10407',
'STATE 4 n 10058 10408',
'STATE 5 t 10042 10409',
'STATE 2 d 10042 10410',
'STATE 2 f 10042 10029',
'STATE 5 c 10042 10411',
'STATE 3 h 10412 10042',
'STATE 4 o 10042 10413',
'STATE 3 t 10058 10414',
'STATE 4 s 10416 10415',
'STATE 4 i 10418 10417',
'STATE 2 w 10420 10419',
'STATE 2 o 10042 10421',
'STATE 2 n 10058 10422',
'STATE 4 e 10275 10042',
'STATE 5 t 10042 10423',
'STATE 5 d 10425 10424',
'STATE 2 m 10058 10426',
'STATE 5 l 10058 10042',
'STATE 4 s 10428 10427',
'STATE 4 # 10058 10144',
'STATE 1 w 10430 10429',
'STATE 2 o 10042 10431',
'STATE 5 r 10058 10432',
'STATE 1 g 10042 10433',
'STATE 4 i 10042 10058',
'STATE 4 e 10435 10434',
'STATE 4 e 10093 10436',
'STATE 4 u 10042 10437',
'STATE 1 a 10058 10438',
'STATE 5 b 10042 10439',
'STATE 5 s 10282 10440',
'STATE 5 f 10042 10058',
'STATE 1 s 10442 10441',
'STATE 5 u 10444 10443',
'STATE 1 o 10058 10445',
'STATE 1 r 10058 10446',
'STATE 4 g 10448 10447',
'STATE 4 # 10450 10449',
'STATE 2 e 10042 10451',
'STATE 5 a 10042 10452',
'STATE 1 a 10058 10042',
'STATE 1 r 10029 10453',
'STATE 1 s 10058 10454',
'STATE 5 i 10093 10455',
'STATE 2 p 10058 10093',
'STATE 4 p 10058 10093',
'STATE 1 s 10042 10456',
'STATE 5 y 10029 10457',
'STATE 4 v 10459 10458',
'STATE 4 m 10093 10460',
'STATE 2 b 10058 10461',
'STATE 2 d 10058 10462',
'STATE 1 x 10093 10463',
'STATE 4 y 10058 10464',
'STATE 5 s 10466 10465',
'STATE 5 i 10468 10467',
'STATE 4 m 10058 10469',
'STATE 2 b 10470 10093',
'STATE 2 c 10093 10471',
'STATE 4 c 10065 10472',
'STATE 2 g 10058 10473',
'STATE 5 i 10029 10474',
'STATE 4 # 10042 10475',
'STATE 2 a 10058 10042',
'STATE 2 m 10042 10476',
'STATE 1 a 10058 10477',
'STATE 5 z 10479 10478',
'STATE 5 # 10480 10042',
'STATE 1 s 10042 10481',
'STATE 2 l 10042 10482',
'STATE 5 s 10484 10483',
'STATE 5 l 10058 10485',
'STATE 2 w 10058 10486',
'STATE 2 a 10042 10487',
'STATE 2 p 10042 10488',
'STATE 5 # 10042 10489',
'STATE 2 p 10058 10042',
'STATE 5 u 10058 10249',
'STATE 4 # 10042 10490',
'STATE 5 # 10042 10491',
'STATE 1 u 10493 10492',
'STATE 4 i 10093 10042',
'STATE 2 i 10093 10494',
'STATE 4 o 10042 10495',
'STATE 5 s 10042 10496',
'STATE 5 b 10058 10497',
'STATE 5 s 10042 10498',
'STATE 4 o 10500 10499',
'STATE 4 y 10042 10501',
'STATE 1 q 10058 10502',
'STATE 5 l 10042 10503',
'STATE 5 t 10058 10504',
'STATE 2 b 10506 10505',
'STATE 4 t 10042 10507',
'STATE 2 b 10058 10508',
'STATE 1 t 10093 10058',
'STATE 5 b 10058 10509',
'STATE 1 c 10058 10510',
'STATE 5 z 10058 10511',
'STATE 2 f 10179 10042',
'STATE 4 u 10042 10512',
'STATE 1 a 10058 10513',
'STATE 5 # 10042 10058',
'STATE 5 z 10058 10514',
'STATE 1 e 10515 10042',
'STATE 1 l 10058 10516',
'STATE 4 v 10093 10517',
'STATE 2 v 10042 10029',
'STATE 2 d 10029 10518',
'STATE 4 n 10520 10519',
'STATE 5 a 10521 10058',
'STATE 1 t 10523 10522',
'STATE 2 t 10525 10524',
'STATE 4 c 10527 10526',
'STATE 2 t 10058 10528',
'STATE 4 m 10093 10529',
'STATE 1 l 10058 10530',
'STATE 2 v 10042 10058',
'STATE 5 e 10531 10093',
'STATE 4 l 10228 10532',
'STATE 2 m 10058 10093',
'STATE 4 t 10058 10093',
'STATE 2 f 10058 10533',
'STATE 4 k 10093 10534',
'STATE 5 r 10058 10535',
'STATE 2 s 10042 10029',
'STATE 3 d 10537 10536',
'STATE 2 s 10538 10042',
'STATE 3 d 10042 10539',
'STATE 5 n 10541 10540',
'STATE 4 i 10058 10542',
'STATE 1 s 10042 10543',
'STATE 1 i 10058 10544',
'STATE 2 m 10042 10545',
'STATE 5 l 10058 10546',
'STATE 1 b 10042 10547',
'STATE 1 k 10058 10548',
'STATE 2 c 10042 10549',
'STATE 1 l 10551 10550',
'STATE 5 e 10552 10042',
'STATE 5 m 10042 10553',
'STATE 5 n 10042 10554',
'STATE 5 e 10093 10555',
'STATE 4 h 10058 10556',
'STATE 5 # 10042 10557',
'STATE 4 u 10058 10558',
'STATE 4 e 10042 10559',
'STATE 1 f 10042 10560',
'STATE 5 p 10058 10561',
'STATE 5 l 10042 10058',
'STATE 4 # 10093 10562',
'STATE 5 w 10093 10042',
'STATE 2 o 10042 10563',
'STATE 2 c 10058 10042',
'STATE 2 t 10042 10564',
'STATE 5 r 10058 10565',
'STATE 5 d 10567 10566',
'STATE 4 t 10042 10568',
'STATE 4 e 10042 10569',
'STATE 1 o 10058 10042',
'STATE 2 h 10042 10570',
'STATE 1 i 10058 10571',
'STATE 4 e 10042 10572',
'STATE 5 t 10058 10573',
'STATE 2 e 10575 10574',
'STATE 2 t 10577 10576',
'STATE 2 c 10042 10029',
'STATE 4 a 10042 10578',
'STATE 5 l 10093 10579',
'STATE 2 p 10029 10580',
'STATE 2 r 10042 10581',
'STATE 1 o 10058 10582',
'STATE 2 s 10583 10058',
'STATE 4 e 10042 10584',
'STATE 5 p 10042 10585',
'STATE 1 o 10058 10586',
'STATE 1 n 10058 10587',
'STATE 4 t 10589 10588',
'STATE 2 x 10058 10093',
'STATE 2 d 10093 10058',
'STATE 4 k 10093 10590',
'STATE 2 m 10592 10591',
'STATE 4 m 10594 10593',
'STATE 4 d 10093 10595',
'STATE 2 g 10058 10596',
'STATE 5 i 10598 10597',
'STATE 5 f 10058 10599',
'STATE 1 # 10042 10600',
'STATE 5 n 10601 10042',
'STATE 5 a 10042 10058',
'STATE 2 a 10058 10602',
'STATE 5 v 10058 10603',
'STATE 2 m 10058 10604',
'STATE 4 c 10058 10042',
'STATE 1 l 10058 10605',
'STATE 1 p 10058 10606',
'STATE 2 g 10058 10042',
'STATE 5 i 10607 10042',
'STATE 2 u 10042 10608',
'STATE 5 e 10058 10324',
'STATE 2 a 10042 10609',
'STATE 2 s 10611 10610',
'STATE 2 i 10058 10612',
'STATE 2 v 10058 10042',
'STATE 2 g 10042 10613',
'STATE 1 y 10042 10614',
'STATE 1 h 10093 10042',
'STATE 1 g 10616 10615',
'STATE 4 i 10058 10029',
'STATE 5 n 10617 10042',
'STATE 5 l 10093 10618',
'STATE 1 i 10058 10619',
'STATE 4 a 10058 10620',
'STATE 5 e 10093 10621',
'STATE 1 # 10623 10622',
'STATE 2 v 10042 10624',
'STATE 2 t 10114 10042',
'STATE 4 y 10626 10625',
'STATE 4 e 10058 10042',
'STATE 5 n 10058 10627',
'STATE 2 t 10042 10628',
'STATE 5 s 10042 10629',
'STATE 2 p 10058 10630',
'STATE 2 b 10042 10631',
'STATE 2 b 10633 10632',
'STATE 1 s 10093 10634',
'STATE 1 t 10042 10635',
'STATE 5 n 10636 10042',
'STATE 5 n 10058 10324',
'STATE 4 # 10042 10637',
'STATE 2 t 10093 10638',
'STATE 2 h 10042 10639',
'STATE 4 t 10641 10640',
'STATE 2 b 10058 10642',
'STATE 1 e 10058 10093',
'STATE 5 y 10058 10643',
'STATE 4 t 10093 10644',
'STATE 1 t 10058 10645',
'STATE 1 e 10093 10058',
'STATE 4 n 10647 10646',
'STATE 2 b 10058 10648',
'STATE 2 v 10058 10649',
'STATE 1 s 10042 10650',
'STATE 1 a 10042 10651',
'STATE 4 g 10652 10093',
'STATE 2 f 10058 10093',
'STATE 2 t 10093 10653',
'STATE 4 m 10093 10654',
'STATE 4 m 10093 10655',
'STATE 4 m 10058 10093',
'STATE 2 c 10058 10656',
'STATE 5 s 10657 10042',
'STATE 4 e 10658 10042',
'STATE 2 e 10042 10659',
'STATE 1 s 10661 10660',
'STATE 4 o 10058 10662',
'STATE 1 u 10058 10663',
'STATE 1 e 10058 10664',
'STATE 2 n 10042 10665',
'STATE 2 y 10042 10058',
'STATE 2 y 10042 10666',
'STATE 1 m 10058 10667',
'STATE 1 s 10058 10042',
'STATE 2 m 10058 10042',
'STATE 5 n 10668 10042',
'STATE 5 o 10670 10669',
'STATE 1 p 10042 10671',
'STATE 2 e 10029 10042',
'STATE 4 a 10058 10093',
'STATE 5 n 10672 10042',
'STATE 5 w 10029 10673',
'STATE 4 o 10058 10042',
'STATE 4 y 10093 10674',
'STATE 5 a 10058 10675',
'STATE 2 v 10676 10093',
'STATE 1 a 10042 10656',
'STATE 4 n 10678 10677',
'STATE 1 a 10042 10679',
'STATE 5 e 10042 10680',
'STATE 2 c 10042 10681',
'STATE 2 p 10058 10682',
'STATE 2 c 10058 10683',
'STATE 4 b 10179 10042',
'STATE 5 a 10685 10684',
'STATE 1 l 10058 10686',
'STATE 2 c 10093 10058',
'STATE 1 s 10058 10093',
'STATE 2 s 10042 10058',
'STATE 4 s 10042 10687',
'STATE 5 o 10689 10688',
'STATE 5 e 10029 10690',
'STATE 5 s 10692 10691',
'STATE 2 b 10694 10693',
'STATE 1 a 10696 10695',
'STATE 4 t 10093 10697',
'STATE 4 e 10163 10058',
'STATE 2 p 10699 10698',
'STATE 1 s 10700 10058',
'STATE 1 o 10058 10701',
'STATE 1 e 10093 10702',
'STATE 4 t 10058 10703',
'STATE 1 x 10042 10704',
'STATE 5 c 10058 10042',
'STATE 2 s 10042 10093',
'STATE 2 v 10093 10705',
'STATE 2 m 10058 10706',
'STATE 4 p 10058 10707',
'STATE 5 n 10058 10042',
'STATE 1 s 10058 10708',
'STATE 2 n 10058 10042',
'STATE 3 b 10058 10042',
'STATE 5 b 10058 10709',
'STATE 5 c 10042 10710',
'STATE 1 g 10042 10711',
'STATE 1 i 10042 10058',
'STATE 1 c 10058 10389',
'STATE 1 e 10042 10712',
'STATE 2 k 10058 10713',
'STATE 2 f 10042 10714',
'STATE 2 b 10058 10042',
'STATE 4 d 10555 10715',
'STATE 1 d 10093 10042',
'STATE 2 i 10029 10716',
'STATE 4 a 10042 10093',
'STATE 4 y 10042 10717',
'STATE 5 # 10042 10430',
'STATE 2 k 10689 10093',
'STATE 5 i 10058 10093',
'STATE 5 # 10719 10718',
'STATE 5 # 10720 10042',
'STATE 1 c 10058 10721',
'STATE 4 g 10042 10722',
'STATE 4 f 10042 10723',
'STATE 1 c 10058 10724',
'STATE 1 o 10058 10725',
'STATE 1 s 10093 10726',
'STATE 4 b 10058 10093',
'STATE 4 s 10093 10727',
'STATE 1 m 10042 10567',
'STATE 4 g 10093 10728',
'STATE 4 k 10058 10093',
'STATE 2 w 10042 10729',
'STATE 2 o 10042 10730',
'STATE 1 i 10042 10731',
'STATE 2 u 10042 10732',
'STATE 5 s 10058 10733',
'STATE 2 t 10735 10734',
'STATE 5 a 10093 10058',
'STATE 4 a 10058 10736',
'STATE 2 g 10058 10737',
'STATE 1 s 10738 10058',
'STATE 4 o 10058 10739',
'STATE 2 t 10740 10093',
'STATE 1 d 10058 10741',
'STATE 2 t 10058 10742',
'STATE 2 h 10042 10743',
'STATE 2 c 10744 10093',
'STATE 2 b 10093 10745',
'STATE 5 u 10746 10470',
'STATE 3 p 10042 10747',
'STATE 5 y 10749 10748',
'STATE 2 t 10042 10750',
'STATE 4 a 10154 10751',
'STATE 2 c 10042 10752',
'STATE 2 u 10042 10753',
'STATE 1 s 10755 10754',
'STATE 4 i 10042 10756',
'STATE 5 # 10042 10757',
'STATE 5 n 10759 10758',
'STATE 1 o 10761 10760',
'STATE 2 f 10762 10042',
'STATE 1 g 10042 10763',
'STATE 2 t 10042 10764',
'STATE 5 # 10765 10058',
'STATE 5 # 10042 10766',
'STATE 2 n 10058 10767',
'STATE 1 a 10058 10612',
'STATE 2 e 10093 10768',
'STATE 5 a 10093 10769',
'STATE 5 a 10470 10093',
'STATE 2 m 10042 10770',
'STATE 4 a 10772 10771',
'STATE 4 u 10058 10773',
'STATE 5 s 10775 10774',
'STATE 5 y 10058 10776',
'STATE 5 a 10058 10777',
'STATE 1 s 10093 10058',
'STATE 4 o 10042 10778',
'STATE 4 l 10058 10779',
'STATE 4 s 10093 10058',
'STATE 4 l 10093 10058',
'STATE 1 a 10093 10058',
'STATE 2 v 10093 10058',
'STATE 4 d 10058 10780',
'STATE 1 n 10058 10781',
'STATE 4 t 10093 10058',
'STATE 2 h 10093 10782',
'STATE 4 s 10058 10093',
'STATE 4 u 10042 10783',
'STATE 4 y 10785 10784',
'STATE 1 u 10058 10786',
'STATE 2 c 10042 10787',
'STATE 4 t 10042 10788',
'STATE 2 u 10042 10789',
'STATE 1 p 10058 10790',
'STATE 2 t 10058 10791',
'STATE 2 t 10042 10425',
'STATE 1 h 10042 10792',
'STATE 4 a 10794 10793',
'STATE 5 c 10042 10795',
'STATE 1 m 10042 10796',
'STATE 1 a 10798 10797',
'STATE 4 t 10042 10799',
'STATE 4 t 10058 10042',
'STATE 2 h 10058 10800',
'STATE 2 s 10058 10801',
'STATE 1 l 10058 10802',
'STATE 5 e 10042 10373',
'STATE 5 n 10804 10803',
'STATE 5 o 10093 10805',
'STATE 1 y 10058 10806',
'STATE 2 n 10042 10029',
'STATE 4 e 10808 10807',
'STATE 1 m 10042 10809',
'STATE 4 a 10042 10810',
'STATE 2 c 10058 10811',
'STATE 2 s 10093 10058',
'STATE 1 l 10093 10042',
'STATE 5 o 10058 10093',
'STATE 4 n 10093 10812',
'STATE 4 e 10058 10813',
'STATE 4 n 10058 10093',
'STATE 2 v 10058 10814',
'STATE 4 b 10058 10815',
'STATE 3 g 10042 10816',
'STATE 5 # 10818 10817',
'STATE 2 n 10042 10819',
'STATE 4 t 10042 10058',
'STATE 5 e 10042 10820',
'STATE 1 s 10042 10821',
'STATE 2 o 10042 10822',
'STATE 1 y 10058 10823',
'STATE 1 u 10058 10824',
'STATE 4 e 10042 10825',
'STATE 5 r 10042 10826',
'STATE 1 s 10029 10042',
'STATE 1 j 10042 10827',
'STATE 4 e 10042 10828',
'STATE 2 r 10058 10829',
'STATE 2 t 10042 10830',
'STATE 4 c 10058 10831',
'STATE 1 a 10042 10124',
'STATE 1 e 10058 10042',
'STATE 1 a 10058 10832',
'STATE 5 l 10833 10042',
'STATE 1 a 10042 10058',
'STATE 5 i 10093 10594',
'STATE 1 n 10093 10834',
'STATE 4 o 10836 10835',
'STATE 5 o 10042 10837',
'STATE 2 v 10838 10058',
'STATE 4 e 10552 10058',
'STATE 2 w 10093 10058',
'STATE 1 s 10093 10839',
'STATE 1 m 10058 10840',
'STATE 5 d 10058 10841',
'STATE 2 s 10093 10741',
'STATE 2 e 10042 10842',
'STATE 4 o 10091 10843',
'STATE 1 g 10545 10042',
'STATE 1 e 10042 10844',
'STATE 5 # 10042 10845',
'STATE 2 l 10042 10846',
'STATE 2 i 10042 10668',
'STATE 1 e 10848 10847',
'STATE 1 g 10042 10849',
'STATE 4 y 10042 10850',
'STATE 1 m 10042 10851',
'STATE 4 i 10853 10852',
'STATE 1 w 10029 10042',
'STATE 4 i 10855 10854',
'STATE 2 v 10058 10856',
'STATE 2 n 10058 10857',
'STATE 1 n 10042 10058',
'STATE 1 e 10042 10858',
'STATE 5 e 10093 10859',
'STATE 4 y 10861 10860',
'STATE 2 t 10058 10862',
'STATE 5 d 10058 10863',
'STATE 1 o 10058 10864',
'STATE 1 p 10058 10042',
'STATE 2 m 10093 10865',
'STATE 2 t 10058 10866',
'STATE 1 a 10042 10867',
'STATE 5 s 10869 10868',
'STATE 2 r 10042 10870',
'STATE 5 s 10042 10871',
'STATE 1 t 10042 10872',
'STATE 2 h 10832 10873',
'STATE 2 r 10042 10658',
'STATE 1 a 10042 10874',
'STATE 4 a 10042 10875',
'STATE 2 e 10876 10042',
'STATE 1 t 10042 10877',
'STATE 1 m 10042 10878',
'STATE 5 t 10058 10879',
'STATE 5 a 10042 10880',
'STATE 5 h 10042 10881',
'STATE 2 t 10058 10294',
'STATE 2 m 10042 10058',
'STATE 1 r 10058 10635',
'STATE 2 u 10883 10882',
'STATE 1 e 10042 10058',
'STATE 5 u 10058 10884',
'STATE 5 r 10058 10885',
'STATE 5 g 10042 10058',
'STATE 4 d 10042 10886',
'STATE 5 c 10058 10366',
'STATE 2 s 10042 10887',
'STATE 4 e 10889 10888',
'STATE 4 i 10890 10042',
'STATE 2 u 10042 10891',
'STATE 5 l 10042 10892',
'STATE 2 p 10042 10893',
'STATE 1 n 10058 10894',
'STATE 1 i 10042 10895',
'STATE 4 m 10042 10896',
'STATE 4 # 10042 10897',
'STATE 1 n 10042 10898',
'STATE 5 e 10042 10029',
'STATE 5 u 10839 10899',
'STATE 5 o 10042 10900',
'STATE 2 m 10058 10901',
'STATE 2 y 10903 10902',
'STATE 1 e 10058 10904',
'STATE 1 e 10058 10905',
'STATE 1 n 10042 10906',
'STATE 2 v 10908 10907',
'STATE 2 o 10042 10909',
'STATE 4 a 10911 10910',
'STATE 1 q 10042 10912',
'STATE 2 t 10058 10913',
'STATE 1 a 10058 10914',
'STATE 2 w 10042 10915',
'STATE 2 h 10058 10916',
'STATE 2 i 10042 10917',
'STATE 2 c 10042 10918',
'STATE 4 h 10042 10919',
'STATE 1 v 10042 10920',
'STATE 1 p 10042 10921',
'STATE 2 f 10923 10922',
'STATE 1 c 10058 10924',
'STATE 5 s 10042 10925',
'STATE 1 a 10058 10926',
'STATE 4 l 10058 10042',
'STATE 4 s 10042 10927',
'STATE 2 p 10058 10928',
'STATE 1 o 10058 10929',
'STATE 4 v 10093 10930',
'STATE 4 t 10093 10931',
'STATE 4 i 10042 10932',
'STATE 4 i 10934 10933',
'STATE 1 i 10058 10935',
'STATE 2 p 10042 10936',
'STATE 1 c 10042 10937',
'STATE 1 i 10042 10938',
'STATE 4 k 10042 10939',
'STATE 1 n 10058 10042',
'STATE 2 r 10042 10940',
'STATE 1 n 10042 10941',
'STATE 1 # 10042 10093',
'STATE 4 o 10042 10942',
'STATE 4 a 10944 10943',
'STATE 4 f 10832 10042',
'STATE 1 m 10058 10945',
'STATE 5 s 10058 10946',
'STATE 2 f 10042 10567',
'STATE 4 m 10058 10947',
'STATE 5 a 10058 10042',
'STATE 1 i 10058 10121',
'STATE 1 e 10058 10948',
'STATE 1 s 10093 10949',
'STATE 4 g 10093 10950',
'STATE 2 t 10042 10412',
'STATE 1 l 10951 10042',
'STATE 2 t 10042 10952',
'STATE 5 c 10042 10953',
'STATE 1 e 10058 10954',
'STATE 2 l 10042 10058',
'STATE 1 c 10042 10955',
'STATE 4 l 10042 10956',
'STATE 2 b 10058 10957',
'STATE 2 d 10058 10958',
'STATE 4 y 10029 10959',
'STATE 5 # 10029 10960',
'STATE 1 s 10058 10961',
'STATE 4 c 10042 10962',
'STATE 1 g 10042 10963',
'STATE 2 p 10965 10964',
'STATE 1 f 10058 10121',
'STATE 1 c 10093 10966',
'STATE 4 s 10968 10967',
'STATE 2 m 10042 10294',
'STATE 2 h 10058 10969',
'STATE 5 m 10058 10042',
'STATE 1 c 10058 10970',
'STATE 1 u 10042 10971',
'STATE 5 d 10042 10972',
'STATE 2 n 10973 10058',
'STATE 1 h 10058 10974',
'STATE 1 d 10976 10975',
'STATE 4 e 10978 10977',
'STATE 1 h 10029 10042',
'STATE 1 n 10979 10042',
'STATE 1 t 10042 10980',
'STATE 5 r 10982 10981',
'STATE 4 s 10983 10058',
'STATE 4 n 10093 10984',
'STATE 4 b 10093 10058',
'STATE 1 n 10058 10093',
'STATE 5 e 10985 10042',
'STATE 5 l 10042 10986',
'STATE 2 t 10058 10987',
'STATE 5 a 10042 10988',
'STATE 1 i 10058 10042',
'STATE 1 k 10058 10989',
'STATE 5 t 10029 10042',
'STATE 4 e 10042 10029',
'STATE 1 h 10029 10990',
'STATE 1 h 10042 10991',
'STATE 5 s 10042 10992',
'STATE 2 i 10042 10993',
'STATE 2 w 10327 10994',
'STATE 1 n 10995 10058',
'STATE 5 t 10058 10996',
'STATE 1 e 10058 10744',
'STATE 2 n 10042 10997',
'STATE 5 k 10042 10937',
'STATE 1 o 10042 10998',
'STATE 2 h 10999 10042',
'STATE 2 b 10042 11000',
'STATE 1 m 10042 10029',
'STATE 5 l 10042 10029',
'STATE 4 m 10927 10058',
'STATE 1 h 10042 11001',
'STATE 1 c 11003 11002',
'STATE 2 g 10058 10093',
'STATE 1 m 10093 10777',
'STATE 2 u 10042 11004',
'STATE 2 d 10042 11005',
'STATE 4 e 10042 10058',
'STATE 2 m 10042 11006',
'STATE 5 n 10937 11007',
'STATE 2 t 10058 11008',
'STATE 2 k 10058 10093',
'STATE 2 i 10042 10937',
'STATE 2 i 10042 11009',
'STATE 1 e 10058 11010',
'STATE 1 i 10042 11011',
'STATE 2 b 10058 11012',
'STATE 1 l 10042 10058',
'STATE 2 v 10042 11013',
'STATE 2 t 10801 11014',
'STATE 5 a 11015 10058',
'STATE 1 o 10058 11016',
'STATE 5 c 10042 10058',
'STATE 4 s 11017 10058',
'STATE 2 i 10058 10042',
'STATE 1 i 10093 10058',
'INDEX 11018 s',
'STATE 4 # 11020 11019',
'STATE 4 h 11022 11021',
'STATE 3 t 11024 11023',
'STATE 4 s 11026 11025',
'STATE 3 t 11028 11027',
'STATE 3 s 11030 11029',
'PHONE s',
'STATE 5 h 11032 11031',
'STATE 3 e 11034 11033',
'STATE 1 # 11036 11035',
'PHONE epsilon',
'STATE 3 u 11038 11037',
'STATE 1 n 11024 11039',
'STATE 2 0 11041 11040',
'STATE 3 t 11028 11042',
'STATE 5 p 11044 11043',
'STATE 6 o 11028 11045',
'PHONE sh',
'STATE 3 i 11046 11035',
'STATE 3 i 11048 11047',
'STATE 2 a 11049 11024',
'STATE 2 e 11050 11024',
'STATE 4 t 11052 11051',
'STATE 4 z 11035 11053',
'STATE 4 c 11054 11024',
'STATE 1 0 11028 11055',
'STATE 3 i 11024 11028',
'STATE 2 n 11057 11056',
'STATE 6 n 11024 11035',
'STATE 2 t 11059 11058',
'STATE 2 a 11028 11060',
'STATE 1 h 11024 11061',
'STATE 1 i 11024 11062',
'STATE 3 s 11064 11063',
'STATE 6 w 11065 11024',
'STATE 4 j 11035 11066',
'STATE 6 o 11068 11067',
'STATE 1 # 11070 11069',
'STATE 2 m 11028 11071',
'STATE 1 # 11024 11028',
'STATE 3 k 11024 11072',
'STATE 3 e 11024 11073',
'STATE 2 o 11028 11074',
'PHONE z',
'STATE 1 l 11024 11075',
'STATE 4 k 11024 11076',
'STATE 5 o 11078 11077',
'STATE 1 # 11024 11079',
'STATE 4 r 11081 11080',
'STATE 6 i 11083 11082',
'STATE 2 0 11035 11024',
'STATE 5 o 11085 11084',
'STATE 2 f 11087 11086',
'STATE 1 0 11028 11088',
'STATE 3 p 11024 11089',
'STATE 3 h 11024 11090',
'STATE 2 e 11061 11091',
'STATE 1 u 11024 11092',
'STATE 6 n 11094 11093',
'STATE 4 u 11096 11095',
'STATE 6 n 11035 11024',
'STATE 2 r 11024 11061',
'STATE 6 a 11024 11097',
'STATE 5 i 11024 11035',
'STATE 3 z 11028 11098',
'STATE 1 0 11035 11024',
'STATE 6 a 11100 11099',
'STATE 1 s 11061 11101',
'STATE 5 t 11024 11102',
'STATE 3 o 11024 11103',
'STATE 2 f 11028 11104',
'STATE 3 c 11024 11105',
'STATE 1 i 11024 11106',
'STATE 2 u 11061 11107',
'STATE 1 t 11028 11024',
'STATE 4 z 11109 11108',
'STATE 4 i 11111 11110',
'STATE 2 e 11113 11112',
'STATE 5 r 11035 11114',
'STATE 5 r 11115 11024',
'STATE 1 # 11116 11035',
'STATE 6 o 11028 11117',
'STATE 3 u 11028 11118',
'STATE 3 u 11024 11119',
'STATE 2 b 11121 11120',
'STATE 3 a 11024 11028',
'STATE 2 i 11028 11122',
'STATE 2 k 11124 11123',
'STATE 1 n 11126 11125',
'STATE 1 o 11024 11127',
'STATE 4 m 11129 11128',
'STATE 6 o 11024 11035',
'STATE 4 z 11035 11130',
'STATE 5 o 11132 11131',
'STATE 5 a 11134 11133',
'STATE 1 m 11024 11135',
'STATE 2 i 11136 11024',
'STATE 4 u 11137 11024',
'STATE 6 a 11024 11138',
'STATE 1 w 11140 11139',
'STATE 5 m 11119 11028',
'STATE 2 l 11024 11028',
'STATE 2 g 11103 11141',
'STATE 3 u 11028 11142',
'STATE 1 p 11144 11143',
'STATE 3 f 11024 11145',
'STATE 3 e 11024 11146',
'STATE 1 o 11061 11147',
'STATE 3 a 11024 11061',
'STATE 2 c 11024 11148',
'STATE 4 e 11150 11149',
'STATE 5 # 11061 11151',
'STATE 4 c 11024 11152',
'STATE 5 g 11154 11153',
'STATE 3 n 11035 11155',
'STATE 6 v 11061 11156',
'STATE 4 i 11035 11157',
'STATE 1 # 11024 11158',
'STATE 1 # 11035 11024',
'STATE 6 e 11035 11024',
'STATE 6 e 11159 11035',
'STATE 3 i 11161 11160',
'STATE 5 # 11028 11024',
'STATE 2 d 11162 11028',
'STATE 3 i 11028 11163',
'STATE 1 # 11165 11164',
'STATE 5 u 11028 11166',
'STATE 3 a 11168 11167',
'STATE 1 i 11024 11061',
'STATE 1 t 11061 11169',
'STATE 1 i 11171 11170',
'STATE 4 u 11173 11172',
'STATE 3 r 11024 11174',
'STATE 5 s 11061 11175',
'STATE 4 p 11024 11176',
'PHONE zh',
'STATE 3 e 11061 11024',
'STATE 3 l 11035 11153',
'STATE 6 s 11178 11177',
'STATE 1 l 11028 11024',
'STATE 1 d 11024 11179',
'STATE 3 u 11035 11180',
'STATE 1 e 11028 11181',
'STATE 5 a 11182 11028',
'STATE 3 i 11028 11183',
'STATE 6 # 11024 11184',
'STATE 1 r 11186 11185',
'STATE 2 b 11028 11187',
'STATE 5 l 11028 11188',
'STATE 2 p 11190 11189',
'STATE 2 z 11061 11191',
'STATE 1 l 11061 11192',
'STATE 1 a 11024 11193',
'STATE 2 t 11024 11061',
'STATE 4 c 11195 11194',
'STATE 5 r 11197 11196',
'STATE 3 n 11199 11198',
'STATE 3 r 11024 11200',
'STATE 3 d 11061 11201',
'STATE 1 # 11024 11202',
'STATE 1 p 11061 11024',
'STATE 1 b 11024 11203',
'STATE 2 r 11024 11204',
'STATE 6 c 11028 11205',
'STATE 2 r 11024 11028',
'STATE 3 o 11028 11024',
'STATE 5 i 11028 11103',
'STATE 2 r 11207 11206',
'STATE 2 g 11028 11208',
'STATE 5 o 11024 11209',
'STATE 6 # 11028 11210',
'STATE 3 o 11212 11211',
'STATE 3 o 11061 11024',
'STATE 2 y 11024 11213',
'STATE 1 s 11061 11214',
'STATE 2 k 11061 11024',
'STATE 4 i 11216 11215',
'STATE 5 i 11218 11217',
'STATE 6 l 11153 11219',
'STATE 3 n 11221 11220',
'STATE 2 o 11223 11222',
'STATE 5 y 11061 11224',
'STATE 5 i 11226 11225',
'STATE 3 t 11024 11227',
'STATE 2 i 11024 11228',
'STATE 6 d 11028 11229',
'STATE 3 o 11024 11035',
'STATE 1 l 11028 11230',
'STATE 2 h 11028 11231',
'STATE 5 l 11028 11232',
'STATE 5 # 11233 11024',
'STATE 2 r 11028 11234',
'STATE 5 e 11024 11235',
'STATE 3 h 11237 11236',
'STATE 2 o 11061 11238',
'STATE 1 u 11061 11239',
'STATE 1 a 11061 11190',
'STATE 3 x 11028 11240',
'STATE 5 a 11242 11241',
'STATE 2 e 11243 11024',
'STATE 2 o 11035 11024',
'STATE 3 e 11244 11024',
'STATE 6 e 11153 11245',
'STATE 6 e 11035 11246',
'STATE 5 # 11248 11247',
'STATE 5 r 11250 11249',
'STATE 1 e 11061 11024',
'STATE 6 l 11061 11251',
'STATE 6 c 11061 11024',
'STATE 3 w 11061 11252',
'STATE 1 c 11024 11253',
'STATE 1 f 11024 11254',
'STATE 1 t 11028 11255',
'STATE 1 l 11024 11256',
'STATE 6 r 11028 11257',
'STATE 2 l 11028 11024',
'STATE 2 h 11024 11258',
'STATE 6 n 11260 11259',
'STATE 3 e 11261 11061',
'STATE 2 c 11024 11262',
'STATE 1 u 11264 11263',
'STATE 2 l 11266 11265',
'STATE 4 p 11268 11267',
'STATE 2 p 11270 11269',
'STATE 6 # 11272 11271',
'STATE 5 o 11024 11061',
'STATE 5 m 11061 11273',
'STATE 2 e 11153 11024',
'STATE 6 g 11024 11274',
'STATE 5 d 11276 11275',
'STATE 3 p 11024 11277',
'STATE 3 i 11061 11278',
'STATE 1 # 11061 11279',
'STATE 2 c 11061 11280',
'STATE 1 0 11282 11281',
'STATE 4 o 11284 11283',
'STATE 1 h 11028 11285',
'STATE 6 i 11287 11286',
'STATE 1 b 11028 11288',
'STATE 6 v 11028 11289',
'STATE 2 l 11291 11290',
'STATE 2 r 11140 11024',
'STATE 5 i 11024 11028',
'STATE 1 q 11024 11292',
'STATE 1 i 11061 11293',
'STATE 2 l 11061 11294',
'STATE 2 l 11024 11061',
'STATE 2 c 11024 11295',
'STATE 1 l 11061 11296',
'STATE 4 o 11298 11297',
'STATE 3 w 11061 11024',
'STATE 3 u 11300 11299',
'STATE 6 i 11061 11301',
'STATE 6 k 11024 11302',
'STATE 2 n 11153 11303',
'STATE 5 l 11061 11024',
'STATE 2 i 11035 11024',
'STATE 5 r 11305 11304',
'STATE 3 a 11024 11306',
'STATE 3 a 11308 11307',
'STATE 5 d 11061 11024',
'STATE 1 r 11061 11309',
'STATE 3 d 11061 11310',
'STATE 3 r 11024 11311',
'STATE 3 i 11028 11312',
'STATE 1 g 11314 11313',
'STATE 1 l 11028 11315',
'STATE 1 i 11024 11316',
'STATE 5 u 11028 11317',
'STATE 5 f 11024 11028',
'STATE 1 o 11028 11318',
'STATE 1 b 11028 11319',
'STATE 2 w 11024 11320',
'STATE 6 n 11028 11024',
'STATE 2 f 11024 11321',
'STATE 2 a 11061 11322',
'STATE 2 g 11061 11323',
'STATE 1 a 11061 11324',
'STATE 1 a 11061 11024',
'STATE 3 g 11326 11325',
'STATE 5 r 11328 11327',
'STATE 3 i 11330 11329',
'STATE 5 v 11024 11331',
'STATE 1 e 11061 11332',
'STATE 1 t 11061 11333',
'STATE 3 o 11024 11334',
'STATE 2 r 11336 11335',
'STATE 2 e 11338 11337',
'STATE 3 p 11024 11339',
'STATE 3 l 11024 11340',
'STATE 2 r 11061 11341',
'STATE 3 u 11024 11061',
'STATE 6 s 11061 11342',
'STATE 2 u 11024 11343',
'STATE 4 m 11061 11024',
'STATE 5 i 11345 11344',
'STATE 2 a 11028 11024',
'STATE 1 h 11024 11346',
'STATE 1 c 11348 11347',
'STATE 2 c 11028 11349',
'STATE 2 l 11028 11350',
'STATE 5 e 11352 11351',
'STATE 6 l 11028 11353',
'STATE 1 e 11264 11061',
'STATE 2 g 11061 11024',
'STATE 2 d 11061 11354',
'STATE 2 e 11356 11355',
'STATE 5 e 11358 11357',
'STATE 4 b 11061 11359',
'STATE 3 a 11361 11360',
'STATE 2 v 11061 11362',
'STATE 3 e 11364 11363',
'STATE 5 t 11061 11365',
'STATE 5 # 11024 11366',
'STATE 6 e 11061 11367',
'STATE 3 a 11024 11035',
'STATE 3 a 11153 11024',
'STATE 3 i 11369 11368',
'STATE 5 n 11371 11370',
'STATE 3 i 11061 11372',
'STATE 3 y 11061 11373',
'STATE 1 o 11024 11061',
'STATE 2 w 11061 11374',
'STATE 2 b 11024 11375',
'STATE 1 u 11024 11376',
'STATE 4 o 11024 11377',
'STATE 1 f 11379 11378',
'STATE 2 o 11028 11024',
'STATE 2 u 11140 11024',
'STATE 1 v 11024 11380',
'STATE 5 r 11381 11024',
'STATE 6 f 11028 11382',
'STATE 5 i 11028 11383',
'STATE 1 d 11024 11384',
'STATE 1 d 11028 11385',
'STATE 2 j 11024 11386',
'STATE 1 a 11388 11387',
'STATE 1 r 11390 11389',
'STATE 1 n 11024 11061',
'STATE 2 d 11392 11391',
'STATE 6 # 11394 11393',
'STATE 1 i 11061 11395',
'STATE 1 # 11397 11396',
'STATE 2 e 11398 11024',
'STATE 3 b 11061 11399',
'STATE 2 h 11401 11400',
'STATE 2 r 11403 11402',
'STATE 2 d 11024 11404',
'STATE 6 e 11061 11405',
'STATE 6 g 11061 11406',
'STATE 3 u 11408 11407',
'STATE 2 w 11061 11409',
'STATE 1 # 11411 11410',
'STATE 6 t 11061 11412',
'STATE 2 i 11024 11413',
'STATE 3 l 11024 11414',
'STATE 2 h 11061 11415',
'STATE 1 c 11024 11416',
'STATE 6 t 11418 11417',
'STATE 3 i 11024 11419',
'STATE 1 b 11421 11420',
'STATE 2 u 11024 11422',
'STATE 4 y 11024 11423',
'STATE 6 # 11024 11424',
'STATE 2 l 11426 11425',
'STATE 2 u 11028 11427',
'STATE 5 # 11028 11428',
'STATE 6 s 11024 11028',
'STATE 5 e 11028 11429',
'STATE 2 m 11061 11430',
'STATE 2 r 11061 11395',
'STATE 2 r 11432 11431',
'STATE 2 g 11024 11061',
'STATE 4 l 11434 11433',
'STATE 4 r 11435 11024',
'STATE 1 0 11437 11436',
'STATE 3 i 11028 11438',
'STATE 2 n 11061 11024',
'STATE 3 i 11178 11439',
'STATE 2 r 11441 11440',
'STATE 6 # 11442 11061',
'STATE 2 r 11061 11024',
'STATE 5 e 11024 11443',
'STATE 3 y 11061 11024',
'STATE 6 a 11061 11444',
'STATE 1 # 11061 11445',
'STATE 2 v 11061 11446',
'STATE 6 g 11448 11447',
'STATE 5 t 11061 11024',
'STATE 3 o 11450 11449',
'STATE 2 a 11061 11451',
'STATE 1 r 11061 11452',
'STATE 5 s 11453 11024',
'STATE 3 o 11061 11454',
'STATE 3 o 11061 11455',
'STATE 3 e 11061 11456',
'STATE 1 h 11061 11024',
'STATE 1 b 11061 11457',
'STATE 1 r 11024 11458',
'STATE 6 y 11024 11459',
'STATE 5 a 11024 11061',
'STATE 3 k 11024 11460',
'STATE 1 l 11462 11461',
'STATE 2 u 11024 11463',
'STATE 4 i 11024 11028',
'STATE 1 j 11465 11464',
'STATE 6 i 11028 11024',
'STATE 1 g 11028 11466',
'STATE 6 s 11028 11467',
'STATE 2 c 11028 11468',
'STATE 5 i 11028 11024',
'STATE 2 t 11024 11028',
'STATE 1 l 11024 11469',
'STATE 1 t 11061 11470',
'STATE 1 t 11061 11224',
'STATE 3 p 11024 11471',
'STATE 2 o 11024 11472',
'STATE 3 e 11028 11024',
'STATE 3 u 11024 11473',
'STATE 3 i 11028 11024',
'STATE 4 n 11028 11024',
'STATE 3 y 11474 11024',
'STATE 2 m 11061 11024',
'STATE 6 a 11061 11273',
'STATE 1 r 11024 11061',
'STATE 6 g 11190 11475',
'STATE 5 r 11061 11024',
'STATE 6 e 11061 11024',
'STATE 1 p 11061 11476',
'STATE 5 b 11061 11477',
'STATE 1 # 11024 11061',
'STATE 3 e 11479 11478',
'STATE 6 # 11481 11480',
'STATE 2 m 11061 11482',
'STATE 2 h 11061 11483',
'STATE 3 i 11061 11024',
'STATE 5 m 11061 11024',
'STATE 6 d 11024 11154',
'STATE 1 h 11061 11484',
'STATE 2 e 11486 11485',
'STATE 2 c 11024 11487',
'STATE 5 a 11489 11488',
'STATE 2 r 11491 11490',
'STATE 5 e 11024 11492',
'STATE 5 f 11024 11493',
'STATE 5 # 11028 11494',
'STATE 4 l 11496 11495',
'STATE 6 # 11028 11024',
'STATE 1 o 11028 11497',
'STATE 5 b 11028 11498',
'STATE 2 s 11028 11499',
'STATE 2 n 11501 11500',
'STATE 1 m 11061 11502',
'STATE 4 q 11024 11503',
'STATE 1 r 11061 11024',
'STATE 6 y 11505 11504',
'STATE 2 a 11061 11024',
'STATE 3 a 11507 11506',
'STATE 2 a 11061 11508',
'STATE 5 c 11061 11024',
'STATE 5 y 11510 11509',
'STATE 1 h 11061 11511',
'STATE 2 j 11024 11512',
'STATE 2 p 11061 11513',
'STATE 2 f 11061 11024',
'STATE 6 h 11061 11514',
'STATE 6 # 11516 11515',
'STATE 1 i 11024 11517',
'STATE 3 e 11442 11024',
'STATE 1 l 11024 11061',
'STATE 2 d 11024 11518',
'STATE 2 a 11024 11519',
'STATE 2 c 11024 11520',
'STATE 1 # 11061 11024',
'STATE 6 a 11024 11521',
'STATE 5 s 11024 11522',
'STATE 5 d 11024 11523',
'STATE 1 r 11525 11524',
'STATE 1 r 11024 11526',
'STATE 5 a 11028 11527',
'STATE 3 a 11528 11028',
'STATE 2 t 11028 11024',
'STATE 1 d 11079 11529',
'STATE 1 i 11061 11024',
'STATE 1 p 11024 11530',
'STATE 4 a 11532 11531',
'STATE 3 i 11024 11533',
'STATE 3 i 11061 11507',
'STATE 1 r 11474 11534',
'STATE 2 e 11061 11024',
'STATE 5 # 11024 11535',
'STATE 3 a 11536 11024',
'STATE 3 t 11024 11537',
'STATE 5 t 11024 11538',
'STATE 6 u 11024 11539',
'STATE 2 l 11061 11491',
'STATE 5 s 11061 11540',
'STATE 3 a 11024 11541',
'STATE 1 # 11061 11542',
'STATE 2 t 11024 11543',
'STATE 5 e 11154 11024',
'STATE 3 i 11061 11544',
'STATE 3 o 11024 11545',
'STATE 6 g 11024 11546',
'STATE 2 a 11547 11024',
'STATE 4 i 11024 11314',
'STATE 5 a 11024 11548',
'STATE 6 g 11028 11549',
'STATE 6 r 11028 11024',
'STATE 2 a 11028 11550',
'STATE 1 c 11028 11551',
'STATE 1 n 11024 11552',
'STATE 1 e 11061 11553',
'STATE 3 t 11024 11554',
'STATE 3 r 11024 11555',
'STATE 2 e 11557 11556',
'STATE 2 r 11491 11024',
'STATE 2 o 11061 11558',
'STATE 2 e 11273 11024',
'STATE 1 # 11024 11559',
'STATE 1 g 11024 11560',
'STATE 1 # 11273 11024',
'STATE 6 # 11024 11561',
'STATE 6 s 11061 11024',
'STATE 2 r 11061 11562',
'STATE 2 c 11564 11563',
'STATE 6 r 11024 11061',
'STATE 1 k 11061 11565',
'STATE 1 o 11024 11566',
'STATE 5 c 11028 11024',
'STATE 4 o 11028 11567',
'STATE 4 m 11028 11568',
'STATE 2 t 11028 11569',
'STATE 5 # 11028 11570',
'STATE 2 h 11024 11061',
'STATE 1 i 11061 11571',
'STATE 3 i 11573 11572',
'STATE 5 n 11575 11574',
'STATE 3 d 11061 11576',
'STATE 1 r 11024 11577',
'STATE 2 u 11061 11578',
'STATE 3 l 11024 11061',
'STATE 2 h 11024 11579',
'STATE 1 w 11024 11580',
'STATE 1 s 11024 11581',
'STATE 1 e 11583 11582',
'STATE 3 u 11061 11024',
'STATE 1 l 11061 11584',
'STATE 1 w 11024 11585',
'STATE 5 n 11024 11586',
'STATE 5 r 11024 11587',
'STATE 6 m 11028 11588',
'STATE 1 g 11024 11028',
'STATE 2 d 11061 11589',
'STATE 3 w 11061 11590',
'STATE 4 r 11061 11024',
'STATE 2 v 11061 11591',
'STATE 2 e 11061 11592',
'STATE 3 a 11024 11593',
'STATE 4 b 11061 11594',
'STATE 6 e 11024 11595',
'STATE 6 s 11061 11596',
'STATE 2 e 11597 11024',
'STATE 3 u 11024 11474',
'STATE 1 a 11154 11598',
'STATE 2 r 11061 11453',
'STATE 1 i 11061 11599',
'STATE 6 l 11600 11024',
'STATE 5 s 11024 11601',
'STATE 6 t 11024 11602',
'STATE 1 r 11604 11603',
'STATE 2 s 11024 11061',
'STATE 5 s 11024 11605',
'STATE 5 l 11607 11606',
'STATE 6 # 11061 11541',
'STATE 1 c 11024 11608',
'STATE 1 n 11061 11609',
'STATE 2 r 11061 11610',
'STATE 5 n 11611 11024',
'STATE 5 n 11061 11024',
'STATE 3 i 11061 11612',
'STATE 2 l 11061 11613',
'STATE 4 a 11028 11024',
'STATE 1 n 11615 11614',
'STATE 5 v 11024 11616',
'STATE 6 e 11028 11617',
'STATE 2 e 11028 11182',
'STATE 3 k 11024 11618',
'STATE 3 a 11024 11619',
'STATE 6 e 11024 11620',
'STATE 6 l 11622 11621',
'STATE 3 n 11024 11061',
'STATE 1 # 11024 11623',
'STATE 6 # 11061 11024',
'STATE 1 r 11625 11624',
'STATE 3 a 11312 11626',
'STATE 6 # 11628 11627',
'STATE 5 e 11024 11028',
'STATE 5 # 11024 11629',
'STATE 6 s 11028 11630',
'STATE 3 a 11507 11631',
'STATE 2 e 11024 11632',
'STATE 3 o 11061 11633',
'STATE 4 f 11024 11634',
'STATE 4 n 11024 11061',
'STATE 2 e 11024 11339',
'STATE 2 p 11061 11635',
'STATE 2 i 11061 11024',
'STATE 2 t 11024 11636',
'STATE 6 r 11024 11637',
'STATE 5 d 11024 11028',
'STATE 4 e 11638 11028',
'STATE 2 m 11028 11639',
'STATE 4 f 11024 11640',
'STATE 6 l 11061 11641',
'STATE 1 # 11024 11609',
'STATE 2 o 11024 11642',
'STATE 1 # 11061 11643',
'STATE 4 b 11061 11644',
'STATE 4 e 11024 11645',
'STATE 5 s 11028 11646',
'STATE 6 d 11028 11647',
'STATE 4 n 11024 11648',
'STATE 6 c 11190 11649',
'STATE 2 c 11024 11650',
'STATE 1 l 11061 11651',
'STATE 2 i 11061 11652',
'STATE 1 s 11024 11653',
'STATE 5 d 11028 11024',
'STATE 6 # 11028 11654',
'STATE 2 l 11061 11655',
'STATE 6 i 11564 11656',
'STATE 4 w 11024 11657',
'STATE 2 i 11024 11061',
'STATE 1 h 11024 11658',
'STATE 1 l 11024 11028',
'STATE 1 a 11028 11659',
'STATE 1 f 11024 11660',
'STATE 5 b 11448 11661',
'STATE 1 g 11024 11662',
'STATE 2 e 11024 11663',
'STATE 1 b 11028 11664',
'STATE 1 c 11061 11665',
'STATE 6 e 11024 11666',
'STATE 4 l 11024 11667',
'STATE 4 m 11024 11564',
'STATE 1 m 11028 11668',
'STATE 3 u 11024 11669',
'STATE 2 a 11671 11670',
'STATE 6 r 11672 11061',
'STATE 1 s 11028 11673',
'STATE 3 o 11675 11674',
'STATE 6 s 11024 11676',
'STATE 5 # 11024 11472',
'STATE 2 a 11024 11677',
'STATE 3 a 11140 11028',
'STATE 4 d 11679 11678',
'STATE 5 # 11061 11024',
'STATE 3 u 11680 11534',
'STATE 1 # 11024 11681',
'STATE 3 e 11395 11682',
'STATE 1 d 11061 11683',
'STATE 6 # 11024 11684',
'STATE 3 n 11061 11024',
'STATE 1 h 11024 11685',
'STATE 2 u 11061 11686',
'STATE 5 # 11024 11061',
'STATE 5 r 11061 11687',
'STATE 2 a 11024 11061',
'STATE 3 m 11061 11688',
'STATE 3 r 11024 11689',
'STATE 6 c 11061 11690',
'STATE 2 u 11024 11691',
'STATE 4 v 11061 11692',
'STATE 6 r 11693 11474',
'STATE 2 e 11061 11694',
'STATE 4 b 11061 11686',
'INDEX 11695 t',
'STATE 4 h 11697 11696',
'STATE 5 o 11699 11698',
'STATE 5 e 11701 11700',
'STATE 4 t 11703 11702',
'STATE 4 i 11705 11704',
'STATE 5 # 11707 11706',
'STATE 6 r 11709 11708',
'STATE 4 c 11711 11710',
'STATE 3 e 11713 11712',
'STATE 4 t 11715 11714',
'STATE 3 s 11717 11716',
'STATE 3 s 11719 11718',
'PHONE th',
'STATE 6 # 11721 11720',
'STATE 3 # 11707 11722',
'STATE 4 u 11724 11723',
'STATE 5 h 11726 11725',
'STATE 5 s 11728 11727',
'STATE 5 s 11726 11729',
'STATE 4 u 11731 11730',
'STATE 3 e 11726 11732',
'STATE 3 n 11734 11733',
'PHONE ch',
'STATE 5 o 11736 11735',
'STATE 6 a 11707 11725',
'STATE 6 a 11738 11737',
'STATE 3 n 11707 11739',
'STATE 3 a 11739 11721',
'STATE 3 t 11741 11740',
'STATE 3 # 11725 11742',
'PHONE t',
'PHONE epsilon',
'STATE 6 d 11726 11743',
'STATE 6 # 11745 11744',
'STATE 5 l 11746 11726',
'STATE 3 t 11748 11747',
'STATE 6 u 11717 11725',
'STATE 6 m 11726 11749',
'STATE 6 n 11734 11750',
'PHONE sh',
'STATE 6 # 11752 11751',
'STATE 6 r 11707 11753',
'STATE 6 d 11755 11754',
'STATE 3 # 11707 11756',
'PHONE dh',
'STATE 5 a 11758 11757',
'STATE 4 s 11726 11759',
'STATE 5 r 11761 11760',
'STATE 5 r 11726 11762',
'STATE 3 i 11725 11726',
'STATE 3 o 11725 11763',
'STATE 6 e 11726 11725',
'STATE 4 a 11764 11725',
'STATE 4 r 11725 11765',
'STATE 6 # 11767 11766',
'STATE 6 u 11734 11768',
'STATE 5 a 11770 11769',
'STATE 3 r 11772 11771',
'STATE 3 l 11725 11773',
'STATE 6 i 11707 11774',
'STATE 3 a 11739 11707',
'STATE 3 r 11707 11725',
'STATE 6 h 11776 11775',
'STATE 4 i 11778 11777',
'STATE 6 0 11725 11779',
'STATE 5 a 11717 11780',
'STATE 6 e 11717 11781',
'STATE 6 s 11726 11782',
'STATE 3 a 11726 11725',
'PHONE d',
'STATE 4 o 11725 11726',
'STATE 3 o 11784 11783',
'STATE 3 o 11725 11726',
'STATE 3 a 11734 11725',
'STATE 6 n 11786 11785',
'STATE 6 i 11725 11707',
'STATE 5 y 11707 11787',
'STATE 5 y 11739 11707',
'STATE 3 o 11725 11788',
'STATE 6 y 11739 11789',
'STATE 4 l 11791 11790',
'STATE 4 s 11726 11725',
'STATE 4 g 11793 11792',
'STATE 3 s 11795 11794',
'STATE 5 d 11725 11796',
'STATE 6 a 11798 11797',
'STATE 6 a 11717 11799',
'STATE 6 n 11726 11800',
'STATE 3 u 11725 11801',
'STATE 6 n 11726 11725',
'STATE 6 l 11803 11802',
'STATE 3 r 11739 11755',
'STATE 5 s 11707 11804',
'STATE 6 d 11707 11805',
'STATE 6 s 11807 11806',
'STATE 5 h 11809 11808',
'STATE 3 s 11810 11725',
'STATE 4 r 11725 11811',
'STATE 3 r 11726 11725',
'STATE 3 n 11813 11812',
'STATE 6 # 11725 11717',
'STATE 4 r 11725 11814',
'STATE 5 e 11717 11815',
'STATE 5 l 11717 11816',
'STATE 6 i 11817 11725',
'STATE 5 # 11726 11818',
'STATE 6 n 11725 11819',
'STATE 3 y 11821 11820',
'STATE 5 i 11725 11707',
'STATE 3 n 11707 11822',
'STATE 6 l 11707 11823',
'STATE 3 # 11707 11824',
'STATE 3 o 11739 11755',
'STATE 4 # 11725 11825',
'STATE 4 s 11826 11725',
'STATE 5 e 11726 11725',
'STATE 3 # 11827 11725',
'STATE 3 # 11725 11734',
'STATE 6 l 11717 11828',
'STATE 5 a 11725 11829',
'STATE 3 s 11725 11830',
'STATE 3 r 11717 11725',
'STATE 3 n 11725 11717',
'STATE 6 a 11726 11831',
'STATE 3 a 11832 11726',
'STATE 6 s 11833 11707',
'STATE 5 m 11739 11707',
'STATE 5 a 11707 11739',
'STATE 6 n 11707 11834',
'STATE 3 o 11835 11707',
'STATE 5 n 11837 11836',
'STATE 6 a 11717 11838',
'STATE 4 e 11725 11839',
'STATE 6 t 11734 11840',
'STATE 6 t 11725 11841',
'STATE 6 # 11725 11842',
'STATE 3 a 11844 11843',
'STATE 6 o 11726 11725',
'STATE 3 # 11739 11707',
'STATE 6 g 11707 11845',
'STATE 6 n 11739 11707',
'STATE 6 n 11847 11846',
'STATE 4 e 11848 11725',
'STATE 6 o 11725 11717',
'STATE 4 o 11725 11849',
'STATE 6 # 11734 11725',
'STATE 4 o 11851 11850',
'STATE 3 i 11725 11852',
'STATE 6 # 11726 11853',
'STATE 5 n 11725 11854',
'STATE 3 # 11707 11855',
'STATE 3 d 11857 11856',
'STATE 4 i 11858 11725',
'STATE 3 s 11860 11859',
'STATE 4 w 11725 11861',
'STATE 5 s 11725 11862',
'STATE 5 m 11725 11863',
'STATE 6 e 11864 11725',
'STATE 5 k 11726 11865',
'STATE 5 y 11726 11866',
'STATE 6 u 11725 11867',
'STATE 5 u 11868 11725',
'STATE 4 k 11726 11725',
'STATE 5 e 11768 11725',
'STATE 3 f 11726 11725',
'STATE 6 i 11726 11746',
'STATE 4 s 11725 11726',
'STATE 5 n 11725 11869',
'STATE 5 n 11870 11726',
'STATE 3 r 11717 11871',
'STATE 6 i 11726 11872',
'STATE 6 t 11725 11726',
'STATE 3 a 11707 11873',
'STATE 4 e 11874 11725',
'STATE 5 w 11726 11875',
'STATE 6 # 11726 11725',
'STATE 3 a 11717 11725',
'STATE 3 r 11725 11876',
'STATE 3 r 11707 11877',
'STATE 6 r 11878 11725',
'STATE 5 g 11725 11879',
'STATE 6 k 11725 11880',
'STATE 3 e 11707 11725',
'STATE 3 s 11717 11725',
'STATE 6 k 11725 11881',
'STATE 3 u 11883 11882',
'STATE 5 m 11725 11884',
'STATE 6 g 11726 11885',
'STATE 6 e 11726 11810',
'STATE 6 f 11725 11886',
'STATE 5 i 11746 11887',
'STATE 6 n 11725 11888',
'STATE 6 m 11726 11889',
'STATE 6 d 11726 11890',
'STATE 5 a 11744 11891',
'STATE 6 o 11726 11892',
'STATE 5 l 11767 11893',
'STATE 6 e 11895 11894',
'STATE 5 e 11896 11726',
'STATE 6 l 11898 11897',
'STATE 5 r 11726 11725',
'STATE 6 r 11726 11899',
'STATE 6 y 11726 11900',
'STATE 4 e 11726 11725',
'STATE 6 l 11726 11725',
'STATE 4 n 11725 11901',
'STATE 6 a 11898 11902',
'STATE 4 l 11904 11903',
'STATE 6 b 11726 11905',
'STATE 6 s 11726 11725',
'STATE 5 l 11907 11906',
'STATE 6 s 11909 11908',
'STATE 6 # 11910 11904',
'STATE 4 k 11725 11911',
'STATE 4 e 11725 11726',
'STATE 4 a 11725 11726',
'STATE 6 i 11725 11912',
'STATE 6 m 11725 11913',
'STATE 6 # 11725 11914',
'STATE 4 i 11725 11915',
'STATE 6 r 11726 11725',
'INDEX 11916 u',
'STATE 4 r 11918 11917',
'STATE 3 a 11920 11919',
'STATE 5 y 11922 11921',
'STATE 3 o 11924 11923',
'PHONE epsilon',
'STATE 5 i 11926 11925',
'STATE 3 b 11927 11920',
'STATE 3 q 11929 11928',
'STATE 4 p 11931 11930',
'STATE 5 o 11933 11932',
'STATE 3 # 11935 11934',
'PHONE eh1',
'STATE 5 # 11937 11936',
'STATE 4 e 11939 11938',
'STATE 4 i 11941 11940',
'STATE 5 l 11943 11942',
'STATE 5 a 11945 11944',
'STATE 3 e 11947 11946',
'STATE 3 p 11949 11948',
'STATE 6 n 11951 11950',
'STATE 5 e 11953 11952',
'STATE 4 e 11955 11954',
'STATE 5 a 11957 11956',
'STATE 5 # 11920 11958',
'STATE 4 s 11960 11959',
'STATE 5 s 11961 11957',
'PHONE uw1',
'PHONE ah1',
'STATE 5 e 11963 11962',
'STATE 6 # 11965 11964',
'STATE 3 a 11920 11966',
'STATE 6 p 11949 11967',
'STATE 3 c 11949 11968',
'PHONE y-uh1',
'PHONE y-er',
'PHONE y-er1',
'STATE 5 i 11970 11969',
'STATE 4 s 11972 11971',
'STATE 4 a 11974 11973',
'STATE 3 g 11920 11975',
'STATE 6 # 11977 11976',
'PHONE w',
'STATE 6 # 11979 11978',
'STATE 4 n 11981 11980',
'STATE 5 s 11942 11982',
'PHONE uw',
'STATE 5 r 11984 11983',
'STATE 3 t 11920 11985',
'STATE 3 t 11920 11986',
'PHONE uh1',
'STATE 6 # 11965 11987',
'STATE 6 l 11989 11988',
'STATE 3 f 11949 11990',
'STATE 5 a 11992 11991',
'STATE 3 # 11994 11993',
'STATE 3 # 11996 11995',
'STATE 6 s 11998 11997',
'STATE 4 s 12000 11999',
'STATE 3 g 11957 12001',
'STATE 3 r 11942 12002',
'STATE 4 a 11957 12003',
'STATE 5 l 11957 12004',
'STATE 5 t 11920 12005',
'STATE 5 z 11957 11920',
'STATE 4 t 12007 12006',
'STATE 5 g 12009 12008',
'STATE 5 # 11920 12010',
'STATE 5 u 12011 11920',
'STATE 3 d 12012 11920',
'STATE 3 s 12014 12013',
'STATE 3 c 12016 12015',
'STATE 6 n 11942 11920',
'STATE 6 s 11965 12017',
'PHONE uh',
'STATE 3 b 11927 12018',
'STATE 4 e 12020 12019',
'STATE 4 l 12022 12021',
'STATE 3 m 12024 12023',
'STATE 6 n 11943 12025',
'STATE 3 r 12027 12026',
'STATE 4 n 12029 12028',
'STATE 3 e 12031 12030',
'STATE 3 r 12000 12032',
'STATE 4 m 12000 12033',
'PHONE ah',
'STATE 3 h 11961 11942',
'STATE 3 s 11961 12034',
'STATE 4 i 11957 12035',
'STATE 5 d 11957 12036',
'STATE 5 u 11920 12037',
'STATE 5 h 12039 12038',
'STATE 6 e 12041 12040',
'STATE 6 r 11943 11920',
'STATE 6 e 11920 11943',
'STATE 5 t 12042 11920',
'STATE 3 a 11920 11965',
'PHONE ao1',
'STATE 3 a 11920 12043',
'STATE 6 # 11920 12044',
'STATE 3 g 11950 12045',
'STATE 6 t 11949 11950',
'PHONE y-uw1',
'STATE 3 g 11949 12046',
'STATE 4 a 12048 12047',
'STATE 3 g 11920 12049',
'STATE 3 # 12051 12050',
'STATE 6 t 12053 12052',
'STATE 3 c 12055 12054',
'STATE 4 n 12057 12056',
'STATE 6 m 11943 12058',
'STATE 3 s 12060 12059',
'STATE 4 m 12062 12061',
'STATE 4 t 12017 12000',
'STATE 6 a 12000 12063',
'STATE 3 m 12017 12064',
'STATE 6 r 11920 12017',
'STATE 3 b 11943 12000',
'STATE 3 f 12000 12065',
'STATE 3 d 11942 12066',
'STATE 4 o 11957 11942',
'STATE 5 t 11957 12067',
'STATE 5 r 11920 12068',
'STATE 5 l 12070 12069',
'STATE 4 g 11920 12071',
'STATE 5 e 12073 12072',
'STATE 5 h 11943 11920',
'STATE 6 # 11920 12074',
'STATE 3 g 11950 12075',
'STATE 6 r 11965 12076',
'STATE 3 o 11920 12077',
'STATE 3 m 11949 12078',
'STATE 4 i 12080 12079',
'STATE 3 g 12082 12081',
'STATE 3 r 12084 12083',
'STATE 6 # 11942 12085',
'STATE 4 n 12086 12017',
'STATE 6 r 12088 12087',
'STATE 3 t 12000 12089',
'STATE 3 b 12091 12090',
'STATE 4 s 12093 12092',
'STATE 6 o 11942 12017',
'STATE 6 c 12017 12094',
'STATE 4 n 12017 12095',
'STATE 3 l 12097 12096',
'STATE 4 l 12000 11942',
'STATE 6 t 11961 12098',
'STATE 6 n 12000 11942',
'STATE 6 x 11943 12099',
'STATE 3 f 12017 12100',
'STATE 3 e 12102 12101',
'STATE 3 n 12104 12103',
'STATE 5 n 11957 12105',
'STATE 6 y 11920 11957',
'STATE 4 x 11942 12106',
'STATE 4 b 11943 12107',
'STATE 6 a 11943 12108',
'STATE 5 a 11961 11920',
'STATE 6 d 11920 12109',
'STATE 6 i 11942 11920',
'STATE 3 o 11920 12110',
'STATE 6 s 11920 12111',
'STATE 3 m 11920 12112',
'STATE 3 d 11965 12113',
'STATE 3 # 12115 12114',
'STATE 3 g 12117 12116',
'STATE 5 t 12119 12118',
'STATE 5 r 11920 12120',
'STATE 6 # 12122 12121',
'STATE 5 n 11965 11942',
'STATE 6 t 12124 12123',
'STATE 6 b 12017 12125',
'STATE 3 t 11942 12126',
'STATE 3 c 12128 12127',
'STATE 3 s 12000 12129',
'STATE 3 l 11942 12130',
'STATE 4 s 12132 12131',
'STATE 6 e 12128 12133',
'STATE 6 n 12017 11942',
'STATE 6 z 12128 12134',
'STATE 4 t 12017 11961',
'STATE 3 t 12136 12135',
'STATE 6 n 12137 11942',
'STATE 6 l 11942 12138',
'STATE 6 m 11943 12000',
'STATE 3 b 12017 12139',
'STATE 4 i 11942 12140',
'STATE 4 x 11920 12141',
'STATE 3 l 11942 12142',
'PHONE y-uw',
'STATE 4 i 11957 11920',
'STATE 4 e 12144 12143',
'STATE 4 e 11961 11920',
'STATE 6 e 11920 12145',
'STATE 6 r 11920 11942',
'STATE 3 b 12147 12146',
'STATE 6 d 11965 11920',
'STATE 3 # 11949 12148',
'STATE 3 k 11920 12149',
'STATE 5 o 12151 12150',
'STATE 4 n 12153 12152',
'STATE 3 b 11920 12154',
'STATE 6 h 11957 12155',
'STATE 5 l 12157 12156',
'STATE 3 t 11961 12158',
'STATE 6 a 11957 12159',
'STATE 3 l 12161 12160',
'STATE 3 s 11942 12162',
'STATE 3 h 12164 12163',
'STATE 4 t 12166 12165',
'STATE 6 n 11943 12167',
'STATE 3 c 12128 12168',
'STATE 3 g 12128 12169',
'PHONE y-ah',
'STATE 3 d 12000 12128',
'STATE 3 p 12171 12170',
'STATE 6 o 12017 12172',
'PHONE ih1',
'STATE 6 v 12128 12173',
'STATE 6 s 12128 12017',
'STATE 3 d 12175 12174',
'STATE 4 d 11942 12176',
'STATE 4 m 11943 11942',
'STATE 6 n 12177 11942',
'STATE 3 # 12017 12178',
'STATE 4 y 11920 12179',
'STATE 4 p 11943 11961',
'STATE 3 c 12104 12017',
'STATE 5 t 11920 12180',
'STATE 6 t 11961 11942',
'STATE 6 i 11920 12181',
'STATE 3 c 12183 12182',
'STATE 6 a 11949 11927',
'STATE 6 s 11965 12184',
'STATE 3 a 11920 12185',
'STATE 4 # 12187 12186',
'STATE 6 # 12189 12188',
'STATE 4 p 12191 12190',
'STATE 5 d 12193 12192',
'STATE 3 c 12195 12194',
'STATE 5 t 11920 12196',
'STATE 3 t 12198 12197',
'STATE 3 n 12104 12199',
'STATE 3 d 12201 12200',
'STATE 5 n 11957 12202',
'STATE 3 b 12204 12203',
'STATE 5 n 11961 12205',
'STATE 3 n 12207 12206',
'STATE 3 s 12209 12208',
'STATE 4 m 12017 11942',
'STATE 3 c 12128 12210',
'STATE 3 p 12128 12104',
'STATE 6 l 12000 11943',
'STATE 3 m 12128 12211',
'STATE 3 p 12128 12000',
'STATE 3 f 12213 12212',
'STATE 6 s 11943 12017',
'STATE 4 t 12104 12093',
'STATE 6 n 12128 12017',
'STATE 3 e 12215 12214',
'STATE 4 l 11961 11942',
'STATE 4 t 11942 12216',
'STATE 4 d 11942 12217',
'STATE 3 c 12017 12218',
'STATE 4 z 11961 12219',
'STATE 4 # 11942 12220',
'STATE 6 # 11920 11943',
'STATE 3 p 11949 12221',
'STATE 6 # 11950 11949',
'STATE 3 a 11920 12222',
'STATE 3 o 11920 12223',
'STATE 4 o 12225 12224',
'STATE 3 c 12104 12226',
'STATE 6 u 12228 12227',
'STATE 3 c 12017 11942',
'STATE 5 u 12017 12229',
'STATE 5 h 12000 12230',
'STATE 5 o 11943 12231',
'STATE 6 o 12000 12232',
'STATE 5 n 12234 12233',
'STATE 5 s 11957 11920',
'STATE 6 g 11920 12235',
'STATE 3 n 11942 12236',
'STATE 6 y 11961 12237',
'STATE 3 d 11961 12238',
'STATE 3 l 12104 12239',
'PHONE ah-w',
'STATE 6 # 11920 11957',
'STATE 3 m 12241 12240',
'STATE 5 n 11957 12017',
'STATE 6 e 11965 11942',
'STATE 3 l 12243 12242',
'STATE 5 l 11989 12104',
'STATE 3 g 12245 12244',
'STATE 6 r 11965 12246',
'STATE 4 m 11942 12247',
'STATE 6 # 11942 12248',
'STATE 3 h 12250 12249',
'STATE 4 j 11942 12017',
'STATE 4 y 11920 12251',
'STATE 4 g 12104 12252',
'STATE 4 n 12253 11942',
'STATE 4 b 11942 12000',
'STATE 6 n 11942 12254',
'STATE 4 l 12000 12255',
'STATE 4 v 12257 12256',
'STATE 3 # 11951 12258',
'STATE 6 l 11920 12259',
'STATE 3 e 11920 12260',
'STATE 3 e 12262 12261',
'STATE 5 u 12264 12263',
'STATE 3 e 11942 12265',
'STATE 3 c 12267 12266',
'STATE 4 l 12128 11943',
'STATE 4 h 12269 12268',
'STATE 5 d 12000 12270',
'STATE 6 o 12272 12271',
'STATE 6 e 11943 12273',
'STATE 3 m 11942 12274',
'STATE 3 r 11942 12275',
'STATE 6 d 11920 12276',
'STATE 5 n 12278 12277',
'STATE 6 i 11961 11942',
'STATE 3 t 12280 12279',
'STATE 3 n 12104 12128',
'STATE 3 f 12282 12281',
'STATE 5 l 12104 12017',
'STATE 3 d 11942 12283',
'STATE 5 d 12104 11942',
'STATE 3 n 12285 12284',
'STATE 6 n 11920 12000',
'STATE 6 n 11961 11942',
'STATE 3 d 12000 12286',
'STATE 3 d 12000 11942',
'STATE 3 k 12288 12287',
'STATE 4 m 12104 12289',
'STATE 6 n 12291 12290',
'STATE 6 r 12293 12292',
'STATE 6 # 11942 12000',
'STATE 3 r 11942 12294',
'STATE 4 h 11961 12295',
'STATE 5 a 12297 12296',
'STATE 5 i 11961 11942',
'STATE 6 k 11965 12298',
'STATE 6 n 12300 12299',
'STATE 6 n 12302 12301',
'STATE 5 u 12304 12303',
'STATE 4 p 11943 12305',
'STATE 5 r 11920 12306',
'STATE 3 n 12104 12307',
'STATE 3 g 11961 12308',
'STATE 3 b 12310 12309',
'STATE 4 t 12128 11942',
'STATE 4 m 11943 12311',
'STATE 6 i 11965 11942',
'STATE 5 p 11943 12312',
'STATE 5 u 12000 12313',
'STATE 5 c 11943 12000',
'STATE 6 i 11943 12000',
'STATE 3 s 11942 12314',
'STATE 6 g 11942 12128',
'STATE 6 e 11920 12315',
'STATE 3 s 11957 12316',
'STATE 3 h 11957 12317',
'STATE 3 s 12201 12318',
'STATE 6 i 12201 11961',
'STATE 5 r 12320 12319',
'STATE 5 l 12017 11957',
'STATE 5 r 11920 12321',
'STATE 3 c 12173 12322',
'STATE 4 f 12128 11942',
'STATE 4 g 12000 12323',
'STATE 3 g 11961 12324',
'STATE 6 c 12017 12325',
'STATE 6 a 11942 12326',
'STATE 3 n 12328 12327',
'STATE 4 m 12128 12329',
'STATE 6 # 11943 12330',
'STATE 4 v 11942 12331',
'STATE 6 # 12017 11942',
'STATE 4 k 12000 12332',
'STATE 4 a 11961 12333',
'STATE 4 g 11942 12334',
'STATE 3 l 11965 12335',
'STATE 3 d 11965 11920',
'STATE 3 d 11920 12336',
'STATE 6 z 11920 12337',
'STATE 3 t 11920 12338',
'STATE 3 s 12340 12339',
'STATE 3 c 12342 12341',
'STATE 5 r 11942 12343',
'STATE 6 # 11920 12344',
'STATE 3 c 12104 12345',
'STATE 3 n 11961 12346',
'STATE 3 m 12104 12347',
'STATE 4 t 12128 12017',
'STATE 5 t 11943 12348',
'STATE 6 i 11943 12349',
'STATE 5 w 12000 12350',
'STATE 6 # 12352 12351',
'STATE 5 l 11920 12353',
'STATE 5 d 11957 12354',
'STATE 3 j 11957 11961',
'STATE 3 x 12201 11961',
'STATE 6 a 11957 12355',
'STATE 3 e 11920 12356',
'STATE 3 c 12017 12357',
'STATE 4 t 12359 12358',
'STATE 3 t 12000 12360',
'STATE 6 e 12362 12361',
'STATE 4 l 12104 12062',
'STATE 4 l 12017 11942',
'STATE 3 j 11942 12363',
'STATE 4 t 12000 11942',
'STATE 4 l 12128 12364',
'STATE 6 n 11920 11942',
'STATE 4 t 11942 12365',
'STATE 3 n 11943 12366',
'STATE 6 u 11920 12367',
'STATE 6 # 11942 12175',
'STATE 6 t 11920 12368',
'STATE 3 s 11965 12017',
'STATE 3 j 11965 12369',
'STATE 3 s 11965 11920',
'STATE 4 l 12371 12370',
'STATE 5 r 12373 12372',
'STATE 4 t 12375 12374',
'STATE 4 m 12017 12128',
'STATE 6 a 12104 12376',
'STATE 5 y 11920 12377',
'STATE 3 t 12000 11961',
'STATE 3 l 11961 12378',
'STATE 4 b 12380 12379',
'STATE 5 o 12104 12381',
'STATE 6 a 11943 12382',
'STATE 6 a 12000 12383',
'STATE 5 d 11961 12384',
'STATE 3 r 11942 11961',
'STATE 6 t 11957 12385',
'STATE 5 r 11957 11942',
'STATE 3 k 11965 12386',
'STATE 3 h 11920 11965',
'STATE 5 l 11942 12017',
'STATE 6 m 11942 12387',
'STATE 3 r 11942 12388',
'STATE 4 n 11961 11942',
'STATE 3 e 12390 12389',
'STATE 4 l 11942 12391',
'STATE 3 h 12393 12392',
'STATE 3 j 11942 12017',
'STATE 4 b 11920 11942',
'STATE 4 g 11943 12394',
'STATE 6 r 11920 12395',
'STATE 3 j 11920 12396',
'STATE 6 s 11920 12397',
'STATE 5 h 12399 12398',
'STATE 3 f 12401 12400',
'STATE 4 b 12403 12402',
'STATE 6 e 11961 12404',
'STATE 6 # 11942 12405',
'STATE 3 m 12017 12406',
'STATE 6 c 11920 12407',
'STATE 3 b 11957 12408',
'STATE 3 r 11961 12409',
'STATE 4 d 11942 12410',
'STATE 3 s 12000 12411',
'STATE 4 z 11961 11943',
'STATE 5 t 11943 12412',
'STATE 5 f 12000 12413',
'STATE 3 h 11920 12414',
'STATE 6 i 11920 12415',
'STATE 3 h 12417 12416',
'STATE 3 e 12419 12418',
'STATE 3 l 11961 12420',
'STATE 3 n 11942 12421',
'STATE 6 c 12017 11942',
'STATE 3 r 11942 11943',
'STATE 4 t 12423 12422',
'STATE 4 l 11942 12424',
'STATE 4 p 12426 12425',
'STATE 6 t 11961 12427',
'STATE 3 e 11920 12428',
'STATE 3 t 11965 12429',
'STATE 6 # 12431 12430',
'STATE 4 s 12433 12432',
'STATE 3 b 12435 12434',
'STATE 6 y 12000 12436',
'STATE 5 p 12438 12437',
'STATE 6 c 12000 12439',
'STATE 6 a 11942 11943',
'STATE 4 k 11961 12440',
'STATE 6 r 12017 11942',
'STATE 6 i 11942 12441',
'STATE 5 m 11957 12442',
'STATE 3 k 11942 12443',
'STATE 3 h 12164 12444',
'STATE 3 d 12000 11943',
'STATE 6 r 12000 12445',
'STATE 5 l 12000 12446',
'STATE 3 d 11961 11942',
'STATE 6 n 11920 12447',
'STATE 3 # 11965 12448',
'STATE 5 t 11942 12449',
'STATE 6 s 12451 12450',
'STATE 6 l 12017 11942',
'STATE 3 b 12128 12452',
'STATE 6 o 11942 12453',
'STATE 3 m 12455 12454',
'STATE 3 b 12104 12017',
'STATE 4 t 11942 12456',
'STATE 3 m 12000 12457',
'STATE 3 r 12000 12458',
'STATE 5 i 12460 12459',
'STATE 3 n 11950 12461',
'STATE 6 a 11965 12462',
'STATE 4 y 11920 12463',
'STATE 5 y 12465 12464',
'STATE 4 t 12467 12466',
'STATE 3 b 11965 12468',
'STATE 3 p 12470 12469',
'STATE 5 l 11965 12471',
'STATE 5 n 12000 12472',
'STATE 5 c 12474 12473',
'STATE 4 m 11943 12475',
'STATE 6 l 11943 12476',
'STATE 6 l 11942 12477',
'STATE 6 # 12478 11920',
'STATE 6 o 11957 12479',
'STATE 3 m 11942 11961',
'STATE 3 e 12481 12480',
'STATE 6 e 12000 11943',
'STATE 5 h 12000 12482',
'STATE 6 o 11920 12483',
'STATE 6 d 11942 12484',
'STATE 5 l 11942 11965',
'STATE 3 k 11961 12485',
'STATE 3 l 11942 12486',
'STATE 3 p 12128 12017',
'STATE 6 # 11942 12487',
'STATE 4 p 11942 12488',
'STATE 4 n 12489 12017',
'STATE 6 # 12017 12164',
'STATE 3 b 12000 12490',
'STATE 3 d 11943 12491',
'STATE 5 f 11920 12492',
'STATE 6 n 12494 12493',
'STATE 6 # 11965 12495',
'STATE 3 l 11965 11920',
'STATE 5 c 12497 12496',
'STATE 5 d 12499 12498',
'STATE 4 d 11942 12500',
'STATE 4 g 12502 12501',
'STATE 6 # 12504 12503',
'STATE 3 p 11965 12505',
'STATE 5 l 12507 12506',
'STATE 5 l 11965 12508',
'STATE 6 a 12000 11943',
'STATE 5 f 11989 12509',
'STATE 4 g 12000 12510',
'STATE 6 e 12000 12511',
'STATE 6 h 11943 12512',
'STATE 6 a 11943 12513',
'STATE 6 s 11942 12514',
'STATE 5 z 11920 11942',
'STATE 5 n 11961 11942',
'STATE 6 w 12000 12515',
'STATE 6 g 12017 11920',
'STATE 5 v 12000 12516',
'STATE 5 n 11920 12517',
'STATE 3 c 11957 12518',
'STATE 3 f 11961 12519',
'STATE 4 d 11943 11942',
'STATE 4 c 11942 12520',
'STATE 3 z 11942 12521',
'STATE 6 # 12017 12104',
'STATE 3 r 12523 12522',
'STATE 3 s 12000 12524',
'STATE 5 # 12526 12525',
'STATE 4 l 11942 11920',
'STATE 4 l 11961 11920',
'STATE 3 d 11965 11949',
'STATE 5 l 12528 12527',
'STATE 4 c 12530 12529',
'STATE 5 r 11920 12531',
'STATE 4 n 12532 11943',
'STATE 3 l 11942 12533',
'STATE 6 i 11942 12534',
'STATE 6 e 12017 11943',
'STATE 3 r 12535 11943',
'STATE 3 m 11961 12536',
'STATE 3 c 11965 11943',
'STATE 5 y 11942 12537',
'STATE 6 o 11942 11943',
'STATE 6 o 11965 12538',
'STATE 6 i 11965 12539',
'STATE 6 a 12541 12540',
'STATE 4 c 12000 11943',
'STATE 4 s 12000 12542',
'STATE 5 m 12000 12543',
'STATE 6 e 12545 12544',
'STATE 3 d 11961 12546',
'STATE 5 t 12000 12547',
'STATE 6 a 11920 11957',
'STATE 3 t 11961 12548',
'STATE 3 b 11961 12549',
'STATE 6 g 11943 12550',
'STATE 6 l 11942 12551',
'STATE 3 p 11943 12552',
'STATE 4 n 11943 12553',
'STATE 3 h 12000 11943',
'STATE 6 a 11942 12554',
'STATE 4 d 11920 11942',
'STATE 5 r 12556 12555',
'STATE 4 h 11942 12557',
'STATE 3 b 12558 11943',
'STATE 6 i 12560 12559',
'STATE 3 i 12000 12561',
'STATE 3 f 11943 12562',
'STATE 4 b 11942 12128',
'STATE 3 b 12564 12563',
'STATE 6 e 11943 11942',
'STATE 3 l 11942 12565',
'STATE 3 h 11943 12566',
'STATE 6 i 11943 12567',
'STATE 6 o 11965 12568',
'STATE 5 h 11943 12569',
'STATE 4 s 12000 12570',
'STATE 6 o 12000 12571',
'STATE 6 o 12000 12572',
'STATE 4 s 12017 12573',
'STATE 4 g 12017 12000',
'STATE 6 t 11961 12574',
'STATE 5 b 12575 12000',
'STATE 3 d 11942 12576',
'STATE 3 m 11961 12577',
'STATE 4 m 12579 12578',
'STATE 6 r 12017 12580',
'STATE 3 h 11943 12581',
'STATE 4 t 12000 11943',
'STATE 5 e 12493 12582',
'STATE 4 z 12584 12583',
'STATE 4 h 11965 12585',
'STATE 4 c 11942 12586',
'STATE 4 n 11943 12587',
'STATE 6 o 11942 11961',
'STATE 3 b 11961 11942',
'STATE 3 p 12589 12588',
'STATE 3 l 12000 12590',
'STATE 3 d 11943 12591',
'STATE 6 a 12104 11943',
'STATE 3 r 11961 11942',
'STATE 5 f 12593 12592',
'STATE 5 t 12000 11943',
'STATE 5 l 11965 12594',
'STATE 4 f 12596 12595',
'STATE 4 m 12000 11943',
'STATE 6 r 12000 12167',
'STATE 5 v 12000 12597',
'STATE 6 a 11942 12598',
'STATE 3 r 11942 12599',
'STATE 6 e 11943 12000',
'STATE 5 l 11961 12600',
'STATE 4 k 11961 12601',
'STATE 3 s 11961 12602',
'STATE 6 n 11942 11961',
'STATE 4 l 12604 12603',
'STATE 3 c 11943 12605',
'STATE 4 l 11920 12606',
'STATE 4 s 12608 12607',
'STATE 6 l 11943 12609',
'STATE 3 n 11961 12610',
'STATE 4 p 12612 12611',
'STATE 4 s 11943 11965',
'STATE 5 l 11943 12613',
'STATE 5 s 11965 11943',
'STATE 3 m 12000 11943',
'STATE 3 r 11943 12614',
'STATE 6 a 12567 12615',
'STATE 3 g 11943 12000',
'STATE 6 # 11965 11943',
'STATE 5 d 11943 12616',
'STATE 6 i 12000 11943',
'STATE 5 l 12000 12617',
'STATE 4 b 12000 12618',
'STATE 4 s 12000 12619',
'STATE 3 s 11957 12620',
'STATE 4 n 12622 12621',
'STATE 6 l 12624 12623',
'STATE 6 # 12017 12625',
'STATE 3 c 12017 12626',
'STATE 3 l 12628 12627',
'STATE 6 e 11942 11920',
'STATE 4 h 12630 12629',
'STATE 5 s 12632 12631',
'STATE 6 e 12634 12633',
'STATE 3 l 11942 12635',
'STATE 4 g 12637 12636',
'STATE 3 d 11942 12391',
'STATE 5 z 12524 12638',
'STATE 3 h 11943 12639',
'STATE 6 # 11943 12640',
'STATE 5 s 12642 12641',
'STATE 6 r 12000 12643',
'STATE 3 d 12000 12644',
'STATE 6 n 12646 12645',
'STATE 3 p 11957 11942',
'STATE 4 s 11961 12647',
'STATE 3 d 11943 11942',
'STATE 4 l 12649 12648',
'STATE 4 b 11942 12017',
'STATE 4 g 12017 12650',
'STATE 3 b 12017 12104',
'STATE 3 g 12652 12651',
'STATE 4 b 11943 12000',
'STATE 6 o 11943 12653',
'STATE 6 e 11942 11943',
'STATE 5 t 12654 11943',
'STATE 3 r 12507 11943',
'STATE 5 z 12655 11943',
'STATE 3 b 11943 11961',
'STATE 6 i 11943 12656',
'STATE 6 e 12658 12657',
'STATE 3 g 11961 11943',
'STATE 5 m 11943 12659',
'STATE 6 # 12000 11943',
'STATE 6 y 12000 12660',
'STATE 4 l 12661 11943',
'STATE 4 s 11943 12652',
'STATE 5 t 11943 12662',
'STATE 6 r 12000 11943',
'STATE 4 l 12000 12663',
'STATE 3 l 11961 11942',
'STATE 4 v 11961 12664',
'STATE 3 j 11942 12665',
'STATE 3 j 11942 12666',
'STATE 6 s 12017 12667',
'STATE 3 t 11943 12000',
'STATE 4 n 11943 12000',
'STATE 3 c 12669 12668',
'STATE 6 r 12671 12670',
'STATE 6 i 11942 12507',
'STATE 3 d 12628 11943',
'STATE 6 a 12672 11943',
'STATE 4 b 11942 11943',
'STATE 4 m 12674 12673',
'STATE 5 c 11943 12675',
'STATE 6 o 12000 11943',
'STATE 5 j 12000 12676',
'STATE 6 s 11942 12677',
'STATE 6 y 11961 12678',
'STATE 4 s 11942 12679',
'STATE 3 d 11942 12680',
'STATE 4 n 12017 12681',
'STATE 3 p 11943 12682',
'STATE 4 m 12683 11943',
'STATE 6 a 12684 11943',
'STATE 3 l 12000 11943',
'STATE 4 n 11943 11942',
'STATE 4 t 12686 12685',
'STATE 5 p 11943 12687',
'STATE 5 t 11943 12688',
'STATE 5 s 12596 11943',
'STATE 3 l 11942 12360',
'STATE 3 r 12690 12689',
'STATE 3 d 11942 12691',
'STATE 3 t 11961 11942',
'STATE 6 d 12017 11942',
'STATE 5 z 12693 12692',
'STATE 5 s 12000 12694',
'STATE 3 g 11943 12695',
'STATE 3 j 11943 12696',
'STATE 5 s 11943 12697',
'STATE 5 b 11943 12000',
'STATE 3 c 11943 12698',
'STATE 3 t 11942 12699',
'STATE 6 l 11961 11942',
'STATE 4 d 11942 12700',
'STATE 3 i 12000 12701',
'STATE 6 i 12702 11943',
'STATE 5 v 12000 11943',
'STATE 3 b 11943 12703',
'STATE 5 n 11943 12704',
'STATE 3 c 12000 11943',
'STATE 3 v 11943 12705',
'STATE 6 r 12707 12706',
'STATE 6 s 11942 12708',
'STATE 4 k 12000 12709',
'STATE 4 d 11943 12710',
'STATE 3 m 11943 12000',
'STATE 3 b 11943 12711',
'STATE 3 g 11943 12712',
'STATE 4 d 11942 12713',
'STATE 3 l 11942 11961',
'STATE 6 c 11942 12714',
'STATE 3 l 12716 12715',
'STATE 3 n 11961 11943',
'STATE 3 g 11943 12717',
'STATE 5 g 11943 12718',
'STATE 6 l 11942 12719',
'STATE 3 t 11942 12720',
'STATE 4 m 11943 12721',
'STATE 5 n 12000 12722',
'STATE 4 g 11943 12723',
'STATE 5 s 11943 12724',
'STATE 4 b 11942 12725',
'STATE 4 b 11942 12726',
'STATE 3 r 11943 12727',
'STATE 6 a 12729 12728',
'STATE 3 d 11943 12730',
'STATE 5 k 11943 12731',
'STATE 3 d 12000 12246',
'STATE 6 n 11942 12732',
'STATE 6 a 12734 12733',
'STATE 5 w 11943 12735',
'STATE 4 n 12000 11943',
'STATE 5 g 12737 12736',
'STATE 6 o 11943 12703',
'STATE 6 a 11942 11961',
'STATE 6 s 12590 11943',
'STATE 5 t 11943 12738',
'STATE 5 t 12739 11943',
'STATE 3 f 11943 12740',
'STATE 3 t 12000 11943',
'STATE 3 n 12000 11943',
'STATE 4 t 11943 12445',
'STATE 4 p 12741 11943',
'STATE 5 p 11943 12742',
'STATE 3 r 12000 11943',
'INDEX 12743 v',
'STATE 4 v 12745 12744',
'PHONE v',
'PHONE epsilon',
'INDEX 12746 w',
'STATE 3 o 12748 12747',
'STATE 3 e 12750 12749',
'STATE 5 k 12752 12751',
'STATE 3 a 12754 12753',
'STATE 5 k 12756 12755',
'STATE 6 z 12758 12757',
'STATE 6 i 12756 12759',
'STATE 4 r 12759 12760',
'STATE 4 a 12762 12761',
'STATE 6 z 12764 12763',
'PHONE f',
'STATE 4 a 12766 12765',
'STATE 5 c 12764 12767',
'PHONE epsilon',
'STATE 5 o 12769 12768',
'STATE 5 k 12756 12770',
'STATE 5 l 12759 12767',
'STATE 4 o 12762 12771',
'PHONE v',
'STATE 6 l 12773 12772',
'STATE 5 y 12767 12774',
'PHONE w',
'STATE 3 u 12776 12775',
'STATE 4 h 12777 12767',
'STATE 4 o 12767 12778',
'STATE 4 a 12780 12779',
'STATE 4 o 12767 12759',
'STATE 5 l 12767 12759',
'STATE 1 c 12767 12759',
'STATE 5 r 12782 12781',
'STATE 1 # 12767 12783',
'STATE 6 l 12759 12784',
'STATE 4 h 12767 12785',
'STATE 4 i 12787 12786',
'STATE 5 y 12767 12788',
'STATE 4 e 12767 12789',
'STATE 3 s 12790 12767',
'STATE 4 e 12767 12759',
'STATE 6 o 12767 12759',
'STATE 4 i 12792 12791',
'STATE 4 e 12794 12793',
'STATE 6 d 12767 12795',
'STATE 1 # 12759 12796',
'STATE 5 s 12798 12797',
'STATE 6 t 12767 12799',
'STATE 4 e 12801 12800',
'STATE 6 g 12759 12802',
'STATE 4 h 12802 12759',
'STATE 6 l 12802 12803',
'STATE 6 e 12767 12804',
'STATE 6 e 12767 12805',
'STATE 3 i 12806 12767',
'STATE 6 i 12764 12767',
'STATE 1 a 12759 12807',
'STATE 5 o 12808 12759',
'STATE 1 # 12809 12759',
'STATE 1 # 12759 12767',
'STATE 5 d 12759 12810',
'STATE 1 e 12767 12811',
'STATE 6 k 12767 12812',
'STATE 1 s 12764 12767',
'STATE 6 d 12759 12767',
'STATE 1 # 12813 12759',
'STATE 6 # 12759 12764',
'STATE 5 l 12759 12814',
'STATE 5 s 12759 12815',
'STATE 6 # 12759 12816',
'STATE 6 n 12759 12764',
'STATE 5 r 12759 12817',
'STATE 5 n 12759 12818',
'STATE 1 s 12759 12819',
'STATE 5 s 12759 12820',
'STATE 6 t 12759 12821',
'STATE 5 r 12767 12773',
'STATE 5 y 12759 12822',
'STATE 6 # 12759 12767',
'STATE 6 t 12759 12823',
'STATE 5 n 12759 12824',
'STATE 5 # 12759 12767',
'INDEX 12825 x',
'STATE 3 u 12827 12826',
'STATE 2 0 12829 12828',
'STATE 4 # 12831 12830',
'STATE 3 e 12832 12830',
'PHONE z',
'PHONE k-s',
'STATE 2 l 12830 12833',
'STATE 4 a 12835 12834',
'PHONE epsilon',
'STATE 4 u 12837 12836',
'STATE 2 # 12839 12838',
'STATE 4 h 12839 12840',
'STATE 2 s 12841 12839',
'STATE 2 l 12839 12830',
'PHONE g-z',
'STATE 4 e 12843 12842',
'PHONE k-sh',
'STATE 4 i 12845 12844',
'STATE 2 # 12839 12830',
'STATE 4 o 12839 12830',
'STATE 2 l 12830 12846',
'STATE 2 m 12830 12843',
'INDEX 12847 y',
'STATE 4 # 12849 12848',
'STATE 2 0 12851 12850',
'STATE 3 a 12853 12852',
'STATE 3 a 12855 12854',
'STATE 5 a 12857 12856',
'STATE 3 o 12853 12858',
'PHONE epsilon',
'STATE 3 o 12860 12859',
'STATE 4 a 12862 12861',
'PHONE y',
'STATE 4 e 12856 12863',
'STATE 3 f 12865 12864',
'STATE 3 e 12867 12866',
'STATE 4 o 12869 12868',
'STATE 4 u 12853 12870',
'STATE 5 m 12856 12871',
'PHONE iy',
'STATE 3 e 12873 12872',
'STATE 2 i 12875 12874',
'STATE 1 0 12877 12876',
'STATE 4 e 12879 12878',
'STATE 5 m 12856 12853',
'STATE 2 t 12856 12853',
'STATE 4 o 12853 12880',
'STATE 2 b 12853 12881',
'STATE 3 u 12875 12882',
'STATE 2 l 12863 12883',
'STATE 2 f 12863 12875',
'PHONE ay1',
'STATE 4 a 12885 12884',
'STATE 3 s 12887 12886',
'STATE 1 # 12853 12888',
'STATE 2 m 12890 12889',
'STATE 5 v 12875 12853',
'STATE 2 r 12853 12891',
'STATE 2 f 12892 12863',
'STATE 2 n 12863 12893',
'STATE 1 # 12895 12894',
'STATE 1 # 12897 12896',
'STATE 5 e 12875 12898',
'STATE 5 o 12900 12899',
'STATE 2 n 12863 12901',
'STATE 5 r 12903 12902',
'STATE 1 # 12875 12904',
'STATE 2 k 12875 12905',
'STATE 3 l 12875 12863',
'STATE 2 s 12863 12906',
'STATE 5 e 12908 12907',
'STATE 2 a 12910 12909',
'STATE 3 i 12912 12911',
'STATE 3 i 12856 12875',
'STATE 4 o 12914 12913',
'STATE 4 r 12916 12915',
'STATE 4 n 12917 12916',
'STATE 5 # 12863 12918',
'STATE 5 d 12853 12875',
'STATE 2 b 12875 12853',
'PHONE ay',
'STATE 5 n 12853 12919',
'STATE 2 v 12920 12863',
'STATE 4 n 12922 12921',
'STATE 3 r 12863 12923',
'STATE 2 o 12925 12924',
'STATE 3 n 12863 12916',
'STATE 5 n 12926 12856',
'STATE 5 # 12853 12856',
'STATE 4 u 12856 12927',
'STATE 3 k 12856 12928',
'STATE 5 e 12875 12916',
'PHONE ih1',
'PHONE ah',
'STATE 2 l 12863 12853',
'STATE 2 p 12853 12875',
'STATE 1 a 12863 12929',
'STATE 5 n 12931 12930',
'STATE 5 a 12904 12932',
'STATE 1 p 12934 12933',
'STATE 2 e 12936 12935',
'STATE 4 m 12916 12917',
'STATE 2 e 12856 12937',
'STATE 4 a 12939 12938',
'STATE 5 n 12875 12940',
'STATE 1 o 12863 12941',
'STATE 4 k 12943 12942',
'STATE 4 i 12944 12937',
'STATE 2 a 12946 12945',
'STATE 2 g 12916 12947',
'STATE 4 m 12917 12863',
'STATE 3 s 12875 12948',
'STATE 3 g 12917 12916',
'STATE 3 n 12856 12863',
'STATE 4 e 12875 12949',
'STATE 5 n 12904 12875',
'STATE 3 m 12875 12856',
'STATE 1 l 12863 12853',
'STATE 4 l 12951 12950',
'STATE 5 # 12952 12916',
'STATE 3 f 12875 12953',
'STATE 2 e 12917 12954',
'STATE 3 r 12952 12916',
'STATE 4 l 12956 12955',
'STATE 5 e 12875 12957',
'STATE 4 r 12959 12958',
'STATE 4 e 12961 12960',
'STATE 5 # 12917 12962',
'PHONE ih',
'STATE 3 l 12964 12963',
'STATE 5 s 12916 12965',
'STATE 3 d 12875 12966',
'STATE 2 s 12875 12917',
'STATE 4 e 12875 12967',
'STATE 3 h 12969 12968',
'STATE 5 o 12875 12970',
'STATE 4 o 12972 12971',
'STATE 3 u 12853 12973',
'STATE 3 r 12975 12974',
'STATE 2 r 12863 12976',
'STATE 2 l 12863 12875',
'STATE 1 r 12917 12977',
'STATE 4 n 12875 12978',
'STATE 3 k 12875 12979',
'STATE 4 n 12981 12980',
'STATE 5 n 12916 12982',
'STATE 5 a 12875 12983',
'STATE 4 u 12856 12984',
'STATE 5 # 12856 12863',
'STATE 5 # 12875 12985',
'STATE 2 a 12917 12986',
'STATE 1 m 12917 12863',
'STATE 3 r 12875 12863',
'STATE 5 i 12916 12987',
'STATE 4 p 12875 12988',
'STATE 4 o 12990 12989',
'STATE 5 o 12992 12991',
'STATE 3 d 12875 12916',
'STATE 5 t 12916 12875',
'STATE 3 l 12916 12993',
'STATE 4 r 12995 12994',
'STATE 3 m 12863 12996',
'STATE 3 b 12916 12917',
'STATE 3 w 12916 12997',
'STATE 1 e 12999 12998',
'STATE 3 u 12875 13000',
'STATE 3 r 12875 12853',
'STATE 3 c 13002 13001',
'STATE 4 c 12904 12875',
'STATE 3 m 12853 13003',
'STATE 5 t 13005 13004',
'STATE 2 o 12863 12853',
'STATE 3 l 13006 12856',
'STATE 3 s 12916 13007',
'STATE 2 a 12875 13008',
'STATE 2 n 12875 12863',
'STATE 4 n 12916 13009',
'STATE 5 a 13011 13010',
'STATE 4 c 12875 13012',
'STATE 5 i 12875 12916',
'STATE 4 c 13014 13013',
'STATE 3 l 12917 12916',
'STATE 5 a 12856 12863',
'STATE 2 r 12916 13015',
'STATE 1 c 12863 13016',
'STATE 5 n 12875 13017',
'STATE 3 m 12916 13018',
'STATE 3 l 12875 13019',
'STATE 4 p 12875 12916',
'STATE 4 i 12972 13020',
'STATE 3 c 13022 13021',
'STATE 1 e 12917 12916',
'STATE 3 l 12863 13023',
'STATE 4 l 12875 13024',
'STATE 4 m 12916 13025',
'STATE 3 b 12875 13026',
'STATE 3 l 13028 13027',
'STATE 3 z 12916 13029',
'STATE 2 i 12952 12875',
'STATE 2 o 12875 13030',
'STATE 5 o 12875 13031',
'STATE 3 r 12916 13032',
'STATE 4 l 12916 13033',
'STATE 3 z 13035 13034',
'STATE 2 l 13037 13036',
'STATE 2 o 13038 12916',
'STATE 4 r 12875 13039',
'STATE 2 f 12875 13040',
'STATE 4 i 12875 13041',
'STATE 4 b 12916 12875',
'STATE 4 d 13043 13042',
'STATE 4 b 12917 12875',
'STATE 4 t 13045 13044',
'STATE 4 s 12917 12863',
'STATE 1 p 12952 12863',
'STATE 3 h 12875 12904',
'STATE 2 s 13047 13046',
'STATE 5 u 12916 13048',
'STATE 1 o 12916 13049',
'STATE 3 h 12875 12863',
'STATE 5 i 12917 13050',
'STATE 2 a 12916 12917',
'STATE 2 p 12916 13051',
'STATE 3 z 12916 12875',
'STATE 3 z 12916 13052',
'STATE 1 r 12916 13053',
'STATE 1 p 13054 12916',
'STATE 4 p 12916 13055',
'STATE 5 h 12875 13056',
'STATE 4 p 12916 13057',
'STATE 4 g 12916 13058',
'STATE 5 h 12916 13059',
'STATE 3 d 12916 13060',
'STATE 3 s 12917 13061',
'STATE 5 a 12917 12863',
'STATE 3 h 12875 13062',
'STATE 5 s 12916 13063',
'STATE 3 n 13064 12863',
'STATE 2 c 12916 13065',
'STATE 3 b 12875 13066',
'STATE 2 o 12917 12863',
'STATE 3 r 12916 12875',
'STATE 4 d 12916 13067',
'STATE 5 i 12875 13068',
'STATE 3 l 12875 13069',
'STATE 3 w 12875 12916',
'INDEX 13070 z',
'STATE 3 t 13072 13071',
'STATE 4 z 13074 13073',
'STATE 4 # 13076 13075',
'STATE 3 c 13074 13077',
'PHONE epsilon',
'STATE 4 e 13079 13078',
'PHONE s',
'STATE 3 s 13074 13080',
'STATE 4 s 13074 13081',
'PHONE z',
'STATE 3 z 13083 13082',
'STATE 4 i 13079 13084',
'STATE 3 d 13086 13085',
'STATE 4 l 13079 13087',
'STATE 4 o 13079 13076',
'STATE 4 s 13074 13088',
'STATE 4 i 13074 13089',
'STATE 4 # 13079 13090',
'STATE 4 h 13092 13091',
'STATE 4 e 13079 13074',
'STATE 4 y 13079 13093',
'STATE 3 r 13079 13094',
'STATE 3 # 13095 13079',
'STATE 4 o 13079 13096',
'STATE 4 # 13079 13097',
'PHONE zh',
'STATE 4 e 13079 13098',
'STATE 3 l 13099 13079',
'PHONE t-s',
'STATE 4 b 13098 13079'
];

function _dict() { return {
'a':['ey1','dt'],
'aback':['ah b-ae1-k','rb'],
'abalone':['ae b-ah l-ow1 n-iy','nn'],
'abandon':['ah b-ae1-n d-ah-n','vb nn vbp'],
'abandoned':['ah b-ae1-n d-ah-n-d','vbn vbd jj'],
'abandoning':['ah b-ae1-n d-ah n-ih-ng','vbg'],
'abandonment':['ah b-ae1-n d-ah-n m-ah-n-t','nn'],
'abandons':['ah b-ae1-n d-ah-n-z','vbz'],
'abashed':['ah b-ae1-sh-t','jj'],
'abate':['ah b-ey1-t','vb'],
'abated':['ah b-ey1 t-ih-d','vbn vbd'],
'abatement':['ah b-ey1-t m-ah-n-t','nn'],
'abates':['ah b-ey1-t-s','vbz'],
'abating':['ah b-ey1 t-ih-ng','vbg'],
'abbey':['ae1 b-iy','nn'],
'abbot':['ae1 b-ah-t','nn'],
'abbreviated':['ah b-r-iy1 v-iy ey t-ah-d','jj'],
'abbreviation':['ah b-r-iy v-iy ey1 sh-ah-n','nn'],
'abdicate':['ae1-b d-ah k-ey-t','vbp vb'],
'abdomen':['ae-b d-ow1 m-ah-n','nn'],
'abdominal':['ae-b d-aa1 m-ah n-ah-l','jj'],
'abducted':['ae-b d-ah1-k t-ih-d','vbn nn'],
'abduction':['ae-b d-ah1-k sh-ah-n','nn'],
'abed':['ah b-eh1-d','rb'],
'aberrant':['ae b-eh1 r-ah-n-t','jj'],
'aberration':['ae b-er ey1 sh-ah-n','nn'],
'abetted':['ah b-eh1 t-ih-d','vbn vbd'],
'abetting':['ah b-eh1 t-ih-ng','vbg'],
'abeyance':['ah b-ey1 ah-n-s','nn'],
'abhor':['ae-b hh-ao1-r','vb'],
'abhorred':['ah-b hh-ao1-r-d','vbd'],
'abhorrent':['ae-b hh-ao1 r-ah-n-t','jj'],
'abide':['ah b-ay1-d','vb vbp'],
'abides':['ah b-ay1-d-z','vbz'],
'abiding':['ah b-ay1 d-ih-ng','jj'],
'ability':['ah b-ih1 l-ah t-iy','nn'],
'abject':['ae1-b jh-eh-k-t','jj'],
'ablaze':['ah b-l-ey1-z','jj rb'],
'able':['ey1 b-ah-l','jj'],
'abler':['ey1 b-ah-l er','jjr'],
'ably':['ey1 b-l-iy','rb'],
'abnormal':['ae-b n-ao1-r m-ah-l','jj'],
'abnormality':['ae-b n-ao-r m-ae1 l-ah t-iy','nn'],
'abnormally':['ae-b n-ao1-r m-ah l-iy','rb'],
'aboard':['ah b-ao1-r-d','in rb rp'],
'abode':['ah b-ow1-d','nn'],
'abolish':['ah b-aa1 l-ih-sh','vb'],
'abolished':['ah b-aa1 l-ih-sh-t','vbn vbd'],
'abolishing':['ah b-aa1 l-ih sh-ih-ng','vbg'],
'abolition':['ae b-ah l-ih1 sh-ah-n','nn'],
'abolitionist':['ae b-ah l-ih1 sh-ah n-ah-s-t','nn'],
'abominable':['ah b-aa1 m-ah n-ah b-ah-l','jj'],
'abomination':['ah b-aa m-ah n-ey1 sh-ah-n','nn'],
'aboriginal':['ae b-er ih1 jh-ah n-ah-l','jj'],
'aborigine':['ae b-er ih1 jh-ah n-iy','nn'],
'abort':['ah b-ao1-r-t','vb'],
'aborted':['ah b-ao1-r t-ih-d','jj vbd vbn'],
'aborting':['ah b-ao1-r t-ih-ng','vbg'],
'abortion':['ah b-ao1-r sh-ah-n','nn'],
'abortionist':['ah b-ao1-r sh-ah-n ih-s-t','nn'],
'abortive':['ah b-ao1-r t-ih-v','jj'],
'abound':['ah b-aw1-n-d','vbp vb'],
'abounded':['ah b-aw1-n d-ih-d','vbd'],
'abounding':['ah b-aw1-n d-ih-ng','vbg'],
'abounds':['ah b-aw1-n-d-z','vbz'],
'about':['ah b-aw1-t','in jj rb rp rbr'],
'above':['ah b-ah1-v','in jj rb'],
'aboveboard':['ah b-ah1-v b-ao-r-d','jj'],
'abrasive':['ah b-r-ey1 s-ih-v','jj nn'],
'abreast':['ah b-r-eh1-s-t','rb'],
'abridged':['ah b-r-ih1-jh-d','vbn'],
'abroad':['ah b-r-ao1-d','rb'],
'abrogate':['ae1 b-r-ah g-ey-t','vb'],
'abrogated':['ae1 b-r-ah g-ey t-ih-d','vbn'],
'abrupt':['ah b-r-ah1-p-t','jj'],
'abruptly':['ah b-r-ah1-p-t l-iy','rb'],
'abruptness':['ah b-r-ah1-p-t n-ah-s','nn'],
'abscess':['ae1-b s-eh-s','nn'],
'absence':['ae1-b s-ah-n-s','nn'],
'absent':['ae1-b s-ah-n-t','jj vb'],
'absentee':['ae-b s-ah-n t-iy1','jj nn'],
'absenteeism':['ae-b s-ah-n t-iy1 ih z-ah-m','nn'],
'absolute':['ae1-b s-ah l-uw-t','jj nn'],
'absolutely':['ae-b s-ah l-uw1-t l-iy','rb'],
'absoluteness':['ae1-b s-ah l-uw-t n-ah-s','nn'],
'absolution':['ae-b s-ah l-uw1 sh-ah-n','nn'],
'absolutism':['ae1-b s-ah l-uw t-ih z-ah-m','nn'],
'absolve':['ah-b z-aa1-l-v','vbp'],
'absolved':['ah-b z-aa1-l-v-d','vbd'],
'absolving':['ah-b z-aa1-l v-ih-ng','vbg'],
'absorb':['ah-b z-ao1-r-b','vb vbp'],
'absorbed':['ah-b z-ao1-r-b-d','vbn jj vbd'],
'absorbency':['ah-b z-ao1-r b-ah-n s-iy','nn'],
'absorbent':['ah-b z-ao1-r b-ah-n-t','jj'],
'absorber':['ah-b z-ao1-r b-er','nn'],
'absorbing':['ah-b z-ao1-r b-ih-ng','vbg jj'],
'absorbs':['ah-b z-ao1-r-b-z','vbz'],
'absorption':['ah-b z-ao1-r-p sh-ah-n','nn'],
'abstain':['ah-b s-t-ey1-n','vb vbp'],
'abstained':['ah-b s-t-ey1-n-d','vbd'],
'abstaining':['ah-b s-t-ey1 n-ih-ng','vbg'],
'abstention':['ah-b s-t-eh1-n ch-ah-n','nn'],
'abstinence':['ae1-b s-t-ah n-ah-n-s','nn'],
'abstract':['ae-b s-t-r-ae1-k-t','jj nn vbp'],
'abstracted':['ae1-b s-t-r-ae-k t-ih-d','jj vbd'],
'abstraction':['ae-b s-t-r-ae1-k sh-ah-n','nn'],
'abstruse':['ah-b s-t-r-uw1-s','jj'],
'absurd':['ah-b s-er1-d','jj'],
'absurdist':['ah-b s-er1 d-ih-s-t','jj nn'],
'absurdity':['ah-b s-er1 d-ah t-iy','nn'],
'absurdly':['ah-b s-er1-d l-iy','rb'],
'abundance':['ah b-ah1-n d-ah-n-s','nn'],
'abundant':['ah b-ah1-n d-ah-n-t','jj'],
'abundantly':['ah b-ah1-n d-ah-n-t l-iy','rb'],
'abuse':['ah b-y-uw1-s','nn vb vbp'],
'abused':['ah b-y-uw1-z-d','vbn jj vbd'],
'abuser':['ah b-y-uw1 z-er','nn'],
'abuses':['ah b-y-uw1 s-ih-z','nns'],
'abusing':['ah b-y-uw1 z-ih-ng','vbg'],
'abusive':['ah b-y-uw1 s-ih-v','jj'],
'abuzz':['ah b-ah1-z','jj'],
'abysmal':['ah b-ih1-z m-ah-l','jj'],
'abyss':['ah b-ih1-s','nn'],
'acacia':['ah k-ey1 sh-ah','nn'],
'academe':['ae1 k-ah d-iy-m','nn'],
'academia':['ae k-ah d-iy1 m-iy ah','nn'],
'academic':['ae k-ah d-eh1 m-ih-k','jj nn'],
'academically':['ae k-ah d-eh1 m-ih-k l-iy','rb'],
'academician':['ae k-ah d-ah m-ih1 sh-ah-n','nn'],
'academy':['ah k-ae1 d-ah m-iy','nn'],
'accede':['ae-k s-iy1-d','vb'],
'acceded':['ae-k s-iy1 d-ih-d','vbd vbn'],
'accelerate':['ae-k s-eh1 l-er ey-t','vb vbp'],
'accelerated':['ae-k s-eh1 l-er ey t-ih-d','vbn jj vbd'],
'accelerates':['ae-k s-eh1 l-er ey-t-s','vbz'],
'accelerating':['ae-k s-eh1 l-er ey t-ih-ng','vbg nn jj'],
'acceleration':['ae-k s-eh l-er ey1 sh-ah-n','nn'],
'accelerator':['ae-k s-eh1 l-er ey t-er','nn'],
'accelerometer':['ae-k s-eh l-er aa1 m-ah t-er','nn'],
'accent':['ah-k s-eh1-n-t','nn vb'],
'accented':['ae1-k s-eh-n t-ih-d','vbn jj'],
'accents':['ae1-k s-eh-n-t-s','nns vbz'],
'accentuate':['ae-k s-eh1-n ch-uw ey-t','vb'],
'accentuated':['ae-k s-eh1-n ch-ah-w ey t-ih-d','vbn vbd'],
'accept':['ae-k s-eh1-p-t','vb vbp'],
'acceptability':['ah-k s-eh-p t-ah b-ih1 l-ah t-iy','nn'],
'acceptable':['ae-k s-eh1-p t-ah b-ah-l','jj'],
'acceptance':['ae-k s-eh1-p t-ah-n-s','nn'],
'accepted':['ae-k s-eh1-p t-ih-d','vbn jj vbd'],
'accepting':['ae-k s-eh1-p t-ih-ng','vbg jj'],
'accepts':['ae-k s-eh1-p-t-s','vbz'],
'access':['ae1-k s-eh-s','nn vb'],
'accessed':['ae1-k s-eh-s-t','vbn'],
'accessibility':['ae-k s-eh s-ah b-ih1 l-ih t-iy','nn'],
'accessible':['ae-k s-eh1 s-ah b-ah-l','jj'],
'accessory':['ae-k s-eh1 s-er iy','nn jj'],
'accident':['ae1-k s-ah d-ah-n-t','nn'],
'accidental':['ae-k s-ah d-eh1-n t-ah-l','jj'],
'accidentally':['ae-k s-ah d-eh1-n t-ah l-iy','rb'],
'accidently':['ae1-k s-ah d-ah-n-t l-iy','rb'],
'acclaim':['ah k-l-ey1-m','nn vb'],
'acclaimed':['ah k-l-ey1-m-d','vbn vbd jj'],
'accolade':['ae1 k-ah l-ey-d','nn'],
'accommodate':['ah k-aa1 m-ah d-ey-t','vb vbp'],
'accommodated':['ah k-aa1 m-ah d-ey t-ah-d','vbn'],
'accommodates':['ah k-aa1 m-ah d-ey-t-s','vbz'],
'accommodating':['ah k-aa1 m-ah d-ey t-ih-ng','vbg'],
'accommodation':['ah k-aa m-ah d-ey1 sh-ah-n','nn'],
'accommodative':['ah k-aa1 m-ah d-ey t-ih-v','jj'],
'accompanied':['ah k-ah1-m p-ah n-iy-d','vbn vbd'],
'accompanies':['ah k-ah1-m p-ah n-iy-z','vbz'],
'accompaniment':['ah k-ah1-m-p n-ih m-ah-n-t','nn'],
'accompanist':['ah k-ah1-m p-ah n-ah-s-t','nn'],
'accompany':['ah k-ah1-m p-ah n-iy','vb vbp'],
'accompanying':['ah k-ah1-m p-ah n-iy ih-ng','vbg jj'],
'accompli':['aa k-aa1-m p-l-iy','nn'],
'accomplice':['ah k-aa1-m p-l-ah-s','nn'],
'accomplish':['ah k-aa1-m p-l-ih-sh','vb vbp'],
'accomplished':['ah k-aa1-m p-l-ih-sh-t','vbn jj vbd'],
'accomplishes':['ah k-aa1-m p-l-ih sh-ih-z','vbz'],
'accomplishing':['ah k-aa1-m p-l-ih sh-ih-ng','vbg'],
'accomplishment':['ah k-aa1-m p-l-ih-sh m-ah-n-t','nn'],
'accord':['ah k-ao1-r-d','nn vbp rb vb'],
'accordance':['ah k-ao1-r d-ah-n-s','nn'],
'accorded':['ah k-ao1-r d-ih-d','vbn vbd'],
'according':['ah k-ao1-r d-ih-ng','vbg'],
'accordingly':['ah k-ao1-r d-ih-ng l-iy','rb'],
'accordion':['ah k-ao1-r d-iy ah-n','nn'],
'accords':['ah k-ao1-r-d-z','nns vbz'],
'accosted':['ah k-aa1 s-t-ah-d','vbn vbd'],
'account':['ah k-aw1-n-t','nn vbp vb'],
'accountability':['ah k-aw1-n t-ah b-ih l-ih t-iy','nn'],
'accountable':['ah k-aw1-n t-ah b-ah-l','jj'],
'accountant':['ah k-aw1-n t-ah-n-t','nn'],
'accounted':['ah k-aw1-n t-ah-d','vbd vbn'],
'accounting':['ah k-aw1-n t-ih-ng','nn vbg jj'],
'accounts':['ah k-aw1-n-t-s','nns vbz'],
'accouterments':['ah k-uw1 t-er m-ah-n-t-s','nns'],
'accreditation':['ah k-r-eh d-ah t-ey1 sh-ah-n','nn'],
'accredited':['ah k-r-eh1 d-ih t-ih-d','vbd vbn'],
'accrediting':['ah k-r-eh1 d-ah t-ih-ng','nn'],
'accretion':['ah k-r-iy1 sh-ah-n','nn'],
'accrual':['ah k-r-uw1 ah-l','nn'],
'accrue':['ah k-r-uw1','vb'],
'accrued':['ah k-r-uw1-d','vbn jj'],
'accrues':['ah k-r-uw1-z','vbz'],
'accruing':['ah k-r-uw1 ih-ng','vbg'],
'accumulate':['ah k-y-uw1 m-y-ah l-ey-t','vb vbp'],
'accumulated':['ah k-y-uw1 m-y-ah l-ey t-ih-d','vbn jj vbd'],
'accumulates':['ah k-y-uw1 m-y-ah l-ey-t-s','vbz'],
'accumulating':['ah k-y-uw1 m-y-ah l-ey t-ih-ng','vbg jj'],
'accumulation':['ah k-y-uw m-y-ah l-ey1 sh-ah-n','nn'],
'accuracy':['ae1 k-y-er ah s-iy','nn'],
'accurate':['ae1 k-y-er ah-t','jj'],
'accurately':['ae1 k-y-er ah-t l-iy','rb'],
'accusation':['ae k-y-ah z-ey1 sh-ah-n','nn'],
'accusatory':['ah k-y-uw1 z-ah t-ao r-iy','jj'],
'accuse':['ah k-y-uw1-z','vb vbp'],
'accused':['ah k-y-uw1-z-d','vbn vbd jj nn'],
'accuser':['ah k-y-uw1 z-er','nn'],
'accuses':['ah k-y-uw1 z-ih-z','vbz'],
'accusing':['ah k-y-uw1 z-ih-ng','vbg jj'],
'accusingly':['ah k-y-uw1 z-ih-ng l-iy','rb'],
'accustomed':['ah k-ah1 s-t-ah-m-d','vbn jj'],
'ace':['ey1-s','nn'],
'acerbic':['ah s-eh1-r b-ih-k','jj'],
'aces':['ey1 s-ih-z','nns vbz'],
'acetate':['ae1 s-ah t-ey-t','nn'],
'acetone':['ae1 s-ah t-ow-n','nn'],
'acetylene':['ah s-eh1 t-ah l-iy-n','nn'],
'ache':['ey1-k','nn vb vbp'],
'aches':['ey1-k-s','nns vbz'],
'achievable':['ah ch-iy1 v-ah b-ah-l','jj'],
'achieve':['ah ch-iy1-v','vb vbp'],
'achieved':['ah ch-iy1-v-d','vbn vbd jj'],
'achievement':['ah ch-iy1-v m-ah-n-t','nn'],
'achieves':['ah ch-iy1-v-z','vbz'],
'achieving':['ah ch-iy1 v-ih-ng','vbg'],
'aching':['ey1 k-ih-ng','vbg jj'],
'acid':['ae1 s-ah-d','nn jj'],
'acidified':['ah s-ih1 d-ah f-ay-d','vbn'],
'acidity':['ah s-ih1 d-ah t-iy','nn'],
'acidly':['ae1 s-ah-d l-iy','rb'],
'aciduria':['ae s-ah d-uh1 r-iy ah','nn'],
'acknowledge':['ae-k n-aa1 l-ih-jh','vbp vb'],
'acknowledged':['ae-k n-aa1 l-ih-jh-d','vbd vbn jj'],
'acknowledgement':['ae-k n-aa1 l-ih-jh m-ah-n-t','nn'],
'acknowledges':['ae-k n-aa1 l-ih jh-ih-z','vbz'],
'acknowledging':['ae-k n-aa1 l-ih jh-ih-ng','vbg'],
'acknowledgment':['ae-k n-aa1 l-ih-jh m-ah-n-t','nn'],
'acne':['ae1-k n-iy','nn'],
'acolyte':['ae1 k-ah l-ay-t','nn'],
'acorns':['ey1 k-ao-r-n-z','nns'],
'acoustic':['ah k-uw1 s-t-ih-k','jj'],
'acoustical':['ah k-uw1 s-t-ih k-ah-l','jj'],
'acoustically':['ah k-uw1 s-t-ih-k l-iy','rb'],
'acquaint':['ah k-w-ey1-n-t','vb'],
'acquaintance':['ah k-w-ey1-n t-ah-n-s','nn'],
'acquainted':['ah k-w-ey1-n t-ih-d','vbn'],
'acquiesce':['ae k-w-iy eh1-s','vb'],
'acquiesced':['ae k-w-iy eh1-s-t','vbd vbn'],
'acquiescence':['ae k-w-iy eh1 s-ah-n-s','nn'],
'acquire':['ah k-w-ay1 er','vb nn vbp'],
'acquired':['ah k-w-ay1 er-d','vbn jj vbd'],
'acquirer':['ah k-w-ay1 er er','nn'],
'acquires':['ah k-w-ay1 er-z','vbz'],
'acquiring':['ah k-w-ay1 r-ih-ng','vbg'],
'acquisition':['ae k-w-ah z-ih1 sh-ah-n','nn'],
'acquisitive':['ah k-w-ih1 z-ah t-ih-v','jj'],
'acquit':['ah k-w-ih1-t','vb'],
'acquittal':['ah k-w-ih1 t-ah-l','nn'],
'acquitted':['ah k-w-ih1 t-ah-d','vbn vbd'],
'acre':['ey1 k-er','nn'],
'acreage':['ey1 k-er ih-jh','nn'],
'acrimonious':['ae k-r-ah m-ow1 n-iy ah-s','jj'],
'acrimony':['ae1 k-r-ih m-ow n-iy','nn'],
'acrobat':['ae1 k-r-ah b-ae-t','nn'],
'acrobatic':['ae k-r-ah b-ae1 t-ih-k','jj'],
'acronym':['ae1 k-r-ah n-ih-m','nn'],
'across':['ah k-r-ao1-s','in rb rp'],
'acrylic':['ah k-r-ih1 l-ih-k','nn'],
'act':['ae1-k-t','nn vbp vb'],
'acted':['ae1-k t-ah-d','vbd vbn'],
'acting':['ae1-k t-ih-ng','vbg jj nn'],
'action':['ae1-k sh-ah-n','nn'],
'actionable':['ae1-k sh-ah-n ah b-ah-l','jj'],
'activate':['ae1-k t-ah v-ey-t','vbp vb'],
'activated':['ae1-k t-ah v-ey t-ah-d','vbn jj'],
'activating':['ae1-k t-ah v-ey t-ih-ng','vbg'],
'activation':['ae-k t-ah v-ey1 sh-ah-n','nn'],
'active':['ae1-k t-ih-v','jj nn'],
'actively':['ae1-k t-ih-v l-iy','rb'],
'activism':['ae1-k t-ih v-ih z-ah-m','nn'],
'activist':['ae1-k t-ah v-ah-s-t','nn jj'],
'activists':['ae1-k t-ah v-ah-s-t-s','nns vbz'],
'activity':['ae-k t-ih1 v-ah t-iy','nn'],
'actor':['ae1-k t-er','nn'],
'actress':['ae1-k t-r-ah-s','nn'],
'acts':['ae1-k-t-s','nns vbz'],
'actual':['ae1-k ch-ah w-ah-l','jj'],
'actuality':['ae-k ch-ah-w ae1 l-ah t-iy','nn'],
'actually':['ae1-k ch-ah-w ah l-iy','rb'],
'actuarial':['ae-k ch-ah-w eh1 r-iy ah-l','jj'],
'actuary':['ae1-k ch-ah-w eh r-iy','nn'],
'acumen':['ah k-y-uw1 m-ah-n','nn'],
'acupuncture':['ae1 k-y-uw p-ah-ng-k ch-er','nn'],
'acute':['ah k-y-uw1-t','jj nn'],
'acutely':['ah k-y-uw1-t l-iy','rb'],
'ad':['ae1-d','nn'],
'adage':['ae1 d-ah-jh','nn'],
'adagio':['ah d-aa1 zh-iy ow','nn rb'],
'adamant':['ae1 d-ah m-ah-n-t','jj'],
'adamantly':['ae1 d-ah m-ah-n-t l-iy','rb'],
'adapt':['ah d-ae1-p-t','vb vbp'],
'adaptability':['ah d-ae-p t-ah b-ih1 l-ah t-iy','nn'],
'adaptable':['ah d-ae1-p t-ah b-ah-l','jj'],
'adaptation':['ae d-ah-p t-ey1 sh-ah-n','nn'],
'adapted':['ah d-ae1-p t-ah-d','vbn vbd'],
'adapter':['ah d-ae1-p t-er','nn'],
'adapting':['ah d-ae1-p t-ih-ng','vbg nn'],
'adaptor':['ah d-ae1-p t-er','nn'],
'adapts':['ah d-ae1-p-t-s','vbz'],
'add':['ae1-d','vb vbp'],
'added':['ae1 d-ah-d','vbd jj vbn'],
'addict':['ah d-ih1-k-t','nn'],
'addicted':['ah d-ih1-k t-ah-d','vbn jj'],
'addiction':['ah d-ih1-k sh-ah-n','nn'],
'addictive':['ah d-ih1-k t-ih-v','jj'],
'adding':['ae1 d-ih-ng','vbg nn'],
'addition':['ah d-ih1 sh-ah-n','nn'],
'additional':['ah d-ih1 sh-ah n-ah-l','jj'],
'additionally':['ah d-ih1 sh-ah-n ah l-iy','rb'],
'additive':['ae1 d-ah t-ih-v','nn'],
'addled':['ae1 d-ah-l-d','jj vbn'],
'address':['ae1 d-r-eh-s','nn vbp vb'],
'addressed':['ah d-r-eh1-s-t','vbn jj vbd'],
'addressee':['ae d-r-eh s-iy1','nn'],
'addresses':['ae1 d-r-eh1 s-ih-z','nns vbz'],
'addressing':['ah d-r-eh1 s-ih-ng','vbg'],
'adds':['ae1-d-z','vbz'],
'adept':['ah d-eh1-p-t','jj'],
'adequacy':['ae1 d-ah k-w-ah s-iy','nn'],
'adequate':['ae1 d-ah k-w-ah-t','jj'],
'adequately':['ae1 d-ah k-w-ah-t l-iy','rb'],
'adhere':['ah-d hh-ih1-r','vb vbp'],
'adhered':['ae-d hh-ih1-r-d','vbn vbd'],
'adherence':['ah-d hh-ih1 r-ah-n-s','nn'],
'adherent':['ah-d hh-ih1 r-ah-n-t','jj nn'],
'adheres':['ah-d hh-ih1-r-z','vbz'],
'adhering':['ah-d hh-ih1 r-ih-ng','vbg'],
'adhesive':['ae-d hh-iy1 s-ih-v','jj nn'],
'adjacent':['ah jh-ey1 s-ah-n-t','jj'],
'adjective':['ae1 jh-ih-k t-ih-v','nn'],
'adjoining':['ah jh-oy1 n-ih-ng','vbg jj'],
'adjoins':['ah jh-oy1-n-z','vbz'],
'adjourn':['ah jh-er1-n','vb'],
'adjourned':['ah jh-er1-n-d','vbd vbn'],
'adjourning':['ah jh-er1 n-ih-ng','nn vbg'],
'adjournment':['ah jh-er1-n m-ah-n-t','nn'],
'adjourns':['ah jh-er1-n-z','vbz'],
'adjudged':['ah jh-ah1-jh-d','vbn'],
'adjudicate':['ah jh-uw1 d-ih k-ey-t','vb'],
'adjudication':['ah jh-uw d-ah k-ey1 sh-ah-n','nn'],
'adjunct':['ae1 jh-ah-ng-k-t','nn jj'],
'adjust':['ah jh-ah1-s-t','vb vbp'],
'adjustable':['ah jh-ah1 s-t-ah b-ah-l','jj'],
'adjusted':['ah jh-ah1 s-t-ah-d','vbn jj vbd'],
'adjuster':['ah jh-ah1 s-t-er','nn'],
'adjusting':['ah jh-ah1 s-t-ih-ng','vbg nn'],
'adjustment':['ah jh-ah1-s-t m-ah-n-t','nn'],
'adjusts':['ah jh-ah1-s-t-s','vbz'],
'adman':['ae1-d m-ah-n','nn'],
'administer':['ah-d m-ih1 n-ah s-t-er','vb vbp'],
'administered':['ah-d m-ih1 n-ah s-t-er-d','vbn vbd'],
'administering':['ae-d m-ih1 n-ih s-t-er ih-ng','vbg'],
'administers':['ae-d m-ih1 n-ih s-t-er-z','vbz'],
'administrate':['ae-d m-ih1 n-ih s-t-r-ey-t','vb'],
'administration':['ae-d m-ih n-ih s-t-r-ey1 sh-ah-n','nn'],
'administrative':['ah-d m-ih1 n-ah s-t-r-ey t-ih-v','jj'],
'administratively':['ae-d m-ih n-ah s-t-r-ey1 t-ih-v l-iy','rb'],
'administrator':['ah-d m-ih1 n-ah s-t-r-ey t-er','nn'],
'admirable':['ae1-d m-er ah b-ah-l','jj'],
'admirably':['ae1-d m-er ah b-l-iy','rb'],
'admiral':['ae1-d m-er ah-l','nn'],
'admiralty':['ae1-d m-er ah-l t-iy','nn'],
'admiration':['ae-d m-er ey1 sh-ah-n','nn'],
'admire':['ae-d m-ay1-r','vb vbp'],
'admired':['ah-d m-ay1 er-d','vbd vbn jj'],
'admirer':['ae-d m-ay1 r-er','nn'],
'admires':['ae-d m-ay1-r-z','vbz'],
'admiring':['ae-d m-ay1 r-ih-ng','vbg'],
'admiringly':['ae-d m-ay1 r-ih-ng l-iy','rb'],
'admissible':['ah-d m-ih1 s-ah b-ah-l','jj'],
'admission':['ae-d m-ih1 sh-ah-n','nn'],
'admit':['ah-d m-ih1-t','vb vbp'],
'admits':['ah-d m-ih1-t-s','vbz nns'],
'admittance':['ah-d m-ih1 t-ah-n-s','nn'],
'admitted':['ah-d m-ih1 t-ah-d','vbd vbn jj'],
'admittedly':['ae-d m-ih1 t-ih-d l-iy','rb'],
'admitting':['ae-d m-ih1 t-ih-ng','vbg'],
'admonished':['ah-d m-aa1 n-ih-sh-t','vbd'],
'admonishing':['ae-d m-aa1 n-ih sh-ih-ng','vbg'],
'admonition':['ae-d m-ah n-ih1 sh-ah-n','nn'],
'adobe':['ah d-ow1 b-iy','nn'],
'adolescence':['ae d-ah l-eh1 s-ah-n-s','nn'],
'adolescent':['ae d-ah l-eh1 s-ah-n-t','nn jj'],
'adopt':['ah d-aa1-p-t','vb vbp'],
'adoptable':['ah d-aa1-p t-ah b-ah-l','jj'],
'adopted':['ah d-aa1-p t-ah-d','vbn vbd jj'],
'adoptees':['ah d-aa1-p t-iy1-z','nns'],
'adopting':['ah d-aa1-p t-ih-ng','vbg'],
'adoption':['ah d-aa1-p sh-ah-n','nn'],
'adoptive':['ah d-aa1-p t-ih-v','jj'],
'adopts':['ah d-aa1-p-t-s','vbz'],
'adorable':['ah d-ao1 r-ah b-ah-l','jj'],
'adore':['ah d-ao1-r','vbp vb'],
'adored':['ah d-ao1-r-d','vbd vbn'],
'adores':['ah d-ao1-r-z','vbz'],
'adoring':['ah d-ao1 r-ih-ng','vbg'],
'adorn':['ah d-ao1-r-n','vb vbp'],
'adorned':['ah d-ao1-r-n-d','vbn vbd'],
'adorns':['ah d-ao1-r-n-z','vbz'],
'adrenal':['ah d-r-iy1 n-ah-l','jj nn'],
'adrenaline':['ah d-r-eh1 n-ah l-ah-n','nn'],
'adrift':['ah d-r-ih1-f-t','rb'],
'adroit':['ah d-r-oy1-t','jj'],
'adroitly':['ah d-r-oy1-t l-iy','rb'],
'adulation':['ae jh-ah l-ey1 sh-ah-n','nn'],
'adult':['ah d-ah1-l-t','nn jj'],
'adulterated':['ah d-ah1-l t-er ey t-ih-d','vbn jj'],
'adulterers':['ah d-ah1-l t-er er-z','nns'],
'adulterous':['ah d-ah1-l t-er ah-s','jj'],
'adultery':['ah d-ah1-l t-er iy','nn'],
'adulthood':['ah d-ah1-l-t hh-uh-d','nn'],
'advance':['ah-d v-ae1-n-s','nn jj vb vbp'],
'advanced':['ah-d v-ae1-n-s-t','vbd vbn jj'],
'advancement':['ah-d v-ae1-n-s m-ah-n-t','nn'],
'advancers':['ah-d v-ae1-n s-er-z','nns'],
'advances':['ah-d v-ae1-n s-ah-z','nns vbz'],
'advancing':['ah-d v-ae1-n s-ih-ng','vbg jj'],
'advantage':['ae-d v-ae1-n t-ih-jh','nn vb'],
'advantageous':['ae-d v-ah-n t-ey1 jh-ah-s','jj'],
'advantages':['ae-d v-ae1-n t-ih jh-ih-z','nns'],
'advent':['ae1-d v-eh-n-t','nn'],
'adventure':['ae-d v-eh1-n ch-er','nn vb'],
'adventurer':['ae-d v-eh1-n ch-er er','nn'],
'adventures':['ae-d v-eh1-n ch-er-z','nns'],
'adventuresome':['ae-d v-eh1-n ch-er s-ah-m','jj'],
'adventurism':['ae-d v-eh1-n ch-er ih z-ah-m','nn'],
'adventurous':['ae-d v-eh1-n ch-er ah-s','jj'],
'adverbial':['ae-d v-er1 b-iy ah-l','jj'],
'adverbs':['ae1-d v-er-b-z','nns'],
'adversarial':['ae-d v-er s-eh1 r-iy ah-l','jj'],
'adversary':['ae1-d v-er s-eh r-iy','nn jj'],
'adverse':['ae-d v-er1-s','jj'],
'adversely':['ae-d v-er1-s l-iy','rb'],
'adversity':['ae-d v-er1 s-ih t-iy','nn'],
'advertise':['ae1-d v-er t-ay-z','vb vbp'],
'advertised':['ae1-d v-er t-ay-z-d','vbn jj vbd'],
'advertisement':['ah-d v-er1 t-ah-z m-ah-n-t','nn'],
'advertiser':['ae1-d v-er t-ay z-er','nn'],
'advertises':['ae1-d v-er t-ay z-ih-z','vbz'],
'advertising':['ae1-d v-er t-ay z-ih-ng','nn vbg'],
'advertorial':['ae-d v-er t-ao1 r-iy ah-l','jj'],
'advice':['ae-d v-ay1-s','nn'],
'advisability':['ae-d v-ay z-ah b-ih1 l-ih t-iy','nn'],
'advisable':['ah-d v-ay1 z-ah b-ah-l','jj'],
'advise':['ae-d v-ay1-z','vb vbp nn'],
'advised':['ae-d v-ay1-z-d','vbn vbd jj'],
'advisedly':['ae-d v-ay1 z-ah-d l-iy','rb'],
'advisement':['ah-d v-ay1-z m-ah-n-t','nn'],
'adviser':['ae-d v-ay1 z-er','nn'],
'advises':['ae-d v-ay1 z-ih-z','vbz'],
'advising':['ae-d v-ay1 z-ih-ng','vbg nn'],
'advisor':['ae-d v-ay1 z-er','nn'],
'advisory':['ae-d v-ay1 z-er iy','jj nn'],
'advocacy':['ae1-d v-ah k-ah s-iy','nn'],
'advocate':['ae1-d v-ah k-ah-t','nn vb vbp'],
'advocated':['ae1-d v-ah k-ey t-ah-d','vbn vbd'],
'advocates':['ae1-d v-ah k-ah-t-s','nns vbz'],
'advocating':['ae1-d v-ah k-ey t-ih-ng','vbg'],
'aegis':['iy1 jh-ah-s','nn'],
'aerial':['eh1 r-iy ah-l','jj'],
'aerobic':['eh r-ow1 b-ih-k','jj'],
'aerobics':['er ow1 b-ih-k-s','nn nns'],
'aerodynamic':['eh r-ow d-ay n-ae1 m-ih-k','jj'],
'aeronautical':['eh r-ow n-aa1 t-ah k-ah-l','jj'],
'aerosol':['eh1 r-ah s-aa-l','nn'],
'aerospace':['eh1 r-ow s-p-ey-s','nn jj'],
'aesthetic':['eh-s th-eh1 t-ih-k','jj nn'],
'aesthetically':['eh-s th-eh1 t-ih-k l-iy','rb'],
'afar':['ah f-aa1-r','rb'],
'affable':['ae1 f-ah b-ah-l','jj'],
'affair':['ah f-eh1-r','nn'],
'affect':['ah f-eh1-k-t','vb vbp nn'],
'affectation':['ae f-eh-k t-ey1 sh-ah-n','nn'],
'affected':['ah f-eh1-k t-ah-d','vbn jj vbd'],
'affecting':['ah f-eh1-k t-ih-ng','vbg'],
'affection':['ah f-eh1-k sh-ah-n','nn'],
'affectionate':['ah f-eh1-k sh-ah-n ah-t','jj'],
'affectionately':['ah f-eh1-k sh-ah-n ah-t l-iy','rb'],
'affects':['ah f-eh1-k-t-s','vbz nns'],
'afferent':['ae1 f-er ah-n-t','jj'],
'affidavit':['ae f-ah d-ey1 v-ah-t','nn'],
'affiliate':['ah f-ih1 l-iy ey-t','nn vb'],
'affiliated':['ah f-ih1 l-iy ey t-ah-d','vbn jj'],
'affiliates':['ah f-ih1 l-iy ah-t-s','nns vbz'],
'affiliating':['ah f-ih1 l-iy ey t-ih-ng','vbg'],
'affiliation':['ah f-ih l-iy ey1 sh-ah-n','nn'],
'affinity':['ah f-ih1 n-ah t-iy','nn'],
'affirm':['ah f-er1-m','vb vbp'],
'affirmation':['ae f-er m-ey1 sh-ah-n','nn'],
'affirmative':['ah f-er1 m-ah t-ih-v','jj nn'],
'affirmed':['ah f-er1-m-d','vbd vbn'],
'affirming':['ah f-er1 m-ih-ng','vbg'],
'affirms':['ah f-er1-m-z','vbz'],
'affix':['ae1 f-ih-k-s','vb'],
'affixed':['ah f-ih1-k-s-t','vbn jj'],
'afflict':['ah f-l-ih1-k-t','vb'],
'afflicted':['ah f-l-ih1-k t-ah-d','vbn jj'],
'afflicting':['ah f-l-ih1-k t-ih-ng','vbg'],
'affliction':['ah f-l-ih1-k sh-ah-n','nn'],
'afflicts':['ah f-l-ih1-k-t-s','vbz'],
'affluence':['ae1 f-l-uw ah-n-s','nn'],
'affluent':['ae1 f-l-uw ah-n-t','jj nn'],
'afford':['ah f-ao1-r-d','vb vbp'],
'affordability':['ah f-ao-r d-ah b-ih1 l-ah t-iy','nn'],
'affordable':['ah f-ao1-r d-ah b-ah-l','jj'],
'afforded':['ah f-ao1-r d-ah-d','vbn vbd'],
'affording':['ah f-ao1-r d-ih-ng','vbg'],
'affords':['ah f-ao1-r-d-z','vbz'],
'affront':['ah f-r-ah1-n-t','nn'],
'affronted':['ah f-r-ah1-n t-ih-d','vbn'],
'afghan':['ae1-f g-ae-n','nn'],
'aficionado':['ah f-iy sh-y-ah n-aa1 d-ow','nn'],
'afield':['ah f-iy1-l-d','rb'],
'afire':['ah f-ay1-r','rb jj'],
'aflame':['ah f-l-ey1-m','jj'],
'aflatoxin':['ae f-l-ah t-aa1-k s-ih-n','nn'],
'afloat':['ah f-l-ow1-t','rb'],
'afoot':['ah f-uh1-t','rb jj'],
'aforementioned':['ah f-ao1-r m-eh-n sh-ah-n-d','jj vbn'],
'aforesaid':['ah f-ao1-r s-eh-d','jj rb'],
'afoul':['ah f-aw1-l','rb'],
'afraid':['ah f-r-ey1-d','jj'],
'afresh':['ah f-r-eh1-sh','rb'],
'aft':['ae1-f-t','jj nn rb'],
'after':['ae1-f t-er','in rb rp'],
'afterburners':['ae1-f t-er b-er n-er-z','nns'],
'aftereffects':['ae1-f t-er ah f-eh-k-t-s','nns'],
'afterglow':['ae1-f t-er g-l-ow','nn'],
'aftermarket':['ae1-f t-er m-aa-r k-ih-t','jj nn'],
'aftermath':['ae1-f t-er m-ae-th','nn'],
'afternoon':['ae-f t-er n-uw1-n','nn'],
'afternoons':['ae-f t-er n-uw1-n-z','nns rb'],
'aftershock':['ae1-f t-er sh-aa-k','nn'],
'aftertax':['ae1-f t-er t-ae-k-s','jj nn'],
'afterthought':['ae1-f t-er th-aa-t','nn'],
'afterward':['ae1-f t-er w-er-d','rb'],
'afterwards':['ae1-f t-er w-er-d-z','rb'],
'again':['ah g-eh1-n','rb'],
'against':['ah g-eh1-n-s-t','in'],
'agates':['ae1 g-ah-t-s','nns'],
'age':['ey1-jh','nn vbp'],
'aged':['ey1-jh-d','vbn jj nns vbd'],
'ageless':['ey1-jh l-ah-s','jj'],
'agency':['ey1 jh-ah-n s-iy','nn'],
'agenda':['ah jh-eh1-n d-ah','nn nns'],
'agent':['ey1 jh-ah-n-t','nn'],
'ages':['ey1 jh-ah-z','nns vbz'],
'agglomeration':['ah g-l-aa m-er ey1 sh-ah-n','nn'],
'aggrandizing':['ae1 g-r-ah-n d-ay z-ih-ng','vbg'],
'aggravate':['ae1 g-r-ah v-ey-t','vbp vb'],
'aggravated':['ae1 g-r-ah v-ey t-ah-d','vbn vbd'],
'aggravates':['ae1 g-r-ah v-ey-t-s','vbz'],
'aggravating':['ae1 g-r-ah v-ey t-ih-ng','vbg jj'],
'aggregate':['ae1 g-r-ah g-ah-t','jj nn'],
'aggression':['ah g-r-eh1 sh-ah-n','nn'],
'aggressive':['ah g-r-eh1 s-ih-v','jj'],
'aggressively':['ah g-r-eh1 s-ih-v l-iy','rb'],
'aggressiveness':['ah g-r-eh1 s-ih-v n-ah-s','nn'],
'aggressor':['ah g-r-eh1 s-er','nn'],
'aggrieved':['ah g-r-iy1-v-d','vbn jj'],
'aghast':['ah g-ae1-s-t','jj'],
'agile':['ae1 jh-ah-l','jj'],
'agility':['ah jh-ih1 l-ah t-iy','nn'],
'aging':['ey1 jh-ih-ng','vbg jj nn'],
'agitate':['ae1 jh-ah t-ey-t','vbp'],
'agitated':['ae1 jh-ah t-ey t-ah-d','vbd vbn'],
'agitating':['ae1 jh-ah t-ey t-ih-ng','vbg'],
'agitation':['ae jh-ah t-ey1 sh-ah-n','nn'],
'agitator':['ae1 jh-ah t-ey t-er','nn'],
'agleam':['ah-g l-iy1-m','jj'],
'ago':['ah g-ow1','rb in'],
'agonize':['ae1 g-ah n-ay-z','vb vbp'],
'agonized':['ae1 g-ah n-ay-z-d','vbd jj'],
'agonizes':['ae1 g-ah n-ay z-ih-z','vbz'],
'agonizing':['ae1 g-ah n-ay z-ih-ng','jj'],
'agony':['ae1 g-ah n-iy','nn'],
'agrarian':['ah g-r-eh1 r-iy ah-n','jj nn'],
'agree':['ah g-r-iy1','vb vbp'],
'agreeable':['ah g-r-iy1 ah b-ah-l','jj'],
'agreed':['ah g-r-iy1-d','vbd vbn jj'],
'agreeing':['ah g-r-iy1 ih-ng','vbg'],
'agreement':['ah g-r-iy1 m-ah-n-t','nn'],
'agrees':['ah g-r-iy1-z','vbz'],
'agricultural':['ae g-r-ah k-ah1-l ch-er ah-l','jj'],
'agriculturally':['ae g-r-ih k-ah1-l ch-er ah l-iy','rb'],
'agriculture':['ae1 g-r-ih k-ah-l ch-er','nn'],
'agrochemical':['ae g-r-ow k-eh1 m-ah k-ah-l','nn'],
'agronomist':['ah g-r-aa1 n-ah m-ih-s-t','nn'],
'aground':['ah g-r-aw1-n-d','rb'],
'ah':['aa1','uh vbp'],
'ahead':['ah hh-eh1-d','rb jj'],
'ahs':['aa1-z','uh'],
'aid':['ey1-d','nn vb vbp'],
'aide':['ey1-d','nn'],
'aided':['ey1 d-ah-d','vbn vbd'],
'aiding':['ey1 d-ih-ng','vbg'],
'aids':['ey1-d-z','nns vbz'],
'ailerons':['ey1 l-er aa-n-z','nns'],
'ailing':['ey1 l-ih-ng','vbg jj'],
'ailment':['ey1-l m-ah-n-t','nn'],
'ails':['ey1-l-z','nns vbz'],
'aim':['ey1-m','nn vbp vb'],
'aimed':['ey1-m-d','vbn vbd'],
'aiming':['ey1 m-ih-ng','vbg'],
'aimless':['ey1-m l-ah-s','jj'],
'aimlessly':['ey1-m l-ah-s l-iy','rb'],
'aims':['ey1-m-z','vbz nns'],
'air':['eh1-r','nn vb vbn vbp'],
'airbags':['eh1-r b-ae-g-z','nns'],
'airborne':['eh1-r b-ao-r-n','jj'],
'aircraft':['eh1-r k-r-ae-f-t','nn nns'],
'aired':['eh1-r-d','vbn vbd'],
'airfare':['eh1-r f-eh-r','nn'],
'airfield':['eh1-r f-iy-l-d','nn'],
'airflow':['eh1-r f-l-ow','nn'],
'airframe':['eh1-r f-r-ey-m','nn'],
'airing':['eh1 r-ih-ng','vbg nn'],
'airless':['eh1-r l-ah-s','jj'],
'airlift':['eh1-r l-ih-f-t','nn'],
'airlifted':['eh1-r l-ih-f t-ih-d','vbn'],
'airlifting':['eh1-r l-ih-f t-ih-ng','vbg'],
'airline':['eh1-r l-ay-n','nn'],
'airliner':['eh1-r l-ay n-er','nn'],
'airmail':['eh1-r m-ey-l','nn'],
'airman':['eh1-r m-ah-n','nn'],
'airplane':['eh1-r p-l-ey-n','nn'],
'airplanes':['eh1-r p-l-ey-n-z','nns vbz'],
'airport':['eh1-r p-ao-r-t','nn'],
'airs':['eh1-r-z','nns vbz'],
'airspeed':['eh1-r s-p-iy-d','nn'],
'airstrip':['eh1-r s-t-r-ih-p','nn'],
'airtime':['eh1-r t-ay-m','nn'],
'airwaves':['eh1-r w-ey-v-z','nns'],
'airway':['eh1-r w-ey','nn'],
'airworthiness':['eh1-r w-er dh-iy n-ah-s','nn'],
'airy':['eh1 r-iy','jj'],
'aisle':['ay1-l','nn'],
'ajar':['ah jh-aa1-r','rb'],
'akin':['ah k-ih1-n','jj rb'],
'alabaster':['ae1 l-ah b-ae s-t-er','nn jjr'],
'alarm':['ah l-aa1-r-m','nn vb'],
'alarmed':['ah l-aa1-r-m-d','vbn jj'],
'alarming':['ah l-aa1-r m-ih-ng','jj vbg'],
'alarmingly':['ah l-aa1-r m-ih-ng l-iy','rb'],
'alarmist':['ah l-aa1-r m-ah-s-t','jj'],
'alarms':['ah l-aa1-r-m-z','nns vbz'],
'alas':['ah l-ae1-s','uh'],
'albatross':['ae1-l b-ah t-r-aa-s','nn'],
'albeit':['ao-l b-iy1 ih-t','in rb'],
'albino':['ae-l b-ay1 n-ow','nn'],
'album':['ae1-l b-ah-m','nn'],
'albumin':['ae-l b-y-uw1 m-ah-n','nn'],
'alchemy':['ae1-l k-ah m-iy','nn'],
'alcohol':['ae1-l k-ah hh-aa-l','nn'],
'alcoholic':['ae-l k-ah hh-aa1 l-ih-k','jj nn'],
'alcoholism':['ae1-l k-ah hh-ao l-ih z-ah-m','nn'],
'alcoves':['ae1-l k-ow-v-z','nns'],
'alderman':['ao1-l d-er m-ah-n','nn'],
'ale':['ey1-l','nn'],
'alert':['ah l-er1-t','jj vbp nn vb'],
'alerted':['ah l-er1 t-ih-d','vbd vbn'],
'alerting':['ah l-er1 t-ih-ng','vbg'],
'alertness':['ah l-er1-t n-ah-s','nn'],
'alerts':['ah l-er1-t-s','vbz nns'],
'alfalfa':['ae-l f-ae1-l f-ah','nn'],
'algae':['ae1-l jh-iy','nns'],
'algebra':['ae1-l jh-ah b-r-ah','nn'],
'algebraic':['ae-l jh-ah b-r-ey1 ih-k','jj'],
'algebraically':['ae-l jh-ah b-r-ey1 ih-k l-iy','rb'],
'algorithm':['ae1-l g-er ih dh-ah-m','nn'],
'alias':['ey1 l-iy ah-s','nn'],
'alibi':['ae1 l-ah b-ay','nn'],
'alien':['ey1 l-iy ah-n','jj nn'],
'alienate':['ey1 l-y-ah n-ey-t','vb'],
'alienated':['ey1 l-iy ah n-ey t-ah-d','vbn vbd'],
'alienates':['ey1 l-iy ah n-ey-t-s','vbz'],
'alienating':['ey1 l-iy ah n-ey t-ih-ng','vbg jj'],
'alienation':['ey l-iy ah n-ey1 sh-ah-n','nn'],
'alight':['ah l-ay1-t','jj vb'],
'align':['ah l-ay1-n','vb vbp'],
'aligned':['ah l-ay1-n-d','vbn jj'],
'alignment':['ah l-ay1-n m-ah-n-t','nn'],
'alike':['ah l-ay1-k','rb jj'],
'alimony':['ae1 l-ah m-ow n-iy','nn'],
'alive':['ah l-ay1-v','jj rb'],
'alkali':['ae1-l k-ah l-ay','nns nn'],
'alkaline':['ae1-l k-ah l-ay-n','jj'],
'alkaloids':['ae1-l k-ah l-oy-d-z','nns'],
'all':['ao1-l','dt rb pdt'],
'allay':['ah l-ey1','vb'],
'allayed':['ah l-ey1-d','vbn'],
'allaying':['ah l-ey1 ih-ng','vbg'],
'allegation':['ae l-ah g-ey1 sh-ah-n','nn'],
'allege':['ah l-eh1-jh','vbp vb'],
'alleged':['ah l-eh1-jh-d','vbn jj vbd'],
'allegedly':['ah l-eh1 jh-ah-d l-iy','rb'],
'alleges':['ah l-eh1 jh-ah-z','vbz nns'],
'allegiance':['ah l-iy1 jh-ah-n-s','nn'],
'alleging':['ah l-eh1 jh-ih-ng','vbg'],
'allegorical':['ae l-ah g-ao1 r-ah k-ah-l','jj'],
'allegory':['ae1 l-ah g-ao r-iy','nn'],
'allegro':['ah l-eh1 g-r-ow','jj'],
'allergic':['ah l-er1 jh-ih-k','jj'],
'allergy':['ae1 l-er jh-iy','nn'],
'alleviate':['ah l-iy1 v-iy ey-t','vb vbp'],
'alleviates':['ah l-iy1 v-iy ey-t-s','vbz'],
'alleviating':['ah l-iy1 v-iy ey t-ih-ng','vbg'],
'alley':['ae1 l-iy','nn'],
'alleyways':['ae1 l-iy w-ey-z','nns'],
'alliance':['ah l-ay1 ah-n-s','nn'],
'allied':['ah l-ay1-d','vbn jj vbd'],
'allies':['ae1 l-ay-z','nns'],
'alligator':['ae1 l-ah g-ey t-er','nn'],
'alliterative':['ah l-ih1 t-er ah t-ih-v','jj'],
'allocate':['ae1 l-ah k-ey-t','vb vbp'],
'allocated':['ae1 l-ah k-ey t-ih-d','vbn vbd'],
'allocates':['ae1 l-ah k-ey-t-s','vbz'],
'allocating':['ae1 l-ah k-ey t-ih-ng','vbg nn'],
'allocation':['ae l-ah k-ey1 sh-ah-n','nn'],
'allocator':['ae1 l-ah k-ey t-er','nn'],
'allot':['ah l-aa1-t','vb'],
'alloted':['ah l-aa1 t-ih-d','vbn'],
'allotment':['ah l-aa1-t m-ah-n-t','nn'],
'allotted':['ah l-aa1 t-ih-d','vbn vbd'],
'allotting':['ah l-aa1 t-ih-ng','vbg'],
'allow':['ah l-aw1','vb vbp'],
'allowable':['ah l-aw1 ah b-ah-l','jj'],
'allowance':['ah l-aw1 ah-n-s','nn'],
'allowed':['ah l-aw1-d','vbn jj vb vbd'],
'allowing':['ah l-aw1 ih-ng','vbg'],
'allows':['ah l-aw1-z','vbz'],
'alloy':['ae1 l-oy','nn'],
'alltime':['ao1-l t-ay-m','nn'],
'alluded':['ah l-uw1 d-ah-d','vbd vbn'],
'alludes':['ah l-uw1-d-z','vbz'],
'alluding':['ah l-uw1 d-ih-ng','vbg'],
'allure':['ah l-uh1-r','nn'],
'alluring':['ah l-uh1 r-ih-ng','jj'],
'allusion':['ah l-uw1 zh-ah-n','nn'],
'alluvial':['ae l-uw1 v-iy ah-l','jj'],
'ally':['ae1 l-ay','nn vb'],
'allying':['ae1 l-ay ih-ng','vbg'],
'alma':['ae1-l m-ah','jj nn'],
'almanac':['ao1-l m-ah n-ae-k','nn'],
'almond':['aa1 m-ah-n-d','nn'],
'almost':['ao1-l m-ow-s-t','rb jj'],
'aloft':['ah l-ao1-f-t','rb'],
'alone':['ah l-ow1-n','rb jj'],
'along':['ah l-ao1-ng','in rb rp'],
'alongside':['ah l-ao1-ng s-ay1-d','in rb'],
'aloof':['ah l-uw1-f','jj rb'],
'aloofness':['ah l-uw1-f n-ah-s','nn'],
'aloud':['ah l-aw1-d','rb'],
'alpha':['ae1-l f-ah','jj nn'],
'alphabet':['ae1-l f-ah b-eh-t','nn'],
'alphabetic':['ae-l f-ah b-eh1 t-ih-k','jj'],
'alphabetical':['ae-l f-ah b-eh1 t-ih k-ah-l','jj'],
'alphabetically':['ae-l f-ah b-eh1 t-ih-k l-iy','rb'],
'already':['ao-l r-eh1 d-iy','rb'],
'also':['ao1-l s-ow','rb .'],
'altar':['ao1-l t-er','nn'],
'alter':['ao1-l t-er','vb vbp jj'],
'alteration':['ao-l t-er ey1 sh-ah-n','nn'],
'altercation':['aa-l t-er k-ey1 sh-ah-n','nn'],
'altered':['ao1-l t-er-d','vbn jj vbd'],
'altering':['ao1-l t-er ih-ng','vbg'],
'alternate':['ao1-l t-er n-ah-t','jj vbp nn vb'],
'alternated':['ao1-l t-er n-ey t-ah-d','vbd vbn'],
'alternately':['ao1-l t-er n-ah-t l-iy','rb'],
'alternates':['ao1-l t-er n-ey-t-s','vbz nns'],
'alternating':['ao1-l t-er n-ey t-ih-ng','vbg'],
'alternative':['ao-l t-er1 n-ah t-ih-v','nn jj'],
'alternatively':['ao-l t-er1 n-ah t-ih-v l-iy','rb'],
'alters':['ao1-l t-er-z','vbz'],
'although':['ao-l dh-ow1','in'],
'altitude':['ae1-l t-ah t-uw-d','nn'],
'alto':['ae1-l t-ow','nn'],
'altogether':['ao-l t-ah g-eh1 dh-er','rb'],
'altruism':['ae1-l t-r-uw ih z-ah-m','nn'],
'altruistic':['ao-l t-r-uw ih1 s-t-ih-k','jj'],
'alum':['ae1 l-ah-m','nn'],
'alumina':['ah l-uw1 m-ah n-ah','nn'],
'aluminum':['ah l-uw1 m-ah n-ah-m','nn jj'],
'alumnus':['ah l-ah1-m n-ah-s','nn'],
'alveolar':['ae-l v-iy1 ah l-er','nn jj'],
'alveoli':['ae-l v-iy1 ah l-ay','nns'],
'always':['ao1-l w-ey-z','rb'],
'am':['ae1-m','vbp rb'],
'amalgam':['ah m-ae1-l g-ah-m','nn'],
'amalgamated':['ah m-ae1-l g-ah m-ey t-ih-d','vbn'],
'amalgamation':['ah m-ae-l g-ah m-ey1 sh-ah-n','nn'],
'amass':['ah m-ae1-s','vb vbp'],
'amassed':['ah m-ae1-s-t','vbn vbd'],
'amassing':['ah m-ae1 s-ih-ng','nn vbg'],
'amateur':['ae1 m-ah t-er','nn jj'],
'amateurish':['ae1 m-ah ch-er ih-sh','jj'],
'amateurism':['ae1 m-ah ch-er ih z-ah-m','nn'],
'amaze':['ah m-ey1-z','vb vbp'],
'amazed':['ah m-ey1-z-d','vbn vbd jj'],
'amazement':['ah m-ey1-z m-ah-n-t','nn'],
'amazing':['ah m-ey1 z-ih-ng','jj'],
'amazingly':['ah m-ey1 z-ih-ng l-iy','rb'],
'amazons':['ae1 m-ah z-aa-n-z','nns'],
'ambassador':['ae-m b-ae1 s-ah d-er','nn'],
'amber':['ae1-m b-er','jj nn'],
'ambiance':['ae1-m b-iy ah-n-s','nn'],
'ambidextrous':['ae-m b-ah d-eh1-k-s t-r-ah-s','jj'],
'ambiguity':['ae-m b-ih g-y-uw1 ah t-iy','nn'],
'ambiguous':['ae-m b-ih1 g-y-uw ah-s','jj'],
'ambition':['ae-m b-ih1 sh-ah-n','nn'],
'ambitious':['ae-m b-ih1 sh-ah-s','jj'],
'ambitiously':['ae-m b-ih1 sh-ah-s l-iy','rb'],
'ambivalence':['ae-m b-ih1 v-ah l-ah-n-s','nn'],
'ambivalent':['ae-m b-ih1 v-ah l-ah-n-t','jj'],
'amble':['ae1-m b-ah-l','vb'],
'ambled':['ae1-m b-ah-l-d','vbd'],
'ambling':['ae1-m b-ah-l ih-ng','vbg'],
'ambrosial':['ae-m b-r-ow1 zh-ah-l','jj'],
'ambulance':['ae1-m b-y-ah l-ah-n-s','nn'],
'ambulatory':['ae1-m b-y-ah l-ah t-ao r-iy','jj'],
'ambush':['ae1-m b-uh-sh','nn vb'],
'ambushed':['ae1-m b-uh-sh-t','vbd vbn'],
'amelioration':['ah m-iy l-y-er ey1 sh-ah-n','nn'],
'amen':['ey m-eh1-n','uh'],
'amenable':['ah m-eh1 n-ah b-ah-l','jj'],
'amend':['ah m-eh1-n-d','vb vbp'],
'amended':['ah m-eh1-n d-ah-d','vbn vbd'],
'amending':['ah m-eh1-n d-ih-ng','vbg'],
'amendment':['ah m-eh1-n-d m-ah-n-t','nn'],
'amenities':['ah m-eh1 n-ah t-iy-z','nns'],
'amiable':['ey1 m-iy ah b-ah-l','jj'],
'amicable':['ae1 m-ih k-ah b-ah-l','jj'],
'amicably':['ae1 m-ih k-ah b-l-iy','rb'],
'amicus':['ah m-iy1 k-ah-s','nn'],
'amid':['ah m-ih1-d','in'],
'amidst':['ah m-ih1-d-s-t','in'],
'amiss':['ah m-ih1-s','jj'],
'amity':['ae1 m-ih t-iy','nn'],
'ammo':['ae1 m-ow','nn'],
'ammonia':['ah m-ow1 n-y-ah','nn'],
'ammonium':['ah m-ow1 n-iy ah-m','nn'],
'ammunition':['ae m-y-ah n-ih1 sh-ah-n','nn'],
'amnesty':['ae1-m n-ah s-t-iy','nn'],
'amniotic':['ae1-m n-iy ao t-ih-k','jj'],
'amok':['ah m-ah1-k','rb'],
'among':['ah m-ah1-ng','in'],
'amongst':['ah m-ah1-ng-s-t','in'],
'amoral':['ey m-ao1 r-ah-l','jj'],
'amorist':['ae1 m-er ah-s-t','nn'],
'amorous':['ae1 m-er ah-s','jj'],
'amorphous':['ah m-ao1-r f-ah-s','jj'],
'amortization':['ae m-er t-ih z-ey1 sh-ah-n','nn'],
'amortize':['ae1 m-er t-ay-z','vb'],
'amortized':['ae1 m-er t-ay-z-d','vbn'],
'amortizing':['ae1 m-er t-ay z-ih-ng','jj vbg'],
'amount':['ah m-aw1-n-t','nn vb vbp'],
'amounted':['ah m-aw1-n t-ih-d','vbd vbn'],
'amounting':['ah m-aw1-n t-ih-ng','vbg'],
'amounts':['ah m-aw1-n-t-s','nns vbz'],
'amphetamines':['ae-m f-eh1 t-ah m-iy-n-z','nns'],
'amphibious':['ae-m f-ih1 b-iy ah-s','jj'],
'amphitheater':['ae1-m f-ah th-iy ah t-er','nn'],
'ample':['ae1-m p-ah-l','jj'],
'amplification':['ae-m p-l-ah f-ah k-ey1 sh-ah-n','nn'],
'amplified':['ae1-m p-l-ah f-ay-d','vbn'],
'amplifier':['ae1-m p-l-ah f-ay er','nn'],
'amplifies':['ae1-m p-l-ah f-ay-z','vbz'],
'amplify':['ae1-m p-l-ah f-ay','vb'],
'amplifying':['ae1-m p-l-ah f-ay ih-ng','vbg'],
'amplitude':['ae1-m p-l-ah t-uw-d','nn'],
'amply':['ae1-m p-l-iy','rb'],
'amputated':['ae1-m p-y-ah t-ey t-ih-d','vbn'],
'amputation':['ae-m p-y-ah t-ey1 sh-ah-n','nn'],
'amulet':['ae1 m-y-ah l-ah-t','nn'],
'amuse':['ah m-y-uw1-z','vb vbp'],
'amused':['ah m-y-uw1-z-d','vbn jj vbd'],
'amusement':['ah m-y-uw1-z m-ah-n-t','nn'],
'amusing':['ah m-y-uw1 z-ih-ng','jj'],
'amusingly':['ah m-y-uw1 z-ih-ng l-iy','rb'],
'an':['ae1-n','dt cc jj nnp'],
'anachronism':['ah n-ae1 k-r-ah n-ih z-ah-m','nn'],
'anachronistic':['ah n-ae k-r-ah n-ih1 s-t-ih-k','jj'],
'anaconda':['ae n-ah k-aa1-n d-ah','nn'],
'anaerobic':['ae n-er ow1 b-ih-k','jj'],
'anaesthesia':['ae n-ah-s th-iy1 zh-ah','nn'],
'anagram':['ae1 n-ah g-r-ae-m','nn'],
'analgesic':['ae n-ah-l jh-iy1 s-ih-k','jj nn'],
'analog':['ae1 n-ah l-ao-g','nn jj'],
'analogous':['ah n-ae1 l-ah g-ah-s','jj'],
'analogue':['ae1 n-ah l-ao-g','nn'],
'analogy':['ah n-ae1 l-ah jh-iy','nn'],
'analyses':['ah n-ae1 l-ah s-iy-z','nns vbz'],
'analysis':['ah n-ae1 l-ah s-ah-s','nn'],
'analyst':['ae1 n-ah l-ah-s-t','nn'],
'analytic':['ae n-ah l-ih1 t-ih-k','jj nn'],
'analytical':['ae n-ah l-ih1 t-ih k-ah-l','jj'],
'analytically':['ae n-ah l-ih1 t-ih-k l-iy','rb'],
'analyticity':['ae n-ah l-ah t-ih1 s-ah t-iy','nn'],
'analyzable':['ae1 n-ah l-ay z-ah b-ah-l','jj'],
'analyze':['ae1 n-ah l-ay-z','vb vbp'],
'analyzed':['ae1 n-ah l-ay-z-d','vbn vbd'],
'analyzer':['ae1 n-ah l-ay z-er','nn'],
'analyzes':['ae1 n-ah l-ay z-ih-z','vbz'],
'analyzing':['ae1 n-ah l-ay z-ih-ng','vbg nn'],
'anarchic':['ae n-aa1-r k-ih-k','jj'],
'anarchical':['ae n-aa1-r k-ah k-ah-l','jj'],
'anarchist':['ae1 n-er k-ah-s-t','nn'],
'anarchy':['ae1 n-er k-iy','nn'],
'anathema':['ah n-ae1 th-ah m-ah','nn'],
'anatomical':['ae n-ah t-aa1 m-ah k-ah-l','jj nn'],
'anatomically':['ae n-ah t-aa1 m-ah-k l-iy','rb'],
'anatomy':['ah n-ae1 t-ah m-iy','nn'],
'ancestor':['ae1-n s-eh s-t-er','nn'],
'ancestral':['ae-n s-eh1-s t-r-ah-l','jj'],
'ancestry':['ae1-n s-eh-s t-r-iy','nn'],
'anchor':['ae1-ng k-er','nn vbp vb'],
'anchorage':['ae1-ng k-er ah-jh','nn'],
'anchored':['ae1-ng k-er-d','vbn vbd jj'],
'anchoring':['ae1-ng k-er ih-ng','vbg'],
'anchors':['ae1-ng k-er-z','nns vbz'],
'anchovy':['ae-n ch-ow1 v-iy','nn'],
'ancient':['ey1-n ch-ah-n-t','jj nn'],
'ancillary':['ae1-n s-ah l-eh r-iy','jj'],
'and':['ae1-n-d','cc vbp jj rb nnp'],
'anecdotal':['ae n-ah-k d-ow1 t-ah-l','jj'],
'anecdote':['ae1 n-ah-k d-ow-t','nn'],
'anemia':['ah n-iy1 m-iy ah','nn'],
'anemic':['ah n-iy1 m-ih-k','jj'],
'anesthetic':['ae n-ah-s th-eh1 t-ih-k','nn'],
'anew':['ah n-uw1','rb'],
'angel':['ey1-n jh-ah-l','nn'],
'angelfish':['ey1-n jh-ah-l f-ih-sh','nn'],
'angelic':['ae-n jh-eh1 l-ih-k','jj'],
'angelica':['ae-n jh-eh1 l-ih k-ah','nn'],
'anger':['ae1-ng g-er','nn vbp'],
'angered':['ae1-ng g-er-d','vbn jj vbd'],
'angering':['ae1-ng g-er ih-ng','vbg'],
'angers':['ae1-ng g-er-z','vbz'],
'angina':['ae-n jh-ay1 n-ah','nn'],
'angiotensin':['ae-n jh-iy ow t-eh1-n s-ih-n','nn'],
'angle':['ae1-ng g-ah-l','nn vb'],
'angler':['ae1-ng g-l-er','nn'],
'angles':['ae1-ng g-ah-l-z','nns'],
'angling':['ae1-ng g-l-ih-ng','vbg nn'],
'angora':['ae-ng g-ao1 r-ah','nn'],
'angrily':['ae1-ng g-r-ah l-iy','rb'],
'angry':['ae1-ng g-r-iy','jj'],
'angst':['aa1-ng-k-s-t','nn'],
'anguish':['ae1-ng g-w-ih-sh','nn vbp'],
'anguished':['ae1-ng g-w-ih-sh-t','jj'],
'angular':['ae1-ng g-y-ah l-er','jj'],
'aniline':['ae1 n-ah l-iy-n','nn'],
'animal':['ae1 n-ah m-ah-l','nn jj'],
'animate':['ae1 n-ah m-ah-t','jj'],
'animated':['ae1 n-ah m-ey t-ah-d','jj vbd vbn'],
'animates':['ae1 n-ah m-ah-t-s','vbz'],
'animation':['ae n-ah m-ey1 sh-ah-n','nn'],
'animism':['ae1 n-ah m-ih z-ah-m','nn'],
'animosity':['ae n-ah m-aa1 s-ah t-iy','nn'],
'anise':['ae1 n-ah-s','nn'],
'ankle':['ae1-ng k-ah-l','nn'],
'annals':['ae1 n-ah-l-z','nns nn'],
'annex':['ae1 n-eh-k-s','nn vb'],
'annexation':['ae n-eh-k s-ey1 sh-ah-n','nn'],
'annexed':['ae1 n-eh-k-s-t','vbd'],
'annihilate':['ah n-ay1 ah l-ey-t','vb'],
'annihilation':['ah n-ay ah l-ey1 sh-ah-n','nn'],
'anniversary':['ae n-ah v-er1 s-er iy','nn'],
'annotated':['ae n-ah t-ey1 t-ih-d','vbn'],
'announce':['ah n-aw1-n-s','vb vbp'],
'announced':['ah n-aw1-n-s-t','vbd vbn jj'],
'announcement':['ah n-aw1-n s-m-ah-n-t','nn'],
'announcer':['ah n-aw1-n s-er','nn'],
'announces':['ah n-aw1-n s-ih-z','vbz'],
'announcing':['ah n-aw1-n s-ih-ng','vbg'],
'announcment':['ah n-aw1-n s-m-ah-n-t','nn'],
'annoy':['ah n-oy1','vb vbp'],
'annoyance':['ah n-oy1 ah-n-s','nn'],
'annoyed':['ah n-oy1-d','vbn vbd jj'],
'annoying':['ah n-oy1 ih-ng','jj nn'],
'annoys':['ah n-oy1-z','vbz'],
'annual':['ae1 n-y-uw ah-l','jj nn'],
'annualized':['ae1 n-y-uw w-ah l-ay-z-d','vbn vbd jj'],
'annually':['ae1 n-y-uw ah l-iy','rb'],
'annuity':['ah n-uw1 ah t-iy','nn'],
'annulled':['ae1 n-ah-l-d','vbd'],
'annum':['ae1 n-ah-m','nn'],
'anode':['ae1 n-ow-d','nn'],
'anomalous':['ah n-aa1 m-ah l-ah-s','jj'],
'anomaly':['ah n-aa1 m-ah l-iy','nn'],
'anonymity':['ae n-ah n-ih1 m-ih t-iy','nn'],
'anonymous':['ah n-aa1 n-ah m-ah-s','jj'],
'anonymously':['ah n-aa1 n-ah m-ah-s l-iy','rb'],
'anorexia':['ae n-er eh1-k s-iy ah','nn'],
'another':['ah n-ah1 dh-er','dt jj nn'],
'answer':['ae1-n s-er','nn vb vbp'],
'answerable':['ae1-n s-er ah b-ah-l','jj'],
'answered':['ae1-n s-er-d','vbd vbn'],
'answering':['ae1-n s-er ih-ng','vbg nn'],
'answers':['ae1-n s-er-z','nns vbz'],
'ant':['ae1-n-t','nn'],
'antacid':['ae-n t-ae1 s-ah-d','nn'],
'antagonism':['ae-n t-ae1 g-ah n-ih z-ah-m','nn'],
'antagonist':['ae-n t-ae1 g-ah n-ah-s-t','nn'],
'antagonistic':['ae-n t-ae g-ah n-ih1 s-t-ih-k','jj'],
'antagonize':['ae-n t-ae1 g-ah n-ay-z','vb vbp'],
'ante':['ae1-n t-iy','nn vb'],
'anteaters':['ae1-n-t iy t-er-z','nns'],
'antebellum':['ae-n t-ih b-eh1 l-ah-m','jj'],
'antecedents':['ae-n t-ih s-iy1 d-ah-n-t-s','nns'],
'antelope':['ae1-n t-ah l-ow-p','nn nns'],
'antenna':['ae-n t-eh1 n-ah','nn'],
'anterior':['ae-n t-ih1 r-iy er','jj'],
'anthem':['ae1-n th-ah-m','nn'],
'anthology':['ae-n th-aa1 l-ah jh-iy','nn'],
'anthrax':['ae1-n th-r-ae-k-s','nn'],
'anthropological':['ae-n th-r-ah p-ah l-aa1 jh-ah k-ah-l','jj'],
'anthropologist':['ae-n th-r-ah p-aa1 l-ah jh-ah-s-t','nn'],
'anthropology':['ae-n th-r-ah p-aa1 l-ah jh-iy','nn'],
'anthropomorphic':['ae-n th-r-ah p-ah m-ao1-r f-ih-k','jj'],
'anti':['ae1-n t-iy','in nn'],
'antibacterial':['ae-n t-iy b-ae-k t-ih1 r-iy ah-l','jj'],
'antibiotic':['ae-n t-iy b-iy aa1 t-ih-k','nn jj'],
'antibody':['ae1-n t-ih b-aa d-iy','nn'],
'antic':['ae1-n t-ih-k','jj'],
'anticipate':['ae-n t-ih1 s-ah p-ey-t','vb vbp'],
'anticipated':['ae-n t-ih1 s-ah p-ey t-ah-d','vbn jj vbd'],
'anticipates':['ae-n t-ih1 s-ah p-ey-t-s','vbz'],
'anticipating':['ae-n t-ih1 s-ah p-ey t-ih-ng','vbg'],
'anticipation':['ae-n t-ih s-ah p-ey1 sh-ah-n','nn'],
'anticipatory':['ae-n t-ih1 s-ah p-ah t-ao r-iy','jj'],
'anticompetitive':['ae-n t-ih k-ah-m p-eh1 t-ih t-ih-v','jj'],
'anticorruption':['ae-n t-ay k-er ah1-p sh-ah-n','nn'],
'antidepressant':['ae-n t-iy d-ih p-r-eh1 s-ah-n-t','nn'],
'antidote':['ae1-n t-ih d-ow-t','nn'],
'antifreeze':['ae1-n t-iy f-r-iy-z','nn'],
'antigen':['ae1-n t-ah jh-ah-n','nn'],
'antihistamine':['ae-n t-iy hh-ih1 s-t-ah m-ah-n','nn'],
'antilock':['ae1-n t-iy l-aa1-k','jj'],
'antimissile':['ae-n t-ay m-ih1 s-ah-l','jj'],
'antipathy':['ae-n t-ih1 p-ah th-iy','nn'],
'antiquated':['ae1-n t-ah k-w-ey t-ah-d','jj'],
'antique':['ae-n t-iy1-k','jj nn'],
'antiquity':['ae-n t-ih1 k-w-ah t-iy','nn'],
'antiseptic':['ae-n t-ah s-eh1-p t-ih-k','jj nn'],
'antismoking':['ae1-n t-iy-s m-ow1 k-ih-ng','jj'],
'antisocial':['ae-n t-ih s-ow1 sh-ah-l','jj'],
'antisubmarine':['ae-n t-ih s-ah1-b m-er iy-n','jj'],
'antitakeover':['ae-n t-iy t-ey1-k ow v-er','jjr'],
'antithesis':['ae-n t-ih1 th-ah s-ah-s','nn'],
'antithetical':['ae-n t-ah th-eh1 t-ih k-ah-l','jj'],
'antitrust':['ae-n t-ay t-r-ah1-s-t','jj nn'],
'antiviral':['ae-n t-iy v-ay1 r-ah-l','jj'],
'antiwar':['ae-n t-ay w-ao1-r','jj'],
'antsy':['ae1-n-t s-iy','jj rb'],
'anvil':['ae1-n v-ah-l','nn'],
'anxiety':['ae-ng z-ay1 ah t-iy','nn'],
'anxious':['ae1-ng-k sh-ah-s','jj'],
'anxiously':['ae1-ng-k sh-ah-s l-iy','rb'],
'any':['eh1 n-iy','dt rb'],
'anybody':['eh1 n-iy b-ah d-iy','nn'],
'anyhow':['eh1 n-iy hh-aw','rb'],
'anymore':['eh n-iy m-ao1-r','rb'],
'anyone':['eh1 n-iy w-ah-n','nn'],
'anyplace':['eh1 n-iy p-l-ey-s','rb'],
'anything':['eh1 n-iy th-ih-ng','nn'],
'anytime':['eh1 n-iy t-ay-m','rb'],
'anyway':['eh1 n-iy w-ey','rb'],
'anyways':['eh1 n-iy w-ey-z','uh'],
'anywhere':['eh1 n-iy w-eh-r','rb'],
'aorta':['ey ao1-r t-ah','nn'],
'apace':['ah p-ey1-s','rb'],
'apart':['ah p-aa1-r-t','rb rp jj'],
'apartheid':['ah p-aa1-r-t ay-t','nn'],
'apartment':['ah p-aa1-r-t m-ah-n-t','nn'],
'apathetic':['ae p-ah th-eh1 t-ih-k','jj'],
'apathy':['ae1 p-ah th-iy','nn'],
'ape':['ey1-p','nn'],
'aperture':['ae1 p-er ch-er','nn'],
'apex':['ey1 p-eh-k-s','nn'],
'aphorisms':['ae1 f-er ih z-ah-m-z','nns'],
'apiece':['ah p-iy1-s','rb jj'],
'aping':['ey1 p-ih-ng','vbg'],
'aplenty':['ah p-l-eh1-n t-iy','jj rb'],
'aplomb':['ah p-l-aa1-m','nn'],
'apocalypse':['ah p-aa1 k-ah l-ih-p-s','nn'],
'apocalyptic':['ah p-aa k-ah l-ih1-p t-ih-k','jj'],
'apocryphal':['ah p-aa1 k-r-ah f-ah-l','jj'],
'apogee':['ae1 p-ah jh-iy','nn'],
'apologetic':['ah p-aa l-ah jh-eh1 t-ih-k','jj'],
'apologetically':['ah p-aa l-ah jh-eh1 t-ih-k l-iy','rb'],
'apologist':['ah p-aa1 l-ah jh-ah-s-t','nn'],
'apologize':['ah p-aa1 l-ah jh-ay-z','vb'],
'apologized':['ah p-aa1 l-ah jh-ay-z-d','vbd vbn'],
'apologizes':['ah p-aa1 l-ah jh-ay z-ih-z','vbz'],
'apologizing':['ah p-aa1 l-ah jh-ay z-ih-ng','vbg'],
'apology':['ah p-aa1 l-ah jh-iy','nn'],
'apoplectic':['ae p-ah p-l-eh1-k t-ih-k','jj'],
'apostle':['ah p-aa1 s-ah-l','nn'],
'apothecary':['ah p-aa1 th-ah k-eh r-iy','nn'],
'apotheosis':['ah p-aa th-iy ow1 s-ah-s','nn'],
'appalled':['ah p-ao1-l-d','vbn jj'],
'appalling':['ah p-ao1 l-ih-ng','jj'],
'appallingly':['ah p-ao1 l-ih-ng l-iy','rb'],
'appalls':['ah p-ao1-l-z','vbz'],
'apparatus':['ae p-er ae1 t-ah-s','nn'],
'apparel':['ah p-ae1 r-ah-l','nn vb'],
'apparent':['ah p-eh1 r-ah-n-t','jj nn'],
'apparently':['ah p-eh1 r-ah-n-t l-iy','rb'],
'apparition':['ae p-er ih1 sh-ah-n','nn'],
'appeal':['ah p-iy1-l','nn vbp vb'],
'appealed':['ah p-iy1-l-d','vbd vbn'],
'appealing':['ah p-iy1 l-ih-ng','jj vbg'],
'appeals':['ah p-iy1-l-z','nns vbz'],
'appear':['ah p-ih1-r','vb vbp'],
'appearance':['ah p-ih1 r-ah-n-s','nn'],
'appeared':['ah p-ih1-r-d','vbd vbn'],
'appearing':['ah p-ih1 r-ih-ng','vbg'],
'appears':['ah p-ih1-r-z','vbz nns'],
'appease':['ah p-iy1-z','vb'],
'appeased':['ah p-iy1-z-d','vbn vbd'],
'appeasement':['ah p-iy1-z m-ah-n-t','nn'],
'appeasing':['ah p-iy1 z-ih-ng','nn'],
'appellate':['ah p-eh1 l-ih-t','jj nn'],
'append':['ah p-eh1-n-d','vb'],
'appendages':['ah p-eh1-n d-ih jh-ih-z','nns'],
'appended':['ah p-eh1-n d-ih-d','vbn'],
'appetite':['ae1 p-ah t-ay-t','nn'],
'appetizer':['ae1 p-ah t-ay z-er','nn'],
'appetizing':['ae1 p-ih t-ay z-ih-ng','jj'],
'applaud':['ah p-l-ao1-d','vbp vb'],
'applauded':['ah p-l-ao1 d-ah-d','vbd vbn'],
'applauding':['ah p-l-ao1 d-ih-ng','vbg'],
'applauds':['ah p-l-ao1-d-z','vbz'],
'applause':['ah p-l-ao1-z','nn'],
'apple':['ae1 p-ah-l','nn'],
'apples':['ae1 p-ah-l-z','nns jj'],
'appliance':['ah p-l-ay1 ah-n-s','nn'],
'applicability':['ae p-l-ah k-ah b-ih1 l-ah t-iy','nn'],
'applicable':['ae1 p-l-ah k-ah b-ah-l','jj'],
'applicant':['ae1 p-l-ih k-ah-n-t','nn'],
'application':['ae p-l-ah k-ey1 sh-ah-n','nn'],
'applicator':['ae1 p-l-ah k-ey t-er','nn'],
'applied':['ah p-l-ay1-d','vbn vbd jj'],
'applies':['ah p-l-ay1-z','vbz'],
'apply':['ah p-l-ay1','vb vbp'],
'applying':['ah p-l-ay1 ih-ng','vbg'],
'appoint':['ah p-oy1-n-t','vb'],
'appointed':['ah p-oy1-n t-ah-d','vbn jj vbd'],
'appointee':['ah p-oy-n t-iy1','nn'],
'appointing':['ah p-oy1-n t-ih-ng','vbg'],
'appointment':['ah p-oy1-n-t m-ah-n-t','nn'],
'appoints':['ah p-oy1-n-t-s','vbz'],
'apportion':['ah p-ao1-r sh-ah-n','vb'],
'apportioned':['ah p-ao1-r sh-ah-n-d','vbn'],
'apportionment':['ah p-ao1-r sh-ah-n m-ah-n-t','nn'],
'appraisal':['ah p-r-ey1 z-ah-l','nn'],
'appraise':['ah p-r-ey1-z','vb'],
'appraised':['ah p-r-ey1-z-d','vbn'],
'appraiser':['ah p-r-ey1 z-er','nn'],
'appraising':['ah p-r-ey1 z-ih-ng','vbg'],
'appreciable':['ah p-r-iy1 sh-ah b-ah-l','jj'],
'appreciably':['ah p-r-iy1 sh-ah b-l-iy','rb'],
'appreciate':['ah p-r-iy1 sh-iy ey-t','vb vbp'],
'appreciated':['ah p-r-iy1 sh-iy ey t-ih-d','vbn vbd jj'],
'appreciates':['ah p-r-iy1 sh-iy ey-t-s','vbz'],
'appreciating':['ah p-r-iy1 sh-iy ey t-ih-ng','vbg nn'],
'appreciation':['ah p-r-iy sh-iy ey1 sh-ah-n','nn'],
'appreciative':['ah p-r-iy1 sh-iy ey t-ih-v','jj'],
'appreciatively':['ah p-r-iy1 sh-iy ey t-ih-v l-iy','rb'],
'apprehend':['ae p-r-ih hh-eh1-n-d','vb'],
'apprehended':['ae p-r-ih hh-eh1-n d-ah-d','vbn'],
'apprehension':['ae p-r-ih hh-eh1-n sh-ah-n','nn'],
'apprehensive':['ae p-r-ih hh-eh1-n s-ih-v','jj'],
'apprentice':['ah p-r-eh1-n t-ah-s','nn'],
'apprenticed':['ah p-r-eh1-n t-ah-s-t','vbn'],
'apprenticeship':['ah p-r-eh1-n t-ah-s sh-ih-p','nn'],
'apprised':['ah p-r-ay1-z-d','vbn'],
'approach':['ah p-r-ow1-ch','nn vb vbp'],
'approachable':['ah p-r-ow1 ch-ah b-ah-l','jj'],
'approached':['ah p-r-ow1-ch-t','vbd vbn'],
'approaching':['ah p-r-ow1 ch-ih-ng','vbg'],
'appropriate':['ah p-r-ow1 p-r-iy ah-t','jj vbp vb'],
'appropriated':['ah p-r-ow1 p-r-iy ey t-ih-d','vbn vbd'],
'appropriately':['ah p-r-ow1 p-r-iy ih-t l-iy','rb'],
'appropriateness':['ah p-r-ow1 p-r-iy ah-t n-ah-s','nn'],
'appropriates':['ah p-r-ow1 p-r-iy ey-t-s','vbz'],
'appropriating':['ah p-r-ow1 p-r-iy ey t-ih-ng','vbg nn'],
'appropriation':['ah p-r-ow p-r-iy ey1 sh-ah-n','nn'],
'appropriators':['ah p-r-ow1 p-r-iy ey t-er-z','nns'],
'approval':['ah p-r-uw1 v-ah-l','nn'],
'approve':['ah p-r-uw1-v','vb vbp'],
'approved':['ah p-r-uw1-v-d','vbd vbn jj'],
'approves':['ah p-r-uw1-v-z','vbz'],
'approving':['ah p-r-uw1 v-ih-ng','vbg'],
'approvingly':['ah p-r-uw1 v-ih-ng l-iy','rb'],
'approximate':['ah p-r-aa1-k s-ah m-ah-t','jj vbp nn vb'],
'approximated':['ah p-r-aa1-k s-ah m-ey t-ah-d','vbn jj vbd'],
'approximately':['ah p-r-aa1-k s-ah m-ah-t l-iy','rb jj'],
'approximates':['ah p-r-aa1-k s-ah m-ah-t-s','vbz'],
'approximation':['ah p-r-aa-k s-ah m-ey1 sh-ah-n','nn'],
'apricot':['ey1 p-r-ah k-aa-t','nn'],
'april':['ey1 p-r-ah-l','nnp'],
'apron':['ey1 p-r-ah-n','nn'],
'apt':['ae1-p-t','jj'],
'aptitude':['ae1-p t-ah t-uw-d','nn'],
'aptly':['ae1-p-t l-iy','rb'],
'aquamarine':['aa k-w-ah m-er iy1-n','nn'],
'aquarium':['ah k-w-eh1 r-iy ah-m','nn'],
'aquatic':['ah k-w-aa1 t-ih-k','jj'],
'aqueducts':['ae1 k-w-ah d-ah-k-t-s','nns'],
'aqueous':['ey1 k-w-iy ah-s','jj'],
'aquifer':['ae1 k-w-ah f-er','nn'],
'arabic':['ae1 r-ah b-ih-k','jj'],
'arable':['eh1 r-ah b-ah-l','jj'],
'aramid':['eh1 r-ah m-ih-d','nn'],
'arb':['aa1-r-b','nn'],
'arbiter':['aa1-r b-ih t-er','nn'],
'arbitrage':['aa1-r b-ih t-r-aa-zh','nn'],
'arbitrager':['aa1-r b-ih t-r-aa zh-er','nn'],
'arbitrageur':['aa1-r b-ih t-r-aa zh-er','nn'],
'arbitraging':['aa1-r b-ah t-r-aa jh-ih-ng','vbg'],
'arbitrarily':['aa1-r b-ih t-r-eh r-ah l-iy','rb'],
'arbitrary':['aa1-r b-ah t-r-eh r-iy','jj'],
'arbitrate':['aa1-r b-ah t-r-ey-t','vb'],
'arbitrated':['aa1-r b-ah t-r-ey t-ih-d','vbn'],
'arbitrates':['aa1-r b-ih t-r-ey-t-s','vbz'],
'arbitrating':['aa1-r b-ah t-r-ey t-ih-ng','vbg'],
'arbitration':['aa-r b-ih t-r-ey1 sh-ah-n','nn'],
'arbitrator':['aa1-r b-ih t-r-ey t-er','nn'],
'arboreal':['aa-r b-ao1 r-iy ah-l','jj'],
'arc':['aa1-r-k','nn vb'],
'arcade':['aa-r k-ey1-d','nn'],
'arcane':['aa-r k-ey1-n','jj'],
'arch':['aa1-r-ch','nn vbp'],
'archaeological':['aa-r k-iy ah l-aa1 jh-ih k-ah-l','jj'],
'archaeologist':['aa-r k-iy aa1 l-ah jh-ih-s-t','nn'],
'archaeology':['aa-r k-iy aa1 l-ah jh-iy','nn'],
'archaic':['aa-r k-ey1 ih-k','jj'],
'archbishop':['aa1-r-ch b-ih1 sh-ah-p','nn'],
'archdiocese':['aa1-r-ch d-ay1 ah s-ah-s','nn'],
'arched':['aa1-r-ch-t','jj vbd vbn'],
'archenemy':['aa1-r ch-eh1 n-ah m-iy','nn'],
'archeological':['aa-r k-iy ah l-aa1 jh-ih k-ah-l','jj'],
'archery':['aa1-r ch-er iy','nn'],
'arches':['aa1-r ch-ah-z','nns vbz'],
'archetype':['aa1-r k-ih t-ay-p','nn'],
'arching':['aa1-r ch-ih-ng','vbg'],
'archipelago':['aa-r k-ah p-eh1 l-ah g-ow','nn'],
'architect':['aa1-r k-ah t-eh-k-t','nn'],
'architectural':['aa-r k-ah t-eh1-k ch-er ah-l','jj'],
'architecturally':['aa-r k-ih t-eh1-k ch-er ah l-iy','rb'],
'architecture':['aa1-r k-ah t-eh-k ch-er','nn'],
'archival':['aa-r k-ay1 v-ah-l','jj'],
'archive':['aa1-r k-ay-v','nn'],
'archivist':['aa1-r k-ah v-ih-s-t','nn'],
'archrival':['aa1-r-ch r-ay v-ah-l','jj nn'],
'arcs':['aa1-r-k-s','nns'],
'arctic':['aa1-r-k t-ih-k','jj'],
'ardent':['aa1-r d-ah-n-t','jj'],
'ardently':['aa1-r d-ah-n-t l-iy','rb'],
'ardor':['aa1-r d-er','nn'],
'arduous':['aa1-r jh-uw ah-s','jj'],
'are':['aa1-r','vbp nnp'],
'area':['eh1 r-iy ah','nn'],
'areas':['eh1 r-iy ah-z','nns vbn'],
'arena':['er iy1 n-ah','nn'],
'argon':['aa1-r g-aa-n','nn'],
'argot':['aa1-r g-ah-t','nn'],
'arguably':['aa1-r g-y-uw ah b-l-iy','rb'],
'argue':['aa1-r g-y-uw','vbp vb'],
'argued':['aa1-r g-y-uw-d','vbd vbn'],
'argues':['aa1-r g-y-uw-z','vbz'],
'arguing':['aa1-r g-y-uw ih-ng','vbg nn'],
'argument':['aa1-r g-y-ah m-ah-n-t','nn'],
'aria':['aa1 r-iy ah','nn'],
'arid':['ae1 r-ah-d','jj'],
'arise':['er ay1-z','vb vbp'],
'arisen':['er ih1 z-ah-n','vbn'],
'arises':['er ay1 z-ah-z','vbz'],
'arising':['er ay1 z-ih-ng','vbg'],
'aristocracy':['eh r-ah s-t-aa1 k-r-ah s-iy','nn'],
'aristocrat':['er ih1 s-t-ah k-r-ae-t','nn'],
'aristocratic':['er ih s-t-ah k-r-ae1 t-ih-k','jj'],
'arithmetic':['eh r-ih-th m-eh1 t-ih-k','nn'],
'arm':['aa1-r-m','nn vb'],
'armadillo':['aa-r m-ah d-ih1 l-ow','nn'],
'armament':['aa1-r m-ah m-ah-n-t','nn'],
'armchair':['aa1-r-m ch-eh-r','nn jj'],
'armed':['aa1-r-m-d','vbn jj'],
'arming':['aa1-r m-ih-ng','nn'],
'armistice':['aa1-r m-ah s-t-ah-s','nn'],
'armor':['aa1-r m-er','nn'],
'armored':['aa1-r m-er-d','jj vbn'],
'armory':['aa1-r m-er iy','nn'],
'armpit':['aa1-r-m p-ih-t','nn'],
'arms':['aa1-r-m-z','nns'],
'army':['aa1-r m-iy','nn'],
'aroma':['er ow1 m-ah','nn'],
'aromatic':['eh r-ah m-ae1 t-ih-k','jj'],
'arose':['er ow1-z','vbd'],
'around':['er aw1-n-d','in rb rp'],
'arousal':['er aw1 z-ah-l','jj'],
'arouse':['er aw1-z','vb vbp'],
'aroused':['er aw1-z-d','vbn vbd'],
'arouses':['er aw1 z-ih-z','vbz'],
'arousing':['er aw1 z-ih-ng','vbg'],
'arraigned':['er ey1-n-d','vbd vbn'],
'arrange':['er ey1-n-jh','vb vbp'],
'arranged':['er ey1-n-jh-d','vbn vbd'],
'arrangement':['er ey1-n-jh m-ah-n-t','nn'],
'arranges':['er ey1-n jh-ih-z','vbz'],
'arranging':['er ey1-n jh-ih-ng','vbg'],
'array':['er ey1','nn'],
'arrayed':['er ey1-d','vbn'],
'arrearage':['er iy1 r-ih-jh','nn'],
'arrears':['er ih1-r-z','nns'],
'arrest':['er eh1-s-t','nn vbp vb'],
'arrested':['er eh1 s-t-ah-d','vbn vbd'],
'arresting':['er eh1 s-t-ih-ng','vbg jj'],
'arrests':['er eh1-s-t-s','nns'],
'arrival':['er ay1 v-ah-l','nn'],
'arrive':['er ay1-v','vb vbp'],
'arrived':['er ay1-v-d','vbd vbn'],
'arrives':['er ay1-v-z','vbz'],
'arriving':['er ay1 v-ih-ng','vbg'],
'arrogance':['eh1 r-ah g-ah-n-s','nn'],
'arrogant':['eh1 r-ah g-ah-n-t','jj'],
'arrogantly':['eh1 r-ah g-ah-n-t l-iy','rb'],
'arrogate':['ae1 r-ow g-ey-t','vb'],
'arrow':['ae1 r-ow','nn'],
'arrowheads':['ae1 r-ow hh-eh-d-z','nns'],
'arroyo':['er oy1 ow','nn'],
'arsenal':['aa1-r s-ah n-ah-l','nn'],
'arsenic':['aa1-r s-ah n-ih-k','nn'],
'arsenide':['aa1-r s-ah n-ay-d','nn'],
'arson':['aa1-r s-ah-n','nn'],
'arsonist':['aa1-r s-ah n-ah-s-t','nn'],
'art':['aa1-r-t','nn vbp'],
'arterial':['aa-r t-ih1 r-iy ah-l','jj'],
'arteriosclerosis':['aa-r t-ih r-iy ow s-k-l-er ow1 s-ah-s','nn'],
'artery':['aa1-r t-er iy','nn'],
'artful':['aa1-r-t f-ah-l','jj'],
'artfully':['aa1-r-t f-ah l-iy','rb'],
'arthritic':['aa-r th-r-ih1 t-ih-k','jj'],
'arthritis':['aa-r-th r-ay1 t-ah-s','nn'],
'artichoke':['aa1-r t-ah ch-ow-k','nn'],
'article':['aa1-r t-ah k-ah-l','nn'],
'articulate':['aa-r t-ih1 k-y-ah l-ey-t','jj vb'],
'articulated':['aa-r t-ih1 k-y-ah l-ey t-ah-d','vbn'],
'articulation':['aa-r t-ih k-y-ah l-ey1 sh-ah-n','nn'],
'artifact':['aa1-r t-ah f-ae-k-t','nn'],
'artifice':['aa1-r t-ah f-ih-s','nn'],
'artificial':['aa-r t-ah f-ih1 sh-ah-l','jj nn'],
'artificially':['aa-r t-ah f-ih1 sh-ah l-iy','rb'],
'artillery':['aa-r t-ih1 l-er iy','nn'],
'artisan':['aa1-r t-ah z-ah-n','nn'],
'artist':['aa1-r t-ah-s-t','nn'],
'artistic':['aa-r t-ih1 s-t-ih-k','jj'],
'artistically':['aa-r t-ih1 s-t-ih-k l-iy','rb'],
'artistry':['aa1-r t-ih-s t-r-iy','nn'],
'artsy':['aa1-r-t s-iy1','jj'],
'artwork':['aa1-r-t w-er-k','nn'],
'arty':['aa1-r t-iy','jj'],
'as':['ae1-z','in nnp jj rb'],
'asbestos':['ae-s b-eh1 s-t-ah-s','nn'],
'asbestosis':['ae-s b-eh s-t-ow1 s-ah-s','nn'],
'ascend':['ah s-eh1-n-d','vb'],
'ascendancy':['ah s-eh1-n d-ah-n s-iy','nn'],
'ascended':['ah s-eh1-n d-ah-d','vbd'],
'ascendency':['ah s-eh1-n d-ah-n s-iy','nn'],
'ascending':['ah s-eh1-n d-ih-ng','vbg'],
'ascension':['ah s-eh1-n sh-ah-n','nn'],
'ascent':['ah s-eh1-n-t','nn'],
'ascertain':['ae s-er t-ey1-n','vb'],
'ascertained':['ae s-er t-ey1-n-d','vbn vbd'],
'ascetic':['ah s-eh1 t-ih-k','nn'],
'ascribe':['ah s-k-r-ay1-b','vbp vb'],
'ascribed':['ah s-k-r-ay1-b-d','vbn vbd'],
'ascribes':['ah s-k-r-ay1-b-z','vbz'],
'ash':['ae1-sh','nn'],
'ashamed':['ah sh-ey1-m-d','jj'],
'ashare':['ae1 sh-eh-r','nn'],
'ashen':['ae1 sh-ah-n','jj'],
'ashore':['ah sh-ao1-r','rb jj'],
'ashtrays':['ae1-sh t-r-ey-z','nns'],
'aside':['ah s-ay1-d','rb rp'],
'asinine':['ae1 s-ah n-ay-n','jj'],
'ask':['ae1-s-k','vb vbp'],
'askance':['ah s-k-ae1-n-s','rb'],
'asked':['ae1-s-k-t','vbd vbn jj'],
'askew':['ah s-k-y-uw1','rb'],
'askin':['ah s-k-ih1-n','vbg'],
'asking':['ae1 s-k-ih-ng','vbg jj nn'],
'asks':['ae1-s-k-s','vbz'],
'asleep':['ah s-l-iy1-p','rb jj'],
'asparagus':['ah s-p-eh1 r-ah g-ah-s','nn'],
'aspect':['ae1 s-p-eh-k-t','nn'],
'aspen':['ae1 s-p-ah-n','nn jj'],
'aspersions':['ah s-p-er1 zh-ah-n-z','nns'],
'asphalt':['ae1-s f-ao-l-t','nn jj'],
'aspirant':['ae1 s-p-er ah-n-t','nn'],
'aspiration':['ae s-p-er ey1 sh-ah-n','nn'],
'aspire':['ah s-p-ay1-r','vb vbp'],
'aspired':['ah s-p-ay1-r-d','vbd'],
'aspires':['ah s-p-ay1 er-z','vbz'],
'aspirin':['ae1 s-p-r-ih-n','nn'],
'aspiring':['ah s-p-ay1 r-ih-ng','jj vbg'],
'ass':['ae1-s','nn'],
'assail':['ah s-ey1-l','vb'],
'assailant':['ah s-ey1 l-ah-n-t','nn'],
'assailed':['ah s-ey1-l-d','vbn vbd'],
'assailing':['ah s-ey1 l-ih-ng','vbg'],
'assails':['ah s-ey1-l-z','vbz'],
'assassin':['ah s-ae1 s-ah-n','nn'],
'assassinate':['ah s-ae1 s-ah n-ey-t','vb'],
'assassinated':['ah s-ae1 s-ah n-ey t-ah-d','vbn vbd'],
'assassination':['ah s-ae s-ah n-ey1 sh-ah-n','nn'],
'assault':['ah s-ao1-l-t','nn vbp'],
'assaulted':['ah s-ao1-l t-ih-d','vbd vbn'],
'assaulting':['ah s-ao1-l t-ih-ng','vbg'],
'assay':['ae1 s-iy','nn'],
'assemblage':['ah s-eh1-m b-l-ah-jh','nn'],
'assemble':['ah s-eh1-m b-ah-l','vb vbp'],
'assembled':['ah s-eh1-m b-ah-l-d','vbn vbd jj'],
'assembles':['ah s-eh1-m b-ah-l-z','vbz'],
'assembling':['ah s-eh1-m b-ah-l ih-ng','vbg nn'],
'assembly':['ah s-eh1-m b-l-iy','nn'],
'assent':['ah s-eh1-n-t','vb nn'],
'assert':['ah s-er1-t','vb vbp'],
'asserted':['ah s-er1 t-ah-d','vbd vbn'],
'assertedly':['ah s-er1 t-ih-d l-iy','rb'],
'asserting':['ah s-er1 t-ih-ng','vbg'],
'assertion':['ah s-er1 sh-ah-n','nn'],
'assertive':['ah s-er1 t-ih-v','jj'],
'assertiveness':['ah s-er1 t-ih-v n-ah-s','nn'],
'asserts':['ah s-er1-t-s','vbz'],
'assess':['ah s-eh1-s','vb vbp'],
'assessed':['ah s-eh1-s-t','vbn vbd'],
'assesses':['ah s-eh1 s-ih-z','vbz'],
'assessing':['ah s-eh1 s-ih-ng','vbg'],
'assessment':['ah s-eh1-s m-ah-n-t','nn'],
'assessor':['ah s-eh1 s-er','nn'],
'asset':['ae1 s-eh-t','nn'],
'assiduously':['ah s-ih1 d-w-ah-s l-iy','rb'],
'assign':['ah s-ay1-n','vb vbp'],
'assigned':['ah s-ay1-n-d','vbn vbd jj'],
'assigning':['ah s-ay1 n-ih-ng','vbg'],
'assignment':['ah s-ay1-n m-ah-n-t','nn'],
'assigns':['ah s-ay1-n-z','vbz nns'],
'assimilate':['ah s-ih1 m-ah l-ey-t','vb'],
'assimilated':['ah s-ih1 m-ah l-ey t-ih-d','vbn jj'],
'assimilating':['ah s-ih1 m-ah l-ey t-ih-ng','vbg'],
'assimilation':['ah s-ih m-ah l-ey1 sh-ah-n','nn'],
'assist':['ah s-ih1-s-t','vb nn vbp'],
'assistance':['ah s-ih1 s-t-ah-n-s','nn'],
'assistant':['ah s-ih1 s-t-ah-n-t','nn jj'],
'assisted':['ah s-ih1 s-t-ah-d','vbn vbd'],
'assisting':['ah s-ih1 s-t-ih-ng','vbg'],
'assists':['ah s-ih1-s-t-s','vbz nns'],
'associate':['ah s-ow1 s-iy ah-t','jj vbp nn vb'],
'associated':['ah s-ow1 s-iy ey t-ah-d','vbn vbd jj'],
'associates':['ah s-ow1 s-iy ah-t-s','nns vbz'],
'associating':['ah s-ow1 s-iy ey t-ih-ng','vbg'],
'association':['ah s-ow s-iy ey1 sh-ah-n','nn'],
'assorted':['ah s-ao1-r t-ih-d','jj vbn'],
'assortment':['ah s-ao1-r-t m-ah-n-t','nn'],
'assuage':['ah s-w-ey1-jh','vb'],
'assuaged':['ah s-w-ey1-jh-d','vbn'],
'assume':['ah s-uw1-m','vb vbp'],
'assumed':['ah s-uw1-m-d','vbn vbd jj'],
'assumes':['ah s-uw1-m-z','vbz'],
'assuming':['ah s-uw1 m-ih-ng','vbg'],
'assumption':['ah s-ah1-m-p sh-ah-n','nn'],
'assurance':['ah sh-uh1 r-ah-n-s','nn'],
'assure':['ah sh-uh1-r','vb vbp'],
'assured':['ah sh-uh1-r-d','vbn vbd jj nn'],
'assuredly':['ah sh-uh1 r-ah-d l-iy','rb'],
'assures':['ah sh-uh1-r-z','vbz'],
'assuring':['ah sh-uh1 r-ih-ng','vbg'],
'asteroid':['ae1 s-t-er oy-d','jj nn'],
'asters':['ae1 s-t-er-z','nns'],
'asthma':['ae1-z m-ah','nn'],
'astigmatism':['ah s-t-ih1-g m-ah t-ih z-ah-m','nn'],
'astonished':['ah s-t-aa1 n-ih-sh-t','vbn vbd'],
'astonishing':['ah s-t-aa1 n-ih sh-ih-ng','jj'],
'astonishingly':['ah s-t-aa1 n-ih sh-ih-ng l-iy','rb'],
'astonishment':['ah s-t-aa1 n-ih-sh m-ah-n-t','nn'],
'astound':['ah s-t-aw1-n-d','vb'],
'astounded':['ah s-t-aw1-n d-ih-d','vbn jj'],
'astounding':['ah s-t-aw1-n d-ih-ng','jj'],
'astoundingly':['ah s-t-aw1-n d-ih-ng l-iy','rb'],
'astounds':['ah s-t-aw1-n-d-z','vbz'],
'astral':['ae1-s t-r-ah-l','jj'],
'astray':['ah s-t-r-ey1','rb'],
'astride':['ah s-t-r-ay1-d','in'],
'astringent':['ah s-t-r-ih1-n jh-ah-n-t','jj'],
'astrological':['ae-s t-r-ah l-aa1 jh-ih k-ah-l','jj'],
'astrology':['ah s-t-r-aa1 l-ah jh-iy','nn'],
'astronaut':['ae1 s-t-r-ah n-aa-t','nn'],
'astronomer':['ah s-t-r-aa1 n-ah m-er','nn'],
'astronomical':['ae s-t-r-ah n-aa1 m-ih k-ah-l','jj'],
'astronomically':['ae s-t-r-ah n-aa1 m-ih-k l-iy','rb'],
'astronomy':['ah s-t-r-aa1 n-ah m-iy','nn'],
'astrophysicist':['ae s-t-r-ow f-ih1 s-ih s-ih-s-t','nn'],
'astrophysics':['ae-s t-r-ow f-ih1 z-ih-k-s','nns'],
'astute':['ah s-t-uw1-t','jj'],
'astuteness':['ah s-t-uw1-t n-ah-s','nn'],
'asunder':['ah s-ah1-n d-er','rb'],
'asylum':['ah s-ay1 l-ah-m','nn'],
'asymmetry':['ey s-ih1 m-ah t-r-iy','nn'],
'at':['ae1-t','in vbd rb rp'],
'atavistic':['ae1 t-ah v-ih s-t-ih-k','jj'],
'ataxia':['ey1 t-ae1-k s-iy ah','nn'],
'ate':['ey1-t','vbd'],
'atheism':['ah th-ay1 s-ah-m','nn'],
'atheist':['ey1 th-iy ah-s-t','jj nn'],
'atheistic':['ey th-iy ih1 s-t-ih-k','jj'],
'athlete':['ae1-th l-iy-t','nn'],
'athletic':['ae-th l-eh1 t-ih-k','jj'],
'athletically':['ae-th l-eh1 t-ih-k l-iy','rb'],
'athletics':['ae-th l-eh1 t-ih-k-s','nns'],
'atmosphere':['ae1-t m-ah s-f-ih-r','nn'],
'atmospheric':['ae-t m-ah-s f-eh1 r-ih-k','jj'],
'atom':['ae1 t-ah-m','nn'],
'atomic':['ah t-aa1 m-ih-k','jj'],
'atonal':['ey t-ow1 n-ah-l','jj'],
'atone':['ah t-ow1-n','vb'],
'atonement':['ah t-ow1-n m-ah-n-t','nn'],
'atop':['ah t-aa1-p','in'],
'atrium':['ey1 t-r-iy ah-m','nn'],
'atrocious':['ah t-r-ow1 sh-ah-s','jj'],
'atrocity':['ah t-r-aa1 s-ah t-iy','nn'],
'atrophied':['ae1 t-r-ah f-iy-d','vbn'],
'atrophy':['ae1 t-r-ah f-iy','nn vbp'],
'attach':['ah t-ae1-ch','vb vbp'],
'attache':['ae t-ah sh-ey1','nn'],
'attached':['ah t-ae1-ch-t','vbn jj vbd'],
'attaches':['ah t-ae1 ch-ih-z','vbz'],
'attaching':['ah t-ae1 ch-ih-ng','vbg'],
'attachment':['ah t-ae1-ch m-ah-n-t','nn'],
'attack':['ah t-ae1-k','nn vbp vb'],
'attacked':['ah t-ae1-k-t','vbn vbd'],
'attacker':['ah t-ae1 k-er','nn'],
'attacking':['ah t-ae1 k-ih-ng','vbg nn'],
'attacks':['ah t-ae1-k-s','nns vbz'],
'attain':['ah t-ey1-n','vb vbp'],
'attainable':['ah t-ey1 n-ah b-ah-l','jj'],
'attained':['ah t-ey1-n-d','vbd vbn'],
'attaining':['ah t-ey1 n-ih-ng','vbg nn'],
'attainment':['ah t-ey1-n m-ah-n-t','nn'],
'attains':['ah t-ey1-n-z','vbz'],
'attempt':['ah t-eh1-m-p-t','nn vbp vb'],
'attempted':['ah t-eh1-m-p t-ah-d','vbd jj vbn'],
'attempting':['ah t-eh1-m-p t-ih-ng','vbg'],
'attempts':['ah t-eh1-m-p-t-s','nns vbz'],
'attend':['ah t-eh1-n-d','vb vbp'],
'attendance':['ah t-eh1-n d-ah-n-s','nn'],
'attendant':['ah t-eh1-n d-ah-n-t','nn jj'],
'attended':['ah t-eh1-n d-ah-d','vbd vbn'],
'attendee':['ah t-eh1-n d-iy1','nn'],
'attending':['ah t-eh1-n d-ih-ng','vbg'],
'attends':['ah t-eh1-n-d-z','vbz'],
'attention':['ah t-eh1-n sh-ah-n','nn'],
'attentive':['ah t-eh1-n t-ih-v','jj'],
'attentively':['ah t-eh1-n t-ih-v l-iy','rb'],
'attest':['ah t-eh1-s-t','vb vbp'],
'attested':['ah t-eh1 s-t-ih-d','vbn vbd'],
'attesting':['ah t-eh1 s-t-ih-ng','vbg'],
'attests':['ah t-eh1-s-t-s','vbz'],
'attic':['ae1 t-ih-k','nn'],
'attire':['ah t-ay1 er','nn'],
'attired':['ah t-ay1-r-d','jj vbn'],
'attitude':['ae1 t-ah t-uw-d','nn'],
'attorney':['ah t-er1 n-iy','nn'],
'attract':['ah t-r-ae1-k-t','vb vbp'],
'attracted':['ah t-r-ae1-k t-ah-d','vbn vbd'],
'attracting':['ah t-r-ae1-k t-ih-ng','vbg'],
'attraction':['ah t-r-ae1-k sh-ah-n','nn'],
'attractive':['ah t-r-ae1-k t-ih-v','jj'],
'attractively':['ah t-r-ae1-k t-ih-v l-iy','rb'],
'attractiveness':['ah t-r-ae1-k t-ih-v n-ah-s','nn'],
'attracts':['ah t-r-ae1-k-t-s','vbz'],
'attributable':['ah t-r-ih1 b-y-ah t-ah b-ah-l','jj'],
'attribute':['ae1 t-r-ah b-y-uw-t','vbp nn vb'],
'attributed':['ah t-r-ih1 b-y-ah t-ah-d','vbd vbn'],
'attributes':['ae1 t-r-ah b-y-uw-t-s','nns vbz'],
'attributing':['ah t-r-ih1 b-y-ah t-ih-ng','vbg'],
'attrition':['ah t-r-ih1 sh-ah-n','nn'],
'attuned':['ah t-uw1-n-d','vbn jj'],
'atypical':['ey t-ih1 p-ih k-ah-l','jj'],
'auburn':['aa1 b-er-n','jj'],
'auction':['aa1-k sh-ah-n','nn vb'],
'auctioned':['ao1-k sh-ah-n-d','vbn vbd'],
'auctioneer':['aa-k sh-ah n-ih1-r','nn'],
'auctioning':['ao1-k sh-ah-n ih-ng','nn'],
'auctions':['aa1-k sh-ah-n-z','nns vbz'],
'audacious':['aa d-ey1 sh-ah-s','jj'],
'audacity':['aa d-ae1 s-ah t-iy','nn'],
'audible':['aa1 d-ah b-ah-l','jj'],
'audibly':['aa1 d-ah b-l-iy','rb'],
'audience':['aa1 d-iy ah-n-s','nn'],
'audio':['aa1 d-iy ow','jj nn'],
'audiophiles':['ao d-iy1 ow f-iy-l-z','nns'],
'audiotex':['ao1 d-iy ow t-eh-k-s','nn'],
'audiovisual':['ao d-iy ow v-ih1 zh-uw ah-l','jj'],
'audit':['ao1 d-ih-t','nn vb'],
'audited':['ao1 d-ih t-ih-d','vbn jj vbd'],
'auditing':['aa1 d-ah t-ih-ng','nn vbg'],
'audition':['aa d-ih1 sh-ah-n','nn vb'],
'auditioning':['aa d-ih1 sh-ah-n ih-ng','vbg'],
'auditions':['ao d-ih1 sh-ah-n-z','nns'],
'auditor':['ao1 d-ih t-er','nn'],
'auditorium':['ao d-ah t-ao1 r-iy ah-m','nn'],
'audits':['aa1 d-ah-t-s','nn nns'],
'augment':['ao-g m-eh1-n-t','vb vbp'],
'augmented':['aa-g m-eh1-n t-ah-d','vbn jj vbd'],
'augmenting':['ao1-g m-eh-n t-ih-ng','vbg'],
'augur':['ao1 g-er','vbp'],
'augurs':['ao1 g-er-z','vbz'],
'august':['aa1 g-ah-s-t','jj'],
'aunt':['ae1-n-t','nn'],
'aura':['ao1 r-ah','nn'],
'aural':['ao1 r-ah-l','jj'],
'auspices':['ao1 s-p-ih s-ih-z','nns'],
'auspicious':['aa s-p-ih1 sh-ah-s','jj'],
'austere':['ao s-t-ih1-r','jj'],
'austerely':['ao s-t-ih1-r l-iy','rb'],
'austerity':['ao s-t-eh1 r-ih t-iy','nn'],
'authentic':['ah th-eh1-n t-ih-k','jj'],
'authentically':['ao th-eh1-n t-ih-k l-iy','rb'],
'authenticate':['ao th-eh1-n t-ah k-ey-t','vbp'],
'authentication':['ao th-eh-n t-ah k-ey1 sh-ah-n','nn'],
'authenticity':['ao th-ah-n t-ih1 s-ih t-iy','nn'],
'author':['ao1 th-er','nn'],
'authored':['ao1 th-er-d','vbn'],
'authoritarian':['ah th-ao r-ah t-eh1 r-iy ah-n','jj'],
'authoritarianism':['ah th-ao r-ah t-eh1 r-iy ah n-ih z-ah-m','nn'],
'authoritative':['ah th-ao1 r-ah t-ey t-ih-v','jj'],
'authority':['ah th-ao1 r-ah t-iy','nn'],
'authorization':['ao th-er ah z-ey1 sh-ah-n','nn'],
'authorize':['ao1 th-er ay-z','vb'],
'authorized':['ao1 th-er ay-z-d','vbn jj vbd'],
'authorizes':['ao1 th-er ay z-ih-z','vbz'],
'authorizing':['ao1 th-er ay z-ih-ng','vbg'],
'authorship':['ao1 th-er sh-ih-p','nn'],
'autism':['ao1 t-ih z-ah-m','nn'],
'autistic':['ao t-ih1 s-t-ih-k','jj'],
'auto':['ao1 t-ow','nn'],
'autobiographical':['ao t-ah b-ay ah g-r-ae1 f-ih k-ah-l','jj'],
'autobiography':['ao t-ah b-ay aa1 g-r-ah f-iy','nn'],
'autoclave':['ao1 t-ow k-l-ey-v','nn'],
'autocracy':['ao t-aa1 k-r-ah s-iy','nn'],
'autocrat':['ao1 t-ah k-r-ae-t','nn'],
'autocratic':['ao t-ah k-r-ae1 t-ih-k','jj'],
'autograph':['ao1 t-ah g-r-ae-f','nn vb'],
'autographed':['ao1 t-ah g-r-ae-f-t','vbn jj'],
'autographs':['ao1 t-ah g-r-ae-f-s','nns'],
'autoimmune':['ao t-oy1 m-y-uw-n','jj'],
'automaker':['ao1 t-ow m-ey k-er','nn'],
'automate':['ao1 t-ah m-ey-t','vb'],
'automated':['ao1 t-ah m-ey t-ih-d','vbn jj vbd'],
'automates':['ao1 t-ah m-ey-t-s','vbz'],
'automatic':['ao t-ah m-ae1 t-ih-k','jj nn'],
'automatically':['ao t-ah m-ae1 t-ih-k l-iy','rb'],
'automating':['ao1 t-ah m-ey t-ih-ng','vbg'],
'automation':['ao t-ah m-ey1 sh-ah-n','nn'],
'automaton':['ao t-aa1 m-ah t-aa-n','nn'],
'automobile':['ao1 t-ah m-ow b-iy-l','nn'],
'automotive':['ao t-ah m-ow1 t-ih-v','jj'],
'autonomic':['ao t-ah n-aa1 m-ih-k','jj'],
'autonomous':['ao t-aa1 n-ah m-ah-s','jj'],
'autonomously':['ao t-aa1 n-ow m-ah-s l-iy','rb'],
'autonomy':['ao t-aa1 n-ah m-iy','nn'],
'autopsies':['ao1 t-aa-p s-iy-z','nns'],
'autopsy':['ao1 t-aa-p s-iy','nn vb'],
'autumn':['ao1 t-ah-m','nn'],
'autumnal':['ao t-ah1-m n-ah-l','jj'],
'auxiliary':['aa-g z-ih1 l-y-er iy','jj nn'],
'avail':['ah v-ey1-l','nn vb'],
'availability':['ah v-ey l-ah b-ih1 l-ah t-iy','nn'],
'available':['ah v-ey1 l-ah b-ah-l','jj'],
'availed':['ah v-ey1-l-d','vbd'],
'avalanche':['ae1 v-ah l-ae-n-ch','nn'],
'avarice':['ae1 v-er ah-s','nn'],
'avenge':['ah v-eh1-n-jh','vb'],
'avenger':['ah v-eh1-n jh-er','nn'],
'avenging':['ah v-eh1-n jh-ih-ng','jj vbg'],
'avenue':['ae1 v-ah n-uw','nn'],
'average':['ae1 v-er ih-jh','jj nn vb vbp'],
'averaged':['ae1 v-r-ah-jh-d','vbd vbn'],
'averages':['ae1 v-r-ih jh-ih-z','nns vbz'],
'averaging':['ae1 v-r-ih jh-ih-ng','vbg jj nn'],
'averse':['ah v-er1-s','jj'],
'aversion':['ah v-er1 zh-ah-n','nn'],
'avert':['ah v-er1-t','vb'],
'averted':['ah v-er1 t-ah-d','vbn vbd'],
'averting':['ah v-er1 t-ih-ng','vbg'],
'averts':['ah v-er1-t-s','vbz'],
'aviary':['ey1 v-iy eh r-iy','nn'],
'aviation':['ey v-iy ey1 sh-ah-n','nn'],
'aviator':['ey1 v-iy ey t-er','nn'],
'avid':['ae1 v-ah-d','jj'],
'avidly':['ae1 v-ah-d l-iy','rb'],
'avionics':['ey v-iy aa1 n-ih-k-s','nns nn'],
'avocado':['ae v-ah k-aa1 d-ow','nn'],
'avocation':['ae v-ah k-ey1 sh-ah-n','nn'],
'avoid':['ah v-oy1-d','vb jj vbp'],
'avoidable':['ah v-oy1 d-ah b-ah-l','jj'],
'avoidance':['ah v-oy1 d-ah-n-s','nn'],
'avoided':['ah v-oy1 d-ah-d','vbn jj vbd'],
'avoiding':['ah v-oy1 d-ih-ng','vbg'],
'avoids':['ah v-oy1-d-z','vbz'],
'avowed':['ah v-aw1-d','jj'],
'avowedly':['ah v-aw1 ah-d l-iy','rb'],
'avuncular':['ah v-ah1-ng k-y-ah l-er','jj'],
'await':['ah w-ey1-t','vb vbp'],
'awaited':['ah w-ey1 t-ah-d','vbd jj vbn'],
'awaiting':['ah w-ey1 t-ih-ng','vbg'],
'awaits':['ah w-ey1-t-s','vbz'],
'awake':['ah w-ey1-k','jj rb vb'],
'awaken':['ah w-ey1 k-ah-n','vb vbp'],
'awakened':['ah w-ey1 k-ah-n-d','vbn vbd'],
'awakening':['ah w-ey1 k-ah n-ih-ng','vbg nn'],
'awakens':['ah w-ey1 k-ah-n-z','vbz'],
'award':['ah w-ao1-r-d','nn vbp vb'],
'awarded':['ah w-ao1-r d-ah-d','vbn vbd'],
'awarding':['ah w-ao1-r d-ih-ng','vbg nn'],
'awards':['ah w-ao1-r-d-z','nns vbz'],
'aware':['ah w-eh1-r','jj'],
'awareness':['ah w-eh1-r n-ah-s','nn'],
'awash':['ah w-aa1-sh','jj rb'],
'away':['ah w-ey1','rb rp'],
'awe':['aa1','nn'],
'awed':['ao1-d','vbn jj'],
'awesome':['aa1 s-ah-m','jj'],
'awful':['aa1 f-ah-l','jj rb'],
'awfully':['aa1-f l-iy','rb'],
'awfulness':['ao1 f-ah-l n-ah-s','nn'],
'awhile':['ah w-ay1-l','rb'],
'awkward':['aa1 k-w-er-d','jj'],
'awkwardly':['ao1 k-w-er-d l-iy','rb'],
'awkwardness':['ao1 k-w-er-d n-ah-s','nn'],
'awoke':['ah w-ow1-k','vbd'],
'awry':['er ay1','rb jj'],
'axe':['ae1-k-s','nn'],
'axiom':['ae1-k s-iy ah-m','nn'],
'axiomatic':['ae-k s-iy ah m-ae1 t-ih-k','jj'],
'axis':['ae1-k s-ah-s','nn'],
'axle':['ae1-k s-ah-l','nn'],
'ayatollah':['ay ah t-ow1 l-ah','nn'],
'azalea':['ah z-ey1 l-y-ah','nn'],
'azure':['ae1 zh-er','jj'],
'b':['b-iy1','nn ls'],
'babble':['b-ae1 b-ah-l','nn'],
'babbled':['b-ae1 b-ah-l-d','vbd'],
'babel':['b-ae1 b-ah-l','nn'],
'babes':['b-ey1-b-z','nns'],
'baboon':['b-ah b-uw1-n','nn'],
'baby':['b-ey1 b-iy','nn uh'],
'babyhood':['b-ey1 b-iy hh-uh-d','nn'],
'baccalaureate':['b-ae k-ah l-ao1 r-iy ah-t','nn'],
'bachelor':['b-ae1 ch-ah l-er','nn'],
'back':['b-ae1-k','rb in jj nn rp vb vbp'],
'backbone':['b-ae1-k b-ow-n','nn'],
'backdated':['b-ae1-k d-ey t-ih-d','vbd'],
'backdoor':['b-ae1-k d-ao-r','jj'],
'backdrop':['b-ae1-k d-r-aa-p','nn'],
'backed':['b-ae1-k-t','vbn jj vbd'],
'backer':['b-ae1 k-er','nn'],
'backfield':['b-ae1-k f-iy-l-d','nn'],
'backfire':['b-ae1-k f-ay-r','vb vbp'],
'backfired':['b-ae1-k f-ay er-d','vbd vbn'],
'backfires':['b-ae1-k f-ay er-z','vbz'],
'backfiring':['b-ae1-k f-ay er ih-ng','vbg'],
'background':['b-ae1-k g-r-aw-n-d','nn jj'],
'backhand':['b-ae1-k hh-ae-n-d','nn'],
'backhanded':['b-ae1-k hh-ae-n d-ah-d','jj'],
'backhoe':['b-ae1-k hh-ow','nn'],
'backing':['b-ae1 k-ih-ng','vbg nn'],
'backlash':['b-ae1-k l-ae-sh','nn'],
'backlog':['b-ae1-k l-aa-g','nn'],
'backpack':['b-ae1-k p-ae-k','nn'],
'backpedaling':['b-ae1-k p-eh d-ah-l ih-ng','vbg'],
'backroom':['b-ae1-k r-uw-m','nn'],
'backs':['b-ae1-k-s','nns vbz'],
'backside':['b-ae1-k s-ay-d','nn'],
'backslapping':['b-ae1-k s-l-ae p-ih-ng','vbg'],
'backstage':['b-ae1-k s-t-ey1-jh','rb'],
'backstitch':['b-ae1-k s-t-ih-ch','nn vb'],
'backstop':['b-ae1-k s-t-aa-p','nn vb'],
'backtracking':['b-ae1-k t-r-ae k-ih-ng','nn vbg'],
'backup':['b-ae1-k ah-p','nn jj'],
'backward':['b-ae1-k w-er-d','rb jj'],
'backwardness':['b-ae1 k-w-er-d n-ah-s','nn'],
'backwards':['b-ae1-k w-er-d-z','rb nns jj'],
'backwater':['b-ae1-k w-ao t-er','nn jj'],
'backwoods':['b-ae1-k w-uh1-d-z','nns jj'],
'backyard':['b-ae1-k y-aa-r-d','nn'],
'bacon':['b-ey1 k-ah-n','nn'],
'bacterial':['b-ae-k t-ih1 r-iy ah-l','jj'],
'bacterium':['b-ae-k t-ih1 r-iy ah-m','nn'],
'bad':['b-ae1-d','jj nn rb'],
'bade':['b-ey1-d','vbd'],
'badge':['b-ae1-jh','nn'],
'badgered':['b-ae1 jh-er-d','vbd'],
'badgering':['b-ae1 jh-er ih-ng','vbg'],
'badly':['b-ae1-d l-iy','rb'],
'badminton':['b-ae1-d m-ih-n t-ah-n','nn'],
'badness':['b-ae1-d n-ah-s','nn'],
'baffle':['b-ae1 f-ah-l','vb'],
'baffled':['b-ae1 f-ah-l-d','vbn jj vbd'],
'baffling':['b-ae1 f-ah-l ih-ng','jj'],
'bag':['b-ae1-g','nn vb'],
'bagel':['b-ey1 g-ah-l','nn'],
'baggage':['b-ae1 g-ah-jh','nn'],
'bagged':['b-ae1-g-d','vbd jj'],
'baggy':['b-ae1 g-iy','jj'],
'bagpipe':['b-ae1-g p-ay-p','nn'],
'bags':['b-ae1-g-z','nns vbz'],
'bail':['b-ey1-l','vb nn'],
'bailed':['b-ey1-l-d','vbd vbn'],
'bailiff':['b-ey1 l-ah-f','nn'],
'bailing':['b-ey1 l-ih-ng','vbg nn'],
'bailout':['b-ey1-l aw-t','nn'],
'bait':['b-ey1-t','nn vb'],
'baited':['b-ey1 t-ih-d','vbn'],
'bake':['b-ey1-k','jj vb'],
'baked':['b-ey1-k-t','jj vbd vbn'],
'baker':['b-ey1 k-er','nn'],
'bakery':['b-ey1 k-er iy','nn'],
'bakes':['b-ey1-k-s','vbz'],
'bakeware':['b-ey1-k w-eh-r','nn'],
'baking':['b-ey1 k-ih-ng','nn vbg'],
'balance':['b-ae1 l-ah-n-s','nn vbp jj rb vb'],
'balanced':['b-ae1 l-ah-n-s-t','jj vbd vbn'],
'balances':['b-ae1 l-ah-n s-ah-z','nns vbz'],
'balancing':['b-ae1 l-ah-n s-ih-ng','nn vbg'],
'balcony':['b-ae1-l k-ah n-iy','nn'],
'bald':['b-ao1-l-d','jj'],
'balding':['b-ao1-l d-ih-ng','jj vbg'],
'baldness':['b-ao1-l-d n-ah-s','nn'],
'bale':['b-ey1-l','nn vbp'],
'baleful':['b-ey1-l f-ah-l','jj'],
'balk':['b-ao1-k','vb vbp'],
'balkanized':['b-ao1-l k-ah n-ay-z-d','jj'],
'balked':['b-ao1-k-t','vbd vbn'],
'balking':['b-ao1 k-ih-ng','vbg'],
'balks':['b-ao1-k-s','vbz'],
'ball':['b-ao1-l','nn'],
'ballad':['b-ae1 l-ah-d','nn'],
'ballast':['b-ae1 l-ah-s-t','nn'],
'balled':['b-ao1-l-d','vbn'],
'ballerina':['b-ae l-er iy1 n-ah','nn'],
'ballet':['b-ae l-ey1','nn'],
'balling':['b-ao1 l-ih-ng','vbg'],
'ballistic':['b-ah l-ih1 s-t-ih-k','jj'],
'ballistics':['b-ah l-ih1 s-t-ih-k-s','nns'],
'balloon':['b-ah l-uw1-n','nn vbp vb'],
'ballooned':['b-ah l-uw1-n-d','vbn vbd'],
'ballooning':['b-ah l-uw1 n-ih-ng','nn vbg'],
'balloons':['b-ah l-uw1-n-z','nns vbz'],
'ballot':['b-ae1 l-ah-t','nn'],
'balloting':['b-ae1 l-ah t-ih-ng','nn'],
'ballpark':['b-ao1-l p-aa-r-k','nn'],
'ballplayer':['b-ao1-l p-l-ey er','nn'],
'ballroom':['b-ao1-l r-uw-m','nn'],
'ballyhoo':['b-ae1 l-iy hh-uw1','nn'],
'ballyhooed':['b-ae1 l-iy hh-uw1-d','vbn'],
'balm':['b-aa1-m','nn'],
'balmy':['b-aa1 m-iy','jj'],
'baloney':['b-ah l-ow1 n-iy','nn'],
'bam':['b-ae1-m','uh'],
'bambino':['b-ae-m b-iy1 n-ow','nn'],
'bamboo':['b-ae-m b-uw1','nn'],
'ban':['b-ae1-n','nn vb vbp'],
'banal':['b-ah n-aa1-l','jj'],
'banalities':['b-ah n-ae1 l-ih t-iy-z','nns'],
'banana':['b-ah n-ae1 n-ah','nn'],
'band':['b-ae1-n-d','nn vb'],
'bandage':['b-ae1-n d-ih-jh','nn'],
'bandaged':['b-ae1-n d-ah-jh-d','vbn jj'],
'banded':['b-ae1-n d-ih-d','vbn'],
'bandied':['b-ae1-n d-iy-d','vbn'],
'banding':['b-ae1-n d-ih-ng','vbg'],
'bandit':['b-ae1-n d-ah-t','nn'],
'bands':['b-ae1-n-d-z','nns'],
'bandstand':['b-ae1-n-d s-t-ae-n-d','nn'],
'bandwagon':['b-ae1-n-d w-ae g-ah-n','nn'],
'bandwidth':['b-ae1-n-d w-ih-d-th','nn'],
'bane':['b-ey1-n','nn'],
'bang':['b-ae1-ng','nn vb uh vbp'],
'banged':['b-ae1-ng-d','vbd vbn'],
'banging':['b-ae1 ng-ih-ng','vbg nn'],
'bangish':['b-ae1 ng-ih-sh','jj'],
'bangs':['b-ae1-ng-z','nns vbz'],
'banish':['b-ae1 n-ih-sh','vb'],
'banished':['b-ae1 n-ih-sh-t','vbn vbd'],
'banishing':['b-ae1 n-ih sh-ih-ng','vbg'],
'banishment':['b-ae1 n-ih-sh m-ah-n-t','nn'],
'banister':['b-ae1 n-ih s-t-er','nn'],
'banjo':['b-ae1-n jh-ow','nn'],
'bank':['b-ae1-ng-k','nn vbp vb'],
'banked':['b-ae1-ng-k-t','vbd jj vbn'],
'banker':['b-ae1-ng k-er','nn'],
'bankholding':['b-ae1-ng-k hh-ow-l d-ih-ng','vbg'],
'banking':['b-ae1-ng k-ih-ng','nn vbg jj'],
'bankroll':['b-ae1-ng-k r-ow-l','vb vbp'],
'bankrolled':['b-ae1-ng-k r-ow-l-d','vbd vbn'],
'bankrolling':['b-ae1-ng-k r-ow l-ih-ng','vbg'],
'bankrupt':['b-ae1-ng k-r-ah-p-t','jj nn vb vbp'],
'bankruptcy':['b-ae1-ng k-r-ah-p s-iy','nn'],
'banks':['b-ae1-ng-k-s','nns vbz'],
'banned':['b-ae1-n-d','vbn vbd'],
'banner':['b-ae1 n-er','nn'],
'banning':['b-ae1 n-ih-ng','vbg nn'],
'banquet':['b-ae1-ng k-w-ah-t','nn'],
'bans':['b-ae1-n-z','nns vbz'],
'banshee':['b-ae-n sh-iy1','nn'],
'banter':['b-ae1-n t-er','nn'],
'baptism':['b-ae1-p t-ih z-ah-m','nn'],
'baptismal':['b-ae-p t-ih1-z m-ah-l','jj'],
'baptistery':['b-ae1-p t-ah-s t-r-iy','nn'],
'baptized':['b-ae-p t-ay1-z-d','vbn'],
'bar':['b-aa1-r','nn vb vbp'],
'barb':['b-aa1-r-b','nn'],
'barbarian':['b-aa-r b-eh1 r-iy ah-n','nn'],
'barbaric':['b-aa-r b-ae1 r-ih-k','jj'],
'barbarous':['b-aa1-r b-er ah-s','jj'],
'barbecue':['b-aa1-r b-ih k-y-uw','nn vb'],
'barbecued':['b-aa1-r b-ih k-y-uw-d','vbn'],
'barbecues':['b-aa1-r b-ih k-y-uw-z','nns'],
'barbed':['b-aa1-r-b-d','jj vbn'],
'barbequed':['b-aa1-r b-ih k-y-uw-d','jj'],
'barber':['b-aa1-r b-er','nn'],
'bard':['b-aa1-r-d','nn'],
'bare':['b-eh1-r','jj vb'],
'bared':['b-eh1-r-d','vbd'],
'barefoot':['b-eh1-r f-uh-t','rb jj'],
'barely':['b-eh1-r l-iy','rb'],
'barest':['b-eh1 r-ah-s-t','jjs'],
'bargain':['b-aa1-r g-ah-n','nn vb'],
'bargained':['b-aa1-r g-ah-n-d','vbd vbn'],
'bargaining':['b-aa1-r g-ih n-ih-ng','nn jj vbg'],
'bargains':['b-aa1-r g-ah-n-z','nns vbz'],
'barge':['b-aa1-r-jh','nn vb vbp'],
'barged':['b-aa1-r-jh-d','vbn vbd'],
'barges':['b-aa1-r jh-ah-z','nns vbz'],
'baring':['b-eh1 r-ih-ng','vbg'],
'baritone':['b-eh1 r-ah t-ow-n','nn jj'],
'barium':['b-eh1 r-iy ah-m','nn'],
'bark':['b-aa1-r-k','nn vb'],
'barked':['b-aa1-r-k-t','vbd'],
'barking':['b-aa1-r k-ih-ng','vbg nn'],
'barks':['b-aa1-r-k-s','vbz'],
'barley':['b-aa1-r l-iy','nn'],
'barn':['b-aa1-r-n','nn'],
'barnacles':['b-aa1-r n-ah k-ah-l-z','nns'],
'barnyard':['b-aa1-r-n y-aa-r-d','nn'],
'barometer':['b-er aa1 m-ih t-er','nn'],
'barometric':['b-ae r-ah m-eh1 t-r-ih-k','jj'],
'baron':['b-ae1 r-ah-n','nn'],
'baroque':['b-er ow1-k','jj nn'],
'barrack':['b-ae1 r-ah-k','nn'],
'barracks':['b-ae1 r-ah-k-s','nn nns'],
'barrage':['b-er aa1-zh','nn'],
'barred':['b-aa1-r-d','vbn vbd'],
'barrel':['b-ae1 r-ah-l','nn vb'],
'barreling':['b-ae1 r-ah-l ih-ng','vbg'],
'barrels':['b-ae1 r-ah-l-z','nns'],
'barren':['b-ae1 r-ah-n','jj'],
'barricade':['b-ae1 r-ah k-ey-d','nn vb'],
'barricades':['b-ae1 r-ah k-ey-d-z','nns'],
'barrier':['b-ae1 r-iy er','nn'],
'barring':['b-aa1 r-ih-ng','vbg'],
'barrio':['b-aa1 r-iy ow','nn'],
'barrister':['b-ae1 r-ih-s t-er','nn'],
'barroom':['b-aa1-r r-uw-m','nn'],
'barrow':['b-ae1 r-ow','nn'],
'bars':['b-aa1-r-z','nns vbz'],
'bartender':['b-aa1-r t-eh-n d-er','nn'],
'barter':['b-aa1-r t-er','nn'],
'bartered':['b-aa1-r t-er-d','vbn'],
'bartering':['b-aa1-r t-er ih-ng','nn'],
'base':['b-ey1-s','nn vbp jj vb'],
'baseball':['b-ey1-s b-ao1-l','nn'],
'based':['b-ey1-s-t','vbn vbd'],
'baseless':['b-ey1-s l-ah-s','jj'],
'baseline':['b-ey1-s l-ay-n','nn jj'],
'baseman':['b-ey1-s m-ah-n','nn'],
'basement':['b-ey1-s m-ah-n-t','nn'],
'bases':['b-ey1 s-ah-z','nns vbz'],
'bash':['b-ae1-sh','nn vb'],
'bashed':['b-ae1-sh-t','vbd vbn'],
'basher':['b-ae1 sh-er','nn'],
'bashes':['b-ae1 sh-ih-z','nns'],
'bashful':['b-ae1-sh f-ah-l','jj'],
'bashing':['b-ae1 sh-ih-ng','jj vbg nn'],
'basic':['b-ey1 s-ih-k','jj nn'],
'basically':['b-ey1 s-ih-k l-iy','rb'],
'basil':['b-ae1 z-ah-l','nn'],
'basin':['b-ey1 s-ah-n','nn'],
'basing':['b-ey1 s-ih-ng','vbg'],
'basis':['b-ey1 s-ah-s','nn'],
'basked':['b-ae1-s-k-t','vbd'],
'basket':['b-ae1 s-k-ah-t','nn'],
'basketball':['b-ae1 s-k-ah-t b-ao-l','nn'],
'basking':['b-ae1 s-k-ih-ng','vbg'],
'bass':['b-ae1-s','nn'],
'bassist':['b-ey1 s-ih-s-t','nn'],
'basso':['b-ae1 s-ow','nn'],
'bassoon':['b-ah s-uw1-n','nn'],
'bastard':['b-ae1 s-t-er-d','nn'],
'bastion':['b-ae1 s-ch-ah-n','nn'],
'bat':['b-ae1-t','nn vb'],
'batch':['b-ae1-ch','nn'],
'bated':['b-ey1 t-ih-d','jj'],
'bath':['b-ae1-th','nn'],
'bathe':['b-ey1-dh','vb'],
'bathed':['b-ey1-dh-d','vbn vbd'],
'bathers':['b-ey1 dh-er-z','nns'],
'bathing':['b-ey1 dh-ih-ng','nn vbg'],
'bathrobe':['b-ae1-th r-ow-b','nn'],
'bathroom':['b-ae1-th r-uw-m','nn'],
'baths':['b-ae1-th-s','nns vbz'],
'bathtub':['b-ae1-th t-ah-b','nn'],
'baton':['b-ah t-aa1-n','nn'],
'bats':['b-ae1-t-s','nns vbz'],
'battalion':['b-ah t-ae1 l-y-ah-n','nn'],
'batted':['b-ae1 t-ih-d','vbd vbn'],
'batten':['b-ae1 t-ah-n','nn'],
'batter':['b-ae1 t-er','nn vb'],
'battered':['b-ae1 t-er-d','vbn vbd jj'],
'battering':['b-ae1 t-er ih-ng','vbg nn'],
'batters':['b-ae1 t-er-z','nns vbz'],
'battery':['b-ae1 t-er iy','nn'],
'batting':['b-ae1 t-ih-ng','vbg nn'],
'battle':['b-ae1 t-ah-l','nn vbp vb'],
'battled':['b-ae1 t-ah-l-d','vbd vbn'],
'battlefield':['b-ae1 t-ah-l f-iy-l-d','nn'],
'battlefront':['b-ae1 t-ah-l f-r-ah-n-t','nn'],
'battleground':['b-ae1 t-ah-l g-r-aw-n-d','nn'],
'battlements':['b-ae1 t-ah-l m-ah-n-t-s','nns'],
'battles':['b-ae1 t-ah-l-z','nns vbz'],
'battling':['b-ae1 t-ah-l ih-ng','vbg nn'],
'baubles':['b-ao1 b-ah-l-z','nns'],
'bawdy':['b-ao1 d-iy','jj'],
'bawled':['b-ao1-l-d','vbd'],
'bay':['b-ey1','nn'],
'bayed':['b-ey1-d','vbd vbn'],
'baying':['b-ey1 ih-ng','nn vbg'],
'bayly':['b-ey1 l-iy','nn'],
'bayonet':['b-ey1 ah n-eh-t','nn'],
'bazaar':['b-ah z-aa1-r','nn'],
'be':['b-iy1','vb vbn vbp'],
'beach':['b-iy1-ch','nn'],
'beached':['b-iy1-ch-t','jj'],
'beachfront':['b-iy1-ch f-r-aa-n-t','nn jj'],
'beachhead':['b-iy1-ch hh-eh-d','nn'],
'beaching':['b-iy1 ch-ih-ng','vbg'],
'beacon':['b-iy1 k-ah-n','nn'],
'beaded':['b-iy1 d-ih-d','vbn'],
'beadle':['b-iy1 d-ah-l','nn'],
'beads':['b-iy1-d-z','nns'],
'beadwork':['b-iy1-d w-er-k','nn'],
'beady':['b-iy1 d-iy','jj'],
'beak':['b-iy1-k','nn'],
'beaker':['b-iy1 k-er','nn'],
'beam':['b-iy1-m','nn vb'],
'beamed':['b-iy1-m-d','vbn'],
'beaming':['b-iy1 m-ih-ng','vbg jj'],
'beams':['b-iy1-m-z','nns vbz'],
'bean':['b-iy1-n','nn vbn jj'],
'bear':['b-eh1-r','vb nn vbp'],
'bearable':['b-eh1 r-ah b-ah-l','jj'],
'beard':['b-ih1-r-d','nn'],
'bearded':['b-ih1-r d-ah-d','jj vbd vbn'],
'bearer':['b-eh1 r-er','nn'],
'bearing':['b-eh1 r-ih-ng','vbg jj nn'],
'bearish':['b-eh1 r-ih-sh','jj'],
'bearishly':['b-eh1 r-ih-sh l-iy','rb'],
'bearishness':['b-eh1 r-ih-sh n-ah-s','nn'],
'bears':['b-eh1-r-z','vbz nns'],
'beast':['b-iy1-s-t','nn'],
'beat':['b-iy1-t','vb jj nn vbd vbn vbp'],
'beaten':['b-iy1 t-ah-n','vbn jj'],
'beating':['b-iy1 t-ih-ng','vbg nn'],
'beatniks':['b-iy1-t n-ih-k-s','nns'],
'beats':['b-iy1-t-s','vbz nns'],
'beau':['b-ow1','nn'],
'beautiful':['b-y-uw1 t-ah f-ah-l','jj'],
'beautifully':['b-y-uw1 t-ah f-l-iy','rb'],
'beautify':['b-y-uw1 t-ih f-ay','vbp'],
'beauty':['b-y-uw1 t-iy','nn'],
'beaver':['b-iy1 v-er','nn'],
'bebop':['b-iy1 b-aa-p','nn'],
'becalmed':['b-ih k-aa1-m-d','jj'],
'became':['b-ih k-ey1-m','vbd'],
'because':['b-ih k-ao1-z','in rb'],
'beckon':['b-eh1 k-ah-n','vbp vb'],
'beckoned':['b-eh1 k-ah-n-d','vbd vbn'],
'beckoning':['b-eh1 k-ah n-ih-ng','vbg nn'],
'beckons':['b-eh1 k-ah-n-z','vbz'],
'become':['b-ih k-ah1-m','vb vbd vbn vbp'],
'becomes':['b-ih k-ah1-m-z','vbz'],
'becoming':['b-ih k-ah1 m-ih-ng','vbg nn'],
'bed':['b-eh1-d','nn vb vbp'],
'bedding':['b-eh1 d-ih-ng','nn'],
'bedevil':['b-ih d-eh1 v-ah-l','vb'],
'bedeviled':['b-ih d-eh1 v-ah-l-d','vbn'],
'bedfellows':['b-eh1-d f-eh l-ow-z','nns'],
'bedlam':['b-eh1-d l-ah-m','nn'],
'bedpans':['b-eh1-d p-ae-n-z','nns'],
'bedraggled':['b-ih d-r-ae1 g-ah-l-d','jj'],
'bedridden':['b-eh1-d r-ih d-ah-n','jj'],
'bedrock':['b-eh1-d r-aa-k','nn'],
'bedroom':['b-eh1-d r-uw-m','nn'],
'beds':['b-eh1-d-z','nns'],
'bedside':['b-eh1-d s-ay-d','nn'],
'bedspread':['b-eh1-d s-p-r-eh-d','nn'],
'bedstraw':['b-eh1-d s-t-r-ao','nn'],
'bedtime':['b-eh1-d t-ay-m','nn'],
'bee':['b-iy1','nn'],
'beech':['b-iy1-ch','nn'],
'beef':['b-iy1-f','nn vb vbp'],
'beefed':['b-iy1-f-t','vbn vbd'],
'beefing':['b-iy1 f-ih-ng','vbg'],
'beefs':['b-iy1-f-s','vbz'],
'beefsteak':['b-iy1-f s-t-ey-k','nn'],
'beefy':['b-iy1 f-iy','jj'],
'beehive':['b-iy1 hh-ay-v','nn'],
'been':['b-ih1-n','vbn vbp'],
'beep':['b-iy1-p','nn'],
'beeper':['b-iy1 p-er','nn jjr'],
'beer':['b-ih1-r','nn'],
'beet':['b-iy1-t','nn'],
'beetle':['b-iy1 t-ah-l','nn'],
'befall':['b-ih f-ao1-l','vb vbp'],
'befallen':['b-ah f-aa1 l-ah-n','vbn'],
'befell':['b-ih f-eh1-l','vbd'],
'befits':['b-ih f-ih1-t-s','vbz'],
'befitting':['b-ih f-ih1 t-ih-ng','vbg jj'],
'before':['b-ih f-ao1-r','in rb rp'],
'beforehand':['b-ih f-ao1-r hh-ae-n-d','rb'],
'befriended':['b-ih f-r-eh1-n d-ih-d','vbd vbn'],
'befriends':['b-ih f-r-eh1-n-d-z','vbz'],
'befuddled':['b-ih f-ah1 d-ah-l-d','vbd vbn'],
'befuddles':['b-ih f-ah1 d-ah-l-z','vbz'],
'beg':['b-eh1-g','vbp vb'],
'began':['b-ih g-ae1-n','vbd'],
'begat':['b-ih g-ae1-t','vbd'],
'beget':['b-ih g-eh1-t','vb vbp'],
'begets':['b-ih g-eh1-t-s','vbz'],
'beggar':['b-eh1 g-er','nn'],
'begged':['b-eh1-g-d','vbd vbn'],
'begging':['b-eh1 g-ih-ng','vbg nn'],
'begin':['b-ih g-ih1-n','vb vbp'],
'beginner':['b-ih g-ih1 n-er','nn'],
'beginning':['b-ih g-ih1 n-ih-ng','vbg jj nn'],
'begins':['b-ih g-ih1-n-z','vbz'],
'begonia':['b-ih g-ow1 n-y-ah','nn'],
'begrudge':['b-ih g-r-ah1-jh','vb'],
'begs':['b-eh1-g-z','vbz'],
'beguiled':['b-ih g-ay1-l-d','vbn'],
'beguiling':['b-ih g-ay1 l-ih-ng','jj'],
'begun':['b-ih g-ah1-n','vbn'],
'behalf':['b-ih hh-ae1-f','nn'],
'behave':['b-ih hh-ey1-v','vb vbp'],
'behaved':['b-ih hh-ey1-v-d','vbd vbn'],
'behaves':['b-ih hh-ey1-v-z','vbz'],
'behaving':['b-ih hh-ey1 v-ih-ng','vbg'],
'behavior':['b-ih hh-ey1 v-y-er','nn'],
'behavioral':['b-ih hh-ey1 v-y-er ah-l','jj'],
'beheaded':['b-ih hh-eh1 d-ih-d','vbd'],
'beheading':['b-ih hh-eh1 d-ih-ng','nn vbg'],
'behemoth':['b-ah hh-iy1 m-ah-th','nn'],
'behest':['b-ih hh-eh1-s-t','nn'],
'behind':['b-ih hh-ay1-n-d','in nn rb rp'],
'behold':['b-ih hh-ow1-l-d','vb'],
'beholden':['b-ih hh-ow1-l d-ah-n','jj'],
'beholder':['b-iy hh-ow1-l d-er','nn'],
'beige':['b-ey1-zh','jj'],
'being':['b-iy1 ih-ng','vbg jj nn'],
'belaboring':['b-ih l-ey1 b-er ih-ng','vbg nn'],
'belated':['b-ih l-ey1 t-ih-d','jj'],
'belatedly':['b-ih l-ey1 t-ah-d l-iy','rb'],
'belch':['b-eh1-l-ch','nn'],
'belched':['b-eh1-l-ch-t','vbd'],
'belching':['b-eh1-l ch-ih-ng','nn vbg'],
'beleaguered':['b-ih l-iy1 g-er-d','jj vbn'],
'belfry':['b-eh1-l f-r-iy','nn'],
'belie':['b-ih l-ay1','vbp vb'],
'belied':['b-ih l-ay1-d','vbd'],
'belief':['b-ih l-iy1-f','nn'],
'belies':['b-ih l-ay1-z','vbz'],
'believability':['b-ah l-iy v-ah b-ih1 l-ih t-iy','nn'],
'believable':['b-ah l-iy1 v-ah b-ah-l','jj'],
'believe':['b-ih l-iy1-v','vbp vb'],
'believed':['b-ih l-iy1-v-d','vbd vbn'],
'believer':['b-ah l-iy1 v-er','nn'],
'believes':['b-ih l-iy1-v-z','vbz'],
'believing':['b-ih l-iy1 v-ih-ng','vbg'],
'belittle':['b-ih l-ih1 t-ah-l','vbp vb'],
'belittled':['b-ih l-ih1 t-ah-l-d','jj'],
'belittling':['b-ih l-ih1 t-ah-l ih-ng','vbg'],
'bell':['b-eh1-l','nn'],
'bellhops':['b-eh1-l hh-aa-p-s','nns'],
'belligerence':['b-ah l-ih1 jh-er ah-n-s','nn'],
'belligerent':['b-ah l-ih1 jh-er ah-n-t','jj'],
'bellow':['b-eh1 l-ow','nn vb'],
'bellowed':['b-eh1 l-ow-d','vbd vbn'],
'bellowing':['b-eh1 l-ow ih-ng','vbg'],
'bellows':['b-eh1 l-ow-z','vbz nn'],
'bellwether':['b-eh1-l w-eh dh-er','nn jj'],
'belly':['b-eh1 l-iy','nn rb'],
'belong':['b-ih l-ao1-ng','vb vbp jj'],
'belonged':['b-ih l-ao1-ng-d','vbd vbn'],
'belonging':['b-ih l-ao1 ng-ih-ng','vbg nn'],
'belongs':['b-ih l-ao1-ng-z','vbz'],
'beloved':['b-ih l-ah1-v-d','jj vbn nn'],
'below':['b-ih l-ow1','in rb'],
'belt':['b-eh1-l-t','nn'],
'belted':['b-eh1-l t-ah-d','vbd'],
'belting':['b-eh1-l t-ih-ng','nn vbg'],
'belts':['b-eh1-l-t-s','nns vbz'],
'beltway':['b-eh1-l-t w-ey','nn'],
'beluga':['b-ih l-uw1 g-ah','nn'],
'belying':['b-ih l-ay1 ih-ng','vbg'],
'bemoan':['b-ih m-ow1-n','vb'],
'bemoaned':['b-ih m-ow1-n-d','vbd'],
'bemoaning':['b-ih m-ow1 n-ih-ng','vbg'],
'bemoans':['b-ih m-ow1-n-z','vbz'],
'bemused':['b-ih m-y-uw1-z-d','jj'],
'bench':['b-eh1-n-ch','nn vb'],
'benches':['b-eh1-n ch-ih-z','nns'],
'benchmark':['b-eh1-n-ch m-aa-r-k','nn jj'],
'bend':['b-eh1-n-d','vb nn'],
'bending':['b-eh1-n d-ih-ng','vbg'],
'bends':['b-eh1-n-d-z','nns vbz'],
'beneath':['b-ih n-iy1-th','in rb'],
'benefactor':['b-eh1 n-ah f-ae-k t-er','nn'],
'beneficence':['b-ah n-eh1 f-ah s-ah-n-s','nn'],
'beneficial':['b-eh n-ah f-ih1 sh-ah-l','jj'],
'beneficially':['b-eh n-ah f-ih1 sh-ah l-iy','rb'],
'beneficiary':['b-eh n-ah f-ih1 sh-iy eh r-iy','nn jj'],
'benefit':['b-eh1 n-ah f-ih-t','nn vb vbp'],
'benefited':['b-eh1 n-ah f-ih t-ih-d','vbd vbn'],
'benefiting':['b-eh1 n-ah f-ih t-ih-ng','vbg'],
'benefits':['b-eh1 n-ah f-ih-t-s','nns vbz'],
'benevolence':['b-ah n-eh1 v-ah l-ah-n-s','nn'],
'benevolent':['b-ah n-eh1 v-ah l-ah-n-t','jj'],
'benighted':['b-ih n-ay1 t-ih-d','jj'],
'benign':['b-ih n-ay1-n','jj'],
'bent':['b-eh1-n-t','vbd vbn jj nn'],
'benzene':['b-eh-n z-iy1-n','nn'],
'bequeath':['b-ih k-w-iy1-th','vb'],
'bequeathed':['b-ah k-w-iy1-th-t','vbn vbd'],
'bequest':['b-ih k-w-eh1-s-t','nn'],
'berated':['b-ih r-ey1 t-ih-d','vbn vbd'],
'bereavement':['b-er iy1-v m-ah-n-t','nn'],
'bereft':['b-er eh1-f-t','jj vbn'],
'beret':['b-eh1 r-ah-t','nn'],
'bergs':['b-er1-g-z','nns'],
'berry':['b-eh1 r-iy','nn'],
'berserk':['b-er s-er1-k','jj rb'],
'berth':['b-er1-th','nn'],
'beryllium':['b-er ih1 l-iy ah-m','nn'],
'beseiged':['b-ih s-iy1-jh-d','vbn'],
'beset':['b-ih s-eh1-t','vbn jj vbd'],
'besetting':['b-ih s-eh1 t-ih-ng','vbg'],
'beside':['b-ih s-ay1-d','in rb'],
'besides':['b-ih s-ay1-d-z','in rb'],
'besiege':['b-ih s-iy1-jh','vb'],
'besieged':['b-ih s-iy1-jh-d','vbn jj'],
'bespeak':['b-ih s-p-iy1-k','vbp'],
'bespeaks':['b-ih s-p-iy1-k-s','vbz'],
'bespectacled':['b-ih s-p-eh1-k t-ah k-ah-l-d','jj'],
'best':['b-eh1-s-t','jjs rbs jjss nn rb'],
'bested':['b-eh1 s-t-ih-d','vbn vbd'],
'bestial':['b-eh1 s-ch-ah-l','jj'],
'besting':['b-eh1 s-t-ih-ng','vbg'],
'bestow':['b-ih s-t-ow1','vb'],
'bestowed':['b-ih s-t-ow1-d','vbn vbd'],
'bestows':['b-ih s-t-ow1-z','vbz'],
'bestseller':['b-eh1-s-t s-eh1 l-er','nn'],
'bestselling':['b-eh1-s-t s-eh1 l-ih-ng','jj'],
'bet':['b-eh1-t','nn rb vb vbd vbn vbp'],
'beta':['b-ey1 t-ah','nn jj'],
'betide':['b-ih t-ay1-d','vb'],
'betray':['b-ih t-r-ey1','vb vbp'],
'betrayal':['b-ih t-r-ey1 ah-l','nn'],
'betrayed':['b-ih t-r-ey1-d','vbn vbd'],
'betraying':['b-ih t-r-ey1 ih-ng','vbg'],
'betrays':['b-ih t-r-ey1-z','vbz'],
'bets':['b-eh1-t-s','nns'],
'better':['b-eh1 t-er','jjr rbr jj rb vb'],
'bettered':['b-eh1 t-er-d','vbd'],
'bettering':['b-eh1 t-er ih-ng','vbg'],
'betterment':['b-eh1 t-er m-ah-n-t','nn'],
'betters':['b-eh1 t-er-z','nns'],
'betting':['b-eh1 t-ih-ng','vbg nn'],
'between':['b-ih t-w-iy1-n','in rb'],
'bevel':['b-eh1 v-ah-l','vb'],
'beverage':['b-eh1 v-er ih-jh','nn'],
'bevy':['b-eh1 v-iy','nn'],
'beware':['b-ih w-eh1-r','vb vbp'],
'bewildered':['b-ih w-ih1-l d-er-d','vbn jj'],
'bewildering':['b-ih w-ih1-l d-er ih-ng','vbg jj'],
'bewilderment':['b-ih w-ih1-l d-er m-ah-n-t','nn'],
'bewilders':['b-ih w-ih1-l d-er-z','vbz'],
'bewitched':['b-ih w-ih1-ch-t','vbn'],
'beyond':['b-ih aa1-n-d','in rb'],
'bi':['b-ay1','in'],
'bianco':['b-iy aa1-ng k-ow','nn'],
'biannual':['b-ay ae1 n-uw ah-l','jj'],
'bias':['b-ay1 ah-s','nn'],
'biased':['b-ay1 ah-s-t','vbn'],
'bib':['b-ih1-b','nn'],
'bible':['b-ay1 b-ah-l','jj nn'],
'biblical':['b-ih1 b-l-ah k-ah-l','jj'],
'bibliography':['b-ih b-l-iy aa1 g-r-ah f-iy','nn'],
'bicarbonate':['b-ay k-aa1-r b-ah n-ah-t','nn'],
'bicentennial':['b-ay s-eh-n t-eh1 n-iy ah-l','nn jj'],
'bickered':['b-ih1 k-er-d','vbn'],
'bickering':['b-ih1 k-er ih-ng','nn vbg'],
'bicycle':['b-ay1 s-ih k-ah-l','nn'],
'bicycling':['b-ay1 s-ih k-ah l-ih-ng','nn'],
'bid':['b-ih1-d','nn vbd vbn vbp vb'],
'bidder':['b-ih1 d-er','nn'],
'bidding':['b-ih1 d-ih-ng','nn vbg jj'],
'bide':['b-ay1-d','vb'],
'bids':['b-ih1-d-z','nns vbz'],
'biennial':['b-ay eh1 n-iy ah-l','jj'],
'bifocals':['b-ay1 f-ow k-ah-l-z','nns'],
'bifurcated':['b-ih1 f-er k-ey t-ih-d','jj'],
'big':['b-ih1-g','jj rb'],
'bigger':['b-ih1 g-er','jjr rbr'],
'biggest':['b-ih1 g-ah-s-t','jjs rbs'],
'biggie':['b-ih1 g-iy','nn'],
'bigness':['b-ih1-g n-ah-s','nn'],
'bigoted':['b-ih1 g-ah t-ih-d','jj'],
'bigotry':['b-ih1 g-ah t-r-iy','nn'],
'bigots':['b-ih1 g-ah-t-s','nns'],
'bike':['b-ay1-k','nn'],
'biker':['b-ay1 k-er','nn'],
'biking':['b-ay1 k-ih-ng','nn'],
'bikini':['b-ih k-iy1 n-iy','nn'],
'bilateral':['b-ay l-ae1 t-er ah-l','jj'],
'bile':['b-ay1-l','nn'],
'bilingual':['b-ay l-ih1-ng g-w-ah-l','jj'],
'bilious':['b-ih1 l-iy ah-s','jj'],
'bilk':['b-ih1-l-k','vb'],
'bilked':['b-ih1-l-k-t','vbn vbd'],
'bilking':['b-ih1-l k-ih-ng','vbg'],
'bill':['b-ih1-l','nn vbp vb'],
'billable':['b-ih1 l-ah b-ah-l','jj'],
'billboard':['b-ih1-l b-ao-r-d','nn'],
'billed':['b-ih1-l-d','vbn jj vbd'],
'billet':['b-ih1 l-ah-t','nn'],
'billfold':['b-ih1-l f-ow-l-d','nn'],
'billiard':['b-ih1 l-y-er-d','nn'],
'billiards':['b-ih1 l-y-er-d-z','nn nns'],
'billing':['b-ih1 l-ih-ng','nn vbg'],
'billionaire':['b-ih l-y-ah n-eh1-r','nn'],
'billions':['b-ih1 l-y-ah-n-z','nns'],
'billon':['b-ih1 l-ah-n','nn'],
'billowed':['b-ih1 l-ow-d','vbd'],
'billowing':['b-ih1 l-ow ih-ng','vbg'],
'billows':['b-ih1 l-ow-z','nns'],
'bills':['b-ih1-l-z','nns vbz'],
'bimbos':['b-ih1-m b-ow-s','nns'],
'bimonthly':['b-ay m-ah1-n-th l-iy','jj'],
'bin':['b-ih1-n','nn'],
'binary':['b-ay1 n-er iy','jj'],
'binational':['b-ay n-ae1 sh-ah n-ah-l','jj'],
'bind':['b-ay1-n-d','nn vb vbp'],
'binder':['b-ay1-n d-er','nn'],
'binding':['b-ay1-n d-ih-ng','nn jj vbg'],
'binds':['b-ay1-n-d-z','vbz'],
'binge':['b-ih1-n-jh','nn'],
'bingo':['b-ih1-ng g-ow','nn'],
'binoculars':['b-ah n-aa1 k-y-ah l-er-z','nns'],
'binomial':['b-ay n-ow1 m-iy ah-l','nn'],
'bio':['b-ay ow1','nn jj'],
'biochemical':['b-ay ow k-eh1 m-ah k-ah-l','jj'],
'biochemist':['b-ay ow k-eh1 m-ah-s-t','nn'],
'biochemistry':['b-ay ow k-eh1 m-ah s-t-r-iy','nn'],
'biodegradable':['b-ay ow d-ah g-r-ey1 d-ah b-ah-l','jj'],
'biofeedback':['b-ay ow f-iy1-d b-ae-k','nn'],
'biographer':['b-ay aa1 g-r-ah f-er','nn'],
'biographical':['b-ay ah g-r-ae1 f-ih k-ah-l','jj'],
'biography':['b-ay aa1 g-r-ah f-iy','nn'],
'biological':['b-ay ah l-aa1 jh-ih k-ah-l','jj nn'],
'biologically':['b-ay ah l-aa1 jh-ih-k l-iy','rb'],
'biologist':['b-ay aa1 l-ah jh-ah-s-t','nn'],
'biology':['b-ay aa1 l-ah jh-iy','nn'],
'biomedical':['b-ay ow m-eh1 d-ih k-ah-l','jj'],
'biopharmaceutical':['b-ay ow f-aa-r m-ah s-uw1 t-ih k-ah-l','jj'],
'biophysics':['b-ay ow f-ih1 s-ih-k-s','nns'],
'biopsy':['b-ay1 aa-p s-iy','nn'],
'biotech':['b-ay1 ow t-eh-k','jj nn'],
'biotechnology':['b-ay ow t-eh-k n-aa1 l-ah jh-iy','nn'],
'bipartisan':['b-ay p-aa1-r t-ih z-ah-n','jj'],
'bipartisanship':['b-ay p-aa1-r t-ah z-ah-n sh-ih-p','nn'],
'biplane':['b-ay1 p-l-ey-n','nn'],
'biracial':['b-ay r-ey1 sh-ah-l','jj'],
'birch':['b-er1-ch','nn'],
'bird':['b-er1-d','nn'],
'birdie':['b-er1 d-iy','nn'],
'birdied':['b-er1 d-iy-d','vbd vbn'],
'birth':['b-er1-th','nn'],
'birthday':['b-er1-th d-ey','nn'],
'birthplace':['b-er1-th p-l-ey-s','nn'],
'birthrate':['b-er1-th r-ey-t','nn'],
'birthright':['b-er1-th r-ay-t','nn'],
'biscuit':['b-ih1 s-k-ah-t','nn'],
'bisexual':['b-ay s-eh1-k sh-uw ah-l','jj'],
'bishop':['b-ih1 sh-ah-p','nn'],
'bison':['b-ay1 s-ah-n','nn'],
'bisque':['b-ih1-s-k','nn'],
'bit':['b-ih1-t','nn vbd vbn jj rb vb'],
'bitch':['b-ih1-ch','nn vb'],
'bitchy':['b-ih1 ch-iy','jj'],
'bite':['b-ay1-t','vb vbp nn'],
'bites':['b-ay1-t-s','nns vbz'],
'biting':['b-ay1 t-ih-ng','vbg jj'],
'bits':['b-ih1-t-s','nns'],
'bitten':['b-ih1 t-ah-n','vbn'],
'bitter':['b-ih1 t-er','jj'],
'bitterest':['b-ih1 t-er ah-s-t','jjs'],
'bitterly':['b-ih1 t-er l-iy','rb'],
'bitterness':['b-ih1 t-er n-ah-s','nn'],
'bittersweet':['b-ih1 t-er s-w-iy-t','jj'],
'bituminous':['b-ih t-uw1 m-ah n-ah-s','jj'],
'bivouac':['b-ih1-v w-ae-k','nn'],
'biweekly':['b-ay w-iy1-k l-iy','jj nn'],
'biz':['b-ih1-z','nn'],
'bizarre':['b-ah z-aa1-r','jj'],
'black':['b-l-ae1-k','jj nn vb'],
'blackberry':['b-l-ae1-k b-eh r-iy','nn'],
'blackboard':['b-l-ae1-k b-ao-r-d','nn'],
'blacked':['b-l-ae1-k-t','vbn'],
'blackened':['b-l-ae1 k-ah-n-d','vbn jj'],
'blackening':['b-l-ae1 k-ah n-ih-ng','nn'],
'blacker':['b-l-ae1 k-er','jjr'],
'blackest':['b-l-ae1 k-ah-s-t','jjs'],
'blackjack':['b-l-ae1-k jh-ae-k','nn'],
'blacklist':['b-l-ae1-k l-ih-s-t','vb'],
'blacklisting':['b-l-ae1-k l-ih s-t-ih-ng','nn'],
'blackmail':['b-l-ae1-k m-ey-l','nn vb'],
'blackmailed':['b-l-ae1-k m-ey-l-d','vbn'],
'blackmailing':['b-l-ae1-k m-ey l-ih-ng','vbg'],
'blackness':['b-l-ae1-k n-ah-s','nn'],
'blackout':['b-l-ae1-k aw-t','nn'],
'blacks':['b-l-ae1-k-s','nns nnps'],
'blacksmith':['b-l-ae1-k s-m-ih-th','nn'],
'bladder':['b-l-ae1 d-er','nn'],
'blade':['b-l-ey1-d','nn'],
'blame':['b-l-ey1-m','vb nn vbp'],
'blamed':['b-l-ey1-m-d','vbd vbn'],
'blames':['b-l-ey1-m-z','vbz'],
'blaming':['b-l-ey1 m-ih-ng','vbg'],
'blanche':['b-l-ae1-n-ch','jj nn'],
'bland':['b-l-ae1-n-d','jj'],
'blander':['b-l-ae1-n d-er','jjr'],
'blandly':['b-l-ae1-n-d l-iy','rb'],
'blandness':['b-l-ae1-n-d n-ah-s','nn'],
'blank':['b-l-ae1-ng-k','jj nn rb vb'],
'blanket':['b-l-ae1-ng k-ah-t','nn vbp jj vb'],
'blanketed':['b-l-ae1-ng k-ah t-ih-d','vbd'],
'blankets':['b-l-ae1-ng k-ah-t-s','nns vbz'],
'blanks':['b-l-ae1-ng-k-s','nns'],
'blared':['b-l-eh1-r-d','vbd'],
'blares':['b-l-eh1-r-z','vbz'],
'blaring':['b-l-eh1 r-ih-ng','vbg'],
'blase':['b-l-ey1-z','jj nn'],
'blasphemous':['b-l-ae1-s f-ah m-ah-s','jj'],
'blasphemy':['b-l-ae1-s f-ah m-iy','nn'],
'blast':['b-l-ae1-s-t','nn vb'],
'blastdown':['b-l-ae1-s-t d-aw-n','nn'],
'blasted':['b-l-ae1 s-t-ah-d','vbd vbn'],
'blasting':['b-l-ae1 s-t-ih-ng','vbg jj'],
'blasts':['b-l-ae1-s-t-s','nns'],
'blatant':['b-l-ey1 t-ah-n-t','jj'],
'blatantly':['b-l-ey1 t-ah-n-t l-iy','rb'],
'blaze':['b-l-ey1-z','nn vb vbp'],
'blazed':['b-l-ey1-z-d','vbd'],
'blazer':['b-l-ey1 z-er','nn'],
'blazing':['b-l-ey1 z-ih-ng','vbg jj'],
'bleach':['b-l-iy1-ch','nn'],
'bleached':['b-l-iy1-ch-t','jj vbn'],
'bleacher':['b-l-iy1 ch-er','nn'],
'bleaching':['b-l-iy1 ch-ih-ng','vbg'],
'bleak':['b-l-iy1-k','jj nn'],
'bleaker':['b-l-iy1 k-er','jjr'],
'bleary':['b-l-ih1 r-iy','jj'],
'bled':['b-l-eh1-d','vbd'],
'bleed':['b-l-iy1-d','vb vbp'],
'bleeding':['b-l-iy1 d-ih-ng','vbg nn'],
'blemish':['b-l-eh1 m-ih-sh','nn'],
'blend':['b-l-eh1-n-d','nn vbp vb'],
'blended':['b-l-eh1-n d-ah-d','jj vbd vbn'],
'blender':['b-l-eh1-n d-er','nn'],
'blending':['b-l-eh1-n d-ih-ng','vbg nn'],
'blends':['b-l-eh1-n-d-z','nns'],
'bless':['b-l-eh1-s','vb'],
'blessed':['b-l-eh1-s-t','vbn jj vbd'],
'blessing':['b-l-eh1 s-ih-ng','nn vbg'],
'blest':['b-l-eh1-s-t','vb vbn'],
'blew':['b-l-uw1','vbd'],
'blight':['b-l-ay1-t','nn'],
'blighted':['b-l-ay1 t-ih-d','vbn vbd jj'],
'blind':['b-l-ay1-n-d','jj nns nn vb'],
'blinded':['b-l-ay1-n d-ih-d','vbn'],
'blindfold':['b-l-ay1-n-d f-ow-l-d','nn'],
'blindfolded':['b-l-ay1-n-d f-ow-l d-ih-d','vbn jj vbd'],
'blinding':['b-l-ay1-n d-ih-ng','jj vbg'],
'blindly':['b-l-ay1-n-d l-iy','rb'],
'blindness':['b-l-ay1-n-d n-ah-s','nn'],
'blinds':['b-l-ay1-n-d-z','nns vbz'],
'blindsided':['b-l-ay1-n-d s-ay d-ih-d','vbn jj'],
'blink':['b-l-ih1-ng-k','vb nn'],
'blinked':['b-l-ih1-ng-k-t','vbd'],
'blinking':['b-l-ih1-ng k-ih-ng','jj vbg rb'],
'blinks':['b-l-ih1-ng-k-s','vbz'],
'blip':['b-l-ih1-p','nn vbp vb'],
'blips':['b-l-ih1-p-s','nns'],
'bliss':['b-l-ih1-s','nn'],
'blissful':['b-l-ih1-s f-ah-l','jj'],
'blissfully':['b-l-ih1-s f-ah l-iy','rb'],
'blister':['b-l-ih1 s-t-er','nn'],
'blistered':['b-l-ih1 s-t-er-d','vbn'],
'blistering':['b-l-ih1 s-t-er ih-ng','vbg jj'],
'blithe':['b-l-ay1-dh','jj'],
'blithely':['b-l-ay1-th l-iy','rb'],
'blitz':['b-l-ih1-t-s','nn'],
'blitzkrieg':['b-l-ih1-t s-k-r-iy-g','nn'],
'blizzard':['b-l-ih1 z-er-d','nn'],
'bloat':['b-l-ow1-t','nn'],
'bloated':['b-l-ow1 t-ih-d','jj vbn'],
'blob':['b-l-aa1-b','nn'],
'blobby':['b-l-aa1 b-iy','jj'],
'bloc':['b-l-aa1-k','nn'],
'block':['b-l-aa1-k','nn vbp jj vb'],
'blockade':['b-l-aa k-ey1-d','nn'],
'blockading':['b-l-aa k-ey1 d-ih-ng','vbg'],
'blockages':['b-l-aa1 k-ih jh-ih-z','nns'],
'blockbuster':['b-l-aa1-k b-ah s-t-er','nn jj'],
'blocked':['b-l-aa1-k-t','vbn jj vbd'],
'blocker':['b-l-aa1 k-er','nn'],
'blocking':['b-l-aa1 k-ih-ng','vbg jj nn'],
'blocks':['b-l-aa1-k-s','nns vbz'],
'blokes':['b-l-ow1-k-s','nns'],
'blond':['b-l-aa1-n-d','jj nn'],
'blonde':['b-l-aa1-n-d','jj nn'],
'blood':['b-l-ah1-d','nn'],
'bloodbath':['b-l-ah1-d b-ae-th','nn'],
'blooded':['b-l-ah1 d-ih-d','vbn'],
'bloodied':['b-l-ah1 d-iy-d','jj vbn'],
'bloodiest':['b-l-ah1 d-iy ah-s-t','jjs'],
'bloodless':['b-l-ah1-d l-ah-s','jj'],
'bloodletting':['b-l-ah1-d l-eh t-ih-ng','vbg nn'],
'bloodshed':['b-l-ah1-d sh-eh-d','nn'],
'bloodstained':['b-l-ah1-d s-t-ey-n-d','jj'],
'bloodstains':['b-l-ah1-d s-t-ey-n-z','nns'],
'bloodstream':['b-l-ah1-d s-t-r-iy-m','nn'],
'bloodsucking':['b-l-ah1-d s-ah k-ih-ng','vbg'],
'bloodthirsty':['b-l-ah1-d th-er s-t-iy','jj'],
'bloody':['b-l-ah1 d-iy','jj'],
'bloom':['b-l-uw1-m','nn vbp vb'],
'bloomed':['b-l-uw1-m-d','vbd'],
'blooming':['b-l-uw1 m-ih-ng','vbg nn'],
'blooms':['b-l-uw1-m-z','nns vbz'],
'blooper':['b-l-uw1 p-er','nn'],
'blossom':['b-l-aa1 s-ah-m','vb jj nn'],
'blossomed':['b-l-aa1 s-ah-m-d','vbd vbn'],
'blossoms':['b-l-aa1 s-ah-m-z','nns'],
'blot':['b-l-aa1-t','nn vb'],
'blots':['b-l-aa1-t-s','nns vbz'],
'blouse':['b-l-aw1-s','nn'],
'blouses':['b-l-aw1 s-ah-z','nns'],
'blow':['b-l-ow1','nn vb vbp'],
'blower':['b-l-ow1 er','nn'],
'blowing':['b-l-ow1 ih-ng','vbg nn'],
'blown':['b-l-ow1-n','vbn'],
'blowout':['b-l-ow1 aw-t','nn'],
'blows':['b-l-ow1-z','nns vbz'],
'blowtorch':['b-l-ow1 t-ao-r-ch','nn'],
'blowup':['b-l-ow1 ah-p','nn'],
'blubber':['b-l-ah1 b-er','nn'],
'bludgeon':['b-l-ah1 jh-ah-n','vb nn'],
'bludgeoned':['b-l-ah1 jh-ah-n-d','vbn'],
'blue':['b-l-uw1','jj nn'],
'blueberry':['b-l-uw1 b-eh r-iy','nn'],
'bluebonnets':['b-l-uw1 b-aa n-ah-t-s','nns'],
'bluechip':['b-l-uw1 ch-ih-p','jj'],
'blueprint':['b-l-uw1 p-r-ih-n-t','nn'],
'blueprints':['b-l-uw1 p-r-ih-n-t-s','nns vbz'],
'blues':['b-l-uw1-z','nns jj'],
'bluest':['b-l-uw1 ah-s-t','jjs'],
'bluff':['b-l-ah1-f','nn jj'],
'bluffing':['b-l-ah1 f-ih-ng','vbg'],
'bluing':['b-l-uw1 ih-ng','nn'],
'bluish':['b-l-uw1 ih-sh','jj'],
'blunder':['b-l-ah1-n d-er','nn vb'],
'blundered':['b-l-ah1-n d-er-d','vbd vbn'],
'blunders':['b-l-ah1-n d-er-z','nns'],
'blunt':['b-l-ah1-n-t','vb jj'],
'blunted':['b-l-ah1-n t-ah-d','vbd jj vbn'],
'blunter':['b-l-ah1-n t-er','nn'],
'bluntest':['b-l-ah1-n t-ah-s-t','rbs'],
'bluntly':['b-l-ah1-n-t l-iy','rb'],
'bluntness':['b-l-ah1-n-t n-ah-s','nn'],
'blunts':['b-l-ah1-n-t-s','vbz'],
'blur':['b-l-er1','nn vb vbp'],
'blurred':['b-l-er1-d','vbn jj vbd'],
'blurring':['b-l-er1 ih-ng','vbg nn'],
'blurry':['b-l-er1 iy','jj'],
'blurt':['b-l-er1-t','nn vbp'],
'blurted':['b-l-er1 t-ih-d','vbd vbn'],
'blush':['b-l-ah1-sh','nn vb'],
'blushed':['b-l-ah1-sh-t','vbd vbn'],
'blushes':['b-l-ah1 sh-ah-z','nns'],
'blushing':['b-l-ah1 sh-ih-ng','vbg'],
'bluster':['b-l-ah1 s-t-er','nn vb'],
'blustery':['b-l-ah1 s-t-er iy','jj'],
'boa':['b-ow1 ah','nn'],
'boar':['b-ao1-r','nns'],
'board':['b-ao1-r-d','nn rb vb'],
'boarded':['b-ao1-r d-ah-d','vbd vbn jj'],
'boarding':['b-ao1-r d-ih-ng','vbg nn'],
'boardinghouses':['b-ao1-r d-ih-ng hh-aw s-ih-z','nns'],
'boardroom':['b-ao1-r-d r-uw-m','nn'],
'boards':['b-ao1-r-d-z','nns vbz'],
'boast':['b-ow1-s-t','vbp nn vb'],
'boasted':['b-ow1 s-t-ah-d','vbd vbn'],
'boastful':['b-ow1-s-t f-ah-l','jj'],
'boasting':['b-ow1 s-t-ih-ng','vbg'],
'boasts':['b-ow1-s-t-s','vbz'],
'boat':['b-ow1-t','nn'],
'boaters':['b-ow1 t-er-z','nns'],
'boating':['b-ow1 t-ih-ng','nn vbg'],
'boatload':['b-ow1-t l-ow-d','nn'],
'boatman':['b-ow1-t m-ah-n','nn'],
'bobbing':['b-aa1 b-ih-ng','vbg'],
'bobby':['b-aa1 b-iy','nn jj'],
'bode':['b-ow1-d','vb vbp'],
'bodegas':['b-ow d-ey1 g-ah-s','nns'],
'bodes':['b-ow1-d-z','vbz'],
'bodied':['b-aa1 d-iy-d','jj'],
'bodies':['b-aa1 d-iy-z','nns vbz'],
'bodily':['b-aa1 d-ah l-iy','jj rb'],
'body':['b-aa1 d-iy','nn'],
'bodyguard':['b-aa1 d-iy g-aa-r-d','nn'],
'bog':['b-aa1-g','vb nn'],
'bogey':['b-ow1 g-iy','nn'],
'bogeyed':['b-ow1 g-iy-d','vbd'],
'bogeymen':['b-ow1 g-iy m-ih-n','nns'],
'bogged':['b-aa1-g-d','vbd vbn'],
'bogging':['b-ao1 g-ih-ng','vbg'],
'boggled':['b-aa1 g-ah-l-d','vbd'],
'bogs':['b-aa1-g-z','vbz'],
'bogus':['b-ow1 g-ah-s','jj'],
'bohemian':['b-ow hh-iy1 m-iy ah-n','jj'],
'boil':['b-oy1-l','vb nn'],
'boiled':['b-oy1-l-d','vbn vbd jj'],
'boiler':['b-oy1 l-er','nn'],
'boilerplate':['b-oy1 l-er p-l-ey-t','nn'],
'boiling':['b-oy1 l-ih-ng','vbg nn'],
'boils':['b-oy1-l-z','vbz'],
'boisterous':['b-oy1 s-t-er ah-s','jj'],
'bold':['b-ow1-l-d','jj'],
'bolder':['b-ow1-l d-er','jjr'],
'boldest':['b-ow1-l d-ih-s-t','jjs'],
'boldly':['b-ow1-l-d l-iy','rb'],
'boldness':['b-ow1-l-d n-ah-s','nn'],
'boll':['b-ow1-l','nn'],
'bolster':['b-ow1-l s-t-er','vb vbp nn'],
'bolstered':['b-ow1-l s-t-er-d','vbn vbd'],
'bolstering':['b-ow1-l s-t-er ih-ng','vbg'],
'bolsters':['b-ow1-l s-t-er-z','vbz'],
'bolt':['b-ow1-l-t','nn vb'],
'bolted':['b-ow1-l t-ah-d','vbn vbd'],
'bolting':['b-ow1-l t-ih-ng','vbg'],
'bolts':['b-ow1-l-t-s','nns'],
'bomb':['b-aa1-m','nn vb'],
'bombard':['b-aa-m b-aa1-r-d','vb'],
'bombarded':['b-aa-m b-aa1-r d-ih-d','vbd vbn'],
'bombarding':['b-aa-m b-aa1-r d-ih-ng','vbg'],
'bombardment':['b-aa-m b-aa1-r-d m-ah-n-t','nn'],
'bombast':['b-aa1-m b-ae-s-t','nn'],
'bombastic':['b-aa-m b-ae1 s-t-ih-k','jj'],
'bombed':['b-aa1-m-d','vbn vbd'],
'bomber':['b-aa1 m-er','nn'],
'bombing':['b-aa1 m-ih-ng','nn vbg'],
'bombs':['b-aa1-m-z','nns'],
'bombshell':['b-aa1-m sh-eh-l','nn'],
'bonanza':['b-ah n-ae1-n z-ah','nn'],
'bond':['b-aa1-n-d','nn jj'],
'bondage':['b-aa1-n d-ih-jh','nn'],
'bonded':['b-aa1-n d-ah-d','vbn jj'],
'bondholder':['b-aa1-n-d hh-ow-l d-er','nn'],
'bonding':['b-aa1-n d-ih-ng','vbg nn'],
'bone':['b-ow1-n','nn'],
'boned':['b-ow1-n-d','vbn'],
'bonfire':['b-aa1-n f-ay er','nn'],
'bonkers':['b-aa1-ng k-er-z','jj'],
'bonnet':['b-aa1 n-ah-t','nn'],
'bonus':['b-ow1 n-ah-s','nn'],
'bonuses':['b-ow1 n-ah s-ih-z','nns vbz'],
'bony':['b-ow1 n-iy','jj'],
'boo':['b-uw1','vb'],
'booby':['b-uw1 b-iy','jj nn'],
'booed':['b-uw1-d','vbd'],
'boogie':['b-uw1 g-iy','nn'],
'booing':['b-uw1 ih-ng','vbg'],
'book':['b-uh1-k','nn vb'],
'bookcase':['b-uh1-k k-ey-s','nn'],
'bookcases':['b-uh1-k k-ey s-ih-z','nns'],
'booked':['b-uh1-k-t','vbn vbd'],
'booker':['b-uh1-k er','nn'],
'bookies':['b-uh1-k iy-z','nns'],
'booking':['b-uh1 k-ih-ng','nn vbg'],
'bookish':['b-uh1-k ih-sh','jj'],
'bookkeeper':['b-uh1-k k-iy p-er','nn'],
'bookkeeping':['b-uh1-k k-iy p-ih-ng','nn vbg'],
'booklet':['b-uh1-k l-ih-t','nn'],
'books':['b-uh1-k-s','nns nn pos'],
'bookseller':['b-uh1-k s-eh l-er','nn'],
'bookshelf':['b-uh1-k sh-eh-l-f','nn'],
'bookshelves':['b-uh1-k sh-eh-l-v-z','nns'],
'bookstore':['b-uh1-k s-t-ao-r','nn'],
'boom':['b-uw1-m','nn vb'],
'boomed':['b-uw1-m-d','vbd vbn'],
'boomerang':['b-uw1 m-er ae-ng','nn vb'],
'boomers':['b-uw1 m-er-z','nns'],
'booming':['b-uw1 m-ih-ng','jj vbg'],
'booms':['b-uw1-m-z','nns vbz'],
'boomtown':['b-uw1-m t-aw-n','nn'],
'boon':['b-uw1-n','nn jj'],
'boondoggle':['b-uw-n d-aa1 g-ah-l','nn'],
'boorish':['b-uh1 r-ih-sh','jj'],
'boors':['b-uh1-r-z','nns'],
'boos':['b-uw1-z','nns'],
'boost':['b-uw1-s-t','vb nn vbp'],
'boosted':['b-uw1 s-t-ih-d','vbd vbn'],
'booster':['b-uw1 s-t-er','nn'],
'boosting':['b-uw1 s-t-ih-ng','vbg'],
'boosts':['b-uw1-s-t-s','nns vbz'],
'boot':['b-uw1-t','nn vb'],
'booted':['b-uw1 t-ih-d','vbn vbd jj'],
'booth':['b-uw1-th','nn'],
'booting':['b-uw1 t-ih-ng','vbg'],
'bootleg':['b-uw1-t l-eh-g','jj nn'],
'bootlegger':['b-uw1-t l-eh g-er','nn'],
'bootlegging':['b-uw1-t l-eh g-ih-ng','nn'],
'boots':['b-uw1-t-s','nns'],
'booty':['b-uw1 t-iy','nn'],
'booze':['b-uw1-z','nn'],
'bop':['b-aa1-p','nn'],
'borax':['b-ao1 r-ae-k-s','nn'],
'border':['b-ao1-r d-er','nn'],
'bordered':['b-ao1-r d-er-d','vbn'],
'bordering':['b-ao1-r d-er ih-ng','vbg'],
'borderline':['b-ao1-r d-er l-ay-n','jj nn rb'],
'borders':['b-ao1-r d-er-z','nns vbz'],
'bore':['b-ao1-r','vbd vbp jj nn vb'],
'bored':['b-ao1-r-d','vbn jj vbd'],
'boredom':['b-ao1-r d-ah-m','nn'],
'borer':['b-ao1 r-er','nn'],
'bores':['b-ao1-r-z','nns vbz'],
'boring':['b-ao1 r-ih-ng','jj vbg nn'],
'born':['b-ao1-r-n','vbn'],
'borne':['b-ao1-r-n','vbn'],
'borough':['b-er1 ow','nn'],
'borrow':['b-aa1 r-ow','vb vbp'],
'borrowed':['b-aa1 r-ow-d','vbn vbd jj'],
'borrower':['b-aa1 r-ow er','nn'],
'borrowing':['b-aa1 r-ow ih-ng','nn jj vbg'],
'borrows':['b-aa1 r-ow-z','vbz'],
'bosom':['b-uh1 z-ah-m','nn'],
'boss':['b-aa1-s','nn'],
'bossman':['b-ao1-s m-ah-n','nn'],
'botanical':['b-ah t-ae1 n-ih k-ah-l','jj'],
'botanist':['b-aa1 t-ah n-ah-s-t','nn'],
'botany':['b-aa1 t-ah n-iy','nn'],
'botched':['b-aa1-ch-t','vbn jj'],
'bother':['b-aa1 dh-er','vb vbp'],
'bothered':['b-aa1 dh-er-d','vbn vbd'],
'bothering':['b-aa1 dh-er ih-ng','vbg'],
'bothers':['b-aa1 dh-er-z','vbz'],
'bothersome':['b-aa1 dh-er s-ah-m','jj'],
'bottle':['b-aa1 t-ah-l','nn vb'],
'bottled':['b-aa1 t-ah-l-d','jj vbd vbn'],
'bottleneck':['b-aa1 t-ah-l n-eh-k','nn'],
'bottler':['b-aa1-t l-er','nn'],
'bottles':['b-aa1 t-ah-l-z','nns vbz'],
'bottling':['b-aa1 t-ah-l ih-ng','nn vbg'],
'bottom':['b-aa1 t-ah-m','nn jj vb'],
'bottomed':['b-aa1 t-ah-m-d','vbn vbd'],
'bottoming':['b-aa1 t-ah m-ih-ng','vbg'],
'bottomless':['b-aa1 t-ah-m l-ah-s','jj'],
'bottoms':['b-aa1 t-ah-m-z','nns'],
'bough':['b-aw1','nn'],
'bought':['b-aa1-t','vbd vbn'],
'boulder':['b-ow1-l d-er','nn'],
'boulevard':['b-uh1 l-ah v-aa-r-d','nn'],
'bounce':['b-aw1-n-s','vb vbp nn'],
'bounced':['b-aw1-n-s-t','vbd vbn'],
'bounces':['b-aw1-n s-ih-z','vbz nns'],
'bouncing':['b-aw1-n s-ih-ng','vbg nn'],
'bouncy':['b-aw1-n s-iy','jj'],
'bound':['b-aw1-n-d','vbn jj nn vbd'],
'boundary':['b-aw1-n d-er iy','nn'],
'bounded':['b-aw1-n d-ah-d','vbn vbd jj'],
'bounding':['b-aw1-n d-ih-ng','vbg'],
'boundless':['b-aw1-n-d l-ah-s','jj'],
'bounds':['b-aw1-n-d-z','nns vbz'],
'bounty':['b-aw1-n t-iy','nn'],
'bouquet':['b-uw k-ey1','nn'],
'bourbon':['b-er1 b-ah-n','nn'],
'bourgeois':['b-uh-r zh-w-aa1','jj'],
'bourgeoisie':['b-uh-r zh-w-aa z-iy1','nns'],
'boutique':['b-uw t-iy1-k','nn'],
'bouts':['b-aw1-t-s','nns'],
'bovine':['b-ow1 v-ay-n','jj nn'],
'bow':['b-aw1','nn vb vbp'],
'bowed':['b-aw1-d','vbd vbn jj'],
'bowel':['b-aw1 ah-l','nn'],
'bower':['b-aw1 er','nn'],
'bowing':['b-ow1 ih-ng','vbg'],
'bowl':['b-ow1-l','nn vbp'],
'bowled':['b-ow1-l-d','vbn'],
'bowling':['b-ow1 l-ih-ng','nn vbg'],
'bowls':['b-ow1-l-z','nns vbz'],
'bows':['b-aw1-z','nns vbz'],
'box':['b-aa1-k-s','nn vb'],
'boxcar':['b-aa1-k-s k-aa-r','nn'],
'boxed':['b-aa1-k-s-t','vbn'],
'boxer':['b-aa1-k s-er','nn'],
'boxes':['b-aa1-k s-ah-z','nns'],
'boxing':['b-aa1-k s-ih-ng','nn'],
'boxy':['b-aa1-k s-iy','jj'],
'boy':['b-oy1','nn uh'],
'boycott':['b-oy1 k-aa-t','nn vb vbp'],
'boycotted':['b-oy1 k-aa t-ih-d','vbn'],
'boycotting':['b-oy1 k-aa t-ih-ng','vbg'],
'boycotts':['b-oy1 k-aa-t-s','nns'],
'boyfriend':['b-oy1 f-r-eh-n-d','nn'],
'boyhood':['b-oy1 hh-uh-d','nn'],
'boyish':['b-oy1 ih-sh','jj'],
'bra':['b-r-aa1','nn'],
'brace':['b-r-ey1-s','nn vb vbp'],
'braced':['b-r-ey1-s-t','vbn vbd'],
'bracelet':['b-r-ey1-s l-ah-t','nn'],
'braces':['b-r-ey1 s-ah-z','nns'],
'bracing':['b-r-ey1 s-ih-ng','vbg nn'],
'bracket':['b-r-ae1 k-ih-t','nn'],
'brackish':['b-r-ae1 k-ih-sh','jj'],
'brad':['b-r-ae1-d','nn'],
'bradykinin':['b-r-ah d-ih1 k-ih n-ih-n','nn'],
'brag':['b-r-ae1-g','vb vbp nn'],
'braggadocio':['b-r-ae g-ah d-ow1 sh-iy ow','nn'],
'bragged':['b-r-ae1-g-d','vbd'],
'bragging':['b-r-ae1 g-ih-ng','vbg nn'],
'brags':['b-r-ae1-g-z','vbz nns'],
'braided':['b-r-ey1 d-ih-d','jj vbn'],
'braiding':['b-r-ey1 d-ih-ng','vbg'],
'braids':['b-r-ey1-d-z','nns'],
'brain':['b-r-ey1-n','nn'],
'brainchild':['b-r-ey1-n ch-ay-l-d','nn'],
'brainpower':['b-r-ey1-n p-aw er','nn'],
'brainstorm':['b-r-ey1-n s-t-ao-r-m','nn'],
'brainwashed':['b-r-ey1-n w-aa-sh-t','vbn'],
'brainwashing':['b-r-ey1-n w-aa sh-ih-ng','nn'],
'brainy':['b-r-ey1 n-iy','jj'],
'brake':['b-r-ey1-k','nn vb'],
'brakes':['b-r-ey1-k-s','nns vbz'],
'braking':['b-r-ey1 k-ih-ng','vbg nn'],
'brambles':['b-r-ae1-m b-ah-l-z','nns'],
'bran':['b-r-ae1-n','nn'],
'branch':['b-r-ae1-n-ch','nn vb'],
'branched':['b-r-ae1-n-ch-t','vbn jj vbd'],
'branches':['b-r-ae1-n ch-ah-z','nns'],
'branching':['b-r-ae1-n ch-ih-ng','vbg nn'],
'brand':['b-r-ae1-n-d','nn vbp jj rb'],
'branded':['b-r-ae1-n d-ih-d','vbn jj'],
'brandished':['b-r-ae1-n d-ih-sh-t','vbd'],
'brandishes':['b-r-ae1-n d-ih sh-ih-z','vbz'],
'brandishing':['b-r-ae1-n d-ih sh-ih-ng','vbg'],
'brands':['b-r-ae1-n-d-z','nns vbz'],
'brandy':['b-r-ae1-n d-iy','nn vb'],
'brash':['b-r-ae1-sh','jj'],
'brashness':['b-r-ae1-sh n-ah-s','nn'],
'brass':['b-r-ae1-s','nn jj'],
'brassy':['b-r-ae1 s-iy','jj'],
'brat':['b-r-ae1-t','nn'],
'bratwurst':['b-r-ae1-t w-er-s-t','nn'],
'bravado':['b-r-ah v-aa1 d-ow','nn'],
'brave':['b-r-ey1-v','jj vb'],
'braved':['b-r-ey1-v-d','vbd vbn'],
'bravely':['b-r-ey1-v l-iy','rb'],
'braver':['b-r-ey1 v-er','jjr'],
'bravery':['b-r-ey1 v-er iy','nn'],
'bravest':['b-r-ey1 v-ah-s-t','jjs'],
'braving':['b-r-ey1 v-ih-ng','vbg'],
'bravura':['b-r-ah v-y-uh1 r-ah','nn jj'],
'brawl':['b-r-ao1-l','nn'],
'brawling':['b-r-ao1 l-ih-ng','nn'],
'brawny':['b-r-ao1 n-iy','jj'],
'brazen':['b-r-ey1 z-ah-n','jj vb'],
'brazenly':['b-r-ey1 z-ah-n l-iy','rb'],
'brazier':['b-r-ey1 z-iy er','nn'],
'brazil':['b-r-ah z-ih1-l','nn'],
'breach':['b-r-iy1-ch','nn vb'],
'breached':['b-r-iy1-ch-t','vbd vbn'],
'breaches':['b-r-iy1 ch-ih-z','nns'],
'breaching':['b-r-iy1 ch-ih-ng','vbg'],
'bread':['b-r-eh1-d','nn'],
'breadbasket':['b-r-eh1-d b-ae s-k-ah-t','nn'],
'breadbox':['b-r-eh1-d b-aa-k-s','nn'],
'breaded':['b-r-eh1 d-ih-d','vbn'],
'breadth':['b-r-eh1-d-th','nn'],
'break':['b-r-ey1-k','vb nn vbp'],
'breakage':['b-r-ey1 k-ih-jh','nn'],
'breakaway':['b-r-ey1-k ah w-ey','nn'],
'breakdown':['b-r-ey1-k d-aw-n','nn'],
'breaker':['b-r-ey1 k-er','nn'],
'breakeven':['b-r-ey1 k-iy v-ah-n','jj nn'],
'breakfast':['b-r-eh1-k f-ah-s-t','nn'],
'breaking':['b-r-ey1 k-ih-ng','vbg jj nn'],
'breakneck':['b-r-ey1-k n-eh-k','jj'],
'breaks':['b-r-ey1-k-s','nns vbz'],
'breakthrough':['b-r-ey1-k th-r-uw','nn'],
'breakup':['b-r-ey1-k ah-p','nn'],
'breakwater':['b-r-ey1-k w-ao t-er','nn'],
'breast':['b-r-eh1-s-t','nn'],
'breath':['b-r-eh1-th','nn vb'],
'breathe':['b-r-iy1-dh','vb vbp'],
'breathed':['b-r-iy1-dh-d','vbd vbn'],
'breather':['b-r-iy1 dh-er','nn'],
'breathes':['b-r-iy1-dh-z','vbz'],
'breathing':['b-r-iy1 dh-ih-ng','nn vbg'],
'breathless':['b-r-eh1-th l-ah-s','jj'],
'breathlessly':['b-r-eh1-th l-ah-s l-iy','rb'],
'breaths':['b-r-eh1-th-s','nns'],
'breathtaking':['b-r-eh1-th t-ey k-ih-ng','jj'],
'breathy':['b-r-eh1 th-iy','jj'],
'bred':['b-r-eh1-d','vbn vbd'],
'breed':['b-r-iy1-d','nn vb vbp'],
'breeder':['b-r-iy1 d-er','nn'],
'breeding':['b-r-iy1 d-ih-ng','vbg nn'],
'breeds':['b-r-iy1-d-z','nns vbz'],
'breeze':['b-r-iy1-z','nn vb'],
'breezes':['b-r-iy1 z-ih-z','nns'],
'breezy':['b-r-iy1 z-iy','jj'],
'brevity':['b-r-eh1 v-ah t-iy','nn'],
'brew':['b-r-uw1','nn vb'],
'brewed':['b-r-uw1-d','vbn'],
'brewer':['b-r-uw1 er','nn'],
'brewery':['b-r-uw1 er iy','nn'],
'brewing':['b-r-uw1 ih-ng','nn vbg'],
'brews':['b-r-uw1-z','vbz nns'],
'bribe':['b-r-ay1-b','nn vb'],
'bribed':['b-r-ay1-b-d','vbd vbn'],
'bribery':['b-r-ay1 b-er iy','nn'],
'bribes':['b-r-ay1-b-z','nns'],
'brick':['b-r-ih1-k','nn'],
'bricklayers':['b-r-ih1-k l-ey er-z','nns'],
'bridal':['b-r-ay1 d-ah-l','jj'],
'bride':['b-r-ay1-d','nn'],
'bridegroom':['b-r-ay1-d g-r-uw-m','nn'],
'bridesmaids':['b-r-ay1-d-z m-ey-d-z','nns'],
'bridge':['b-r-ih1-jh','nn jj vb'],
'bridgehead':['b-r-ih1-jh hh-eh-d','nn'],
'bridges':['b-r-ih1 jh-ah-z','nns vbz'],
'bridging':['b-r-ih1 jh-ih-ng','vbg jj'],
'bridle':['b-r-ay1 d-ah-l','nn'],
'bridled':['b-r-ay1 d-ah-l-d','vbn'],
'brie':['b-r-iy1','nn'],
'brief':['b-r-iy1-f','jj nn vb'],
'briefcase':['b-r-iy1-f k-ey-s','nn'],
'briefcases':['b-r-iy1-f k-ey s-ih-z','nns'],
'briefed':['b-r-iy1-f-t','vbn vbd'],
'briefer':['b-r-iy1 f-er','jjr'],
'briefing':['b-r-iy1 f-ih-ng','nn vbg'],
'briefly':['b-r-iy1-f l-iy','nn rb'],
'briefs':['b-r-iy1-f-s','nns'],
'brig':['b-r-ih1-g','nn'],
'brigade':['b-r-ah g-ey1-d','nn'],
'brigadier':['b-r-ih g-ah d-ih1-r','nn'],
'bright':['b-r-ay1-t','jj rb'],
'brighten':['b-r-ay1 t-ah-n','vb'],
'brightened':['b-r-ay1 t-ah-n-d','vbd vbn'],
'brightening':['b-r-ay1 t-ah-n ih-ng','vbg'],
'brightens':['b-r-ay1 t-ah-n-z','vbz'],
'brighter':['b-r-ay1 t-er','jjr'],
'brightest':['b-r-ay1 t-ah-s-t','jjs'],
'brightly':['b-r-ay1-t l-iy','rb'],
'brightness':['b-r-ay1-t n-ah-s','nn'],
'brilliance':['b-r-ih1-l y-ah-n-s','nn'],
'brilliant':['b-r-ih1-l y-ah-n-t','jj'],
'brilliantly':['b-r-ih1-l y-ah-n-t l-iy','rb'],
'brim':['b-r-ih1-m','nn'],
'brimmed':['b-r-ih1-m-d','vbd vbn'],
'brimming':['b-r-ih1 m-ih-ng','vbg'],
'brindle':['b-r-ih1-n d-ah-l','nn'],
'brine':['b-r-ay1-n','nn'],
'bring':['b-r-ih1-ng','vb vbp'],
'bringing':['b-r-ih1 ng-ih-ng','vbg nn'],
'brings':['b-r-ih1-ng-z','vbz'],
'brink':['b-r-ih1-ng-k','nn'],
'brinkmanship':['b-r-ih1-ng-k m-ah-n sh-ih-p','nn'],
'brinksmanship':['b-r-ih1-ng-k-s m-ah-n sh-ih-p','nn'],
'brisk':['b-r-ih1-s-k','jj'],
'brisker':['b-r-ih1 s-k-er','jjr'],
'briskly':['b-r-ih1-s-k l-iy','rb'],
'bristle':['b-r-ih1 s-ah-l','vbp vb nn'],
'bristled':['b-r-ih1 s-ah-l-d','vbd'],
'bristles':['b-r-ih1 s-ah-l-z','vbz nns'],
'bristling':['b-r-ih1 s-ah-l ih-ng','vbg jj'],
'brittle':['b-r-ih1 t-ah-l','jj'],
'broach':['b-r-ow1-ch','vb'],
'broached':['b-r-ow1-ch-t','vbn'],
'broad':['b-r-ao1-d','jj'],
'broadcast':['b-r-ao1-d k-ae-s-t','nn jj vb vbd vbn'],
'broadcaster':['b-r-ao1-d k-ae s-t-er','nn'],
'broadcasting':['b-r-ao1-d k-ae s-t-ih-ng','nn vbg'],
'broadcasts':['b-r-ao1-d k-ae-s-t-s','nns vbz'],
'broaden':['b-r-ao1 d-ah-n','vb vbp'],
'broadened':['b-r-ao1 d-ah-n-d','vbn vbd'],
'broadening':['b-r-ao1 d-ah-n ih-ng','vbg nn'],
'broadens':['b-r-ao1 d-ah-n-z','vbz'],
'broader':['b-r-ao1 d-er','jjr'],
'broadest':['b-r-ao1 d-ih-s-t','jjs'],
'broadly':['b-r-ao1-d l-iy','rb'],
'broadside':['b-r-ao1-d s-ay-d','jj nn rb'],
'brocade':['b-r-ow k-ey1-d','nn'],
'broccoli':['b-r-aa1 k-ah l-iy','nns'],
'brochure':['b-r-ow sh-uh1-r','nn'],
'broil':['b-r-oy1-l','nn vb'],
'broiled':['b-r-oy1-l-d','vbn'],
'broiler':['b-r-oy1 l-er','nn'],
'broiling':['b-r-oy1 l-ih-ng','vbg'],
'brok':['b-r-aa1-k','vbd'],
'broke':['b-r-ow1-k','vbd vbn jj rb vb'],
'broken':['b-r-ow1 k-ah-n','vbn jj'],
'broker':['b-r-ow1 k-er','nn'],
'brokerage':['b-r-ow1 k-er ih-jh','nn'],
'brokered':['b-r-ow1 k-er-d','jj vbd vbn'],
'brokering':['b-r-ow1 k-er ih-ng','vbg nn'],
'bromides':['b-r-ow1 m-ay-d-z','nns'],
'bronc':['b-r-aa1-ng-k','nn'],
'bronchial':['b-r-aa1-n ch-iy ah-l','jj'],
'bronchitis':['b-r-aa-ng k-ay1 t-ah-s','nn'],
'bronze':['b-r-aa1-n-z','nn jj'],
'bronzed':['b-r-aa1-n-z-d','jj'],
'bronzes':['b-r-aa1-n z-ah-z','nns'],
'brooch':['b-r-uw1-ch','nn'],
'brood':['b-r-uw1-d','nn jj vb'],
'brooded':['b-r-uw1 d-ih-d','vbd'],
'brooding':['b-r-uw1 d-ih-ng','vbg jj nn'],
'broody':['b-r-uw1 d-iy','jj'],
'brook':['b-r-uh1-k','nn'],
'broom':['b-r-uw1-m','nn'],
'broth':['b-r-ao1-th','nn'],
'brothel':['b-r-aa1 th-ah-l','nn'],
'brother':['b-r-ah1 dh-er','nn'],
'brotherhood':['b-r-ah1 dh-er hh-uh-d','nn'],
'brotherly':['b-r-ah1 dh-er l-iy','jj'],
'brought':['b-r-ao1-t','vbn vbd'],
'brouhaha':['b-r-uw1 hh-aa hh-aa','nn'],
'brow':['b-r-aw1','nn'],
'browbeat':['b-r-aw1 b-iy-t','vb'],
'brown':['b-r-aw1-n','jj nn vb'],
'browned':['b-r-aw1-n-d','vbn'],
'brownies':['b-r-aw1 n-iy-z','nns'],
'browning':['b-r-aw1 n-ih-ng','vbg'],
'brownish':['b-r-aw1 n-ih-sh','jj'],
'brownouts':['b-r-aw1-n aw-t-s','nns'],
'browse':['b-r-aw1-z','vb'],
'browsing':['b-r-aw1 z-ih-ng','vbg nn'],
'bruise':['b-r-uw1-z','nn vb'],
'bruised':['b-r-uw1-z-d','vbn jj'],
'bruises':['b-r-uw1 z-ah-z','nns'],
'bruising':['b-r-uw1 z-ih-ng','jj vbg'],
'brunch':['b-r-ah1-n-ch','nn'],
'brunette':['b-r-uw n-eh1-t','jj nn'],
'brunt':['b-r-ah1-n-t','nn'],
'brush':['b-r-ah1-sh','nn jj vb vbp'],
'brushed':['b-r-ah1-sh-t','vbd vbn jj'],
'brushes':['b-r-ah1 sh-ih-z','nns'],
'brushing':['b-r-ah1 sh-ih-ng','vbg nn'],
'brushwork':['b-r-ah1-sh w-er-k','nn'],
'brushy':['b-r-ah1 sh-iy','jj'],
'brusquely':['b-r-ah1-s-k l-iy','rb'],
'brutal':['b-r-uw1 t-ah-l','jj'],
'brutality':['b-r-uw t-ae1 l-ah t-iy','nn'],
'brutalized':['b-r-uw1 t-ah l-ay-z-d','vbn jj'],
'brutally':['b-r-uw1 t-ah l-iy','rb'],
'brute':['b-r-uw1-t','nn jj'],
'brutish':['b-r-uw1 t-ih-sh','jj'],
'bubble':['b-ah1 b-ah-l','nn vbp vb'],
'bubbled':['b-ah1 b-ah-l-d','vbn vbd'],
'bubbles':['b-ah1 b-ah-l-z','nns vbz'],
'bubbling':['b-ah1 b-ah-l ih-ng','vbg nn'],
'bubbly':['b-ah1 b-l-iy','jj nn'],
'buccaneers':['b-ah k-ah n-iy1-r-z','nns'],
'buck':['b-ah1-k','nn vbp vb'],
'bucked':['b-ah1-k-t','vbd vbn'],
'bucket':['b-ah1 k-ah-t','nn'],
'bucking':['b-ah1 k-ih-ng','vbg'],
'buckle':['b-ah1 k-ah-l','vb nn'],
'buckled':['b-ah1 k-ah-l-d','vbd vbn jj'],
'buckles':['b-ah1 k-ah-l-z','nns'],
'buckling':['b-ah1 k-l-ih-ng','vbg nn'],
'bucks':['b-ah1-k-s','nns vbz'],
'buckshot':['b-ah1-k sh-aa-t','nn'],
'buckskin':['b-ah1-k s-k-ih-n','nn'],
'buckwheat':['b-ah1-k w-iy-t','nn'],
'bucolic':['b-y-uw k-aa1 l-ih-k','jj'],
'bud':['b-ah1-d','nn vb'],
'budding':['b-ah1 d-ih-ng','vbg jj'],
'buddy':['b-ah1 d-iy','nn'],
'budge':['b-ah1-jh','vb'],
'budged':['b-ah1-jh-d','vbd vbn'],
'budget':['b-ah1 jh-ih-t','nn vb'],
'budgetary':['b-ah1 jh-ih t-eh r-iy','jj'],
'budgeted':['b-ah1 jh-ih t-ah-d','vbn vbd'],
'budgeteers':['b-ah jh-ih t-ih1-r-z','nns'],
'budgeting':['b-ah1 jh-ih t-ih-ng','nn vbg'],
'budgets':['b-ah1 jh-ih-t-s','nns'],
'buds':['b-ah1-d-z','nns'],
'buff':['b-ah1-f','nn jj'],
'buffalo':['b-ah1 f-ah l-ow','nn nns'],
'buffer':['b-ah1 f-er','nn vb'],
'buffered':['b-ah1 f-er-d','vbn vbd'],
'buffet':['b-ah1 f-ah-t','nn vb vbp'],
'buffeted':['b-ah f-ey1-d','vbn'],
'buffets':['b-ah1 f-ah-t-s','nns'],
'buffoon':['b-ah f-uw1-n','nn'],
'bug':['b-ah1-g','nn vbp'],
'bugaboo':['b-ah1 g-ah b-uw','nn'],
'bugeyed':['b-ah1 g-ay-d','jj'],
'bugged':['b-ah1-g-d','vbn vbd'],
'buggers':['b-ah1 g-er-z','nns'],
'bugging':['b-ah1 g-ih-ng','nn'],
'buggy':['b-ah1 g-iy','nn'],
'bugle':['b-y-uw1 g-ah-l','nn'],
'bugs':['b-ah1-g-z','nns vbz'],
'build':['b-ih1-l-d','vb vbn vbp nn'],
'builder':['b-ih1-l d-er','nn'],
'building':['b-ih1-l d-ih-ng','nn vbg'],
'builds':['b-ih1-l-d-z','vbz'],
'buildup':['b-ih1-l-d ah-p','nn'],
'built':['b-ih1-l-t','vbn jj vbd'],
'bulb':['b-ah1-l-b','nn'],
'bulge':['b-ah1-l-jh','nn vb'],
'bulged':['b-ah1-l-jh-d','vbd'],
'bulging':['b-ah1-l jh-ih-ng','vbg jj'],
'bulk':['b-ah1-l-k','nn jj'],
'bulkhead':['b-ah1-l-k hh-eh-d','nn'],
'bulky':['b-ah1-l k-iy','jj'],
'bull':['b-uh1-l','nn vb'],
'bulldog':['b-uh1-l d-ao-g','jj'],
'bulldoze':['b-uh1-l d-ow-z','vb'],
'bulldozed':['b-uh1-l d-ow-z-d','vbn'],
'bulldozer':['b-uh1-l d-ow z-er','nn'],
'bullet':['b-uh1 l-ah-t','nn'],
'bulletin':['b-uh1 l-ih t-ah-n','nn vb'],
'bulletins':['b-uh1 l-ah t-ah-n-z','nns vbz'],
'bulletproof':['b-uh1 l-ah-t p-r-uw-f','jj'],
'bullfighter':['b-uh1-l f-ay t-er','nn'],
'bullhorn':['b-uh1-l hh-ao-r-n','nn'],
'bullied':['b-uh1 l-iy-d','vbd vbn'],
'bullies':['b-uh1 l-iy-z','vbz nns'],
'bullion':['b-uh1 l-y-ah-n','nn'],
'bullish':['b-uh1 l-ih-sh','jj'],
'bullishly':['b-uh1 l-ih-sh l-iy','rb'],
'bullishness':['b-uh1 l-ih-sh n-ah-s','nn'],
'bullock':['b-uh1 l-ah-k','nn'],
'bulls':['b-uh1-l-z','nns'],
'bully':['b-uh1 l-iy','nn vbp vb'],
'bullying':['b-uh1 l-iy ih-ng','vbg'],
'bulwark':['b-uh1-l w-er-k','nn'],
'bum':['b-ah1-m','nn vbp jj'],
'bumble':['b-ah1-m b-ah-l','vb vbp'],
'bumbling':['b-ah1-m b-ah-l ih-ng','jj vbg'],
'bummed':['b-ah1-m-d','vbn'],
'bump':['b-ah1-m-p','vb nn vbp'],
'bumped':['b-ah1-m-p-t','vbd vbn'],
'bumper':['b-ah1-m p-er','nn jj jjr'],
'bumping':['b-ah1-m p-ih-ng','vbg'],
'bumps':['b-ah1-m-p-s','nns vbz'],
'bumpy':['b-ah1-m p-iy','jj'],
'bun':['b-ah1-n','nn'],
'bunch':['b-ah1-n-ch','nn'],
'bunched':['b-ah1-n-ch-t','vbn vbd'],
'bunching':['b-ah1-n ch-ih-ng','vbg'],
'bundle':['b-ah1-n d-ah-l','nn vb'],
'bundled':['b-ah1-n d-ah-l-d','vbn'],
'bundles':['b-ah1-n d-ah-l-z','nns'],
'bundling':['b-ah1-n d-ah-l ih-ng','vbg'],
'bungalow':['b-ah1-ng g-ah l-ow','nn'],
'bungled':['b-ah1-ng g-ah-l-d','vbd vbn'],
'bungling':['b-ah1-ng g-ah-l ih-ng','vbg'],
'bunk':['b-ah1-ng-k','nn'],
'bunker':['b-ah1-ng k-er','nn'],
'bunny':['b-ah1 n-iy','nn'],
'bunt':['b-ah1-n-t','nn vb'],
'buoy':['b-uw1 iy','vb'],
'buoyancy':['b-oy1 ah-n s-iy','nn'],
'buoyant':['b-oy1 ah-n-t','jj'],
'buoyed':['b-uw1 iy-d','vbn vbd'],
'buoying':['b-oy1 ih-ng','vbg'],
'buoys':['b-uw1 iy-z','nns vbz'],
'burden':['b-er1 d-ah-n','nn vb vbp'],
'burdened':['b-er1 d-ah-n-d','vbn vbd'],
'burdening':['b-er1 d-ah-n ih-ng','vbg'],
'burdens':['b-er1 d-ah-n-z','nns vbz'],
'burdensome':['b-er1 d-ah-n s-ah-m','jj'],
'bureau':['b-y-uh1 r-ow','nn'],
'bureaucracy':['b-y-uh r-aa1 k-r-ah s-iy','nn'],
'bureaucrat':['b-y-uh1 r-ah k-r-ae-t','nn'],
'bureaucratic':['b-y-uh r-ah k-r-ae1 t-ih-k','jj'],
'bureaus':['b-y-uh1 r-ow-z','nn nns'],
'burgeoned':['b-er1 jh-ah-n-d','vbd vbn'],
'burgeoning':['b-er1 jh-ah n-ih-ng','vbg jj'],
'burger':['b-er1 g-er','nn'],
'burglar':['b-er1 g-l-er','nn'],
'burglary':['b-er1 g-l-er iy','nn'],
'burgundy':['b-er1 g-ah-n d-iy','nn'],
'burial':['b-eh1 r-iy ah-l','nn'],
'buried':['b-eh1 r-iy-d','vbn jj vbd'],
'burl':['b-er1-l','nn'],
'burlap':['b-er1 l-ae-p','nn'],
'burlesque':['b-er l-eh1-s-k','jj'],
'burley':['b-er1 l-iy','nn'],
'burly':['b-er1 l-iy','jj'],
'burn':['b-er1-n','vb vbp nn'],
'burne':['b-er1-n','vb'],
'burned':['b-er1-n-d','vbn jj vbd'],
'burner':['b-er1 n-er','nn'],
'burning':['b-er1 n-ih-ng','vbg jj nn'],
'burnished':['b-er1 n-ih-sh-t','vbn'],
'burnout':['b-er1-n aw-t','nn'],
'burns':['b-er1-n-z','nns vbz'],
'burnt':['b-er1-n-t','vbn jj'],
'burping':['b-er1 p-ih-ng','vbg'],
'burr':['b-er1','nn'],
'burrow':['b-er1 ow','nn vb'],
'burrowing':['b-er1 ow ih-ng','vbg'],
'burrows':['b-er1 ow-z','nns'],
'burst':['b-er1-s-t','nn vbd vbn vbp vb'],
'bursting':['b-er1 s-t-ih-ng','vbg'],
'bursts':['b-er1-s-t-s','nns vbz'],
'bury':['b-eh1 r-iy','vb vbp'],
'burying':['b-eh1 r-iy ih-ng','vbg'],
'bus':['b-ah1-s','nn'],
'busboy':['b-ah1-s b-oy','nn'],
'bused':['b-ah1-s-t','vbn'],
'bush':['b-uh1-sh','nn'],
'bushel':['b-uh1 sh-ah-l','nn'],
'bushy':['b-uh1 sh-iy','jj'],
'busied':['b-ih1 z-iy-d','vbd'],
'busier':['b-ih1 z-iy er','jjr'],
'busiest':['b-ih1 z-iy ah-s-t','jjs'],
'busily':['b-ih1 z-ah l-iy','rb'],
'business':['b-ih1-z n-ah-s','nn nnp'],
'businesslike':['b-ih1-z n-ih-s l-ay-k','jj'],
'businessman':['b-ih1-z n-ah-s m-ae-n','nn'],
'businesspeople':['b-ih1-z n-ah-s p-iy1 p-ah-l','nn'],
'businesswoman':['b-ih1-z n-ih-s w-uh m-ah-n','nn'],
'busing':['b-ah1 s-ih-ng','vbg'],
'busload':['b-ah-s l-ow1-d','nn'],
'buss':['b-ah1-s','nn'],
'bust':['b-ah1-s-t','nn jj rb vb'],
'busted':['b-ah1 s-t-ih-d','jj vbd vbn'],
'buster':['b-ah1 s-t-er','nn'],
'bustin':['b-ah1 s-t-ih-n','vbg'],
'busting':['b-ah1 s-t-ih-ng','vbg'],
'bustle':['b-ah1 s-ah-l','nn'],
'bustling':['b-ah1 s-ah l-ih-ng','jj vbg'],
'busts':['b-ah1-s-t-s','nns vbz'],
'busy':['b-ih1 z-iy','jj'],
'but':['b-ah1-t','cc in jj rb'],
'butane':['b-y-uw t-ey1-n','nn'],
'butcher':['b-uh1 ch-er','nn'],
'butchered':['b-uh1 ch-er-d','vbn'],
'butchering':['b-uh1 ch-er ih-ng','nn'],
'butchers':['b-uh1 ch-er-z','nns vbz'],
'butchery':['b-uh1 ch-er iy','nn'],
'butler':['b-ah1-t l-er','nn'],
'butt':['b-ah1-t','nn cc vbp vb'],
'butter':['b-ah1 t-er','nn'],
'butterfat':['b-ah1 t-er f-ae-t','nn'],
'butterfly':['b-ah1 t-er f-l-ay','nn'],
'buttery':['b-ah1 t-er iy','jj'],
'butting':['b-ah1 t-ih-ng','vbg'],
'buttocks':['b-ah1 t-ah-k-s','nns'],
'button':['b-ah1 t-ah-n','nn vb'],
'buttoned':['b-ah1 t-ah-n-d','vbn'],
'buttonholes':['b-ah1 t-ah-n hh-ow-l-z','nns'],
'buttons':['b-ah1 t-ah-n-z','nns'],
'buttress':['b-ah1 t-r-ah-s','vb'],
'buttressed':['b-ah1 t-r-ah-s-t','vbn'],
'buttresses':['b-ah1 t-r-ah s-ah-z','nns vbz'],
'butts':['b-ah1-t-s','nns vbz'],
'buxom':['b-ah1-k s-ah-m','jj'],
'buy':['b-ay1','vb vbp nn jj'],
'buyback':['b-ay1 b-ae-k','jj nn'],
'buyer':['b-ay1 er','nn'],
'buying':['b-ay1 ih-ng','vbg nn jj'],
'buyout':['b-ay1 aw-t','nn'],
'buys':['b-ay1-z','vbz nns'],
'buzz':['b-ah1-z','nn jj vb'],
'buzzed':['b-ah1-z-d','vbd'],
'buzzer':['b-ah1 z-er','nn'],
'buzzing':['b-ah1 z-ih-ng','vbg jj'],
'buzzword':['b-ah1-z w-er-d','nn'],
'by':['b-ay1','in rb rp'],
'bye':['b-ay1','vb'],
'bygone':['b-ay1 g-ao-n','jj'],
'bylaw':['b-ay1 l-ao','nn'],
'byline':['b-ay1 l-ay-n','nn'],
'bypass':['b-ay1 p-ae-s','vb nn'],
'bypassed':['b-ay1 p-ae-s-t','vbn vbd'],
'bypassing':['b-ay1 p-ae s-ih-ng','vbg'],
'byproduct':['b-ay1 p-r-aa d-ah-k-t','nn'],
'bystander':['b-ay1 s-t-ae-n d-er','nn'],
'bytes':['b-ay1-t-s','nns'],
'byways':['b-ay1 w-ey-z','nns'],
'byword':['b-ay1 w-er-d','nn'],
'byzantine':['b-ih1 z-ah-n t-ay-n','jj'],
'c':['s-iy1','nn ls nnp nns'],
'cab':['k-ae1-b','nn'],
'cabal':['k-ah b-aa1-l','nn'],
'cabana':['k-ah b-ae1 n-ah','nn'],
'cabaret':['k-ae b-er ey1','nn'],
'cabbage':['k-ae1 b-ih-jh','nn'],
'cabdriver':['k-ae1-b d-r-ay v-er','nn'],
'cabin':['k-ae1 b-ah-n','nn'],
'cabinet':['k-ae1 b-ah n-ah-t','nn jj'],
'cable':['k-ey1 b-ah-l','nn vbp jj'],
'cabled':['k-ey1 b-ah-l-d','vbd'],
'cacao':['k-ah k-ey1 ow','nn'],
'cache':['k-ae1-sh','nn'],
'cachet':['k-ae1 sh-ey','nn'],
'cacophony':['k-ae k-aa1 f-ah n-iy','nn'],
'cactus':['k-ae1-k t-ah-s','nn'],
'cadaver':['k-ah d-ae1 v-er','nn'],
'cadence':['k-ey1 d-ah-n-s','nn'],
'cadet':['k-ah d-eh1-t','nn'],
'cadmium':['k-ae1-d m-iy ah-m','nn'],
'cadre':['k-ae1 d-r-iy','nn'],
'cafe':['k-ah f-ey1','nn'],
'cafeteria':['k-ae f-ah t-ih1 r-iy ah','nn'],
'caffeine':['k-ae f-iy1-n','nn'],
'cage':['k-ey1-jh','nn'],
'caged':['k-ey1-jh-d','vbn'],
'cagey':['k-ey1 jh-iy','jj'],
'cahoots':['k-ah hh-uw1-t-s','nns'],
'cain':['k-ey1-n','md'],
'cajole':['k-ah jh-ow1-l','vb'],
'cajun':['k-ey1 jh-ah-n','jj'],
'cake':['k-ey1-k','nn'],
'calamitous':['k-ah l-ae1 m-ah t-ah-s','jj'],
'calamity':['k-ah l-ae1 m-ah t-iy','nn'],
'calcified':['k-ae1-l s-ah f-ay-d','vbd'],
'calcium':['k-ae1-l s-iy ah-m','nn'],
'calculate':['k-ae1-l k-y-ah l-ey-t','vb vbp'],
'calculated':['k-ae1-l k-y-ah l-ey t-ah-d','vbn vbd jj'],
'calculates':['k-ae1-l k-y-ah l-ey-t-s','vbz'],
'calculating':['k-ae1-l k-y-ah l-ey t-ih-ng','vbg jj nn'],
'calculation':['k-ae-l k-y-ah l-ey1 sh-ah-n','nn'],
'calculator':['k-ae1-l k-y-ah l-ey t-er','nn'],
'calculus':['k-ae1-l k-y-ah l-ah-s','nn'],
'calendar':['k-ae1 l-ah-n d-er','nn'],
'calf':['k-ae1-f','nn'],
'caliber':['k-ae1 l-ah b-er','nn'],
'calibrated':['k-ae1 l-ah b-r-ey t-ah-d','vbn vbd'],
'calibration':['k-ae l-ah b-r-ey1 sh-ah-n','nn'],
'calico':['k-ae1 l-ah k-ow','jj nn'],
'calipers':['k-ae1 l-ah p-er-z','nns'],
'calisthenics':['k-ae l-ah-s th-eh1 n-ih-k-s','nns'],
'call':['k-ao1-l','vb jj nn vbp'],
'callable':['k-ao1 l-ah b-ah-l','jj'],
'called':['k-ao1-l-d','vbn vbd vb'],
'caller':['k-ao1 l-er','nn'],
'calligraphy':['k-ah l-ih1 g-r-ah f-iy','nn'],
'calling':['k-ao1 l-ih-ng','vbg nn'],
'callous':['k-ae1 l-ah-s','jj'],
'calloused':['k-ae1 l-ah-s-t','jj'],
'callously':['k-ae1 l-ah-s l-iy','rb'],
'callousness':['k-ae1 l-ah-s n-ah-s','nn'],
'calls':['k-ao1-l-z','vbz nns nn'],
'calm':['k-aa1-m','jj nn vb'],
'calmed':['k-aa1-m-d','vbd vbn'],
'calmer':['k-aa1 m-er','jjr'],
'calming':['k-aa1 m-ih-ng','vbg jj'],
'calmly':['k-aa1-m l-iy','rb'],
'calmness':['k-aa1-m n-ah-s','nn'],
'caloric':['k-ah l-ao1 r-ih-k','jj'],
'calorie':['k-ae1 l-er iy','nn'],
'calving':['k-ae1 v-ih-ng','vbg nn'],
'calypso':['k-ah l-ih1-p s-ow','nn'],
'camaraderie':['k-aa m-er aa1 d-er iy','nn'],
'camcorder':['k-ae1-m k-ao-r d-er','nn'],
'came':['k-ey1-m','vbd'],
'camel':['k-ae1 m-ah-l','nn'],
'cameo':['k-ae1 m-iy ow','nn jj'],
'camera':['k-ae1 m-er ah','nn'],
'cameraman':['k-ae1 m-er ah m-ah-n','nn'],
'camouflage':['k-ae1 m-ah f-l-aa-zh','nn vb'],
'camouflaged':['k-ae1 m-ah f-l-aa-zh-d','vbn jj'],
'camp':['k-ae1-m-p','nn vb'],
'campaign':['k-ae-m p-ey1-n','nn vb'],
'campaigned':['k-ae-m p-ey1-n-d','vbd vbn'],
'campaigners':['k-ae-m p-ey1 n-er-z','nns'],
'campaigning':['k-ae-m p-ey1 n-ih-ng','vbg nn'],
'campaigns':['k-ae-m p-ey1-n-z','nns'],
'camped':['k-ae1-m-p-t','vbd vbn'],
'camper':['k-ae1-m p-er','nn'],
'campfire':['k-ae1-m-p f-ay er','nn'],
'campground':['k-ae1-m-p g-r-aw-n-d','nn'],
'camping':['k-ae1-m p-ih-ng','nn vbg'],
'camps':['k-ae1-m-p-s','nns'],
'campsites':['k-ae1-m-p s-ay-t-s','nns'],
'campus':['k-ae1-m p-ah-s','nn'],
'can':['k-ae1-n','md nn vb'],
'canal':['k-ah n-ae1-l','nn'],
'canard':['k-ah n-aa1-r-d','nn'],
'cancel':['k-ae1-n s-ah-l','vb nn'],
'canceled':['k-ae1-n s-ah-l-d','vbn vbd jj'],
'canceling':['k-ae1-n s-ah l-ih-ng','vbg'],
'cancellation':['k-ae-n s-ah l-ey1 sh-ah-n','nn'],
'cancelled':['k-ae1-n s-ah-l-d','vbn'],
'cancelling':['k-ae1-n s-ah l-ih-ng','vbg'],
'cancels':['k-ae1-n s-ah-l-z','vbz'],
'cancer':['k-ae1-n s-er','nn'],
'cancerous':['k-ae1-n s-er ah-s','jj'],
'candid':['k-ae1-n d-ah-d','jj'],
'candidacy':['k-ae1-n d-ih d-ah s-iy','nn'],
'candidate':['k-ae1-n d-ah d-ey-t','nn'],
'candidly':['k-ae1-n d-ih-d l-iy','rb'],
'candle':['k-ae1-n d-ah-l','nn'],
'candlelight':['k-ae1-n d-ah-l l-ay-t','nn'],
'candor':['k-ae1-n d-er','nn'],
'candy':['k-ae1-n d-iy','nn'],
'cane':['k-ey1-n','nn'],
'canine':['k-ey1 n-ay-n','jj nn'],
'caning':['k-ey1 n-ih-ng','nn'],
'canister':['k-ae1 n-ah s-t-er','nn'],
'canker':['k-ae1-ng k-er','nn'],
'canned':['k-ae1-n-d','jj vbn'],
'cannery':['k-ae1 n-er iy','nn'],
'cannibalism':['k-ae1 n-ah b-ah l-ih z-ah-m','nn'],
'cannibalize':['k-ae1 n-ah b-ah l-ay-z','vb'],
'cannibalizing':['k-ae1 n-ah b-ah l-ay z-ih-ng','vbg'],
'canning':['k-ae1 n-ih-ng','nn vbg jj'],
'cannister':['k-ae1 n-ih s-t-er','nn'],
'cannon':['k-ae1 n-ah-n','nn nns'],
'cannot':['k-ae1 n-aa-t','md'],
'canny':['k-ae1 n-iy','jj'],
'canoe':['k-ah n-uw1','nn vbp'],
'canon':['k-ae1 n-ah-n','nn'],
'canopy':['k-ae1 n-ah p-iy','nn'],
'cans':['k-ae1-n-z','nns vbz'],
'cant':['k-ae1-n-t','nn'],
'cantaloupe':['k-ae1-n t-ah l-ow-p','nn'],
'canteen':['k-ae-n t-iy1-n','nn'],
'canter':['k-ae1-n t-er','nn'],
'cantons':['k-ae1-n t-ah-n-z','nns'],
'canvas':['k-ae1-n v-ah-s','nn'],
'canvass':['k-ae1-n v-ah-s','nn vb'],
'canvassed':['k-ae1-n v-ah-s-t','vbn'],
'canvassers':['k-ae1-n v-ah s-er-z','nns'],
'canvassing':['k-ae1-n v-ah s-ih-ng','vbg'],
'canyon':['k-ae1 n-y-ah-n','nn'],
'cap':['k-ae1-p','nn vbp vb'],
'capability':['k-ey p-ah b-ih1 l-ah t-iy','nn'],
'capable':['k-ey1 p-ah b-ah-l','jj'],
'capacious':['k-ah p-ey1 sh-ah-s','jj'],
'capacitance':['k-ah p-ae1 s-ah t-ah-n-s','nn'],
'capacitor':['k-ah p-ae1 s-ah t-er','nn'],
'capacity':['k-ah p-ae1 s-ah t-iy','nn'],
'cape':['k-ey1-p','nn'],
'caper':['k-ey1 p-er','nn'],
'capillary':['k-ae1 p-ah l-eh r-iy','nn'],
'capital':['k-ae1 p-ah t-ah-l','nn jj'],
'capitalism':['k-ae1 p-ih t-ah l-ih z-ah-m','nn'],
'capitalist':['k-ae1 p-ah t-ah l-ah-s-t','jj nn'],
'capitalistic':['k-ae p-ih t-ah l-ih1 s-t-ih-k','jj'],
'capitalization':['k-ae p-ih t-ah l-ih z-ey1 sh-ah-n','nn'],
'capitalize':['k-ae1 p-ah t-ah l-ay-z','vb'],
'capitalized':['k-ae1 p-ih t-ah l-ay-z-d','vbn jj vbd'],
'capitalizes':['k-ae1 p-ah t-ah l-ay z-ih-z','vbz'],
'capitalizing':['k-ae1 p-ih t-ah l-ay z-ih-ng','vbg'],
'capitol':['k-ae1 p-ih t-ah-l','nn'],
'capitulated':['k-ah p-ih1 ch-ah l-ey t-ih-d','vbd vbn'],
'capitulation':['k-ah p-ih ch-ah l-ey1 sh-ah-n','nn'],
'capo':['k-aa1 p-ow','nn'],
'capped':['k-ae1-p-t','vbd vbn jj'],
'capping':['k-ae1 p-ih-ng','vbg nn'],
'cappuccino':['k-ae p-uw ch-iy1 n-ow','nn'],
'capricious':['k-ah p-r-ih1 sh-ah-s','jj'],
'capriciously':['k-ah p-r-ih1 sh-ih-s l-iy','rb'],
'caps':['k-ae1-p-s','nns vbz'],
'capstan':['k-ae1-p s-t-ah-n','nn'],
'capsule':['k-ae1-p s-ah-l','nn'],
'captain':['k-ae1-p t-ah-n','nn vbp'],
'caption':['k-ae1-p sh-ah-n','nn'],
'captivated':['k-ae1-p t-ih v-ey t-ih-d','vbn vbd jj'],
'captivating':['k-ae1-p t-ih v-ey t-ih-ng','jj vbg'],
'captive':['k-ae1-p t-ih-v','jj nn'],
'captivity':['k-ae-p t-ih1 v-ah t-iy','nn'],
'captors':['k-ae1-p t-er-z','nns'],
'capture':['k-ae1-p ch-er','vb vbp nn'],
'captured':['k-ae1-p ch-er-d','vbn jj vbd'],
'captures':['k-ae1-p ch-er-z','vbz nns'],
'capturing':['k-ae1-p ch-er ih-ng','vbg'],
'car':['k-aa1-r','nn'],
'caramel':['k-eh1 r-ah m-ah-l','nn'],
'carat':['k-eh1 r-ah-t','nn'],
'caravan':['k-ae1 r-ah v-ae-n','nn'],
'caraway':['k-ae1 r-ah w-ey','jj nn'],
'carbide':['k-aa1-r b-ay-d','nn'],
'carbine':['k-aa1-r b-ay-n','nn'],
'carbohydrate':['k-aa-r b-ow hh-ay1 d-r-ey-t','nn'],
'carbon':['k-aa1-r b-ah-n','nn'],
'carbonates':['k-aa1-r b-ah n-ey-t-s','nns'],
'carborundum':['k-aa-r b-er ah1-n d-ah-m','jj'],
'carcass':['k-aa1-r k-ah-s','nn'],
'carcinogen':['k-aa-r s-ih1 n-ah jh-ah-n','nn'],
'carcinogenic':['k-aa-r s-ah n-ow jh-eh1 n-ih-k','jj'],
'carcinoma':['k-aa-r s-ah n-ow1 m-ah','nn'],
'card':['k-aa1-r-d','nn'],
'cardboard':['k-aa1-r-d b-ao-r-d','nn jj'],
'cardholder':['k-aa1-r-d hh-ow-l d-er','nn'],
'cardiac':['k-aa1-r d-iy ae-k','jj'],
'cardigan':['k-aa1-r d-ih g-ah-n','nn'],
'cardinal':['k-aa1-r d-ah n-ah-l','jj nn'],
'cardiologist':['k-aa-r d-iy aa1 l-ah jh-ih-s-t','nn'],
'cardiovascular':['k-aa-r d-iy ow v-ae1 s-k-y-ah l-er','jj'],
'care':['k-eh1-r','nn vb vbp'],
'cared':['k-eh1-r-d','vbd vbn'],
'careen':['k-er iy1-n','vb'],
'careened':['k-er iy1-n-d','vbd'],
'careening':['k-er iy1 n-ih-ng','vbg'],
'career':['k-er ih1-r','nn nnp'],
'careerists':['k-er ih1 r-ih-s-t-s','nns'],
'carefree':['k-eh1-r f-r-iy','jj'],
'careful':['k-eh1-r f-ah-l','jj'],
'carefully':['k-eh1-r f-ah l-iy','rb'],
'caregivers':['k-eh1-r g-ih v-er-z','nns'],
'careless':['k-eh1-r l-ah-s','jj'],
'carelessly':['k-eh1-r l-ah-s l-iy','rb'],
'carelessness':['k-eh1-r l-ah-s n-ah-s','nn'],
'cares':['k-eh1-r-z','vbz nns'],
'caress':['k-er eh1-s','vb'],
'caressing':['k-er eh1 s-ih-ng','vbg nn'],
'caretaker':['k-eh1-r t-ey k-er','nn'],
'cargo':['k-aa1-r g-ow','nn'],
'caribou':['k-eh1 r-ih b-uw','nn nns'],
'caricature':['k-eh1 r-ah k-ah ch-er','nn vb'],
'caricatured':['k-er ih1 k-ah ch-er-d','vbn'],
'caricatures':['k-er ih1 k-ah ch-er-z','nns'],
'caricaturist':['k-eh1 r-ah k-ah ch-er ah-s-t','nn'],
'caring':['k-eh1 r-ih-ng','vbg jj nn'],
'carinii':['k-eh r-iy1 n-iy','nn'],
'carload':['k-aa1-r l-ow-d','nn'],
'carnage':['k-aa1-r n-ih-jh','nn'],
'carnal':['k-aa1-r n-ah-l','jj'],
'carnival':['k-aa1-r n-ah v-ah-l','nn'],
'carnivores':['k-aa1-r n-ah v-ao-r-z','nns'],
'carnivorous':['k-aa-r n-ih1 v-er ah-s','jj'],
'carob':['k-eh1 r-ah-b','nn'],
'carousel':['k-eh1 r-ah s-eh-l','nn'],
'carousing':['k-er aw1 z-ih-ng','nn'],
'carp':['k-aa1-r-p','vbp vb nn'],
'carpenter':['k-aa1-r p-ah-n t-er','nn'],
'carpentry':['k-aa1-r p-ah-n t-r-iy','nn'],
'carpet':['k-aa1-r p-ah-t','nn'],
'carpetbaggers':['k-aa1-r p-ah-t b-ae g-er-z','nns'],
'carpeted':['k-aa1-r p-ah t-ih-d','vbn vbd'],
'carpeting':['k-aa1-r p-ah t-ih-ng','nn'],
'carping':['k-aa1-r p-ih-ng','vbg jj'],
'carport':['k-aa1-r p-ao-r-t','nn'],
'carriage':['k-ae1 r-ih-jh','nn'],
'carried':['k-ae1 r-iy-d','vbd vbn'],
'carrier':['k-ae1 r-iy er','nn'],
'carries':['k-ae1 r-iy-z','vbz nns'],
'carrion':['k-eh1 r-iy ah-n','jj nn'],
'carrot':['k-ae1 r-ah-t','nn'],
'carry':['k-ae1 r-iy','vb nn vbp'],
'carryforwards':['k-eh1 r-iy f-ao-r w-er-d-z','nns'],
'carrying':['k-ae1 r-iy ih-ng','vbg jj nn'],
'carryover':['k-eh1 r-y-ow v-er','nn'],
'cart':['k-aa1-r-t','nn vbp vb'],
'carte':['k-aa1-r-t','nn'],
'carted':['k-aa1-r t-ih-d','vbd vbn'],
'cartel':['k-aa-r t-eh1-l','nn'],
'cartilage':['k-aa1-r t-ah l-ah-jh','nn'],
'carting':['k-aa1-r t-ih-ng','vbg'],
'carton':['k-aa1-r t-ah-n','nn'],
'cartoon':['k-aa-r t-uw1-n','nn'],
'cartoonist':['k-aa-r t-uw1 n-ah-s-t','nn'],
'cartridge':['k-aa1-r t-r-ah-jh','nn'],
'carts':['k-aa1-r-t-s','nns'],
'cartwheels':['k-aa1-r-t w-iy-l-z','nns'],
'carve':['k-aa1-r-v','vb vbp'],
'carved':['k-aa1-r-v-d','vbn jj vbd'],
'carver':['k-aa1-r v-er','nn'],
'carves':['k-aa1-r-v-z','vbz'],
'carving':['k-aa1-r v-ih-ng','vbg'],
'cascade':['k-ae s-k-ey1-d','nn vb'],
'cascaded':['k-ae s-k-ey1 d-ih-d','vbd'],
'cascades':['k-ae s-k-ey1-d-z','vbz'],
'cascading':['k-ae s-k-ey1 d-ih-ng','vbg jj'],
'case':['k-ey1-s','nn vb'],
'casein':['k-ey s-iy1-n','nn'],
'caseload':['k-ey1-s l-ow-d','nn'],
'cases':['k-ey1 s-ah-z','nns vbz'],
'caseworkers':['k-ey1-s w-er k-er-z','nns'],
'cash':['k-ae1-sh','nn vb'],
'cashed':['k-ae1-sh-t','vbd vbn'],
'cashews':['k-ae1 sh-uw-z','nns'],
'cashflow':['k-ae1-sh f-l-ow','nn'],
'cashier':['k-ae sh-ih1-r','nn'],
'cashing':['k-ae1 sh-ih-ng','vbg'],
'cashmere':['k-ae1-zh m-ih-r','nn'],
'casings':['k-ey1 s-ih-ng-z','nns'],
'casino':['k-ah s-iy1 n-ow','nn'],
'casinos':['k-ah s-iy1 n-ow-z','nns vbz'],
'cask':['k-ae1-s-k','nn'],
'casket':['k-ae1 s-k-ah-t','nn'],
'cassette':['k-ah s-eh1-t','nn'],
'cast':['k-ae1-s-t','nn jj vb vbd vbn vbp'],
'caste':['k-ae1-s-t','nn'],
'caster':['k-ae1 s-t-er','nn'],
'castigate':['k-ae1 s-t-ah g-ey-t','vb'],
'castigated':['k-ae1 s-t-ah g-ey t-ih-d','vbn vbd'],
'castigating':['k-ae1 s-t-ah g-ey t-ih-ng','vbg'],
'casting':['k-ae1 s-t-ih-ng','vbg nn'],
'castle':['k-ae1 s-ah-l','nn'],
'castor':['k-ae1 s-t-er','nn'],
'casts':['k-ae1-s-t-s','vbz nns'],
'casual':['k-ae1 zh-ah w-ah-l','jj rb'],
'casually':['k-ae1 zh-ah-w ah l-iy','rb'],
'casualty':['k-ae1 zh-ah w-ah-l t-iy','nn'],
'cat':['k-ae1-t','nn'],
'cataclysmic':['k-ae t-ah k-l-ih1-z m-ih-k','jj'],
'catalog':['k-ae1 t-ah l-ao-g','nn'],
'cataloging':['k-ae1 t-ah l-aa g-ih-ng','vbg'],
'catalogs':['k-ae1 t-ah l-aa-g-z','nns vbz'],
'catalogue':['k-ae1 t-ah l-ao-g','nn'],
'catalogued':['k-ae1 t-ah l-ao-g-d','vbn vbd'],
'catalyst':['k-ae1 t-ah l-ah-s-t','nn'],
'catalytic':['k-ae t-ah l-ih1 t-ih-k','jj'],
'catalyzed':['k-ae1 t-ah l-ay-z-d','vbn'],
'catamaran':['k-ae t-ah m-er ae1-n','nn'],
'catapult':['k-ae1 t-ah p-ah-l-t','vb vbp'],
'catapulted':['k-ae1 t-ah p-ah-l t-ih-d','vbd vbn'],
'catapulting':['k-ae1 t-ah p-ah-l t-ih-ng','vbg'],
'catapults':['k-ae1 t-ah p-ah-l-t-s','vbz'],
'cataract':['k-ae1 t-er ae-k-t-s','nn'],
'catastrophe':['k-ah t-ae1-s t-r-ah f-iy','nn'],
'catastrophic':['k-ae t-ah s-t-r-aa1 f-ih-k','jj'],
'catbird':['k-ae1-t b-er-d','jj'],
'catcalls':['k-ae1-t k-ao-l-z','nns'],
'catch':['k-ae1-ch','vb vbp nn'],
'catchall':['k-ae1 ch-ao-l','nn'],
'catcher':['k-ae1 ch-er','nn'],
'catches':['k-ae1 ch-ah-z','vbz nns'],
'catching':['k-ae1 ch-ih-ng','vbg nn'],
'catchup':['k-ae1 ch-ah-p','jj nn'],
'catchword':['k-ae1-ch w-er-d','nn'],
'catchy':['k-ae1 ch-iy','jj'],
'catechism':['k-ae1 t-ah k-ih z-ah-m','nn'],
'categorical':['k-ae t-ah g-aa1 r-ih k-ah-l','jj'],
'categorically':['k-ae t-ah g-aa1 r-ih-k l-iy','rb'],
'categorize':['k-ae1 t-ah g-er ay-z','vb'],
'categorized':['k-ae1 t-ah g-er ay-z-d','vbn'],
'categorizing':['k-ae1 t-ah g-er ay z-ih-ng','vbg'],
'category':['k-ae1 t-ah g-ao r-iy','nn'],
'cater':['k-ey1 t-er','vbp vb'],
'catered':['k-ey1 t-er-d','vbd jj'],
'caterer':['k-ey1 t-er er','nn'],
'catering':['k-ey1 t-er ih-ng','nn vbg'],
'caterpillar':['k-ae1 t-ah p-ih l-er','nn'],
'caters':['k-ey1 t-er-z','vbz'],
'catfish':['k-ae1-t f-ih-sh','nn'],
'catharsis':['k-ah th-aa1-r s-ah-s','nn'],
'cathartic':['k-ah th-aa1-r t-ih-k','jj'],
'cathedral':['k-ah th-iy1 d-r-ah-l','nn'],
'catheter':['k-ae1 th-ah t-er','nn'],
'cathode':['k-ae1 th-ow-d','nn'],
'catkins':['k-ae1-t k-ah-n-z','nns'],
'catlike':['k-ae1-t l-ay-k','jj'],
'catsup':['k-eh1 ch-ah-p','nn'],
'cattle':['k-ae1 t-ah-l','nns nn'],
'cattlemen':['k-ae1 t-ah-l m-ah-n','nns'],
'catty':['k-ae1 t-iy','jj'],
'caucus':['k-ao1 k-ah-s','nn vb'],
'caucuses':['k-ao1 k-ah s-ih-z','nns'],
'caught':['k-aa1-t','vbn vbd'],
'cauliflower':['k-aa1 l-ah f-l-aw er','nn'],
'causal':['k-ao1 z-ah-l','jj'],
'causative':['k-aa1 z-ah t-ih-v','jj'],
'cause':['k-aa1-z','nn vb vbg vbp'],
'caused':['k-aa1-z-d','vbn vbd'],
'causes':['k-aa1 z-ah-z','nns vbz'],
'causeway':['k-aa1-z w-ey','nn'],
'causing':['k-aa1 z-ih-ng','vbg nn'],
'caustic':['k-aa1 s-t-ih-k','jj'],
'caution':['k-aa1 sh-ah-n','nn vbp vb'],
'cautionary':['k-ao1 sh-ah-n eh r-iy','jj'],
'cautioned':['k-aa1 sh-ah-n-d','vbd vbn'],
'cautioning':['k-ao1 sh-ah-n ih-ng','vbg'],
'cautions':['k-ao1 sh-ah-n-z','vbz nns'],
'cautious':['k-ao1 sh-ah-s','jj'],
'cautiously':['k-ao1 sh-ah-s l-iy','rb'],
'cautiousness':['k-ao1 sh-ah-s n-ah-s','nn'],
'cavalier':['k-ae v-ah l-ih1-r','jj'],
'cavalry':['k-ae1 v-ah-l r-iy','nn'],
'cave':['k-ey1-v','nn vb'],
'caveat':['k-ey1 v-iy ae-t','nn'],
'caved':['k-ey1-v-d','vbd vbn'],
'cavern':['k-ae1 v-er-n','nn'],
'cavernous':['k-ae1 v-er n-ah-s','jj'],
'caves':['k-ey1-v-z','nns'],
'caviar':['k-ae1 v-iy aa-r','nn'],
'cavin':['k-ae1 v-ih-n','vbg'],
'caving':['k-ey1 v-ih-ng','nn'],
'cavity':['k-ae1 v-ah t-iy','nn'],
'cavort':['k-ah v-ao1-r-t','vbp vb'],
'cavorting':['k-ah v-ao1-r t-ih-ng','vbg nn'],
'cayenne':['k-ay eh1-n','nn'],
'cease':['s-iy1-s','vb vbp'],
'ceased':['s-iy1-s-t','vbd vbn'],
'ceasefire':['s-iy1-s f-ay1 er','nn'],
'ceaseless':['s-iy1-s l-ih-s','jj'],
'ceaselessly':['s-iy1-z l-ah-s l-iy','rb'],
'ceases':['s-iy1 s-ih-z','vbz'],
'ceasing':['s-iy1 s-ih-ng','vbg'],
'cedar':['s-iy1 d-er','nn'],
'cede':['s-iy1-d','vb'],
'ceded':['s-iy1 d-ah-d','vbd vbn'],
'ceding':['s-iy1 d-ih-ng','vbg'],
'ceiling':['s-iy1 l-ih-ng','nn'],
'celebrants':['s-eh1 l-ah b-r-ah-n-t-s','nns'],
'celebrate':['s-eh1 l-ah b-r-ey-t','vb vbp'],
'celebrated':['s-eh1 l-ah b-r-ey t-ah-d','vbd jj vbn'],
'celebrates':['s-eh1 l-ah b-r-ey-t-s','vbz'],
'celebrating':['s-eh1 l-ah b-r-ey t-ih-ng','vbg'],
'celebration':['s-eh l-ah b-r-ey1 sh-ah-n','nn'],
'celebrity':['s-ah l-eh1 b-r-ih t-iy','nn'],
'celery':['s-eh1 l-er iy','nn'],
'celestial':['s-ah l-eh1 s-ch-ah-l','jj'],
'cell':['s-eh1-l','nn'],
'cellar':['s-eh1 l-er','nn'],
'cellist':['ch-eh1 l-ah-s-t','nn'],
'cellophane':['s-eh1 l-ah f-ey-n','nn'],
'cellular':['s-eh1-l y-ah l-er','jj'],
'cellulose':['s-eh1-l y-ah l-ow-s','nn'],
'cement':['s-ah m-eh1-n-t','nn vb'],
'cemented':['s-ah m-eh1-n t-ah-d','vbn'],
'cementing':['s-ih m-eh1-n t-ih-ng','vbg'],
'cemetery':['s-eh1 m-ah t-eh r-iy','nn'],
'censor':['s-eh1-n s-er','vbp vb'],
'censored':['s-eh1-n s-er-d','vbn'],
'censors':['s-eh1-n s-er-z','nns vbz'],
'censorship':['s-eh1-n s-er sh-ih-p','nn'],
'censure':['s-eh1-n sh-er','nn'],
'censured':['s-eh1-n sh-er-d','vbd vbn'],
'census':['s-eh1-n s-ah-s','nn'],
'cent':['s-eh1-n-t','nn'],
'centaur':['s-eh1-n t-ao-r','nn'],
'centenarians':['s-eh-n t-ah n-eh1 r-iy ah-n-z','nns'],
'centenary':['s-eh1-n t-ah n-eh r-iy','jj'],
'centennial':['s-eh-n t-eh1 n-iy ah-l','nn jj'],
'center':['s-eh1-n t-er','nn jj rb vb vbp'],
'centered':['s-eh1-n t-er-d','vbn vbd'],
'centerfielder':['s-eh1-n t-er f-iy-l d-er','nn'],
'centerfold':['s-eh1-n t-er f-ow-l-d','nn'],
'centering':['s-eh1-n t-er ih-ng','vbg nn'],
'centerpiece':['s-eh1-n t-er p-iy-s','nn'],
'centers':['s-eh1-n t-er-z','nns vbz'],
'centigrade':['s-eh1-n t-ah g-r-ey-d','jj'],
'centimeter':['s-eh1-n t-ah m-iy t-er','nn'],
'central':['s-eh1-n t-r-ah-l','jj nn'],
'centralization':['s-eh-n t-r-ah l-ih z-ey1 sh-ah-n','nn'],
'centralize':['s-eh1-n t-r-ah l-ay-z','vb'],
'centralized':['s-eh1-n t-r-ah l-ay-z-d','jj vbd vbn'],
'centralizing':['s-eh1-n t-r-ah l-ay z-ih-ng','vbg jj'],
'centrally':['s-eh1-n t-r-ah l-iy','rb'],
'centre':['s-eh1-n t-er','nn'],
'centrex':['s-eh1-n t-r-ah-k-s','nn'],
'centrifugal':['s-eh-n t-r-ih1 f-y-ih g-ah-l','jj'],
'centrifuge':['s-eh1-n t-r-ah f-y-uw-jh','nn'],
'centrist':['s-eh1-n t-r-ih-s-t','jj'],
'centum':['k-eh1-n t-ah-m','nn'],
'century':['s-eh1-n ch-er iy','nn'],
'ceramic':['s-er ae1 m-ih-k','jj'],
'cereal':['s-ih1 r-iy ah-l','nn'],
'cerebral':['s-eh1 r-ah b-r-ah-l','jj'],
'ceremonial':['s-eh r-ah m-ow1 n-iy ah-l','jj'],
'ceremony':['s-eh1 r-ah m-ow n-iy','nn'],
'certain':['s-er1 t-ah-n','jj rb'],
'certainly':['s-er1 t-ah-n l-iy','rb'],
'certainty':['s-er1 t-ah-n t-iy','nn'],
'certificate':['s-er t-ih1 f-ih k-ah-t','nn'],
'certification':['s-er t-ah f-ah k-ey1 sh-ah-n','nn'],
'certified':['s-er1 t-ah f-ay-d','vbn jj'],
'certifies':['s-er1 t-ah f-ay-z','vbz'],
'certify':['s-er1 t-ah f-ay','vb'],
'certifying':['s-er1 t-ah f-ay ih-ng','vbg'],
'cervical':['s-er1 v-ah k-ah-l','jj'],
'cervix':['s-er1 v-ih-k-s','nn'],
'cessation':['s-eh s-ey1 sh-ah-n','nn'],
'chafe':['ch-ey1-f','vbp vb'],
'chafed':['ch-ey1-f-t','vbn'],
'chafes':['ch-ey1-f-s','vbz'],
'chaff':['ch-ae1-f','nn'],
'chafing':['ch-ey1 f-ih-ng','vbg jj'],
'chagrin':['sh-ah g-r-ih1-n','nn'],
'chain':['ch-ey1-n','nn vbp'],
'chained':['ch-ey1-n-d','vbd vbn jj'],
'chair':['ch-eh1-r','nn vb'],
'chaired':['ch-eh1-r-d','vbn jj vbd'],
'chairing':['ch-eh1 r-ih-ng','nn'],
'chairman':['ch-eh1-r m-ah-n','nn'],
'chairmanship':['ch-eh1-r m-ah-n sh-ih-p','nn'],
'chairs':['ch-eh1-r-z','nns vbz'],
'chairwoman':['ch-eh1-r w-uh m-ah-n','nn'],
'chaise':['sh-ey1-z','nn'],
'chalk':['ch-aa1-k','nn vbp vb'],
'chalked':['ch-ao1-k-t','vbn vbd'],
'chalking':['ch-ao1 k-ih-ng','vbg'],
'challenge':['ch-ae1 l-ah-n-jh','nn vb vbp'],
'challenged':['ch-ae1 l-ah-jh-d','vbd vbn'],
'challenger':['ch-ae1 l-ah-n jh-er','nn'],
'challenges':['ch-ae1 l-ah-n jh-ih-z','nns vbz'],
'challenging':['ch-ae1 l-ah-n jh-ih-ng','vbg jj'],
'chamber':['ch-ey1-m b-er','nn'],
'chambered':['ch-ey1-m b-er-d','vbn'],
'chameleon':['ch-ah m-eh1 l-iy ah-n','nn'],
'champ':['ch-ae1-m-p','nn'],
'champagne':['sh-ae-m p-ey1-n','nn'],
'champion':['ch-ae1-m p-iy ah-n','nn jj vb vbp'],
'championed':['ch-ae1-m p-iy ah-n-d','vbn vbd'],
'championing':['ch-ae1-m p-iy ah n-ih-ng','vbg'],
'champions':['ch-ae1-m p-iy ah-n-z','nns vbz'],
'championship':['ch-ae1-m p-iy ah-n sh-ih-p','nn'],
'chance':['ch-ae1-n-s','nn jj vb vbp'],
'chancellor':['ch-ae1-n s-ah l-er','nn'],
'chancery':['ch-ae1-n s-er iy','nn'],
'chances':['ch-ae1-n s-ah-z','nns'],
'chancy':['ch-ae1-n s-iy','jj'],
'chandelier':['sh-ae-n d-ah l-ih1-r','nn'],
'change':['ch-ey1-n-jh','nn vbp vb'],
'changeable':['ch-ey1-n jh-ah b-ah-l','jj'],
'changed':['ch-ey1-n-jh-d','vbn vbd jj'],
'changeover':['ch-ey1-n-jh ow v-er','nn'],
'changes':['ch-ey1-n jh-ah-z','nns vbz'],
'changing':['ch-ey1-n jh-ih-ng','vbg jj nn'],
'channel':['ch-ae1 n-ah-l','nn vb vbp'],
'channeled':['ch-ae1 n-ah-l-d','vbn vbd'],
'channeling':['ch-ae1 n-ah-l ih-ng','vbg'],
'channels':['ch-ae1 n-ah-l-z','nns'],
'chant':['ch-ae1-n-t','nn vb'],
'chanted':['ch-ae1-n t-ih-d','vbd vbn'],
'chanting':['ch-ae1-n t-ih-ng','vbg nn'],
'chants':['ch-ae1-n-t-s','nns'],
'chaos':['k-ey1 aa-s','nn'],
'chaotic':['k-ey aa1 t-ih-k','jj'],
'chap':['ch-ae1-p','nn'],
'chapel':['ch-ae1 p-ah-l','nn'],
'chaperone':['sh-ae1 p-er ow-n','nn'],
'chaplain':['ch-ae1-p l-ah-n','nn'],
'chapter':['ch-ae1-p t-er','nn'],
'char':['ch-aa1-r','vb'],
'character':['k-eh1 r-ih-k t-er','nn'],
'characteristic':['k-eh r-ah-k t-er ih1 s-t-ih-k','jj nn'],
'characteristically':['k-eh r-ah-k t-er ih1 s-t-ih-k l-iy','rb'],
'characterization':['k-eh r-ah-k t-er ih z-ey1 sh-ah-n','nn'],
'characterize':['k-eh1 r-ah-k t-er ay-z','vb vbp'],
'characterized':['k-eh1 r-ah-k t-er ay-z-d','vbn vbd jj'],
'characterizes':['k-eh1 r-ah-k t-er ay z-ah-z','vbz'],
'characterizing':['k-eh1 r-ih-k t-er ay z-ih-ng','vbg'],
'charcoal':['ch-aa1-r k-ow-l','nn'],
'charge':['ch-aa1-r-jh','nn vbp vb'],
'chargeable':['ch-aa1-r jh-ah b-ah-l','jj'],
'charged':['ch-aa1-r-jh-d','vbn vbd jj'],
'charges':['ch-aa1-r jh-ah-z','nns vbz'],
'charging':['ch-aa1-r jh-ih-ng','vbg nn'],
'chariot':['ch-eh1 r-iy ah-t','nn'],
'charisma':['k-er ih1-z m-ah','nn'],
'charismatic':['k-eh r-ih-z m-ae1 t-ih-k','jj'],
'charitable':['ch-ae1 r-ah t-ah b-ah-l','jj'],
'charitably':['ch-eh1 r-ih t-ah b-l-iy','rb'],
'charity':['ch-eh1 r-ih t-iy','nn'],
'charlatans':['sh-aa1-r l-ah t-ah-n-z','nns'],
'charlotte':['sh-aa1-r l-ah-t','nn'],
'charm':['ch-aa1-r-m','nn vb'],
'charmed':['ch-aa1-r-m-d','vbn vbd'],
'charmer':['ch-aa1-r m-er','nn'],
'charming':['ch-aa1-r m-ih-ng','jj'],
'charmingly':['ch-aa1-r m-ih-ng l-iy','rb'],
'charms':['ch-aa1-r-m-z','nns vbz'],
'charred':['ch-aa1-r-d','jj vbn'],
'chart':['ch-aa1-r-t','nn vb vbp'],
'charted':['ch-aa1-r t-ah-d','vbn'],
'charter':['ch-aa1-r t-er','nn jj'],
'chartered':['ch-aa1-r t-er-d','jj vbn vbd'],
'charters':['ch-aa1-r t-er-z','nns vbz'],
'charting':['ch-aa1-r t-ih-ng','nn vbg'],
'chartist':['ch-aa1-r t-ih-s-t','nn'],
'charts':['ch-aa1-r-t-s','nns vbz'],
'chary':['ch-aa1 r-iy','jj'],
'chase':['ch-ey1-s','nn jj vb vbp'],
'chased':['ch-ey1-s-t','vbn vbd'],
'chasers':['ch-ey1 s-er-z','nns'],
'chasing':['ch-ey1 s-ih-ng','vbg nn'],
'chasm':['k-ae1 z-ah-m','nn'],
'chassis':['ch-ae1 s-iy','nn nns'],
'chaste':['ch-ey1-s-t','jj'],
'chastened':['ch-ey1 s-ah-n-d','vbd vbn'],
'chastised':['ch-ae s-t-ay1-z-d','vbd vbn'],
'chastises':['ch-ae s-t-ay1 z-ih-z','vbz'],
'chastity':['ch-ae1 s-t-ah t-iy','nn'],
'chat':['ch-ae1-t','nn vbp vb'],
'chateau':['sh-ae t-ow1','nn'],
'chateaux':['sh-ah t-ow1','nn'],
'chatted':['ch-ae1 t-ah-d','vbd'],
'chatter':['ch-ae1 t-er','nn vb vbp'],
'chattering':['ch-ae1 t-er ih-ng','vbg nn'],
'chatting':['ch-ae1 t-ih-ng','vbg'],
'chatty':['ch-ae1 t-iy','jj'],
'chauffeur':['sh-ow f-er1','nn vb'],
'chauffeured':['sh-ow f-er1-d','vbn'],
'chauffeurs':['sh-ow f-er1-z','nns'],
'chauvinism':['sh-ow1 v-ah n-ih z-ah-m','nn'],
'chauvinistic':['ch-ow v-ah n-ih1 s-t-ih-k','jj'],
'chaw':['ch-ao1','nn'],
'cheap':['ch-iy1-p','jj nn rb'],
'cheapening':['ch-iy1 p-ah n-ih-ng','vbg'],
'cheaper':['ch-iy1 p-er','jjr rbr'],
'cheapest':['ch-iy1 p-ah-s-t','jjs'],
'cheaply':['ch-iy1-p l-iy','rb'],
'cheat':['ch-iy1-t','vb vbp'],
'cheated':['ch-iy1 t-ah-d','vbn jj vbd'],
'cheaters':['ch-iy1 t-er-z','nns'],
'cheating':['ch-iy1 t-ih-ng','nn vbg'],
'cheats':['ch-iy1-t-s','vbz'],
'check':['ch-eh1-k','nn vbp vb'],
'checkbook':['ch-eh1-k b-uh-k','nn'],
'checked':['ch-eh1-k-t','vbn vbd jj'],
'checker':['ch-eh1 k-er','nn'],
'checking':['ch-eh1 k-ih-ng','vbg nn'],
'checklist':['ch-eh1-k l-ih-s-t','nn'],
'checkout':['ch-eh1-k aw-t','nn'],
'checkpoints':['ch-eh1-k p-oy-n-t-s','nns'],
'checks':['ch-eh1-k-s','nns vbz'],
'checkup':['ch-eh1-k ah-p','nn'],
'cheek':['ch-iy1-k','nn'],
'cheekbones':['ch-iy1-k b-ow-n-z','nns'],
'cheeky':['ch-iy1 k-iy','jj'],
'cheer':['ch-ih1-r','nn vb vbp'],
'cheered':['ch-ih1-r-d','vbd vbn'],
'cheerful':['ch-ih1-r f-ah-l','jj'],
'cheerfully':['ch-ih1-r f-ah l-iy','rb'],
'cheering':['ch-ih1 r-ih-ng','vbg'],
'cheerleader':['ch-ih1-r l-iy d-er','nn'],
'cheerleading':['ch-ih1-r l-iy d-ih-ng','nn'],
'cheers':['ch-ih1-r-z','nns vbz'],
'cheery':['ch-ih1 r-iy','jj'],
'cheese':['ch-iy1-z','nn'],
'cheeseburgers':['ch-iy1-z b-er g-er-z','nns'],
'cheesecake':['ch-iy1-z k-ey-k','nn'],
'cheeses':['ch-iy1 z-ih-z','nns'],
'cheesy':['ch-iy1 z-iy','jj'],
'cheetah':['ch-iy1 t-ah','nn'],
'cheetal':['ch-iy1 t-ah-l','jj'],
'chef':['sh-eh1-f','nn'],
'chemical':['k-eh1 m-ah k-ah-l','nn jj'],
'chemically':['k-eh1 m-ah k-l-iy','rb'],
'chemise':['sh-ah m-iy1-z','nn'],
'chemist':['k-eh1 m-ih-s-t','nn'],
'chemistry':['k-eh1 m-ah s-t-r-iy','nn'],
'chemotherapy':['k-iy m-ow th-eh1 r-ah p-iy','nn'],
'cherish':['ch-eh1 r-ih-sh','vb vbp'],
'cherished':['ch-eh1 r-ih-sh-t','vbn jj vbd'],
'cherishes':['ch-eh1 r-ih sh-ih-z','vbz'],
'cherry':['ch-eh1 r-iy','jj nn'],
'cherubs':['ch-eh1 r-ah-b-z','nns'],
'chess':['ch-eh1-s','nn'],
'chest':['ch-eh1-s-t','nn'],
'chestnut':['ch-eh1-s n-ah-t','nn'],
'chew':['ch-uw1','vb vbp'],
'chewed':['ch-uw1-d','vbd vbn'],
'chewing':['ch-uw1 ih-ng','vbg jj nn'],
'chews':['ch-uw1-z','nns'],
'chic':['sh-iy1-k','jj nn'],
'chicago':['sh-ah k-aa1 g-ow','nnp'],
'chicanery':['sh-ih k-ey1 n-er iy','nn'],
'chick':['ch-ih1-k','nn'],
'chicken':['ch-ih1 k-ah-n','nn'],
'chide':['ch-ay1-d','vb vbp'],
'chided':['ch-ay1 d-ih-d','vbn vbd'],
'chides':['ch-ay1-d-z','vbz'],
'chiding':['ch-ay1 d-ih-ng','vbg'],
'chief':['ch-iy1-f','jj nn'],
'chiefdom':['ch-iy1-f d-ah-m','nn'],
'chiefly':['ch-iy1-f l-iy','rb'],
'chieftain':['ch-iy1-f t-ah-n','nn'],
'child':['ch-ay1-l-d','nn'],
'childbearing':['ch-ay1-l-d b-eh r-ih-ng','vbg'],
'childbirth':['ch-ay1-l-d b-er-th','nn'],
'childcare':['ch-ay1-l-d k-eh-r','nn'],
'childhood':['ch-ay1-l-d hh-uh-d','nn'],
'childish':['ch-ay1-l d-ih-sh','jj'],
'childless':['ch-ay1-l-d l-ah-s','jj'],
'childlike':['ch-ay1-l-d l-ay-k','jj'],
'chili':['ch-ih1 l-iy','nn nns'],
'chill':['ch-ih1-l','nn jj vb'],
'chilled':['ch-ih1-l-d','vbn jj vbd'],
'chillier':['ch-ih1 l-iy er','nn'],
'chilling':['ch-ih1 l-ih-ng','vbg jj'],
'chillingly':['ch-ih1 l-ih-ng l-iy','rb'],
'chills':['ch-ih1-l-z','nns vbz'],
'chilly':['ch-ih1 l-iy','jj'],
'chimed':['ch-ay1-m-d','vbd'],
'chimes':['ch-ay1-m-z','vbz nns'],
'chimney':['ch-ih1-m n-iy','nn'],
'chimp':['ch-ih1-m-p','nn'],
'chimpanzees':['ch-ih-m p-ae1-n z-iy-z','nns'],
'chin':['ch-ih1-n','nn vb'],
'china':['ch-ay1 n-ah','nn'],
'chip':['ch-ih1-p','nn'],
'chipped':['ch-ih1-p-t','vbn vbd jj'],
'chipper':['ch-ih1 p-er','jj'],
'chipping':['ch-ih1 p-ih-ng','vbg nn'],
'chiropractor':['k-ay1 r-ah p-r-ae-k t-er','nn'],
'chirping':['ch-er1 p-ih-ng','vbg nn'],
'chirpy':['ch-er1 p-iy','jj'],
'chisel':['ch-ih1 z-ah-l','nn vb'],
'chiseled':['ch-ih1 z-ah-l-d','vbn jj'],
'chisels':['ch-ih1 z-ah-l-z','nns'],
'chit':['ch-ih1-t','nn'],
'chitchat':['ch-ih1-t ch-ae-t','nn'],
'chivalry':['sh-ih1 v-ah-l r-iy','nn'],
'chives':['ch-ay1-v-z','nns'],
'chloride':['k-l-ao1 r-ay-d','nn'],
'chlorine':['k-l-ao1 r-iy-n','nn'],
'chocolate':['ch-ao1-k l-ah-t','nn jj'],
'choice':['ch-oy1-s','nn jj'],
'choicest':['ch-oy1 s-ah-s-t','jjs'],
'choir':['k-w-ay1 er','nn'],
'choke':['ch-ow1-k','vb vbp nn'],
'choked':['ch-ow1-k-t','vbd vbn'],
'choking':['ch-ow1 k-ih-ng','vbg jj nn'],
'cholera':['k-aa1 l-er ah','nn'],
'cholesterol':['k-ah l-eh1-s t-er ao-l','nn'],
'cholla':['ch-aa1 l-ah','nn'],
'chomp':['ch-aa1-m-p','nn vbp'],
'chomping':['ch-aa1-m p-ih-ng','vbg'],
'choose':['ch-uw1-z','vb vbp'],
'chooses':['ch-uw1 z-ah-z','vbz'],
'choosing':['ch-uw1 z-ih-ng','vbg nn'],
'choosy':['ch-uw1 z-iy','jj'],
'chop':['ch-aa1-p','vb nn'],
'chopped':['ch-aa1-p-t','jj vbd vbn'],
'chopper':['ch-aa1 p-er','nn'],
'chopping':['ch-aa1 p-ih-ng','vbg nn'],
'choppy':['ch-aa1 p-iy','jj'],
'chops':['ch-aa1-p-s','nns'],
'chopsticks':['ch-aa1-p s-t-ih-k-s','nns'],
'choral':['k-ao1 r-ah-l','jj'],
'chord':['k-ao1-r-d','nn'],
'chore':['ch-ao1-r','nn'],
'choreographed':['k-ao1 r-iy ah g-r-ae-f-t','vbn jj'],
'choreographer':['k-ao r-iy aa1 g-r-ah f-er','nn'],
'choreography':['k-ao r-iy aa1 g-r-ah f-iy','nn'],
'chortled':['ch-ao1-r t-ah-l-d','vbd vbn'],
'chortles':['ch-ao1-r t-ah-l-z','vbz'],
'chortling':['ch-ao1-r t-ah-l ih-ng','vbg'],
'chorus':['k-ao1 r-ah-s','nn'],
'chose':['ch-ow1-z','vbd'],
'chosen':['ch-ow1 z-ah-n','vbn jj'],
'chow':['ch-aw1','nn'],
'chowder':['ch-aw1 d-er','nn'],
'christen':['k-r-ih1 s-ah-n','vb'],
'christened':['k-r-ih1 s-ah-n-d','vbd vbn'],
'christening':['k-r-ih1 s-ah-n ih-ng','nn'],
'chrome':['k-r-ow1-m','nn'],
'chromium':['k-r-ow1 m-iy ah-m','nn'],
'chromosome':['k-r-ow1 m-ah s-ow-m','nn'],
'chronic':['k-r-aa1 n-ih-k','jj nn'],
'chronically':['k-r-aa1 n-ih k-ah l-iy','rb'],
'chronicle':['k-r-aa1 n-ih k-ah-l','nn'],
'chronicled':['k-r-aa1 n-ih k-ah-l-d','vbd vbn'],
'chronicler':['k-r-aa1 n-ih k-l-er','nn'],
'chronicles':['k-r-aa1 n-ih k-ah-l-z','vbz nns'],
'chronicling':['k-r-aa1 n-ih k-l-ih-ng','vbg'],
'chronological':['k-r-aa n-ah l-aa1 jh-ih k-ah-l','jj'],
'chronologically':['k-r-aa n-ah l-aa1 jh-ih-k l-iy','rb'],
'chronology':['k-r-ah n-aa1 l-ah jh-iy','nn'],
'chrysanthemums':['k-r-ih s-ae1-n th-ah m-ah-m-z','nns'],
'chubby':['ch-ah1 b-iy','jj'],
'chuck':['ch-ah1-k','nn vb'],
'chucked':['ch-ah1-k-t','vbd vbn'],
'chucking':['ch-ah1 k-ih-ng','vbg'],
'chuckle':['ch-ah1 k-ah-l','nn vb vbp'],
'chuckled':['ch-ah1 k-ah-l-d','vbd'],
'chuckles':['ch-ah1 k-ah-l-z','nns vbz'],
'chuckling':['ch-ah1 k-l-ih-ng','vbg'],
'chug':['ch-ah1-g','vbp'],
'chugging':['ch-ah1 g-ih-ng','vbg'],
'chum':['ch-ah1-m','nn'],
'chump':['ch-ah1-m-p','nn'],
'chunk':['ch-ah1-ng-k','nn'],
'chunky':['ch-ah1-ng k-iy','jj'],
'church':['ch-er1-ch','nn nnp'],
'churchgoers':['ch-er1-ch g-ow er-z','nns'],
'churchmen':['ch-er1-ch m-ah-n','nns'],
'churchyard':['ch-er1-ch y-aa-r-d','nn'],
'churn':['ch-er1-n','vb vbp'],
'churned':['ch-er1-n-d','vbd'],
'churning':['ch-er1 n-ih-ng','vbg nn'],
'churns':['ch-er1-n-z','vbz'],
'chute':['sh-uw1-t','nn'],
'chutney':['ch-ah1-t n-iy','nn'],
'chutzpah':['ch-ah1-t s-p-aa','nn'],
'cicadas':['s-ih k-ey1 d-ah-z','nns'],
'cider':['s-ay1 d-er','nn'],
'cigar':['s-ih g-aa1-r','nn'],
'cigarette':['s-ih g-er eh1-t','nn'],
'ciliates':['s-ih1 l-iy ah-t-s','nns'],
'cinch':['s-ih1-n-ch','nn vb'],
'cinder':['s-ih1-n d-er','nn'],
'cinema':['s-ih1 n-ah m-ah','nn'],
'cinematic':['s-ih n-ah m-ae1 t-ih-k','jj'],
'cinematographer':['s-ih n-ih m-ah t-aa1 g-r-ah f-er','nn'],
'cinematography':['s-ih n-ih m-ah t-aa1 g-r-ah f-iy','nn'],
'cipher':['s-ay1 f-er','vb'],
'circle':['s-er1 k-ah-l','nn vb'],
'circled':['s-er1 k-ah-l-d','vbd'],
'circles':['s-er1 k-ah-l-z','nns vbz'],
'circling':['s-er1 k-ah-l ih-ng','vbg'],
'circuit':['s-er1 k-ah-t','nn'],
'circuitous':['s-er k-y-uw1 ih t-ah-s','jj'],
'circuitry':['s-er1 k-ah t-r-iy','nn'],
'circuits':['s-er1 k-ah-t-s','nns vbz'],
'circular':['s-er1 k-y-ah l-er','jj nn'],
'circulate':['s-er1 k-y-ah l-ey-t','vb vbp'],
'circulated':['s-er1 k-y-ah l-ey t-ah-d','vbd vbn'],
'circulates':['s-er1 k-y-ah l-ey-t-s','vbz'],
'circulating':['s-er1 k-y-ah l-ey t-ih-ng','vbg'],
'circulation':['s-er1 k-y-ah l-ey sh-ah-n','nn'],
'circulatory':['s-er1 k-y-ah l-ah t-ao r-iy','jj'],
'circumcision':['s-er k-ah-m s-ih1 zh-ah-n','nn'],
'circumference':['s-er k-ah1-m f-r-ah-n-s','nn'],
'circumscribed':['s-er k-ah-m s-k-r-ay1-b-d','jj'],
'circumspect':['s-er1 k-ah-m s-p-eh-k-t','jj'],
'circumspection':['s-er k-ah-m s-p-eh1-k sh-ah-n','nn'],
'circumstance':['s-er1 k-ah-m s-t-ae-n-s','nn'],
'circumstantial':['s-er k-ah-m s-t-ae1-n ch-ah-l','jj'],
'circumvent':['s-er k-ah-m v-eh1-n-t','vb'],
'circumventing':['s-er k-ah-m v-eh1-n t-ih-ng','vbg'],
'circumvention':['s-er k-ah-m v-eh1-n ch-ah-n','nn'],
'circumvents':['s-er k-ah-m v-eh1-n-t-s','vbz'],
'circus':['s-er1 k-ah-s','nn'],
'cirrhosis':['s-er ow1 s-ah-s','nn'],
'citation':['s-ay t-ey1 sh-ah-n','nn'],
'cite':['s-ay1-t','vbp vb'],
'cited':['s-ay1 t-ah-d','vbd vbn'],
'cites':['s-ay1-t-s','vbz'],
'citing':['s-ay1 t-ih-ng','vbg'],
'citizen':['s-ih1 t-ah z-ah-n','nn'],
'citizenry':['s-ih1 t-ih z-ah-n r-iy','nn'],
'citizenship':['s-ih1 t-ih z-ah-n sh-ih-p','nn'],
'citric':['s-ih1 t-r-ih-k','jj'],
'citron':['s-ih1 t-r-ah-n','jj'],
'citrus':['s-ih1 t-r-ah-s','jj nn'],
'city':['s-ih1 t-iy','nn'],
'citywide':['s-ih1 t-iy w-ay-d','jj'],
'civic':['s-ih1 v-ih-k','jj'],
'civics':['s-ih1 v-ih-k-s','nns'],
'civil':['s-ih1 v-ah-l','jj'],
'civilian':['s-ah v-ih1-l y-ah-n','jj nn'],
'civility':['s-ah v-ih1 l-ah t-iy','nn'],
'civilizations':['s-ih v-ah l-ih z-ey1 sh-ah-n-z','nns'],
'civilize':['s-ih1 v-ah l-ay-z','vb'],
'civilized':['s-ih1 v-ah l-ay-z-d','jj vbn'],
'clad':['k-l-ae1-d','vbn jj'],
'claim':['k-l-ey1-m','nn vbp vb'],
'claimant':['k-l-ey1 m-ah-n-t','nn'],
'claimed':['k-l-ey1-m-d','vbd vbn'],
'claiming':['k-l-ey1 m-ih-ng','vbg'],
'claims':['k-l-ey1-m-z','nns vbz'],
'clairvoyance':['k-l-eh-r v-oy1 ah-n-s','nn'],
'clam':['k-l-ae1-m','nn vbp'],
'clambered':['k-l-ae1-m b-er-d','vbd'],
'clammy':['k-l-ae1 m-iy','jj'],
'clamor':['k-l-ae1 m-er','vbp nn'],
'clamored':['k-l-ae1 m-er-d','vbd'],
'clamoring':['k-l-ae1 m-er ih-ng','vbg'],
'clamp':['k-l-ae1-m-p','vb nn'],
'clampdown':['k-l-ae1-m-p d-aw-n','nn'],
'clamped':['k-l-ae1-m-p-t','vbd vbn'],
'clamping':['k-l-ae1-m p-ih-ng','vbg'],
'clamps':['k-l-ae1-m-p-s','nns'],
'clamshell':['k-l-ae1-m sh-eh-l','nn'],
'clan':['k-l-ae1-n','nn'],
'clandestine':['k-l-ae-n d-eh1 s-t-ih-n','jj'],
'clang':['k-l-ae1-ng','nn'],
'clanging':['k-l-ae1 ng-ih-ng','nn vbg'],
'clanking':['k-l-ae1-ng k-ih-ng','vbg'],
'clannish':['k-l-ae1 n-ih-sh','jj'],
'clap':['k-l-ae1-p','nn'],
'clapboard':['k-l-ae1-p b-ao-r-d','nn'],
'clapped':['k-l-ae1-p-t','vbd'],
'clapping':['k-l-ae1 p-ih-ng','vbg nn'],
'claps':['k-l-ae1-p-s','nns vbz'],
'clarification':['k-l-eh r-ah f-ah k-ey1 sh-ah-n','nn'],
'clarified':['k-l-eh1 r-ah f-ay-d','vbn vbd'],
'clarifies':['k-l-eh1 r-ah f-ay-z','vbz'],
'clarify':['k-l-eh1 r-ah f-ay','vb vbp'],
'clarifying':['k-l-eh1 r-ah f-ay ih-ng','vbg'],
'clarinet':['k-l-eh r-ah n-eh1-t','nn'],
'clarinetist':['k-l-eh r-ah n-eh1 t-ih-s-t','nn'],
'clarity':['k-l-eh1 r-ah t-iy','nn'],
'clash':['k-l-ae1-sh','nn vbp vb'],
'clashed':['k-l-ae1-sh-t','vbn vbd'],
'clashes':['k-l-ae1 sh-ih-z','nns vbz'],
'clashing':['k-l-ae1 sh-ih-ng','vbg'],
'clasped':['k-l-ae1-s-p-t','vbd vbn'],
'class':['k-l-ae1-s','nn'],
'classed':['k-l-ae1-s-t','vbn'],
'classic':['k-l-ae1 s-ih-k','jj nn'],
'classical':['k-l-ae1 s-ih k-ah-l','jj'],
'classically':['k-l-ae1 s-ih-k l-iy','rb'],
'classicism':['k-l-ae1 s-ih s-ih z-ah-m','nn'],
'classification':['k-l-ae s-ah f-ah k-ey1 sh-ah-n','nn'],
'classified':['k-l-ae1 s-ah f-ay-d','vbn vbd jj'],
'classifies':['k-l-ae1 s-ah f-ay-z','vbz'],
'classify':['k-l-ae1 s-ah f-ay','vb vbp'],
'classifying':['k-l-ae1 s-ah f-ay ih-ng','vbg'],
'classless':['k-l-ae1-s l-ah-s','jj'],
'classmate':['k-l-ae1-s m-ey-t','nn'],
'classroom':['k-l-ae1-s r-uw-m','nn'],
'classy':['k-l-ae1 s-iy','jj'],
'clatter':['k-l-ae1 t-er','nn vb'],
'clause':['k-l-ao1-z','nn'],
'clauses':['k-l-ao1 z-ah-z','nns'],
'claustrophobia':['k-l-ao s-t-r-ah f-ow1 b-iy ah','nn'],
'claustrophobic':['k-l-ao s-t-r-ah f-ow1 b-ih-k','jj'],
'claw':['k-l-ao1','nn'],
'clawed':['k-l-ao1-d','vbn'],
'clawing':['k-l-ao1 ih-ng','vbg'],
'clay':['k-l-ey1','nn'],
'clean':['k-l-iy1-n','jj vbp rb vb'],
'cleaned':['k-l-iy1-n-d','vbn vbd'],
'cleaner':['k-l-iy1 n-er','jjr nn'],
'cleanest':['k-l-iy1 n-ah-s-t','jjs'],
'cleaning':['k-l-iy1 n-ih-ng','nn vbg'],
'cleanliness':['k-l-eh1-n l-iy n-ih-s','nn'],
'cleanly':['k-l-iy1-n l-iy','rb'],
'cleans':['k-l-iy1-n-z','vbz'],
'cleanse':['k-l-eh1-n-z','vb'],
'cleansed':['k-l-eh1-n-z-d','vbd vbn'],
'cleanser':['k-l-eh1-n z-er','nn'],
'cleansing':['k-l-eh1-n z-ih-ng','nn vbg'],
'cleanup':['k-l-iy1-n ah-p','nn'],
'clear':['k-l-ih1-r','jj rb vb vbp'],
'clearance':['k-l-ih1 r-ah-n-s','nn'],
'cleared':['k-l-ih1-r-d','vbn vbd'],
'clearer':['k-l-ih1 r-er','jjr rbr'],
'clearest':['k-l-ih1 r-ih-s-t','jjs'],
'clearing':['k-l-ih1 r-ih-ng','vbg jj nn'],
'clearinghouse':['k-l-ih1 r-ih-ng hh-aw-s','nn'],
'clearly':['k-l-ih1-r l-iy','rb'],
'clears':['k-l-ih1-r-z','vbz'],
'cleat':['k-l-iy1-t','nn'],
'cleavage':['k-l-iy1 v-ah-jh','nn'],
'cleaver':['k-l-iy1 v-er','nn'],
'clemency':['k-l-eh1 m-ah-n s-iy','nn'],
'clenched':['k-l-eh1-n-ch-t','jj vbd vbn'],
'clergy':['k-l-er1 jh-iy','nn nns'],
'clergyman':['k-l-er1 jh-iy m-ah-n','nn'],
'cleric':['k-l-eh1 r-ih-k','nn'],
'clerical':['k-l-eh1 r-ah k-ah-l','jj nn'],
'clerk':['k-l-er1-k','nn'],
'clever':['k-l-eh1 v-er','jj'],
'cleverly':['k-l-eh1 v-er l-iy','rb'],
'cleverness':['k-l-eh1 v-er n-ah-s','nn'],
'cliche':['k-l-iy sh-ey1','nn'],
'cliched':['k-l-iy sh-ey1-d','jj'],
'click':['k-l-ih1-k','nn'],
'clicked':['k-l-ih1-k-t','vbd vbn'],
'clicking':['k-l-ih1 k-ih-ng','vbg nn'],
'clicks':['k-l-ih1-k-s','nns vbz'],
'client':['k-l-ay1 ah-n-t','nn'],
'clientele':['k-l-ay ah-n t-eh1-l','nn nns'],
'cliff':['k-l-ih1-f','nn'],
'climactic':['k-l-ay m-ae1-k t-ih-k','jj'],
'climate':['k-l-ay1 m-ah-t','nn'],
'climatic':['k-l-ay m-ae1 t-ih-k','jj'],
'climax':['k-l-ay1 m-ae-k-s','nn vb'],
'climaxed':['k-l-ay1 m-ae-k-s-t','vbd vbn'],
'climaxes':['k-l-ih1 m-ae-k s-ih-z','nns'],
'climb':['k-l-ay1-m','vb vbp nn'],
'climbed':['k-l-ay1-m-d','vbd vbn'],
'climber':['k-l-ay1 m-er','nn'],
'climbing':['k-l-ay1 m-ih-ng','vbg nn'],
'climbs':['k-l-ay1-m-z','vbz nns'],
'climes':['k-l-ay1-m-z','nns'],
'clinch':['k-l-ih1-n-ch','vb'],
'clinched':['k-l-ih1-n-ch-t','vbd'],
'clincher':['k-l-ih1-n ch-er','nn'],
'clinches':['k-l-ih1-n ch-ah-z','nns'],
'clinching':['k-l-ih1-n ch-ih-ng','vbg'],
'cling':['k-l-ih1-ng','vbp vb'],
'clinging':['k-l-ih1 ng-ih-ng','vbg'],
'clings':['k-l-ih1-ng-z','vbz'],
'clinic':['k-l-ih1 n-ih-k','nn'],
'clinical':['k-l-ih1 n-ah k-ah-l','jj'],
'clinically':['k-l-ih1 n-ih k-ah l-iy','rb'],
'clinician':['k-l-ih n-ih1 sh-ah-n','nn'],
'clip':['k-l-ih1-p','nn vb'],
'clipboard':['k-l-ih1-p b-ao-r-d','nn'],
'clipped':['k-l-ih1-p-t','vbn vbd'],
'clipping':['k-l-ih1 p-ih-ng','nn'],
'clips':['k-l-ih1-p-s','nns'],
'clique':['k-l-iy1-k','nn'],
'cloak':['k-l-ow1-k','nn vbp'],
'clobber':['k-l-aa1 b-er','vb'],
'clobbered':['k-l-aa1 b-er-d','vbn vbd'],
'clock':['k-l-aa1-k','nn vbp'],
'clocked':['k-l-aa1-k-t','vbn'],
'clocks':['k-l-aa1-k-s','nns vbz'],
'clockwork':['k-l-aa1-k w-er-k','nn'],
'clog':['k-l-aa1-g','vb'],
'clogged':['k-l-ao1-g-d','vbn jj vbd'],
'clogging':['k-l-aa1 g-ih-ng','vbg'],
'clogs':['k-l-aa1-g-z','vbz'],
'cloistered':['k-l-oy1 s-t-er-d','jj'],
'cloisters':['k-l-oy1 s-t-er-z','nns'],
'clone':['k-l-ow1-n','nn vb'],
'cloned':['k-l-ow1-n-d','vbn'],
'clones':['k-l-ow1-n-z','nns'],
'cloning':['k-l-ow1 n-ih-ng','vbg nn'],
'close':['k-l-ow1-s','vb vbp jj nn rb'],
'closed':['k-l-ow1-z-d','vbd vbn jj nn vb'],
'closedown':['k-l-ow1-z d-aw-n','nn'],
'closely':['k-l-ow1-s l-iy','rb'],
'closeness':['k-l-ow1-s n-ih-s','nn'],
'closer':['k-l-ow1 s-er','jjr jj rb rbr'],
'closes':['k-l-ow1 z-ih-z','vbz nns nn'],
'closest':['k-l-ow1 s-ah-s-t','jjs'],
'closet':['k-l-aa1 z-ah-t','nn'],
'closeted':['k-l-aa1 z-ah t-ih-d','jj vbd vbn'],
'closeup':['k-l-ow1 s-ah1-p','jj'],
'closing':['k-l-ow1 z-ih-ng','vbg jj nn'],
'closure':['k-l-ow1 zh-er','nn'],
'clot':['k-l-aa1-t','nn vb'],
'cloth':['k-l-ao1-th','nn'],
'clothe':['k-l-ow1-dh','vb'],
'clothed':['k-l-ow1-dh-d','vbn'],
'clothes':['k-l-ow1-dh-z','nns'],
'clotheshorse':['k-l-ow1-z hh-ao-r-s','nn'],
'clothier':['k-l-ow1 dh-y-er','nn'],
'clothing':['k-l-ow1 dh-ih-ng','nn'],
'clots':['k-l-aa1-t-s','nns'],
'clotted':['k-l-aa1 t-ah-d','jj'],
'clotting':['k-l-aa1 t-ih-ng','vbg'],
'cloture':['k-l-ow1 ch-er','nn'],
'cloud':['k-l-aw1-d','nn vb vbp'],
'cloudburst':['k-l-aw1-d b-er-s-t','nn'],
'clouded':['k-l-aw1 d-ih-d','vbn jj vbd'],
'clouding':['k-l-aw1 d-ih-ng','nn'],
'cloudless':['k-l-aw1-d l-ah-s','jj'],
'clouds':['k-l-aw1-d-z','nns vbz'],
'cloudy':['k-l-aw1 d-iy','jj'],
'clout':['k-l-aw1-t','nn'],
'clover':['k-l-ow1 v-er','nn'],
'cloves':['k-l-ow1-v-z','nns'],
'clown':['k-l-aw1-n','nn'],
'clowning':['k-l-aw1 n-ih-ng','nn'],
'cloying':['k-l-oy1 ih-ng','jj vbg'],
'clozapine':['k-l-ow1 z-ah p-ay-n','nn'],
'club':['k-l-ah1-b','nn'],
'clubbed':['k-l-ah1-b-d','jj vbd'],
'clubby':['k-l-ah1 b-iy','jj'],
'clubhouse':['k-l-ah1-b hh-aw-s','nn'],
'clubhouses':['k-l-ah1-b hh-aw s-ih-z','nns'],
'cluck':['k-l-ah1-k','nn'],
'clucking':['k-l-ah1 k-ih-ng','vbg'],
'clucks':['k-l-ah1-k-s','vbz nns'],
'clue':['k-l-uw1','nn'],
'clump':['k-l-ah1-m-p','nn'],
'clumsily':['k-l-ah1-m s-ah l-iy','rb'],
'clumsy':['k-l-ah1-m z-iy','jj'],
'clung':['k-l-ah1-ng','vbd vbn'],
'clunky':['k-l-ah1-ng k-iy','jj'],
'cluster':['k-l-ah1 s-t-er','nn vbp'],
'clustered':['k-l-ah1 s-t-er-d','vbn vbd'],
'clustering':['k-l-ah1 s-t-er ih-ng','vbg nn'],
'clusters':['k-l-ah1 s-t-er-z','nns vbz'],
'clutch':['k-l-ah1-ch','nn vbp vb'],
'clutched':['k-l-ah1-ch-t','vbd vbn'],
'clutches':['k-l-ah1 ch-ah-z','nns vbz'],
'clutching':['k-l-ah1 ch-ih-ng','vbg'],
'clutter':['k-l-ah1 t-er','nn vb'],
'cluttered':['k-l-ah1 t-er-d','vbn jj'],
'coach':['k-ow1-ch','nn'],
'coached':['k-ow1-ch-t','vbn'],
'coaching':['k-ow1 ch-ih-ng','nn vbg'],
'coachman':['k-ow1-ch m-ah-n','nn'],
'coagulating':['k-ow ae1 g-y-ah l-ey t-ih-ng','vbg'],
'coal':['k-ow1-l','nn'],
'coalesce':['k-ow ah l-eh1-s','vb vbp'],
'coalesced':['k-ow ah l-eh1-s-t','vbn'],
'coalition':['k-ow ah l-ih1 sh-ah-n','nn'],
'coarse':['k-ao1-r-s','jj'],
'coast':['k-ow1-s-t','nn'],
'coastal':['k-ow1 s-t-ah-l','jj'],
'coasted':['k-ow1 s-t-ih-d','vbd vbn'],
'coaster':['k-ow1 s-t-er','nn'],
'coastline':['k-ow1-s-t l-ay-n','nn'],
'coat':['k-ow1-t','nn vb'],
'coated':['k-ow1 t-ah-d','vbn jj'],
'coating':['k-ow1 t-ih-ng','nn vbg'],
'coats':['k-ow1-t-s','nns'],
'coattails':['k-ow1-t t-ey-l-z','nns'],
'coax':['k-ow1-k-s','vb'],
'coaxed':['k-ow1-k-s-t','vbn vbd'],
'coaxes':['k-ow1-k s-ih-z','vbz'],
'coaxial':['k-ow1 ae1-k s-iy ah-l','jj'],
'coaxing':['k-ow1-k s-ih-ng','jj vbg'],
'cobalt':['k-ow1 b-ao-l-t','nn'],
'cobbled':['k-aa1 b-ah-l-d','vbn vbd'],
'cobbler':['k-aa1-b l-er','nn'],
'cobblestones':['k-aa1 b-ah-l s-t-ow-n-z','nns'],
'cobra':['k-ow1 b-r-ah','nn'],
'cobwebs':['k-aa1-b w-eh-b-z','nns'],
'coca':['k-ow1 k-ah','nn'],
'cocaine':['k-ow k-ey1-n','nn'],
'cock':['k-aa1-k','nn'],
'cockatoo':['k-aa1 k-ah t-uw','nn'],
'cocked':['k-aa1-k-t','vbd vbn'],
'cockeyed':['k-aa1-k ay-d','jj'],
'cockiness':['k-aa1 k-iy n-ah-s','nn'],
'cockpit':['k-aa1-k p-ih-t','nn'],
'cockroach':['k-aa1-k r-ow-ch','nn'],
'cocktail':['k-aa1-k t-ey-l','nn'],
'cocky':['k-aa1 k-iy','jj'],
'coco':['k-ow1 k-ow','nn'],
'cocoa':['k-ow1 k-ow','nn'],
'coconut':['k-ow1 k-ah n-ah-t','nn'],
'cocoon':['k-ah k-uw1-n','nn'],
'cod':['k-aa1-d','nn'],
'coddle':['k-aa1 d-ah-l','vbp'],
'coddled':['k-aa1 d-ah-l-d','vbn'],
'coddling':['k-aa1 d-ah-l ih-ng','nn'],
'code':['k-ow1-d','nn vb'],
'coded':['k-ow1 d-ih-d','vbn jj'],
'codes':['k-ow1-d-z','nns'],
'codification':['k-aa d-ah f-ah k-ey1 sh-ah-n','nn'],
'codified':['k-aa1 d-ah f-ay-d','vbn'],
'codifies':['k-ow1 d-ah f-ay-z','vbz'],
'codifying':['k-ow1 d-ah f-ay ih-ng','vbg'],
'coding':['k-ow1 d-ih-ng','nn'],
'coed':['k-ow1 eh-d','nn'],
'coefficient':['k-ow ah f-ih1 sh-ah-n-t','nn'],
'coerce':['k-ow er1-s','vb'],
'coerced':['k-ow er1-s-t','vbn vbd jj'],
'coercion':['k-ow er1 sh-ah-n','nn'],
'coercive':['k-ow er1 s-ih-v','jj'],
'coexist':['k-ow ah-g z-ih1-s-t','vb vbp'],
'coexistence':['k-ow ih-g z-ih1 s-t-ah-n-s','nn'],
'coffee':['k-aa1 f-iy','nn'],
'coffers':['k-aa1 f-er-z','nns'],
'coffin':['k-ao1 f-ih-n','nn'],
'cofounder':['k-ow1 f-aw1-n d-er','nn'],
'cog':['k-ao1-g','nn'],
'cogeneration':['k-ow1 jh-eh1 n-er ey1 sh-ah-n','nn'],
'cognac':['k-ow1-n y-ae-k','nn'],
'cognitive':['k-aa1-g n-ih t-ih-v','jj'],
'cognizance':['k-aa1-g n-ah z-ah-n-s','nn'],
'cognizant':['k-aa1-g n-ah z-ah-n-t','jj'],
'cognoscenti':['k-aa-g n-ao sh-eh1-n t-iy','nns'],
'coherence':['k-ow hh-ih1 r-ah-n-s','nn'],
'coherent':['k-ow hh-ih1 r-ah-n-t','jj'],
'coherently':['k-ow hh-iy1 r-ah-n-t l-iy','rb'],
'cohesion':['k-ow hh-iy1 zh-ah-n','nn'],
'cohesive':['k-ow hh-iy1 s-ih-v','jj'],
'cohesively':['k-ow hh-iy1 s-ih-v l-iy','rb'],
'cohesiveness':['k-ow hh-iy1 s-ih-v n-ah-s','nn'],
'cohort':['k-ow1 hh-ao-r-t','nn'],
'coiffed':['k-oy1-f-t','jj'],
'coil':['k-oy1-l','nn'],
'coiled':['k-oy1-l-d','vbd'],
'coin':['k-oy1-n','nn vb'],
'coincide':['k-ow ih-n s-ay1-d','vb vbp'],
'coincided':['k-ow ah-n s-ay1 d-ah-d','vbd vbn'],
'coincidence':['k-ow ih1-n s-ih d-ah-n-s','nn'],
'coincident':['k-ow ih1-n s-ah d-ah-n-t','jj'],
'coincidental':['k-ow ih-n s-ah d-eh1-n t-ah-l','jj'],
'coincidentally':['k-ow ih-n s-ih d-eh1-n t-ah l-iy','rb'],
'coincides':['k-ow ih-n s-ay1-d-z','vbz'],
'coinciding':['k-ow ah-n s-ay1 d-ih-ng','vbg jj'],
'coined':['k-oy1-n-d','vbn vbd'],
'coins':['k-oy1-n-z','nns'],
'coke':['k-ow1-k','nn'],
'cola':['k-ow1 l-ah','nn'],
'cold':['k-ow1-l-d','jj nn'],
'colder':['k-ow1-l d-er','jjr rbr'],
'coldest':['k-ow1-l d-ah-s-t','jjs'],
'coldly':['k-ow1-l-d l-iy','rb'],
'coldness':['k-ow1-l-d n-ah-s','nn'],
'coliseum':['k-aa l-ah s-iy1 ah-m','nn'],
'collaborate':['k-ah l-ae1 b-er ey-t','vb vbp'],
'collaborated':['k-ah l-ae1 b-er ey t-ah-d','vbd vbn'],
'collaborating':['k-ah l-ae1 b-er ey t-ih-ng','vbg'],
'collaboration':['k-ah l-ae b-er ey1 sh-ah-n','nn'],
'collaborative':['k-ah l-ae1 b-er ey t-ih-v','jj'],
'collaborator':['k-ah l-ae1 b-er ey t-er','nn'],
'collage':['k-ah l-aa1-zh','nn'],
'collagen':['k-aa1 l-ah g-ah-n','nn'],
'collapse':['k-ah l-ae1-p-s','nn vb'],
'collapsed':['k-ah l-ae1-p-s-t','vbd jj vbn'],
'collapses':['k-ah l-ae1-p s-ih-z','vbz nns'],
'collapsible':['k-ah l-ae1-p s-ah b-ah-l','jj'],
'collapsing':['k-ah l-ae1-p s-ih-ng','vbg'],
'collar':['k-aa1 l-er','nn vb'],
'collarbone':['k-aa1 l-er b-ow-n','nn'],
'collared':['k-aa1 l-er-d','vbn'],
'collars':['k-aa1 l-er-z','nns'],
'collateral':['k-ah l-ae1 t-er ah-l','nn jj'],
'collateralized':['k-ah l-ae1 t-er ah l-ay-z-d','jj vbn'],
'colleague':['k-aa1 l-iy-g','nn'],
'collect':['k-ah l-eh1-k-t','vb jj vbp'],
'collected':['k-ah l-eh1-k t-ah-d','vbn vbd'],
'collectible':['k-ah l-eh1-k t-ah b-ah-l','jj'],
'collecting':['k-ah l-eh1-k t-ih-ng','vbg nn'],
'collection':['k-ah l-eh1-k sh-ah-n','nn'],
'collective':['k-ah l-eh1-k t-ih-v','jj nn'],
'collectively':['k-ah l-eh1-k t-ih-v l-iy','rb'],
'collectivization':['k-ah l-eh-k t-ih v-ih z-ey1 sh-ah-n','nn'],
'collector':['k-ah l-eh1-k t-er','nn'],
'collects':['k-ah l-eh1-k-t-s','vbz'],
'college':['k-aa1 l-ih-jh','nn'],
'collegial':['k-ah l-iy1 jh-iy ah-l','jj'],
'collegians':['k-ah l-iy1 jh-ah-n-z','nns'],
'collegiate':['k-ah l-iy1 jh-ih-t','jj'],
'collided':['k-ah l-ay1 d-ih-d','vbd'],
'collie':['k-aa1 l-iy','nn'],
'collision':['k-ah l-ih1 zh-ah-n','nn'],
'colloidal':['k-ah l-oy1 d-ah-l','jj'],
'colloquial':['k-ah l-ow1-k w-iy ah-l','jj'],
'colloquium':['k-ah l-ow1 k-w-iy ah-m','nn'],
'colloquy':['k-aa1 l-ah k-w-iy','nn'],
'collude':['k-ah l-uw1-d','vb'],
'colluded':['k-ah l-uw1 d-ih-d','vbd'],
'collusion':['k-ah l-uw1 zh-ah-n','nn'],
'cologne':['k-ah l-ow1-n','nn'],
'colon':['k-ow1 l-ah-n','nn'],
'colonel':['k-er1 n-ah-l','nn'],
'colonial':['k-ah l-ow1 n-iy ah-l','jj nn'],
'colonialism':['k-ah l-ow1 n-iy ah l-ih z-ah-m','nn'],
'colonialists':['k-ah l-ow1 n-iy ah l-ih-s-t-s','nns'],
'colonists':['k-aa1 l-ah n-ih-s-t-s','nns'],
'colonization':['k-aa l-ah n-ih z-ey1 sh-ah-n','nn'],
'colonized':['k-aa1 l-ah n-ay-z-d','vbd vbn'],
'colonnade':['k-aa l-ah n-ey1-d','nn'],
'colony':['k-aa1 l-ah n-iy','nn'],
'color':['k-ah1 l-er','nn jj vb vbp'],
'coloration':['k-ah l-er ey1 sh-ah-n','nn'],
'coloratura':['k-ah l-er ah t-uh1 r-ah','nn'],
'colorblind':['k-ah1 l-er b-l-ay-n-d','jj'],
'colored':['k-ah1 l-er-d','jj vbn'],
'colorful':['k-ah1 l-er f-ah-l','jj'],
'coloring':['k-ah1 l-er ih-ng','nn'],
'colorization':['k-ah l-er ah z-ey1 sh-ah-n','nn'],
'colorless':['k-ah1 l-er l-ah-s','jj'],
'colors':['k-ah1 l-er-z','nns vbz'],
'colossal':['k-ah l-aa1 s-ah-l','jj'],
'colossus':['k-ah l-aa1 s-ah-s','nn'],
'coloured':['k-ah1 l-er-d','jj'],
'colt':['k-ow1-l-t','nn'],
'columbines':['k-aa1 l-ah-m b-ay-n-z','nns'],
'column':['k-aa1 l-ah-m','nn'],
'columnist':['k-aa1 l-ah-m n-ah-s-t','nn'],
'coma':['k-ow1 m-ah','nn'],
'comb':['k-ow1-m','nn vb'],
'combat':['k-aa1-m b-ae-t','nn vb'],
'combatant':['k-ah-m b-ae1 t-ah-n-t','jj nn'],
'combating':['k-ah-m b-ae1 t-ih-ng','vbg'],
'combative':['k-ah-m b-ae1 t-ih-v','jj'],
'combed':['k-ow1-m-d','vbd vbn'],
'combinable':['k-ah-m b-ay1 n-ah b-ah-l','jj'],
'combination':['k-aa-m b-ah n-ey1 sh-ah-n','nn'],
'combine':['k-aa1-m b-ay-n','vb nn vbp'],
'combined':['k-ah-m b-ay1-n-d','vbn jj vbd'],
'combines':['k-ah-m b-ay1-n-z','vbz nns'],
'combing':['k-ow1 m-ih-ng','vbg jj'],
'combining':['k-ah-m b-ay1 n-ih-ng','vbg'],
'combo':['k-aa1-m b-ow','nn'],
'combustion':['k-ah-m b-ah1-s ch-ah-n','nn'],
'come':['k-ah1-m','vb vbd vbn vbp vbz jj'],
'comeback':['k-ah1-m b-ae-k','nn'],
'comedian':['k-ah m-iy1 d-iy ah-n','nn'],
'comedic':['k-ah m-iy1 d-ih-k','jj'],
'comedy':['k-aa1 m-ah d-iy','nn'],
'comely':['k-ah1-m l-iy','jj'],
'comer':['k-ah1 m-er','nn'],
'comes':['k-ah1-m-z','vbz nns'],
'comet':['k-aa1 m-ah-t','nn'],
'cometary':['k-aa1 m-ah t-eh r-iy','jj'],
'comeuppance':['k-ah m-ah1 p-ah-n-s','nn'],
'comfort':['k-ah1-m f-er-t','nn vb'],
'comfortable':['k-ah1-m f-er t-ah b-ah-l','jj nn'],
'comfortably':['k-ah1-m f-er t-ah b-l-iy','rb'],
'comforted':['k-ah1-m f-er t-ih-d','vbn'],
'comforting':['k-ah1-m f-er t-ih-ng','vbg jj'],
'comforts':['k-ah1-m f-er-t-s','nns'],
'comfy':['k-ah1-m f-iy','jj'],
'comic':['k-aa1 m-ih-k','jj nn'],
'comical':['k-aa1 m-ih k-ah-l','jj'],
'comically':['k-aa1 m-ih k-ah l-iy','rb'],
'coming':['k-ah1 m-ih-ng','vbg jj nn'],
'comma':['k-aa1 m-ah','nn'],
'command':['k-ah m-ae1-n-d','nn vbp vb'],
'commandant':['k-aa m-ah-n d-aa1-n-t','nn'],
'commanded':['k-ah m-ae1-n d-ah-d','vbd vbn'],
'commandeered':['k-aa m-ah-n d-ih1-r-d','vbn'],
'commander':['k-ah m-ae1-n d-er','nn'],
'commanding':['k-ah m-ae1-n d-ih-ng','vbg jj nn'],
'commandment':['k-ah m-ae1-n-d m-ah-n-t','nn'],
'commando':['k-ah m-ae1-n d-ow','nn'],
'commands':['k-ah m-ae1-n-d-z','nns vbz'],
'commemorate':['k-ah m-eh1 m-er ey-t','vb vbp'],
'commemorated':['k-ah m-eh1 m-er ey t-ih-d','vbn vbd'],
'commemorates':['k-ah m-eh1 m-er ey-t-s','vbz'],
'commemorating':['k-ah m-eh1 m-er ey t-ih-ng','vbg'],
'commemorative':['k-ah m-eh1-m r-ah t-ih-v','jj nn'],
'commence':['k-ah m-eh1-n-s','vb vbp'],
'commenced':['k-ah m-eh1-n-s-t','vbd vbn'],
'commencement':['k-ah m-eh1-n s-m-ah-n-t','nn'],
'commences':['k-ah m-eh1-n s-ah-z','vbz'],
'commencing':['k-ah m-eh1-n s-ih-ng','vbg'],
'commend':['k-ah m-eh1-n-d','vb vbp'],
'commendable':['k-ah m-eh1-n d-ah b-ah-l','jj'],
'commendation':['k-aa m-ah-n d-ey1 sh-ah-n','nn'],
'commended':['k-ah m-eh1-n d-ih-d','vbn vbd'],
'commensurate':['k-ah m-eh1-n s-er ah-t','jj'],
'comment':['k-aa1 m-eh-n-t','vb vbp nn'],
'commentary':['k-aa1 m-ah-n t-eh r-iy','nn'],
'commentator':['k-aa1 m-ah-n t-ey t-er','nn'],
'commented':['k-aa1 m-eh-n t-ah-d','vbd vbn'],
'commenting':['k-aa1 m-eh-n t-ih-ng','vbg'],
'comments':['k-aa1 m-eh-n-t-s','nns vbz'],
'commerce':['k-aa1 m-er-s','nn'],
'commercial':['k-ah m-er1 sh-ah-l','jj nn'],
'commercialism':['k-ah m-er1 sh-ah l-ih z-ah-m','nn'],
'commercialization':['k-ah m-er sh-ah l-ih z-ey1 sh-ah-n','nn'],
'commercialize':['k-ah m-er1 sh-ah l-ay-z','vb'],
'commercialized':['k-ah m-er1 sh-ah l-ay-z-d','vbn'],
'commercializing':['k-ah m-er1 sh-ah l-ay z-ih-ng','vbg'],
'commercially':['k-ah m-er1 sh-ah l-iy','rb'],
'commerical':['k-ah m-er1 sh-ah-l','jj'],
'commie':['k-aa1 m-iy','jj'],
'commingled':['k-aa m-ih1-ng g-ah-l-d','vbn'],
'commiserate':['k-ah m-ih1 s-er ey-t','vb'],
'commissar':['k-aa1 m-ah s-aa-r','nn'],
'commissary':['k-aa1 m-ah s-eh r-iy','nn'],
'commission':['k-ah m-ih1 sh-ah-n','nn vb'],
'commissioned':['k-ah m-ih1 sh-ah-n-d','vbn vbd'],
'commissioner':['k-ah m-ih1 sh-ah-n er','nn'],
'commissioning':['k-ah m-ih1 sh-ah-n ih-ng','nn vbg'],
'commissions':['k-ah m-ih1 sh-ah-n-z','nns vbz nn'],
'commit':['k-ah m-ih1-t','vb jj vbp'],
'commitment':['k-ah m-ih1-t m-ah-n-t','nn'],
'commits':['k-ah m-ih1-t-s','vbz'],
'committed':['k-ah m-ih1 t-ah-d','vbn jj vb vbd'],
'committee':['k-ah m-ih1 t-iy','nn'],
'committees':['k-ah m-ih1 t-iy-z','nns vbz'],
'committing':['k-ah m-ih1 t-ih-ng','vbg'],
'commodity':['k-ah m-aa1 d-ah t-iy','nn'],
'common':['k-aa1 m-ah-n','jj nn'],
'commonality':['k-aa m-ah n-ae1 l-ah t-iy','nn'],
'commoner':['k-aa1 m-ah n-er','jjr'],
'commonly':['k-aa1 m-ah-n l-iy','rb'],
'commonplace':['k-aa1 m-ah-n p-l-ey-s','jj nn'],
'commons':['k-aa1 m-ah-n-z','nn'],
'commonwealth':['k-aa1 m-ah-n w-eh-l-th','nn'],
'commotion':['k-ah m-ow1 sh-ah-n','nn'],
'communal':['k-ah m-y-uw1 n-ah-l','jj'],
'commune':['k-aa1 m-y-uw-n','nn vb'],
'communes':['k-aa1 m-y-uw-n-z','nns'],
'communicable':['k-ah m-y-uw1 n-ah k-ah b-ah-l','jj'],
'communicate':['k-ah m-y-uw1 n-ah k-ey-t','vb vbp'],
'communicated':['k-ah m-y-uw1 n-ah k-ey t-ih-d','vbn vbd'],
'communicating':['k-ah m-y-uw1 n-ah k-ey t-ih-ng','vbg nn'],
'communication':['k-ah m-y-uw n-ah k-ey1 sh-ah-n','nn'],
'communications':['k-ah m-y-uw n-ah k-ey1 sh-ah-n-z','nns nn'],
'communicative':['k-ah m-y-uw1 n-ah k-ah t-ih-v','jj'],
'communicator':['k-ah m-y-uw1 n-ah k-ey t-er','nn'],
'communion':['k-ah m-y-uw1 n-y-ah-n','nn'],
'communique':['k-ah m-y-uw1 n-ah k-ey','nn'],
'communism':['k-aa1 m-y-ah n-ih z-ah-m','nn'],
'communist':['k-aa1 m-y-ah n-ah-s-t','jj nn'],
'community':['k-ah m-y-uw1 n-ah t-iy','nn'],
'communize':['k-aa1 m-y-ah n-ay-z','vb'],
'commute':['k-ah m-y-uw1-t','vbp nn'],
'commuted':['k-ah m-y-uw1 t-ih-d','vbn vbd'],
'commuter':['k-ah m-y-uw1 t-er','nn'],
'commutes':['k-ah m-y-uw1-t-s','nns vbz'],
'commuting':['k-ah m-y-uw1 t-ih-ng','vbg nn'],
'compact':['k-aa1-m p-ae-k-t','jj nn'],
'compacted':['k-ah-m p-ae1-k t-ih-d','jj vbn'],
'companion':['k-ah-m p-ae1 n-y-ah-n','nn jj'],
'companionship':['k-ah-m p-ae1 n-y-ah-n sh-ih-p','nn'],
'companionway':['k-ah-m p-ae1 n-y-ah-n w-ey','nn'],
'company':['k-ah1-m p-ah n-iy','nn'],
'companywide':['k-ah1-m p-ah n-iy w-ay-d','jj rb'],
'comparability':['k-aa-m p-er ah b-ih1 l-ih t-iy','nn'],
'comparable':['k-aa1-m p-er ah b-ah-l','jj'],
'comparably':['k-aa1-m p-er ah b-l-iy','rb'],
'comparative':['k-ah-m p-eh1 r-ah t-ih-v','jj nn'],
'comparatively':['k-ah-m p-eh1 r-ah t-ih-v l-iy','rb'],
'compare':['k-ah-m p-eh1-r','vb vbp nn'],
'compared':['k-ah-m p-eh1-r-d','vbn vbd'],
'compares':['k-ah-m p-eh1-r-z','vbz nns'],
'comparing':['k-ah-m p-eh1 r-ih-ng','vbg'],
'comparison':['k-ah-m p-eh1 r-ah s-ah-n','nn'],
'compartment':['k-ah-m p-aa1-r-t m-ah-n-t','nn'],
'compass':['k-ah1-m p-ah-s','nn vb'],
'compassion':['k-ah-m p-ae1 sh-ah-n','nn'],
'compassionate':['k-ah-m p-ae1 sh-ah n-ah-t','jj'],
'compatibility':['k-ah-m p-ae t-ah b-ih1 l-ah t-iy','nn'],
'compatible':['k-ah-m p-ae1 t-ah b-ah-l','jj'],
'compatriots':['k-ah-m p-ey1 t-r-iy ah-t-s','nns'],
'compel':['k-ah-m p-eh1-l','vb'],
'compelled':['k-ah-m p-eh1-l-d','vbn vbd jj'],
'compelling':['k-ah-m p-eh1 l-ih-ng','jj vbg'],
'compellingly':['k-ah-m p-eh1 l-ih-ng l-iy','rb'],
'compels':['k-ah-m p-eh1-l-z','vbz'],
'compendium':['k-ah-m p-eh1-n d-iy ah-m','nn'],
'compensate':['k-aa1-m p-ah-n s-ey-t','vb vbp'],
'compensated':['k-aa1-m p-ah-n s-ey t-ah-d','vbn vbd'],
'compensates':['k-aa1-m p-ah-n s-ey-t-s','vbz'],
'compensating':['k-aa1-m p-ah-n s-ey t-ih-ng','vbg'],
'compensation':['k-aa-m p-ah-n s-ey1 sh-ah-n','nn'],
'compensatory':['k-ah-m p-eh1-n s-ah t-ao r-iy','jj'],
'compete':['k-ah-m p-iy1-t','vb vbp'],
'competed':['k-ah-m p-iy1 t-ih-d','vbd vbn'],
'competence':['k-aa1-m p-ah t-ih-n-s','nn'],
'competency':['k-aa1-m p-ah t-ih-n s-iy','nn'],
'competent':['k-aa1-m p-ah t-ih-n-t','jj'],
'competently':['k-aa1-m p-ah t-ih-n-t l-iy','rb'],
'competes':['k-ah-m p-iy1-t-s','vbz'],
'competing':['k-ah-m p-iy1 t-ih-ng','vbg jj'],
'competition':['k-aa-m p-ah t-ih1 sh-ah-n','nn'],
'competitive':['k-ah-m p-eh1 t-ah t-ih-v','jj'],
'competitively':['k-ah-m p-eh1 t-ih t-ih-v l-iy','rb'],
'competitiveness':['k-ah-m p-eh1 t-ih t-ih-v n-ih-s','nn'],
'competitor':['k-ah-m p-eh1 t-ah t-er','nn'],
'compilation':['k-aa-m p-ah l-ey1 sh-ah-n','nn'],
'compile':['k-ah-m p-ay1-l','vb'],
'compiled':['k-ah-m p-ay1-l-d','vbn vbd'],
'compiles':['k-ah-m p-ay1-l-z','vbz'],
'compiling':['k-ah-m p-ay1 l-ih-ng','vbg'],
'complacency':['k-ah-m p-l-ey1 s-ah-n s-iy','nn'],
'complacent':['k-ah-m p-l-ey1 s-ah-n-t','jj'],
'complacently':['k-ah-m p-l-ey1 s-ah-n-t l-iy','rb'],
'complain':['k-ah-m p-l-ey1-n','vbp vb'],
'complained':['k-ah-m p-l-ey1-n-d','vbd vbn'],
'complaining':['k-ah-m p-l-ey1 n-ih-ng','vbg'],
'complains':['k-ah-m p-l-ey1-n-z','vbz'],
'complaint':['k-ah-m p-l-ey1-n-t','nn'],
'complaisant':['k-ah-m p-l-ey1 s-ah-n-t','jj'],
'complement':['k-aa1-m p-l-ah m-ah-n-t','nn vbp vb'],
'complementary':['k-aa-m p-l-ah m-eh1-n t-r-iy','jj'],
'complemented':['k-aa1-m p-l-ah m-eh-n t-ih-d','vbd'],
'complements':['k-aa1-m p-l-ah m-ah-n-t-s','vbz nns'],
'complete':['k-ah-m p-l-iy1-t','jj vb vbp'],
'completed':['k-ah-m p-l-iy1 t-ah-d','vbn jj vbd'],
'completely':['k-ah-m p-l-iy1-t l-iy','rb'],
'completeness':['k-ah-m p-l-iy1-t n-ah-s','nn'],
'completes':['k-ah-m p-l-iy1-t-s','vbz'],
'completing':['k-ah-m p-l-iy1 t-ih-ng','vbg'],
'completion':['k-ah-m p-l-iy1 sh-ah-n','nn'],
'complex':['k-aa1-m p-l-eh-k-s','jj nn'],
'complexion':['k-ah-m p-eh1-k sh-ah-n','nn'],
'complexity':['k-ah-m p-l-eh1-k s-ah t-iy','nn'],
'compliance':['k-ah-m p-l-ay1 ah-n-s','nn'],
'compliant':['k-ah-m p-l-ay1 ah-n-t','jj'],
'complicate':['k-aa1-m p-l-ah k-ey-t','vb vbp'],
'complicated':['k-aa1-m p-l-ah k-ey t-ah-d','vbn jj vbd'],
'complicates':['k-aa1-m p-l-ih k-ey-t-s','vbz'],
'complicating':['k-aa1-m p-l-ah k-ey t-ih-ng','vbg'],
'complication':['k-aa-m p-l-ah k-ey1 sh-ah-n','nn'],
'complicity':['k-ah-m p-l-ih1 s-ah t-iy','nn'],
'complied':['k-ah-m p-l-ay1-d','vbn vbd'],
'complies':['k-ah-m p-l-ay1-z','vbz'],
'compliment':['k-aa1-m p-l-ah m-eh-n-t','nn vbp vb'],
'complimentary':['k-aa-m p-l-ah m-eh1-n t-er iy','jj'],
'compliments':['k-aa1-m p-l-ah m-eh-n-t-s','nns'],
'comply':['k-ah-m p-l-ay1','vb vbp'],
'complying':['k-ah-m p-l-ay1 ih-ng','vbg'],
'component':['k-ah-m p-ow1 n-ah-n-t','nn'],
'comportment':['k-ah-m p-ao1-r-t m-ah-n-t','nn'],
'compose':['k-ah-m p-ow1-z','vb vbp'],
'composed':['k-ah-m p-ow1-z-d','vbn vbd jj'],
'composer':['k-ah-m p-ow1 z-er','nn'],
'composes':['k-ah-m p-ow1 z-ih-z','vbz'],
'composing':['k-ah-m p-ow1 z-ih-ng','vbg'],
'composite':['k-ah-m p-aa1 z-ah-t','jj nn'],
'composition':['k-aa-m p-ah z-ih1 sh-ah-n','nn'],
'compositional':['k-aa-m p-ah z-ih1 sh-ah n-ah-l','jj'],
'compost':['k-aa1-m p-ow-s-t','nn'],
'composting':['k-aa1-m p-ow s-t-ih-ng','nn'],
'composure':['k-ah-m p-ow1 zh-er','nn'],
'compote':['k-aa1-m p-ow-t','nn'],
'compound':['k-aa1-m p-aw-n-d','nn jj vb vbp'],
'compounded':['k-ah-m p-aw1-n d-ah-d','vbn jj vbd'],
'compounding':['k-ah-m p-aw1-n d-ih-ng','vbg nn'],
'compounds':['k-aa1-m p-aw-n-d-z','nns vbz'],
'comprehend':['k-aa-m p-r-iy hh-eh1-n-d','vb'],
'comprehended':['k-aa-m p-r-ih hh-eh1-n d-ih-d','vbd vbn'],
'comprehending':['k-aa-m p-r-ih hh-eh1-n d-ih-ng','vbg'],
'comprehension':['k-aa-m p-r-iy hh-eh1-n sh-ah-n','nn'],
'comprehensive':['k-aa-m p-r-iy hh-eh1-n s-ih-v','jj'],
'comprehensively':['k-aa-m p-r-ih hh-eh1-n s-ih-v l-iy','rb'],
'compress':['k-aa1-m p-r-eh-s','vb'],
'compressed':['k-ah-m p-r-eh1-s-t','vbn vbd'],
'compresses':['k-aa1-m p-r-eh s-ah-z','nns vbz'],
'compressing':['k-ah-m p-r-eh1 s-ih-ng','vbg'],
'compression':['k-ah-m p-r-eh1 sh-ah-n','nn'],
'compressor':['k-ah-m p-r-eh1 s-er','nn'],
'comprise':['k-ah-m p-r-ay1-z','vbp vb'],
'comprised':['k-ah-m p-r-ay1-z-d','vbn vbd'],
'comprises':['k-ah-m p-r-ay1 z-ah-z','vbz'],
'comprising':['k-ah-m p-r-ay1 z-ih-ng','vbg'],
'compromise':['k-aa1-m p-r-ah m-ay-z','nn jj vb'],
'compromised':['k-aa1-m p-r-ah m-ay-z-d','vbn vbd'],
'compromises':['k-aa1-m p-r-ah m-ay z-ih-z','nns vbz'],
'compromising':['k-aa1-m p-r-ah m-ay z-ih-ng','vbg jj'],
'comptroller':['k-ah-n t-r-ow1 l-er','nn'],
'compulsion':['k-ah-m p-ah1-l sh-ah-n','nn'],
'compulsive':['k-ah-m p-ah1-l s-ih-v','jj'],
'compulsively':['k-ah-m p-ah1-l s-ih-v l-iy','rb'],
'compulsory':['k-ah-m p-ah1-l s-er iy','jj'],
'computation':['k-aa-m p-y-ah t-ey1 sh-ah-n','nn'],
'computational':['k-aa-m p-y-uw t-ey1 sh-ah n-ah-l','jj'],
'compute':['k-ah-m p-y-uw1-t','vb'],
'computed':['k-ah-m p-y-uw1 t-ah-d','vbn jj'],
'computer':['k-ah-m p-y-uw1 t-er','nn'],
'computerize':['k-ah-m p-y-uw1 t-er ay-z','vb'],
'computerized':['k-ah-m p-y-uw1 t-er ay-z-d','jj nn vb vbn'],
'computerizing':['k-ah-m p-y-uw1 t-er ay z-ih-ng','vbg'],
'computes':['k-ah-m p-y-uw1-t-s','vbz'],
'computing':['k-ah-m p-y-uw1 t-ih-ng','vbg nn'],
'comrade':['k-aa1-m r-ae-d','nn'],
'concave':['k-aa-n k-ey1-v','jj'],
'conceal':['k-ah-n s-iy1-l','vb'],
'concealed':['k-ah-n s-iy1-l-d','vbn jj vbd'],
'concealing':['k-ah-n s-iy1 l-ih-ng','vbg'],
'concealment':['k-ah-n s-iy1-l m-ah-n-t','nn'],
'conceals':['k-ah-n s-iy1-l-z','vbz'],
'concede':['k-ah-n s-iy1-d','vbp vb'],
'conceded':['k-ah-n s-iy1 d-ih-d','vbd vbn'],
'concedes':['k-ah-n s-iy1-d-z','vbz'],
'conceding':['k-ah-n s-iy1 d-ih-ng','vbg'],
'conceit':['k-ah-n s-iy1-t','nn'],
'conceivable':['k-ah-n s-iy1 v-ah b-ah-l','jj'],
'conceivably':['k-ah-n s-iy1 v-ah b-l-iy','rb'],
'conceive':['k-ah-n s-iy1-v','vb vbp'],
'conceived':['k-ah-n s-iy1-v-d','vbn vbd jj'],
'conceiving':['k-ah-n s-iy1 v-ih-ng','vbg'],
'concentrate':['k-aa1-n s-ah-n t-r-ey-t','vb vbp nn'],
'concentrated':['k-aa1-n s-ah-n t-r-ey t-ah-d','vbn jj vbd'],
'concentrates':['k-aa1-n s-ah-n t-r-ey-t-s','vbz nns'],
'concentrating':['k-aa1-n s-ah-n t-r-ey t-ih-ng','vbg'],
'concentration':['k-aa-n s-ah-n t-r-ey1 sh-ah-n','nn'],
'concentric':['k-ah-n s-eh1-n t-r-ih-k','jj'],
'concept':['k-aa1-n s-eh-p-t','nn'],
'conception':['k-ah-n s-eh1-p sh-ah-n','nn'],
'conceptual':['k-ah-n s-eh1-p ch-uw ah-l','jj'],
'conceptually':['k-ah-n s-eh1-p ch-uw ah l-iy','rb'],
'concern':['k-ah-n s-er1-n','nn vbp vb'],
'concerned':['k-ah-n s-er1-n-d','vbn jj vb vbd'],
'concerning':['k-ah-n s-er1 n-ih-ng','vbg'],
'concerns':['k-ah-n s-er1-n-z','nns vbz'],
'concert':['k-aa1-n s-er-t','nn'],
'concerted':['k-ah-n s-er1 t-ah-d','jj vbn'],
'concerti':['k-ah-n ch-eh1-r t-iy','nns'],
'concertmaster':['k-aa1-n s-er-t m-ae s-t-er','nn'],
'concerto':['k-ah-n ch-eh1-r t-ow','nn'],
'concession':['k-ah-n s-eh1 sh-ah-n','nn'],
'concessionaire':['k-ah-n s-eh sh-ah n-eh1-r','nn'],
'concierge':['k-aa-n s-iy eh1-r-zh','nn'],
'conciliator':['k-ah-n s-ih1 l-iy ey t-er','nn'],
'conciliatory':['k-ah-n s-ih1 l-iy ah t-ao r-iy','jj'],
'concise':['k-ah-n s-ay1-s','jj'],
'concisely':['k-ah-n s-ay1-s l-iy','rb'],
'conclave':['k-aa1-n k-l-ey-v','nn'],
'conclude':['k-ah-n k-l-uw1-d','vb vbp'],
'concluded':['k-ah-n k-l-uw1 d-ah-d','vbd vbn'],
'concludes':['k-ah-n k-l-uw1-d-z','vbz'],
'concluding':['k-ah-n k-l-uw1 d-ih-ng','vbg jj'],
'conclusion':['k-ah-n k-l-uw1 zh-ah-n','nn'],
'conclusive':['k-ah-n k-l-uw1 s-ih-v','jj'],
'conclusively':['k-ah-n k-l-uw1 s-ih-v l-iy','rb'],
'concoct':['k-ah-n k-aa1-k-t','vb'],
'concocted':['k-ah-n k-aa1-k t-ah-d','vbn vbd'],
'concoction':['k-ah-n k-aa1-k sh-ah-n','nn'],
'concomitant':['k-aa-n k-aa1 m-ah t-ah-n-t','jj nn'],
'concomitantly':['k-aa-n k-aa1 m-ah t-ah-n-t l-iy','rb'],
'concord':['k-aa1-n k-ao-r-d','nn'],
'concrete':['k-ah-n k-r-iy1-t','jj nn'],
'concretely':['k-aa1-n k-r-iy-t l-iy','rb'],
'concur':['k-ah-n k-er1','vbp vb'],
'concurred':['k-ah-n k-er1-d','vbd vbn'],
'concurrence':['k-ah-n k-er1 ah-n-s','nn'],
'concurrent':['k-ah-n k-er1 ah-n-t','jj'],
'concurrently':['k-ah-n k-er1 ah-n-t l-iy','rb'],
'concurring':['k-ah-n k-er1 ih-ng','vbg'],
'concurs':['k-ah-n k-er1-z','vbz'],
'concussion':['k-ah-n k-ah1 sh-ah-n','nn'],
'condemn':['k-ah-n d-eh1-m','vb vbp'],
'condemnation':['k-aa-n d-ah-m n-ey1 sh-ah-n','nn'],
'condemned':['k-ah-n d-eh1-m-d','vbn jj vbd'],
'condemning':['k-ah-n d-eh1 m-ih-ng','vbg'],
'condemns':['k-ah-n d-eh1-m-z','vbz'],
'condensate':['k-aa1-n d-ah-n s-ey-t','nn'],
'condensation':['k-aa-n d-ah-n s-ey1 sh-ah-n','nn'],
'condense':['k-ah-n d-eh1-n-s','vb'],
'condensed':['k-ah-n d-eh1-n-s-t','jj vbn'],
'condenser':['k-ah-n d-eh1-n s-er','nn'],
'condescending':['k-aa-n d-ih s-eh1-n d-ih-ng','jj vbg'],
'condescension':['k-aa-n d-ah s-eh1-n sh-ah-n','nn'],
'condiments':['k-aa1-n d-ah m-ah-n-t-s','nns'],
'condition':['k-ah-n d-ih1 sh-ah-n','nn vbp vb'],
'conditional':['k-ah-n d-ih1 sh-ah n-ah-l','jj nn'],
'conditionally':['k-ah-n d-ih1 sh-ah-n ah l-iy','rb'],
'conditioned':['k-ah-n d-ih1 sh-ah-n-d','vbn vbd jj'],
'conditioner':['k-ah-n d-ih1 sh-ah-n er','nn'],
'conditioning':['k-ah-n d-ih1 sh-ah-n ih-ng','nn vbg'],
'conditions':['k-ah-n d-ih1 sh-ah-n-z','nns'],
'condo':['k-aa1-n d-ow','nn'],
'condolences':['k-ah-n d-ow1 l-ah-n s-ah-z','nns'],
'condom':['k-aa1-n d-ah-m','nn'],
'condominium':['k-aa-n d-ah m-ih1 n-iy ah-m','nn'],
'condone':['k-ah-n d-ow1-n','vb'],
'condoned':['k-ah-n d-ow1-n-d','vbn vbd'],
'conducive':['k-ah-n d-uw1 s-ih-v','jj'],
'conduct':['k-ah-n d-ah1-k-t','nn vbp vb'],
'conducted':['k-ah-n d-ah1-k t-ah-d','vbn vbd'],
'conducting':['k-ah-n d-ah1-k t-ih-ng','vbg nn'],
'conduction':['k-ah-n d-ah1-k sh-ah-n','nn'],
'conductivity':['k-aa-n d-ah-k t-ih1 v-ah t-iy','nn'],
'conductor':['k-ah-n d-ah1-k t-er','nn'],
'conducts':['k-ah-n d-ah1-k-t-s','vbz'],
'conduit':['k-aa1-n d-uw ih-t','nn'],
'cone':['k-ow1-n','nn vb'],
'cones':['k-ow1-n-z','nns'],
'confabulation':['k-ah-n f-ae b-y-ah l-ey1 sh-ah-n','nn'],
'confectionary':['k-ah-n f-eh1-k sh-ah-n eh r-iy','jj'],
'confectioner':['k-ah-n f-eh1-k sh-ah-n er','nn'],
'confectionery':['k-ah-n f-eh1-k sh-ah-n eh r-iy','nn'],
'confederacy':['k-ah-n f-eh1 d-er ah s-iy','nn'],
'confederation':['k-ah-n f-eh d-er ey1 sh-ah-n','nn'],
'confer':['k-ah-n f-er1','vb vbp'],
'conferees':['k-aa-n f-er iy1-z','nns'],
'conference':['k-aa1-n f-er ah-n-s','nn'],
'conferred':['k-ah-n f-er1-d','vbn vbd'],
'conferring':['k-ah-n f-er1 ih-ng','vbg'],
'confers':['k-ah-n f-er1-z','vbz'],
'confess':['k-ah-n f-eh1-s','vb vbp'],
'confessed':['k-ah-n f-eh1-s-t','vbd vbn'],
'confesses':['k-ah-n f-eh1 s-ih-z','vbz'],
'confessing':['k-ah-n f-eh1 s-ih-ng','vbg'],
'confession':['k-ah-n f-eh1 sh-ah-n','nn'],
'confessional':['k-ah-n f-eh1 sh-ah n-ah-l','nn jj'],
'confidant':['k-aa1-n f-ah d-aa-n-t','nn'],
'confidante':['k-aa1-n f-ah d-ae-n-t','nn'],
'confide':['k-ah-n f-ay1-d','vb vbp'],
'confided':['k-ah-n f-ay1 d-ah-d','vbd vbn'],
'confidence':['k-aa1-n f-ah d-ah-n-s','nn'],
'confident':['k-aa1-n f-ah d-ah-n-t','jj'],
'confidential':['k-aa-n f-ah d-eh1-n sh-ah-l','jj'],
'confidentiality':['k-aa-n f-ah d-eh-n sh-iy ae1 l-ah t-iy','nn'],
'confidentially':['k-aa-n f-ah d-eh1-n sh-ah l-iy','rb'],
'confidently':['k-aa1-n f-ah d-ah-n-t l-iy','rb'],
'confides':['k-ah-n f-ay1-d-z','vbz'],
'confiding':['k-ah-n f-ay1 d-ih-ng','vbg jj'],
'configuration':['k-ah-n f-ih g-y-er ey1 sh-ah-n','nn'],
'confine':['k-ah-n f-ay1-n','vb vbp'],
'confined':['k-ah-n f-ay1-n-d','vbn jj vbd'],
'confinement':['k-ah-n f-ay1-n m-ah-n-t','nn'],
'confines':['k-aa1-n f-ay-n-z','nns'],
'confining':['k-ah-n f-ay1 n-ih-ng','vbg'],
'confirm':['k-ah-n f-er1-m','vb vbp'],
'confirmation':['k-aa-n f-er m-ey1 sh-ah-n','nn'],
'confirmed':['k-ah-n f-er1-m-d','vbd vbn jj'],
'confirming':['k-ah-n f-er1 m-ih-ng','vbg'],
'confirms':['k-ah-n f-er1-m-z','vbz'],
'confiscate':['k-aa1-n f-ah s-k-ey-t','vb'],
'confiscated':['k-aa1-n f-ah s-k-ey t-ah-d','vbn vbd'],
'confiscating':['k-aa1-n f-ah s-k-ey t-ih-ng','vbg'],
'confiscation':['k-aa-n f-ah s-k-ey1 sh-ah-n','nn'],
'confiscatory':['k-ah-n f-ih1-s k-ah t-ao r-iy','jj'],
'conflagration':['k-aa-n f-l-ah g-r-ey1 sh-ah-n','nn'],
'conflict':['k-aa1-n f-l-ih-k-t','nn vb vbp'],
'conflicted':['k-ah-n f-l-ih1-k t-ih-d','vbd vbn'],
'conflicting':['k-ah-n f-l-ih1-k t-ih-ng','vbg jj'],
'conflicts':['k-ah-n f-l-ih1-k-t-s','nns vbz'],
'confluence':['k-aa1-n f-l-uw ah-n-s','nn'],
'conform':['k-ah-n f-ao1-r-m','vb vbp'],
'conformance':['k-ah-n f-ao1-r m-ah-n-s','nn'],
'conformational':['k-aa-n f-er m-ey1 sh-ah n-ah-l','jj'],
'conformed':['k-ah-n f-ao1-r-m-d','vbn vbd'],
'conforming':['k-ah-n f-ao1-r m-ih-ng','vbg'],
'conformist':['k-ah-n f-ao1-r m-ih-s-t','jj nn'],
'conformity':['k-ah-n f-ao1-r m-ah t-iy','nn'],
'conforms':['k-ah-n f-ao1-r-m-z','vbz'],
'confound':['k-aa-n f-aw1-n-d','vb'],
'confounded':['k-ah-n f-aw1-n d-ih-d','vbd vbn'],
'confounding':['k-ah-n f-aw1-n d-ih-ng','vbg'],
'confront':['k-ah-n f-r-ah1-n-t','vb vbp'],
'confrontation':['k-aa-n f-r-ah-n t-ey1 sh-ah-n','nn'],
'confrontational':['k-aa-n f-r-ah-n t-ey1 sh-ah n-ah-l','jj'],
'confronted':['k-ah-n f-r-ah1-n t-ah-d','vbn vbd'],
'confronting':['k-ah-n f-r-ah1-n t-ih-ng','vbg'],
'confronts':['k-ah-n f-r-ah1-n-t-s','vbz'],
'confuse':['k-ah-n f-y-uw1-z','vb vbp'],
'confused':['k-ah-n f-y-uw1-z-d','vbn jj vbd'],
'confuses':['k-ah-n f-y-uw1 z-ih-z','vbz'],
'confusing':['k-ah-n f-y-uw1 z-ih-ng','jj vbg'],
'confusion':['k-ah-n f-y-uw1 zh-ah-n','nn'],
'congeal':['k-ah-n jh-iy1-l','vb'],
'congealed':['k-ah-n jh-iy1-l-d','vbd vbn'],
'congenial':['k-ah-n jh-iy1 n-y-ah-l','jj'],
'congeniality':['k-ah-n jh-iy n-iy ae1 l-ah t-iy','nn'],
'congenital':['k-ah-n jh-eh1 n-ah t-ah-l','jj'],
'congested':['k-ah-n jh-eh1 s-t-ah-d','jj vbn'],
'congestion':['k-ah-n jh-eh1-s ch-ah-n','nn'],
'congestive':['k-ah-n jh-eh1 s-t-ih-v','jj'],
'conglomerate':['k-ah-n g-l-aa1 m-er ah-t','nn jj'],
'congratulate':['k-ah-n g-r-ae1 ch-ah l-ey-t','vbp vb'],
'congratulated':['k-ah-n g-r-ae1 ch-ah l-ey t-ih-d','vbn vbd'],
'congratulating':['k-ah-n g-r-ae1 ch-ah l-ey t-ih-ng','nn'],
'congratulation':['k-ah-n g-r-ae ch-ah l-ey1 sh-ah-n','nn'],
'congratulations':['k-ah-n g-r-ae ch-ah l-ey1 sh-ah-n-z','nns uh'],
'congratulatory':['k-ah-n g-r-ae1 ch-ah l-ah t-ao r-iy','jj'],
'congregate':['k-aa1-ng g-r-ah g-ey-t','vb'],
'congregated':['k-aa1-ng g-r-ih g-ey t-ih-d','vbd'],
'congregation':['k-aa-ng g-r-ah g-ey1 sh-ah-n','nn'],
'congregational':['k-aa-ng g-r-ah g-ey1 sh-ah n-ah-l','jj'],
'congress':['k-aa1-ng g-r-ah-s','nn'],
'congressional':['k-ah-n g-r-eh1 sh-ah n-ah-l','jj'],
'congressionally':['k-ah-n g-r-eh1 sh-ah-n ah l-iy','rb'],
'congressman':['k-aa1-ng g-r-ah-s m-ah-n','nn'],
'congruence':['k-ao1-n g-r-uw ah-n-s','nn'],
'conjecture':['k-ah-n jh-eh1-k ch-er','nn'],
'conjectures':['k-ah-n jh-eh1-k ch-er-z','nns vbz'],
'conjunction':['k-ah-n jh-ah1-ng-k sh-ah-n','nn'],
'conjure':['k-aa1-n jh-er','vb vbp'],
'conjured':['k-aa1-n jh-er-d','vbn vbd'],
'conjures':['k-aa1-n jh-er-z','vbz'],
'conjuring':['k-aa1-n jh-er ih-ng','vbg'],
'connect':['k-ah n-eh1-k-t','vb vbp'],
'connected':['k-ah n-eh1-k t-ah-d','vbn jj vbd'],
'connecting':['k-ah n-eh1-k t-ih-ng','vbg jj'],
'connection':['k-ah n-eh1-k sh-ah-n','nn'],
'connective':['k-ah n-eh1-k t-ih-v','jj'],
'connector':['k-ah n-eh1-k t-er','nn'],
'connects':['k-ah n-eh1-k-t-s','vbz'],
'conned':['k-aa1-n-d','vbn'],
'conning':['k-aa1 n-ih-ng','vbg'],
'connivance':['k-ah n-ay1 v-ah-n-s','nn'],
'connoisseur':['k-aa n-ah s-er1','nn'],
'connotation':['k-aa n-ah t-ey1 sh-ah-n','nn'],
'connote':['k-ah n-ow1-t','vb vbp'],
'connotes':['k-ah n-ow1-t-s','vbz'],
'conquer':['k-aa1-ng k-er','vb vbp'],
'conquered':['k-aa1-ng k-er-d','vbn vbd'],
'conquering':['k-aa1-ng k-er ih-ng','vbg'],
'conqueror':['k-aa1-ng k-er er','nn'],
'conquest':['k-aa1-ng k-w-eh-s-t','nn'],
'cons':['k-aa1-n-z','nns'],
'conscience':['k-aa1-n sh-ah-n-s','nn'],
'conscientious':['k-aa-n sh-iy eh1-n sh-ah-s','jj'],
'conscious':['k-aa1-n sh-ah-s','jj nn'],
'consciously':['k-aa1-n sh-ah-s l-iy','rb'],
'consciousness':['k-aa1-n sh-ah-s n-ah-s','nn'],
'conscript':['k-aa1-n s-k-r-ih-p-t','nn jj'],
'conscripted':['k-ah-n s-k-r-ih1-p t-ih-d','vbn'],
'conscription':['k-ah-n s-k-r-ih1-p sh-ah-n','nn'],
'consecutive':['k-ah-n s-eh1 k-y-ah t-ih-v','jj'],
'consensual':['k-ah-n s-eh1-n s-uw ah-l','jj'],
'consensus':['k-ah-n s-eh1-n s-ah-s','nn'],
'consent':['k-ah-n s-eh1-n-t','nn vb'],
'consented':['k-ah-n s-eh1-n t-ih-d','vbd vbn'],
'consenting':['k-ah-n s-eh1-n t-ih-ng','vbg jj'],
'consents':['k-ah-n s-eh1-n-t-s','nns vbz'],
'consequence':['k-aa1-n s-ah k-w-ah-n-s','nn'],
'consequent':['k-aa1-n s-ah k-w-ah-n-t','jj'],
'consequential':['k-aa-n s-ah k-w-eh1-n ch-ah-l','jj'],
'consequently':['k-aa1-n s-ah k-w-ah-n-t l-iy','rb'],
'conservancy':['k-ah-n s-er1 v-ah-n s-iy','nn'],
'conservation':['k-aa-n s-er v-ey1 sh-ah-n','nn'],
'conservationist':['k-aa-n s-er v-ey1 sh-ah-n ah-s-t','nn'],
'conservatism':['k-ah-n s-er1 v-ah t-ih z-ah-m','nn'],
'conservative':['k-ah-n s-er1 v-ah t-ih-v','jj nn'],
'conservatively':['k-ah-n s-er1 v-ah t-ih-v l-iy','rb'],
'conservator':['k-ah-n s-er1 v-ah t-er','nn'],
'conservatorship':['k-ah-n s-er1 v-ah t-er sh-ih-p','nn'],
'conservatory':['k-ah-n s-er1 v-ah t-ao r-iy','nn'],
'conserve':['k-ah-n s-er1-v','vb'],
'conserved':['k-ah-n s-er1-v-d','vbn'],
'conserving':['k-ah-n s-er1 v-ih-ng','vbg'],
'consider':['k-ah-n s-ih1 d-er','vb vbp'],
'considerable':['k-ah-n s-ih1 d-er ah b-ah-l','jj'],
'considerably':['k-ah-n s-ih1 d-er ah b-l-iy','rb'],
'considerate':['k-ah-n s-ih1 d-er ah-t','jj'],
'consideration':['k-ah-n s-ih d-er ey1 sh-ah-n','nn'],
'considered':['k-ah-n s-ih1 d-er-d','vbn vbd jj'],
'considering':['k-ah-n s-ih1 d-er ih-ng','vbg'],
'considers':['k-ah-n s-ih1 d-er-z','vbz'],
'consign':['k-ah-n s-ay1-n','vb'],
'consigned':['k-ah-n s-ay1-n-d','vbd vbn'],
'consist':['k-ah-n s-ih1-s-t','vb vbp'],
'consisted':['k-ah-n s-ih1 s-t-ah-d','vbd vbn'],
'consistence':['k-ah-n s-ih1 s-t-ah-n-s','nn'],
'consistency':['k-ah-n s-ih1 s-t-ah-n s-iy','nn'],
'consistent':['k-ah-n s-ih1 s-t-ah-n-t','jj'],
'consistently':['k-ah-n s-ih1 s-t-ah-n-t l-iy','rb'],
'consisting':['k-ah-n s-ih1 s-t-ih-ng','vbg'],
'consists':['k-ah-n s-ih1-s-t-s','vbz'],
'consolation':['k-aa-n s-ah l-ey1 sh-ah-n','nn'],
'console':['k-aa1-n s-ow-l','vb vbp'],
'consoled':['k-ah-n s-ow1-l-d','vbd vbn'],
'consoles':['k-ah-n s-ow1-l-z','nns vbz'],
'consolidate':['k-ah-n s-aa1 l-ih d-ey-t','vb vbp'],
'consolidated':['k-ah-n s-aa1 l-ah d-ey t-ah-d','jj vbd vbn'],
'consolidates':['k-ah-n s-aa1 l-ih d-ey-t-s','vbz'],
'consolidating':['k-ah-n s-aa1 l-ah d-ey t-ih-ng','vbg jj'],
'consolidation':['k-ah-n s-aa l-ah d-ey1 sh-ah-n','nn'],
'consoling':['k-ah-n s-ow1 l-ih-ng','vbg'],
'consonant':['k-aa1-n s-ah n-ah-n-t','jj nn'],
'consonantal':['k-aa-n s-ah n-aa1-n t-ah-l','jj'],
'consortia':['k-ah-n s-ao1-r sh-ah','nns nn'],
'consorting':['k-ah-n s-ao1-r t-ih-ng','vbg'],
'consortium':['k-ah-n s-ao1-r sh-iy ah-m','nn'],
'conspicuous':['k-ah-n s-p-ih1 k-y-uw ah-s','jj'],
'conspicuously':['k-ah-n s-p-ih1 k-y-uw ah-s l-iy','rb'],
'conspiracy':['k-ah-n s-p-ih1 r-ah s-iy','nn'],
'conspirator':['k-ah-n s-p-ih1 r-ah t-er','nn'],
'conspiratorial':['k-ah-n s-p-ih r-ah t-ao1 r-iy ah-l','jj'],
'conspire':['k-ah-n s-p-ay1 er','vbp vb'],
'conspired':['k-ah-n s-p-ay1 er-d','vbd vbn'],
'conspiring':['k-ah-n s-p-ay1 r-ih-ng','vbg'],
'constables':['k-aa1-n s-t-ah b-ah-l-z','nns'],
'constancy':['k-aa1-n s-t-ah-n s-iy','nn'],
'constant':['k-aa1-n s-t-ah-n-t','jj nn rb'],
'constantly':['k-aa1-n s-t-ah-n-t l-iy','rb'],
'constellation':['k-aa-n s-t-ah l-ey1 sh-ah-n','nn'],
'consternation':['k-aa-n s-t-er n-ey1 sh-ah-n','nn'],
'constipation':['k-aa-n s-t-ah p-ey1 sh-ah-n','nn'],
'constituency':['k-ah-n s-t-ih1 ch-uw ah-n s-iy','nn'],
'constituent':['k-ah-n s-t-ih1 ch-uw ah-n-t','nn jj'],
'constitute':['k-aa1-n s-t-ah t-uw-t','vbp vb'],
'constituted':['k-aa1-n s-t-ah t-uw t-ah-d','vbd vbn'],
'constitutes':['k-aa1-n s-t-ah t-uw-t-s','vbz'],
'constituting':['k-aa1-n s-t-ah t-uw t-ih-ng','vbg'],
'constitution':['k-aa-n s-t-ah t-uw1 sh-ah-n','nn'],
'constitutional':['k-aa-n s-t-ah t-uw1 sh-ah n-ah-l','jj'],
'constitutionality':['k-aa-n s-t-ih t-uw sh-ah n-ae1 l-ih t-iy','nn'],
'constitutionally':['k-aa-n s-t-ah t-uw1 sh-ah-n ah-l iy','rb'],
'constrain':['k-ah-n s-t-r-ey1-n','vb'],
'constrained':['k-ah-n s-t-r-ey1-n-d','vbn jj'],
'constraining':['k-ah-n s-t-r-ey1 n-ih-ng','vbg'],
'constrains':['k-ah-n s-t-r-ey1-n-z','vbz'],
'constraint':['k-ah-n s-t-r-ey1-n-t','nn'],
'constricted':['k-ah-n s-t-r-ih1-k t-ah-d','jj vbd vbn'],
'constricting':['k-ah-n s-t-r-ih1-k t-ih-ng','vbg'],
'constriction':['k-ah-n s-t-r-ih1-k sh-ah-n','nn'],
'constrictor':['k-ah-n s-t-r-ih1-k t-er','nn'],
'construct':['k-ah-n s-t-r-ah1-k-t','vb vbp'],
'constructed':['k-ah-n s-t-r-ah1-k t-ah-d','vbn vbd'],
'constructing':['k-ah-n s-t-r-ah1-k t-ih-ng','vbg'],
'construction':['k-ah-n s-t-r-ah1-k sh-ah-n','nn'],
'constructive':['k-ah-n s-t-r-ah1-k t-ih-v','jj'],
'constructively':['k-ah-n s-t-r-ah1-k t-ih-v l-iy','rb'],
'constructs':['k-ah-n s-t-r-ah1-k-t-s','vbz'],
'construe':['k-ah-n s-t-r-uw1','vb vbp'],
'construed':['k-ah-n s-t-r-uw1-d','vbn vbd'],
'consul':['k-aa1-n s-ah-l','nn'],
'consular':['k-aa1-n s-ah l-er','jj'],
'consulate':['k-aa1-n s-ah l-ah-t','nn'],
'consult':['k-ah-n s-ah1-l-t','vb'],
'consultancy':['k-ah-n s-ah1-l t-ah-n s-iy','nn'],
'consultant':['k-ah-n s-ah1-l t-ah-n-t','nn'],
'consultation':['k-aa-n s-ah-l t-ey1 sh-ah-n','nn'],
'consultative':['k-ah-n s-ah1-l t-ah t-ih-v','jj'],
'consulted':['k-ah-n s-ah1-l t-ah-d','vbn vbd'],
'consulting':['k-ah-n s-ah1-l t-ih-ng','nn jj vbg'],
'consume':['k-ah-n s-uw1-m','vbp vb'],
'consumed':['k-ah-n s-uw1-m-d','vbn vbd'],
'consumer':['k-ah-n s-uw1 m-er','nn'],
'consumerism':['k-ah-n s-uw1 m-er ih z-ah-m','nn'],
'consumes':['k-ah-n s-uw1-m-z','vbz'],
'consuming':['k-ah-n s-uw1 m-ih-ng','vbg nn'],
'consummate':['k-aa1-n s-ah m-ah-t','jj'],
'consummated':['k-aa1-n s-ah m-ey t-ah-d','vbn vbd'],
'consummation':['k-aa-n s-ah m-ey1 sh-ah-n','nn'],
'consumption':['k-ah-n s-ah1-m-p sh-ah-n','nn'],
'contact':['k-aa1-n t-ae-k-t','nn jj vb'],
'contacted':['k-aa1-n t-ae-k t-ih-d','vbn vbd'],
'contacting':['k-aa1-n t-ae-k t-ih-ng','vbg'],
'contacts':['k-aa1-n t-ae-k-t-s','nns vbz'],
'contagion':['k-ah-n t-ey1 jh-ah-n','nn'],
'contagious':['k-ah-n t-ey1 jh-ah-s','jj'],
'contain':['k-ah-n t-ey1-n','vb vbp'],
'contained':['k-ah-n t-ey1-n-d','vbd vbn jj'],
'container':['k-ah-n t-ey1 n-er','nn'],
'containerboard':['k-ah-n t-ey1 n-er b-ao-r-d','nn'],
'containing':['k-ah-n t-ey1 n-ih-ng','vbg'],
'containment':['k-ah-n t-ey1-n m-ah-n-t','nn'],
'contains':['k-ah-n t-ey1-n-z','vbz'],
'contaminants':['k-ah-n t-ae1 m-ah n-ah-n-t-s','nns'],
'contaminate':['k-ah-n t-ae1 m-ah n-ey-t','vb'],
'contaminated':['k-ah-n t-ae1 m-ah n-ey t-ah-d','vbn vbd jj'],
'contaminating':['k-ah-n t-ae1 m-ah n-ey t-ih-ng','vbg'],
'contamination':['k-ah-n t-ae m-ah n-ey1 sh-ah-n','nn'],
'contemplate':['k-aa1-n t-ah-m p-l-ey-t','vb vbp'],
'contemplated':['k-aa1-n t-ah-m p-l-ey t-ih-d','vbn jj vbd'],
'contemplates':['k-aa1-n t-ah-m p-l-ey-t-s','vbz'],
'contemplating':['k-aa1-n t-ah-m p-l-ey t-ih-ng','vbg'],
'contemplation':['k-aa-n t-ah-m p-l-ey1 sh-ah-n','nn'],
'contemplative':['k-ah-n t-eh1-m p-l-ah t-ih-v','jj'],
'contemporaneous':['k-ah-n t-eh-m p-er ey1 n-iy ah-s','jj'],
'contemporary':['k-ah-n t-eh1-m p-er eh r-iy','jj nn'],
'contempt':['k-ah-n t-eh1-m-p-t','nn'],
'contemptible':['k-ah-n t-eh1-m-p t-ah b-ah-l','jj'],
'contemptuous':['k-ah-n t-eh1-m-p ch-uw ah-s','jj'],
'contemptuously':['k-ah-n t-eh1-m-p ch-w-ah-s l-iy','rb'],
'contend':['k-ah-n t-eh1-n-d','vbp vb'],
'contended':['k-ah-n t-eh1-n d-ih-d','vbd vbn'],
'contender':['k-ah-n t-eh1-n d-er','nn'],
'contending':['k-ah-n t-eh1-n d-ih-ng','vbg'],
'contends':['k-ah-n t-eh1-n-d-z','vbz'],
'content':['k-aa1-n t-eh-n-t','nn jj vb'],
'contented':['k-ah-n t-eh1-n t-ah-d','vbn jj'],
'contentedly':['k-ah-n t-eh1-n t-ah-d l-iy','rb'],
'contention':['k-ah-n t-eh1-n sh-ah-n','nn'],
'contentious':['k-ah-n t-eh1-n sh-ah-s','jj'],
'contentment':['k-ah-n t-eh1-n-t m-ah-n-t','nn'],
'contents':['k-aa1-n t-eh-n-t-s','nns'],
'contest':['k-aa1-n t-eh-s-t','nn vb'],
'contestant':['k-ah-n t-eh1 s-t-ah-n-t','nn'],
'contested':['k-ah-n t-eh1 s-t-ah-d','vbn jj vbd'],
'contesting':['k-ah-n t-eh1 s-t-ih-ng','vbg'],
'contests':['k-aa1-n t-eh-s-t-s','nns'],
'context':['k-aa1-n t-eh-k-s-t','nn'],
'contiguous':['k-ah-n t-ih1 g-y-uw ah-s','jj'],
'continent':['k-aa1-n t-ah n-ah-n-t','nn'],
'continental':['k-aa-n t-ah n-eh1-n t-ah-l','jj'],
'continentally':['k-aa-n t-ah n-eh1-n t-ah l-iy','rb'],
'contingency':['k-ah-n t-ih1-n jh-ah-n s-iy','nn'],
'contingent':['k-ah-n t-ih1-n jh-ah-n-t','jj nn'],
'continual':['k-ah-n t-ih1 n-y-uw ah-l','jj'],
'continually':['k-ah-n t-ih1 n-y-uw ah l-iy','rb'],
'continuance':['k-ah-n t-ih1 n-y-uw ah-n-s','nn'],
'continuation':['k-ah-n t-ih n-y-uw ey1 sh-ah-n','nn'],
'continue':['k-ah-n t-ih1 n-y-uw','vb vbp'],
'continued':['k-ah-n t-ih1 n-y-uw-d','vbd jj vbn'],
'continues':['k-ah-n t-ih1 n-y-uw-z','vbz'],
'continuing':['k-ah-n t-ih1 n-y-uw ih-ng','vbg jj'],
'continuity':['k-aa-n t-ah n-uw1 ah t-iy','nn'],
'continuous':['k-ah-n t-ih1 n-y-uw ah-s','jj'],
'continuously':['k-ah-n t-ih1 n-y-uw ah-s l-iy','rb'],
'continuum':['k-ah-n t-ih1 n-y-uw ah-m','nn'],
'contorted':['k-ah-n t-ao1-r t-ah-d','jj'],
'contortion':['k-ah-n t-ao1-r sh-ah-n','nn'],
'contour':['k-aa1-n t-uh-r','nn'],
'contraband':['k-aa1-n t-r-ah b-ae-n-d','jj nn'],
'contraception':['k-aa-n t-r-ah s-eh1-p sh-ah-n','nn'],
'contraceptive':['k-aa-n t-r-ah s-eh1-p t-ih-v','jj nn'],
'contract':['k-aa1-n t-r-ae-k-t','nn vb vbp'],
'contracted':['k-aa1-n t-r-ae-k t-ah-d','vbd vbn'],
'contracting':['k-aa1-n t-r-ae-k t-ih-ng','nn vbg'],
'contraction':['k-ah-n t-r-ae1-k sh-ah-n','nn'],
'contractor':['k-aa1-n t-r-ae-k t-er','nn'],
'contracts':['k-aa1-n t-r-ae-k-t-s','nns nn vbz'],
'contractual':['k-ah-n t-r-ae1-k ch-uw ah-l','jj'],
'contractually':['k-ah-n t-r-ae1-k ch-uw ah l-iy','rb'],
'contradict':['k-aa-n t-r-ah d-ih1-k-t','vb vbp'],
'contradicted':['k-aa-n t-r-ah d-ih1-k t-ah-d','vbd vbn'],
'contradicting':['k-aa-n t-r-ah d-ih1-k t-ih-ng','vbg'],
'contradiction':['k-aa-n t-r-ah d-ih1-k sh-ah-n','nn'],
'contradictorily':['k-aa-n t-r-ah d-ih1-k t-er ah l-iy','rb'],
'contradictory':['k-aa-n t-r-ah d-ih1-k t-er iy','jj nn'],
'contradicts':['k-aa-n t-r-ah d-ih1-k-t-s','vbz'],
'contraption':['k-ah-n t-r-ae1-p sh-ah-n','nn'],
'contrarian':['k-aa-n t-r-eh1 r-iy ah-n','jj nn'],
'contrary':['k-aa1-n t-r-eh r-iy','jj nn'],
'contrast':['k-aa1-n t-r-ae-s-t','nn vb vbp'],
'contrasted':['k-ah-n t-r-ae1 s-t-ah-d','vbn vbd'],
'contrasting':['k-ah-n t-r-ae1 s-t-ih-ng','vbg jj'],
'contrasts':['k-aa1-n t-r-ae-s-t-s','nns vbz'],
'contretemps':['k-aa1-n t-r-ah t-eh-m-p-s','nn'],
'contribute':['k-ah-n t-r-ih1 b-y-uw-t','vb vbp'],
'contributed':['k-ah-n t-r-ih1 b-y-uw t-ih-d','vbd vbn'],
'contributes':['k-ah-n t-r-ih1 b-y-uw-t-s','vbz'],
'contributing':['k-ah-n t-r-ih1 b-y-uw t-ih-ng','vbg jj'],
'contribution':['k-aa-n t-r-ah b-y-uw1 sh-ah-n','nn'],
'contributor':['k-ah-n t-r-ih1 b-y-ah t-er','nn'],
'contributory':['k-ah-n t-r-ih1 b-y-ah t-ao r-iy','jj'],
'contrite':['k-ah-n t-r-ay1-t','jj'],
'contrition':['k-ah-n t-r-ih1 sh-ah-n','nn'],
'contrivance':['k-ah-n t-r-ay1 v-ah-n-s','nn'],
'contrive':['k-ah-n t-r-ay1-v','vb'],
'contrived':['k-ah-n t-r-ay1-v-d','vbn jj'],
'control':['k-ah-n t-r-ow1-l','nn jj vb vbp'],
'controllable':['k-ah-n t-r-ow1 l-ah b-ah-l','jj'],
'controlled':['k-ah-n t-r-ow1-l-d','vbn jj vbd'],
'controller':['k-ah-n t-r-ow1 l-er','nn'],
'controlling':['k-ah-n t-r-ow1 l-ih-ng','vbg jj nn'],
'controls':['k-ah-n t-r-ow1-l-z','nns vbz'],
'controversial':['k-aa-n t-r-ah v-er1 sh-ah-l','jj'],
'controversy':['k-aa1-n t-r-ah v-er s-iy','nn'],
'conundrum':['k-ah n-ah1-n d-r-ah-m','nn'],
'convalescence':['k-aa-n v-ah l-eh1 s-ah-n-s','nn'],
'convene':['k-ah-n v-iy1-n','vb vbp'],
'convened':['k-ah-n v-iy1-n-d','vbd vbn'],
'convenes':['k-ah-n v-iy1-n-z','vbz'],
'convenience':['k-ah-n v-iy1 n-y-ah-n-s','nn'],
'convenient':['k-ah-n v-iy1 n-y-ah-n-t','jj'],
'conveniently':['k-ah-n v-iy1 n-y-ah-n-t l-iy','rb'],
'convening':['k-ah-n v-iy1 n-ih-ng','vbg nn'],
'convent':['k-aa1-n v-ah-n-t','nn'],
'convention':['k-ah-n v-eh1-n sh-ah-n','nn'],
'conventional':['k-ah-n v-eh1-n sh-ah n-ah-l','jj'],
'conventionally':['k-ah-n v-eh1-n sh-ah-n ah l-iy','rb'],
'conventioneers':['k-ah-n v-eh-n sh-ah n-ih1-r-z','nns'],
'converge':['k-ah-n v-er1-jh','vb vbp'],
'converged':['k-ah-n v-er1-jh-d','vbd'],
'converging':['k-ah-n v-er1 jh-ih-ng','vbg'],
'conversant':['k-ah-n v-er1 s-ah-n-t','nn'],
'conversation':['k-aa-n v-er s-ey1 sh-ah-n','nn'],
'conversational':['k-aa-n v-er s-ey1 sh-ah n-ah-l','jj'],
'converse':['k-aa1-n v-er-s','vb nn'],
'conversely':['k-aa1-n v-er-s l-iy','rb'],
'conversing':['k-ah-n v-er1 s-ih-ng','vbg'],
'conversion':['k-ah-n v-er1 zh-ah-n','nn'],
'convert':['k-aa1-n v-er-t','vb vbp nn'],
'converted':['k-ah-n v-er1 t-ih-d','vbn jj vbd'],
'converter':['k-ah-n v-er1 t-er','nn'],
'convertibility':['k-aa-n v-er t-ih b-ih1 l-ih t-iy','nn'],
'convertible':['k-ah-n v-er1 t-ah b-ah-l','jj nn'],
'converting':['k-ah-n v-er1 t-ih-ng','vbg'],
'converts':['k-aa1-n v-er-t-s','nns vbz'],
'convex':['k-ah-n v-eh1-k-s','jj'],
'convey':['k-ah-n v-ey1','vb vbp'],
'conveyance':['k-ah-n v-ey1 ah-n-s','nn'],
'conveyed':['k-ah-n v-ey1-d','vbd vbn'],
'conveying':['k-ah-n v-ey1 ih-ng','vbg'],
'conveyor':['k-ah-n v-ey1 er','nn'],
'conveys':['k-ah-n v-ey1-z','vbz'],
'convict':['k-aa1-n v-ih-k-t','nn vb'],
'convicted':['k-ah-n v-ih1-k t-ah-d','vbn vbd'],
'convicting':['k-ah-n v-ih1-k t-ih-ng','vbg'],
'conviction':['k-ah-n v-ih1-k sh-ah-n','nn'],
'convicts':['k-aa1-n v-ih-k-t-s','nns'],
'convince':['k-ah-n v-ih1-n-s','vb vbp'],
'convinced':['k-ah-n v-ih1-n-s-t','vbn jj vbd'],
'convinces':['k-ah-n v-ih1-n s-ih-z','vbz'],
'convincing':['k-ah-n v-ih1-n s-ih-ng','jj vbg nn'],
'convincingly':['k-ah-n v-ih1-n s-ih-ng l-iy','rb'],
'convivial':['k-ah-n v-ih1 v-iy ah-l','jj'],
'convocation':['k-aa-n v-ah k-ey1 sh-ah-n','nn'],
'convoluted':['k-aa1-n v-ah l-uw t-ah-d','jj vbn'],
'convoy':['k-aa1-n v-oy','nn'],
'convulsion':['k-ah-n v-ah1-l sh-ah-n','nn'],
'convulsive':['k-ah-n v-ah1-l s-ih-v','jj'],
'cook':['k-uh1-k','nn vb vbp'],
'cookbook':['k-uh1-k b-uh-k','nn'],
'cooked':['k-uh1-k-t','vbn vbd'],
'cooker':['k-uh1 k-er','nn'],
'cookie':['k-uh1 k-iy','nn'],
'cooking':['k-uh1 k-ih-ng','nn vbg jj'],
'cooks':['k-uh1-k-s','nns'],
'cookware':['k-uh1-k w-eh-r','nn'],
'cool':['k-uw1-l','jj nn rb vb vbp'],
'coolant':['k-uw1 l-ah-n-t','nn'],
'cooled':['k-uw1-l-d','vbn jj vbd'],
'cooler':['k-uw1 l-er','jjr nn'],
'coolest':['k-uw1 l-ah-s-t','jjs'],
'cooling':['k-uw1 l-ih-ng','nn vbg'],
'coolly':['k-uw1 l-iy','rb'],
'coolness':['k-uw1-l n-ah-s','nn'],
'cools':['k-uw1-l-z','vbz'],
'coop':['k-uw1-p','nn'],
'cooped':['k-uw1-p-t','jj nn vbn'],
'cooperate':['k-ow aa1 p-er ey-t','vb vbp'],
'cooperated':['k-ow aa1 p-er ey t-ah-d','vbn vbd'],
'cooperates':['k-ow aa1 p-er ey-t-s','vbz'],
'cooperating':['k-ow aa1 p-er ey t-ih-ng','vbg'],
'cooperation':['k-ow aa p-er ey1 sh-ah-n','nn'],
'cooperative':['k-ow aa1 p-er ey t-ih-v','jj nn'],
'cooperatively':['k-ow aa1 p-r-ah t-ih-v l-iy','rb'],
'coordinate':['k-ow ao1-r d-ah-n ah-t','vb jj nn vbp'],
'coordinated':['k-ow ao1-r d-ah n-ey t-ih-d','vbn vbd jj'],
'coordinates':['k-ow ao1-r d-ah-n ah-t-s','nns vbz'],
'coordinating':['k-ow ao1-r d-ah n-ey t-ih-ng','vbg nn'],
'coordination':['k-ow ao-r d-ah-n ey1 sh-ah-n','nn'],
'coordinator':['k-ow ao1-r d-ah-n ey t-er','nn'],
'cop':['k-aa1-p','nn'],
'cope':['k-ow1-p','vb nn'],
'copes':['k-ow1-p-s','vbz'],
'copied':['k-aa1 p-iy-d','vbn vbd'],
'copier':['k-aa1 p-iy er','nn'],
'copies':['k-aa1 p-iy-z','nns'],
'coping':['k-ow1 p-ih-ng','vbg nn'],
'copious':['k-ow1 p-iy ah-s','jj'],
'copper':['k-aa1 p-er','nn'],
'coppery':['k-aa1 p-er iy','jj nn'],
'copra':['k-ow1 p-r-ah','nn'],
'copy':['k-aa1 p-iy','nn vbp vb'],
'copycat':['k-aa1 p-iy k-ae-t','nn'],
'copying':['k-aa1 p-iy ih-ng','nn vbg'],
'copyright':['k-aa1 p-iy r-ay-t','nn jj'],
'copyrighted':['k-aa1 p-iy r-ay t-ih-d','vbn'],
'copywriter':['k-aa1 p-iy r-ay t-er','nn'],
'coral':['k-ao1 r-ah-l','jj'],
'cord':['k-ao1-r-d','nn'],
'corded':['k-ao1-r d-ah-d','vbn'],
'cordial':['k-ao1-r jh-ah-l','jj nn'],
'cordless':['k-ao1-r-d l-ah-s','jj'],
'cordon':['k-ao1-r d-ah-n','nn vb'],
'corduroy':['k-ao1-r d-er oy','nn'],
'core':['k-ao1-r','nn jj'],
'coriander':['k-ao r-iy ae1-n d-er','nn'],
'corinthian':['k-er ih1-n th-iy ah-n','jj'],
'cork':['k-ao1-r-k','nn'],
'corkscrew':['k-ao1-r-k s-k-r-uw','nn'],
'corn':['k-ao1-r-n','nn'],
'corne':['k-ao1-r-n','nn'],
'cornea':['k-ao1-r n-iy ah','nn'],
'corneal':['k-ao-r n-iy1-l','jj'],
'corner':['k-ao1-r n-er','nn jj vb'],
'cornered':['k-ao1-r n-er-d','vbn vbd jj'],
'cornering':['k-ao1-r n-er ih-ng','vbg'],
'corners':['k-ao1-r n-er-z','nns'],
'cornerstone':['k-ao1-r n-er s-t-ow-n','nn'],
'cornfield':['k-ao1-r-n f-iy-l-d','nn'],
'cornmeal':['k-ao1-r-n m-iy-l','nn'],
'cornstarch':['k-ao1-r-n s-t-aa-r-ch','nn'],
'cornucopia':['k-ao-r n-ah k-ow1 p-iy ah','nn'],
'corny':['k-ao1-r n-iy','jj'],
'corollary':['k-ao1 r-ah l-eh r-iy','nn'],
'corona':['k-er ow1 n-ah','nn'],
'coronary':['k-ao1 r-ah n-eh r-iy','jj'],
'coroner':['k-ao1 r-ah n-er','nn'],
'corporal':['k-ao1-r p-er ah-l','nn'],
'corporate':['k-ao1-r p-er ah-t','jj'],
'corporatewide':['k-ao1-r p-er ih-t w-ay-d','jj'],
'corporation':['k-ao-r p-er ey1 sh-ah-n','nn'],
'corporatism':['k-ao1-r p-er ah t-ih z-ah-m','nn'],
'corporatist':['k-ao1-r p-er ah t-ih-s-t','nn'],
'corpse':['k-ao1-r-p-s','nn'],
'corpses':['k-ao1-r-p s-ah-z','nns'],
'corpsman':['k-ao1-r m-ah-n','nn'],
'corpus':['k-ao1-r p-ah-s','nn'],
'corral':['k-er ae1-l','nn vb'],
'corralled':['k-er ae1-l-d','vbn'],
'corralling':['k-er ae1 l-ih-ng','vbg'],
'correct':['k-er eh1-k-t','jj vbp vb'],
'corrected':['k-er eh1-k t-ah-d','vbn vbd jj'],
'correcting':['k-er eh1-k t-ih-ng','vbg nn'],
'correction':['k-er eh1-k sh-ah-n','nn'],
'correctional':['k-er eh1-k sh-ah n-ah-l','jj'],
'corrective':['k-er eh1-k t-ih-v','jj'],
'correctly':['k-er eh1-k-t l-iy','rb'],
'correctness':['k-er eh1-k-t n-ah-s','nn'],
'corrects':['k-er eh1-k-t-s','vbz'],
'correlate':['k-ao1 r-ah l-ey-t','vb vbp'],
'correlated':['k-ao1 r-ah l-ey t-ah-d','jj vbd vbn'],
'correlation':['k-ao r-ah l-ey1 sh-ah-n','nn'],
'correspond':['k-ao r-ah s-p-aa1-n-d','vb vbp'],
'corresponded':['k-ao r-ah s-p-aa1-n d-ih-d','vbd'],
'correspondence':['k-ao r-ah s-p-aa1-n d-ah-n-s','nn'],
'correspondent':['k-ao r-ah s-p-aa1-n d-ah-n-t','nn'],
'corresponding':['k-ao r-ah s-p-aa1-n d-ih-ng','jj vbg'],
'correspondingly':['k-ao r-ah s-p-aa1-n d-ih-ng l-iy','rb'],
'corresponds':['k-ao r-ah s-p-aa1-n-d-z','vbz'],
'corridor':['k-ao1 r-ah d-er','nn'],
'corroborate':['k-er aa1 b-er ey-t','vb vbp'],
'corroborated':['k-er aa1 b-er ey t-ih-d','vbd vbn'],
'corrode':['k-er ow1-d','vbp vb'],
'corrosion':['k-er ow1 zh-ah-n','nn'],
'corrosive':['k-er ow1 s-ih-v','jj'],
'corrugated':['k-ao1 r-ah g-ey t-ah-d','jj vbn'],
'corrupt':['k-er ah1-p-t','jj vb'],
'corrupted':['k-er ah1-p t-ih-d','vbn'],
'corrupting':['k-er ah1-p t-ih-ng','vbg jj nn'],
'corruption':['k-er ah1-p sh-ah-n','nn'],
'corrupts':['k-er ah1-p-t-s','vbz'],
'corsage':['k-ao-r s-aa1-zh','nn'],
'corset':['k-ao1-r s-ah-t','nn'],
'cortex':['k-ao1-r t-eh-k-s','nn'],
'corticosteroids':['k-ao-r t-ih k-ow s-t-eh1 r-oy-d-z','nns'],
'cortisone':['k-ao1-r t-ah z-ow-n','nn'],
'corvettes':['k-ao-r v-eh1-t-s','nns'],
'cosmetic':['k-aa-z m-eh1 t-ih-k','jj nn'],
'cosmetology':['k-aa-z m-ah t-aa1 l-ah jh-iy','nn'],
'cosmic':['k-aa1-z m-ih-k','jj'],
'cosmopolitan':['k-aa-z m-ah p-aa1 l-ah t-ah-n','jj'],
'cosmos':['k-aa1-z m-ow-s','nn'],
'cost':['k-aa1-s-t','nn vbd vbn vbp vb'],
'costcutting':['k-ao1-s-t k-ah t-ih-ng','nn'],
'coste':['k-ow1-s-t','vb'],
'costing':['k-ao1 s-t-ih-ng','vbg nn'],
'costlier':['k-ao1-s-t l-iy er','jjr'],
'costliest':['k-ao1-s-t l-iy ih-s-t','jjs'],
'costly':['k-aa1-s-t l-iy','jj rb'],
'costs':['k-aa1-s-t-s','nns vbz'],
'costume':['k-aa s-t-uw1-m','nn jj'],
'costumed':['k-ao1 s-t-uw-m-d','vbn'],
'cot':['k-aa1-t','nn'],
'coterie':['k-ow1 t-er iy','nn'],
'cottage':['k-aa1 t-ah-jh','nn'],
'cotter':['k-aa1 t-er','nn'],
'cotton':['k-aa1 t-ah-n','nn'],
'cottonmouth':['k-aa1 t-ah-n m-aw-th','nn'],
'cottonseed':['k-aa1 t-ah-n s-iy-d','nn'],
'couch':['k-aw1-ch','nn'],
'couched':['k-aw1-ch-t','vbn vbd'],
'cough':['k-aa1-f','nn vb'],
'coughed':['k-ao1-f-t','vbd vbn'],
'coughing':['k-aa1 f-ih-ng','vbg nn'],
'coughs':['k-ao1-f-s','nns'],
'could':['k-uh1-d','md'],
'council':['k-aw1-n s-ah-l','nn'],
'councilman':['k-aw1-n s-ah-l m-ah-n','nn'],
'councilor':['k-aw1-n s-ah-l er','nn'],
'councilwoman':['k-aw1-n s-ah-l w-uh m-ah-n','nn'],
'counsel':['k-aw1-n s-ah-l','nn vb vbp'],
'counseled':['k-aw1-n s-ah-l-d','vbn vbd'],
'counseling':['k-aw1-n s-ah-l ih-ng','nn vbg'],
'counselor':['k-aw1-n s-ah-l er','nn'],
'counsels':['k-aw1-n s-ah-l-z','vbz'],
'count':['k-aw1-n-t','nn vb vbp'],
'countdown':['k-aw1-n-t d-aw-n','nn'],
'counted':['k-aw1-n t-ah-d','vbn vbd'],
'countenance':['k-aw1-n t-ah n-ah-n-s','nn vb'],
'countenances':['k-aw1-n t-ah n-ah-n s-ih-z','nns'],
'counter':['k-aw1-n t-er','nn in jj rb vb vbp'],
'counteract':['k-aw1-n t-er ae-k-t','vb'],
'counteracted':['k-aw1-n t-er ae-k t-ih-d','vbn vbd'],
'counteracting':['k-aw-n t-er ae1-k t-ih-ng','vbg'],
'counterattack':['k-aw1-n t-er ah t-ae-k','nn vb'],
'counterattacked':['k-aw-n t-er ah t-ae1-k-t','vbd'],
'counterbalance':['k-aw1-n t-er b-ae l-ah-n-s','vb nn'],
'counterbalanced':['k-aw-n t-er b-ae1 l-ah-n-s-t','vbn'],
'counterbid':['k-aw-n t-er b-ih1-d','nn'],
'counterchallenge':['k-aw1-n t-er ch-ae l-ah-n-jh','vb'],
'countercharges':['k-aw1-n t-er ch-aa-r jh-ih-z','nns'],
'counterclaim':['k-aw1-n t-er k-l-ey-m','nn'],
'countercultural':['k-aw-n t-er k-ah1-l ch-er ah-l','jj'],
'counterculture':['k-aw1-n t-er k-ah-l ch-er','jj'],
'countered':['k-aw1-n t-er-d','vbd vbn'],
'counterespionage':['k-aw-n t-er eh1 s-p-iy ah n-aa-jh','nn'],
'counterfeit':['k-aw1-n t-er f-ih-t','jj'],
'counterforce':['k-aw1-n t-er f-ao-r-s','nn'],
'countering':['k-aw1-n t-er ih-ng','vbg'],
'counterman':['k-aw1-n t-er m-ae-n','nn'],
'countermeasures':['k-aw1-n t-er m-eh zh-er-z','nns'],
'countermove':['k-aw1-n t-er m-uw-v','nn'],
'counteroffensive':['k-aw-n t-er ao f-eh1-n s-ih-v','nn'],
'counteroffer':['k-aw1-n t-er ao f-er','nn'],
'counterpart':['k-aw1-n t-er p-aa-r-t','nn jj'],
'counterpoint':['k-aw1-n t-er p-oy-n-t','nn'],
'counterproductive':['k-aw1-n t-er p-r-ah d-ah-k t-ih-v','jj'],
'counterproposal':['k-aw1-n t-er p-r-ah p-ow z-ah-l','nn'],
'counterrevolutionary':['k-aw-n t-er r-eh v-ah l-uw1 sh-ah-n eh r-iy','jj nn'],
'counters':['k-aw1-n t-er-z','nns vbz'],
'countersued':['k-aw1-n t-er s-uw-d','vbd vbn'],
'countersuit':['k-aw1-n t-er s-uw-t','nn'],
'countertenor':['k-aw1-n t-er t-eh n-er','nn'],
'counterterrorism':['k-aw1-n t-er t-eh r-er ih z-ah-m','nn'],
'countertop':['k-aw1-n t-er t-aa-p','nn'],
'countervailing':['k-aw1-n t-er v-ey l-ih-ng','jj vbg'],
'counterweight':['k-aw1-n t-er w-ey-t','nn'],
'countess':['k-aw1-n t-ah-s','nn'],
'counting':['k-aw1-n t-ih-ng','vbg nn'],
'countless':['k-aw1-n-t l-ah-s','jj'],
'country':['k-ah1-n t-r-iy','nn'],
'countryman':['k-ah1-n t-r-iy m-ah-n','nn'],
'countryside':['k-ah1-n t-r-iy s-ay-d','nn'],
'countrywide':['k-ah1-n t-r-iy w-ay-d','jj'],
'counts':['k-aw1-n-t-s','nns vbz'],
'county':['k-aw1-n t-iy','nn'],
'coup':['k-uw1','nn'],
'coupe':['k-uw1-p','nn'],
'couple':['k-ah1 p-ah-l','nn jj vb'],
'coupled':['k-ah1 p-ah-l-d','vbn vbd'],
'couplers':['k-ah1-p l-er-z','nns'],
'couples':['k-ah1 p-ah-l-z','nns'],
'coupling':['k-ah1 p-l-ih-ng','vbg nn'],
'coupon':['k-uw1 p-ao-n','nn'],
'couponing':['k-uw1 p-aa n-ih-ng','nn vbg'],
'courage':['k-er1 ah-jh','nn'],
'courageous':['k-er ey1 jh-ah-s','jj'],
'courageously':['k-er ey1 jh-ah-s l-iy','rb'],
'courier':['k-er1 iy er','nn'],
'course':['k-ao1-r-s','nn rb'],
'courses':['k-ao1-r s-ah-z','nns'],
'coursing':['k-ao1-r s-ih-ng','vbg'],
'court':['k-ao1-r-t','nn vbp vb'],
'courted':['k-ao1-r t-ih-d','vbn vbd'],
'courteous':['k-er1 t-iy ah-s','jj'],
'courtesy':['k-er1 t-ah s-iy','nn'],
'courthouse':['k-ao1-r-t hh-aw-s','nn'],
'courthouses':['k-ao1-r-t hh-aw s-ih-z','nns'],
'courtier':['k-ao1-r t-iy er','nn'],
'courting':['k-ao1-r t-ih-ng','vbg nn'],
'courtly':['k-ao1-r-t l-iy','jj'],
'courtroom':['k-ao1-r-t r-uw-m','nn'],
'courts':['k-ao1-r-t-s','nns vbz'],
'courtship':['k-ao1-r ch-ih-p','nn'],
'courtyard':['k-ao1-r-t y-aa-r-d','nn'],
'cousin':['k-ah1 z-ah-n','nn'],
'couture':['k-uw t-uh1-r','nn'],
'cove':['k-ow1-v','nn'],
'covenant':['k-ah1 v-ah n-ah-n-t','nn'],
'covenants':['k-ah1 v-ah n-ah-n-t-s','nns vbz'],
'cover':['k-ah1 v-er','vb nn vbp'],
'coverage':['k-ah1 v-er ah-jh','nn'],
'covered':['k-ah1 v-er-d','vbn jj vbd'],
'covering':['k-ah1 v-er ih-ng','vbg nn'],
'covers':['k-ah1 v-er-z','vbz nns'],
'covert':['k-ow1 v-er-t','jj'],
'covertly':['k-ah1 v-er-t l-iy','rb'],
'coverup':['k-ah1 v-er ah-p','nn'],
'covet':['k-ah1 v-ah-t','vb vbp'],
'coveted':['k-ah1 v-ah t-ih-d','vbn jj'],
'covets':['k-ah1 v-ah-t-s','vbz'],
'cow':['k-aw1','nn vb'],
'coward':['k-aw1 er-d','nn'],
'cowardice':['k-aw1 er d-ah-s','nn'],
'cowardly':['k-aw1 er-d l-iy','jj'],
'cowboy':['k-aw1 b-oy','nn'],
'cowed':['k-aw1-d','vbn'],
'cower':['k-aw1 er','vbp vb'],
'cowering':['k-aw1 er ih-ng','vbg'],
'cowling':['k-aw1 l-ih-ng','nn'],
'cowman':['k-aw1 m-ah-n','nn'],
'coworkers':['k-ow1 w-er1 k-er-z','nns'],
'cows':['k-aw1-z','nns'],
'coy':['k-oy1','jj'],
'coyly':['k-oy1 l-iy','rb'],
'coyote':['k-ay ow1 t-iy','nn'],
'coziness':['k-ow1 z-iy n-ah-s','nn'],
'cozy':['k-ow1 z-iy','jj rb'],
'crab':['k-r-ae1-b','nn'],
'crabbed':['k-r-ae1-b-d','jj'],
'crabby':['k-r-ae1 b-iy','jj'],
'crabmeat':['k-r-ae1-b m-iy-t','nn'],
'crack':['k-r-ae1-k','nn jj vb vbp'],
'crackdown':['k-r-ae1-k d-aw-n','nn'],
'cracked':['k-r-ae1-k-t','vbd vbn jj'],
'cracker':['k-r-ae1 k-er','nn'],
'cracking':['k-r-ae1 k-ih-ng','vbg jj nn'],
'crackle':['k-r-ae1 k-ah-l','nn vbp'],
'crackled':['k-r-ae1 k-ah-l-d','vbd'],
'crackles':['k-r-ae1 k-ah-l-z','vbz'],
'crackling':['k-r-ae1 k-l-ih-ng','nn'],
'crackpot':['k-r-ae1-k p-aa-t','nn'],
'cracks':['k-r-ae1-k-s','nns vbz'],
'cradle':['k-r-ey1 d-ah-l','nn vb'],
'cradles':['k-r-ey1 d-ah-l-z','nns'],
'craft':['k-r-ae1-f-t','nn vb'],
'crafted':['k-r-ae1-f t-ih-d','vbn vbd'],
'crafting':['k-r-ae1-f t-ih-ng','vbg'],
'crafts':['k-r-ae1-f-t-s','nns'],
'craftsman':['k-r-ae1-f-t-s m-ah-n','nn'],
'craftsmanship':['k-r-ae1-f-t-s m-ah-n sh-ih-p','nn'],
'crafty':['k-r-ae1-f t-iy','jj'],
'craggy':['k-r-ae1 g-iy','jj'],
'cram':['k-r-ae1-m','jj'],
'crammed':['k-r-ae1-m-d','vbn vbd'],
'cramming':['k-r-ae1 m-ih-ng','vbg'],
'cramp':['k-r-ae1-m-p','nn'],
'cramped':['k-r-ae1-m-p-t','jj'],
'crams':['k-r-ae1-m-z','vbz'],
'cranberry':['k-r-ae1-n b-eh r-iy','nn'],
'crane':['k-r-ey1-n','nn vb'],
'cranes':['k-r-ey1-n-z','nns'],
'crank':['k-r-ae1-ng-k','vb nn'],
'cranked':['k-r-ae1-ng-k-t','vbd vbn'],
'cranking':['k-r-ae1-ng k-ih-ng','vbg'],
'cranks':['k-r-ae1-ng-k-s','nns'],
'crankshaft':['k-r-ae1-ng-k sh-ae-f-t','nn'],
'cranky':['k-r-ae1-ng k-iy','jj'],
'cranny':['k-r-ae1 n-iy','nn'],
'crap':['k-r-ae1-p','nn uh'],
'crapshoot':['k-r-ae1-p sh-uw-t','nn vb'],
'crash':['k-r-ae1-sh','nn vbp jj vb'],
'crashed':['k-r-ae1-sh-t','vbd jj vbn'],
'crashes':['k-r-ae1 sh-ih-z','nns vbz'],
'crashing':['k-r-ae1 sh-ih-ng','vbg'],
'crass':['k-r-ae1-s','jj'],
'crate':['k-r-ey1-t','nn'],
'crater':['k-r-ey1 t-er','nn'],
'cratered':['k-r-ey1 t-er-d','vbn'],
'crave':['k-r-ey1-v','vbp vb'],
'craven':['k-r-ey1 v-ah-n','jj'],
'craves':['k-r-ey1-v-z','vbz'],
'craving':['k-r-ey1 v-ih-ng','nn vbg'],
'crawl':['k-r-ao1-l','vb nn'],
'crawled':['k-r-ao1-l-d','vbd vbn'],
'crawling':['k-r-ao1 l-ih-ng','vbg'],
'crawls':['k-r-ao1-l-z','nns vbz'],
'crayons':['k-r-ey1 aa-n-z','nns'],
'craze':['k-r-ey1-z','nn'],
'crazed':['k-r-ey1-z-d','jj vbd'],
'crazily':['k-r-ey1 z-ah l-iy','rb'],
'craziness':['k-r-ey1 z-iy n-ah-s','nn'],
'crazy':['k-r-ey1 z-iy','jj'],
'creaked':['k-r-iy1-k-t','vbd'],
'creaking':['k-r-iy1 k-ih-ng','vbg nn'],
'cream':['k-r-iy1-m','nn jj'],
'creamed':['k-r-iy1-m-d','vbn'],
'creamer':['k-r-iy1 m-er','nn'],
'creamier':['k-r-iy1 m-iy er','jjr'],
'creamy':['k-r-iy1 m-iy','jj'],
'crease':['k-r-iy1-s','nn'],
'create':['k-r-iy ey1-t','vb vbp'],
'created':['k-r-iy ey1 t-ah-d','vbn vbd jj'],
'creates':['k-r-iy ey1-t-s','vbz'],
'creating':['k-r-iy ey1 t-ih-ng','vbg'],
'creation':['k-r-iy ey1 sh-ah-n','nn'],
'creationism':['k-r-iy ey1 sh-ah-n ih z-ah-m','nn'],
'creative':['k-r-iy ey1 t-ih-v','jj'],
'creatively':['k-r-iy ey1 t-ih-v l-iy','rb'],
'creativeness':['k-r-iy ey1 t-ih-v n-ah-s','nn'],
'creativity':['k-r-iy ey t-ih1 v-ah t-iy','nn'],
'creator':['k-r-iy ey1 t-er','nn'],
'creature':['k-r-iy1 ch-er','nn'],
'credence':['k-r-iy1 d-ah-n-s','nn'],
'credential':['k-r-ih d-eh1-n ch-ah-l','nn'],
'credibility':['k-r-eh d-ah b-ih1 l-ih t-iy','nn'],
'credible':['k-r-eh1 d-ah b-ah-l','jj'],
'credibly':['k-r-eh1 d-ah b-l-iy','rb'],
'credit':['k-r-eh1 d-ah-t','nn vb vbp'],
'creditable':['k-r-eh1 d-ah t-ah b-ah-l','jj'],
'creditably':['k-r-eh1 d-ah t-ah b-l-iy','rb'],
'credited':['k-r-eh1 d-ah t-ah-d','vbn vbd'],
'crediting':['k-r-eh1 d-ah t-ih-ng','nn vbg'],
'creditor':['k-r-eh1 d-ah t-er','nn'],
'credits':['k-r-eh1 d-ih-t-s','nns vbz'],
'creditworthiness':['k-r-eh1 d-ih-t w-er dh-iy n-ah-s','nn'],
'creditworthy':['k-r-eh1 d-ih-t w-er dh-iy','nn'],
'credo':['k-r-ey1 d-ow','nn'],
'credulity':['k-r-ih d-uw1 l-ah t-iy','nn'],
'creed':['k-r-iy1-d','nn'],
'creek':['k-r-iy1-k','nn'],
'creep':['k-r-iy1-p','vb vbp jj nn'],
'creeping':['k-r-iy1 p-ih-ng','vbg jj'],
'creeps':['k-r-iy1-p-s','vbz'],
'creepy':['k-r-iy1 p-iy','jj'],
'cremate':['k-r-iy1 m-ey-t','vb'],
'cremation':['k-r-iy m-ey1 sh-ah-n','nn'],
'creole':['k-r-iy1 ow-l','nn'],
'crepe':['k-r-ey1-p','jj'],
'crept':['k-r-eh1-p-t','vbd vbn'],
'crescendo':['k-r-ih sh-eh1-n d-ow','nn'],
'crescent':['k-r-eh1 s-ah-n-t','nn'],
'crest':['k-r-eh1-s-t','nn vb'],
'crested':['k-r-eh1 s-t-ah-d','jj'],
'crestfallen':['k-r-eh1-s-t f-ao l-ah-n','jj'],
'crevice':['k-r-eh1 v-ah-s','nn'],
'crew':['k-r-uw1','nn'],
'crewel':['k-r-uw1 ah-l','nn'],
'crewmen':['k-r-uw1 m-ih-n','nns'],
'crib':['k-r-ih1-b','nn jj'],
'cribs':['k-r-ih1-b-z','nns vbz'],
'cricket':['k-r-ih1 k-ah-t','nn'],
'cried':['k-r-ay1-d','vbd vbn'],
'cries':['k-r-ay1-z','nns vbz'],
'crime':['k-r-ay1-m','nn'],
'criminal':['k-r-ih1 m-ah n-ah-l','jj nn'],
'criminality':['k-r-ih m-ah n-ae1 l-ih t-iy','nn'],
'criminalize':['k-r-ih1 m-ah n-ah l-ay-z','vb'],
'criminalizing':['k-r-ih1 m-ah n-ah l-ay z-ih-ng','vbg'],
'criminally':['k-r-ih1 m-ah n-ah l-iy','rb'],
'criminologist':['k-r-ih m-ih n-aa1 l-ah jh-ih-s-t','nn'],
'criminology':['k-r-ih m-ih n-aa1 l-ah jh-iy','nn'],
'crimp':['k-r-ih1-m-p','vb nn'],
'crimped':['k-r-ih1-m-p-t','jj vbn'],
'crimping':['k-r-ih1-m p-ih-ng','vbg'],
'crimps':['k-r-ih1-m-p-s','vbz'],
'crimson':['k-r-ih1-m z-ah-n','jj nn'],
'cringe':['k-r-ih1-n-jh','vbp vb'],
'cringed':['k-r-ih1-n-jh-d','vbd'],
'cripple':['k-r-ih1 p-ah-l','vb nn vbp'],
'crippled':['k-r-ih1 p-ah-l-d','vbn jj vbd'],
'cripples':['k-r-ih1 p-ah-l-z','nns'],
'crippling':['k-r-ih1 p-ah-l ih-ng','jj vbg'],
'crisis':['k-r-ay1 s-ah-s','nn'],
'crisp':['k-r-ih1-s-p','jj nn'],
'crisper':['k-r-ih1 s-p-er','nn'],
'crisply':['k-r-ih1-s-p l-iy','rb'],
'crispness':['k-r-ih1-s-p n-ah-s','nn'],
'crispy':['k-r-ih1 s-p-iy','jj'],
'crisscross':['k-r-ih1-s k-r-ao-s','vbp'],
'crisscrossed':['k-r-ih1-s k-r-ao-s-t','vbn vbd'],
'crisscrossing':['k-r-ih1-s k-r-ao s-ih-ng','vbg'],
'criterion':['k-r-ay t-ih1 r-iy ah-n','nn'],
'critic':['k-r-ih1 t-ih-k','nn'],
'critical':['k-r-ih1 t-ih k-ah-l','jj'],
'criticality':['k-r-ih t-ah k-ae1 l-ah t-iy','nn'],
'critically':['k-r-ih1 t-ih k-ah l-iy','rb'],
'criticism':['k-r-ih1 t-ih s-ih z-ah-m','nn'],
'criticize':['k-r-ih1 t-ih s-ay-z','vb vbp'],
'criticized':['k-r-ih1 t-ah s-ay-z-d','vbn vbd'],
'criticizes':['k-r-ih1 t-ih s-ay z-ih-z','vbz'],
'criticizing':['k-r-ih1 t-ih s-ay z-ih-ng','vbg'],
'critique':['k-r-ah t-iy1-k','nn'],
'critter':['k-r-ih1 t-er','nn'],
'croak':['k-r-ow1-k','nn vbp'],
'croaker':['k-r-ow1 k-er','nn'],
'croaking':['k-r-ow1 k-ih-ng','nn'],
'crochet':['k-r-ow sh-ey1','vb'],
'crocodile':['k-r-aa1 k-ah d-ay-l','nn'],
'croissants':['k-w-aa s-aa1-n-t-s','nns'],
'crone':['k-r-ow1-n','nn'],
'crony':['k-r-ow1 n-iy','nn'],
'cronyism':['k-r-ow1 n-iy ih z-ah-m','nn'],
'crook':['k-r-uh1-k','nn'],
'crooked':['k-r-uh1 k-ah-d','jj'],
'crooner':['k-r-uw1 n-er','nn'],
'crooning':['k-r-uw1 n-ih-ng','vbg'],
'croons':['k-r-uw1-n-z','vbz'],
'crop':['k-r-aa1-p','nn rp vb vbp'],
'cropped':['k-r-aa1-p-t','vbd vbn'],
'cropping':['k-r-aa1 p-ih-ng','vbg'],
'crops':['k-r-aa1-p-s','nns vbz'],
'cross':['k-r-ao1-s','vb jj nn rb vbp'],
'crossborder':['k-r-ao1-s b-ao-r d-er','jj'],
'crosscurrents':['k-r-ao1-s k-er ah-n-t-s','nns'],
'crossed':['k-r-ao1-s-t','vbd vbn'],
'crosses':['k-r-ao1 s-ih-z','vbz nns'],
'crossfire':['k-r-ao1-s f-ay-r','nn'],
'crossing':['k-r-ao1 s-ih-ng','vbg nn'],
'crossover':['k-r-ao1-s ow v-er','nn'],
'crossroads':['k-r-ao1-s r-ow-d-z','nns nn'],
'crosswise':['k-r-ao1-s w-ay-z','rb'],
'crotch':['k-r-aa1-ch','nn'],
'crotchety':['k-r-aa1 ch-ah t-iy','jj'],
'crouch':['k-r-aw1-ch','nn vb'],
'crouched':['k-r-aw1-ch-t','vbd vbn'],
'crow':['k-r-ow1','nn vb'],
'crowd':['k-r-aw1-d','nn vbp vb'],
'crowded':['k-r-aw1 d-ah-d','vbn vbd jj'],
'crowding':['k-r-aw1 d-ih-ng','vbg'],
'crowds':['k-r-aw1-d-z','nns vbp vbz'],
'crowed':['k-r-ow1-d','vbd vbn'],
'crowing':['k-r-ow1 ih-ng','vbg nn'],
'crown':['k-r-aw1-n','nn vb'],
'crowned':['k-r-aw1-n-d','vbn'],
'crowning':['k-r-aw1 n-ih-ng','jj vbg'],
'crowns':['k-r-aw1-n-z','nns'],
'crows':['k-r-ow1-z','nns vbz'],
'crucial':['k-r-uw1 sh-ah-l','jj'],
'crucially':['k-r-uw1 sh-ah l-l-iy','rb'],
'crucible':['k-r-uw1 s-ah b-ah-l','nn'],
'crucified':['k-r-uw1 s-ah f-ay-d','vbd vbn'],
'crucifix':['k-r-uw1 s-ah f-ih-k-s','nn'],
'crucifixion':['k-r-uw s-ih f-ih1-k sh-ah-n','nn'],
'crude':['k-r-uw1-d','nn jj'],
'crudely':['k-r-uw1-d l-iy','rb'],
'cruel':['k-r-uw1 ah-l','jj'],
'cruelest':['k-r-uw1 l-ah-s-t','jjs'],
'cruelly':['k-r-uw1 l-iy','rb'],
'cruelty':['k-r-uw1-l t-iy','nn'],
'cruise':['k-r-uw1-z','nn vbp'],
'cruiser':['k-r-uw1 z-er','nn'],
'cruising':['k-r-uw1 z-ih-ng','vbg nn'],
'crumble':['k-r-ah1-m b-ah-l','vb'],
'crumbled':['k-r-ah1-m b-ah-l-d','vbd vbn jj'],
'crumbles':['k-r-ah1-m b-ah-l-z','vbz'],
'crumbling':['k-r-ah1-m b-ah-l ih-ng','vbg jj'],
'crummy':['k-r-ah1 m-iy','jj'],
'crumpled':['k-r-ah1-m p-ah-l-d','jj vbd vbn'],
'crunch':['k-r-ah1-n-ch','nn vb'],
'crunched':['k-r-ah1-n-ch-t','vbd'],
'crunches':['k-r-ah1-n ch-ih-z','nns'],
'crunching':['k-r-ah1-n ch-ih-ng','vbg'],
'crusade':['k-r-uw s-ey1-d','nn'],
'crusader':['k-r-uw s-ey1 d-er','nn'],
'crusading':['k-r-uw s-ey1 d-ih-ng','vbg'],
'crush':['k-r-ah1-sh','nn vb'],
'crushed':['k-r-ah1-sh-t','vbn jj vbd'],
'crusher':['k-r-ah1 sh-er','nn'],
'crushes':['k-r-ah1 sh-ih-z','vbz'],
'crushing':['k-r-ah1 sh-ih-ng','vbg jj nn'],
'crust':['k-r-ah1-s-t','nn'],
'crusty':['k-r-ah1 s-t-iy','jj'],
'crutch':['k-r-ah1-ch','nn'],
'crux':['k-r-ah1-k-s','nn'],
'cruzado':['k-r-uw z-aa1 d-ow','nn'],
'cry':['k-r-ay1','nn vb vbp'],
'crying':['k-r-ay1 ih-ng','vbg nn'],
'crypt':['k-r-ih1-p-t','nn'],
'cryptic':['k-r-ih1-p t-ih-k','jj'],
'crystal':['k-r-ih1 s-t-ah-l','nn jj'],
'crystalline':['k-r-ih1 s-t-ah l-ay-n','jj'],
'crystallize':['k-r-ih1 s-t-ah l-ay-z','vb'],
'crystallized':['k-r-ih1 s-t-ah l-ay-z-d','vbd vbn'],
'crystallizing':['k-r-ih1 s-t-ah l-ay z-ih-ng','vbg'],
'crystallography':['k-r-ih s-t-ah l-aa1 g-r-ah f-iy','nn'],
'cub':['k-ah1-b','nn'],
'cubbyhole':['k-ah1 b-iy hh-ow-l','nn'],
'cube':['k-y-uw1-b','nn'],
'cubic':['k-y-uw1 b-ih-k','jj'],
'cubism':['k-y-uw1 b-ih z-ah-m','nn'],
'cubist':['k-y-uw1 b-ih-s-t','jj'],
'cuckoo':['k-ah1 k-uw','nn'],
'cucumber':['k-y-uw1 k-ah-m b-er','nn'],
'cuddled':['k-ah1 d-ah-l-d','vbd'],
'cuddly':['k-ah1-d l-iy','jj'],
'cudgels':['k-ah1 jh-ah-l-z','nns'],
'cue':['k-y-uw1','nn'],
'cued':['k-y-uw1-d','vbd'],
'cuff':['k-ah1-f','nn vb'],
'cuffs':['k-ah1-f-s','nns'],
'cuisine':['k-w-ih z-iy1-n','nn'],
'cul':['k-ah1-l','nn'],
'culinary':['k-y-uw1 l-ih n-eh r-iy','jj'],
'cull':['k-ah1-l','vb vbp'],
'culled':['k-ah1-l-d','vbn'],
'culling':['k-ah1 l-ih-ng','vbg'],
'culminate':['k-ah1-l m-ih n-ey-t','vb vbp'],
'culminated':['k-ah1-l m-ah n-ey t-ah-d','vbd vbn'],
'culminates':['k-ah1-l m-ih n-ey-t-s','vbz'],
'culminating':['k-ah1-l m-ah n-ey t-ih-ng','vbg'],
'culmination':['k-ah-l m-ah n-ey1 sh-ah-n','nn'],
'culpable':['k-ah1-l p-ah b-ah-l','jj'],
'culprit':['k-ah1-l p-r-ih-t','nn'],
'cult':['k-ah1-l-t','nn'],
'cultist':['k-ah1-l t-ah-s-t','nn'],
'cultivate':['k-ah1-l t-ah v-ey-t','vb'],
'cultivated':['k-ah1-l t-ah v-ey t-ah-d','vbn jj vbd'],
'cultivates':['k-ah1-l t-ih v-ey-t-s','vbz'],
'cultivating':['k-ah1-l t-ih v-ey t-ih-ng','vbg'],
'cultivation':['k-ah-l t-ih v-ey1 sh-ah-n','nn'],
'cultural':['k-ah1-l ch-er ah-l','jj'],
'culturally':['k-ah1-l ch-er ah l-iy','rb'],
'culture':['k-ah1-l ch-er','nn'],
'cultured':['k-ah1-l ch-er-d','jj vbn'],
'cumbersome':['k-ah1-m b-er s-ah-m','jj'],
'cumulative':['k-y-uw1 m-y-ah l-ah t-ih-v','jj'],
'cumulatively':['k-y-uw1 m-y-ah l-ah t-ih-v l-iy','rb'],
'cunning':['k-ah1 n-ih-ng','jj nn'],
'cup':['k-ah1-p','nn vb'],
'cupboard':['k-ah1 b-er-d','nn'],
'cupid':['k-y-uw1 p-ih-d','nn'],
'cups':['k-ah1-p-s','nns'],
'curative':['k-y-uh1 r-ah t-ih-v','jj'],
'curator':['k-y-uh r-ey1 t-er','nn'],
'curb':['k-er1-b','vb nn'],
'curbed':['k-er1-b-d','vbn vbd'],
'curbing':['k-er1 b-ih-ng','vbg'],
'curbs':['k-er1-b-z','nns'],
'curbside':['k-er1-b s-ay-d','nn jj'],
'curd':['k-er1-d','nn'],
'cure':['k-y-uh1-r','nn vbp vb'],
'cured':['k-y-uh1-r-d','vbn vbd'],
'cures':['k-y-uh1-r-z','nns vbz'],
'curfew':['k-er1 f-y-uw','nn'],
'curing':['k-y-uh1 r-ih-ng','vbg nn'],
'curiosity':['k-y-uh r-iy aa1 s-ah t-iy','nn'],
'curious':['k-y-uh1 r-iy ah-s','jj'],
'curiously':['k-y-uh1 r-iy ah-s l-iy','rb'],
'curl':['k-er1-l','vb vbp'],
'curled':['k-er1-l-d','vbd vbn'],
'curling':['k-er1 l-ih-ng','nn vbg'],
'curls':['k-er1-l-z','nns vbz'],
'curly':['k-er1 l-iy','jj'],
'currency':['k-er1 ah-n s-iy','nn'],
'current':['k-er1 ah-n-t','jj nn'],
'currently':['k-er1 ah-n-t l-iy','rb jj'],
'curricular':['k-er ih1 k-y-ah l-er','jj'],
'curriculum':['k-er ih1 k-y-ah l-ah-m','nn'],
'curry':['k-ah1 r-iy','vb nn'],
'curse':['k-er1-s','nn vb'],
'cursed':['k-er1-s-t','vbd vbn'],
'curses':['k-er1 s-ih-z','nns'],
'cursing':['k-er1 s-ih-ng','vbg nn'],
'cursory':['k-er1 s-er iy','jj'],
'curt':['k-er1-t','jj'],
'curtail':['k-er t-ey1-l','vb vbp'],
'curtailed':['k-er t-ey1-l-d','vbn vbd'],
'curtailing':['k-er t-ey1 l-ih-ng','vbg'],
'curtailment':['k-er t-ey1-l m-ah-n-t','nn'],
'curtails':['k-er t-ey1-l-z','vbz'],
'curtain':['k-er1 t-ah-n','nn'],
'curtly':['k-er1-t l-iy','rb'],
'curve':['k-er1-v','nn'],
'curved':['k-er1-v-d','jj vbn'],
'curving':['k-er1 v-ih-ng','vbg'],
'curvy':['k-er1 v-iy','jj'],
'cushion':['k-uh1 sh-ah-n','nn vb'],
'cushioned':['k-uh1 sh-ah-n-d','vbn vbd'],
'cushioning':['k-uh1 sh-ah-n ih-ng','nn vbg'],
'cushions':['k-uh1 sh-ah-n-z','nns'],
'cusp':['k-ah1-s-p','nn'],
'custodial':['k-ah s-t-ow1 d-iy ah-l','jj'],
'custodian':['k-ah s-t-ow1 d-iy ah-n','nn'],
'custody':['k-ah1 s-t-ah d-iy','nn'],
'custom':['k-ah1 s-t-ah-m','nn jj rb'],
'customarily':['k-ah s-t-ah m-eh1 r-ah l-iy','rb'],
'customary':['k-ah1 s-t-ah m-eh r-iy','jj'],
'customer':['k-ah1 s-t-ah m-er','nn'],
'customized':['k-ah1 s-t-ah m-ay-z-d','vbn jj'],
'cut':['k-ah1-t','vb vbd vbn vbp jj nn'],
'cutback':['k-ah1-t b-ae-k','nn'],
'cute':['k-y-uw1-t','jj'],
'cuteness':['k-y-uw1-t n-ah-s','nn'],
'cutlass':['k-ah1-t l-ah-s','nn'],
'cutlery':['k-ah1-t l-er iy','nn'],
'cutoff':['k-ah1 t-ao-f','nn jj'],
'cutouts':['k-ah1-t aw-t-s','nns'],
'cuts':['k-ah1-t-s','nns vbz nn'],
'cutter':['k-ah1 t-er','nn'],
'cutthroat':['k-ah1-t th-r-ow-t','jj'],
'cutting':['k-ah1 t-ih-ng','vbg nn jj'],
'cyanide':['s-ay1 ah n-ay-d','nn'],
'cycads':['s-ay1 k-ae-d-z','nns'],
'cycle':['s-ay1 k-ah-l','nn vb'],
'cycled':['s-ay1 k-ah-l-d','vbn'],
'cycles':['s-ay1 k-ah-l-z','nns vbz'],
'cyclical':['s-ay1 k-l-ih k-ah-l','jj'],
'cyclicality':['s-ih k-l-ih k-ae1 l-ih t-iy','nn'],
'cycling':['s-ay1 k-ah-l ih-ng','nn vbg'],
'cyclist':['s-ay1 k-ah l-ih-s-t','nn'],
'cyclosporine':['s-ih k-l-ao1 s-p-er iy-n','nn jj'],
'cylinder':['s-ih1 l-ah-n d-er','nn'],
'cylindrical':['s-ah l-ih1-n d-r-ih k-ah-l','jj'],
'cynic':['s-ih1 n-ih-k','nn jj'],
'cynical':['s-ih1 n-ih k-ah-l','jj'],
'cynically':['s-ih1 n-ih k-ah l-iy','rb'],
'cynicism':['s-ih1 n-ih s-ih z-ah-m','nn'],
'cypress':['s-ay1 p-r-ah-s','nn'],
'cyst':['s-ih1-s-t','nn'],
'cystic':['s-ih1 s-t-ih-k','jj'],
'cytoplasm':['s-ay1 t-ah p-l-ae z-ah-m','nn'],
'czar':['z-aa1-r','nn'],
'd':['d-iy1','ls nn'],
'dabble':['d-ae1 b-ah-l','vb'],
'dabbled':['d-ae1 b-ah-l-d','vbd'],
'dabbles':['d-ae1 b-ah-l-z','vbz'],
'dabbling':['d-ae1 b-ah-l ih-ng','vbg'],
'dad':['d-ae1-d','nn'],
'dada':['d-aa1 d-aa','nn'],
'daddy':['d-ae1 d-iy','nn'],
'daffodils':['d-ae1 f-ah d-ih-l-z','nns'],
'daft':['d-ae1-f-t','jj'],
'daily':['d-ey1 l-iy','jj nn rb'],
'dainty':['d-ey1-n t-iy','jj'],
'dairy':['d-eh1 r-iy','nn jj'],
'dairymen':['d-ey1 r-iy m-ah-n','nns'],
'dais':['d-ey1-z','nn'],
'daisy':['d-ey1 z-iy','nn'],
'dales':['d-ey1-l-z','nns'],
'dally':['d-ae1 l-iy','vb'],
'dam':['d-ae1-m','nn jj uh'],
'damage':['d-ae1 m-ah-jh','nn vbp vb'],
'damaged':['d-ae1 m-ah-jh-d','vbn jj vbd'],
'damages':['d-ae1 m-ah jh-ah-z','nns vbz'],
'damaging':['d-ae1 m-ih jh-ih-ng','jj vbg'],
'dame':['d-ey1-m','nn'],
'daminozide':['d-ae m-ih1 n-ah z-ay-d','nn'],
'dammed':['d-ae1-m-d','vbd'],
'damn':['d-ae1-m','jj nn rb vb uh'],
'damnation':['d-ae-m n-ey1 sh-ah-n','nn'],
'damned':['d-ae1-m-d','jj vbn rb'],
'damning':['d-ae1 m-ih-ng','vbg'],
'damp':['d-ae1-m-p','vb vbp jj nn'],
'damped':['d-ae1-m-p-t','vbn vbd vbp'],
'dampen':['d-ae1-m p-ah-n','vb'],
'dampened':['d-ae1-m p-ah-n-d','vbd vbn'],
'dampening':['d-ae1-m p-ah n-ih-ng','jj'],
'damper':['d-ae1-m p-er','nn'],
'damping':['d-ae1-m p-ih-ng','vbg jj'],
'damsel':['d-ae1-m z-ah-l','nn'],
'dance':['d-ae1-n-s','nn vb vbp'],
'danced':['d-ae1-n-s-t','vbd vbn'],
'dancer':['d-ae1-n s-er','nn'],
'dances':['d-ae1-n s-ah-z','nns vbz'],
'dancing':['d-ae1-n s-ih-ng','nn vbg'],
'dandelion':['d-ae1-n d-ah l-ay ah-n','nn'],
'dandy':['d-ae1-n d-iy','jj'],
'dang':['d-ae1-ng','jj'],
'danger':['d-ey1-n jh-er','nn'],
'dangerous':['d-ey1-n jh-er ah-s','jj'],
'dangerously':['d-ey1-n jh-er ah-s l-iy','rb'],
'dangle':['d-ae1-ng g-ah-l','vb'],
'dangled':['d-ae1-ng g-ah-l-d','vbd vbn'],
'dangles':['d-ae1-ng g-ah-l-z','vbz'],
'dangling':['d-ae1-ng g-ah l-ih-ng','vbg jj'],
'dank':['d-ae1-ng-k','jj'],
'dapper':['d-ae1 p-er','jj'],
'dare':['d-eh1-r','vb vbp md nn'],
'dared':['d-eh1-r-d','vbd vbn'],
'dares':['d-eh1-r-z','vbz'],
'darin':['d-ae1 r-ih-n','jj'],
'daring':['d-eh1 r-ih-ng','jj nn vbg'],
'dark':['d-aa1-r-k','jj nn rb'],
'darken':['d-aa1-r k-ah-n','vbp'],
'darkened':['d-aa1-r k-ah-n-d','vbd vbn jj'],
'darkening':['d-aa1-r k-ah n-ih-ng','vbg'],
'darker':['d-aa1-r k-er','jjr'],
'darkest':['d-aa1-r k-ah-s-t','jjs'],
'darkly':['d-aa1-r-k l-iy','rb'],
'darkness':['d-aa1-r-k n-ah-s','nn'],
'darkroom':['d-aa1-r-k r-uw-m','nn'],
'darlin':['d-aa1-r l-ih-n','nn'],
'darling':['d-aa1-r l-ih-ng','nn jj'],
'darn':['d-aa1-r-n','jj vb'],
'darned':['d-aa1-r-n-d','rb'],
'dart':['d-aa1-r-t','nn vbp'],
'dartboard':['d-aa1-r-t b-ao-r-d','nn'],
'darted':['d-aa1-r t-ih-d','vbd'],
'darting':['d-aa1-r t-ih-ng','vbg'],
'darts':['d-aa1-r-t-s','nn nns'],
'das':['d-ae1-s','nns'],
'dash':['d-ae1-sh','nn vb'],
'dashboard':['d-ae1-sh b-ao-r-d','nn'],
'dashed':['d-ae1-sh-t','vbn jj vbd'],
'dashes':['d-ae1 sh-ih-z','nns vbz'],
'dashing':['d-ae1 sh-ih-ng','vbg jj'],
'dastardly':['d-ae1 s-t-er-d l-iy','jj'],
'data':['d-ey1 t-ah','nns nn'],
'database':['d-ey1 t-ah b-ey-s','nn'],
'databases':['d-ey1 t-ah b-ey s-ih-z','nns'],
'date':['d-ey1-t','nn vbp vb'],
'dated':['d-ey1 t-ih-d','vbn jj vbd'],
'dates':['d-ey1-t-s','nns vbz'],
'dating':['d-ey1 t-ih-ng','vbg nn'],
'datum':['d-ae1 t-ah-m','nn'],
'daubed':['d-ao1-b-d','vbd'],
'daughter':['d-ao1 t-er','nn'],
'daunted':['d-ao1-n t-ih-d','vbd vbn jj'],
'daunting':['d-ao1-n t-ih-ng','jj vbg'],
'dauphin':['d-aw1 f-ih-n','nn'],
'davenport':['d-ae1 v-ah-n p-ao-r-t','nn'],
'dawdling':['d-ao1-d l-ih-ng','vbg'],
'dawn':['d-ao1-n','nn vb'],
'dawned':['d-ao1-n-d','vbd'],
'dawning':['d-ao1 n-ih-ng','vbg nn'],
'dawns':['d-ao1-n-z','vbz'],
'day':['d-ey1','nn'],
'daybreak':['d-ey1 b-r-ey-k','nn'],
'daydream':['d-ey1 d-r-iy-m','nn'],
'daydreamed':['d-ey1 d-r-iy-m-d','vbd'],
'daylight':['d-ey1 l-ay-t','nn'],
'daylong':['d-ey1 l-ao-ng','jj'],
'daytime':['d-ey1 t-ay-m','jj nn'],
'daze':['d-ey1-z','nn'],
'dazed':['d-ey1-z-d','jj vbd vbn'],
'dazzle':['d-ae1 z-ah-l','vb'],
'dazzled':['d-ae1 z-ah-l-d','vbn jj'],
'dazzling':['d-ae1 z-ah-l ih-ng','jj vbg'],
'deacon':['d-iy1 k-ah-n','nn'],
'deactivated':['d-iy ae1-k t-ih v-ey t-ih-d','vbn'],
'dead':['d-eh1-d','jj nn rb vbn'],
'deadbeats':['d-eh1-d b-iy-t-s','nns'],
'deader':['d-eh1 d-er','jjr'],
'deadliest':['d-eh1-d l-iy ah-s-t','jjs'],
'deadline':['d-eh1-d l-ay-n','nn'],
'deadliness':['d-eh1-d l-iy n-ah-s','nn'],
'deadlock':['d-eh1-d l-aa-k','nn'],
'deadlocked':['d-eh1-d l-aa-k-t','vbn jj'],
'deadly':['d-eh1-d l-iy','jj rb'],
'deadpan':['d-eh1-d p-ae-n','jj'],
'deadweight':['d-eh1-d w-ey-t','nn'],
'deadwood':['d-eh1-d w-uh-d','nn'],
'deaf':['d-eh1-f','jj'],
'deafening':['d-eh1 f-ah-n ih-ng','vbg jj'],
'deafness':['d-eh1-f n-ah-s','nn'],
'deal':['d-iy1-l','nn vb vbp'],
'dealer':['d-iy1 l-er','nn'],
'dealership':['d-iy1 l-er sh-ih-p','nn'],
'dealing':['d-iy1 l-ih-ng','vbg nn'],
'dealmaker':['d-iy1-l m-ey k-er','nn'],
'deals':['d-iy1-l-z','nns vbz'],
'dealt':['d-eh1-l-t','vbn vbd'],
'dean':['d-iy1-n','nn nnp'],
'dear':['d-ih1-r','jj nn rb uh'],
'dearest':['d-ih1 r-ah-s-t','jjs'],
'dearly':['d-ih1-r l-iy','rb'],
'dearth':['d-er1-th','nn'],
'death':['d-eh1-th','nn'],
'deathbed':['d-eh1-th b-eh-d','nn'],
'deathly':['d-eh1-th l-iy','jj'],
'deathwatch':['d-eh1-th w-aa-ch','nn'],
'debacle':['d-ah b-aa1 k-ah-l','nn'],
'debasement':['d-ah b-ey1-s m-ah-n-t','nn'],
'debasing':['d-ih b-ey1 s-ih-ng','vbg'],
'debatable':['d-ah b-ey1 t-ah b-ah-l','jj'],
'debate':['d-ah b-ey1-t','nn vb vbp'],
'debated':['d-ah b-ey1 t-ih-d','vbn vbd'],
'debates':['d-ah b-ey1-t-s','nns vbz'],
'debating':['d-ah b-ey1 t-ih-ng','vbg jj'],
'debauchery':['d-ah b-ao1 ch-er iy','nn'],
'debenture':['d-ah b-eh1-n ch-er','nn'],
'debilitated':['d-ah b-ih1 l-ah t-ey t-ih-d','vbn jj'],
'debilitating':['d-ah b-ih1 l-ah t-ey t-ih-ng','jj'],
'debility':['d-ah b-ih1 l-ah t-iy','nn'],
'debit':['d-eh1 b-ih-t','nn'],
'debonair':['d-eh b-ah n-eh1-r','jj'],
'debris':['d-ah b-r-iy1','nn'],
'debs':['d-eh1-b-z','nns'],
'debt':['d-eh1-t','nn'],
'debtholder':['d-eh1-t hh-ow-l d-er','nn'],
'debtor':['d-eh1 t-er','nn jj'],
'debunk':['d-ih b-ah1-ng-k','vb'],
'debunked':['d-ih b-ah1-ng-k-t','vbn'],
'debunking':['d-ih b-ah1-ng k-ih-ng','nn'],
'debut':['d-ey b-y-uw1','nn vb vbp'],
'debutante':['d-eh1 b-y-ah t-aa1-n-t','nn'],
'debuted':['d-ey b-y-uw1-d','vbd'],
'debuts':['d-ey1 b-y-uw-z','vbz'],
'decade':['d-eh k-ey1-d','nn'],
'decadence':['d-eh1 k-ah d-ah-n-s','nn'],
'decadent':['d-eh1 k-ah d-ah-n-t','jj'],
'decaffeinated':['d-iy k-ae1 f-ah n-ey t-ih-d','vbn'],
'decamped':['d-iy k-ae1-m-p-t','vbd'],
'decanting':['d-ah k-ae1-n t-ih-ng','vbg'],
'decay':['d-ah k-ey1','nn vb'],
'decayed':['d-ah k-ey1-d','jj vbd vbn'],
'decaying':['d-ah k-ey1 ih-ng','vbg jj'],
'decays':['d-ah k-ey1-z','vbz'],
'deceased':['d-ih s-iy1-s-t','jj nn vbn'],
'decedent':['d-eh s-iy1 d-ah-n-t','nn'],
'deceit':['d-ah s-iy1-t','nn'],
'deceitful':['d-ah s-iy1-t f-ah-l','jj'],
'deceive':['d-ih s-iy1-v','vb'],
'deceived':['d-ih s-iy1-v-d','vbn vbd'],
'deceiving':['d-ih s-iy1 v-ih-ng','vbg'],
'decelerate':['d-ih s-eh1 l-er ey-t','vb'],
'decelerated':['d-ih s-eh1 l-er ey t-ih-d','vbn'],
'decelerating':['d-ih s-eh1 l-er ey t-ih-ng','vbg'],
'deceleration':['d-ih s-eh l-er ey1 sh-ah-n','nn'],
'decency':['d-iy1 s-ah-n s-iy','nn'],
'decent':['d-iy1 s-ah-n-t','jj'],
'decently':['d-iy1 s-ah-n-t l-iy','rb'],
'decentralization':['d-ih s-eh-n t-r-ah l-ih z-ey1 sh-ah-n','nn'],
'decentralize':['d-ih s-eh1-n t-r-ah l-ay-z','vb'],
'decentralized':['d-ih s-eh1-n t-r-ah l-ay-z-d','jj vbd vbn'],
'decentralizing':['d-ih s-eh1-n t-r-ah l-ay z-ih-ng','vbg'],
'deception':['d-ih s-eh1-p sh-ah-n','nn'],
'deceptive':['d-ih s-eh1-p t-ih-v','jj'],
'deceptively':['d-ih s-eh1-p t-ih-v l-iy','rb'],
'decertified':['d-iy s-er1 t-ah f-ay-d','vbn'],
'decide':['d-ih s-ay1-d','vb vbp'],
'decided':['d-ih s-ay1 d-ih-d','vbd vbn jj'],
'decidedly':['d-ih s-ay1 d-ah-d l-iy','rb'],
'decides':['d-ih s-ay1-d-z','vbz'],
'deciding':['d-ah s-ay1 d-ih-ng','vbg jj'],
'decimal':['d-eh1 s-ah m-ah-l','nn jj'],
'decimated':['d-eh1 s-ah m-ey t-ih-d','vbn vbd'],
'decipher':['d-ih s-ay1 f-er','vb'],
'deciphered':['d-ih s-ay1 f-er-d','vbd'],
'decision':['d-ih s-ih1 zh-ah-n','nn'],
'decisionmaking':['d-ih s-ih1 zh-ah-n m-ey k-ih-ng','nn'],
'decisive':['d-ih s-ay1 s-ih-v','jj'],
'decisively':['d-ih s-ay1 s-ih-v l-iy','rb'],
'decisiveness':['d-ih s-ay1 s-ih-v n-ah-s','nn'],
'deck':['d-eh1-k','nn vb'],
'decked':['d-eh1-k-t','vbn'],
'decking':['d-eh1 k-ih-ng','nn vbg'],
'decks':['d-eh1-k-s','nns'],
'declaimed':['d-ih k-l-ey1-m-d','vbd'],
'declaration':['d-eh k-l-er ey1 sh-ah-n','nn'],
'declaratory':['d-ih k-l-eh1 r-ah t-ao r-iy','jj'],
'declare':['d-ih k-l-eh1-r','vb vbp'],
'declared':['d-ih k-l-eh1-r-d','vbd vbn jj'],
'declares':['d-ih k-l-eh1-r-z','vbz'],
'declaring':['d-ih k-l-eh1 r-ih-ng','vbg'],
'decline':['d-ih k-l-ay1-n','nn vb vbp'],
'declined':['d-ih k-l-ay1-n-d','vbd vbn'],
'decliners':['d-ih k-l-ay1 n-er-z','nns'],
'declines':['d-ih k-l-ay1-n-z','nns vbz'],
'declining':['d-ih k-l-ay1 n-ih-ng','vbg jj nn'],
'deco':['d-eh1 k-ow','nn'],
'decode':['d-ih k-ow1-d','vb'],
'decommissioned':['d-iy k-ah m-ih1 sh-ah-n-d','vbn'],
'decompose':['d-iy k-ah-m p-ow1-z','vb'],
'decomposed':['d-iy k-ah-m p-ow1-z-d','jj'],
'decomposes':['d-iy k-ah-m p-ow1 z-ih-z','vbz'],
'decompression':['d-iy k-ah-m p-r-eh1 sh-ah-n','nn'],
'decontamination':['d-iy k-ah-n t-ae m-ah n-ey1 sh-ah-n','nn'],
'decontrol':['d-iy k-ah-n t-r-ow1-l','nn vb'],
'decor':['d-ih k-ao1-r','nn'],
'decorate':['d-eh1 k-er ey-t','vbp vb'],
'decorated':['d-eh1 k-er ey t-ah-d','vbn vbd jj'],
'decorating':['d-eh1 k-er ey t-ih-ng','nn vbg'],
'decoration':['d-eh k-er ey1 sh-ah-n','nn'],
'decorative':['d-eh1 k-r-ah t-ih-v','jj'],
'decorator':['d-eh1 k-er ey t-er','nn'],
'decorous':['d-eh1 k-er ah-s','jj'],
'decorum':['d-ih k-ao1 r-ah-m','nn'],
'decoy':['d-ah k-oy1','nn'],
'decrease':['d-ih k-r-iy1-s','nn vb vbp'],
'decreased':['d-ih k-r-iy1-s-t','vbd vbn'],
'decreases':['d-ih k-r-iy1 s-ah-z','nns vbp vbz'],
'decreasing':['d-ih k-r-iy1 s-ih-ng','vbg'],
'decree':['d-ih k-r-iy1','nn'],
'decreed':['d-ih k-r-iy1-d','vbd vbn'],
'decrees':['d-ih k-r-iy1-z','nns vbz'],
'decrepit':['d-ah k-r-eh1 p-ih-t','jj'],
'decried':['d-ih k-r-ay1-d','vbd'],
'decries':['d-ih k-r-ay1-z','vbz'],
'decriminalization':['d-iy k-r-ih m-ah n-ah l-ah z-ey1 sh-ah-n','nn'],
'decry':['d-ih k-r-ay1','vb vbp'],
'decrying':['d-ih k-r-ay1 ih-ng','vbg'],
'dedicate':['d-eh1 d-ah k-ey-t','vb'],
'dedicated':['d-eh1 d-ah k-ey t-ah-d','vbn jj vbd'],
'dedicates':['d-eh1 d-ih k-ey-t-s','vbz'],
'dedication':['d-eh d-ah k-ey1 sh-ah-n','nn'],
'deduce':['d-ih d-uw1-s','vb'],
'deduced':['d-ih d-uw1-s-t','vbn vbd'],
'deduct':['d-ih d-ah1-k-t','vb vbp'],
'deducted':['d-ih d-ah1-k t-ih-d','vbn vbd'],
'deductibility':['d-ih d-ah-k t-ah b-ih1 l-ah t-iy','nn'],
'deductible':['d-ih d-ah1-k t-ah b-ah-l','jj'],
'deducting':['d-ih d-ah1-k t-ih-ng','vbg'],
'deduction':['d-ih d-ah1-k sh-ah-n','nn'],
'deed':['d-iy1-d','nn'],
'deeds':['d-iy1-d-z','nns vbz'],
'deem':['d-iy1-m','vbp vb'],
'deemed':['d-iy1-m-d','vbn vbd'],
'deems':['d-iy1-m-z','vbz'],
'deep':['d-iy1-p','jj rb'],
'deepen':['d-iy1 p-ah-n','vb'],
'deepened':['d-iy1 p-ah-n-d','vbd vbn'],
'deepening':['d-iy1 p-ah n-ih-ng','vbg'],
'deepens':['d-iy1 p-ah-n-z','vbz'],
'deeper':['d-iy1 p-er','jjr rbr rb'],
'deepest':['d-iy1 p-ah-s-t','jjs'],
'deeply':['d-iy1-p l-iy','rb'],
'deer':['d-ih1-r','nn nns'],
'defamation':['d-eh f-ah m-ey1 sh-ah-n','nn'],
'defamatory':['d-ih f-ae1 m-ah t-ao r-iy','jj'],
'defamed':['d-ih f-ey1-m-d','vbn vbd'],
'default':['d-ih f-ao1-l-t','nn vb'],
'defaulted':['d-ih f-ao1-l t-ih-d','vbd vbn jj'],
'defaulters':['d-ih f-ao1-l t-er-z','nns'],
'defaulting':['d-ih f-ao1-l t-ih-ng','vbg'],
'defaults':['d-ih f-ao1-l-t-s','nns vbz'],
'defeat':['d-ih f-iy1-t','nn vb vbp'],
'defeated':['d-ih f-iy1 t-ah-d','vbn vbd'],
'defeating':['d-ih f-iy1 t-ih-ng','vbg'],
'defeatism':['d-ih f-iy1 t-ih z-ah-m','nn'],
'defeats':['d-ih f-iy1-t-s','nns vbz'],
'defect':['d-iy1 f-eh-k-t','nn vb'],
'defected':['d-ih f-eh1-k t-ih-d','vbd vbn'],
'defecting':['d-ih f-eh1-k t-ih-ng','vbg'],
'defection':['d-ih f-eh1-k sh-ah-n','nn'],
'defective':['d-ih f-eh1-k t-ih-v','jj'],
'defector':['d-ih f-eh1-k t-er','nn'],
'defects':['d-iy1 f-eh-k-t-s','nns'],
'defend':['d-ih f-eh1-n-d','vb vbp'],
'defendant':['d-ih f-eh1-n d-ah-n-t','nn'],
'defended':['d-ih f-eh1-n d-ah-d','vbd vbn'],
'defender':['d-ih f-eh1-n d-er','nn'],
'defending':['d-ih f-eh1-n d-ih-ng','vbg jj'],
'defends':['d-ih f-eh1-n-d-z','vbz'],
'defense':['d-ih f-eh1-n-s','nn'],
'defenseless':['d-ih f-eh1-n-s l-ah-s','jj'],
'defenses':['d-ih f-eh1-n s-ah-z','nns'],
'defensible':['d-ih f-eh1-n s-ah b-ah-l','jj'],
'defensive':['d-ih f-eh1-n s-ih-v','jj nn'],
'defensively':['d-ih f-eh1-n s-ih-v l-iy','rb'],
'defensiveness':['d-ih f-eh1-n s-ih-v n-ah-s','nn'],
'defer':['d-ih f-er1','vb'],
'deference':['d-eh1 f-er ah-n-s','nn'],
'deferment':['d-ih f-er1 m-ah-n-t','nn'],
'deferred':['d-ih f-er1-d','vbn vbd jj'],
'deferring':['d-ih f-er1 ih-ng','vbg'],
'defers':['d-ih f-er1-z','vbz'],
'defiance':['d-ih f-ay1 ah-n-s','nn'],
'defiant':['d-ih f-ay1 ah-n-t','jj'],
'defiantly':['d-ih f-ay1 ah-n-t l-iy','rb'],
'deficiency':['d-ih f-ih1 sh-ah-n s-iy','nn'],
'deficient':['d-ih f-ih1 sh-ah-n-t','jj'],
'deficit':['d-eh1 f-ah s-ah-t','nn'],
'defied':['d-ih f-ay1-d','vbd vbn'],
'defies':['d-ih f-ay1-z','vbz'],
'definable':['d-ih f-ay1 n-ah b-ah-l','jj'],
'define':['d-ih f-ay1-n','vb vbp'],
'defined':['d-ih f-ay1-n-d','vbn jj vbd'],
'defines':['d-ih f-ay1-n-z','vbz'],
'defining':['d-ih f-ay1 n-ih-ng','vbg'],
'definite':['d-eh1 f-ah n-ah-t','jj'],
'definitely':['d-eh1 f-ah n-ah-t l-iy','rb'],
'definition':['d-eh f-ah n-ih1 sh-ah-n','nn'],
'definitive':['d-ih f-ih1 n-ih t-ih-v','jj'],
'definitively':['d-ih f-ih1 n-ih t-ih-v l-iy','rb'],
'deflate':['d-ih f-l-ey1-t','vb'],
'deflated':['d-ih f-l-ey1 t-ih-d','vbd jj vbn'],
'deflating':['d-ih f-l-ey1 t-ih-ng','vbg'],
'deflationary':['d-ih f-l-ey1 sh-ah-n eh r-iy','jj'],
'deflator':['d-ih f-l-ey1 t-er','nn'],
'deflect':['d-ih f-l-eh1-k-t','vb'],
'deflected':['d-ih f-l-eh1-k t-ih-d','vbd vbn'],
'deflecting':['d-ih f-l-eh1-k t-ih-ng','vbg'],
'deflects':['d-ih f-l-eh1-k-t-s','vbz'],
'deforestation':['d-ih f-ao r-ih s-t-ey1 sh-ah-n','nn'],
'deformation':['d-iy f-ao-r m-ey1 sh-ah-n','nn'],
'deformed':['d-ih f-ao1-r-m-d','jj vbn'],
'deformities':['d-ih f-ao1-r m-ah t-iy-z','nns'],
'defraud':['d-ih f-r-ao1-d','vb'],
'defrauded':['d-ih f-r-ao1 d-ih-d','vbd vbn'],
'defrauding':['d-ih f-r-ao1 d-ih-ng','vbg'],
'defray':['d-ih f-r-ey1','vb'],
'deft':['d-eh1-f-t','jj'],
'deftly':['d-eh1-f-t l-iy','rb'],
'defunct':['d-ih f-ah1-ng-k-t','jj vb'],
'defuse':['d-ih f-y-uw1-z','vb'],
'defused':['d-ih f-y-uw1-z-d','vbn'],
'defy':['d-ih f-ay1','vb vbp'],
'defying':['d-ih f-ay1 ih-ng','vbg'],
'degeneracy':['d-ih jh-eh1 n-er ah s-iy','nn'],
'degenerate':['d-ih jh-eh1 n-er ah-t','jj vb'],
'degenerated':['d-ih jh-eh1 n-er ey t-ih-d','vbd vbn'],
'degenerates':['d-ih jh-eh1 n-er ah-t-s','vbz'],
'degeneration':['d-ih jh-eh n-er ey1 sh-ah-n','nn'],
'degenerative':['d-ih jh-eh1 n-er ah t-ih-v','jj'],
'degradation':['d-eh g-r-ah d-ey1 sh-ah-n','nn'],
'degrade':['d-ih g-r-ey1-d','vb'],
'degraded':['d-ih g-r-ey1 d-ah-d','jj'],
'degrading':['d-ih g-r-ey1 d-ih-ng','jj vbg'],
'degree':['d-ih g-r-iy1','nn'],
'dehumanized':['d-ih hh-y-uw1 m-ah n-ay-z-d','vbn'],
'dehydrated':['d-ih hh-ay1 d-r-ey t-ah-d','jj vbn'],
'dehydration':['d-iy hh-ay d-r-ey1 sh-ah-n','nn'],
'deification':['d-iy ah f-ah k-ey1 sh-ah-n','nn'],
'deinstitutionalization':['d-iy ih-n s-t-ih t-uw sh-ah n-ah-l ah z-ey1 sh-ah-n','nn'],
'deities':['d-iy1 ah t-iy-z','nns'],
'delay':['d-ih l-ey1','nn vb vbp'],
'delayed':['d-ih l-ey1-d','vbn jj vbd'],
'delaying':['d-ih l-ey1 ih-ng','vbg jj nn'],
'delays':['d-ih l-ey1-z','nns vbz'],
'delectable':['d-ih l-eh1-k t-ah b-ah-l','jj'],
'delegate':['d-eh1 l-ah g-ey-t','nn vb'],
'delegated':['d-eh1 l-ah g-ey t-ah-d','vbn'],
'delegates':['d-eh1 l-ah g-ey-t-s','nns'],
'delegating':['d-eh1 l-ah g-ey t-ih-ng','vbg'],
'delegation':['d-eh l-ah g-ey1 sh-ah-n','nn'],
'delete':['d-ih l-iy1-t','vb'],
'deleted':['d-ih l-iy1 t-ah-d','vbn vbd'],
'deleterious':['d-eh l-ah t-ih1 r-iy ah-s','jj'],
'deleting':['d-ih l-iy1 t-ih-ng','vbg'],
'deletion':['d-ih l-iy1 sh-ah-n','nn'],
'deli':['d-eh1 l-iy','nns'],
'deliberate':['d-ih l-ih1 b-er ah-t','jj vb'],
'deliberated':['d-ih l-ih1 b-er ey t-ih-d','vbd'],
'deliberately':['d-ih l-ih1 b-er ah-t l-iy','rb'],
'deliberating':['d-ih l-ih1 b-er ey t-ih-ng','vbg'],
'deliberation':['d-ih l-ih b-er ey1 sh-ah-n','nn'],
'deliberative':['d-ih l-ih1 b-er ey t-ih-v','jj'],
'delicacy':['d-eh1 l-ah k-ah s-iy','nn'],
'delicate':['d-eh1 l-ah k-ah-t','jj'],
'delicately':['d-eh1 l-ah k-ah-t l-iy','rb'],
'delicatessen':['d-eh l-ih k-ah t-eh1 s-ah-n','nn'],
'delicious':['d-ih l-ih1 sh-ah-s','jj'],
'deliciously':['d-ih l-ih1 sh-ah-sh l-iy','rb'],
'delight':['d-ih l-ay1-t','nn vb vbp'],
'delighted':['d-ih l-ay1 t-ah-d','vbn jj vbd'],
'delightful':['d-ih l-ay1-t f-ah-l','jj'],
'delightfully':['d-ih l-ay1-t f-ah l-iy','rb'],
'delighting':['d-ih l-ay1 t-ih-ng','vbg'],
'delights':['d-ih l-ay1-t-s','vbz nns'],
'delineate':['d-ih l-ih1 n-iy ey-t','vb'],
'delineated':['d-ih l-ih1 n-iy ey t-ih-d','vbn'],
'delineating':['d-ih l-ih1 n-iy ey t-ih-ng','vbg'],
'delineation':['d-ih l-ih n-iy ey1 sh-ah-n','nn'],
'delinquency':['d-ih l-ih1-ng k-w-ah-n s-iy','nn'],
'delinquent':['d-ih l-ih1-ng k-w-ah-n-t','jj nn'],
'delirium':['d-ih l-ih1 r-iy ah-m','nn'],
'delist':['d-iy l-ih1-s-t','vb'],
'delisted':['d-iy l-ih1 s-t-ih-d','vbn vbd'],
'delisting':['d-iy l-ih1 s-t-ih-ng','nn vbg'],
'deliver':['d-ih l-ih1 v-er','vb vbp'],
'deliverable':['d-ih l-ih1 v-er ah b-ah-l','jj'],
'deliverance':['d-ih l-ih1 v-er ah-n-s','nn'],
'delivered':['d-ih l-ih1 v-er-d','vbn vbd'],
'deliverers':['d-ih l-ih1 v-er er-z','nns'],
'deliveries':['d-ih l-ih1 v-er iy-z','nns nn'],
'delivering':['d-ih l-ih1 v-er ih-ng','vbg'],
'delivers':['d-ih l-ih1 v-er-z','vbz'],
'delivery':['d-ih l-ih1 v-er iy','nn'],
'dell':['d-eh1-l','nn'],
'delle':['d-eh1-l','nnp'],
'delphic':['d-eh1-l f-ih-k','jj'],
'delta':['d-eh1-l t-ah','nn jj'],
'delude':['d-ih l-uw1-d','vb vbp'],
'deluded':['d-ih l-uw1 d-ih-d','jj vbd'],
'deluding':['d-ih l-uw1 d-ih-ng','vbg'],
'deluge':['d-eh1 l-y-uw-jh','nn'],
'deluged':['d-eh1 l-y-uw-jh-d','vbn vbd'],
'delusion':['d-ih l-uw1 zh-ah-n','nn'],
'deluxe':['d-ah l-ah1-k-s','jj'],
'delve':['d-eh1-l-v','vb'],
'delved':['d-eh1-l-v-d','vbn'],
'delves':['d-eh1-l-v-z','vbz'],
'delving':['d-eh1-l v-ih-ng','vbg nn'],
'demagogic':['d-eh m-ah g-aa1 jh-ih-k','jj'],
'demagoguery':['d-eh1 m-ah g-aa g-er iy','nn'],
'demagogues':['d-eh1 m-ah g-aa-g-z','nns'],
'demand':['d-ih m-ae1-n-d','nn vb vbp'],
'demanded':['d-ih m-ae1-n d-ah-d','vbd vbn'],
'demanding':['d-ih m-ae1-n d-ih-ng','vbg jj'],
'demandingly':['d-ih m-ae1-n d-ih-ng l-iy','rb'],
'demands':['d-ih m-ae1-n-d-z','nns vbz'],
'demarcation':['d-iy m-aa-r k-ey1 sh-ah-n','nn'],
'demeaning':['d-ih m-iy1 n-ih-ng','jj'],
'demeanor':['d-ih m-iy1 n-er','nn'],
'demeans':['d-ih m-iy1-n-z','vbz'],
'demented':['d-ih m-eh1-n t-ih-d','jj vbn'],
'dementia':['d-ih m-eh1-n sh-iy ah','nn'],
'demilitarize':['d-iy m-ih1 l-ah t-er ay-z','vb'],
'demise':['d-ih m-ay1-z','nn'],
'demo':['d-eh1 m-ow','nn'],
'demobilize':['d-ih m-ow1 b-ah l-ay-z','vb'],
'democracy':['d-ih m-aa1 k-r-ah s-iy','nn'],
'democrat':['d-eh1 m-ah k-r-ae-t','nn'],
'democratic':['d-eh m-ah k-r-ae1 t-ih-k','jj'],
'democratically':['d-eh m-ah k-r-ae1 t-ih-k l-iy','rb'],
'democratization':['d-ih m-aa k-r-ah t-ah z-ey1 sh-ah-n','nn'],
'democratize':['d-ih m-aa1 k-r-ah t-ay-z','vb'],
'demographic':['d-eh m-ah g-r-ae1 f-ih-k','jj'],
'demographically':['d-eh m-ah g-r-ae1 f-ih-k l-iy','rb'],
'demography':['d-ih m-aa1 g-r-ah f-iy','nn'],
'demolish':['d-ih m-aa1 l-ih-sh','vb'],
'demolished':['d-ih m-aa1 l-ih-sh-t','vbn vbd'],
'demolishing':['d-ih m-aa1 l-ih sh-ih-ng','vbg'],
'demolition':['d-eh m-ah l-ih1 sh-ah-n','nn jj'],
'demon':['d-iy1 m-ah-n','nn'],
'demonic':['d-ih m-aa1 n-ih-k','jj'],
'demonize':['d-iy1 m-ah n-ay-z','vb'],
'demonized':['d-iy1 m-ah n-ay-z-d','vbn'],
'demonizing':['d-iy1 m-ah n-ay z-ih-ng','nn'],
'demonstrable':['d-eh1 m-ah-n s-t-r-ah b-ah-l','jj'],
'demonstrably':['d-ih m-aa1-n s-t-r-ah b-l-iy','rb'],
'demonstrate':['d-eh1 m-ah-n s-t-r-ey-t','vb vbp'],
'demonstrated':['d-eh1 m-ah-n s-t-r-ey t-ah-d','vbn vbd jj'],
'demonstrates':['d-eh1 m-ah-n s-t-r-ey-t-s','vbz'],
'demonstrating':['d-eh1 m-ah-n s-t-r-ey t-ih-ng','vbg'],
'demonstration':['d-eh m-ah-n s-t-r-ey1 sh-ah-n','nn'],
'demonstrators':['d-eh1 m-ah-n s-t-r-ey t-er-z','nns'],
'demoralization':['d-ih m-ao r-ah l-ih z-ey1 sh-ah-n','nn'],
'demoralize':['d-ih m-ao1 r-ah l-ay-z','vb vbp'],
'demoralized':['d-ih m-ao1 r-ah l-ay-z-d','vbn jj'],
'demoralizing':['d-ih m-ao1 r-ah l-ay z-ih-ng','vbg'],
'demoted':['d-ih m-ow1 t-ih-d','vbn vbd'],
'demotion':['d-ih m-ow1 sh-ah-n','nn'],
'demur':['d-ih m-er1','vbp'],
'demure':['d-ih m-y-uh1-r','jj'],
'demurred':['d-ih m-er1-d','vbd'],
'demurs':['d-ih m-er1-z','vbz'],
'den':['d-eh1-n','nn nnp'],
'denationalization':['d-iy n-ae sh-ah-n ah l-ih z-ey1 sh-ah-n','nn'],
'denationalized':['d-ih n-ae1 sh-ah-n ah l-ay-z-d','vbn'],
'denial':['d-ih n-ay1 ah-l','nn'],
'denied':['d-ih n-ay1-d','vbn vbd'],
'denies':['d-ih n-ay1-z','vbz'],
'denigrate':['d-eh1 n-ah g-r-ey-t','vb'],
'denizens':['d-eh1 n-ah z-ah-n-z','nns'],
'denominated':['d-ih n-aa1 m-ah n-ey t-ih-d','vbn'],
'denomination':['d-ih n-ao m-ah n-ey1 sh-ah-n','nn'],
'denominator':['d-ih n-aa1 m-ah n-ey t-er','nn'],
'denote':['d-ih n-ow1-t','vb vbp'],
'denoted':['d-ih n-ow1 t-ah-d','vbn vbd'],
'denotes':['d-ih n-ow1-t-s','vbz'],
'denouement':['d-ey n-uw m-aa1-n','nn'],
'denounce':['d-ih n-aw1-n-s','vbp vb'],
'denounced':['d-ih n-aw1-n-s-t','vbd vbn'],
'denounces':['d-ih n-aw1-n s-ih-z','vbz'],
'denouncing':['d-ih n-aw1-n s-ih-ng','vbg'],
'dense':['d-eh1-n-s','jj'],
'densely':['d-eh1-n-s l-iy','rb'],
'densest':['d-eh1-n s-ah-s-t','jjs'],
'density':['d-eh1-n s-ah t-iy','nn'],
'dent':['d-eh1-n-t','nn vb'],
'dental':['d-eh1-n t-ah-l','jj nn'],
'dented':['d-eh1-n t-ih-d','vbd vbn'],
'denting':['d-eh1-n t-ih-ng','vbg'],
'dentist':['d-eh1-n t-ah-s-t','nn'],
'dentistry':['d-eh1-n t-ah-s t-r-iy','nn'],
'dents':['d-eh1-n-t-s','nns'],
'dentures':['d-eh1-n ch-er-z','nns'],
'denuclearized':['d-ih n-uw1 k-l-iy er ay-z-d','vbn'],
'denuded':['d-ih n-uw1 d-ih-d','vbn'],
'denunciation':['d-ih n-ah-n s-iy ey1 sh-ah-n','nn'],
'deny':['d-ih n-ay1','vb vbp'],
'denying':['d-ih n-ay1 ih-ng','vbg nn'],
'deodorant':['d-iy ow1 d-er ah-n-t','nn jj'],
'deoxyribonucleic':['d-iy aa-k s-iy r-ay b-ow n-uw k-l-ey1 ih-k','jj'],
'depart':['d-ih p-aa1-r-t','vb vbp'],
'departed':['d-ih p-aa1-r t-ah-d','vbd jj vbn'],
'departing':['d-ih p-aa1-r t-ih-ng','vbg'],
'department':['d-ih p-aa1-r-t m-ah-n-t','nn'],
'departmental':['d-ih p-aa-r-t m-eh1 n-ah-l','jj'],
'departs':['d-ih p-aa1-r-t-s','vbz'],
'departure':['d-ih p-aa1-r ch-er','nn'],
'depend':['d-ih p-eh1-n-d','vb vbp'],
'dependable':['d-ih p-eh1-n d-ah b-ah-l','jj'],
'depended':['d-ih p-eh1-n d-ah-d','vbd vbn'],
'dependence':['d-ih p-eh1-n d-ah-n-s','nn'],
'dependency':['d-ih p-eh1-n d-ah-n s-iy','nn'],
'dependent':['d-ih p-eh1-n d-ah-n-t','jj nn'],
'depending':['d-ih p-eh1-n d-ih-ng','vbg'],
'depends':['d-ih p-eh1-n-d-z','vbz'],
'depict':['d-ih p-ih1-k-t','vb vbp'],
'depicted':['d-ih p-ih1-k t-ah-d','vbn vbd'],
'depicting':['d-ih p-ih1-k t-ih-ng','vbg'],
'depiction':['d-ih p-ih1-k sh-ah-n','nn'],
'depicts':['d-ih p-ih1-k-t-s','vbz'],
'deplete':['d-ih p-l-iy1-t','vb'],
'depleted':['d-ih p-l-iy1 t-ih-d','vbn vbd'],
'depletes':['d-ih p-l-iy1-t-s','vbz'],
'depletion':['d-ih p-l-iy1 sh-ah-n','nn'],
'deplorable':['d-ih p-l-ao1 r-ah b-ah-l','jj'],
'deplore':['d-ih p-l-ao1-r','vb'],
'deplored':['d-ih p-l-ao1-r-d','vbd'],
'deplores':['d-ih p-l-ao1-r-z','vbz'],
'deploring':['d-ih p-l-ao1 r-ih-ng','vbg'],
'deploy':['d-ih p-l-oy1','vb vbp'],
'deployable':['d-ih p-l-oy1 ah b-ah-l','jj'],
'deployed':['d-ih p-l-oy1-d','vbn vbd'],
'deploying':['d-ih p-l-oy1 ih-ng','vbg'],
'deployment':['d-ih p-l-oy1 m-ah-n-t','nn'],
'deport':['d-ih p-ao1-r-t','vb'],
'deportation':['d-iy p-ao-r t-ey1 sh-ah-n','nn'],
'deported':['d-ih p-ao1-r t-ah-d','vbn vbd'],
'deportees':['d-iy p-ao-r t-iy1-z','nns'],
'depose':['d-ih p-ow1-z','vb'],
'deposed':['d-ih p-ow1-z-d','vbn vbd'],
'deposit':['d-ah p-aa1 z-ih-t','nn vb vbp'],
'depositary':['d-ah p-aa1 z-ih t-eh r-iy','nn jj'],
'deposited':['d-ah p-aa1 z-ih t-ih-d','vbn vbd'],
'depositing':['d-ah p-aa1 z-ih t-ih-ng','vbg'],
'deposition':['d-eh p-ah z-ih1 sh-ah-n','nn'],
'depositor':['d-ah p-aa1 z-ih t-er','nn'],
'depository':['d-ih p-aa1 z-ah t-ao r-iy','nn jj'],
'deposits':['d-ah p-aa1 z-ih-t-s','nns vbz'],
'depot':['d-iy1 p-ow','nn'],
'depravity':['d-ih p-r-ae1 v-ah t-iy','nn'],
'depreciable':['d-ih p-r-ih1 sh-ah b-ah-l','jj'],
'depreciated':['d-ih p-r-iy1 sh-iy ey t-ih-d','vbd'],
'depreciating':['d-ih p-r-iy1 sh-iy ey t-ih-ng','vbg'],
'depreciation':['d-ih p-r-iy sh-iy ey1 sh-ah-n','nn'],
'depredations':['d-eh p-r-ah d-ey1 sh-ah-n-z','nns'],
'depress':['d-ih p-r-eh1-s','vb vbp'],
'depressant':['d-ih p-r-eh1 s-ah-n-t','nn'],
'depressed':['d-ih p-r-eh1-s-t','jj vbd vbn'],
'depresses':['d-ih p-r-eh1 s-ah-z','vbz'],
'depressing':['d-ih p-r-eh1 s-ih-ng','jj vbg'],
'depressingly':['d-ih p-r-eh1 s-ih-ng l-iy','rb'],
'depression':['d-ih p-r-eh1 sh-ah-n','nn'],
'deprivation':['d-eh p-r-ah v-ey1 sh-ah-n','nn'],
'deprive':['d-ih p-r-ay1-v','vb'],
'deprived':['d-ih p-r-ay1-v-d','vbn jj vbd'],
'deprives':['d-ih p-r-ay1-v-z','vbz'],
'depriving':['d-ih p-r-ay1 v-ih-ng','vbg'],
'depth':['d-eh1-p-th','nn'],
'deputy':['d-eh1 p-y-ah t-iy','nn jj'],
'derail':['d-ih r-ey1-l','vb vbp'],
'derailed':['d-ih r-ey1-l-d','vbd vbn'],
'derailing':['d-ih r-ey1 l-ih-ng','vbg'],
'derailments':['d-ih r-ey1-l m-ah-n-t-s','nns'],
'deranged':['d-ih r-ey1-n-jh-d','jj vbn'],
'derby':['d-er1 b-iy','nn'],
'deregulate':['d-iy r-eh1 g-y-ah l-ey-t','vb nn'],
'deregulated':['d-iy r-eh1 g-y-ah l-ey t-ih-d','vbn jj vbd'],
'deregulating':['d-iy r-eh1 g-y-ah l-ey t-ih-ng','vbg'],
'deregulation':['d-iy r-eh g-y-ah l-ey1 sh-ah-n','nn'],
'derelict':['d-eh1 r-ah l-ih-k-t','nn jj'],
'dereliction':['d-eh r-ah l-ih1-k sh-ah-n','nn'],
'deride':['d-ih r-ay1-d','vbp'],
'derided':['d-ih r-ay1 d-ih-d','vbd vbn'],
'derision':['d-er ih1 zh-ah-n','nn'],
'derisively':['d-er ih1 s-ih-v l-iy','rb'],
'derivation':['d-eh r-ah v-ey1 sh-ah-n','nn'],
'derivative':['d-er ih1 v-ah t-ih-v','jj nn'],
'derive':['d-er ay1-v','vbp vb'],
'derived':['d-er ay1-v-d','vbn vbd'],
'derives':['d-er ay1-v-z','vbz'],
'deriving':['d-er ay1 v-ih-ng','vbg'],
'dermal':['d-er1 m-ah-l','jj'],
'dermatological':['d-er m-ah t-ah l-aa1 jh-ih k-ah-l','jj'],
'derogatory':['d-er aa1 g-ah t-ao r-iy','jj'],
'derrick':['d-eh1 r-ih-k','nn'],
'derring':['d-eh1 r-ih-ng','nn'],
'dervish':['d-er1 v-ih-sh','nn'],
'descend':['d-ih s-eh1-n-d','vb vbp'],
'descendant':['d-ih s-eh1-n d-ah-n-t','nn'],
'descended':['d-ih s-eh1-n d-ah-d','vbd vbn'],
'descendents':['d-ih s-eh1-n d-ah-n-t-s','nns'],
'descending':['d-ih s-eh1-n d-ih-ng','vbg'],
'descends':['d-ih s-eh1-n-d-z','vbz'],
'descent':['d-ih s-eh1-n-t','nn'],
'describe':['d-ih s-k-r-ay1-b','vb vbp'],
'described':['d-ih s-k-r-ay1-b-d','vbn vbd'],
'describes':['d-ih s-k-r-ay1-b-z','vbz'],
'describing':['d-ih s-k-r-ay1 b-ih-ng','vbg'],
'description':['d-ih s-k-r-ih1-p sh-ah-n','nn'],
'descriptive':['d-ih s-k-r-ih1-p t-ih-v','jj'],
'desecrated':['d-eh z-ah k-r-ey1 t-ih-d','vbn'],
'desecration':['d-eh s-ah k-r-ey1 sh-ah-n','nn'],
'desegregate':['d-ih s-eh1 g-r-ah g-ey-t','vb'],
'desegregated':['d-ih s-eh1 g-r-ih g-ey t-ih-d','vbn'],
'desegregation':['d-ih s-eh g-r-ah g-ey1 sh-ah-n','nn'],
'desert':['d-eh1 z-er-t','nn jj vb vbp'],
'deserted':['d-ih z-er1 t-ih-d','vbn vbd jj'],
'desertion':['d-ih z-er1 sh-ah-n','nn'],
'deserts':['d-eh1 z-er-t-s','nns vbz'],
'deserve':['d-ih z-er1-v','vbp vb'],
'deserved':['d-ih z-er1-v-d','vbd vbn'],
'deserves':['d-ih z-er1-v-z','vbz'],
'deserving':['d-ih z-er1 v-ih-ng','jj nn'],
'design':['d-ih z-ay1-n','nn vb vbp'],
'designate':['d-eh1 z-ah-g n-ey-t','vb jj nn'],
'designated':['d-eh1 z-ih-g n-ey t-ih-d','vbn vbd jj'],
'designates':['d-eh1 z-ih-g n-ey-t-s','vbz'],
'designating':['d-eh1 z-ih-g n-ey t-ih-ng','vbg'],
'designation':['d-eh z-ah-g n-ey1 sh-ah-n','nn'],
'designed':['d-ih z-ay1-n-d','vbn vbd'],
'designees':['d-eh z-ih-g n-iy1-z','nns'],
'designer':['d-ih z-ay1 n-er','nn'],
'designing':['d-ih z-ay1 n-ih-ng','vbg'],
'designs':['d-ih z-ay1-n-z','nns vbz'],
'desirability':['d-ih z-ay r-ah b-ih1 l-ih t-iy','nn'],
'desirable':['d-ih z-ay1 r-ah b-ah-l','jj'],
'desire':['d-ih z-ay1 er','nn vbp vb'],
'desired':['d-ih z-ay1 er-d','vbn vbd jj'],
'desires':['d-ih z-ay1 er-z','nns vbz'],
'desiring':['d-ih z-ay1 er ih-ng','vbg'],
'desist':['d-ih s-ih1-s-t','vb vbp'],
'desk':['d-eh1-s-k','nn'],
'desktop':['d-eh1-s-k t-aa-p','nn jj'],
'desolate':['d-eh1 s-ah l-ah-t','jj'],
'desolation':['d-eh s-ah l-ey1 sh-ah-n','nn'],
'despair':['d-ih s-p-eh1-r','nn vb'],
'despaired':['d-ih s-p-eh1-r-d','vbd'],
'despairing':['d-ih s-p-eh1 r-ih-ng','jj vbg'],
'despairs':['d-ih s-p-eh1-r-z','vbz'],
'desperadoes':['d-eh s-p-er aa1 d-ow-z','nns'],
'desperate':['d-eh1 s-p-r-ih-t','jj'],
'desperately':['d-eh1 s-p-er ah-t l-iy','rb'],
'desperation':['d-eh s-p-er ey1 sh-ah-n','nn'],
'despicable':['d-ih s-p-ih1 k-ah b-ah-l','jj'],
'despise':['d-ih s-p-ay1-z','vbp vb'],
'despised':['d-ih s-p-ay1-z-d','vbd vbn'],
'despises':['d-ih s-p-ay1 z-ih-z','vbz'],
'despite':['d-ih s-p-ay1-t','in'],
'despondency':['d-ih s-p-aa1-n d-ah-n s-iy','nn'],
'despondent':['d-ih s-p-aa1-n d-ah-n-t','jj'],
'despot':['d-eh1 s-p-ah-t','nn'],
'despotism':['d-eh1 s-p-ah t-ih z-ah-m','nn'],
'dessert':['d-ih z-er1-t','nn'],
'destabilize':['d-ih s-t-ey1 b-ah l-ay-z','vb'],
'destabilizing':['d-ih s-t-ey1 b-ah l-ay z-ih-ng','vbg jj'],
'destination':['d-eh s-t-ah n-ey1 sh-ah-n','nn'],
'destined':['d-eh1 s-t-ih-n-d','vbn'],
'destiny':['d-eh1 s-t-ah n-iy','nn'],
'destitute':['d-eh1 s-t-ah t-uw-t','jj'],
'destroy':['d-ih s-t-r-oy1','vb vbp'],
'destroyed':['d-ih s-t-r-oy1-d','vbn vbd jj'],
'destroyer':['d-ih s-t-r-oy1 er','nn'],
'destroying':['d-ih s-t-r-oy1 ih-ng','vbg'],
'destroys':['d-ih s-t-r-oy1-z','vbz'],
'destruction':['d-ih-s t-r-ah1-k sh-ah-n','nn'],
'destructive':['d-ih s-t-r-ah1-k t-ih-v','jj'],
'desultory':['d-eh1 s-ah-l t-ao r-iy','jj'],
'detach':['d-ih t-ae1-ch','vb'],
'detachable':['d-ih t-ae1 ch-ah b-ah-l','jj'],
'detached':['d-ih t-ae1-ch-t','vbn jj vbd'],
'detachment':['d-ih t-ae1-ch m-ah-n-t','nn'],
'detail':['d-ih t-ey1-l','nn vb'],
'detailed':['d-ih t-ey1-l-d','vbn jj vbd'],
'detailing':['d-ih t-ey1 l-ih-ng','vbg'],
'details':['d-ih t-ey1-l-z','nns vbz'],
'detain':['d-ih t-ey1-n','vb'],
'detained':['d-ih t-ey1-n-d','vbn vbd'],
'detaining':['d-ih t-ey1 n-ih-ng','vbg'],
'detect':['d-ih t-eh1-k-t','vb vbp'],
'detectable':['d-ih t-eh1-k t-ah b-ah-l','jj'],
'detected':['d-ih t-eh1-k t-ah-d','vbn vbd'],
'detecting':['d-ih t-eh1-k t-ih-ng','vbg nn'],
'detection':['d-ih t-eh1-k sh-ah-n','nn'],
'detective':['d-ih t-eh1-k t-ih-v','nn'],
'detector':['d-ih t-eh1-k t-er','nn'],
'detects':['d-ih t-eh1-k-t-s','vbz'],
'detente':['d-ey t-aa1-n-t','nn'],
'detention':['d-ih t-eh1-n sh-ah-n','nn'],
'deter':['d-ih t-er1','vb vbp'],
'detergent':['d-ih t-er1 jh-ah-n-t','nn'],
'deteriorate':['d-ih t-ih1 r-iy er ey-t','vb vbp'],
'deteriorated':['d-ih t-ih1 r-iy er ey t-ih-d','vbn jj vbd'],
'deteriorates':['d-ih t-ih1 r-iy er ey-t-s','vbz'],
'deteriorating':['d-ih t-ih1 r-iy er ey t-ih-ng','vbg jj'],
'deterioration':['d-ih t-ih1 r-iy er ey sh-ah-n','nn'],
'determinable':['d-ih t-er1 m-ah n-ah b-ah-l','jj'],
'determinant':['d-ih t-er1 m-ah n-ah-n-t','nn'],
'determinate':['d-ih t-er1 m-ah n-ey-t','jj'],
'determination':['d-ih t-er m-ah n-ey1 sh-ah-n','nn'],
'determine':['d-ah t-er1 m-ah-n','vb vbp'],
'determined':['d-ih t-er1 m-ah-n-d','vbn vbd jj'],
'determinedly':['d-ah t-er1 m-ah n-ah-d l-iy','rb'],
'determines':['d-ah t-er1 m-ah-n-z','vbz'],
'determining':['d-ih t-er1 m-ah n-ih-ng','vbg jj nn'],
'determinism':['d-ih t-er1 m-ah n-ih z-ah-m','nn'],
'deterministic':['d-ih t-er m-ah n-ih1 s-t-ih-k','jj'],
'deterred':['d-ih t-er1-d','vbn vbd'],
'deterrence':['d-ih t-er1 ah-n-s','nn'],
'deterrent':['d-ih t-er1 r-ah-n-t','nn'],
'deterring':['d-ih t-er1 ih-ng','vbg'],
'deters':['d-ih t-er1-z','vbz'],
'detest':['d-ih t-eh1-s-t','vbp vb'],
'detested':['d-ih t-eh1 s-t-ih-d','vbd vbn'],
'dethroned':['d-ih th-r-ow1-n-d','vbn'],
'detonate':['d-eh1 t-ah n-ey-t','vb vbp'],
'detonated':['d-eh1 t-ah n-ey t-ah-d','vbn vbd'],
'detonating':['d-eh1 t-ah n-ey t-ih-ng','vbg'],
'detonation':['d-eh t-ah n-ey1 sh-ah-n','nn'],
'detour':['d-ih t-uh1-r','nn'],
'detoxification':['d-ih t-aa-k s-ih f-ih k-ey1 sh-ah-n','nn'],
'detoxify':['d-ih t-aa1-k s-ah f-ay','vb'],
'detract':['d-ih t-r-ae1-k-t','vb vbp'],
'detracted':['d-ih t-r-ae1-k t-ih-d','vbn'],
'detracting':['d-ih t-r-ae1-k t-ih-ng','vbg'],
'detractors':['d-ih t-r-ae1-k t-er-z','nns'],
'detracts':['d-ih t-r-ae1-k-t-s','vbz'],
'detriment':['d-eh1 t-r-ah m-ah-n-t','nn'],
'detrimental':['d-eh t-r-ah m-eh1-n t-ah-l','jj'],
'deuterium':['d-uw t-iy1 r-iy ah-m','nn'],
'devaluation':['d-ih v-ae-l y-uw ey1 sh-ah-n','nn'],
'devalue':['d-ih v-ae1-l y-uw','vb'],
'devalued':['d-ih v-ae1-l y-uw-d','vbd vbn'],
'devastate':['d-eh1 v-ah s-t-ey-t','vb'],
'devastated':['d-eh1 v-ah s-t-ey t-ah-d','vbn vbd jj'],
'devastating':['d-eh1 v-ah s-t-ey t-ih-ng','jj vbg'],
'devastatingly':['d-eh1 v-ah s-t-ey t-ih-ng l-iy','rb'],
'devastation':['d-eh v-ah s-t-ey1 sh-ah-n','nn'],
'develop':['d-ih v-eh1 l-ah-p','vb vbp'],
'developed':['d-ih v-eh1 l-ah-p-t','vbn jj vbd'],
'developer':['d-ih v-eh1 l-ah p-er','nn'],
'developing':['d-ih v-eh1 l-ah p-ih-ng','vbg jj nn'],
'development':['d-ih v-eh1 l-ah-p m-ah-n-t','nn'],
'developmental':['d-ih v-eh l-ah-p m-eh1-n t-ah-l','jj'],
'develops':['d-ih v-eh1 l-ah-p-s','vbz'],
'deviance':['d-iy1 v-iy ah-n-s','nn'],
'deviant':['d-iy1 v-iy ah-n-t','jj'],
'deviate':['d-iy1 v-iy ey-t','vb'],
'deviated':['d-iy1 v-iy ey t-ih-d','vbd vbn'],
'deviates':['d-iy1 v-iy ey-t-s','vbz'],
'deviation':['d-iy v-iy ey1 sh-ah-n','nn'],
'device':['d-ih v-ay1-s','nn'],
'devil':['d-eh1 v-ah-l','nn'],
'devilish':['d-eh1-v l-ih-sh','jj'],
'devils':['d-eh1 v-ah-l-z','nns vbz'],
'devious':['d-iy1 v-iy ah-s','jj'],
'devise':['d-ih v-ay1-z','vb vbp'],
'devised':['d-ih v-ay1-z-d','vbn vbd'],
'devises':['d-ih v-ay1 z-ih-z','vbz'],
'devising':['d-ih v-ay1 z-ih-ng','vbg'],
'devoid':['d-ih v-oy1-d','jj'],
'devote':['d-ih v-ow1-t','vb vbp'],
'devoted':['d-ih v-ow1 t-ah-d','vbn vbd jj'],
'devotee':['d-eh v-ah t-iy1','nn'],
'devotes':['d-ih v-ow1-t-s','vbz'],
'devoting':['d-ih v-ow1 t-ih-ng','vbg'],
'devotion':['d-ih v-ow1 sh-ah-n','nn'],
'devotional':['d-ih v-ow1 sh-ah n-ah-l','jj'],
'devour':['d-ih v-aw1 er','vb vbp'],
'devoured':['d-ih v-aw1 er-d','vbn vbd'],
'devouring':['d-ih v-aw1 er ih-ng','vbg'],
'devours':['d-ih v-aw1 er-z','vbz'],
'devout':['d-ih v-aw1-t','jj'],
'devoutly':['d-ih v-aw1-t l-iy','rb'],
'dew':['d-uw1','nn'],
'dexterity':['d-eh-k s-t-eh1 r-ah t-iy','nn'],
'dextrous':['d-eh1-k s-t-r-ah-s','jj'],
'diabetes':['d-ay ah b-iy1 t-iy-z','nn'],
'diabetic':['d-ay ah b-eh1 t-ih-k','jj nn'],
'diabolical':['d-ay ah b-aa1 l-ih k-ah-l','jj'],
'diagnose':['d-ay ah-g n-ow1-s','vb vbp'],
'diagnosed':['d-ay ah-g n-ow1-s-t','vbn vbd'],
'diagnoses':['d-ay ah-g n-ow1 s-iy-z','nns vbz'],
'diagnosing':['d-ay ah-g n-ow1 s-ih-ng','vbg nn'],
'diagnosis':['d-ay ah-g n-ow1 s-ah-s','nn'],
'diagnostic':['d-ay ah-g n-aa1 s-t-ih-k','jj nn'],
'diagnostics':['d-ay ah-g n-aa1 s-t-ih-k-s','nns'],
'diagonal':['d-ay ae1 g-ah n-ah-l','jj'],
'diagonally':['d-ay ae1 g-ah n-ah l-iy','rb'],
'diagram':['d-ay1 ah g-r-ae-m','nn'],
'dial':['d-ay1 ah-l','nn vb'],
'dialect':['d-ay1 ah l-eh-k-t','nn'],
'dialectic':['d-ay ah l-eh1-k t-ih-k','nn'],
'dialectical':['d-ay ah l-eh1-k t-ih k-ah-l','jj'],
'dialed':['d-ay1 ah-l-d','vbd vbn'],
'dialing':['d-ay1 ah l-ih-ng','vbg nn'],
'dialogue':['d-ay1 ah l-ao-g','nn'],
'dials':['d-ay1 ah-l-z','nns vbz'],
'dialysis':['d-ay ae1 l-ah s-ah-s','nn'],
'diameter':['d-ay ae1 m-ah t-er','nn'],
'diametrically':['d-ay ah m-eh1 t-r-ih k-ah l-iy','rb'],
'diamond':['d-ay1 m-ah-n-d','nn'],
'diaper':['d-ay1 p-er','nn'],
'diaphragm':['d-ay1 ah f-r-ae-m','nn'],
'diarrhea':['d-ay er iy1 ah','nn'],
'diarrhoea':['d-ay er iy1 ah','nn'],
'diary':['d-ay1 er iy','nn'],
'diathermy':['d-ay1 ah th-er m-iy','nn'],
'diatomic':['d-ay ah t-aa1 m-ih-k','jj'],
'diatoms':['d-ay1 ah t-aa-m-z','nns'],
'diatribe':['d-ay1 ah t-r-ay-b','nn'],
'dibs':['d-ih1-b-z','nns'],
'dice':['d-ay1-s','nns nn'],
'dichotomy':['d-ay k-aa1 t-ah m-iy','nn'],
'dicker':['d-ih1 k-er','vb'],
'dickered':['d-ih1 k-er-d','vbd'],
'dickering':['d-ih1 k-er ih-ng','nn'],
'dicks':['d-ih1-k-s','nns'],
'dictate':['d-ih-k t-ey1-t','vb vbp'],
'dictated':['d-ih-k t-ey1 t-ah-d','vbn vbd'],
'dictates':['d-ih-k t-ey1-t-s','nns vbz'],
'dictating':['d-ih1-k t-ey t-ih-ng','vbg nn'],
'dictation':['d-ih-k t-ey1 sh-ah-n','nn'],
'dictator':['d-ih-k t-ey1 t-er','nn'],
'dictatorial':['d-ih-k t-ah t-ao1 r-iy ah-l','jj'],
'dictatorship':['d-ih-k t-ey1 t-er sh-ih-p','nn'],
'diction':['d-ih1-k sh-ah-n','nn'],
'dictionary':['d-ih1-k sh-ah n-eh r-iy','nn'],
'dictum':['d-ih1-k t-ah-m','nn'],
'did':['d-ih1-d','vbd'],
'didactic':['d-ay d-ae1-k t-ih-k','jj'],
'die':['d-ay1','vb vbp nn'],
'died':['d-ay1-d','vbd vbn'],
'diehard':['d-ay1 hh-aa-r-d','jj'],
'dies':['d-ay1-z','vbz nns'],
'diesel':['d-iy1 s-ah-l','nn jj'],
'diet':['d-ay1 ah-t','nn vb'],
'dietary':['d-ay1 ah t-eh r-iy','jj'],
'dieters':['d-ay1 ah t-er-z','nns'],
'dieting':['d-ay1 ah t-ih-ng','nn vbg'],
'diets':['d-ay1 ih-t-s','nns'],
'differ':['d-ih1 f-er','vbp vb'],
'differed':['d-ih1 f-er-d','vbd vbn'],
'difference':['d-ih1 f-er ah-n-s','nn'],
'different':['d-ih1 f-er ah-n-t','jj'],
'differential':['d-ih f-er eh1-n ch-ah-l','jj nn'],
'differentiate':['d-ih f-er eh1-n sh-iy ey-t','vb vbp'],
'differentiated':['d-ih f-er eh1-n ch-iy ey t-ih-d','vbn jj vbd'],
'differentiates':['d-ih f-er eh1-n sh-iy ey-t-s','vbz'],
'differentiating':['d-ih f-er eh1-n ch-iy ey t-ih-ng','vbg'],
'differentiation':['d-ih f-er eh-n sh-iy ey1 sh-ah-n','nn'],
'differently':['d-ih1 f-r-ah-n-t l-iy','rb'],
'differing':['d-ih1 f-er ih-ng','vbg jj'],
'differs':['d-ih1 f-er-z','vbz'],
'difficult':['d-ih1 f-ah k-ah-l-t','jj'],
'difficulty':['d-ih1 f-ah k-ah-l t-iy','nn'],
'diffraction':['d-ih f-r-ae1-k sh-ah-n','nn'],
'diffuse':['d-ih f-y-uw1-s','jj vb'],
'diffused':['d-ih f-y-uw1-z-d','vbn'],
'diffuses':['d-ih f-y-uw1 z-ah-z','vbz'],
'diffusion':['d-ih f-y-uw1 zh-ah-n','nn'],
'dig':['d-ih1-g','vb nn vbp'],
'digest':['d-ay jh-eh1-s-t','vb nn'],
'digested':['d-ay1 jh-eh s-t-ih-d','vbn'],
'digestible':['d-ay jh-eh1 s-t-ah b-ah-l','jj'],
'digesting':['d-ay jh-eh1 s-t-ih-ng','vbg'],
'digestion':['d-ay jh-eh1-s ch-ah-n','nn'],
'digestive':['d-ay jh-eh1 s-t-ih-v','jj'],
'digger':['d-ih1 g-er','nn'],
'digging':['d-ih1 g-ih-ng','vbg nn'],
'digit':['d-ih1 jh-ah-t','nn'],
'digital':['d-ih1 jh-ah t-ah-l','jj'],
'digitalis':['d-ih jh-ah t-ae1 l-ah-s','nn'],
'dignified':['d-ih1-g n-ah f-ay-d','vbn jj'],
'dignify':['d-ih1-g n-ah f-ay','vb'],
'dignitaries':['d-ih1-g n-ah t-eh r-iy-z','nns'],
'dignity':['d-ih1-g n-ah t-iy','nn'],
'digress':['d-ay g-r-eh1-s','vb'],
'digressions':['d-ay g-r-eh1 sh-ah-n-z','nns'],
'digs':['d-ih1-g-z','vbz nns'],
'dike':['d-ay1-k','nn'],
'dilapidated':['d-ah l-ae1 p-ah d-ey t-ih-d','jj vbn'],
'dilatation':['d-ih l-ah t-ey1 sh-ah-n','nn'],
'dilate':['d-ay l-ey1-t','vb'],
'dilated':['d-ay l-ey1 t-ah-d','vbn'],
'dilation':['d-ay l-ey1 sh-ah-n','nn'],
'dilemma':['d-ih l-eh1 m-ah','nn'],
'dilemmas':['d-ah l-eh1 m-ah-z','nn nns'],
'diligence':['d-ih1 l-ah jh-ah-n-s','nn'],
'diligent':['d-ih1 l-ih jh-ah-n-t','jj'],
'diligently':['d-ih1 l-ah jh-ah-n-t l-iy','rb'],
'dill':['d-ih1-l','nn'],
'dilute':['d-ay l-uw1-t','vb jj'],
'diluted':['d-ay l-uw1 t-ah-d','vbn jj vbd'],
'dilutes':['d-ay l-uw1-t-s','vbz'],
'diluting':['d-ay l-uw1 t-ih-ng','vbg'],
'dilution':['d-ay l-uw1 sh-ah-n','nn'],
'dilutive':['d-ah l-uw1 t-ih-v','jj'],
'dim':['d-ih1-m','jj'],
'dime':['d-ay1-m','nn'],
'dimension':['d-ih m-eh1-n sh-ah-n','nn vb'],
'dimensional':['d-ih m-eh1-n sh-ah n-ah-l','jj'],
'dimensions':['d-ih m-eh1-n sh-ah-n-z','nns'],
'diming':['d-ay1 m-ih-ng','vbg nn'],
'diminish':['d-ih m-ih1 n-ih-sh','vb vbp'],
'diminished':['d-ih m-ih1 n-ih-sh-t','vbn jj vbd'],
'diminishes':['d-ih m-ih1 n-ih sh-ih-z','vbz'],
'diminishing':['d-ih m-ih1 n-ih sh-ih-ng','vbg'],
'diminution':['d-ih m-ah n-uw1 sh-ah-n','nn'],
'diminutive':['d-ih m-ih1 n-y-ah t-ih-v','jj'],
'dimly':['d-ih1-m l-iy','rb'],
'dimmed':['d-ih1-m-d','vbn vbd'],
'dimmer':['d-ih1 m-er','rb jjr'],
'dimming':['d-ih1 m-ih-ng','vbg'],
'din':['d-ih1-n','nn'],
'dine':['d-ay1-n','vb'],
'dined':['d-ay1-n-d','vbd vbn'],
'diner':['d-ay1 n-er','nn'],
'dines':['d-ay1-n-z','vbz'],
'dinghy':['d-ih1-ng iy','nn'],
'dingo':['d-ih1-ng g-ow','nn'],
'dingy':['d-ih1-n jh-iy','jj'],
'dining':['d-ay1 n-ih-ng','nn jj vbg'],
'dinner':['d-ih1 n-er','nn'],
'dinnertime':['d-ih1 n-er t-ay-m','nn'],
'dinnerware':['d-ih1 n-er w-eh-r','nn'],
'dinosaur':['d-ay1 n-ah s-ao-r','nn'],
'diocesan':['d-ay aa1 s-ah s-ah-n','jj'],
'diocese':['d-ay1 ah s-iy-z','nn'],
'dioceses':['d-ay1 ah s-iy-z','nns'],
'dioxide':['d-ay aa1-k s-ay-d','nn'],
'dioxin':['d-ay aa1-k s-ih-n','nn'],
'dip':['d-ih1-p','nn vbp vb'],
'diphtheria':['d-ih-f th-ih1 r-iy ah','nn'],
'diploma':['d-ih p-l-ow1 m-ah','nn'],
'diplomacy':['d-ih p-l-ow1 m-ah s-iy','nn'],
'diplomat':['d-ih1 p-l-ah m-ae-t','nn'],
'diplomatic':['d-ih p-l-ah m-ae1 t-ih-k','jj'],
'diplomatically':['d-ih p-l-ah m-ae1 t-ih-k l-iy','rb'],
'dipole':['d-ay1 p-ow-l','jj nn'],
'dipped':['d-ih1-p-t','vbd vbn'],
'dipper':['d-ih1 p-er','nn'],
'dipping':['d-ih1 p-ih-ng','vbg'],
'dippy':['d-ih1 p-iy','jj'],
'dips':['d-ih1-p-s','nns'],
'dire':['d-ay1-r','jj'],
'direct':['d-er eh1-k-t','jj vbp rb vb'],
'directed':['d-er eh1-k t-ah-d','vbn vbd jj'],
'directing':['d-er eh1-k t-ih-ng','vbg nn'],
'direction':['d-er eh1-k sh-ah-n','nn'],
'directional':['d-er eh1-k sh-ah n-ah-l','jj'],
'directionless':['d-er eh1-k sh-ah-n l-ah-s','jj'],
'directive':['d-er eh1-k t-ih-v','nn'],
'directly':['d-er eh1-k-t l-iy','rb'],
'directness':['d-er eh1-k-t n-ah-s','nn'],
'director':['d-er eh1-k t-er','nn'],
'directorate':['d-er eh1-k t-er ah-t','nn'],
'directorial':['d-er eh-k t-ao1 r-iy ah-l','jj'],
'directorship':['d-er eh1-k t-er sh-ih-p','nn'],
'directory':['d-er eh1-k t-er iy','nn jj'],
'directs':['d-er eh1-k-t-s','vbz'],
'dirge':['d-er1-jh','nn'],
'dirhams':['d-er1 ah-m-z','nns'],
'dirt':['d-er1-t','nn jj'],
'dirtier':['d-er1 t-iy er','jjr'],
'dirtiest':['d-er1 t-iy ah-s-t','jjs'],
'dirty':['d-er1 t-iy','jj vb'],
'disability':['d-ih-s ah b-ih1 l-ih t-iy','nn'],
'disable':['d-ih-s ey1 b-ah-l','vb'],
'disabled':['d-ih-s ey1 b-ah-l-d','jj nn vbn'],
'disabling':['d-ih-s ey1 b-ah-l ih-ng','vbg'],
'disadvantage':['d-ih-s ah-d v-ae1-n t-ih-jh','nn'],
'disadvantaged':['d-ih-s ah-d v-ae1-n t-ih-jh-d','jj'],
'disadvantageous':['d-ih-s ae-d v-ae-n t-ey1 jh-ah-s','jj'],
'disaffected':['d-ih-s ah f-eh1-k t-ih-d','jj'],
'disaffection':['d-ih-s ah f-eh1-k sh-ah-n','nn'],
'disagree':['d-ih-s ah g-r-iy1','vbp vb'],
'disagreeable':['d-ih-s ah g-r-iy1 ah b-ah-l','jj'],
'disagreed':['d-ih-s ah g-r-iy1-d','vbd vbn'],
'disagreement':['d-ih-s ah g-r-iy1 m-ah-n-t','nn'],
'disagrees':['d-ih-s ah g-r-iy1-z','vbz'],
'disallow':['d-ih-s ah l-aw1','vb'],
'disallowance':['d-ih-s ah l-aw1 ah-n-s','nn'],
'disallowed':['d-ih-s ah l-aw1-d','vbd vbn'],
'disappear':['d-ih-s ah p-ih1-r','vb vbp'],
'disappearance':['d-ih-s ah p-ih1 r-ah-n-s','nn'],
'disappeared':['d-ih-s ah p-ih1-r-d','vbd vbn'],
'disappearing':['d-ih-s ah p-ih1 r-ih-ng','vbg'],
'disappears':['d-ih-s ah p-ih1-r-z','vbz'],
'disappoint':['d-ih-s ah p-oy1-n-t','vb'],
'disappointed':['d-ih-s ah p-oy1-n t-ih-d','vbn jj vbd'],
'disappointing':['d-ih-s ah p-oy1-n t-ih-ng','jj'],
'disappointingly':['d-ih-s ah p-oy1-n t-ih-ng l-iy','rb'],
'disappointment':['d-ih-s ah p-oy1-n-t m-ah-n-t','nn'],
'disappoints':['d-ih-s ah p-oy1-n-t-s','vbz'],
'disapproval':['d-ih-s ah p-r-uw1 v-ah-l','nn'],
'disapprove':['d-ih-s ah p-r-uw1-v','vbp vb'],
'disapproved':['d-ih-s ah p-r-uw1-v-d','vbd vbn'],
'disapproves':['d-ih-s ah p-r-uw1-v-z','vbz'],
'disapproving':['d-ih-s ah p-r-uw1 v-ih-ng','jj'],
'disarm':['d-ih-s aa1-r-m','vb'],
'disarmament':['d-ih-s aa1-r m-ah m-ah-n-t','nn'],
'disarmed':['d-ih-s aa1-r-m-d','jj vbn'],
'disarming':['d-ih-s aa1-r m-ih-ng','vbg jj'],
'disarmingly':['d-ih-s aa1-r m-ih-ng l-iy','rb'],
'disarray':['d-ih-s er ey1','nn'],
'disassembled':['d-ih-s ah s-eh1-m b-ah-l-d','vbd'],
'disassociate':['d-ih-s ah s-ow1 sh-iy ey-t','vb'],
'disaster':['d-ih z-ae1 s-t-er','nn'],
'disastrous':['d-ih z-ae1 s-t-r-ah-s','jj'],
'disastrously':['d-ih z-ae1 s-t-r-ah-s l-iy','rb'],
'disavowed':['d-ih-s ah v-aw1-d','vbd'],
'disband':['d-ih-s b-ae1-n-d','vb'],
'disbanded':['d-ih-s b-ae1-n d-ih-d','vbn vbd jj'],
'disbanding':['d-ih-s b-ae1-n d-ih-ng','vbg nn'],
'disbelief':['d-ih-s b-ih l-iy1-f','nn'],
'disbelieve':['d-ih-s b-ah l-iy1-v','vb'],
'disbursed':['d-ih-s b-er1-s-t','vbn vbd'],
'disbursement':['d-ih-s b-er1-s m-ah-n-t','nn'],
'disc':['d-ih1-s-k','nn'],
'discard':['d-ih-s k-aa1-r-d','vb vbp'],
'discarded':['d-ih-s k-aa1-r d-ih-d','vbn vbd'],
'discern':['d-ih s-er1-n','vb vbp'],
'discerned':['d-ih s-er1-n-d','vbn'],
'discernible':['d-ih s-er1 n-ah b-ah-l','jj'],
'discerning':['d-ih s-er1 n-ih-ng','jj'],
'discernment':['d-ih s-er1-n m-ah-n-t','nn'],
'discharge':['d-ih-s ch-aa1-r-jh','nn vb'],
'discharged':['d-ih-s ch-aa1-r-jh-d','vbn vbd'],
'discharges':['d-ih-s ch-aa1-r jh-ah-z','nns vbz'],
'discharging':['d-ih-s ch-aa1-r jh-ih-ng','vbg'],
'disciple':['d-ih s-ay1 p-ah-l','nn'],
'disciplinary':['d-ih1 s-ah p-l-ah n-eh r-iy','jj'],
'discipline':['d-ih1 s-ah p-l-ah-n','nn vb vbp'],
'disciplined':['d-ih1 s-ah p-l-ah-n-d','vbn vbd jj'],
'disciplines':['d-ih1 s-ah p-l-ah-n-z','nns'],
'disciplining':['d-ih1 s-ah p-l-ah n-ih-ng','vbg'],
'disclaimed':['d-ih-s k-l-ey1-m-d','vbd'],
'disclaimer':['d-ih-s k-l-ey1 m-er','nn'],
'disclaims':['d-ih-s k-l-ey1-m-z','vbz'],
'disclose':['d-ih-s k-l-ow1-z','vb vbp'],
'disclosed':['d-ih-s k-l-ow1-z-d','vbn jj vbd'],
'discloses':['d-ih-s k-l-ow1 z-ih-z','vbz'],
'disclosing':['d-ih-s k-l-ow1 z-ih-ng','vbg'],
'disclosure':['d-ih-s k-l-ow1 zh-er','nn'],
'disclosures':['d-ih-s k-l-ow1 zh-er-z','nns vbz'],
'disco':['d-ih1 s-k-ow','nn'],
'discolors':['d-ih s-k-ah1 l-er-z','vbz'],
'discomfort':['d-ih-s k-ah1-m f-er-t','nn'],
'disconcerting':['d-ih-s k-ah-n s-er1 t-ih-ng','jj vbg'],
'disconnect':['d-ih-s k-ah n-eh1-k-t','vb'],
'disconnected':['d-ih-s k-ah n-eh1-k t-ih-d','vbn jj'],
'disconnecting':['d-ih-s k-ah n-eh1-k t-ih-ng','vbg'],
'discontent':['d-ih-s k-ah-n t-eh1-n-t','nn'],
'discontented':['d-ih-s k-ah-n t-eh1-n t-ih-d','jj'],
'discontinuance':['d-ih-s k-ah-n t-ih1 n-y-uw ah-n-s','nn'],
'discontinuation':['d-ih-s k-ah-n t-ih n-y-uw ey1 sh-ah-n','nn'],
'discontinue':['d-ih-s k-ah-n t-ih1 n-y-uw','vb'],
'discontinued':['d-ih-s k-ah-n t-ih1 n-y-uw-d','vbn jj vbd'],
'discontinuing':['d-ih-s k-ah-n t-ih1 n-y-uw ih-ng','vbg'],
'discord':['d-ih1 s-k-ao-r-d','nn'],
'discordant':['d-ih s-k-ao1-r d-ah-n-t','jj'],
'discotheque':['d-ih1 s-k-ow t-eh-k','nn'],
'discount':['d-ih-s k-aw1-n-t','nn vbp jj vb'],
'discounted':['d-ih1-s k-aw-n t-ih-d','vbn jj vbd'],
'discounter':['d-ih1-s k-aw-n t-er','nn'],
'discounting':['d-ih1-s k-aw-n t-ih-ng','nn vbg'],
'discounts':['d-ih-s k-aw1-n-t-s','nns vbz'],
'discourage':['d-ih s-k-er1 ih-jh','vb vbp'],
'discouraged':['d-ih s-k-er1 ah-jh-d','vbn vbd jj'],
'discouragement':['d-ih s-k-er1 ih-jh m-ah-n-t','nn'],
'discourages':['d-ih s-k-er1 ih jh-ih-z','vbz'],
'discouraging':['d-ih s-k-er1 ah jh-ih-ng','vbg jj'],
'discourse':['d-ih1-s k-ao-r-s','nn vb'],
'discourses':['d-ih-s k-ao1-r s-ih-z','nns'],
'discover':['d-ih s-k-ah1 v-er','vb vbp'],
'discovered':['d-ih s-k-ah1 v-er-d','vbn jj vbd'],
'discovering':['d-ih s-k-ah1 v-er ih-ng','vbg'],
'discovers':['d-ih s-k-ah1 v-er-z','vbz'],
'discovery':['d-ih s-k-ah1 v-er iy','nn'],
'discredit':['d-ih-s k-r-eh1 d-ah-t','vb nn'],
'discredited':['d-ih-s k-r-eh1 d-ih t-ih-d','vbn jj'],
'discrediting':['d-ih-s k-r-eh1 d-ah t-ih-ng','nn vbg'],
'discreet':['d-ih s-k-r-iy1-t','jj'],
'discreetly':['d-ih s-k-r-iy1-t l-iy','rb'],
'discrepancy':['d-ih s-k-r-eh1 p-ah-n s-iy','nn'],
'discrete':['d-ih s-k-r-iy1-t','jj'],
'discretion':['d-ih s-k-r-eh1 sh-ah-n','nn'],
'discretionary':['d-ih s-k-r-eh1 sh-ah-n eh r-iy','jj'],
'discriminate':['d-ih s-k-r-ih1 m-ah n-ey-t','vb vbp jj'],
'discriminated':['d-ih s-k-r-ih1 m-ah n-ey t-ah-d','vbd jj nn vb'],
'discriminates':['d-ih s-k-r-ih1 m-ah n-ey-t-s','nns vbz'],
'discriminating':['d-ih s-k-r-ih1 m-ah n-ey t-ih-ng','vbg jj'],
'discrimination':['d-ih s-k-r-ih m-ah n-ey1 sh-ah-n','nn'],
'discriminatory':['d-ih s-k-r-ih1 m-ah n-ah t-ao r-iy','jj'],
'discursive':['d-ih s-k-er1 s-ih-v','jj'],
'discus':['d-ih1 s-k-ah-s','nn'],
'discuss':['d-ih s-k-ah1-s','vb vbp'],
'discussed':['d-ih s-k-ah1-s-t','vbn vbd'],
'discusses':['d-ih s-k-ah1 s-ah-z','vbz'],
'discussing':['d-ih s-k-ah1 s-ih-ng','vbg'],
'discussion':['d-ih s-k-ah1 sh-ah-n','nn'],
'disdain':['d-ih-s d-ey1-n','nn vbp vb'],
'disdained':['d-ih-s d-ey1-n-d','vbn'],
'disdainful':['d-ih-s d-ey1-n f-ah-l','jj'],
'disdaining':['d-ih-s d-ey1 n-ih-ng','vbg'],
'disdains':['d-ih-s d-ey1-n-z','vbz'],
'disease':['d-ih z-iy1-z','nn'],
'diseased':['d-ih z-iy1-z-d','jj'],
'diseases':['d-ih z-iy1 z-ah-z','nns'],
'disembodied':['d-ih-s ih b-aa1 d-iy-d','jj'],
'disenchanted':['d-ih-s ih-n ch-ae1-n t-ih-d','vbn jj'],
'disenfranchised':['d-ih-s ih-n f-r-ae1-n ch-ay-z-d','vbn'],
'disenfranchisement':['d-ih-s ih-n f-r-ae1-n ch-ay-z m-ah-n-t','nn'],
'disengage':['d-ih-s ih-n g-ey1-jh','vb'],
'disengaged':['d-ih-s ih-n g-ey1-jh-d','vbn'],
'disengagement':['d-ih-s ih-n g-ey1-jh m-ah-n-t','nn'],
'disentangle':['d-ih-s ah-n t-ae1-ng g-ah-l','vb'],
'disfavor':['d-ih-s f-ey1 v-er','nn'],
'disfigured':['d-ih-s f-ih1 g-y-er-d','vbd vbn jj'],
'disgorge':['d-ih-s g-ao1-r-jh','vb'],
'disgorgement':['d-ih-s g-ao1-r-jh m-ah-n-t','nn'],
'disgrace':['d-ih-s g-r-ey1-s','nn'],
'disgraced':['d-ih-s g-r-ey1-s-t','vbn'],
'disgraceful':['d-ih-s g-r-ey1-s f-ah-l','jj'],
'disgruntled':['d-ih-s g-r-ah1-n t-ah-l-d','jj vbn'],
'disguise':['d-ih-s g-ay1-z','vb vbp nn'],
'disguised':['d-ih-s g-ay1-z-d','vbn vbd jj'],
'disguises':['d-ih-s g-ay1 z-ih-z','vbz nns'],
'disgust':['d-ih-s g-ah1-s-t','nn'],
'disgusted':['d-ih-s g-ah1 s-t-ah-d','vbn jj'],
'disgusting':['d-ih-s g-ah1 s-t-ih-ng','jj'],
'dish':['d-ih1-sh','nn vb'],
'disharmony':['d-ih-s hh-aa1-r m-ah n-iy','nn'],
'disheartened':['d-ih-s hh-aa1-r t-ah-n-d','vbn'],
'disheartening':['d-ih-s hh-aa1-r t-ah-n ih-ng','jj'],
'dished':['d-ih1-sh-t','vbd'],
'dishes':['d-ih1 sh-ah-z','nns vbz'],
'disheveled':['d-ih sh-eh1 v-ah-l-d','jj'],
'dishing':['d-ih1 sh-ih-ng','vbg'],
'dishonest':['d-ih-s aa1 n-ah-s-t','jj'],
'dishonestly':['d-ih-s ao1 n-ah-s-t l-iy','rb'],
'dishonesty':['d-ih-s aa1 n-ah s-t-iy','nn'],
'dishonor':['d-ih-s aa1 n-er','nn vb'],
'dishonorable':['d-ih-s aa1 n-er ah b-ah-l','jj'],
'dishwasher':['d-ih1-sh w-aa sh-er','nn'],
'disillusioned':['d-ih-s ih l-uw1 zh-ah-n-d','vbn jj'],
'disillusioning':['d-ih-s ih l-uw1 zh-ah-n ih-ng','jj'],
'disillusionment':['d-ih-s ih l-uw1 zh-ah-n m-ah-n-t','nn'],
'disincentive':['d-ih-s ih-n s-eh1-n t-ih-v','nn'],
'disinclination':['d-ih-s ih-n k-l-ah n-ey1 sh-ah-n','nn'],
'disinclined':['d-ih-s ih-n k-l-ay1-n-d','vbn jj'],
'disinfectant':['d-ih-s ih-n f-eh1-k t-ah-n-t','nn'],
'disinflation':['d-ih-s ih-n f-l-ey1 sh-ah-n','nn'],
'disinflationary':['d-ih-s ih-n f-l-ey1 sh-ah-n eh r-iy','jj'],
'disingenuous':['d-ih-s ih-n jh-eh1 n-y-uw ah-s','jj'],
'disintegrate':['d-ih-s ih1-n t-ah g-r-ey-t','vb'],
'disintegrated':['d-ih-s ih1-n t-ah g-r-ey t-ih-d','vbd'],
'disintegrating':['d-ih-s ih1-n t-ah g-r-ey t-ih-ng','vbg'],
'disintegration':['d-ih-s ih-n t-ah g-r-ey1 sh-ah-n','nn'],
'disinterest':['d-ih-s ih1-n t-er ah-s-t','nn'],
'disinterested':['d-ih-s ih1-n t-r-ih s-t-ih-d','jj'],
'disjointed':['d-ih-s jh-oy1-n t-ih-d','vbn jj'],
'disk':['d-ih1-s-k','nn'],
'diskette':['d-ih s-k-eh1-t','nn'],
'dislike':['d-ih-s l-ay1-k','nn vb vbp'],
'disliked':['d-ih-s l-ay1-k-t','vbd jj vbn'],
'dislikes':['d-ih-s l-ay1-k-s','vbz nn nns'],
'dislocated':['d-ih1-s l-ow k-ey t-ih-d','jj'],
'dislocation':['d-ih-s l-ow1 k-ey1 sh-ah-n','nn'],
'dislodge':['d-ih-s l-aa1-jh','vb'],
'dislodged':['d-ih-s l-aa1-jh-d','vbd vbn'],
'disloyal':['d-ih-s l-oy1 ah-l','jj'],
'disloyalty':['d-ih-s l-oy1 ah-l t-iy','nn'],
'dismal':['d-ih1-z m-ah-l','jj'],
'dismally':['d-ih1-z m-ah l-iy','rb'],
'dismantle':['d-ih-s m-ae1-n t-ah-l','vb'],
'dismantled':['d-ih-s m-ae1-n t-ah-l-d','vbn vbd'],
'dismantles':['d-ih-s m-ae1-n t-ah-l-z','vbz'],
'dismantling':['d-ih-s m-ae1-n t-ah-l ih-ng','vbg nn'],
'dismay':['d-ih-s m-ey1','nn'],
'dismayed':['d-ih-s m-ey1-d','vbn jj'],
'dismaying':['d-ih-s m-ey1 ih-ng','jj'],
'dismember':['d-ih-s m-eh1-m b-er','vb'],
'dismembered':['d-ih-s m-eh1-m b-er-d','vbd'],
'dismembering':['d-ih-s m-eh1-m b-er ih-ng','vbg'],
'dismemberment':['d-ih-s m-eh1-m b-er m-ah-n-t','nn'],
'dismiss':['d-ih-s m-ih1-s','vb vbp'],
'dismissal':['d-ih-s m-ih1 s-ah-l','nn'],
'dismissed':['d-ih-s m-ih1-s-t','vbd vbn'],
'dismisses':['d-ih-s m-ih1 s-ih-z','vbz'],
'dismissing':['d-ih-s m-ih1 s-ih-ng','vbg'],
'disobedience':['d-ih-s ah b-iy1 d-iy ah-n-s','nn'],
'disobey':['d-ih-s ah b-ey1','vb'],
'disobeyed':['d-ih-s ow b-ey1-d','vbn vbd'],
'disorder':['d-ih-s ao1-r d-er','nn vb'],
'disorderly':['d-ih-s ao1-r d-er l-iy','jj'],
'disorders':['d-ih-s ao1-r d-er-z','nns'],
'disorganization':['d-ih-s ao-r g-ah n-ah z-ey1 sh-ah-n','nn'],
'disorganized':['d-ih-s ao1-r g-ah n-ay-z-d','jj vbn'],
'disoriented':['d-ih-s ao1 r-iy eh-n t-ih-d','vbn jj'],
'disowned':['d-ih-s ow1-n-d','vbd vbn'],
'disparage':['d-ih s-p-eh1 r-ih-jh','vb'],
'disparaged':['d-ih s-p-eh1 r-ih-jh-d','vbd vbn'],
'disparaging':['d-ih s-p-eh1 r-ih jh-ih-ng','vbg'],
'disparate':['d-ih1 s-p-er ih-t','jj'],
'disparity':['d-ih s-p-eh1 r-ah t-iy','nn'],
'dispassionate':['d-ih s-p-ae1 sh-ah-n ah-t','jj'],
'dispassionately':['d-ih s-p-ae1 sh-ah-n ah-t l-iy','rb'],
'dispatch':['d-ih s-p-ae1-ch','nn vb'],
'dispatched':['d-ih s-p-ae1-ch-t','vbd vbn'],
'dispatchers':['d-ih s-p-ae1 ch-er-z','nns'],
'dispatches':['d-ih s-p-ae1 ch-ih-z','nns'],
'dispatching':['d-ih s-p-ae1 ch-ih-ng','vbg'],
'dispel':['d-ih s-p-eh1-l','vb'],
'dispell':['d-ih s-p-eh1-l','vb'],
'dispelled':['d-ih s-p-eh1-l-d','vbn vbd'],
'dispensable':['d-ih s-p-eh1-n s-ah b-ah-l','jj'],
'dispensation':['d-ih s-p-ah-n s-ey1 sh-ah-n','nn'],
'dispense':['d-ih s-p-eh1-n-s','vb vbp'],
'dispensed':['d-ih s-p-eh1-n-s-t','vbn vbd'],
'dispenser':['d-ih s-p-eh1-n s-er','nn'],
'dispenses':['d-ih s-p-eh1-n s-ih-z','vbz'],
'dispensing':['d-ih s-p-eh1-n s-ih-ng','vbg nn'],
'dispersal':['d-ih s-p-er1 s-ah-l','nn'],
'dispersant':['d-ih s-p-er1 s-ah-n-t','nn'],
'disperse':['d-ih s-p-er1-s','vb'],
'dispersed':['d-ih s-p-er1-s-t','vbn jj vbd'],
'dispersing':['d-ih s-p-er1 s-ih-ng','vbg'],
'dispersion':['d-ih s-p-er1 zh-ah-n','nn'],
'displace':['d-ih-s p-l-ey1-s','vb'],
'displaced':['d-ih-s p-l-ey1-s-t','vbn vbd'],
'displacement':['d-ih-s p-l-ey1-s m-ah-n-t','nn'],
'displaces':['d-ih-s p-l-ey1 s-ih-z','vbz'],
'displacing':['d-ih-s p-l-ey1 s-ih-ng','nn vbg'],
'display':['d-ih s-p-l-ey1','nn vbp vb'],
'displayed':['d-ih s-p-l-ey1-d','vbn vbd'],
'displaying':['d-ih s-p-l-ey1 ih-ng','vbg'],
'displays':['d-ih s-p-l-ey1-z','nns vbz'],
'displeased':['d-ih-s p-l-iy1-z-d','vbn jj'],
'displeasure':['d-ih-s p-l-eh1 zh-er','nn'],
'disposable':['d-ih s-p-ow1 z-ah b-ah-l','jj nn'],
'disposal':['d-ih s-p-ow1 z-ah-l','nn'],
'dispose':['d-ih s-p-ow1-z','vb'],
'disposed':['d-ih s-p-ow1-z-d','vbn jj vbd'],
'disposes':['d-ih s-p-ow1 z-ih-z','vbz'],
'disposing':['d-ih s-p-ow1 z-ih-ng','vbg'],
'disposition':['d-ih s-p-ah z-ih1 sh-ah-n','nn'],
'dispossessed':['d-ih-s p-ah z-eh1-s-t','vbn jj'],
'disproportionate':['d-ih s-p-r-ah p-ao1-r sh-ah-n ih-t','jj'],
'disproportionately':['d-ih s-p-r-ah p-ao1-r sh-ah-n ah-t l-iy','rb'],
'disprove':['d-ih-s p-r-uw1-v','vb'],
'dispute':['d-ih s-p-y-uw1-t','nn vbp vb'],
'disputed':['d-ih s-p-y-uw1 t-ah-d','vbn vbd jj'],
'disputes':['d-ih s-p-y-uw1-t-s','nns vbz'],
'disqualification':['d-ih-s k-w-aa l-ah f-ah k-ey1 sh-ah-n','nn'],
'disqualified':['d-ih-s k-w-aa1 l-ah f-ay-d','vbn vbd'],
'disqualify':['d-ih-s k-w-aa1 l-ah f-ay','vb'],
'disquiet':['d-ih-s k-w-ay1 ah-t','nn'],
'disquieting':['d-ih-s k-w-ay1 ah t-ih-ng','jj'],
'disregard':['d-ih-s r-ih g-aa1-r-d','nn vb'],
'disregarded':['d-ih-s r-ih g-aa1-r d-ih-d','vbd vbn'],
'disregarding':['d-ih-s r-ih g-aa1-r d-ih-ng','vbg'],
'disrepair':['d-ih-s r-ih p-eh1-r','nn'],
'disreputable':['d-ih-s r-eh1 p-y-ah t-ah b-ah-l','jj'],
'disrepute':['d-ih-s r-ih p-y-uw1-t','nn'],
'disrespect':['d-ih-s r-ih s-p-eh1-k-t','nn'],
'disrupt':['d-ih-s r-ah1-p-t','vb vbp'],
'disrupted':['d-ih-s r-ah1-p t-ih-d','vbn jj vbd'],
'disrupting':['d-ih-s r-ah1-p t-ih-ng','vbg'],
'disruption':['d-ih-s r-ah1-p sh-ah-n','nn'],
'disruptive':['d-ih-s r-ah1-p t-ih-v','jj'],
'disrupts':['d-ih-s r-ah1-p-t-s','vbz'],
'dissatisfaction':['d-ih-s ae t-ih-s f-ae1-k sh-ah-n','nn'],
'dissatisfied':['d-ih-s ae1 t-ah-s f-ay-d','jj vbn'],
'dissected':['d-ay s-eh1-k t-ah-d','vbd vbn'],
'dissecting':['d-ay s-eh1-k t-ih-ng','vbg'],
'dissection':['d-ay s-eh1-k sh-ah-n','nn'],
'dissects':['d-ay s-eh1-k-t-s','vbz'],
'disseminate':['d-ih s-eh1 m-ah n-ey-t','vb'],
'disseminated':['d-ih s-eh1 m-ah n-ey t-ah-d','vbn'],
'disseminating':['d-ih s-eh1 m-ah n-ey t-ih-ng','vbg'],
'dissemination':['d-ih s-eh m-ah n-ey1 sh-ah-n','nn'],
'dissension':['d-ih s-eh1-n sh-ah-n','nn'],
'dissent':['d-ih s-eh1-n-t','nn vbp vb'],
'dissented':['d-ih s-eh1-n t-ih-d','vbd vbn'],
'dissenter':['d-ih s-eh1-n t-er','nn'],
'dissenting':['d-ih s-eh1-n t-ih-ng','jj vbg'],
'dissents':['d-ih s-eh1-n-t-s','nns'],
'disservice':['d-ih-s s-er1 v-ah-s','nn'],
'dissident':['d-ih1 s-ah d-ih-n-t','jj nn'],
'dissimilar':['d-ih s-s-ih1 m-ah l-er','jj'],
'dissipate':['d-ih1 s-ah p-ey-t','vb'],
'dissipated':['d-ih1 s-ah p-ey t-ih-d','vbn vbd'],
'dissipates':['d-ih1 s-ah p-ey-t-s','vbz'],
'dissipating':['d-ih1 s-ah p-ey t-ih-ng','vbg'],
'dissociate':['d-ih s-ow1 s-iy ey-t','vb'],
'dissociation':['d-ih s-ow s-iy ey1 sh-ah-n','nn'],
'dissolution':['d-ih s-ah l-uw1 sh-ah-n','nn'],
'dissolve':['d-ih z-aa1-l-v','vb vbp nn'],
'dissolved':['d-ih z-aa1-l-v-d','vbn vbd'],
'dissolves':['d-ih z-aa1-l-v-z','vbz'],
'dissolving':['d-ih z-ao1-l v-ih-ng','vbg jj'],
'dissonance':['d-ih1 s-ah n-ah-n-s','nn'],
'dissuade':['d-ih s-w-ey1-d','vb'],
'dissuaded':['d-ih s-w-ey1 d-ih-d','vbd vbn'],
'distal':['d-ih1 s-t-ah-l','jj'],
'distance':['d-ih1 s-t-ah-n-s','nn vb'],
'distances':['d-ih1 s-t-ah-n s-ah-z','nns vbz'],
'distancing':['d-ih1 s-t-ah-n s-ih-ng','vbg nn'],
'distant':['d-ih1 s-t-ah-n-t','jj'],
'distaste':['d-ih-s t-ey1-s-t','nn'],
'distasteful':['d-ih-s t-ey1-s-t f-ah-l','jj'],
'distillates':['d-ih1 s-t-ah l-ey-t-s','nns'],
'distillation':['d-ih s-t-ah l-ey1 sh-ah-n','nn'],
'distilled':['d-ih s-t-ih1-l-d','vbn jj'],
'distiller':['d-ih s-t-ih1 l-er','nn'],
'distillery':['d-ih s-t-ih1 l-er iy','nn'],
'distilling':['d-ih s-t-ih1 l-ih-ng','vbg nn'],
'distills':['d-ih s-t-ih1-l-z','vbz'],
'distinct':['d-ih s-t-ih1-ng-k-t','jj'],
'distinction':['d-ih s-t-ih1-ng-k sh-ah-n','nn'],
'distinctive':['d-ih s-t-ih1-ng-k t-ih-v','jj'],
'distinctively':['d-ih s-t-ih1-ng-k t-ih-v l-iy','rb'],
'distinctiveness':['d-ih s-t-ih1-ng-k t-ih-v n-ah-s','nn'],
'distinctly':['d-ih s-t-ih1-ng-k-t l-iy','rb'],
'distinguish':['d-ih s-t-ih1-ng g-w-ih-sh','vb vbp'],
'distinguishable':['d-ih s-t-ih1-ng g-w-ih sh-ah b-ah-l','jj'],
'distinguished':['d-ih s-t-ih1-ng g-w-ih-sh-t','vbn jj vbd'],
'distinguishes':['d-ih s-t-ih1-ng g-w-ih sh-ih-z','vbz'],
'distinguishing':['d-ih s-t-ih1-ng g-w-ih sh-ih-ng','vbg jj'],
'distort':['d-ih s-t-ao1-r-t','vb vbp'],
'distorted':['d-ih s-t-ao1-r t-ah-d','vbn vbd jj'],
'distorting':['d-ih s-t-ao1-r t-ih-ng','vbg'],
'distortion':['d-ih s-t-ao1-r sh-ah-n','nn'],
'distorts':['d-ih s-t-ao1-r-t-s','vbz'],
'distract':['d-ih s-t-r-ae1-k-t','vb'],
'distracted':['d-ih s-t-r-ae1-k t-ah-d','vbn vbd'],
'distracting':['d-ih s-t-r-ae1-k t-ih-ng','vbg jj'],
'distraction':['d-ih-s t-r-ae1-k sh-ah-n','nn'],
'distraught':['d-ih s-t-r-ao1-t','jj'],
'distress':['d-ih s-t-r-eh1-s','nn'],
'distressed':['d-ih s-t-r-eh1-s-t','jj vbn'],
'distresses':['d-ih s-t-r-eh1 s-ih-z','nns vbz'],
'distressing':['d-ih s-t-r-eh1 s-ih-ng','jj'],
'distressingly':['d-ih s-t-r-eh1 s-ih-ng l-iy','rb'],
'distribute':['d-ih s-t-r-ih1 b-y-uw-t','vb vbp'],
'distributed':['d-ih-s t-r-ih1 b-y-ah t-ah-d','vbn vbd jj'],
'distributes':['d-ih s-t-r-ih1 b-y-uw-t-s','vbz'],
'distributing':['d-ih s-t-r-ih1 b-y-uw t-ih-ng','vbg nn'],
'distribution':['d-ih-s t-r-ah b-y-uw1 sh-ah-n','nn'],
'distributive':['d-ih s-t-r-ih1 b-y-uw t-ih-v','jj'],
'distributor':['d-ih-s t-r-ih1 b-y-ah t-er','nn'],
'distributorship':['d-ih s-t-r-ih1 b-y-uw t-er sh-ih-p','nn'],
'district':['d-ih1 s-t-r-ih-k-t','nn jj'],
'distrust':['d-ih-s t-r-ah1-s-t','nn vbp vb'],
'distrusted':['d-ih-s t-r-ah1 s-t-ah-d','vbn vbd'],
'distrusts':['d-ih-s t-r-ah1-s-t-s','vbz'],
'disturb':['d-ih s-t-er1-b','vb vbp'],
'disturbance':['d-ih s-t-er1 b-ah-n-s','nn'],
'disturbed':['d-ih s-t-er1-b-d','vbn vbd jj'],
'disturbing':['d-ih s-t-er1 b-ih-ng','jj vbg'],
'disturbingly':['d-ih s-t-er1 b-ih-ng l-iy','rb'],
'disturbs':['d-ih s-t-er1-b-z','vbz'],
'disunion':['d-ih-s y-uw1 n-y-ah-n','nn'],
'disunity':['d-ih-s y-uw1 n-ah t-iy','nn'],
'ditch':['d-ih1-ch','nn'],
'ditched':['d-ih1-ch-t','vbd'],
'ditches':['d-ih1 ch-ah-z','nns vbz'],
'dithering':['d-ih1 dh-er ih-ng','vbg'],
'dithers':['d-ih1 dh-er-z','vbz'],
'ditto':['d-ih1 t-ow','nn'],
'ditty':['d-ih1 t-iy','nn'],
'diuretic':['d-ay uw r-eh1 t-ih-k','nn jj'],
'diurnal':['d-ay er1 n-ah-l','jj'],
'diva':['d-iy1 v-ah','nn'],
'divan':['d-ih v-ae1-n','nn'],
'dive':['d-ay1-v','nn vbp vb'],
'dived':['d-ay1-v-d','vbd vbn'],
'diver':['d-ay1 v-er','nn'],
'diverge':['d-ih v-er1-jh','vb vbp'],
'divergence':['d-ay v-er1 jh-ah-n-s','nn'],
'divergent':['d-ay v-er1 jh-ah-n-t','jj'],
'diverging':['d-ay v-er1 jh-ih-ng','vbg'],
'divers':['d-ay1 v-er-z','nns jj'],
'diverse':['d-ay v-er1-s','jj'],
'diversification':['d-ay v-er s-ah f-ah k-ey1 sh-ah-n','nn'],
'diversified':['d-ay v-er1 s-ah f-ay-d','jj vbd vbn'],
'diversify':['d-ay v-er1 s-ah f-ay','vb'],
'diversifying':['d-ay v-er1 s-ah f-ay ih-ng','vbg'],
'diversion':['d-ay v-er1 zh-ah-n','nn'],
'diversionary':['d-ay v-er1 zh-ah-n eh r-iy','jj'],
'diversity':['d-ih v-er1 s-ah t-iy','nn'],
'divert':['d-ay v-er1-t','vb vbp'],
'diverted':['d-ay v-er1 t-ih-d','vbn vbd'],
'divertimento':['d-ih v-er t-ah m-eh1-n t-ow','jj'],
'diverting':['d-ay v-er1 t-ih-ng','vbg'],
'dives':['d-ay1-v-z','nns vbz'],
'divest':['d-ay v-eh1-s-t','vb vbp'],
'divested':['d-ay v-eh1 s-t-ih-d','vbn jj vbd'],
'divesting':['d-ay v-eh1 s-t-ih-ng','vbg'],
'divestiture':['d-ih v-eh1 s-t-ih ch-er','nn'],
'divestment':['d-ay v-eh1-s-t m-ah-n-t','nn'],
'divide':['d-ih v-ay1-d','vb nn vbp'],
'divided':['d-ih v-ay1 d-ah-d','vbn vbd jj'],
'dividend':['d-ih1 v-ih d-eh-n-d','nn'],
'divider':['d-ih v-ay1 d-er','nn'],
'divides':['d-ih v-ay1-d-z','vbz'],
'dividing':['d-ih v-ay1 d-ih-ng','vbg nn jj'],
'divination':['d-ih v-ah n-ey1 sh-ah-n','nn'],
'divine':['d-ih v-ay1-n','jj nn'],
'divinely':['d-ih v-ay1-n l-iy','rb'],
'diving':['d-ay1 v-ih-ng','vbg jj nn'],
'divining':['d-ah v-ay1 n-ih-ng','vbg'],
'divinity':['d-ih v-ih1 n-ah t-iy','nn'],
'divisible':['d-ih v-ih1 z-ah b-ah-l','jj'],
'division':['d-ih v-ih1 zh-ah-n','nn'],
'divisional':['d-ih v-ih1 zh-ah n-ah-l','jj'],
'divisive':['d-ih v-ay1 s-ih-v','jj'],
'divisiveness':['d-ih v-ay1 s-ih-v n-ah-s','nn'],
'divorce':['d-ih v-ao1-r-s','nn vb vbp'],
'divorced':['d-ih v-ao1-r-s-t','vbn jj vbd'],
'divorcee':['d-ah v-ao1-r s-iy','nn'],
'divorces':['d-ih v-ao1-r s-ih-z','nns'],
'divulge':['d-ih v-ah1-l-jh','vb'],
'divulging':['d-ih v-ah1-l jh-ih-ng','vbg'],
'divvied':['d-ih1 v-iy-d','vbn'],
'dizziness':['d-ih1 z-iy n-ah-s','nn'],
'dizzy':['d-ih1 z-iy','jj'],
'dizzying':['d-ih1 z-iy ih-ng','jj vbg'],
'do':['d-uw1','vbp nn vb nns vbz'],
'doable':['d-uw1 ah b-ah-l','jj'],
'doan':['d-ow1-n','vbp'],
'doble':['d-ow1 b-ah-l','nn'],
'doc':['d-aa1-k','nn'],
'docile':['d-aa1 s-ah-l','jj'],
'dock':['d-aa1-k','nn vb'],
'docked':['d-aa1-k-t','vbn'],
'docket':['d-aa1 k-ah-t','nn'],
'docks':['d-aa1-k-s','nns'],
'dockside':['d-aa1-k s-ay-d','nn'],
'dockworkers':['d-aa1-k w-er k-er-z','nns'],
'doctor':['d-aa1-k t-er','nn vb'],
'doctoral':['d-aa1-k t-er ah-l','jj'],
'doctorate':['d-aa1-k t-er ah-t','nn'],
'doctored':['d-aa1-k t-er-d','vbn'],
'doctoring':['d-aa1-k t-er ih-ng','nn'],
'doctors':['d-aa1-k t-er-z','nns'],
'doctrinaire':['d-aa-k t-r-ah n-eh1-r','jj nn'],
'doctrinal':['d-aa1-k t-r-ah n-ah-l','jj'],
'doctrine':['d-aa1-k t-r-ah-n','nn'],
'document':['d-aa1 k-y-ah m-eh-n-t','nn vb vbp'],
'documentary':['d-aa k-y-ah m-eh1-n t-er iy','nn jj'],
'documentation':['d-aa k-y-ah m-eh-n t-ey1 sh-ah-n','nn'],
'documented':['d-aa1 k-y-ah m-eh-n t-ah-d','vbn vbd'],
'documenting':['d-aa1 k-y-ah m-ah-n t-ih-ng','vbg'],
'documents':['d-aa1 k-y-ah m-ah-n-t-s','nns vbz'],
'doddering':['d-aa1 d-er ih-ng','jj'],
'dodge':['d-aa1-jh','vbp nn vb'],
'dodged':['d-aa1-jh-d','vbd vbn'],
'dodging':['d-aa1 jh-ih-ng','vbg'],
'doe':['d-ow1','nn'],
'doer':['d-uw1-r','nn'],
'does':['d-ah1-z','vbz'],
'doffing':['d-ao1 f-ih-ng','vbg'],
'dog':['d-ao1-g','nn vbp'],
'dogfight':['d-aa1-g f-ay-t','nn'],
'dogged':['d-ao1-g-d','vbn jj vbd'],
'doggedly':['d-ao1 g-ah-d l-iy','rb'],
'doggerel':['d-aa1 g-er ah-l','nn'],
'dogging':['d-ao1 g-ih-ng','vbg'],
'doghouse':['d-ao1-g hh-aw-s','nn'],
'dogma':['d-aa1-g m-ah','nn'],
'dogmatic':['d-aa-g m-ae1 t-ih-k','jj'],
'dogmatically':['d-aa-g m-ae1 t-ih-k l-iy','rb'],
'dogmatism':['d-aa1-g m-ah t-ih z-ah-m','nn'],
'dogs':['d-aa1-g-z','nns vbz'],
'dogwood':['d-ao1-g w-uh-d','nn'],
'doing':['d-uw1 ih-ng','vbg nn'],
'doldrums':['d-ow1-l d-r-ah-m-z','nns nn'],
'dole':['d-ow1-l','vb nn vbp'],
'doled':['d-ow1-l-d','vbd vbn'],
'doleful':['d-ow1-l f-ah-l','jj'],
'doles':['d-ow1-l-z','vbz'],
'doling':['d-ow1 l-ih-ng','vbg'],
'doll':['d-aa1-l','nn'],
'dollar':['d-aa1 l-er','nn'],
'dollop':['d-aa1 l-ah-p','nn'],
'dolphin':['d-aa1-l f-ah-n','nn'],
'domain':['d-ow m-ey1-n','nn'],
'dome':['d-ow1-m','nn'],
'domed':['d-ow1-m-d','jj'],
'domestic':['d-ah m-eh1 s-t-ih-k','jj nn'],
'domestically':['d-ah m-eh1 s-t-ih-k l-iy','rb'],
'domesticity':['d-ow m-eh s-t-ih1 s-ah t-iy','nn'],
'dominance':['d-aa1 m-ah n-ah-n-s','nn'],
'dominant':['d-aa1 m-ah n-ah-n-t','jj'],
'dominate':['d-aa1 m-ah n-ey-t','vb vbp'],
'dominated':['d-aa1 m-ah n-ey t-ah-d','vbn vbd'],
'dominates':['d-aa1 m-ah n-ey-t-s','vbz'],
'dominating':['d-aa1 m-ah n-ey t-ih-ng','vbg jj'],
'domination':['d-aa m-ah n-ey1 sh-ah-n','nn'],
'domineering':['d-aa m-ah n-ih1 r-ih-ng','vbg jj'],
'dominion':['d-ah m-ih1 n-y-ah-n','nn'],
'domino':['d-aa1 m-ah n-ow','nn'],
'dominoes':['d-aa1 m-ah n-ow-z','nn'],
'don':['d-aa1-n','vb nn vbp'],
'donate':['d-ow1 n-ey-t','vb vbp'],
'donated':['d-ow1 n-ey t-ah-d','vbn vbd'],
'donates':['d-ow1 n-ey-t-s','vbz'],
'donating':['d-ow1 n-ey t-ih-ng','vbg'],
'donation':['d-ow n-ey1 sh-ah-n','nn'],
'done':['d-ah1-n','vbn jj rb vbd'],
'donkey':['d-aa1-ng k-iy','nn'],
'donna':['d-aa1 n-ah','nn'],
'donned':['d-aa1-n-d','vbd'],
'donning':['d-aa1 n-ih-ng','vbg'],
'donnybrook':['d-aa1 n-iy b-r-uh-k','nn'],
'donor':['d-ow1 n-er','nn'],
'dons':['d-aa1-n-z','vbz nns'],
'donut':['d-ow1 n-ah-t','nn'],
'doo':['d-uw1','nn vb'],
'doodads':['d-uw1 d-ae-d-z','nns'],
'doom':['d-uw1-m','nn vb'],
'doomed':['d-uw1-m-d','vbn vbd jj'],
'dooming':['d-uw1 m-ih-ng','vbg'],
'dooms':['d-uw1-m-z','nns'],
'doomsayers':['d-uw-m s-ey1 er-z','nns'],
'doomsday':['d-uw1-m-z d-ey','nn'],
'door':['d-ao1-r','nn rb'],
'doorkeeper':['d-ao1-r k-iy p-er','nn'],
'doorknob':['d-uw1-r n-aa-b','nn'],
'doorman':['d-ao1-r m-ae-n','nn'],
'doorstep':['d-ao1-r s-t-eh-p','nn'],
'doorway':['d-ao1-r w-ey','nn'],
'dope':['d-ow1-p','nn'],
'doped':['d-ow1-p-t','jj'],
'dorm':['d-ao1-r-m','nn'],
'dormant':['d-ao1-r m-ah-n-t','jj'],
'dormitory':['d-ao1-r m-ah t-ao r-iy','nn'],
'dosage':['d-ow1 s-ah-jh','nn'],
'dose':['d-ow1-s','nn'],
'doses':['d-ow1 s-ah-z','nns'],
'dossier':['d-ao s-y-ey1','nn'],
'dost':['d-aa1-s-t','vbp'],
'dot':['d-aa1-t','nn vbp vb'],
'doting':['d-ow1 t-ih-ng','vbg jj'],
'dots':['d-aa1-t-s','nns'],
'dotted':['d-aa1 t-ah-d','vbn vbd jj'],
'dotting':['d-aa1 t-ih-ng','vbg nn'],
'double':['d-ah1 b-ah-l','jj vbp nn rb vb'],
'doubled':['d-ah1 b-ah-l-d','vbd vbn'],
'doubles':['d-ah1 b-ah-l-z','nns vbz'],
'doubling':['d-ah1 b-ah-l ih-ng','vbg nn'],
'doubly':['d-ah1 b-l-iy','rb'],
'doubt':['d-aw1-t','nn vbp rb vb'],
'doubted':['d-aw1 t-ih-d','vbd vbn'],
'doubters':['d-aw1 t-er-z','nns'],
'doubtful':['d-aw1-t f-ah-l','jj'],
'doubting':['d-aw1 t-ih-ng','vbg jj'],
'doubtless':['d-aw1-t l-ah-s','rb'],
'doubts':['d-aw1-t-s','nns vbz'],
'dough':['d-ow1','nn'],
'doughnut':['d-ow1 n-ah-t','nn'],
'doughty':['d-ao1 t-iy','jj'],
'dour':['d-aw1 er','jj'],
'douse':['d-aw1-s','vb'],
'doused':['d-aw1-s-t','vbd vbn'],
'dove':['d-ah1-v','nn vbd'],
'dovetail':['d-ah1-v t-ey-l','vbp vb'],
'dovetails':['d-ah1-v t-ey-l-z','vbz'],
'dovish':['d-ow1 v-ih-sh','jj'],
'dowdy':['d-aw1 d-iy','jj'],
'dowel':['d-aw1 ah-l','nn'],
'dower':['d-aw1-r','nn'],
'down':['d-aw1-n','rb in rbr vbp jj nn rp vb'],
'downbeat':['d-aw-n b-iy1-t','jj nn'],
'downcast':['d-aw1-n k-ae-s-t','jj'],
'downdraft':['d-aw1-n d-r-ae-f-t','nn'],
'downed':['d-aw1-n-d','vbd vbn'],
'downer':['d-aw1 n-er','nn'],
'downfall':['d-aw1-n f-ao-l','nn'],
'downgrade':['d-aw1-n g-r-ey1-d','nn vb'],
'downgraded':['d-aw1-n g-r-ey1 d-ah-d','vbd vbn'],
'downgrades':['d-aw1-n g-r-ey1-d-z','nns'],
'downgrading':['d-aw1-n g-r-ey1 d-ih-ng','nn vbg'],
'downhill':['d-aw1-n hh-ih1-l','rb jj'],
'downing':['d-aw1 n-ih-ng','vbg'],
'downpayment':['d-aw-n p-ey1 m-ah-n-t','nn'],
'downplay':['d-aw1-n p-l-ey','vb'],
'downplayed':['d-aw1-n p-l-ey-d','vbd'],
'downplaying':['d-aw1-n p-l-ey ih-ng','vbg'],
'downplays':['d-aw1-n p-l-ey-z','vbz'],
'downpour':['d-aw1-n p-ao-r','nn'],
'downright':['d-aw1-n r-ay-t','rb jj'],
'downs':['d-aw1-n-z','nns'],
'downside':['d-aw1-n s-ay-d','nn jj'],
'downsize':['d-aw1-n s-ay-z','vb'],
'downsized':['d-aw1-n s-ay-z-d','vbn'],
'downsizing':['d-aw1-n s-ay z-ih-ng','vbg nn'],
'downstairs':['d-aw1-n s-t-eh1-r-z','nn rb'],
'downstream':['d-aw1-n s-t-r-iy1-m','rb jj'],
'downtime':['d-aw1-n t-ay-m','nn'],
'downtown':['d-aw1-n t-aw1-n','nn jj rb'],
'downtrend':['d-aw1-n t-r-eh-n-d','nn'],
'downtrodden':['d-aw-n t-r-aa1 d-ah-n','jj'],
'downturn':['d-aw1-n t-er-n','nn'],
'downward':['d-aw1-n w-er-d','jj rb'],
'dowry':['d-aw1 r-iy','nn'],
'dozed':['d-ow1-z-d','vbd'],
'dozens':['d-ah1 z-ah-n-z','nns'],
'dozing':['d-ow1 z-ih-ng','vbg'],
'drab':['d-r-ae1-b','jj'],
'drachma':['d-r-ae1-k m-ah','nn'],
'drachmas':['d-r-aa1-k m-ah-z','nns nn'],
'draconian':['d-r-ey k-ow1 n-iy ah-n','jj'],
'draft':['d-r-ae1-f-t','nn jj vb vbp'],
'drafted':['d-r-ae1-f t-ih-d','vbn vbd'],
'draftees':['d-r-ae1-f t-iy1-z','nns'],
'drafters':['d-r-ae1-f t-er-z','nns'],
'drafting':['d-r-ae1-f t-ih-ng','vbg nn'],
'drafts':['d-r-ae1-f-t-s','nns'],
'draftsmen':['d-r-ae1-f-t-s m-ah-n','nns'],
'drafty':['d-r-ae1-f t-iy','jj'],
'drag':['d-r-ae1-g','nn vbp vb'],
'dragged':['d-r-ae1-g-d','vbn vbd'],
'dragging':['d-r-ae1 g-ih-ng','vbg nn'],
'dragnet':['d-r-ae1-g n-eh-t','nn'],
'dragon':['d-r-ae1 g-ah-n','nn'],
'drags':['d-r-ae1-g-z','vbz'],
'drain':['d-r-ey1-n','nn vbp vb'],
'drainage':['d-r-ey1 n-ah-jh','nn'],
'drained':['d-r-ey1-n-d','vbn vbd'],
'draining':['d-r-ey1 n-ih-ng','vbg nn'],
'drains':['d-r-ey1-n-z','nns vbz'],
'dram':['d-r-ae1-m','nn'],
'drama':['d-r-aa1 m-ah','nn'],
'dramas':['d-r-aa1 m-ah-z','nn'],
'dramatic':['d-r-ah m-ae1 t-ih-k','jj'],
'dramatically':['d-r-ah m-ae1 t-ih-k l-iy','rb'],
'dramatist':['d-r-aa1 m-ah t-ih-s-t','nn'],
'dramatization':['d-r-ae m-ah t-ah z-ey1 sh-ah-n','nn'],
'dramatize':['d-r-aa1 m-ah t-ay-z','vb vbp'],
'dramatized':['d-r-ae1 m-ah t-ay-z-d','vbn'],
'dramatizes':['d-r-ae1 m-ah t-ay z-ih-z','vbz'],
'dramatizing':['d-r-ae1 m-ah t-ay z-ih-ng','vbg'],
'drank':['d-r-ae1-ng-k','vbd'],
'drape':['d-r-ey1-p','nn vb'],
'draped':['d-r-ey1-p-t','vbd jj vbn'],
'draper':['d-r-ey1 p-er','nn'],
'drapery':['d-r-ey1 p-er iy','nn'],
'drapes':['d-r-ey1-p-s','nns'],
'draping':['d-r-ey1 p-ih-ng','vbg'],
'drastic':['d-r-ae1 s-t-ih-k','jj rb'],
'drastically':['d-r-ae1 s-t-ih-k l-iy','rb'],
'draught':['d-r-ae1-f-t','nn'],
'draughts':['d-r-ae1-f-t-s','nn nns'],
'draw':['d-r-ao1','vb vbp nn'],
'drawback':['d-r-ao1 b-ae-k','nn'],
'drawbridge':['d-r-ao1 b-r-ih-jh','nn'],
'drawdown':['d-r-ao1 d-aw-n','nn'],
'drawer':['d-r-ao1-r','nn'],
'drawing':['d-r-ao1 ih-ng','vbg jj nn'],
'drawl':['d-r-ao1-l','nn'],
'drawled':['d-r-ao1-l-d','vbd'],
'drawn':['d-r-ao1-n','vbn jj'],
'draws':['d-r-ao1-z','vbz nns'],
'dread':['d-r-eh1-d','nn jj'],
'dreaded':['d-r-eh1 d-ih-d','vbn vbd jj'],
'dreadful':['d-r-eh1-d f-ah-l','jj nn'],
'dreadfully':['d-r-eh1-d f-ah l-iy','rb'],
'dreading':['d-r-eh1 d-ih-ng','vbg'],
'dream':['d-r-iy1-m','nn vb vbp'],
'dreamed':['d-r-iy1-m-d','vbd vbn'],
'dreamer':['d-r-iy1 m-er','nn'],
'dreaming':['d-r-iy1 m-ih-ng','vbg nn'],
'dreamlike':['d-r-iy1-m l-ay-k','jj'],
'dreams':['d-r-iy1-m-z','nns vbz'],
'dreamt':['d-r-eh1-m-t','vbd vbn'],
'dreamy':['d-r-iy1 m-iy','jj'],
'dreariness':['d-r-iy1 r-iy n-ah-s','nn'],
'dreary':['d-r-ih1 r-iy','jj'],
'dredged':['d-r-eh1-jh-d','vbd'],
'dredges':['d-r-eh1 jh-ah-z','vbz'],
'dregs':['d-r-eh1-g-z','nns'],
'drenching':['d-r-eh1-n ch-ih-ng','nn'],
'dress':['d-r-eh1-s','nn vbp vb'],
'dressed':['d-r-eh1-s-t','vbn jj vbd'],
'dresser':['d-r-eh1 s-er','nn'],
'dresses':['d-r-eh1 s-ah-z','nns vbz'],
'dressing':['d-r-eh1 s-ih-ng','nn vbg jj'],
'dressmaking':['d-r-eh1-s m-ey k-ih-ng','nn'],
'dressy':['d-r-eh1 s-iy','jj'],
'drew':['d-r-uw1','vbd'],
'dribble':['d-r-ih1 b-ah-l','nn'],
'dribbled':['d-r-ih1 b-ah-l-d','vbd'],
'dried':['d-r-ay1-d','vbn vbd jj'],
'drier':['d-r-ay1 er','nn jjr rbr'],
'dries':['d-r-ay1-z','nns vbz'],
'drift':['d-r-ih1-f-t','nn vbp vb'],
'drifted':['d-r-ih1-f t-ah-d','vbd vbn'],
'drifter':['d-r-ih1-f t-er','nn'],
'drifting':['d-r-ih1-f t-ih-ng','vbg jj nn'],
'driftnet':['d-r-ih1-f-t n-eh-t','nn'],
'drifts':['d-r-ih1-f-t-s','vbz nns'],
'driftwood':['d-r-ih1-f-t w-uh-d','nn'],
'drill':['d-r-ih1-l','nn vb'],
'drilled':['d-r-ih1-l-d','vbn vbd'],
'drillers':['d-r-ih1 l-er-z','nns'],
'drilling':['d-r-ih1 l-ih-ng','nn jj vbg'],
'drills':['d-r-ih1-l-z','nns vbz'],
'drink':['d-r-ih1-ng-k','nn vbp vb'],
'drinker':['d-r-ih1-ng k-er','nn'],
'drinking':['d-r-ih1-ng k-ih-ng','nn jj vbg'],
'drinks':['d-r-ih1-ng-k-s','nns vbz'],
'drip':['d-r-ih1-p','nn'],
'dripped':['d-r-ih1-p-t','vbd'],
'dripping':['d-r-ih1 p-ih-ng','vbg'],
'drips':['d-r-ih1-p-s','vbz'],
'drive':['d-r-ay1-v','nn vbp vb'],
'driven':['d-r-ih1 v-ah-n','vbn jj'],
'driver':['d-r-ay1 v-er','nn'],
'drives':['d-r-ay1-v-z','nns vbz'],
'driveway':['d-r-ay1-v w-ey','nn'],
'driving':['d-r-ay1 v-ih-ng','vbg jj nn'],
'drizzle':['d-r-ih1 z-ah-l','nn'],
'drizzling':['d-r-ih1 z-ah-l ih-ng','vbg jj'],
'drizzly':['d-r-ih1-z l-iy','jj'],
'drone':['d-r-ow1-n','nn jj'],
'drooling':['d-r-uw1 l-ih-ng','vbg'],
'droop':['d-r-uw1-p','vbp nn'],
'drooped':['d-r-uw1-p-t','vbd'],
'drooping':['d-r-uw1 p-ih-ng','vbg'],
'drop':['d-r-aa1-p','nn jj vb vbp'],
'droplets':['d-r-aa1 p-l-ah-t-s','nns'],
'dropoff':['d-r-aa1 p-ao-f','nn'],
'dropout':['d-r-aa1-p aw-t','nn'],
'dropped':['d-r-aa1-p-t','vbd vbn'],
'droppers':['d-r-aa1 p-er-z','nns'],
'dropping':['d-r-aa1 p-ih-ng','vbg nn'],
'drops':['d-r-aa1-p-s','vbz nns'],
'dross':['d-r-ao1-s','nn'],
'drought':['d-r-aw1-t','nn'],
'drove':['d-r-ow1-v','vbd nn'],
'drover':['d-r-ow1 v-er','nn'],
'drown':['d-r-aw1-n','vb vbp'],
'drowned':['d-r-aw1-n-d','vbn jj vbd'],
'drowning':['d-r-aw1 n-ih-ng','vbg'],
'drowns':['d-r-aw1-n-z','vbz'],
'drowsy':['d-r-aw1 z-iy','jj'],
'drubbed':['d-r-ah1-b-d','vbn'],
'drubbing':['d-r-ah1 b-ih-ng','nn'],
'drudgery':['d-r-ah1 jh-er iy','nn'],
'drug':['d-r-ah1-g','nn'],
'drugged':['d-r-ah1-g-d','vbn jj'],
'druggist':['d-r-ah1 g-ih-s-t','nn'],
'drugstore':['d-r-ah1-g s-t-ao-r','nn'],
'drum':['d-r-ah1-m','vb nn'],
'drumbeat':['d-r-ah1-m b-iy-t','nn'],
'drummed':['d-r-ah1-m-d','vbd vbn'],
'drummer':['d-r-ah1 m-er','nn'],
'drumming':['d-r-ah1 m-ih-ng','vbg'],
'drums':['d-r-ah1-m-z','nns'],
'drunk':['d-r-ah1-ng-k','jj nn vbn'],
'drunkard':['d-r-ah1-ng k-er-d','nn'],
'drunken':['d-r-ah1-ng k-ah-n','jj'],
'drunkenness':['d-r-ah1-ng k-ah-n n-ah-s','nn'],
'dry':['d-r-ay1','jj vb vbp'],
'dryer':['d-r-ay1 er','nn'],
'drying':['d-r-ay1 ih-ng','vbg nn'],
'dryly':['d-r-ay1 l-iy','rb'],
'dryness':['d-r-ay1 n-ah-s','nn'],
'dual':['d-uw1 ah-l','jj'],
'dualism':['d-uw1 ah l-ih z-ah-m','nn'],
'dubbed':['d-ah1-b-d','vbn vbd'],
'dubious':['d-uw1 b-iy ah-s','jj'],
'dubs':['d-ah1-b-z','vbz'],
'duck':['d-ah1-k','nn vb'],
'ducked':['d-ah1-k-t','vbd'],
'ducking':['d-ah1 k-ih-ng','vbg'],
'duckling':['d-ah1 k-l-ih-ng','nn'],
'ducks':['d-ah1-k-s','nns vbz'],
'duct':['d-ah1-k-t','nn'],
'dud':['d-ah1-d','nn'],
'dudgeon':['d-ah1 jh-ah-n','nn'],
'due':['d-uw1','jj in nn rb nns'],
'duel':['d-uw1 ah-l','nn'],
'dueling':['d-uw1 l-ih-ng','vbg nn'],
'duet':['d-uw eh1-t','nn'],
'duffel':['d-ah1 f-ah-l','nn'],
'duffer':['d-ah1 f-er','nn'],
'dug':['d-ah1-g','vbd vbn'],
'dugout':['d-ah1-g aw-t','nn'],
'duke':['d-uw1-k','nn'],
'dull':['d-ah1-l','jj'],
'dulled':['d-ah1-l-d','vbn'],
'duller':['d-ah1 l-er','jjr rbr'],
'dullest':['d-ah1 l-ah-s-t','jjs'],
'dulling':['d-ah1 l-ih-ng','vbg'],
'dullness':['d-ah1-l n-ah-s','nn'],
'duly':['d-uw1 l-iy','rb'],
'dumb':['d-ah1-m','jj'],
'dumbbell':['d-ah1-m b-eh-l','nn'],
'dumber':['d-ah1 m-er','jjr'],
'dumbest':['d-ah1 m-ah-s-t','jjs'],
'dumbfounded':['d-ah1-m f-aw-n d-ih-d','jj vbn'],
'dummy':['d-ah1 m-iy','jj nn'],
'dump':['d-ah1-m-p','vb vbp nn'],
'dumped':['d-ah1-m-p-t','vbd vbn'],
'dumping':['d-ah1-m p-ih-ng','vbg nn'],
'dumps':['d-ah1-m-p-s','vbz nns'],
'dumpster':['d-ah1-m-p s-t-er','nn'],
'dumpy':['d-ah1-m p-iy','jj'],
'dun':['d-ah1-n','nn'],
'dune':['d-uw1-n','nn'],
'dung':['d-ah1-ng','nn'],
'dungeon':['d-ah1-n jh-ah-n','nn'],
'dunk':['d-ah1-ng-k','nn vb'],
'dunked':['d-ah1-ng-k-t','vbd'],
'dunks':['d-ah1-ng-k-s','vbz'],
'duo':['d-uw1 ow','nn'],
'duodenal':['d-uw1 ah d-iy1 n-ah-l','jj'],
'duopoly':['d-uw1 ow p-aa l-iy','rb'],
'duped':['d-uw1-p-t','vbn'],
'duplex':['d-uw1 p-l-eh-k-s','nn jj'],
'duplicate':['d-uw1 p-l-ah k-ah-t','vb jj'],
'duplicated':['d-uw1 p-l-ih k-ey t-ih-d','vbn'],
'duplicates':['d-y-uw1 p-l-ah k-ey-t-s','vbz'],
'duplicating':['d-uw1 p-l-ih k-ey t-ih-ng','vbg'],
'duplication':['d-y-uw p-l-ah k-ey1 sh-ah-n','nn'],
'duplicative':['d-uw p-l-ih1 k-ah t-ih-v','jj'],
'duplicity':['d-uw p-l-ih1 s-ih t-iy','nn'],
'durability':['d-er ah b-ih1 l-ih t-iy','nn'],
'durable':['d-uh1 r-ah b-ah-l','jj'],
'duration':['d-uh1 r-ey1 sh-ah-n','nn'],
'duress':['d-uh1 r-eh-s','nn'],
'during':['d-uh1 r-ih-ng','in'],
'durn':['d-er1-n','jj'],
'dusk':['d-ah1-s-k','nn'],
'dust':['d-ah1-s-t','nn vb vbp'],
'dusted':['d-ah1 s-t-ih-d','vbn vbd'],
'dustin':['d-ah1 s-t-ih-n','vbg'],
'dusting':['d-ah1 s-t-ih-ng','vbg nn'],
'dusts':['d-ah1-s-t-s','nns'],
'dusty':['d-ah1 s-t-iy','jj'],
'dutiful':['d-uw1 t-iy f-ah-l','jj'],
'dutifully':['d-uw1 t-iy f-ah l-iy','rb'],
'duty':['d-uw1 t-iy','nn'],
'dwarf':['d-w-ao1-r-f','nn vbp vb'],
'dwarfed':['d-w-ao1-r-f-t','vbn vbd'],
'dwarfism':['d-w-ao1-r f-ih z-ah-m','nn'],
'dwarfs':['d-w-ao1-r-f-s','vbz nns'],
'dwell':['d-w-eh1-l','vbp vb'],
'dwelled':['d-w-eh1-l-d','vbn'],
'dweller':['d-w-eh1 l-er','nn'],
'dwelling':['d-w-eh1 l-ih-ng','nn vbg'],
'dwells':['d-w-eh1-l-z','vbz'],
'dwelt':['d-w-eh1-l-t','vbd'],
'dwindle':['d-w-ih1-n d-ah-l','vb'],
'dwindled':['d-w-ih1-n d-ah-l-d','vbd vbn'],
'dwindles':['d-w-ih1-n d-ah-l-z','vbz'],
'dwindling':['d-w-ih1-n d-ah-l ih-ng','vbg'],
'dye':['d-ay1','nn vb'],
'dyed':['d-ay1-d','vbn'],
'dyeing':['d-ay1 ih-ng','nn'],
'dyes':['d-ay1-z','nns'],
'dying':['d-ay1 ih-ng','vbg jj nn'],
'dykes':['d-ay1-k-s','nns'],
'dynamic':['d-ay n-ae1 m-ih-k','jj nn'],
'dynamism':['d-ay1 n-ah m-ih z-ah-m','nn'],
'dynamite':['d-ay1 n-ah m-ay-t','nn jj'],
'dynamo':['d-ay1 n-ah m-ow','nn'],
'dynastic':['d-ay n-ae1 s-t-ih-k','jj'],
'dynasty':['d-ay1 n-ah s-t-iy','nn'],
'dysentery':['d-ih1-s ah-n t-eh r-iy','nn'],
'dysfunction':['d-ih-s f-ah1-ng-k sh-ah-n','nn'],
'dyslexia':['d-ih-s l-eh1-k s-iy ah','nn'],
'dysplasia':['d-ih s-p-l-ey1 zh-ah','nn'],
'dystrophy':['d-ih1-s t-r-ah f-iy','nn'],
'e':['iy1','nn nnp'],
'each':['iy1-ch','dt'],
'eager':['iy1 g-er','jj'],
'eagerly':['iy1 g-er l-iy','rb'],
'eagerness':['iy1 g-er n-ah-s','nn'],
'eagle':['iy1 g-ah-l','nn'],
'ear':['ih1-r','nn'],
'eared':['ih1-r-d','jj'],
'earful':['iy1-r f-ah-l','nn'],
'earl':['er1-l','nn'],
'earlier':['er1 l-iy er','rbr jjr jj nn rb'],
'earliest':['er1 l-iy ah-s-t','jjs rbs'],
'early':['er1 l-iy','jj rb'],
'earmark':['ih1-r m-aa-r-k','vb'],
'earmarked':['ih1-r m-aa-r-k-t','vbn vbd jj'],
'earmarking':['ih1-r m-aa-r k-ih-ng','vbg'],
'earn':['er1-n','vb vbp'],
'earned':['er1-n-d','vbd jj vbn'],
'earner':['er1 n-er','nn'],
'earnest':['er1 n-ih-s-t','nn jj'],
'earnestly':['er1 n-ah-s-t l-iy','rb'],
'earnestness':['er1 n-ah-s-t n-ah-s','nn'],
'earning':['er1 n-ih-ng','vbg nn'],
'earnings':['er1 n-ih-ng-z','nns nn'],
'earns':['er1-n-z','vbz'],
'earphones':['ih1-r f-ow-n-z','nns'],
'earring':['ih1 r-ih-ng','nn'],
'earth':['er1-th','nn'],
'earthbound':['er1-th b-aw-n-d','jj'],
'earthenware':['er1 th-ah-n w-eh-r','nn'],
'earthlings':['er1-th l-ih-ng-z','nns'],
'earthly':['er1-th l-iy','jj'],
'earthmoving':['er1-th m-uw v-ih-ng','jj nn'],
'earthquake':['er1-th k-w-ey-k','nn'],
'earthshaking':['er1-th sh-ey k-ih-ng','jj'],
'earthworm':['er1-th w-er-m','nn'],
'earthy':['er1 th-iy','jj'],
'ease':['iy1-z','vb nn vbp'],
'eased':['iy1-z-d','vbd vbn'],
'easel':['iy1 z-ah-l','nn'],
'easement':['iy1-z m-ah-n-t','nn'],
'eases':['iy1 z-ih-z','vbz'],
'easier':['iy1 z-iy er','jjr rbr rb'],
'easiest':['iy1 z-iy ah-s-t','jjs'],
'easily':['iy1 z-ah l-iy','rb'],
'easing':['iy1 z-ih-ng','vbg jj nn'],
'east':['iy1-s-t','jj nn rb'],
'eastern':['iy1 s-t-er-n','jj'],
'eastward':['iy1-s-t w-er-d','rb'],
'easy':['iy1 z-iy','jj rb'],
'easygoing':['iy1 z-iy g-ow1 ih-ng','jj'],
'eat':['iy1-t','vb vbp'],
'eatable':['iy1 t-ah b-ah-l','jj'],
'eaten':['iy1 t-ah-n','vbn'],
'eaters':['iy1 t-er-z','nns'],
'eatery':['iy1 t-er iy','nn'],
'eating':['iy1 t-ih-ng','vbg jj nn'],
'eats':['iy1-t-s','vbz'],
'eave':['iy1-v','nn'],
'eavesdrop':['iy1-v-z d-r-aa-p','vb'],
'eavesdropping':['iy1-v-z d-r-aa p-ih-ng','nn'],
'ebb':['eh1-b','nn vb'],
'ebbed':['eh1-b-d','vbd vbn'],
'ebbing':['eh1 b-ih-ng','vbg'],
'ebbs':['eh1-b-z','vbz'],
'ebony':['eh1 b-ah n-iy','nn jj'],
'ebullient':['ih b-ah1-l y-ah-n-t','jj'],
'eccentric':['ih-k s-eh1-n t-r-ih-k','jj nn'],
'eccentricity':['eh-k s-ah-n t-r-ih1 s-ah t-iy','nn'],
'ecclesiastical':['ih k-l-iy z-iy ae1 s-t-ih k-ah-l','jj'],
'echelon':['eh1 sh-ah l-aa-n','nn'],
'echo':['eh1 k-ow','nn vbp vb'],
'echoed':['eh1 k-ow-d','vbd vbn'],
'echoes':['eh1 k-ow-z','nns vbz'],
'echoing':['eh1 k-ow ih-ng','vbg'],
'eclectic':['ih k-l-eh1-k t-ih-k','jj nn'],
'eclipse':['ih k-l-ih1-p-s','vb nn vbp'],
'eclipsed':['ih k-l-ih1-p-s-t','vbd vbn'],
'eclipses':['ih k-l-ih1-p s-ah-z','nns'],
'eclipsing':['ih k-l-ih1-p s-ih-ng','vbg'],
'ecliptic':['ih k-l-ih1-p t-ih-k','nn jj'],
'ecological':['iy k-ah l-aa1 jh-ih k-ah-l','jj'],
'ecologically':['iy k-ah l-aa1 jh-ih-k l-iy','rb'],
'ecologists':['ih k-aa1 l-ah jh-ih-s-t-s','nns'],
'ecology':['ih k-aa1 l-ah jh-iy','nn'],
'econometric':['ih k-aa n-ah m-eh1 t-r-ih-k','jj'],
'econometrics':['ih k-aa n-ah m-eh1 t-r-ih-k-s','nn'],
'economic':['eh k-ah n-aa1 m-ih-k','jj'],
'economical':['eh k-ah n-aa1 m-ih k-ah-l','jj'],
'economically':['eh k-ah n-aa1 m-ih-k l-iy','rb'],
'economics':['eh k-ah n-aa1 m-ih-k-s','nns nn'],
'economist':['ih k-aa1 n-ah m-ih-s-t','nn'],
'economize':['ih k-aa1 n-ah m-ay-z','vb'],
'economy':['ih k-aa1 n-ah m-iy','nn'],
'ecstasy':['eh1-k s-t-ah s-iy','nn'],
'ecstatic':['eh-k s-t-ae1 t-ih-k','jj nn'],
'ecstatically':['eh-k s-t-ae1 t-ih-k l-iy','rb'],
'ecumenical':['eh k-y-uw m-eh1 n-ih k-ah-l','jj'],
'edema':['ih d-iy1 m-ah','nn'],
'edge':['eh1-jh','nn vb'],
'edged':['eh1-jh-d','vbd jj vbn'],
'edges':['eh1 jh-ah-z','nns'],
'edginess':['eh1 jh-iy n-ah-s','nn'],
'edging':['eh1 jh-ih-ng','vbg nn'],
'edgy':['eh1 jh-iy','jj'],
'edible':['eh1 d-ah b-ah-l','jj'],
'edict':['iy1 d-ih-k-t','nn'],
'edifice':['eh1 d-ah f-ah-s','nn'],
'edifying':['eh1 d-ah f-ay ih-ng','jj'],
'edit':['eh1 d-ah-t','vb'],
'edited':['eh1 d-ah t-ah-d','vbn vbd'],
'editing':['eh1 d-ah t-ih-ng','nn vbg'],
'edition':['ah d-ih1 sh-ah-n','nn'],
'editor':['eh1 d-ah t-er','nn'],
'editorial':['eh d-ah t-ao1 r-iy ah-l','nn jj'],
'editorialist':['eh d-ah t-ao1 r-iy ah l-ih-s-t','nn'],
'editorialize':['eh d-ah t-ao1 r-iy ah l-ay-z','vb'],
'editorially':['eh d-ah t-ao1 r-iy ah l-iy','rb'],
'editorship':['eh1 d-ah t-er sh-ih-p','nn'],
'edits':['eh1 d-ih-t-s','vbz'],
'educate':['eh1 jh-ah k-ey-t','vb'],
'educated':['eh1 jh-ah k-ey t-ah-d','vbn jj'],
'educating':['eh1 jh-ah k-ey t-ih-ng','vbg nn'],
'education':['eh jh-ah k-ey1 sh-ah-n','nn'],
'educational':['eh jh-ah k-ey1 sh-ah n-ah-l','jj'],
'educator':['eh1 jh-ah k-ey t-er','nn'],
'eel':['iy1-l','nn'],
'eerie':['ih1 r-iy','jj'],
'eerily':['ih1 r-ah l-iy','rb'],
'effect':['ih f-eh1-k-t','nn jj vb vbp'],
'effected':['ih f-eh1-k t-ah-d','vbn vbd'],
'effecting':['ih f-eh1-k t-ih-ng','vbg'],
'effective':['ih f-eh1-k t-ih-v','jj vbn nn'],
'effectively':['ih f-eh1-k t-ih-v l-iy','rb'],
'effectiveness':['ih f-eh1-k t-ih-v n-ah-s','nn'],
'effects':['ih f-eh1-k-t-s','nns vbz'],
'effete':['eh f-iy1-t','jj'],
'efficacious':['eh f-ah k-ey1 sh-ah-s','jj'],
'efficacy':['eh1 f-ih k-ae s-iy','nn'],
'efficiency':['ih f-ih1 sh-ah-n s-iy','nn'],
'efficient':['ih f-ih1 sh-ah-n-t','jj'],
'efficiently':['ih f-ih1 sh-ah-n-t l-iy','rb'],
'effigy':['eh1 f-ih jh-iy','nn'],
'effluent':['eh1 f-l-uw ah-n-t','nn jj'],
'effort':['eh1 f-er-t','nn'],
'effortless':['eh1 f-er-t l-ah-s','jj'],
'effortlessly':['eh1 f-er-t l-ah-s l-iy','rb'],
'effrontery':['ih f-r-ah1-n t-er iy','nn'],
'effusive':['eh1 f-y-uw s-ih-v','jj'],
'egalitarian':['ih g-ae l-ah t-eh1 r-iy ah-n','jj'],
'egalitarianism':['iy g-ae l-ah t-eh1 r-iy ah n-ih z-ah-m','nn'],
'egg':['eh1-g','nn'],
'egged':['eh1-g-d','vbn'],
'egghead':['eh1-g hh-eh-d','nn'],
'eggplants':['eh1-g p-l-ae-n-t-s','nns'],
'eggshell':['eh1-g sh-eh-l','jj'],
'ego':['iy1 g-ow','nn'],
'egocentric':['iy g-ow s-eh1-n t-r-ih-k','jj'],
'egotism':['iy1 g-ah t-ih z-ah-m','nn'],
'egotist':['iy1 g-ah t-ih-s-t','nn'],
'egotistical':['iy g-ah t-ih1 s-t-ih k-ah-l','jj'],
'egregious':['ih g-r-iy1 jh-ah-s','jj'],
'egregiously':['ih g-r-iy1 jh-ah-s l-iy','rb'],
'eh':['eh1','uh'],
'eighteenth':['ey t-iy1-n-th','jj'],
'eighth':['ey1-t-th','jj nn'],
'eighties':['ey1 t-iy-z','nns'],
'either':['iy1 dh-er','dt cc in rb rbr'],
'eject':['ih jh-eh1-k-t','vb'],
'ejected':['ih jh-eh1-k t-ih-d','vbn vbd'],
'ejection':['ih jh-eh1-k sh-ah-n','nn'],
'eke':['iy1-k','vb'],
'eked':['iy1-k-t','vbd vbn'],
'eking':['iy1 k-ih-ng','vbg'],
'elaborate':['ih l-ae1 b-r-ah-t','vb jj'],
'elaborated':['ih l-ae1 b-er ey t-ah-d','vbn'],
'elaborately':['ih l-ae1 b-r-ah-t l-iy','rb'],
'elaborates':['ih l-ae1 b-er ey-t-s','vbz'],
'elaborating':['ih l-ae1 b-er ey t-ih-ng','vbg'],
'elaboration':['ih l-ae b-er ey1 sh-ah-n','nn'],
'elan':['iy1 l-ah-n','nn'],
'elapse':['ih l-ae1-p-s','vb'],
'elapsed':['ih l-ae1-p-s-t','vbn'],
'elastic':['ih l-ae1 s-t-ih-k','jj nn'],
'elasticity':['iy l-ae s-t-ih1 s-ah t-iy','nn'],
'elastomers':['ih l-ae1 s-t-ah m-er-z','nns'],
'elated':['ih l-ey1 t-ah-d','jj vbn'],
'elation':['ih l-ey1 sh-ah-n','nn'],
'elbow':['eh1-l b-ow','nn'],
'elbowing':['eh1-l b-ow ih-ng','vbg'],
'elder':['eh1-l d-er','jjr jj nn'],
'elderly':['eh1-l d-er l-iy','jj nn'],
'eldest':['eh1-l d-ah-s-t','jjs'],
'elect':['ih l-eh1-k-t','vb vbp'],
'elected':['ih l-eh1-k t-ah-d','vbn jj vbd'],
'electing':['ih l-eh1-k t-ih-ng','vbg'],
'election':['ih l-eh1-k sh-ah-n','nn'],
'elective':['ih l-eh1-k t-ih-v','jj'],
'electoral':['ih l-eh1-k t-er ah-l','jj'],
'electorate':['ih l-eh1-k t-er ah-t','nn'],
'electors':['ih l-eh1-k t-er-z','nns'],
'electric':['ih l-eh1-k t-r-ih-k','jj nn'],
'electrical':['ih l-eh1-k t-r-ih k-ah-l','jj'],
'electrically':['ih l-eh1-k t-r-ih k-ah l-iy','rb'],
'electrician':['ih l-eh-k t-r-ih1 sh-ah-n','nn'],
'electricity':['ih l-eh-k t-r-ih1 s-ah t-iy','nn'],
'electrified':['ih l-eh1-k t-r-ah f-ay-d','vbn'],
'electrifying':['ih l-eh1-k t-r-ah f-ay ih-ng','jj'],
'electrocardiogram':['ih l-eh-k t-r-ow k-aa1-r d-iy ah g-r-ae-m','nn'],
'electrode':['ih l-eh1-k t-r-ow-d','nn'],
'electrodynamics':['ih l-eh-k t-r-ow d-ay n-ae1 m-ih-k-s','nns'],
'electrolysis':['ih l-eh-k t-r-aa1 l-ah s-ah-s','nn'],
'electrolytic':['ih l-eh-k t-r-ah l-ih1 t-ih-k','jj'],
'electromagnet':['ih l-eh-k t-r-ow m-ae1-g n-ah-t','nn'],
'electromagnetism':['ih l-eh-k t-r-ow m-ae1-g n-ah t-ih z-ah-m','nn'],
'electromechanical':['ih l-eh-k t-r-ow m-ah k-ae1 n-ih k-ah-l','jj'],
'electron':['ih l-eh1-k t-r-aa-n','nn'],
'electronic':['ih l-eh-k t-r-aa1 n-ih-k','jj'],
'electronically':['ih l-eh-k t-r-aa1 n-ih k-ah l-iy','rb'],
'electronics':['ih l-eh-k t-r-aa1 n-ih-k-s','nns nn'],
'electroplating':['ih l-eh1-k t-r-ah p-l-ey t-ih-ng','nn'],
'electroshock':['ih l-eh1-k t-r-ow sh-aa-k','nn'],
'electrostatic':['ih l-eh-k t-r-ow s-t-ae1 t-ih-k','jj'],
'elects':['ih l-eh1-k-t-s','vbz'],
'elegance':['eh1 l-ah g-ah-n-s','nn'],
'elegant':['eh1 l-ah g-ah-n-t','jj'],
'elegantly':['eh1 l-ih g-ah-n-t l-iy','rb'],
'element':['eh1 l-ah m-ah-n-t','nn'],
'elemental':['eh l-ah m-eh1-n t-ah-l','jj nn'],
'elementary':['eh l-ah m-eh1-n t-r-iy','jj'],
'elephant':['eh1 l-ah f-ah-n-t','nn'],
'elephantine':['eh l-ah f-ae1-n t-iy-n','jj'],
'elevate':['eh1 l-ah v-ey-t','vb vbp'],
'elevated':['eh1 l-ah v-ey t-ih-d','vbn vbd jj'],
'elevates':['eh1 l-ah v-ey-t-s','vbz'],
'elevating':['eh1 l-ah v-ey t-ih-ng','vbg'],
'elevation':['eh l-ah v-ey1 sh-ah-n','nn'],
'elevator':['eh1 l-ah v-ey t-er','nn'],
'eleventh':['ih l-eh1 v-ah-n-th','jj'],
'elfin':['eh1-l f-ih-n','jj'],
'elicit':['ih l-ih1 s-ih-t','vb'],
'elicited':['ih l-ih1 s-ih t-ih-d','vbn vbd'],
'elicits':['ih l-ih1 s-ah-t-s','vbz'],
'eligibility':['eh l-ih jh-ah b-ih1 l-ih t-iy','nn'],
'eligible':['eh1 l-ah jh-ah b-ah-l','jj nn'],
'eliminate':['ih l-ih1 m-ah n-ey-t','vb vbp'],
'eliminated':['ih l-ih1 m-ah n-ey t-ah-d','vbn vbd'],
'eliminates':['ih l-ih1 m-ah n-ey-t-s','vbz'],
'eliminating':['ih l-ih1 m-ah n-ey t-ih-ng','vbg'],
'elimination':['ih l-ih m-ah n-ey1 sh-ah-n','nn'],
'elite':['ih l-iy1-t','nn nns jj'],
'elitist':['ey l-iy1 t-ih-s-t','jj'],
'elixir':['ih l-ih1-k s-er','nn'],
'elk':['eh1-l-k','nns nn'],
'ell':['eh1-l','nn'],
'ellipsoids':['ih l-ih1-p s-oy-d-z','nns'],
'elliptical':['ih l-ih1-p t-ih k-ah-l','jj'],
'elm':['eh1-l-m','nn'],
'elongate':['ih l-ao1-ng g-ey-t','vb'],
'elongated':['ih l-ao1-ng g-ey t-ah-d','vbn jj'],
'elongation':['iy l-ao-ng g-ey1 sh-ah-n','nn'],
'eloquence':['eh1 l-ah k-w-ah-n-s','nn'],
'eloquent':['eh1 l-ah k-w-ah-n-t','jj'],
'eloquently':['eh1 l-ah k-w-ah-n-t l-iy','rb'],
'else':['eh1-l-s','rb jj nn'],
'elsewhere':['eh1-l s-w-eh-r','rb nn'],
'elucidated':['ih l-uw1 s-ah d-ey t-ah-d','vbn'],
'eluded':['ih l-uw1 d-ih-d','vbd vbn'],
'eludes':['ih l-uw1-d-z','vbz'],
'eluding':['ih l-uw1 d-ih-ng','vbg'],
'elusive':['ih l-uw1 s-ih-v','jj'],
'elusiveness':['ih l-uw1 s-ih-v n-ah-s','nn'],
'em':['eh1-m','prp'],
'emaciated':['ih m-ey1 sh-iy ey t-ih-d','vbn jj'],
'emanated':['eh1 m-ah n-ey t-ih-d','vbd vbn'],
'emanating':['eh1 m-ah n-ey t-ih-ng','vbg'],
'emanations':['eh m-ah n-ey1 sh-ah-n-z','nns'],
'emancipate':['ih m-ae1-n s-ah p-ey-t','vb'],
'emancipated':['ih m-ae1-n s-ah p-ey t-ih-d','vbn jj'],
'emancipation':['ih m-ae-n s-ah p-ey1 sh-ah-n','nn'],
'emasculate':['ah m-ae1-s k-y-uw l-ih-t','vb'],
'emasculated':['ah m-ae1-s k-y-uw l-ey t-ih-d','vbd'],
'embankment':['eh-m b-ae1-ng-k m-ah-n-t','nn'],
'embargo':['eh-m b-aa1-r g-ow','nn vb'],
'embargoed':['ih-m b-aa1-r g-ow-d','jj vbd vbn'],
'embargoes':['eh-m b-aa1-r g-ow-z','nns'],
'embark':['eh-m b-aa1-r-k','vb vbp'],
'embarked':['eh-m b-aa1-r-k-t','vbd vbn'],
'embarking':['eh-m b-aa1-r k-ih-ng','vbg'],
'embarks':['ih-m b-aa1-r-k-s','vbz'],
'embarrass':['ih-m b-eh1 r-ah-s','vb'],
'embarrassed':['ih-m b-eh1 r-ah-s-t','vbn vbd jj'],
'embarrassing':['ih-m b-eh1 r-ah s-ih-ng','jj vbg'],
'embarrassingly':['ih-m b-eh1 r-ah s-ih-ng l-iy','rb'],
'embarrassment':['ih-m b-eh1 r-ah-s m-ah-n-t','nn'],
'embassy':['eh1-m b-ah s-iy','nn'],
'embattled':['eh-m b-ae1 t-ah-l-d','jj vbn'],
'embedded':['eh-m b-eh1 d-ih-d','vbn jj'],
'embellish':['ih-m b-eh1 l-ih-sh','vb'],
'embellished':['eh-m b-eh1 l-ih-sh-t','vbn'],
'embezzle':['ih-m b-eh1 z-ah-l','vb'],
'embezzled':['ih-m b-eh1 z-ah-l-d','vbd'],
'embezzlement':['eh-m b-eh1 z-ah-l m-ah-n-t','nn'],
'embezzler':['ih-m b-eh1 z-ah-l er','nn'],
'embezzling':['ih-m b-eh1 z-ah-l ih-ng','vbg'],
'embittered':['eh-m b-ih1 t-er-d','vbn jj'],
'emblazoned':['eh-m b-l-ey1 z-ah-n-d','vbn'],
'emblem':['eh1-m b-l-ah-m','nn'],
'emblematic':['eh-m b-l-ah m-ae1 t-ih-k','jj'],
'embodied':['ih-m b-aa1 d-iy-d','vbn vbd'],
'embodies':['eh-m b-aa1 d-iy-z','vbz'],
'embodiment':['eh-m b-aa1 d-iy m-ah-n-t','nn'],
'embody':['ih-m b-aa1 d-iy','vbp'],
'embodying':['ih-m b-aa1 d-iy ih-ng','vbg'],
'emboldened':['eh-m b-ow1-l d-ah-n-d','vbn vbd'],
'embolisms':['eh1-m b-ow l-ih z-ah-m-z','nns'],
'embossed':['ih-m b-ao1-s-t','vbd vbn'],
'embrace':['eh-m b-r-ey1-s','vb vbp nn'],
'embraced':['eh-m b-r-ey1-s-t','vbn vbd'],
'embraces':['eh-m b-r-ey1 s-ih-z','vbz nns'],
'embracing':['eh-m b-r-ey1 s-ih-ng','vbg'],
'embroider':['ih-m b-r-oy1 d-er','vbp vb'],
'embroidered':['eh-m b-r-oy1 d-er-d','vbn'],
'embroidery':['eh-m b-r-oy1 d-er iy','nn'],
'embroiled':['eh-m b-r-oy1-l-d','vbn'],
'embryo':['eh1-m b-r-iy ow','nn'],
'embryonic':['eh-m b-r-iy aa1 n-ih-k','jj'],
'emcee':['eh1-m s-iy1','nn'],
'emerald':['eh1-m r-ah-l-d','jj nn'],
'emerge':['ih m-er1-jh','vb vbp'],
'emerged':['ih m-er1-jh-d','vbd vbn'],
'emergence':['ih m-er1 jh-ah-n-s','nn'],
'emergency':['ih m-er1 jh-ah-n s-iy','nn jj'],
'emergent':['ih m-er1 jh-ah-n-t','jj'],
'emerges':['ih m-er1 jh-ih-z','vbz'],
'emerging':['ih m-er1 jh-ih-ng','vbg'],
'emeritus':['ih m-eh1 r-ah t-ah-s','nn jj'],
'emigrants':['eh1 m-ah g-r-ah-n-t-s','nns'],
'emigrate':['eh1 m-ah g-r-ey-t','vb vbp'],
'emigrated':['eh1 m-ah g-r-ey t-ih-d','vbd vbn'],
'emigrating':['eh1 m-ah g-r-ey t-ih-ng','vbg'],
'emigration':['eh m-ah g-r-ey1 sh-ah-n','nn'],
'emigre':['eh1 m-ah g-r-ey','nn'],
'eminence':['eh1 m-ah n-ah-n-s','nn'],
'eminent':['eh1 m-ah n-ah-n-t','jj'],
'eminently':['eh1 m-ah n-ah-n-t l-iy','rb'],
'emissary':['eh1 m-ah s-eh r-iy','nn'],
'emission':['ih m-ih1 sh-ah-n','nn'],
'emit':['ih m-ih1-t','vb vbp'],
'emits':['ih m-ih1-t-s','vbz'],
'emitted':['ih m-ih1 t-ah-d','vbn vbd'],
'emitting':['ih m-ih1 t-ih-ng','vbg'],
'emotion':['ih m-ow1 sh-ah-n','nn'],
'emotional':['ih m-ow1 sh-ah n-ah-l','jj'],
'emotionalism':['ih m-ow1 sh-ah-n ah l-ih z-ah-m','nn'],
'emotionally':['ih m-ow1-sh n-ah l-iy','rb'],
'empathetic':['eh-m p-ah th-eh1 t-ih-k','jj'],
'empathize':['eh1-m p-ah th-ay-z','vb'],
'empathy':['eh1-m p-ah th-iy','nn'],
'emperor':['eh1-m p-er er','nn'],
'emphasis':['eh1-m f-ah s-ah-s','nn'],
'emphasize':['eh1-m f-ah s-ay-z','vb vbp'],
'emphasized':['eh1-m f-ah s-ay-z-d','vbd vbn'],
'emphasizes':['eh1-m f-ah s-ay z-ah-z','vbz'],
'emphasizing':['eh1-m f-ah s-ay z-ih-ng','vbg'],
'emphatic':['eh-m f-ae1 t-ih-k','jj'],
'emphatically':['eh-m f-ae1 t-ih-k l-iy','rb'],
'emphysema':['eh-m f-ah z-iy1 m-ah','nn'],
'empire':['eh1-m p-ay er','nn'],
'empirical':['eh-m p-ih1 r-ih k-ah-l','jj'],
'empirically':['eh-m p-ih1 r-ih k-ah l-iy','rb'],
'empiricism':['eh-m p-ih1 r-ah s-ih z-ah-m','nn'],
'employ':['eh-m p-l-oy1','vb nn vbp'],
'employed':['eh-m p-l-oy1-d','vbn vbd'],
'employee':['eh-m p-l-oy1 iy','nn'],
'employees':['eh-m p-l-oy1 iy-z','nns vbz'],
'employer':['eh-m p-l-oy1 er','nn'],
'employing':['eh-m p-l-oy1 ih-ng','vbg'],
'employment':['eh-m p-l-oy1 m-ah-n-t','nn'],
'employs':['eh-m p-l-oy1-z','vbz'],
'empower':['ih-m p-aw1 er','vbp'],
'empowered':['ih-m p-aw1 er-d','vbn vbd'],
'empowering':['ih-m p-aw1 er ih-ng','vbg'],
'empowerment':['ih-m p-aw1 er m-ah-n-t','nn'],
'empowers':['ih-m p-aw1 er-z','vbz'],
'emptied':['eh1-m-p t-iy-d','vbn vbd'],
'emptier':['eh1-m-p t-iy er','jjr rbr'],
'empties':['eh1-m-p t-iy-z','vbz'],
'emptiness':['eh1-m-p t-iy n-ah-s','nn'],
'empty':['eh1-m-p t-iy','jj nn rb vb vbp'],
'emptying':['eh1-m-p t-iy ih-ng','vbg'],
'emulate':['eh1 m-y-ah l-ey-t','vb'],
'emulated':['eh1 m-y-ah l-ey t-ih-d','vbn'],
'emulating':['eh1 m-y-ah l-ey t-ih-ng','vbg'],
'emulsion':['ih m-ah1-l sh-ah-n','nn'],
'enable':['eh n-ey1 b-ah-l','vb vbp'],
'enabled':['eh n-ey1 b-ah-l-d','vbd vbn'],
'enables':['eh n-ey1 b-ah-l-z','vbz'],
'enabling':['eh n-ey1 b-ah-l ih-ng','vbg'],
'enact':['ih n-ae1-k-t','vb'],
'enacted':['eh n-ae1-k t-ah-d','vbn vbd'],
'enacting':['eh n-ae1-k t-ih-ng','vbg'],
'enactment':['eh n-ae1-k-t m-ah-n-t','nn'],
'enacts':['ih n-ae1-k-t-s','vbz'],
'enamel':['ih n-ae1 m-ah-l','nn'],
'enamored':['eh n-ae1 m-er-d','jj'],
'encamped':['ih-n k-ae1-m-p-t','vbn'],
'encased':['eh-n k-ey1-s-t','vbd'],
'encephalitis':['eh-n s-eh f-ah l-ay1 t-ah-s','nn'],
'enchanted':['eh-n ch-ae1-n t-ih-d','vbn jj'],
'enchanting':['eh-n ch-ae1-n t-ih-ng','jj'],
'enchantment':['eh-n ch-ae1-n-t m-ah-n-t','nn'],
'encircle':['eh-n s-er1 k-ah-l','vb'],
'encircled':['ih-n s-er1 k-ah-l-d','vbd jj'],
'encircling':['eh-n s-er1 k-ah-l ih-ng','vbg'],
'enclave':['aa1-n k-l-ey-v','nn'],
'enclosed':['eh-n k-l-ow1-z-d','vbn vbd'],
'enclosing':['eh-n k-l-ow1 z-ih-ng','vbg'],
'enclosure':['eh-n k-l-ow1 zh-er','nn'],
'encoded':['eh-n k-ow1 d-ih-d','vbn'],
'encompass':['eh-n k-ah1-m p-ah-s','vb'],
'encompassed':['eh-n k-ah1-m p-ah-s-t','vbn vbd'],
'encompasses':['eh-n k-ah1-m p-ah s-ah-z','vbz'],
'encompassing':['eh-n k-ah1-m p-ah s-ih-ng','vbg'],
'encore':['aa1-n k-ao-r','nn'],
'encounter':['ih-n k-aw1-n t-er','nn vb vbp'],
'encountered':['ih-n k-aw1-n t-er-d','vbn vbd'],
'encountering':['eh-n k-aw1-n t-er ih-ng','vbg'],
'encounters':['ih-n k-aw1-n t-er-z','nns vbz'],
'encourage':['eh-n k-er1 ih-jh','vb vbp'],
'encouraged':['eh-n k-er1 ih-jh-d','vbn vbd jj'],
'encouragement':['eh-n k-er1 ih-jh m-ah-n-t','nn'],
'encourages':['eh-n k-er1 ih jh-ih-z','vbz'],
'encouraging':['eh-n k-er1 ih jh-ih-ng','vbg jj'],
'encroach':['ih-n k-r-ow1-ch','vb'],
'encroached':['ih-n k-r-ow1-ch-t','vbd'],
'encroaching':['ih-n k-r-ow1 ch-ih-ng','vbg'],
'encroachment':['eh-n k-r-ow1-ch m-ah-n-t','nn'],
'encrusted':['eh-n k-r-ah1 s-t-ih-d','vbn'],
'encumber':['eh-n k-ah1-m b-er','vb'],
'encumbered':['eh-n k-ah1-m b-er-d','vbn'],
'encyclopedia':['ih-n s-ay k-l-ah p-iy1 d-iy ah','nn'],
'encyclopedic':['ih-n s-ay k-l-ah p-iy1 d-ih-k','jj'],
'end':['eh1-n-d','nn vbp jj rb vb'],
'endanger':['eh-n d-ey1-n jh-er','vb vbp'],
'endangered':['eh-n d-ey1-n jh-er-d','vbn vbd jj'],
'endangering':['eh-n d-ey1-n jh-er ih-ng','vbg'],
'endangerment':['eh-n d-ey1-n jh-er m-ah-n-t','nn'],
'endeared':['eh-n d-iy1-r-d','vbd vbn'],
'endearing':['eh-n d-iy1 r-ih-ng','jj'],
'endearment':['ih-n d-ih1-r m-ah-n-t','nn'],
'endeavor':['ih-n d-eh1 v-er','nn vb'],
'endeavored':['ih-n d-eh1 v-er-d','vbd vbn'],
'endeavors':['ih-n d-eh1 v-er-z','nns'],
'endeavour':['ih-n d-eh1 v-er','nn'],
'ended':['eh1-n d-ah-d','vbd jj vb vbn'],
'endemic':['eh-n d-eh1 m-ih-k','jj'],
'ending':['eh1-n d-ih-ng','vbg jj nn vbn'],
'endless':['eh1-n-d l-ah-s','jj'],
'endlessly':['eh1-n-d l-ah-s l-iy','rb'],
'endocrine':['eh1-n d-ow k-r-ay-n','jj'],
'endocrinologists':['eh-n d-ow k-r-ah n-aa1 l-ah jh-ah-s-t-s','nns'],
'endometriosis':['eh-n d-ow m-eh t-r-iy ow1 s-ih-s','nn'],
'endorse':['eh-n d-ao1-r-s','vb vbp'],
'endorsed':['eh-n d-ao1-r-s-t','vbn vbd'],
'endorsement':['eh-n d-ao1-r-s m-ah-n-t','nn'],
'endorsers':['ih-n d-ao1-r s-er-z','nns'],
'endorses':['eh-n d-ao1-r s-ih-z','vbz'],
'endorsing':['eh-n d-ao1-r s-ih-ng','vbg nn'],
'endosperm':['eh1-n d-ah s-p-er-m','nn'],
'endothermic':['eh-n d-ow th-er1 m-ih-k','jj'],
'endow':['eh-n d-aw1','vb vbp'],
'endowed':['eh-n d-aw1-d','vbn vbd jj'],
'endowment':['eh-n d-aw1 m-ah-n-t','nn'],
'ends':['eh1-n-d-z','nns vbz'],
'endurance':['eh1-n d-er ah-n-s','nn'],
'endure':['eh-n d-y-uh1-r','vb vbp'],
'endured':['eh-n d-y-uh1-r-d','vbd vbn'],
'endures':['eh-n d-y-uh1-r-z','vbz'],
'enduring':['eh-n d-y-uh1 r-ih-ng','vbg jj'],
'enemy':['eh1 n-ah m-iy','nn'],
'energetic':['eh n-er jh-eh1 t-ih-k','jj'],
'energetically':['eh n-er jh-eh1 t-ih-k l-iy','rb'],
'energize':['eh1 n-er jh-ay-z','vb'],
'energized':['eh1 n-er jh-ay-z-d','vbn'],
'energizing':['eh1 n-er jh-ay z-ih-ng','vbg'],
'energy':['eh1 n-er jh-iy','nn'],
'enforce':['eh-n f-ao1-r-s','vb vbp'],
'enforceable':['eh-n f-ao1-r s-ah b-ah-l','jj'],
'enforced':['eh-n f-ao1-r-s-t','vbn vbd jj'],
'enforcement':['eh-n f-ao1-r-s m-ah-n-t','nn'],
'enforcer':['eh-n f-ao1-r s-er','nn'],
'enforces':['eh-n f-ao1-r s-ih-z','vbz'],
'enforcing':['eh-n f-ao1-r s-ih-ng','vbg'],
'engage':['eh-n g-ey1-jh','vb vbp'],
'engaged':['eh-n g-ey1-jh-d','vbn jj vbd'],
'engagement':['eh-n g-ey1-jh m-ah-n-t','nn'],
'engages':['eh-n g-ey1 jh-ih-z','nns vbn vbz'],
'engaging':['eh-n g-ey1 jh-ih-ng','vbg jj'],
'engender':['eh-n jh-eh1-n d-er','vb'],
'engendered':['eh-n jh-eh1-n d-er-d','vbn vbd'],
'engenders':['eh1-ng g-ah-n d-er-z','vbz'],
'engine':['eh1-n jh-ah-n','nn'],
'engineer':['eh1-n jh-ah n-ih1-r','nn vb'],
'engineered':['eh-n jh-ah n-iy1-r-d','vbn vbd jj'],
'engineering':['eh1-n jh-ah n-ih1 r-ih-ng','nn vbg'],
'engineers':['eh1-n jh-ah n-ih1-r-z','nns'],
'engrave':['ih-n g-r-ey1-v','vb'],
'engraved':['ih-n g-r-ey1-v-d','vbn jj'],
'engraver':['ih-n g-r-ey1 v-er','nn'],
'engraving':['ih-n g-r-ey1 v-ih-ng','nn'],
'engrossed':['ih-n g-r-ow1-s-t','jj vbn'],
'engrossing':['ih-n g-r-ow1 s-ih-ng','jj'],
'engulf':['ih-n g-ah1-l-f','vb'],
'engulfed':['ih-n g-ah1-l-f-t','vbn vbd'],
'engulfing':['ih-n g-ah1-l f-ih-ng','vbg jj'],
'enhance':['eh-n hh-ae1-n-s','vb vbp'],
'enhanced':['eh-n hh-ae1-n-s-t','vbn vbd jj'],
'enhancement':['eh-n hh-ae1-n-s m-ah-n-t','nn'],
'enhances':['eh-n hh-ae1-n s-ih-z','vbz'],
'enhancing':['eh-n hh-ae1-n s-ih-ng','vbg'],
'enigma':['ih n-ih1-g m-ah','nn'],
'enigmatic':['eh n-ih-g m-ae1 t-ih-k','jj'],
'enjoin':['eh-n jh-oy1-n','vb'],
'enjoined':['eh-n jh-oy1-n-d','vbn vbd'],
'enjoining':['eh-n jh-oy1 n-ih-ng','vbg'],
'enjoy':['eh-n jh-oy1','vb vbp'],
'enjoyable':['eh-n jh-oy1 ah b-ah-l','jj'],
'enjoyed':['eh-n jh-oy1-d','vbd vbn'],
'enjoying':['eh-n jh-oy1 ih-ng','vbg'],
'enjoyment':['eh-n jh-oy1 m-ah-n-t','nn'],
'enjoys':['eh-n jh-oy1-z','vbz'],
'enlarge':['eh-n l-aa1-r-jh','vb'],
'enlarged':['eh-n l-aa1-r-jh-d','vbn jj vbd'],
'enlargement':['ih-n l-aa1-r-jh m-ah-n-t','nn'],
'enlarges':['ih-n l-aa1-r jh-ih-z','vbz'],
'enlarging':['ih-n l-aa1-r jh-ih-ng','vbg nn'],
'enlighten':['eh-n l-ay1 t-ah-n','vb'],
'enlightened':['eh-n l-ay1 t-ah-n-d','jj vbn'],
'enlightening':['eh-n l-ay1 t-ah-n ih-ng','vbg jj'],
'enlightenment':['eh-n l-ay1 t-ah-n m-ah-n-t','nn'],
'enlist':['eh-n l-ih1-s-t','vb'],
'enlisted':['eh-n l-ih1 s-t-ih-d','vbd jj vbn'],
'enlisting':['eh-n l-ih1 s-t-ih-ng','vbg'],
'enlistment':['eh-n l-ih1-s-t m-ah-n-t','nn'],
'enlists':['eh-n l-ih1-s-t-s','vbz'],
'enliven':['eh-n l-ay1 v-ah-n','vbp'],
'enlivened':['eh-n l-ay1 v-ah-n-d','vbn'],
'enmeshed':['eh-n m-eh1-sh-t','vbn'],
'enmity':['eh1-n m-ah t-iy','nn'],
'ennui':['eh n-uw1 iy','nn'],
'enormity':['ih n-ao1-r m-ah t-iy','nn'],
'enormous':['ih n-ao1-r m-ah-s','jj'],
'enormously':['ih n-ao1-r m-ah-s l-iy','rb'],
'enough':['ih-n ah1-f','rb jj nn'],
'enrage':['eh-n r-ey1-jh','nn'],
'enraged':['eh-n r-ey1-jh-d','jj vbd vbn'],
'enraptured':['eh-n r-ae1-p ch-er-d','vbd vbn'],
'enrich':['eh-n r-ih1-ch','vb vbp'],
'enriched':['eh-n r-ih1-ch-t','vbn'],
'enriches':['eh-n r-ih1 ch-ih-z','vbz'],
'enriching':['eh-n r-ih1 ch-ih-ng','vbg'],
'enrichment':['eh-n r-ih1-ch m-ah-n-t','nn'],
'enroll':['eh-n r-ow1-l','vb vbp'],
'enrolled':['eh-n r-ow1-l-d','vbn vbd jj'],
'enrollees':['eh-n r-ow1 l-iy1-z','nns'],
'enrolling':['eh-n r-ow1 l-ih-ng','nn'],
'enrollment':['eh-n r-ow1-l m-ah-n-t','nn'],
'enroute':['eh-n r-uw1-t','rb'],
'ensconced':['ih-n s-k-aa1-n-s-t','vbn vbd'],
'ensemble':['aa-n s-aa1-m b-ah-l','nn'],
'enshrined':['eh-n sh-r-ay1-n-d','vbn'],
'enslaved':['eh-n s-l-ey1-v-d','vbn'],
'enslavement':['eh-n s-l-ey1-v m-ah-n-t','nn'],
'ensnare':['ih-n s-n-eh1-r','vb'],
'ensnared':['ih-n s-n-eh1-r-d','vbn'],
'ensnarled':['ih-n s-n-aa1-r-l-d','vbn'],
'ensue':['ih-n s-uw1','vb vbp'],
'ensued':['ih-n s-uw1-d','vbd vbn'],
'ensues':['ih-n s-uw1-z','vbz'],
'ensuing':['eh1-n s-uw ih-ng','vbg'],
'ensure':['eh-n sh-uh1-r','vb vbp'],
'ensured':['eh-n sh-uh1-r-d','vbd'],
'ensures':['eh-n sh-uh1-r-z','vbz'],
'ensuring':['eh-n sh-uh1 r-ih-ng','vbg'],
'entail':['eh-n t-ey1-l','vb vbp'],
'entailed':['ih-n t-ey1-l-d','vbd vbn'],
'entailing':['ih-n t-ey1 l-ih-ng','vbg'],
'entails':['ih-n t-ey1-l-z','vbz'],
'entangled':['eh-n t-ae1-ng g-ah-l-d','jj vbd vbn'],
'entanglement':['eh-n t-ae1-ng g-ah-l m-ah-n-t','nn'],
'enter':['eh1-n t-er','vb vbn vbp'],
'entered':['eh1-n t-er-d','vbd vbn'],
'entering':['eh1-n t-er ih-ng','vbg'],
'enterprise':['eh1-n t-er p-r-ay-z','nn'],
'enterprises':['eh1-n t-er p-r-ay z-ih-z','nns'],
'enterprising':['eh1-n t-er p-r-ay z-ih-ng','jj'],
'enters':['eh1-n t-er-z','vbz'],
'entertain':['eh-n t-er t-ey1-n','vb'],
'entertained':['eh-n t-er t-ey1-n-d','vbn vbd'],
'entertainer':['eh-n t-er t-ey1 n-er','nn'],
'entertaining':['eh-n t-er t-ey1 n-ih-ng','vbg jj nn'],
'entertainment':['eh-n t-er t-ey1-n m-ah-n-t','nn'],
'entertains':['eh-n t-er t-ey1-n-z','vbz'],
'enthralled':['eh-n th-r-ao1-l-d','jj vbn'],
'enthusiasm':['ih-n th-uw1 z-iy ae z-ah-m','nn'],
'enthusiast':['eh-n th-uw1 z-iy ae-s-t','nn'],
'enthusiastic':['ih-n th-uw z-iy ae1 s-t-ih-k','jj'],
'enthusiastically':['ih-n th-uw z-iy ae1 s-t-ih-k l-iy','rb'],
'entice':['ih-n t-ay1-s','vb vbp'],
'enticed':['ih-n t-ay1-s-t','vbd'],
'enticements':['ih-n t-ay1-s m-ah-n-t-s','nns'],
'enticing':['eh-n t-ay1 s-ih-ng','vbg jj'],
'entire':['ih-n t-ay1 er','jj nn'],
'entirely':['ih-n t-ay1 er l-iy','rb'],
'entirety':['ih-n t-ay1 er t-iy','nn'],
'entitle':['eh-n t-ay1 t-ah-l','vb vbp'],
'entitled':['eh-n t-ay1 t-ah-l-d','vbn vbd'],
'entitlement':['eh-n t-ay1 t-ah-l m-ah-n-t','nn jj'],
'entitles':['eh-n t-ay1 t-ah-l-z','vbz'],
'entitling':['eh-n t-ay1 t-ah-l ih-ng','vbg'],
'entity':['eh1-n t-ah t-iy','nn'],
'entombed':['eh-n t-uw1-m-d','vbn'],
'entomologist':['eh-n t-ah m-aa1 l-ah jh-ah-s-t','nn'],
'entourage':['aa-n t-uh r-aa1-zh','nn'],
'entrance':['eh1-n t-r-ah-n-s','nn'],
'entranced':['ih-n t-r-ae1-n-s-t','vbn'],
'entrant':['eh1-n t-r-ah-n-t','nn'],
'entrapment':['ih-n t-r-ae1-p m-ah-n-t','nn'],
'entre':['aa1-n t-r-ey','nn'],
'entreaties':['eh-n t-r-iy1 t-iy-z','nns'],
'entrench':['eh-n t-r-eh1-n-ch','vb'],
'entrenched':['eh-n t-r-eh1-n-ch-t','vbn vbd jj'],
'entrenchment':['eh-n t-r-eh1-n-ch m-ah-n-t','nn'],
'entrepreneur':['aa-n t-r-ah p-r-ah n-er1','nn'],
'entrepreneurial':['aa-n t-r-ah p-r-ah n-er1 iy ah-l','jj'],
'entrepreneurship':['aa-n t-r-ah p-r-ah n-er1 sh-ih-p','nn'],
'entropy':['eh1-n t-r-ah p-iy','nn'],
'entrust':['eh-n t-r-ah1-s-t','vb vbp'],
'entrusted':['eh-n t-r-ah1 s-t-ih-d','vbn'],
'entrusting':['eh-n t-r-ah1 s-t-ih-ng','vbg'],
'entry':['eh1-n t-r-iy','nn'],
'entwined':['eh-n t-w-ay1-n-d','vbn jj'],
'enumerated':['ih n-uw1 m-er ey t-ih-d','vbn jj'],
'enumeration':['ih n-uw m-er ey1 sh-ah-n','nn'],
'enunciated':['ih n-ah1-n s-iy ey t-ih-d','vbd'],
'envelope':['eh1-n v-ah l-ow-p','nn'],
'enveloped':['eh-n v-eh1 l-ah-p-t','vbn'],
'enveloping':['ih-n v-eh1 l-ah p-ih-ng','vbg'],
'enviable':['eh1-n v-iy ah b-ah-l','jj'],
'envied':['eh1-n v-iy-d','vbd'],
'envious':['eh1-n v-iy ah-s','jj'],
'enviously':['eh1-n v-iy ah-s l-iy','rb'],
'environment':['ih-n v-ay1 r-ah-n m-ah-n-t','nn'],
'environmental':['ih-n v-ay r-ah-n m-eh1-n t-ah-l','jj'],
'environmentalism':['eh-n v-ay1 r-ah-n m-eh-n t-ah l-ih z-ah-m','nn'],
'environmentalist':['ih-n v-ay r-ah-n m-eh1-n t-ah l-ih-s-t','nn jj'],
'environmentally':['ih-n v-ay r-ah-n m-eh1-n t-ah l-iy','rb'],
'environs':['ih-n v-ay1 r-ah-n-z','nns'],
'envisage':['eh-n v-ih1 z-ih-jh','vb'],
'envisaged':['eh-n v-ih1 z-ih-jh-d','vbd vbn'],
'envisages':['eh-n v-ih1 z-ih jh-ih-z','vbz'],
'envision':['eh-n v-ih1 zh-ah-n','vbp vb'],
'envisioned':['eh-n v-ih1 zh-ah-n-d','vbd vbn'],
'envisioning':['eh-n v-ih1 zh-ah-n ih-ng','vbg'],
'envisions':['eh-n v-ih1 zh-ah-n-z','vbz'],
'envoy':['eh1-n v-oy','nn'],
'envy':['eh1-n v-iy','nn vbp vb'],
'enzymatic':['eh-n z-ay m-ae1 t-ih-k','jj'],
'enzyme':['eh1-n z-ay-m','nn'],
'eons':['iy1 ah-n-z','nns'],
'ephemeral':['ih f-eh1 m-er ah-l','jj'],
'epic':['eh1 p-ih-k','nn jj'],
'epicenter':['eh1 p-ah s-eh-n t-er','nn'],
'epicure':['eh1 p-ih k-y-uh-r','nn'],
'epicurean':['eh p-ah k-y-uh r-iy1 ah-n','nn'],
'epidemic':['eh p-ah d-eh1 m-ih-k','nn jj'],
'epidemiological':['eh p-ah d-iy m-iy ah l-aa1 jh-ih k-ah-l','jj'],
'epidemiologist':['eh p-ah d-iy m-iy aa1 l-ah jh-ih-s-t','nn'],
'epidermis':['eh p-ah d-er1 m-ah-s','nn'],
'epigenetic':['eh p-ah jh-ah n-eh1 t-ih-k','jj'],
'epilepsy':['eh1 p-ah l-eh-p s-iy','nn'],
'epileptic':['eh p-ah l-eh1-p t-ih-k','jj'],
'epilogue':['eh1 p-ah l-ao-g','nn'],
'epiphany':['ih p-ih1 f-ah n-iy','nn'],
'episode':['eh1 p-ah s-ow-d','nn'],
'episodic':['eh p-ah s-aa1 d-ih-k','jj'],
'epistemology':['ih p-ih s-t-ah m-aa1 l-ah jh-iy','nn'],
'epitaph':['eh1 p-ah t-ae-f','nn'],
'epithet':['eh1 p-ah th-eh-t','nn'],
'epitome':['ih p-ih1 t-ah m-iy','nn'],
'epitomize':['ih p-ih1 t-ah m-ay-z','vb vbp'],
'epitomized':['ih p-ih1 t-ah m-ay-z-d','vbn vbd'],
'epitomizes':['ih p-ih1 t-ah m-ay z-ih-z','vbz'],
'epoch':['eh1 p-ah-k','nn'],
'epochal':['eh1 p-ah k-ah-l','jj'],
'epoxy':['ih p-aa1-k s-iy','jj'],
'epsom':['eh1-p s-ah-m','nn'],
'equal':['iy1 k-w-ah-l','jj nn vb vbp'],
'equaled':['iy1 k-w-ah-l-d','vbd vbn'],
'equaling':['iy1 k-w-ah-l ih-ng','vbg'],
'equality':['ih k-w-aa1 l-ah t-iy','nn'],
'equalization':['iy k-w-ah l-ih z-ey1 sh-ah-n','nn'],
'equalize':['iy1 k-w-ah l-ay-z','vb'],
'equalizer':['iy1 k-w-ah l-ay z-er','nn'],
'equalizing':['iy1 k-w-ah l-ay z-ih-ng','nn'],
'equally':['iy1 k-w-ah l-iy','rb'],
'equals':['iy1 k-w-ah-l-z','vbz nns'],
'equanimity':['iy k-w-ah n-ih1 m-ih t-iy','nn'],
'equate':['ih k-w-ey1-t','vb vbp'],
'equated':['ih k-w-ey1 t-ih-d','vbn vbd'],
'equates':['ih k-w-ey1-t-s','vbz'],
'equating':['ih k-w-ey1 t-ih-ng','vbg'],
'equation':['ih k-w-ey1 zh-ah-n','nn'],
'equator':['ih k-w-ey1 t-er','nn'],
'equatorial':['iy k-w-ah t-ao1 r-iy ah-l','jj'],
'equestrian':['ih k-w-eh1-s t-r-iy ah-n','jj'],
'equilibrium':['iy k-w-ah l-ih1 b-r-iy ah-m','nn'],
'equine':['iy1 k-w-ay-n','nn'],
'equip':['ih k-w-ih1-p','vb vbp'],
'equipment':['ih k-w-ih1-p m-ah-n-t','nn'],
'equipped':['ih k-w-ih1-p-t','vbn jj vbd'],
'equipping':['ih k-w-ih1 p-ih-ng','vbg nn'],
'equips':['ih k-w-ih1-p-s','vbz'],
'equitable':['eh1 k-w-ah t-ah b-ah-l','jj'],
'equitably':['eh1 k-w-ih t-ah b-l-iy','rb'],
'equity':['eh1 k-w-ah t-iy','nn'],
'equivalence':['ih k-w-ih1 v-ah l-ah-n-s','nn'],
'equivalent':['ih k-w-ih1 v-ah l-ah-n-t','nn jj'],
'equivocal':['ih k-w-ih1 v-ah k-ah-l','jj'],
'era':['eh1 r-ah','nn'],
'eradicate':['ih r-ae1 d-ah k-ey-t','vb'],
'eradicated':['ih r-ae1 d-ah k-ey t-ih-d','vbn vbd'],
'eradicating':['ih r-ae1 d-ah k-ey t-ih-ng','vbg'],
'eradication':['ih r-ae d-ah k-ey1 sh-ah-n','nn'],
'erasable':['ih r-ey1 s-ah b-ah-l','jj'],
'erase':['ih r-ey1-s','vb'],
'erased':['ih r-ey1-s-t','vbn vbd'],
'eraser':['ih r-ey1 s-er','nn'],
'erases':['ih r-ey1 s-ih-z','vbz'],
'erasing':['ih r-ey1 s-ih-ng','vbg'],
'erect':['ih r-eh1-k-t','vb vbp jj'],
'erected':['ih r-eh1-k t-ah-d','vbn vbd'],
'erecting':['ih r-eh1-k t-ih-ng','vbg'],
'erection':['ih r-eh1-k sh-ah-n','nn'],
'erects':['ih r-eh1-k-t-s','vbz'],
'ergonomics':['er g-ah n-aa1 m-ih-k-s','nns'],
'erode':['ih r-ow1-d','vb vbp'],
'eroded':['ih r-ow1 d-ah-d','vbn jj vbd'],
'erodes':['ih r-ow1-d-z','vbz'],
'eroding':['ih r-ow1 d-ih-ng','vbg'],
'erosion':['ih r-ow1 zh-ah-n','nn'],
'erotic':['ih r-aa1 t-ih-k','jj'],
'errand':['eh1 r-ah-n-d','nn'],
'errant':['eh1 r-ah-n-t','jj'],
'erratic':['ih r-ae1 t-ih-k','jj'],
'erratically':['eh r-ae1 t-ih-k l-iy','rb'],
'erred':['eh1-r-d','vbn vbd'],
'erroneous':['eh r-ow1 n-iy ah-s','jj'],
'erroneously':['eh r-ow1 n-iy ah-s l-iy','rb'],
'error':['eh1 r-er','nn'],
'ersatz':['eh1-r s-aa-t-s','jj'],
'erstwhile':['er1-s-t w-ay-l','jj'],
'erudite':['eh1 r-ah d-ay-t','jj'],
'erudition':['eh r-ah d-ih1 sh-ah-n','nn'],
'erupt':['ih r-ah1-p-t','vb vbp'],
'erupted':['ih r-ah1-p t-ah-d','vbd vbn'],
'erupting':['ih r-ah1-p t-ih-ng','vbg'],
'eruption':['ih r-ah1-p sh-ah-n','nn'],
'erupts':['ih r-ah1-p-t-s','vbz'],
'erythropoietin':['eh r-ih th-r-ah p-oy1 t-ih-n','nn'],
'escalate':['eh1 s-k-ah l-ey-t','vb vbp'],
'escalated':['eh1 s-k-ah l-ey t-ih-d','vbn vbd'],
'escalating':['eh1 s-k-ah l-ey t-ih-ng','vbg'],
'escalation':['eh s-k-ah l-ey1 sh-ah-n','nn'],
'escalators':['eh1 s-k-ah l-ey t-er-z','nns'],
'escapade':['eh1 s-k-ah p-ey-d','nn'],
'escape':['ih s-k-ey1-p','vb nn vbp'],
'escaped':['ih s-k-ey1-p-t','vbd vbn'],
'escapees':['ih s-k-ey p-iy1-z','nns'],
'escapes':['ih s-k-ey1-p-s','nns vbz'],
'escaping':['ih s-k-ey1 p-ih-ng','vbg'],
'eschew':['eh-s ch-uw1','vb'],
'eschewed':['eh-sh uw1-d','vbn'],
'eschewing':['eh-sh uw1 ih-ng','vbg'],
'eschews':['eh-s ch-uw1-z','vbz'],
'escort':['eh s-k-ao1-r-t','nn vb'],
'escorted':['eh s-k-ao1-r t-ih-d','vbd vbn'],
'escorting':['eh1 s-k-ao-r t-ih-ng','vbg'],
'escorts':['eh1 s-k-ao-r-t-s','vbz nns'],
'escrow':['eh s-k-r-ow1','nn jj'],
'escrowed':['eh1 s-k-r-ow-d','vbn'],
'escudo':['eh s-k-uw1 d-ow','nn'],
'esophagus':['ih s-aa1 f-ah g-ah-s','nn'],
'esoteric':['eh s-ah t-eh1 r-ih-k','jj'],
'especially':['ah s-p-eh1-sh l-iy','rb'],
'espionage':['eh1 s-p-iy ah n-aa-jh','nn'],
'esplanade':['eh-s p-l-ah n-aa1-d','nn'],
'espouse':['ih s-p-aw1-z','vbp'],
'espoused':['ih s-p-aw1-z-d','vbd vbn'],
'espouses':['ih s-p-aw1 z-ih-z','vbz'],
'espousing':['ih s-p-aw1 z-ih-ng','vbg'],
'espresso':['eh s-p-r-eh1 s-ow','nn'],
'essay':['eh s-ey1','nn'],
'essayist':['eh1 s-ey ih-s-t','nn'],
'essence':['eh1 s-ah-n-s','nn'],
'essential':['ih s-eh1-n sh-ah-l','jj nn'],
'essentially':['ih s-eh1-n sh-ah l-iy','rb'],
'establish':['ih s-t-ae1 b-l-ih-sh','vb vbp'],
'established':['ih s-t-ae1 b-l-ih-sh-t','vbn jj vbd'],
'establishes':['ih s-t-ae1 b-l-ih sh-ih-z','vbz'],
'establishing':['ih s-t-ae1 b-l-ih sh-ih-ng','vbg'],
'establishment':['ih s-t-ae1 b-l-ih-sh m-ah-n-t','nn'],
'estate':['ih s-t-ey1-t','nn'],
'esteem':['ah s-t-iy1-m','nn'],
'esteemed':['ih s-t-iy1-m-d','vbd vbn'],
'esters':['eh1 s-t-er-z','nns'],
'estimable':['eh1 s-t-ah m-ah b-ah-l','jj'],
'estimate':['eh1 s-t-ah m-ah-t','nn vb vbp'],
'estimated':['eh1 s-t-ah m-ey t-ah-d','vbn jj vbd'],
'estimates':['eh1 s-t-ah m-ah-t-s','nns vbz'],
'estimating':['eh1 s-t-ah m-ey t-ih-ng','vbg nn'],
'estimation':['eh s-t-ah m-ey1 sh-ah-n','nn'],
'estimators':['eh s-t-ih1 m-ah t-er-z','nns'],
'estranged':['eh s-t-r-ey1-n-jh-d','vbn vbd jj'],
'estrangement':['ih s-t-r-ey1-n-jh m-ah-n-t','nn'],
'estrogen':['eh1 s-t-r-ah jh-ah-n','nn'],
'estuary':['eh1-s ch-uw eh r-iy','nn'],
'etcetera':['eh1-t s-eh1 t-er ah','nn'],
'etch':['eh1-ch','vb'],
'etched':['eh1-ch-t','vbd vbn'],
'etching':['eh1 ch-ih-ng','nn'],
'eternal':['ih t-er1 n-ah-l','jj'],
'eternally':['ih t-er1 n-ah l-iy','rb'],
'eternity':['ih t-er1 n-ah t-iy','nn'],
'ethane':['eh1 th-ey-n','nn'],
'ethanol':['eh1 th-ah n-ao-l','nn'],
'ether':['iy1 th-er','nn cc'],
'ethereal':['ih th-ih1 r-iy ah-l','jj'],
'ethic':['eh1 th-ih-k','nn'],
'ethical':['eh1 th-ah k-ah-l','jj'],
'ethically':['eh1 th-ih k-ah l-iy','rb'],
'ethicist':['eh1 th-ih s-ih-s-t','nn'],
'ethics':['eh1 th-ih-k-s','nns nn'],
'ethnic':['eh1-th n-ih-k','jj'],
'ethnically':['eh1-th n-ih-k l-iy','rb'],
'ethnicity':['eh-th n-ih1 s-ih t-iy','nn'],
'ethos':['iy1 th-aa-s','nn'],
'ethyl':['eh1 th-ah-l','nn'],
'ethylene':['eh1 th-ah l-iy-n','nn'],
'etiquette':['eh1 t-ah k-ah-t','nn'],
'etymology':['eh t-ah m-aa1 l-ah jh-iy','nn'],
'eucalyptus':['y-uw k-ah l-ih1-p t-ah-s','nn'],
'eulogy':['y-uw1 l-ah jh-iy','nn'],
'euphemism':['y-uw1 f-ah m-ih z-ah-m','nn'],
'euphoria':['y-uw f-ao1 r-iy ah','nn'],
'euphoric':['y-uw f-aa1 r-ih-k','jj'],
'eva':['ey1 v-ah','nn'],
'evacuate':['ih v-ae1 k-y-ah ey-t','vb'],
'evacuated':['ih v-ae1 k-y-ah-w ey t-ih-d','vbn'],
'evacuation':['ih v-ae k-y-ah-w ey1 sh-ah-n','nn'],
'evade':['ih v-ey1-d','vb'],
'evaded':['ih v-ey1 d-ih-d','vbn vbd'],
'evaders':['ih v-ey1 d-er-z','nns'],
'evades':['ih v-ey1-d-z','vbz'],
'evading':['ih v-ey1 d-ih-ng','vbg'],
'evaluate':['ih v-ae1-l y-uw ey-t','vb vbp'],
'evaluated':['ih v-ae1-l y-uw ey t-ah-d','vbn vbd'],
'evaluates':['ih v-ae1-l y-uw ey-t-s','vbz'],
'evaluating':['ih v-ae1-l y-uw ey t-ih-ng','vbg'],
'evaluation':['ih v-ae-l y-uw ey1 sh-ah-n','nn'],
'evangelical':['iy v-ae-n jh-eh1 l-ih k-ah-l','jj'],
'evangelism':['ih v-ae1-n jh-ah l-ih z-ah-m','nn'],
'evangelist':['ih v-ae1-n jh-ah l-ah-s-t','nn'],
'evaporate':['ih v-ae1 p-er ey-t','vb vbp'],
'evaporated':['ih v-ae1 p-er ey t-ah-d','vbd vbn'],
'evaporates':['ih v-ae1 p-er ey-t-s','vbz'],
'evaporation':['ih v-ae p-er ey1 sh-ah-n','nn'],
'evasion':['ih v-ey1 zh-ah-n','nn'],
'evasive':['ih v-ey1 s-ih-v','jj'],
'eve':['iy1-v','nn'],
'even':['iy1 v-ih-n','rb jj vb'],
'evened':['iy1 v-ah-n-d','vbn'],
'evenhanded':['iy1 v-ah-n hh-ae1-n d-ih-d','jj'],
'evening':['iy1-v n-ih-ng','nn vbg'],
'evenly':['iy1 v-ah-n l-iy','rb'],
'evens':['iy1 v-ah-n-z','vbz'],
'event':['ih v-eh1-n-t','nn'],
'eventful':['ih v-eh1-n-t f-ah-l','jj'],
'eventual':['ah v-eh1-n ch-uw ah-l','jj'],
'eventuality':['ah v-eh-n ch-uw ae1 l-ih t-iy','nn'],
'eventually':['ih v-eh1-n ch-ah-w ah l-iy','rb'],
'ever':['eh1 v-er','rb rbr rp'],
'evergreen':['eh1 v-er g-r-iy-n','nn'],
'everlasting':['eh v-er l-ae1 s-t-ih-ng','jj nn'],
'every':['eh1 v-er iy','dt'],
'everybody':['eh1 v-r-iy b-aa d-iy','nn'],
'everyday':['eh1 v-r-iy d-ey1','jj'],
'everyone':['eh1 v-r-iy w-ah-n','nn'],
'everything':['eh1 v-r-iy th-ih-ng','nn'],
'everywhere':['eh1 v-r-iy w-eh-r','rb'],
'evict':['ih v-ih1-k-t','vb'],
'evicted':['ih v-ih1-k t-ih-d','vbn'],
'evicting':['ih v-ih1-k t-ih-ng','vbg'],
'evidence':['eh1 v-ah d-ah-n-s','nn vbp'],
'evidenced':['eh1 v-ih d-ah-n-s-t','vbn'],
'evident':['eh1 v-ah d-ah-n-t','jj'],
'evidently':['eh1 v-ah d-ah-n-t l-iy','rb'],
'evil':['iy1 v-ah-l','jj nn'],
'evildoers':['iy1 v-ah-l d-uw er-z','nns'],
'evinced':['ih v-ih1-n-s-t','vbn vbd'],
'eviscerate':['ah v-ih1 s-er ey-t','vb'],
'evocation':['iy v-ow k-ey1 sh-ah-n','nn'],
'evocative':['ih v-aa1 k-ah t-ih-v','jj'],
'evoke':['ih v-ow1-k','vb vbp'],
'evoked':['ih v-ow1-k-t','vbn vbd'],
'evokes':['ih v-ow1-k-s','vbz'],
'evoking':['ih v-ow1 k-ih-ng','vbg'],
'evolution':['eh v-ah l-uw1 sh-ah-n','nn'],
'evolutionary':['eh v-ah l-uw1 sh-ah-n eh r-iy','jj'],
'evolve':['ih v-aa1-l-v','vb vbp'],
'evolved':['ih v-aa1-l-v-d','vbn jj vbd'],
'evolves':['ih v-aa1-l-v-z','vbz'],
'evolving':['ih v-aa1-l v-ih-ng','vbg'],
'exacerbate':['ih-g z-ae1 s-er b-ey-t','vb vbp'],
'exacerbated':['ih-g z-ae1 s-er b-ey t-ah-d','vbn vbd'],
'exacerbates':['ih-g z-ae1 s-er b-ey-t-s','vbz'],
'exacerbating':['ih-g z-ae1 s-er b-ey t-ih-ng','vbg'],
'exacerbations':['ih-g z-ae s-er b-ey1 sh-ah-n-z','nns'],
'exact':['ih-g z-ae1-k-t','jj vb'],
'exacted':['ih-g z-ae1-k t-ih-d','vbd'],
'exacting':['ih-g z-ae1-k t-ih-ng','jj'],
'exactly':['ih-g z-ae1-k-t l-iy','rb'],
'exaggerate':['ih-g z-ae1 jh-er ey-t','vb vbp'],
'exaggerated':['ih-g z-ae1 jh-er ey t-ah-d','vbn jj vbd'],
'exaggerating':['ih-g z-ae1 jh-er ey t-ih-ng','vbg'],
'exaggeration':['ih-g z-ae jh-er ey1 sh-ah-n','nn'],
'exalt':['ih-g z-ao1-l-t','vbp vb'],
'exalted':['ih-g z-ao1-l t-ih-d','jj vbd'],
'exam':['ih-g z-ae1-m','nn'],
'examination':['ih-g z-ae m-ah n-ey1 sh-ah-n','nn'],
'examine':['ih-g z-ae1 m-ih-n','vb vbp'],
'examined':['ih-g z-ae1 m-ah-n-d','vbd vbn'],
'examiner':['ih-g z-ae1 m-ah n-er','nn'],
'examines':['ih-g z-ae1 m-ah-n-z','vbz'],
'examining':['ih-g z-ae1 m-ih n-ih-ng','vbg nn'],
'example':['ih-g z-ae1-m p-ah-l','nn'],
'exasperated':['ih-g z-ae1 s-p-er ey t-ih-d','jj'],
'exasperating':['ih-g z-ae1 s-p-er ey t-ih-ng','vbg jj'],
'exasperation':['eh-k s-ae s-p-er ey1 sh-ah-n','nn'],
'excavate':['eh1-k s-k-ah v-ey-t','vb'],
'excavated':['eh1-k s-k-ah v-ey t-ih-d','vbn'],
'excavating':['eh1-k s-k-ah v-ey t-ih-ng','vbg'],
'excavation':['eh-k s-k-ah v-ey1 sh-ah-n','nn'],
'excavators':['eh1-k s-k-ah v-ey t-er-z','nns'],
'exceed':['ih-k s-iy1-d','vb vbp'],
'exceeded':['ih-k s-iy1 d-ah-d','vbd vbn'],
'exceeding':['ih-k s-iy1 d-ih-ng','vbg jj'],
'exceedingly':['ih-k s-iy1 d-ih-ng l-iy','rb'],
'exceeds':['ih-k s-iy1-d-z','vbz'],
'excel':['ih-k s-eh1-l','vbp vb'],
'excelled':['ih-k s-eh1-l-d','vbd vbn'],
'excellence':['eh1-k s-ah l-ah-n-s','nn'],
'excellent':['eh1-k s-ah l-ah-n-t','jj'],
'excellently':['eh1-k s-ah l-ah-n-t l-iy','rb'],
'excels':['ih-k s-eh1-l-z','vbz'],
'excelsior':['ih-k s-eh1-l s-iy er','nn'],
'except':['ih-k s-eh1-p-t','in vb'],
'excepting':['eh-k s-eh1-p t-ih-ng','vbg'],
'exception':['ih-k s-eh1-p sh-ah-n','nn'],
'exceptional':['ih-k s-eh1-p sh-ah n-ah-l','jj'],
'exceptionally':['ih-k s-eh1-p sh-ah-n ah l-iy','rb'],
'excerpt':['eh1-k s-er-p-t','nn'],
'excess':['eh1-k s-eh-s','jj nn'],
'excessive':['ih-k s-eh1 s-ih-v','jj'],
'excessively':['ih-k s-eh1 s-ih-v l-iy','rb'],
'exchange':['ih-k-s ch-ey1-n-jh','nn vb vbp'],
'exchangeable':['ih-k-s ch-ey1-n jh-ah b-ah-l','jj'],
'exchanged':['ih-k-s ch-ey1-n-jh-d','vbn vbd'],
'exchanger':['ih-k-s ch-ey1-n jh-er','nn'],
'exchanges':['ih-k-s ch-ey1-n jh-ah-z','nns vbz'],
'exchanging':['ih-k-s ch-ey1-n jh-ih-ng','vbg'],
'exchequer':['eh1-k-s ch-eh k-er','nn'],
'excise':['eh-k s-ay1-s','nn jj vb'],
'excised':['eh1-k s-ay-z-d','vbd vbn'],
'excision':['ih-k s-ih1 zh-ah-n','nn'],
'excite':['ih-k s-ay1-t','vb'],
'excited':['ih-k s-ay1 t-ah-d','vbn jj vbd'],
'excitedly':['ih-k s-ay1 t-ah-d l-iy','rb'],
'excitement':['ih-k s-ay1-t m-ah-n-t','nn'],
'excites':['ih-k s-ay1-t-s','vbz'],
'exciting':['ih-k s-ay1 t-ih-ng','jj vbg'],
'exclaim':['ih-k s-k-l-ey1-m','vb'],
'exclaimed':['ih-k s-k-l-ey1-m-d','vbd'],
'exclaiming':['ih-k s-k-l-ey1 m-ih-ng','vbg'],
'exclaims':['ih-k s-k-l-ey1-m-z','vbz'],
'exclamation':['eh-k s-k-l-ah m-ey1 sh-ah-n','nn'],
'exclude':['ih-k s-k-l-uw1-d','vb vbp'],
'excluded':['ih-k s-k-l-uw1 d-ah-d','vbn vbd'],
'excludes':['ih-k s-k-l-uw1-d-z','vbz'],
'excluding':['ih-k s-k-l-uw1 d-ih-ng','vbg'],
'exclusion':['ih-k s-k-l-uw1 zh-ah-n','nn'],
'exclusionary':['ih-k s-k-l-uw1 zh-ah-n eh r-iy','jj'],
'exclusive':['ih-k s-k-l-uw1 s-ih-v','jj nn'],
'exclusively':['ih-k s-k-l-uw1 s-ih-v l-iy','rb'],
'exclusivity':['eh-k s-k-l-uw s-ih1 v-ah t-iy','nn'],
'excommunicated':['eh-k-s k-ah m-y-uw1 n-ah k-ey t-ah-d','vbn'],
'excoriated':['eh-k-s k-ao1 r-iy ey t-ih-d','vbd vbn'],
'excoriating':['eh-k-s k-ao1 r-iy ey t-ih-ng','vbg'],
'excrement':['eh1-k s-k-r-ah m-ah-n-t','nn'],
'excrete':['ih-k s-k-r-iy1-t','vb'],
'excretion':['ih-k s-k-r-iy1 sh-ah-n','nn'],
'excretory':['eh1-k s-k-r-ah t-ao r-iy','jj'],
'excruciating':['ih-k s-k-r-uw1 sh-iy ey t-ih-ng','jj'],
'excursion':['ih-k s-k-er1 zh-ah-n','nn'],
'excuse':['ih-k s-k-y-uw1-s','nn vbp vb'],
'excused':['ih-k s-k-y-uw1-z-d','vbn vbd'],
'excuses':['ih-k s-k-y-uw1 s-ih-z','nns vbz'],
'exec':['eh-g z-eh1-k','nn'],
'execute':['eh1-k s-ah k-y-uw-t','vb vbp'],
'executed':['eh1-k s-ah k-y-uw t-ah-d','vbn jj vbd'],
'executes':['eh1-k s-ah k-y-uw-t-s','vbz'],
'executing':['eh1-k s-ah k-y-uw t-ih-ng','vbg'],
'execution':['eh-k s-ah k-y-uw1 sh-ah-n','nn'],
'executioner':['eh-k s-ah k-y-uw1 sh-ah-n er','nn'],
'executive':['ih-g z-eh1 k-y-ah t-ih-v','nn jj'],
'executor':['ih-g z-eh1 k-y-ah t-er','nn'],
'exemplary':['ih-g z-eh1-m p-l-er iy','jj'],
'exemplified':['ih-g z-eh1-m p-l-ah f-ay-d','vbn'],
'exemplifies':['ih-g z-eh1-m p-l-ah f-ay-z','vbz'],
'exemplify':['ih-g z-eh1-m p-l-ah f-ay','vbp vb'],
'exempt':['ih-g z-eh1-m-p-t','jj vb'],
'exempted':['ih-g z-eh1-m-p t-ih-d','vbn vbd'],
'exempting':['ih-g z-eh1-m-p t-ih-ng','vbg'],
'exemption':['ih-g z-eh1-m-p sh-ah-n','nn'],
'exempts':['ih-g z-eh1-m-p-t-s','vbz'],
'exercisable':['eh1-k s-er s-ay z-ah b-ah-l','jj'],
'exercise':['eh1-k s-er s-ay-z','nn vbp vb'],
'exercised':['eh1-k s-er s-ay-z-d','vbn vbd jj'],
'exercises':['eh1-k s-er s-ay z-ah-z','nns vbz'],
'exercising':['eh1-k s-er s-ay z-ih-ng','vbg nn'],
'exert':['ih-g z-er1-t','vb vbp'],
'exerted':['ih-g z-er1 t-ih-d','vbn vbd'],
'exerting':['ih-g z-er1 t-ih-ng','vbg'],
'exertion':['ih-g z-er1 sh-ah-n','nn'],
'exerts':['ih-g z-er1-t-s','vbz'],
'exhaled':['eh-k-s hh-ey1-l-d','vbd'],
'exhaust':['ih-g z-ao1-s-t','nn vb vbp'],
'exhausted':['ih-g z-ao1 s-t-ah-d','vbn vbd'],
'exhausting':['ih-g z-ao1 s-t-ih-ng','vbg jj'],
'exhaustion':['ih-g z-ao1-s ch-ah-n','nn'],
'exhaustive':['ih-g z-ao1 s-t-ih-v','jj'],
'exhaustively':['ih-g z-aa1 s-t-ih-v l-iy','rb'],
'exhausts':['ih-g z-ao1-s-t-s','nns vbz'],
'exhibit':['ih-g z-ih1 b-ih-t','nn vbp vb'],
'exhibited':['ih-g z-ih1 b-ah t-ah-d','vbn vbd'],
'exhibiting':['ih-g z-ih1 b-ih t-ih-ng','vbg'],
'exhibition':['eh-k s-ah b-ih1 sh-ah-n','nn'],
'exhibitors':['ih-g z-ih1 b-ah t-er-z','nns'],
'exhibits':['ih-g z-ih1 b-ah-t-s','nns vbz'],
'exhilarated':['ih-g z-ih1 l-er ey t-ih-d','vbn'],
'exhilarating':['ih-g z-ih1 l-er ey t-ih-ng','jj'],
'exhilaration':['ih-g z-ih l-er ey1 sh-ah-n','nn'],
'exhort':['ih-g z-ao1-r-t','vb'],
'exhortations':['eh-g z-ao-r t-ey1 sh-ah-n-z','nns'],
'exhorting':['ih-g z-ao1-r t-ih-ng','vbg'],
'exhorts':['ih-g z-ao1-r-t-s','vbz'],
'exigencies':['eh-k s-ih1 jh-ah-n s-iy-z','nns'],
'exile':['eh1-g z-ay-l','nn vb'],
'exiled':['eh1-g z-ay-l-d','vbn vbd jj'],
'exiles':['eh1-g z-ay-l-z','nns'],
'exist':['ih-g z-ih1-s-t','vb vbp nnp'],
'existed':['ih-g z-ih1 s-t-ah-d','vbd vbn'],
'existence':['eh-g z-ih1 s-t-ah-n-s','nn'],
'existent':['eh-g z-ih1 s-t-ah-n-t','jj nn'],
'existential':['eh-k s-ih s-t-eh1-n ch-ah-l','jj'],
'existing':['ih-g z-ih1 s-t-ih-ng','vbg jj'],
'exists':['ih-g z-ih1-s-t-s','vbz'],
'exit':['eh1-g z-ih-t','nn vb'],
'exited':['eh1-g z-ah t-ih-d','vbd vbn'],
'exiting':['eh1-g z-ih t-ih-ng','vbg'],
'exits':['eh1-g z-ih-t-s','nns vbz'],
'exodus':['eh1-k s-ah d-ah-s','nn'],
'exonerate':['ih-g z-aa1 n-er ey-t','vb'],
'exonerated':['ih-g z-aa1 n-er ey t-ih-d','vbn vbd'],
'exonerating':['ih-g z-aa1 n-er ey t-ih-ng','vbg'],
'exorbitant':['ih-g z-ao1-r b-ih t-ah-n-t','jj'],
'exorcism':['eh1-k s-er s-ih1-z ah-m','nn'],
'exorcist':['eh1-k s-er s-ah-s-t','nn'],
'exothermic':['eh-k s-ow th-er1 m-ih-k','jj'],
'exotic':['ih-g z-aa1 t-ih-k','jj'],
'expand':['ih-k s-p-ae1-n-d','vb vbp'],
'expandable':['ih-k s-p-ae1-n d-ah b-ah-l','jj'],
'expanded':['ih-k s-p-ae1-n d-ah-d','vbn vbd jj'],
'expanding':['ih-k s-p-ae1-n d-ih-ng','vbg jj'],
'expands':['ih-k s-p-ae1-n-d-z','vbz'],
'expanse':['ih-k s-p-ae1-n-s','nn jj'],
'expansion':['ih-k s-p-ae1-n sh-ah-n','nn jj'],
'expansionism':['ih-k s-p-ae1-n sh-ah n-ih z-ah-m','nn'],
'expansive':['ih-k s-p-ae1-n s-ih-v','jj'],
'expatriate':['eh-k-s p-ey1 t-r-iy ey-t','nn'],
'expect':['ih-k s-p-eh1-k-t','vbp vb in'],
'expectancy':['ih-k s-p-eh1-k t-ah-n s-iy','nn'],
'expectant':['ih-k s-p-eh1-k t-ah-n-t','jj'],
'expectation':['eh-k s-p-eh-k t-ey1 sh-ah-n','nn'],
'expected':['ih-k s-p-eh1-k t-ah-d','vbn vbd jj vb'],
'expecting':['ih-k s-p-eh1-k t-ih-ng','vbg'],
'expects':['ih-k s-p-eh1-k-t-s','vbz'],
'expediency':['ih-k s-p-iy1 d-iy ah-n s-iy','nn'],
'expedient':['ih-k s-p-iy1 d-iy ah-n-t','jj nn'],
'expedite':['eh1-k s-p-ih d-ay-t','vb'],
'expedited':['eh1-k s-p-ih d-ay t-ih-d','vbn vbd jj'],
'expediting':['eh1-k s-p-ah d-ay t-ih-ng','vbg'],
'expedition':['eh-k s-p-ah d-ih1 sh-ah-n','nn'],
'expeditious':['eh-k s-p-ah d-ih1 sh-ah-s','jj'],
'expeditiously':['eh-k s-p-ah d-ih1 sh-ah-s l-iy','rb'],
'expel':['ih-k s-p-eh1-l','vb'],
'expelled':['ih-k s-p-eh1-l-d','vbn vbd'],
'expelling':['ih-k s-p-eh1 l-ih-ng','vbg'],
'expend':['ih-k s-p-eh1-n-d','vb vbp'],
'expendable':['ih-k s-p-eh1-n d-ah b-ah-l','jj'],
'expended':['ih-k s-p-eh1-n d-ih-d','vbn vbd'],
'expenditure':['ih-k s-p-eh1-n d-ah ch-er','nn'],
'expenditures':['ih-k s-p-eh1-n d-ah ch-er-z','nns vbz'],
'expense':['ih-k s-p-eh1-n-s','nn'],
'expenses':['ih-k s-p-eh1-n s-ah-z','nns'],
'expensive':['ih-k s-p-eh1-n s-ih-v','jj rb'],
'experience':['ih-k s-p-ih1 r-iy ah-n-s','nn vbp vb'],
'experienced':['ih-k s-p-ih1 r-iy ah-n-s-t','vbn jj vbd'],
'experiences':['ih-k s-p-ih1 r-iy ah-n s-ih-z','nns vbz'],
'experiencing':['ih-k s-p-ih1 r-iy ah-n s-ih-ng','vbg'],
'experiment':['ih-k s-p-eh1 r-ah m-ah-n-t','nn vbp vb'],
'experimental':['ih-k s-p-eh r-ah m-eh1-n t-ah-l','jj'],
'experimentally':['ih-k s-p-eh r-ah m-eh1-n t-ah l-iy','rb'],
'experimentation':['ih-k s-p-eh r-ah m-ah-n t-ey1 sh-ah-n','nn'],
'experimented':['ih-k s-p-eh1 r-ah m-ah-n t-ah-d','vbd vbn'],
'experimenters':['ih-k s-p-eh1 r-ah m-eh-n t-er-z','nns'],
'experimenting':['eh-k s-p-eh1 r-ah m-eh-n t-ih-ng','vbg'],
'experiments':['ih-k s-p-eh1 r-ah m-ah-n-t-s','nns'],
'expert':['eh1-k s-p-er-t','nn jj'],
'expertise':['eh-k s-p-er t-iy1-z','nn'],
'expertly':['eh1-k s-p-er-t l-iy','rb'],
'expiration':['eh-k s-p-er ey1 sh-ah-n','nn'],
'expire':['ih-k s-p-ay1-r','vb vbp'],
'expired':['ih-k s-p-ay1-r-d','vbd vbn jj'],
'expires':['ih-k s-p-ay1 er-z','vbz'],
'expiring':['ih-k s-p-ay1 r-ih-ng','vbg'],
'explain':['ih-k s-p-l-ey1-n','vb vbp'],
'explained':['ih-k s-p-l-ey1-n-d','vbd vbn'],
'explaining':['ih-k s-p-l-ey1 n-ih-ng','vbg nn'],
'explains':['ih-k s-p-l-ey1-n-z','vbz'],
'explanation':['eh-k s-p-l-ah n-ey1 sh-ah-n','nn'],
'explanatory':['ih-k s-p-l-ae1 n-ah t-ao r-iy','jj'],
'expletive':['eh1-k s-p-l-ah t-ih-v','nn'],
'explicit':['ih-k s-p-l-ih1 s-ah-t','jj'],
'explicitly':['ih-k s-p-l-ih1 s-ah-t l-iy','rb'],
'explode':['ih-k s-p-l-ow1-d','vb vbp'],
'exploded':['ih-k s-p-l-ow1 d-ah-d','vbd vbn'],
'explodes':['ih-k s-p-l-ow1-d-z','vbz'],
'exploding':['ih-k s-p-l-ow1 d-ih-ng','vbg'],
'exploit':['eh1-k s-p-l-oy-t','vb nn vbp'],
'exploitation':['eh-k s-p-l-oy t-ey1 sh-ah-n','nn'],
'exploitative':['eh-k s-p-l-oy1 t-ah t-ih-v','jj'],
'exploited':['eh1-k s-p-l-oy t-ah-d','vbn vbd'],
'exploiting':['eh1-k s-p-l-oy t-ih-ng','vbg'],
'exploits':['eh1-k s-p-l-oy-t-s','nns vbz'],
'exploration':['eh-k s-p-l-er ey1 sh-ah-n','nn'],
'exploratory':['ih-k s-p-l-ao1 r-ah t-ao r-iy','jj'],
'explore':['ih-k s-p-l-ao1-r','vb vbp'],
'explored':['ih-k s-p-l-ao1-r-d','vbn vbd'],
'explorer':['ih-k s-p-l-ao1 r-er','nn'],
'explores':['ih-k s-p-l-ao1-r-z','vbz'],
'exploring':['ih-k s-p-l-ao1 r-ih-ng','vbg'],
'explosion':['ih-k s-p-l-ow1 zh-ah-n','nn'],
'explosive':['ih-k s-p-l-ow1 s-ih-v','jj nn'],
'explosively':['eh-k s-p-l-ow1 s-ih-v l-iy','rb'],
'expo':['eh1-k s-p-ow','nn'],
'exponential':['eh-k s-p-ow n-eh1-n ch-ah-l','jj'],
'exponentially':['eh-k s-p-ow n-eh1-n sh-ah l-iy','rb'],
'exponents':['ih-k s-p-ow1 n-ah-n-t-s','nns'],
'export':['eh1-k s-p-ao-r-t','nn vbp jj vb'],
'exported':['ih-k s-p-ao1-r t-ah-d','vbn vbd'],
'exporter':['ih-k s-p-ao1-r t-er','nn'],
'exporting':['ih-k s-p-ao1-r t-ih-ng','vbg'],
'exports':['eh1-k s-p-ao-r-t-s','nns vbz'],
'expose':['ih-k s-p-ow1-z','vb nn vbp'],
'exposed':['ih-k s-p-ow1-z-d','vbn jj vbd'],
'exposes':['ih-k s-p-ow1 z-ih-z','vbz'],
'exposing':['ih-k s-p-ow1 z-ih-ng','vbg jj'],
'exposition':['eh-k s-p-ah z-ih1 sh-ah-n','nn'],
'exposure':['ih-k s-p-ow1 zh-er','nn'],
'expounded':['ih-k s-p-aw1-n d-ah-d','vbd vbn'],
'expounding':['ih-k s-p-aw1-n d-ih-ng','vbg'],
'express':['ih-k s-p-r-eh1-s','vb vbp jj nn'],
'expressed':['ih-k s-p-r-eh1-s-t','vbn vbd jj'],
'expresses':['ih-k s-p-r-eh1 s-ah-z','vbz'],
'expressing':['ih-k s-p-r-eh1 s-ih-ng','vbg'],
'expression':['ih-k s-p-r-eh1 sh-ah-n','nn'],
'expressionism':['ih-k s-p-r-eh1 sh-ah-n ih z-ah-m','nn'],
'expressionist':['ih-k s-p-r-eh1 sh-ah-n ah-s-t','nn'],
'expressionistic':['ih-k s-p-r-eh sh-ah n-ih1 s-t-ih-k','jj'],
'expressive':['ih-k s-p-r-eh1 s-ih-v','jj'],
'expressly':['eh-k s-p-r-eh1-s l-iy','rb'],
'expressway':['ih-k s-p-r-eh1-s w-ey','nn'],
'expropriated':['eh-k-s p-r-ow1 p-r-iy ey t-ih-d','jj'],
'expulsion':['ih-k s-p-ah1-l sh-ah-n','nn'],
'expunged':['ih-k s-p-ah1-n-jh-d','vbn'],
'exquisite':['eh1-k s-k-w-ah z-ah-t','jj'],
'exquisitely':['eh-k s-k-w-ih1 z-ih-t l-iy','rb'],
'extant':['eh1-k s-t-ah-n-t','jj'],
'extend':['ih-k s-t-eh1-n-d','vb vbp vbz'],
'extended':['ih-k s-t-eh1-n d-ah-d','vbn vbd jj'],
'extendible':['eh-k s-t-eh1-n d-ih b-ah-l','jj'],
'extending':['ih-k s-t-eh1-n d-ih-ng','vbg'],
'extends':['ih-k s-t-eh1-n-d-z','vbz'],
'extension':['ih-k s-t-eh1-n sh-ah-n','nn'],
'extensive':['ih-k s-t-eh1-n s-ih-v','jj'],
'extensively':['ih-k s-t-eh1-n s-ih-v l-iy','rb'],
'extent':['ih-k s-t-eh1-n-t','nn'],
'extenuating':['ih-k s-t-eh1 n-y-uw ey t-ih-ng','vbg'],
'exterior':['ih-k s-t-ih1 r-iy er','jj nn'],
'exterminating':['ih-k s-t-er1 m-ah n-ey t-ih-ng','vbg'],
'extermination':['ih-k s-t-er m-ah n-ey1 sh-ah-n','nn'],
'exterminator':['ih-k s-t-er1 m-ah n-ey t-er','nn'],
'external':['ih-k s-t-er1 n-ah-l','jj'],
'externally':['ih-k s-t-er1 n-ah l-iy','rb'],
'extinct':['ih-k s-t-ih1-ng-k-t','jj'],
'extinction':['ih-k s-t-ih1-ng-k sh-ah-n','nn'],
'extinguish':['ih-k s-t-ih1-ng g-w-ih-sh','vb'],
'extinguished':['ih-k s-t-ih1-ng g-w-ih-sh-t','vbn'],
'extinguisher':['ih-k s-t-ih1-ng g-w-ih sh-er','nn'],
'extinguishment':['ih-k s-t-ih1-ng g-w-ih-sh m-ah-n-t','nn'],
'extolled':['ih-k s-t-ow1-l-d','vbd'],
'extolling':['ih-k s-t-ow1 l-ih-ng','vbg'],
'extols':['ih-k s-t-ow1-l-z','vbz'],
'extort':['ih-k s-t-ao1-r-t','vb'],
'extorted':['ih-k s-t-ao1-r t-ih-d','vbd'],
'extorting':['ih-k s-t-ao1-r t-ih-ng','vbg'],
'extortion':['ih-k s-t-ao1-r sh-ah-n','nn'],
'extortionate':['ih-k s-t-ao1-r sh-ah-n ah-t','jj'],
'extra':['eh1-k s-t-r-ah','jj nn rb'],
'extract':['ih-k s-t-r-ae1-k-t','vb nn vbp'],
'extracted':['ih-k s-t-r-ae1-k t-ah-d','vbn vbd'],
'extracting':['ih-k s-t-r-ae1-k t-ih-ng','vbg'],
'extraction':['ih-k s-t-r-ae1-k sh-ah-n','nn'],
'extracts':['ih-k s-t-r-ae1-k-t-s','nns vbz'],
'extracurricular':['eh-k s-t-r-ah k-er ih1 k-y-ah l-er','jj'],
'extradite':['eh1-k s-t-r-ah d-ay-t','vb'],
'extradited':['eh1-k s-t-r-ah d-ay t-ih-d','vbn'],
'extraditing':['eh1-k s-t-r-ah d-ay t-ih-ng','vbg'],
'extradition':['eh-k s-t-r-ah d-ih1 sh-ah-n','nn'],
'extralegal':['eh-k s-t-r-ah l-iy1 g-ah-l','jj'],
'extramarital':['eh-k s-t-r-ah m-eh1 r-ah t-ah-l','jj'],
'extraneous':['eh-k s-t-r-ey1 n-iy ah-s','jj'],
'extraordinarily':['ih-k s-t-r-ao-r d-ah n-eh1 r-ah l-iy','rb'],
'extraordinary':['ih-k s-t-r-ao1-r d-ah-n eh r-iy','jj nn'],
'extrapolate':['ih-k s-t-r-ae1 p-ah l-ey-t','vb'],
'extrapolated':['ih-k s-t-r-ae1 p-ah l-ey t-ih-d','vbn'],
'extrapolation':['ih-k s-t-r-ae p-ah l-ey1 sh-ah-n','nn'],
'extraterrestrial':['eh-k s-t-r-ah t-er eh1-s t-r-iy ah-l','jj'],
'extraterritorial':['eh-k s-t-r-ah t-eh r-ih t-ao1 r-iy ah-l','jj'],
'extravagance':['ih-k s-t-r-ae1 v-ah g-ah-n-s','nn'],
'extravagant':['ih-k s-t-r-ae1 v-ah g-ah-n-t','jj'],
'extravagantly':['eh-k s-t-r-ae1 v-ah g-ah-n-t l-iy','rb'],
'extravaganza':['ih-k s-t-r-ae v-ah g-ae1-n z-ah','nn'],
'extreme':['ih-k s-t-r-iy1-m','jj nn'],
'extremely':['ih-k s-t-r-iy1-m l-iy','rb'],
'extremist':['ih-k s-t-r-iy1 m-ih-s-t','jj nn'],
'extremity':['ih-k-s t-r-eh1 m-ah t-iy','nn'],
'extricate':['eh1-k s-t-r-ah k-ey-t','vb'],
'extruded':['ih-k s-t-r-uw1 d-ah-d','vbn jj'],
'extruding':['ih-k s-t-r-uw1 d-ih-ng','vbg'],
'extrusion':['ih-k-s t-r-uw1 zh-ah-n','nn'],
'exuberance':['ih-g z-uw1 b-er ah-n-s','nn'],
'exuberant':['ih-g z-uw1 b-er ah-n-t','jj'],
'exude':['ih-g z-uw1-d','vbp'],
'exuded':['ih-g z-uw1 d-ih-d','vbd'],
'exultantly':['ih-g z-ah1-l t-ah-n-t l-iy','rb'],
'exulted':['ih-g z-ah1-l t-ih-d','vbd'],
'exults':['ih-g z-ah1-l-t-s','vbz'],
'eye':['ay1','nn'],
'eyeball':['ay1 b-ao-l','nn'],
'eyebrow':['ay1 b-r-aw','nn'],
'eyed':['ay1-d','vbd'],
'eyeglasses':['ay1 g-l-ae s-ah-z','nns'],
'eyeing':['ay1 ih-ng','vbg'],
'eyelashes':['ay1 l-ae sh-ih-z','nns'],
'eyelets':['ay1 l-ah-t-s','nns'],
'eyelids':['ay1 l-ih-d-z','nns'],
'eyepiece':['ay1 p-iy-s','nn'],
'eyesight':['ay1 s-ay-t','nn'],
'eyesore':['ay1 s-ao-r','nn'],
'eyewear':['ay1 w-eh-r','jj nn'],
'eyewitness':['ay1 w-ih1-t n-ah-s','nn jj'],
'f':['eh1-f','nn'],
'fable':['f-ey1 b-ah-l','nn'],
'fabled':['f-ey1 b-ah-l-d','jj'],
'fabric':['f-ae1 b-r-ih-k','nn'],
'fabricate':['f-ae1 b-r-ah k-ey-t','vb vbp'],
'fabricated':['f-ae1 b-r-ih k-ey t-ah-d','vbn vbd jj'],
'fabricates':['f-ae1 b-r-ih k-ey-t-s','vbz'],
'fabricating':['f-ae1 b-r-ih k-ey t-ih-ng','vbg nn'],
'fabrication':['f-ae b-r-ih k-ey1 sh-ah-n','nn'],
'fabricator':['f-ae1 b-r-ih k-ey t-er','nn'],
'fabulous':['f-ae1 b-y-ah l-ah-s','jj'],
'fabulously':['f-ae1 b-y-uw l-ah-s l-iy','rb'],
'facade':['f-ah s-aa1-d','nn'],
'face':['f-ey1-s','nn vbp jj rb vb'],
'faced':['f-ey1-s-t','vbn vbd'],
'faceless':['f-ey1-s l-ah-s','jj'],
'facelift':['f-ey1-s l-ih-f-t','nn'],
'faces':['f-ey1 s-ah-z','vbz nns'],
'facet':['f-ae1 s-ah-t','nn'],
'facetious':['f-ah s-iy1 sh-ah-s','jj'],
'facetiously':['f-ah s-iy1 sh-ah-s l-iy','rb'],
'facial':['f-ey1 sh-ah-l','jj'],
'facile':['f-ae1 s-ah-l','jj'],
'facilitate':['f-ah s-ih1 l-ah t-ey-t','vb vbp'],
'facilitated':['f-ah s-ih1 l-ah t-ey t-ih-d','vbn'],
'facilitates':['f-ah s-ih1 l-ah t-ey-t-s','vbz'],
'facilitating':['f-ah s-ih1 l-ah t-ey t-ih-ng','vbg'],
'facilitators':['f-ah s-ih1 l-ah t-ey t-er-z','nns'],
'facility':['f-ah s-ih1 l-ih t-iy','nn'],
'facing':['f-ey1 s-ih-ng','vbg jj'],
'facsimile':['f-ae-k s-ih1 m-ah l-iy','nn jj'],
'fact':['f-ae1-k-t','nn'],
'faction':['f-ae1-k sh-ah-n','nn'],
'factor':['f-ae1-k t-er','nn vb vbp'],
'factored':['f-ae1-k t-er-d','vbn jj'],
'factoring':['f-ae1-k t-er ih-ng','nn vbg'],
'factors':['f-ae1-k t-er-z','nns vbz'],
'factory':['f-ae1-k t-er iy','nn jj'],
'factual':['f-ae1-k ch-uw ah-l','jj'],
'factually':['f-ae1-k ch-uw ah l-iy','rb'],
'faculty':['f-ae1 k-ah-l t-iy','nn'],
'fad':['f-ae1-d','nn'],
'faddish':['f-ae1 d-ih-sh','jj'],
'fade':['f-ey1-d','vb vbp nn'],
'faded':['f-ey1 d-ah-d','vbn vbd jj'],
'fades':['f-ey1-d-z','nns vbz'],
'fading':['f-ey1 d-ih-ng','vbg jj nn'],
'fail':['f-ey1-l','vb vbp nn'],
'failed':['f-ey1-l-d','vbd vbn jj'],
'failing':['f-ey1 l-ih-ng','vbg nn'],
'fails':['f-ey1-l-z','vbz nns'],
'failure':['f-ey1 l-y-er','nn'],
'faint':['f-ey1-n-t','jj nn vb'],
'fainted':['f-ey1-n t-ih-d','vbd vbn'],
'fainter':['f-ey1-n t-er','jjr'],
'faintest':['f-ey1-n t-ah-s-t','jjs'],
'fainting':['f-ey1-n t-ih-ng','nn vbg'],
'faintly':['f-ey1-n-t l-iy','rb'],
'fair':['f-eh1-r','jj nn rb'],
'fairer':['f-eh1 r-er','jjr'],
'faires':['f-eh1-r-z','nns'],
'fairest':['f-eh1 r-ih-s-t','jjs'],
'fairly':['f-eh1-r l-iy','rb'],
'fairness':['f-eh1-r n-ah-s','nn'],
'fairway':['f-eh1-r w-ey','nn'],
'fairy':['f-eh1 r-iy','jj nn'],
'fait':['f-ey1-t','nn'],
'faith':['f-ey1-th','nn'],
'faithful':['f-ey1-th f-ah-l','jj nn'],
'faithfully':['f-ey1-th f-ah l-iy','rb'],
'fajitas':['f-ah jh-iy1 t-ah-z','nns'],
'fake':['f-ey1-k','jj nn vbp'],
'faked':['f-ey1-k-t','vbn vbd'],
'faking':['f-ey1 k-ih-ng','vbg'],
'falcon':['f-ae1-l k-ah-n','nn'],
'falconers':['f-ae1-l k-ah n-er-z','nns'],
'falconry':['f-ae1-l k-ah-n r-iy','nn'],
'fall':['f-ao1-l','nn vbp vb'],
'fallacious':['f-ah l-ey1 sh-ah-s','jj'],
'fallacy':['f-ae1 l-ah s-iy','nn'],
'fallback':['f-ao1-l b-ae-k','nn jj'],
'fallen':['f-aa1 l-ah-n','vbn jj'],
'fallible':['f-ae1 l-ah b-ah-l','jj'],
'falling':['f-aa1 l-ih-ng','vbg jj nn'],
'falloff':['f-ao1-l ao-f','nn'],
'fallout':['f-ao1-l aw-t','nn'],
'fallow':['f-ae1 l-ow','jj'],
'falls':['f-ao1-l-z','vbz nns'],
'false':['f-ao1-l-s','jj rb'],
'falsehood':['f-ae1-l-s hh-uh-d','nn'],
'falsely':['f-ao1-l-s l-iy','rb'],
'falsification':['f-ae-l s-ah f-ah k-ey1 sh-ah-n','nn'],
'falsified':['f-ao1-l s-ah f-ay-d','vbn'],
'falsify':['f-ao1-l s-ah f-ay','vb'],
'falsifying':['f-ao1-l s-ah f-ay ih-ng','vbg'],
'falsity':['f-ao1-l s-ah t-iy','nn'],
'falter':['f-ao1-l t-er','vb vbp'],
'faltered':['f-aa1-l t-er-d','vbd vbn'],
'faltering':['f-ao1-l t-er ih-ng','vbg nn'],
'falters':['f-ao1-l t-er-z','vbz'],
'fame':['f-ey1-m','nn vb'],
'famed':['f-ey1-m-d','jj vbn'],
'familial':['f-ah m-ih1 l-y-ah-l','jj'],
'familiar':['f-ah m-ih1 l-y-er','jj nn'],
'familiarity':['f-ah m-ih l-y-eh1 r-ah t-iy','nn'],
'familiarize':['f-ah m-ih1 l-y-er ay-z','vb'],
'family':['f-ae1 m-ah l-iy','nn'],
'famine':['f-ae1 m-ah-n','nn'],
'famous':['f-ey1 m-ah-s','jj nns'],
'famously':['f-ey1 m-ah-s l-iy','rb'],
'fan':['f-ae1-n','nn vbp vb'],
'fanatic':['f-ah n-ae1 t-ih-k','nn jj'],
'fanatical':['f-ah n-ae1 t-ih k-ah-l','jj'],
'fanatically':['f-ah n-ae1 t-ih k-ah l-iy','rb'],
'fanaticism':['f-ah n-ae1 t-ah s-ih z-ah-m','nn'],
'fancied':['f-ae1-n s-iy-d','vbd'],
'fancier':['f-ae1-n s-iy er','jjr'],
'fancies':['f-ae1-n s-iy-z','vbz'],
'fanciest':['f-ae1-n s-iy ah-s-t','jjs'],
'fanciful':['f-ae1-n s-ih f-ah-l','jj'],
'fancy':['f-ae1-n s-iy','jj nn vb'],
'fanfare':['f-ae1-n f-eh-r','nn'],
'fangs':['f-ae1-ng-z','nns'],
'fanned':['f-ae1-n-d','vbd vbn'],
'fanning':['f-ae1 n-ih-ng','vbg nn'],
'fanny':['f-ae1 n-iy','nn'],
'fans':['f-ae1-n-z','nns vbz'],
'fantasia':['f-ae-n t-ey1 zh-ah','nn'],
'fantasize':['f-ae1-n t-ah s-ay-z','vbp vb'],
'fantasized':['f-ae1-n t-ih s-ay-z-d','vbn'],
'fantastic':['f-ae-n t-ae1 s-t-ih-k','jj'],
'fantastically':['f-ae-n t-ae1 s-t-ih-k l-iy','rb'],
'fantasy':['f-ae1-n t-ah s-iy','nn'],
'far':['f-aa1-r','rb in jj'],
'faraway':['f-aa1 r-ah w-ey1','jj'],
'farce':['f-aa1-r-s','nn'],
'fare':['f-eh1-r','nn vbp vb'],
'fared':['f-eh1-r-d','vbd vbn'],
'fares':['f-eh1-r-z','nns nn vbz'],
'farewell':['f-eh-r w-eh1-l','nn uh'],
'farfetched':['f-aa1-r f-eh1-ch-t','jj'],
'faring':['f-eh1 r-iy-ng','vbg'],
'farm':['f-aa1-r-m','nn vb'],
'farmed':['f-aa1-r-m-d','vbd jj'],
'farmer':['f-aa1-r m-er','nn'],
'farmhands':['f-aa1-r-m hh-ae-n-d-z','nns'],
'farmhouse':['f-aa1-r-m hh-aw-s','nn'],
'farmhouses':['f-aa1-r-m hh-aw s-ah-z','nns'],
'farming':['f-aa1-r m-ih-ng','nn vbg'],
'farmland':['f-aa1-r-m l-ae-n-d','nn'],
'farms':['f-aa1-r-m-z','nns vbz'],
'faro':['f-eh1 r-ow','nn'],
'farther':['f-aa1-r dh-er','rb rbr'],
'farthest':['f-aa1-r dh-ah-s-t','jjs rbs'],
'fascinate':['f-ae1 s-ah n-ey-t','vb vbp'],
'fascinated':['f-ae1 s-ah n-ey t-ah-d','vbn jj vbd'],
'fascinates':['f-ae1 s-ah n-ey-t-s','vbz'],
'fascinating':['f-ae1 s-ah n-ey t-ih-ng','jj'],
'fascination':['f-ae s-ah n-ey1 sh-ah-n','nn'],
'fascism':['f-ae1 sh-ih z-ah-m','nn'],
'fascist':['f-ae1 sh-ah-s-t','jj'],
'fashion':['f-ae1 sh-ah-n','nn vb'],
'fashionable':['f-ae1 sh-ah-n ah b-ah-l','jj nn'],
'fashioned':['f-ae1 sh-ah-n-d','vbn vbd'],
'fashioning':['f-ae1 sh-ah-n ih-ng','vbg'],
'fashions':['f-ae1 sh-ah-n-z','nns vbz'],
'fast':['f-ae1-s-t','rb jj nn rp'],
'fastball':['f-ae1-s-t b-ao-l','nn'],
'fasten':['f-ae1 s-ah-n','vb vbp'],
'fastened':['f-ae1 s-ah-n-d','vbn vbd'],
'fastener':['f-ae1 s-ah n-er','nn'],
'fastening':['f-ae1 s-ah n-ih-ng','nn'],
'faster':['f-ae1 s-t-er','rbr jjr rb'],
'fastest':['f-ae1 s-t-ah-s-t','jjs rbs'],
'fastidious':['f-ae s-t-ih1 d-iy ah-s','jj'],
'fastness':['f-ae1-s-t n-ah-s','nn'],
'fat':['f-ae1-t','jj nn'],
'fatal':['f-ey1 t-ah-l','jj'],
'fatality':['f-ah t-ae1 l-ah t-iy','nn'],
'fatally':['f-ey1 t-ah l-iy','rb'],
'fate':['f-ey1-t','nn'],
'fateful':['f-ey1-t f-ah-l','jj'],
'father':['f-aa1 dh-er','nn vb'],
'fathered':['f-aa1 dh-er-d','vbd'],
'fatherly':['f-aa1 dh-er l-iy','jj'],
'fathers':['f-aa1 dh-er-z','nns'],
'fathom':['f-ae1 dh-ah-m','vb'],
'fathoms':['f-ae1 dh-ah-m-z','nns'],
'fatigue':['f-ah t-iy1-g','nn vbp'],
'fatigued':['f-ah t-iy1-g-d','vbn jj'],
'fatten':['f-ae1 t-ah-n','vb vbp'],
'fattened':['f-ae1 t-ah-n-d','vbn vbd'],
'fattening':['f-ae1 t-ah-n ih-ng','vbg nn'],
'fatter':['f-ae1 t-er','jjr'],
'fatty':['f-ae1 t-iy','jj'],
'fatuous':['f-ae1 ch-ah-w ah-s','jj'],
'faucet':['f-ao1 s-ah-t','nn'],
'fault':['f-ao1-l-t','nn vb vbp'],
'faulted':['f-ao1-l t-ih-d','vbn vbd'],
'faulting':['f-ao1-l t-ih-ng','vbg'],
'faults':['f-ao1-l-t-s','nns vbz'],
'faulty':['f-ao1-l t-iy','jj'],
'fauna':['f-ao1 n-ah','nns nn'],
'faux':['f-ao1-k-s','jj'],
'favor':['f-ey1 v-er','nn vbp vb'],
'favorable':['f-ey1 v-er ah b-ah-l','jj'],
'favorably':['f-ey1 v-er ah b-l-iy','rb'],
'favored':['f-ey1 v-er-d','vbn vbd jj'],
'favoring':['f-ey1 v-er ih-ng','vbg'],
'favorite':['f-ey1 v-er ih-t','jj nn'],
'favoritism':['f-ey1 v-er ih t-ih z-ah-m','nn'],
'favors':['f-ey1 v-er-z','vbz nns'],
'fawn':['f-ao1-n','nn'],
'fawning':['f-ao1 n-ih-ng','jj vbg'],
'fax':['f-ae1-k-s','nn jj'],
'faxed':['f-ae1-k-s-t','vbd vbn'],
'faxing':['f-ae1-k s-ih-ng','vbg'],
'faze':['f-ey1-z','vb'],
'fazed':['f-ey1-z-d','vbd'],
'fealty':['f-iy1 ah-l t-iy','nn'],
'fear':['f-ih1-r','nn vb vbp'],
'feared':['f-ih1-r-d','vbd vbn jj'],
'fearful':['f-ih1-r f-ah-l','jj'],
'fearing':['f-ih1 r-ih-ng','vbg'],
'fearless':['f-ih1-r l-ah-s','jj'],
'fears':['f-ih1-r-z','nns vbz'],
'fearsome':['f-ih1-r s-ah-m','jj'],
'feasibility':['f-iy z-ah b-ih1 l-ah t-iy','nn'],
'feasible':['f-iy1 z-ah b-ah-l','jj'],
'feast':['f-iy1-s-t','nn vb'],
'feasted':['f-iy1 s-t-ih-d','vbn'],
'feasting':['f-iy1 s-t-ih-ng','vbg'],
'feasts':['f-iy1-s-t-s','nns'],
'feat':['f-iy1-t','nn'],
'feather':['f-eh1 dh-er','nn vb'],
'featherbedding':['f-eh1 dh-er b-eh d-ih-ng','nn'],
'feathered':['f-eh1 dh-er-d','jj vbn'],
'featherless':['f-eh1 dh-er l-ah-s','jj'],
'feathers':['f-eh1 dh-er-z','nns'],
'feathery':['f-eh1 dh-er iy','jj'],
'feature':['f-iy1 ch-er','nn vbp jj vb'],
'featured':['f-iy1 ch-er-d','vbn vbd jj'],
'featureless':['f-iy1 ch-er l-ah-s','jj'],
'features':['f-iy1 ch-er-z','nns vbz'],
'featuring':['f-iy1 ch-er ih-ng','vbg'],
'fecal':['f-iy1 k-ah-l','jj'],
'feckless':['f-eh1-k l-ih-s','jj'],
'fed':['f-eh1-d','vbn jj nn vbd'],
'federal':['f-eh1 d-er ah-l','jj'],
'federalism':['f-eh1 d-er ah l-ih z-ah-m','nn jj'],
'federalist':['f-eh1 d-er ah l-ih-s-t','nn'],
'federalized':['f-eh1 d-er ah l-ay-z-d','jj'],
'federally':['f-eh1 d-er ah l-iy','rb'],
'federation':['f-eh d-er ey1 sh-ah-n','nn'],
'fedora':['f-ih d-ao1 r-ah','nn'],
'fee':['f-iy1','nn'],
'feeble':['f-iy1 b-ah-l','jj'],
'feebly':['f-iy1 b-l-iy','rb'],
'feed':['f-iy1-d','nn vb vbd vbp'],
'feedback':['f-iy1-d b-ae-k','nn'],
'feeder':['f-iy1 d-er','nn'],
'feeding':['f-iy1 d-ih-ng','vbg nn'],
'feedlot':['f-iy1-d l-aa-t','nn'],
'feeds':['f-iy1-d-z','vbz nns'],
'feedstock':['f-iy1-d s-t-aa-k','nn'],
'feel':['f-iy1-l','vb vbp nn'],
'feelers':['f-iy1 l-er-z','nns'],
'feeling':['f-iy1 l-ih-ng','nn vbg'],
'feels':['f-iy1-l-z','vbz nns'],
'feet':['f-iy1-t','nns nn'],
'feigned':['f-ey1-n-d','jj vbn'],
'feigning':['f-ey1 n-ih-ng','vbg'],
'feint':['f-ey1-n-t','nn vb'],
'feisty':['f-ay1 s-t-iy','jj'],
'felicitous':['f-ih l-ih1 s-ah t-ah-s','jj'],
'felicity':['f-ih l-ih1 s-ah t-iy','nn'],
'feline':['f-iy1 l-ay-n','jj'],
'fell':['f-eh1-l','vbd jj nn vbn'],
'fella':['f-eh1 l-ah','nn uh'],
'felled':['f-eh1-l-d','vbn vbd'],
'feller':['f-eh1 l-er','nn'],
'felling':['f-eh1 l-ih-ng','vbg'],
'fellow':['f-eh1 l-ow','nn jj'],
'fellowship':['f-eh1 l-ow sh-ih-p','nn'],
'felon':['f-eh1 l-ah-n','nn'],
'felony':['f-eh1 l-ah n-iy','nn'],
'felt':['f-eh1-l-t','vbd vbn'],
'female':['f-iy1 m-ey-l','jj nn'],
'feminine':['f-eh1 m-ah n-ah-n','jj nn'],
'femininity':['f-eh m-ah n-ih1 n-ah t-iy','nn'],
'feminism':['f-eh1 m-ih n-ih z-ah-m','nn'],
'feminist':['f-eh1 m-ah n-ah-s-t','jj nn'],
'fence':['f-eh1-n-s','nn'],
'fenced':['f-eh1-n-s-t','jj vbn'],
'fencing':['f-eh1-n s-ih-ng','nn'],
'fend':['f-eh1-n-d','vb'],
'fended':['f-eh1-n d-ih-d','vbd vbn'],
'fender':['f-eh1-n d-er','nn'],
'fending':['f-eh1-n d-ih-ng','vbg'],
'fennel':['f-eh1 n-ah-l','nn'],
'fenugreek':['f-eh1 n-uw g-r-iy-k','nn'],
'fer':['f-er1','in'],
'ferment':['f-er m-eh1-n-t','nn'],
'fermentation':['f-er m-ah-n t-ey1 sh-ah-n','jj nn'],
'fermented':['f-er m-eh1-n t-ah-d','vbn'],
'fermenting':['f-er m-eh1-n t-ih-ng','vbg'],
'fern':['f-er1-n','nn'],
'fernery':['f-er1 n-er iy','nn'],
'ferocious':['f-er ow1 sh-ah-s','jj'],
'ferociously':['f-er ow1 sh-ih-s l-iy','rb'],
'ferocity':['f-er aa1 s-ah t-iy','nn'],
'ferret':['f-eh1 r-ah-t','vb nn'],
'ferreted':['f-eh1 r-ah t-ah-d','vbd vbn'],
'ferreting':['f-eh1 r-ah t-ih-ng','vbg'],
'ferrets':['f-eh1 r-ah-t-s','vbz'],
'ferried':['f-eh1 r-iy-d','vbd vbn'],
'ferries':['f-eh1 r-iy-z','nns'],
'ferris':['f-eh1 r-ih-s','jj'],
'ferromagnetic':['f-eh r-ow m-ae-g n-eh1 t-ih-k','jj'],
'ferry':['f-eh1 r-iy','nn vb'],
'ferrying':['f-eh1 r-iy ih-ng','vbg'],
'fertile':['f-er1 t-ah-l','jj'],
'fertility':['f-er t-ih1 l-ah t-iy','nn'],
'fertilization':['f-er t-ah l-ih z-ey1 sh-ah-n','nn'],
'fertilized':['f-er1 t-ah l-ay-z-d','vbn jj'],
'fertilizer':['f-er1 t-ah l-ay z-er','nn'],
'fertilizing':['f-er1 t-ah l-ay z-ih-ng','vbg'],
'fervent':['f-er1 v-ah-n-t','jj'],
'fervently':['f-er1 v-ah-n-t l-iy','rb'],
'fervid':['f-er1 v-ah-d','nn'],
'fervor':['f-er1 v-er','nn'],
'fess':['f-eh1-s','nn'],
'fest':['f-eh1-s-t','nn jjs'],
'fester':['f-eh1 s-t-er','vb'],
'festering':['f-eh1 s-t-er ih-ng','vbg'],
'festival':['f-eh1 s-t-ah v-ah-l','nn'],
'festive':['f-eh1 s-t-ih-v','jj'],
'festivity':['f-eh s-t-ih1 v-ah t-iy','nn'],
'festooned':['f-eh s-t-uw1-n-d','vbn'],
'fetal':['f-iy1 t-ah-l','jj'],
'fetch':['f-eh1-ch','vb vbp'],
'fetched':['f-eh1-ch-t','vbd vbn'],
'fetches':['f-eh1 ch-ih-z','vbz'],
'fetching':['f-eh1 ch-ih-ng','vbg jj'],
'fete':['f-ey1-t','vb'],
'feted':['f-ey1 t-ih-d','vbd jj vbn'],
'fetid':['f-eh1 t-ah-d','jj'],
'fetish':['f-eh1 t-ih-sh','nn'],
'fettered':['f-eh1 t-er-d','vbn'],
'fetus':['f-iy1 t-ah-s','nn'],
'feud':['f-y-uw1-d','nn'],
'feudal':['f-y-uw1 d-ah-l','jj'],
'feudalism':['f-y-uw1 d-ah l-ih z-ah-m','nn'],
'feudalistic':['f-y-uw d-ah l-ih1 s-t-ih-k','jj'],
'feuded':['f-y-uw1 d-ih-d','vbd'],
'feuding':['f-y-uw1 d-ih-ng','vbg nn'],
'feuds':['f-y-uw1-d-z','nns vbz'],
'fever':['f-iy1 v-er','nn'],
'fevered':['f-iy1 v-er-d','jj'],
'feverish':['f-iy1 v-er ih-sh','jj'],
'feverishly':['f-eh1 v-er ih-sh l-iy','rb'],
'few':['f-y-uw1','jj nn rb nns'],
'fewer':['f-y-uw1 er','jjr rb rbr'],
'fewest':['f-y-uw1 ah-s-t','jjs'],
'fiance':['f-iy aa1-n s-ey','nn'],
'fiancee':['f-iy ae1-n s-iy','nn'],
'fiasco':['f-iy ae1-s k-ow','nn'],
'fiat':['f-ay1 ae-t','nn'],
'fiber':['f-ay1 b-er','nn'],
'fiberglas':['f-ay1 b-er g-l-ae-s','nns'],
'fiberglass':['f-ay1 b-er g-l-ae-s','nns nn'],
'fiberoptic':['f-ay1 b-er aa-p t-ih-k','jj'],
'fibrillation':['f-ih b-r-ih l-ey1 sh-ah-n','nn'],
'fibrin':['f-ay1 b-r-ah-n','nn'],
'fibrosis':['f-ay b-r-ow1 s-ah-s','nn'],
'fibrous':['f-ay1 b-r-ah-s','jj'],
'fickle':['f-ih1 k-ah-l','jj'],
'fickleness':['f-ih1 k-ah-l n-ah-s','nn'],
'fiction':['f-ih1-k sh-ah-n','nn'],
'fictional':['f-ih1-k sh-ah n-ah-l','jj'],
'fictionalized':['f-ih1-k sh-ah n-ah-l ay-z-d','vbn'],
'fictitious':['f-ih-k t-ih1 sh-ah-s','jj'],
'fiddle':['f-ih1 d-ah-l','nn vb'],
'fiddling':['f-ih1-d l-ih-ng','nn jj vbg'],
'fidelity':['f-ah d-eh1 l-ah t-iy','nn'],
'fidgeting':['f-ih1 jh-ah t-ih-ng','vbg'],
'fiduciary':['f-ah d-uw1 sh-iy eh r-iy','jj'],
'fiefdom':['f-iy1-f d-ah-m','nn'],
'field':['f-iy1-l-d','nn jj vb vbp'],
'fielded':['f-iy1-l d-ih-d','vbd vbn'],
'fielder':['f-iy1-l d-er','nn'],
'fielding':['f-iy1-l d-ih-ng','nn vbg'],
'fields':['f-iy1-l-d-z','nns vbz'],
'fieldwork':['f-iy1-l-d w-er-k','nn'],
'fiend':['f-iy1-n-d','nn'],
'fiendish':['f-iy1-n d-ih-sh','jj'],
'fierce':['f-ih1-r-s','jj'],
'fiercely':['f-ih1-r-s l-iy','rb'],
'fiercer':['f-ih1-r s-er','jjr'],
'fiercest':['f-ih1-r s-ah-s-t','jjs'],
'fiery':['f-ay1 er iy','jj'],
'fiesta':['f-iy eh1 s-t-ah','nn'],
'fifteenth':['f-ih-f t-iy1-n-th','jj'],
'fifth':['f-ih1-f-th','jj nn rb'],
'fiftieth':['f-ih1-f t-iy ih-th','jj'],
'fig':['f-ih1-g','nn'],
'fight':['f-ay1-t','nn vb vbp'],
'fighter':['f-ay1 t-er','nn'],
'fighting':['f-ay1 t-ih-ng','vbg nn jj'],
'fights':['f-ay1-t-s','nns vbz'],
'figurative':['f-ih1 g-y-er ah t-ih-v','jj'],
'figuratively':['f-ih g-y-uh1 r-ah t-ih-v l-iy','rb'],
'figure':['f-ih1 g-y-er','nn vb vbp vbz'],
'figured':['f-ih1 g-y-er-d','vbd jj vbn'],
'figurehead':['f-ih1 g-y-er hh-eh-d','nn'],
'figures':['f-ih1 g-y-er-z','nns vbz'],
'figuring':['f-ih1 g-y-er ih-ng','vbg'],
'filament':['f-ih1 l-ah m-ah-n-t','nn'],
'filbert':['f-ih1-l b-er-t','jj'],
'filched':['f-ih1-l-ch-t','vbd vbn'],
'file':['f-ay1-l','vb nn vbp'],
'filed':['f-ay1-l-d','vbn vbd'],
'filers':['f-ay1 l-er-z','nns'],
'files':['f-ay1-l-z','nns vbz'],
'filet':['f-ih l-ey1','nn'],
'filial':['f-ih1 l-iy ah-l','jj'],
'filibuster':['f-ih1 l-ah b-ah s-t-er','nn vb'],
'filibusters':['f-ih1 l-ah b-ah s-t-er-z','nns'],
'filigree':['f-ih1 l-ah g-r-iy','jj nn'],
'filing':['f-ay1 l-ih-ng','nn vbg'],
'fill':['f-ih1-l','vb vbp nn'],
'filled':['f-ih1-l-d','vbn jj vbd'],
'filler':['f-ih1 l-er','nn'],
'filleted':['f-ih1 l-ah t-ih-d','vbn'],
'filling':['f-ih1 l-ih-ng','vbg nn'],
'fillip':['f-ih1 l-ah-p','nn'],
'fills':['f-ih1-l-z','vbz'],
'filly':['f-ih1 l-iy','nn'],
'film':['f-ih1-l-m','nn vb'],
'filmed':['f-ih1-l-m-d','vbn vbd jj'],
'filming':['f-ih1-l m-ih-ng','vbg'],
'filmmakers':['f-ih1-l m-ey k-er-z','nns'],
'filmmaking':['f-ih1-l m-ey k-ih-ng','nn'],
'films':['f-ih1-l-m-z','nns'],
'filter':['f-ih1-l t-er','nn vb'],
'filtered':['f-ih1-l t-er-d','vbn jj'],
'filtering':['f-ih1-l t-er ih-ng','vbg nn'],
'filters':['f-ih1-l t-er-z','nns vbz'],
'filth':['f-ih1-l-th','nn'],
'filthy':['f-ih1-l th-iy','jj'],
'filtration':['f-ih-l t-r-ey1 sh-ah-n','nn'],
'fin':['f-ih1-n','nn'],
'final':['f-ay1 n-ah-l','jj'],
'finale':['f-ah n-ae1 l-iy','nn'],
'finalist':['f-ay1 n-ah l-ih-s-t','nn'],
'finality':['f-ay n-ae1 l-ah t-iy','nn'],
'finalized':['f-ay1 n-ah l-ay-z-d','vbn vbd'],
'finalizing':['f-ay1 n-ah l-ay z-ih-ng','vbg'],
'finally':['f-ay1 n-ah l-iy','rb'],
'finance':['f-ah n-ae1-n-s','nn vbp vb'],
'financed':['f-ih n-ae1-n-s-t','vbn jj vbd'],
'finances':['f-ih n-ae1-n s-ih-z','nns vbz'],
'financial':['f-ah n-ae1-n sh-ah-l','jj'],
'financially':['f-ah n-ae1-n sh-ah l-iy','rb'],
'financier':['f-ih n-ah-n s-ih1-r','nn'],
'financing':['f-ah n-ae1-n s-ih-ng','nn jj vbg'],
'find':['f-ay1-n-d','vb vbp nn'],
'finder':['f-ay1-n d-er','nn'],
'finding':['f-ay1-n d-ih-ng','vbg nn'],
'finds':['f-ay1-n-d-z','vbz nns'],
'fine':['f-ay1-n','jj nn rb vb'],
'fined':['f-ay1-n-d','vbn vbd'],
'finely':['f-ay1-n l-iy','rb'],
'finer':['f-ay1 n-er','jjr'],
'finery':['f-ay1 n-er iy','nn'],
'fines':['f-ay1-n-z','nns nn'],
'finesse':['f-ih n-eh1-s','nn'],
'finessed':['f-ih n-eh1-s-t','vbd vbn'],
'finest':['f-ay1 n-ah-s-t','jjs'],
'finger':['f-ih1-ng g-er','nn vb'],
'fingered':['f-ih1-ng g-er-d','vbd vbn'],
'fingering':['f-ih1-ng g-er ih-ng','vbg nn'],
'fingernails':['f-ih1-ng g-er n-ey-l-z','nns'],
'fingerprint':['f-ih1-ng g-er p-r-ih-n-t','nn'],
'fingerprinting':['f-ih1-ng g-er p-r-ih-n t-ih-ng','nn vbg'],
'fingers':['f-ih1-ng g-er-z','nns'],
'fingertips':['f-ih1-ng g-er t-ih-p-s','nns'],
'finicky':['f-ih1 n-ah k-iy','jj'],
'fining':['f-ay1 n-ih-ng','vbg'],
'finish':['f-ih1 n-ih-sh','vb nn vbp'],
'finished':['f-ih1 n-ih-sh-t','vbd jj vb vbn'],
'finisher':['f-ih1 n-ih sh-er','nn'],
'finishes':['f-ih1 n-ih sh-ih-z','vbz nns'],
'finishing':['f-ih1 n-ih sh-ih-ng','vbg jj nn'],
'finite':['f-ay1 n-ay-t','jj nn'],
'fir':['f-er1','nn'],
'fire':['f-ay1 er','nn vb'],
'firearm':['f-ay1 er aa-r-m','nn'],
'fireball':['f-ay1 er b-ao-l','nn'],
'firebombed':['f-ay1-r b-aa-m-d','vbn'],
'firebombing':['f-ay1 er b-aa m-ih-ng','nn'],
'firebombs':['f-ay1-r b-aa-m-z','nns'],
'firebrand':['f-ay1 er b-r-ae-n-d','nn'],
'firebug':['f-ay1 er b-ah-g','nn'],
'firecracker':['f-ay1 er k-r-ae k-er','nn'],
'fired':['f-ay1 er-d','vbn vbd'],
'firefighter':['f-ay1-r f-ay t-er','nn'],
'firefighting':['f-ay1-r f-ay t-ih-ng','nn'],
'firehouses':['f-ay1 er hh-aw s-ih-z','nns'],
'fireman':['f-ay1-r m-ah-n','nn'],
'fireplace':['f-ay1 er p-l-ey-s','nn'],
'firepower':['f-ay1-r p-aw er','nn'],
'fireproofing':['f-ay1 er p-r-uw f-ih-ng','nn vbg'],
'fires':['f-ay1 er-z','nns vbz'],
'firestorm':['f-ay1-r s-t-ao-r-m','nn'],
'fireweed':['f-ay1 er w-iy-d','nn'],
'firewood':['f-ay1 er w-uh-d','nn'],
'fireworks':['f-ay1-r w-er-k-s','nns'],
'firing':['f-ay1 r-ih-ng','vbg jj nn'],
'firm':['f-er1-m','nn jj vb'],
'firmed':['f-er1-m-d','vbd vbn'],
'firmer':['f-er1 m-er','jjr rb rbr'],
'firming':['f-er1 m-ih-ng','vbg nn'],
'firmly':['f-er1-m l-iy','rb'],
'firmness':['f-er1-m n-ah-s','nn'],
'firms':['f-er1-m-z','nns vbz'],
'firsthand':['f-er-s-t hh-ae1-n-d','rb jj nn'],
'firsts':['f-er1-s-t-s','nns'],
'fiscal':['f-ih1-s k-ah-l','jj in nn'],
'fiscally':['f-ih1-s k-ah l-iy','rb'],
'fish':['f-ih1-sh','nn vb nns'],
'fishbowl':['f-ih1-sh b-ow-l','nn'],
'fished':['f-ih1-sh-t','vbn'],
'fisherman':['f-ih1 sh-er m-ae-n','nn'],
'fishers':['f-ih1 sh-er-z','nns'],
'fishery':['f-ih1 sh-er iy','nn'],
'fishes':['f-ih1 sh-ah-z','nns vbz'],
'fishing':['f-ih1 sh-ih-ng','nn vbg jj'],
'fishy':['f-ih1 sh-iy','jj'],
'fission':['f-ih1 sh-ah-n','nn'],
'fissionable':['f-ih1 sh-ah-n ah b-ah-l','jj'],
'fissured':['f-ih1 sh-er-d','vbn'],
'fissures':['f-ih1 sh-er-z','nns'],
'fist':['f-ih1-s-t','nn rb'],
'fisted':['f-ih1 s-t-ih-d','vbd'],
'fit':['f-ih1-t','vb vbn vbp jj nn rb vbd'],
'fitful':['f-ih1-t f-ah-l','jj'],
'fitfully':['f-ih1-t f-ah l-iy','rb'],
'fitness':['f-ih1-t n-ah-s','nn'],
'fits':['f-ih1-t-s','vbz nns'],
'fitted':['f-ih1 t-ah-d','vbn vbd jj'],
'fittest':['f-ih1 t-ah-s-t','jjs'],
'fitting':['f-ih1 t-ih-ng','jj nn vbg'],
'fivefold':['f-ay1-v f-ow-l-d','jj rb'],
'fix':['f-ih1-k-s','vb vbp nn'],
'fixable':['f-ih1-k s-ah b-ah-l','jj'],
'fixated':['f-ih1-k s-ey t-ih-d','vbn'],
'fixation':['f-ih-k s-ey1 sh-ah-n','nn'],
'fixed':['f-ih1-k-s-t','vbn jj vbd'],
'fixes':['f-ih1-k s-ih-z','nns'],
'fixing':['f-ih1-k s-ih-ng','vbg nn'],
'fixture':['f-ih1-k-s ch-er','nn'],
'fizzled':['f-ih1 z-ah-l-d','vbd vbn'],
'fizzles':['f-ih1 z-ah-l-z','vbz'],
'fjords':['f-y-ao1-r-d-z','nns'],
'flabbergasted':['f-l-ae1 b-er g-ae s-t-ih-d','jj vbn'],
'flaccid':['f-l-ae1-k s-ih-d','jj'],
'flag':['f-l-ae1-g','nn vb'],
'flagellated':['f-l-ae1 jh-ah l-ey t-ah-d','vbn'],
'flageolet':['f-l-ae jh-ah l-eh1-t','nn'],
'flagged':['f-l-ae1-g-d','vbd'],
'flagging':['f-l-ae1 g-ih-ng','jj vbg'],
'flagpole':['f-l-ae1-g p-ow-l','nn'],
'flagrant':['f-l-ey1 g-r-ah-n-t','jj'],
'flagrantly':['f-l-ey1 g-r-ah-n-t l-iy','rb'],
'flags':['f-l-ae1-g-z','nns vbz'],
'flagship':['f-l-ae1-g sh-ih-p','nn jj'],
'flail':['f-l-ey1-l','nn'],
'flailing':['f-l-ey1 l-ih-ng','vbg'],
'flair':['f-l-eh1-r','nn'],
'flak':['f-l-ae1-k','nn'],
'flakes':['f-l-ey1-k-s','nns'],
'flaky':['f-l-ey1 k-iy','jj'],
'flamboyant':['f-l-ae-m b-oy1 ah-n-t','jj'],
'flamboyantly':['f-l-ae-m b-oy1 ah-n-t l-iy','rb'],
'flame':['f-l-ey1-m','nn vbp vb'],
'flamed':['f-l-ey1-m-d','vbd vbn'],
'flames':['f-l-ey1-m-z','nns'],
'flaming':['f-l-ey1 m-ih-ng','jj vbg'],
'flammable':['f-l-ae1 m-ah b-ah-l','jj'],
'flange':['f-l-ae1-n-jh','nn'],
'flank':['f-l-ae1-ng-k','nn'],
'flanked':['f-l-ae1-ng-k-t','vbd vbn'],
'flanking':['f-l-ae1-ng k-ih-ng','vbg'],
'flannel':['f-l-ae1 n-ah-l','nn'],
'flap':['f-l-ae1-p','nn'],
'flappers':['f-l-ae1 p-er-z','nns'],
'flapping':['f-l-ae1 p-ih-ng','vbg jj'],
'flare':['f-l-eh1-r','nn vb vbp'],
'flared':['f-l-eh1-r-d','vbd vbn jj'],
'flares':['f-l-eh1-r-z','nns vbz'],
'flaring':['f-l-eh1 r-ih-ng','vbg jj'],
'flash':['f-l-ae1-sh','nn vbp jj vb'],
'flashback':['f-l-ae1-sh b-ae-k','nn'],
'flashed':['f-l-ae1-sh-t','vbd vbn'],
'flasher':['f-l-ae1 sh-er','nn'],
'flashes':['f-l-ae1 sh-ih-z','nns vbz'],
'flashier':['f-l-ae1 sh-iy er','jjr'],
'flashing':['f-l-ae1 sh-ih-ng','vbg nn'],
'flashlight':['f-l-ae1-sh l-ay-t','nn'],
'flashpoint':['f-l-ae1-sh p-oy-n-t','nn'],
'flashy':['f-l-ae1 sh-iy','jj'],
'flask':['f-l-ae1-s-k','nn'],
'flat':['f-l-ae1-t','jj nn rb'],
'flathead':['f-l-ae1-t hh-eh-d','jj'],
'flatland':['f-l-ae1-t l-ae-n-d','nn'],
'flatly':['f-l-ae1-t l-iy','rb'],
'flatness':['f-l-ae1-t n-ah-s','nn'],
'flatten':['f-l-ae1 t-ah-n','vb'],
'flattened':['f-l-ae1 t-ah-n-d','vbn vbd jj'],
'flattening':['f-l-ae1 t-ah n-ih-ng','vbg nn'],
'flatter':['f-l-ae1 t-er','vb'],
'flattered':['f-l-ae1 t-er-d','vbn jj vbd'],
'flattering':['f-l-ae1 t-er ih-ng','jj'],
'flattery':['f-l-ae1 t-er iy','nn'],
'flattish':['f-l-ae1 t-ih-sh','jj'],
'flatulent':['f-l-ae1 ch-ah l-ah-n-t','jj'],
'flatware':['f-l-ae1-t w-eh-r','nn'],
'flaunt':['f-l-ao1-n-t','vb vbp'],
'flaunted':['f-l-ao1-n t-ih-d','vbd'],
'flaunting':['f-l-ao1-n t-ih-ng','vbg'],
'flaunts':['f-l-ao1-n-t-s','vbz'],
'flavor':['f-l-ey1 v-er','nn'],
'flavored':['f-l-ey1 v-er-d','jj vbn'],
'flavorful':['f-l-ey1 v-er f-ah-l','jj'],
'flavoring':['f-l-ey1 v-er ih-ng','nn vbg'],
'flaw':['f-l-ao1','nn'],
'flawed':['f-l-ao1-d','jj vbn'],
'flawless':['f-l-ao1 l-ah-s','jj'],
'flax':['f-l-ae1-k-s','nn'],
'flaxseed':['f-l-ae1-k s-iy-d','nn'],
'flea':['f-l-iy1','nn'],
'fleck':['f-l-eh1-k','nn'],
'fled':['f-l-eh1-d','vbd vbn'],
'fledging':['f-l-eh1 jh-ih-ng','vbg'],
'fledgling':['f-l-eh1-jh l-ih-ng','nn jj'],
'flee':['f-l-iy1','vb'],
'fleece':['f-l-iy1-s','nn'],
'fleeced':['f-l-iy1-s-t','vbn'],
'fleeing':['f-l-iy1 ih-ng','vbg'],
'flees':['f-l-iy1-z','vbz'],
'fleet':['f-l-iy1-t','nn jj'],
'fleeting':['f-l-iy1 t-ih-ng','jj'],
'fleets':['f-l-iy1-t-s','nns vbz'],
'flesh':['f-l-eh1-sh','nn vb'],
'fleshy':['f-l-eh1 sh-iy','jj'],
'flew':['f-l-uw1','vbd'],
'flex':['f-l-eh1-k-s','nn jj vb'],
'flexed':['f-l-eh1-k-s-t','vbd'],
'flexibility':['f-l-eh-k s-ah b-ih1 l-ah t-iy','nn'],
'flexible':['f-l-eh1-k s-ah b-ah-l','jj'],
'flexing':['f-l-eh1-k s-ih-ng','vbg'],
'flextime':['f-l-eh1-k s-t-ay-m','nn'],
'flick':['f-l-ih1-k','nn'],
'flicker':['f-l-ih1 k-er','nn vb'],
'flickered':['f-l-ih1 k-er-d','vbd vbn'],
'flickering':['f-l-ih1 k-er ih-ng','vbg'],
'flicking':['f-l-ih1 k-ih-ng','vbg'],
'flied':['f-l-ay1-d','vbd'],
'flier':['f-l-ay1 er','nn'],
'flies':['f-l-ay1-z','vbz nns'],
'flight':['f-l-ay1-t','nn'],
'flighty':['f-l-ay1 t-iy','jj'],
'flimflam':['f-l-ih1-m f-l-ae-m','nn'],
'flimsy':['f-l-ih1-m z-iy','jj'],
'flinch':['f-l-ih1-n-ch','vb'],
'flinched':['f-l-ih1-n-ch-t','vbd'],
'fling':['f-l-ih1-ng','nn vb'],
'flinging':['f-l-ih1 ng-ih-ng','vbg'],
'flings':['f-l-ih1-ng-z','nns'],
'flint':['f-l-ih1-n-t','nn'],
'flinty':['f-l-ih1-n t-iy','jj'],
'flip':['f-l-ih1-p','jj vb'],
'flippant':['f-l-ih1 p-ah-n-t','jj'],
'flipped':['f-l-ih1-p-t','vbd'],
'flippers':['f-l-ih1 p-er-z','nns'],
'flipping':['f-l-ih1 p-ih-ng','vbg jj rb'],
'flips':['f-l-ih1-p-s','vbz nns'],
'flirt':['f-l-er1-t','vbp'],
'flirtation':['f-l-er t-ey1 sh-ah-n','nn'],
'flirtatious':['f-l-er t-ey1 sh-ah-s','jj'],
'flirted':['f-l-er1 t-ih-d','vbd vbn'],
'flirting':['f-l-er1 t-ih-ng','vbg'],
'flit':['f-l-ih1-t','vbp'],
'flitting':['f-l-ih1 t-ih-ng','vbg'],
'float':['f-l-ow1-t','vb nn vbp'],
'floated':['f-l-ow1 t-ah-d','vbd vbn'],
'floater':['f-l-ow1 t-er','nn'],
'floating':['f-l-ow1 t-ih-ng','vbg jj'],
'floats':['f-l-ow1-t-s','vbz'],
'floc':['f-l-aa1-k','nn'],
'flock':['f-l-aa1-k','nn vb vbp'],
'flocked':['f-l-aa1-k-t','vbd vbn'],
'flocking':['f-l-aa1 k-ih-ng','vbg'],
'flocks':['f-l-aa1-k-s','nns'],
'floe':['f-l-ow1','nn'],
'flood':['f-l-ah1-d','nn vb vbp'],
'flooded':['f-l-ah1 d-ah-d','vbn vbd'],
'flooding':['f-l-ah1 d-ih-ng','vbg nn'],
'floods':['f-l-ah1-d-z','nns'],
'floor':['f-l-ao1-r','nn'],
'floorboards':['f-l-ao1-r b-ao-r-d-z','nns'],
'flooring':['f-l-ao1 r-ih-ng','nn'],
'flop':['f-l-aa1-p','nn vbp vb'],
'flopped':['f-l-aa1-p-t','vbd vbn'],
'floppy':['f-l-aa1 p-iy','jj'],
'flops':['f-l-aa1-p-s','vbz nns'],
'flora':['f-l-ao1 r-ah','nns nn'],
'floral':['f-l-ao1 r-ah-l','jj'],
'florid':['f-l-ao1 r-ah-d','jj'],
'florist':['f-l-aa1 r-ah-s-t','nn'],
'flotation':['f-l-ow t-ey1 sh-ah-n','nn'],
'flotilla':['f-l-ow t-ih1 l-ah','nn'],
'flounder':['f-l-aw1-n d-er','vb'],
'floundered':['f-l-aw1-n d-er-d','vbn vbd'],
'floundering':['f-l-aw1-n d-er ih-ng','vbg jj'],
'flounders':['f-l-aw1-n d-er-z','vbz'],
'flour':['f-l-aw1 er','nn'],
'flourish':['f-l-er1 ih-sh','vb nn vbp'],
'flourished':['f-l-er1 ih-sh-t','vbd vbn'],
'flourishes':['f-l-er1 ih sh-ah-z','nns vbz'],
'flourishing':['f-l-er1 ih sh-ih-ng','vbg jj'],
'flout':['f-l-aw1-t','vb'],
'flouted':['f-l-aw1 t-ih-d','vbn'],
'flouting':['f-l-aw1 t-ih-ng','nn vbg'],
'flow':['f-l-ow1','nn vbp vb'],
'flowed':['f-l-ow1-d','vbd vbn'],
'flower':['f-l-aw1 er','nn vbp'],
'flowered':['f-l-aw1 er-d','jj vbd vbn'],
'flowering':['f-l-aw1 er ih-ng','nn vbg'],
'flowerpot':['f-l-aw1 er p-aa-t','nn'],
'flowing':['f-l-ow1 ih-ng','vbg jj'],
'flown':['f-l-ow1-n','vbn'],
'flows':['f-l-ow1-z','nns vbz'],
'flu':['f-l-uw1','nn'],
'fluctuate':['f-l-ah1-k ch-ah-w ey-t','vbp vb'],
'fluctuated':['f-l-ah1-k ch-ah-w ey t-ih-d','vbd vbn'],
'fluctuates':['f-l-ah1-k ch-uw ey-t-s','vbz'],
'fluctuating':['f-l-ah1-k ch-ah-w ey t-ih-ng','vbg'],
'fluctuation':['f-l-ah-k ch-uw ey1 sh-ah-n','nn'],
'flue':['f-l-uw1','nn'],
'fluency':['f-l-uw1 ah-n s-iy','nn'],
'fluent':['f-l-uw1 ah-n-t','jj'],
'fluff':['f-l-ah1-f','nn'],
'fluffy':['f-l-ah1 f-iy','jj'],
'fluid':['f-l-uw1 ah-d','nn jj'],
'fluidity':['f-l-uw ih1 d-ah t-iy','nn'],
'fluke':['f-l-uw1-k','nn'],
'flung':['f-l-ah1-ng','vbd vbn'],
'flunk':['f-l-ah1-ng-k','vbp'],
'flunked':['f-l-ah1-ng-k-t','vbd vbn'],
'flunking':['f-l-ah1-ng k-ih-ng','vbg'],
'fluorescent':['f-l-uh r-eh1 s-ah-n-t','jj'],
'fluoride':['f-l-uh1 r-ay-d','nn'],
'fluorine':['f-l-uh1 r-iy-n','nn'],
'flurried':['f-l-er1 iy-d','vbd'],
'flurry':['f-l-er1 iy','nn vbp'],
'flush':['f-l-ah1-sh','jj nn rb vb vbp'],
'flushed':['f-l-ah1-sh-t','vbn vbd'],
'flushes':['f-l-ah1 sh-ih-z','vbz'],
'flushing':['f-l-ah1 sh-ih-ng','nn vbg'],
'flustered':['f-l-ah1 s-t-er-d','vbn'],
'flute':['f-l-uw1-t','nn'],
'flutist':['f-l-uw1 t-ih-s-t','nn'],
'flutter':['f-l-ah1 t-er','nn'],
'fluttered':['f-l-ah1 t-er-d','vbd'],
'fluttering':['f-l-ah1 t-er ih-ng','vbg'],
'flux':['f-l-ah1-k-s','nn'],
'fly':['f-l-ay1','vb nn vbp'],
'flyer':['f-l-ay1 er','nn'],
'flying':['f-l-ay1 ih-ng','vbg nn jj'],
'foal':['f-ow1-l','nn'],
'foam':['f-ow1-m','nn vb vbp'],
'foams':['f-ow1-m-z','nns'],
'foamy':['f-ow1 m-iy','jj'],
'focal':['f-ow1 k-ah-l','jj'],
'focus':['f-ow1 k-ah-s','nn vbp vb'],
'focused':['f-ow1 k-ah-s-t','vbn jj vbd'],
'focuses':['f-ow1 k-ah s-ih-z','vbz nns'],
'focusing':['f-ow1 k-ah s-ih-ng','vbg nn'],
'focussed':['f-ow1 k-ah-s-t','vbn vbd'],
'fodder':['f-aa1 d-er','nn'],
'foe':['f-ow1','nn'],
'fog':['f-aa1-g','nn vb'],
'foggy':['f-aa1 g-iy','jj'],
'fogy':['f-ow1 g-iy','nn'],
'foibles':['f-oy1 b-ah-l-z','nns'],
'foil':['f-oy1-l','nn vb'],
'foiled':['f-oy1-l-d','vbn jj vbd'],
'foiling':['f-oy1 l-ih-ng','vbg'],
'foisted':['f-oy1 s-t-ih-d','vbd vbn'],
'fold':['f-ow1-l-d','vb nn vbp'],
'folded':['f-ow1-l d-ah-d','vbn vbd jj'],
'folder':['f-ow1-l d-er','nn'],
'folding':['f-ow1-l d-ih-ng','vbg jj nn'],
'folds':['f-ow1-l-d-z','nns vbz'],
'foliage':['f-ow1 l-ih-jh','nn'],
'folk':['f-ow1-k','nn nns'],
'folkish':['f-ow1 k-ih-sh','jj'],
'folklore':['f-ow1-k l-ao-r','nn'],
'folksy':['f-ow1-k s-iy','jj'],
'follow':['f-aa1 l-ow','vb vbp'],
'followed':['f-aa1 l-ow-d','vbd vbn jj'],
'follower':['f-aa1 l-ow er','nn'],
'following':['f-aa1 l-ow ih-ng','vbg jj nn'],
'follows':['f-aa1 l-ow-z','vbz'],
'followup':['f-aa1 l-ow ah-p','jj'],
'folly':['f-aa1 l-iy','nn'],
'foment':['f-ow1 m-eh-n-t','vb'],
'fomented':['f-ow1 m-eh-n t-ih-d','vbd'],
'fomenting':['f-ow1 m-eh-n t-ih-ng','vbg'],
'fond':['f-aa1-n-d','jj nn vb'],
'fonder':['f-aa1-n d-er','jjr'],
'fondest':['f-aa1-n d-ah-s-t','jjs'],
'fondly':['f-aa1-n-d l-iy','rb'],
'fondness':['f-aa1-n-d n-ah-s','nn'],
'font':['f-aa1-n-t','nn'],
'food':['f-uw1-d','nn'],
'foodservice':['f-uw1-d s-er1 v-ih-s','nn'],
'foodstuff':['f-uw1-d s-t-ah-f','nn'],
'fool':['f-uw1-l','nn jj vb'],
'fooled':['f-uw1-l-d','vbn'],
'foolhardy':['f-uw1-l hh-aa-r d-iy','jj'],
'fooling':['f-uw1 l-ih-ng','vbg nn'],
'foolish':['f-uw1 l-ih-sh','jj'],
'foolishly':['f-uw1 l-ih-sh l-iy','rb'],
'foolishness':['f-uw1 l-ih-sh n-ah-s','nn'],
'foolproof':['f-uw1-l p-r-uw-f','jj'],
'fools':['f-uw1-l-z','nns'],
'foot':['f-uh1-t','nn vbp jj vb'],
'footage':['f-uh1 t-ih-jh','nn'],
'football':['f-uh1-t b-ao-l','nn'],
'footfall':['f-uh1-t f-ao-l','nn'],
'foothill':['f-uh1-t hh-ih-l','nn'],
'foothold':['f-uh1-t hh-ow-l-d','nn'],
'footing':['f-uh1 t-ih-ng','nn vbg'],
'footlights':['f-uh1-t l-ay-t-s','nns'],
'footloose':['f-uh1-t l-uw-s','jj'],
'footman':['f-uh1-t m-ah-n','nn'],
'footnote':['f-uh1-t n-ow-t','nn'],
'footnoted':['f-uh1-t n-ow t-ih-d','vbn'],
'footpath':['f-uh1-t p-ae-th','nn'],
'footprint':['f-uh1-t p-r-ih-n-t','nn'],
'footsteps':['f-uh1-t s-t-eh-p-s','nns'],
'footwear':['f-uh1-t w-eh-r','nn'],
'footwork':['f-uh1-t w-er-k','nn'],
'for':['f-ao1-r','in nnp cc jj rb rp'],
'forage':['f-ao1 r-ih-jh','nn jj'],
'foraging':['f-ao1 r-ih jh-ih-ng','nn vbg'],
'foray':['f-ao1 r-ey','nn'],
'forays':['f-ao1 r-ey-z','nns vbz'],
'forbade':['f-er b-ey1-d','vbd'],
'forbearance':['f-ao-r b-eh1 r-ah-n-s','nn'],
'forbid':['f-er b-ih1-d','vb vbp uh'],
'forbidden':['f-ao1-r b-ih d-ah-n','vbn jj'],
'forbidding':['f-er b-ih1 d-ih-ng','vbg jj'],
'forbids':['f-er b-ih1-d-z','vbz'],
'force':['f-ao1-r-s','nn vb nnp vbp'],
'forced':['f-ao1-r-s-t','vbn jj vbd'],
'forceful':['f-ao1-r-s f-ah-l','jj'],
'forcefully':['f-ao1-r-s f-ah l-iy','rb'],
'forcefulness':['f-ao1-r-s f-ah-l n-ah-s','nn'],
'forces':['f-ao1-r s-ih-z','nns vbz'],
'forcibly':['f-ao1-r s-ah b-l-iy','rb'],
'forcing':['f-ao1-r s-ih-ng','vbg'],
'fords':['f-ao1-r-d-z','nns'],
'fore':['f-ao1-r','nn rb'],
'forearm':['f-ao r-aa1-r-m','nn'],
'forebears':['f-ao1-r b-eh-r-z','nns'],
'foreboding':['f-ao-r b-ow1 d-ih-ng','nn jj'],
'forecast':['f-ao1-r k-ae-s-t','nn vbd vbn vbp vb'],
'forecasted':['f-ao1-r k-ae s-t-ih-d','vbn'],
'forecaster':['f-ao1-r k-ae s-t-er','nn'],
'forecasting':['f-ao1-r k-ae s-t-ih-ng','nn jj vbg'],
'forecasts':['f-ao-r k-ae1-s-t-s','nns vbz'],
'foreclose':['f-ao-r k-l-ow1-z','vb'],
'foreclosed':['f-ao-r k-l-ow1-z-d','vbn jj vbd'],
'foreclosing':['f-ao-r k-l-ow1 z-ih-ng','vbg'],
'foreclosure':['f-ao-r k-l-ow1 zh-er','nn'],
'forefathers':['f-ao1-r f-aa dh-er-z','nns'],
'forefinger':['f-ao1-r f-ih-ng g-er','nn'],
'forefront':['f-ao1-r f-r-ah-n-t','nn'],
'forego':['f-ao-r g-ow1','vb vbp'],
'foregoing':['f-ao-r g-ow1 ih-ng','nn jj vbg'],
'foregone':['f-ao1-r g-ao1-n','jj vbn'],
'foreground':['f-ao1-r g-r-aw-n-d','nn'],
'forehead':['f-ao1 r-hh-eh-d','nn'],
'foreign':['f-ao1 r-ah-n','jj'],
'foreigner':['f-ao1 r-ah n-er','nn'],
'foreman':['f-ao1-r m-ah-n','nn'],
'foremost':['f-ao1-r m-ow-s-t','jj rb'],
'forensic':['f-er eh1-n s-ih-k','jj'],
'forerunner':['f-ao1 r-ah n-er','nn'],
'foresaw':['f-ao-r s-ao1','vbd'],
'foresee':['f-ao-r s-iy1','vbp vb'],
'foreseeable':['f-ao-r s-iy1 ah b-ah-l','jj'],
'foreseeing':['f-ao-r s-iy1 ih-ng','vbg'],
'foreseen':['f-ao-r s-iy1-n','vbn'],
'foresees':['f-ao-r s-iy1-z','vbz'],
'foreshadow':['f-ao-r sh-ae1 d-ow','vb vbp'],
'foreshadowed':['f-ao-r sh-ae1 d-ow-d','vbn'],
'foreshadowing':['f-ao-r sh-ae1 d-ow ih-ng','vbg'],
'foresight':['f-ao1-r s-ay-t','nn'],
'forest':['f-ao1 r-ah-s-t','nn'],
'forestall':['f-ao-r s-t-ao1-l','vb'],
'forestalled':['f-ao-r s-t-aa1-l-d','vbn'],
'forestry':['f-ao1 r-ah s-t-r-iy','nn'],
'foretell':['f-ao-r t-eh1-l','vb'],
'forethought':['f-ao1-r th-ao-t','nn'],
'forever':['f-er eh1 v-er','rb'],
'foreword':['f-ao1-r w-er-d','nn'],
'forfeit':['f-ao1-r f-ih-t','vb nn'],
'forfeitable':['f-ao1-r f-ah t-ah b-ah-l','jj'],
'forfeited':['f-ao1-r f-ih t-ih-d','vbn'],
'forfeiting':['f-ao1-r f-ah t-ih-ng','vbg'],
'forfeiture':['f-ao1-r f-ah ch-er','nn'],
'forgave':['f-er g-ey1-v','vbd'],
'forge':['f-ao1-r-jh','vb vbp nn'],
'forged':['f-ao1-r-jh-d','vbn jj vbd'],
'forger':['f-ao1-r jh-er','nn'],
'forgery':['f-ao1-r jh-er iy','nn'],
'forges':['f-ao1-r jh-ih-z','vbz'],
'forget':['f-er g-eh1-t','vb vbp'],
'forgetful':['f-ao-r g-eh1-t f-ah-l','jj'],
'forgets':['f-er g-eh1-t-s','vbz'],
'forgettable':['f-ao-r g-eh1 t-ah b-ah-l','jj'],
'forgetting':['f-er g-eh1 t-ih-ng','vbg'],
'forging':['f-ao1-r jh-ih-ng','vbg'],
'forgive':['f-er g-ih1-v','vb vbp'],
'forgiven':['f-er g-ih1 v-ah-n','vbn'],
'forgiveness':['f-er g-ih1-v n-ah-s','nn'],
'forgiving':['f-er g-ih1 v-ih-ng','vbg jj'],
'forgo':['f-ao-r g-ow1','vb vbp'],
'forgoes':['f-ao-r g-ow1-z','vbz'],
'forgone':['f-ao-r g-aa1-n','jj'],
'forgot':['f-er g-aa1-t','vbd vbn'],
'forgotten':['f-er g-aa1 t-ah-n','vbn jj'],
'forint':['f-ao1 r-ih-n-t','nn'],
'fork':['f-ao1-r-k','nn vb'],
'forked':['f-ao1-r-k-t','jj vbd vbn'],
'forking':['f-ao1-r k-ih-ng','vbg'],
'forklift':['f-ao1-r-k l-ih-f-t','nn'],
'forks':['f-ao1-r-k-s','nns'],
'forlorn':['f-er l-ao1-r-n','jj'],
'form':['f-ao1-r-m','nn vbp jj vb'],
'formal':['f-ao1-r m-ah-l','jj'],
'formaldehyde':['f-ao-r m-ae1-l d-ah hh-ay-d','nn'],
'formalism':['f-ao1-r m-ah l-ih z-ah-m','nn'],
'formality':['f-ao-r m-ae1 l-ah t-iy','nn'],
'formalize':['f-ao1-r m-ah l-ay-z','vb'],
'formalized':['f-ao1-r m-ah l-ay-z-d','jj vbn'],
'formalizes':['f-ao1-r m-ah l-ay z-ih-z','vbz'],
'formally':['f-ao1-r m-ah l-iy','rb'],
'format':['f-ao1-r m-ae-t','nn'],
'formation':['f-ao-r m-ey1 sh-ah-n','nn'],
'formative':['f-ao1-r m-ah t-ih-v','jj nn'],
'formed':['f-ao1-r-m-d','vbn vbd jj'],
'former':['f-ao1-r m-er','jj nn'],
'formerly':['f-ao1-r m-er l-iy','rb'],
'formidable':['f-ao1-r m-ah d-ah b-ah-l','jj'],
'formidably':['f-ao1-r m-ah d-ah b-l-iy','rb'],
'forming':['f-ao1-r m-ih-ng','vbg nn'],
'forms':['f-ao1-r-m-z','nns vbz'],
'formula':['f-ao1-r m-y-ah l-ah','nn'],
'formulaic':['f-ao-r m-y-ah l-ey1 ih-k','jj'],
'formulate':['f-ao1-r m-y-ah l-ey-t','vb'],
'formulated':['f-ao1-r m-y-ah l-ey t-ah-d','vbn vbd'],
'formulates':['f-ao1-r m-y-ah l-ey-t-s','vbz'],
'formulating':['f-ao1-r m-y-ah l-ey t-ih-ng','vbg'],
'formulation':['f-ao-r m-y-ah l-ey1 sh-ah-n','nn'],
'forsake':['f-ao-r s-ey1-k','vb'],
'forsaken':['f-ao-r s-ey1 k-ah-n','vbn'],
'forseeable':['f-ao-r s-iy1 ah b-ah-l','jj'],
'fort':['f-ao1-r-t','nn vb'],
'forte':['f-ao1-r t-ey','nn'],
'forth':['f-ao1-r-th','rb rp'],
'forthcoming':['f-ao1-r-th k-ah1 m-ih-ng','jj vbg'],
'forthright':['f-ao1-r-th r-ay1-t','jj'],
'forthrightly':['f-ao1-r-th r-ay1-t l-iy','rb'],
'forthrightness':['f-ao1-r-th r-ay1-t n-ah-s','nn'],
'forthwith':['f-ao1-r-th w-ih1-th','rb'],
'fortifications':['f-ao-r t-ah f-ah k-ey1 sh-ah-n-z','nns'],
'fortified':['f-ao1-r t-ah f-ay-d','vbn vbd jj'],
'fortify':['f-ao1-r t-ih f-ay','vb'],
'fortitude':['f-ao1-r t-ih t-uw-d','nn'],
'fortnight':['f-ao1-r-t n-ay-t','nn'],
'fortress':['f-ao1-r t-r-ah-s','nn'],
'forts':['f-ao1-r-t-s','nns'],
'fortuitous':['f-ao-r t-uw1 ih t-ah-s','jj'],
'fortunate':['f-ao1-r ch-ah n-ah-t','jj'],
'fortunately':['f-ao1-r ch-ah n-ah-t l-iy','rb'],
'fortune':['f-ao1-r ch-ah-n','nn'],
'forum':['f-ao1 r-ah-m','nn'],
'forward':['f-ao1-r w-er-d','rb jj nn vb'],
'forwarded':['f-ao1-r w-er d-ih-d','vbn'],
'forwarders':['f-ao1-r w-er d-er-z','nns'],
'forwarding':['f-ao1-r w-er d-ih-ng','nn vbg'],
'forwards':['f-ao1-r w-er-d-z','rb'],
'fossil':['f-aa1 s-ah-l','jj nn'],
'fossilized':['f-aa1 s-ah l-ay-z-d','jj'],
'foster':['f-aa1 s-t-er','vb jj'],
'fostered':['f-aa1 s-t-er-d','vbn vbd'],
'fostering':['f-aa1 s-t-er ih-ng','vbg nn'],
'fosters':['f-aa1 s-t-er-z','vbz'],
'fought':['f-ao1-t','vbd vbn'],
'foul':['f-aw1-l','jj rb vb uh'],
'fouled':['f-aw1-l-d','vbd jj vbn'],
'fouling':['f-aw1 l-ih-ng','nn vbg'],
'found':['f-aw1-n-d','vbd vbn vb'],
'foundation':['f-aw-n d-ey1 sh-ah-n','nn'],
'founded':['f-aw1-n d-ah-d','vbn vbd'],
'founder':['f-aw1-n d-er','nn vb'],
'foundered':['f-aw1-n d-er-d','vbd'],
'foundering':['f-aw1-n d-er ih-ng','vbg jj'],
'founders':['f-aw1-n d-er-z','nns'],
'founding':['f-aw1-n d-ih-ng','nn vbg jj'],
'foundling':['f-aw1-n-d l-ih-ng','nn'],
'foundry':['f-aw1-n d-r-iy','nn'],
'fountain':['f-aw1-n t-ah-n','nn'],
'four':['f-ao1-r','cd'],
'fourfold':['f-ao1-r f-ow1-l-d','rb jj'],
'foursome':['f-ao1-r s-ah-m','nn'],
'fourteenth':['f-ao1-r t-iy1-n-th','jj'],
'fourth':['f-ao1-r-th','jj rb'],
'fourthquarter':['f-ao1-r-th k-w-ao1-r t-er','nn'],
'fowl':['f-aw1-l','nn'],
'fox':['f-aa1-k-s','nn'],
'foxholes':['f-aa1-k-s hh-ow-l-z','nns'],
'foxtail':['f-aa1-k-s t-ey-l','nn'],
'foyer':['f-oy1 er','nn'],
'fracas':['f-r-ey1 k-ah-s','nn'],
'fraction':['f-r-ae1-k sh-ah-n','nn'],
'fractional':['f-r-ae1-k sh-ah n-ah-l','jj'],
'fractionally':['f-r-ae1-k sh-ah-n ah-l iy','rb'],
'fractious':['f-r-ae1-k sh-ah-s','jj'],
'fracture':['f-r-ae1-k ch-er','nn'],
'fractured':['f-r-ae1-k ch-er-d','vbn jj vbd'],
'fractures':['f-r-ae1-k ch-er-z','nns vbz'],
'fracturing':['f-r-ae1-k ch-er ih-ng','vbg'],
'fragile':['f-r-ae1 jh-ah-l','jj'],
'fragility':['f-r-ah jh-ih1 l-ah t-iy','nn'],
'fragment':['f-r-ae1-g m-ah-n-t','nn'],
'fragmentary':['f-r-ae1-g m-ah-n t-eh r-iy','jj rb'],
'fragmentation':['f-r-ae-g m-ah-n t-ey1 sh-ah-n','nn'],
'fragmented':['f-r-ae1-g m-ah-n t-ih-d','jj vbn'],
'fragments':['f-r-ae1-g m-ah-n-t-s','nns vbz'],
'fragrance':['f-r-ey1 g-r-ah-n-s','nn'],
'fragrant':['f-r-ey1 g-r-ah-n-t','jj'],
'frail':['f-r-ey1-l','jj'],
'frailties':['f-r-ey1-l t-iy-z','nns'],
'frame':['f-r-ey1-m','nn vbp vb'],
'framed':['f-r-ey1-m-d','vbn vbd jj'],
'framer':['f-r-ey1 m-er','nn'],
'frames':['f-r-ey1-m-z','nns'],
'framework':['f-r-ey1-m w-er-k','nn'],
'framing':['f-r-ey1 m-ih-ng','nn vbg'],
'franc':['f-r-ae1-ng-k','nn'],
'franchise':['f-r-ae1-n ch-ay-z','nn vb'],
'franchised':['f-r-ae1-n ch-ay-z-d','vbn jj'],
'franchisee':['f-r-ae1-n ch-ay z-iy1','nn'],
'franchiser':['f-r-ae1-n ch-ay z-er','nn'],
'franchises':['f-r-ae1-n ch-ay z-ih-z','nns vbz'],
'franchising':['f-r-ae1-n ch-ay z-ih-ng','nn vbg'],
'frank':['f-r-ae1-ng-k','jj nn'],
'franked':['f-r-ae1-ng-k-t','jj vbn'],
'frankfurter':['f-r-ae1-ng-k f-er t-er','nn'],
'franking':['f-r-ae1-ng k-ih-ng','nn vbg'],
'frankly':['f-r-ae1-ng k-l-iy','rb'],
'frankness':['f-r-ae1-ng-k n-ah-s','nn'],
'frantic':['f-r-ae1-n t-ih-k','jj'],
'frantically':['f-r-ae1-n t-ah k-ah l-iy','rb'],
'fraternity':['f-r-ah t-er1 n-ah t-iy','nn'],
'frau':['f-r-aw1','nn'],
'fraud':['f-r-ao1-d','nn'],
'fraudulent':['f-r-ao1 jh-ah l-ah-n-t','jj'],
'fraudulently':['f-r-ao1 d-uw l-ah-n-t l-iy','rb'],
'fraught':['f-r-ao1-t','jj'],
'fray':['f-r-ey1','nn vb'],
'frayed':['f-r-ey1-d','jj vbn'],
'fraying':['f-r-ey1 ih-ng','vbg'],
'frazzled':['f-r-ae1 z-ah-l-d','vbn'],
'freak':['f-r-iy1-k','nn vbp'],
'freaked':['f-r-iy1-k-t','vbn'],
'freckled':['f-r-eh1 k-ah-l-d','jj'],
'free':['f-r-iy1','jj rb vb vbp'],
'freebies':['f-r-iy1 b-iy-z','nns'],
'freed':['f-r-iy1-d','vbn jj vbd'],
'freedom':['f-r-iy1 d-ah-m','nn'],
'freefall':['f-r-iy1 f-ao-l','nn'],
'freehand':['f-r-iy1 hh-ae-n-d','jj rb'],
'freeholders':['f-r-iy1 hh-ow-l d-er-z','nns'],
'freeing':['f-r-iy1 ih-ng','vbg nn'],
'freelance':['f-r-iy1 l-ae-n-s','jj'],
'freely':['f-r-iy1 l-iy','rb'],
'freeman':['f-r-iy1 m-ah-n','nn'],
'freer':['f-r-iy1 er','jjr'],
'frees':['f-r-iy1-z','vbz'],
'freest':['f-r-iy1 ih-s-t','jjs'],
'freethinkers':['f-r-iy1 th-ih1-ng k-er-z','nns'],
'freeway':['f-r-iy1 w-ey','nn'],
'freewheeling':['f-r-iy1 w-iy l-ih-ng','jj'],
'freeze':['f-r-iy1-z','nn vb vbp'],
'freezer':['f-r-iy1 z-er','nn'],
'freezes':['f-r-iy1 z-ih-z','vbz nns'],
'freezing':['f-r-iy1 z-ih-ng','vbg jj nn'],
'freight':['f-r-ey1-t','nn vb'],
'freighter':['f-r-ey1 t-er','nn'],
'french':['f-r-eh1-n-ch','jj nn nns'],
'frenetic':['f-r-ah n-eh1 t-ih-k','jj'],
'frenzied':['f-r-eh1-n z-iy-d','jj'],
'frenzy':['f-r-eh1-n z-iy','nn'],
'frequency':['f-r-iy1 k-w-ah-n s-iy','nn'],
'frequent':['f-r-iy1 k-w-ah-n-t','jj vbp vb'],
'frequented':['f-r-iy1 k-w-ah-n t-ih-d','vbd vbn'],
'frequently':['f-r-iy1 k-w-ah-n-t l-iy','rb'],
'frequents':['f-r-iy1 k-w-ah-n-t-s','vbz'],
'fresco':['f-r-eh1-s k-ow','nn'],
'frescoed':['f-r-eh1-s k-ow-d','jj'],
'fresh':['f-r-eh1-sh','jj rb'],
'fresher':['f-r-eh1 sh-er','jjr'],
'freshly':['f-r-eh1-sh l-iy','rb'],
'freshman':['f-r-eh1-sh m-ah-n','nn'],
'freshness':['f-r-eh1-sh n-ah-s','nn'],
'freshwater':['f-r-eh1-sh w-ao t-er','jjr'],
'fret':['f-r-eh1-t','vbp nn vb'],
'frets':['f-r-eh1-t-s','vbz'],
'fretted':['f-r-eh1 t-ih-d','vbd'],
'fretting':['f-r-eh1 t-ih-ng','vbg'],
'friar':['f-r-ay1 er','nn'],
'friction':['f-r-ih1-k sh-ah-n','nn'],
'frictionless':['f-r-ih1-k sh-ah-n l-ah-s','jj'],
'fridge':['f-r-ih1-jh','nn'],
'fried':['f-r-ay1-d','jj vbn'],
'friend':['f-r-eh1-n-d','nn'],
'friendlier':['f-r-eh1-n-d l-iy er','jjr rbr'],
'friendliness':['f-r-eh1-n-d l-iy n-ih-s','nn'],
'friendly':['f-r-eh1-n-d l-iy','jj'],
'friendship':['f-r-eh1-n-d sh-ih-p','nn'],
'fries':['f-r-ay1-z','nns'],
'frieze':['f-r-iy1-z','nn'],
'frigate':['f-r-ih1 g-ah-t','nn'],
'fright':['f-r-ay1-t','nn'],
'frighten':['f-r-ay1 t-ah-n','vb vbp'],
'frightened':['f-r-ay1 t-ah-n-d','vbn jj vbd'],
'frightening':['f-r-ay1 t-ah-n ih-ng','jj vbg'],
'frighteningly':['f-r-ay1 t-ah-n ih-ng l-iy','rb'],
'frightens':['f-r-ay1 t-ah-n-z','vbz'],
'frightful':['f-r-ay1-t f-ah-l','jj'],
'frightfully':['f-r-ay1-t f-ah l-iy','rb'],
'frigid':['f-r-ih1 jh-ah-d','jj'],
'frill':['f-r-ih1-l','nn'],
'frilly':['f-r-ih1 l-iy','jj'],
'fringe':['f-r-ih1-n-jh','nn jj'],
'frisky':['f-r-ih1-s k-iy','jj'],
'frittered':['f-r-ih1 t-er-d','vbn'],
'frittering':['f-r-ih1 t-er ih-ng','vbg'],
'frivolity':['f-r-ah v-aa1 l-ah t-iy','nn'],
'frivolous':['f-r-ih1 v-ah l-ah-s','jj'],
'frock':['f-r-aa1-k','nn'],
'frog':['f-r-aa1-g','nn'],
'frogmen':['f-r-aa1-g m-ih-n','nns'],
'frolic':['f-r-aa1 l-ih-k','nn vb'],
'frolicking':['f-r-aa1 l-ih k-ih-ng','vbg'],
'from':['f-r-ah1-m','in rb rp'],
'fronds':['f-r-aa1-n-d-z','nns'],
'front':['f-r-ah1-n-t','nn jj rb'],
'frontage':['f-r-ah1-n t-ih-jh','nn'],
'frontal':['f-r-ah1-n t-ah-l','jj'],
'fronted':['f-r-ah1-n t-ih-d','vbd'],
'frontier':['f-r-ah-n t-ih1-r','nn'],
'fronting':['f-r-ah1-n t-ih-ng','vbg nn'],
'frontrunner':['f-r-ah1-n t-r-ah n-er','nn'],
'frost':['f-r-ao1-s-t','nn'],
'frostbite':['f-r-ao1-s-t b-ay-t','nn'],
'frosted':['f-r-ao1 s-t-ah-d','vbd'],
'frosting':['f-r-ao1 s-t-ih-ng','nn'],
'frosty':['f-r-ao1 s-t-iy','jj'],
'froth':['f-r-ao1-th','nn vb'],
'frothy':['f-r-ao1 th-iy','jj'],
'frown':['f-r-aw1-n','vbp'],
'frowned':['f-r-aw1-n-d','vbd vbn'],
'frowning':['f-r-aw1 n-ih-ng','vbg'],
'frowningly':['f-r-aw1 n-ih-ng l-iy','rb'],
'frowns':['f-r-aw1-n-z','vbz'],
'froze':['f-r-ow1-z','vbd vbn'],
'frozen':['f-r-ow1 z-ah-n','vbn jj'],
'fructose':['f-r-ah1-k t-ow-s','nn'],
'frugal':['f-r-uw1 g-ah-l','jj'],
'frugality':['f-r-uw g-ae1 l-ah t-iy','nn'],
'frugally':['f-r-uw1 g-ah l-iy','rb'],
'fruit':['f-r-uw1-t','nn'],
'fruitful':['f-r-uw1-t f-ah-l','jj'],
'fruition':['f-r-uw ih1 sh-ah-n','nn'],
'fruitless':['f-r-uw1-t l-ah-s','jj'],
'fruitlessly':['f-r-uw1-t l-ah-s l-iy','rb'],
'fruity':['f-r-uw1 t-iy','jj'],
'frumpy':['f-r-ah1-m p-iy','jj'],
'frustrate':['f-r-ah1 s-t-r-ey-t','vb'],
'frustrated':['f-r-ah1 s-t-r-ey t-ah-d','vbn jj vbd'],
'frustrates':['f-r-ah1 s-t-r-ey-t-s','vbz'],
'frustrating':['f-r-ah1 s-t-r-ey t-ih-ng','jj vbg'],
'frustration':['f-r-ah-s t-r-ey1 sh-ah-n','nn'],
'fry':['f-r-ay1','nn vbp vb'],
'fryers':['f-r-ay1 er-z','nns'],
'frying':['f-r-ay1 ih-ng','vbg nn'],
'fudge':['f-ah1-jh','vb vbp nn'],
'fudging':['f-ah1 jh-ih-ng','nn'],
'fuel':['f-y-uw1 ah-l','nn vbp vb'],
'fueled':['f-y-uw1 ah-l-d','vbn vbd'],
'fueling':['f-y-uw1 l-ih-ng','vbg nn'],
'fuels':['f-y-uw1 ah-l-z','nns vbz'],
'fuer':['f-y-uw1-r','nnp'],
'fugitive':['f-y-uw1 jh-ah t-ih-v','jj nn'],
'fulcrum':['f-uh1-l k-r-ah-m','nn'],
'fulfill':['f-uh-l f-ih1-l','vb vbp'],
'fulfilled':['f-uh-l f-ih1-l-d','vbn vbd'],
'fulfilling':['f-uh-l f-ih1 l-ih-ng','vbg'],
'fulfillment':['f-uh-l f-ih1-l m-ah-n-t','nn'],
'fulfills':['f-uh-l f-ih1-l-z','vbz'],
'full':['f-uh1-l','jj rb'],
'fullback':['f-uh1-l b-ae-k','nn'],
'fuller':['f-uh1 l-er','jjr'],
'fullest':['f-uh1 l-ah-s-t','jjs'],
'fullness':['f-uh1-l n-ah-s','nn'],
'fully':['f-uh1 l-iy','rb'],
'fulminate':['f-uh1-l m-ah n-ey-t','vb'],
'fumble':['f-ah1-m b-ah-l','nn'],
'fumbled':['f-ah1-m b-ah-l-d','vbd'],
'fumbling':['f-ah1-m b-ah-l ih-ng','vbg jj'],
'fumed':['f-y-uw1-m-d','vbd'],
'fumes':['f-y-uw1-m-z','nns vbz'],
'fuming':['f-y-uw1 m-ih-ng','vbg'],
'fun':['f-ah1-n','nn jj'],
'function':['f-ah1-ng-k sh-ah-n','nn vb vbp'],
'functional':['f-ah1-ng-k sh-ah n-ah-l','jj nn'],
'functionally':['f-ah1-ng-k sh-ah-n ah-l iy','rb'],
'functionary':['f-ah1-ng-k sh-ah-n eh r-iy','nn'],
'functioned':['f-ah1-ng-k sh-ah-n-d','vbd vbn'],
'functioning':['f-ah1-ng-k sh-ah-n ih-ng','vbg nn'],
'functions':['f-ah1-ng-k sh-ah-n-z','nns vbz'],
'fund':['f-ah1-n-d','nn vbp vb'],
'fundamental':['f-ah-n d-ah m-eh1-n t-ah-l','jj'],
'fundamentalism':['f-ah-n d-ah m-eh1-n t-ah l-ih z-ah-m','nn'],
'fundamentalist':['f-ah-n d-ah m-eh1-n t-ah l-ih-s-t','jj nn'],
'fundamentally':['f-ah-n d-ah m-eh1-n t-ah l-iy','rb'],
'funded':['f-ah1-n d-ah-d','vbn vbd jj'],
'funding':['f-ah1-n d-ih-ng','nn vbg jj'],
'fundraiser':['f-ah1-n d-r-ey z-er','nn'],
'fundraising':['f-ah1-n d-r-ey s-ih-ng','vbg nn'],
'funds':['f-ah1-n-d-z','nns vbz'],
'funeral':['f-y-uw1 n-er ah-l','nn jj'],
'fungal':['f-ah1-ng g-ah-l','jj'],
'fungible':['f-ah1-n jh-ih b-ah-l','jj'],
'fungicides':['f-ah1-n jh-ah s-ay-d-z','nns'],
'fungus':['f-ah1-ng g-ah-s','nn'],
'funks':['f-ah1-ng-k-s','nns'],
'funky':['f-ah1-ng k-iy','jj'],
'funnel':['f-ah1 n-ah-l','vb nn'],
'funneled':['f-ah1 n-ah-l-d','vbd vbn'],
'funneling':['f-ah1 n-ah-l ih-ng','vbg'],
'funnels':['f-ah1 n-ah-l-z','nns'],
'funnier':['f-ah1 n-iy er','jjr'],
'funniest':['f-ah1 n-iy ah-s-t','jjs'],
'funny':['f-ah1 n-iy','jj'],
'fur':['f-er1','nn'],
'furbishing':['f-er1 b-ih sh-ih-ng','nn'],
'furious':['f-y-uh1 r-iy ah-s','jj'],
'furiouser':['f-y-uh1 r-iy ah s-er','rbr'],
'furiously':['f-y-uh1 r-iy ah-s l-iy','rb'],
'furlough':['f-er1 l-ow','nn'],
'furloughed':['f-er1 l-ow-d','vbn'],
'furnace':['f-er1 n-ah-s','nn'],
'furnish':['f-er1 n-ih-sh','vb vbp'],
'furnished':['f-er1 n-ih-sh-t','vbn vbd jj'],
'furnishes':['f-er1 n-ih sh-ah-z','vbz'],
'furnishing':['f-er1 n-ih sh-ih-ng','nn vbg'],
'furniture':['f-er1 n-ih ch-er','nn'],
'furor':['f-y-uh1 r-ao-r','nn'],
'furrier':['f-er1 iy er','nn'],
'furrow':['f-er1 ow','nn'],
'furry':['f-er1 iy','jj'],
'further':['f-er1 dh-er','jj rb rbr jjr vb'],
'furthered':['f-er1 dh-er-d','vbd vbn'],
'furthering':['f-er1 dh-er ih-ng','vbg'],
'furthermore':['f-er1 dh-er m-ao-r','rb'],
'furthers':['f-er1 dh-er-z','vbz'],
'furthest':['f-er1 th-ah-s-t','jjs rbs'],
'furtive':['f-er1 t-ih-v','jj'],
'furtively':['f-er1 t-ih-v l-iy','rb'],
'fury':['f-y-uh1 r-iy','nn'],
'fuse':['f-y-uw1-z','nn vb'],
'fused':['f-y-uw1-z-d','vbn vbd'],
'fuselage':['f-y-uw1 s-ah l-aa-jh','nn'],
'fuses':['f-y-uw1 z-ah-z','nns'],
'fusillade':['f-y-uw1 s-ih l-ey-d','nn'],
'fusing':['f-y-uw1 z-ih-ng','vbg nn'],
'fusion':['f-y-uw1 zh-ah-n','nn'],
'fuss':['f-ah1-s','nn vb'],
'fusses':['f-ah1 s-ih-z','vbz'],
'fussing':['f-ah1 s-ih-ng','vbg'],
'fussy':['f-ah1 s-iy','jj'],
'futile':['f-y-uw1 t-ah-l','jj'],
'futility':['f-y-uw t-ih1 l-ah t-iy','nn'],
'future':['f-y-uw1 ch-er','nn jj'],
'futurist':['f-y-uw1 ch-er ih-s-t','nn'],
'futuristic':['f-y-uw ch-er ih1 s-t-ih-k','jj'],
'fuzz':['f-ah1-z','nn'],
'fuzzier':['f-ah1 z-iy er','jjr'],
'fuzzy':['f-ah1 z-iy','jj'],
'g':['jh-iy1','nn'],
'gab':['g-ae1-b','nn'],
'gabardine':['g-ae1 b-er d-iy-n','nn'],
'gabbing':['g-ae1 b-ih-ng','vbg'],
'gable':['g-ey1 b-ah-l','nn'],
'gadfly':['g-ae1-d f-l-ay','nn'],
'gadget':['g-ae1 jh-ah-t','nn'],
'gadgetry':['g-ae1 jh-ah t-r-iy','nn'],
'gaffes':['g-ae1-f-s','nns'],
'gag':['g-ae1-g','nn'],
'gage':['g-ey1-jh','nn vb'],
'gagged':['g-ae1-g-d','vbn vbd'],
'gaggle':['g-ae1 g-ah-l','nn'],
'gaily':['g-ey1 l-iy','rb'],
'gain':['g-ey1-n','nn vbp vb'],
'gained':['g-ey1-n-d','vbd vbn vb'],
'gainer':['g-ey1 n-er','nn'],
'gainful':['g-ey1-n f-ah-l','jj'],
'gaining':['g-ey1 n-ih-ng','vbg'],
'gains':['g-ey1-n-z','nns nn vbz'],
'gait':['g-ey1-t','nn'],
'gal':['g-ae1-l','nn jj'],
'gala':['g-ae1 l-ah','jj nn'],
'galactic':['g-ah l-ae1-k t-ih-k','jj'],
'galaxy':['g-ae1 l-ah-k s-iy','nn'],
'gale':['g-ey1-l','nn'],
'gall':['g-ao1-l','nn vb'],
'gallant':['g-ae1 l-ah-n-t','jj'],
'gallantry':['g-ae1 l-ah-n t-r-iy','nn'],
'gallbladder':['g-ao1-l b-l-ae d-er','nn'],
'gallery':['g-ae1 l-er iy','nn'],
'galley':['g-ae1 l-iy','nn'],
'galling':['g-ao1 l-ih-ng','jj'],
'gallium':['g-ae1 l-iy ah-m','nn'],
'gallon':['g-ae1 l-ah-n','nn'],
'gallop':['g-ae1 l-ah-p','nn vb'],
'galloped':['g-ae1 l-ah-p-t','vbn'],
'galloping':['g-ae1 l-ah p-ih-ng','vbg jj'],
'gallows':['g-ae1 l-ow-z','nn nns'],
'galls':['g-ao1-l-z','nns'],
'gallstone':['g-ao1-l s-t-ow-n','nn'],
'galvanic':['g-ae-l v-ae1 n-ih-k','jj'],
'galvanize':['g-ae1-l v-ah n-ay-z','vb'],
'galvanized':['g-ae1-l v-ah n-ay-z-d','jj vbd vbn'],
'galvanizing':['g-ae1-l v-ah n-ay z-ih-ng','vbg nn'],
'gambit':['g-ae1-m b-ih-t','nn'],
'gamble':['g-ae1-m b-ah-l','nn vb'],
'gambled':['g-ae1-m b-ah-l-d','vbn'],
'gambler':['g-ae1-m b-l-er','nn'],
'gambles':['g-ae1-m b-ah-l-z','nns'],
'gambling':['g-ae1-m b-ah-l ih-ng','nn vbg'],
'game':['g-ey1-m','nn'],
'gaming':['g-ey1 m-ih-ng','nn'],
'gamma':['g-ae1 m-ah','nn jj'],
'gamut':['g-ae1 m-ah-t','nn'],
'gander':['g-ae1-n d-er','nn'],
'gang':['g-ae1-ng','nn vb'],
'gangbusters':['g-ae1-ng b-ah s-t-er-z','nns'],
'ganging':['g-ae1 ng-ih-ng','vbg'],
'gangplank':['g-ae1-ng p-l-ae-ng-k','nn'],
'gangs':['g-ae1-ng-z','nns'],
'gangster':['g-ae1-ng s-t-er','nn'],
'gantlet':['g-ao1-n-t l-ah-t','nn'],
'gap':['g-ae1-p','nn'],
'gaping':['g-ey1 p-ih-ng','vbg jj'],
'garage':['g-er aa1-zh','nn vbp'],
'garb':['g-aa1-r-b','nn'],
'garbage':['g-aa1-r b-ih-jh','nn'],
'garbed':['g-aa1-r-b-d','vbn'],
'garbled':['g-aa1-r b-ah-l-d','vbn'],
'garden':['g-aa1-r d-ah-n','nn vb'],
'gardened':['g-aa1-r d-ah-n-d','vbd'],
'gardener':['g-aa1-r d-ah-n er','nn'],
'gardenia':['g-aa-r d-iy1 n-y-ah','nn'],
'gardening':['g-aa1-r d-ah-n ih-ng','nn vbg'],
'gardens':['g-aa1-r d-ah-n-z','nns vbz'],
'gargantuan':['g-aa-r g-ae1-n ch-uw ah-n','jj'],
'garish':['g-eh1 r-ih-sh','jj'],
'garland':['g-aa1-r l-ah-n-d','nn'],
'garlic':['g-aa1-r l-ih-k','nn'],
'garment':['g-aa1-r m-ah-n-t','nn'],
'garner':['g-aa1-r n-er','vb vbp'],
'garnered':['g-aa1-r n-er-d','vbd vbn'],
'garnet':['g-aa1-r n-ah-t','nn'],
'garnish':['g-aa1-r n-ih-sh','nn'],
'garrison':['g-ae1 r-ih s-ah-n','nn vb'],
'garrisoned':['g-ae1 r-ah s-ah-n-d','vbn'],
'garrulous':['g-eh1 r-ah l-ah-s','jj'],
'garter':['g-aa1-r t-er','nn'],
'gas':['g-ae1-s','nn vb'],
'gaseous':['g-ae1 s-iy ah-s','jj'],
'gases':['g-ae1 s-ah-z','nns'],
'gash':['g-ae1-sh','nn'],
'gasket':['g-ae1-s k-ah-t','nn'],
'gasoline':['g-ae1 s-ah l-iy-n','nn'],
'gasp':['g-ae1-s-p','nn vb'],
'gasped':['g-ae1-s-p-t','vbd'],
'gasping':['g-ae1 s-p-ih-ng','vbg'],
'gasps':['g-ae1-s-p-s','nns'],
'gassed':['g-ae1-s-t','vbn'],
'gasser':['g-ae1 s-er','nn'],
'gasses':['g-ae1 s-ih-z','nns'],
'gassing':['g-ae1 s-ih-ng','nn'],
'gastric':['g-ae1 s-t-r-ih-k','jj'],
'gastrointestinal':['g-ae s-t-r-ow ih-n t-eh1 s-t-ah n-ah-l','jj'],
'gastronomy':['g-ae-s t-r-aa1 n-ah m-iy','nn'],
'gate':['g-ey1-t','nn'],
'gates':['g-ey1-t-s','nns vbz'],
'gateway':['g-ey1-t w-ey','nn'],
'gather':['g-ae1 dh-er','vb vbp'],
'gathered':['g-ae1 dh-er-d','vbd vbn'],
'gathering':['g-ae1 dh-er ih-ng','nn vbg'],
'gathers':['g-ae1 dh-er-z','vbz'],
'gauche':['g-ow1-sh','jj'],
'gaucho':['g-aw1 ch-ow','nn'],
'gaudy':['g-ao1 d-iy','jj'],
'gauge':['g-ey1-jh','nn vb'],
'gauged':['g-ey1-jh-d','vbn vbd'],
'gauges':['g-ey1 jh-ah-z','vbz nns'],
'gauging':['g-ey1 jh-ih-ng','vbg'],
'gaunt':['g-ao1-n-t','jj'],
'gauntlet':['g-ao1-n-t l-ah-t','nn'],
'gauss':['g-aw1-s','nn'],
'gauze':['g-ao1-z','nn'],
'gave':['g-ey1-v','vbd'],
'gawky':['g-ao1 k-iy','jj'],
'gay':['g-ey1','jj'],
'gaze':['g-ey1-z','nn vb vbp'],
'gazed':['g-ey1-z-d','vbd'],
'gazelle':['g-ah z-eh1-l','nn'],
'gazes':['g-ey1 z-ah-z','vbz'],
'gazing':['g-ey1 z-ih-ng','vbg'],
'gear':['g-ih1-r','nn vb vbp'],
'gearboxes':['g-ih1-r b-aa-k s-ih-z','nns'],
'geared':['g-ih1-r-d','vbn vbd'],
'gearing':['g-ih1 r-ih-ng','vbg'],
'gears':['g-ih1-r-z','nns vbz'],
'gee':['jh-iy1','uh'],
'geeing':['jh-iy1 ih-ng','vbg'],
'geeks':['g-iy1-k-s','nns'],
'gel':['jh-eh1-l','nn'],
'gelatin':['jh-eh1 l-ah t-ah-n','nn'],
'gels':['jh-eh1-l-z','vbz'],
'gem':['jh-eh1-m','nn'],
'gemsbok':['g-eh1-m-z b-aa-k','nns'],
'gemstone':['jh-eh1-m s-t-ow-n','nn'],
'gen':['jh-eh1 n-er ah-l','nn'],
'gendarme':['zh-aa1-n d-aa-r-m','nn'],
'gender':['jh-eh1-n d-er','nn vb'],
'genders':['jh-eh1-n d-er-z','nns'],
'gene':['jh-iy1-n','nn'],
'genera':['jh-eh1 n-er ah','nn'],
'general':['jh-eh1 n-er ah-l','jj nn'],
'generalist':['jh-eh1 n-er ah l-ih-s-t','nn'],
'generality':['jh-eh n-er ae1 l-ah t-iy','nn'],
'generalization':['jh-eh n-er ah l-ih z-ey1 sh-ah-n','nn'],
'generalize':['jh-eh1 n-er ah l-ay-z','vb vbp'],
'generalized':['jh-eh1 n-er ah l-ay-z-d','jj vbn'],
'generally':['jh-eh1 n-er ah l-iy','rb'],
'generalship':['jh-eh1 n-er ah-l sh-ih-p','nn'],
'generate':['jh-eh1 n-er ey-t','vb vbp'],
'generated':['jh-eh1 n-er ey t-ah-d','vbn vbd vbg'],
'generates':['jh-eh1 n-er ey-t-s','vbz'],
'generating':['jh-eh1 n-er ey t-ih-ng','vbg nn'],
'generation':['jh-eh n-er ey1 sh-ah-n','nn'],
'generational':['jh-eh n-er ey1 sh-ah n-ah-l','jj'],
'generator':['jh-eh1 n-er ey t-er','nn'],
'generic':['jh-ah n-eh1 r-ih-k','jj'],
'generically':['jh-ah n-eh1 r-ih-k l-iy','rb'],
'generosity':['jh-eh n-er aa1 s-ah t-iy','nn'],
'generous':['jh-eh1 n-er ah-s','jj'],
'generously':['jh-eh1 n-er ah-s l-iy','rb'],
'genesis':['jh-eh1 n-ah s-ah-s','nn'],
'genetic':['jh-ah n-eh1 t-ih-k','jj'],
'genetically':['jh-ah n-eh1 t-ih-k l-iy','rb'],
'geneticist':['jh-ah n-eh1 t-ah s-ah-s-t','nn'],
'genetics':['jh-ah n-eh1 t-ih-k-s','nns'],
'genial':['jh-iy1 n-y-ah-l','jj'],
'genie':['jh-iy1 n-iy','nn'],
'genital':['jh-eh1 n-ah t-ah-l','jj'],
'genius':['jh-iy1 n-y-ah-s','nn'],
'genocide':['jh-eh1 n-ah s-ay-d','nn'],
'genome':['jh-iy1 n-ow-m','nn'],
'genre':['zh-aa1-n r-ah','nn'],
'genteel':['jh-eh-n t-iy1-l','jj'],
'gentian':['jh-eh1-n sh-ah-n','nn'],
'gentile':['jh-eh1-n t-ay-l','nn jj'],
'gentility':['jh-eh-n t-ih1 l-ih t-iy','nn'],
'gentle':['jh-eh1-n t-ah-l','jj vb'],
'gentleman':['jh-eh1-n t-ah-l m-ah-n','nn'],
'gentlemanly':['jh-eh1-n t-ah-l m-ah-n l-iy','jj'],
'gentleness':['jh-eh1-n t-ah-l n-ah-s','nn'],
'gentler':['jh-eh1-n-t l-er','jjr'],
'gently':['jh-eh1-n-t l-iy','rb'],
'gentrified':['jh-eh1-n t-r-ih f-ay-d','vbn'],
'gentry':['jh-eh1-n t-r-iy','nn'],
'genuine':['jh-eh1 n-y-ah-w ah-n','jj'],
'genuinely':['jh-eh1 n-y-ah-w ah-n l-iy','rb'],
'genus':['jh-iy1 n-ah-s','nn'],
'geocentric':['jh-iy ow s-eh1-n t-r-ih-k','jj'],
'geochemistry':['jh-iy ow k-eh1 m-ah s-t-r-iy','nn'],
'geographer':['jh-iy aa1 g-r-ah f-er','nn'],
'geographic':['jh-iy ah g-r-ae1 f-ih-k','jj'],
'geographical':['jh-iy ah g-r-ae1 f-ih k-ah-l','jj'],
'geographically':['jh-iy ah g-r-ae1 f-ih k-ah l-iy','rb'],
'geography':['jh-iy aa1 g-r-ah f-iy','nn'],
'geologic':['jh-iy ah l-aa1 jh-ih-k','jj'],
'geological':['jh-iy ah l-aa1 jh-ih k-ah-l','jj'],
'geologist':['jh-iy aa1 l-ah jh-ah-s-t','nn'],
'geology':['jh-iy aa1 l-ah jh-iy','nn'],
'geometric':['jh-iy ah m-eh1 t-r-ih-k','jj'],
'geometrical':['jh-iy ah m-eh1 t-r-ih k-ah-l','jj'],
'geometrically':['jh-iy ah m-eh1 t-r-ih-k l-iy','rb'],
'geometry':['jh-iy aa1 m-ah t-r-iy','nn'],
'geopolitical':['jh-iy ow p-ah l-ih1 t-ih k-ah-l','jj'],
'geothermal':['jh-iy ow th-er1 m-ah-l','jj'],
'geranium':['jh-er ey1 n-iy ah-m','nn'],
'geriatric':['jh-eh r-iy ae1 t-r-ih-k','jj'],
'germ':['jh-er1-m','nn'],
'germane':['jh-er m-ey1-n','jj'],
'germinate':['jh-er1 m-ah n-ey-t','vbp'],
'germinated':['jh-er1 m-ah n-ey t-ih-d','jj'],
'gerrymandering':['jh-eh r-iy m-ae1-n d-er ih-ng','nn'],
'gesture':['jh-eh1-s ch-er','nn'],
'gestures':['jh-eh1-s ch-er-z','nns vbz'],
'gesturing':['jh-eh1-s ch-er ih-ng','vbg'],
'get':['g-eh1-t','vb vbp'],
'getaway':['g-eh1-t ah w-ey','nn'],
'gets':['g-eh1-t-s','vbz'],
'getting':['g-eh1 t-ih-ng','vbg'],
'geyser':['g-ay1 z-er','nn'],
'ghastly':['g-ae1-s-t l-iy','jj'],
'ghetto':['g-eh1 t-ow','nn'],
'ghost':['g-ow1-s-t','nn'],
'ghostbusters':['g-ow1-s-t b-ah s-t-er-z','nns'],
'ghostlike':['g-ow1-s-t l-ay-k','jj'],
'ghostly':['g-ow1-s-t l-iy','jj'],
'ghoulish':['g-uw1 l-ih-sh','jj'],
'giant':['jh-ay1 ah-n-t','nn jj'],
'gibberish':['g-ih1 b-er ih-sh','nn'],
'giblet':['jh-ih1 b-l-ah-t','nn'],
'giddy':['g-ih1 d-iy','jj'],
'gift':['g-ih1-f-t','nn'],
'gifted':['g-ih1-f t-ah-d','jj vbn'],
'gig':['g-ih1-g','nn'],
'gigantic':['jh-ay g-ae1-n t-ih-k','jj'],
'giggle':['g-ih1 g-ah-l','nn'],
'giggled':['g-ih1 g-ah-l-d','vbd vbn'],
'giggling':['g-ih1 g-ah-l ih-ng','vbg'],
'gigolo':['jh-ih1-g ah l-ow','nn'],
'gilded':['g-ih1-l d-ih-d','jj'],
'gilding':['g-ih1-l d-ih-ng','nn'],
'gilt':['g-ih1-l-t','jj nn'],
'gimmick':['g-ih1 m-ih-k','nn'],
'gimmickry':['g-ih1 m-ih k-r-iy','nn'],
'gimmicky':['g-ih m-ih1 k-iy','jj'],
'gin':['jh-ih1-n','nn'],
'ginger':['jh-ih1-n jh-er','nn'],
'gingerly':['jh-ih1-n jh-er l-iy','rb jj'],
'ginning':['jh-ih1 n-ih-ng','nn'],
'ginseng':['jh-ih1-n s-eh-ng','nn'],
'gird':['g-er1-d','vb'],
'girders':['g-er1 d-er-z','nns'],
'girding':['g-er1 d-ih-ng','vbg'],
'girdle':['g-er1 d-ah-l','nn'],
'girds':['g-er1-d-z','vbz'],
'girl':['g-er1-l','nn'],
'girlfriend':['g-er1-l f-r-eh-n-d','nn'],
'girlie':['g-er1 l-iy','nn'],
'girlish':['g-er1 l-ih-sh','jj'],
'girlishly':['g-er1 l-ih-sh l-iy','rb'],
'girth':['g-er1-th','nn'],
'gist':['jh-ih1-s-t','nn'],
'give':['g-ih1-v','vb nn vbp'],
'giveaway':['g-ih1 v-ah w-ey','nn jj'],
'givebacks':['g-ih1-v b-ae-k-s','nns'],
'given':['g-ih1 v-ah-n','vbn jj'],
'givenness':['g-ih1 v-ah n-ah-s','nn'],
'giver':['g-ih1 v-er','nn'],
'gives':['g-ih1-v-z','vbz'],
'giveth':['g-ih1 v-eh-th','vbz vb'],
'giving':['g-ih1 v-ih-ng','vbg nn'],
'gizmo':['g-ih1-z m-ow','nn'],
'glacial':['g-l-ey1 sh-ah-l','jj'],
'glacier':['g-l-ey1 sh-er','nn'],
'glad':['g-l-ae1-d','jj'],
'glade':['g-l-ey1-d','nn'],
'gladiator':['g-l-ae1 d-iy ey t-er','nn'],
'gladly':['g-l-ae1-d l-iy','rb'],
'glamor':['g-l-ae1 m-er','nn'],
'glamorous':['g-l-ae1 m-er ah-s','jj'],
'glamour':['g-l-ae1 m-er','nn'],
'glance':['g-l-ae1-n-s','nn vb'],
'glanced':['g-l-ae1-n-s-t','vbd'],
'glances':['g-l-ae1-n s-ih-z','nns vbz'],
'glancing':['g-l-ae1-n s-ih-ng','vbg jj'],
'gland':['g-l-ae1-n-d','nn'],
'glandular':['g-l-ae1-n jh-ah l-er','jj'],
'glare':['g-l-eh1-r','nn vb'],
'glared':['g-l-eh1-r-d','vbd'],
'glares':['g-l-eh1-r-z','vbz'],
'glaring':['g-l-eh1 r-ih-ng','jj vbg'],
'glasnost':['g-l-ae1-s n-aa-s-t','nn'],
'glass':['g-l-ae1-s','nn'],
'glassware':['g-l-ae1-s w-eh-r','nn'],
'glassy':['g-l-ae1 s-iy','jj'],
'glaucoma':['g-l-ao k-ow1 m-ah','nn'],
'glaze':['g-l-ey1-z','nn vb vbp'],
'glazed':['g-l-ey1-z-d','vbn'],
'glazes':['g-l-ey1 z-ah-z','nns'],
'glazing':['g-l-ey1 z-ih-ng','vbg'],
'gleam':['g-l-iy1-m','nn vbp'],
'gleamed':['g-l-iy1-m-d','vbd'],
'gleaming':['g-l-iy1 m-ih-ng','vbg'],
'glean':['g-l-iy1-n','vb'],
'gleaned':['g-l-iy1-n-d','vbn'],
'glee':['g-l-iy1','nn'],
'gleeful':['g-l-iy1 f-ah-l','jj'],
'gleefully':['g-l-iy1 f-ah l-iy','rb'],
'glen':['g-l-eh1-n','nn'],
'glib':['g-l-ih1-b','jj'],
'glibly':['g-l-ih1 b-l-iy','rb'],
'glide':['g-l-ay1-d','vb vbp'],
'glided':['g-l-ay1 d-ih-d','vbd'],
'glider':['g-l-ay1 d-er','nn'],
'glides':['g-l-ay1-d-z','vbz'],
'gliding':['g-l-ay1 d-ih-ng','vbg'],
'glimmer':['g-l-ih1 m-er','nn'],
'glimmering':['g-l-ih1 m-er ih-ng','vbg'],
'glimpse':['g-l-ih1-m-p-s','nn'],
'glimpsed':['g-l-ih1-m-p-s-t','vbn vbd'],
'glimpses':['g-l-ih1-m-p s-ih-z','nns'],
'glint':['g-l-ih1-n-t','nn'],
'glistened':['g-l-ih1 s-ah-n-d','vbd'],
'glistening':['g-l-ih1 s-ah-n ih-ng','vbg'],
'glitch':['g-l-ih1-ch','nn'],
'glitter':['g-l-ih1 t-er','nn'],
'glittered':['g-l-ih1 t-er-d','vbd vbn'],
'glittering':['g-l-ih1 t-er ih-ng','vbg'],
'glittery':['g-l-ih1 t-er iy','nn'],
'glitz':['g-l-ih1-t-s','nn'],
'glitzy':['g-l-ih1-t s-iy','jj nn'],
'gloat':['g-l-ow1-t','vb nn'],
'gloated':['g-l-ow1 t-ih-d','vbd'],
'gloating':['g-l-ow1 t-ih-ng','vbg'],
'gloats':['g-l-ow1-t-s','vbz'],
'global':['g-l-ow1 b-ah-l','jj'],
'globalization':['g-l-ow b-ah-l ih z-ey1 sh-ah-n','nn'],
'globalized':['g-l-ow1 b-ah-l ay-z-d','jj'],
'globally':['g-l-ow1 b-ah-l iy','rb'],
'globe':['g-l-ow1-b','nn'],
'globulin':['g-l-aa1 b-y-ah l-ih-n','nn'],
'gloom':['g-l-uw1-m','nn'],
'gloomier':['g-l-uw1 m-iy er','jjr rbr'],
'gloomily':['g-l-uw1 m-ah l-iy','rb'],
'gloomy':['g-l-uw1 m-iy','jj'],
'glop':['g-l-aa1-p','nn'],
'glories':['g-l-ao1 r-iy-z','nns vbz'],
'glorified':['g-l-ao1 r-ah f-ay-d','vbn jj'],
'glorify':['g-l-ao1 r-ah f-ay','vb vbp'],
'glorious':['g-l-ao1 r-iy ah-s','jj'],
'gloriously':['g-l-ao1 r-iy ah-s l-iy','rb'],
'glory':['g-l-ao1 r-iy','nn'],
'gloss':['g-l-ao1-s','vb nn vbp'],
'glossary':['g-l-ao1 s-er iy','nn'],
'glossed':['g-l-ao1-s-t','vbd vbn'],
'glossy':['g-l-ao1 s-iy','jj'],
'glottal':['g-l-aa1 t-ah-l','jj'],
'glove':['g-l-ah1-v','nn'],
'gloved':['g-l-ah1-v-d','vbn jj'],
'glover':['g-l-ah1 v-er','nn'],
'glow':['g-l-ow1','nn vb'],
'glowed':['g-l-ow1-d','vbd'],
'glowered':['g-l-aw1 er-d','vbd vbn'],
'glowering':['g-l-aw1 er ih-ng','vbg'],
'glowing':['g-l-ow1 ih-ng','vbg jj'],
'glows':['g-l-ow1-z','nns'],
'glucose':['g-l-uw1 k-ow-s','nn'],
'glue':['g-l-uw1','nn'],
'glued':['g-l-uw1-d','vbn vbd'],
'glum':['g-l-ah1-m','jj nn'],
'glumly':['g-l-ah1-m l-iy','rb'],
'glut':['g-l-ah1-t','nn vb'],
'glutamate':['g-l-uw1 t-ah m-ey-t','nn'],
'glutamic':['g-l-uw t-ae1 m-ih-k','jj'],
'glutaric':['g-l-uw t-ae1 r-ih-k','jj'],
'gluten':['g-l-uw1 t-ah-n','nn'],
'gluts':['g-l-ah1-t-s','nns'],
'glutted':['g-l-ah1 t-ih-d','vbn jj'],
'gluttons':['g-l-ah1 t-ah-n-z','nns'],
'gluttony':['g-l-ah1 t-ah-n iy','nn'],
'glycerol':['g-l-ih1 s-er ow-l','nn'],
'glycol':['g-l-ay1 k-ao-l','nn'],
'gnarled':['n-aa1-r-l-d','jj'],
'gnashing':['n-ae1 sh-ih-ng','vbg'],
'gnaw':['n-ao1','vb'],
'gnawed':['n-ao1-d','vbd'],
'gnawing':['n-ao1 ih-ng','nn'],
'gnomes':['n-ow1-m-z','nns'],
'go':['g-ow1','vb jj nn rp vbp'],
'goad':['g-ow1-d','nn'],
'goaded':['g-ow1 d-ih-d','vbd vbn'],
'goal':['g-ow1-l','nn'],
'goat':['g-ow1-t','nn'],
'goatee':['g-ow1 t-iy1','nn'],
'gob':['g-aa1-b','nn'],
'gobble':['g-aa1 b-ah-l','nn vb'],
'gobbled':['g-aa1 b-ah-l-d','vbd vbn'],
'gobbledygook':['g-aa b-ah-l d-iy g-uh1-k','nn'],
'gobblers':['g-aa1 b-ah-l er-z','nns'],
'gobbling':['g-aa1 b-ah-l ih-ng','vbg'],
'god':['g-aa1-d','nn'],
'goddam':['g-aa1 d-d-ae-m','jj'],
'goddamn':['g-aa1 d-d-ae-m','uh vbn jj'],
'goddess':['g-aa1 d-ah-s','nn'],
'godfather':['g-aa1-d f-aa dh-er','nn'],
'godhead':['g-aa1-d hh-eh-d','nn'],
'godless':['g-aa1-d l-ah-s','jj'],
'godmother':['g-aa1-d m-ah dh-er','nn'],
'godsend':['g-aa1-d s-eh-n-d','nn'],
'goes':['g-ow1-z','vbz'],
'goggles':['g-aa1 g-ah-l-z','nns'],
'goin':['g-oy1-n','vbg nn vb'],
'going':['g-ow1 ih-ng','vbg jj nn'],
'gold':['g-ow1-l-d','nn jj'],
'golden':['g-ow1-l d-ah-n','jj'],
'goldfish':['g-ow1-l-d f-ih-sh','nn'],
'goldsmith':['g-ow1-l-d s-m-ih-th','nn'],
'goldstock':['g-ow1-l-d s-t-aa-k','nn'],
'golf':['g-aa1-l-f','nn'],
'golfed':['g-aa1-l-f-t','vbn'],
'golfer':['g-aa1-l f-er','nn'],
'golfing':['g-aa1-l f-ih-ng','nn vbg'],
'golly':['g-aa1 l-iy','uh'],
'gone':['g-ao1-n','vbn jj'],
'gonna':['g-aa1 n-ah','vbg'],
'goo':['g-uw1','nn'],
'good':['g-uh1-d','jj nn rb'],
'goodbye':['g-uh-d b-ay1','nn uh'],
'goodness':['g-uh1-d n-ah-s','nn'],
'goodnight':['g-uh1-d n-ay-t','nn'],
'goodwill':['g-uh1-d w-ih1-l','nn'],
'goody':['g-uh1 d-iy','uh'],
'gooey':['g-uw1 iy','jj'],
'goofed':['g-uw1-f-t','vbd'],
'goofing':['g-uw1 f-ih-ng','vbg'],
'goofy':['g-uw1 f-iy','jj'],
'google':['g-uw1 g-ah-l','nn vb'],
'goons':['g-uw1-n-z','nns'],
'goose':['g-uw1-s','nn'],
'gooseberry':['g-uw1-s b-eh r-iy','nn'],
'goosey':['g-uw1 s-iy','jj'],
'gore':['g-ao1-r','vb'],
'gored':['g-ao1-r-d','vbn'],
'gorge':['g-ao1-r-jh','nn'],
'gorgeous':['g-ao1-r jh-ah-s','jj'],
'gorges':['g-ao1-r jh-ah-z','nns vbz'],
'gorilla':['g-er ih1 l-ah','nn'],
'gory':['g-ao1 r-iy','jj'],
'gosh':['g-aa1-sh','uh'],
'gospel':['g-aa1 s-p-ah-l','nn'],
'gossamer':['g-aa1 s-ah m-er','nn'],
'gossip':['g-aa1 s-ah-p','nn'],
'gossiping':['g-aa1 s-ah p-ih-ng','vbg'],
'gossips':['g-aa1 s-ah-p-s','nns vbz'],
'gossipy':['g-aa1 s-ah p-iy','jj'],
'got':['g-aa1-t','vbd vbn vbp vb'],
'gothic':['g-aa1 th-ih-k','jj'],
'gotta':['g-aa1 t-ah','vb vbn vbp to'],
'gotten':['g-aa1 t-ah-n','vbn'],
'gouge':['g-aw1-jh','vb vbp'],
'gouged':['g-aw1-jh-d','vbd'],
'gouging':['g-aw1 jh-ih-ng','vbg nn'],
'gourd':['g-ao1-r-d','nn'],
'gourmet':['g-uh1-r m-ey','nn jj'],
'gout':['g-aw1-t','nn'],
'gouty':['g-aw1 t-iy','jj'],
'govern':['g-ah1 v-er-n','vb vbp'],
'governance':['g-ah1 v-er n-ah-n-s','nn'],
'governed':['g-ah1 v-er-n-d','vbn jj vbd'],
'governess':['g-ah1 v-er n-ah-s','nn'],
'governing':['g-ah1 v-er n-ih-ng','vbg jj nn'],
'government':['g-ah1 v-er m-ah-n-t','nn'],
'governmental':['g-ah1 v-er m-eh-n t-ah-l','jj'],
'governmentally':['g-ah1 v-er m-eh-n t-ah l-iy','rb'],
'governor':['g-ah1 v-er n-er','nn'],
'governorship':['g-ah1 v-er n-er sh-ih-p','nn'],
'governs':['g-ah1 v-er-n-z','vbz'],
'gown':['g-aw1-n','nn'],
'grab':['g-r-ae1-b','vb jj nn vbp'],
'grabbed':['g-r-ae1-b-d','vbd vbn'],
'grabbing':['g-r-ae1 b-ih-ng','vbg nn'],
'grabs':['g-r-ae1-b-z','nns vbz'],
'grace':['g-r-ey1-s','nn vb vbp'],
'graced':['g-r-ey1-s-t','vbd vbn'],
'graceful':['g-r-ey1-s f-ah-l','jj'],
'gracefully':['g-r-ey1-s f-ah l-iy','rb'],
'graces':['g-r-ey1 s-ih-z','nns'],
'gracious':['g-r-ey1 sh-ah-s','jj'],
'graciously':['g-r-ey1 sh-ah-s l-iy','rb'],
'grad':['g-r-ae1-d','nn'],
'grade':['g-r-ey1-d','nn vb'],
'graded':['g-r-ey1 d-ah-d','vbn'],
'grader':['g-r-ey1 d-er','nn'],
'grades':['g-r-ey1-d-z','nns'],
'grading':['g-r-ey1 d-ih-ng','vbg nn'],
'gradual':['g-r-ae1 jh-uw ah-l','jj rb'],
'gradualism':['g-r-ae1 jh-ah-w ah l-ih z-ah-m','nn'],
'gradualist':['g-r-ae1 jh-ah-w ah l-ih-s-t','nn'],
'gradually':['g-r-ae1 jh-uw ah l-iy','rb'],
'graduate':['g-r-ae1 jh-ah-w ah-t','nn jj vb vbp'],
'graduated':['g-r-ae1 jh-uw ey t-ih-d','vbn vbd jj'],
'graduates':['g-r-ae1 jh-ah-w ah-t-s','nns vbz'],
'graduating':['g-r-ae1 jh-ah-w ey t-ih-ng','vbg nn'],
'graduation':['g-r-ae jh-uw ey1 sh-ah-n','nn'],
'graffiti':['g-r-ah f-iy1 t-iy','nn'],
'graft':['g-r-ae1-f-t','nn vb'],
'grafted':['g-r-ae1-f t-ih-d','jj vbn'],
'grafting':['g-r-ae1-f t-ih-ng','vbg'],
'graham':['g-r-ey1 ah-m','nn'],
'grain':['g-r-ey1-n','nn'],
'grained':['g-r-ey1-n-d','jj'],
'grainy':['g-r-ey1 n-iy','jj'],
'gram':['g-r-ae1-m','nn'],
'grammar':['g-r-ae1 m-er','nn'],
'grammatical':['g-r-ah m-ae1 t-ah k-ah-l','jj'],
'grand':['g-r-ae1-n-d','jj'],
'grandchild':['g-r-ae1-n-d ch-ay-l-d','nn'],
'granddaughter':['g-r-ae1-n d-ao t-er','nn'],
'grander':['g-r-ae1-n d-er','jjr rbr'],
'grandest':['g-r-ae1-n d-ah-s-t','jjs'],
'grandeur':['g-r-ae-n d-uw1-r','nn'],
'grandfather':['g-r-ae1-n-d f-aa dh-er','nn vb'],
'grandfathers':['g-r-ae1-n-d f-aa dh-er-z','nns'],
'grandiloquent':['g-r-ae-n d-ih1 l-ah k-w-ah-n-t','jj'],
'grandiose':['g-r-ae-n d-iy ow1-s','jj'],
'grandkids':['g-r-ae1-n-d k-ih-d-z','nns'],
'grandly':['g-r-ae1-n-d l-iy','rb'],
'grandma':['g-r-ae1 m-aa','nn'],
'grandmother':['g-r-ae1-n-d m-ah dh-er','nn'],
'grandmotherly':['g-r-ae1-n-d m-ah dh-er l-iy','jj'],
'grandparents':['g-r-ae1-n-d p-eh r-ah-n-t-s','nns'],
'grandson':['g-r-ae1-n-d s-ah-n','nn'],
'grandstand':['g-r-ae1-n-d s-t-ae-n-d','nn'],
'grandstanding':['g-r-ae1-n-d s-t-ae-n d-ih-ng','vbg nn'],
'granite':['g-r-ae1 n-ah-t','nn'],
'grant':['g-r-ae1-n-t','nn vbp vb'],
'granted':['g-r-ae1-n t-ah-d','vbn jj vbd'],
'granting':['g-r-ae1-n t-ih-ng','vbg nn'],
'grantors':['g-r-ae1-n t-er-z','nns'],
'grants':['g-r-ae1-n-t-s','nns vbz'],
'granular':['g-r-ae1 n-y-ah l-er','jj'],
'grape':['g-r-ey1-p','nn'],
'grapefruit':['g-r-ey1-p f-r-uw-t','nn nns'],
'grapeshot':['g-r-ey1-p sh-aa-t','nn'],
'grapevine':['g-r-ey1-p v-ay-n','nn'],
'graph':['g-r-ae1-f','nn'],
'graphic':['g-r-ae1 f-ih-k','jj'],
'graphical':['g-r-ae1 f-ih k-ah-l','jj'],
'graphically':['g-r-ae1 f-ih-k l-iy','rb'],
'graphics':['g-r-ae1 f-ih-k-s','nns nn'],
'graphite':['g-r-ae1 f-ay-t','nn'],
'grapple':['g-r-ae1 p-ah-l','vb'],
'grappled':['g-r-ae1 p-ah-l-d','vbd'],
'grapples':['g-r-ae1 p-ah-l-z','vbz'],
'grappling':['g-r-ae1 p-l-ih-ng','vbg nn'],
'grasp':['g-r-ae1-s-p','vb vbp nn'],
'grasped':['g-r-ae1-s-p-t','vbn vbd'],
'grasping':['g-r-ae1 s-p-ih-ng','vbg'],
'grass':['g-r-ae1-s','nn'],
'grassed':['g-r-ae1-s-t','vbn'],
'grassers':['g-r-ae1 s-er-z','nns'],
'grasshoppers':['g-r-ae1-s hh-aa p-er-z','nns'],
'grassland':['g-r-ae1-s l-ae-n-d','nn'],
'grassroots':['g-r-ae1-s r-uw1-t-s','nns'],
'grassy':['g-r-ae1 s-iy','jj'],
'grate':['g-r-ey1-t','nn'],
'grated':['g-r-ey1 t-ih-d','vbd jj'],
'grateful':['g-r-ey1-t f-ah-l','jj'],
'gratefully':['g-r-ey1-t f-ah l-iy','rb'],
'gratification':['g-r-ae t-ah f-ah k-ey1 sh-ah-n','nn'],
'gratified':['g-r-ae1 t-ah f-ay-d','vbn jj'],
'gratify':['g-r-ae1 t-ah f-ay','vb'],
'gratifying':['g-r-ae1 t-ah f-ay ih-ng','jj vbg'],
'grating':['g-r-ey1 t-ih-ng','nn'],
'gratis':['g-r-ae1 t-ah-s','jj rb'],
'gratitude':['g-r-ae1 t-ah t-uw-d','nn'],
'gratuitous':['g-r-ah t-uw1 ah t-ah-s','jj'],
'gratuitously':['g-r-ah t-uw1 ah t-ah-s l-iy','rb'],
'gratuity':['g-r-ah t-uw1 ih t-iy','nn'],
'grave':['g-r-ey1-v','jj nn'],
'gravel':['g-r-ae1 v-ah-l','nn'],
'gravely':['g-r-ey1-v l-iy','rb'],
'graven':['g-r-ey1 v-ah-n','jj'],
'graver':['g-r-ey1 v-er','jjr rbr'],
'gravest':['g-r-ae1 v-ah-s-t','jjs'],
'graveyard':['g-r-ey1-v y-aa-r-d','nn'],
'gravitates':['g-r-ae1 v-ih t-ey-t-s','vbz'],
'gravitating':['g-r-ae1 v-ih t-ey t-ih-ng','vbg'],
'gravitational':['g-r-ae v-ih t-ey1 sh-ah n-ah-l','jj'],
'gravity':['g-r-ae1 v-ah t-iy','nn'],
'gravy':['g-r-ey1 v-iy','nn'],
'gray':['g-r-ey1','jj nn'],
'grayer':['g-r-ey1 er','jjr'],
'graying':['g-r-ey1 ih-ng','vbg nn'],
'graze':['g-r-ey1-z','vbp vb'],
'grazer':['g-r-ey1 z-er','nn'],
'grazing':['g-r-ey1 z-ih-ng','vbg jj nn'],
'grease':['g-r-iy1-s','nn'],
'greased':['g-r-iy1-s-t','vbd vbn'],
'greasy':['g-r-iy1 s-iy','jj'],
'great':['g-r-ey1-t','jj rb'],
'greater':['g-r-ey1 t-er','jjr rbr'],
'greatest':['g-r-ey1 t-ah-s-t','jjs'],
'greatly':['g-r-ey1-t l-iy','rb'],
'greatness':['g-r-ey1-t n-ah-s','nn'],
'greed':['g-r-iy1-d','nn'],
'greedily':['g-r-iy1 d-ah l-iy','rb'],
'greedy':['g-r-iy1 d-iy','jj'],
'green':['g-r-iy1-n','jj nn vb'],
'greenback':['g-r-iy1-n b-ae-k','nn'],
'greener':['g-r-iy1 n-er','jjr'],
'greenfield':['g-r-iy1-n f-iy-l-d','nn'],
'greenhouse':['g-r-iy1-n hh-aw-s','nn'],
'greenhouses':['g-r-iy1-n hh-aw s-ih-z','nns'],
'greening':['g-r-iy1 n-ih-ng','jj nn vbg'],
'greenish':['g-r-iy1 n-ih-sh','jj'],
'greenly':['g-r-iy1-n l-iy','rb'],
'greenmail':['g-r-iy1-n m-ey-l','nn'],
'greenmailer':['g-r-iy1-n m-ey l-er','nn'],
'greenness':['g-r-iy1-n n-ah-s','nn'],
'greens':['g-r-iy1-n-z','nns'],
'greet':['g-r-iy1-t','vb vbp'],
'greeted':['g-r-iy1 t-ah-d','vbd vbn'],
'greeting':['g-r-iy1 t-ih-ng','nn vbg jj'],
'greets':['g-r-iy1-t-s','vbz'],
'gregarious':['g-r-ah g-eh1 r-iy ah-s','jj'],
'grenade':['g-r-ah n-ey1-d','nn'],
'grew':['g-r-uw1','vbd vbn'],
'grey':['g-r-ey1','jj nn'],
'grid':['g-r-ih1-d','nn'],
'gridiron':['g-r-ih1-d ay er-n','nn'],
'gridlock':['g-r-ih1-d l-aa-k','nn'],
'gridlocked':['g-r-ih1-d l-aa-k-t','vbn jj'],
'grief':['g-r-iy1-f','nn'],
'grievance':['g-r-iy1 v-ah-n-s','nn'],
'grieving':['g-r-iy1 v-ih-ng','vbg'],
'grievous':['g-r-iy1 v-ah-s','jj'],
'grill':['g-r-ih1-l','nn vb vbp'],
'grille':['g-r-ih1-l','nn'],
'grilled':['g-r-ih1-l-d','jj vbn'],
'grim':['g-r-ih1-m','jj'],
'grimaced':['g-r-ih1 m-ah-s-t','vbd'],
'grimaces':['g-r-ih1 m-ah s-ih-z','nns'],
'grime':['g-r-ay1-m','nn'],
'grimly':['g-r-ih1-m l-iy','rb'],
'grimmer':['g-r-ih1 m-er','rbr'],
'grimmest':['g-r-ih1 m-ah-s-t','jjs'],
'grimness':['g-r-ih1-m n-ah-s','nn'],
'grimy':['g-r-ay1 m-iy','jj nn'],
'grin':['g-r-ih1-n','nn vb'],
'grind':['g-r-ay1-n-d','vbp nn vb'],
'grinder':['g-r-ay1-n d-er','nn'],
'grinding':['g-r-ay1-n d-ih-ng','vbg nn'],
'grinds':['g-r-ay1-n-d-z','vbz nns'],
'grindstone':['g-r-ay1-n-d s-t-ow-n','nn'],
'gringos':['g-r-ih1-ng g-ow-z','nns'],
'grinned':['g-r-ih1-n-d','vbd'],
'grinning':['g-r-ih1 n-ih-ng','vbg jj'],
'grins':['g-r-ih1-n-z','nns'],
'grip':['g-r-ih1-p','nn vbp'],
'gripe':['g-r-ay1-p','vbp nn vb'],
'griped':['g-r-ay1-p-t','vbd'],
'gripes':['g-r-ay1-p-s','nns vbz'],
'griping':['g-r-ay1 p-ih-ng','nn'],
'gripped':['g-r-ih1-p-t','vbd vbn'],
'gripping':['g-r-ih1 p-ih-ng','vbg jj'],
'grisly':['g-r-ih1-z l-iy','jj'],
'grist':['g-r-ih1-s-t','nn'],
'grit':['g-r-ih1-t','nn vb'],
'grits':['g-r-ih1-t-s','nns'],
'gritty':['g-r-ih1 t-iy','jj'],
'grizzled':['g-r-ih1 z-ah-l-d','jj'],
'grizzly':['g-r-ih1-z l-iy','nn'],
'groan':['g-r-ow1-n','nn'],
'groaned':['g-r-ow1-n-d','vbd'],
'groaning':['g-r-ow1 n-ih-ng','vbg'],
'groans':['g-r-ow1-n-z','vbz'],
'grocer':['g-r-ow1 s-er','nn'],
'grocery':['g-r-ow1 s-er iy','nn'],
'groggy':['g-r-aa1 g-iy','jj'],
'groin':['g-r-oy1-n','nn'],
'groom':['g-r-uw1-m','nn'],
'groomed':['g-r-uw1-m-d','vbn'],
'grooming':['g-r-uw1 m-ih-ng','nn vbg'],
'groove':['g-r-uw1-v','nn'],
'grope':['g-r-ow1-p','vb vbp'],
'groped':['g-r-ow1-p-t','vbd'],
'groping':['g-r-ow1 p-ih-ng','vbg nn'],
'gross':['g-r-ow1-s','jj nn vb'],
'grossed':['g-r-ow1-s-t','vbd'],
'grosses':['g-r-ow1 s-ih-z','vbz'],
'grossing':['g-r-ow1 s-ih-ng','vbg'],
'grossly':['g-r-ow1-s l-iy','rb'],
'grotesque':['g-r-ow t-eh1-s-k','jj'],
'grotesquely':['g-r-ow t-eh1-s-k l-iy','rb'],
'ground':['g-r-aw1-n-d','nn jj vb vbd vbn'],
'grounded':['g-r-aw1-n d-ih-d','vbn vbd jj'],
'grounding':['g-r-aw1-n d-ih-ng','vbg nn'],
'groundless':['g-r-aw1-n-d l-ah-s','jj'],
'grounds':['g-r-aw1-n-d-z','nns vbz'],
'groundwater':['g-r-aw1-n-d w-aa t-er','nn'],
'groundwork':['g-r-aw1-n-d w-er-k','nn'],
'group':['g-r-uw1-p','nn vb vbp'],
'grouped':['g-r-uw1-p-t','vbn'],
'grouper':['g-r-uw1 p-er','nn'],
'grouping':['g-r-uw1 p-ih-ng','nn vbg'],
'groups':['g-r-uw1-p-s','nns'],
'grouse':['g-r-aw1-s','vbp nn'],
'groused':['g-r-aw1-s-t','vbd'],
'grouses':['g-r-aw1 s-ih-z','vbz'],
'grousing':['g-r-aw1 s-ih-ng','vbg'],
'grove':['g-r-ow1-v','nn'],
'grovel':['g-r-aa1 v-ah-l','vb'],
'groveling':['g-r-ao1 v-ah-l ih-ng','vbg nn'],
'grow':['g-r-ow1','vb vbp'],
'grower':['g-r-ow1 er','nn'],
'growing':['g-r-ow1 ih-ng','vbg jj nn'],
'growl':['g-r-aw1-l','nn'],
'growled':['g-r-aw1-l-d','vbd'],
'growling':['g-r-ow1 l-ih-ng','vbg'],
'growls':['g-r-aw1-l-z','vbz'],
'grown':['g-r-ow1-n','vbn vbd jj'],
'grownups':['g-r-ow1 n-ah-p-s','nns'],
'grows':['g-r-ow1-z','vbz'],
'growth':['g-r-ow1-th','nn jj vb'],
'growths':['g-r-ow1-th-s','nns'],
'grub':['g-r-ah1-b','nn'],
'grubby':['g-r-ah1 b-iy','jj'],
'grudge':['g-r-ah1-jh','nn'],
'grudging':['g-r-ah1 jh-ih-ng','jj'],
'grudgingly':['g-r-ah1 jh-ih-ng l-iy','rb'],
'grueling':['g-r-uw1 ih l-ih-ng','jj vbg'],
'gruesome':['g-r-uw1 s-ah-m','jj'],
'gruff':['g-r-ah1-f','jj'],
'grumble':['g-r-ah1-m b-ah-l','vbp vb'],
'grumbled':['g-r-ah1-m b-ah-l-d','vbd vbn'],
'grumbles':['g-r-ah1-m b-ah-l-z','vbz'],
'grumbling':['g-r-ah1-m b-ah-l ih-ng','vbg nn'],
'grunt':['g-r-ah1-n-t','vb nn'],
'grunted':['g-r-ah1-n t-ih-d','vbd'],
'guarantee':['g-eh r-ah-n t-iy1','nn vb vbp'],
'guaranteed':['g-eh r-ah-n t-iy1-d','vbn jj vbd'],
'guaranteeing':['g-eh r-ah-n t-iy1 ih-ng','vbg nn'],
'guarantees':['g-eh r-ah-n t-iy1-z','nns vbz'],
'guarantor':['g-eh r-ah-n t-ao1-r','nn'],
'guaranty':['g-eh r-ah-n t-iy1','nn'],
'guard':['g-aa1-r-d','nn jj vb vbp'],
'guarded':['g-aa1-r d-ah-d','vbn jj vbd'],
'guardedly':['g-aa1-r d-ih-d l-iy','rb'],
'guardian':['g-aa1-r d-iy ah-n','nn'],
'guardianship':['g-aa1-r d-iy ah-n sh-ih-p','nn'],
'guarding':['g-aa1-r d-ih-ng','vbg'],
'guards':['g-aa1-r-d-z','nns vbz'],
'gubernatorial':['g-uw b-er n-ah t-ao1 r-iy ah-l','jj'],
'guerilla':['g-er ih1 l-ah','nn'],
'guerrilla':['g-er ih1 l-ah','nn jj'],
'guess':['g-eh1-s','vbp nn vb'],
'guessed':['g-eh1-s-t','vbd vbn'],
'guesses':['g-eh1 s-ah-z','nns vbz'],
'guessing':['g-eh1 s-ih-ng','vbg nn'],
'guesstimates':['g-eh1 s-t-ih m-ih-t-s','nns'],
'guesswork':['g-eh1-s w-er-k','nn'],
'guest':['g-eh1-s-t','nn jj'],
'guffaws':['g-ah f-ao1-z','nns'],
'guidance':['g-ay1 d-ah-n-s','nn'],
'guide':['g-ay1-d','nn vbp vb'],
'guidebook':['g-ay1-d b-uh-k','nn'],
'guided':['g-ay1 d-ah-d','vbn vbd jj'],
'guideline':['g-ay1-d l-ay-n','nn'],
'guidepost':['g-ay1-d p-ow-s-t','nn'],
'guides':['g-ay1-d-z','nns vbz'],
'guiding':['g-ay1 d-ih-ng','vbg jj'],
'guild':['g-ih1-l-d','nn'],
'guilder':['g-ih1-l d-er','nn'],
'guile':['g-ay1-l','nn'],
'guileless':['g-ay1-l l-ah-s','jj'],
'guillotine':['g-ih1 l-ah t-iy-n','nn'],
'guilt':['g-ih1-l-t','nn'],
'guilty':['g-ih1-l t-iy','jj rb'],
'guinea':['g-ih1 n-iy','nn'],
'guise':['g-ay1-z','nn'],
'guises':['g-ay1 z-ah-z','nns'],
'guitar':['g-ih t-aa1-r','nn'],
'guitarist':['g-ih t-aa1 r-ah-s-t','nn'],
'gulag':['g-y-uw1 l-ae-g','nn'],
'gulf':['g-ah1-l-f','nn'],
'gull':['g-ah1-l','nn'],
'gullet':['g-ah1 l-ah-t','nn'],
'gulley':['g-ah1 l-iy','nn'],
'gullibility':['g-ah l-ah b-ih1 l-ih t-iy','nn'],
'gullible':['g-ah1 l-ah b-ah-l','jj'],
'gully':['g-ah1 l-iy','nn'],
'gulp':['g-ah1-l-p','nn'],
'gulped':['g-ah1-l-p-t','vbd'],
'gulping':['g-ah1-l p-ih-ng','vbg'],
'gum':['g-ah1-m','nn vb'],
'gummed':['g-ah1-m-d','vbn'],
'gummy':['g-ah1 m-iy','jj'],
'gumption':['g-ah1-m-p sh-ah-n','nn'],
'gums':['g-ah1-m-z','nns'],
'gun':['g-ah1-n','nn vb'],
'gunboats':['g-ah1-n b-ow-t-s','nns'],
'gunfighter':['g-ah1-n f-ay t-er','nn'],
'gunfights':['g-ah1-n f-ay-t-s','nns'],
'gunfire':['g-ah1-n f-ay er','nn'],
'gunflint':['g-ah1-n f-l-ih-n-t','nn'],
'gunman':['g-ah1-n m-ah-n','nn'],
'gunned':['g-ah1-n-d','vbn vbd'],
'gunner':['g-ah1 n-er','nn'],
'gunning':['g-ah1 n-ih-ng','vbg nn'],
'gunny':['g-ah1 n-iy','nn'],
'gunpoint':['g-ah1-n p-oy-n-t','nn'],
'gunpowder':['g-ah1-n p-aw d-er','nn'],
'guns':['g-ah1-n-z','nns vbz'],
'gunship':['g-ah1-n sh-ih-p','nn'],
'gunshot':['g-ah1-n sh-aa-t','nn'],
'gunslinger':['g-ah1-n s-l-ih ng-er','nn'],
'guppies':['g-ah1 p-iy-z','nns'],
'gurgle':['g-er1 g-ah-l','nn'],
'gurgling':['g-er1 g-ah-l ih-ng','vbg'],
'guru':['g-uw1 r-uw','nn'],
'gush':['g-ah1-sh','vbp vb'],
'gushed':['g-ah1-sh-t','vbd'],
'gusher':['g-ah1 sh-er','nn'],
'gushes':['g-ah1 sh-ih-z','vbz'],
'gushing':['g-ah1 sh-ih-ng','vbg'],
'gussied':['g-ah1 s-iy-d','vbn'],
'gust':['g-ah1-s-t','nn rb'],
'gusto':['g-ah1 s-t-ow','nn'],
'gusty':['g-ah1 s-t-iy','jj'],
'gut':['g-ah1-t','nn vb'],
'guts':['g-ah1-t-s','nns'],
'gutsy':['g-ah1-t s-iy1','jj'],
'gutted':['g-ah1 t-ih-d','vbn vbd'],
'gutter':['g-ah1 t-er','nn'],
'guttered':['g-ah1 t-er-d','vbd'],
'guttural':['g-ah1 t-er ah-l','jj nn'],
'guy':['g-ay1','nn'],
'guzzle':['g-ah1 z-ah-l','vb'],
'guzzler':['g-ah1-z l-er','nn'],
'guzzling':['g-ah1 z-ah-l ih-ng','nn'],
'gym':['jh-ih1-m','nn'],
'gymnast':['jh-ih1-m n-ah-s-t','nn'],
'gymnastic':['jh-ih-m n-ae1 s-t-ih-k','jj'],
'gymnastics':['jh-ih-m n-ae1 s-t-ih-k-s','nns'],
'gynecologic':['g-ay n-ah k-ah l-aa1 jh-ih-k','jj'],
'gynecological':['g-ay n-ah k-ah l-aa1 jh-ih k-ah-l','jj'],
'gynecologist':['g-ay n-ah k-aa1 l-ah jh-ah-s-t','nn'],
'gynecology':['g-ay n-ah k-aa1 l-ah jh-iy','nn'],
'gypsum':['jh-ih1-p s-ah-m','nn'],
'gypsy':['jh-ih1-p s-iy','nn'],
'gyrate':['jh-ay1 r-ey-t','vb'],
'gyrated':['jh-ay1 r-ey t-ih-d','vbd vbn'],
'gyrating':['jh-ay1 r-ey t-ih-ng','vbg'],
'gyrations':['jh-ay r-ey1 sh-ah-n-z','nns'],
'gyro':['jh-ay1 r-ow','nn'],
'gyroscopes':['jh-ay1 r-ah s-k-ow-p-s','nns'],
'h':['ey1-ch','nn'],
'haberdashery':['hh-ae1 b-er d-ae sh-er iy','nn'],
'habit':['hh-ae1 b-ah-t','nn'],
'habitable':['hh-ae1 b-ah t-ah b-ah-l','jj'],
'habitat':['hh-ae1 b-ah t-ae-t','nn'],
'habitual':['hh-ah b-ih1 ch-uw ah-l','jj'],
'habitually':['hh-ah b-ih1 ch-uw ah l-iy','rb'],
'habitues':['hh-ae1 b-ih ch-uw-z','nns'],
'hable':['hh-ey1 b-ah-l','jj'],
'hack':['hh-ae1-k','nn vb'],
'hacked':['hh-ae1-k-t','vbd vbn'],
'hacker':['hh-ae1 k-er','nn'],
'hacking':['hh-ae1 k-ih-ng','jj nn'],
'hackles':['hh-ae1 k-ah-l-z','nns'],
'hackneyed':['hh-ae1-k n-iy-d','jj'],
'hacks':['hh-ae1-k-s','nns'],
'hacksaw':['hh-ae1-k s-ao','nn'],
'had':['hh-ae1-d','vbd vbn'],
'hadd':['hh-ae1-d','vbn'],
'haddock':['hh-ae1 d-ah-k','nn'],
'hafta':['hh-ae1-f t-ah','vb vbp'],
'haggard':['hh-ae1 g-er-d','jj'],
'haggle':['hh-ae1 g-ah-l','vb nn'],
'haggling':['hh-ae1 g-ah l-ih-ng','vbg nn'],
'hail':['hh-ey1-l','nn vb vbp'],
'hailed':['hh-ey1-l-d','vbd vbn'],
'hailing':['hh-ey1 l-ih-ng','vbg'],
'hails':['hh-ey1-l-z','vbz'],
'hailstorm':['hh-ey1-l s-t-ao-r-m','nn'],
'hair':['hh-eh1-r','nn'],
'haircut':['hh-eh1-r k-ah-t','nn'],
'hairdos':['hh-eh1-r d-uw-z','nns'],
'hairdresser':['hh-eh1-r d-r-eh s-er','nn'],
'hairless':['hh-eh1-r l-ah-s','jj'],
'hairline':['hh-eh1-r l-ay-n','nn'],
'hairspray':['hh-eh1-r s-p-r-ey','nn'],
'hairy':['hh-eh1 r-iy','jj'],
'halcyon':['hh-ae1-l s-iy ah-n','jj'],
'halfback':['hh-ae1-f b-ae-k','nn'],
'halfhearted':['hh-ae1-f hh-aa-r t-ih-d','jj'],
'halftime':['hh-ae1-f t-ay-m','nn'],
'halfway':['hh-ae1-f w-ey1','rb jj'],
'hall':['hh-ao1-l','nn'],
'hallmark':['hh-aa1-l m-aa-r-k','nn'],
'hallowed':['hh-ae1 l-ow-d','jj'],
'hallucinations':['hh-ah l-uw s-ah n-ey1 sh-ah-n-z','nns'],
'hallucinatory':['hh-ah l-uw1 s-ah n-ah t-ao r-iy','jj'],
'hallway':['hh-ao1-l w-ey','nn'],
'halo':['hh-ey1 l-ow','nn'],
'halogen':['hh-ae1 l-ah jh-ah-n','nn'],
'halogenated':['hh-ae1 l-ah jh-ah n-ey t-ih-d','vbd'],
'halt':['hh-ao1-l-t','nn jj vb vbp'],
'halted':['hh-ao1-l t-ah-d','vbn nn vbd'],
'halter':['hh-ao1-l t-er','nn'],
'halting':['hh-ao1-l t-ih-ng','vbg jj'],
'haltingly':['hh-ao1-l t-ih-ng l-iy','rb'],
'halts':['hh-ao1-l-t-s','nns vbz'],
'halve':['hh-ae1-v','vb'],
'halved':['hh-ae1-v-d','vbn jj vbd'],
'halves':['hh-ae1-v-z','nns vbz'],
'halving':['hh-ae1 v-ih-ng','vbg'],
'ham':['hh-ae1-m','nn'],
'hamburger':['hh-ae1-m b-er g-er','nn'],
'hamlet':['hh-ae1-m l-ah-t','nn'],
'hammer':['hh-ae1 m-er','nn vbp vb'],
'hammered':['hh-ae1 m-er-d','vbn vbd'],
'hammering':['hh-ae1 m-er ih-ng','vbg nn'],
'hammerlock':['hh-ae1 m-er l-aa-k','nn'],
'hammock':['hh-ae1 m-ah-k','nn'],
'hamper':['hh-ae1-m p-er','vb nn'],
'hampered':['hh-ae1-m p-er-d','vbn vbd'],
'hampering':['hh-ae1-m p-er ih-ng','vbg'],
'hampers':['hh-ae1-m p-er-z','vbz'],
'hamstring':['hh-ae1-m s-t-r-ih-ng','vb'],
'hamstrung':['hh-ae1-m s-t-r-ah-ng','jj vbn vbp'],
'han':['hh-aa1-n','nn'],
'hand':['hh-ae1-n-d','nn rb vb vbp jj'],
'handbag':['hh-ae1-n-d b-ae-g','nn'],
'handbills':['hh-ae1-n-d b-ih-l-z','nns'],
'handbook':['hh-ae1-n-d b-uh-k','nn'],
'handclasp':['hh-ae1-n-d k-l-ae-s-p','nn'],
'handcrafted':['hh-ae1-n-d k-r-ae-f t-ih-d','vbn'],
'handcuff':['hh-ae1-n-d k-ah-f','vbp'],
'handcuffed':['hh-ae1-n-d k-ah-f-t','vbn'],
'handed':['hh-ae1-n d-ah-d','vbd vbn'],
'hander':['hh-ae1-n d-er','nn'],
'handful':['hh-ae1-n-d f-uh-l','nn'],
'handgun':['hh-ae1-n-d g-ah-n','nn'],
'handheld':['hh-ae1-n-d hh-eh1-l-d','jj'],
'handicap':['hh-ae1-n d-iy k-ae-p','nn vb'],
'handicapped':['hh-ae1-n d-iy k-ae-p-t','jj vbn nns'],
'handicaps':['hh-ae1-n d-iy k-ae-p-s','nns'],
'handicrafts':['hh-ae1-n d-iy k-r-ae-f-t-s','nns'],
'handier':['hh-ae1-n d-iy er','jjr'],
'handily':['hh-ae1-n d-ah l-iy','rb'],
'handing':['hh-ae1-n d-ih-ng','vbg'],
'handiwork':['hh-ae1-n d-iy w-er-k','nn'],
'handkerchief':['hh-ae1-ng k-er ch-ih-f','nn'],
'handle':['hh-ae1-n d-ah-l','vb nn vbp'],
'handlebars':['hh-ae1-n d-ah-l b-aa-r-z','nns'],
'handled':['hh-ae1-n d-ah-l-d','vbn vbd'],
'handlers':['hh-ae1-n-d l-er-z','nns'],
'handles':['hh-ae1-n d-ah-l-z','vbz nns'],
'handling':['hh-ae1-n-d l-ih-ng','nn vbg'],
'handmade':['hh-ae1-n-d m-ey1-d','jj'],
'handout':['hh-ae1-n-d aw-t','nn'],
'handpicked':['hh-ae1-n-d p-ih1-k-t','vbn vbd'],
'hands':['hh-ae1-n-d-z','nns vbz'],
'handset':['hh-ae1-n-d s-eh-t','nn'],
'handshake':['hh-ae1-n-d sh-ey-k','nn'],
'handsome':['hh-ae1-n s-ah-m','jj'],
'handsomely':['hh-ae1-n s-ah-m l-iy','rb'],
'handstands':['hh-ae1-n-d s-t-ae-n-d-z','nns'],
'handwriting':['hh-ae1-n-d r-ay t-ih-ng','nn'],
'handwritten':['hh-ae1-n-d r-ih t-ah-n','jj'],
'handy':['hh-ae1-n d-iy','jj'],
'handyman':['hh-ae1-n d-iy m-ae-n','nn'],
'hang':['hh-ae1-ng','vb jj vbp nn'],
'hangar':['hh-ae1 ng-er','nn'],
'hanged':['hh-ae1-ng-d','vbn vbd'],
'hangers':['hh-ae1 ng-er-z','nns'],
'hanging':['hh-ae1 ng-ih-ng','vbg jj nn'],
'hangman':['hh-ae1-ng m-ah-n','nn'],
'hangout':['hh-ae1-ng aw-t','nn'],
'hangover':['hh-ae1-ng ow v-er','nn'],
'hangs':['hh-ae1-ng-z','vbz'],
'hank':['hh-ae1-ng-k','nn'],
'hanky':['hh-ae1-ng k-iy','nn'],
'hansom':['hh-ae1-n s-ah-m','jj'],
'haphazard':['hh-ae-p hh-ae1 z-er-d','jj'],
'haphazardly':['hh-ae1-f ah z-er-d l-iy','rb'],
'hapless':['hh-ae1-p l-ah-s','jj'],
'happen':['hh-ae1 p-ah-n','vb vbp'],
'happened':['hh-ae1 p-ah-n-d','vbd vbn'],
'happening':['hh-ae1 p-ah n-ih-ng','vbg nn'],
'happens':['hh-ae1 p-ah-n-z','vbz'],
'happenstance':['hh-ae1 p-ah-n s-t-ae-n-s','nn'],
'happier':['hh-ae1 p-iy er','jjr'],
'happiest':['hh-ae1 p-iy ah-s-t','jjs'],
'happily':['hh-ae1 p-ah l-iy','rb'],
'happiness':['hh-ae1 p-iy n-ah-s','nn'],
'happy':['hh-ae1 p-iy','jj'],
'harangued':['hh-er ae1-ng-d','vbd'],
'harangues':['hh-er ae1-ng-z','nns vbz'],
'haranguing':['hh-er ae1 ng-ih-ng','vbg'],
'harass':['hh-er ae1-s','vb vbp'],
'harassed':['hh-er ae1-s-t','vbd vbn'],
'harassing':['hh-er ae1 s-ih-ng','vbg'],
'harassment':['hh-er ae1-s m-ah-n-t','nn'],
'harbinger':['hh-aa1-r b-ih-n jh-er','nn'],
'harbor':['hh-aa1-r b-er','nn vb vbp'],
'harbored':['hh-aa1-r b-er-d','vbn vbd'],
'harboring':['hh-aa1-r b-er ih-ng','vbg'],
'harbors':['hh-aa1-r b-er-z','nns vbz'],
'harborside':['hh-aa1-r b-er s-ay-d','nn'],
'hard':['hh-aa1-r-d','jj rb'],
'hardball':['hh-aa1-r-d b-ao-l','nn'],
'hardboard':['hh-aa1-r-d b-ao-r-d','nn'],
'hardcore':['hh-aa1-r-d k-ao1-r','jj nn'],
'hardcover':['hh-aa1-r-d k-ah v-er','nn'],
'harden':['hh-aa1-r d-ah-n','vb'],
'hardened':['hh-aa1-r d-ah-n-d','vbn jj vbd'],
'hardener':['hh-aa1-r d-ah-n er','nn'],
'hardening':['hh-aa1-r d-ah-n ih-ng','vbg'],
'harder':['hh-aa1-r d-er','jjr rbr jj rb'],
'hardest':['hh-aa1-r d-ah-s-t','jjs rbs rb'],
'hardier':['hh-aa1-r d-iy er','jjr'],
'hardline':['hh-aa1-r-d l-ay-n','jj nn'],
'hardliner':['hh-aa1-r-d l-ay n-er','nn'],
'hardly':['hh-aa1-r-d l-iy','rb'],
'hardness':['hh-aa1-r-d n-ah-s','nn'],
'hardscrabble':['hh-aa1-r-d s-k-r-ae b-ah-l','jj'],
'hardship':['hh-aa1-r-d sh-ih-p','nn'],
'hardware':['hh-aa1-r-d w-eh-r','nn'],
'hardwood':['hh-aa1-r-d w-uh-d','nn'],
'hardworking':['hh-aa1-r-d w-er k-ih-ng','jj'],
'hardy':['hh-aa1-r d-iy','jj'],
'hare':['hh-eh1-r','nn'],
'harem':['hh-eh1 r-ah-m','nn'],
'hark':['hh-aa1-r-k','vbp'],
'harking':['hh-aa1-r k-ih-ng','vbg'],
'harm':['hh-aa1-r-m','nn vb vbp'],
'harmed':['hh-aa1-r-m-d','vbn vbd'],
'harmful':['hh-aa1-r-m f-ah-l','jj'],
'harming':['hh-aa1-r m-ih-ng','vbg'],
'harmless':['hh-aa1-r-m l-ah-s','jj'],
'harmlessly':['hh-aa1-r-m l-ah-s l-iy','rb'],
'harmonic':['hh-aa-r m-aa1 n-ih-k','jj'],
'harmonious':['hh-aa-r m-ow1 n-iy ah-s','jj'],
'harmonization':['hh-aa-r m-ah n-ih z-ey1 sh-ah-n','nn'],
'harmonize':['hh-aa1-r m-ah n-ay-z','vb'],
'harmony':['hh-aa1-r m-ah n-iy','nn'],
'harms':['hh-aa1-r-m-z','vbz nns'],
'harness':['hh-aa1-r n-ah-s','nn vbp vb'],
'harnessed':['hh-aa1-r n-ah-s-t','vbn'],
'harnessing':['hh-aa1-r n-ah s-ih-ng','vbg'],
'harp':['hh-aa1-r-p','nn vb'],
'harped':['hh-aa1-r-p-t','vbd'],
'harping':['hh-aa1-r p-ih-ng','vbg nn'],
'harpsichord':['hh-aa1-r-p s-ah k-ao-r-d','nn'],
'harried':['hh-eh1 r-iy-d','vbn vbd'],
'harrowing':['hh-eh1 r-ow ih-ng','jj vbg'],
'harry':['hh-eh1 r-iy','vb'],
'harsh':['hh-aa1-r-sh','jj'],
'harsher':['hh-aa1-r sh-er','jjr rbr'],
'harshest':['hh-aa1-r sh-ah-s-t','jjs'],
'harshly':['hh-aa1-r-sh l-iy','rb'],
'harshness':['hh-aa1-r-sh n-ah-s','nn'],
'harvest':['hh-aa1-r v-ah-s-t','nn vb vbp'],
'harvested':['hh-aa1-r v-ah s-t-ah-d','vbn'],
'harvesting':['hh-aa1-r v-ah s-t-ih-ng','nn vbg'],
'harvests':['hh-aa1-r v-ah-s-t-s','nns'],
'has':['hh-ae1-z','vbz vbn .'],
'hash':['hh-ae1-sh','nn vb vbp'],
'hashers':['hh-ae1 sh-er-z','nns'],
'hashing':['hh-ae1 sh-ih-ng','nn vbg'],
'hashish':['hh-ae1 sh-ih-sh','nn'],
'hassle':['hh-ae1 s-ah-l','nn vb'],
'hassled':['hh-ae1 s-ah-l-d','vbn'],
'hassles':['hh-ae1 s-ah-l-z','nns'],
'hast':['hh-ae1-s-t','vbp'],
'haste':['hh-ey1-s-t','nn'],
'hasten':['hh-ey1 s-ah-n','vb vbp'],
'hastened':['hh-ey1 s-ah-n-d','vbd vbn'],
'hastening':['hh-ey1 s-ah-n ih-ng','vbg'],
'hastens':['hh-ey1 s-ah-n-z','vbz'],
'hastily':['hh-ey1 s-t-ah l-iy','rb'],
'hasty':['hh-ey1 s-t-iy','jj'],
'hat':['hh-ae1-t','nn'],
'hatch':['hh-ae1-ch','nn vbp vb'],
'hatchback':['hh-ae1-ch b-ae-k','nn'],
'hatched':['hh-ae1-ch-t','vbn vbd'],
'hatchery':['hh-ae1 ch-er iy','nn'],
'hatchet':['hh-ae1 ch-ah-t','nn'],
'hatching':['hh-ae1 ch-ih-ng','nn vbg'],
'hate':['hh-ey1-t','vbp nn vb'],
'hated':['hh-ey1 t-ah-d','vbd jj vbn'],
'hateful':['hh-ey1-t f-ah-l','jj'],
'haters':['hh-ey1 t-er-z','nns'],
'hates':['hh-ey1-t-s','vbz nns'],
'hath':['hh-ae1-th','vbz'],
'hating':['hh-ey1 t-ih-ng','vbg'],
'hatred':['hh-ey1 t-r-ah-d','nn'],
'haughty':['hh-ao1 t-iy','jj'],
'haul':['hh-ao1-l','nn vb'],
'hauled':['hh-ao1-l-d','vbd vbn'],
'haulers':['hh-ao1 l-er-z','nns'],
'hauling':['hh-ao1 l-ih-ng','vbg nn'],
'hauls':['hh-ao1-l-z','nns vbz'],
'haunt':['hh-ao1-n-t','vb vbp nn'],
'haunted':['hh-ao1-n t-ah-d','vbn vbd jj'],
'haunting':['hh-ao1-n t-ih-ng','jj vbg nn'],
'hauntingly':['hh-ao1-n t-ih-ng l-iy','rb'],
'haunts':['hh-ao1-n-t-s','nns vbz'],
'have':['hh-ae1-v','vbp jj nn vb vbn'],
'haven':['hh-ey1 v-ah-n','nn'],
'haves':['hh-ae1-v-z','nns'],
'having':['hh-ae1 v-ih-ng','vbg'],
'havoc':['hh-ae1 v-ah-k','nn'],
'hawing':['hh-ao1 ih-ng','vbg'],
'hawk':['hh-ao1-k','nn vbp vb'],
'hawked':['hh-ao1-k-t','vbd vbn'],
'hawker':['hh-ao1 k-er','nn'],
'hawking':['hh-ao1 k-ih-ng','vbg nn'],
'hawkish':['hh-ao1 k-ih-sh','jj'],
'hawks':['hh-ao1-k-s','nns vbz'],
'hay':['hh-ey1','nn vb'],
'hayfields':['hh-ey1 f-iy-l-d-z','nns'],
'haystack':['hh-ey1 s-t-ae-k','nn'],
'hazard':['hh-ae1 z-er-d','nn vb'],
'hazardous':['hh-ae1 z-er d-ah-s','jj'],
'hazards':['hh-ae1 z-er-d-z','nns'],
'haze':['hh-ey1-z','nn'],
'hazel':['hh-ey1 z-ah-l','jj'],
'hazelnut':['hh-ey1 z-ah-l n-ah-t','nn'],
'hazy':['hh-ey1 z-iy','jj'],
'he':['hh-iy1','prp vb'],
'head':['hh-eh1-d','nn jj rb vb vbp'],
'headache':['hh-eh1-d ey-k','nn'],
'headboard':['hh-eh1-d b-ao-r-d','nn'],
'headcount':['hh-eh1-d k-aw-n-t','nn'],
'headdress':['hh-eh1-d r-eh-s','nn'],
'headed':['hh-eh1 d-ah-d','vbn vbd'],
'header':['hh-eh1 d-er','nn'],
'headhunters':['hh-eh1-d hh-ah-n t-er-z','nns'],
'heading':['hh-eh1 d-ih-ng','vbg nn'],
'headless':['hh-eh1-d l-ah-s','jj'],
'headlights':['hh-eh1-d l-ay-t-s','nns'],
'headline':['hh-eh1-d l-ay-n','nn'],
'headlined':['hh-eh1-d l-ay-n-d','vbn vbd'],
'headlining':['hh-eh1-d l-ay n-ih-ng','vbg'],
'headlong':['hh-eh1-d l-ao-ng','rb'],
'headmaster':['hh-eh1-d m-ae1 s-t-er','nn'],
'headphones':['hh-eh1-d f-ow-n-z','nns'],
'headquartered':['hh-eh1-d k-ao-r t-er-d','vbn'],
'headquarters':['hh-eh1-d k-w-ao-r t-er-z','nn nns'],
'headrests':['hh-eh1-d r-eh-s-t-s','nns'],
'headroom':['hh-eh1-d r-uw-m','nn'],
'heads':['hh-eh1-d-z','nns vbz'],
'headsets':['hh-eh1-d s-eh-t-s','nns'],
'headship':['hh-eh1-d sh-ih-p','nn'],
'headsman':['hh-eh1-d-z m-ah-n','nn'],
'headstrong':['hh-eh1-d s-t-r-ao-ng','jj'],
'headwaters':['hh-eh1-d w-ao t-er-z','nns'],
'headway':['hh-eh1-d w-ey','nn'],
'heady':['hh-eh1 d-iy','jj'],
'heal':['hh-iy1-l','vb'],
'healed':['hh-iy1-l-d','vbn vbd'],
'healer':['hh-iy1 l-er','nn'],
'healing':['hh-iy1 l-ih-ng','nn jj vbg'],
'health':['hh-eh1-l-th','nn'],
'healthcare':['hh-eh1-l-th k-eh-r','nn'],
'healthful':['hh-eh1-l-th f-ah-l','jj'],
'healthier':['hh-eh1-l th-iy er','jjr rbr'],
'healthiest':['hh-eh1-l th-iy ih-s-t','jjs'],
'healthy':['hh-eh1-l th-iy','jj'],
'heap':['hh-iy1-p','nn'],
'heaped':['hh-iy1-p-t','vbn vbd'],
'heaping':['hh-iy1 p-ih-ng','vbg'],
'hear':['hh-ih1-r','vb vbp'],
'heard':['hh-er1-d','vbn vbd'],
'hearers':['hh-ih1 r-er-z','nns'],
'hearing':['hh-ih1 r-ih-ng','nn vbg'],
'hears':['hh-ih1-r-z','vbz'],
'hearsay':['hh-ih1-r s-ey','nn'],
'hearse':['hh-er1-s','nn'],
'heart':['hh-aa1-r-t','nn rb vb'],
'heartbeat':['hh-aa1-r-t b-iy-t','nn'],
'heartbreak':['hh-aa1-r-t b-r-ey-k','nn'],
'heartbreaking':['hh-aa1-r-t b-r-ey k-ih-ng','jj'],
'heartburn':['hh-aa1-r-t b-er-n','nn'],
'heartened':['hh-aa1-r t-ah-n-d','vbn'],
'heartening':['hh-aa1-r t-ah-n ih-ng','jj'],
'heartfelt':['hh-aa1-r-t f-eh-l-t','jj'],
'hearth':['hh-aa1-r-th','nn'],
'heartily':['hh-aa1-r t-ah l-iy','rb'],
'heartland':['hh-aa1-r-t l-ae-n-d','nn'],
'heartless':['hh-aa1-r-t l-ah-s','jj'],
'hearts':['hh-aa1-r-t-s','nns'],
'heartthrob':['hh-aa1-r-t th-r-aa-b','nn'],
'hearty':['hh-aa1-r t-iy','jj'],
'heat':['hh-iy1-t','nn vb vbp'],
'heated':['hh-iy1 t-ah-d','vbn jj vbd'],
'heatedly':['hh-iy1 t-ih-d l-iy','rb'],
'heater':['hh-iy1 t-er','nn'],
'heath':['hh-iy1-th','nn'],
'heathen':['hh-iy1 dh-ah-n','jj nn'],
'heather':['hh-eh1 dh-er','nn'],
'heating':['hh-iy1 t-ih-ng','nn vbg'],
'heats':['hh-iy1-t-s','vbz'],
'heave':['hh-iy1-v','nn'],
'heaved':['hh-iy1-v-d','vbd vbn'],
'heaven':['hh-eh1 v-ah-n','nn'],
'heavenly':['hh-eh1 v-ah-n l-iy','jj'],
'heavens':['hh-eh1 v-ah-n-z','nns uh'],
'heaves':['hh-iy1-v-z','nn vbd'],
'heavier':['hh-eh1 v-iy er','jjr rbr'],
'heaviest':['hh-eh1 v-iy ah-s-t','jjs'],
'heavily':['hh-eh1 v-ah l-iy','rb'],
'heaving':['hh-iy1 v-ih-ng','vbg'],
'heavy':['hh-eh1 v-iy','jj nn rb'],
'heavyweight':['hh-eh1 v-iy w-ey-t','nn jj'],
'heck':['hh-eh1-k','nn uh'],
'heckled':['hh-eh1 k-ah-l-d','vbn'],
'hectares':['hh-eh1-k t-aa-r-z','nns'],
'hectic':['hh-eh1-k t-ih-k','jj'],
'hedge':['hh-eh1-jh','vb jj nn vbp'],
'hedged':['hh-eh1-jh-d','vbn vbd'],
'hedgehogs':['hh-eh1-jh hh-aa-g-z','nns'],
'hedgers':['hh-eh1 jh-er-z','nns'],
'hedges':['hh-eh1 jh-ih-z','nns'],
'hedging':['hh-eh1 jh-ih-ng','vbg jj nn'],
'hedonism':['hh-iy1 d-ah n-ih z-ah-m','nn'],
'hedonistic':['hh-iy d-ah n-ih1 s-t-ih-k','jj'],
'heed':['hh-iy1-d','vb vbp nn'],
'heeded':['hh-iy1 d-ah-d','vbd vbn'],
'heeds':['hh-iy1-d-z','vbz'],
'heel':['hh-iy1-l','nn'],
'heft':['hh-eh1-f-t','nn'],
'heftier':['hh-eh1-f t-iy er','jjr'],
'heftiest':['hh-eh1-f t-iy ah-s-t','jjs'],
'hefty':['hh-eh1-f t-iy','jj'],
'hegemonic':['hh-eh g-ah m-aa1 n-ih-k','jj'],
'hegemony':['hh-iy jh-eh1 m-ah n-iy','nn'],
'heifers':['hh-eh1 f-er-z','nns'],
'height':['hh-ay1-t','nn'],
'heighten':['hh-ay1 t-ah-n','vb'],
'heightened':['hh-ay1 t-ah-n-d','vbn vbd jj'],
'heightening':['hh-ay1 t-ah-n ih-ng','vbg nn'],
'heightens':['hh-ay1 t-ah-n-z','vbz'],
'heinous':['hh-ey1 n-ah-s','jj'],
'heir':['eh1-r','nn'],
'heiress':['eh1 r-ah-s','nn'],
'heist':['hh-ay1-s-t','nn'],
'held':['hh-eh1-l-d','vbn vbd jj'],
'helicopter':['hh-eh1 l-ih k-aa-p t-er','nn'],
'heliotrope':['hh-iy1 l-iy ah t-r-ow-p','nn'],
'helium':['hh-iy1 l-iy ah-m','nn'],
'helix':['hh-iy1 l-ih-k-s','nn'],
'hell':['hh-eh1-l','nn uh'],
'hellfire':['hh-eh1-l f-ay-r','nn'],
'hello':['hh-ah l-ow1','uh nn'],
'helluva':['hh-eh l-uw1 v-ah','jj'],
'helm':['hh-eh1-l-m','nn'],
'helmet':['hh-eh1-l m-ah-t','nn'],
'helmsman':['hh-eh1-l-m-z m-ae-n','nn'],
'help':['hh-eh1-l-p','vb nn vbp'],
'helped':['hh-eh1-l-p-t','vbd vbn vbp'],
'helper':['hh-eh1-l p-er','nn'],
'helpful':['hh-eh1-l-p f-ah-l','jj'],
'helpfully':['hh-eh1-l-p f-ah l-iy','rb'],
'helping':['hh-eh1-l p-ih-ng','vbg nn'],
'helpless':['hh-eh1-l-p l-ah-s','jj'],
'helplessly':['hh-eh1-l-p l-ah-s l-iy','rb'],
'helplessness':['hh-eh1-l-p l-ah-s n-ah-s','nn'],
'helps':['hh-eh1-l-p-s','vbz vbp nns'],
'hem':['hh-eh1-m','nn'],
'hemisphere':['hh-eh1 m-ih s-f-ih-r','nn'],
'hemispheric':['hh-eh m-ah s-f-ih1 r-ih-k','jj'],
'hemlines':['hh-eh1-m l-ay-n-z','nns'],
'hemmed':['hh-eh1-m-d','vbn'],
'hemming':['hh-eh1 m-ih-ng','vbg'],
'hemoglobin':['hh-iy m-ah g-l-ow1 b-ah-n','nn'],
'hemolytic':['hh-iy m-ah l-ih1 t-ih-k','jj'],
'hemophilia':['hh-iy m-ah f-iy1 l-iy ah','nn'],
'hemophiliacs':['hh-iy m-ah f-ih1 l-iy ae-k-s','nns'],
'hemorrhage':['hh-eh1 m-er ih-jh','nn'],
'hemorrhaged':['hh-eh1 m-er ih-jh-d','vbn'],
'hemorrhaging':['hh-eh1 m-er ih jh-ih-ng','vbg jj nn'],
'hemorrhoids':['hh-eh1 m-er oy-d-z','nns'],
'hen':['hh-eh1-n','nn'],
'hence':['hh-eh1-n-s','rb'],
'henceforth':['hh-eh1-n-s f-ao1-r-th','rb'],
'henchman':['hh-eh1-n-ch m-ah-n','nn'],
'henpecked':['hh-eh1-n p-eh-k-t','jj'],
'hepatitis':['hh-eh p-ah t-ay1 t-ah-s','nn nnp'],
'her':['hh-er','prp$ prp prp$r'],
'herald':['hh-eh1 r-ah-l-d','vb vbp'],
'heralded':['hh-eh1 r-ah-l d-ih-d','vbn vbd'],
'heraldic':['hh-eh r-ae1-l d-ih-k','jj'],
'heralding':['hh-eh1 r-ah-l d-ih-ng','vbg'],
'herb':['er1-b','nn'],
'herbaceous':['er b-ey1 sh-ah-s','jj'],
'herbal':['er1 b-ah-l','jj'],
'herbicide':['hh-er1 b-ih s-ay-d','nn'],
'herculean':['hh-er k-y-uw1 l-iy ah-n','jj'],
'herd':['hh-er1-d','nn vb'],
'herded':['hh-er1 d-ih-d','vbn vbd'],
'herding':['hh-er1 d-ih-ng','vbg nn'],
'herds':['hh-er1-d-z','nns'],
'herdsmen':['hh-er1-d-z m-ih-n','nns'],
'here':['hh-ih1-r','rb'],
'hereabout':['hh-ih1 r-ah b-aw-t','jj'],
'hereabouts':['hh-ih1 r-ah b-aw-t-s','rb'],
'hereafter':['hh-ih r-ae1-f t-er','rb'],
'hereby':['hh-ih-r b-ay1','rb'],
'hereditary':['hh-er eh1 d-ah t-eh r-iy','jj'],
'heredity':['hh-er eh1 d-ah t-iy','nn'],
'herein':['hh-ih r-ih1-n','rb'],
'heresy':['hh-eh1 r-ah s-iy','nn'],
'heretical':['hh-er eh1 t-ih k-ah-l','jj'],
'heretofore':['hh-ih-r t-ah f-ao1-r','rb'],
'herewith':['hh-ih1-r w-ih1-th','rb'],
'heritage':['hh-eh1 r-ah t-ah-jh','nn'],
'hermetically':['hh-er m-eh1 t-ih k-ah l-iy','rb'],
'hero':['hh-ih1 r-ow','nn'],
'heroic':['hh-ih r-ow1 ih-k','jj'],
'heroically':['hh-ih r-ow1 ih-k l-iy','rb'],
'heroin':['hh-eh1 r-ow ah-n','nn'],
'heroine':['hh-eh1 r-ow ah-n','nn'],
'heroism':['hh-eh1 r-ow ih z-ah-m','nn'],
'herons':['hh-eh1 r-ah-n-z','nns'],
'herring':['hh-eh1 r-ih-ng','nn'],
'hers':['hh-er-z','prp jj'],
'herself':['hh-er s-eh1-l-f','prp'],
'hesitancy':['hh-eh1 z-ih t-ah-n s-iy','nn'],
'hesitant':['hh-eh1 z-ih t-ah-n-t','jj'],
'hesitantly':['hh-eh1 z-ih t-ah-n-t l-iy','rb'],
'hesitate':['hh-eh1 z-ah t-ey-t','vb vbp'],
'hesitated':['hh-eh1 z-ih t-ey t-ih-d','vbd vbn'],
'hesitates':['hh-eh1 z-ah t-ey-t-s','vbz'],
'hesitating':['hh-eh1 z-ah t-ey t-ih-ng','vbg'],
'hesitation':['hh-eh z-ah t-ey1 sh-ah-n','nn'],
'heterogeneity':['hh-eh t-er ah jh-ih n-iy1 ah t-iy','nn'],
'heterogeneous':['hh-eh t-er ah jh-iy1 n-y-ah-s','jj'],
'heterozygous':['hh-eh t-er ah z-ay1 g-ah-s','jj'],
'hewn':['hh-y-uw1-n','vbn'],
'hews':['hh-y-uw1-z','vbz'],
'hexagon':['hh-eh1-k s-ah g-aa-n','nn'],
'hexagonal':['hh-eh-k s-ae1 g-ah n-ah-l','jj'],
'hey':['hh-ey1','uh'],
'heyday':['hh-ey1 d-ey','nn'],
'hiatus':['hh-ay ey1 t-ah-s','nn'],
'hibachi':['hh-ah b-aa1 ch-iy','nn'],
'hibernate':['hh-ay1 b-er n-ey-t','vbp vb'],
'hiccup':['hh-ih1 k-ah-p','nn'],
'hick':['hh-ih1-k','nn'],
'hickory':['hh-ih1 k-er iy','nn'],
'hid':['hh-ih1-d','vbd vbn'],
'hidden':['hh-ih1 d-ah-n','vbn jj'],
'hide':['hh-ay1-d','vb vbp nn'],
'hideaway':['hh-ay1 d-ah w-ey','nn'],
'hidebound':['hh-ay1-d b-aw-n-d','jj'],
'hideous':['hh-ih1 d-iy ah-s','jj'],
'hideously':['hh-ih1 d-iy ah-s l-iy','rb'],
'hideout':['hh-ay1 d-aw-t','nn'],
'hides':['hh-ay1-d-z','nns vbz'],
'hiding':['hh-ay1 d-ih-ng','vbg nn'],
'hierarchical':['hh-ay r-aa1-r k-ah k-ah-l','jj'],
'hierarchy':['hh-ay1 er aa-r k-iy','nn'],
'high':['hh-ay1','jj nn rb rp'],
'highboy':['hh-ay1 b-oy','nn'],
'highbrow':['hh-ay1 b-r-aw','jj'],
'higher':['hh-ay1 er','jjr rb rbr'],
'highest':['hh-ay1 ah-s-t','jjs rb rbs'],
'highfalutin':['hh-ay f-ah l-uw1 t-ih-n','jj'],
'highflying':['hh-ay1 f-l-ay ih-ng','jj'],
'highland':['hh-ay1 l-ah-n-d','nn'],
'highlight':['hh-ay1 l-ay-t','vb vbp nn'],
'highlighted':['hh-ay1 l-ay t-ih-d','vbn vbd'],
'highlighting':['hh-ay1 l-ay t-ih-ng','vbg'],
'highlights':['hh-ay1 l-ay-t-s','vbz nns'],
'highly':['hh-ay1 l-iy','rb'],
'highness':['hh-ay1 n-ah-s','nn'],
'highway':['hh-ay1 w-ey','nn'],
'highways':['hh-ay1 w-ey-z','nns jj'],
'highyield':['hh-ay1 y-iy1-l-d','jj'],
'hijacked':['hh-ay1 jh-ae-k-t','vbn vbd'],
'hijacker':['hh-ay1 jh-ae k-er','nn'],
'hijacking':['hh-ay1 jh-ae k-ih-ng','nn vbg'],
'hike':['hh-ay1-k','nn vb'],
'hiked':['hh-ay1-k-t','vbd vbn'],
'hiker':['hh-ay1 k-er','nn'],
'hikes':['hh-ay1-k-s','nns'],
'hiking':['hh-ay1 k-ih-ng','vbg nn'],
'hilarious':['hh-ih l-eh1 r-iy ah-s','jj'],
'hilariously':['hh-ih l-eh1 r-iy ah-s l-iy','rb'],
'hilarity':['hh-ih l-eh1 r-ah t-iy','nn'],
'hill':['hh-ih1-l','nn'],
'hillbilly':['hh-ih1-l b-ih l-iy','nn'],
'hillside':['hh-ih1-l s-ay-d','nn'],
'hilltop':['hh-ih1-l t-aa-p','nn'],
'hilly':['hh-ih1 l-iy','jj'],
'hilt':['hh-ih1-l-t','nn'],
'him':['hh-ih1-m','prp'],
'himself':['hh-ih-m s-eh1-l-f','prp'],
'hind':['hh-ay1-n-d','jj'],
'hinder':['hh-ih1-n d-er','vb'],
'hindered':['hh-ih1-n d-er-d','vbn vbd'],
'hindering':['hh-ih1-n d-er ih-ng','vbg jj'],
'hinders':['hh-ih1-n d-er-z','vbz'],
'hindquarters':['hh-ay1-n-d k-w-ao-r t-er-z','nns'],
'hindrance':['hh-ih1-n d-r-ah-n-s','nn'],
'hindsight':['hh-ay1-n-d s-ay-t','nn'],
'hinge':['hh-ih1-n-jh','vb nn'],
'hinged':['hh-ih1-n-jh-d','vbn'],
'hinges':['hh-ih1-n jh-ih-z','nns vbz'],
'hint':['hh-ih1-n-t','nn vbp vb'],
'hinted':['hh-ih1-n t-ah-d','vbd vbn'],
'hinterlands':['hh-ih1-n t-er l-ae-n-d-z','nns'],
'hinting':['hh-ih1-n t-ih-ng','vbg'],
'hints':['hh-ih1-n-t-s','nns vbz'],
'hip':['hh-ih1-p','nn jj'],
'hipper':['hh-ih1 p-er','jjr'],
'hippie':['hh-ih1 p-iy','nn'],
'hire':['hh-ay1 er','vb vbp nn'],
'hired':['hh-ay1 er-d','vbn jj vbd'],
'hires':['hh-ay1 er-z','vbz nns'],
'hiring':['hh-ay1 r-ih-ng','vbg nn'],
'his':['hh-ih1-z','prp$ prp'],
'hiss':['hh-ih1-s','nns'],
'hissed':['hh-ih1-s-t','vbd'],
'hissing':['hh-ih1 s-ih-ng','nn vbg'],
'histology':['hh-ih s-t-aa1 l-ah jh-iy','nn'],
'historian':['hh-ih s-t-ao1 r-iy ah-n','nn'],
'historic':['hh-ih s-t-ao1 r-ih-k','jj'],
'historical':['hh-ih s-t-ao1 r-ih k-ah-l','jj'],
'historically':['hh-ih s-t-ao1 r-ih k-ah l-iy','rb'],
'historiography':['hh-ih s-t-ao r-iy aa1 g-r-ah f-iy','nn'],
'history':['hh-ih1 s-t-er iy','nn'],
'histrionics':['hh-ih-s t-r-iy aa1 n-ih-k-s','nns'],
'hit':['hh-ih1-t','vbd jj nn vb vbn vbp'],
'hitch':['hh-ih1-ch','nn'],
'hitched':['hh-ih1-ch-t','vbn vbd'],
'hitches':['hh-ih1 ch-ih-z','nns vbz'],
'hitching':['hh-ih1 ch-ih-ng','nn vbg'],
'hither':['hh-ih1 dh-er','rb'],
'hitherto':['hh-ih1 dh-er t-uw1','rb'],
'hits':['hh-ih1-t-s','nns vbz'],
'hitter':['hh-ih1 t-er','nn'],
'hitting':['hh-ih1 t-ih-ng','vbg nn'],
'hive':['hh-ay1-v','nn'],
'hoard':['hh-ao1-r-d','nn vb vbp'],
'hoarding':['hh-ao1-r d-ih-ng','nn vbg'],
'hoards':['hh-ao1-r-d-z','nns'],
'hoarse':['hh-ao1-r-s','jj'],
'hoarseness':['hh-ao1-r-s n-ah-s','nn'],
'hoaxes':['hh-ow1-k s-ih-z','nns'],
'hob':['hh-aa1-b','nn'],
'hobbing':['hh-aa1 b-ih-ng','vbg'],
'hobble':['hh-aa1 b-ah-l','vb'],
'hobbled':['hh-aa1 b-ah-l-d','vbn'],
'hobbles':['hh-aa1 b-ah-l-z','vbz'],
'hobbling':['hh-aa1 b-ah-l ih-ng','vbg'],
'hobby':['hh-aa1 b-iy','nn'],
'hobbyist':['hh-aa1 b-iy ih-s-t','nn'],
'hobnob':['hh-aa1-b n-aa-b','vb'],
'hock':['hh-aa1-k','nn vb'],
'hockey':['hh-aa1 k-iy','nn'],
'hocking':['hh-aa1 k-ih-ng','vbg'],
'hodgepodge':['hh-aa1-jh p-aa-jh','nn'],
'hoes':['hh-ow1-z','nns'],
'hog':['hh-aa1-g','nn nns'],
'hoist':['hh-oy1-s-t','vb'],
'hoisted':['hh-oy1 s-t-ah-d','vbn vbd'],
'hoisting':['hh-oy1 s-t-ih-ng','nn'],
'hokey':['hh-ow1 k-iy','jj'],
'hold':['hh-ow1-l-d','vb nn rb vbp'],
'holder':['hh-ow1-l d-er','nn'],
'holding':['hh-ow1-l d-ih-ng','vbg jj nn'],
'holdouts':['hh-ow1-l-d aw-t-s','nns'],
'holdovers':['hh-ow1-l-d ow v-er-z','nns'],
'holds':['hh-ow1-l-d-z','vbz nns'],
'holdup':['hh-ow1-l-d ah-p','nn'],
'hole':['hh-ow1-l','nn vbp vb'],
'holed':['hh-ow1-l-d','vbn vbd'],
'holes':['hh-ow1-l-z','nns'],
'holiday':['hh-aa1 l-ah d-ey','nn'],
'holidays':['hh-aa1 l-ah d-ey-z','nns vbz'],
'holiest':['hh-ow1 l-iy ih-s-t','jjs'],
'holistic':['hh-ow l-ih1 s-t-ih-k','jj'],
'holler':['hh-aa1 l-er','vb'],
'hollered':['hh-aa1 l-er-d','vbd'],
'hollering':['hh-aa1 l-er ih-ng','vbg jj'],
'hollers':['hh-aa1 l-er-z','vbz'],
'hollow':['hh-aa1 l-ow','jj nn'],
'hollyhocks':['hh-aa1 l-iy hh-aa-k-s','nns'],
'holocaust':['hh-aa1 l-ah k-ao-s-t','nn'],
'holster':['hh-ow1-l s-t-er','nn'],
'holy':['hh-ow1 l-iy','jj'],
'homage':['aa1 m-ah-jh','nn'],
'home':['hh-ow1-m','nn vbp rb vb'],
'homebuilder':['hh-ow1-m b-ih-l d-er','nn'],
'homebuilding':['hh-ow1-m b-ih-l d-ih-ng','nn'],
'homecoming':['hh-ow1-m k-ah m-ih-ng','nn'],
'homeland':['hh-ow1-m l-ae-n-d','nn'],
'homeless':['hh-ow1-m l-ah-s','jj nn'],
'homelessness':['hh-ow1-m l-ah-s n-ah-s','nn'],
'homely':['hh-ow1-m l-iy','jj'],
'homemade':['hh-ow1-m m-ey1-d','jj'],
'homemaker':['hh-ow1-m m-ey k-er','nn'],
'homeowner':['hh-ow1 m-ow n-er','nn jjr'],
'homeownership':['hh-ow1 m-ow n-er sh-ih-p','nn'],
'homer':['hh-ow1 m-er','nn vb'],
'homers':['hh-ow1 m-er-z','nns'],
'homes':['hh-ow1-m-z','nns'],
'homesick':['hh-ow1-m s-ih-k','jj'],
'homesickness':['hh-ow1-m s-ih-k n-ah-s','nn'],
'homespun':['hh-ow1-m s-p-ah-n','jj'],
'homestead':['hh-ow1-m s-t-eh-d','nn'],
'hometown':['hh-ow1-m t-aw-n','nn jj'],
'homeward':['hh-ow1-m w-er-d','rb'],
'homework':['hh-ow1-m w-er-k','nn'],
'homey':['hh-ow1 m-iy','jj'],
'homicidal':['hh-aa m-ah s-ay1 d-ah-l','jj'],
'homicide':['hh-aa1 m-ah s-ay-d','nn'],
'homilies':['hh-ow1 m-ah l-iy-z','nns'],
'homing':['hh-ow1 m-ih-ng','jj vbg'],
'homogeneity':['hh-aa m-ah jh-ah n-iy1 ah t-iy','nn'],
'homogeneous':['hh-ow m-ah jh-iy1 n-iy ah-s','jj'],
'homogenized':['hh-ow m-aa1 jh-ah n-ay-z-d','vbn'],
'homogenous':['hh-ah m-aa1 jh-ah n-ah-s','jj'],
'homologous':['hh-ah m-aa1 l-ah g-ah-s','rb'],
'homophobia':['hh-ow m-ah f-ow1 b-iy ah','nn'],
'homosexual':['hh-ow m-ow s-eh1-k sh-ah w-ah-l','jj nn'],
'homosexuality':['hh-ow m-ow s-eh-k sh-ah-w ae1 l-ah t-iy','nn'],
'homozygous':['hh-ow m-ow z-ay1 g-ah-s','jj'],
'hon':['hh-aa1-n','nn'],
'hone':['hh-ow1-n','vb vbp'],
'honed':['hh-ow1-n-d','vbn jj'],
'honest':['aa1 n-ah-s-t','jj'],
'honestly':['aa1 n-ah-s-t l-iy','rb'],
'honesty':['aa1 n-ah s-t-iy','nn'],
'honey':['hh-ah1 n-iy','nn vb uh jj'],
'honeybee':['hh-ah1 n-iy b-iy','nn'],
'honeymoon':['hh-ah1 n-iy m-uw-n','nn vb'],
'honeymooned':['hh-ah1 n-iy m-uw-n-d','vbd vbn'],
'honeymooners':['hh-ah1 n-iy m-uw n-er-z','nns'],
'honk':['hh-aa1-ng-k','vbp'],
'honor':['aa1 n-er','nn vbp vb'],
'honorable':['aa1 n-er ah b-ah-l','jj nn'],
'honorably':['aa1 n-er ah b-l-iy','rb'],
'honorarium':['aa n-er eh1 r-iy ah-m','nn'],
'honorary':['aa1 n-er eh r-iy','jj'],
'honored':['aa1 n-er-d','vbn jj vbd'],
'honorific':['aa n-er ih1 f-ih-k','nn'],
'honoring':['aa1 n-er ih-ng','vbg'],
'honors':['aa1 n-er-z','nns vbz jj'],
'hooch':['hh-uw1-ch','nn'],
'hood':['hh-uh1-d','nn'],
'hoodlum':['hh-uh1-d l-ah-m','nn'],
'hoodwinked':['hh-uh1-d w-ih-ng-k-t','vbn'],
'hoof':['hh-uw1-f','nn'],
'hook':['hh-uh1-k','nn vb'],
'hooked':['hh-uh1-k-t','vbn jj vbd'],
'hooker':['hh-uh1 k-er','nn'],
'hooking':['hh-uh1 k-ih-ng','vbg'],
'hooks':['hh-uh1-k-s','nns vbz'],
'hookup':['hh-uh1-k ah-p','nn'],
'hookworm':['hh-uh1-k w-er-m','nn'],
'hooliganism':['hh-uw1 l-ih g-ah n-ih z-ah-m','nn'],
'hoop':['hh-uw1-p','nn'],
'hoopla':['hh-uw1 p-l-aa','nn'],
'hoot':['hh-uw1-t','nn vbp'],
'hooted':['hh-uw1 t-ih-d','vbd vbn'],
'hooves':['hh-uh1-v-z','nns'],
'hop':['hh-aa1-p','nn vb'],
'hope':['hh-ow1-p','nn vb vbd vbp'],
'hoped':['hh-ow1-p-t','vbd vbn'],
'hopeful':['hh-ow1-p f-ah-l','jj nn'],
'hopefully':['hh-ow1-p f-ah l-iy','rb'],
'hopeless':['hh-ow1-p l-ah-s','jj'],
'hopelessly':['hh-ow1-p l-ah-s l-iy','rb'],
'hopelessness':['hh-ow1-p l-ah-s n-ah-s','nn'],
'hopes':['hh-ow1-p-s','vbz nns'],
'hoping':['hh-ow1 p-ih-ng','vbg'],
'hopped':['hh-aa1-p-t','vbd'],
'hopper':['hh-aa1 p-er','nn'],
'hopping':['hh-aa1 p-ih-ng','vbg'],
'hops':['hh-aa1-p-s','vbz nns'],
'hopscotch':['hh-aa1-p s-k-aa-ch','nn'],
'horde':['hh-ao1-r-d','nn'],
'horizon':['hh-er ay1 z-ah-n','nn'],
'horizontal':['hh-ao r-ah z-aa1-n t-ah-l','jj nn'],
'horizontally':['hh-ao r-ih z-aa1-n t-ah l-iy','rb'],
'hormonal':['hh-ao1-r m-ow n-ah-l','jj'],
'hormone':['hh-ao1-r m-ow-n','nn'],
'horn':['hh-ao1-r-n','nn vb'],
'horned':['hh-ao1-r-n-d','jj'],
'hornet':['hh-ao1-r n-ih-t','nn'],
'horns':['hh-ao1-r-n-z','nns'],
'horoscope':['hh-ao1 r-ah s-k-ow-p','nn'],
'horrendous':['hh-ao r-eh1-n d-ah-s','jj'],
'horrible':['hh-ao1 r-ah b-ah-l','jj nn'],
'horribly':['hh-ao1 r-ah b-l-iy','rb'],
'horrid':['hh-ao1 r-ah-d','jj'],
'horrific':['hh-ao r-ih1 f-ih-k','jj'],
'horrified':['hh-ao1 r-ah f-ay-d','vbn jj'],
'horrifying':['hh-ao1 r-ah f-ay ih-ng','jj'],
'horror':['hh-ao1 r-er','nn'],
'horse':['hh-ao1-r-s','nn'],
'horseback':['hh-ao1-r-s b-ae-k','nn jj rb'],
'horseflesh':['hh-ao1-r-s f-l-eh-sh','nn'],
'horseman':['hh-ao1-r-s m-ah-n','nn'],
'horsemanship':['hh-ao1-r-s m-ah-n sh-ih-p','nn'],
'horseplay':['hh-ao1-r-s p-l-ey','nn'],
'horsepower':['hh-ao1-r-s p-aw er','nn'],
'horseradish':['hh-ao1-r-s r-ae d-ih-sh','nn'],
'horses':['hh-ao1-r s-ah-z','nns'],
'horticultural':['hh-ao-r t-ah k-ah1-l ch-er ah-l','jj'],
'horticulture':['hh-ao1-r t-ih k-ah-l ch-er','nn'],
'horticulturist':['hh-ao-r t-ih k-ah1-l ch-er ih-s-t','nn'],
'hose':['hh-ow1-z','nn vb'],
'hoses':['hh-ow1 z-ih-z','nns vbz'],
'hospice':['hh-aa1-s p-ah-s','nn'],
'hospitable':['hh-aa1 s-p-ih1 t-ah b-ah-l','jj'],
'hospital':['hh-aa1-s p-ih t-ah-l','nn'],
'hospitality':['hh-aa-s p-ah t-ae1 l-ah t-iy','nn'],
'hospitalization':['hh-aa-s p-ih t-ah-l ah z-ey1 sh-ah-n','nn'],
'hospitalized':['hh-aa1-s p-ih t-ah-l ay-z-d','vbn jj'],
'hoss':['hh-aa1-s','nn nns'],
'host':['hh-ow1-s-t','nn vb'],
'hostage':['hh-aa1 s-t-ih-jh','nn'],
'hosted':['hh-ow1 s-t-ih-d','vbn jj vbd'],
'hostess':['hh-ow1 s-t-ah-s','nn'],
'hostile':['hh-aa1 s-t-ah-l','jj nn'],
'hostility':['hh-aa s-t-ih1 l-ah t-iy','nn'],
'hosting':['hh-ow1 s-t-ih-ng','vbg'],
'hostler':['hh-aa1-s l-er','nn'],
'hosts':['hh-ow1-s-t-s','nns vbz'],
'hot':['hh-aa1-t','jj'],
'hotbed':['hh-aa1-t b-eh-d','nn'],
'hotdog':['hh-aa1-t d-ao-g','nn'],
'hotel':['hh-ow t-eh1-l','nn'],
'hotelier':['hh-ow t-eh1 l-y-er','nn'],
'hothouse':['hh-aa1-t hh-aw-s','jj'],
'hotline':['hh-aa1-t l-ay-n','nn'],
'hotly':['hh-aa1-t l-iy','rb'],
'hotter':['hh-aa1 t-er','jjr'],
'hottest':['hh-aa1 t-ah-s-t','jjs'],
'hound':['hh-aw1-n-d','nn'],
'hounded':['hh-aw1-n d-ih-d','vbd'],
'hounding':['hh-aw1-n d-ih-ng','vbg'],
'hour':['aw1 er','nn'],
'hourlong':['aw1-r l-ao-ng','jj'],
'hourly':['aw1-r l-iy','jj'],
'hours':['aw1 er-z','nns vbz'],
'house':['hh-aw1-s','nn vbp vb'],
'houseboat':['hh-aw1-s b-ow-t','nn'],
'housebroken':['hh-aw1-s b-r-ow k-ah-n','jj'],
'housecleaning':['hh-aw1-s-k l-iy n-ih-ng','nn'],
'housed':['hh-aw1-z-d','vbn vbd'],
'houseful':['hh-aw1-s f-ah-l','nn'],
'household':['hh-aw1-s hh-ow-l-d','nn jj'],
'householder':['hh-aw1-s hh-ow-l d-er','nn'],
'housekeeper':['hh-aw1-s k-iy p-er','nn'],
'housekeeping':['hh-aw1-s k-iy p-ih-ng','nn'],
'houseman':['hh-aw1-s m-ah-n','nn'],
'houses':['hh-aw1 s-ah-z','nns vbz'],
'housewares':['hh-aw1-s w-eh-r-z','nns'],
'housewife':['hh-aw1-s w-ay-f','nn'],
'housework':['hh-aw1-s w-er-k','nn'],
'housing':['hh-aw1 z-ih-ng','nn vbg jj'],
'hove':['hh-ow1-v','vbd'],
'hovel':['hh-ah1 v-ah-l','nn'],
'hover':['hh-ah1 v-er','vb vbp'],
'hovered':['hh-ah1 v-er-d','vbd vbn'],
'hovering':['hh-ah1 v-er ih-ng','vbg'],
'hovers':['hh-ah1 v-er-z','vbz'],
'how':['hh-aw1','wrb'],
'howdy':['hh-aw1 d-iy','uh'],
'howe':['hh-aw1','wrb'],
'however':['hh-aw eh1 v-er','rb rbr wrb'],
'howitzer':['hh-aw1 ah-t s-er','nn'],
'howl':['hh-aw1-l','nn vb'],
'howled':['hh-aw1-l-d','vbd'],
'howling':['hh-aw1 l-ih-ng','vbg'],
'howls':['hh-aw1-l-z','nns vbz'],
'hub':['hh-ah1-b','nn'],
'hubbub':['hh-ah1 b-ah-b','nn'],
'hubby':['hh-ah1 b-iy','nn'],
'hubris':['hh-y-uw1 b-r-ah-s','nn'],
'huckster':['hh-ah1-k s-t-er','nn vb'],
'huddle':['hh-ah1 d-ah-l','nn'],
'huddled':['hh-ah1 d-ah-l-d','vbd vbn jj'],
'huddling':['hh-ah1 d-ah-l ih-ng','vbg'],
'hue':['hh-y-uw1','nn'],
'huffed':['hh-ah1-f-t','vbd'],
'hug':['hh-ah1-g','nn vb vbp'],
'huge':['hh-y-uw1-jh','jj'],
'hugely':['hh-y-uw1-jh l-iy','rb'],
'hugged':['hh-ah1-g-d','vbd'],
'hugging':['hh-ah1 g-ih-ng','vbg nn'],
'hugs':['hh-ah1-g-z','nns'],
'huh':['hh-ah1','uh'],
'hula':['hh-uw1 l-ah','nn'],
'hulk':['hh-ah1-l-k','nn'],
'hulking':['hh-ah1-l k-ih-ng','jj vbg'],
'hull':['hh-ah1-l','nn'],
'hullabaloo':['hh-ah l-ah b-ah l-uw1','nn'],
'hum':['hh-ah1-m','nn vb'],
'human':['hh-y-uw1 m-ah-n','jj nn'],
'humane':['hh-y-uw m-ey1-n','jj'],
'humanism':['hh-y-uw1 m-ah n-ih z-ah-m','nn'],
'humanist':['hh-y-uw1 m-ah n-ih-s-t','jj nn'],
'humanistic':['hh-y-uw m-ah n-ih1 s-t-ih-k','jj'],
'humanitarian':['hh-y-uw m-ae n-ah t-eh1 r-iy ah-n','jj nn'],
'humanity':['hh-y-uw m-ae1 n-ih t-iy','nn'],
'humanize':['hh-y-uw1 m-ah n-ay-z','vb'],
'humanizing':['hh-y-uw1 m-ah n-ay z-ih-ng','vbg'],
'humankind':['hh-y-uw1 m-ah-n k-ay-n-d','nn'],
'humanly':['hh-y-uw1 m-ah-n l-iy','rb'],
'humanness':['hh-y-uw1 m-ah-n n-ah-s','nn'],
'humble':['hh-ah1-m b-ah-l','jj vb'],
'humbled':['hh-ah1-m b-ah-l-d','vbn vbd'],
'humblest':['hh-ah1-m b-ah l-ah-s-t','jjs'],
'humbling':['hh-ah1-m b-ah-l ih-ng','jj'],
'humbly':['hh-ah1-m b-l-iy','rb'],
'humid':['hh-y-uw1 m-ah-d','jj'],
'humidity':['hh-y-uw m-ih1 d-ah t-iy','nn'],
'humiliated':['hh-y-uw m-ih1 l-iy ey t-ih-d','vbn jj'],
'humiliating':['hh-y-uw m-ih1 l-iy ey t-ih-ng','jj'],
'humiliation':['hh-y-uw m-ih l-iy ey1 sh-ah-n','nn'],
'humility':['hh-y-uw m-ih1 l-ih t-iy','nn'],
'humming':['hh-ah1 m-ih-ng','nn vbg'],
'humongous':['hh-y-uw m-ao1-ng g-ah-s','jj'],
'humor':['hh-y-uw1 m-er','nn vb'],
'humorist':['hh-y-uw1 m-er ah-s-t','nn'],
'humorless':['hh-y-uw1 m-er l-ah-s','jj'],
'humorous':['hh-y-uw1 m-er ah-s','jj'],
'hump':['hh-ah1-m-p','nn'],
'humped':['hh-ah1-m-p-t','nn'],
'hums':['hh-ah1-m-z','vbz'],
'hunch':['hh-ah1-n-ch','nn vb'],
'hunched':['hh-ah1-n-ch-t','vbn vbd jj'],
'hunches':['hh-ah1-n ch-ih-z','nns'],
'hundredth':['hh-ah1-n d-r-ah-d-th','jj'],
'hundredweight':['hh-ah1-n d-r-ah-d w-ey-t','nn'],
'hung':['hh-ah1-ng','vbd jj vbn'],
'hunger':['hh-ah1-ng g-er','nn'],
'hungrier':['hh-ah1-ng g-r-iy er','jjr'],
'hungrily':['hh-ah1-ng g-r-ah l-iy','rb'],
'hungry':['hh-ah1-ng g-r-iy','jj'],
'hunk':['hh-ah1-ng-k','nn'],
'hunker':['hh-ah1-ng k-er','vb'],
'hunkered':['hh-ah1-ng k-er-d','vbn vbd'],
'hunt':['hh-ah1-n-t','nn vb vbp'],
'hunted':['hh-ah1-n t-ah-d','vbn vbd jj'],
'hunter':['hh-ah1-n t-er','nn'],
'hunting':['hh-ah1-n t-ih-ng','nn vbg'],
'hunts':['hh-ah1-n-t-s','vbz nns'],
'hurdle':['hh-er1 d-ah-l','nn vb'],
'hurdles':['hh-er1 d-ah-l-z','nns'],
'hurl':['hh-er1-l','vb vbp'],
'hurled':['hh-er1-l-d','vbd vbn'],
'hurley':['hh-er1 l-iy','nn'],
'hurling':['hh-er1 l-ih-ng','vbg'],
'hurrah':['hh-uh r-aa1','nn'],
'hurricane':['hh-er1 ah k-ey-n','nn'],
'hurried':['hh-er1 iy-d','vbd vbn jj'],
'hurriedly':['hh-er1 iy-d l-iy','rb'],
'hurries':['hh-er1 iy-z','vbz'],
'hurry':['hh-er1 iy','nn vbp vb'],
'hurrying':['hh-er1 iy ih-ng','vbg jj'],
'hurt':['hh-er1-t','vbn jj nn vb vbd vbp'],
'hurting':['hh-er1 t-ih-ng','vbg'],
'hurtling':['hh-er1-t l-ih-ng','vbg'],
'hurts':['hh-er1-t-s','vbz'],
'husband':['hh-ah1-z b-ah-n-d','nn'],
'husbandry':['hh-ah1-z b-ah-n d-r-iy','nn'],
'hush':['hh-ah1-sh','jj nn uh'],
'hushed':['hh-ah1-sh-t','jj'],
'husk':['hh-ah1-s-k','nn'],
'husky':['hh-ah1-s k-iy','jj'],
'hustings':['hh-ah1 s-t-ih-ng-z','nn'],
'hustle':['hh-ah1 s-ah-l','vb nn'],
'hustled':['hh-ah1 s-ah-l-d','vbd vbn'],
'hustler':['hh-ah1 s-ah l-er','nn'],
'hustles':['hh-ah1 s-ah-l-z','vbz'],
'hustling':['hh-ah1 s-ah l-ih-ng','vbg'],
'hut':['hh-ah1-t','nn'],
'hutch':['hh-ah1-ch','nn'],
'hyacinths':['hh-ay1 ah s-ih-n-th-s','nns'],
'hyaluronic':['hh-ay ah l-er aa1 n-ih-k','jj'],
'hybrid':['hh-ay1 b-r-ah-d','jj nn'],
'hydrated':['hh-ay1 d-r-ey t-ah-d','jj'],
'hydration':['hh-ay d-r-ey1 sh-ah-n','nn'],
'hydraulic':['hh-ay d-r-ao1 l-ih-k','jj'],
'hydraulics':['hh-ay d-r-ao1 l-ih-k-s','nns nn'],
'hydride':['hh-ay1 d-r-ay-d','nn'],
'hydrocarbon':['hh-ay d-r-ow k-aa1-r b-ah-n','nn'],
'hydroelectric':['hh-ay d-r-ow ih l-eh1-k t-r-ih-k','jj'],
'hydrogen':['hh-ay1 d-r-ah jh-ah-n','nn'],
'hydrolysis':['hh-ay d-r-aa1 l-ah s-ah-s','nn'],
'hydrolyzed':['hh-ay1 d-r-ah l-ay-z-d','vbn'],
'hydrophilic':['hh-ay d-r-ah f-ih1 l-ih-k','jj'],
'hydrotherapy':['hh-ay d-r-ow th-eh1 r-ah p-iy','nn'],
'hydrous':['hh-ay1 d-r-ah-s','jj'],
'hyenas':['hh-ay iy1 n-ah-z','nns'],
'hygiene':['hh-ay1 jh-iy-n','nn'],
'hymn':['hh-ih1-m','nn'],
'hype':['hh-ay1-p','nn'],
'hyped':['hh-ay1-p-t','vbd nn'],
'hyper':['hh-ay1 p-er','jj'],
'hyperactive':['hh-ay p-er ae1-k t-ih-v','jj'],
'hyperbole':['hh-ay p-er1 b-ah l-iy','nn'],
'hyperbolic':['hh-ay p-er b-aa1 l-ih-k','jj'],
'hyperinflation':['hh-ay p-er ih-n f-l-ey1 sh-ah-n','nn'],
'hypermarket':['hh-ay1 p-er m-aa-r k-ih-t','nn'],
'hypersonic':['hh-ay p-er s-aa1 n-ih-k','jj'],
'hypertension':['hh-ay p-er t-eh1-n sh-ah-n','nn'],
'hyphenated':['hh-ay1 f-ah n-ey t-ih-d','jj vbn'],
'hyping':['hh-ay1 p-ih-ng','vbg nn'],
'hypnosis':['hh-ih-p n-ow1 s-ah-s','nn'],
'hypnotic':['hh-ih-p n-aa1 t-ih-k','jj'],
'hypnotized':['hh-ih1-p n-ah t-ay-z-d','vbn'],
'hypocrisy':['hh-ih p-aa1 k-r-ah s-iy','nn'],
'hypocrite':['hh-ih1 p-ah k-r-ih-t','nn'],
'hypocritical':['hh-ih p-ah k-r-ih1 t-ih k-ah-l','jj'],
'hypodermic':['hh-ay p-ah d-er1 m-ih-k','jj'],
'hypoglycemia':['hh-ay p-ow g-l-ay s-iy1 m-iy ah','nn'],
'hypoglycemic':['hh-ay p-ow g-l-ay s-iy1 m-ih-k','jj'],
'hypothalamic':['hh-ay p-ow th-ah l-ae1 m-ih-k','jj'],
'hypothesis':['hh-ay p-aa1 th-ah s-ah-s','nn'],
'hypothesize':['hh-ay p-aa1 th-ah s-ay-z','vb'],
'hypothesized':['hh-ay p-aa1 th-ah s-ay-z-d','vbn'],
'hypothetical':['hh-ay p-ah th-eh1 t-ah k-ah-l','jj'],
'hypothetically':['hh-ay p-ah th-eh1 t-ih-k l-iy','rb'],
'hysterectomy':['hh-ih s-t-er eh1-k t-ah m-iy','nn'],
'hysteria':['hh-ih s-t-eh1 r-iy ah','nn'],
'hysterical':['hh-ih s-t-eh1 r-ih k-ah-l','jj'],
'hysterically':['hh-ih s-t-eh1 r-ih-k l-iy','rb'],
'i':['ay1','nn nnp nns'],
'ice':['ay1-s','nn jj'],
'iceberg':['ay1-s b-er-g','nn'],
'icebox':['ay1-s b-aa-k-s','nn'],
'iced':['ay1-s-t','jj'],
'icing':['ay1 s-ih-ng','nn'],
'icon':['ay1 k-aa-n','nn'],
'iconoclasm':['ay k-aa1 n-ah k-l-ae z-ah-m','nn'],
'iconoclast':['ay k-aa1 n-ah k-l-ae-s-t','nn'],
'iconoclastic':['ay k-ah n-ah k-l-ae1 s-t-ih-k','jj'],
'icy':['ay1 s-iy','jj'],
'idea':['ay d-iy1 ah','nn'],
'ideal':['ay d-iy1-l','jj nn'],
'idealism':['ay d-iy1 l-ih z-ah-m','nn'],
'idealist':['ay d-iy1 l-ih-s-t','nn'],
'idealistic':['ay d-iy ah l-ih1 s-t-ih-k','jj'],
'idealized':['ay d-iy1 ah l-ay-z-d','vbn jj'],
'ideally':['ay d-iy1 l-iy','rb'],
'identical':['ay d-eh1-n t-ih k-ah-l','jj'],
'identifiable':['ay d-eh1-n t-ah f-ay ah b-ah-l','jj'],
'identification':['ay d-eh-n t-ah f-ah k-ey1 sh-ah-n','nn'],
'identified':['ay d-eh1-n t-ah f-ay-d','vbn jj vbd'],
'identifies':['ay d-eh1-n t-ah f-ay-z','vbz'],
'identify':['ay d-eh1-n t-ah f-ay','vb vbp'],
'identifying':['ay d-eh1-n t-ah f-ay ih-ng','vbg'],
'identity':['ay d-eh1-n t-ah t-iy','nn'],
'ideological':['ay d-iy ah l-aa1 jh-ih k-ah-l','jj'],
'ideologically':['ay d-iy ah l-aa1 jh-ih-k l-iy','rb'],
'ideologist':['ay d-iy aa1 l-ah jh-ah-s-t','nn'],
'ideologues':['ay1 d-iy ah l-ao-g-z','nns'],
'ideology':['ay d-iy aa1 l-ah jh-iy','nn'],
'idiocy':['ih1 d-iy ah s-iy','nn'],
'idiom':['ih1 d-iy ah-m','nn'],
'idiomatic':['ih d-iy ah m-ae1 t-ih-k','jj'],
'idiosyncratic':['ih d-iy ow s-ih-n k-r-ae1 t-ih-k','jj'],
'idiot':['ih1 d-iy ah-t','jj nn'],
'idiotic':['ih d-iy aa1 t-ih-k','jj'],
'idiotically':['ih d-iy aa1 t-ih-k l-iy','rb'],
'idle':['ay1 d-ah-l','jj vb'],
'idled':['ay1 d-ah-l-d','vbn vbd'],
'idleness':['ay1 d-ah-l n-ah-s','nn'],
'idler':['ay1 d-ah l-er','nn'],
'idling':['ay1 d-ah-l ih-ng','vbg nn'],
'idly':['ay1-d l-iy','rb'],
'idol':['ay1 d-ah-l','nn'],
'idolize':['ay1 d-ah l-ay-z','vbp'],
'idolized':['ay1 d-ah l-ay-z-d','jj'],
'idyll':['ay1 d-ah-l','nn'],
'idyllic':['ay d-ih1 l-ih-k','jj'],
'if':['ih1-f','in'],
'iffy':['ih1 f-iy','jj'],
'igloo':['ih1 g-l-uw','nn'],
'igneous':['ih1-g n-iy ah-s','jj'],
'ignite':['ih-g n-ay1-t','vb'],
'ignited':['ih-g n-ay1 t-ih-d','vbd vbn'],
'ignition':['ih-g n-ih1 sh-ah-n','nn'],
'ignoble':['ih-g n-ow1 b-ah-l','jj'],
'ignominious':['ih-g n-ah m-ih1 n-iy ah-s','jj'],
'ignoramus':['ih-g n-er ey1 m-ah-s','nn'],
'ignorance':['ih1-g n-er ah-n-s','nn'],
'ignorant':['ih1-g n-er ah-n-t','jj'],
'ignore':['ih-g n-ao1-r','vb vbp'],
'ignored':['ih-g n-ao1-r-d','vbn vbd'],
'ignores':['ih-g n-ao1-r-z','vbz'],
'ignoring':['ih-g n-ao1 r-ih-ng','vbg'],
'iguana':['ay g-w-aa1 n-ah','jj'],
'ilk':['ih1-l-k','nn'],
'ill':['ih1-l','jj nns nn rb'],
'illegal':['ih l-iy1 g-ah-l','jj'],
'illegality':['ih l-iy g-ae1 l-ih t-iy','nn'],
'illegally':['ih l-iy1 g-ah l-iy','rb'],
'illegitimacy':['ih l-ih jh-ih1 t-ah m-ah s-iy','nn'],
'illegitimate':['ih l-ih jh-ih1 t-ah m-ih-t','jj'],
'illicit':['ih l-ih1 s-ah-t','jj'],
'illicitly':['ih1 l-ih s-ih-t l-iy','rb'],
'illiquid':['ih l-ih1 k-w-ih-d','jj'],
'illiquidity':['ih l-ih k-w-ih1 d-ih t-iy','nn'],
'illiteracy':['ih l-ih1 t-er ah s-iy','nn'],
'illiterate':['ih l-ih1 t-er ah-t','jj'],
'illness':['ih1-l n-ah-s','nn'],
'illogic':['ih l-aa1 jh-ih-k','nn'],
'illogical':['ih l-aa1 jh-ih k-ah-l','jj'],
'illuminate':['ih l-uw1 m-ih n-ih-t','vb'],
'illuminated':['ih l-uw1 m-ah n-ey t-ah-d','vbn jj vbd'],
'illuminates':['ih l-uw1 m-ah n-ey-t-s','vbz'],
'illuminating':['ih l-uw1 m-ah n-ey t-ih-ng','jj vbg'],
'illumination':['ih l-uw m-ah n-ey1 sh-ah-n','nn'],
'illumined':['ih l-uw1 m-ah-n-d','vbd'],
'illusion':['ih l-uw1 zh-ah-n','nn'],
'illusionary':['ih l-uw1 zh-ah n-eh r-iy','jj'],
'illusionist':['ih l-uw1 zh-ah n-ah-s-t','nn'],
'illusory':['ih l-uw1 s-er iy','jj'],
'illustrate':['ih1 l-ah-s t-r-ey-t','vb vbp'],
'illustrated':['ih1 l-ah-s t-r-ey t-ah-d','vbn jj vbd'],
'illustrates':['ih1 l-ah-s t-r-ey-t-s','vbz'],
'illustrating':['ih1 l-ah-s t-r-ey t-ih-ng','vbg'],
'illustration':['ih l-ah-s t-r-ey1 sh-ah-n','nn'],
'illustrative':['ih l-ah1-s t-r-ah t-ih-v','jj'],
'illustrator':['ih1 l-ah-s t-r-ey t-er','nn'],
'illustrious':['ih l-ah1-s t-r-iy ah-s','jj'],
'image':['ih1 m-ah-jh','nn vb'],
'imagery':['ih1 m-ah jh-r-iy','nn'],
'images':['ih1 m-ah jh-ah-z','nns'],
'imaginable':['ih m-ae1 jh-ah n-ah b-ah-l','jj'],
'imaginary':['ih m-ae1 jh-ah n-eh r-iy','jj nn'],
'imagination':['ih m-ae jh-ah n-ey1 sh-ah-n','nn'],
'imaginative':['ih m-ae1 jh-ah n-ah t-ih-v','jj'],
'imaginatively':['ih m-ae1 jh-ah n-ah t-ih-v l-iy','rb'],
'imagine':['ih m-ae1 jh-ah-n','vb vbp'],
'imagined':['ih m-ae1 jh-ah-n-d','vbn vbd jj'],
'imagines':['ih m-ae1 jh-ah-n-z','vbz'],
'imaging':['ih1 m-ih jh-ih-ng','nn jj'],
'imagining':['ih m-ae1 jh-ah n-ih-ng','vbg'],
'imbalance':['ih-m b-ae1 l-ah-n-s','nn'],
'imbedded':['ih-m b-eh1 d-ih-d','vbn vbd'],
'imbibe':['ih-m b-ay1-b','vb'],
'imbibed':['ih-m b-ay1-b-d','vbn vbd'],
'imbroglio':['ih-m b-r-ow1 l-y-ow','nn'],
'imbued':['ih-m b-y-uw1-d','vbn'],
'imitate':['ih1 m-ah t-ey-t','vb vbp'],
'imitated':['ih1 m-ah t-ey t-ih-d','vbn vbd'],
'imitates':['ih1 m-ah t-ey-t-s','vbz'],
'imitating':['ih1 m-ah t-ey t-ih-ng','vbg'],
'imitation':['ih m-ah t-ey1 sh-ah-n','nn'],
'imitative':['ih1 m-ah t-ey t-ih-v','jj'],
'imitators':['ih1 m-ih t-ey t-er-z','nns'],
'immaculate':['ih m-ae1 k-y-uw l-ih-t-s','jj'],
'immanent':['ih1 m-ah n-ah-n-t','jj'],
'immaterial':['ih m-ah t-ih1 r-iy ah-l','jj'],
'immature':['ih m-ah t-y-uh1-r','jj'],
'immaturity':['ih m-ah ch-uh1 r-ih t-iy','nn'],
'immeasurably':['ih m-eh1 zh-er ae b-l-iy','rb'],
'immediacy':['ih m-iy1 d-iy ah s-iy','nn'],
'immediate':['ih m-iy1 d-iy ah-t','jj'],
'immediately':['ih m-iy1 d-iy ah-t l-iy','rb'],
'immemorial':['ih m-ah m-ao1 r-iy ah-l','jj'],
'immense':['ih m-eh1-n-s','jj nn'],
'immensely':['ih m-eh1-n-s l-iy','rb'],
'immersed':['ih m-er1-s-t','vbn vbd'],
'immersion':['ih m-er1 zh-ah-n','nn'],
'immigrant':['ih1 m-ah g-r-ah-n-t','nn jj'],
'immigrated':['ih1 m-ah g-r-ey t-ih-d','vbd vbn'],
'immigration':['ih m-ah g-r-ey1 sh-ah-n','nn'],
'imminent':['ih1 m-ah n-ah-n-t','jj'],
'imminently':['ih1 m-ah n-ah-n-t l-iy','rb'],
'immobility':['ih m-ow b-ih1 l-ih t-iy','nn'],
'immobilized':['ih m-ow1 b-ah l-ay-z-d','vbn'],
'immoral':['ih m-ao1 r-ah-l','jj'],
'immorality':['ih m-er ae1 l-ih t-iy','nn'],
'immortal':['ih m-ao1-r t-ah-l','jj'],
'immortality':['ih m-ao-r t-ae1 l-ih t-iy','nn'],
'immortalized':['ih m-ao1-r t-ah l-ay-z-d','vbn'],
'immovable':['ih m-uw1 v-ah b-ah-l','jj'],
'immune':['ih m-y-uw1-n','jj nn'],
'immunity':['ih m-y-uw1 n-ah t-iy','nn'],
'immunization':['ih m-y-uw n-ah z-ey1 sh-ah-n','nn'],
'immunized':['ih1 m-y-ah n-ay-z-d','vbn'],
'immunodeficiency':['ih m-y-uw n-ow d-ih f-ih1 sh-ah-n s-iy','nn'],
'immunological':['ih m-y-uw n-ah l-aa1 jh-ih k-ah-l','jj'],
'immunologist':['ih m-y-uw n-aa1 l-ah jh-ih-s-t','nn'],
'immunology':['ih m-y-uw n-aa1 l-ah jh-iy','nn'],
'immutable':['ih m-y-uw1 t-ah b-ah-l','jj'],
'impact':['ih-m p-ae1-k-t','nn vb'],
'impacted':['ih1-m p-ae-k t-ih-d','vbn jj'],
'impacts':['ih-m p-ae1-k-t-s','nns'],
'impair':['ih-m p-eh1-r','vb nn'],
'impaired':['ih-m p-eh1-r-d','vbn jj'],
'impairment':['ih-m p-eh1-r m-ah-n-t','nn'],
'impaled':['ih-m p-ey1-l-d','vbn'],
'impart':['ih-m p-aa1-r-t','vb'],
'imparted':['ih-m p-aa1-r t-ih-d','vbn vbd'],
'impartial':['ih-m p-aa1-r sh-ah-l','jj'],
'impartiality':['ih-m p-aa-r sh-iy ae1 l-ih t-iy','nn'],
'imparting':['ih-m p-aa1-r t-ih-ng','vbg'],
'impassable':['ih-m p-ae1 s-ah b-ah-l','jj'],
'impasse':['ih-m p-ae1-s','nn'],
'impassioned':['ih-m p-ae1 sh-ah-n-d','jj'],
'impassive':['ih-m p-ae1 s-ih-v','jj'],
'impassively':['ih-m p-ae1 s-ih-v l-iy','rb'],
'impatience':['ih-m p-ey1 sh-ah-n-s','nn'],
'impatiens':['ih-m p-ey1 sh-ah-n-z','nns'],
'impatient':['ih-m p-ey1 sh-ah-n-t','jj nn'],
'impatiently':['ih-m p-ey1 sh-ah-n-t l-iy','rb'],
'impeached':['ih-m p-iy1-ch-t','vbn'],
'impeachment':['ih-m p-iy1-ch m-ah-n-t','nn'],
'impeccable':['ih-m p-eh1 k-ah b-ah-l','jj'],
'impeccably':['ih-m p-eh1 k-ah b-l-iy','rb'],
'impede':['ih-m p-iy1-d','vb vbp'],
'impeded':['ih-m p-iy1 d-ih-d','vbn vbd'],
'impedes':['ih-m p-iy1-d-z','vbz'],
'impediment':['ih-m p-eh1 d-ah m-ah-n-t','nn'],
'impeding':['ih-m p-iy1 d-ih-ng','vbg'],
'impelled':['ih-m p-eh1-l-d','vbn vbd'],
'impending':['ih-m p-eh1-n d-ih-ng','jj vbg'],
'impenetrable':['ih-m p-eh1 n-ah t-r-ah b-ah-l','jj'],
'imperative':['ih-m p-eh1 r-ah t-ih-v','jj nn'],
'imperceptible':['ih-m p-er s-eh1-p t-ih b-ah-l','jj'],
'imperceptibly':['ih-m p-er s-eh1-p t-ih b-l-iy','rb'],
'imperfect':['ih-m p-er1 f-ih-k-t','jj'],
'imperfection':['ih-m p-er f-eh1-k sh-ah-n','nn'],
'imperfectly':['ih-m p-er1 f-ih-k-t l-iy','rb'],
'imperial':['ih-m p-ih1 r-iy ah-l','jj'],
'imperialism':['ih-m p-ih1 r-iy ah l-ih z-ah-m','nn'],
'imperialist':['ih-m p-ih1 r-iy ah l-ih-s-t','nn'],
'imperil':['ih-m p-eh1 r-ah-l','vb'],
'imperiled':['ih-m p-eh1 r-ah-l-d','vbn jj'],
'imperiling':['ih-m p-eh1 r-ah l-ih-ng','vbg'],
'imperious':['ih-m p-ih1 r-iy ah-s','jj'],
'impermissible':['ih-m p-er m-ih1 s-ih b-ah-l','jj'],
'impersonal':['ih-m p-er1 s-ah n-ah-l','jj'],
'impersonation':['ih-m p-er s-ah n-ey1 sh-ah-n','nn'],
'impersonator':['ih-m p-er1 s-ah n-ey t-er','nn'],
'impertinent':['ih-m p-er1 t-ah n-ah-n-t','jj'],
'impervious':['ih-m p-er1 v-iy ah-s','jj'],
'impetuous':['ih-m p-eh1 ch-w-ah-s','jj'],
'impetus':['ih1-m p-ah t-ah-s','nn'],
'impinge':['ih-m p-ih1-n-jh','vb vbp'],
'implacable':['ih-m p-l-ae1 k-ah b-ah-l','jj'],
'implant':['ih-m p-l-ae1-n-t','nn vb'],
'implantable':['ih1-m p-l-ae-n t-ah b-ah-l','jj'],
'implantation':['ih-m p-l-ae-n t-ey1 sh-ah-n','nn'],
'implanted':['ih-m p-l-ae1-n t-ih-d','vbn vbd'],
'implanting':['ih-m p-l-ae1-n t-ih-ng','vbg'],
'implausible':['ih-m p-l-ao1 z-ah b-ah-l','jj'],
'implausibly':['ih-m p-l-ao1 z-ah b-l-iy','rb'],
'implement':['ih1-m p-l-ah m-ah-n-t','vb vbp'],
'implementation':['ih-m p-l-ah m-eh-n t-ey1 sh-ah-n','nn'],
'implemented':['ih1-m p-l-ah m-eh-n t-ah-d','vbn vbd'],
'implementing':['ih1-m p-l-ah m-eh-n t-ih-ng','vbg'],
'implements':['ih1-m p-l-ah m-ah-n-t-s','nns vbz'],
'implicate':['ih1-m p-l-ih k-ey-t','vb'],
'implicated':['ih1-m p-l-ih k-ey t-ih-d','vbn'],
'implication':['ih-m p-l-ah k-ey1 sh-ah-n','nn'],
'implicit':['ih-m p-l-ih1 s-ah-t','jj'],
'implicitly':['ih-m p-l-ih1 s-ah-t l-iy','rb'],
'implied':['ih-m p-l-ay1-d','vbn jj vbd'],
'implies':['ih-m p-l-ay1-z','vbz'],
'implore':['ih-m p-l-ao1-r','vb'],
'implored':['ih-m p-l-ao1-r-d','vbn vbd'],
'implores':['ih-m p-l-ao1-r-z','vbz'],
'imploring':['ih-m p-l-ao1 r-ih-ng','vbg'],
'imply':['ih-m p-l-ay1','vb vbp'],
'implying':['ih-m p-l-ay1 ih-ng','vbg'],
'impolite':['ih-m p-ah l-ay-t','jj'],
'impolitic':['ih-m p-ao1 l-ih1 t-ih-k','jj'],
'imponderable':['ih-m p-aa1-n d-er ah b-ah-l','jj nn'],
'import':['ih-m p-ao1-r-t','nn vb vbp'],
'importance':['ih-m p-ao1-r t-ah-n-s','nn'],
'important':['ih-m p-ao1-r t-ah-n-t','jj'],
'importantly':['ih-m p-ao1-r t-ah-n-t l-iy','rb'],
'importation':['ih-m p-ao-r t-ey1 sh-ah-n','nn'],
'imported':['ih-m p-ao1-r t-ih-d','vbn vbd jj'],
'importer':['ih-m p-ao1-r t-er','nn'],
'importing':['ih-m p-ao1-r t-ih-ng','vbg nn'],
'imports':['ih-m p-ao1-r-t-s','nns vbz'],
'impose':['ih-m p-ow1-z','vb vbp'],
'imposed':['ih-m p-ow1-z-d','vbn vbd jj'],
'imposes':['ih-m p-ow1 z-ah-z','vbz'],
'imposing':['ih-m p-ow1 z-ih-ng','vbg jj'],
'imposition':['ih-m p-ah z-ih1 sh-ah-n','nn'],
'impossibility':['ih-m p-ao s-ih b-ih1 l-ih t-iy','nn'],
'impossible':['ih-m p-aa1 s-ah b-ah-l','jj'],
'impossibly':['ih-m p-aa1 s-ah b-l-iy','rb'],
'impotence':['ih1-m p-ah t-ah-n-s','nn'],
'impotent':['ih1-m p-ah t-ah-n-t','jj'],
'impound':['ih-m p-aw1-n-d','vb'],
'impounded':['ih-m p-aw1-n d-ih-d','vbn'],
'impoundment':['ih-m p-aw1-n-d m-ah-n-t','nn'],
'impoverished':['ih-m p-aa1 v-r-ih-sh-t','jj vbn'],
'impoverishment':['ih-m p-aa1 v-r-ih-sh m-ah-n-t','nn'],
'impracticable':['ih-m p-r-ae1-k t-ih k-ah b-ah-l','jj'],
'impractical':['ih-m p-r-ae1-k t-ah k-ah-l','jj'],
'imprecise':['ih1-m p-r-ah s-ay-s','jj'],
'impregnable':['ih-m p-r-eh1-g n-ah b-ah-l','jj'],
'impresario':['ih-m p-r-ih s-aa1 r-iy ow','nn'],
'impress':['ih-m p-r-eh1-s','vb nn vbp'],
'impressed':['ih-m p-r-eh1-s-t','vbn jj vbd'],
'impresses':['ih-m p-r-eh1 s-ih-z','vbz'],
'impressing':['ih-m p-r-eh1 s-ih-ng','vbg'],
'impression':['ih-m p-r-eh1 sh-ah-n','nn'],
'impressionism':['ih-m p-r-eh1 sh-ah-n ih z-ah-m','nn'],
'impressionist':['ih-m p-r-eh1 sh-ah-n ah-s-t','nn jj'],
'impressionistic':['ih-m p-r-eh sh-ah n-ih1 s-t-ih-k','jj'],
'impressive':['ih-m p-r-eh1 s-ih-v','jj'],
'impressively':['ih-m p-r-eh1 s-ih-v l-iy','rb'],
'imprimatur':['ih-m p-r-ih m-aa1 t-er','nn'],
'imprint':['ih-m p-r-ih1-n-t','vb nn'],
'imprinted':['ih-m p-r-ih1-n t-ih-d','vbn'],
'imprints':['ih-m p-r-ih1-n-t-s','nns'],
'imprison':['ih-m p-r-ih1 z-ah-n','vb'],
'imprisoned':['ih-m p-r-ih1 z-ah-n-d','vbn vbd jj'],
'imprisoning':['ih-m p-r-ih1 z-ah-n ih-ng','vbg'],
'imprisonment':['ih-m p-r-ih1 z-ah-n m-ah-n-t','nn'],
'improbable':['ih-m p-r-aa1 b-ah b-ah-l','jj'],
'improbably':['ih-m p-r-aa1 b-ah b-l-iy','rb'],
'impromptu':['ih-m p-r-aa1-m-p t-uw','jj nn'],
'improper':['ih-m p-r-aa1 p-er','jj'],
'improperly':['ih-m p-r-aa1 p-er l-iy','rb'],
'impropriety':['ih-m p-r-ah p-r-ay1 ah t-iy','nn'],
'improve':['ih-m p-r-uw1-v','vb vbp'],
'improved':['ih-m p-r-uw1-v-d','vbn jj vbd'],
'improvement':['ih-m p-r-uw1-v m-ah-n-t','nn'],
'improves':['ih-m p-r-uw1-v-z','vbz'],
'improving':['ih-m p-r-uw1 v-ih-ng','vbg nn'],
'improvisation':['ih-m p-r-aa v-ih z-ey1 sh-ah-n','nn'],
'improvisational':['ih-m p-r-aa v-ih z-ey1 sh-ah n-ah-l','jj'],
'improvise':['ih1-m p-r-ah v-ay-z','vb'],
'improvised':['ih1-m p-r-ah v-ay-z-d','vbd vbn'],
'improvising':['ih1-m p-r-ah v-ay z-ih-ng','nn'],
'imprudence':['ih-m p-r-uw1 d-ah-n-s','nn'],
'imprudent':['ih-m p-r-uw1 d-ah-n-t','jj'],
'imprudently':['ih-m p-r-uw1 d-ah-n-t l-iy','rb'],
'impugn':['ih-m p-y-uw1-n','vb'],
'impugned':['ih-m p-y-uw1-n-d','vbn'],
'impugning':['ih-m p-y-uw1 n-ih-ng','vbg'],
'impulse':['ih1-m p-ah-l-s','nn jj'],
'impulses':['ih1-m p-ah-l s-ih-z','nns'],
'impulsive':['ih-m p-ah1-l s-ih-v','jj'],
'impulsively':['ih-m p-ah1-l s-ih-v l-iy','rb'],
'impunity':['ih-m p-y-uw1 n-ih t-iy','nn'],
'impure':['ih-m p-y-uh1-r','jj'],
'impurity':['ih-m p-y-uh1 r-ah t-iy','nn'],
'imputation':['ih-m p-y-ah t-ey1 sh-ah-n','nn'],
'imputed':['ih-m p-y-uw1 t-ih-d','vbn jj'],
'in':['ih-n','in nn rb rp nnp rbr vbd'],
'inability':['ih-n ah b-ih1 l-ih t-iy','nn'],
'inaccessible':['ih-n ah-k s-eh1 s-ah b-ah-l','jj'],
'inaccuracy':['ih-n ae1 k-y-er ah s-iy','nn'],
'inaccurate':['ih-n ae1 k-y-er ah-t','jj'],
'inaccurately':['ih-n ae1 k-y-er ah-t l-iy','rb'],
'inaction':['ih-n ae1-k sh-ah-n','nn'],
'inactivated':['ih-n ae1-k t-ih v-ey t-ih-d','vbd vbn'],
'inactivation':['ih-n ae-k t-ih v-ey1 sh-ah-n','nn'],
'inactive':['ih-n ae1-k t-ih-v','jj'],
'inactivity':['ih-n ae-k t-ih1 v-ih t-iy','nn'],
'inadequacy':['ih-n ae1 d-ih k-w-ah s-iy','nn'],
'inadequate':['ih-n ae1 d-ah k-w-ah-t','jj'],
'inadequately':['ih-n ae1 d-ah k-w-ah-t l-iy','rb'],
'inadvertent':['ih-n ah-d v-er1 t-ah-n-t','jj'],
'inadvertently':['ih-n ah-d v-er1 t-ah-n-t l-iy','rb'],
'inadvisable':['ih-n ah-d v-ay1 z-ah b-ah-l','jj'],
'inalienable':['ih-n ey1 l-y-ah n-ah b-ah-l','jj'],
'inane':['ih-n ey1-n','jj'],
'inanimate':['ih-n ae1 n-ah m-ah-t','jj'],
'inapplicable':['ih-n ae1 p-l-ih k-ah b-ah-l','jj'],
'inappropriate':['ih-n ah p-r-ow1 p-r-iy ih-t','jj'],
'inappropriately':['ih-n ah p-r-aa1 p-r-iy ah-t l-iy','rb'],
'inarticulate':['ih-n aa-r t-ih1 k-y-ah l-ah-t','jj'],
'inasmuch':['ih-n ae1-s m-ah-k','rb'],
'inattention':['ih-n ah t-eh1-n ch-ah-n','nn'],
'inattentive':['ih-n ah t-eh1-n t-ih-v','jj'],
'inaudible':['ih-n ao1 d-ah b-ah-l','jj'],
'inaugural':['ih n-ao1 g-er ah-l','jj nn'],
'inaugurated':['ih n-ao1 g-er ey t-ih-d','vbn vbd'],
'inaugurating':['ih n-ao1 g-y-er ey t-ih-ng','vbg'],
'inauguration':['ih n-ao g-y-er ey1 sh-ah-n','nn'],
'inauspicious':['ih-n aw s-p-ih1 sh-ih-s','jj'],
'inboard':['ih1-n b-ao-r-d','jj rb'],
'inborn':['ih1-n b-ao-r-n','jj'],
'inbound':['ih-n b-aw1-n-d','jj'],
'incalculable':['ih-n k-ae1-l k-y-ah l-ah b-ah-l','jj'],
'incandescent':['ih-n k-ah-n d-eh1 s-ah-n-t','jj'],
'incantation':['ih-n k-ae1-n t-ey1 sh-ah-n','nn'],
'incapable':['ih-n k-ey1 p-ah b-ah-l','jj nn'],
'incapacitated':['ih-n k-ah p-ae1 s-ih t-ey t-ih-d','vbn'],
'incapacity':['ih-n k-ah p-ae1 s-ah t-iy','nn'],
'incarcerate':['ih-n k-aa1-r s-er ey-t','vb'],
'incarcerated':['ih-n k-aa1-r s-er ey t-ih-d','vbn'],
'incarceration':['ih-n k-aa-r s-er ey1 sh-ah-n','nn'],
'incarnate':['ih-n k-aa1-r n-ah-t','jj vb'],
'incarnation':['ih-n k-aa1-r n-ey1 sh-ah-n','nn'],
'incendiary':['ih-n s-eh1-n d-iy eh r-iy','jj'],
'incense':['ih-n s-eh1-n-s','nn'],
'incensed':['ih1-n s-eh-n-s-t','vbn jj'],
'incentive':['ih-n s-eh1-n t-ih-v','nn jj'],
'inception':['ih-n s-eh1-p sh-ah-n','nn'],
'incessant':['ih-n s-eh1 s-ah-n-t','jj'],
'incessantly':['ih-n s-eh1 s-ah-n-t l-iy','rb'],
'incest':['ih1-n s-eh-s-t','nn'],
'incestuous':['ih-n s-eh1 s-ch-w-ah-s','jj'],
'inch':['ih1-n-ch','nn rb vb'],
'inched':['ih1-n-ch-t','vbd vbn'],
'inches':['ih1-n ch-ah-z','nns nn vbz'],
'inching':['ih1-n ch-ih-ng','vbg'],
'incidence':['ih1-n s-ah d-ah-n-s','nn'],
'incident':['ih1-n s-ah d-ah-n-t','nn jj'],
'incidental':['ih-n s-ih d-eh1-n t-ah-l','jj'],
'incidentally':['ih-n s-ih d-eh1-n t-ah l-iy','rb'],
'incineration':['ih-n s-ih1 n-er ey sh-ah-n','nn'],
'incinerator':['ih-n s-ih1 n-er ey t-er','nn'],
'incipient':['ih-n s-ih1 p-iy ah-n-t','jj'],
'incisive':['ih-n s-ay1 s-ih-v','jj'],
'incite':['ih-n s-ay1-t','vb vbp'],
'incited':['ih-n s-ay1 t-ih-d','vbn vbd'],
'inciting':['ih-n s-ay1 t-ih-ng','vbg'],
'inclement':['ih-n k-l-eh1 m-ah-n-t','jj'],
'inclination':['ih-n k-l-ah n-ey1 sh-ah-n','nn'],
'incline':['ih-n k-l-ay1-n','nn'],
'inclined':['ih-n k-l-ay1-n-d','vbn jj'],
'include':['ih-n k-l-uw1-d','vbp vbn vb'],
'included':['ih-n k-l-uw1 d-ah-d','vbd vbn jj'],
'includes':['ih-n k-l-uw1-d-z','vbz'],
'including':['ih-n k-l-uw1 d-ih-ng','vbg'],
'inclusion':['ih-n k-l-uw1 zh-ah-n','nn'],
'inclusive':['ih-n k-l-uw1 s-ih-v','jj'],
'incoherence':['ih-n k-ow hh-ih1 r-ah-n-s','nn'],
'incoherent':['ih-n k-ow hh-ih1 r-ah-n-t','jj'],
'incoherently':['ih-n k-ow hh-ih1 r-ah-n-t l-iy','rb'],
'income':['ih1-n k-ah-m','nn'],
'incoming':['ih1-n k-ah m-ih-ng','jj vbg'],
'incomparable':['ih-n k-aa1-m p-er ah b-ah-l','jj'],
'incomparably':['ih-n k-aa1-m p-er ah b-l-iy','rb'],
'incompatibility':['ih-n k-aa-m p-ah t-ih b-ih1 l-ih t-iy','nn'],
'incompatible':['ih-n k-ah-m p-ae1 t-ah b-ah-l','jj'],
'incompetence':['ih-n k-aa1-m p-ah t-ah-n-s','nn'],
'incompetent':['ih-n k-aa1-m p-ah t-ah-n-t','jj'],
'incomplete':['ih-n k-ah-m p-l-iy1-t','jj'],
'incomprehensible':['ih-ng k-aa-m p-r-ah hh-eh1-n s-ih b-ah-l','jj nn'],
'inconceivable':['ih-n k-ah-n s-iy1 v-ah b-ah-l','jj'],
'inconclusive':['ih-n k-ah-n k-l-uw1 s-ih-v','jj'],
'inconclusively':['ih-ng k-aa1-n k-l-uw s-ih-v l-iy','rb'],
'incongruity':['ih-ng k-ao-ng r-uw1 ih t-iy','nn'],
'incongruous':['ih-ng k-ao1-ng r-uw ah-s','jj'],
'inconsequential':['ih-ng k-aa-n s-ah k-w-eh1-n ch-ah-l','jj'],
'inconsistency':['ih-n k-ah-n s-ih1 s-t-ah-n s-iy','nn'],
'inconsistent':['ih-n k-ah-n s-ih1 s-t-ah-n-t','jj'],
'inconspicuous':['ih-ng k-aa1-n s-p-ih k-w-ah-s','jj'],
'incontrovertible':['ih-ng k-aa-n t-r-ow v-er1 t-ih b-ah-l','jj'],
'inconvenience':['ih-n k-ah-n v-iy1 n-y-ah-n-s','nn'],
'inconvenient':['ih-n k-ah-n v-iy1 n-y-ah-n-t','jj'],
'incorporate':['ih-n k-ao1-r p-er ey-t','vb vbp jj'],
'incorporated':['ih-n k-ao1-r p-er ey t-ah-d','vbn jj vbd'],
'incorporates':['ih-n k-ao1-r p-er ey-t-s','vbz'],
'incorporating':['ih-n k-ao1-r p-er ey t-ih-ng','jj vbg'],
'incorporation':['ih-n k-ao-r p-er ey1 sh-ah-n','nn'],
'incorrect':['ih-n k-er eh1-k-t','jj'],
'incorrectly':['ih-n k-er eh1-k-t l-iy','rb'],
'incorrigible':['ih-n k-aa1 r-ah jh-ah b-ah-l','jj'],
'increase':['ih-n k-r-iy1-s','nn vb vbp'],
'increased':['ih-n k-r-iy1-s-t','vbn jj vb vbd'],
'increases':['ih-n k-r-iy1 s-ah-z','nns vbz'],
'increasing':['ih-n k-r-iy1 s-ih-ng','vbg jj nn'],
'increasingly':['ih-n k-r-iy1 s-ih-ng l-iy','rb'],
'incredible':['ih-n k-r-eh1 d-ah b-ah-l','jj'],
'incredibly':['ih-n k-r-eh1 d-ah b-l-iy','rb'],
'incredulity':['ih-n k-r-ah d-uw1 l-ih t-iy','nn'],
'increment':['ih1-n k-r-ah m-ah-n-t','nn'],
'incremental':['ih-n k-r-ah m-eh1-n t-ah-l','jj'],
'incriminating':['ih-n k-r-ih1 m-ah n-ey t-ih-ng','vbg jj'],
'incubating':['ih1-n k-y-ah b-ey t-ih-ng','vbg'],
'incubation':['ih-ng k-y-uw b-ey1 sh-ah-n','nn'],
'incumbency':['ih-n k-ah1-m b-ah-n s-iy','nn'],
'incumbent':['ih-n k-ah1-m b-ah-n-t','jj nn'],
'incur':['ih-n k-er1','vb vbp'],
'incurable':['ih-n k-y-uh1 r-ah b-ah-l','jj'],
'incurred':['ih-n k-er1-d','vbn vbd'],
'incurring':['ih-n k-er1 ih-ng','vbg'],
'incurs':['ih-n k-er1-z','vbz'],
'incursion':['ih-n k-er1 zh-ah-n','nn'],
'indebted':['ih-n d-eh1 t-ah-d','jj'],
'indebtedness':['ih-n d-eh1 t-ih-d n-ih-s','nn'],
'indecency':['ih-n d-iy1 s-ah-n s-iy','nn'],
'indecent':['ih-n d-iy1 s-ah-n-t','jj'],
'indecision':['ih-n d-ih s-ih1 zh-ah-n','nn'],
'indecisive':['ih-n d-ih s-ay1 s-ih-v','jj'],
'indecisiveness':['ih-n d-eh1 s-ih s-ih-v n-ah-s','nn'],
'indeed':['ih-n d-iy1-d','rb uh'],
'indefatigable':['ih-n d-ih f-ae1 t-ih g-ah b-ah-l','jj'],
'indefensible':['ih-n d-ih f-eh1-n s-ah b-ah-l','jj'],
'indefinable':['ih-n d-ih f-ay1 n-ah b-ah-l','jj'],
'indefinite':['ih-n d-eh1 f-ah n-ah-t','jj nn'],
'indefinitely':['ih-n d-eh1 f-ah n-ah-t l-iy','rb'],
'indelible':['ih-n d-eh1 l-ih b-ah-l','jj'],
'indelibly':['ih-n d-eh1 l-ah b-l-iy','rb'],
'indelicate':['ih-n d-eh1 l-ih k-ah-t','jj'],
'indemnification':['ih-n d-eh-m n-ah f-ih k-ey1 sh-ah-n','nn'],
'indemnify':['ih-n d-eh1-m n-ah f-ay','vb'],
'indemnifying':['ih-n d-eh-m n-ih f-ay1 ih-ng','vbg'],
'indemnity':['ih-n d-eh1-m n-ah t-iy','nn'],
'indenture':['ih-n d-eh1-n ch-er','nn'],
'indentured':['ih-n d-eh1-n ch-er-d','vbn'],
'independence':['ih-n d-ih p-eh1-n d-ah-n-s','nn'],
'independent':['ih-n d-ih p-eh1-n d-ah-n-t','jj nn'],
'independently':['ih-n d-ih p-eh1-n d-ah-n-t l-iy','rb'],
'indestructibility':['ih-n d-ah-s-t r-ah-k t-ih b-ih1 l-ih t-iy','nn'],
'indestructible':['ih-n d-ah-s-t r-ah1-k t-ih b-ah-l','jj'],
'indeterminate':['ih-n d-ih t-er1 m-ih n-ih-t','jj'],
'index':['ih1-n d-eh-k-s','nn vb'],
'indexation':['ih1-n d-eh-k s-ey1 sh-ah-n','nn'],
'indexed':['ih1-n d-eh-k-s-t','vbn jj'],
'indexers':['ih1-n d-eh-k s-er-z','nns'],
'indexes':['ih1-n d-eh-k s-ih-z','nns'],
'indexing':['ih1-n d-eh-k s-ih-ng','nn vbg'],
'indicate':['ih1-n d-ah k-ey-t','vb vbp'],
'indicated':['ih1-n d-ah k-ey t-ah-d','vbd jj vbn'],
'indicates':['ih1-n d-ih k-ey-t-s','vbz'],
'indicating':['ih1-n d-ah k-ey t-ih-ng','vbg'],
'indication':['ih-n d-ah k-ey1 sh-ah-n','nn'],
'indicative':['ih-n d-ih1 k-ah t-ih-v','jj'],
'indicator':['ih1-n d-ah k-ey t-er','nn'],
'indices':['ih1-n d-ih s-iy-z','nns'],
'indict':['ih-n d-ay1-t','vb'],
'indicted':['ih-n d-ay1 t-ah-d','vbn vbd'],
'indictment':['ih-n d-ay1-t m-ah-n-t','nn'],
'indifference':['ih-n d-ih1 f-er ah-n-s','nn'],
'indifferent':['ih-n d-ih1 f-r-ah-n-t','jj'],
'indigenous':['ih-n d-ih1 jh-ah n-ah-s','jj'],
'indigent':['ih1-n d-ih jh-ah-n-t','jj'],
'indigestion':['ih-n d-ay jh-eh1-s ch-ah-n','nn'],
'indignant':['ih-n d-ih1-g n-ah-n-t','jj'],
'indignantly':['ih-n d-ih1-g n-ah-n-t l-iy','rb'],
'indignation':['ih-n d-ih-g n-ey1 sh-ah-n','nn'],
'indignity':['ih-n d-ih1-g n-ah t-iy','nn'],
'indirect':['ih-n d-er eh1-k-t','jj'],
'indirectly':['ih-n d-er eh1-k-t l-iy','rb'],
'indiscreet':['ih-n d-ih s-k-r-iy1-t','jj'],
'indiscretions':['ih-n d-ih s-k-r-eh1 sh-ah-n-z','nns'],
'indiscriminate':['ih-n d-ih s-k-r-ih1 m-ah n-ah-t','jj'],
'indispensable':['ih-n d-ih s-p-eh1-n s-ah b-ah-l','jj nn'],
'indispensible':['ih-n d-ih s-p-eh1-n s-ih b-ah-l','jj'],
'indisputable':['ih-n d-ih-s p-y-uw1 t-ah b-ah-l','jj'],
'indisputably':['ih-n d-ih1 s-p-y-uw t-ae b-l-iy','rb'],
'indistinct':['ih-n d-ih s-t-ih1-ng-k-t','jj'],
'indistinguishable':['ih-n d-ih s-t-ih1-ng g-w-ih sh-ah b-ah-l','jj'],
'indium':['ih1-n d-iy ah-m','nn'],
'individual':['ih-n d-ah v-ih1 jh-ah w-ah-l','jj nn'],
'individualism':['ih-n d-ih v-ih d-uw1 ah l-ih z-ah-m','nn'],
'individualist':['ih-n d-ih v-ih d-uw1 ah l-ih-s-t','nn'],
'individualistic':['ih-n d-ih v-ih jh-uw ah l-ih1 s-t-ih-k','jj'],
'individuality':['ih-n d-ih v-ih jh-uw ae1 l-ih t-iy','nn'],
'individualized':['ih-n d-ih v-ih1 jh-uw ah l-ay-z-d','jj vbn'],
'individually':['ih-n d-ih v-ih1 jh-uw ah l-iy','rb'],
'indivisible':['ih-n d-ih v-ih1 s-ih b-ah-l','jj'],
'indoctrinated':['ih-n d-aa1-k t-r-ah n-ey t-ih-d','vbn'],
'indoctrination':['ih-n d-aa-k t-r-ah n-ey1 sh-ah-n','nn'],
'indolent':['ih1-n d-ah l-ah-n-t','jj'],
'indomitable':['ih-n d-aa1 m-ah t-ah b-ah-l','jj'],
'indoor':['ih1-n d-ao-r','jj'],
'indoors':['ih1-n d-ao-r-z','nn rb'],
'induce':['ih-n d-uw1-s','vb'],
'induced':['ih-n d-uw1-s-t','vbn vbd'],
'inducement':['ih-n d-uw1-s m-ah-n-t','nn'],
'induces':['ih-n d-uw1 s-ih-z','vbz'],
'inducing':['ih-n d-uw1 s-ih-ng','vbg'],
'inducted':['ih-n d-ah1-k t-ah-d','vbn'],
'inductees':['ih-n d-ah1-k t-iy1-z','nns'],
'induction':['ih-n d-ah1-k sh-ah-n','nn'],
'indulge':['ih-n d-ah1-l-jh','vb vbp'],
'indulged':['ih-n d-ah1-l-jh-d','vbd'],
'indulgence':['ih-n d-ah1-l jh-ah-n-s','nn'],
'indulgent':['ih-n d-ah1-l jh-ah-n-t','jj'],
'indulges':['ih-n d-ah1-l jh-ih-z','vbz'],
'indulging':['ih-n d-ah1-l jh-ih-ng','vbg'],
'industrial':['ih-n d-ah1-s t-r-iy ah-l','jj'],
'industrialist':['ih-n d-ah1-s t-r-iy ah l-ih-s-t','nn'],
'industrialization':['ih-n d-ah-s t-r-iy ah l-ih z-ey1 sh-ah-n','nn'],
'industrialize':['ih-n d-ah1-s t-r-iy ah l-ay-z','vb'],
'industrialized':['ih-n d-ah1-s t-r-iy ah l-ay-z-d','vbn jj'],
'industrially':['ih-n d-ah1-s t-r-iy ah l-iy','rb'],
'industrials':['ih-n d-ah1-s t-r-iy ah-l-z','nns vbz'],
'industrious':['ih-n d-ah1-s t-r-iy ah-s','jj'],
'industry':['ih1-n d-ah-s t-r-iy','nn'],
'industrywide':['ih1-n d-ah-s t-r-iy w-ay-d','jj rb'],
'inedible':['ih-n eh1 d-ah b-ah-l','jj'],
'ineffable':['ih-n eh1 f-ah b-ah-l','jj'],
'ineffective':['ih-n ih f-eh1-k t-ih-v','jj'],
'ineffectiveness':['ih-n ah f-eh1-k t-ih-v n-ah-s','nn'],
'ineffectual':['ih-n ah f-eh1-k ch-uw ah-l','jj'],
'inefficiency':['ih-n ih f-ih1 sh-ah-n s-iy','nn'],
'inefficient':['ih-n ih f-ih1 sh-ah-n-t','jj'],
'ineligible':['ih-n eh1 l-ih jh-ah b-ah-l','jj'],
'inept':['ih-n eh1-p-t','jj'],
'ineptitude':['ih-n eh1-p t-ih t-uw-d','nn'],
'ineptly':['ih-n eh1-p-t l-iy','rb'],
'inequality':['ih-n ih k-w-aa1 l-ah t-iy','nn'],
'inequitable':['ih-n eh1 k-w-ah t-ah b-ah-l','jj'],
'inequity':['ih-n eh1 k-w-ah t-iy','nn'],
'inert':['ih n-er1-t','jj'],
'inertia':['ih n-er1 sh-ah','nn'],
'inertial':['ih n-er1 sh-ah-l','jj'],
'inescapable':['ih-n ih-s k-ey1 p-ah b-ah-l','jj'],
'inevitability':['ih-n eh v-ih t-ah b-ih1 l-ih t-iy','nn'],
'inevitable':['ih-n eh1 v-ah t-ah b-ah-l','jj'],
'inevitably':['ih-n eh1 v-ah t-ah b-l-iy','rb'],
'inexact':['ih-n ih-g z-ae1-k-t','jj'],
'inexcusable':['ih-n ih-k s-k-y-uw1 z-ah b-ah-l','jj'],
'inexhaustible':['ih-n ih-g z-ao1 s-t-ah b-ah-l','jj'],
'inexorable':['ih-n eh1-k s-er ah b-ah-l','jj'],
'inexorably':['ih-n eh1-k s-er ah b-l-iy','rb'],
'inexpensive':['ih-n ih-k s-p-eh1-n s-ih-v','jj'],
'inexperience':['ih-n ih-k s-p-ih1 r-iy ah-n-s','nn'],
'inexperienced':['ih-n ih-k s-p-ih1 r-iy ah-n-s-t','jj'],
'inexplicable':['ih-n ah-k s-p-l-ih1 s-ah b-ah-l','jj'],
'inexplicably':['ih-n ih-k s-p-l-ih1 k-ah b-l-iy','rb'],
'inextricably':['ih-n eh1-k s-t-r-ih k-ah b-l-iy','rb'],
'infallibility':['ih-n f-ae l-ih b-ih1 l-ih t-iy','nn'],
'infallible':['ih-n f-ae1 l-ah b-ah-l','jj nn'],
'infamous':['ih1-n f-ah m-ah-s','jj'],
'infamy':['ih1-n f-ah m-iy','nn'],
'infancy':['ih1-n f-ah-n s-iy','nn'],
'infant':['ih1-n f-ah-n-t','nn jj'],
'infantile':['ih1-n f-ah-n t-ih-l','jj'],
'infantry':['ih1-n f-ah-n t-r-iy','nn'],
'infantryman':['ih1-n f-ae-n t-r-iy m-ah-n','nn'],
'infarction':['ih-n f-aa1-r-k sh-ah-n','nn'],
'infatuation':['ih-n f-ae ch-uw ey1 sh-ah-n','nn'],
'infect':['ih-n f-eh1-k-t','vb'],
'infected':['ih-n f-eh1-k t-ah-d','vbn vbd jj'],
'infecting':['ih-n f-eh1-k t-ih-ng','vbg'],
'infection':['ih-n f-eh1-k sh-ah-n','nn'],
'infectious':['ih-n f-eh1-k sh-ah-s','jj'],
'infer':['ih-n f-er1','vb'],
'inference':['ih1-n f-er ah-n-s','nn'],
'inferior':['ih-n f-ih1 r-iy er','jj'],
'inferiority':['ih-n f-ih r-iy ao1 r-ih t-iy','nn'],
'infernal':['ih-n f-er1 n-ah-l','jj'],
'inferno':['ih-n f-er1 n-ow','nn'],
'inferred':['ih-n f-er1-d','vbd vbn'],
'infertile':['ih-n f-er1 t-ah-l','jj'],
'infertility':['ih-n f-er t-ih1 l-ih t-iy','nn'],
'infest':['ih-n f-eh1-s-t','vb'],
'infestation':['ih-n f-eh1 s-t-ey1 sh-ah-n','nn'],
'infested':['ih-n f-eh1 s-t-ah-d','vbn jj'],
'infests':['ih-n f-eh1-s-t-s','vbz'],
'infidel':['ih1-n f-ih d-eh-l','jj'],
'infidelity':['ih-n f-ih d-eh1 l-ih t-iy','nn'],
'infield':['ih1-n f-iy-l-d','nn'],
'infielder':['ih1-n f-iy-l d-er','nn'],
'infighting':['ih1-n f-ay t-ih-ng','nn'],
'infiltrate':['ih-n f-ih1-l t-r-ey-t','vb'],
'infiltrated':['ih-n f-ih1-l t-r-ey t-ih-d','vbn vbd'],
'infiltrating':['ih-n f-ih1-l t-r-ey t-ih-ng','vbg'],
'infiltration':['ih-n f-ih-l t-r-ey1 sh-ah-n','nn'],
'infinite':['ih1-n f-ah n-ah-t','jj nn'],
'infinitely':['ih1-n f-ah n-ah-t l-iy','rb'],
'infinitesimal':['ih-n f-ih n-ih t-eh1 s-ih m-ah-l','jj'],
'infinity':['ih-n f-ih1 n-ah t-iy','nn'],
'infirm':['ih-n f-er1-m','jj'],
'infirmary':['ih-n f-er1 m-er iy','nn'],
'inflame':['ih-n f-l-ey1-m','vb'],
'inflamed':['ih-n f-l-ey1-m-d','jj vbd'],
'inflammation':['ih-n f-l-ah m-ey1 sh-ah-n','nn'],
'inflammatory':['ih-n f-l-ae1 m-ah t-ao r-iy','jj'],
'inflate':['ih-n f-l-ey1-t','vb'],
'inflated':['ih-n f-l-ey1 t-ah-d','jj vbd vbn'],
'inflates':['ih-n f-l-ey1-t-s','vbz'],
'inflating':['ih-n f-l-ey1 t-ih-ng','vbg'],
'inflation':['ih-n f-l-ey1 sh-ah-n','nn'],
'inflationary':['ih-n f-l-ey1 sh-ah-n eh r-iy','jj'],
'inflected':['ih-n f-l-eh1-k t-ah-d','vbn jj'],
'inflection':['ih-n f-l-eh1-k sh-ah-n','nn'],
'inflexible':['ih-n f-l-eh1-k s-ah b-ah-l','jj'],
'inflict':['ih-n f-l-ih1-k-t','vb'],
'inflicted':['ih-n f-l-ih1-k t-ih-d','vbd vbn'],
'inflicting':['ih-n f-l-ih1-k t-ih-ng','vbg'],
'infliction':['ih-n f-l-ih1-k sh-ah-n','nn'],
'inflight':['ih1-n f-l-ay-t','jj'],
'inflow':['ih1-n f-l-ow','nn'],
'influence':['ih1-n f-l-uw ah-n-s','nn vb vbp'],
'influenced':['ih1-n f-l-uw ah-n-s-t','vbn vbd'],
'influences':['ih1-n f-l-uw ah-n s-ih-z','nns vbz'],
'influencing':['ih1-n f-l-uw ah-n s-ih-ng','vbg'],
'influential':['ih-n f-l-uw eh1-n ch-ah-l','jj'],
'influenza':['ih-n f-l-uw eh1-n z-ah','nn'],
'influx':['ih1-n f-l-ah-k-s','nn'],
'inform':['ih-n f-ao1-r-m','vb vbp'],
'informal':['ih-n f-ao1-r m-ah-l','jj'],
'informality':['ih-n f-er m-ae1 l-ih t-iy','nn'],
'informally':['ih-n f-ao1-r m-ah l-iy','rb'],
'informant':['ih-n f-ao1-r m-ah-n-t','nn'],
'informatics':['ih-n f-er m-ae1 t-ih-k-s','nns'],
'information':['ih-n f-er m-ey1 sh-ah-n','nn'],
'informational':['ih-n f-er m-ey1 sh-ah n-ah-l','jj'],
'informative':['ih-n f-ao1-r m-ah t-ih-v','jj'],
'informed':['ih-n f-ao1-r-m-d','vbn vbd jj'],
'informer':['ih-n f-ao1-r m-er','jj'],
'informing':['ih-n f-ao1-r m-ih-ng','vbg'],
'informs':['ih-n f-ao1-r-m-z','vbz'],
'infraction':['ih-n f-r-ae1-k sh-ah-n','nn'],
'infrared':['ih-n f-r-er eh1-d','jj nn'],
'infrastructural':['ih-n f-r-ah s-t-r-ah1-k ch-er ah-l','jj'],
'infrastructure':['ih-n f-r-ah s-t-r-ah1-k ch-er','nn'],
'infrequent':['ih-n f-r-iy1 k-w-ah-n-t','jj'],
'infrequently':['ih-n f-r-iy1 k-w-ah-n-t l-iy','rb'],
'infringe':['ih-n f-r-ih1-n-jh','vb vbp'],
'infringed':['ih-n f-r-ih1-n-jh-d','vbd vbn'],
'infringement':['ih-n f-r-ih1-n-jh m-ah-n-t','nn'],
'infringes':['ih-n f-r-ih1-n jh-ih-z','vbz'],
'infringing':['ih-n f-r-ih1-n jh-ih-ng','vbg'],
'infuriate':['ih-n f-y-uh1 r-iy ey-t','vb'],
'infuriated':['ih-n f-y-uh1 r-iy ey t-ah-d','vbd vbn'],
'infuriating':['ih-n f-y-uh1 r-iy ey t-ih-ng','jj vbg'],
'infuse':['ih-n f-y-uw1-z','vb'],
'infused':['ih-n f-y-uw1-z-d','vbn'],
'infusion':['ih-n f-y-uw1 zh-ah-n','nn'],
'ingenious':['ih-n jh-iy1 n-y-ah-s','jj'],
'ingeniously':['ih-n jh-iy1 n-y-ah-s l-iy','rb'],
'ingenuity':['ih-n jh-ah n-uw1 ah t-iy','nn'],
'ingest':['ih-n jh-eh1-s-t','vbp vb'],
'ingested':['ih-n jh-eh1 s-t-ah-d','vbd vbn'],
'ingestion':['ih-n jh-eh1-s ch-ah-n','nn'],
'inglorious':['ih-n g-l-ao1 r-iy ah-s','jj'],
'ingot':['ih1-ng g-ah-t','nn'],
'ingrained':['ih-n g-r-ey1-n-d','jj'],
'ingratiate':['ih-ng g-r-ey1 sh-iy ey-t','vb'],
'ingratiating':['ih-ng g-r-ey1 sh-iy ey t-ih-ng','jj'],
'ingredient':['ih-n g-r-iy1 d-iy ah-n-t','nn'],
'inhabit':['ih-n hh-ae1 b-ah-t','vbp'],
'inhabitants':['ih-n hh-ae1 b-ah t-ah-n-t-s','nns'],
'inhabitation':['ih-n hh-ae b-ah t-ey1 sh-ah-n','nn'],
'inhabited':['ih-n hh-ae1 b-ah t-ah-d','vbn vbd jj'],
'inhalation':['ih-n ah l-ey1 sh-ah-n','nn'],
'inhaling':['ih-n hh-ey1 l-ih-ng','vbg'],
'inherent':['ih-n hh-ih1 r-ah-n-t','jj'],
'inherently':['ih-n hh-ih1 r-ah-n-t l-iy','rb'],
'inherit':['ih-n hh-eh1 r-ah-t','vb vbp'],
'inheritable':['ih-n hh-eh1 r-ah t-ah b-ah-l','jj'],
'inheritance':['ih-n hh-eh1 r-ah t-ah-n-s','nn'],
'inherited':['ih-n hh-eh1 r-ah t-ih-d','vbn vbd jj'],
'inheriting':['ih-n hh-eh1 r-ah t-ih-ng','vbg'],
'inherits':['ih-n hh-eh1 r-ah-t-s','vbz'],
'inhibit':['ih-n hh-ih1 b-ah-t','vb vbp'],
'inhibited':['ih-n hh-ih1 b-ah t-ih-d','vbd vbn'],
'inhibiting':['ih-n hh-ih1 b-ah t-ih-ng','vbg jj'],
'inhibition':['ih-n hh-ah b-ih1 sh-ah-n','nn'],
'inhibitor':['ih-n hh-ih1 b-ah t-er','nn'],
'inhibitory':['ih-n hh-ih1 b-ah t-ao r-iy','jj'],
'inhibits':['ih-n hh-ih1 b-ah-t-s','vbz'],
'inhospitable':['ih-n hh-aa1 s-p-ah t-ah b-ah-l','jj'],
'inhuman':['ih-n hh-y-uw1 m-ah-n','jj'],
'inhumane':['ih-n hh-y-uw m-ey1-n','jj'],
'inimical':['ih n-ih1 m-ih k-ah-l','jj'],
'inimitable':['ih n-ih1 m-ah t-ah b-ah-l','jj'],
'initial':['ih n-ih1 sh-ah-l','jj nn vb'],
'initialed':['ih n-ih1 sh-ah-l-d','vbd vbn'],
'initially':['ih n-ih1 sh-ah l-iy','rb'],
'initials':['ih n-ih1 sh-ah-l-z','nns'],
'initiate':['ih n-ih1 sh-iy ey-t','vb nn vbp'],
'initiated':['ih n-ih1 sh-iy ey t-ah-d','vbn vbd'],
'initiates':['ih n-ih1 sh-iy ah-t-s','vbz nns'],
'initiating':['ih n-ih1 sh-iy ey t-ih-ng','vbg'],
'initiation':['ih n-ih sh-iy ey1 sh-ah-n','nn'],
'initiative':['ih n-ih1 sh-ah t-ih-v','nn'],
'initiatives':['ih n-ih1 sh-ah t-ih-v-z','nns vbz'],
'initiator':['ih n-ih sh-iy1 ey t-er','nn'],
'inject':['ih-n jh-eh1-k-t','vb vbp'],
'injected':['ih-n jh-eh1-k t-ah-d','vbn vbd'],
'injecting':['ih-n jh-eh1-k t-ih-ng','vbg'],
'injection':['ih-n jh-eh1-k sh-ah-n','nn'],
'injects':['ih-n jh-eh1-k-t-s','vbz'],
'injunction':['ih-n jh-ah1-ng-k sh-ah-n','nn'],
'injunctive':['ih-n jh-ah1-ng-k t-ih-v','jj'],
'injure':['ih1-n jh-er','vb vbp'],
'injured':['ih1-n jh-er-d','vbn vbd jj'],
'injures':['ih1-n jh-er-z','vbz'],
'injuring':['ih1-n jh-er ih-ng','vbg'],
'injurious':['ih-n jh-uh1 r-iy ah-s','jj'],
'injury':['ih1-n jh-er iy','nn'],
'injustice':['ih-n jh-ah1 s-t-ih-s','nn'],
'ink':['ih1-ng-k','nn'],
'inkling':['ih1-ng k-l-ih-ng','nn'],
'inlaid':['ih1-n l-ey-d','vbn'],
'inland':['ih1-n l-ae-n-d','rb jj'],
'inlay':['ih1-n l-ey','nn'],
'inlet':['ih1-n l-eh-t','nn'],
'inmate':['ih1-n m-ey-t','nn'],
'inn':['ih1-n','nn'],
'innards':['ih1 n-er-d-z','nns nn'],
'innate':['ih n-ey1-t','jj'],
'inner':['ih1 n-er','jj'],
'innermost':['ih1 n-er m-ow-s-t','jj'],
'inning':['ih1 n-ih-ng','nn'],
'innocence':['ih1 n-ah s-ah-n-s','nn'],
'innocent':['ih1 n-ah s-ah-n-t','jj'],
'innocently':['ih1 n-ah s-ah-n-t l-iy','rb'],
'innocuous':['ih-n aa1 k-y-uw ah-s','jj'],
'innovate':['ih1 n-ah v-ey-t','vb'],
'innovated':['ih1 n-ah v-ey t-ih-d','vbd'],
'innovation':['ih n-ah v-ey1 sh-ah-n','nn'],
'innovative':['ih1 n-ah v-ey t-ih-v','jj'],
'innovator':['ih1 n-ah v-ey t-er','nn'],
'innuendo':['ih n-y-uw eh1-n d-ow','nn'],
'innumerable':['ih n-uw1 m-er ah b-ah-l','jj'],
'inoculate':['ih-n aa1 k-y-ah l-ey-t','vb'],
'inoculation':['ih-n aa k-y-ah l-ey1 sh-ah-n','nn'],
'inoperable':['ih-n aa1 p-er ah b-ah-l','jj'],
'inoperative':['ih-n aa1 p-er ah t-ih-v','jj'],
'inopportune':['ih-n aa p-er t-uw1-n','jj'],
'inordinate':['ih-n ao1-r d-ah n-ih-t','jj'],
'inordinately':['ih-n ao1-r d-ah n-ah-t l-iy','rb'],
'inorganic':['ih-n ao-r g-ae1 n-ih-k','jj'],
'inpatient':['ih1-n p-ey sh-ah-n-t','nn'],
'input':['ih1-n p-uh-t','nn'],
'inquest':['ih1-n k-w-eh-s-t','nn'],
'inquire':['ih-n k-w-ay1-r','vb'],
'inquired':['ih-n k-w-ay1 er-d','vbd vbn'],
'inquiring':['ih-n k-w-ay1 er ih-ng','jj vbg'],
'inquiry':['ih-n k-w-ay1 r-iy','nn'],
'inquisitive':['ih-n k-w-ih1 z-ih t-ih-v','jj'],
'inroad':['ih1-n r-ow-d','nn'],
'insane':['ih-n s-ey1-n','jj'],
'insanity':['ih-n s-ae1 n-ah t-iy','nn'],
'insatiable':['ih-n s-ey1 sh-ah b-ah-l','jj'],
'inscribed':['ih-n s-k-r-ay1-b-d','vbn vbd jj'],
'inscription':['ih-n s-k-r-ih1-p sh-ah-n','nn'],
'inscrutable':['ih-n s-k-r-uw1 t-ah b-ah-l','jj'],
'insect':['ih1-n s-eh-k-t','nn jj'],
'insecticide':['ih-n s-eh1-k t-ah s-ay-d','nn'],
'insecure':['ih1-n s-ah k-y-er','jj'],
'insecurity':['ih-n s-ih k-y-uh1 r-ih t-iy','nn'],
'insemination':['ih-n s-eh m-ah n-ey1 sh-ah-n','nn'],
'insensitive':['ih-n s-eh1-n s-ah t-ih-v','jj'],
'insensitivity':['ih-n s-eh-n s-ah t-ih1 v-ah t-iy','nn'],
'inseparable':['ih-n s-eh1 p-er ah b-ah-l','jj'],
'insert':['ih-n s-er1-t','vb vbp nn'],
'inserted':['ih-n s-er1 t-ah-d','vbn vbd'],
'inserting':['ih-n s-er1 t-ih-ng','vbg'],
'insertion':['ih-n s-er1 sh-ah-n','nn'],
'inserts':['ih-n s-er1-t-s','nns'],
'inset':['ih1-n s-eh-t','nn'],
'inshore':['ih1-n sh-ao1-r','jj rb'],
'inside':['ih-n s-ay1-d','in jj nn rb'],
'insider':['ih-n s-ay1 d-er','nn jj'],
'insidious':['ih-n s-ih1 d-iy ah-s','jj'],
'insight':['ih1-n s-ay-t','nn'],
'insightful':['ih1-n s-ay-t f-ah-l','jj'],
'insignificance':['ih-n s-ih-g n-y-ih1 f-ih k-ah-n-s','nn'],
'insignificant':['ih-n s-ih-g n-y-ih1 f-ih k-ah-n-t','jj'],
'insincere':['ih-n s-ih-n s-ih1-r','jj'],
'insinuate':['ih-n s-ih1 n-y-uw ey-t','vb'],
'insinuating':['ih-n s-ih1 n-y-uw ey t-ih-ng','vbg'],
'insinuation':['ih-n s-ih n-y-uw ey1 sh-ah-n','nn'],
'insist':['ih-n s-ih1-s-t','vbp vb'],
'insisted':['ih-n s-ih1 s-t-ah-d','vbd vbn'],
'insistence':['ih-n s-ih1 s-t-ah-n-s','nn'],
'insistent':['ih-n s-ih1 s-t-ah-n-t','jj'],
'insistently':['ih-n s-ih1 s-t-ah-n-t l-iy','rb'],
'insisting':['ih-n s-ih1 s-t-ih-ng','vbg'],
'insists':['ih-n s-ih1-s-t-s','vbz'],
'insofar':['ih1-n s-ah f-aa-r','rb in'],
'insolence':['ih1-n s-ah l-ah-n-s','nn'],
'insolent':['ih1-n s-ah l-ah-n-t','jj'],
'insoluble':['ih-n s-aa1 l-y-ah b-ah-l','jj nn'],
'insolvency':['ih-n s-aa1-l v-ah-n s-iy','nn'],
'insolvent':['ih-n s-aa1-l v-ah-n-t','jj nn'],
'insomnia':['ih-n s-aa1-m n-iy ah','nn'],
'inspect':['ih-n s-p-eh1-k-t','vb vbp'],
'inspected':['ih-n s-p-eh1-k t-ih-d','vbn vbd'],
'inspecting':['ih-n s-p-eh1-k t-ih-ng','vbg'],
'inspection':['ih-n s-p-eh1-k sh-ah-n','nn'],
'inspector':['ih-n s-p-eh1-k t-er','nn'],
'inspectors':['ih-n s-p-eh1-k t-er-z','nns vbz'],
'inspiration':['ih-n s-p-er ey1 sh-ah-n','nn'],
'inspirational':['ih-n s-p-er ey1 sh-ah n-ah-l','jj'],
'inspire':['ih-n s-p-ay1-r','vb vbp'],
'inspired':['ih-n s-p-ay1 er-d','vbn jj vbd'],
'inspires':['ih-n s-p-ay1-r-z','vbz'],
'inspiring':['ih-n s-p-ay1 r-ih-ng','jj vbg'],
'instability':['ih-n s-t-ah b-ih1 l-ih t-iy','nn'],
'install':['ih-n s-t-ao1-l','vb vbp'],
'installation':['ih-n s-t-ah l-ey1 sh-ah-n','nn'],
'installed':['ih-n s-t-ao1-l-d','vbn jj vbd'],
'installer':['ih-n s-t-ao1 l-er','nn'],
'installing':['ih-n s-t-ao1 l-ih-ng','vbg'],
'installment':['ih-n s-t-ao1-l m-ah-n-t','nn jj'],
'instance':['ih1-n s-t-ah-n-s','nn'],
'instant':['ih1-n s-t-ah-n-t','nn jj'],
'instantaneous':['ih-n s-t-ah-n t-ae1 n-iy ah-s','jj'],
'instantaneously':['ih-n s-t-ah-n t-ae1 n-iy ah-s l-iy','rb'],
'instantly':['ih1-n s-t-ah-n-t l-iy','rb'],
'instead':['ih-n s-t-eh1-d','rb'],
'instigate':['ih1-n s-t-ah g-ey-t','vb'],
'instigated':['ih1-n s-t-ah g-ey t-ih-d','vbd vbn'],
'instigating':['ih1-n s-t-ah g-ey t-ih-ng','vbg'],
'instigation':['ih-n s-t-ih g-ey1 sh-ah-n','nn'],
'instigator':['ih1-n s-t-ah g-ey t-er','nn'],
'instill':['ih-n s-t-ih1-l','vb'],
'instinct':['ih1-n s-t-ih-ng-k-t','nn'],
'instinctive':['ih-n s-t-ih1-ng-k t-ih-v','jj'],
'instinctively':['ih-n s-t-ih1-ng-k t-ih-v l-iy','rb'],
'institute':['ih1-n s-t-ah t-uw-t','nn vb'],
'instituted':['ih1-n s-t-ah t-uw t-ah-d','vbn vbd'],
'institutes':['ih1-n s-t-ah t-uw-t-s','nn'],
'instituting':['ih1-n s-t-ih t-uw t-ih-ng','vbg'],
'institution':['ih-n s-t-ih t-uw1 sh-ah-n','nn'],
'institutional':['ih-n s-t-ih t-uw1 sh-ah n-ah-l','jj'],
'institutionalized':['ih-n s-t-ih t-uw1 sh-ah n-ah-l ay-z-d','vbn jj'],
'institutionally':['ih-n s-t-ih t-uw1 sh-ah n-ah-l iy','rb'],
'instruct':['ih-n s-t-r-ah1-k-t','vb vbp'],
'instructed':['ih-n s-t-r-ah1-k t-ah-d','vbn vbd'],
'instructing':['ih-n s-t-r-ah1-k t-ih-ng','vbg'],
'instruction':['ih-n s-t-r-ah1-k sh-ah-n','nn'],
'instructional':['ih-n s-t-r-ah1-k sh-ah n-ah-l','jj'],
'instructive':['ih-n s-t-r-ah1-k t-ih-v','jj'],
'instructor':['ih-n s-t-r-ah1-k t-er','nn'],
'instructs':['ih-n s-t-r-ah1-k-t-s','vbz'],
'instrument':['ih1-n s-t-r-ah m-ah-n-t','nn'],
'instrumental':['ih-n-s t-r-ah m-eh1-n t-ah-l','jj nn'],
'instrumentalists':['ih-n-s t-r-ah m-eh1-n t-ah l-ih-s-t-s','nns'],
'instrumentation':['ih-n-s t-r-ah m-eh-n t-ey1 sh-ah-n','nn'],
'insubordination':['ih-n s-ah b-ao-r d-ah n-ey1 sh-ah-n','nn'],
'insubstantial':['ih-n s-ah-b s-t-ae1-n ch-ah-l','jj'],
'insufferable':['ih-n s-ah1 f-er ah b-ah-l','jj'],
'insufficient':['ih-n s-ah f-ih1 sh-ah-n-t','jj'],
'insufficiently':['ih-n s-ah f-ih1 sh-ah-n-t l-iy','rb'],
'insularity':['ih-n s-ah l-eh1 r-ih t-iy','nn'],
'insulate':['ih1-n s-ah l-ey-t','vb vbp'],
'insulated':['ih1-n s-ah l-ey t-ah-d','vbn . jj'],
'insulating':['ih1-n s-ah l-ey t-ih-ng','vbg jj nn'],
'insulation':['ih-n s-ah l-ey1 sh-ah-n','nn'],
'insulator':['ih1-n s-ah l-ey t-er','nn'],
'insulin':['ih1-n s-ah l-ah-n','nn'],
'insult':['ih-n s-ah1-l-t','nn vb'],
'insulted':['ih-n s-ah1-l t-ih-d','vbn vbd'],
'insulting':['ih-n s-ah1-l t-ih-ng','jj vbg'],
'insults':['ih-n s-ah1-l-t-s','nns'],
'insupportable':['ih-n s-ah p-ao1-r t-ah b-ah-l','jj'],
'insurance':['ih-n sh-uh1 r-ah-n-s','nn'],
'insure':['ih-n sh-uh1-r','vb vbp'],
'insured':['ih-n sh-uh1-r-d','vbn vbd jj nn'],
'insurer':['ih-n sh-uh1 r-er','nn'],
'insures':['ih-n sh-uh1-r-z','vbz'],
'insurgency':['ih-n s-er1 jh-ah-n s-iy','nn'],
'insurgent':['ih-n s-er1 jh-ah-n-t','jj'],
'insuring':['ih-n sh-uh1 r-ih-ng','vbg'],
'insurmountable':['ih-n s-er m-aw1-n t-ah b-ah-l','jj'],
'insurrection':['ih-n s-er eh1-k sh-ah-n','nn jj'],
'intact':['ih-n t-ae1-k-t','jj'],
'intake':['ih1-n t-ey-k','nn'],
'intangible':['ih-n t-ae1-n jh-ah b-ah-l','jj nn'],
'integers':['ih1-n t-ah jh-er-z','nns'],
'integral':['ih1-n t-ah g-r-ah-l','nn jj'],
'integrate':['ih1-n t-ah g-r-ey-t','vb vbp'],
'integrated':['ih1-n t-ah g-r-ey t-ah-d','vbn jj'],
'integrates':['ih1-n t-ah g-r-ey-t-s','vbz'],
'integrating':['ih1-n t-ah g-r-ey t-ih-ng','vbg'],
'integration':['ih-n t-ah g-r-ey1 sh-ah-n','nn'],
'integrator':['ih1-n t-ah g-r-ey t-er','nn'],
'integrity':['ih-n t-eh1 g-r-ah t-iy','nn'],
'intellect':['ih1-n t-ah l-eh-k-t','nn'],
'intellectual':['ih-n t-ah l-eh1-k ch-uw ah-l','jj nn'],
'intellectually':['ih-n t-ah l-eh1-k ch-uw ah l-iy','rb'],
'intelligence':['ih-n t-eh1 l-ah jh-ah-n-s','nn'],
'intelligent':['ih-n t-eh1 l-ah jh-ah-n-t','jj'],
'intelligently':['ih-n t-eh1 l-ih jh-ah-n-t l-iy','rb'],
'intelligentsia':['ih-n t-eh l-ah jh-eh1-n-t s-iy ah','nn nns'],
'intelligible':['ih-n t-eh1 l-ah jh-ah b-ah-l','jj nn'],
'intend':['ih-n t-eh1-n-d','vbp vb'],
'intended':['ih-n t-eh1-n d-ah-d','vbn vbd jj'],
'intending':['ih-n t-eh1-n d-ih-ng','vbg'],
'intends':['ih-n t-eh1-n-d-z','vbz'],
'intense':['ih-n t-eh1-n-s','jj'],
'intensely':['ih-n t-eh1-n-s l-iy','rb'],
'intensification':['ih-n t-eh-n s-ah f-ah k-ey1 sh-ah-n','nn'],
'intensified':['ih-n t-eh1-n s-ah f-ay-d','vbn vbd jj'],
'intensify':['ih-n t-eh1-n s-ah f-ay','vb vbp'],
'intensifying':['ih-n t-eh1-n s-ah f-ay ih-ng','vbg jj'],
'intensity':['ih-n t-eh1-n s-ah t-iy','nn'],
'intensive':['ih-n t-eh1-n s-ih-v','jj nn'],
'intensively':['ih-n t-eh1-n s-ih-v l-iy','rb'],
'intent':['ih-n t-eh1-n-t','nn jj'],
'intention':['ih-n t-eh1-n ch-ah-n','nn'],
'intentional':['ih-n t-eh1-n sh-ah n-ah-l','jj'],
'intentionally':['ih-n t-eh1-n sh-ah-n ah l-iy','rb'],
'intentioned':['ih-n t-eh1-n ch-ah-n-d','jj'],
'intently':['ih-n t-eh1-n-t l-iy','rb'],
'interact':['ih-n t-er ae1-k-t','vbp vb'],
'interacting':['ih-n t-er ae1-k t-ih-ng','vbg nn'],
'interaction':['ih-n t-er ae1-k sh-ah-n','nn'],
'interactive':['ih-n t-er ae1-k t-ih-v','jj'],
'interacts':['ih-n t-er ae1-k-t-s','vbz'],
'interagency':['ih-n t-er ey1 jh-ah-n s-iy','nn jj'],
'interbank':['ih-n t-er b-ae1-ng-k','nn jj rb'],
'intercede':['ih-n t-er s-iy1-d','vb'],
'interceded':['ih-n t-er s-iy1 d-ih-d','vbd'],
'intercept':['ih-n t-er s-eh1-p-t','nn vbp vb'],
'intercepted':['ih-n t-er s-eh1-p t-ah-d','vbd vbn'],
'interception':['ih-n t-er s-eh1-p sh-ah-n','nn'],
'interceptor':['ih-n t-er s-eh1-p t-er','nn'],
'intercepts':['ih-n t-er s-eh1-p-t-s','nns'],
'interchange':['ih-n t-er ch-ey1-n-jh','nn'],
'interchangeable':['ih-n t-er ch-ey1-n jh-ah b-ah-l','jj'],
'intercity':['ih1-n t-er s-ih t-iy','jj nn'],
'intercollegiate':['ih-n t-er k-ah l-iy1 jh-ah-t','jj'],
'intercom':['ih1-n t-er k-aa-m','nn'],
'intercompany':['ih-n t-er k-ah1-m p-ah n-iy','nn'],
'interconnect':['ih-n t-er k-ah n-eh1-k-t','nn vb'],
'interconnected':['ih-n t-er k-ah n-eh1-k t-ih-d','vbn'],
'interconnections':['ih-n t-er k-ah n-eh1-k sh-ah-n-z','nns'],
'intercontinental':['ih-n t-er k-aa-n t-ah n-eh1-n t-ah-l','jj'],
'intercourse':['ih1-n t-er k-ao-r-s','nn'],
'interdependence':['ih-n t-er d-ah p-eh1-n d-ah-n-s','nn'],
'interdependent':['ih-n t-er d-ih p-eh1-n d-ah-n-t','jj'],
'interdicting':['ih-n t-er d-ih1-k t-ih-ng','vbg'],
'interdiction':['ih-n t-er d-ih1-k sh-ah-n','nn'],
'interest':['ih1-n t-r-ah-s-t','nn vbp vb'],
'interested':['ih1-n t-r-ah s-t-ah-d','jj vbd vbn'],
'interesting':['ih1-n t-r-ah s-t-ih-ng','jj'],
'interestingly':['ih1-n t-er eh s-t-ih-ng l-iy','rb'],
'interestrate':['ih1-n t-er ah s-t-r-ey-t','nn'],
'interests':['ih1-n t-r-ah-s-t-s','nns vbz'],
'interface':['ih1-n t-er f-ey-s','nn vb'],
'interfaces':['ih1-n t-er f-ey s-ih-z','nns'],
'interfaith':['ih-n t-er f-ey1-th','jj'],
'interfere':['ih-n t-er f-ih1-r','vb vbp'],
'interfered':['ih-n t-er f-ih1-r-d','vbd vbn'],
'interference':['ih-n t-er f-ih1 r-ah-n-s','nn'],
'interferes':['ih-n t-er f-ih1-r-z','vbz'],
'interfering':['ih-n t-er f-ih1 r-ih-ng','vbg'],
'interferometer':['ih-n t-er f-er aa1 m-ah t-er','nn'],
'interferon':['ih-n t-er f-eh1 r-aa-n','nn'],
'intergenerational':['ih-n t-er jh-eh1 n-er ey1 sh-ah n-ah-l','jj'],
'intergovernmental':['ih-n t-er g-ah v-er-n m-eh1-n t-ah-l','jj'],
'intergroup':['ih-n t-er g-r-uw1-p','jj nn'],
'interim':['ih1-n t-er ah-m','jj nn'],
'interior':['ih-n t-ih1 r-iy er','jj nn'],
'interject':['ih-n t-er jh-eh1-k-t','vbp'],
'interjected':['ih-n t-er jh-eh1-k t-ih-d','vbd'],
'interjects':['ih-n t-er jh-eh1-k-t-s','vbz'],
'interlaced':['ih-n t-er l-ey1-s-t','vbn jj'],
'interlinked':['ih-n t-er l-ih1-ng-k-t','vbn'],
'interlocking':['ih1-n t-er l-aa k-ih-ng','vbg jj'],
'interloper':['ih1-n t-er l-ow p-er','nn'],
'interlude':['ih1-n t-er l-uw-d','nn'],
'intermarket':['ih-n t-er m-aa1-r k-ih-t','jj nn'],
'intermediary':['ih-n t-er m-iy1 d-iy eh r-iy','nn jj'],
'intermediate':['ih-n t-er m-iy1 d-iy ih-t','jj'],
'interminable':['ih-n t-er1 m-ah n-ah b-ah-l','jj'],
'intermingle':['ih-n t-er m-ih1-ng g-ah-l','vbp'],
'intermission':['ih-n t-er m-ih1 sh-ah-n','nn'],
'intermittent':['ih-n t-er m-ih1 t-ah-n-t','jj'],
'intermittently':['ih-n t-er m-ih1 t-ah-n-t l-iy','rb'],
'intermixed':['ih-n t-er m-ih1-k-s-t','vbd'],
'intermodal':['ih-n t-er m-ow1 d-ah-l','jj'],
'intermolecular':['ih-n t-er m-ah l-eh1 k-y-ah l-er','jj'],
'intern':['ih1-n t-er-n','nn vb'],
'internal':['ih-n t-er1 n-ah-l','jj'],
'internally':['ih-n t-er1 n-ah l-iy','rb'],
'international':['ih-n t-er n-ae1 sh-ah n-ah-l','jj nn'],
'internationalism':['ih-n t-er n-ae1 sh-ah-n ah l-ih z-ah-m','nn'],
'internationalist':['ih-n t-er n-ae1 sh-ah-n ah l-ih-s-t','jj'],
'internationalization':['ih-n t-er n-ae sh-ah-n ah l-ih z-ey1 sh-ah-n','nn'],
'internationalize':['ih-n t-er n-ae1 sh-ah-n ah l-ay-z','vb'],
'internationalized':['ih-n t-er n-ae1 sh-ah-n ah l-ay-z-d','vbn'],
'internationally':['ih-n t-er n-ae1 sh-ah-n ah l-iy','rb'],
'interned':['ih1-n t-er-n-d','vbn'],
'internist':['ih-n t-er1 n-ih-s-t','nn'],
'internment':['ih-n t-er1-n m-ah-n-t','nn'],
'interns':['ih1-n t-er-n-z','nns'],
'interoffice':['ih-n t-er ao1 f-ah-s','jj'],
'interpersonal':['ih-n t-er p-er1 s-ah n-ah-l','jj'],
'interplanetary':['ih-n t-er p-l-ae1 n-ah t-eh r-iy','jj'],
'interplay':['ih1-n t-er p-l-ey','nn'],
'interpolated':['ih t-er1 p-ah l-ey t-ih-d','vbd'],
'interpret':['ih-n t-er1 p-r-ah-t','vb vbp'],
'interpretation':['ih-n t-er p-r-ih t-ey1 sh-ah-n','nn'],
'interpreted':['ih-n t-er1 p-r-ah t-ah-d','vbn vbd'],
'interpreter':['ih-n t-er1 p-r-ah t-er','nn'],
'interpreting':['ih1-n t-er p-r-eh t-ih-ng','vbg'],
'interpretive':['ih-n t-er1 p-r-ah t-ih-v','jj'],
'interprets':['ih-n t-er1 p-r-ah-t-s','vbz'],
'interprovincial