# Detect Clicks Outside and Inside an Element

[View, Screenshot, Remix, or Edit on COTR](https://cotr.dev/snippet/347)

## Code Snippet
```
export function useOutsideAlerter({
  ref,
  onClickOutside,
  onClickInside,
}: {
  ref: React.RefObject<HTMLElement>;
  onClickOutside: () => void;
  onClickInside: () => void;
}) {
  useEffect(() => {
    function handleClickOutside(event: MouseEvent) {
      if (ref.current && !ref.current.contains(event.target as Node)) {
        console.log("You clicked outside of me!");
        onClickOutside();
      } else {
        console.log("You clicked inside of me!");
        onClickInside();
      }
    }
    // Bind the event listener
    document.addEventListener("mousedown", handleClickOutside);
    return () => {
      // Unbind the event listener on clean up
      document.removeEventListener("mousedown", handleClickOutside);
    };
  }, [ref]);
}
```

## Description
This code snippet defines a custom React hook called `useOutsideAlerter`. The purpose of this hook is to detect clicks outside or inside a specific component and execute corresponding callback functions.

Here's a breakdown of the code:

1. The hook takes an object as a parameter with three properties:
   - `ref`: A React ref object that refers to the HTML element of the component.
   - `onClickOutside`: A callback function to be executed when a click occurs outside the component.
   - `onClickInside`: A callback function to be executed when a click occurs inside the component.

2. Inside the hook, the `useEffect` hook is used to add and remove an event listener for the `mousedown` event on the `document` object.

3. The `handleClickOutside` function is defined within the `useEffect` hook. It receives the `event` object as a parameter, which represents the `MouseEvent` triggered by the click.

4. Inside `handleClickOutside`, the code checks if the clicked target is not contained within the component referenced by `ref.current`.
   - If the click occurred outside the component, it logs a message and calls the `onClickOutside` callback function.
   - If the click occurred inside the component, it logs a message and calls the `onClickInside` callback function.

5. The event listener is added to the `document` object using `document.addEventListener("mousedown", handleClickOutside)`. This means that whenever a `mousedown` event occurs anywhere in the document, the `handleClickOutside` function will be executed.

6. The `useEffect` hook returns a cleanup function that removes the event listener when the component unmounts or when the dependencies (in this case, `[ref]`) change.

To use this custom hook in a React component, you would need to provide a ref to the component's HTML element and define the `onClickOutside` and `onClickInside` callback functions. For example:

```jsx
const MyComponent = () => {
  const ref = useRef(null);

  const handleClickOutside = () => {
    // Logic to handle clicks outside the component
  };

  const handleClickInside = () => {
    // Logic to handle clicks inside the component
  };

  useOutsideAlerter({ ref, onClickOutside: handleClickOutside, onClickInside: handleClickInside });

  return <div ref={ref}>My Component</div>;
};
```

In this example, the `useOutsideAlerter` hook is used within the `MyComponent` component. The `ref` is created using `useRef` and passed to the hook along with the `handleClickOutside` and `handleClickInside` callback functions. The `ref` is then attached to the `div` element representing the component.

Now, whenever a click occurs outside or inside the `MyComponent`, the corresponding callback function will be executed based on the click target.

## Tags
javascript, react, hooks, useeffect, outside-click-detection
