import type { Runnable } from '@langchain/core/runnables'
import { BaseOutputParser } from '@langchain/core/output_parsers'
import type { AgentAction, AgentFinish } from '@langchain/core/agents'
import type { BaseMessage } from '@langchain/core/messages'
import type { ChainValues } from '@langchain/core/utils/types'

export interface ChatCompletionMessageToolCall {
    /**
     * The ID of the tool call.
     */
    id: string

    /**
     * The function that the model called.
     */
    function: ChatCompletionMessageToolCall.Function

    /**
     * The type of the tool. Currently, only `function` is supported.
     */
    type: 'function'
}

/**
 * The name and arguments of a function that should be called, as generated by the
 * model.
 */
export interface ChatCompletionMessageFunctionCall {
    /**
     * The arguments to call the function with, as generated by the model in JSON
     * format. Note that the model does not always generate valid JSON, and may
     * hallucinate parameters not defined by your function schema. Validate the
     * arguments in your code before calling your function.
     */
    arguments: string

    /**
     * The name of the function to call.
     */
    name: string
}

// eslint-disable-next-line @typescript-eslint/no-namespace
export namespace ChatCompletionMessageToolCall {
    /**
     * The function that the model called.
     */
    export interface Function {
        /**
         * The arguments to call the function with, as generated by the model in JSON
         * format. Note that the model does not always generate valid JSON, and may
         * hallucinate parameters not defined by your function schema. Validate the
         * arguments in your code before calling your function.
         */
        arguments: string

        /**
         * The name of the function to call.
         */
        name: string
    }
}

/**
 * Specifies a tool the model should use. Use to force the model to call a specific
 * function.
 */
export interface ChatCompletionNamedToolChoice {
    function?: ChatCompletionNamedToolChoice.Function

    /**
     * The type of the tool. Currently, only `function` is supported.
     */
    type?: 'function'
}

// eslint-disable-next-line @typescript-eslint/no-namespace
export namespace ChatCompletionNamedToolChoice {
    export interface Function {
        /**
         * The name of the function to call.
         */
        name: string
    }
}

export interface FunctionDefinition {
    /**
     * The name of the function to be called. Must be a-z, A-Z, 0-9, or contain
     * underscores and dashes, with a maximum length of 64.
     */
    name: string

    /**
     * The parameters the functions accepts, described as a JSON Schema object. See the
     * [guide](https://platform.openai.com/docs/guides/gpt/function-calling) for
     * examples, and the
     * [JSON Schema reference](https://json-schema.org/understanding-json-schema/) for
     * documentation about the format.
     *
     * To describe a function that accepts no parameters, provide the value
     * `{"type": "object", "properties": {}}`.
     */
    parameters: FunctionParameters

    /**
     * A description of what the function does, used by the model to choose when and
     * how to call the function.
     */
    description?: string
}

/**
 * The parameters the functions accepts, described as a JSON Schema object. See the
 * [guide](https://platform.openai.com/docs/guides/gpt/function-calling) for
 * examples, and the
 * [JSON Schema reference](https://json-schema.org/understanding-json-schema/) for
 * documentation about the format.
 *
 * To describe a function that accepts no parameters, provide the value
 * `{"type": "object", "properties": {}}`.
 */
export type FunctionParameters = Record<string, unknown>

export interface ChatCompletionTool {
    function: FunctionDefinition

    /**
     * The type of the tool. Currently, only `function` is supported.
     */
    type: 'function'
}

/**
 * Controls which (if any) function is called by the model. `none` means the model
 * will not call a function and instead generates a message. `auto` means the model
 * can pick between generating a message or calling a function. Specifying a
 * particular function via
 * `{"type: "function", "function": {"name": "my_function"}}` forces the model to
 * call that function.
 *
 * `none` is the default when no functions are present. `auto` is the default if
 * functions are present.
 */
export type ChatCompletionToolChoiceOption =
    | 'none'
    | 'auto'
    | ChatCompletionNamedToolChoice

export interface ChatCompletionToolMessageParam {
    /**
     * The contents of the tool message.
     */
    content: string | null

    /**
     * The role of the messages author, in this case `tool`.
     */
    role: 'tool'

    /**
     * Tool call that this message is responding to.
     */
    tool_call_id: string
}

/**
 * Interface defining the input for creating an agent. It includes the
 * LLMChain instance, an optional output parser, and an optional list of
 * allowed tools.
 */
export interface AgentInput {
    // llmChain: LLMChain
    outputParser: AgentActionOutputParser | undefined
    allowedTools?: string[]
}

/**
 * Interface defining the input for creating a single action agent
 * that uses runnables.
 */
export interface RunnableSingleActionAgentInput {
    runnable: Runnable<
        ChainValues & {
            agent_scratchpad?: string | BaseMessage[]
            stop?: string[]
        },
        AgentAction | AgentFinish
    >
    streamRunnable?: boolean
    defaultRunName?: string
}

/**
 * Interface defining the input for creating a multi-action agent that uses
 * runnables. It includes the Runnable instance, and an optional list of
 * stop strings.
 */
export interface RunnableMultiActionAgentInput {
    runnable: Runnable<
        ChainValues & {
            agent_scratchpad?: string | BaseMessage[]
            stop?: string[]
        },
        AgentAction[] | AgentAction | AgentFinish
    >
    streamRunnable?: boolean
    defaultRunName?: string
    stop?: string[]
}

/** @deprecated Renamed to RunnableMultiActionAgentInput. */
export interface RunnableAgentInput extends RunnableMultiActionAgentInput {}

/**
 * Abstract class representing an output parser specifically for agent
 * actions and finishes in LangChain. It extends the `BaseOutputParser`
 * class.
 */
export abstract class AgentActionOutputParser extends BaseOutputParser<
    AgentAction | AgentFinish
> {}

/**
 * Abstract class representing an output parser specifically for agents
 * that return multiple actions.
 */
export abstract class AgentMultiActionOutputParser extends BaseOutputParser<
    AgentAction[] | AgentFinish
> {}

/**
 * Type representing the stopping method for an agent. It can be either
 * 'force' or 'generate'.
 */
export type StoppingMethod = 'force' | 'generate'
