<?php

namespace CML\Classes;

/**
 * Class Router
 *
 * The Router class handles routing in the CML framework.
 * It matches the requested URL and HTTP method to the defined routes,
 * executes the corresponding callback functions, and handles error cases.
 *
 * @author CallMeLeon <kontakt@callmeleon.de>
 * @see https://docs.callmeleon.de/the-basics#Routing
 */
class Router extends \CML\Classes\HTMLBuilder
{
    use Functions\Functions;
    use Functions\Session;

    /**
     * @var array Stores the indexing status and sitemap data for each route
     */
    protected array $routeIndexData = [];

    /**
     * @var array Stores the indexing status for each route
     */
    protected array $routeIndexStatus = [];

    /**
     * @var array Stores custom hardcoded routes for the sitemap.
     */
    protected array $customSitemapRoutes = [];

    /**
     * Stores the defined routes.
     *
     * @var array
     */
    protected array $routes = [];

    /**
     * Stores the named routes with their corresponding URLs.
     *
     * @var array
     */
    public array $namedRoutes = [];

    /**
     * An array to store metadata for routes.
     * 
     * @var array
     */
    public array $routeMetadata = [];

    /**
     * Stores route-specific middleware functions.
     *
     * @var array
     */
    protected array $middlewares = [];

    /**
     * Stores global middleware functions.
     *
     * @var array
     */
    protected array $globalMiddleware = [];

    /**
     * Stores route aliases.
     *
     * @var array
     */
    protected array $aliases = [];

    /**
     * Stores the Page to show if a route is not defined.
     *
     * @var array
     */
    public array $errorPage = [];

    /**
     * Array of error page variables.
     *
     * @var array
     */
    public array $errorPageVariables = [];

    /**
     * Stores the parameters of the current route.
     *
     * @var array
     */
    protected array $currentRouteParams = [];

    /**
     * Stores the currently requested route.
     *
     * @var string
     */
    public string $currentRoute = '';

    /**
     * Stores the currently url.
     *
     * @var string
     */
    public string $currentUrl = '';


    /**
     * The current name of the router.
     *
     * @var string
     */
    public string $currentRouteName = '';

    /**
     * Stores the current HTTP request method.
     *
     * @var string
     */
    protected string $currentMethod = '';


    /**
     * Stores the URL to redirect to if a route is not defined.
     *
     * @var string
     */
    public string $redirectUrl = "";

    /**
     * Stores sites path.
     *
     * @var string
     */
    public string $sitesPath = "";

    /**
     * Indicates whether the route is an API route.
     *
     * @var bool
     */
    public bool $isApi = false;

    /**
     * Indicates whether the sitemap is enabled or not.
     *
     * @var bool
     */
    private bool $sitemapEnabled = true;

    /**
     * Constructor method for the Router class.
     * Merges the $_GET superglobal array with the query parameters obtained from the getQueryParams() method.
     */
    public function __construct()
    {
        $this->sitesPath = cml_config('SITES_PATH');
        $_GET = array_merge($_GET, $this->getQueryParams());
    }

    /**
     * Match the defined routes.
     */
    public function __destruct()
    {
        $this->registerSitemapRoute();
        $this->_matchRoute();
    }

    /**
     * Set the route as an API route.
     *
     * @return bool
     */
    public function isApi(): bool
    {
        self::setHeader('Content-Type', 'application/json');
        return $this->isApi = true;
    }

    /**
     * Sets the indexing status and sitemap data for the current route
     *
     * @param bool $indexable True if the route should be indexable, false otherwise
     * @param array $sitemapData Additional sitemap data (lastmod, changefreq, priority)
     * @return $this
     */
    public function setIndexable(bool $indexable = true, array $sitemapData = [])
    {
        if (!empty($this->currentRoute)) {
            $this->routeIndexData[$this->currentRoute] = [
                'indexable' => $indexable,
                'lastmod' => $sitemapData['lastmod'] ?? null,
                'changefreq' => $sitemapData['changefreq'] ?? null,
                'priority' => $sitemapData['priority'] ?? null,
            ];
        }
        return $this;
    }


    /**
     * Returns the indexing status of a route
     *
     * @param string $route The route to check
     * @return bool True if the route is indexable, false otherwise
     */
    public function isIndexable(string $route): bool
    {
        if (!cml_config('PRODUCTION')) {
            return false;
        }
        return $this->routeIndexData[$route]['indexable'] ?? true; // Default is indexable
    }

    /**
     * Generates the sitemap
     *
     * @return string The generated sitemap as an XML string
     */
    public function generateSitemap(): string
    {
        $xml = new \SimpleXMLElement('<?xml version="1.0" encoding="UTF-8"?><urlset xmlns="http://www.sitemaps.org/schemas/sitemap/0.9"></urlset>');

        foreach ($this->getAllRoutes() as $route) {
            if ($this->isIndexable($route['url'])) {
                $this->addUrlToSitemap($xml, $route['url'], $this->routeIndexData[$route['url']] ?? []);
            }
        }

        foreach ($this->customSitemapRoutes as $url => $sitemapData) {
            $this->addUrlToSitemap($xml, $url, $sitemapData);
        }

        return $xml->asXML();
    }

    /**
     * Add a custom route to the sitemap.
     *
     * @param string $url The URL of the custom route
     * @param array $sitemapData Additional sitemap data (lastmod, changefreq, priority)
     * @return $this
     */
    public function sitemapAdd(string $url, array $sitemapData = [])
    {
        $this->customSitemapRoutes[$url] = [
            'lastmod' => $sitemapData['lastmod'] ?? null,
            'changefreq' => $sitemapData['changefreq'] ?? null,
            'priority' => $sitemapData['priority'] ?? null,
        ];
        return $this;
    }

    /**
     * Helper method to add a URL to the sitemap XML
     *
     * @param \SimpleXMLElement $xml The XML object
     * @param string $url The URL to add
     * @param array $sitemapData The sitemap data for the URL
     */
    private function addUrlToSitemap(\SimpleXMLElement $xml, string $url, array $sitemapData)
    {
        $urlElement = $xml->addChild('url');
        $urlElement->addChild('loc', $this->url($url));

        if (!empty($sitemapData['lastmod'])) {
            $urlElement->addChild('lastmod', date('c', strtotime($sitemapData['lastmod'])));
        }

        if (!empty($sitemapData['changefreq'])) {
            $urlElement->addChild('changefreq', $sitemapData['changefreq']);
        }

        if (!empty($sitemapData['priority'])) {
            $urlElement->addChild('priority', number_format($sitemapData['priority'], 1));
        }
    }

    /**
     * Disables the sitemap functionality.
     *
     * This method sets the $sitemapEnabled property to false, indicating that the sitemap functionality is disabled.
     */
    public function disableSitemap()
    {
        $this->sitemapEnabled = false;
    }

    /**
     * Registers the sitemap route
     */
    protected  function registerSitemapRoute()
    {
        if ($this->sitemapEnabled == true) {
            $this->addRoute('GET', '/sitemap.xml', function () {
                $this->isApi();
                self::setHeader('Content-Type', 'application/xml');
                echo $this->generateSitemap();
            })->setIndexable(false); // The sitemap itself should not be indexed
        }
    }

    /**
     * Sets the robots meta tag based on the indexing status of the current route
     */
    protected function setRobotsMetaTag()
    {
        $indexable = $this->isIndexable($this->currentUrl);
        $content = $indexable ? 'index, follow' : 'noindex, nofollow';
        $this->addMeta('name="robots" content="' . $content . '"');
    }

    /**
     * Get all defined routes.
     *
     * @return array An array containing all defined routes.
     */
    public function getAllRoutes(): array
    {
        $allRoutes = [];
        foreach ($this->routes as $method => $routes) {
            foreach ($routes as $url => $route) {
                $allRoutes[] = [
                    'method' => $method,
                    'url' => $url,
                    'name' => $route['name'],
                    'meta' => $this->meta(null, $url),
                ];
            }
        }
        return $allRoutes;
    }

    /**
     * Get the value of a route parameter.
     *
     * If $paramName is empty, returns an array of all current route parameters.
     * If $paramName is specified, returns the value of the corresponding route parameter.
     *
     * @param mixed $paramName The name of the route parameter to retrieve. (optional)
     * @return mixed|array|null The value of the specified route parameter, an array of all route parameters, or null if the parameter is not found.
     */
    public function getRouteParam($paramName = null)
    {
        if (is_null($paramName)) {
            // Return all current route parameters
            return $this->currentRouteParams;
        } else {
            // Return the value of the specified route parameter or null if not found
            return isset($this->currentRouteParams[$paramName]) ? $this->currentRouteParams[$paramName] : null;
        }
    }

    /**
     * Handle a route not found error.
     *
     * @param string $url The URL for the route
     * @param string $method The HTTP method (e.g., "GET" or "POST")
     */
    private function handleRouteNotFound(string $url, string $method)
    {
        self::setHeader("HTTP/1.1 404 Not Found");
        trigger_error("Route not found for URL: '$url' (Method: $method)", E_USER_ERROR);
    }

    /**
     * Set a URL to redirect to if the route is not defined.
     *
     * @param string $url The URL for the redirect
     */
    public function setErrorRedirect(string $url)
    {
        $this->redirectUrl = parent::assetUrl($url);
    }

    /**
     * Set a path to show an error page if the route is not defined.
     *
     * @param string $siteName The name of the desired file.
     */
    public function setErrorPage(string $siteName, array $variables = [], string $htmlTitle = "404 - Not Found")
    {
        if (file_exists(self::getRootPath($this->sitesPath . $siteName))) {
            $this->errorPage['page'] = $siteName;
            $this->errorPage['title'] = $htmlTitle;
            $this->errorPageVariables = $variables;
        } else {
            return trigger_error(htmlentities("Could not find the file $this->sitesPath" . "$siteName", E_USER_ERROR));
        }
    }

    /**
     * Add a global middleware for routes not in the specified array of URLs.
     *
     * @param array $urls An array of URLs for the redirect
     * @param Closure $gloMiddleware
     */
    public function addGlobalMiddleware(array $urls, \Closure $gloMiddleware)
    {
        $this->globalMiddleware['function'][] = $gloMiddleware;
        foreach ($urls as $url) {
            $this->globalMiddleware['url'][] = $url;
        }
    }

    /**
     * Add a middleware to be executed before or after a route callback.
     *
     * @param Closure $middleware The middleware function to be added.
     * @param string $position The position (before or after)
     * @return $this
     */
    public function addMiddleware(\Closure $middleware, string $position = "before")
    {
        $this->middlewares["function"][] = $middleware;
        $this->middlewares["route"][] = $this->currentRoute;
        $this->middlewares["position"][] = $position;
        return $this;
    }

    /**
     * Add a route to the application with multiple HTTP methods.
     *
     * @param array|string $methods The HTTP methods (e.g., ['GET', 'POST']) or a single method as a string.
     * @param string $url The URL for the route
     * @param Closure $target The callback function for the route
     * @param string $name The name for the route
     * @return object
     */
    public function addRoute($methods, string $url, \Closure $target, string $name = '')
    {
        $methods = (is_array($methods)) ? $methods : [$methods]; // Convert to an array if it's a single method
        $this->currentRoute = $url;

        foreach ($methods as $method) {
            $this->currentMethod = $method;
            $this->routes[$method][$url] = [
                'target' => $target,
                'name' => $name,
                'ajaxOnly' => false,
            ];

            if (!empty($name)) {
                $this->namedRoutes[$name] = $url;
            }
        }

        global $cml_namedRoutes;
        $cml_namedRoutes = $this->namedRoutes;

        return $this;
    }

    /**
     * Retrieves or sets the metadata for the current route.
     *
     * @param mixed $metadata The metadata to retrieve or set. If null, retrieves the entire metadata array.
     *                        If a string, retrieves the value of the specified metadata key.
     *                        If an array, sets the entire metadata array.
     * @return mixed The requested metadata or null if not found.
     */
    public function meta($metadata = null, string $routeUrl = '')
    {
        if (empty($routeUrl)) {
            $routeUrl = $this->currentRoute;
        }

        if (is_null($metadata)) {
            return $this->routeMetadata[$routeUrl] ?? null;
        }

        if (is_array($metadata) && !empty($metadata)) {
            return $this->routeMetadata[$routeUrl] = $metadata;
        }

        if (is_string($metadata) && !empty($metadata) && isset($this->routeMetadata[$routeUrl][$metadata])) {
            return $this->routeMetadata[$routeUrl][$metadata];
        }

        return null;
    }

    /**
     * Retrieves the metadata for all routes or a specific route URL.
     *
     * @param string|null $url The URL of the specific route to retrieve metadata for. If null, retrieves metadata for all routes.
     * @return null|array The metadata for the routes. The keys are the route URLs and the values are the corresponding metadata.
     */
    public function allMetas(string $url = null): ?array
    {
        $metadata = [];

        if ($url) {
            $metadata[$url] = $this->meta(null, $url);
        } else {
            $routes = $this->getAllRoutes();
            foreach ($routes as $data) {
                $metadata[$data["url"]] = $this->meta(null, $data["url"]);
            }
        }

        return $metadata;
    }

    /**
     * Get the URL for a named route with placeholder replacement.
     *
     * @param string $name The name of the route
     * @param array $parameters Associative array of parameter values to replace placeholders
     * @return string|null The URL for the named route with replaced placeholders or null if not found
     */
    public function getNamedRouteUrl(string $name, array $parameters = []): ?string
    {
        if (isset($this->namedRoutes[$name])) {
            $url = $this->namedRoutes[$name];

            // Replace placeholders in the URL with actual values from the parameters array
            foreach ($parameters as $paramName => $paramValue) {
                $url = str_replace(":$paramName", $paramValue, $url);
            }

            return $url;
        }
        return null;
    }

    /**
     * Sets a "where" condition for the current route.
     *
     * @param string $param The name of the parameter to which the condition applies.
     * @param string $condition The regular expression condition for the parameter.
     * @return $this The router instance for method chaining.
     */
    public function where(string $param, string $condition)
    {
        if (!empty($this->currentRoute)) {
            $this->routes[$this->currentMethod][$this->currentRoute]['where'] = [$param => $condition];
        }

        return $this;
    }

    /**
     * Sets a "where in" condition for the current route.
     *
     * @param string $param The parameter to apply the "where in" condition to.
     * @param array $in The array of values to check against for the "where in" condition.
     * @return $this The current instance of the Router class.
     */
    public function whereIn(string $param, array $in)
    {
        if (!empty($this->currentRoute)) {
            $this->routes[$this->currentMethod][$this->currentRoute]['whereIn'] = [$param => $in];
        }

        return $this;
    }


    /**
     * Add a route group to the application.
     *
     * The addGroup feature allows bundling of related routes under a common URL prefix for better structure and organization in routing.
     *
     * @param string $prefix The common initial part of the URLs for the bundled routes (e.g., "/admin").
     * @param Closure $callback A function that defines the specific routes for this group and adds them to the router.
     * @return object
     */
    public function addGroup(string $prefix, \Closure $callback)
    {
        // Backup current middlewares
        $originalMiddlewares = $this->middlewares;

        // Add the group prefix to the current route
        $this->currentRoute = $prefix . $this->currentRoute;

        // Execute the callback to define routes within the group
        $callback($this, $prefix);

        // Restore the original middlewares and apply them to all routes within the group
        $this->middlewares = $originalMiddlewares;

        return $this;
    }

    /**
     * Compares the requested method and URL to defined routes, processes them, and throws an exception if no match is found.
     */
    protected function _matchRoute()
    {

        # URL Validation
        $url = $_SERVER['REQUEST_URI']; // Get the URL from the current page
        if ($url != "/") {
            $url = rtrim($_SERVER['REQUEST_URI'], '/');
        } // Remove slash from the end
        $url = strtok($url, '?'); // Remove query parameters from the URL
        $url = str_replace(ltrim(rtrim(parent::assetUrl(""), "/"), "/"), "", $url); // Add the basename of the application folder
        if ($url == "/index.php") {
            $url = "/";
        } // Check if the URL is "index.php" and redirect to the root route
        $url = str_replace("//", "/", $url); // Remove double slashes from the URL
        $this->currentUrl = $url;
        $this->currentRouteName = array_flip($this->namedRoutes)[$this->currentUrl] ?? $this->currentUrl;

        // Clear current site cache and preload ist
        if ($this->cacheEnabled && isset($_GET[$this->cacheOptions['config']['clearCurrent']])) {
            $this->purgeCache($url);
        }

        // Clear all cache
        if ($this->cacheEnabled && isset($_GET[$this->cacheOptions['config']['clearAll']])) {
            $this->purgeAll();
        }

        // Check for cache and load it
        $this->checkCache($this->currentUrl);

        $method = $_SERVER['REQUEST_METHOD']; // Indicates the client's HTTP request method (e.g., GET, POST, etc.)

        $alias = $this->_findOriginalUrlForAlias($url);
        if ($alias !== null) {
            $url = $alias; // Use the original URL
        }

        // Process the given action based on the HTTP request method (e.g., GET, POST, etc.)
        if (isset($this->routes[$method])) {
            $process = $this->_processRoutes($url, $this->routes[$method]);
        }

        // Process wildcard routes for all methods
        if (isset($this->routes['*'])) {
            $process = $this->_processRoutes($url, $this->routes['*']);
        }

        // Redirect to the specified URL if the route is not found and a redirect URL is set
        if (!isset($this->routes[$method][$url]) || $process === false) {
            if (!empty($this->redirectUrl)) {
                self::setHeader("Location", $this->redirectUrl);
                die;
            } elseif (isset($this->errorPage['page'])) {
                $this->setTitle($this->errorPage['title']);
                ob_start();
                $this->getSite($this->errorPage['page'], $this->errorPageVariables);
                $this->buildHTML(ob_get_clean());
            } else {
                $this->handleRouteNotFound($url, $method);
            }
        }
    }

    /**
     * Loop through the routes to find a match for the given URL.
     *
     * @param string $url The route URL
     * @param array $routes Routes with the appropriate action
     */
    protected function _processRoutes(string $url, array $routes)
    {
        // Sort the routes by the number of parameters in descending order
        uksort($routes, fn ($key1, $key2) => (strpos($key1, ':') !== false ? 1 : 0) - (strpos($key2, ':') !== false ? 1 : 0));

        foreach ($routes as $routeUrl => $routeData) {
            // Check if the current route is restricted to AJAX requests
            if ($routeData['ajaxOnly'] && !$this->isAjaxRequest()) {
                http_response_code(403);
                continue; // Skip this route if it's not an AJAX request
            }

            // Convert route URL to a regular expression pattern for ':' parameters
            $pattern = preg_replace('/\/:([^\/]+)/', '/(?P<$1>[^/]+)', $routeUrl);

            // Check if the current URL matches the pattern
            if (preg_match('#^' . $pattern . '$#', $url, $matches)) {

                // Check "whereIn" conditions
                if (!empty($routeData['whereIn']) && $this->_checkWhereIn($matches, $routeData['whereIn'])) {
                    return false;
                }

                // Check "where" conditions
                if (!$this->_checkWhereConditions($matches, $routeData['where'] ?? array())) {
                    return false;
                }

                ob_start();
                // Execute global middleware function
                if (!empty($this->globalMiddleware) && !in_array($url, $this->globalMiddleware["url"])) {
                    call_user_func($this->globalMiddleware["function"][0]);
                }

                // Execute middleware functions before the target function
                $this->_executeMiddleware('before', $url);

                // Call the target function with the extracted parameter values
                $parameterValues = $this->_sanitizeStringsArray(array_filter($matches, 'is_string', ARRAY_FILTER_USE_KEY));
                $this->currentRouteParams = $parameterValues;
                call_user_func_array($routeData['target'], $parameterValues);

                // Execute middleware functions after the target function
                $this->_executeMiddleware('after', $url);

                // Close the application
                (!$this->isApi && !$routeData['ajaxOnly']) ? $this->buildHTML(ob_get_clean()) : exit;
            }
        }
    }

    /**
     * Check if parameter values meet the specified "where" conditions.
     *
     * @param array $parameterValues An array of parameter values to be checked.
     * @param array $whereConditions An array of "where" conditions for parameter validation.
     *
     * @return bool True if all conditions are met, false otherwise.
     */
    protected function _checkWhereConditions(array $parameterValues, array $whereConditions): bool
    {
        foreach ($whereConditions as $paramName => $condition) {
            if (isset($parameterValues[$paramName]) && !preg_match($condition, $parameterValues[$paramName])) {
                return false;
            }
        }
        return true;
    }

    /**
     * Checks if the parameter values satisfy the given where conditions.
     *
     * @param array $parameterValues The parameter values to check.
     * @param array $whereConditions The where conditions to satisfy.
     * @return bool Returns true if the parameter values satisfy the where conditions, false otherwise.
     */
    protected function _checkWhereIn(array $parameterValues, array $whereConditions): bool
    {
        foreach ($whereConditions as $paramName => $conditions) {
            if (isset($parameterValues[$paramName]) && in_array($parameterValues[$paramName], $conditions)) {
                return false;
            }
        }
        return true;
    }


    /**
     * Execute middleware functions based on the specified position (before or after).
     *
     * @param string $position The position of the middleware (before or after)
     * @param string $url The URL for which the middleware should be executed
     */
    protected function _executeMiddleware(string $position, string $url)
    {
        if (!empty($this->middlewares)) {
            $mdPosition = array_search($url, $this->middlewares["route"]);
            if (is_int($mdPosition) && $this->middlewares['position'][$mdPosition] === $position) {
                call_user_func($this->middlewares["function"][$mdPosition]);
            }
        }
    }


    /**
     * Processes a file by including it and capturing the output.
     *
     * @param string $sitePath The path to the file to be processed.
     * @param array $variables An optional array of variables to be extracted and made available to the included file.
     * @return string The captured output of the included file.
     */
    protected function _processFile(string $sitePath, array $variables = []): string
    {
        if (!file_exists($sitePath)) {
            trigger_error(htmlentities("'$sitePath' | Site not found"), E_USER_ERROR);
            return '';
        }

        extract($variables);
        ob_start();
        require $sitePath;
        return ob_get_clean();
    }

    /**
     * Loads and displays a file with PHP components embedded in HTML tags.
     *
     * @param string $siteName The name of the view to render.
     * @param array $variables An optional array of variables to pass to the view.
     * @return void
     */
    public function view(string $siteName, array $variables = [])
    {
        $sitePath = self::getRootPath($this->sitesPath . $siteName);
        $content = $this->_processFile($sitePath, $variables);

        $content = preg_replace_callback('/<(\w+)([^>]*)>([\s\S]*?)<\/\1>|<(\w+)([^>]*)>/', function ($matches) {
            $tag = $matches[4] ?? $matches[1];
            $attributes = $matches[5] ?? $matches[2];
            $slot = $matches[3] ?? null;

            $componentName = $tag . '.cml.php';
            $componentPath = self::getRootPath(cml_config('COMPONENTS_PATH') . $componentName);

            if (file_exists($componentPath) && preg_match('~^\p{Lu}~u', $tag)) {
                $attributeValues = [];
                preg_match_all('/(\w+)(?:="([^"]*)")?/', $attributes, $attributeMatches);
                foreach ($attributeMatches[1] as $index => $attributeName) {
                    $attributeValues[$attributeName] = $attributeMatches[2][$index] ?? null;
                }

                foreach ($attributeValues as $attributeName => $attributeValue) {
                    $$attributeName = $attributeValue;
                }

                if (!empty($slot)) {
                    $slot = trim($slot);
                }

                ob_start();
                require $componentPath;
                return ob_get_clean();
            } else {
                return $matches[0];
            }
        }, $content);

        echo $content;
    }

    /**
     * Renders the specified view with optional variables.
     *
     * @param string $siteName The name of the desired file.
     * @param array $variables An associative array of variables to be made available in the loaded file.
     */
    public function getSite(string $siteName, array $variables = [])
    {
        $sitePath = self::getRootPath($this->sitesPath . $siteName);
        $content = $this->_processFile($sitePath, $variables);

        echo $content;
    }



    /**
     * Set the route to be accessible only via AJAX requests.
     *
     * @return $this
     */
    public function onlyAjax()
    {
        if (!empty($this->currentRoute)) {
            $this->routes[$this->currentMethod][$this->currentRoute]['ajaxOnly'] = true;
        }

        return $this;
    }

    /**
     * Add an alias for the current route.
     *
     * @param string $alias The alias URL
     * @return Router This router instance
     */
    public function setAlias(string $alias)
    {
        if (!empty($this->currentRoute)) {
            $this->aliases[$alias] = $this->currentRoute;
        }

        return $this;
    }

    /**
     * Find the original URL for a given alias.
     *
     * @param string $alias The alias URL
     * @return string|null The original URL if found, null otherwise
     */
    protected function _findOriginalUrlForAlias(string $alias)
    {
        return $this->aliases[$alias] ?? null;
    }

    /**
     * Sanitize an array of strings using various filters.
     *
     * @param array $inputArray The array of strings to sanitize.
     * @return array The sanitized array.
     */
    protected function _sanitizeStringsArray(array $inputArray): array
    {
        $sanitizedArray = [];

        foreach ($inputArray as $k => $input) {
            // Remove HTML tags
            $sanitized = strip_tags($input);

            // Convert illegal characters
            $sanitized = htmlspecialchars($input, ENT_QUOTES, 'UTF-8');

            // Allow only certain characters (example: letters, numbers, spaces)
            $sanitized = preg_replace('/[^a-zA-Z0-9\s]/', '', $sanitized);

            // Replace unauthorized SQL keywords
            $sql_keywords = ["SELECT", "INSERT", "UPDATE", "DELETE", "DROP", "TABLE", "UNION"];
            $sanitized = str_ireplace($sql_keywords, "", $sanitized);

            // Add more customizations or filters as needed.

            $sanitizedArray[$k] = $sanitized;
        }

        return $sanitizedArray;
    }
}
