# Adopted from JOS/xv6

#include <include/mmu.h>
#include <kernel/version.h>

# Start the CPU: switch to 32-bit protected mode, jump into C.
# The BIOS loads this code from the first sector of the hard disk into
# memory at physical address 0x7c00 and starts executing in real mode
# with %cs=0 %ip=7c00.

.set PROT_MODE_CSEG, 0x8         # kernel code segment selector
.set PROT_MODE_DSEG, 0x10        # kernel data segment selector
.set CR0_PE_ON,      0x1         # protected mode enable flag

.globl start
start:
  .code16                     # Assemble for 16-bit mode

  cli                         # Disable interrupts
  cld                         # String operations increment

  # Set up the important data segment registers (DS, ES, SS).
  xorw    %ax,%ax             # Segment number zero
  movw    %ax,%ds             # -> Data Segment
  movw    %ax,%es             # -> Extra Segment
  movw    %ax,%ss             # -> Stack Segment

  # Hello message
  mov     $msg, %si           # loads the address of msg into si
  mov     $0xe, %ah           # loads 0xe (function number for int 0x10) into ah
printChar:
  lodsb                       # loads the byte from the address in si into al and increments si
  cmp     $0, %al             # compares content in AL with zero
  je      done                # if al == 0, go to "done"
  int     $0x10               # prints the character in al to screen
  jmp     printChar

done:
  # Enable A20:
  #   For backwards compatibility with the earliest PCs, physical
  #   address line 20 is tied low, so that addresses higher than
  #   1MB wrap around to zero by default.  This code undoes this.
  # BUGENC: a simpler method to enable A20
  inb	  $0x92, %al
  orb	  $0x02, %al
  out	  %al, $0x92

  # Switch from real to protected mode, using a bootstrap GDT
  # and segment translation that makes virtual addresses
  # identical to their physical addresses, so that the
  # effective memory map does not change during the switch.
  lgdt    gdtdesc
  movl    %cr0, %eax
  orl     $CR0_PE_ON, %eax
  movl    %eax, %cr0

  # Jump to next instruction, but in 32-bit code segment.
  # Switches processor into 32-bit mode.
  # BUGENC: set %cs
  ljmp    $PROT_MODE_CSEG, $protcseg

  .code32                     # Assemble for 32-bit mode
protcseg:
  # Set up the protected-mode data segment registers
  movw    $PROT_MODE_DSEG, %ax    # Our data segment selector
  movw    %ax, %ds                # -> DS: Data Segment
  movw    %ax, %es                # -> ES: Extra Segment
  movw    %ax, %fs                # -> FS
  movw    %ax, %gs                # -> GS
  movw    %ax, %ss                # -> SS: Stack Segment

testa20:
  # Test if high memory wrap now
  pushal
  movl    $0x112345, %edi         # Odd MB
  movl    $0x012345, %esi         # Even MB
  movl    %esi, (%esi)
  movl    %edi, (%edi)
  cmpsd
  popal
  jne      a20on

a20fail:
  movl    $0x0c41, %eax           # Red 'A'
  movl    $0xb809c, %ebx
  movl    %eax, (%ebx)
  jmp     spin

a20on:
  # A20 is on, print a green 'A' on the screen
  movl    $0x0a41, %eax
  movl    $0xb809c, %ebx
  movl    %eax, (%ebx)
  # Set up the stack pointer and call into C.
  movl    $start, %esp
  call    bootmain

  # If bootmain returns (it shouldn't), loop.
spin:
  jmp     spin

msg:
.string LITCHI_LDR_INFO

# Bootstrap GDT
.p2align 2                                # force 4 byte alignment
gdt:
  SEG_NULL				# null seg
  SEG(STA_X|STA_R, 0x0, 0xffffffff)	# code seg
  SEG(STA_W, 0x0, 0xffffffff)	        # data seg

gdtdesc:
  .word   0x17                            # sizeof(gdt) - 1
  .long   gdt                             # address gdt
