/**
 * @copyright
 * Copyright (C) 2020 Assured Information Security, Inc.
 *
 * @copyright
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * @copyright
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * @copyright
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

#include <debug.h>
#include <mutable_span_t.h>
#include <platform.h>
#include <types.h>

/**
 * <!-- description -->
 *   @brief This function gets the addr of the microkernel's page pool
 *
 * <!-- inputs/outputs -->
 *   @param page_pool a pointer to a mutable_span_t that stores the page pool
 *   @param base_virt provide the base virtual address that the page pool
 *     was mapped to.
 *   @param pmut_addr where to store the resulting addr of the page pool
 *   @return LOADER_SUCCESS on success, LOADER_FAILURE on failure.
 */
NODISCARD int64_t
get_mk_page_pool_addr(
    struct mutable_span_t const *const page_pool,
    uint64_t const base_virt,
    uint8_t **const pmut_addr) NOEXCEPT
{
    uint64_t const phys = platform_virt_to_phys(page_pool->addr);
    if (((uint64_t)0) == phys) {
        bferror("platform_virt_to_phys failed");
        return LOADER_FAILURE;
    }

    *pmut_addr = ((uint8_t *)(base_virt + phys));
    return LOADER_SUCCESS;
}
