/// @copyright
/// Copyright (C) 2020 Assured Information Security, Inc.
///
/// @copyright
/// Permission is hereby granted, free of charge, to any person obtaining a copy
/// of this software and associated documentation files (the "Software"), to deal
/// in the Software without restriction, including without limitation the rights
/// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
/// copies of the Software, and to permit persons to whom the Software is
/// furnished to do so, subject to the following conditions:
///
/// @copyright
/// The above copyright notice and this permission notice shall be included in
/// all copies or substantial portions of the Software.
///
/// @copyright
/// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
/// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
/// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
/// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
/// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
/// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
/// SOFTWARE.

#include "../../../mocks/ext_t.hpp"

#include <huge_pool_t.hpp>
#include <intrinsic_t.hpp>
#include <page_pool_t.hpp>
#include <tls_t.hpp>

#include <bsl/discard.hpp>
#include <bsl/ut.hpp>
namespace
{
    /// @brief verify constinit it supported
    constinit mk::ext_t const g_verify_constinit{};
}

/// <!-- description -->
///   @brief Main function for this unit test. If a call to bsl::ut_check() fails
///     the application will fast fail. If all calls to bsl::ut_check() pass, this
///     function will successfully return with bsl::exit_success.
///
/// <!-- inputs/outputs -->
///   @return Always returns bsl::exit_success.
///
[[nodiscard]] auto
main() noexcept -> bsl::exit_code
{
    bsl::ut_scenario{"verify supports constinit/constexpr"} = []() noexcept {
        bsl::discard(g_verify_constinit);
    };

    bsl::ut_scenario{"verify noexcept"} = []() noexcept {
        bsl::ut_given{} = []() noexcept {
            mk::ext_t mut_ext{};
            mk::ext_t const ext{};
            mk::tls_t mut_tls{};
            mk::page_pool_t mut_page_pool{};
            mk::huge_pool_t mut_huge_pool{};
            mk::intrinsic_t mut_intrinsic{};
            bsl::ut_then{} = []() noexcept {
                static_assert(noexcept(mk::ext_t{}));

                static_assert(noexcept(mut_ext.initialize(mut_tls, mut_page_pool, {}, {}, {})));
                static_assert(noexcept(mut_ext.release(mut_tls, mut_page_pool, mut_huge_pool)));
                static_assert(noexcept(mut_ext.id()));
                static_assert(noexcept(mut_ext.bootstrap_ip()));
                static_assert(noexcept(mut_ext.set_bootstrap_ip({})));
                static_assert(noexcept(mut_ext.vmexit_ip()));
                static_assert(noexcept(mut_ext.set_vmexit_ip({})));
                static_assert(noexcept(mut_ext.fail_ip()));
                static_assert(noexcept(mut_ext.set_fail_ip({})));
                static_assert(noexcept(mut_ext.open_handle()));
                static_assert(noexcept(mut_ext.close_handle()));
                static_assert(noexcept(mut_ext.is_handle_valid({})));
                static_assert(noexcept(mut_ext.handle()));
                static_assert(noexcept(mut_ext.is_started()));
                static_assert(noexcept(mut_ext.is_executing_fail()));
                static_assert(noexcept(mut_ext.alloc_page(mut_tls, mut_page_pool)));
                static_assert(
                    noexcept(mut_ext.alloc_huge(mut_tls, mut_page_pool, mut_huge_pool, {})));
                static_assert(noexcept(mut_ext.map_page_direct(mut_tls, mut_page_pool, {}, {})));
                static_assert(
                    noexcept(mut_ext.unmap_page_direct(mut_tls, mut_page_pool, {}, {}, {})));
                static_assert(noexcept(mut_ext.signal_vm_created(mut_tls, mut_page_pool, {})));
                static_assert(noexcept(mut_ext.signal_vm_destroyed(mut_tls, mut_page_pool, {})));
                static_assert(noexcept(mut_ext.signal_vm_active(mut_tls, mut_intrinsic, {})));
                static_assert(noexcept(mut_ext.start(mut_tls, mut_intrinsic)));
                static_assert(noexcept(mut_ext.bootstrap(mut_tls, mut_intrinsic)));
                static_assert(noexcept(mut_ext.vmexit(mut_tls, mut_intrinsic, {})));
                static_assert(noexcept(mut_ext.fail(mut_tls, mut_intrinsic, {}, {})));
                static_assert(noexcept(mut_ext.dump({})));

                static_assert(noexcept(ext.id()));
                static_assert(noexcept(ext.bootstrap_ip()));
                static_assert(noexcept(ext.vmexit_ip()));
                static_assert(noexcept(ext.fail_ip()));
                static_assert(noexcept(ext.is_handle_valid({})));
                static_assert(noexcept(ext.handle()));
                static_assert(noexcept(ext.is_started()));
                static_assert(noexcept(ext.is_executing_fail()));
                static_assert(noexcept(ext.dump({})));
            };
        };
    };

    return bsl::ut_success();
}
