<?php

namespace Aza\Components\PhpGen\Tests;
use Aza\Components\Benchmark;
use Aza\Components\Common\CArray;
use Aza\Components\Common\Date;
use Aza\Components\PhpGen\PhpGen;
use PHPUnit_Framework_TestCase as TestCase;
use SplFixedArray;

/**
 * PhpGen benchmarks
 *
 * @project Anizoptera CMF
 * @package system.phpgen
 * @author  Amal Samally <amal.samally at gmail.com>
 * @license MIT
 *
 * @group benchmark
 * @coversNothing
 */
class PhpGenBenchmarkTest extends TestCase
{
	/**
	 * Check strings variants
	 *
	 * @author amal
	 */
	public function testStrings()
	{
		/**
		 * Escaped string vs. raw
		 *
		 * Escaped strings can be insignificantly faster than raw (<1%)
		 */
		$iteratons = 100000;
		$tests     = 20;

		$res = array();
		for ($j = 0; $j < $tests; $j++) {
			$start = microtime(true);
			for ($i = 0; $i < $iteratons; $i++) {
				$v = "\n\t1\r\0\1\2\3\4\5\6\7'\"\$v";
			}
			$res['escaped'][] = Date::timeEnd($start);

			$start = microtime(true);
			for ($i = 0; $i < $iteratons; $i++) {
				$v = "
	1
 '\"\$v";
			}
			$res['raw'][] = Date::timeEnd($start);
		}
		$results = Benchmark::analyzeResults($res);

		print_r($results);
	}


	/**
	 * Check strings variants
	 *
	 * @author amal
	 */
	public function testBinaryStringsRecognition()
	{
		$binaryStrings = array(
			"\0",
			"\1\2\3\4\5\6\7",
			"some\0example",
			"some\1exa\3mple\5",
			"\xB6\xBA\x06\xDC\xD3\x18\x14\xC5\x19\xD0\xB1\xC1\x64\x9E\x25\x82",
			"\x48\x6D\x08\x05\xD0\xA2\xE4\xA1\x28\xB4\xA3\x67\x24\x67\xE4\xD2",
			"\x46\x58\xD0\xF7\xF3\xD3\x24\x98\x1A\xBC\xA7\xB5\xD9\x74\x8E\x3A",
			"\x9A\x61\x75\xD6\x0E\x69\x34\x90\xD8\xEC\xFA\x20\xBF\xA4\x8F\x4A",
			"\r\x26\xA8\x49\x7F\x6E\x34\xDD\x48\x39\xF6\xCD\xA2\x18\x6E\x57",
			"\xB8\xF5\xE7\x3C\x92\xD5\x04\x74\xBA\x28\x69\xEF\x53\x33\x65\xDA",
			"\x14\x2E\xA2\x2C\x76\xFD\x64\x43\x3B\x2F\x9F\xE2\x59\x8E\x86\x1E",
			"\x3A\xFD\x6D\xDB\x10\x3A\x64\x56\xEB\x21\x3B\xEF\xB7\xBA\xB7\xA5",
			"\xF0\x9D\x9F\x4F\x5D\x20\xB4\x1E\xEA\x56\xCC\x1A\xF7\x21\x2A\xE6",
			"\x8C\x80\x52\xDC\x60\x01\x14\x4D\x8B\xB8\xF8\x86\x97\x8B\xEA\x56",
			"\x98\x58\x49\x22\x1C\x68\xC4\xD9\t\x2E\xDF\x31\x13\x18\xAC\x66",
			"\xE6\xF3\xD7\x30\xE4\x71\xA4\xF5\x6B\x54\x9A\x7F\x18\x08\x1A\xDD",
			"\x5A\x4B\x17\xE9\xF9\xAB\xD4\x08\xAB\x76\x6B\x7D\x57\xD6\x55\x9F",
			"\x86\x47\xCF\x42\x30\x8E\x04\x31\x8A\xD8\x9D\x9D\xBD\xE2\x1B\f",
			"\x92\x7B\x97\xF6\xB2\x11\x54\x93\x4B\x3D\xC9\x90\xDC\x03\x96\x05",
			"\x86\xB9\x2F\v\x9C\x6E\xE4\xCF\x2B\x63\xFA\x11\x5C\xE8\xD5\x6C",
			"\xAF\x47\xB5\x33\x0E\x2C\x64\xBB\x1B\f\xBE\xCF\x0F\x66\x5A\x5C",
			"\xC1\xE0\x90\xB9\x58\x3E\x34\x92\xAB\x33\x3E\xC0\xD5\xC5\xA0\x00",
			"\x85\xA7\xEC\x06\x71\x7C\x94\x14\x68\x97\xB2\x6F\xAD\xAA\x41\xF5",
			"\t\xFB\x2C\xA7\xEF\xFA\x74\x40\xBB\xF5\xF8\x7F\x3F\x8F\xB4\xA5",
			"\xB6\x47\x4A\x2B\x9C\r\xB4\x0F\x9B\xEB\xEF\xD0\x75\x8C\x6E\x6F",
			"\x38\xCA\xC3\xE3\xA5\xC2\xD4\x3B\x5B\x7F\xD0\x29\x19\x6E\x77\xE3",
			"\x16\x3B\x5A\x9F\x2F\xB8\x04\x82\xC8\x48\xB9\xB7\xC3\x07\xB5\x67",
			"\x8B\xCB\x88\x54\xA5\xB8\x14\x7C\xF8\x6B\xDA\x13\xB2\x3A\xED\x43",
			"\xC8\x10\x81\x70\x8F\xCB\x64\x27\xD8\x43\x73\x28\x05\x07\xA9\xAA",
			"\x3E\xBE\x45\x2A\xC1\xA8\xA4\xB3\x4A\xB1\x2A\x3B\x1A\x69\x2E\x55",
			"\x3C\x5D\xD3\x1A\xBB\x27\xE4\xB7\xCA\x83\x71\x95\x38\x96\xEE\x65",
			"\xEF\x13\xBC\xCB\x6E\xF5\x14\x5E\x79\xE1\x29\xFB\xFD\xE8\x7D\xAB",
			"\x60\xF8\x16\x93\x1D\xB9\xE4\xAB\xA9\x95\xF5\x19\x46\x50\xEB\xAB",
			"\x56\x16\xF4\v\x93\x7B\xA4\xDD\x7B\x78\x86\x59\xE1\x67\x93\xC8",
			"\xBA\x6E\x39\x14\x55\x55\x84\x9C\x9A\x1A\xDB\x10\x75\x5D\x32\x17",
			"\xCC\x70\x33\x35\n\xFF\xD4\xBD\x58\x1B\xEE\xA8\x76\x75\xBE\xF3",
			"\xB1\x77\x27\xE3\xCC\x60\xD4\x98\x5B\xB5\x3D\x43\x24\xD0\x54\x3E",
			"\xB5\xB9\x1C\x96\x1E\x96\x24\x2E\x59\xBD\xF4\x66\x67\xBB\x9E\xDC",
			"\xED\x49\xDF\x07\xC8\x47\xF4\xB6\x88\x29\x5E\x5E\xAE\x7A\xF7\xBE",
			"\xF6\x76\x44\xA7\x20\x60\xD4\x2A\x49\x9E\x9F\f\x70\x11\x78\x77",
			"\x21\xDD\xFE\xDF\xC8\x36\xD4\x4E\x5A\x50\x28\xE1\x3F\x84\x2C\x10",
			"\x56\x32\xAE\xE1\x6F\xA7\x74\x11\x5A\xB8\xF1\x3A\x3C\xFD\x4B\xDC",
			"\x53\x90\x36\xD6\x50\x14\x54\x52\x08\xC2\r\x49\xE2\xF2\x9B\xC7",
			"\x9B\xC0\x54\xD2\xFB\xCB\x64\x86\xAB\x66\xC4\x77\x80\x5F\x35\xB7",
			"\v\x9F\xEF\xC0\x5B\x5D\xC4\x21\xD9\x10\xB7\xB4\xE7\x7F\xD8\xE3",
			"\x6F\xF6\xEC\x8E\x80\x52\x94\xF4\x99\x95\xDC\x3A\xFF\x19\x95\x94",
			"\f\x98\x7A\x48\xDF\x07\x74\x0E\xDA\xED\xB1\x10\x87\xD0\xD5\x17",
			"\x20\x3A\x72\x53\x19\x88\xF4\x9F\x6A\xB6\xCB\x64\x14\xB9\x47\x2F",
			"\xBB\x87\xCD\x47\x31\x8B\x14\x28\x7A\x0E\x9B\xB9\v\x2F\xB9\xA1",
			"\xB1\x18\x1E\x04\x44\v\xB4\x28\x3A\xF7\x2E\xC5\xDA\x40\x29\x0E",
			"\xF1\xA0\x5A\xA9\x06\x66\x14\x64\x1A\xDA\x29\xF5\x10\xAA\x66\xD6",
			"\x98\xF3\x36\xC3\x40\x5A\x74\x20\xD8\x76\x6F\f\xCF\xEA\x90\xDE",
			"\x34\xCB\x76\x3D\xE7\x60\x94\x34\x4A\x1F\x29\xBB\x27\x47\xCB\x53",
			"\xAD\x87\x06\x84\xF8\xA7\x94\x87\x88\xBC\xB9\x91\x3C\xD1\x75\xA5",
			"\xA6\x0F\xC2\xAF\x4E\x7A\xD4\xB5\xC8\xF8\x8F\x12\xB4\x1F\xA0\x7D",
			"\xD6\x1B\x47\x02\x64\x29\xA4\xFE\xFA\xBA\xFA\xE8\xB5\x16\xD6\xF2",
			"\xB2\xFF\x02\x44\xF8\xCE\x44\x04\n\x52\x4E\x1D\x78\xC5\x2B\xAB",
			"\xF2\x3C\x56\x6E\x95\x54\x34\x16\x9A\xC0\x50\v\x71\x58\x29\r",
			"\xA3\x81\xAE\x05\x5F\x38\xF4\x1F\x5A\xAE\x50\x87\x62\x45\xD9\xC9",
			"\xB8\x8C\x42\x6D\xD0\x55\xF4\x1B\x39\xDC\xA5\x85\x40\x7C\xFD\xC4",
			"\r\xCB\x07\x82\x32\x84\xC4\x08\x6B\x19\v\xBD\xF8\x02\x86\x94",
			"\x9C\x85\x11\x07\x9D\xE4\xB4\x39\n\x92\xE6\x2E\x98\x5A\xF1\x1D",
			"\xBF\x4B\x5D\x03\xBE\xAE\x64\x48\xE9\x20\x80\xFE\x7F\x22\x8C\xAF",
			"\xA5\xE3\x4E\x61\x7B\x78\x94\xA7\x8B\xE6\x6F\x36\xA3\xE9\x7C\xF4",
			"\x56\x5E\x4C\x80\x35\x59\x44\xC8\xBA\x75\n\x87\x45\x72\xF1\x95",
			"\x67\x13\xE0\xEE\xD3\x78\xC4\x68\xBB\x6E\xC3\xA7\xE5\x66\xE5\xE1",
			"\x28\x53\xCD\x44\xA3\x43\x44\xA2\xDA\xAD\xA2\xEC\xBD\x20\xC2\x13",
			"\xEC\x68\x2B\x4B\xB7\x34\xE4\x68\xC9\x23\x23\xCC\x47\xBC\xEA\xE3",
			"\xC4\x9C\xD8\x2F\xDA\xE0\x14\xE4\x1A\xDF\xA1\x7C\x3B\x41\x69\x22",
			"\xC9\xC8\x3F\xC7\x40\x37\x64\x01\xDB\xE9\x93\xE9\xB8\x19\x69\x83",
			"\xC9\xA8\xBE\x07\xBD\x90\x04\xD2\x38\x30\x54\xF1\x2E\x5F\xE3\r",
			"\xF9\xCA\x1C\xAC\x29\x6C\x54\x84\x88\x9C\x9D\x8F\x10\xD4\x9F\x73",
			"\xB9\xBF\x2C\x24\xD7\x82\x64\x08\x8B\x25\x57\x8D\xF5\x94\xF6\x9A",
			"\xAB\xF8\x4B\x89\x1D\x77\x24\x2D\xA9\xDF\x71\xAC\x48\x60\xA4\x21",
			"\xE6\xE7\x91\x62\xA5\xDD\xE4\x21\x2B\xC1\x24\xDE\x32\xA0\x10\xFA",
			"\x59\x8C\x5A\xCA\xD7\xE1\xF4\xAD\x5A\x9E\x29\x59\x32\x84\x94\x65",
			"\xF4\x01\x78\x89\xCD\x50\x54\xA4\xB8\xB4\xC1\x59\x18\xF3\x52\x0E",
			"\x33\x44\x57\xC2\x5D\x41\xB4\x4E\x5A\x97\x37\x32\x47\x64\x83\x75",
			"\x57\x37\x3A\xA8\xED\xEB\x44\xA3\xB8\xFC\x86\xC7\x16\xB5\xF1\xAC",
			"\xB2\xC8\xD8\xAB\x3C\x5F\xB4\x26\x18\x99\xDA\xC7\x51\xE5\x6D\x40",
			"\x6B\x26\x87\x81\x51\x82\xD4\x97\x79\xA3\x98\x1F\x1D\x6A\x43\x51",
			"\x0E\x1A\xAC\xAA\xD8\xF8\x94\x2D\x08\xFF\xC6\x1D\x35\x73\xE8\x89",
			"\xB3\x50\x62\xEF\x42\xCE\xE4\x45\x5B\xC2\x34\xBC\xF3\xCB\xDF\x85",
			"\x4D\x12\x8E\xA2\xE0\x5C\x34\x90\xD9\x57\x53\x7F\xAE\x77\x31\x5B",
			"\x27\x00\xEA\x24\xF3\x8C\x44\xAC\x68\xA3\xCF\x2F\x40\x20\x3C\x27",
			"\xB8\x34\x93\x05\x96\xDF\x54\x7B\x59\x40\x75\xED\x8C\x65\x14\x45",
			"\x03\x55\xDA\xE0\x9E\xDE\x04\x7A\xBB\x9B\x56\xBD\xBB\x67\x68\x85",
			"\x7F\xE3\x7E\xA0\x18\xFE\x24\x68\x5A\xE4\f\x7A\x6C\x43\xC6\x28",
			"\x7B\x53\xE4\xA5\x75\xEB\xC4\xBF\xCA\xC8\xB1\xB8\x45\n\x68\xEA",
			"\xF8\x91\xF6\x65\x1E\xD7\xF4\x8A\x5A\x8D\x5A\x17\x91\x1F\x97\x91",
			"\x37\x5C\x20\x00\xE1\xB4\x54\n\x7B\x15\x2C\xC8\x9C\x1B\xEB\x10",
			"\x9E\x65\xD6\x2F\xB9\x27\x24\x42\x5A\x7D\x97\xF7\xF1\xC6\x5E\x11",
			"\x63\x28\x13\x64\xB0\xE8\x54\xCA\xCA\xCF\xB2\x7C\xD8\x3D\xC6\xB6",
			"\xA5\x4B\x40\xE4\xFF\x7E\x64\x6A\x6A\x15\x3F\x02\x49\x74\x6A\xB8",
			"\xF6\x76\xB7\xEF\xB7\x9F\x74\x9B\x1B\x57\x64\x10\x4A\x31\xB9\x3E",
			"\x46\x63\x76\x66\x7D\xA8\xF4\x00\x48\xF2\x5A\x5A\xBC\xFB\xA2\x62",
			"\x84\x68\xEE\xA6\x36\xD4\xE4\xAF\x99\x19\x58\x15\x43\x44\xFD\x79",
			"\xB4\xD1\f\x25\x53\xEA\xF4\x21\x59\xFC\x50\xE1\x4C\xC2\xD7\x66",
			"\x44\f\xF7\x2C\x7F\x54\xA4\x3F\xFA\xC1\x15\xBD\x28\x90\x3C\x97",
			"\x52\x52\x28\x89\x4B\xBE\xA4\xD5\t\x72\x44\x4D\x6C\x4F\xAC\x65",
			"\x2C\x05\x2E\x41\x38\xB2\x44\xBE\xDA\x6D\x12\v\x74\xBB\x3C\xC6",
			"\xE5\xF7\xAB\x86\xD4\x49\x24\x81\x2B\x22\x8B\xB1\xC6\xF2\x7C\xCD",
			"\xCF\x36\x3F\x62\x6E\xE1\xF4\x8B\xAA\x38\x75\x96\x46\x4D\xAC\r",
			"\x05\xA8\x21\x67\x98\xDC\x64\xE2\xA8\x57\xE1\xF1\x47\x7E\x61\xD3",
			"\x2E\x63\x93\xD9\x3F\xD9\x14\x29\xDA\xAB\x9E\xAA\x22\x74\x92\x65",
			"\xF8\x07\x40\xE5\x80\x28\x94\x64\x79\xF5\x64\xE5\xB4\x3F\x33\x8C",
			"\x36\xA7\x38\xA6\xC9\x80\x44\x27\xC8\xA5\xAE\t\x18\x87\x97\x89",
			"\xEF\xAF\xFB\x8D\xE2\xEF\xB4\x2A\x9B\xED\x37\x1E\x17\x24\x53\xDB",
			"\x4D\x92\x0E\x96\xB5\x59\x74\x2A\x08\x00\xD6\x02\xAB\x22\xE3\x17",
			"\x20\x0F\v\xB7\x4C\x49\x54\x9A\x1A\xF7\xA9\xAE\x8E\x21\x82\n",
			"\xA9\x7B\x5D\x33\x86\xA9\x44\xB6\xB9\x93\xE0\x38\xCD\xE9\xB5\x5A",
			"\xAF\xA0\x50\x4B\xA1\x88\x94\x05\x4A\xC1\xF7\xA7\x23\xB0\x62\x95",
			"\x3F\x64\x5B\r\x70\x6C\x24\x38\x1B\xE8\x78\xBD\x25\xCC\x48\x79",
			"\xEA\xC9\xF3\x8B\x6D\x21\xF4\x01\x58\xB5\x34\xD2\xA8\xB6\x51\xA3",
			"\x50\x6C\xAF\x7D\xA8\xE2\xB4\x3F\x9A\xF4\xE4\x76\xC5\t\x98\x51",
			"\x3D\x53\x1E\x3E\xBA\xC1\xA4\x93\x49\xFF\x06\xFA\xB8\x60\x3F\x33",
			"\x56\x66\x27\x75\x19\xB2\x04\xE4\x7A\xF8\x8D\x15\x23\x7A\xFC\xDD",
			"\x20\xBF\x26\x6B\xE1\x76\x64\xF3\x9B\x35\r\x07\xE1\x2D\xCA\x5B",
			"\xA4\xE3\xEE\x10\x5A\x69\x74\x9A\x59\xF6\x90\xC1\xDC\x36\x97\xB5",
			"\xCF\xA1\xB1\x83\x59\x8D\x04\xB7\xD8\x83\xE0\xC2\xF4\x04\x8A\xFC",
			"\x6E\xBB\xCB\x36\x72\x41\xD4\xF2\x3B\xBE\x8E\x1B\x05\x92\xC6\xB0",
			"\x88\x89\x32\xDB\x3C\x77\x54\xFA\xEB\x2E\x14\x99\xC6\x06\x33\x89",
			"\xA8\x07\xD8\x81\x9A\x0F\x94\x7A\x79\xAE\x35\x8A\x28\xC4\xF0\x5B",
			"\xA1\x86\x8F\xE2\xB2\x70\x64\x83\n\x2B\x00\xC2\xA4\x43\x1B\x13",
			"\xFC\x91\x2B\x38\x4D\xD2\xB4\xE1\x58\xD7\x07\x24\xB8\x6F\x70\x13",
			"\xCB\xA2\x3D\x6F\x1D\xFD\x84\x2A\xE9\x49\x7F\xF5\x5C\f\xA8\xF3",
			"\xB7\xC4\xB9\x8A\x3C\xDF\x84\x37\xBA\x65\xC3\x13\xE1\x5E\x06\x7E",
			"\x14\x62\x6A\x05\x59\xE9\xB4\xF3\xB8\x36\xFE\xE2\x80\x29\x19\x04",
			"\xBD\x48\x50\x83\x72\x30\x94\xFB\xA9\xFF\x45\xC3\x13\xCD\x75\x3C",
			"\xE0\x39\x1F\x43\xA3\x76\x54\x6F\v\x4D\x44\xDC\x94\xAB\x5D\x8C",
			"\xF4\x48\xAC\xFA\xF4\xF1\x74\xD8\x98\xAF\x8A\xCC\x89\x35\x7E\x87",
			"\xB0\x24\x3A\x82\x41\xC0\xC4\x93\x9B\x54\x79\x99\x8F\x4E\xE5\xAB",
			"\x3F\x1C\x66\x4A\xAC\x40\xD4\x71\xA9\xBD\x2E\x58\x53\x17\xF1\xD0",
			"\x28\xE2\xAA\xE6\xF0\x32\x94\xD9\x1A\xC5\x2C\xFC\xA1\xA3\x3D\n",
			"\x75\xD0\x6B\x27\xA0\xAB\x54\x57\x1A\xFB\x9E\xC3\x3C\x4C\x66\xB0",
			"\x24\x8C\x7D\xD0\x2E\x7A\x74\xF4\x68\xA8\x9A\x8F\x7B\x73\x40\xB2",
			"\x59\x08\xF1\xB6\x81\x70\xF4\xAF\xF9\xBF\x5F\x8F\xA4\x71\xEA\xA3",
			"\x9B\xB0\x05\t\xE2\x79\x64\xCD\v\xA5\xB5\xA5\x40\x7D\x07\x2A",
			"\xD4\x37\xAC\x39\xD0\xE3\x94\x93\v\x22\f\x9A\xD5\xEB\xF6\x81",
			"\xEC\xDA\x36\xC9\x5C\x76\xB4\xC8\xC9\x8F\x51\x1F\xD3\x2F\x11\xF2",
			"\x5C\xEE\xA4\x9A\xE4\xD5\x74\xDA\xCA\x5B\x7C\x3E\xD4\x26\xC6\xEB",
			"\x8B\x3A\x5E\xE1\xB3\x04\xA4\x07\xE8\x02\xCC\x23\x5F\x5D\xC3\xF6",
			"\x52\xA2\x00\x08\x9E\xCB\xA4\xE7\x18\x69\x47\x1A\x03\x82\xED\xDD",
			"\x7C\xE4\x3D\xE8\xD6\xAD\x14\xBF\x88\x2C\x2B\r\xA7\x2B\xCC\x59",
			"\xB6\x55\xE1\x59\x95\x74\x14\xF4\x19\x93\x9A\x7D\xA3\x12\x1F\xD6",
			"\x84\xF5\x94\x87\x15\xF9\x64\xB3\xB8\xBC\xBB\x8A\xA5\x00\x61\x2B",
			"\x52\x2C\x72\x92\x46\x4A\xD4\x34\x89\xAD\x49\x63\xEC\x26\x1E\x72",
			"\x1D\x15\xE0\xF2\xF8\xD9\xE4\x67\x89\xD9\xB1\x8A\x7F\xDD\x16\xD5",
			"\xA2\xC2\x36\x61\x39\x00\x24\x4E\x9A\x04\x9C\x1E\x12\x65\xB0\x84",
			"\xA5\x82\x2A\x8B\xCA\x5B\xA4\x39\xEB\xD7\x4F\x46\xD5\x8D\xF9\xA8",
			"\x3B\x94\xD6\x01\x81\x28\x34\xCA\xCA\xE7\xC0\x58\xCD\x5C\xBC\x5D",
			"\xD2\x27\x78\xB4\x94\xB6\x44\xB1\xFA\x96\x7B\x96\x13\xF8\xE8\x62",
			"\x65\x3B\x9A\x31\xE1\xD5\x84\x1E\xE9\xE2\x07\xB2\x1A\xB2\x20\x7A",
			"\x51\xD3\xE8\x47\x19\x06\x34\x44\x1A\x82\xFB\x16\xB2\x40\xA2\x4D",
			"\xF1\xC1\x2D\xB2\x28\x71\xF4\x3F\x19\xB6\xB2\x91\x2D\x5C\x25\x04",
			"\x35\x3F\x1E\xAA\x29\xCA\x44\xD1\xA8\x73\xBA\x6A\x78\x34\xCF\x2A",
			"\x02\x4A\xA3\x43\x87\x2A\x54\x4F\xDA\xE2\x56\x8A\x5A\x5A\xED\xDD",
			"\x1A\x4B\xC9\xA5\x42\x39\x34\xD3\x3A\xDB\x93\x1E\xED\x04\x0E\x77",
			"\x84\x6D\xD4\x41\xEC\x7E\x14\x21\x3B\xC2\x99\xDA\xAE\x6F\xC7\xAC",
			"\xFA\x6E\t\xAD\x30\x0E\x24\xD3\x48\x55\x5B\xD9\x9C\xB3\x25\x94",
			"\xA8\x65\x8C\x3D\x89\x88\x74\x72\xAA\xB8\x6C\x28\x3C\x35\x27\x76",
			"\r\xA6\xB2\xF4\x48\x98\x24\xA7\x7A\x08\xE0\xD2\x58\x21\x4B\x5D",
			"\x98\x73\x25\x4B\xE5\x35\xA4\xB7\xF9\x01\xB3\x2F\x86\xAF\x30\xE1",
			"\x2F\xCD\x53\x49\x5B\xB3\x04\xAE\xC8\xD7\x53\x31\xBB\xC1\xC3\x43",
			"\xB4\xC4\x59\xD3\x0F\t\x34\x69\xDA\xE4\xD9\xED\x83\x2C\x3C\x7B",
			"\xAB\xD6\xD3\xEB\xE6\xC8\xE4\xE1\x38\x62\xCC\x68\x78\x59\x8F\x6C",
			"\x23\x33\x38\x92\xBB\xC1\x04\x1F\x5A\x2C\xAB\xC4\xFE\x2E\x8A\x24",
			"\x2B\xA7\xD7\x9A\xF9\x9C\x04\xFF\x8B\xD9\xA8\x7E\x60\xB8\x40\x04",
			"\xFD\x10\x2A\xF5\xFC\x19\x44\x5A\x6A\xB3\x00\x82\xE7\x77\x56\x42",
			"\x31\xBF\x1A\xF2\x2B\xA1\x04\xD2\t\xE8\x76\x91\x8C\xFF\x19\xD2",
			"\xE4\xE2\x9D\xA4\xE4\x51\xA4\x25\x9A\x46\x12\x6F\x76\x08\x12\xE6",
			"\x75\r\xCC\xAE\x02\x91\x54\x35\xF9\x89\x4C\xB3\xE1\xF7\x93\x41",
			"\x7E\x7B\x2D\r\xCA\xE5\xD4\x8F\xAB\x62\xE8\x84\x9A\xFC\x1C\x4F",
			"\x56\x2F\xE2\xC2\x76\x1B\xC4\x61\x6A\xEB\x63\xD7\xAA\x5A\x5E\x59",
			"\x43\x06\x6E\xC3\xEC\v\x64\x5D\xA8\xC4\x29\x96\xAE\x56\xE5\xA6",
			"\xFA\xBC\x20\xE8\x73\x37\x84\x12\x48\x7A\x66\x21\x44\x47\x6D\xE6",
			"\x5A\x8C\x04\x21\xE1\x58\x14\xF9\x89\xF9\x66\x0E\xD8\x29\x1C\r",
			"\x57\xAA\x60\x95\xF3\x45\x44\x9A\x48\x06\xB1\x2F\x32\xB0\xE9\x18",
			"\xBA\xE1\x30\x74\x7D\x8D\x04\x6F\xD9\x60\xEB\xF0\xDE\xC3\x53\x81",
			"\x8E\xDE\x35\xB1\xED\x56\xD4\x3B\xA9\xA8\x60\x24\xD3\x70\xDB\x4B",
			"\x5D\x4A\x9C\x84\x9C\xC0\xF4\x1E\xF8\x31\xA3\x22\x65\xBB\x72\x53",
			"\x99\x33\xB8\x70\x18\xBB\x64\x6A\x19\xFF\x7D\xC0\x74\xDA\x90\x2D",
			"\x6F\x37\xB4\xD6\xE9\x29\x94\xCA\x28\xC3\x17\xB1\xD7\x16\x2C\x18",
			"\x9A\x75\xD2\xC4\xA7\x58\x04\x77\xBA\xBE\x95\x37\x61\x5E\x02\xB9",
			"\xA3\xC6\xA2\x7A\x9C\xC2\x44\x57\x7B\x1B\x4D\x68\xDB\xE7\xCC\x6D",
			"\xC2\x58\xEC\x44\xA5\x19\x44\xC3\xFA\xB6\xD7\xC2\xFB\x53\xA0\xDA",
			"\x6C\xCC\xC1\x26\xC1\xB1\x94\xC8\x7A\x4D\x78\x2C\xB6\x20\x4B\x88",
			"\xAF\x5B\xC6\xE7\x96\x8E\x74\xA9\xDA\x10\xA8\x0E\x39\x90\x6C\x71",
			"\x2B\x08\x13\xF5\x97\x66\x74\x18\x39\x37\xB1\x04\n\xAF\x7B\xFC",
			"\x27\xEB\xCE\xD8\x4F\f\x44\x6B\x2B\x7C\xC6\x08\x71\x95\xB9\x14",
			"\x7D\xE4\x50\xD4\xB4\xA3\x94\x76\xBB\x5F\x27\x1D\x74\xA4\x56\x61",
			"\x4A\xCE\xF1\xAF\x21\xDD\x74\xD9\xCB\x7C\xAC\x44\x31\xEA\x41\x35",
			"\x36\x30\x3E\x3D\x12\xDB\x84\xDA\xDB\x9C\x21\xF2\x13\x62\x28\x8A",
			"\xF9\x54\x32\xC9\xB2\xEC\x24\xE3\xA8\x06\x14\x92\x30\xFC\x07\x3B",
			"\xF7\x48\xED\x3A\x6C\xDC\xD4\x38\x8A\x7E\x51\xCF\x6F\xE3\x67\x3C",
			"\x38\xC2\x37\x46\x4D\xFB\xA4\x6E\x2B\xF2\x1F\xF7\x5C\x9F\x8B\xB7",
			"\r\x59\x41\x01\x21\x88\x34\xEE\x69\x9F\xD5\xF2\xB3\xE0\x5D\x90",
			"\x37\xF0\x6C\x92\x05\x1D\x04\xAD\xDA\x47\xBF\x3E\xF2\x77\x85\x2E",
			"\xEA\x00\x18\x04\xD2\xD9\x04\x87\x78\x15\x54\xFD\xED\xA8\x6C\xA0",
			"\xED\x60\x73\x73\x6B\xD4\x04\xF2\xBA\x14\x07\x84\xCB\x80\xC3\x6E",
			"\x1B\x7D\x43\x72\x90\x98\xB4\x1B\xDB\xFF\x6D\x47\xE8\x66\xF8\xB7",
			"\x7B\x2F\xB5\xBE\xFD\x1D\xC4\xBD\x29\x0E\x5E\x20\x4E\x5B\xC8\xD8",
			"\xDE\x17\x9F\x41\x89\x2E\x74\x2B\x8A\x34\xF4\x75\xF2\xA1\xE9\x5A",
			"\xA9\x24\x3A\x8F\x50\xFD\x44\x30\x3A\xE4\x1E\x47\x3A\x69\xC5\x49",
			"\x43\x36\x29\xED\x0F\x90\xB4\x2E\xFA\xA2\x48\x41\x36\x36\x43\v",
			"\x1F\xA6\x9B\x57\x39\x9C\x14\x25\x7A\x72\xF5\x3B\xB7\x82\x48\x38",
			"\x39\xED\x83\x39\xE6\x71\x04\x91\x69\x08\x47\x62\x58\xA5\xFD\x9D",
			"\x1A\xCE\x73\x99\n\x7C\x14\x54\x3B\xC2\x88\x76\x51\x27\x3A\x60",
			"\r\x53\x6D\x61\x85\x50\xE4\x0F\x88\x86\v\xB6\xF1\xE9\xBD\x26",
			"\x69\x98\xCC\x5F\x4D\x06\x74\x49\xA8\xB3\xF3\x80\xBA\x47\x63\x1A",
			"\xE8\x74\x28\xCC\xB1\x82\x34\x1B\xBB\xA1\x66\xA2\x29\x24\x6B\xA0",
			"\xE8\x5A\x08\xF3\x9D\x36\x14\x03\v\xF3\x0E\x3E\xED\xC4\x40\x69",
			"\xAC\x31\x35\xCF\x34\x81\x14\x8A\x9B\x89\r\x96\x44\xAF\xD2\xBE",
			"\x8F\x4F\xF2\xE8\x13\x6D\xF4\x32\x48\xA9\x30\x8E\xB5\x7E\n\x94",
			"\x21\x89\x92\xC1\x13\x8A\x34\x04\x1A\x54\x3C\x74\x22\x8C\xF6\x44",
			"\xBC\xAB\x13\x3C\x2A\x5B\xD4\x98\xBB\x12\x0E\xDE\xFE\xEB\x5E\x65",
			"\xEB\x5F\xDB\x67\x6D\xDF\x74\x05\x68\xE2\x95\x90\x65\x44\xFF\xA8",
			"\xCF\x62\xED\x02\xBC\x75\x84\x10\xFB\x94\xAA\xFC\x2E\xE9\xDE\x98",
			"\x63\x58\x6F\xD2\xD6\xE4\x84\x52\x58\xEA\x00\x3E\x60\x87\x55\x61",
			"\xD1\x86\x03\x6D\x87\x05\xA4\x76\x2A\x88\x10\xE9\x41\x11\x0F\x03",
			"\xD9\x3F\x1B\xD8\xAB\xF5\x64\v\xF8\x08\x07\x6B\xBD\xED\xFB\x9D",
			"\xD3\xA9\x1F\xF7\x07\x7C\x54\x76\x9A\xB7\xEF\xC8\x2F\x96\x38\x60",
			"\x6A\xC7\xF8\x9C\x43\xB8\x34\xF9\xD9\x3F\xCA\x6D\xFE\xCB\x04\x24",
			"\x5E\xE8\x05\xB3\x59\xF2\x44\xE9\n\x9D\xF6\x5F\xAB\r\x05\x8D",
			"\x9C\x5A\x5E\x38\x32\x8B\x04\xC2\xCB\x86\x11\x52\xD7\x48\xB6\xB0",
			"\x58\xC2\x2D\xFB\x5E\x3A\x34\x75\xA8\x89\n\x16\x5B\x05\x4E\xAB",
			"\x27\x8A\x27\x32\xF7\xDD\x84\x95\x78\x99\x65\x6B\x91\x99\x8C\x8A",
			"\n\x8A\x89\xC7\x46\x5A\xD4\xBE\x9B\r\xA9\xB3\xE0\x61\x4C\x3D",
			"\xC6\xBE\xEB\x6B\xD8\xA8\x84\x5B\x1A\xE3\xF8\x80\x47\x84\r\x0E",
			"\xAA\x7B\xED\xBC\x81\x10\x54\x5E\x29\x6E\x78\xE9\x54\x42\x92\xE1",
			"\xDE\x17\xC7\xC5\x67\xCF\x54\x16\xAB\xEA\x67\x6C\xE8\xB2\x92\xDD",
			"\v\xC1\x41\x0E\x6D\x35\x14\xC8\x18\xF4\x55\x68\x6B\xD8\x2E\x26",
			"\x9B\xBD\xC9\x02\xAA\xF0\x44\xE3\x2B\x49\xDA\xD3\x96\x78\x0E\x6C",
			"\x2A\xCC\x54\xE0\x8A\x7F\xD4\x88\xFB\x17\xEE\x9B\x46\x3F\x1D\x62",
			"\xE0\x8B\xCC\xB9\xC7\x9E\xF4\xB0\xF8\x7D\x44\xDA\x1E\xF2\xBF\x27",
			"\xEF\x1B\xF2\x80\x4A\xDE\xB4\x8A\x1B\x52\x12\x7D\x40\xB2\x70\x37",
			"\x2D\x34\x73\x4C\x7F\x80\x04\x04\x7A\x4F\v\xDB\xFE\x85\x86\n",
			"\x7D\x79\x39\x06\x59\x7C\xE4\x6D\x1B\xB8\xEE\x75\x82\x00\x2E\x5E",
			"\xC8\x6C\xEF\x98\xDB\x12\xF4\xC0\xDA\x6F\x0E\xDA\xA4\xD2\x19\x68",
			"\xAE\x7A\x3C\x01\x12\xAE\xC4\x4D\n\x10\xD2\x69\x55\x94\xB2\x9E",
			"\x3F\xD7\x47\x92\x53\x6F\x84\xBB\xFA\x1D\x78\x39\x4A\xFF\xF2\xF6",
			"\x5E\xB5\xCE\x1D\x6C\x2C\x74\x34\x79\xD7\x2F\t\x78\x08\xB9\x00",
			"\x00\xB6\x71\xC1\x1E\x54\x64\x73\x58\x55\v\xFA\x7C\x88\x3F\xEE",
			"\xEA\xE8\xB3\xA3\x79\xE1\x84\xED\x48\x12\x5D\x3F\x65\xC5\xFF\x2F",
			"\xB4\x12\x44\xBE\x17\xEE\xD4\x76\xBA\x94\x0E\xE3\xFB\x67\x63\xCF",
			"\xC8\x4C\x4A\xA3\xDD\xDB\xC4\xE2\x7A\x98\xCE\x30\xAA\x17\x58\x78",
			"\x92\x01\xD9\x84\x7C\xDF\x24\x4D\v\x28\xB2\x50\x61\x87\x29\xD5",
			"\x3C\xEE\xA0\xE3\x65\xCD\x94\xBB\xEB\xC6\x89\xD5\x1A\x6C\x91\x45",
			"\xFA\x7C\xFA\x9B\x1D\xC2\x94\xEA\x8B\xE1\xF6\x85\x49\x4A\xB7\xDE",
			"\x6E\x28\xC7\x16\x23\x42\xA4\xA3\xDB\xDB\x82\x0F\x74\x20\xB6\x45",
			"\x25\x40\x83\x38\x91\x9F\x24\xF2\x18\x2B\xBF\xF7\x05\xC3\x2E\x56",
			"\xB1\xBE\xA1\xE4\xE3\r\x14\xF5\x28\x20\x38\x7F\xFA\xEC\xAF\xB8",
			"\xB6\xD4\x33\x9E\x32\xAE\x04\x91\x8A\x78\x99\xAA\x4A\x8A\x96\x34",
			"\xA7\xB4\x85\x26\xCF\xCF\x24\x04\n\x6C\xAF\x99\xB7\x58\x26\x71",
			"\x1F\xA2\x40\xED\x75\x6D\x14\x40\x58\x36\x34\x11\xEF\x85\x76\x85",
			"\x44\x6E\xC2\x4F\xE2\x27\x74\x92\x5A\x64\x49\x1F\x63\x12\x5A\x66",
			"\x7C\xA3\x9E\x80\x18\x12\x44\x9B\x19\x1A\x76\x03\x9D\x42\xF5\x80",
			"\x45\xCF\xCD\x82\x9F\xFF\x04\x50\x79\t\xC6\x6A\xD6\xAD\xFA\x86",
			"\xCA\xD1\xAC\xCA\xD4\xA0\x94\xE8\x79\x52\x2B\x7B\x72\x59\xB1\x13",
			"\xE1\xB9\x68\xAD\xBB\x73\xE4\x42\x2B\x4D\x4E\x17\x4E\x2B\x15\x88",
			"\x47\x24\xCA\x63\xB1\x31\x14\x8B\x88\xED\x7C\x5B\x26\x3B\x94\xF1",
			"\x07\x39\xD2\x21\x73\x71\x14\x97\xEB\x2D\xA2\x07\xE3\xC7\xFF\x46",
			"\x54\xEC\x8B\x60\xF1\xE7\xF4\x6A\xD9\xB7\x63\xB0\x8B\xD4\x97\xAD",
			"\xB6\x93\xA9\x88\xDF\x34\xB4\x94\xFA\xC2\x5C\x52\xD3\xC8\x63\x5F",
			"\x42\x8D\xE1\xC5\xA9\x32\x84\xCF\x98\x3B\x17\x4A\xD6\x30\x95\xB6",
			"\x53\x88\xFD\xAC\x83\x28\xB4\xA6\x39\x6B\xB7\xCD\xE9\x02\x59\x77",
			"\x94\x40\xE0\x6D\xCA\xB3\x64\x7F\x88\x82\xD0\xEC\x99\x03\x43\xC9",
			"\x6E\x47\xA1\xC7\x77\x66\x44\x8A\x8B\xA1\x53\x64\x88\x3D\x40\x6D",
			"\xCE\x4F\xF6\x82\x4C\x95\x54\f\xDB\xB8\x2D\xA4\x8C\xC8\xDB\xC2",
			"\x05\x92\xFA\xCA\x10\x6C\x84\x2E\x69\xAB\x7F\xBA\x7C\xD8\x98\x5A",
			"\xBD\xAD\xFE\x44\t\xFE\x44\x65\x18\x62\x9D\xBB\x5C\xCF\x9B\xB2",
			"\xF5\xEE\x93\x63\x3E\xBB\x94\xAA\x2B\xE9\x1A\x4E\x27\xBE\x7F\x9A",
			"\x67\xA2\x97\x69\x23\xD0\xA4\x49\xDA\x98\x8A\x27\x5F\xB2\x7E\x64",
			"\xEB\xD9\xFB\xC8\x08\x76\xE4\x83\x6A\xA5\r\x5E\x10\t\xF0\x27",
			"\x7B\x74\x4A\x04\x10\x94\x84\xBF\x5B\xED\x93\xDA\x43\x53\x17\xB8",
			"\xBF\x50\x12\x60\xE7\x90\x54\x60\x9B\x26\xD1\xF9\xB4\x7E\t\x90",
			"\x28\x3E\x37\x2E\x61\x69\xE4\xB3\x88\x18\x43\x5A\xC3\xE6\x82\x3A",
			"\x52\x62\x88\xA5\x79\xF5\x84\xDF\xDA\xA0\x73\xA0\x2C\xD4\x78\x2D",
			"\x0E\xD2\x49\xA7\x11\xC5\x74\x34\xB9\xB8\x8B\x89\x0E\x5A\xE4\xC8",
			"\x00\x23\x43\x61\x69\xFF\xB4\x85\x2B\x19\x34\x57\x83\x9D\x13\x95",
			"\x7D\x3C\x55\x05\x20\x93\x14\xF0\xC8\x69\xC8\xB5\x4F\xB8\x26\xE7",
			"\x3F\x47\xC6\x2A\x01\xA9\x84\x83\x29\xA0\x10\x72\x06\xDD\xF8\x2C",
			"\x9C\x6E\xC7\xA0\x56\xD6\x84\x1E\x7A\xF3\xE3\xDE\x2F\x8C\x50\x19",
			"\xA0\x56\xA0\x45\x79\xA7\x64\x5E\x68\x8D\xA4\x08\xB0\xEB\xDB\xD6",
			"\xC0\x41\f\x6C\xBD\xC8\xA4\xB8\xE9\x91\xC1\xE2\xE0\xDF\x32\x93",
			"\x2A\x28\xE5\xC3\xC0\xD8\x74\x81\xBA\xB1\xC2\xAD\x79\x26\x18\x43",
			"\x23\x98\x4B\x2A\x3A\x25\xF4\xDD\xBB\xDB\x50\x8F\x2C\x7E\x6F\xD1",
			"\xE8\xF2\xD3\x86\x9B\x67\x24\x32\xB8\xE8\x89\xF4\xA4\xD6\x32\x42",
			"\xBC\x34\x13\x84\xFD\x82\xE4\xFF\x39\xF9\xE7\f\x58\xB6\xF7\x8F",
			"\x16\xD8\xC3\xC2\x7C\xEE\xA4\x65\xDA\x88\xC7\x6E\x22\x8B\xA1\x60",
			"\x47\x4F\x14\xA9\xB2\xFD\x04\x5E\xE8\x5F\x67\x2D\xF2\xA0\x2C\x44",
			"\x55\x8E\xDB\x42\x74\x6A\xD4\x81\x99\x03\xF5\x29\x73\x2F\x63\x38",
			"\xB5\x8C\x69\f\x7C\x74\x94\x81\xCA\x74\x11\xE4\xA2\xC2\x42\xF8",
			"\x8E\xD9\x1F\xE7\x58\xF0\xB4\x30\xD9\xF1\x18\xF4\x65\x50\x9F\xDC",
			"\xB3\xF6\x8F\xE6\xEA\x85\x54\xB4\x9A\x7C\x45\xF1\x14\xDD\xCC\xE2",
			"\x40\x21\xAB\xA6\x39\x4B\x64\x43\x6A\xC5\xDC\x2F\x28\xF4\xB6\x92",
			"\x3D\xBD\x3B\x65\x74\x61\xF4\x20\xB8\x1A\x29\x99\x13\r\x26\xE1",
			"\xA1\xB9\x24\x8D\xFE\x98\xE4\xDD\xFB\x76\x11\xD9\x2F\xB7\xC1\x9E",
			"\x36\x6B\xC8\xAC\xD7\x5C\xB4\xF7\x8B\x3D\x13\x1A\xFC\xC2\x5F\xB1",
			"\x37\x4D\xF9\xEB\x8E\x77\x34\xFF\xF8\x5C\x6D\xFF\xE3\x50\x8D\x1C",
			"\x77\x11\xE1\x42\x98\x61\x04\x29\x18\x83\xB6\x99\x7B\x4F\xA8\x6D",
			"\x46\x63\xBA\x2F\xD8\x7E\xD4\xE3\n\xE7\x78\x9A\x25\xEA\xBB\xF1",
			"\x91\x58\x80\xEF\xD1\x07\x44\x92\xCA\x1B\xB1\x57\xFA\x40\x3E\x37",
			"\x9E\x88\xA0\x25\x4D\x5A\xD4\x69\x4B\xB4\x90\x2E\xBF\x81\x8E\x17",
			"\xBB\xB9\x5F\x6B\x83\x9F\x74\x1B\x69\xA1\x30\x9D\x57\xF2\x7F\x6B",
			"\x76\xCB\xEA\xEB\x3B\x14\xA4\x68\x5A\x4E\x4A\xD6\x84\xE1\xAC\xB6",
			"\x5B\xDA\x36\x4E\xF7\x40\x84\x67\x98\x74\x85\xEF\x08\n\x0F\xDE",
			"\x36\xDA\xDA\xAE\xA3\x58\x14\x27\xEB\xBF\x6D\x5B\xA6\x9E\xDD\x32",
			"\x61\xA6\xD4\xAC\xDD\x82\x14\x93\xAB\xBE\x65\x54\x27\x74\x47\xA8",
			"\x9E\x78\xA2\xEE\x1B\x82\x54\xAD\xD8\xE0\x47\x7A\x46\xEA\x9B\xE1",
			"\xB6\xEC\xC6\x26\xDB\x5C\x64\x98\x08\x69\xEC\xB6\xCE\x81\xE1\x79",
			"\xB5\x32\x37\x05\x52\x55\xA4\x92\t\x73\x22\x39\x01\xD2\xD5\xDA",
			"\xA3\xB2\xAB\x41\x15\x4D\x14\x25\xD8\x93\x57\xEE\x1B\x44\xED\x16",
			"\xC8\x03\x94\xEF\x86\xC5\x64\xD2\xDB\x67\x8E\x81\x53\x2B\x20\x2F",
			"\x9A\x36\x3A\x24\xDF\x75\xF4\x32\x1A\xCC\x68\xC9\x15\x2E\x6F\xA6",
			"\xFA\xB3\xEB\xCE\x3D\x18\x64\x27\x3B\xAC\x18\xA2\x22\x5B\xC1\xE9",
			"\x1B\xFE\x95\xE5\xD9\x81\x74\xD2\xFB\x70\xC5\x06\xDB\xEA\x56\x29",
			"\x39\x6F\xAE\x20\x87\xBE\x24\xC8\x8A\xB7\xD5\x0E\x56\xB3\xC6\x72",
			"\xD4\x02\x66\x04\xC5\xF0\x94\xEF\x29\x91\xE3\x11\x84\xEA\n\x00",
			"\x89\xB5\xA8\x29\x65\xED\x44\xE3\x6A\xCA\x8D\x75\x1C\xB2\xEA\x2B",
			"\x43\x87\xA2\xAB\xE3\x9D\xF4\xFA\xD8\x80\xE3\x46\x28\xAA\xA1\x2E",
			"\xA1\x61\x8A\x2A\xBC\x80\xC4\x95\x68\x34\x21\xC1\xBD\x3E\xD0\xE5",
			"\x4C\xD1\xFA\x88\x41\r\xB4\x62\x39\xFE\xB9\x07\xFB\x23\xB2\x56",
			"\xDA\x94\xB6\x4E\xB0\x9A\x54\xE5\n\x83\x11\x23\xB1\xD5\xE5\xFF",
			"\xC5\xE2\x5D\xA5\x17\xDE\x94\x85\x3A\x9B\x04\x73\x5A\xCD\x76\xDE",
			"\xB5\x40\xE4\x60\xC0\xB4\xE4\x86\xE9\x01\x27\x72\x77\xD7\x02\x98",
			"\x0E\x07\x60\xC0\v\x66\x44\x50\x6B\x03\xAC\x74\xED\x23\x5D\x58",
			"\x87\x24\x5F\xCC\x15\x88\x24\xAB\x7A\xA8\x39\xCB\x32\x1D\xCF\x9B",
			"\xFE\xD6\x49\xA9\xBF\xC0\x94\x40\x29\x93\x3F\x41\xBD\xE6\x1D\x55",
			"\x87\xCD\xDE\xE3\x9C\x7D\x54\xB4\xCB\x8C\xB9\x6F\xF3\x01\x16\x50",
			"\x5D\x7F\x83\xC0\x29\xDC\x04\xDF\x18\x2C\xFE\xEB\xA8\x3C\xD0\x05",
			"\xD4\x32\x94\x29\xDC\xFE\xE4\xE6\xCA\x4B\x8E\x07\xAB\xEE\x2A\xB0",
			"\x36\xBF\xBC\x43\xCC\x5B\x64\xBD\xCA\xC9\x65\xFE\xFE\x26\x6A\xE5",
			"\xEF\x49\xCA\x8F\xDD\xE7\xF4\xEB\xC9\x89\x0E\x44\xB8\x18\x4E\xEE",
			"\x88\x72\xB4\xC1\x12\x36\x44\x04\x38\xCA\xE3\x6F\xC2\xAE\xA4\xC5",
			"\xA9\x74\x10\x06\xA6\x90\x44\x7C\xDB\xB9\xE2\xA5\x52\x4D\x9F\xF4",
			"\xE8\x38\x1D\x05\xDD\x60\x34\x19\xA8\x9C\xB7\x8D\x73\x05\x34\xA0",
			"\x59\x98\x21\x0F\x2A\xBA\x94\x54\xA9\x47\xE7\x9F\x2C\xAF\x6F\x99",
			"\x9B\xE0\x73\xE5\x25\xBA\x94\x67\x29\x44\x54\x42\x81\x89\x27\xEB",
			"\x52\x34\xD1\f\xEA\x08\x44\x3A\x6A\x98\x12\x27\x56\x62\x95\xC1",
			"\x3A\x78\x54\xA2\x89\x7A\xE4\xF3\xAA\x3E\x22\x9C\x5E\xA0\xA9\x90",
			"\xF3\xD7\xD2\xE4\xE9\xAF\x24\x28\v\xED\x64\xA3\xCC\x4E\x55\x6F",
			"\xC7\xBF\x54\x81\x6C\xA3\x04\xBB\x6A\xA8\xC9\x01\x53\xF5\x3A\x10",
			"\x70\xD8\xC0\x86\x4F\xC7\x94\x5F\n\x4A\x24\x1B\x2D\xBF\xFE\x03",
			"\xF0\x15\x31\n\x16\xF6\x94\x74\x2A\x75\x06\xE0\x18\xC0\xA2\x7F",
			"\x6D\x82\x15\x62\x71\x89\xB4\x67\xEB\x1B\xB2\xE3\x1A\x3E\xBA\x05",
			"\x7F\x9A\x91\xC1\xFD\x31\x04\x4E\x6B\x26\xEB\x43\xE3\x6A\x34\xA6",
			"\xF0\xD8\x20\x68\x06\xC8\x64\x7C\x48\f\xB4\xF5\x3E\x04\xBA\x2F",
			"\xBB\x47\xCF\xCF\x33\xD2\x84\x02\x9A\xFC\x93\xE2\x81\x9C\xCC\x33",
			"\xF7\x6D\x63\xCD\xEC\x6D\xA4\x99\xC8\x52\x87\x36\xAA\x30\x8C\xE4",
			"\xFD\xCB\xA0\xCA\xCC\x90\x34\x6E\x3A\x60\xD5\x38\xCF\x57\xC4\x83",
			"\xF0\x6D\xC7\x43\x3C\xBB\x64\x79\x79\x05\x9F\x74\x81\x7C\x97\xA6",
			"\xC1\xC8\xC4\x00\n\xA9\x34\x14\xF8\x7C\x29\xEA\x9B\xFF\xA1\x3D",
			"\x7C\xA0\x12\x07\x2E\x46\x24\x26\xB9\xC7\xD7\x9A\x7E\x55\x9F\x0F",
			"\x82\x3F\x74\x0E\x26\xCF\xD4\x19\xA8\x14\x17\x51\xA0\x27\x52\x2D",
			"\x6B\xD1\x1E\x6C\xA4\xEE\x54\x4B\xEB\x9D\xAE\xFE\x84\x78\xF0\x87",
			"\xB6\xA4\xD4\x66\x2E\xD4\x04\xBB\x3A\x69\x80\x4C\xF4\x84\xC9\x06",
			"\x19\xDB\xD5\xEE\x98\xF8\xF4\x07\x99\x6B\x13\xB6\x77\x82\xBF\x77",
			"\x99\xD5\xDC\xAA\x9B\xF5\x64\x31\v\xB7\x01\x05\xCF\xA7\x6D\x65",
			"\xBF\v\xD3\x02\xC5\xBD\x94\xFD\xAB\xBC\xD6\x04\xE8\x2B\xD3\r",
			"\xDF\x7D\x00\x42\x6D\x2B\x94\x9F\xEB\v\xD3\x36\xDB\x8E\x28\x46",
			"\xA4\xFE\x69\x25\x54\xAA\x04\xCC\xDB\x02\xB7\x78\x78\x18\x2E\xE7",
			"\xAA\x31\x7B\x20\xD8\xF9\x84\x22\x6B\x5F\xA1\xE5\x5C\x06\x3E\x69",
			"\xB7\x73\x8E\x44\x7E\xE3\xD4\x7F\x88\xDD\x34\xF2\xC1\x21\r\x74",
			"\x87\x43\x1C\xA4\x11\x7D\xB4\x6F\xF9\xC6\xAD\xED\xB8\x11\x8D\x98",
			"\x2F\x84\x70\x60\x62\x6C\x74\xE9\x5A\xE9\x4D\x41\x7A\x40\xD4\x0E",
			"\xFA\x00\xEB\xC6\x1A\x12\xF4\x94\xE9\x5E\x67\x77\x05\xBA\xC5\xB8",
			"\xF5\x3F\xCF\xA0\x71\xFF\x34\x2A\xEB\x36\xE0\xF6\x00\xF2\xA8\x02",
			"\xB8\x34\r\t\xA0\x26\xB4\x2B\v\xD6\x25\x5D\x16\xE3\x1D\xCE",
			"\x79\x81\x90\x61\xD9\xE0\x94\x60\x3A\xA2\x21\x1D\x46\xD9\x85\x6A",
			"\xDF\x2A\xA8\xEA\x5B\x9F\x04\x71\x9B\xE9\x77\xDB\x22\x77\x90\xA8",
			"\xE4\xA3\x60\x88\x03\xFC\x84\x81\x99\x75\x03\xD3\xBA\x43\x2B\x08",
			"\x4B\x4B\x9C\xE0\x99\x9F\x14\x8F\x88\x56\x80\xCF\xAB\x4E\x8C\x80",
			"\x5E\xCC\xE1\xAC\x97\x5A\xC4\x5A\xFA\x41\v\xC9\xB2\xAB\x83\x68",
			"\x70\xAF\x7E\xA9\xC1\x9E\x04\x60\xEB\x4A\x93\x1E\xEC\x7E\x61\x36",
			"\x2B\x23\x76\x80\xD0\x87\xC4\x91\xDA\xC0\x11\x3F\x15\x38\xEC\xD0",
			"\xCB\xDF\x91\xC6\xF6\x39\xD4\xF1\xC9\x29\xF7\xFA\xDB\x91\x65\v",
			"\xF7\x62\x57\x0E\x45\xB2\x14\x40\x89\x03\x81\x28\x9C\xEC\x98\x4E",
			"\xE8\x96\xBC\x66\x8A\x1E\xB4\xBD\xDB\xA0\xCD\x57\x30\xC7\xA7\x7B",
			"\x96\xDE\xEB\x67\xD1\x84\x64\xDD\xDA\x3B\xC3\x68\x23\x58\x1B\x59",
			"\x98\x67\x2A\x03\x42\x55\x84\xC3\xA9\x48\x91\x5D\xE7\xD3\xD2\xCE",
			"\x90\x3D\xED\xE7\x47\x6B\x24\x5A\n\xA9\x36\x1A\x71\xE5\x2D\x65",
			"\x93\n\x33\x88\x55\x28\x34\x06\x39\x44\x32\x9A\xD5\x54\xC0\xFA",
			"\x26\x8F\x14\x65\x22\x2F\x04\x52\x78\x4C\x2A\xBF\x65\xDF\xA5\xA7",
			"\x9B\xB5\x21\x83\x86\x70\xE4\xAB\xF9\x62\xF3\xD4\x00\xE4\x07\x1C",
			"\xE2\xEC\xC1\xCF\xD6\xA0\xE4\xC2\xBB\x8B\xA8\x02\xB7\x64\x29\x77",
			"\x59\xF0\x77\x88\x65\x2A\x04\x3F\xF8\xEC\x67\x4E\x1C\xE8\x7F\xC0",
			"\x77\x99\x4F\xE8\xCF\x9D\x04\xF5\xAB\x74\x40\x76\x5A\x78\x7F\xAE",
			"\x70\x57\x54\xAD\x34\x6F\x14\xCE\x5A\xB0\xC2\x2B\t\x82\x59\xBB",
			"\x46\x5F\xBE\xE3\xE4\x4E\xC4\x10\xBB\x0E\x8E\x5D\x24\x69\xD9\xA2",
			"\x10\x5B\x5F\xC7\xF1\x4C\x34\x5B\x79\x93\x20\x35\x5A\x2B\x4A\x1D",
			"\x2E\x13\x82\xA4\xFC\x14\x54\x02\x8A\x01\x3C\xE5\x7A\x82\xBF\xB0",
			"\f\x45\xE4\x41\x1E\x02\x34\x51\x6A\x45\n\x51\x45\n\x2F\xA9",
			"\x0F\x48\xC2\xA5\x90\xC2\x94\xAD\x4B\x8D\x2B\x56\xFC\xFE\x73\xEE",
			"\x3B\xA7\x82\xA5\x66\xBC\x04\x6D\xCB\x93\x3E\x96\xEB\n\x3B\x12",
			"\xC3\x46\x6E\xCB\x4F\xEA\xF4\x1D\x2A\x65\xB3\x11\xDD\xD4\x2B\x01",
			"\x7F\x4B\x67\xA4\x96\xC3\x74\xE6\x7A\x1E\x50\x17\xD6\x84\xB7\x21",
			"\x7B\x19\x3C\x86\x86\xCA\xD4\x7D\x58\x76\x51\x16\xFC\x96\x65\x8F",
			"\xB6\xC5\xBB\x06\x18\x22\xD4\x43\x48\xCE\xA1\xEE\x50\xE5\x75\x86",
			"\xE4\xEE\xAD\x00\x38\x8D\xD4\x5F\xBA\xC9\x32\xB2\xB2\xB3\r\xDC",
			"\x03\xC6\x3B\xC1\xA0\xD2\xF4\x8E\n\xB1\x96\xA6\x2F\x96\x18\xAB",
			"\xC0\xE8\x2A\x4D\x1B\x21\x94\x3D\xAB\x6C\x83\v\x33\x5C\x62\xC1",
			"\xB5\x6D\x8A\x64\x53\x76\xF4\x7B\xB8\x1A\x70\x01\x0E\xE5\x9F\x2A",
			"\x45\xEC\x24\xA3\x47\x8E\x74\x29\x38\x85\x5E\xA2\x4B\xA8\x9E\xCA",
		);
		for ($i = 0, $binary = ''; $i < 256; $i++) $binary .= chr($i);
		$binaryStrings[] = $binary;
		unset($binary, $i);

		$notBinaryStrings = array(
			"Одним из преимуществ является совместимость с ASCII — любые их 7-битные символы отображаются как есть, а остальные выдают пользователю мусор (шум).",
			"\r\n\t",
			"some\vexa\fmple",
			"UTF-8 fa servir grups de bytes per a representar l'estàndard d'Unicode per als alfabets de moltes de les llengües del món. És molt útil especialment per a la transmissió",
			"是一種針對Unicode的可變長度字元編碼（定长码），也是一种前缀码。它可以用來表示Unicode標準中",
			"對上述提及的第四種字元而言，UTF-8使用四個位元組來編碼似乎太耗費資源了。但UTF-8對所有常用的字元都可以用三個位元組表示，而且它的另一種選",
			"Cho đến đầu năm 1992, người ta vẫn đang tìm kiếm một bộ mã hóa thành byte cho các tập ký tự có nhiều byte. Tiêu chuẩn ISO 10646 ở dạng nháp có một phần phụ không bắt",
			"é um tipo de codificação Unicode de comprimento variável criado por",
			"Текст, що складається тільки з символів з кодом меншим, ніж 128, при записі в UTF-8 перетворюється",
			"karışık olan harf ve rakam dizgileri kullanmak gerekiyordu. Mesela Ş harfini yazmak için Ş küçük ş içinse ş yazmak gibi.",
			"UTF-8 är konstruerat så att tecken som tillhör ASCII-tabellen (som täcker A-Z, a-z, 0-9, vanligt förekommande interpunktion samt ett antal kontrolltecken) kommer att kodas på",
			"Latin 1 ‑lohkoon sisältyvät merkit (mm. Å/å, Ä/ä ja Ö/ö) koodautuvat siis kaksitavuisina ja muut monikielisen perustason merkit kaksi- tai kolmitavuisina.",
			"Kán-tan kóng, thong-iōng-bé sī 1-ê kok-chè piau-chún.",
			"مجموعه‌ای از استاندارد رمزگذاری کاراکتر، شمارش خواص شخصیت مانند حروف بزرگ و کوچک، مجموعه‌ای ازمرجع فایل‌های داده‌های ",
			"यूनिकोड को ग्राहक-सर्वर अथवा बहु-आयामी उपकरणों और वेबसाइटों में शामिल करने से, परंपरागत उपकरणों के प्रयोग की अपेक्षा खर्च में अत्यधिक बचत होती है। यूनिकोड से एक ऐसा अकेला सॉफ्टवेयर उत्पाद अथवा ",
			"収録されている文字は、各国で標準として規定されている文字集合や実際に使用されている文字を持ち寄り、委員会により取捨選択されている。",
			"სიმბოლოებს, შესაბამისი კოდებით. შემდეგ ჯგუფებში მოთავსებულია სხვადასხვა დამწერლობის ნიშნები, პუნქტუაციის",
			"कॅरॅक्टर एनकोडिंग' ह्या संज्ञेची अगदी सोपी व्याख्या म्हणजे कोठल्याही 'कोणत्याही एका मानवी भाषेतील सर्व अक्षरे, चिन्हे यांना काही विशिष्ट क्रमाने ठरवून दिलेले गणितीय आकडे'.",
			"கணியுலகில் வெவ்வேறு வரிவடிவங்களுக்காக வெவ்வேறு குறிமுறைகள் இன்று பயன்பாட்டிலுள்ளன.",
			"یونی کوڈ جسے اردو میں یکرمزی کہ سکتے ہیں [1] ایک معیاری ضابطہُ تختی ہے جس میں تمام زبانوں کے حروف کو یکساں طور پر پیش کیا جا سکتا ہے اس ",
			"The standard consists in a set of code charts for visual reference, an encoding methodology and set of standard character encodings",
			"Στους υπολογιστές, το διεθνές πρότυπο Unicode στοχεύει στην κωδικοποίηση όλων των συστημάτων γραφής που χρησιμοποιούνται στον πλανήτη, ώστε να γίνει δυνατή η αποθήκευση -στη μνήμη ενός υπολογιστή- γραπτού κειμένου όλων των γλωσσών συμπεριλαμβανομένων και συμβόλων επιστημών, όπως μαθηματικά, φυσική κτλ.",
			"",
		);

		// Variants quality check
		if (true) {
			$variants = array(
				// Worst option. Often does not detect binary strings
				0 => function($string) {
					return (bool)preg_match('~[^\x09-\x0d\x20-\xff]~S', $string);
				},
				// Not always determine the NON-binary strings
				1 => function($string) {
					return !preg_match('~^.+$~Dsu', $string);
				},
				// Good variant! But not work for valid ASCII
				2 => function($string) {
					return false === preg_match('~~u', $string);
				},
				// The best variant! Found no false positives!
				3 => function($string) {
					return false === preg_match('~~u', $string)
					       || preg_match('~[^\x09-\x0d\x20-\xff]~S', $string);
				},
			);

			// Test
			$results = array();
			foreach ($variants as $variant => $fun) {
				$res = array('false' => 0, 'true' => 0);
				foreach ($notBinaryStrings as $string) {
					if ($fun($string)) {
						$res['false']++;
					} else {
						$res['true']++;
					}
				}
				$results["variant_$variant"]['notBinary'] = $res;
			}
			foreach ($variants as $variant => $fun) {
				$res = array('false' => 0, 'true' => 0);
				foreach ($binaryStrings as $string) {
					if ($fun($string)) {
						$res['true']++;
					} else {
						$res['false']++;
					}
				}
				$results["variant_$variant"]['binary'] = $res;
			}
			unset($res, $fun, $string, $variant);

			// Format
			$results = array_map(function($e) {
				$e['result'] = $e['notBinary']['true']
				               - $e['notBinary']['false']*10
				               + $e['binary']['true']*1.5
				               - $e['binary']['false'];
				return $e;
			}, $results);
			CArray::sortByAttribute($results, 'result', true, true);

			print_r($results);
		}

		// PCRE error variant speed test
		if (false) {
			/**
			 * All options are similar, but we can still choose '~~u' as best
			 */
			$iteratons = 100;
			$tests     = 8;

			$regexps = array(
				'~~u',
				'~~Su',
				'~.~Su',
				'~.~u',
				'~.~su',
			);

			$res = array();
			$yMax = (int)round(count($binaryStrings)/count($notBinaryStrings));
			for ($j = 0; $j < $tests; $j++) {
				foreach ($regexps as $regexp) {
					$start = microtime(true);
					for ($i = 0; $i < $iteratons; $i++) {
						foreach ($binaryStrings as $string) {
							preg_match($regexp, $string);
						}
						for ($y = 0; $y < $yMax; $y++) {
							foreach ($notBinaryStrings as $string) {
								preg_match($regexp, $string);
							}
						}
					}
					$res[$regexp][] = Date::timeEnd($start);
				}
			}
			$results = Benchmark::analyzeResults($res);

			print_r($results);
		}

		// Sebastian Bergmann variant speed test
		if (false) {
			/**
			 * Flag 'S' is very usefull. More then 130% faster!
			 */
			$iteratons = 300;
			$tests     = 10;

			$regexps = array(
				'~[^\x09-\x0d\x20-\xff]~',
				'~[^\x09-\x0d\x20-\xff]~S',
			);

			$res = array();
			$yMax = (int)round(count($binaryStrings)/count($notBinaryStrings));
			for ($j = 0; $j < $tests; $j++) {
				foreach ($regexps as $regexp) {
					$start = microtime(true);
					for ($i = 0; $i < $iteratons; $i++) {
						foreach ($binaryStrings as $string) {
							preg_match($regexp, $string);
						}
						for ($y = 0; $y < $yMax; $y++) {
							foreach ($notBinaryStrings as $string) {
								preg_match($regexp, $string);
							}
						}
					}
					$res[$regexp][] = Date::timeEnd($start);
				}
			}
			$results = Benchmark::analyzeResults($res);

			print_r($results);
		}
	}


	/**
	 * Compare with var_export
	 *
	 * @author amal
	 */
	public function testCompareWithVarExport()
	{
		$phpGen = new PhpGen;

		$var = false;
		echo $phpGen->getCode($var) . PHP_EOL; // false;
		echo var_export($var, true) . PHP_EOL; // false

		$var = true;
		echo $phpGen->getCode($var) . PHP_EOL; // true;
		echo var_export($var, true) . PHP_EOL; // true

		$var = null;
		echo $phpGen->getCode($var) . PHP_EOL; // null;
		echo var_export($var, true) . PHP_EOL; // NULL

		$var = fopen('php://stderr', 'w');
		echo $phpGen->getCode($var) . PHP_EOL; // "Resource id #68";
		echo var_export($var, true) . PHP_EOL; // NULL

		$var = (object)array('a' => 1, 'b' => 2);
		echo $phpGen->getCode($var) . PHP_EOL; // unserialize("O:8:\"stdClass\":2:{s:1:\"a\";i:1;s:1:\"b\";i:2;}");
		echo var_export($var, true) . PHP_EOL; // stdClass::__set_state(array('a' => 1, 'b' => 2))

		$var = "\n\t\r\0\1\2\3\4\5\6\7'\"\$v";
		echo $phpGen->getCode($var) . PHP_EOL; // "\n\t\r\x00\x01\x02\x03\x04\x05\x06\x07\x27\x22\x24\x76";
		echo var_export($var, true) . PHP_EOL; // '		' . "\0" . '\'"$v'
		$this->assertSame($var, eval("return " . $phpGen->getCode($var)));
		$this->assertSame($var, eval("return " . var_export($var, true) . ';'));

		$var = new SplFixedArray(3);
		$var[0] = 'a';
		$var[1] = 'b';
		echo $phpGen->getCode($var) . PHP_EOL; // ["a","b",null,];
		echo var_export($var, true) . PHP_EOL; // SplFixedArray::__set_state(array(0 => 'a', 1 => 'b', 2 => NULL))

		$var = function($a, $b) {
			return $a + $b;
		};
		echo $phpGen->getCode($var) . PHP_EOL; // function($a, $b) {return $a + $b;};
		echo var_export($var, true) . PHP_EOL; // Closure::__set_state(array())
	}
}
