namespace AngleSharp.Core.Tests.Library
{
    using AngleSharp.Browser;
    using AngleSharp.Core.Tests.Mocks;
    using AngleSharp.Dom;
    using NUnit.Framework;
    using System;
    using System.Linq;
    using System.Threading.Tasks;

    [TestFixture]
    public class MutationObserverTests
    {
        private static IDocument Html(String code, Boolean defer = false)
        {
            var config = Configuration.Default;

            if (defer)
            {
                config = config.With<IEventLoop>(_ => new StandingEventLoop());
            }

            return code.ToHtmlDocument(config);
        }

        [Test]
        public void ConnectMutationObserverChildNodesTriggerManually()
        {
            var called = false;

            var observer = new MutationObserver((mut, _) =>
            {
                called = true;
                Assert.AreEqual(1, mut.Length);
                var record = mut[0];
                Assert.IsNotNull(record.Added);
                Assert.AreEqual(1, record.Added.Length);
            });

            var document = Html("");

            observer.Connect(document.Body, childList: true);
            document.Body.AppendChild(document.CreateElement("span"));
            Assert.IsTrue(called);
        }

        [Test]
        public void ConnectMutationObserverChildNodesTriggerManuallyFromDocument()
        {
            var called = false;

            var observer = new MutationObserver((mut, _) =>
            {
                called = true;
                Assert.AreEqual(1, mut.Length);
                var record = mut[0];
                Assert.IsNotNull(record.Added);
                Assert.AreEqual(1, record.Added.Length);
            });

            var document = Html("");

            observer.Connect(document, childList: true, subtree: true);
            document.Body.AppendChild(document.CreateElement("span"));
            Assert.IsTrue(called);
        }

        [Test]
        public void ConnectMutationObserverAttributesTriggerManually()
        {
            var called = false;
            var attrName = "something";
            var attrValue = "test";

            var observer = new MutationObserver((mut, _) =>
            {
                called = true;
                Assert.AreEqual(1, mut.Length);
                Assert.AreEqual(attrName, mut[0].AttributeName);
                Assert.IsNull(mut[0].PreviousValue);
            });

            var document = Html("");

            observer.Connect(document.Body, attributes: true);

            document.Body.SetAttribute(attrName, attrValue);
            Assert.IsTrue(called);
        }

        [Test]
        public void ConnectMutationObserverAttributesTriggerManuallyFromDocument()
        {
            var called = false;
            var attrName = "something";
            var attrValue = "test";

            var observer = new MutationObserver((mut, _) =>
            {
                called = true;
                Assert.AreEqual(1, mut.Length);
                Assert.AreEqual(attrName, mut[0].AttributeName);
                Assert.IsNull(mut[0].PreviousValue);
            });

            var document = Html("");

            observer.Connect(document, attributes: true, subtree: true);

            document.Body.SetAttribute(attrName, attrValue);
            Assert.IsTrue(called);
        }

        [Test]
        public void ConnectMutationObserverMultipleAttributesDescendantTriggerManually()
        {
            var called1 = false;
            var called2 = false;
            var called3 = false;
            var attrName = "something";
            var attrValue = "test";

            var document = Html("");

            var observer1 = new MutationObserver((mut, _) =>
            {
                called1 = true;
                Assert.AreEqual(1, mut.Length);
            });

            observer1.Connect(document.DocumentElement, attributes: true, subtree: true);

            var observer2 = new MutationObserver((mut, _) =>
            {
                called2 = true;
                Assert.AreEqual(0, mut.Length);
            });

            observer2.Connect(document.DocumentElement, attributes: true, subtree: false);

            var observer3 = new MutationObserver((mut, _) =>
            {
                called3 = true;
                Assert.AreEqual(1, mut.Length);
            });

            observer3.Connect(document.Body, attributes: true);

            document.Body.SetAttribute(attrName, attrValue);
            Assert.IsTrue(called1);
            Assert.IsFalse(called2);
            Assert.IsTrue(called3);
        }

        [Test]
        public void ConnectMutationObserverTextWithDescendantsAndClearOldValueTriggerManually()
        {
            var called = false;
            var text = "something";
            var replaced = "different";

            var observer = new MutationObserver((mut, _) =>
            {
                called = true;
                Assert.AreEqual(1, mut.Length);
                Assert.IsNull(mut[0].PreviousValue);
                var tn = mut[0].Target as TextNode;
                Assert.IsNotNull(tn);
                Assert.AreEqual(text + replaced, tn.TextContent);
            });

            var document = Html("");

            observer.Connect(document.Body, characterData: true, subtree: true, characterDataOldValue: false);

            document.Body.TextContent = text;
            var textNode = document.Body.ChildNodes[0] as TextNode;
            textNode.Replace(text.Length, 0, replaced);
            Assert.IsTrue(called);
        }

        [Test]
        public void ConnectMutationObserverTextWithDescendantsAndExaminingOldValueTriggerManually()
        {
            var called = false;
            var text = "something";
            var replaced = "different";

            var observer = new MutationObserver((mut, _) =>
            {
                called = true;
                Assert.AreEqual(1, mut.Length);
                Assert.AreEqual(text, mut[0].PreviousValue);
                var tn = mut[0].Target as TextNode;
                Assert.IsNotNull(tn);
                Assert.AreEqual(text + replaced, tn.TextContent);
            });

            var document = Html("");

            observer.Connect(document.Body, characterData: true, subtree: true, characterDataOldValue: true);

            document.Body.TextContent = text;
            var textNode = document.Body.ChildNodes[0] as TextNode;
            textNode.Replace(text.Length, 0, replaced);
            Assert.IsTrue(called);
        }

        [Test]
        public void ConnectMutationObserverTextNoDescendantsTriggerManually()
        {
            var called = false;
            var text = "something";
            var replaced = "different";

            var observer = new MutationObserver((mut, _) =>
            {
                called = true;
                Assert.AreEqual(0, mut.Length);
            });

            var document = Html("");

            observer.Connect(document.Body, characterData: true, subtree: false);

            document.Body.TextContent = text;
            var textNode = document.Body.ChildNodes[0] as TextNode;
            textNode.Replace(text.Length, 0, replaced);
            Assert.IsFalse(called);
        }

        [Test]
        public void ConnectMutationObserverTextNoDescendantsButCreatedTriggerManually()
        {
            var called = false;
            var text = "something";

            var observer = new MutationObserver((mut, _) =>
            {
                called = true;
                Assert.AreEqual(1, mut.Length);
                Assert.AreEqual(1, mut[0].Added.Length);
                Assert.AreEqual(text, mut[0].Added[0].TextContent);
            });

            var document = Html("");

            observer.Connect(document.Body, subtree: false, childList: true);

            document.Body.TextContent = text;
            Assert.IsTrue(called);
        }

        [Test]
        public void MutationObserverAttr()
        {
            var document = Html("", true);

            var div = document.CreateElement("div");
            var observer = new MutationObserver((_, _) => { });
            observer.Connect(div, attributes: true);
            div.SetAttribute("a", "A");
            div.SetAttribute("a", "B");

            var records = observer.Flush().ToArray();
            Assert.AreEqual(records.Count(), 2);

            AssertRecord(records[0], new TestMutationRecord
            {
                Type = "attributes",
                Target = div,
                AttributeName = "a",
                AttributeNamespace = null
            });
            AssertRecord(records[1], new TestMutationRecord
            {
                Type = "attributes",
                Target = div,
                AttributeName = "a",
                AttributeNamespace = null
            });
        }

        [Test]
        public void MutationObserverAttrWithOldvalue()
        {
            var document = Html("", true);

            var div = document.CreateElement("div");
            var observer = new MutationObserver((_, _) => { });
            observer.Connect(div, attributes: true, attributeOldValue: true);
            div.SetAttribute("a", "A");
            div.SetAttribute("a", "B");

            var records = observer.Flush().ToArray();
            Assert.AreEqual(records.Count(), 2);

            AssertRecord(records[0], new TestMutationRecord
            {
                Type = "attributes",
                Target = div,
                AttributeName = "a",
                AttributeNamespace = null,
                PreviousValue = null
            });
            AssertRecord(records[1], new TestMutationRecord
            {
                Type = "attributes",
                Target = div,
                AttributeName = "a",
                AttributeNamespace = null,
                PreviousValue = "A"
            });
        }

        [Test]
        public void MutationObserverAttrChangeInSubtreeShouldNotGenereateARecord()
        {
            var document = Html("");

            var div = document.CreateElement("div");
            var child = document.CreateElement("div");
            div.AppendChild(child);

            var observer = new MutationObserver((_, _) => { });
            observer.Connect(div, attributes: true);
            child.SetAttribute("a", "A");
            child.SetAttribute("a", "B");

            var records = observer.Flush().ToArray();
            Assert.AreEqual(records.Count(), 0);
        }

        [Test]
        public void MutationObserverAttrChangeSubtree()
        {
            var document = Html("", true);

            var div = document.CreateElement("div");
            var child = document.CreateElement("div");
            div.AppendChild(child);

            var observer = new MutationObserver((_, _) => { });
            observer.Connect(div, attributes: true, subtree: true);
            child.SetAttribute("a", "A");
            child.SetAttribute("a", "B");

            var records = observer.Flush().ToArray();
            Assert.AreEqual(records.Count(), 2);

            AssertRecord(records[0], new TestMutationRecord
            {
                Type = "attributes",
                Target = child,
                AttributeName = "a"
            });
            AssertRecord(records[1], new TestMutationRecord
            {
                Type = "attributes",
                Target = child,
                AttributeName = "a"
            });
        }

        [Test]
        public void MutationObserverMultipleObserversOnSameTarget()
        {
            var document = Html("", true);
            var div = document.CreateElement("div");

            var observer1 = new MutationObserver((_, _) => { });
            observer1.Connect(div, attributes: true, attributeOldValue: true);

            var observer2 = new MutationObserver((_, _) => { });
            observer2.Connect(div, attributes: true, attributeFilter: new []{ "b" });
            div.SetAttribute("a", "A");
            div.SetAttribute("a", "A2");
            div.SetAttribute("b", "B");

            var records = observer1.Flush().ToArray();
            Assert.AreEqual(records.Count(), 3);

            AssertRecord(records[0], new TestMutationRecord
            {
                Type = "attributes",
                Target = div,
                AttributeName = "a"
            });
            AssertRecord(records[1], new TestMutationRecord
            {
                Type = "attributes",
                Target = div,
                AttributeName = "a",
                PreviousValue = "A"
            });
            AssertRecord(records[2], new TestMutationRecord
            {
                Type = "attributes",
                Target = div,
                AttributeName = "b"
            });

            records = observer2.Flush().ToArray();
            Assert.AreEqual(1, records.Count());

            AssertRecord(records[0], new TestMutationRecord
            {
                Type = "attributes",
                Target = div,
                AttributeName = "b"
            });
        }

        [Test]
        public void MutationObserverObserverObservesOnDifferentTarget()
        {
            var document = Html("", true);
            var div = document.CreateElement("div");
            var child = document.CreateElement("div");
            div.AppendChild(child);

            var observer = new MutationObserver((_, _) => { });
            observer.Connect(child, attributes: true);
            observer.Connect(div, attributes: true, subtree: true, attributeOldValue: true);

            child.SetAttribute("a", "A");
            child.SetAttribute("a", "A2");
            child.SetAttribute("b", "B");

            var records = observer.Flush().ToArray();
            Assert.AreEqual(3, records.Count());

            AssertRecord(records[0], new TestMutationRecord
            {
                Type = "attributes",
                Target = child,
                AttributeName = "a"
            });
            AssertRecord(records[1], new TestMutationRecord
            {
                Type = "attributes",
                Target = child,
                AttributeName = "a",
                PreviousValue = "A"
            });
            AssertRecord(records[2], new TestMutationRecord
            {
                Type = "attributes",
                Target = child,
                AttributeName = "b"
            });
        }

        [Test]
        public void MutationObserverObservingOnTheSameNodeShouldUpdateTheOptions()
        {
            var document = Html("", true);
            var div = document.CreateElement("div");
            var observer = new MutationObserver((_, _) => { });
            observer.Connect(div, attributes: true, attributeFilter: new[] { "a" });
            observer.Connect(div, attributes: true, attributeFilter: new[] { "b" });

            div.SetAttribute("a", "A");
            div.SetAttribute("b", "B");

            var records = observer.Flush().ToArray();
            Assert.AreEqual(records.Count(), 1);

            AssertRecord(records[0], new TestMutationRecord
            {
                Type = "attributes",
                Target = div,
                AttributeName = "b"
            });
        }

        [Test]
        public void MutationObserverDisconnectShouldStopAllEventsAndEmptyTheRecords()
        {
            var document = Html("");
            var div = document.CreateElement("div");
            var observer = new MutationObserver((_, _) => { });
            observer.Connect(div, attributes: true);

            div.SetAttribute("a", "A");
            observer.Disconnect();
            var records = observer.Flush().ToArray();
            Assert.AreEqual(records.Count(), 0);

            div.SetAttribute("b", "B");
            records = observer.Flush().ToArray();
            Assert.AreEqual(records.Count(), 0);
        }

        [Test]
        public void MutationObserverDisconnectShouldNotAffectOtherObservers()
        {
            var document = Html("", true);
            var div = document.CreateElement("div");
            var observer1 = new MutationObserver((_, _) => { });
            observer1.Connect(div, attributes: true);
            var observer2 = new MutationObserver((_, _) => { });
            observer2.Connect(div, attributes: true);

            div.SetAttribute("a", "A");

            observer1.Disconnect();
            var records1 = observer1.Flush().ToArray();
            Assert.AreEqual(records1.Count(), 0);

            var records2 = observer2.Flush().ToArray();
            Assert.AreEqual(records2.Count(), 1);
            AssertRecord(records2[0], new TestMutationRecord
            {
                Type = "attributes",
                Target = div,
                AttributeName = "a"
            });

            div.SetAttribute("b", "B");

            records1 = observer1.Flush().ToArray();
            Assert.AreEqual(records1.Count(), 0);

            records2 = observer2.Flush().ToArray();
            Assert.AreEqual(records2.Count(), 1);
            AssertRecord(records2[0], new TestMutationRecord
            {
                Type = "attributes",
                Target = div,
                AttributeName = "b"
            });
        }

        [Test]
        public async Task MutationObserverOneObserverTwoAttributeChanges()
        {
            var document = Html("", true);
            var div = document.CreateElement("div");
            var tcs = new TaskCompletionSource<Boolean>();
            var observer = new MutationObserver((records, _) =>
            {
                Assert.AreEqual(records.Count(), 2);

                AssertRecord(records[0], new TestMutationRecord
                {
                    Type = "attributes",
                    Target = div,
                    AttributeName = "a",
                    AttributeNamespace = null
                });
                AssertRecord(records[1], new TestMutationRecord
                {
                    Type = "attributes",
                    Target = div,
                    AttributeName = "a",
                    AttributeNamespace = null
                });

                tcs.SetResult(true);
            });

            observer.Connect(div, attributes: true);

            div.SetAttribute("a", "A");
            div.SetAttribute("a", "B");
            await tcs.Task.ConfigureAwait(false);
        }

        [Test]
        public async Task MutationObserverNestedChanges()
        {
            var document = Html("", true);
            var div = document.CreateElement("div");
            var fresh = true;
            var tcs = new TaskCompletionSource<Boolean>();
            var observer = new MutationObserver((records, _) =>
            {
                Assert.AreEqual(records.Count(), 1);

                if (fresh)
                {
                    AssertRecord(records[0], new TestMutationRecord
                    {
                        Type = "attributes",
                        Target = div,
                        AttributeName = "a",
                        AttributeNamespace = null
                    });
                    div.SetAttribute("b", "B");
                    fresh = false;
                }
                else
                {
                    AssertRecord(records[0], new TestMutationRecord
                    {
                        Type = "attributes",
                        Target = div,
                        AttributeName = "b",
                        AttributeNamespace = null
                    });
                    tcs.SetResult(true);
                }
            });

            observer.Connect(div, attributes: true);

            div.SetAttribute("a", "A");
            await tcs.Task.ConfigureAwait(false);
        }

        [Test]
        public void MutationObserverCharacterdata()
        {
            var document = Html("", true);

            var text = document.CreateTextNode("abc");
            var observer = new MutationObserver((_, _) => { });
            observer.Connect(text, characterData: true);
            text.TextContent = "def";
            text.TextContent = "ghi";

            var records = observer.Flush().ToArray();
            Assert.AreEqual(records.Count(), 2);

            AssertRecord(records[0], new TestMutationRecord
            {
                Type = "characterData",
                Target = text
            });
            AssertRecord(records[1], new TestMutationRecord
            {
                Type = "characterData",
                Target = text
            });
        }

        [Test]
        public void MutationObserverCharacterdataWithOldValue()
        {
            var document = Html("", true);
            var testDiv = document.Body.AppendChild(document.CreateElement("div"));
            var text = testDiv.AppendChild(document.CreateTextNode("abc"));
            var observer = new MutationObserver((_, _) => { });
            observer.Connect(text, characterData: true, characterDataOldValue: true);
            text.TextContent = "def";
            text.TextContent = "ghi";

            var records = observer.Flush().ToArray();
            Assert.AreEqual(records.Count(), 2);

            AssertRecord(records[0], new TestMutationRecord
            {
                Type = "characterData",
                Target = text,
                PreviousValue = "abc"
            });
            AssertRecord(records[1], new TestMutationRecord
            {
                Type = "characterData",
                Target = text,
                PreviousValue = "def"
            });
        }

        [Test]
        public void MutationObserverCharacterdataChangeInSubtreeShouldNotGenerateARecord()
        {
            var document = Html("", true);
            var div = document.CreateElement("div");
            var text = div.AppendChild(document.CreateTextNode("abc"));
            var observer = new MutationObserver((_, _) => { });
            observer.Connect(div, characterData: true);
            text.TextContent = "def";
            text.TextContent = "ghi";

            var records = observer.Flush().ToArray();
            Assert.AreEqual(records.Count(), 0);
        }

        [Test]
        public void MutationObserverCharacterdataChangeInSubtree()
        {
            var document = Html("", true);
            var div = document.CreateElement("div");
            var text = div.AppendChild(document.CreateTextNode("abc"));
            var observer = new MutationObserver((_, _) => { });
            observer.Connect(div, characterData: true, subtree: true);
            text.TextContent = "def";
            text.TextContent = "ghi";

            var records = observer.Flush().ToArray();
            Assert.AreEqual(records.Count(), 2);

            AssertRecord(records[0], new TestMutationRecord
            {
                Type = "characterData",
                Target = text
            });
            AssertRecord(records[1], new TestMutationRecord
            {
                Type = "characterData",
                Target = text
            });
        }

        [Test]
        public void MutationObserverAppendChild()
        {
            var document = Html("", true);
            var div = document.CreateElement("div");
            var observer = new MutationObserver((_, _) => { });
            observer.Connect(div, childList: true);
            var a = document.CreateElement("a");
            var b = document.CreateElement("b");

            div.AppendChild(a);
            div.AppendChild(b);

            var records = observer.Flush().ToArray();
            Assert.AreEqual(records.Count(), 2);

            AssertRecord(records[0], new TestMutationRecord
            {
                Type = "childList",
                Target = div,
                Added = ToNodeList(a)
            });

            AssertRecord(records[1], new TestMutationRecord
            {
                Type = "childList",
                Target = div,
                Added = ToNodeList(b),
                PreviousSibling = a
            });

        }

        [Test]
        public void MutationObserverInsertBefore()
        {
            var document = Html("", true);
            var div = document.CreateElement("div");
            var a = document.CreateElement("a");
            var b = document.CreateElement("b");
            var c = document.CreateElement("c");
            div.AppendChild(a);

            var observer = new MutationObserver((_, _) => { });
            observer.Connect(div, childList: true);

            div.InsertBefore(b, a);
            div.InsertBefore(c, a);

            var records = observer.Flush().ToArray();
            Assert.AreEqual(records.Count(), 2);

            AssertRecord(records[0], new TestMutationRecord
            {
                Type = "childList",
                Target = div,
                Added = ToNodeList(b),
                NextSibling = a
            });

            AssertRecord(records[1], new TestMutationRecord
            {
                Type = "childList",
                Target = div,
                Added = ToNodeList(c),
                NextSibling = a,
                PreviousSibling = b
            });
        }

        [Test]
        public void MutationObserverRemovechild()
        {
            var document = Html("", true);
            var testDiv = document.Body.AppendChild(document.CreateElement("div"));
            var div = testDiv.AppendChild(document.CreateElement("div"));
            var a = div.AppendChild(document.CreateElement("a"));
            var b = div.AppendChild(document.CreateElement("b"));
            var c = div.AppendChild(document.CreateElement("c"));

            var observer = new MutationObserver((_, _) => { });
            observer.Connect(div, childList: true);

            div.RemoveChild(b);
            div.RemoveChild(a);

            var records = observer.Flush().ToArray();
            Assert.AreEqual(records.Count(), 2);

            AssertRecord(records[0], new TestMutationRecord
            {
                Type = "childList",
                Target = div,
                Removed = ToNodeList(b),
                NextSibling = c,
                PreviousSibling = a
            });

            AssertRecord(records[1], new TestMutationRecord
            {
                Type = "childList",
                Target = div,
                Removed = ToNodeList(a),
                NextSibling = c
            });
        }

        [Test]
        public void MutationObserverDirectChildren()
        {
            var document = Html("", true);
            var testDiv = document.Body.AppendChild(document.CreateElement("div"));
            var div = testDiv.AppendChild(document.CreateElement("div"));
            var observer = new MutationObserver((_, _) => { });
            observer.Connect(div, childList: true);
            var a = document.CreateElement("a");
            var b = document.CreateElement("b");

            div.AppendChild(a);
            div.InsertBefore(b, a);
            div.RemoveChild(b);

            var records = observer.Flush().ToArray();
            Assert.AreEqual(records.Count(), 3);

            AssertRecord(records[0], new TestMutationRecord
            {
                Type = "childList",
                Target = div,
                Added = ToNodeList(a)
            });

            AssertRecord(records[1], new TestMutationRecord
            {
                Type = "childList",
                Target = div,
                NextSibling = a,
                Added = ToNodeList(b)
            });

            AssertRecord(records[2], new TestMutationRecord
            {
                Type = "childList",
                Target = div,
                NextSibling = a,
                Removed = ToNodeList(b)
            });
        }

        [Test]
        public void MutationObserverSubtree()
        {
            var document = Html("", true);
            var div = document.CreateElement("div");
            var child = div.AppendChild(document.CreateElement("div"));
            var observer = new MutationObserver((_, _) => { });
            observer.Connect(child, childList: true);
            var a = document.CreateTextNode("a");
            var b = document.CreateTextNode("b");

            child.AppendChild(a);
            child.InsertBefore(b, a);
            child.RemoveChild(b);

            var records = observer.Flush().ToArray();
            Assert.AreEqual(records.Count(), 3);

            AssertRecord(records[0], new TestMutationRecord
            {
                Type = "childList",
                Target = child,
                Added = ToNodeList(a)
            });

            AssertRecord(records[1], new TestMutationRecord
            {
                Type = "childList",
                Target = child,
                NextSibling = a,
                Added = ToNodeList(b)
            });

            AssertRecord(records[2], new TestMutationRecord
            {
                Type = "childList",
                Target = child,
                NextSibling = a,
                Removed = ToNodeList(b)
            });
        }

        [Test]
        public void MutationObserverBothDirectAndSubtree()
        {
            var document = Html("", true);
            var div = document.CreateElement("div");
            var child = div.AppendChild(document.CreateElement("div"));
            var observer = new MutationObserver((_, _) => { });
            observer.Connect(div, childList: true, subtree: true);
            observer.Connect(child, childList: true);

            var a = document.CreateTextNode("a");
            var b = document.CreateTextNode("b");

            child.AppendChild(a);
            div.AppendChild(b);

            var records = observer.Flush().ToArray();
            Assert.AreEqual(records.Count(), 2);

            AssertRecord(records[0], new TestMutationRecord
            {
                Type = "childList",
                Target = child,
                Added = ToNodeList(a)
            });

            AssertRecord(records[1], new TestMutationRecord
            {
                Type = "childList",
                Target = div,
                Added = ToNodeList(b),
                PreviousSibling = child
            });
        }

        [Test]
        public void MutationObserverAppendMultipleAtOnceAtTheEnd()
        {
            var document = Html("", true);
            var testDiv = document.Body.AppendChild(document.CreateElement("div"));
            var div = testDiv.AppendChild(document.CreateElement("div"));
            div.AppendChild(document.CreateTextNode("a"));

            var observer = new MutationObserver((_, _) => { });
            observer.Connect(div, childList: true);

            var df = document.CreateDocumentFragment();
            var b = df.AppendChild(document.CreateTextNode("b"));
            var c = df.AppendChild(document.CreateTextNode("c"));
            var d = df.AppendChild(document.CreateTextNode("d"));

            div.AppendChild(df);

            var records = observer.Flush().ToArray();
            var merged = MergeRecords(records);

            AssertArrayEqual(merged.Item1, ToNodeList(b, c, d));
            AssertArrayEqual(merged.Item2, ToNodeList());
            AssertAll(records, new TestMutationRecord
            {
                Type = "childList",
			    Target = div
            });
        }

        [Test]
        public void MutationObserverAppendMultipleAtOnceAtTheFront()
        {
            var document = Html("", true);
            var testDiv = document.Body.AppendChild(document.CreateElement("div"));
            var div = testDiv.AppendChild(document.CreateElement("div"));
            var a = div.AppendChild(document.CreateTextNode("a"));

            var observer = new MutationObserver((_, _) => { });
            observer.Connect(div, childList: true);

            var df = document.CreateDocumentFragment();
            var b = df.AppendChild(document.CreateTextNode("b"));
            var c = df.AppendChild(document.CreateTextNode("c"));
            var d = df.AppendChild(document.CreateTextNode("d"));

            div.InsertBefore(df, a);

            var records = observer.Flush().ToArray();
            var merged = MergeRecords(records);

            AssertArrayEqual(merged.Item1, ToNodeList(b, c, d));
            AssertArrayEqual(merged.Item2, ToNodeList());
            AssertAll(records, new TestMutationRecord
            {
                Type = "childList",
			    Target = div
            });
        }

        [Test]
        public void MutationObserverAppendMultipleAtOnceInTheMiddle()
        {
            var document = Html("", true);
            var testDiv = document.Body.AppendChild(document.CreateElement("div"));
            var div = document.CreateElement("div");
            testDiv.AppendChild(div);
            div.AppendChild(document.CreateTextNode("a"));
            var b = div.AppendChild(document.CreateTextNode("b"));

            var observer = new MutationObserver((_, _) => { });
            observer.Connect(div, childList: true);

            var df = document.CreateDocumentFragment();
            var c = df.AppendChild(document.CreateTextNode("c"));
            var d = df.AppendChild(document.CreateTextNode("d"));

            div.InsertBefore(df, b);

            var records = observer.Flush().ToArray();
            var merged = MergeRecords(records);

            AssertArrayEqual(merged.Item1, ToNodeList(c, d));
            AssertArrayEqual(merged.Item2, ToNodeList());
            AssertAll(records, new TestMutationRecord
            {
                Type = "childList",
			    Target = div
            });
        }

        [Test]
        public void MutationObserverRemoveAllChildren()
        {
            var document = Html("", true);
            var testDiv = document.Body.AppendChild(document.CreateElement("div"));
            var div = document.CreateElement("div");
            testDiv.AppendChild(div);
            var a = div.AppendChild(document.CreateTextNode("a"));
            var b = div.AppendChild(document.CreateTextNode("b"));
            var c = div.AppendChild(document.CreateTextNode("c"));

            var observer = new MutationObserver((_, _) => { });
            observer.Connect(div, childList: true);

            div.InnerHtml = "";

            var records = observer.Flush().ToArray();
            var merged = MergeRecords(records);

            AssertArrayEqual(merged.Item1, ToNodeList());
            AssertArrayEqual(merged.Item2, ToNodeList(a, b, c));
            AssertAll(records, new TestMutationRecord
            {
                Type = "childList",
			    Target = div
            });
        }

        [Test]
        public void MutationObserverReplaceAllChildrenUsingInnerhtml()
        {
            var document = Html("", true);
            var testDiv = document.Body.AppendChild(document.CreateElement("div"));
            var div = document.CreateElement("div");
            testDiv.AppendChild(div);
            var a = div.AppendChild(document.CreateTextNode("a"));
            var b = div.AppendChild(document.CreateTextNode("b"));

            var observer = new MutationObserver((_, _) => { });
            observer.Connect(div, childList: true);

            div.InnerHtml = "<c></c><d></d>";

            var c = div.FirstChild;
            var d = div.LastChild;
            var records = observer.Flush().ToArray();
            var merged = MergeRecords(records);

            AssertArrayEqual(merged.Item1, ToNodeList(c, d));
            AssertArrayEqual(merged.Item2, ToNodeList(a, b));
            AssertAll(records, new TestMutationRecord
            {
                Type = "childList",
			    Target = div
            });
        }

        [Test]
        public void MutationObserverAttrAndCharacterdata()
        {
            var document = Html("", true);
            var div = document.CreateElement("div");
            div.AppendChild(document.CreateTextNode("text"));
            var observer = new MutationObserver((_, _) => { });
            observer.Connect(div, attributes: true, subtree: true, characterData: true);

            div.SetAttribute("a", "A");
            div.FirstChild.TextContent = "changed";

            var records = observer.Flush().ToArray();
            Assert.AreEqual(records.Count(), 2);

            AssertRecord(records[0], new TestMutationRecord
            {
                Type = "attributes",
                Target = div,
                AttributeName = "a",
                AttributeNamespace = null
            });
            AssertRecord(records[1], new TestMutationRecord
            {
                Type = "characterData",
                Target = div.FirstChild
            });
        }

        [Test]
        public void MutationObserverAttrChanged()
        {
            var document = Html("", true);
            var testDiv = document.Body.AppendChild(document.CreateElement("div"));
            var div = testDiv.AppendChild(document.CreateElement("div"));
            var child = document.CreateElement("div");
            div.AppendChild(child);
            var observer = new MutationObserver((_, _) => { });
            observer.Connect(div, attributes: true, subtree: true);
            div.RemoveChild(child);
            child.SetAttribute("a", "A");

            var records = observer.Flush().ToArray();
            Assert.AreEqual(1, records.Count());

            AssertRecord(records[0], new TestMutationRecord
            {
                Type = "attributes",
                Target = child,
                AttributeName = "a",
                AttributeNamespace = null
            });

            child.SetAttribute("b", "B");
            records = observer.Flush().ToArray();
            Assert.AreEqual(records.Count(), 1);

            AssertRecord(records[0], new TestMutationRecord
            {
                Type = "attributes",
                Target = child,
                AttributeName = "b",
                AttributeNamespace = null
            });
        }

        [Test]
        public void MutationObserverAttrCallback()
        {
            var document = Html("");
            var testDiv = document.Body.AppendChild(document.CreateElement("div"));
            var div = testDiv.AppendChild(document.CreateElement("div"));
            var child = document.CreateElement("div");
            div.AppendChild(child);
            var i = 0;
            var observer = new MutationObserver((records, obs) =>
            {
                Assert.LessOrEqual(++i, 2);
                Assert.AreEqual(1, records.Count());

                AssertRecord(records[0], new TestMutationRecord
                {
                    Type = "attributes",
                    Target = child,
                    AttributeName = "a",
                    AttributeNamespace = null
                });

                // The transient observers are removed before the callback is called.
                child.SetAttribute("b", "B");
                records = obs.Flush().ToArray();
                Assert.AreEqual(0, records.Count());
            });

            observer.Connect(div, attributes: true, subtree: true);

            div.RemoveChild(child);
            child.SetAttribute("a", "A");
            observer.Trigger();
        }

        [Test]
        public void MutationObserverAttrMakeSureTransientGetsRemoved()
        {
            var document = Html("");
            var testDiv = document.Body.AppendChild(document.CreateElement("div"));
            var div = testDiv.AppendChild(document.CreateElement("div"));
            var child = document.CreateElement("div");
            div.AppendChild(child);
            var i = 0;
            var observer = new MutationObserver((records, _) =>
            {
                Assert.AreNotEqual(2, ++i);
                Assert.AreEqual(records.Count(), 1);

                AssertRecord(records[0], new TestMutationRecord
                {
                    Type = "attributes",
                    Target = child,
                    AttributeName = "a",
                    AttributeNamespace = null
                });
            });

            observer.Connect(div, subtree: true, attributes: true);

            div.RemoveChild(child);
            child.SetAttribute("a", "A");
            observer.Trigger();

            var div2 = document.CreateElement("div");
            var observer2 = new MutationObserver((records, _) =>
            {
                Assert.LessOrEqual(++i, 3);
                Assert.AreEqual(records.Count(), 1);

                AssertRecord(records[0], new TestMutationRecord
                {
                    Type = "attributes",
                    Target = child,
                    AttributeName = "b",
                    AttributeNamespace = null
                });
            });

            observer2.Connect(div2, attributes: true, subtree: true);

            div2.AppendChild(child);
            child.SetAttribute("b", "B");
            observer2.Trigger();
        }

        [Test]
        public void MutationObserverChildListCharacterdata()
        {
            var document = Html("", true);
            var div = document.CreateElement("div");
            var child = div.AppendChild(document.CreateTextNode("text"));
            var observer = new MutationObserver((_, _) => { });
            observer.Connect(div, characterData: true, subtree: true);
            div.RemoveChild(child);
            child.TextContent = "changed";

            var records = observer.Flush().ToArray();
            Assert.AreEqual(records.Count(), 1);

            AssertRecord(records[0], new TestMutationRecord
            {
                Type = "characterData",
                Target = child
            });

            child.TextContent += " again";

            records = observer.Flush().ToArray();
            Assert.AreEqual(records.Count(), 1);

            AssertRecord(records[0], new TestMutationRecord
            {
                Type = "characterData",
                Target = child
            });
        }

        [Test]
        public void MutationObserverCharacterdataCallback()
        {
            var document = Html("");
            var div = document.CreateElement("div");
            var child = div.AppendChild(document.CreateTextNode("text"));
            var i = 0;
            var observer = new MutationObserver((records, obs) =>
            {
                Assert.LessOrEqual(++i, 2);
                Assert.AreEqual(1, records.Count());

                AssertRecord(records[0], new TestMutationRecord
                {
                    Type = "characterData",
                    Target = child
                });

                // The transient observers are removed before the callback is called.
                child.TextContent += " again";
                records = obs.Flush().ToArray();
                Assert.AreEqual(0, records.Count());
            });
            observer.Connect(div, characterData: true, subtree: true);
            div.RemoveChild(child);
            child.TextContent = "changed";
            observer.Trigger();
        }

        [Test]
        public void MutationObserverChildlist()
        {
            var document = Html("", true);
            var testDiv = document.Body.AppendChild(document.CreateElement("div"));
            var div = testDiv.AppendChild(document.CreateElement("div"));
            var child = div.AppendChild(document.CreateElement("div"));
            var observer = new MutationObserver((_, _) => { });
            observer.Connect(div, childList: true, subtree: true);
            div.RemoveChild(child);
            var grandChild = child.AppendChild(document.CreateElement("span"));

            var records = observer.Flush().ToArray();
            Assert.AreEqual(records.Count(), 2);

            AssertRecord(records[0], new TestMutationRecord
            {
                Type = "childList",
                Target = div,
                Removed = ToNodeList(child)
            });

            AssertRecord(records[1], new TestMutationRecord
            {
                Type = "childList",
                Target = child,
                Added = ToNodeList(grandChild)
            });

            child.RemoveChild(grandChild);

            records = observer.Flush().ToArray();
            Assert.AreEqual(records.Count(), 1);

            AssertRecord(records[0], new TestMutationRecord
            {
                Type = "childList",
                Target = child,
                Removed = ToNodeList(grandChild)
            });
        }

        [Test]
        public async Task MutationObserverChildlistCallback()
        {
            var document = Html("", true);
            var testDiv = document.Body.AppendChild(document.CreateElement("div"));
            var div = testDiv.AppendChild(document.CreateElement("div"));
            var child = div.AppendChild(document.CreateElement("div"));
            var grandChild = document.CreateElement("span");
            var i = 0;
            var tcs = new TaskCompletionSource<Boolean>();
            var observer = new MutationObserver((records, obs) =>
            {
                Assert.LessOrEqual(++i, 2);
                Assert.AreEqual(2, records.Count());

                AssertRecord(records[0], new TestMutationRecord
                {
                    Type = "childList",
                    Target = div,
                    Removed = ToNodeList(child)
                });

                AssertRecord(records[1], new TestMutationRecord
                {
                    Type = "childList",
                    Target = child,
                    Added = ToNodeList(grandChild)
                });

                // The transient observers are removed before the callback is called.
                child.RemoveChild(grandChild);

                records = obs.Flush().ToArray();
                Assert.AreEqual(0, records.Count());
                tcs.SetResult(true);
            });
            observer.Connect(div, childList: true, subtree: true);
            div.RemoveChild(child);
            child.AppendChild(grandChild);
            observer.Trigger();
            await tcs.Task.ConfigureAwait(false);
        }

        private static Tuple<NodeList, NodeList> MergeRecords(IMutationRecord[] records)
        {
            var added = new NodeList();
            var removed = new NodeList();

            foreach (var record in records)
            {
                if (record.Added != null)
                {
                    added.AddRange((NodeList)record.Added);
                }

                if (record.Removed != null)
                {
                    removed.AddRange((NodeList)record.Removed);
                }
            }

            return Tuple.Create(added, removed);
        }

        private static void AssertArrayEqual(INodeList actual, INodeList expected)
        {
            Assert.AreEqual(expected.Length, actual.Length);

            for (int i = 0; i < expected.Length; i++)
            {
                Assert.AreSame(expected[i], actual[i]);
            }
        }

        private static void AssertAll(IMutationRecord[] actualRecords, TestMutationRecord expected)
        {
            foreach (var actualRecord in actualRecords)
            {
                Assert.AreEqual(expected.Type, actualRecord.Type);
                Assert.AreEqual(expected.Target, actualRecord.Target);
            }
        }

        private static void AssertRecord(IMutationRecord actual, TestMutationRecord expected)
        {
            Assert.AreEqual(expected.AttributeName, actual.AttributeName);
            Assert.AreEqual(expected.AttributeNamespace, actual.AttributeNamespace);
            Assert.AreEqual(expected.NextSibling, actual.NextSibling);
            Assert.AreEqual(expected.PreviousSibling, actual.PreviousSibling);
            Assert.AreEqual(expected.PreviousValue, actual.PreviousValue);
            Assert.AreEqual(expected.Type, actual.Type);
            Assert.AreEqual(expected.Target, actual.Target);
        }

        private static INodeList ToNodeList(params INode[] nodes)
        {
            var list = new NodeList();

            foreach (var node in nodes)
            {
                list.Add((Node)node);
            }

            return list;
        }

        private class TestMutationRecord : IMutationRecord
        {
            public INodeList Added
            {
                get;
                set;
            }

            public string AttributeName
            {
                get;
                set;
            }

            public string AttributeNamespace
            {
                get;
                set;
            }

            public INode NextSibling
            {
                get;
                set;
            }

            public INode PreviousSibling
            {
                get;
                set;
            }

            public string PreviousValue
            {
                get;
                set;
            }

            public INodeList Removed
            {
                get;
                set;
            }

            public INode Target
            {
                get;
                set;
            }

            public string Type
            {
                get;
                set;
            }
        }
    }
}
