#include "BasicRenderer.h"
#include "cstr.h"
#include "cmath.h"

BasicRenderer* GlobalRenderer;

BasicRenderer::BasicRenderer(Framebuffer* targetFramebuffer, PSF1_FONT* psf1_Font){
    TargetFramebuffer = targetFramebuffer;
    PSF1_Font = psf1_Font;
    Color = 0xFFFFFFFF;
    ClearColor = 0x00000000;
    CursorPosition = {0,0};
}

uint32_t BasicRenderer::GetPix(uint32_t x, uint32_t y){
    return *(uint32_t*)((uint64_t)TargetFramebuffer->BaseAddress + (x*4) + (y * TargetFramebuffer->PixelsPerScanline * 4));
}

void BasicRenderer::PutPix(uint32_t x, uint32_t y, uint32_t color){
    *(uint32_t*)((uint64_t)TargetFramebuffer->BaseAddress + (x*4) + (y * TargetFramebuffer->PixelsPerScanline * 4)) = color;
}

void BasicRenderer::DrawRectangle(uint32_t width, uint32_t height, uint32_t x, uint32_t y, uint32_t color){
    for(int w=x; w<=x+width; ++w){
        for(int h=y; h<=y+height; ++h){
            PutPix(w, h, color);
        }
    }
}
void BasicRenderer::DrawLine(uint32_t x1, uint32_t y1, uint32_t x2, uint32_t y2, uint32_t color)
{
    double m = (x2 - x1) / (y2 - y1);
    for(int x = x1; x <= x2; ++x)
        PutPix(x, y1 + (uint32_t)round(m * (double) x), color);
}

void BasicRenderer::ClearMouseCursor(uint8_t* mouseCursor, Point position){
    if(!MouseDrawn) return;

    int xMax = 16;
    int yMax = 16;
    int differenceX = TargetFramebuffer->Width - position.X;
    int differenceY = TargetFramebuffer->Height - position.Y;

    if(differenceX < 16) xMax = differenceX;
    if(differenceY < 16) yMax = differenceY;

    for(int y = 0; y < yMax; ++y){
        for(int x = 0; x < xMax; ++x){
            int bit = y * 16 + x;
            int byte = bit / 8;
            if((mouseCursor[byte] & (0b10000000 >> (x % 8)))){
                if(GetPix(position.X + x, position.Y + y) == MouseCursorBufferAfter[x+y*16]){
                    PutPix(position.X + x, position.Y + y, MouseCursorBuffer[x+y*16]);
                }
            }
        }
    }
}

void BasicRenderer::DrawOverlayMouseCursor(uint8_t* mouseCursor, Point position, uint32_t color){
    int xMax = 16;
    int yMax = 16;
    int differenceX = TargetFramebuffer->Width - position.X;
    int differenceY = TargetFramebuffer->Height - position.Y;

    if(differenceX < 16) xMax = differenceX;
    if(differenceY < 16) yMax = differenceY;

    for(int y = 0; y < yMax; ++y){
        for(int x = 0; x < xMax; ++x){
            int bit = y * 16 + x;
            int byte = bit / 8;
            if((mouseCursor[byte] & (0b10000000 >> (x % 8)))){
                MouseCursorBuffer[x+y*16] = GetPix(position.X + x, position.Y + y);
                PutPix(position.X + x, position.Y + y, color);
                MouseCursorBufferAfter[x+y*16] = GetPix(position.X + x, position.Y + y);
            }
        }
    }
    MouseDrawn = true;
}

void BasicRenderer::Print(const char* str){
    char* chr = (char*)str;
    while(*chr != 0){
        PutChar(*chr, CursorPosition.X, CursorPosition.Y);
        CursorPosition.X+=8;
        if(CursorPosition.X + 8 > TargetFramebuffer->Width){
            CursorPosition.X = 0;
            CursorPosition.Y += 16;
        }
        chr++;
    }
}

void BasicRenderer::Next(){
    CursorPosition.X = 0;
    CursorPosition.Y += 16;
}

void BasicRenderer::Println(const char* str){
    Print(str);
    Next();
}

void BasicRenderer::Clear(){
    uint64_t fbBase = (uint64_t)TargetFramebuffer->BaseAddress;
    uint64_t bytesPerScanline = TargetFramebuffer->PixelsPerScanline * 4;
    uint64_t fbHeight = TargetFramebuffer->Height;
    for(int verticalScanline = 0; verticalScanline < fbHeight; ++verticalScanline){
        uint64_t pixPtrBase = fbBase + (bytesPerScanline * verticalScanline);
        for(uint32_t* pixPtr = (uint32_t*)pixPtrBase; pixPtr < (uint32_t*)(pixPtrBase + bytesPerScanline); ++pixPtr){
            *pixPtr = ClearColor;
        }
    }
}

void BasicRenderer::ClearChar(){
    if(CursorPosition.X == 0){
        CursorPosition.X = TargetFramebuffer->Width;
        CursorPosition.Y -= 16;
        if(CursorPosition.Y < 0) CursorPosition.Y = 0;
    }

    unsigned int xOff = CursorPosition.X;
    unsigned int yOff = CursorPosition.Y;

    unsigned int* pixPtr = (unsigned int*)TargetFramebuffer->BaseAddress;
    for(unsigned long y = yOff; y < yOff + 16; ++y){
        for(unsigned long x = xOff - 8; x < xOff; ++x){
            *(unsigned int*)(pixPtr + x + (y * TargetFramebuffer->PixelsPerScanline)) = ClearColor;

        }
    }

    CursorPosition.X -= 8;
    if(CursorPosition.X < 0){
        CursorPosition.X = TargetFramebuffer->Width;
        CursorPosition.Y -= 16;
        if(CursorPosition.Y < 0) CursorPosition.Y = 0;
    }
}

void BasicRenderer::PutChar(char chr, unsigned int xOff, unsigned int yOff){
    unsigned int* pixPtr = (unsigned int*)TargetFramebuffer->BaseAddress;
    char* fontPtr = (char*)PSF1_Font->glyphBuffer + (chr * PSF1_Font->psf1_Header->charsize);
    for(unsigned long y = yOff; y < yOff + 16; ++y){
        for(unsigned long x = xOff; x < xOff+8; ++x){
            if((*fontPtr & (0b10000000 >> (x - xOff))) > 0){
                *(unsigned int*)(pixPtr + x + (y * TargetFramebuffer->PixelsPerScanline)) = Color;
            }
        }
        fontPtr++;
    }
}

void BasicRenderer::PutChar(char chr){
    PutChar(chr, CursorPosition.X, GlobalRenderer->CursorPosition.Y);
    CursorPosition.X += 8;
    if(CursorPosition.X + 8 > TargetFramebuffer->Width){
        CursorPosition = {0, CursorPosition.Y+16};
    }
}