#include <stdlib.h>
#include <stdint.h>
#include <string.h>
#include <stdio.h>

#ifdef _WIN32
	#include <windows.h>
#else
	#include <unistd.h>
	#include <sys/types.h>
	#include <sys/shm.h>

	#ifdef __APPLE__
		#include <semaphore.h>
		#include <mach/mach_time.h>
	#else
		#include <sys/ipc.h>
		#include <sys/sem.h>
		#include <sys/time.h>
		#include <sys/sysinfo.h>
	#endif
#endif

#include "flshm.h"


uint32_t flshm_amf0_read_string(flshm_amf0_string * str, const char * src, size_t max) {
	// Bounds check the header.
	if (max < 3) {
		return false;
	}

	// Check the type marker.
	if (*src != '\x02') {
		return false;
	}
	src++;

	// Read the string size, big endian.
	uint16_t size =
		((*((uint8_t *)src + 1)     ) & 0xFF  ) |
		((*((uint8_t *)src    ) << 8) & 0xFF00);
	src += 2;

	// Compute total data size.
	uint32_t encode_size = size + 3;

	// Bounds check the length.
	if (encode_size > max) {
		return false;
	}

	// Set the string size.
	str->size = size;

	// Copy string data, not null terminated.
	memcpy(str->data, src, size);

	// Return the amount of data read.
	return encode_size;
}


uint32_t flshm_amf0_read_boolean(bool * flag, const char * src, size_t max) {
	// Bounds check.
	if (max < 2) {
		return 0;
	}

	// Check the type marker.
	if (*src != '\x01') {
		return 0;
	}
	src++;

	// Read flag, anything but null is considered true.
	*flag = *src != '\x00';

	return 2;
}


uint32_t flshm_amf0_read_double(double * number, const char * src, size_t max) {
	// Bounds check.
	if (max < 9) {
		return 0;
	}

	// Check the type marker.
	if (*src != '\x00') {
		return 0;
	}
	src++;

	// Read big endian double into native via union.
	union {
		double d;
		uint64_t i;
	} u;
	u.i = 0;
	u.i |= (((uint64_t)src[0]) & 0xFF) << 56;
	u.i |= (((uint64_t)src[1]) & 0xFF) << 48;
	u.i |= (((uint64_t)src[2]) & 0xFF) << 40;
	u.i |= (((uint64_t)src[3]) & 0xFF) << 32;
	u.i |= (((uint64_t)src[4]) & 0xFF) << 24;
	u.i |= (((uint64_t)src[5]) & 0xFF) << 16;
	u.i |= (((uint64_t)src[6]) & 0xFF) << 8;
	u.i |= (((uint64_t)src[7]) & 0xFF);

	// Set the double variable.
	*number = u.d;

	return 9;
}


uint32_t flshm_amf0_write_string(const flshm_amf0_string * str, char * dest, size_t max) {
	// Get encode length and bounds check.
	uint16_t size = str->size;
	uint32_t encode_size = size + 3;
	if (encode_size > max) {
		return 0;
	}

	// Write the string marker.
	*dest = '\x02';
	dest++;

	// Write the string size, big endian.
	*((uint8_t *)dest    ) = (size >> 8) & 0xFF;
	*((uint8_t *)dest + 1) = (size     ) & 0xFF;
	dest += 2;

	// Copy the string without null byte.
	memcpy(dest, str->data, size);

	return encode_size;
}


uint32_t flshm_amf0_write_boolean(bool flag, char * dest, size_t max) {
	// Bounds check.
	if (max < 2) {
		return 0;
	}

	// Write the type marker.
	*dest = '\x01';
	dest++;

	// Write flag.
	*dest = flag ? '\x01' : '\x00';

	return 2;
}


uint32_t flshm_amf0_write_double(double number, char * dest, size_t max) {
	// Bounds check.
	if (max < 9) {
		return 0;
	}

	// Write the type marker.
	*dest = '\x00';
	dest++;

	// Write big endian double from native via union.
	union {
		double d;
		uint64_t i;
	} u;
	u.d = number;
	dest[0] = (char)(u.i >> 56);
	dest[1] = (char)(u.i >> 48);
	dest[2] = (char)(u.i >> 40);
	dest[3] = (char)(u.i >> 32);
	dest[4] = (char)(u.i >> 24);
	dest[5] = (char)(u.i >> 16);
	dest[6] = (char)(u.i >> 8);
	dest[7] = (char)(u.i);

	return 9;
}


bool flshm_amf0_decode_string_cstr(const flshm_amf0_string * str, char * to, bool skip_null) {
	bool r = true;
	uint16_t size = str->size;
	uint16_t i = 0;
	for (; i < size; i++) {
		char c = str->data[i];
		if (c == '\x00') {
			if (skip_null) {
				r = false;
				continue;
			}
			break;
		}
		to[i] = c;
	}
	to[i] = '\x00';
	if (i != size) {
		r = false;
	}
	return r;
}


bool flshm_amf0_encode_string_cstr(flshm_amf0_string * str, const char * from) {
	size_t size = strlen(from);
	if (size > FLSHM_AMF0_STRING_DATA_MAX_SIZE) {
		return false;
	}
	memcpy(str->data, from, size);
	str->size = (uint16_t)size;
	return true;
}


uint32_t flshm_hash_uid(uint32_t uid) {
	// Hash the uid into something unique.
	uint32_t a = 9 * ((uid + ~(uid << 15)) ^ ((uid + ~(uid << 15)) >> 10));
	uint32_t b = 16389 * (a ^ (a >> 6)) ^ (16389 * (a ^ (a >> 6)) >> 16);

	// Return 1 if hash is 0.
	return b ? b : 1;
}


bool flshm_shm_inited(const void * shmdata) {
	// Check for 2 uint32 flag in the head.
	return (
		*((uint32_t *)shmdata) == 1 &&
		*((uint32_t *)((char *)shmdata + 4)) == 1
	);
}


uint32_t flshm_tick() {
	uint32_t ret;

	#ifdef _WIN32
		ret = GetTickCount();
	#elif __APPLE__
		// Generated by TickCount, but TickCount is deprecated, reimplement.
		uint64_t mat = mach_absolute_time();
		uint32_t mul = 0x80D9594E;
		ret = ((((0xFFFFFFFF & mat) * mul) >> 32) + (mat >> 32) * mul) >> 23;
	#else
		// Variables to cache across multiple calls.
		static bool inited = false;
		static struct timeval tvbase;
		static uint32_t base = 0;

		if (inited) {
			// Get the current time.
			struct timeval tv;
			gettimeofday(&tv, 0);

			// Calculate offset from base.
			tv.tv_sec -= tvbase.tv_sec;
			tv.tv_usec -= tvbase.tv_usec;

			// Calculate the tick from the base.
			ret = base + 1000 * tv.tv_sec + tv.tv_usec / 1000;
		}
		else {
			// Get the system uptime, first time.
			struct sysinfo si;
			sysinfo(&si);

			// Get the current time as base.
			gettimeofday(&tvbase, 0);

			// Mark initialized.
			inited = true;

			// Calculate the time base, and use it as first tick.
			base = 1000 * si.uptime + tvbase.tv_usec / 1000 % 1000;
			ret = base;
		}
	#endif

	return ret;
}


void flshm_keys_init(flshm_keys * keys, bool is_per_user) {
	#ifdef _WIN32
		// Not used.
		(void)is_per_user;

		// Copy keys.
		strcpy(keys->sem, "MacromediaMutexOmega");
		strcpy(keys->shm, "MacromediaFMOmega");
	#elif __APPLE__
		// Check if user-specific keys for isPerUser, else user-shared.
		if (is_per_user) {
			// The isPerUser generated keys.
			uint32_t shm = flshm_hash_uid(getuid());
			snprintf(keys->sem, 20, "%u", shm);
			keys->shm = (key_t)shm;
		}
		else {
			// The default user-shared keys.
			strcpy(keys->sem, "MacromediaSemaphoreDig");
			keys->shm = (key_t)0x53414E44; // SAND
		}
	#else
		// Not used.
		(void)is_per_user;

		// Generate key.
		key_t key = (key_t)flshm_hash_uid(getuid());
		keys->sem = key;
		keys->shm = key;
	#endif
}


bool flshm_open(flshm_info * info, const flshm_keys * keys) {
	bool r;

	#ifdef _WIN32
		// Open semaphore.
		info->sem = OpenMutex(MUTEX_ALL_ACCESS, FALSE, keys->sem);

		// Open the shared memory.
		info->shm = OpenFileMapping(FILE_MAP_ALL_ACCESS, FALSE, keys->shm);

		// Attach to shared memory.
		info->shmaddr = info->shm ?
			MapViewOfFile(info->shm, FILE_MAP_ALL_ACCESS, 0, 0, FLSHM_SIZE) :
			NULL;
		info->data = info->shmaddr;

		// Check success, cleanup if any failed.
		r = info->sem &&
			info->shm &&
			info->shmaddr &&
			flshm_shm_inited(info->data);
		if (!r) {
			if (info->shmaddr) {
				UnmapViewOfFile(info->shmaddr);
			}
			if (info->shm) {
				CloseHandle(info->shm);
			}
			if (info->sem) {
				CloseHandle(info->sem);
			}
			return false;
		}
	#else
		bool semopen;
		void * badptr = (void *)-1;

		// Open semaphore.
		#if __APPLE__
			info->sem = sem_open(keys->sem, 0);
			semopen = info->sem != SEM_FAILED;
		#else
			info->sem = semget(keys->sem, 1, 0);
			semopen = info->sem > -1;
		#endif

		// Open the shared memory.
		info->shm = shmget(keys->shm, FLSHM_SIZE, 0);
		bool shmgot = info->shm > -1;

		// Attach to shared memory.
		info->shmaddr = shmgot ? shmat(info->shm, NULL, 0) : badptr;
		info->data = (void *)info->shmaddr;
		bool shmopen = info->shmaddr != badptr;

		// Cleanup if any failed.
		r = semopen &&
			shmgot &&
			shmopen &&
			flshm_shm_inited(info->data);
		if (!r) {
			if (shmopen) {
				shmdt(info->shmaddr);
			}
			if (shmgot) {
				close(info->shm);
			}
			if (semopen) {
				#if __APPLE__
					sem_close(info->sem);
				#else
					close(info->sem);
				#endif
			}
			return false;
		}
	#endif

	return r;
}


void flshm_close(flshm_info * info) {
	#ifdef _WIN32
		// Detach from shared memory.
		UnmapViewOfFile(info->shmaddr);

		// Close the shared memory.
		CloseHandle(info->shm);

		// Close semaphore.
		CloseHandle(info->sem);
	#else
		// Detach from shared memory.
		shmdt(info->shmaddr);

		// Try to lock to check if another process is using the memory.
		if (flshm_lock(info)) {
			// Check how many processes are connect to the shared memory.
			struct shmid_ds ds;
			shmctl(info->shm, IPC_STAT, &ds);

			// If no open connection, then remove shared memory.
			if (!ds.shm_nattch) {
				shmctl(info->shm, IPC_RMID, &ds);
			}

			flshm_unlock(info);
		}

		// Close the shared memory.
		close(info->shm);

		// Close semaphore.
		#if __APPLE__
			sem_close(info->sem);
		#else
			// Nothing to close here.
		#endif
	#endif
}


bool flshm_lock(flshm_info * info) {
	#ifdef _WIN32
		return WaitForSingleObject(info->sem, INFINITE) == WAIT_OBJECT_0;
	#elif __APPLE__
		return !sem_wait(info->sem);
	#else
		struct sembuf sb;
		sb.sem_num = 0;
		sb.sem_op = -1;
		sb.sem_flg = SEM_UNDO;
		return !semop(info->sem, &sb, 1);
	#endif
}


bool flshm_unlock(flshm_info * info) {
	#ifdef _WIN32
		return ReleaseMutex(info->sem) == TRUE;
	#elif __APPLE__
		return !sem_post(info->sem);
	#else
		struct sembuf sb;
		sb.sem_num = 0;
		sb.sem_op = 1;
		sb.sem_flg = SEM_UNDO;
		return !semop(info->sem, &sb, 1);
	#endif
}


bool flshm_connection_name_valid(const char * name, size_t size) {
	// Check if too large.
	if (size > FLSHM_CONNECTION_NAME_MAX_LENGTH) {
		return false;
	}

	// First character must exist and not be ':' or null.
	if (size == 0 || name[0] == ':' || name[0] == '\x00') {
		return false;
	}

	// Check how many colons are expected, and keep track of those seen.
	uint8_t colons_valid = name[0] == '_' ? 0 : 1;
	uint8_t colons = 0;
	for (uint32_t i = 0; i < size; i++) {
		char c = name[i];

		// Should not encounter a null byte within size.
		if (c == '\x00') {
			return false;
		}

		// If colon, make sure not more than expected.
		if (c == ':') {
			colons++;
			if (colons > colons_valid) {
				return false;
			}
		}
	}

	// If saw the expected colon count, return valid.
	return colons == colons_valid;
}


bool flshm_connection_name_valid_cstr(const char * name) {
	size_t len = strlen(name);
	return flshm_connection_name_valid(name, len);
}


bool flshm_connection_name_valid_amf0(const flshm_amf0_string * name) {
	return flshm_connection_name_valid(name->data, name->size);
}


void flshm_connection_list(flshm_info * info, flshm_connected * list) {
	// Reset count.
	list->count = 0;

	// Map out the memory, and loop over connections.
	char * memory = ((char *)info->data) + FLSHM_CONNECTIONS_OFFSET;
	for (uint32_t offset = 0; list->count < FLSHM_CONNECTIONS_MAX_COUNT;) {
		// Read a connection entry, end of list if nothing read.
		bool valid = false;
		uint32_t read = flshm_connection_read(
			&list->connections[list->count],
			memory + offset,
			FLSHM_CONNECTIONS_SIZE - offset,
			&valid
		);
		if (!read) {
			break;
		}

		// If a valid connection, move to next entry, invalid skipped.
		if (valid) {
			list->count++;
		}

		// Move to next connection.
		offset += read;
	}
}


bool flshm_connection_add(flshm_info * info, const flshm_connection * connection) {
	// Validate the connection name.
	if (!flshm_connection_name_valid_cstr(connection->name)) {
		return false;
	}

	// Get the current connections.
	flshm_connected connected;
	flshm_connection_list(info, &connected);

	// Fail if maxed out on connections.
	if (connected.count >= FLSHM_CONNECTIONS_MAX_COUNT) {
		return false;
	}

	// Loop over the connections, make sure the name is unique.
	for (uint32_t i = 0; i < connected.count; i++) {
		if (!strcmp(connection->name, connected.connections[i].name)) {
			return false;
		}
	}

	// Seek out end of connection list, if entries are present.
	char * memory = ((char *)info->data) + FLSHM_CONNECTIONS_OFFSET;
	uint32_t offset = 0;
	if (*memory != '\x00') {
		// Seek for double null, make default offset invalid.
		offset = FLSHM_CONNECTIONS_SIZE;
		for (uint32_t i = 0; i < FLSHM_CONNECTIONS_SIZE - 1; i++) {
			char * p = memory + i;
			if (*p == '\x00' && *(p + 1) == '\x00') {
				offset = i + 1;
				break;
			}
		}
	}

	// Return now if no space remaining.
	if (offset >= FLSHM_CONNECTIONS_SIZE) {
		return false;
	}

	// Calculate the remaining space.
	uint32_t max = (FLSHM_CONNECTIONS_SIZE - offset) - 1;

	// Write the connection to the list, or fail.
	char * addr = memory + offset;
	uint32_t wrote = flshm_connection_write(connection, addr, max);
	if (!wrote) {
		return false;
	}
	addr += wrote;

	// Add list terminating null.
	*(addr + 1) = '\x00';

	return true;
}


bool flshm_connection_remove(flshm_info * info, const flshm_connection * connection) {
	// Get the current connections.
	flshm_connected connected;
	flshm_connection_list(info, &connected);

	// Get the offset of connection list.
	char * addr = ((char *)info->data) + FLSHM_CONNECTIONS_OFFSET;

	// Loop over them all, rewrite everything to ensure clean reflow.
	bool found = false;
	for (uint32_t i = 0; i < connected.count; i++) {
		flshm_connection * con = &connected.connections[i];

		// Check if this is the one to remove.
		if (
			!found &&
			con->version == connection->version &&
			con->sandbox == connection->sandbox &&
			!strcmp(con->name, connection->name)
		) {
			found = true;
			continue;
		}

		// Rewrite this connection entry, no need for max write limit.
		// The rewritten list will never be larger than the old list.
		// Pass size of maximum possible to skip computing encode size.
		addr += flshm_connection_write(
			con,
			addr,
			FLSHM_CONNECTION_ENCODE_MAX_SIZE
		);
	}

	// Add list terminating null.
	*(addr) = '\x00';

	return found;
}


uint32_t flshm_connection_encode_size(const flshm_connection * connection) {
	// Get connection name size.
	uint32_t name_size = (uint32_t)strlen(connection->name);

	// Compute size requirements for serialized connection, includes null.
	uint32_t r = name_size + 1;
	if (connection->version != FLSHM_VERSION_1) {
		r += connection->sandbox != FLSHM_SECURITY_NONE ? 8 : 4;
	}
	return r;
}


uint32_t flshm_connection_read(flshm_connection * connection, const char * src, size_t max, bool * valid) {
	*valid = false;

	// Default values.
	const char * name = NULL;
	flshm_version version = FLSHM_VERSION_1;
	flshm_security sandbox = FLSHM_SECURITY_NONE;

	// Seek out name end.
	bool ended = false;
	uint32_t i = 0;
	for (; i < max; i++) {
		char c = src[i];
		if (c == '\x00') {
			ended = true;
			break;
		}
	}

	//  If string did not end in max or no string data, no connection here.
	if (!ended || !i) {
		return 0;
	}
	i++;

	// Name found, now validate, return now if invalid.
	name = src;
	if (!flshm_connection_name_valid_cstr(name)) {
		return i;
	}

	// Check if matches "::[^\x00]\x00" up to 2 times in the remaining space.
	if (
		i + 4 < max &&
		src[i + 0] == ':' &&
		src[i + 1] == ':' &&
		src[i + 2] != '\x00' &&
		src[i + 3] == '\x00'
	) {
		version = src[i + 2] - (uint8_t)'0';
		i += 4;

		if (
			i + 4 < max &&
			src[i + 0] == ':' &&
			src[i + 1] == ':' &&
			src[i + 2] != '\x00' &&
			src[i + 3] == '\x00'
		) {
			sandbox = src[i + 2] - (uint8_t)'1';
			i += 4;
		}
	}

	// Copy the connection data.
	strcpy(connection->name, name);
	connection->version = version;
	connection->sandbox = sandbox;

	*valid = true;
	return i;
}


uint32_t flshm_connection_write(const flshm_connection * connection, char * dest, size_t max) {
	// If the max size is less than the max encode size, check encode size.
	// Optimization trick to avoid computing size when unnecessary.
	if (
		max < FLSHM_CONNECTION_ENCODE_MAX_SIZE &&
		max < flshm_connection_encode_size(connection)
	) {
		return 0;
	}

	// Copy connection name and remember length.
	uint32_t i = 0;
	for (char c; (c = connection->name[i]) != '\x00'; i++) {
		dest[i] = c;
	}
	dest[i++] = '\x00';

	// Add the meta data if present, '0' and '1' indexed.
	if (connection->version != FLSHM_VERSION_1) {
		dest[i++] = ':';
		dest[i++] = ':';
		dest[i++] = (char)((uint8_t)'0' + connection->version);
		dest[i++] = '\x00';
		if (connection->sandbox != FLSHM_SECURITY_NONE) {
			dest[i++] = ':';
			dest[i++] = ':';
			dest[i++] = (char)((uint8_t)'1' + connection->sandbox);
			dest[i++] = '\x00';
		}
	}

	return i;
}


uint32_t flshm_message_tick(const flshm_info * info) {
	// Read the tick from the memory.
	return *((uint32_t *)((char *)info->data + FLSHM_MESSAGE_TICK_OFFSET));
}


bool flshm_message_read(const flshm_info * info, flshm_message * message) {
	// All the properties to be set.
	uint32_t tick;
	flshm_version version = FLSHM_VERSION_1;
	bool sandboxed = false;
	bool https = false;
	flshm_security sandbox = FLSHM_SECURITY_NONE;
	uint32_t swfv = 0;
	flshm_amf amfv = FLSHM_AMF0;
	uint32_t size = 0;

	double d2i;

	// Pointer to shared memory.
	char * shmdata = (char *)info->data;

	// Read the tick count and check if set (only valid if non-zero).
	tick = *((uint32_t *)(shmdata + FLSHM_MESSAGE_TICK_OFFSET));
	if (!tick) {
		return false;
	}

	// Read the message size if present and sanity check it.
	uint32_t amf_size = *((uint32_t *)(shmdata + FLSHM_MESSAGE_SIZE_OFFSET));
	if (!amf_size || amf_size > FLSHM_MESSAGE_MAX_SIZE) {
		return false;
	}

	// Keep track of position and bounds.
	uint32_t i = FLSHM_MESSAGE_BODY_OFFSET;
	uint32_t max = FLSHM_MESSAGE_BODY_OFFSET + amf_size;
	uint32_t re;

	// Start a block that can be broken from, assume failure on an early break.
	bool success = false;
	for (;;) {
		// Read the connection name, or fail.
		re = flshm_amf0_read_string(
			&message->name,
			shmdata + i,
			max - i
		);
		if (!re) {
			break;
		}
		i += re;

		// Read the host name, or fail.
		re = flshm_amf0_read_string(
			&message->host,
			shmdata + i,
			max - i
		);
		if (!re) {
			break;
		}
		i += re;

		// Read the optional data, if present, based on first boolean.

		// Read version 2 data if present.
		re = flshm_amf0_read_boolean(
			&sandboxed,
			shmdata + i,
			max - i
		);
		if (re) {
			// Read sandboxed successfully.
			i += re;

			// We have version 2 at least, read data.
			version = FLSHM_VERSION_2;

			// Read HTTPS or fail.
			re = flshm_amf0_read_boolean(
				&https,
				shmdata + i,
				max - i
			);
			if (!re) {
				break;
			}
			i += re;

			// Read version 3 data if present, based on first double.
			re = flshm_amf0_read_double(
				&d2i,
				shmdata + i,
				max - i
			);
			if (re) {
				// Read sandbox successfully.
				sandbox = (int32_t)d2i;
				i += re;

				// We have version 3 at least, read data.
				version = FLSHM_VERSION_3;

				// Read version or fail.
				re = flshm_amf0_read_double(
					&d2i,
					shmdata + i,
					max - i
				);
				if (!re) {
					break;
				}
				swfv = (uint32_t)d2i;
				i += re;

				// If sandbox local-with-file, includes sender filepath.
				if (sandbox == FLSHM_SECURITY_LOCAL_WITH_FILE) {
					re = flshm_amf0_read_string(
						&message->filepath,
						shmdata + i,
						max - i
					);
					if (!re) {
						break;
					}
					i += re;
				}
				else {
					message->size = 0;
				}

				// Read AMF version if present, else ignore.
				re = flshm_amf0_read_double(
					&d2i,
					shmdata + i,
					max - i
				);
				if (re) {
					amfv = (uint32_t)d2i;
					i += re;

					// Version must be 4.
					version = FLSHM_VERSION_4;
				}
			}
		}

		// Read the method name or fail.
		re = flshm_amf0_read_string(
			&message->method,
			shmdata + i,
			max - i
		);
		if (!re) {
			break;
		}
		i += re;

		// Copy the message data.
		size = max - i;
		if (size) {
			memcpy(message->data, shmdata + i, size);
		}

		// Finish setting the members.
		message->tick = tick;
		message->version = version;
		message->sandboxed = sandboxed;
		message->https = https;
		message->sandbox = sandbox;
		message->swfv = swfv;
		message->amfv = amfv;
		message->size = size;

		success = true;
		break;
	}

	return success;
}


bool flshm_message_write(flshm_info * info, const flshm_message * message) {
	// The buffer to encode message into.
	char buffer[FLSHM_MESSAGE_MAX_SIZE];

	// Start a block that can be broken from, assume failure on break.
	bool success = false;
	for (;;) {
		// Keep track of offset and bounds.
		uint32_t i = 0;
		uint32_t max = sizeof(buffer);
		uint32_t wr;

		// Write the connection name or fail.
		wr = flshm_amf0_write_string(
			&message->name,
			buffer + i,
			max - i
		);
		if (!wr) {
			break;
		}
		i += wr;

		// Write the connection host or fail.
		wr = flshm_amf0_write_string(
			&message->host,
			buffer + i,
			max - i
		);
		if (!wr) {
			break;
		}
		i += wr;

		// Add version 2 data if specified.
		if (message->version >= FLSHM_VERSION_2) {
			// Write sandboxed or fail.
			wr = flshm_amf0_write_boolean(
				message->sandboxed,
				buffer + i,
				max - i
			);
			if (!wr) {
				break;
			}
			i += wr;

			// Write HTTPS or fail.
			wr = flshm_amf0_write_boolean(
				message->https,
				buffer + i,
				max - i
			);
			if (!wr) {
				break;
			}
			i += wr;

			// Add version 3 data if specified.
			if (message->version >= FLSHM_VERSION_3) {
				// Write sandbox or fail.
				wr = flshm_amf0_write_double(
					(double)message->sandbox,
					buffer + i,
					max - i
				);
				if (!wr) {
					break;
				}
				i += wr;

				// Write version or fail.
				wr = flshm_amf0_write_double(
					(double)message->swfv,
					buffer + i,
					max - i
				);
				if (!wr) {
					break;
				}
				i += wr;

				// Write filepath if local-with-file or fail.
				if (
					message->sandboxed &&
					message->sandbox == FLSHM_SECURITY_LOCAL_WITH_FILE
				) {
					wr = flshm_amf0_write_string(
						&message->filepath,
						buffer + i,
						max - i
					);
					if (!wr) {
						break;
					}
					i += wr;
				}

				// Add version 4 data if specified.
				if (message->version >= FLSHM_VERSION_4) {
					// Write the AMF version or fail.
					wr = flshm_amf0_write_double(
						(double)message->amfv,
						buffer + i,
						max - i
					);
					if (!wr) {
						break;
					}
					i += wr;
				}
			}
		}

		// Write method or fail.
		wr = flshm_amf0_write_string(
			&message->method,
			buffer + i,
			max - i
		);
		if (!wr) {
			break;
		}
		i += wr;

		// Check that message data will fit.
		if (message->size > max - i) {
			break;
		}

		// Write message data to the buffer.
		memcpy(buffer + i, message->data, message->size);
		i += message->size;

		// Pointer to shared memory.
		char * shmdata = (char *)info->data;

		// Copy buffer to the shared memory.
		memcpy(shmdata + FLSHM_MESSAGE_BODY_OFFSET, buffer, i);

		// Set the size of the message.
		*((uint32_t *)(shmdata + FLSHM_MESSAGE_SIZE_OFFSET)) = i;

		// Set the message tick.
		*((uint32_t *)(shmdata + FLSHM_MESSAGE_TICK_OFFSET)) = message->tick;

		success = true;
		break;
	}

	return success;
}


void flshm_message_clear(flshm_info * info) {
	// Pointer to shared memory.
	char * shmdata = (char *)info->data;

	// Write 0 to both the tick and size.
	*((uint32_t *)(shmdata + FLSHM_MESSAGE_SIZE_OFFSET)) = 0;
	*((uint32_t *)(shmdata + FLSHM_MESSAGE_TICK_OFFSET)) = 0;
}
