#include <string_view>
#include <zasm/zasm.hpp>

namespace zasm::tests::data
{
    using InstrEmitter = zasm::Error (*)(class x86::Assembler& a);

    struct InstrTest
    {
        std::string_view instrBytes;
        std::string_view operation;
        InstrEmitter emitter;
    };

#define INSTRUCTION_TEST(INSTR_BYTES, ...)                                                                                     \
    InstrTest                                                                                                                  \
    {                                                                                                                          \
#INSTR_BYTES, #__VA_ARGS__, [](auto& a) { using namespace zasm::x86; return a.__VA_ARGS__; }   \
    }

    // clang-format off
#if defined(__clang__)
    // Clang seems to be failing to compile this as constexpr.
    inline const InstrTest Instructions[] = {
#else
    // GCC struggles with a long compile time when this is not constexpr.
    inline constexpr InstrTest Instructions[] = {
#endif
        INSTRUCTION_TEST(10BC1180000000              , adc(byte_ptr(rcx, rdx, 1, 128), bh)),
        INSTRUCTION_TEST(668394118000000001          , adc(word_ptr(rcx, rdx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(66119C1180000000            , adc(word_ptr(rcx, rdx, 1, 128), bx)),
        INSTRUCTION_TEST(4811D1                      , adc(rcx, rdx)),
        INSTRUCTION_TEST(48138C1A80000000            , adc(rcx, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(48119C1180000000            , adc(qword_ptr(rcx, rdx, 1, 128), rbx)),
        INSTRUCTION_TEST(488394118000000001          , adc(qword_ptr(rcx, rdx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(411351FD                    , adc(edx, dword_ptr(r9, -3))),
        INSTRUCTION_TEST(83D101                      , adc(ecx, Imm(1))),
        INSTRUCTION_TEST(11D1                        , adc(ecx, edx)),
        INSTRUCTION_TEST(138C1A80000000              , adc(ecx, dword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(8394118000000001            , adc(dword_ptr(rcx, rdx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(119C1180000000              , adc(dword_ptr(rcx, rdx, 1, 128), ebx)),
        INSTRUCTION_TEST(4883D101                    , adc(rcx, Imm(1))),
        INSTRUCTION_TEST(6683D101                    , adc(cx, Imm(1))),
        INSTRUCTION_TEST(6611D1                      , adc(cx, dx)),
        INSTRUCTION_TEST(80D101                      , adc(cl, Imm(1))),
        INSTRUCTION_TEST(10D1                        , adc(cl, dl)),
        INSTRUCTION_TEST(10F1                        , adc(cl, dh)),
        INSTRUCTION_TEST(128C1A80000000              , adc(cl, byte_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(80D501                      , adc(ch, Imm(1))),
        INSTRUCTION_TEST(10D5                        , adc(ch, dl)),
        INSTRUCTION_TEST(10F5                        , adc(ch, dh)),
        INSTRUCTION_TEST(12AC1A80000000              , adc(ch, byte_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(8094118000000001            , adc(byte_ptr(rcx, rdx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(109C1180000000              , adc(byte_ptr(rcx, rdx, 1, 128), bl)),
        INSTRUCTION_TEST(66138C1A80000000            , adc(cx, word_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(66480F38F6CA                , adcx(rcx, rdx)),
        INSTRUCTION_TEST(66480F38F68C1A80000000      , adcx(rcx, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660F38F68C1A80000000        , adcx(ecx, dword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660F38F6CA                  , adcx(ecx, edx)),
        INSTRUCTION_TEST(0504030201                  , add(eax, Imm(0x01020304))),
        INSTRUCTION_TEST(83C001                      , add(eax, Imm(1))),
        INSTRUCTION_TEST(01CB                        , add(ebx, ecx)),
        INSTRUCTION_TEST(038C1A80000000              , add(ecx, dword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(01D1                        , add(ecx, edx)),
        INSTRUCTION_TEST(83C101                      , add(ecx, Imm(1))),
        INSTRUCTION_TEST(488384118000000001          , add(qword_ptr(rcx, rdx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(4883C101                    , add(rcx, Imm(1))),
        INSTRUCTION_TEST(48038C1A80000000            , add(rcx, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(4801D1                      , add(rcx, rdx)),
        INSTRUCTION_TEST(66019C1180000000            , add(word_ptr(rcx, rdx, 1, 128), bx)),
        INSTRUCTION_TEST(668384118000000001          , add(word_ptr(rcx, rdx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(8384118000000001            , add(dword_ptr(rcx, rdx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(48019C1180000000            , add(qword_ptr(rcx, rdx, 1, 128), rbx)),
        INSTRUCTION_TEST(019C1180000000              , add(dword_ptr(rcx, rdx, 1, 128), ebx)),
        INSTRUCTION_TEST(80C501                      , add(ch, Imm(1))),
        INSTRUCTION_TEST(6683C101                    , add(cx, Imm(1))),
        INSTRUCTION_TEST(6601D1                      , add(cx, dx)),
        INSTRUCTION_TEST(80C101                      , add(cl, Imm(1))),
        INSTRUCTION_TEST(00D1                        , add(cl, dl)),
        INSTRUCTION_TEST(00F1                        , add(cl, dh)),
        INSTRUCTION_TEST(028C1A80000000              , add(cl, byte_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(00D5                        , add(ch, dl)),
        INSTRUCTION_TEST(00F5                        , add(ch, dh)),
        INSTRUCTION_TEST(02AC1A80000000              , add(ch, byte_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(8084118000000001            , add(byte_ptr(rcx, rdx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(009C1180000000              , add(byte_ptr(rcx, rdx, 1, 128), bl)),
        INSTRUCTION_TEST(00BC1180000000              , add(byte_ptr(rcx, rdx, 1, 128), bh)),
        INSTRUCTION_TEST(66050201                    , add(ax, Imm(0x0102))),
        INSTRUCTION_TEST(66038C1A80000000            , add(cx, word_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660F588C1A80000000          , addpd(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660F58CA                    , addpd(xmm1, xmm2)),
        INSTRUCTION_TEST(0F58CA                      , addps(xmm1, xmm2)),
        INSTRUCTION_TEST(0F588C1A80000000            , addps(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(F20F588C1A80000000          , addsd(xmm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(F20F58CA                    , addsd(xmm1, xmm2)),
        INSTRUCTION_TEST(F30F588C1A80000000          , addss(xmm1, dword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(F30F58CA                    , addss(xmm1, xmm2)),
        INSTRUCTION_TEST(660FD0CA                    , addsubpd(xmm1, xmm2)),
        INSTRUCTION_TEST(660FD08C1A80000000          , addsubpd(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(F20FD0CA                    , addsubps(xmm1, xmm2)),
        INSTRUCTION_TEST(F20FD08C1A80000000          , addsubps(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(F30F38F68C1A80000000        , adox(ecx, dword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(F30F38F6CA                  , adox(ecx, edx)),
        INSTRUCTION_TEST(F3480F38F68C1A80000000      , adox(rcx, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(F3480F38F6CA                , adox(rcx, rdx)),
        INSTRUCTION_TEST(660F38DE8C1A80000000        , aesdec(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660F38DECA                  , aesdec(xmm1, xmm2)),
        INSTRUCTION_TEST(660F38DFCA                  , aesdeclast(xmm1, xmm2)),
        INSTRUCTION_TEST(660F38DF8C1A80000000        , aesdeclast(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660F38DCCA                  , aesenc(xmm1, xmm2)),
        INSTRUCTION_TEST(660F38DC8C1A80000000        , aesenc(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660F38DDCA                  , aesenclast(xmm1, xmm2)),
        INSTRUCTION_TEST(660F38DD8C1A80000000        , aesenclast(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660F38DBCA                  , aesimc(xmm1, xmm2)),
        INSTRUCTION_TEST(660F38DB8C1A80000000        , aesimc(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660F3ADFCA01                , aeskeygenassist(xmm1, xmm2, Imm(1))),
        INSTRUCTION_TEST(660F3ADF8C1A8000000001      , aeskeygenassist(xmm1, xmmword_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(83A4118000000001            , and_(dword_ptr(rcx, rdx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(238C1A80000000              , and_(ecx, dword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(21D1                        , and_(ecx, edx)),
        INSTRUCTION_TEST(83E101                      , and_(ecx, Imm(1))),
        INSTRUCTION_TEST(4883A4118000000001          , and_(qword_ptr(rcx, rdx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(6683A4118000000001          , and_(word_ptr(rcx, rdx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(4883E101                    , and_(rcx, Imm(1))),
        INSTRUCTION_TEST(48238C1A80000000            , and_(rcx, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(4821D1                      , and_(rcx, rdx)),
        INSTRUCTION_TEST(66219C1180000000            , and_(word_ptr(rcx, rdx, 1, 128), bx)),
        INSTRUCTION_TEST(219C1180000000              , and_(dword_ptr(rcx, rdx, 1, 128), ebx)),
        INSTRUCTION_TEST(48219C1180000000            , and_(qword_ptr(rcx, rdx, 1, 128), rbx)),
        INSTRUCTION_TEST(66238C1A80000000            , and_(cx, word_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(20F1                        , and_(cl, dh)),
        INSTRUCTION_TEST(6621D1                      , and_(cx, dx)),
        INSTRUCTION_TEST(80E101                      , and_(cl, Imm(1))),
        INSTRUCTION_TEST(20D1                        , and_(cl, dl)),
        INSTRUCTION_TEST(228C1A80000000              , and_(cl, byte_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(80E501                      , and_(ch, Imm(1))),
        INSTRUCTION_TEST(20D5                        , and_(ch, dl)),
        INSTRUCTION_TEST(20F5                        , and_(ch, dh)),
        INSTRUCTION_TEST(22AC1A80000000              , and_(ch, byte_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(80A4118000000001            , and_(byte_ptr(rcx, rdx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(209C1180000000              , and_(byte_ptr(rcx, rdx, 1, 128), bl)),
        INSTRUCTION_TEST(20BC1180000000              , and_(byte_ptr(rcx, rdx, 1, 128), bh)),
        INSTRUCTION_TEST(6683E101                    , and_(cx, Imm(1))),
        INSTRUCTION_TEST(C4E2E8F2CB                  , andn(rcx, rdx, rbx)),
        INSTRUCTION_TEST(C4E2E8F28C2B80000000        , andn(rcx, rdx, qword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E268F2CB                  , andn(ecx, edx, ebx)),
        INSTRUCTION_TEST(C4E268F28C2B80000000        , andn(ecx, edx, dword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(660F55CA                    , andnpd(xmm1, xmm2)),
        INSTRUCTION_TEST(660F558C1A80000000          , andnpd(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(0F55CA                      , andnps(xmm1, xmm2)),
        INSTRUCTION_TEST(0F558C1A80000000            , andnps(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660F54CA                    , andpd(xmm1, xmm2)),
        INSTRUCTION_TEST(660F548C1A80000000          , andpd(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(0F54CA                      , andps(xmm1, xmm2)),
        INSTRUCTION_TEST(0F548C1A80000000            , andps(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(C4E2E0F7CA                  , bextr(rcx, rdx, rbx)),
        INSTRUCTION_TEST(C4E2D8F78C1A80000000        , bextr(rcx, qword_ptr(rdx, rbx, 1, 128), rsp)),
        INSTRUCTION_TEST(C4E258F78C1A80000000        , bextr(ecx, dword_ptr(rdx, rbx, 1, 128), esp)),
        INSTRUCTION_TEST(C4E260F7CA                  , bextr(ecx, edx, ebx)),
        INSTRUCTION_TEST(8FE970018C1A80000000        , blcfill(ecx, dword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(8FE97001CA                  , blcfill(ecx, edx)),
        INSTRUCTION_TEST(8FE9F0018C1A80000000        , blcfill(rcx, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(8FE9F001CA                  , blcfill(rcx, rdx)),
        INSTRUCTION_TEST(8FE9F002F2                  , blci(rcx, rdx)),
        INSTRUCTION_TEST(8FE9F002B41A80000000        , blci(rcx, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(8FE97002F2                  , blci(ecx, edx)),
        INSTRUCTION_TEST(8FE97002B41A80000000        , blci(ecx, dword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(8FE97001AC1A80000000        , blcic(ecx, dword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(8FE97001EA                  , blcic(ecx, edx)),
        INSTRUCTION_TEST(8FE9F001AC1A80000000        , blcic(rcx, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(8FE9F001EA                  , blcic(rcx, rdx)),
        INSTRUCTION_TEST(8FE9F0028C1A80000000        , blcmsk(rcx, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(8FE9F002CA                  , blcmsk(rcx, rdx)),
        INSTRUCTION_TEST(8FE970028C1A80000000        , blcmsk(ecx, dword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(8FE97002CA                  , blcmsk(ecx, edx)),
        INSTRUCTION_TEST(8FE970019C1A80000000        , blcs(ecx, dword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(8FE97001DA                  , blcs(ecx, edx)),
        INSTRUCTION_TEST(8FE9F0019C1A80000000        , blcs(rcx, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(8FE9F001DA                  , blcs(rcx, rdx)),
        INSTRUCTION_TEST(660F3A0DCA01                , blendpd(xmm1, xmm2, Imm(1))),
        INSTRUCTION_TEST(660F3A0D8C1A8000000001      , blendpd(xmm1, xmmword_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(660F3A0CCA01                , blendps(xmm1, xmm2, Imm(1))),
        INSTRUCTION_TEST(660F3A0C8C1A8000000001      , blendps(xmm1, xmmword_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(660F38158C1A80000000        , blendvpd(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660F3815CA                  , blendvpd(xmm1, xmm2)),
        INSTRUCTION_TEST(660F38148C1A80000000        , blendvps(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660F3814CA                  , blendvps(xmm1, xmm2)),
        INSTRUCTION_TEST(8FE97001941A80000000        , blsfill(ecx, dword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(8FE97001D2                  , blsfill(ecx, edx)),
        INSTRUCTION_TEST(8FE9F001941A80000000        , blsfill(rcx, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(8FE9F001D2                  , blsfill(rcx, rdx)),
        INSTRUCTION_TEST(C4E270F39C1A80000000        , blsi(ecx, dword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(C4E270F3DA                  , blsi(ecx, edx)),
        INSTRUCTION_TEST(C4E2F0F39C1A80000000        , blsi(rcx, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(C4E2F0F3DA                  , blsi(rcx, rdx)),
        INSTRUCTION_TEST(8FE9F001F2                  , blsic(rcx, rdx)),
        INSTRUCTION_TEST(8FE9F001B41A80000000        , blsic(rcx, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(8FE97001B41A80000000        , blsic(ecx, dword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(8FE97001F2                  , blsic(ecx, edx)),
        INSTRUCTION_TEST(C4E270F3941A80000000        , blsmsk(ecx, dword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(C4E270F3D2                  , blsmsk(ecx, edx)),
        INSTRUCTION_TEST(C4E2F0F3941A80000000        , blsmsk(rcx, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(C4E2F0F3D2                  , blsmsk(rcx, rdx)),
        INSTRUCTION_TEST(C4E2F0F3CA                  , blsr(rcx, rdx)),
        INSTRUCTION_TEST(C4E2F0F38C1A80000000        , blsr(rcx, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(C4E270F3CA                  , blsr(ecx, edx)),
        INSTRUCTION_TEST(C4E270F38C1A80000000        , blsr(ecx, dword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(F30F1A8C1A80000000          , bndcl(bnd1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(F30F1ACA                    , bndcl(bnd1, rdx)),
        INSTRUCTION_TEST(F20F1B8C1A80000000          , bndcn(bnd1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(F20F1BCA                    , bndcn(bnd1, rdx)),
        INSTRUCTION_TEST(F20F1A8C1A80000000          , bndcu(bnd1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(F20F1ACA                    , bndcu(bnd1, rdx)),
        INSTRUCTION_TEST(660F1ACA                    , bndmov(bnd1, bnd2)),
        INSTRUCTION_TEST(480FBCCA                    , bsf(rcx, rdx)),
        INSTRUCTION_TEST(480FBC8C1A80000000          , bsf(rcx, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(0FBCCA                      , bsf(ecx, edx)),
        INSTRUCTION_TEST(660FBCCA                    , bsf(cx, dx)),
        INSTRUCTION_TEST(660FBC8C1A80000000          , bsf(cx, word_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(0FBC8C1A80000000            , bsf(ecx, dword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(480FBD8C1A80000000          , bsr(rcx, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(480FBDCA                    , bsr(rcx, rdx)),
        INSTRUCTION_TEST(0FBDCA                      , bsr(ecx, edx)),
        INSTRUCTION_TEST(0FBD8C1A80000000            , bsr(ecx, dword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660FBD8C1A80000000          , bsr(cx, word_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660FBDCA                    , bsr(cx, dx)),
        INSTRUCTION_TEST(660FC9                      , bswap(cx)),
        INSTRUCTION_TEST(0FC9                        , bswap(ecx)),
        INSTRUCTION_TEST(480FC9                      , bswap(rcx)),
        INSTRUCTION_TEST(660FA39C1180000000          , bt(word_ptr(rcx, rdx, 1, 128), bx)),
        INSTRUCTION_TEST(480FA3D1                    , bt(rcx, rdx)),
        INSTRUCTION_TEST(480FBAE101                  , bt(rcx, Imm(1))),
        INSTRUCTION_TEST(480FA39C1180000000          , bt(qword_ptr(rcx, rdx, 1, 128), rbx)),
        INSTRUCTION_TEST(480FBAA4118000000001        , bt(qword_ptr(rcx, rdx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(660FBAA4118000000001        , bt(word_ptr(rcx, rdx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(0FA3D1                      , bt(ecx, edx)),
        INSTRUCTION_TEST(0FBAA4118000000001          , bt(dword_ptr(rcx, rdx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(0FA39C1180000000            , bt(dword_ptr(rcx, rdx, 1, 128), ebx)),
        INSTRUCTION_TEST(660FBAE101                  , bt(cx, Imm(1))),
        INSTRUCTION_TEST(660FA3D1                    , bt(cx, dx)),
        INSTRUCTION_TEST(0FBAE101                    , bt(ecx, Imm(1))),
        INSTRUCTION_TEST(660FBABC118000000001        , btc(word_ptr(rcx, rdx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(660FBB9C1180000000          , btc(word_ptr(rcx, rdx, 1, 128), bx)),
        INSTRUCTION_TEST(480FBBD1                    , btc(rcx, rdx)),
        INSTRUCTION_TEST(480FBAF901                  , btc(rcx, Imm(1))),
        INSTRUCTION_TEST(480FBB9C1180000000          , btc(qword_ptr(rcx, rdx, 1, 128), rbx)),
        INSTRUCTION_TEST(480FBABC118000000001        , btc(qword_ptr(rcx, rdx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(0FBAF901                    , btc(ecx, Imm(1))),
        INSTRUCTION_TEST(0FBABC118000000001          , btc(dword_ptr(rcx, rdx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(0FBB9C1180000000            , btc(dword_ptr(rcx, rdx, 1, 128), ebx)),
        INSTRUCTION_TEST(660FBAF901                  , btc(cx, Imm(1))),
        INSTRUCTION_TEST(660FBBD1                    , btc(cx, dx)),
        INSTRUCTION_TEST(0FBBD1                      , btc(ecx, edx)),
        INSTRUCTION_TEST(480FB39C1180000000          , btr(qword_ptr(rcx, rdx, 1, 128), rbx)),
        INSTRUCTION_TEST(660FB39C1180000000          , btr(word_ptr(rcx, rdx, 1, 128), bx)),
        INSTRUCTION_TEST(480FB3D1                    , btr(rcx, rdx)),
        INSTRUCTION_TEST(480FBAF101                  , btr(rcx, Imm(1))),
        INSTRUCTION_TEST(480FBAB4118000000001        , btr(qword_ptr(rcx, rdx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(660FBAB4118000000001        , btr(word_ptr(rcx, rdx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(0FB3D1                      , btr(ecx, edx)),
        INSTRUCTION_TEST(660FB3D1                    , btr(cx, dx)),
        INSTRUCTION_TEST(660FBAF101                  , btr(cx, Imm(1))),
        INSTRUCTION_TEST(0FBAF101                    , btr(ecx, Imm(1))),
        INSTRUCTION_TEST(0FBAB4118000000001          , btr(dword_ptr(rcx, rdx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(0FB39C1180000000            , btr(dword_ptr(rcx, rdx, 1, 128), ebx)),
        INSTRUCTION_TEST(660FABD1                    , bts(cx, dx)),
        INSTRUCTION_TEST(660FBAE901                  , bts(cx, Imm(1))),
        INSTRUCTION_TEST(0FAB9C1180000000            , bts(dword_ptr(rcx, rdx, 1, 128), ebx)),
        INSTRUCTION_TEST(0FBAAC118000000001          , bts(dword_ptr(rcx, rdx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(0FABD1                      , bts(ecx, edx)),
        INSTRUCTION_TEST(0FBAE901                    , bts(ecx, Imm(1))),
        INSTRUCTION_TEST(480FBAAC118000000001        , bts(qword_ptr(rcx, rdx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(480FAB9C1180000000          , bts(qword_ptr(rcx, rdx, 1, 128), rbx)),
        INSTRUCTION_TEST(480FBAE901                  , bts(rcx, Imm(1))),
        INSTRUCTION_TEST(480FABD1                    , bts(rcx, rdx)),
        INSTRUCTION_TEST(660FAB9C1180000000          , bts(word_ptr(rcx, rdx, 1, 128), bx)),
        INSTRUCTION_TEST(660FBAAC118000000001        , bts(word_ptr(rcx, rdx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(C4E2E0F5CA                  , bzhi(rcx, rdx, rbx)),
        INSTRUCTION_TEST(C4E2D8F58C1A80000000        , bzhi(rcx, qword_ptr(rdx, rbx, 1, 128), rsp)),
        INSTRUCTION_TEST(C4E258F58C1A80000000        , bzhi(ecx, dword_ptr(rdx, rbx, 1, 128), esp)),
        INSTRUCTION_TEST(C4E260F5CA                  , bzhi(ecx, edx, ebx)),
        INSTRUCTION_TEST(FF10                        , call(qword_ptr(rax))),
        INSTRUCTION_TEST(6698                        , cbw()),
        INSTRUCTION_TEST(99                          , cdq()),
        INSTRUCTION_TEST(4898                        , cdqe()),
        INSTRUCTION_TEST(0F01CA                      , clac()),
        INSTRUCTION_TEST(F8                          , clc()),
        INSTRUCTION_TEST(FC                          , cld()),
        INSTRUCTION_TEST(0F01DD                      , clgi()),
        INSTRUCTION_TEST(FA                          , cli()),
        INSTRUCTION_TEST(0F06                        , clts()),
        INSTRUCTION_TEST(F5                          , cmc()),
        INSTRUCTION_TEST(660F46CA                    , cmovbe(cx, dx)),
        INSTRUCTION_TEST(660F468C1A80000000          , cmovbe(cx, word_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(0F468C1A80000000            , cmovbe(ecx, dword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(0F46CA                      , cmovbe(ecx, edx)),
        INSTRUCTION_TEST(480F468C1A80000000          , cmovbe(rcx, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(480F46CA                    , cmovbe(rcx, rdx)),
        INSTRUCTION_TEST(480F4C8C1A80000000          , cmovl(rcx, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(480F4CCA                    , cmovl(rcx, rdx)),
        INSTRUCTION_TEST(0F4CCA                      , cmovl(ecx, edx)),
        INSTRUCTION_TEST(0F4C8C1A80000000            , cmovl(ecx, dword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660F4CCA                    , cmovl(cx, dx)),
        INSTRUCTION_TEST(660F4C8C1A80000000          , cmovl(cx, word_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660F4ECA                    , cmovle(cx, dx)),
        INSTRUCTION_TEST(660F4E8C1A80000000          , cmovle(cx, word_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(0F4E8C1A80000000            , cmovle(ecx, dword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(0F4ECA                      , cmovle(ecx, edx)),
        INSTRUCTION_TEST(480F4E8C1A80000000          , cmovle(rcx, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(480F4ECA                    , cmovle(rcx, rdx)),
        INSTRUCTION_TEST(480F43CA                    , cmovnb(rcx, rdx)),
        INSTRUCTION_TEST(0F43CA                      , cmovnb(ecx, edx)),
        INSTRUCTION_TEST(480F438C1A80000000          , cmovnb(rcx, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660F438C1A80000000          , cmovnb(cx, word_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(0F438C1A80000000            , cmovnb(ecx, dword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660F47CA                    , cmovnbe(cx, dx)),
        INSTRUCTION_TEST(660F4DCA                    , cmovnl(cx, dx)),
        INSTRUCTION_TEST(660F4D8C1A80000000          , cmovnl(cx, word_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(0F4D8C1A80000000            , cmovnl(ecx, dword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(0F4DCA                      , cmovnl(ecx, edx)),
        INSTRUCTION_TEST(480F4D8C1A80000000          , cmovnl(rcx, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(480F4DCA                    , cmovnl(rcx, rdx)),
        INSTRUCTION_TEST(480F4F8C1A80000000          , cmovnle(rcx, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(480F4FCA                    , cmovnle(rcx, rdx)),
        INSTRUCTION_TEST(0F4FCA                      , cmovnle(ecx, edx)),
        INSTRUCTION_TEST(660F4FCA                    , cmovnle(cx, dx)),
        INSTRUCTION_TEST(660F4F8C1A80000000          , cmovnle(cx, word_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(0F4F8C1A80000000            , cmovnle(ecx, dword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660F41CA                    , cmovno(cx, dx)),
        INSTRUCTION_TEST(660F418C1A80000000          , cmovno(cx, word_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(0F418C1A80000000            , cmovno(ecx, dword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(0F41CA                      , cmovno(ecx, edx)),
        INSTRUCTION_TEST(480F418C1A80000000          , cmovno(rcx, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(480F41CA                    , cmovno(rcx, rdx)),
        INSTRUCTION_TEST(480F4BCA                    , cmovnp(rcx, rdx)),
        INSTRUCTION_TEST(480F4B8C1A80000000          , cmovnp(rcx, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(0F4BCA                      , cmovnp(ecx, edx)),
        INSTRUCTION_TEST(660F4BCA                    , cmovnp(cx, dx)),
        INSTRUCTION_TEST(660F4B8C1A80000000          , cmovnp(cx, word_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(0F4B8C1A80000000            , cmovnp(ecx, dword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(480F498C1A80000000          , cmovns(rcx, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(480F49CA                    , cmovns(rcx, rdx)),
        INSTRUCTION_TEST(0F49CA                      , cmovns(ecx, edx)),
        INSTRUCTION_TEST(660F49CA                    , cmovns(cx, dx)),
        INSTRUCTION_TEST(660F498C1A80000000          , cmovns(cx, word_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(0F498C1A80000000            , cmovns(ecx, dword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660F45CA                    , cmovnz(cx, dx)),
        INSTRUCTION_TEST(660F458C1A80000000          , cmovnz(cx, word_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(0F458C1A80000000            , cmovnz(ecx, dword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(0F45CA                      , cmovnz(ecx, edx)),
        INSTRUCTION_TEST(480F458C1A80000000          , cmovnz(rcx, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(480F45CA                    , cmovnz(rcx, rdx)),
        INSTRUCTION_TEST(480F40CA                    , cmovo(rcx, rdx)),
        INSTRUCTION_TEST(480F408C1A80000000          , cmovo(rcx, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(0F40CA                      , cmovo(ecx, edx)),
        INSTRUCTION_TEST(660F408C1A80000000          , cmovo(cx, word_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660F40CA                    , cmovo(cx, dx)),
        INSTRUCTION_TEST(0F408C1A80000000            , cmovo(ecx, dword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660F4ACA                    , cmovp(cx, dx)),
        INSTRUCTION_TEST(660F4A8C1A80000000          , cmovp(cx, word_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(0F4A8C1A80000000            , cmovp(ecx, dword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(0F4ACA                      , cmovp(ecx, edx)),
        INSTRUCTION_TEST(480F4A8C1A80000000          , cmovp(rcx, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(480F4ACA                    , cmovp(rcx, rdx)),
        INSTRUCTION_TEST(480F48CA                    , cmovs(rcx, rdx)),
        INSTRUCTION_TEST(480F488C1A80000000          , cmovs(rcx, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(0F48CA                      , cmovs(ecx, edx)),
        INSTRUCTION_TEST(660F488C1A80000000          , cmovs(cx, word_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660F48CA                    , cmovs(cx, dx)),
        INSTRUCTION_TEST(0F488C1A80000000            , cmovs(ecx, dword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660F44CA                    , cmovz(cx, dx)),
        INSTRUCTION_TEST(660F448C1A80000000          , cmovz(cx, word_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(0F448C1A80000000            , cmovz(ecx, dword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(0F44CA                      , cmovz(ecx, edx)),
        INSTRUCTION_TEST(480F448C1A80000000          , cmovz(rcx, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(480F44CA                    , cmovz(rcx, rdx)),
        INSTRUCTION_TEST(399C1180000000              , cmp(dword_ptr(rcx, rdx, 1, 128), ebx)),
        INSTRUCTION_TEST(83BC118000000001            , cmp(dword_ptr(rcx, rdx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(3B8C1A80000000              , cmp(ecx, dword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(39D1                        , cmp(ecx, edx)),
        INSTRUCTION_TEST(83F901                      , cmp(ecx, Imm(1))),
        INSTRUCTION_TEST(4883BC118000000001          , cmp(qword_ptr(rcx, rdx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(66399C1180000000            , cmp(word_ptr(rcx, rdx, 1, 128), bx)),
        INSTRUCTION_TEST(4883F901                    , cmp(rcx, Imm(1))),
        INSTRUCTION_TEST(483B8C1A80000000            , cmp(rcx, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(4839D1                      , cmp(rcx, rdx)),
        INSTRUCTION_TEST(6683BC118000000001          , cmp(word_ptr(rcx, rdx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(48399C1180000000            , cmp(qword_ptr(rcx, rdx, 1, 128), rbx)),
        INSTRUCTION_TEST(663B8C1A80000000            , cmp(cx, word_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(80FD01                      , cmp(ch, Imm(1))),
        INSTRUCTION_TEST(6639D1                      , cmp(cx, dx)),
        INSTRUCTION_TEST(80F901                      , cmp(cl, Imm(1))),
        INSTRUCTION_TEST(38D1                        , cmp(cl, dl)),
        INSTRUCTION_TEST(38F1                        , cmp(cl, dh)),
        INSTRUCTION_TEST(3A8C1A80000000              , cmp(cl, byte_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(38D5                        , cmp(ch, dl)),
        INSTRUCTION_TEST(38F5                        , cmp(ch, dh)),
        INSTRUCTION_TEST(3AAC1A80000000              , cmp(ch, byte_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(80BC118000000001            , cmp(byte_ptr(rcx, rdx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(389C1180000000              , cmp(byte_ptr(rcx, rdx, 1, 128), bl)),
        INSTRUCTION_TEST(38BC1180000000              , cmp(byte_ptr(rcx, rdx, 1, 128), bh)),
        INSTRUCTION_TEST(6683F901                    , cmp(cx, Imm(1))),
        INSTRUCTION_TEST(660FC2CA01                  , cmppd(xmm1, xmm2, Imm(1))),
        INSTRUCTION_TEST(660FC28C1A8000000001        , cmppd(xmm1, xmmword_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(0FC28C1A8000000001          , cmpps(xmm1, xmmword_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(0FC2CA01                    , cmpps(xmm1, xmm2, Imm(1))),
        INSTRUCTION_TEST(A6                          , cmpsb()),
        INSTRUCTION_TEST(A7                          , cmpsd()),
        INSTRUCTION_TEST(F20FC28C1A8000000001        , cmpsd(xmm1, qword_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(F20FC2CA01                  , cmpsd(xmm1, xmm2, Imm(1))),
        INSTRUCTION_TEST(48A7                        , cmpsq()),
        INSTRUCTION_TEST(F30FC28C1A8000000001        , cmpss(xmm1, dword_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(F30FC2CA01                  , cmpss(xmm1, xmm2, Imm(1))),
        INSTRUCTION_TEST(66A7                        , cmpsw()),
        INSTRUCTION_TEST(660FB19C1180000000          , cmpxchg(word_ptr(rcx, rdx, 1, 128), bx)),
        INSTRUCTION_TEST(480FB1D1                    , cmpxchg(rcx, rdx)),
        INSTRUCTION_TEST(480FB19C1180000000          , cmpxchg(qword_ptr(rcx, rdx, 1, 128), rbx)),
        INSTRUCTION_TEST(660FB1D1                    , cmpxchg(cx, dx)),
        INSTRUCTION_TEST(0FB0D1                      , cmpxchg(cl, dl)),
        INSTRUCTION_TEST(0FB1D1                      , cmpxchg(ecx, edx)),
        INSTRUCTION_TEST(0FB0D5                      , cmpxchg(ch, dl)),
        INSTRUCTION_TEST(0FB0F1                      , cmpxchg(cl, dh)),
        INSTRUCTION_TEST(0FB0BC1180000000            , cmpxchg(byte_ptr(rcx, rdx, 1, 128), bh)),
        INSTRUCTION_TEST(0FB09C1180000000            , cmpxchg(byte_ptr(rcx, rdx, 1, 128), bl)),
        INSTRUCTION_TEST(0FB19C1180000000            , cmpxchg(dword_ptr(rcx, rdx, 1, 128), ebx)),
        INSTRUCTION_TEST(0FB0F5                      , cmpxchg(ch, dh)),
        INSTRUCTION_TEST(480FC78C1180000000          , cmpxchg16b(xmmword_ptr(rcx, rdx, 1, 128))),
        INSTRUCTION_TEST(0FC78C1180000000            , cmpxchg8b(qword_ptr(rcx, rdx, 1, 128))),
        INSTRUCTION_TEST(660F2F8C1A80000000          , comisd(xmm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660F2FCA                    , comisd(xmm1, xmm2)),
        INSTRUCTION_TEST(0F2F8C1A80000000            , comiss(xmm1, dword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(0F2FCA                      , comiss(xmm1, xmm2)),
        INSTRUCTION_TEST(0FA2                        , cpuid()),
        INSTRUCTION_TEST(4899                        , cqo()),
        INSTRUCTION_TEST(F2480F38F1CA                , crc32(rcx, rdx)),
        INSTRUCTION_TEST(F2480F38F0CA                , crc32(rcx, dl)),
        INSTRUCTION_TEST(F2480F38F08C1A80000000      , crc32(rcx, byte_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(F2660F38F18C1A80000000      , crc32(ecx, word_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(F20F38F1CA                  , crc32(ecx, edx)),
        INSTRUCTION_TEST(F2480F38F18C1A80000000      , crc32(rcx, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(F20F38F18C1A80000000        , crc32(ecx, dword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(F20F38F0CA                  , crc32(ecx, dl)),
        INSTRUCTION_TEST(F20F38F0CE                  , crc32(ecx, dh)),
        INSTRUCTION_TEST(F2660F38F1CA                , crc32(ecx, dx)),
        INSTRUCTION_TEST(F20F38F08C1A80000000        , crc32(ecx, byte_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(F30FE68C1A80000000          , cvtdq2pd(xmm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(F30FE6CA                    , cvtdq2pd(xmm1, xmm2)),
        INSTRUCTION_TEST(0F5BCA                      , cvtdq2ps(xmm1, xmm2)),
        INSTRUCTION_TEST(0F5B8C1A80000000            , cvtdq2ps(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(F20FE68C1A80000000          , cvtpd2dq(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(F20FE6CA                    , cvtpd2dq(xmm1, xmm2)),
        INSTRUCTION_TEST(660F2DCA                    , cvtpd2pi(mm1, xmm2)),
        INSTRUCTION_TEST(660F2D8C1A80000000          , cvtpd2pi(mm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660F5ACA                    , cvtpd2ps(xmm1, xmm2)),
        INSTRUCTION_TEST(660F5A8C1A80000000          , cvtpd2ps(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660F2ACA                    , cvtpi2pd(xmm1, mm2)),
        INSTRUCTION_TEST(660F2A8C1A80000000          , cvtpi2pd(xmm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(0F2ACA                      , cvtpi2ps(xmm1, mm2)),
        INSTRUCTION_TEST(0F2A8C1A80000000            , cvtpi2ps(xmm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660F5B8C1A80000000          , cvtps2dq(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660F5BCA                    , cvtps2dq(xmm1, xmm2)),
        INSTRUCTION_TEST(0F5ACA                      , cvtps2pd(xmm1, xmm2)),
        INSTRUCTION_TEST(0F5A8C1A80000000            , cvtps2pd(xmm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(0F2D8C1A80000000            , cvtps2pi(mm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(0F2DCA                      , cvtps2pi(mm1, xmm2)),
        INSTRUCTION_TEST(F20F2D8C1A80000000          , cvtsd2si(ecx, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(F20F2DCA                    , cvtsd2si(ecx, xmm2)),
        INSTRUCTION_TEST(F2480F2D8C1A80000000        , cvtsd2si(rcx, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(F2480F2DCA                  , cvtsd2si(rcx, xmm2)),
        INSTRUCTION_TEST(F20F5A8C1A80000000          , cvtsd2ss(xmm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(F20F5ACA                    , cvtsd2ss(xmm1, xmm2)),
        INSTRUCTION_TEST(F20F2A8C1A80000000          , cvtsi2sd(xmm1, dword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(F20F2ACA                    , cvtsi2sd(xmm1, edx)),
        INSTRUCTION_TEST(F2480F2A8C1A80000000        , cvtsi2sd(xmm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(F2480F2ACA                  , cvtsi2sd(xmm1, rdx)),
        INSTRUCTION_TEST(F30F2A8C1A80000000          , cvtsi2ss(xmm1, dword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(F30F2ACA                    , cvtsi2ss(xmm1, edx)),
        INSTRUCTION_TEST(F3480F2A8C1A80000000        , cvtsi2ss(xmm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(F3480F2ACA                  , cvtsi2ss(xmm1, rdx)),
        INSTRUCTION_TEST(F30F5ACA                    , cvtss2sd(xmm1, xmm2)),
        INSTRUCTION_TEST(F30F5A8C1A80000000          , cvtss2sd(xmm1, dword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(F30F2D8C1A80000000          , cvtss2si(ecx, dword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(F30F2DCA                    , cvtss2si(ecx, xmm2)),
        INSTRUCTION_TEST(F3480F2D8C1A80000000        , cvtss2si(rcx, dword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(F3480F2DCA                  , cvtss2si(rcx, xmm2)),
        INSTRUCTION_TEST(660FE6CA                    , cvttpd2dq(xmm1, xmm2)),
        INSTRUCTION_TEST(660FE68C1A80000000          , cvttpd2dq(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660F2C8C1A80000000          , cvttpd2pi(mm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660F2CCA                    , cvttpd2pi(mm1, xmm2)),
        INSTRUCTION_TEST(F30F5B8C1A80000000          , cvttps2dq(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(F30F5BCA                    , cvttps2dq(xmm1, xmm2)),
        INSTRUCTION_TEST(0F2C8C1A80000000            , cvttps2pi(mm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(0F2CCA                      , cvttps2pi(mm1, xmm2)),
        INSTRUCTION_TEST(F20F2C8C1A80000000          , cvttsd2si(ecx, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(F20F2CCA                    , cvttsd2si(ecx, xmm2)),
        INSTRUCTION_TEST(F2480F2C8C1A80000000        , cvttsd2si(rcx, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(F2480F2CCA                  , cvttsd2si(rcx, xmm2)),
        INSTRUCTION_TEST(F3480F2C8C1A80000000        , cvttss2si(rcx, dword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(F3480F2CCA                  , cvttss2si(rcx, xmm2)),
        INSTRUCTION_TEST(F30F2CCA                    , cvttss2si(ecx, xmm2)),
        INSTRUCTION_TEST(F30F2C8C1A80000000          , cvttss2si(ecx, dword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(6699                        , cwd()),
        INSTRUCTION_TEST(98                          , cwde()),
        INSTRUCTION_TEST(FE8C1180000000              , dec(byte_ptr(rcx, rdx, 1, 128))),
        INSTRUCTION_TEST(FECD                        , dec(ch)),
        INSTRUCTION_TEST(FEC9                        , dec(cl)),
        INSTRUCTION_TEST(66FFC9                      , dec(cx)),
        INSTRUCTION_TEST(FF8C1180000000              , dec(dword_ptr(rcx, rdx, 1, 128))),
        INSTRUCTION_TEST(FFC9                        , dec(ecx)),
        INSTRUCTION_TEST(48FF8C1180000000            , dec(qword_ptr(rcx, rdx, 1, 128))),
        INSTRUCTION_TEST(48FFC9                      , dec(rcx)),
        INSTRUCTION_TEST(66FF8C1180000000            , dec(word_ptr(rcx, rdx, 1, 128))),
        INSTRUCTION_TEST(48F7F1                      , div(rcx)),
        INSTRUCTION_TEST(66F7B41180000000            , div(word_ptr(rcx, rdx, 1, 128))),
        INSTRUCTION_TEST(48F7B41180000000            , div(qword_ptr(rcx, rdx, 1, 128))),
        INSTRUCTION_TEST(F7B41180000000              , div(dword_ptr(rcx, rdx, 1, 128))),
        INSTRUCTION_TEST(F7F1                        , div(ecx)),
        INSTRUCTION_TEST(F6F1                        , div(cl)),
        INSTRUCTION_TEST(F6F5                        , div(ch)),
        INSTRUCTION_TEST(F6B41180000000              , div(byte_ptr(rcx, rdx, 1, 128))),
        INSTRUCTION_TEST(66F7F1                      , div(cx)),
        INSTRUCTION_TEST(660F5ECA                    , divpd(xmm1, xmm2)),
        INSTRUCTION_TEST(660F5E8C1A80000000          , divpd(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(0F5ECA                      , divps(xmm1, xmm2)),
        INSTRUCTION_TEST(0F5E8C1A80000000            , divps(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(F20F5ECA                    , divsd(xmm1, xmm2)),
        INSTRUCTION_TEST(F20F5E8C1A80000000          , divsd(xmm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(F30F5E8C1A80000000          , divss(xmm1, dword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(F30F5ECA                    , divss(xmm1, xmm2)),
        INSTRUCTION_TEST(660F3A41CA01                , dppd(xmm1, xmm2, Imm(1))),
        INSTRUCTION_TEST(660F3A418C1A8000000001      , dppd(xmm1, xmmword_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(660F3A40CA01                , dpps(xmm1, xmm2, Imm(1))),
        INSTRUCTION_TEST(660F3A408C1A8000000001      , dpps(xmm1, xmmword_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(0F77                        , emms()),
        INSTRUCTION_TEST(F30F1EFB                    , endbr32()),
        INSTRUCTION_TEST(F30F1EFA                    , endbr64()),
        INSTRUCTION_TEST(C8010002                    , enter(1, 2)),
        INSTRUCTION_TEST(660F3A179C118000000001      , extractps(dword_ptr(rcx, rdx, 1, 128), xmm3, Imm(1))),
        INSTRUCTION_TEST(660F3A17D101                , extractps(ecx, xmm2, Imm(1))),
        INSTRUCTION_TEST(660F79CA                    , extrq(xmm1, xmm2)),
        INSTRUCTION_TEST(660F78C10102                , extrq(xmm1, 1, 2)),
        INSTRUCTION_TEST(D800                        , fadd(dword_ptr(rax))),
        INSTRUCTION_TEST(DC00                        , fadd(qword_ptr(rax))),
        INSTRUCTION_TEST(9B                          , fwait()),
        INSTRUCTION_TEST(D9F4                        , fxtract()),
        INSTRUCTION_TEST(0F37                        , getsec()),
        INSTRUCTION_TEST(660F3ACFCA01                , gf2p8affineinvqb(xmm1, xmm2, Imm(1))),
        INSTRUCTION_TEST(660F3ACF8C1A8000000001      , gf2p8affineinvqb(xmm1, xmmword_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(660F3ACECA01                , gf2p8affineqb(xmm1, xmm2, Imm(1))),
        INSTRUCTION_TEST(660F3ACE8C1A8000000001      , gf2p8affineqb(xmm1, xmmword_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(660F38CFCA                  , gf2p8mulb(xmm1, xmm2)),
        INSTRUCTION_TEST(660F38CF8C1A80000000        , gf2p8mulb(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660F7CCA                    , haddpd(xmm1, xmm2)),
        INSTRUCTION_TEST(660F7C8C1A80000000          , haddpd(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(F20F7CCA                    , haddps(xmm1, xmm2)),
        INSTRUCTION_TEST(F20F7C8C1A80000000          , haddps(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(F4                          , hlt()),
        INSTRUCTION_TEST(F30F3AF0C001                , hreset(1)),
        INSTRUCTION_TEST(660F7DCA                    , hsubpd(xmm1, xmm2)),
        INSTRUCTION_TEST(660F7D8C1A80000000          , hsubpd(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(F20F7DCA                    , hsubps(xmm1, xmm2)),
        INSTRUCTION_TEST(F20F7D8C1A80000000          , hsubps(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(F6BC1180000000              , idiv(byte_ptr(rcx, rdx, 1, 128))),
        INSTRUCTION_TEST(F6FD                        , idiv(ch)),
        INSTRUCTION_TEST(F6F9                        , idiv(cl)),
        INSTRUCTION_TEST(66F7F9                      , idiv(cx)),
        INSTRUCTION_TEST(F7BC1180000000              , idiv(dword_ptr(rcx, rdx, 1, 128))),
        INSTRUCTION_TEST(F7F9                        , idiv(ecx)),
        INSTRUCTION_TEST(48F7BC1180000000            , idiv(qword_ptr(rcx, rdx, 1, 128))),
        INSTRUCTION_TEST(48F7F9                      , idiv(rcx)),
        INSTRUCTION_TEST(66F7BC1180000000            , idiv(word_ptr(rcx, rdx, 1, 128))),
        INSTRUCTION_TEST(486B8C1A8000000001          , imul(rcx, qword_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(48F7AC1180000000            , imul(qword_ptr(rcx, rdx, 1, 128))),
        INSTRUCTION_TEST(486BCA01                    , imul(rcx, rdx, Imm(1))),
        INSTRUCTION_TEST(66F7E9                      , imul(cx)),
        INSTRUCTION_TEST(66F7AC1180000000            , imul(word_ptr(rcx, rdx, 1, 128))),
        INSTRUCTION_TEST(66F7AC1A80000000            , imul(word_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(F7E9                        , imul(ecx)),
        INSTRUCTION_TEST(48F7E9                      , imul(rcx)),
        INSTRUCTION_TEST(6BCA01                      , imul(ecx, edx, Imm(1))),
        INSTRUCTION_TEST(F6E9                        , imul(cl)),
        INSTRUCTION_TEST(F7AC1A80000000              , imul(dword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(F7AC1180000000              , imul(dword_ptr(rcx, rdx, 1, 128))),
        INSTRUCTION_TEST(666B8C1A8000000001          , imul(cx, word_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(666BCA01                    , imul(cx, dx, Imm(1))),
        INSTRUCTION_TEST(6B8C1A8000000001            , imul(ecx, dword_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(F6ED                        , imul(ch)),
        INSTRUCTION_TEST(F6AC1180000000              , imul(byte_ptr(rcx, rdx, 1, 128))),
        INSTRUCTION_TEST(48FFC1                      , inc(rcx)),
        INSTRUCTION_TEST(48FF841180000000            , inc(qword_ptr(rcx, rdx, 1, 128))),
        INSTRUCTION_TEST(FFC1                        , inc(ecx)),
        INSTRUCTION_TEST(FF841180000000              , inc(dword_ptr(rcx, rdx, 1, 128))),
        INSTRUCTION_TEST(66FF841180000000            , inc(word_ptr(rcx, rdx, 1, 128))),
        INSTRUCTION_TEST(FEC1                        , inc(cl)),
        INSTRUCTION_TEST(FEC5                        , inc(ch)),
        INSTRUCTION_TEST(FE841180000000              , inc(byte_ptr(rcx, rdx, 1, 128))),
        INSTRUCTION_TEST(66FFC1                      , inc(cx)),
        INSTRUCTION_TEST(F30FAEE9                    , incsspd(ecx)),
        INSTRUCTION_TEST(F3480FAEE9                  , incsspq(rcx)),
        INSTRUCTION_TEST(660F3A218C1A8000000001      , insertps(xmm1, dword_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(660F3A21CA01                , insertps(xmm1, xmm2, Imm(1))),
        INSTRUCTION_TEST(F20F79CA                    , insertq(xmm1, xmm2)),
        INSTRUCTION_TEST(F20F78CA0102                , insertq(xmm1, xmm2, 1, 2)),
        INSTRUCTION_TEST(CD01                        , int_(1)),
        INSTRUCTION_TEST(CC                          , int3()),
        INSTRUCTION_TEST(0F08                        , invd()),
        INSTRUCTION_TEST(660F38808C1A80000000        , invept(rcx, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660F38828C1A80000000        , invpcid(rcx, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660F38818C1A80000000        , invvpid(rcx, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(66CF                        , iret()),
        INSTRUCTION_TEST(CF                          , iretd()),
        INSTRUCTION_TEST(48CF                        , iretq()),
        INSTRUCTION_TEST(C5ED4ACB                    , kaddb(k1, k2, k3)),
        INSTRUCTION_TEST(C4E1ED4ACB                  , kaddd(k1, k2, k3)),
        INSTRUCTION_TEST(C4E1EC4ACB                  , kaddq(k1, k2, k3)),
        INSTRUCTION_TEST(C5EC4ACB                    , kaddw(k1, k2, k3)),
        INSTRUCTION_TEST(C5ED41CB                    , kandb(k1, k2, k3)),
        INSTRUCTION_TEST(C4E1ED41CB                  , kandd(k1, k2, k3)),
        INSTRUCTION_TEST(C5ED42CB                    , kandnb(k1, k2, k3)),
        INSTRUCTION_TEST(C4E1ED42CB                  , kandnd(k1, k2, k3)),
        INSTRUCTION_TEST(C4E1EC42CB                  , kandnq(k1, k2, k3)),
        INSTRUCTION_TEST(C5EC42CB                    , kandnw(k1, k2, k3)),
        INSTRUCTION_TEST(C5CC42E7                    , kandnw(k4, k6, k7)),
        INSTRUCTION_TEST(C5DC42EE                    , kandnw(k5, k4, k6)),
        INSTRUCTION_TEST(C4E1EC41CB                  , kandq(k1, k2, k3)),
        INSTRUCTION_TEST(C5EC41CB                    , kandw(k1, k2, k3)),
        INSTRUCTION_TEST(C5D441D6                    , kandw(k2, k5, k6)),
        INSTRUCTION_TEST(C5DC41EE                    , kandw(k5, k4, k6)),
        INSTRUCTION_TEST(C5F990CA                    , kmovb(k1, k2)),
        INSTRUCTION_TEST(C5F992CA                    , kmovb(k1, edx)),
        INSTRUCTION_TEST(C5F9908C1A80000000          , kmovb(k1, byte_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(C5F993CA                    , kmovb(ecx, k2)),
        INSTRUCTION_TEST(C5F9919C1180000000          , kmovb(byte_ptr(rcx, rdx, 1, 128), k3)),
        INSTRUCTION_TEST(C4E1F9919C1180000000        , kmovd(dword_ptr(rcx, rdx, 1, 128), k3)),
        INSTRUCTION_TEST(C5FB93CA                    , kmovd(ecx, k2)),
        INSTRUCTION_TEST(C4E1F9908C1A80000000        , kmovd(k1, dword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(C5FB92CA                    , kmovd(k1, edx)),
        INSTRUCTION_TEST(C4E1F990CA                  , kmovd(k1, k2)),
        INSTRUCTION_TEST(C4E1F890CA                  , kmovq(k1, k2)),
        INSTRUCTION_TEST(C4E1F8908C1A80000000        , kmovq(k1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(C4E1FB92CA                  , kmovq(k1, rdx)),
        INSTRUCTION_TEST(C4E1F8919C1180000000        , kmovq(qword_ptr(rcx, rdx, 1, 128), k3)),
        INSTRUCTION_TEST(C4E1FB93CA                  , kmovq(rcx, k2)),
        INSTRUCTION_TEST(C5F890E5                    , kmovw(k4, k5)),
        INSTRUCTION_TEST(C4A17890A4F023010000        , kmovw(k4, word_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(C5F89021                    , kmovw(k4, word_ptr(rcx))),
        INSTRUCTION_TEST(C57893EB                    , kmovw(r13d, k3)),
        INSTRUCTION_TEST(C4C17892DD                  , kmovw(k3, r13d)),
        INSTRUCTION_TEST(C4A17891ACF034120000        , kmovw(word_ptr(rax, r14, 8, 4660), k5)),
        INSTRUCTION_TEST(C5F8919C1180000000          , kmovw(word_ptr(rcx, rdx, 1, 128), k3)),
        INSTRUCTION_TEST(C5F89121                    , kmovw(word_ptr(rcx), k4)),
        INSTRUCTION_TEST(C5F89129                    , kmovw(word_ptr(rcx), k5)),
        INSTRUCTION_TEST(C4A17891A4F023010000        , kmovw(word_ptr(rax, r14, 8, 291), k4)),
        INSTRUCTION_TEST(C5F892DD                    , kmovw(k3, ebp)),
        INSTRUCTION_TEST(C57893EA                    , kmovw(r13d, k2)),
        INSTRUCTION_TEST(C5F89011                    , kmovw(k2, word_ptr(rcx))),
        INSTRUCTION_TEST(C5F893C2                    , kmovw(eax, k2)),
        INSTRUCTION_TEST(C5F893C3                    , kmovw(eax, k3)),
        INSTRUCTION_TEST(C5F893EA                    , kmovw(ebp, k2)),
        INSTRUCTION_TEST(C5F893EB                    , kmovw(ebp, k3)),
        INSTRUCTION_TEST(C5F893CA                    , kmovw(ecx, k2)),
        INSTRUCTION_TEST(C5F892D8                    , kmovw(k3, eax)),
        INSTRUCTION_TEST(C5F892CA                    , kmovw(k1, edx)),
        INSTRUCTION_TEST(C5F890CA                    , kmovw(k1, k2)),
        INSTRUCTION_TEST(C5F8908C1A80000000          , kmovw(k1, word_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(C5F890D5                    , kmovw(k2, k5)),
        INSTRUCTION_TEST(C4A1789094F034120000        , kmovw(k2, word_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(C5F944CA                    , knotb(k1, k2)),
        INSTRUCTION_TEST(C4E1F944CA                  , knotd(k1, k2)),
        INSTRUCTION_TEST(C4E1F844CA                  , knotq(k1, k2)),
        INSTRUCTION_TEST(C5F844E3                    , knotw(k4, k3)),
        INSTRUCTION_TEST(C5F844CA                    , knotw(k1, k2)),
        INSTRUCTION_TEST(C5F844DE                    , knotw(k3, k6)),
        INSTRUCTION_TEST(C5ED45CB                    , korb(k1, k2, k3)),
        INSTRUCTION_TEST(C4E1ED45CB                  , kord(k1, k2, k3)),
        INSTRUCTION_TEST(C4E1EC45CB                  , korq(k1, k2, k3)),
        INSTRUCTION_TEST(C5F998CA                    , kortestb(k1, k2)),
        INSTRUCTION_TEST(C4E1F998CA                  , kortestd(k1, k2)),
        INSTRUCTION_TEST(C4E1F898CA                  , kortestq(k1, k2)),
        INSTRUCTION_TEST(C5F898DE                    , kortestw(k3, k6)),
        INSTRUCTION_TEST(C5F898CA                    , kortestw(k1, k2)),
        INSTRUCTION_TEST(C5F898D6                    , kortestw(k2, k6)),
        INSTRUCTION_TEST(C5EC45CB                    , korw(k1, k2, k3)),
        INSTRUCTION_TEST(C5E445D5                    , korw(k2, k3, k5)),
        INSTRUCTION_TEST(C5CC45E7                    , korw(k4, k6, k7)),
        INSTRUCTION_TEST(C4E37932CA01                , kshiftlb(k1, k2, Imm(1))),
        INSTRUCTION_TEST(C4E37933CA01                , kshiftld(k1, k2, Imm(1))),
        INSTRUCTION_TEST(C4E3F933CA01                , kshiftlq(k1, k2, Imm(1))),
        INSTRUCTION_TEST(C4E3F932D57B                , kshiftlw(k2, k5, Imm(123))),
        INSTRUCTION_TEST(C4E3F932CA01                , kshiftlw(k1, k2, Imm(1))),
        INSTRUCTION_TEST(C4E3F932D5AB                , kshiftlw(k2, k5, Imm(171))),
        INSTRUCTION_TEST(C4E3F932DB7B                , kshiftlw(k3, k3, Imm(123))),
        INSTRUCTION_TEST(C4E3F932DBAB                , kshiftlw(k3, k3, Imm(171))),
        INSTRUCTION_TEST(C4E37930CA01                , kshiftrb(k1, k2, Imm(1))),
        INSTRUCTION_TEST(C4E37931CA01                , kshiftrd(k1, k2, Imm(1))),
        INSTRUCTION_TEST(C4E3F931CA01                , kshiftrq(k1, k2, Imm(1))),
        INSTRUCTION_TEST(C4E3F930E4AB                , kshiftrw(k4, k4, Imm(171))),
        INSTRUCTION_TEST(C4E3F930E47B                , kshiftrw(k4, k4, Imm(123))),
        INSTRUCTION_TEST(C4E3F930DDAB                , kshiftrw(k3, k5, Imm(171))),
        INSTRUCTION_TEST(C4E3F930CA01                , kshiftrw(k1, k2, Imm(1))),
        INSTRUCTION_TEST(C4E3F930DD7B                , kshiftrw(k3, k5, Imm(123))),
        INSTRUCTION_TEST(C5F999CA                    , ktestb(k1, k2)),
        INSTRUCTION_TEST(C4E1F999CA                  , ktestd(k1, k2)),
        INSTRUCTION_TEST(C4E1F899CA                  , ktestq(k1, k2)),
        INSTRUCTION_TEST(C5F899CA                    , ktestw(k1, k2)),
        INSTRUCTION_TEST(C5ED4BCB                    , kunpckbw(k1, k2, k3)),
        INSTRUCTION_TEST(C5E54BD7                    , kunpckbw(k2, k3, k7)),
        INSTRUCTION_TEST(C5D54BEE                    , kunpckbw(k5, k5, k6)),
        INSTRUCTION_TEST(C4E1EC4BCB                  , kunpckdq(k1, k2, k3)),
        INSTRUCTION_TEST(C5EC4BCB                    , kunpckwd(k1, k2, k3)),
        INSTRUCTION_TEST(C5ED46CB                    , kxnorb(k1, k2, k3)),
        INSTRUCTION_TEST(C4E1ED46CB                  , kxnord(k1, k2, k3)),
        INSTRUCTION_TEST(C4E1EC46CB                  , kxnorq(k1, k2, k3)),
        INSTRUCTION_TEST(C5D446DD                    , kxnorw(k3, k5, k5)),
        INSTRUCTION_TEST(C5CC46D7                    , kxnorw(k2, k6, k7)),
        INSTRUCTION_TEST(C5EC46CB                    , kxnorw(k1, k2, k3)),
        INSTRUCTION_TEST(C5ED47CB                    , kxorb(k1, k2, k3)),
        INSTRUCTION_TEST(C4E1ED47CB                  , kxord(k1, k2, k3)),
        INSTRUCTION_TEST(C4E1EC47CB                  , kxorq(k1, k2, k3)),
        INSTRUCTION_TEST(C5EC47CB                    , kxorw(k1, k2, k3)),
        INSTRUCTION_TEST(C5CC47D7                    , kxorw(k2, k6, k7)),
        INSTRUCTION_TEST(C5DC47DE                    , kxorw(k3, k4, k6)),
        INSTRUCTION_TEST(9F                          , lahf()),
        INSTRUCTION_TEST(660F02CA                    , lar(cx, dx)),
        INSTRUCTION_TEST(660F028C1A80000000          , lar(cx, word_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(0F02CA                      , lar(ecx, edx)),
        INSTRUCTION_TEST(0F028C1A80000000            , lar(ecx, word_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(F20FF08C1A80000000          , lddqu(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(0FAE941180000000            , ldmxcsr(dword_ptr(rcx, rdx, 1, 128))),
        INSTRUCTION_TEST(C9                          , leave()),
        INSTRUCTION_TEST(0FAEE8                      , lfence()),
        INSTRUCTION_TEST(660FB48C1A80000000          , lfs(cx, dword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(480FB48C1A80000000          , lfs(rcx, tbyte_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(480FB58C1A80000000          , lgs(rcx, tbyte_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660FB58C1A80000000          , lgs(cx, dword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(0F00D1                      , lldt(cx)),
        INSTRUCTION_TEST(0F00941180000000            , lldt(word_ptr(rcx, rdx, 1, 128))),
        INSTRUCTION_TEST(8FE97812C1                  , llwpcb(ecx)),
        INSTRUCTION_TEST(8FE9F812C1                  , llwpcb(rcx)),
        INSTRUCTION_TEST(0F01F1                      , lmsw(cx)),
        INSTRUCTION_TEST(0F01B41180000000            , lmsw(word_ptr(rcx, rdx, 1, 128))),
        INSTRUCTION_TEST(AC                          , lodsb()),
        INSTRUCTION_TEST(AD                          , lodsd()),
        INSTRUCTION_TEST(48AD                        , lodsq()),
        INSTRUCTION_TEST(480F038C1A80000000          , lsl(rcx, word_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(480F03CA                    , lsl(rcx, edx)),
        INSTRUCTION_TEST(0F038C1A80000000            , lsl(ecx, word_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(0F03CA                      , lsl(ecx, edx)),
        INSTRUCTION_TEST(660F038C1A80000000          , lsl(cx, word_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660F03CA                    , lsl(cx, dx)),
        INSTRUCTION_TEST(660FB28C1A80000000          , lss(cx, dword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(480FB28C1A80000000          , lss(rcx, tbyte_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(0F00D9                      , ltr(cx)),
        INSTRUCTION_TEST(0F009C1180000000            , ltr(word_ptr(rcx, rdx, 1, 128))),
        INSTRUCTION_TEST(8FEA7012841A8000000001000000, lwpins(ecx, dword_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(8FEA7012C201000000          , lwpins(ecx, edx, Imm(1))),
        INSTRUCTION_TEST(8FEAF012841A8000000001000000, lwpins(rcx, dword_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(8FEAF012C201000000          , lwpins(rcx, edx, Imm(1))),
        INSTRUCTION_TEST(8FEAF0128C1A8000000001000000, lwpval(rcx, dword_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(8FEAF012CA01000000          , lwpval(rcx, edx, Imm(1))),
        INSTRUCTION_TEST(8FEA70128C1A8000000001000000, lwpval(ecx, dword_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(8FEA7012CA01000000          , lwpval(ecx, edx, Imm(1))),
        INSTRUCTION_TEST(F3480FBDCA                  , lzcnt(rcx, rdx)),
        INSTRUCTION_TEST(F3480FBD8C1A80000000        , lzcnt(rcx, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(F30FBDCA                    , lzcnt(ecx, edx)),
        INSTRUCTION_TEST(F3660FBD8C1A80000000        , lzcnt(cx, word_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(F3660FBDCA                  , lzcnt(cx, dx)),
        INSTRUCTION_TEST(F30FBD8C1A80000000          , lzcnt(ecx, dword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660FF7CA                    , maskmovdqu(xmm1, xmm2)),
        INSTRUCTION_TEST(0FF7CA                      , maskmovq(mm1, mm2)),
        INSTRUCTION_TEST(660F5FCA                    , maxpd(xmm1, xmm2)),
        INSTRUCTION_TEST(660F5F8C1A80000000          , maxpd(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(0F5FCA                      , maxps(xmm1, xmm2)),
        INSTRUCTION_TEST(0F5F8C1A80000000            , maxps(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(F20F5F8C1A80000000          , maxsd(xmm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(F20F5FCA                    , maxsd(xmm1, xmm2)),
        INSTRUCTION_TEST(F30F5F8C1A80000000          , maxss(xmm1, dword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(F30F5FCA                    , maxss(xmm1, xmm2)),
        INSTRUCTION_TEST(F30F01FA                    , mcommit()),
        INSTRUCTION_TEST(0FAEF0                      , mfence()),
        INSTRUCTION_TEST(660F5DCA                    , minpd(xmm1, xmm2)),
        INSTRUCTION_TEST(660F5D8C1A80000000          , minpd(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(0F5DCA                      , minps(xmm1, xmm2)),
        INSTRUCTION_TEST(0F5D8C1A80000000            , minps(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(F20F5D8C1A80000000          , minsd(xmm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(F20F5DCA                    , minsd(xmm1, xmm2)),
        INSTRUCTION_TEST(F30F5D8C1A80000000          , minss(xmm1, dword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(F30F5DCA                    , minss(xmm1, xmm2)),
        INSTRUCTION_TEST(440F2000                    , mov(rax, cr8)),
        INSTRUCTION_TEST(0F2000                      , mov(rax, cr0)),
        INSTRUCTION_TEST(48899C1180000000            , mov(qword_ptr(rcx, rdx, 1, 128), rbx)),
        INSTRUCTION_TEST(B901000000                  , mov(ecx, Imm(1))),
        INSTRUCTION_TEST(48BB8877665544332211        , mov(rbx, Imm64(0x001122334455667788LL))),
        INSTRUCTION_TEST(8EE2                        , mov(fs, dx)),
        INSTRUCTION_TEST(48C784118000000001000000    , mov(qword_ptr(rcx, rdx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(0F2011                      , mov(rcx, cr2)),
        INSTRUCTION_TEST(8EA41A80000000              , mov(fs, word_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(488CE1                      , mov(rcx, fs)),
        INSTRUCTION_TEST(48C7C101000000              , mov(rcx, Imm(1))),
        INSTRUCTION_TEST(488B8C1A80000000            , mov(rcx, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(4889D1                      , mov(rcx, rdx)),
        INSTRUCTION_TEST(66899C1180000000            , mov(word_ptr(rcx, rdx, 1, 128), bx)),
        INSTRUCTION_TEST(8CA41180000000              , mov(word_ptr(rcx, rdx, 1, 128), fs)),
        INSTRUCTION_TEST(66C78411800000000100        , mov(word_ptr(rcx, rdx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(89D1                        , mov(ecx, edx)),
        INSTRUCTION_TEST(0F2111                      , mov(rcx, dr2)),
        INSTRUCTION_TEST(8B8C1A80000000              , mov(ecx, dword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(8CE1                        , mov(ecx, fs)),
        INSTRUCTION_TEST(C784118000000001000000      , mov(dword_ptr(rcx, rdx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(88BC1180000000              , mov(byte_ptr(rcx, rdx, 1, 128), bh)),
        INSTRUCTION_TEST(889C1180000000              , mov(byte_ptr(rcx, rdx, 1, 128), bl)),
        INSTRUCTION_TEST(8AAC1A80000000              , mov(ch, byte_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(88F5                        , mov(ch, dh)),
        INSTRUCTION_TEST(88D5                        , mov(ch, dl)),
        INSTRUCTION_TEST(B501                        , mov(ch, Imm(1))),
        INSTRUCTION_TEST(8A8C1A80000000              , mov(cl, byte_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(88F1                        , mov(cl, dh)),
        INSTRUCTION_TEST(88D1                        , mov(cl, dl)),
        INSTRUCTION_TEST(C684118000000001            , mov(byte_ptr(rcx, rdx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(6689D1                      , mov(cx, dx)),
        INSTRUCTION_TEST(668CE1                      , mov(cx, fs)),
        INSTRUCTION_TEST(66B90100                    , mov(cx, Imm(1))),
        INSTRUCTION_TEST(668B8C1A80000000            , mov(cx, word_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(0F230A                      , mov(dr1, rdx)),
        INSTRUCTION_TEST(B101                        , mov(cl, Imm(1))),
        INSTRUCTION_TEST(B8E8030000                  , mov(eax, Imm(1000))),
        INSTRUCTION_TEST(899C1180000000              , mov(dword_ptr(rcx, rdx, 1, 128), ebx)),
        INSTRUCTION_TEST(660F288C1A80000000          , movapd(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660F299C1180000000          , movapd(xmmword_ptr(rcx, rdx, 1, 128), xmm3)),
        INSTRUCTION_TEST(660F28CA                    , movapd(xmm1, xmm2)),
        INSTRUCTION_TEST(0F28CA                      , movaps(xmm1, xmm2)),
        INSTRUCTION_TEST(0F288C1A80000000            , movaps(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(0F299C1180000000            , movaps(xmmword_ptr(rcx, rdx, 1, 128), xmm3)),
        INSTRUCTION_TEST(660F38F08C1A80000000        , movbe(cx, word_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(0F38F19C1180000000          , movbe(dword_ptr(rcx, rdx, 1, 128), ebx)),
        INSTRUCTION_TEST(0F38F08C1A80000000          , movbe(ecx, dword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(480F38F19C1180000000        , movbe(qword_ptr(rcx, rdx, 1, 128), rbx)),
        INSTRUCTION_TEST(480F38F08C1A80000000        , movbe(rcx, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660F38F19C1180000000        , movbe(word_ptr(rcx, rdx, 1, 128), bx)),
        INSTRUCTION_TEST(660F6E8C1A80000000          , movd(xmm1, dword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660F6ECA                    , movd(xmm1, edx)),
        INSTRUCTION_TEST(0F6E8C1A80000000            , movd(mm1, dword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(0F6ECA                      , movd(mm1, edx)),
        INSTRUCTION_TEST(0F7ED1                      , movd(ecx, mm2)),
        INSTRUCTION_TEST(660F7ED1                    , movd(ecx, xmm2)),
        INSTRUCTION_TEST(660F7E9C1180000000          , movd(dword_ptr(rcx, rdx, 1, 128), xmm3)),
        INSTRUCTION_TEST(0F7E9C1180000000            , movd(dword_ptr(rcx, rdx, 1, 128), mm3)),
        INSTRUCTION_TEST(F20F128C1A80000000          , movddup(xmm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(F20F12CA                    , movddup(xmm1, xmm2)),
        INSTRUCTION_TEST(0F38F99C1180000000          , movdiri(dword_ptr(rcx, rdx, 1, 128), ebx)),
        INSTRUCTION_TEST(480F38F99C1180000000        , movdiri(qword_ptr(rcx, rdx, 1, 128), rbx)),
        INSTRUCTION_TEST(F20FD6CA                    , movdq2q(mm1, xmm2)),
        INSTRUCTION_TEST(660F6FCA                    , movdqa(xmm1, xmm2)),
        INSTRUCTION_TEST(660F6F8C1A80000000          , movdqa(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660F7F9C1180000000          , movdqa(xmmword_ptr(rcx, rdx, 1, 128), xmm3)),
        INSTRUCTION_TEST(F30F6FCA                    , movdqu(xmm1, xmm2)),
        INSTRUCTION_TEST(F30F6F8C1A80000000          , movdqu(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(F30F7F9C1180000000          , movdqu(xmmword_ptr(rcx, rdx, 1, 128), xmm3)),
        INSTRUCTION_TEST(0F12CA                      , movhlps(xmm1, xmm2)),
        INSTRUCTION_TEST(660F179C1180000000          , movhpd(qword_ptr(rcx, rdx, 1, 128), xmm3)),
        INSTRUCTION_TEST(660F168C1A80000000          , movhpd(xmm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(0F168C1A80000000            , movhps(xmm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(0F179C1180000000            , movhps(qword_ptr(rcx, rdx, 1, 128), xmm3)),
        INSTRUCTION_TEST(0F16CA                      , movlhps(xmm1, xmm2)),
        INSTRUCTION_TEST(660F139C1180000000          , movlpd(qword_ptr(rcx, rdx, 1, 128), xmm3)),
        INSTRUCTION_TEST(660F128C1A80000000          , movlpd(xmm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(0F139C1180000000            , movlps(qword_ptr(rcx, rdx, 1, 128), xmm3)),
        INSTRUCTION_TEST(0F128C1A80000000            , movlps(xmm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660F50CA                    , movmskpd(ecx, xmm2)),
        INSTRUCTION_TEST(0F50CA                      , movmskps(ecx, xmm2)),
        INSTRUCTION_TEST(660FE79C1180000000          , movntdq(xmmword_ptr(rcx, rdx, 1, 128), xmm3)),
        INSTRUCTION_TEST(660F382A8C1A80000000        , movntdqa(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(480FC39C1180000000          , movnti(qword_ptr(rcx, rdx, 1, 128), rbx)),
        INSTRUCTION_TEST(0FC39C1180000000            , movnti(dword_ptr(rcx, rdx, 1, 128), ebx)),
        INSTRUCTION_TEST(660F2B9C1180000000          , movntpd(xmmword_ptr(rcx, rdx, 1, 128), xmm3)),
        INSTRUCTION_TEST(0F2B9C1180000000            , movntps(xmmword_ptr(rcx, rdx, 1, 128), xmm3)),
        INSTRUCTION_TEST(0FE79C1180000000            , movntq(qword_ptr(rcx, rdx, 1, 128), mm3)),
        INSTRUCTION_TEST(F20F2B9C1180000000          , movntsd(qword_ptr(rcx, rdx, 1, 128), xmm3)),
        INSTRUCTION_TEST(F30F2B9C1180000000          , movntss(dword_ptr(rcx, rdx, 1, 128), xmm3)),
        INSTRUCTION_TEST(F30F7E8C1A80000000          , movq(xmm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(66480F7ED1                  , movq(rcx, xmm2)),
        INSTRUCTION_TEST(F30F7ECA                    , movq(xmm1, xmm2)),
        INSTRUCTION_TEST(480F7ED1                    , movq(rcx, mm2)),
        INSTRUCTION_TEST(66480F6ECA                  , movq(xmm1, rdx)),
        INSTRUCTION_TEST(0F7F9C1180000000            , movq(qword_ptr(rcx, rdx, 1, 128), mm3)),
        INSTRUCTION_TEST(480F6ECA                    , movq(mm1, rdx)),
        INSTRUCTION_TEST(0F6F8C1A80000000            , movq(mm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(0F6FCA                      , movq(mm1, mm2)),
        INSTRUCTION_TEST(660FD69C1180000000          , movq(qword_ptr(rcx, rdx, 1, 128), xmm3)),
        INSTRUCTION_TEST(F30FD6CA                    , movq2dq(xmm1, mm2)),
        INSTRUCTION_TEST(A4                          , movsb()),
        INSTRUCTION_TEST(F20F10CA                    , movsd(xmm1, xmm2)),
        INSTRUCTION_TEST(F20F108C1A80000000          , movsd(xmm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(A5                          , movsd()),
        INSTRUCTION_TEST(F20F119C1180000000          , movsd(qword_ptr(rcx, rdx, 1, 128), xmm3)),
        INSTRUCTION_TEST(F30F16CA                    , movshdup(xmm1, xmm2)),
        INSTRUCTION_TEST(F30F168C1A80000000          , movshdup(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(F30F12CA                    , movsldup(xmm1, xmm2)),
        INSTRUCTION_TEST(F30F128C1A80000000          , movsldup(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(48A5                        , movsq()),
        INSTRUCTION_TEST(F30F108C1A80000000          , movss(xmm1, dword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(F30F119C1180000000          , movss(dword_ptr(rcx, rdx, 1, 128), xmm3)),
        INSTRUCTION_TEST(F30F10CA                    , movss(xmm1, xmm2)),
        INSTRUCTION_TEST(66A5                        , movsw()),
        INSTRUCTION_TEST(660FBE8C1A80000000          , movsx(cx, byte_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(480FBF8C1A80000000          , movsx(rcx, word_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(480FBFCA                    , movsx(rcx, dx)),
        INSTRUCTION_TEST(480FBECA                    , movsx(rcx, dl)),
        INSTRUCTION_TEST(480FBE8C1A80000000          , movsx(rcx, byte_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(480FBF07                    , movsx(rax, word_ptr(rdi))),
        INSTRUCTION_TEST(480FBE07                    , movsx(rax, byte_ptr(rdi))),
        INSTRUCTION_TEST(0FBFCA                      , movsx(ecx, dx)),
        INSTRUCTION_TEST(0FBF8C1A80000000            , movsx(ecx, word_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(0FBECE                      , movsx(ecx, dh)),
        INSTRUCTION_TEST(0FBE8C1A80000000            , movsx(ecx, byte_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(0FBF07                      , movsx(eax, word_ptr(rdi))),
        INSTRUCTION_TEST(0FBE07                      , movsx(eax, byte_ptr(rdi))),
        INSTRUCTION_TEST(660FBECA                    , movsx(cx, dl)),
        INSTRUCTION_TEST(660FBECE                    , movsx(cx, dh)),
        INSTRUCTION_TEST(0FBECA                      , movsx(ecx, dl)),
        INSTRUCTION_TEST(4863CA                      , movsxd(rcx, edx)),
        INSTRUCTION_TEST(48638C1A80000000            , movsxd(rcx, dword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(4863C3                      , movsxd(rax, ebx)),
        INSTRUCTION_TEST(63CA                        , movsxd(ecx, edx)),
        INSTRUCTION_TEST(486307                      , movsxd(rax, dword_ptr(rdi))),
        INSTRUCTION_TEST(63C3                        , movsxd(eax, ebx)),
        INSTRUCTION_TEST(66638C1A80000000            , movsxd(cx, dword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(6663CA                      , movsxd(cx, dx)),
        INSTRUCTION_TEST(6663C3                      , movsxd(ax, bx)),
        INSTRUCTION_TEST(638C1A80000000              , movsxd(ecx, dword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660F10CA                    , movupd(xmm1, xmm2)),
        INSTRUCTION_TEST(660F108C1A80000000          , movupd(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660F119C1180000000          , movupd(xmmword_ptr(rcx, rdx, 1, 128), xmm3)),
        INSTRUCTION_TEST(0F119C1180000000            , movups(xmmword_ptr(rcx, rdx, 1, 128), xmm3)),
        INSTRUCTION_TEST(0F108C1A80000000            , movups(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(0F10CA                      , movups(xmm1, xmm2)),
        INSTRUCTION_TEST(480FB78C1A80000000          , movzx(rcx, word_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(480FB7CA                    , movzx(rcx, dx)),
        INSTRUCTION_TEST(480FB6CA                    , movzx(rcx, dl)),
        INSTRUCTION_TEST(480FB68C1A80000000          , movzx(rcx, byte_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(480FB707                    , movzx(rax, word_ptr(rdi))),
        INSTRUCTION_TEST(480FB607                    , movzx(rax, byte_ptr(rdi))),
        INSTRUCTION_TEST(440FB6FA                    , movzx(r15d, dl)),
        INSTRUCTION_TEST(440FB6FD                    , movzx(r15d, bpl)),
        INSTRUCTION_TEST(0FB7CA                      , movzx(ecx, dx)),
        INSTRUCTION_TEST(0FB78C1A80000000            , movzx(ecx, word_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(0FB6CE                      , movzx(ecx, dh)),
        INSTRUCTION_TEST(0FB68C1A80000000            , movzx(ecx, byte_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(0FB707                      , movzx(eax, word_ptr(rdi))),
        INSTRUCTION_TEST(0FB6C6                      , movzx(eax, dh)),
        INSTRUCTION_TEST(0FB607                      , movzx(eax, byte_ptr(rdi))),
        INSTRUCTION_TEST(660FB6CA                    , movzx(cx, dl)),
        INSTRUCTION_TEST(660FB6CE                    , movzx(cx, dh)),
        INSTRUCTION_TEST(660FB68C1A80000000          , movzx(cx, byte_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(0FB6CA                      , movzx(ecx, dl)),
        INSTRUCTION_TEST(660F3A42CA01                , mpsadbw(xmm1, xmm2, Imm(1))),
        INSTRUCTION_TEST(660F3A428C1A8000000001      , mpsadbw(xmm1, xmmword_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(66F7A41180000000            , mul(word_ptr(rcx, rdx, 1, 128))),
        INSTRUCTION_TEST(48F7E1                      , mul(rcx)),
        INSTRUCTION_TEST(48F7A41180000000            , mul(qword_ptr(rcx, rdx, 1, 128))),
        INSTRUCTION_TEST(F7E1                        , mul(ecx)),
        INSTRUCTION_TEST(F6A41180000000              , mul(byte_ptr(rcx, rdx, 1, 128))),
        INSTRUCTION_TEST(66F7E1                      , mul(cx)),
        INSTRUCTION_TEST(F6E1                        , mul(cl)),
        INSTRUCTION_TEST(F6E5                        , mul(ch)),
        INSTRUCTION_TEST(F7A41180000000              , mul(dword_ptr(rcx, rdx, 1, 128))),
        INSTRUCTION_TEST(660F598C1A80000000          , mulpd(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660F59CA                    , mulpd(xmm1, xmm2)),
        INSTRUCTION_TEST(0F598C1A80000000            , mulps(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(0F59CA                      , mulps(xmm1, xmm2)),
        INSTRUCTION_TEST(F20F598C1A80000000          , mulsd(xmm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(F20F59CA                    , mulsd(xmm1, xmm2)),
        INSTRUCTION_TEST(F30F598C1A80000000          , mulss(xmm1, dword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(F30F59CA                    , mulss(xmm1, xmm2)),
        INSTRUCTION_TEST(C4E26BF68C2B80000000        , mulx(ecx, edx, dword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E26BF6CB                  , mulx(ecx, edx, ebx)),
        INSTRUCTION_TEST(C4E2EBF68C2B80000000        , mulx(rcx, rdx, qword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E2EBF6CB                  , mulx(rcx, rdx, rbx)),
        INSTRUCTION_TEST(0F01C9                      , mwait()),
        INSTRUCTION_TEST(0F01FB                      , mwaitx()),
        INSTRUCTION_TEST(66F79C1180000000            , neg(word_ptr(rcx, rdx, 1, 128))),
        INSTRUCTION_TEST(40F6DE                      , neg(sil)),
        INSTRUCTION_TEST(48F79C1180000000            , neg(qword_ptr(rcx, rdx, 1, 128))),
        INSTRUCTION_TEST(F7D9                        , neg(ecx)),
        INSTRUCTION_TEST(F7D8                        , neg(eax)),
        INSTRUCTION_TEST(F79C1180000000              , neg(dword_ptr(rcx, rdx, 1, 128))),
        INSTRUCTION_TEST(48F7D9                      , neg(rcx)),
        INSTRUCTION_TEST(F6D9                        , neg(cl)),
        INSTRUCTION_TEST(F6DD                        , neg(ch)),
        INSTRUCTION_TEST(F69C1180000000              , neg(byte_ptr(rcx, rdx, 1, 128))),
        INSTRUCTION_TEST(F6D8                        , neg(al)),
        INSTRUCTION_TEST(F6DC                        , neg(ah)),
        INSTRUCTION_TEST(66F7D9                      , neg(cx)),
        INSTRUCTION_TEST(660F18A41180000000          , nop(word_ptr(rcx, rdx, 1, 128))),
        INSTRUCTION_TEST(660F199C1180000000          , nop(word_ptr(rcx, rdx, 1, 128), bx)),
        INSTRUCTION_TEST(480F18C1                    , nop(rcx)),
        INSTRUCTION_TEST(480F0DD1                    , nop(rcx, rdx)),
        INSTRUCTION_TEST(480F18A41180000000          , nop(qword_ptr(rcx, rdx, 1, 128))),
        INSTRUCTION_TEST(480F199C1180000000          , nop(qword_ptr(rcx, rdx, 1, 128), rbx)),
        INSTRUCTION_TEST(0F0DD1                      , nop(ecx, edx)),
        INSTRUCTION_TEST(0F18A41180000000            , nop(dword_ptr(rcx, rdx, 1, 128))),
        INSTRUCTION_TEST(0F199C1180000000            , nop(dword_ptr(rcx, rdx, 1, 128), ebx)),
        INSTRUCTION_TEST(660F18C1                    , nop(cx)),
        INSTRUCTION_TEST(660F0DD1                    , nop(cx, dx)),
        INSTRUCTION_TEST(90                          , nop()),
        INSTRUCTION_TEST(0F18C1                      , nop(ecx)),
        INSTRUCTION_TEST(F7D1                        , not_(ecx)),
        INSTRUCTION_TEST(66F7941180000000            , not_(word_ptr(rcx, rdx, 1, 128))),
        INSTRUCTION_TEST(48F7D1                      , not_(rcx)),
        INSTRUCTION_TEST(48F7941180000000            , not_(qword_ptr(rcx, rdx, 1, 128))),
        INSTRUCTION_TEST(F7D0                        , not_(eax)),
        INSTRUCTION_TEST(F6D1                        , not_(cl)),
        INSTRUCTION_TEST(66F7D1                      , not_(cx)),
        INSTRUCTION_TEST(F6D5                        , not_(ch)),
        INSTRUCTION_TEST(F6941180000000              , not_(byte_ptr(rcx, rdx, 1, 128))),
        INSTRUCTION_TEST(F7941180000000              , not_(dword_ptr(rcx, rdx, 1, 128))),
        INSTRUCTION_TEST(66838C118000000001          , or_(word_ptr(rcx, rdx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(66099C1180000000            , or_(word_ptr(rcx, rdx, 1, 128), bx)),
        INSTRUCTION_TEST(4809D1                      , or_(rcx, rdx)),
        INSTRUCTION_TEST(480B8C1A80000000            , or_(rcx, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(4883C901                    , or_(rcx, Imm(1))),
        INSTRUCTION_TEST(48099C1180000000            , or_(qword_ptr(rcx, rdx, 1, 128), rbx)),
        INSTRUCTION_TEST(48838C118000000001          , or_(qword_ptr(rcx, rdx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(83C901                      , or_(ecx, Imm(1))),
        INSTRUCTION_TEST(09D1                        , or_(ecx, edx)),
        INSTRUCTION_TEST(0B8C1A80000000              , or_(ecx, dword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(838C118000000001            , or_(dword_ptr(rcx, rdx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(099C1180000000              , or_(dword_ptr(rcx, rdx, 1, 128), ebx)),
        INSTRUCTION_TEST(660B8C1A80000000            , or_(cx, word_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(6609D1                      , or_(cx, dx)),
        INSTRUCTION_TEST(08BC1180000000              , or_(byte_ptr(rcx, rdx, 1, 128), bh)),
        INSTRUCTION_TEST(089C1180000000              , or_(byte_ptr(rcx, rdx, 1, 128), bl)),
        INSTRUCTION_TEST(808C118000000001            , or_(byte_ptr(rcx, rdx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(6683C901                    , or_(cx, Imm(1))),
        INSTRUCTION_TEST(08F5                        , or_(ch, dh)),
        INSTRUCTION_TEST(0AAC1A80000000              , or_(ch, byte_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(80CD01                      , or_(ch, Imm(1))),
        INSTRUCTION_TEST(0A8C1A80000000              , or_(cl, byte_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(08F1                        , or_(cl, dh)),
        INSTRUCTION_TEST(08D1                        , or_(cl, dl)),
        INSTRUCTION_TEST(80C901                      , or_(cl, Imm(1))),
        INSTRUCTION_TEST(08D5                        , or_(ch, dl)),
        INSTRUCTION_TEST(660F56CA                    , orpd(xmm1, xmm2)),
        INSTRUCTION_TEST(660F568C1A80000000          , orpd(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(0F568C1A80000000            , orps(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(0F56CA                      , orps(xmm1, xmm2)),
        INSTRUCTION_TEST(660F381C8C1A80000000        , pabsb(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660F381CCA                  , pabsb(xmm1, xmm2)),
        INSTRUCTION_TEST(0F381C8C1A80000000          , pabsb(mm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(0F381CCA                    , pabsb(mm1, mm2)),
        INSTRUCTION_TEST(0F381ECA                    , pabsd(mm1, mm2)),
        INSTRUCTION_TEST(0F381E8C1A80000000          , pabsd(mm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660F381ECA                  , pabsd(xmm1, xmm2)),
        INSTRUCTION_TEST(660F381E8C1A80000000        , pabsd(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660F381DCA                  , pabsw(xmm1, xmm2)),
        INSTRUCTION_TEST(660F381D8C1A80000000        , pabsw(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(0F381DCA                    , pabsw(mm1, mm2)),
        INSTRUCTION_TEST(0F381D8C1A80000000          , pabsw(mm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(0F6BCA                      , packssdw(mm1, mm2)),
        INSTRUCTION_TEST(0F6B8C1A80000000            , packssdw(mm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660F6BCA                    , packssdw(xmm1, xmm2)),
        INSTRUCTION_TEST(660F6B8C1A80000000          , packssdw(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(0F63CA                      , packsswb(mm1, mm2)),
        INSTRUCTION_TEST(0F638C1A80000000            , packsswb(mm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660F63CA                    , packsswb(xmm1, xmm2)),
        INSTRUCTION_TEST(660F638C1A80000000          , packsswb(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660F382B8C1A80000000        , packusdw(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660F382BCA                  , packusdw(xmm1, xmm2)),
        INSTRUCTION_TEST(0F67CA                      , packuswb(mm1, mm2)),
        INSTRUCTION_TEST(0F678C1A80000000            , packuswb(mm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660F67CA                    , packuswb(xmm1, xmm2)),
        INSTRUCTION_TEST(660F678C1A80000000          , packuswb(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(0FFCCA                      , paddb(mm1, mm2)),
        INSTRUCTION_TEST(0FFC8C1A80000000            , paddb(mm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660FFCCA                    , paddb(xmm1, xmm2)),
        INSTRUCTION_TEST(660FFC8C1A80000000          , paddb(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(0FFECA                      , paddd(mm1, mm2)),
        INSTRUCTION_TEST(0FFE8C1A80000000            , paddd(mm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660FFECA                    , paddd(xmm1, xmm2)),
        INSTRUCTION_TEST(660FFE8C1A80000000          , paddd(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(0FD4CA                      , paddq(mm1, mm2)),
        INSTRUCTION_TEST(0FD48C1A80000000            , paddq(mm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660FD4CA                    , paddq(xmm1, xmm2)),
        INSTRUCTION_TEST(660FD48C1A80000000          , paddq(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660FEC8C1A80000000          , paddsb(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660FECCA                    , paddsb(xmm1, xmm2)),
        INSTRUCTION_TEST(0FEC8C1A80000000            , paddsb(mm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(0FECCA                      , paddsb(mm1, mm2)),
        INSTRUCTION_TEST(0FEDCA                      , paddsw(mm1, mm2)),
        INSTRUCTION_TEST(0FED8C1A80000000            , paddsw(mm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660FEDCA                    , paddsw(xmm1, xmm2)),
        INSTRUCTION_TEST(660FED8C1A80000000          , paddsw(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(0FDCCA                      , paddusb(mm1, mm2)),
        INSTRUCTION_TEST(0FDC8C1A80000000            , paddusb(mm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660FDCCA                    , paddusb(xmm1, xmm2)),
        INSTRUCTION_TEST(660FDC8C1A80000000          , paddusb(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660FDD8C1A80000000          , paddusw(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660FDDCA                    , paddusw(xmm1, xmm2)),
        INSTRUCTION_TEST(0FDDCA                      , paddusw(mm1, mm2)),
        INSTRUCTION_TEST(0FDD8C1A80000000            , paddusw(mm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(0FFDCA                      , paddw(mm1, mm2)),
        INSTRUCTION_TEST(0FFD8C1A80000000            , paddw(mm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660FFDCA                    , paddw(xmm1, xmm2)),
        INSTRUCTION_TEST(660FFD8C1A80000000          , paddw(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660F3A0F8C1A8000000001      , palignr(xmm1, xmmword_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(660F3A0FCA01                , palignr(xmm1, xmm2, Imm(1))),
        INSTRUCTION_TEST(0F3A0F8C1A8000000001        , palignr(mm1, qword_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(0F3A0FCA01                  , palignr(mm1, mm2, Imm(1))),
        INSTRUCTION_TEST(0FDBCA                      , pand(mm1, mm2)),
        INSTRUCTION_TEST(0FDB8C1A80000000            , pand(mm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660FDBCA                    , pand(xmm1, xmm2)),
        INSTRUCTION_TEST(660FDB8C1A80000000          , pand(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660FDFCA                    , pandn(xmm1, xmm2)),
        INSTRUCTION_TEST(660FDF8C1A80000000          , pandn(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(0FDFCA                      , pandn(mm1, mm2)),
        INSTRUCTION_TEST(0FDF8C1A80000000            , pandn(mm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(F390                        , pause()),
        INSTRUCTION_TEST(0FE0CA                      , pavgb(mm1, mm2)),
        INSTRUCTION_TEST(0FE08C1A80000000            , pavgb(mm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660FE0CA                    , pavgb(xmm1, xmm2)),
        INSTRUCTION_TEST(660FE08C1A80000000          , pavgb(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(0F0F8C1A80000000BF          , pavgusb(mm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(0F0FCABF                    , pavgusb(mm1, mm2)),
        INSTRUCTION_TEST(0FE38C1A80000000            , pavgw(mm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660FE3CA                    , pavgw(xmm1, xmm2)),
        INSTRUCTION_TEST(0FE3CA                      , pavgw(mm1, mm2)),
        INSTRUCTION_TEST(660FE38C1A80000000          , pavgw(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660F38108C1A80000000        , pblendvb(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660F3810CA                  , pblendvb(xmm1, xmm2)),
        INSTRUCTION_TEST(660F3A0ECA01                , pblendw(xmm1, xmm2, Imm(1))),
        INSTRUCTION_TEST(660F3A0E8C1A8000000001      , pblendw(xmm1, xmmword_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(660F3A44CA01                , pclmulqdq(xmm1, xmm2, Imm(1))),
        INSTRUCTION_TEST(660F3A448C1A8000000001      , pclmulqdq(xmm1, xmmword_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(0F74CA                      , pcmpeqb(mm1, mm2)),
        INSTRUCTION_TEST(0F748C1A80000000            , pcmpeqb(mm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660F74CA                    , pcmpeqb(xmm1, xmm2)),
        INSTRUCTION_TEST(660F748C1A80000000          , pcmpeqb(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660F768C1A80000000          , pcmpeqd(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660F76CA                    , pcmpeqd(xmm1, xmm2)),
        INSTRUCTION_TEST(0F76CA                      , pcmpeqd(mm1, mm2)),
        INSTRUCTION_TEST(0F768C1A80000000            , pcmpeqd(mm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660F3829CA                  , pcmpeqq(xmm1, xmm2)),
        INSTRUCTION_TEST(660F38298C1A80000000        , pcmpeqq(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660F758C1A80000000          , pcmpeqw(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(0F75CA                      , pcmpeqw(mm1, mm2)),
        INSTRUCTION_TEST(0F758C1A80000000            , pcmpeqw(mm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660F75CA                    , pcmpeqw(xmm1, xmm2)),
        INSTRUCTION_TEST(660F3A618C1A8000000001      , pcmpestri(xmm1, xmmword_ptr(rdx, rbx, 1, 128), 1)),
        INSTRUCTION_TEST(660F3A61CA01                , pcmpestri(xmm1, xmm2, 1)),
        INSTRUCTION_TEST(660F3A60CA01                , pcmpestrm(xmm1, xmm2, 1)),
        INSTRUCTION_TEST(660F3A608C1A8000000001      , pcmpestrm(xmm1, xmmword_ptr(rdx, rbx, 1, 128), 1)),
        INSTRUCTION_TEST(0F64CA                      , pcmpgtb(mm1, mm2)),
        INSTRUCTION_TEST(0F648C1A80000000            , pcmpgtb(mm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660F64CA                    , pcmpgtb(xmm1, xmm2)),
        INSTRUCTION_TEST(660F648C1A80000000          , pcmpgtb(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660F66CA                    , pcmpgtd(xmm1, xmm2)),
        INSTRUCTION_TEST(660F668C1A80000000          , pcmpgtd(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(0F66CA                      , pcmpgtd(mm1, mm2)),
        INSTRUCTION_TEST(0F668C1A80000000            , pcmpgtd(mm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660F3837CA                  , pcmpgtq(xmm1, xmm2)),
        INSTRUCTION_TEST(660F38378C1A80000000        , pcmpgtq(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(0F65CA                      , pcmpgtw(mm1, mm2)),
        INSTRUCTION_TEST(0F658C1A80000000            , pcmpgtw(mm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660F65CA                    , pcmpgtw(xmm1, xmm2)),
        INSTRUCTION_TEST(660F658C1A80000000          , pcmpgtw(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660F3A638C1A8000000001      , pcmpistri(xmm1, xmmword_ptr(rdx, rbx, 1, 128), 1)),
        INSTRUCTION_TEST(660F3A63CA01                , pcmpistri(xmm1, xmm2, 1)),
        INSTRUCTION_TEST(660F3A628C1A8000000001      , pcmpistrm(xmm1, xmmword_ptr(rdx, rbx, 1, 128), 1)),
        INSTRUCTION_TEST(660F3A62CA01                , pcmpistrm(xmm1, xmm2, 1)),
        INSTRUCTION_TEST(0F01C5                      , pconfig()),
        INSTRUCTION_TEST(C4E26BF58C2B80000000        , pdep(ecx, edx, dword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E26BF5CB                  , pdep(ecx, edx, ebx)),
        INSTRUCTION_TEST(C4E2EBF58C2B80000000        , pdep(rcx, rdx, qword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E2EBF5CB                  , pdep(rcx, rdx, rbx)),
        INSTRUCTION_TEST(C4E2EAF58C2B80000000        , pext(rcx, rdx, qword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E2EAF5CB                  , pext(rcx, rdx, rbx)),
        INSTRUCTION_TEST(C4E26AF58C2B80000000        , pext(ecx, edx, dword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E26AF5CB                  , pext(ecx, edx, ebx)),
        INSTRUCTION_TEST(660F3A149C118000000001      , pextrb(byte_ptr(rcx, rdx, 1, 128), xmm3, Imm(1))),
        INSTRUCTION_TEST(660F3A14D101                , pextrb(ecx, xmm2, Imm(1))),
        INSTRUCTION_TEST(660F3A169C118000000001      , pextrd(dword_ptr(rcx, rdx, 1, 128), xmm3, Imm(1))),
        INSTRUCTION_TEST(660F3A16D101                , pextrd(ecx, xmm2, Imm(1))),
        INSTRUCTION_TEST(66480F3A169C118000000001    , pextrq(qword_ptr(rcx, rdx, 1, 128), xmm3, Imm(1))),
        INSTRUCTION_TEST(66480F3A16D101              , pextrq(rcx, xmm2, Imm(1))),
        INSTRUCTION_TEST(660FC5CA01                  , pextrw(ecx, xmm2, Imm(1))),
        INSTRUCTION_TEST(0FC5CA01                    , pextrw(ecx, mm2, Imm(1))),
        INSTRUCTION_TEST(660F3A159C118000000001      , pextrw(word_ptr(rcx, rdx, 1, 128), xmm3, Imm(1))),
        INSTRUCTION_TEST(0F0F8C1A800000001D          , pf2id(mm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(0F0FCA1D                    , pf2id(mm1, mm2)),
        INSTRUCTION_TEST(0F0FCA1C                    , pf2iw(mm1, mm2)),
        INSTRUCTION_TEST(0F0F8C1A800000001C          , pf2iw(mm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(0F0FCAAE                    , pfacc(mm1, mm2)),
        INSTRUCTION_TEST(0F0F8C1A80000000AE          , pfacc(mm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(0F0FCA9E                    , pfadd(mm1, mm2)),
        INSTRUCTION_TEST(0F0F8C1A800000009E          , pfadd(mm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(0F0F8C1A80000000B0          , pfcmpeq(mm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(0F0FCAB0                    , pfcmpeq(mm1, mm2)),
        INSTRUCTION_TEST(0F0F8C1A8000000090          , pfcmpge(mm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(0F0FCA90                    , pfcmpge(mm1, mm2)),
        INSTRUCTION_TEST(0F0FCAA0                    , pfcmpgt(mm1, mm2)),
        INSTRUCTION_TEST(0F0F8C1A80000000A0          , pfcmpgt(mm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(0F0FCAA4                    , pfmax(mm1, mm2)),
        INSTRUCTION_TEST(0F0F8C1A80000000A4          , pfmax(mm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(0F0FCA94                    , pfmin(mm1, mm2)),
        INSTRUCTION_TEST(0F0F8C1A8000000094          , pfmin(mm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(0F0F8C1A80000000B4          , pfmul(mm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(0F0FCAB4                    , pfmul(mm1, mm2)),
        INSTRUCTION_TEST(0F0F8C1A800000008A          , pfnacc(mm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(0F0FCA8A                    , pfnacc(mm1, mm2)),
        INSTRUCTION_TEST(0F0FCA8E                    , pfpnacc(mm1, mm2)),
        INSTRUCTION_TEST(0F0F8C1A800000008E          , pfpnacc(mm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(0F0FCA96                    , pfrcp(mm1, mm2)),
        INSTRUCTION_TEST(0F0F8C1A8000000096          , pfrcp(mm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(0F0FCAB6                    , pfrcpit2(mm1, mm2)),
        INSTRUCTION_TEST(0F0F8C1A80000000B6          , pfrcpit2(mm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(0F0FCAA7                    , pfrsqit1(mm1, mm2)),
        INSTRUCTION_TEST(0F0F8C1A80000000A7          , pfrsqit1(mm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(0F0F8C1A800000009A          , pfsub(mm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(0F0FCA9A                    , pfsub(mm1, mm2)),
        INSTRUCTION_TEST(0F0FCAAA                    , pfsubr(mm1, mm2)),
        INSTRUCTION_TEST(0F0F8C1A80000000AA          , pfsubr(mm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(0F3802CA                    , phaddd(mm1, mm2)),
        INSTRUCTION_TEST(0F38028C1A80000000          , phaddd(mm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660F3802CA                  , phaddd(xmm1, xmm2)),
        INSTRUCTION_TEST(660F38028C1A80000000        , phaddd(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660F38038C1A80000000        , phaddsw(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660F3803CA                  , phaddsw(xmm1, xmm2)),
        INSTRUCTION_TEST(0F38038C1A80000000          , phaddsw(mm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(0F3803CA                    , phaddsw(mm1, mm2)),
        INSTRUCTION_TEST(0F3801CA                    , phaddw(mm1, mm2)),
        INSTRUCTION_TEST(0F38018C1A80000000          , phaddw(mm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660F3801CA                  , phaddw(xmm1, xmm2)),
        INSTRUCTION_TEST(660F38018C1A80000000        , phaddw(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660F3841CA                  , phminposuw(xmm1, xmm2)),
        INSTRUCTION_TEST(660F38418C1A80000000        , phminposuw(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(0F3806CA                    , phsubd(mm1, mm2)),
        INSTRUCTION_TEST(0F38068C1A80000000          , phsubd(mm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660F3806CA                  , phsubd(xmm1, xmm2)),
        INSTRUCTION_TEST(660F38068C1A80000000        , phsubd(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(0F3807CA                    , phsubsw(mm1, mm2)),
        INSTRUCTION_TEST(0F38078C1A80000000          , phsubsw(mm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660F3807CA                  , phsubsw(xmm1, xmm2)),
        INSTRUCTION_TEST(660F38078C1A80000000        , phsubsw(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660F3805CA                  , phsubw(xmm1, xmm2)),
        INSTRUCTION_TEST(660F38058C1A80000000        , phsubw(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(0F3805CA                    , phsubw(mm1, mm2)),
        INSTRUCTION_TEST(0F38058C1A80000000          , phsubw(mm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(0F0FCA0D                    , pi2fd(mm1, mm2)),
        INSTRUCTION_TEST(0F0F8C1A800000000D          , pi2fd(mm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(0F0FCA0C                    , pi2fw(mm1, mm2)),
        INSTRUCTION_TEST(0F0F8C1A800000000C          , pi2fw(mm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660F3A208C1A8000000001      , pinsrb(xmm1, byte_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(660F3A20CA01                , pinsrb(xmm1, edx, Imm(1))),
        INSTRUCTION_TEST(660F3A22CA01                , pinsrd(xmm1, edx, Imm(1))),
        INSTRUCTION_TEST(660F3A228C1A8000000001      , pinsrd(xmm1, dword_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(66480F3A228C1A8000000001    , pinsrq(xmm1, qword_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(66480F3A22CA01              , pinsrq(xmm1, rdx, Imm(1))),
        INSTRUCTION_TEST(0FC4CA01                    , pinsrw(mm1, edx, Imm(1))),
        INSTRUCTION_TEST(0FC48C1A8000000001          , pinsrw(mm1, word_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(660FC4CA01                  , pinsrw(xmm1, edx, Imm(1))),
        INSTRUCTION_TEST(660FC48C1A8000000001        , pinsrw(xmm1, word_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(660F3804CA                  , pmaddubsw(xmm1, xmm2)),
        INSTRUCTION_TEST(660F38048C1A80000000        , pmaddubsw(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(0F3804CA                    , pmaddubsw(mm1, mm2)),
        INSTRUCTION_TEST(0F38048C1A80000000          , pmaddubsw(mm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(0FF5CA                      , pmaddwd(mm1, mm2)),
        INSTRUCTION_TEST(0FF58C1A80000000            , pmaddwd(mm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660FF5CA                    , pmaddwd(xmm1, xmm2)),
        INSTRUCTION_TEST(660FF58C1A80000000          , pmaddwd(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660F383CCA                  , pmaxsb(xmm1, xmm2)),
        INSTRUCTION_TEST(660F383C8C1A80000000        , pmaxsb(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660F383DCA                  , pmaxsd(xmm1, xmm2)),
        INSTRUCTION_TEST(660F383D8C1A80000000        , pmaxsd(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660FEE8C1A80000000          , pmaxsw(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660FEECA                    , pmaxsw(xmm1, xmm2)),
        INSTRUCTION_TEST(0FEECA                      , pmaxsw(mm1, mm2)),
        INSTRUCTION_TEST(0FEE8C1A80000000            , pmaxsw(mm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(0FDECA                      , pmaxub(mm1, mm2)),
        INSTRUCTION_TEST(0FDE8C1A80000000            , pmaxub(mm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660FDECA                    , pmaxub(xmm1, xmm2)),
        INSTRUCTION_TEST(660FDE8C1A80000000          , pmaxub(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660F383FCA                  , pmaxud(xmm1, xmm2)),
        INSTRUCTION_TEST(660F383F8C1A80000000        , pmaxud(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660F383ECA                  , pmaxuw(xmm1, xmm2)),
        INSTRUCTION_TEST(660F383E8C1A80000000        , pmaxuw(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660F38388C1A80000000        , pminsb(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660F3838CA                  , pminsb(xmm1, xmm2)),
        INSTRUCTION_TEST(660F38398C1A80000000        , pminsd(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660F3839CA                  , pminsd(xmm1, xmm2)),
        INSTRUCTION_TEST(0FEACA                      , pminsw(mm1, mm2)),
        INSTRUCTION_TEST(0FEA8C1A80000000            , pminsw(mm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660FEACA                    , pminsw(xmm1, xmm2)),
        INSTRUCTION_TEST(660FEA8C1A80000000          , pminsw(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(0FDACA                      , pminub(mm1, mm2)),
        INSTRUCTION_TEST(0FDA8C1A80000000            , pminub(mm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660FDACA                    , pminub(xmm1, xmm2)),
        INSTRUCTION_TEST(660FDA8C1A80000000          , pminub(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660F383B8C1A80000000        , pminud(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660F383BCA                  , pminud(xmm1, xmm2)),
        INSTRUCTION_TEST(660F383ACA                  , pminuw(xmm1, xmm2)),
        INSTRUCTION_TEST(660F383A8C1A80000000        , pminuw(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(0FD7CA                      , pmovmskb(ecx, mm2)),
        INSTRUCTION_TEST(660FD7CA                    , pmovmskb(ecx, xmm2)),
        INSTRUCTION_TEST(660F38218C1A80000000        , pmovsxbd(xmm1, dword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660F3821CA                  , pmovsxbd(xmm1, xmm2)),
        INSTRUCTION_TEST(660F3822CA                  , pmovsxbq(xmm1, xmm2)),
        INSTRUCTION_TEST(660F38228C1A80000000        , pmovsxbq(xmm1, word_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660F3820CA                  , pmovsxbw(xmm1, xmm2)),
        INSTRUCTION_TEST(660F38208C1A80000000        , pmovsxbw(xmm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660F38258C1A80000000        , pmovsxdq(xmm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660F3825CA                  , pmovsxdq(xmm1, xmm2)),
        INSTRUCTION_TEST(660F38238C1A80000000        , pmovsxwd(xmm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660F3823CA                  , pmovsxwd(xmm1, xmm2)),
        INSTRUCTION_TEST(660F38248C1A80000000        , pmovsxwq(xmm1, dword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660F3824CA                  , pmovsxwq(xmm1, xmm2)),
        INSTRUCTION_TEST(660F38318C1A80000000        , pmovzxbd(xmm1, dword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660F3831CA                  , pmovzxbd(xmm1, xmm2)),
        INSTRUCTION_TEST(660F3832CA                  , pmovzxbq(xmm1, xmm2)),
        INSTRUCTION_TEST(660F38328C1A80000000        , pmovzxbq(xmm1, word_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660F38308C1A80000000        , pmovzxbw(xmm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660F3830CA                  , pmovzxbw(xmm1, xmm2)),
        INSTRUCTION_TEST(660F38358C1A80000000        , pmovzxdq(xmm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660F3835CA                  , pmovzxdq(xmm1, xmm2)),
        INSTRUCTION_TEST(660F38338C1A80000000        , pmovzxwd(xmm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660F3833CA                  , pmovzxwd(xmm1, xmm2)),
        INSTRUCTION_TEST(660F38348C1A80000000        , pmovzxwq(xmm1, dword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660F3834CA                  , pmovzxwq(xmm1, xmm2)),
        INSTRUCTION_TEST(660F38288C1A80000000        , pmuldq(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660F3828CA                  , pmuldq(xmm1, xmm2)),
        INSTRUCTION_TEST(0F380BCA                    , pmulhrsw(mm1, mm2)),
        INSTRUCTION_TEST(0F380B8C1A80000000          , pmulhrsw(mm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660F380BCA                  , pmulhrsw(xmm1, xmm2)),
        INSTRUCTION_TEST(660F380B8C1A80000000        , pmulhrsw(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(0F0FCAB7                    , pmulhrw(mm1, mm2)),
        INSTRUCTION_TEST(0F0F8C1A80000000B7          , pmulhrw(mm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660FE48C1A80000000          , pmulhuw(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660FE4CA                    , pmulhuw(xmm1, xmm2)),
        INSTRUCTION_TEST(0FE4CA                      , pmulhuw(mm1, mm2)),
        INSTRUCTION_TEST(0FE48C1A80000000            , pmulhuw(mm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(0FE5CA                      , pmulhw(mm1, mm2)),
        INSTRUCTION_TEST(0FE58C1A80000000            , pmulhw(mm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660FE5CA                    , pmulhw(xmm1, xmm2)),
        INSTRUCTION_TEST(660FE58C1A80000000          , pmulhw(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660F3840CA                  , pmulld(xmm1, xmm2)),
        INSTRUCTION_TEST(660F38408C1A80000000        , pmulld(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660FD58C1A80000000          , pmullw(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660FD5CA                    , pmullw(xmm1, xmm2)),
        INSTRUCTION_TEST(0FD58C1A80000000            , pmullw(mm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(0FD5CA                      , pmullw(mm1, mm2)),
        INSTRUCTION_TEST(0FF4CA                      , pmuludq(mm1, mm2)),
        INSTRUCTION_TEST(0FF48C1A80000000            , pmuludq(mm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660FF4CA                    , pmuludq(xmm1, xmm2)),
        INSTRUCTION_TEST(660FF48C1A80000000          , pmuludq(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(668F841180000000            , pop(word_ptr(rcx, rdx, 1, 128))),
        INSTRUCTION_TEST(8F841180000000              , pop(qword_ptr(rcx, rdx, 1, 128))),
        INSTRUCTION_TEST(59                          , pop(rcx)),
        INSTRUCTION_TEST(6659                        , pop(cx)),
        INSTRUCTION_TEST(0FA1                        , pop(fs)),
        INSTRUCTION_TEST(F3660FB8CA                  , popcnt(cx, dx)),
        INSTRUCTION_TEST(F3660FB88C1A80000000        , popcnt(cx, word_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(F30FB88C1A80000000          , popcnt(ecx, dword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(F30FB8CA                    , popcnt(ecx, edx)),
        INSTRUCTION_TEST(F3480FB88C1A80000000        , popcnt(rcx, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(F3480FB8CA                  , popcnt(rcx, rdx)),
        INSTRUCTION_TEST(669D                        , popf()),
        INSTRUCTION_TEST(9D                          , popfq()),
        INSTRUCTION_TEST(660FEB8C1A80000000          , por(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660FEBCA                    , por(xmm1, xmm2)),
        INSTRUCTION_TEST(0FEBCA                      , por(mm1, mm2)),
        INSTRUCTION_TEST(0FEB8C1A80000000            , por(mm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(0FF6CA                      , psadbw(mm1, mm2)),
        INSTRUCTION_TEST(0FF68C1A80000000            , psadbw(mm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660FF6CA                    , psadbw(xmm1, xmm2)),
        INSTRUCTION_TEST(660FF68C1A80000000          , psadbw(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(0F3800CA                    , pshufb(mm1, mm2)),
        INSTRUCTION_TEST(0F38008C1A80000000          , pshufb(mm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660F3800CA                  , pshufb(xmm1, xmm2)),
        INSTRUCTION_TEST(660F38008C1A80000000        , pshufb(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660F708C1A8000000001        , pshufd(xmm1, xmmword_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(660F70CA01                  , pshufd(xmm1, xmm2, Imm(1))),
        INSTRUCTION_TEST(F30F70CA01                  , pshufhw(xmm1, xmm2, Imm(1))),
        INSTRUCTION_TEST(F30F708C1A8000000001        , pshufhw(xmm1, xmmword_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(F20F70CA01                  , pshuflw(xmm1, xmm2, Imm(1))),
        INSTRUCTION_TEST(F20F708C1A8000000001        , pshuflw(xmm1, xmmword_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(0F70CA01                    , pshufw(mm1, mm2, Imm(1))),
        INSTRUCTION_TEST(0F708C1A8000000001          , pshufw(mm1, qword_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(660F38088C1A80000000        , psignb(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660F3808CA                  , psignb(xmm1, xmm2)),
        INSTRUCTION_TEST(0F3808CA                    , psignb(mm1, mm2)),
        INSTRUCTION_TEST(0F38088C1A80000000          , psignb(mm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(0F380ACA                    , psignd(mm1, mm2)),
        INSTRUCTION_TEST(0F380A8C1A80000000          , psignd(mm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660F380ACA                  , psignd(xmm1, xmm2)),
        INSTRUCTION_TEST(660F380A8C1A80000000        , psignd(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(0F3809CA                    , psignw(mm1, mm2)),
        INSTRUCTION_TEST(0F38098C1A80000000          , psignw(mm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660F3809CA                  , psignw(xmm1, xmm2)),
        INSTRUCTION_TEST(660F38098C1A80000000        , psignw(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660FF28C1A80000000          , pslld(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(0F72F101                    , pslld(mm1, Imm(1))),
        INSTRUCTION_TEST(0FF2CA                      , pslld(mm1, mm2)),
        INSTRUCTION_TEST(0FF28C1A80000000            , pslld(mm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660F72F101                  , pslld(xmm1, Imm(1))),
        INSTRUCTION_TEST(660FF2CA                    , pslld(xmm1, xmm2)),
        INSTRUCTION_TEST(660F73F901                  , pslldq(xmm1, Imm(1))),
        INSTRUCTION_TEST(0FF3CA                      , psllq(mm1, mm2)),
        INSTRUCTION_TEST(660FF38C1A80000000          , psllq(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660FF3CA                    , psllq(xmm1, xmm2)),
        INSTRUCTION_TEST(0F73F101                    , psllq(mm1, Imm(1))),
        INSTRUCTION_TEST(0FF38C1A80000000            , psllq(mm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660F73F101                  , psllq(xmm1, Imm(1))),
        INSTRUCTION_TEST(0F71F101                    , psllw(mm1, Imm(1))),
        INSTRUCTION_TEST(0FF1CA                      , psllw(mm1, mm2)),
        INSTRUCTION_TEST(0FF18C1A80000000            , psllw(mm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660F71F101                  , psllw(xmm1, Imm(1))),
        INSTRUCTION_TEST(660FF1CA                    , psllw(xmm1, xmm2)),
        INSTRUCTION_TEST(660FF18C1A80000000          , psllw(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(F30F01FF                    , psmash(rax)),
        INSTRUCTION_TEST(0F72E101                    , psrad(mm1, Imm(1))),
        INSTRUCTION_TEST(0FE2CA                      , psrad(mm1, mm2)),
        INSTRUCTION_TEST(0FE28C1A80000000            , psrad(mm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660F72E101                  , psrad(xmm1, Imm(1))),
        INSTRUCTION_TEST(660FE2CA                    , psrad(xmm1, xmm2)),
        INSTRUCTION_TEST(660FE28C1A80000000          , psrad(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660FE1CA                    , psraw(xmm1, xmm2)),
        INSTRUCTION_TEST(660FE18C1A80000000          , psraw(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660F71E101                  , psraw(xmm1, Imm(1))),
        INSTRUCTION_TEST(0FE1CA                      , psraw(mm1, mm2)),
        INSTRUCTION_TEST(0F71E101                    , psraw(mm1, Imm(1))),
        INSTRUCTION_TEST(0FE18C1A80000000            , psraw(mm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(0F72D101                    , psrld(mm1, Imm(1))),
        INSTRUCTION_TEST(0FD2CA                      , psrld(mm1, mm2)),
        INSTRUCTION_TEST(0FD28C1A80000000            , psrld(mm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660F72D101                  , psrld(xmm1, Imm(1))),
        INSTRUCTION_TEST(660FD2CA                    , psrld(xmm1, xmm2)),
        INSTRUCTION_TEST(660FD28C1A80000000          , psrld(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660F73D901                  , psrldq(xmm1, Imm(1))),
        INSTRUCTION_TEST(660FD3CA                    , psrlq(xmm1, xmm2)),
        INSTRUCTION_TEST(0F73D101                    , psrlq(mm1, Imm(1))),
        INSTRUCTION_TEST(0FD3CA                      , psrlq(mm1, mm2)),
        INSTRUCTION_TEST(0FD38C1A80000000            , psrlq(mm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660F73D101                  , psrlq(xmm1, Imm(1))),
        INSTRUCTION_TEST(660FD38C1A80000000          , psrlq(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660FD18C1A80000000          , psrlw(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660FD1CA                    , psrlw(xmm1, xmm2)),
        INSTRUCTION_TEST(660F71D101                  , psrlw(xmm1, Imm(1))),
        INSTRUCTION_TEST(0FD18C1A80000000            , psrlw(mm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(0FD1CA                      , psrlw(mm1, mm2)),
        INSTRUCTION_TEST(0F71D101                    , psrlw(mm1, Imm(1))),
        INSTRUCTION_TEST(0FF8CA                      , psubb(mm1, mm2)),
        INSTRUCTION_TEST(0FF88C1A80000000            , psubb(mm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660FF8CA                    , psubb(xmm1, xmm2)),
        INSTRUCTION_TEST(660FF88C1A80000000          , psubb(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660FFACA                    , psubd(xmm1, xmm2)),
        INSTRUCTION_TEST(660FFA8C1A80000000          , psubd(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(0FFACA                      , psubd(mm1, mm2)),
        INSTRUCTION_TEST(0FFA8C1A80000000            , psubd(mm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(0FFBCA                      , psubq(mm1, mm2)),
        INSTRUCTION_TEST(0FFB8C1A80000000            , psubq(mm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660FFBCA                    , psubq(xmm1, xmm2)),
        INSTRUCTION_TEST(660FFB8C1A80000000          , psubq(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(0FE8CA                      , psubsb(mm1, mm2)),
        INSTRUCTION_TEST(0FE88C1A80000000            , psubsb(mm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660FE8CA                    , psubsb(xmm1, xmm2)),
        INSTRUCTION_TEST(660FE88C1A80000000          , psubsb(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660FE98C1A80000000          , psubsw(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660FE9CA                    , psubsw(xmm1, xmm2)),
        INSTRUCTION_TEST(0FE98C1A80000000            , psubsw(mm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(0FE9CA                      , psubsw(mm1, mm2)),
        INSTRUCTION_TEST(0FD8CA                      , psubusb(mm1, mm2)),
        INSTRUCTION_TEST(0FD88C1A80000000            , psubusb(mm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660FD8CA                    , psubusb(xmm1, xmm2)),
        INSTRUCTION_TEST(660FD88C1A80000000          , psubusb(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(0FD9CA                      , psubusw(mm1, mm2)),
        INSTRUCTION_TEST(0FD98C1A80000000            , psubusw(mm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660FD9CA                    , psubusw(xmm1, xmm2)),
        INSTRUCTION_TEST(660FD98C1A80000000          , psubusw(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660FF98C1A80000000          , psubw(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660FF9CA                    , psubw(xmm1, xmm2)),
        INSTRUCTION_TEST(0FF9CA                      , psubw(mm1, mm2)),
        INSTRUCTION_TEST(0FF98C1A80000000            , psubw(mm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(0F0FCABB                    , pswapd(mm1, mm2)),
        INSTRUCTION_TEST(0F0F8C1A80000000BB          , pswapd(mm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660F38178C1A80000000        , ptest(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660F3817CA                  , ptest(xmm1, xmm2)),
        INSTRUCTION_TEST(F3480FAEA41180000000        , ptwrite(qword_ptr(rcx, rdx, 1, 128))),
        INSTRUCTION_TEST(F3480FAEE1                  , ptwrite(rcx)),
        INSTRUCTION_TEST(F30FAEE1                    , ptwrite(ecx)),
        INSTRUCTION_TEST(F30FAEA41180000000          , ptwrite(dword_ptr(rcx, rdx, 1, 128))),
        INSTRUCTION_TEST(0F68CA                      , punpckhbw(mm1, mm2)),
        INSTRUCTION_TEST(0F688C1A80000000            , punpckhbw(mm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660F68CA                    , punpckhbw(xmm1, xmm2)),
        INSTRUCTION_TEST(660F688C1A80000000          , punpckhbw(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660F6ACA                    , punpckhdq(xmm1, xmm2)),
        INSTRUCTION_TEST(660F6A8C1A80000000          , punpckhdq(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(0F6ACA                      , punpckhdq(mm1, mm2)),
        INSTRUCTION_TEST(0F6A8C1A80000000            , punpckhdq(mm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660F6DCA                    , punpckhqdq(xmm1, xmm2)),
        INSTRUCTION_TEST(660F6D8C1A80000000          , punpckhqdq(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(0F69CA                      , punpckhwd(mm1, mm2)),
        INSTRUCTION_TEST(0F698C1A80000000            , punpckhwd(mm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660F69CA                    , punpckhwd(xmm1, xmm2)),
        INSTRUCTION_TEST(660F698C1A80000000          , punpckhwd(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660F60CA                    , punpcklbw(xmm1, xmm2)),
        INSTRUCTION_TEST(660F608C1A80000000          , punpcklbw(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(0F60CA                      , punpcklbw(mm1, mm2)),
        INSTRUCTION_TEST(0F608C1A80000000            , punpcklbw(mm1, dword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(0F628C1A80000000            , punpckldq(mm1, dword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(0F62CA                      , punpckldq(mm1, mm2)),
        INSTRUCTION_TEST(660F62CA                    , punpckldq(xmm1, xmm2)),
        INSTRUCTION_TEST(660F628C1A80000000          , punpckldq(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660F6CCA                    , punpcklqdq(xmm1, xmm2)),
        INSTRUCTION_TEST(660F6C8C1A80000000          , punpcklqdq(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660F618C1A80000000          , punpcklwd(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(0F618C1A80000000            , punpcklwd(mm1, dword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(0F61CA                      , punpcklwd(mm1, mm2)),
        INSTRUCTION_TEST(660F61CA                    , punpcklwd(xmm1, xmm2)),
        INSTRUCTION_TEST(53                          , push(rbx)),
        INSTRUCTION_TEST(51                          , push(rcx)),
        INSTRUCTION_TEST(55                          , push(rbp)),
        INSTRUCTION_TEST(57                          , push(rdi)),
        INSTRUCTION_TEST(4155                        , push(r13)),
        INSTRUCTION_TEST(56                          , push(rsi)),
        INSTRUCTION_TEST(54                          , push(rsp)),
        INSTRUCTION_TEST(66FFB41180000000            , push(word_ptr(rcx, rdx, 1, 128))),
        INSTRUCTION_TEST(50                          , push(rax)),
        INSTRUCTION_TEST(52                          , push(rdx)),
        INSTRUCTION_TEST(4151                        , push(r9)),
        INSTRUCTION_TEST(4154                        , push(r12)),
        INSTRUCTION_TEST(4157                        , push(r15)),
        INSTRUCTION_TEST(6651                        , push(cx)),
        INSTRUCTION_TEST(0FA0                        , push(fs)),
        INSTRUCTION_TEST(0FA8                        , push(gs)),
        INSTRUCTION_TEST(6A01                        , push(Imm(1))),
        INSTRUCTION_TEST(4150                        , push(r8)),
        INSTRUCTION_TEST(4152                        , push(r10)),
        INSTRUCTION_TEST(4153                        , push(r11)),
        INSTRUCTION_TEST(FFB41180000000              , push(qword_ptr(rcx, rdx, 1, 128))),
        INSTRUCTION_TEST(4156                        , push(r14)),
        INSTRUCTION_TEST(669C                        , pushf()),
        INSTRUCTION_TEST(9C                          , pushfq()),
        INSTRUCTION_TEST(660FEFCA                    , pxor(xmm1, xmm2)),
        INSTRUCTION_TEST(660FEF8C1A80000000          , pxor(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(0FEFCA                      , pxor(mm1, mm2)),
        INSTRUCTION_TEST(0FEF8C1A80000000            , pxor(mm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(D3D1                        , rcl(ecx, cl)),
        INSTRUCTION_TEST(66D3941180000000            , rcl(word_ptr(rcx, rdx, 1, 128), cl)),
        INSTRUCTION_TEST(48D1D1                      , rcl(rcx, Imm(1))),
        INSTRUCTION_TEST(48D3D1                      , rcl(rcx, cl)),
        INSTRUCTION_TEST(48D1941180000000            , rcl(qword_ptr(rcx, rdx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(48D3941180000000            , rcl(qword_ptr(rcx, rdx, 1, 128), cl)),
        INSTRUCTION_TEST(66D1941180000000            , rcl(word_ptr(rcx, rdx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(D1D1                        , rcl(ecx, Imm(1))),
        INSTRUCTION_TEST(D1941180000000              , rcl(dword_ptr(rcx, rdx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(D0D1                        , rcl(cl, Imm(1))),
        INSTRUCTION_TEST(66D1D1                      , rcl(cx, Imm(1))),
        INSTRUCTION_TEST(66D3D1                      , rcl(cx, cl)),
        INSTRUCTION_TEST(D2D1                        , rcl(cl, cl)),
        INSTRUCTION_TEST(D0D5                        , rcl(ch, Imm(1))),
        INSTRUCTION_TEST(D0941180000000              , rcl(byte_ptr(rcx, rdx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(D2941180000000              , rcl(byte_ptr(rcx, rdx, 1, 128), cl)),
        INSTRUCTION_TEST(D3941180000000              , rcl(dword_ptr(rcx, rdx, 1, 128), cl)),
        INSTRUCTION_TEST(D2D5                        , rcl(ch, cl)),
        INSTRUCTION_TEST(0F538C1A80000000            , rcpps(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(0F53CA                      , rcpps(xmm1, xmm2)),
        INSTRUCTION_TEST(F30F53CA                    , rcpss(xmm1, xmm2)),
        INSTRUCTION_TEST(F30F538C1A80000000          , rcpss(xmm1, dword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(D29C1180000000              , rcr(byte_ptr(rcx, rdx, 1, 128), cl)),
        INSTRUCTION_TEST(66D39C1180000000            , rcr(word_ptr(rcx, rdx, 1, 128), cl)),
        INSTRUCTION_TEST(48D1D9                      , rcr(rcx, Imm(1))),
        INSTRUCTION_TEST(48D3D9                      , rcr(rcx, cl)),
        INSTRUCTION_TEST(48D19C1180000000            , rcr(qword_ptr(rcx, rdx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(48D39C1180000000            , rcr(qword_ptr(rcx, rdx, 1, 128), cl)),
        INSTRUCTION_TEST(D1D9                        , rcr(ecx, Imm(1))),
        INSTRUCTION_TEST(D3D9                        , rcr(ecx, cl)),
        INSTRUCTION_TEST(D19C1180000000              , rcr(dword_ptr(rcx, rdx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(66D19C1180000000            , rcr(word_ptr(rcx, rdx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(66D1D9                      , rcr(cx, Imm(1))),
        INSTRUCTION_TEST(66D3D9                      , rcr(cx, cl)),
        INSTRUCTION_TEST(D0D9                        , rcr(cl, Imm(1))),
        INSTRUCTION_TEST(D2D9                        , rcr(cl, cl)),
        INSTRUCTION_TEST(D0DD                        , rcr(ch, Imm(1))),
        INSTRUCTION_TEST(D2DD                        , rcr(ch, cl)),
        INSTRUCTION_TEST(D09C1180000000              , rcr(byte_ptr(rcx, rdx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(D39C1180000000              , rcr(dword_ptr(rcx, rdx, 1, 128), cl)),
        INSTRUCTION_TEST(F30FAEC1                    , rdfsbase(ecx)),
        INSTRUCTION_TEST(F3480FAEC1                  , rdfsbase(rcx)),
        INSTRUCTION_TEST(F30FAEC9                    , rdgsbase(ecx)),
        INSTRUCTION_TEST(F3480FAEC9                  , rdgsbase(rcx)),
        INSTRUCTION_TEST(0F32                        , rdmsr()),
        INSTRUCTION_TEST(F30FC7F9                    , rdpid(rcx)),
        INSTRUCTION_TEST(0F01EE                      , rdpkru()),
        INSTRUCTION_TEST(0F33                        , rdpmc()),
        INSTRUCTION_TEST(0F01FD                      , rdpru()),
        INSTRUCTION_TEST(660FC7F1                    , rdrand(cx)),
        INSTRUCTION_TEST(0FC7F1                      , rdrand(ecx)),
        INSTRUCTION_TEST(480FC7F1                    , rdrand(rcx)),
        INSTRUCTION_TEST(660FC7F9                    , rdseed(cx)),
        INSTRUCTION_TEST(0FC7F9                      , rdseed(ecx)),
        INSTRUCTION_TEST(480FC7F9                    , rdseed(rcx)),
        INSTRUCTION_TEST(F30F1EC9                    , rdsspd(ecx)),
        INSTRUCTION_TEST(F3480F1EC9                  , rdsspq(rcx)),
        INSTRUCTION_TEST(0F31                        , rdtsc()),
        INSTRUCTION_TEST(0F01F9                      , rdtscp()),
        INSTRUCTION_TEST(C20100                      , ret(Imm(1))),
        INSTRUCTION_TEST(C3                          , ret()),
        INSTRUCTION_TEST(D2841180000000              , rol(byte_ptr(rcx, rdx, 1, 128), cl)),
        INSTRUCTION_TEST(66D1841180000000            , rol(word_ptr(rcx, rdx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(66D3841180000000            , rol(word_ptr(rcx, rdx, 1, 128), cl)),
        INSTRUCTION_TEST(48D1C1                      , rol(rcx, Imm(1))),
        INSTRUCTION_TEST(48D3C1                      , rol(rcx, cl)),
        INSTRUCTION_TEST(48D3841180000000            , rol(qword_ptr(rcx, rdx, 1, 128), cl)),
        INSTRUCTION_TEST(D1C1                        , rol(ecx, Imm(1))),
        INSTRUCTION_TEST(D3C1                        , rol(ecx, cl)),
        INSTRUCTION_TEST(D1841180000000              , rol(dword_ptr(rcx, rdx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(48D1841180000000            , rol(qword_ptr(rcx, rdx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(66D1C1                      , rol(cx, Imm(1))),
        INSTRUCTION_TEST(66D3C1                      , rol(cx, cl)),
        INSTRUCTION_TEST(D0C1                        , rol(cl, Imm(1))),
        INSTRUCTION_TEST(D2C1                        , rol(cl, cl)),
        INSTRUCTION_TEST(D0C5                        , rol(ch, Imm(1))),
        INSTRUCTION_TEST(D2C5                        , rol(ch, cl)),
        INSTRUCTION_TEST(D0841180000000              , rol(byte_ptr(rcx, rdx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(D3841180000000              , rol(dword_ptr(rcx, rdx, 1, 128), cl)),
        INSTRUCTION_TEST(D1C9                        , ror(ecx, Imm(1))),
        INSTRUCTION_TEST(48D38C1180000000            , ror(qword_ptr(rcx, rdx, 1, 128), cl)),
        INSTRUCTION_TEST(48D18C1180000000            , ror(qword_ptr(rcx, rdx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(66D18C1180000000            , ror(word_ptr(rcx, rdx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(48D1C9                      , ror(rcx, Imm(1))),
        INSTRUCTION_TEST(66D38C1180000000            , ror(word_ptr(rcx, rdx, 1, 128), cl)),
        INSTRUCTION_TEST(D3C9                        , ror(ecx, cl)),
        INSTRUCTION_TEST(48D3C9                      , ror(rcx, cl)),
        INSTRUCTION_TEST(D18C1180000000              , ror(dword_ptr(rcx, rdx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(D2C9                        , ror(cl, cl)),
        INSTRUCTION_TEST(66D1C9                      , ror(cx, Imm(1))),
        INSTRUCTION_TEST(66D3C9                      , ror(cx, cl)),
        INSTRUCTION_TEST(D0C9                        , ror(cl, Imm(1))),
        INSTRUCTION_TEST(D0CD                        , ror(ch, Imm(1))),
        INSTRUCTION_TEST(D2CD                        , ror(ch, cl)),
        INSTRUCTION_TEST(D08C1180000000              , ror(byte_ptr(rcx, rdx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(D28C1180000000              , ror(byte_ptr(rcx, rdx, 1, 128), cl)),
        INSTRUCTION_TEST(D38C1180000000              , ror(dword_ptr(rcx, rdx, 1, 128), cl)),
        INSTRUCTION_TEST(C4E3FBF0CA01                , rorx(rcx, rdx, Imm(1))),
        INSTRUCTION_TEST(C4E3FBF08C1A8000000001      , rorx(rcx, qword_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(C4E37BF08C1A8000000001      , rorx(ecx, dword_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(C4E37BF0CA01                , rorx(ecx, edx, Imm(1))),
        INSTRUCTION_TEST(660F3A09CA01                , roundpd(xmm1, xmm2, Imm(1))),
        INSTRUCTION_TEST(660F3A098C1A8000000001      , roundpd(xmm1, xmmword_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(660F3A088C1A8000000001      , roundps(xmm1, xmmword_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(660F3A08CA01                , roundps(xmm1, xmm2, Imm(1))),
        INSTRUCTION_TEST(660F3A0BCA01                , roundsd(xmm1, xmm2, Imm(1))),
        INSTRUCTION_TEST(660F3A0B8C1A8000000001      , roundsd(xmm1, qword_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(660F3A0A8C1A8000000001      , roundss(xmm1, dword_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(660F3A0ACA01                , roundss(xmm1, xmm2, Imm(1))),
        INSTRUCTION_TEST(0F52CA                      , rsqrtps(xmm1, xmm2)),
        INSTRUCTION_TEST(0F528C1A80000000            , rsqrtps(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(F30F528C1A80000000          , rsqrtss(xmm1, dword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(F30F52CA                    , rsqrtss(xmm1, xmm2)),
        INSTRUCTION_TEST(F30F01AC1180000000          , rstorssp(qword_ptr(rcx, rdx, 1, 128))),
        INSTRUCTION_TEST(9E                          , sahf()),
        INSTRUCTION_TEST(66D1BC1180000000            , sar(word_ptr(rcx, rdx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(66D3BC1180000000            , sar(word_ptr(rcx, rdx, 1, 128), cl)),
        INSTRUCTION_TEST(48D1F9                      , sar(rcx, Imm(1))),
        INSTRUCTION_TEST(48D3F9                      , sar(rcx, cl)),
        INSTRUCTION_TEST(48D3BC1180000000            , sar(qword_ptr(rcx, rdx, 1, 128), cl)),
        INSTRUCTION_TEST(D1F9                        , sar(ecx, Imm(1))),
        INSTRUCTION_TEST(D3F9                        , sar(ecx, cl)),
        INSTRUCTION_TEST(D1BC1180000000              , sar(dword_ptr(rcx, rdx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(48D1BC1180000000            , sar(qword_ptr(rcx, rdx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(66D1F9                      , sar(cx, Imm(1))),
        INSTRUCTION_TEST(66D3F9                      , sar(cx, cl)),
        INSTRUCTION_TEST(D0F9                        , sar(cl, Imm(1))),
        INSTRUCTION_TEST(D2F9                        , sar(cl, cl)),
        INSTRUCTION_TEST(D3BC1180000000              , sar(dword_ptr(rcx, rdx, 1, 128), cl)),
        INSTRUCTION_TEST(D0FD                        , sar(ch, Imm(1))),
        INSTRUCTION_TEST(D2FD                        , sar(ch, cl)),
        INSTRUCTION_TEST(D0BC1180000000              , sar(byte_ptr(rcx, rdx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(D2BC1180000000              , sar(byte_ptr(rcx, rdx, 1, 128), cl)),
        INSTRUCTION_TEST(C4E25AF78C1A80000000        , sarx(ecx, dword_ptr(rdx, rbx, 1, 128), esp)),
        INSTRUCTION_TEST(C4E262F7CA                  , sarx(ecx, edx, ebx)),
        INSTRUCTION_TEST(C4E2DAF78C1A80000000        , sarx(rcx, qword_ptr(rdx, rbx, 1, 128), rsp)),
        INSTRUCTION_TEST(C4E2E2F7CA                  , sarx(rcx, rdx, rbx)),
        INSTRUCTION_TEST(F30F01EA                    , saveprevssp()),
        INSTRUCTION_TEST(66839C118000000001          , sbb(word_ptr(rcx, rdx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(66199C1180000000            , sbb(word_ptr(rcx, rdx, 1, 128), bx)),
        INSTRUCTION_TEST(4819D1                      , sbb(rcx, rdx)),
        INSTRUCTION_TEST(481B8C1A80000000            , sbb(rcx, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(4883D901                    , sbb(rcx, Imm(1))),
        INSTRUCTION_TEST(48199C1180000000            , sbb(qword_ptr(rcx, rdx, 1, 128), rbx)),
        INSTRUCTION_TEST(48839C118000000001          , sbb(qword_ptr(rcx, rdx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(83D901                      , sbb(ecx, Imm(1))),
        INSTRUCTION_TEST(19D1                        , sbb(ecx, edx)),
        INSTRUCTION_TEST(1B8C1A80000000              , sbb(ecx, dword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(839C118000000001            , sbb(dword_ptr(rcx, rdx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(199C1180000000              , sbb(dword_ptr(rcx, rdx, 1, 128), ebx)),
        INSTRUCTION_TEST(661B8C1A80000000            , sbb(cx, word_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(189C1180000000              , sbb(byte_ptr(rcx, rdx, 1, 128), bl)),
        INSTRUCTION_TEST(6619D1                      , sbb(cx, dx)),
        INSTRUCTION_TEST(18BC1180000000              , sbb(byte_ptr(rcx, rdx, 1, 128), bh)),
        INSTRUCTION_TEST(809C118000000001            , sbb(byte_ptr(rcx, rdx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(1AAC1A80000000              , sbb(ch, byte_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(18F5                        , sbb(ch, dh)),
        INSTRUCTION_TEST(18D5                        , sbb(ch, dl)),
        INSTRUCTION_TEST(6683D901                    , sbb(cx, Imm(1))),
        INSTRUCTION_TEST(1A8C1A80000000              , sbb(cl, byte_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(18F1                        , sbb(cl, dh)),
        INSTRUCTION_TEST(18D1                        , sbb(cl, dl)),
        INSTRUCTION_TEST(80DD01                      , sbb(ch, Imm(1))),
        INSTRUCTION_TEST(80D901                      , sbb(cl, Imm(1))),
        INSTRUCTION_TEST(AE                          , scasb()),
        INSTRUCTION_TEST(AF                          , scasd()),
        INSTRUCTION_TEST(48AF                        , scasq()),
        INSTRUCTION_TEST(66AF                        , scasw()),
        INSTRUCTION_TEST(0F01E8                      , serialize()),
        INSTRUCTION_TEST(0F92841180000000            , setb(byte_ptr(rcx, rdx, 1, 128))),
        INSTRUCTION_TEST(0F92C5                      , setb(ch)),
        INSTRUCTION_TEST(0F92C1                      , setb(cl)),
        INSTRUCTION_TEST(0F96841180000000            , setbe(byte_ptr(rcx, rdx, 1, 128))),
        INSTRUCTION_TEST(0F96C5                      , setbe(ch)),
        INSTRUCTION_TEST(0F96C1                      , setbe(cl)),
        INSTRUCTION_TEST(0F9C841180000000            , setl(byte_ptr(rcx, rdx, 1, 128))),
        INSTRUCTION_TEST(0F9CC5                      , setl(ch)),
        INSTRUCTION_TEST(0F9CC1                      , setl(cl)),
        INSTRUCTION_TEST(0F9E841180000000            , setle(byte_ptr(rcx, rdx, 1, 128))),
        INSTRUCTION_TEST(0F9EC5                      , setle(ch)),
        INSTRUCTION_TEST(0F9EC1                      , setle(cl)),
        INSTRUCTION_TEST(0F93841180000000            , setnb(byte_ptr(rcx, rdx, 1, 128))),
        INSTRUCTION_TEST(0F93C5                      , setnb(ch)),
        INSTRUCTION_TEST(0F93C1                      , setnb(cl)),
        INSTRUCTION_TEST(0F97C1                      , setnbe(cl)),
        INSTRUCTION_TEST(0F97841180000000            , setnbe(byte_ptr(rcx, rdx, 1, 128))),
        INSTRUCTION_TEST(0F97C5                      , setnbe(ch)),
        INSTRUCTION_TEST(0F9D841180000000            , setnl(byte_ptr(rcx, rdx, 1, 128))),
        INSTRUCTION_TEST(0F9DC5                      , setnl(ch)),
        INSTRUCTION_TEST(0F9DC1                      , setnl(cl)),
        INSTRUCTION_TEST(0F9F841180000000            , setnle(byte_ptr(rcx, rdx, 1, 128))),
        INSTRUCTION_TEST(0F9FC5                      , setnle(ch)),
        INSTRUCTION_TEST(0F9FC1                      , setnle(cl)),
        INSTRUCTION_TEST(0F91841180000000            , setno(byte_ptr(rcx, rdx, 1, 128))),
        INSTRUCTION_TEST(0F91C5                      , setno(ch)),
        INSTRUCTION_TEST(0F91C1                      , setno(cl)),
        INSTRUCTION_TEST(0F9B841180000000            , setnp(byte_ptr(rcx, rdx, 1, 128))),
        INSTRUCTION_TEST(0F9BC5                      , setnp(ch)),
        INSTRUCTION_TEST(0F9BC1                      , setnp(cl)),
        INSTRUCTION_TEST(0F99C1                      , setns(cl)),
        INSTRUCTION_TEST(0F99C5                      , setns(ch)),
        INSTRUCTION_TEST(0F99841180000000            , setns(byte_ptr(rcx, rdx, 1, 128))),
        INSTRUCTION_TEST(0F95C3                      , setnz(bl)),
        INSTRUCTION_TEST(0F95841180000000            , setnz(byte_ptr(rcx, rdx, 1, 128))),
        INSTRUCTION_TEST(0F95C5                      , setnz(ch)),
        INSTRUCTION_TEST(0F95C1                      , setnz(cl)),
        INSTRUCTION_TEST(0F90841180000000            , seto(byte_ptr(rcx, rdx, 1, 128))),
        INSTRUCTION_TEST(0F90C5                      , seto(ch)),
        INSTRUCTION_TEST(0F90C1                      , seto(cl)),
        INSTRUCTION_TEST(0F9A841180000000            , setp(byte_ptr(rcx, rdx, 1, 128))),
        INSTRUCTION_TEST(0F9AC5                      , setp(ch)),
        INSTRUCTION_TEST(0F9AC1                      , setp(cl)),
        INSTRUCTION_TEST(0F98841180000000            , sets(byte_ptr(rcx, rdx, 1, 128))),
        INSTRUCTION_TEST(0F98C5                      , sets(ch)),
        INSTRUCTION_TEST(0F98C1                      , sets(cl)),
        INSTRUCTION_TEST(F30F01E8                    , setssbsy()),
        INSTRUCTION_TEST(410F94C7                    , setz(r15b)),
        INSTRUCTION_TEST(0F94C1                      , setz(cl)),
        INSTRUCTION_TEST(0F94841180000000            , setz(byte_ptr(rcx, rdx, 1, 128))),
        INSTRUCTION_TEST(0F94C7                      , setz(bh)),
        INSTRUCTION_TEST(0F94C5                      , setz(ch)),
        INSTRUCTION_TEST(0FAEF8                      , sfence()),
        INSTRUCTION_TEST(0F38C9CA                    , sha1msg1(xmm1, xmm2)),
        INSTRUCTION_TEST(0F38C98C1A80000000          , sha1msg1(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(0F38CACA                    , sha1msg2(xmm1, xmm2)),
        INSTRUCTION_TEST(0F38CA8C1A80000000          , sha1msg2(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(0F38C8CA                    , sha1nexte(xmm1, xmm2)),
        INSTRUCTION_TEST(0F38C88C1A80000000          , sha1nexte(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(0F3ACCCA01                  , sha1rnds4(xmm1, xmm2, Imm(1))),
        INSTRUCTION_TEST(0F3ACC8C1A8000000001        , sha1rnds4(xmm1, xmmword_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(0F38CC8C1A80000000          , sha256msg1(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(0F38CCCA                    , sha256msg1(xmm1, xmm2)),
        INSTRUCTION_TEST(0F38CDCA                    , sha256msg2(xmm1, xmm2)),
        INSTRUCTION_TEST(0F38CD8C1A80000000          , sha256msg2(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(0F38CBCA                    , sha256rnds2(xmm1, xmm2)),
        INSTRUCTION_TEST(0F38CB8C1A80000000          , sha256rnds2(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(D1E1                        , shl(ecx, Imm(1))),
        INSTRUCTION_TEST(48D3A41180000000            , shl(qword_ptr(rcx, rdx, 1, 128), cl)),
        INSTRUCTION_TEST(48D1A41180000000            , shl(qword_ptr(rcx, rdx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(66D1A41180000000            , shl(word_ptr(rcx, rdx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(48D1E1                      , shl(rcx, Imm(1))),
        INSTRUCTION_TEST(66D3A41180000000            , shl(word_ptr(rcx, rdx, 1, 128), cl)),
        INSTRUCTION_TEST(D3E1                        , shl(ecx, cl)),
        INSTRUCTION_TEST(48D3E1                      , shl(rcx, cl)),
        INSTRUCTION_TEST(D1A41180000000              , shl(dword_ptr(rcx, rdx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(D0E5                        , shl(ch, Imm(1))),
        INSTRUCTION_TEST(66D1E1                      , shl(cx, Imm(1))),
        INSTRUCTION_TEST(66D3E1                      , shl(cx, cl)),
        INSTRUCTION_TEST(D0E1                        , shl(cl, Imm(1))),
        INSTRUCTION_TEST(D2E1                        , shl(cl, cl)),
        INSTRUCTION_TEST(D2E5                        , shl(ch, cl)),
        INSTRUCTION_TEST(D0A41180000000              , shl(byte_ptr(rcx, rdx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(D2A41180000000              , shl(byte_ptr(rcx, rdx, 1, 128), cl)),
        INSTRUCTION_TEST(D3A41180000000              , shl(dword_ptr(rcx, rdx, 1, 128), cl)),
        INSTRUCTION_TEST(480FA49C118000000001        , shld(qword_ptr(rcx, rdx, 1, 128), rbx, Imm(1))),
        INSTRUCTION_TEST(660FA49C118000000001        , shld(word_ptr(rcx, rdx, 1, 128), bx, Imm(1))),
        INSTRUCTION_TEST(660FA59C1180000000          , shld(word_ptr(rcx, rdx, 1, 128), bx, cl)),
        INSTRUCTION_TEST(480FA4D101                  , shld(rcx, rdx, Imm(1))),
        INSTRUCTION_TEST(480FA5D1                    , shld(rcx, rdx, cl)),
        INSTRUCTION_TEST(480FA59C1180000000          , shld(qword_ptr(rcx, rdx, 1, 128), rbx, cl)),
        INSTRUCTION_TEST(660FA4D101                  , shld(cx, dx, Imm(1))),
        INSTRUCTION_TEST(0FA5D1                      , shld(ecx, edx, cl)),
        INSTRUCTION_TEST(0FA49C118000000001          , shld(dword_ptr(rcx, rdx, 1, 128), ebx, Imm(1))),
        INSTRUCTION_TEST(0FA59C1180000000            , shld(dword_ptr(rcx, rdx, 1, 128), ebx, cl)),
        INSTRUCTION_TEST(660FA5D1                    , shld(cx, dx, cl)),
        INSTRUCTION_TEST(0FA4D101                    , shld(ecx, edx, Imm(1))),
        INSTRUCTION_TEST(C4E2D9F78C1A80000000        , shlx(rcx, qword_ptr(rdx, rbx, 1, 128), rsp)),
        INSTRUCTION_TEST(C4E2E1F7CA                  , shlx(rcx, rdx, rbx)),
        INSTRUCTION_TEST(C4E259F78C1A80000000        , shlx(ecx, dword_ptr(rdx, rbx, 1, 128), esp)),
        INSTRUCTION_TEST(C4E261F7CA                  , shlx(ecx, edx, ebx)),
        INSTRUCTION_TEST(D3E9                        , shr(ecx, cl)),
        INSTRUCTION_TEST(48D1E9                      , shr(rcx, Imm(1))),
        INSTRUCTION_TEST(48D3E9                      , shr(rcx, cl)),
        INSTRUCTION_TEST(48D1AC1180000000            , shr(qword_ptr(rcx, rdx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(48D3AC1180000000            , shr(qword_ptr(rcx, rdx, 1, 128), cl)),
        INSTRUCTION_TEST(66D3AC1180000000            , shr(word_ptr(rcx, rdx, 1, 128), cl)),
        INSTRUCTION_TEST(D1E9                        , shr(ecx, Imm(1))),
        INSTRUCTION_TEST(D1AC1180000000              , shr(dword_ptr(rcx, rdx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(D2E9                        , shr(cl, cl)),
        INSTRUCTION_TEST(66D1E9                      , shr(cx, Imm(1))),
        INSTRUCTION_TEST(66D3E9                      , shr(cx, cl)),
        INSTRUCTION_TEST(D0E9                        , shr(cl, Imm(1))),
        INSTRUCTION_TEST(D0ED                        , shr(ch, Imm(1))),
        INSTRUCTION_TEST(D2ED                        , shr(ch, cl)),
        INSTRUCTION_TEST(D0AC1180000000              , shr(byte_ptr(rcx, rdx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(D2AC1180000000              , shr(byte_ptr(rcx, rdx, 1, 128), cl)),
        INSTRUCTION_TEST(66D1AC1180000000            , shr(word_ptr(rcx, rdx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(D3AC1180000000              , shr(dword_ptr(rcx, rdx, 1, 128), cl)),
        INSTRUCTION_TEST(480FACD101                  , shrd(rcx, rdx, Imm(1))),
        INSTRUCTION_TEST(480FADD1                    , shrd(rcx, rdx, cl)),
        INSTRUCTION_TEST(660FAC9C118000000001        , shrd(word_ptr(rcx, rdx, 1, 128), bx, Imm(1))),
        INSTRUCTION_TEST(660FAD9C1180000000          , shrd(word_ptr(rcx, rdx, 1, 128), bx, cl)),
        INSTRUCTION_TEST(480FAD9C1180000000          , shrd(qword_ptr(rcx, rdx, 1, 128), rbx, cl)),
        INSTRUCTION_TEST(480FAC9C118000000001        , shrd(qword_ptr(rcx, rdx, 1, 128), rbx, Imm(1))),
        INSTRUCTION_TEST(0FADD1                      , shrd(ecx, edx, cl)),
        INSTRUCTION_TEST(0FACD101                    , shrd(ecx, edx, Imm(1))),
        INSTRUCTION_TEST(0FAC9C118000000001          , shrd(dword_ptr(rcx, rdx, 1, 128), ebx, Imm(1))),
        INSTRUCTION_TEST(0FAD9C1180000000            , shrd(dword_ptr(rcx, rdx, 1, 128), ebx, cl)),
        INSTRUCTION_TEST(660FACD101                  , shrd(cx, dx, Imm(1))),
        INSTRUCTION_TEST(660FADD1                    , shrd(cx, dx, cl)),
        INSTRUCTION_TEST(C4E2E3F7CA                  , shrx(rcx, rdx, rbx)),
        INSTRUCTION_TEST(C4E2DBF78C1A80000000        , shrx(rcx, qword_ptr(rdx, rbx, 1, 128), rsp)),
        INSTRUCTION_TEST(C4E25BF78C1A80000000        , shrx(ecx, dword_ptr(rdx, rbx, 1, 128), esp)),
        INSTRUCTION_TEST(C4E263F7CA                  , shrx(ecx, edx, ebx)),
        INSTRUCTION_TEST(660FC6CA01                  , shufpd(xmm1, xmm2, Imm(1))),
        INSTRUCTION_TEST(660FC68C1A8000000001        , shufpd(xmm1, xmmword_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(0FC6CA01                    , shufps(xmm1, xmm2, Imm(1))),
        INSTRUCTION_TEST(0FC68C1A8000000001          , shufps(xmm1, xmmword_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(480F00C1                    , sldt(rcx)),
        INSTRUCTION_TEST(660F00C1                    , sldt(cx)),
        INSTRUCTION_TEST(0F00C1                      , sldt(ecx)),
        INSTRUCTION_TEST(0F00841180000000            , sldt(word_ptr(rcx, rdx, 1, 128))),
        INSTRUCTION_TEST(8FE9F812C9                  , slwpcb(rcx)),
        INSTRUCTION_TEST(8FE97812C9                  , slwpcb(ecx)),
        INSTRUCTION_TEST(660F01E1                    , smsw(cx)),
        INSTRUCTION_TEST(0F01E1                      , smsw(ecx)),
        INSTRUCTION_TEST(480F01E1                    , smsw(rcx)),
        INSTRUCTION_TEST(0F01A41180000000            , smsw(word_ptr(rcx, rdx, 1, 128))),
        INSTRUCTION_TEST(660F51CA                    , sqrtpd(xmm1, xmm2)),
        INSTRUCTION_TEST(660F518C1A80000000          , sqrtpd(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(0F518C1A80000000            , sqrtps(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(0F51CA                      , sqrtps(xmm1, xmm2)),
        INSTRUCTION_TEST(F20F51CA                    , sqrtsd(xmm1, xmm2)),
        INSTRUCTION_TEST(F20F518C1A80000000          , sqrtsd(xmm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(F30F518C1A80000000          , sqrtss(xmm1, dword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(F30F51CA                    , sqrtss(xmm1, xmm2)),
        INSTRUCTION_TEST(0F01CB                      , stac()),
        INSTRUCTION_TEST(F9                          , stc()),
        INSTRUCTION_TEST(FD                          , std()),
        INSTRUCTION_TEST(0F01DC                      , stgi()),
        INSTRUCTION_TEST(FB                          , sti()),
        INSTRUCTION_TEST(0FAE9C1180000000            , stmxcsr(dword_ptr(rcx, rdx, 1, 128))),
        INSTRUCTION_TEST(AA                          , stosb()),
        INSTRUCTION_TEST(AB                          , stosd()),
        INSTRUCTION_TEST(48AB                        , stosq()),
        INSTRUCTION_TEST(66AB                        , stosw()),
        INSTRUCTION_TEST(660F00C9                    , str(cx)),
        INSTRUCTION_TEST(0F00C9                      , str(ecx)),
        INSTRUCTION_TEST(480F00C9                    , str(rcx)),
        INSTRUCTION_TEST(0F008C1180000000            , str(word_ptr(rcx, rdx, 1, 128))),
        INSTRUCTION_TEST(F30F01EF                    , stui()),
        INSTRUCTION_TEST(83AC118000000001            , sub(dword_ptr(rcx, rdx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(2B8C1A80000000              , sub(ecx, dword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(29D1                        , sub(ecx, edx)),
        INSTRUCTION_TEST(83E901                      , sub(ecx, Imm(1))),
        INSTRUCTION_TEST(4883AC118000000001          , sub(qword_ptr(rcx, rdx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(4829D1                      , sub(rcx, rdx)),
        INSTRUCTION_TEST(4883E901                    , sub(rcx, Imm(1))),
        INSTRUCTION_TEST(482B8C1A80000000            , sub(rcx, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(299C1180000000              , sub(dword_ptr(rcx, rdx, 1, 128), ebx)),
        INSTRUCTION_TEST(66299C1180000000            , sub(word_ptr(rcx, rdx, 1, 128), bx)),
        INSTRUCTION_TEST(6683AC118000000001          , sub(word_ptr(rcx, rdx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(48299C1180000000            , sub(qword_ptr(rcx, rdx, 1, 128), rbx)),
        INSTRUCTION_TEST(662B8C1A80000000            , sub(cx, word_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(6683E901                    , sub(cx, Imm(1))),
        INSTRUCTION_TEST(6629D1                      , sub(cx, dx)),
        INSTRUCTION_TEST(28BC1180000000              , sub(byte_ptr(rcx, rdx, 1, 128), bh)),
        INSTRUCTION_TEST(289C1180000000              , sub(byte_ptr(rcx, rdx, 1, 128), bl)),
        INSTRUCTION_TEST(80AC118000000001            , sub(byte_ptr(rcx, rdx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(28F5                        , sub(ch, dh)),
        INSTRUCTION_TEST(28D5                        , sub(ch, dl)),
        INSTRUCTION_TEST(2AAC1A80000000              , sub(ch, byte_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(2A8C1A80000000              , sub(cl, byte_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(28F1                        , sub(cl, dh)),
        INSTRUCTION_TEST(28D1                        , sub(cl, dl)),
        INSTRUCTION_TEST(80E901                      , sub(cl, Imm(1))),
        INSTRUCTION_TEST(80ED01                      , sub(ch, Imm(1))),
        INSTRUCTION_TEST(660F5CCA                    , subpd(xmm1, xmm2)),
        INSTRUCTION_TEST(660F5C8C1A80000000          , subpd(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(0F5CCA                      , subps(xmm1, xmm2)),
        INSTRUCTION_TEST(0F5C8C1A80000000            , subps(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(F20F5C8C1A80000000          , subsd(xmm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(F20F5CCA                    , subsd(xmm1, xmm2)),
        INSTRUCTION_TEST(F30F5C8C1A80000000          , subss(xmm1, dword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(F30F5CCA                    , subss(xmm1, xmm2)),
        INSTRUCTION_TEST(0F01F8                      , swapgs()),
        INSTRUCTION_TEST(0F05                        , syscall()),
        INSTRUCTION_TEST(0F34                        , sysenter()),
        INSTRUCTION_TEST(0F35                        , sysexit()),
        INSTRUCTION_TEST(0F07                        , sysret()),
        INSTRUCTION_TEST(8FE97001BC1A80000000        , t1mskc(ecx, dword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(8FE97001FA                  , t1mskc(ecx, edx)),
        INSTRUCTION_TEST(8FE9F001BC1A80000000        , t1mskc(rcx, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(8FE9F001FA                  , t1mskc(rcx, rdx)),
        INSTRUCTION_TEST(F784118000000001000000      , test(dword_ptr(rcx, rdx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(85D1                        , test(ecx, edx)),
        INSTRUCTION_TEST(F7C101000000                , test(ecx, Imm(1))),
        INSTRUCTION_TEST(48F700FF000000              , test(qword_ptr(rax), Imm(0xFF))),
        INSTRUCTION_TEST(48F784118000000001000000    , test(qword_ptr(rcx, rdx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(66F700FF00                  , test(word_ptr(rax), Imm(0xFF))),
        INSTRUCTION_TEST(48F7C101000000              , test(rcx, Imm(1))),
        INSTRUCTION_TEST(4885D1                      , test(rcx, rdx)),
        INSTRUCTION_TEST(859C1180000000              , test(dword_ptr(rcx, rdx, 1, 128), ebx)),
        INSTRUCTION_TEST(66859C1180000000            , test(word_ptr(rcx, rdx, 1, 128), bx)),
        INSTRUCTION_TEST(66F78411800000000100        , test(word_ptr(rcx, rdx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(48859C1180000000            , test(qword_ptr(rcx, rdx, 1, 128), rbx)),
        INSTRUCTION_TEST(F700FF000000                , test(dword_ptr(rax), Imm(0xFF))),
        INSTRUCTION_TEST(F6C101                      , test(cl, Imm(1))),
        INSTRUCTION_TEST(6685D1                      , test(cx, dx)),
        INSTRUCTION_TEST(F6C436                      , test(ah, Imm(0x36))),
        INSTRUCTION_TEST(A836                        , test(al, Imm(0x36))),
        INSTRUCTION_TEST(F600FF                      , test(byte_ptr(rax), Imm(-1))),
        INSTRUCTION_TEST(84BC1180000000              , test(byte_ptr(rcx, rdx, 1, 128), bh)),
        INSTRUCTION_TEST(66F7C10100                  , test(cx, Imm(1))),
        INSTRUCTION_TEST(F684118000000001            , test(byte_ptr(rcx, rdx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(849C1180000000              , test(byte_ptr(rcx, rdx, 1, 128), bl)),
        INSTRUCTION_TEST(84D5                        , test(ch, dl)),
        INSTRUCTION_TEST(F6C501                      , test(ch, Imm(1))),
        INSTRUCTION_TEST(84F1                        , test(cl, dh)),
        INSTRUCTION_TEST(84D1                        , test(cl, dl)),
        INSTRUCTION_TEST(84F5                        , test(ch, dh)),
        INSTRUCTION_TEST(F30F01ED                    , testui()),
        INSTRUCTION_TEST(660FAEF1                    , tpause(ecx)),
        INSTRUCTION_TEST(F3480FBC8C1A80000000        , tzcnt(rcx, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(F3480FBCCA                  , tzcnt(rcx, rdx)),
        INSTRUCTION_TEST(F30FBCCA                    , tzcnt(ecx, edx)),
        INSTRUCTION_TEST(F3660FBCCA                  , tzcnt(cx, dx)),
        INSTRUCTION_TEST(F3660FBC8C1A80000000        , tzcnt(cx, word_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(F30FBC8C1A80000000          , tzcnt(ecx, dword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(8FE97001A41A80000000        , tzmsk(ecx, dword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(8FE97001E2                  , tzmsk(ecx, edx)),
        INSTRUCTION_TEST(8FE9F001A41A80000000        , tzmsk(rcx, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(8FE9F001E2                  , tzmsk(rcx, rdx)),
        INSTRUCTION_TEST(660F2E8C1A80000000          , ucomisd(xmm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660F2ECA                    , ucomisd(xmm1, xmm2)),
        INSTRUCTION_TEST(0F2ECA                      , ucomiss(xmm1, xmm2)),
        INSTRUCTION_TEST(0F2E8C1A80000000            , ucomiss(xmm1, dword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(0FFF8C1A80000000            , ud0(ecx, dword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(0FFFCA                      , ud0(ecx, edx)),
        INSTRUCTION_TEST(0FB98C1A80000000            , ud1(ecx, dword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(0FB9CA                      , ud1(ecx, edx)),
        INSTRUCTION_TEST(0F0B                        , ud2()),
        INSTRUCTION_TEST(F30F01EC                    , uiret()),
        INSTRUCTION_TEST(F20FAEF1                    , umwait(ecx)),
        INSTRUCTION_TEST(660F158C1A80000000          , unpckhpd(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660F15CA                    , unpckhpd(xmm1, xmm2)),
        INSTRUCTION_TEST(0F15CA                      , unpckhps(xmm1, xmm2)),
        INSTRUCTION_TEST(0F158C1A80000000            , unpckhps(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(660F14CA                    , unpcklpd(xmm1, xmm2)),
        INSTRUCTION_TEST(660F148C1A80000000          , unpcklpd(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(0F14CA                      , unpcklps(xmm1, xmm2)),
        INSTRUCTION_TEST(0F148C1A80000000            , unpcklps(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(6271A540588200200000        , vaddpd(zmm8, k0, zmm27, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(6271A54058427F              , vaddpd(zmm8, k0, zmm27, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(6271A5405882C0DFFFFF        , vaddpd(zmm8, k0, zmm27, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(6271A540584280              , vaddpd(zmm8, k0, zmm27, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(6271A5405801                , vaddpd(zmm8, k0, zmm27, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(6271A54058C6                , vaddpd(zmm8, k0, zmm27, zmm6)),
        INSTRUCTION_TEST(6231A5405884F023010000      , vaddpd(zmm8, k0, zmm27, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62E1A548588A00200000        , vaddpd(zmm17, k0, zmm11, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62E1A548588AC0DFFFFF        , vaddpd(zmm17, k0, zmm11, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62E1A548584A80              , vaddpd(zmm17, k0, zmm11, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62E1A548584A7F              , vaddpd(zmm17, k0, zmm11, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62A1A548588CF034120000      , vaddpd(zmm17, k0, zmm11, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(C5E958CB                    , vaddpd(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(C5E9588C2B80000000          , vaddpd(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C5ED58CB                    , vaddpd(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(C5ED588C2B80000000          , vaddpd(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62E1A5485809                , vaddpd(zmm17, k0, zmm11, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62F1ED4858CB                , vaddpd(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(62F1ED48584C2B02            , vaddpd(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62E1A54858CA                , vaddpd(zmm17, k0, zmm11, zmm2)),
        INSTRUCTION_TEST(62F1F54858CA                , vaddpd(zmm1, k0, zmm1, zmm2)),
        INSTRUCTION_TEST(62F14C48589A00200000        , vaddps(zmm3, k0, zmm6, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62F14C48585A7F              , vaddps(zmm3, k0, zmm6, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62F14C48589AC0DFFFFF        , vaddps(zmm3, k0, zmm6, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62F14C48585A80              , vaddps(zmm3, k0, zmm6, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62F14C485819                , vaddps(zmm3, k0, zmm6, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62B14C48589CF034120000      , vaddps(zmm3, k0, zmm6, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62F14C4858DD                , vaddps(zmm3, k0, zmm6, zmm5)),
        INSTRUCTION_TEST(62E11448589200200000        , vaddps(zmm18, k0, zmm13, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62E1144858527F              , vaddps(zmm18, k0, zmm13, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62E11448585280              , vaddps(zmm18, k0, zmm13, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62E114485892C0DFFFFF        , vaddps(zmm18, k0, zmm13, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(C5E858CB                    , vaddps(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(62E114485811                , vaddps(zmm18, k0, zmm13, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(C5E8588C2B80000000          , vaddps(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C5EC58CB                    , vaddps(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(C5EC588C2B80000000          , vaddps(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62A114485894F023010000      , vaddps(zmm18, k0, zmm13, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62F16C4858CB                , vaddps(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(62F16C48584C2B02            , vaddps(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62E1144858D2                , vaddps(zmm18, k0, zmm13, zmm2)),
        INSTRUCTION_TEST(62F174485808                , vaddps(zmm1, k0, zmm1, zmmword_ptr(rax))),
        INSTRUCTION_TEST(62D1F70058D8                , vaddsd(xmm3, k0, xmm17, xmm8)),
        INSTRUCTION_TEST(62F1F700589A00040000        , vaddsd(xmm3, k0, xmm17, qword_ptr(rdx, 1024))),
        INSTRUCTION_TEST(62F1F700585A7F              , vaddsd(xmm3, k0, xmm17, qword_ptr(rdx, 1016))),
        INSTRUCTION_TEST(62F1F700589AF8FBFFFF        , vaddsd(xmm3, k0, xmm17, qword_ptr(rdx, -1032))),
        INSTRUCTION_TEST(62F1F700585A80              , vaddsd(xmm3, k0, xmm17, qword_ptr(rdx, -1024))),
        INSTRUCTION_TEST(62F1F7005819                , vaddsd(xmm3, k0, xmm17, qword_ptr(rcx))),
        INSTRUCTION_TEST(62B1F700589CF023010000      , vaddsd(xmm3, k0, xmm17, qword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(C5EB58CB                    , vaddsd(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(6291EF0058CB                , vaddsd(xmm1, k0, xmm18, xmm27)),
        INSTRUCTION_TEST(62F1EF00588A00040000        , vaddsd(xmm1, k0, xmm18, qword_ptr(rdx, 1024))),
        INSTRUCTION_TEST(62F1EF00584A7F              , vaddsd(xmm1, k0, xmm18, qword_ptr(rdx, 1016))),
        INSTRUCTION_TEST(62F1EF00588AF8FBFFFF        , vaddsd(xmm1, k0, xmm18, qword_ptr(rdx, -1032))),
        INSTRUCTION_TEST(62F1EF00584A80              , vaddsd(xmm1, k0, xmm18, qword_ptr(rdx, -1024))),
        INSTRUCTION_TEST(62F1EF005809                , vaddsd(xmm1, k0, xmm18, qword_ptr(rcx))),
        INSTRUCTION_TEST(62B1EF00588CF034120000      , vaddsd(xmm1, k0, xmm18, qword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(C5EB588C2B80000000          , vaddsd(xmm1, xmm2, qword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62B1560858FB                , vaddss(xmm7, k0, xmm5, xmm19)),
        INSTRUCTION_TEST(C5D258BA00020000            , vaddss(xmm7, xmm5, dword_ptr(rdx, 512))),
        INSTRUCTION_TEST(C5D258BAFC010000            , vaddss(xmm7, xmm5, dword_ptr(rdx, 508))),
        INSTRUCTION_TEST(C5D258BA00FEFFFF            , vaddss(xmm7, xmm5, dword_ptr(rdx, -512))),
        INSTRUCTION_TEST(C5D25839                    , vaddss(xmm7, xmm5, dword_ptr(rcx))),
        INSTRUCTION_TEST(C4A15258BCF023010000        , vaddss(xmm7, xmm5, dword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62C10E0858CF                , vaddss(xmm17, k0, xmm14, xmm15)),
        INSTRUCTION_TEST(C5D258BAFCFDFFFF            , vaddss(xmm7, xmm5, dword_ptr(rdx, -516))),
        INSTRUCTION_TEST(62E10E08584A7F              , vaddss(xmm17, k0, xmm14, dword_ptr(rdx, 508))),
        INSTRUCTION_TEST(62E10E08588AFCFDFFFF        , vaddss(xmm17, k0, xmm14, dword_ptr(rdx, -516))),
        INSTRUCTION_TEST(62E10E08584A80              , vaddss(xmm17, k0, xmm14, dword_ptr(rdx, -512))),
        INSTRUCTION_TEST(62E10E085809                , vaddss(xmm17, k0, xmm14, dword_ptr(rcx))),
        INSTRUCTION_TEST(62A10E08588CF034120000      , vaddss(xmm17, k0, xmm14, dword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(C5EA58CB                    , vaddss(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(C5EA588C2B80000000          , vaddss(xmm1, xmm2, dword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62E10E08588A00020000        , vaddss(xmm17, k0, xmm14, dword_ptr(rdx, 512))),
        INSTRUCTION_TEST(C5E9D0CB                    , vaddsubpd(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(C5E9D08C2B80000000          , vaddsubpd(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C5EDD0CB                    , vaddsubpd(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(C5EDD08C2B80000000          , vaddsubpd(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C5EFD08C2B80000000          , vaddsubps(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C5EFD0CB                    , vaddsubps(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(C5EBD0CB                    , vaddsubps(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(C5EBD08C2B80000000          , vaddsubps(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E269DECB                  , vaesdec(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(C4E269DE8C2B80000000        , vaesdec(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E26DDECB                  , vaesdec(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(C4E26DDE8C2B80000000        , vaesdec(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F26D48DECB                , vaesdec(zmm1, zmm2, zmm3)),
        INSTRUCTION_TEST(62F26D48DE4C2B02            , vaesdec(zmm1, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F26D48DFCB                , vaesdeclast(zmm1, zmm2, zmm3)),
        INSTRUCTION_TEST(62F26D48DF4C2B02            , vaesdeclast(zmm1, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E26DDF8C2B80000000        , vaesdeclast(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E269DF8C2B80000000        , vaesdeclast(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E26DDFCB                  , vaesdeclast(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(C4E269DFCB                  , vaesdeclast(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(C4E269DCCB                  , vaesenc(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(C4E269DC8C2B80000000        , vaesenc(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E26DDCCB                  , vaesenc(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(C4E26DDC8C2B80000000        , vaesenc(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F26D48DCCB                , vaesenc(zmm1, zmm2, zmm3)),
        INSTRUCTION_TEST(62F26D48DC4C2B02            , vaesenc(zmm1, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F26D48DDCB                , vaesenclast(zmm1, zmm2, zmm3)),
        INSTRUCTION_TEST(C4E26DDD8C2B80000000        , vaesenclast(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F26D48DD4C2B02            , vaesenclast(zmm1, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E269DD8C2B80000000        , vaesenclast(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E269DDCB                  , vaesenclast(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(C4E26DDDCB                  , vaesenclast(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(C4E279DBCA                  , vaesimc(xmm1, xmm2)),
        INSTRUCTION_TEST(C4E279DB8C1A80000000        , vaesimc(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(C4E379DFCA01                , vaeskeygenassist(xmm1, xmm2, Imm(1))),
        INSTRUCTION_TEST(C4E379DF8C1A8000000001      , vaeskeygenassist(xmm1, xmmword_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(6273454803BA002000007B      , valignd(zmm15, k0, zmm7, zmmword_ptr(rdx, 8192), Imm(123))),
        INSTRUCTION_TEST(62136D4003CA7B              , valignd(zmm9, k0, zmm18, zmm26, Imm(123))),
        INSTRUCTION_TEST(62136D4003CAAB              , valignd(zmm9, k0, zmm18, zmm26, Imm(171))),
        INSTRUCTION_TEST(62336D40038CF0230100007B    , valignd(zmm9, k0, zmm18, zmmword_ptr(rax, r14, 8, 291), Imm(123))),
        INSTRUCTION_TEST(62736D40038AC0DFFFFF7B      , valignd(zmm9, k0, zmm18, zmmword_ptr(rdx, -8256), Imm(123))),
        INSTRUCTION_TEST(62736D40034A807B            , valignd(zmm9, k0, zmm18, zmmword_ptr(rdx, -8192), Imm(123))),
        INSTRUCTION_TEST(62736D40038A002000007B      , valignd(zmm9, k0, zmm18, zmmword_ptr(rdx, 8192), Imm(123))),
        INSTRUCTION_TEST(62734548037A7F7B            , valignd(zmm15, k0, zmm7, zmmword_ptr(rdx, 8128), Imm(123))),
        INSTRUCTION_TEST(62736D4003097B              , valignd(zmm9, k0, zmm18, zmmword_ptr(rcx), Imm(123))),
        INSTRUCTION_TEST(6273454803BAC0DFFFFF7B      , valignd(zmm15, k0, zmm7, zmmword_ptr(rdx, -8256), Imm(123))),
        INSTRUCTION_TEST(62736D40034A7F7B            , valignd(zmm9, k0, zmm18, zmmword_ptr(rdx, 8128), Imm(123))),
        INSTRUCTION_TEST(6273454803397B              , valignd(zmm15, k0, zmm7, zmmword_ptr(rcx), Imm(123))),
        INSTRUCTION_TEST(62734548037A807B            , valignd(zmm15, k0, zmm7, zmmword_ptr(rdx, -8192), Imm(123))),
        INSTRUCTION_TEST(62F36D08034C2B0801          , valignd(xmm1, k0, xmm2, xmmword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F36D2803CB01              , valignd(ymm1, k0, ymm2, ymm3, Imm(1))),
        INSTRUCTION_TEST(62F36D28034C2B0401          , valignd(ymm1, k0, ymm2, ymmword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F36D0803CB01              , valignd(xmm1, k0, xmm2, xmm3, Imm(1))),
        INSTRUCTION_TEST(62F36D48034C2B0201          , valignd(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(6233454803BCF0341200007B    , valignd(zmm15, k0, zmm7, zmmword_ptr(rax, r14, 8, 4660), Imm(123))),
        INSTRUCTION_TEST(6273454803FD7B              , valignd(zmm15, k0, zmm7, zmm5, Imm(123))),
        INSTRUCTION_TEST(6273454803FDAB              , valignd(zmm15, k0, zmm7, zmm5, Imm(171))),
        INSTRUCTION_TEST(62F36D4803CB01              , valignd(zmm1, k0, zmm2, zmm3, Imm(1))),
        INSTRUCTION_TEST(6263DD4803A2002000007B      , valignq(zmm28, k0, zmm4, zmmword_ptr(rdx, 8192), Imm(123))),
        INSTRUCTION_TEST(6263DD4803627F7B            , valignq(zmm28, k0, zmm4, zmmword_ptr(rdx, 8128), Imm(123))),
        INSTRUCTION_TEST(6263DD4803A2C0DFFFFF7B      , valignq(zmm28, k0, zmm4, zmmword_ptr(rdx, -8256), Imm(123))),
        INSTRUCTION_TEST(6263DD480362807B            , valignq(zmm28, k0, zmm4, zmmword_ptr(rdx, -8192), Imm(123))),
        INSTRUCTION_TEST(6263DD4803217B              , valignq(zmm28, k0, zmm4, zmmword_ptr(rcx), Imm(123))),
        INSTRUCTION_TEST(6223DD4803A4F0230100007B    , valignq(zmm28, k0, zmm4, zmmword_ptr(rax, r14, 8, 291), Imm(123))),
        INSTRUCTION_TEST(6223DD4803E7AB              , valignq(zmm28, k0, zmm4, zmm23, Imm(171))),
        INSTRUCTION_TEST(6223DD4803E77B              , valignq(zmm28, k0, zmm4, zmm23, Imm(123))),
        INSTRUCTION_TEST(6263F540038A002000007B      , valignq(zmm25, k0, zmm17, zmmword_ptr(rdx, 8192), Imm(123))),
        INSTRUCTION_TEST(6263F540038AC0DFFFFF7B      , valignq(zmm25, k0, zmm17, zmmword_ptr(rdx, -8256), Imm(123))),
        INSTRUCTION_TEST(6263F540034A7F7B            , valignq(zmm25, k0, zmm17, zmmword_ptr(rdx, 8128), Imm(123))),
        INSTRUCTION_TEST(6263F54003097B              , valignq(zmm25, k0, zmm17, zmmword_ptr(rcx), Imm(123))),
        INSTRUCTION_TEST(6223F540038CF0341200007B    , valignq(zmm25, k0, zmm17, zmmword_ptr(rax, r14, 8, 4660), Imm(123))),
        INSTRUCTION_TEST(6223F54003C8AB              , valignq(zmm25, k0, zmm17, zmm16, Imm(171))),
        INSTRUCTION_TEST(6223F54003C87B              , valignq(zmm25, k0, zmm17, zmm16, Imm(123))),
        INSTRUCTION_TEST(62F3ED48034C2B0201          , valignq(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F3ED4803CB01              , valignq(zmm1, k0, zmm2, zmm3, Imm(1))),
        INSTRUCTION_TEST(62F3ED28034C2B0401          , valignq(ymm1, k0, ymm2, ymmword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F3ED2803CB01              , valignq(ymm1, k0, ymm2, ymm3, Imm(1))),
        INSTRUCTION_TEST(62F3ED08034C2B0801          , valignq(xmm1, k0, xmm2, xmmword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F3ED0803CB01              , valignq(xmm1, k0, xmm2, xmm3, Imm(1))),
        INSTRUCTION_TEST(6263F540034A807B            , valignq(zmm25, k0, zmm17, zmmword_ptr(rdx, -8192), Imm(123))),
        INSTRUCTION_TEST(62F1ED48554C2B02            , vandnpd(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F1ED4855CB                , vandnpd(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(C5ED558C2B80000000          , vandnpd(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C5E9558C2B80000000          , vandnpd(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C5E955CB                    , vandnpd(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(C5ED55CB                    , vandnpd(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(C5E855CB                    , vandnps(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(C5E8558C2B80000000          , vandnps(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C5EC55CB                    , vandnps(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(C5EC558C2B80000000          , vandnps(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F16C4855CB                , vandnps(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(62F16C48554C2B02            , vandnps(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F1ED4854CB                , vandpd(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(62F1ED48544C2B02            , vandpd(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C5ED54CB                    , vandpd(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(C5E954CB                    , vandpd(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(C5E9548C2B80000000          , vandpd(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C5ED548C2B80000000          , vandpd(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C5E854CB                    , vandps(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(C5E8548C2B80000000          , vandps(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C5EC54CB                    , vandps(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(C5EC548C2B80000000          , vandps(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F16C4854CB                , vandps(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(62F16C48544C2B02            , vandps(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62529D4065C4                , vblendmpd(zmm8, k0, zmm28, zmm12)),
        INSTRUCTION_TEST(62329D406584F023010000      , vblendmpd(zmm8, k0, zmm28, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62729D406501                , vblendmpd(zmm8, k0, zmm28, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62729D40658200200000        , vblendmpd(zmm8, k0, zmm28, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62729D406582C0DFFFFF        , vblendmpd(zmm8, k0, zmm28, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62729D4065427F              , vblendmpd(zmm8, k0, zmm28, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62E2ED4865BA00200000        , vblendmpd(zmm23, k0, zmm2, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62729D40654280              , vblendmpd(zmm8, k0, zmm28, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62E2ED48657A7F              , vblendmpd(zmm23, k0, zmm2, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62E2ED4865BAC0DFFFFF        , vblendmpd(zmm23, k0, zmm2, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62F2ED0865CB                , vblendmpd(xmm1, k0, xmm2, xmm3)),
        INSTRUCTION_TEST(62E2ED486539                , vblendmpd(zmm23, k0, zmm2, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62A2ED4865BCF034120000      , vblendmpd(zmm23, k0, zmm2, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62C2ED4865F8                , vblendmpd(zmm23, k0, zmm2, zmm8)),
        INSTRUCTION_TEST(62F2ED48654C2B02            , vblendmpd(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F2ED4865CB                , vblendmpd(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(62F2ED28654C2B04            , vblendmpd(ymm1, k0, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F2ED2865CB                , vblendmpd(ymm1, k0, ymm2, ymm3)),
        INSTRUCTION_TEST(62F2ED08654C2B08            , vblendmpd(xmm1, k0, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62E2ED48657A80              , vblendmpd(zmm23, k0, zmm2, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62E2354865BA00200000        , vblendmps(zmm23, k0, zmm9, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62721D4065BA00200000        , vblendmps(zmm15, k0, zmm28, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62E2354865FF                , vblendmps(zmm23, k0, zmm9, zmm7)),
        INSTRUCTION_TEST(62A2354865BCF034120000      , vblendmps(zmm23, k0, zmm9, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62721D40657A7F              , vblendmps(zmm15, k0, zmm28, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62E23548657A80              , vblendmps(zmm23, k0, zmm9, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62E2354865BAC0DFFFFF        , vblendmps(zmm23, k0, zmm9, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62E23548657A7F              , vblendmps(zmm23, k0, zmm9, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62E235486539                , vblendmps(zmm23, k0, zmm9, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62721D4065BAC0DFFFFF        , vblendmps(zmm15, k0, zmm28, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62721D4065FC                , vblendmps(zmm15, k0, zmm28, zmm4)),
        INSTRUCTION_TEST(62721D406539                , vblendmps(zmm15, k0, zmm28, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62321D4065BCF023010000      , vblendmps(zmm15, k0, zmm28, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62F26D48654C2B02            , vblendmps(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F26D4865CB                , vblendmps(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(62F26D28654C2B04            , vblendmps(ymm1, k0, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F26D2865CB                , vblendmps(ymm1, k0, ymm2, ymm3)),
        INSTRUCTION_TEST(62F26D08654C2B08            , vblendmps(xmm1, k0, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F26D0865CB                , vblendmps(xmm1, k0, xmm2, xmm3)),
        INSTRUCTION_TEST(62721D40657A80              , vblendmps(zmm15, k0, zmm28, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(C4E3690DCB01                , vblendpd(xmm1, xmm2, xmm3, Imm(1))),
        INSTRUCTION_TEST(C4E3690D8C2B8000000001      , vblendpd(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(C4E36D0DCB01                , vblendpd(ymm1, ymm2, ymm3, Imm(1))),
        INSTRUCTION_TEST(C4E36D0D8C2B8000000001      , vblendpd(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(C4E36D0C8C2B8000000001      , vblendps(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(C4E36D0CCB01                , vblendps(ymm1, ymm2, ymm3, Imm(1))),
        INSTRUCTION_TEST(C4E3690CCB01                , vblendps(xmm1, xmm2, xmm3, Imm(1))),
        INSTRUCTION_TEST(C4E3690C8C2B8000000001      , vblendps(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(C4E3694BCB40                , vblendvpd(xmm1, xmm2, xmm3, xmm4)),
        INSTRUCTION_TEST(C4E3694B8C2B8000000060      , vblendvpd(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128), xmm6)),
        INSTRUCTION_TEST(C4E36D4BCB40                , vblendvpd(ymm1, ymm2, ymm3, ymm4)),
        INSTRUCTION_TEST(C4E36D4B8C2B8000000060      , vblendvpd(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128), ymm6)),
        INSTRUCTION_TEST(C4E36D4A8C2B8000000060      , vblendvps(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128), ymm6)),
        INSTRUCTION_TEST(C4E36D4ACB40                , vblendvps(ymm1, ymm2, ymm3, ymm4)),
        INSTRUCTION_TEST(C4E3694ACB40                , vblendvps(xmm1, xmm2, xmm3, xmm4)),
        INSTRUCTION_TEST(C4E3694A8C2B8000000060      , vblendvps(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128), xmm6)),
        INSTRUCTION_TEST(C4E27D1A8C1A80000000        , vbroadcastf128(ymm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F27D28194C1A10            , vbroadcastf32x2(ymm1, k0, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F27D2819CA                , vbroadcastf32x2(ymm1, k0, xmm2)),
        INSTRUCTION_TEST(62F27D48194C1A10            , vbroadcastf32x2(zmm1, k0, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F27D4819CA                , vbroadcastf32x2(zmm1, k0, xmm2)),
        INSTRUCTION_TEST(62F27D481A19                , vbroadcastf32x4(zmm3, k0, xmmword_ptr(rcx))),
        INSTRUCTION_TEST(62F27D481A9AF0F7FFFF        , vbroadcastf32x4(zmm3, k0, xmmword_ptr(rdx, -2064))),
        INSTRUCTION_TEST(62B27D481A9CF023010000      , vbroadcastf32x4(zmm3, k0, xmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62F27D481A9A00080000        , vbroadcastf32x4(zmm3, k0, xmmword_ptr(rdx, 2048))),
        INSTRUCTION_TEST(62F27D481A5A7F              , vbroadcastf32x4(zmm3, k0, xmmword_ptr(rdx, 2032))),
        INSTRUCTION_TEST(62727D481ABA00080000        , vbroadcastf32x4(zmm15, k0, xmmword_ptr(rdx, 2048))),
        INSTRUCTION_TEST(62F27D481A5A80              , vbroadcastf32x4(zmm3, k0, xmmword_ptr(rdx, -2048))),
        INSTRUCTION_TEST(62727D481ABAF0F7FFFF        , vbroadcastf32x4(zmm15, k0, xmmword_ptr(rdx, -2064))),
        INSTRUCTION_TEST(62F27D281A4C1A08            , vbroadcastf32x4(ymm1, k0, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F27D481A4C1A08            , vbroadcastf32x4(zmm1, k0, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62327D481ABCF034120000      , vbroadcastf32x4(zmm15, k0, xmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62727D481A7A7F              , vbroadcastf32x4(zmm15, k0, xmmword_ptr(rdx, 2032))),
        INSTRUCTION_TEST(62727D481A39                , vbroadcastf32x4(zmm15, k0, xmmword_ptr(rcx))),
        INSTRUCTION_TEST(62727D481A7A80              , vbroadcastf32x4(zmm15, k0, xmmword_ptr(rdx, -2048))),
        INSTRUCTION_TEST(62F27D481B4C1A04            , vbroadcastf32x8(zmm1, k0, ymmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F2FD281A4C1A08            , vbroadcastf64x2(ymm1, k0, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F2FD481A4C1A08            , vbroadcastf64x2(zmm1, k0, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(6262FD481B9200100000        , vbroadcastf64x4(zmm26, k0, ymmword_ptr(rdx, 4096))),
        INSTRUCTION_TEST(6262FD481B527F              , vbroadcastf64x4(zmm26, k0, ymmword_ptr(rdx, 4064))),
        INSTRUCTION_TEST(6262FD481B92E0EFFFFF        , vbroadcastf64x4(zmm26, k0, ymmword_ptr(rdx, -4128))),
        INSTRUCTION_TEST(6262FD481B5280              , vbroadcastf64x4(zmm26, k0, ymmword_ptr(rdx, -4096))),
        INSTRUCTION_TEST(6262FD481B11                , vbroadcastf64x4(zmm26, k0, ymmword_ptr(rcx))),
        INSTRUCTION_TEST(6222FD481B94F034120000      , vbroadcastf64x4(zmm26, k0, ymmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(6262FD481B4A7F              , vbroadcastf64x4(zmm25, k0, ymmword_ptr(rdx, 4064))),
        INSTRUCTION_TEST(6262FD481B8AE0EFFFFF        , vbroadcastf64x4(zmm25, k0, ymmword_ptr(rdx, -4128))),
        INSTRUCTION_TEST(6262FD481B4A80              , vbroadcastf64x4(zmm25, k0, ymmword_ptr(rdx, -4096))),
        INSTRUCTION_TEST(6262FD481B09                , vbroadcastf64x4(zmm25, k0, ymmword_ptr(rcx))),
        INSTRUCTION_TEST(6222FD481B8CF023010000      , vbroadcastf64x4(zmm25, k0, ymmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62F2FD481B4C1A04            , vbroadcastf64x4(zmm1, k0, ymmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(6262FD481B8A00100000        , vbroadcastf64x4(zmm25, k0, ymmword_ptr(rdx, 4096))),
        INSTRUCTION_TEST(C4E27D5A8C1A80000000        , vbroadcasti128(ymm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F27D48594C1A10            , vbroadcasti32x2(zmm1, k0, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F27D4859CA                , vbroadcasti32x2(zmm1, k0, xmm2)),
        INSTRUCTION_TEST(62F27D2859CA                , vbroadcasti32x2(ymm1, k0, xmm2)),
        INSTRUCTION_TEST(62F27D08594C1A10            , vbroadcasti32x2(xmm1, k0, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F27D0859CA                , vbroadcasti32x2(xmm1, k0, xmm2)),
        INSTRUCTION_TEST(62F27D28594C1A10            , vbroadcasti32x2(ymm1, k0, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62627D485A727F              , vbroadcasti32x4(zmm30, k0, xmmword_ptr(rdx, 2032))),
        INSTRUCTION_TEST(62627D485AB2F0F7FFFF        , vbroadcasti32x4(zmm30, k0, xmmword_ptr(rdx, -2064))),
        INSTRUCTION_TEST(62627D485A7280              , vbroadcasti32x4(zmm30, k0, xmmword_ptr(rdx, -2048))),
        INSTRUCTION_TEST(62627D485A31                , vbroadcasti32x4(zmm30, k0, xmmword_ptr(rcx))),
        INSTRUCTION_TEST(62227D485AB4F023010000      , vbroadcasti32x4(zmm30, k0, xmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62F27D485A9200080000        , vbroadcasti32x4(zmm2, k0, xmmword_ptr(rdx, 2048))),
        INSTRUCTION_TEST(62F27D485A527F              , vbroadcasti32x4(zmm2, k0, xmmword_ptr(rdx, 2032))),
        INSTRUCTION_TEST(62F27D485A5280              , vbroadcasti32x4(zmm2, k0, xmmword_ptr(rdx, -2048))),
        INSTRUCTION_TEST(62F27D485A11                , vbroadcasti32x4(zmm2, k0, xmmword_ptr(rcx))),
        INSTRUCTION_TEST(62B27D485A94F034120000      , vbroadcasti32x4(zmm2, k0, xmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62F27D485A4C1A08            , vbroadcasti32x4(zmm1, k0, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F27D285A4C1A08            , vbroadcasti32x4(ymm1, k0, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62627D485AB200080000        , vbroadcasti32x4(zmm30, k0, xmmword_ptr(rdx, 2048))),
        INSTRUCTION_TEST(62F27D485A92F0F7FFFF        , vbroadcasti32x4(zmm2, k0, xmmword_ptr(rdx, -2064))),
        INSTRUCTION_TEST(62F27D485B4C1A04            , vbroadcasti32x8(zmm1, k0, ymmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F2FD485A4C1A08            , vbroadcasti64x2(zmm1, k0, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F2FD285A4C1A08            , vbroadcasti64x2(ymm1, k0, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62E2FD485B8AE0EFFFFF        , vbroadcasti64x4(zmm17, k0, ymmword_ptr(rdx, -4128))),
        INSTRUCTION_TEST(62E2FD485B8A00100000        , vbroadcasti64x4(zmm17, k0, ymmword_ptr(rdx, 4096))),
        INSTRUCTION_TEST(62E2FD485B4A7F              , vbroadcasti64x4(zmm17, k0, ymmword_ptr(rdx, 4064))),
        INSTRUCTION_TEST(62E2FD485B4A80              , vbroadcasti64x4(zmm17, k0, ymmword_ptr(rdx, -4096))),
        INSTRUCTION_TEST(62E2FD485B09                , vbroadcasti64x4(zmm17, k0, ymmword_ptr(rcx))),
        INSTRUCTION_TEST(6272FD485BAA00100000        , vbroadcasti64x4(zmm13, k0, ymmword_ptr(rdx, 4096))),
        INSTRUCTION_TEST(62A2FD485B8CF034120000      , vbroadcasti64x4(zmm17, k0, ymmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(6272FD485BAAE0EFFFFF        , vbroadcasti64x4(zmm13, k0, ymmword_ptr(rdx, -4128))),
        INSTRUCTION_TEST(6272FD485B6A80              , vbroadcasti64x4(zmm13, k0, ymmword_ptr(rdx, -4096))),
        INSTRUCTION_TEST(6272FD485B29                , vbroadcasti64x4(zmm13, k0, ymmword_ptr(rcx))),
        INSTRUCTION_TEST(6232FD485BACF023010000      , vbroadcasti64x4(zmm13, k0, ymmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62F2FD485B4C1A04            , vbroadcasti64x4(zmm1, k0, ymmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(6272FD485B6A7F              , vbroadcasti64x4(zmm13, k0, ymmword_ptr(rdx, 4064))),
        INSTRUCTION_TEST(6262FD481931                , vbroadcastsd(zmm30, k0, qword_ptr(rcx))),
        INSTRUCTION_TEST(6222FD4819B4F023010000      , vbroadcastsd(zmm30, k0, qword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(6262FD48197280              , vbroadcastsd(zmm30, k0, qword_ptr(rdx, -1024))),
        INSTRUCTION_TEST(6222FD48199CF034120000      , vbroadcastsd(zmm27, k0, qword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(6262FD4819727F              , vbroadcastsd(zmm30, k0, qword_ptr(rdx, 1016))),
        INSTRUCTION_TEST(6262FD4819B200040000        , vbroadcastsd(zmm30, k0, qword_ptr(rdx, 1024))),
        INSTRUCTION_TEST(6262FD48199A00040000        , vbroadcastsd(zmm27, k0, qword_ptr(rdx, 1024))),
        INSTRUCTION_TEST(6262FD4819B2F8FBFFFF        , vbroadcastsd(zmm30, k0, qword_ptr(rdx, -1032))),
        INSTRUCTION_TEST(6262FD48195A7F              , vbroadcastsd(zmm27, k0, qword_ptr(rdx, 1016))),
        INSTRUCTION_TEST(62F2FD4819CA                , vbroadcastsd(zmm1, k0, xmm2)),
        INSTRUCTION_TEST(6262FD48195A80              , vbroadcastsd(zmm27, k0, qword_ptr(rdx, -1024))),
        INSTRUCTION_TEST(6262FD481919                , vbroadcastsd(zmm27, k0, qword_ptr(rcx))),
        INSTRUCTION_TEST(62A2FD4819EE                , vbroadcastsd(zmm21, k0, xmm22)),
        INSTRUCTION_TEST(6282FD4819D9                , vbroadcastsd(zmm19, k0, xmm25)),
        INSTRUCTION_TEST(62F2FD48194C1A10            , vbroadcastsd(zmm1, k0, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(C4E27D19CA                  , vbroadcastsd(ymm1, xmm2)),
        INSTRUCTION_TEST(C4E27D198C1A80000000        , vbroadcastsd(ymm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(6262FD48199AF8FBFFFF        , vbroadcastsd(zmm27, k0, qword_ptr(rdx, -1032))),
        INSTRUCTION_TEST(62B27D48189CF023010000      , vbroadcastss(zmm3, k0, dword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62F27D48189200020000        , vbroadcastss(zmm2, k0, dword_ptr(rdx, 512))),
        INSTRUCTION_TEST(62C27D4818FF                , vbroadcastss(zmm23, k0, xmm15)),
        INSTRUCTION_TEST(62F27D481819                , vbroadcastss(zmm3, k0, dword_ptr(rcx))),
        INSTRUCTION_TEST(62F27D4818527F              , vbroadcastss(zmm2, k0, dword_ptr(rdx, 508))),
        INSTRUCTION_TEST(62F27D48189AFCFDFFFF        , vbroadcastss(zmm3, k0, dword_ptr(rdx, -516))),
        INSTRUCTION_TEST(62F27D48185A7F              , vbroadcastss(zmm3, k0, dword_ptr(rdx, 508))),
        INSTRUCTION_TEST(62F27D48189A00020000        , vbroadcastss(zmm3, k0, dword_ptr(rdx, 512))),
        INSTRUCTION_TEST(62F27D48185A80              , vbroadcastss(zmm3, k0, dword_ptr(rdx, -512))),
        INSTRUCTION_TEST(62F27D481892FCFDFFFF        , vbroadcastss(zmm2, k0, dword_ptr(rdx, -516))),
        INSTRUCTION_TEST(62F27D48184C1A20            , vbroadcastss(zmm1, k0, dword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F27D481811                , vbroadcastss(zmm2, k0, dword_ptr(rcx))),
        INSTRUCTION_TEST(62F27D48185280              , vbroadcastss(zmm2, k0, dword_ptr(rdx, -512))),
        INSTRUCTION_TEST(C4E279188C1A80000000        , vbroadcastss(xmm1, dword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(C4E27D188C1A80000000        , vbroadcastss(ymm1, dword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(C4E27D18CA                  , vbroadcastss(ymm1, xmm2)),
        INSTRUCTION_TEST(C4E27918CA                  , vbroadcastss(xmm1, xmm2)),
        INSTRUCTION_TEST(62F27D4818CA                , vbroadcastss(zmm1, k0, xmm2)),
        INSTRUCTION_TEST(62B27D481894F034120000      , vbroadcastss(zmm2, k0, dword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62A27D4818D2                , vbroadcastss(zmm18, k0, xmm18)),
        INSTRUCTION_TEST(C5EDC28C2B8000000001        , vcmppd(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(C5EDC2CB01                  , vcmppd(ymm1, ymm2, ymm3, Imm(1))),
        INSTRUCTION_TEST(C5E9C28C2B8000000001        , vcmppd(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(C5E9C2CB01                  , vcmppd(xmm1, xmm2, xmm3, Imm(1))),
        INSTRUCTION_TEST(62F18548C2AA002000007B      , vcmppd(k5, k0, zmm15, zmmword_ptr(rdx, 8192), Imm(123))),
        INSTRUCTION_TEST(62F18548C26A7F7B            , vcmppd(k5, k0, zmm15, zmmword_ptr(rdx, 8128), Imm(123))),
        INSTRUCTION_TEST(62F18548C2AAC0DFFFFF7B      , vcmppd(k5, k0, zmm15, zmmword_ptr(rdx, -8256), Imm(123))),
        INSTRUCTION_TEST(62F18548C26A807B            , vcmppd(k5, k0, zmm15, zmmword_ptr(rdx, -8192), Imm(123))),
        INSTRUCTION_TEST(62B18548C2ACF0341200007B    , vcmppd(k5, k0, zmm15, zmmword_ptr(rax, r14, 8, 4660), Imm(123))),
        INSTRUCTION_TEST(62B18548C2E8AB              , vcmppd(k5, k0, zmm15, zmm16, Imm(171))),
        INSTRUCTION_TEST(62B18548C2E87B              , vcmppd(k5, k0, zmm15, zmm16, Imm(123))),
        INSTRUCTION_TEST(62F19D48C292002000007B      , vcmppd(k2, k0, zmm12, zmmword_ptr(rdx, 8192), Imm(123))),
        INSTRUCTION_TEST(62F18548C2297B              , vcmppd(k5, k0, zmm15, zmmword_ptr(rcx), Imm(123))),
        INSTRUCTION_TEST(62F19D48C292C0DFFFFF7B      , vcmppd(k2, k0, zmm12, zmmword_ptr(rdx, -8256), Imm(123))),
        INSTRUCTION_TEST(62F19D48C2527F7B            , vcmppd(k2, k0, zmm12, zmmword_ptr(rdx, 8128), Imm(123))),
        INSTRUCTION_TEST(62F1ED08C24C2B0801          , vcmppd(k1, k0, xmm2, xmmword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F1ED28C2CB01              , vcmppd(k1, k0, ymm2, ymm3, Imm(1))),
        INSTRUCTION_TEST(62F1ED28C24C2B0401          , vcmppd(k1, k0, ymm2, ymmword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F1ED48C2CB01              , vcmppd(k1, k0, zmm2, zmm3, Imm(1))),
        INSTRUCTION_TEST(62F1ED48C24C2B0201          , vcmppd(k1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F1ED08C2CB01              , vcmppd(k1, k0, xmm2, xmm3, Imm(1))),
        INSTRUCTION_TEST(62919D48C2D2AB              , vcmppd(k2, k0, zmm12, zmm26, Imm(171))),
        INSTRUCTION_TEST(62B19D48C294F0230100007B    , vcmppd(k2, k0, zmm12, zmmword_ptr(rax, r14, 8, 291), Imm(123))),
        INSTRUCTION_TEST(62F19D48C2117B              , vcmppd(k2, k0, zmm12, zmmword_ptr(rcx), Imm(123))),
        INSTRUCTION_TEST(62F19D48C252807B            , vcmppd(k2, k0, zmm12, zmmword_ptr(rdx, -8192), Imm(123))),
        INSTRUCTION_TEST(62919D48C2D27B              , vcmppd(k2, k0, zmm12, zmm26, Imm(123))),
        INSTRUCTION_TEST(62916448C2D5AB              , vcmpps(k2, k0, zmm3, zmm29, Imm(171))),
        INSTRUCTION_TEST(62B16448C294F0341200007B    , vcmpps(k2, k0, zmm3, zmmword_ptr(rax, r14, 8, 4660), Imm(123))),
        INSTRUCTION_TEST(62F16448C2117B              , vcmpps(k2, k0, zmm3, zmmword_ptr(rcx), Imm(123))),
        INSTRUCTION_TEST(62F16448C252807B            , vcmpps(k2, k0, zmm3, zmmword_ptr(rdx, -8192), Imm(123))),
        INSTRUCTION_TEST(62F16448C292C0DFFFFF7B      , vcmpps(k2, k0, zmm3, zmmword_ptr(rdx, -8256), Imm(123))),
        INSTRUCTION_TEST(C5E8C2CB01                  , vcmpps(xmm1, xmm2, xmm3, Imm(1))),
        INSTRUCTION_TEST(62F16448C292002000007B      , vcmpps(k2, k0, zmm3, zmmword_ptr(rdx, 8192), Imm(123))),
        INSTRUCTION_TEST(C5E8C28C2B8000000001        , vcmpps(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(C5ECC2CB01                  , vcmpps(ymm1, ymm2, ymm3, Imm(1))),
        INSTRUCTION_TEST(C5ECC28C2B8000000001        , vcmpps(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62916448C2D57B              , vcmpps(k2, k0, zmm3, zmm29, Imm(123))),
        INSTRUCTION_TEST(62F16448C2527F7B            , vcmpps(k2, k0, zmm3, zmmword_ptr(rdx, 8128), Imm(123))),
        INSTRUCTION_TEST(62F17440C292002000007B      , vcmpps(k2, k0, zmm17, zmmword_ptr(rdx, 8192), Imm(123))),
        INSTRUCTION_TEST(62F16C28C24C2B0401          , vcmpps(k1, k0, ymm2, ymmword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F17440C292C0DFFFFF7B      , vcmpps(k2, k0, zmm17, zmmword_ptr(rdx, -8256), Imm(123))),
        INSTRUCTION_TEST(62F17440C2527F7B            , vcmpps(k2, k0, zmm17, zmmword_ptr(rdx, 8128), Imm(123))),
        INSTRUCTION_TEST(62F16C28C2CB01              , vcmpps(k1, k0, ymm2, ymm3, Imm(1))),
        INSTRUCTION_TEST(62F16C48C2CB01              , vcmpps(k1, k0, zmm2, zmm3, Imm(1))),
        INSTRUCTION_TEST(62F16C48C24C2B0201          , vcmpps(k1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F16C08C24C2B0801          , vcmpps(k1, k0, xmm2, xmmword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62B17440C2D6AB              , vcmpps(k2, k0, zmm17, zmm22, Imm(171))),
        INSTRUCTION_TEST(62B17440C294F0230100007B    , vcmpps(k2, k0, zmm17, zmmword_ptr(rax, r14, 8, 291), Imm(123))),
        INSTRUCTION_TEST(62F17440C2117B              , vcmpps(k2, k0, zmm17, zmmword_ptr(rcx), Imm(123))),
        INSTRUCTION_TEST(62F17440C252807B            , vcmpps(k2, k0, zmm17, zmmword_ptr(rdx, -8192), Imm(123))),
        INSTRUCTION_TEST(62B17440C2D67B              , vcmpps(k2, k0, zmm17, zmm22, Imm(123))),
        INSTRUCTION_TEST(62F16C08C2CB01              , vcmpps(k1, k0, xmm2, xmm3, Imm(1))),
        INSTRUCTION_TEST(C5EBC2CB01                  , vcmpsd(xmm1, xmm2, xmm3, Imm(1))),
        INSTRUCTION_TEST(62B1E700C2ACF0230100007B    , vcmpsd(k5, k0, xmm19, qword_ptr(rax, r14, 8, 291), Imm(123))),
        INSTRUCTION_TEST(C5EBC28C2B8000000001        , vcmpsd(xmm1, xmm2, qword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F1E700C2ECAB              , vcmpsd(k5, k0, xmm19, xmm4, Imm(171))),
        INSTRUCTION_TEST(62F1E700C2EC7B              , vcmpsd(k5, k0, xmm19, xmm4, Imm(123))),
        INSTRUCTION_TEST(62F1E700C2AA000400007B      , vcmpsd(k5, k0, xmm19, qword_ptr(rdx, 1024), Imm(123))),
        INSTRUCTION_TEST(62F1E700C26A7F7B            , vcmpsd(k5, k0, xmm19, qword_ptr(rdx, 1016), Imm(123))),
        INSTRUCTION_TEST(62F1E700C2AAF8FBFFFF7B      , vcmpsd(k5, k0, xmm19, qword_ptr(rdx, -1032), Imm(123))),
        INSTRUCTION_TEST(62F1E700C26A807B            , vcmpsd(k5, k0, xmm19, qword_ptr(rdx, -1024), Imm(123))),
        INSTRUCTION_TEST(62F1E700C2297B              , vcmpsd(k5, k0, xmm19, qword_ptr(rcx), Imm(123))),
        INSTRUCTION_TEST(62F1D708C2E6AB              , vcmpsd(k4, k0, xmm5, xmm6, Imm(171))),
        INSTRUCTION_TEST(62F1D708C2A2000400007B      , vcmpsd(k4, k0, xmm5, qword_ptr(rdx, 1024), Imm(123))),
        INSTRUCTION_TEST(62F1D708C2627F7B            , vcmpsd(k4, k0, xmm5, qword_ptr(rdx, 1016), Imm(123))),
        INSTRUCTION_TEST(62F1D708C2A2F8FBFFFF7B      , vcmpsd(k4, k0, xmm5, qword_ptr(rdx, -1032), Imm(123))),
        INSTRUCTION_TEST(62F1D708C262807B            , vcmpsd(k4, k0, xmm5, qword_ptr(rdx, -1024), Imm(123))),
        INSTRUCTION_TEST(62F1D708C2217B              , vcmpsd(k4, k0, xmm5, qword_ptr(rcx), Imm(123))),
        INSTRUCTION_TEST(62B1D708C2A4F0341200007B    , vcmpsd(k4, k0, xmm5, qword_ptr(rax, r14, 8, 4660), Imm(123))),
        INSTRUCTION_TEST(62F1EF08C2CB01              , vcmpsd(k1, k0, xmm2, xmm3, Imm(1))),
        INSTRUCTION_TEST(62F1EF08C24C2B1001          , vcmpsd(k1, k0, xmm2, qword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F1D708C2E67B              , vcmpsd(k4, k0, xmm5, xmm6, Imm(123))),
        INSTRUCTION_TEST(62F10608C262807B            , vcmpss(k4, k0, xmm15, dword_ptr(rdx, -512), Imm(123))),
        INSTRUCTION_TEST(62F10608C2A2FCFDFFFF7B      , vcmpss(k4, k0, xmm15, dword_ptr(rdx, -516), Imm(123))),
        INSTRUCTION_TEST(62F10608C2627F7B            , vcmpss(k4, k0, xmm15, dword_ptr(rdx, 508), Imm(123))),
        INSTRUCTION_TEST(62F10608C2A2000200007B      , vcmpss(k4, k0, xmm15, dword_ptr(rdx, 512), Imm(123))),
        INSTRUCTION_TEST(C5EAC2CB01                  , vcmpss(xmm1, xmm2, xmm3, Imm(1))),
        INSTRUCTION_TEST(62D10608C2E4AB              , vcmpss(k4, k0, xmm15, xmm12, Imm(171))),
        INSTRUCTION_TEST(C5EAC28C2B8000000001        , vcmpss(xmm1, xmm2, dword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F10608C2217B              , vcmpss(k4, k0, xmm15, dword_ptr(rcx), Imm(123))),
        INSTRUCTION_TEST(62D10608C2E47B              , vcmpss(k4, k0, xmm15, xmm12, Imm(123))),
        INSTRUCTION_TEST(62B10608C2A4F0230100007B    , vcmpss(k4, k0, xmm15, dword_ptr(rax, r14, 8, 291), Imm(123))),
        INSTRUCTION_TEST(62916600C2D0AB              , vcmpss(k2, k0, xmm19, xmm24, Imm(171))),
        INSTRUCTION_TEST(62916600C2D07B              , vcmpss(k2, k0, xmm19, xmm24, Imm(123))),
        INSTRUCTION_TEST(62F16E08C2CB01              , vcmpss(k1, k0, xmm2, xmm3, Imm(1))),
        INSTRUCTION_TEST(62B16600C294F0341200007B    , vcmpss(k2, k0, xmm19, dword_ptr(rax, r14, 8, 4660), Imm(123))),
        INSTRUCTION_TEST(62F16600C2117B              , vcmpss(k2, k0, xmm19, dword_ptr(rcx), Imm(123))),
        INSTRUCTION_TEST(62F16E08C24C2B2001          , vcmpss(k1, k0, xmm2, dword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F16600C292FCFDFFFF7B      , vcmpss(k2, k0, xmm19, dword_ptr(rdx, -516), Imm(123))),
        INSTRUCTION_TEST(62F16600C2527F7B            , vcmpss(k2, k0, xmm19, dword_ptr(rdx, 508), Imm(123))),
        INSTRUCTION_TEST(62F16600C292000200007B      , vcmpss(k2, k0, xmm19, dword_ptr(rdx, 512), Imm(123))),
        INSTRUCTION_TEST(62F16600C252807B            , vcmpss(k2, k0, xmm19, dword_ptr(rdx, -512), Imm(123))),
        INSTRUCTION_TEST(62A1FD082FBCF023010000      , vcomisd(xmm23, qword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62A1FD082FFD                , vcomisd(xmm23, xmm21)),
        INSTRUCTION_TEST(62E1FD082FBA00040000        , vcomisd(xmm23, qword_ptr(rdx, 1024))),
        INSTRUCTION_TEST(62E1FD082F7A7F              , vcomisd(xmm23, qword_ptr(rdx, 1016))),
        INSTRUCTION_TEST(62E1FD082FBAF8FBFFFF        , vcomisd(xmm23, qword_ptr(rdx, -1032))),
        INSTRUCTION_TEST(62E1FD082F7A80              , vcomisd(xmm23, qword_ptr(rdx, -1024))),
        INSTRUCTION_TEST(62E1FD082F39                , vcomisd(xmm23, qword_ptr(rcx))),
        INSTRUCTION_TEST(C441792FEB                  , vcomisd(xmm13, xmm11)),
        INSTRUCTION_TEST(C5792FAAF8030000            , vcomisd(xmm13, qword_ptr(rdx, 1016))),
        INSTRUCTION_TEST(C5792FAAF8FBFFFF            , vcomisd(xmm13, qword_ptr(rdx, -1032))),
        INSTRUCTION_TEST(C5792FAA00FCFFFF            , vcomisd(xmm13, qword_ptr(rdx, -1024))),
        INSTRUCTION_TEST(C5792F29                    , vcomisd(xmm13, qword_ptr(rcx))),
        INSTRUCTION_TEST(C421792FACF034120000        , vcomisd(xmm13, qword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(C5F92FCA                    , vcomisd(xmm1, xmm2)),
        INSTRUCTION_TEST(C5F92F8C1A80000000          , vcomisd(xmm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(C5792FAA00040000            , vcomisd(xmm13, qword_ptr(rdx, 1024))),
        INSTRUCTION_TEST(62E17C082F01                , vcomiss(xmm16, dword_ptr(rcx))),
        INSTRUCTION_TEST(C441782FC7                  , vcomiss(xmm8, xmm15)),
        INSTRUCTION_TEST(C5782F8200020000            , vcomiss(xmm8, dword_ptr(rdx, 512))),
        INSTRUCTION_TEST(C5782F82FC010000            , vcomiss(xmm8, dword_ptr(rdx, 508))),
        INSTRUCTION_TEST(C5782F82FCFDFFFF            , vcomiss(xmm8, dword_ptr(rdx, -516))),
        INSTRUCTION_TEST(C5782F8200FEFFFF            , vcomiss(xmm8, dword_ptr(rdx, -512))),
        INSTRUCTION_TEST(C5782F01                    , vcomiss(xmm8, dword_ptr(rcx))),
        INSTRUCTION_TEST(C421782F84F034120000        , vcomiss(xmm8, dword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(C5782FB2FC010000            , vcomiss(xmm14, dword_ptr(rdx, 508))),
        INSTRUCTION_TEST(C5782FB200020000            , vcomiss(xmm14, dword_ptr(rdx, 512))),
        INSTRUCTION_TEST(C5782FB2FCFDFFFF            , vcomiss(xmm14, dword_ptr(rdx, -516))),
        INSTRUCTION_TEST(C5782FB200FEFFFF            , vcomiss(xmm14, dword_ptr(rdx, -512))),
        INSTRUCTION_TEST(C5782F31                    , vcomiss(xmm14, dword_ptr(rcx))),
        INSTRUCTION_TEST(C421782FB4F023010000        , vcomiss(xmm14, dword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(C5F82FCA                    , vcomiss(xmm1, xmm2)),
        INSTRUCTION_TEST(C5F82F8C1A80000000          , vcomiss(xmm1, dword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62117C082FF4                , vcomiss(xmm14, xmm28)),
        INSTRUCTION_TEST(6272FD488A4A80              , vcompresspd(zmmword_ptr(rdx, -1024), k0, zmm9)),
        INSTRUCTION_TEST(6262FD488A7280              , vcompresspd(zmmword_ptr(rdx, -1024), k0, zmm30)),
        INSTRUCTION_TEST(6262FD488AB2F8FBFFFF        , vcompresspd(zmmword_ptr(rdx, -1032), k0, zmm30)),
        INSTRUCTION_TEST(6272FD488A09                , vcompresspd(zmmword_ptr(rcx), k0, zmm9)),
        INSTRUCTION_TEST(6262FD488A727F              , vcompresspd(zmmword_ptr(rdx, 1016), k0, zmm30)),
        INSTRUCTION_TEST(6272FD488A8A00040000        , vcompresspd(zmmword_ptr(rdx, 1024), k0, zmm9)),
        INSTRUCTION_TEST(6262FD488AB200040000        , vcompresspd(zmmword_ptr(rdx, 1024), k0, zmm30)),
        INSTRUCTION_TEST(6272FD488A8AF8FBFFFF        , vcompresspd(zmmword_ptr(rdx, -1032), k0, zmm9)),
        INSTRUCTION_TEST(6262FD488A31                , vcompresspd(zmmword_ptr(rcx), k0, zmm30)),
        INSTRUCTION_TEST(6272FD488A4A7F              , vcompresspd(zmmword_ptr(rdx, 1016), k0, zmm9)),
        INSTRUCTION_TEST(6222FD488AB4F034120000      , vcompresspd(zmmword_ptr(rax, r14, 8, 4660), k0, zmm30)),
        INSTRUCTION_TEST(62F2FD488A5C1110            , vcompresspd(zmmword_ptr(rcx, rdx, 1, 128), k0, zmm3)),
        INSTRUCTION_TEST(62F2FD088A5C1110            , vcompresspd(xmmword_ptr(rcx, rdx, 1, 128), k0, xmm3)),
        INSTRUCTION_TEST(62F2FD288AD1                , vcompresspd(ymm1, k0, ymm2)),
        INSTRUCTION_TEST(62F2FD288A5C1110            , vcompresspd(ymmword_ptr(rcx, rdx, 1, 128), k0, ymm3)),
        INSTRUCTION_TEST(62F2FD088AD1                , vcompresspd(xmm1, k0, xmm2)),
        INSTRUCTION_TEST(6262FD488AC9                , vcompresspd(zmm1, k0, zmm25)),
        INSTRUCTION_TEST(62D2FD488AE0                , vcompresspd(zmm8, k0, zmm4)),
        INSTRUCTION_TEST(6232FD488A8CF023010000      , vcompresspd(zmmword_ptr(rax, r14, 8, 291), k0, zmm9)),
        INSTRUCTION_TEST(62F2FD488AD1                , vcompresspd(zmm1, k0, zmm2)),
        INSTRUCTION_TEST(62F27D488A5C1120            , vcompressps(zmmword_ptr(rcx, rdx, 1, 128), k0, zmm3)),
        INSTRUCTION_TEST(62727D488A9200020000        , vcompressps(zmmword_ptr(rdx, 512), k0, zmm10)),
        INSTRUCTION_TEST(62727D488A527F              , vcompressps(zmmword_ptr(rdx, 508), k0, zmm10)),
        INSTRUCTION_TEST(62727D488A92FCFDFFFF        , vcompressps(zmmword_ptr(rdx, -516), k0, zmm10)),
        INSTRUCTION_TEST(62727D488A5280              , vcompressps(zmmword_ptr(rdx, -512), k0, zmm10)),
        INSTRUCTION_TEST(62727D488A11                , vcompressps(zmmword_ptr(rcx), k0, zmm10)),
        INSTRUCTION_TEST(62327D488A94F034120000      , vcompressps(zmmword_ptr(rax, r14, 8, 4660), k0, zmm10)),
        INSTRUCTION_TEST(62727D488AF4                , vcompressps(zmm4, k0, zmm14)),
        INSTRUCTION_TEST(62B27D488AFB                , vcompressps(zmm19, k0, zmm7)),
        INSTRUCTION_TEST(62F27D488AD1                , vcompressps(zmm1, k0, zmm2)),
        INSTRUCTION_TEST(62F27D288A5C1120            , vcompressps(ymmword_ptr(rcx, rdx, 1, 128), k0, ymm3)),
        INSTRUCTION_TEST(62F27D288AD1                , vcompressps(ymm1, k0, ymm2)),
        INSTRUCTION_TEST(62F27D088A5C1120            , vcompressps(xmmword_ptr(rcx, rdx, 1, 128), k0, xmm3)),
        INSTRUCTION_TEST(62F27D088AD1                , vcompressps(xmm1, k0, xmm2)),
        INSTRUCTION_TEST(62327D488A94F023010000      , vcompressps(zmmword_ptr(rax, r14, 8, 291), k0, zmm10)),
        INSTRUCTION_TEST(62E17E48E6727F              , vcvtdq2pd(zmm22, k0, ymmword_ptr(rdx, 4064))),
        INSTRUCTION_TEST(62617E48E6A200100000        , vcvtdq2pd(zmm28, k0, ymmword_ptr(rdx, 4096))),
        INSTRUCTION_TEST(62617E48E6A2E0EFFFFF        , vcvtdq2pd(zmm28, k0, ymmword_ptr(rdx, -4128))),
        INSTRUCTION_TEST(62617E48E66280              , vcvtdq2pd(zmm28, k0, ymmword_ptr(rdx, -4096))),
        INSTRUCTION_TEST(62617E48E621                , vcvtdq2pd(zmm28, k0, ymmword_ptr(rcx))),
        INSTRUCTION_TEST(62217E48E6A4F034120000      , vcvtdq2pd(zmm28, k0, ymmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62217E48E6E4                , vcvtdq2pd(zmm28, k0, ymm20)),
        INSTRUCTION_TEST(62E17E48E6B200100000        , vcvtdq2pd(zmm22, k0, ymmword_ptr(rdx, 4096))),
        INSTRUCTION_TEST(62E17E48E6B2E0EFFFFF        , vcvtdq2pd(zmm22, k0, ymmword_ptr(rdx, -4128))),
        INSTRUCTION_TEST(62617E48E6627F              , vcvtdq2pd(zmm28, k0, ymmword_ptr(rdx, 4064))),
        INSTRUCTION_TEST(62E17E48E631                , vcvtdq2pd(zmm22, k0, ymmword_ptr(rcx))),
        INSTRUCTION_TEST(62A17E48E6B4F023010000      , vcvtdq2pd(zmm22, k0, ymmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62817E48E6F0                , vcvtdq2pd(zmm22, k0, ymm24)),
        INSTRUCTION_TEST(62F17E48E64C1A04            , vcvtdq2pd(zmm1, k0, ymmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F17E48E6CA                , vcvtdq2pd(zmm1, k0, ymm2)),
        INSTRUCTION_TEST(C5FEE68C1A80000000          , vcvtdq2pd(ymm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(C5FEE6CA                    , vcvtdq2pd(ymm1, xmm2)),
        INSTRUCTION_TEST(C5FAE6CA                    , vcvtdq2pd(xmm1, xmm2)),
        INSTRUCTION_TEST(C5FAE68C1A80000000          , vcvtdq2pd(xmm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62E17E48E67280              , vcvtdq2pd(zmm22, k0, ymmword_ptr(rdx, -4096))),
        INSTRUCTION_TEST(62E17C485B9A00200000        , vcvtdq2ps(zmm19, k0, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62917C485BF9                , vcvtdq2ps(zmm7, k0, zmm25)),
        INSTRUCTION_TEST(62B17C485BBCF023010000      , vcvtdq2ps(zmm7, k0, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62F17C485BBA00200000        , vcvtdq2ps(zmm7, k0, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62F17C485BBAC0DFFFFF        , vcvtdq2ps(zmm7, k0, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62F17C485B7A7F              , vcvtdq2ps(zmm7, k0, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62E17C485B5A7F              , vcvtdq2ps(zmm19, k0, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62F17C485B7A80              , vcvtdq2ps(zmm7, k0, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62E17C485B9AC0DFFFFF        , vcvtdq2ps(zmm19, k0, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62F17C485B39                , vcvtdq2ps(zmm7, k0, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62E17C485B19                , vcvtdq2ps(zmm19, k0, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62A17C485B9CF034120000      , vcvtdq2ps(zmm19, k0, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62E17C485BDC                , vcvtdq2ps(zmm19, k0, zmm4)),
        INSTRUCTION_TEST(62F17C485B4C1A02            , vcvtdq2ps(zmm1, k0, zmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F17C485BCA                , vcvtdq2ps(zmm1, k0, zmm2)),
        INSTRUCTION_TEST(C5FC5B8C1A80000000          , vcvtdq2ps(ymm1, ymmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(C5FC5BCA                    , vcvtdq2ps(ymm1, ymm2)),
        INSTRUCTION_TEST(C5F85B8C1A80000000          , vcvtdq2ps(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(C5F85BCA                    , vcvtdq2ps(xmm1, xmm2)),
        INSTRUCTION_TEST(62E17C485B5A80              , vcvtdq2ps(zmm19, k0, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62F26F4872CB                , vcvtne2ps2bf16(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(62F26F28724C2B04            , vcvtne2ps2bf16(ymm1, k0, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F26F48724C2B02            , vcvtne2ps2bf16(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F26F08724C2B08            , vcvtne2ps2bf16(xmm1, k0, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F26F0872CB                , vcvtne2ps2bf16(xmm1, k0, xmm2, xmm3)),
        INSTRUCTION_TEST(62F26F2872CB                , vcvtne2ps2bf16(ymm1, k0, ymm2, ymm3)),
        INSTRUCTION_TEST(62F27E0872CA                , vcvtneps2bf16(xmm1, k0, xmm2)),
        INSTRUCTION_TEST(62F27E08724C1A08            , vcvtneps2bf16(xmm1, k0, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F27E2872CA                , vcvtneps2bf16(xmm1, k0, ymm2)),
        INSTRUCTION_TEST(62F27E28724C1A04            , vcvtneps2bf16(xmm1, k0, ymmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F27E4872CA                , vcvtneps2bf16(ymm1, k0, zmm2)),
        INSTRUCTION_TEST(62F27E48724C1A02            , vcvtneps2bf16(ymm1, k0, zmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(6261FF48E68200200000        , vcvtpd2dq(ymm24, k0, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62F1FF48E6F2                , vcvtpd2dq(ymm6, k0, zmm2)),
        INSTRUCTION_TEST(62B1FF48E6B4F034120000      , vcvtpd2dq(ymm6, k0, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62F1FF48E631                , vcvtpd2dq(ymm6, k0, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62F1FF48E6B200200000        , vcvtpd2dq(ymm6, k0, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62F1FF48E6B2C0DFFFFF        , vcvtpd2dq(ymm6, k0, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62F1FF48E6727F              , vcvtpd2dq(ymm6, k0, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(6261FF48E6427F              , vcvtpd2dq(ymm24, k0, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62F1FF48E67280              , vcvtpd2dq(ymm6, k0, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(6261FF48E682C0DFFFFF        , vcvtpd2dq(ymm24, k0, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62F1FF48E6CA                , vcvtpd2dq(ymm1, k0, zmm2)),
        INSTRUCTION_TEST(6261FF48E601                , vcvtpd2dq(ymm24, k0, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(6221FF48E684F023010000      , vcvtpd2dq(ymm24, k0, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(6241FF48E6C7                , vcvtpd2dq(ymm24, k0, zmm15)),
        INSTRUCTION_TEST(62F1FF48E64C1A02            , vcvtpd2dq(ymm1, k0, zmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(6261FF48E64280              , vcvtpd2dq(ymm24, k0, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(C5FFE68C1A80000000          , vcvtpd2dq(xmm1, ymmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(C5FFE6CA                    , vcvtpd2dq(xmm1, ymm2)),
        INSTRUCTION_TEST(C5FBE68C1A80000000          , vcvtpd2dq(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(C5FBE6CA                    , vcvtpd2dq(xmm1, xmm2)),
        INSTRUCTION_TEST(62E1FD485A8200200000        , vcvtpd2ps(ymm16, k0, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62B1FD485AEF                , vcvtpd2ps(ymm5, k0, zmm23)),
        INSTRUCTION_TEST(62B1FD485AACF023010000      , vcvtpd2ps(ymm5, k0, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62F1FD485A29                , vcvtpd2ps(ymm5, k0, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62F1FD485AAA00200000        , vcvtpd2ps(ymm5, k0, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62F1FD485AAAC0DFFFFF        , vcvtpd2ps(ymm5, k0, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62F1FD485A6A7F              , vcvtpd2ps(ymm5, k0, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62E1FD485A427F              , vcvtpd2ps(ymm16, k0, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62F1FD485A6A80              , vcvtpd2ps(ymm5, k0, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62E1FD485A82C0DFFFFF        , vcvtpd2ps(ymm16, k0, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62C1FD485AC2                , vcvtpd2ps(ymm16, k0, zmm10)),
        INSTRUCTION_TEST(62E1FD485A01                , vcvtpd2ps(ymm16, k0, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62A1FD485A84F034120000      , vcvtpd2ps(ymm16, k0, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62F1FD485A4C1A02            , vcvtpd2ps(ymm1, k0, zmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F1FD485ACA                , vcvtpd2ps(ymm1, k0, zmm2)),
        INSTRUCTION_TEST(C5FD5A8C1A80000000          , vcvtpd2ps(xmm1, ymmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(C5FD5ACA                    , vcvtpd2ps(xmm1, ymm2)),
        INSTRUCTION_TEST(C5F95A8C1A80000000          , vcvtpd2ps(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(C5F95ACA                    , vcvtpd2ps(xmm1, xmm2)),
        INSTRUCTION_TEST(62E1FD485A4280              , vcvtpd2ps(ymm16, k0, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62F1FD487BCA                , vcvtpd2qq(zmm1, k0, zmm2)),
        INSTRUCTION_TEST(62F1FD287B4C1A04            , vcvtpd2qq(ymm1, k0, ymmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F1FD487B4C1A02            , vcvtpd2qq(zmm1, k0, zmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F1FD087B4C1A08            , vcvtpd2qq(xmm1, k0, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F1FD087BCA                , vcvtpd2qq(xmm1, k0, xmm2)),
        INSTRUCTION_TEST(62F1FD287BCA                , vcvtpd2qq(ymm1, k0, ymm2)),
        INSTRUCTION_TEST(6261FC48798A00200000        , vcvtpd2udq(ymm25, k0, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(6261FC48794A7F              , vcvtpd2udq(ymm25, k0, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(6261FC48798AC0DFFFFF        , vcvtpd2udq(ymm25, k0, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(6261FC48794A80              , vcvtpd2udq(ymm25, k0, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(6221FC48798CF034120000      , vcvtpd2udq(ymm25, k0, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(6201FC4879C9                , vcvtpd2udq(ymm25, k0, zmm25)),
        INSTRUCTION_TEST(6271FC4879BA00200000        , vcvtpd2udq(ymm15, k0,zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(6271FC48797A7F              , vcvtpd2udq(ymm15, k0,zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(6271FC4879BAC0DFFFFF        , vcvtpd2udq(ymm15, k0,zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(6261FC487909                , vcvtpd2udq(ymm25, k0, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(6271FC487939                , vcvtpd2udq(ymm15, k0, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(6231FC4879BCF023010000      , vcvtpd2udq(ymm15, k0, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(6231FC4879FB                , vcvtpd2udq(ymm15, k0, zmm19)),
        INSTRUCTION_TEST(62F1FC48794C1A02            , vcvtpd2udq(ymm1, k0, zmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F1FC4879CA                , vcvtpd2udq(ymm1, k0, zmm2)),
        INSTRUCTION_TEST(62F1FC28794C1A04            , vcvtpd2udq(xmm1, k0, ymmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F1FC2879CA                , vcvtpd2udq(xmm1, k0, ymm2)),
        INSTRUCTION_TEST(62F1FC08794C1A08            , vcvtpd2udq(xmm1, k0, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F1FC0879CA                , vcvtpd2udq(xmm1, k0, xmm2)),
        INSTRUCTION_TEST(6271FC48797A80              , vcvtpd2udq(ymm15, k0, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62F1FD48794C1A02            , vcvtpd2uqq(zmm1, k0, zmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F1FD4879CA                , vcvtpd2uqq(zmm1, k0, zmm2)),
        INSTRUCTION_TEST(62F1FD28794C1A04            , vcvtpd2uqq(ymm1, k0, ymmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F1FD2879CA                , vcvtpd2uqq(ymm1, k0, ymm2)),
        INSTRUCTION_TEST(62F1FD0879CA                , vcvtpd2uqq(xmm1, k0, xmm2)),
        INSTRUCTION_TEST(62F1FD08794C1A08            , vcvtpd2uqq(xmm1, k0, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62727D48136A7F              , vcvtph2ps(zmm13, k0, ymmword_ptr(rdx, 4064))),
        INSTRUCTION_TEST(62727D48134A7F              , vcvtph2ps(zmm9, k0, ymmword_ptr(rdx, 4064))),
        INSTRUCTION_TEST(62727D48138AE0EFFFFF        , vcvtph2ps(zmm9, k0, ymmword_ptr(rdx, -4128))),
        INSTRUCTION_TEST(62727D48134A80              , vcvtph2ps(zmm9, k0, ymmword_ptr(rdx, -4096))),
        INSTRUCTION_TEST(62727D481309                , vcvtph2ps(zmm9, k0, ymmword_ptr(rcx))),
        INSTRUCTION_TEST(62327D48138CF034120000      , vcvtph2ps(zmm9, k0, ymmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62327D4813CD                , vcvtph2ps(zmm9, k0, ymm21)),
        INSTRUCTION_TEST(62727D4813AA00100000        , vcvtph2ps(zmm13, k0, ymmword_ptr(rdx, 4096))),
        INSTRUCTION_TEST(62727D4813AAE0EFFFFF        , vcvtph2ps(zmm13, k0, ymmword_ptr(rdx, -4128))),
        INSTRUCTION_TEST(62727D48138A00100000        , vcvtph2ps(zmm9, k0, ymmword_ptr(rdx, 4096))),
        INSTRUCTION_TEST(62727D481329                , vcvtph2ps(zmm13, k0, ymmword_ptr(rcx))),
        INSTRUCTION_TEST(C4E279138C1A80000000        , vcvtph2ps(xmm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(C4E27913CA                  , vcvtph2ps(xmm1, xmm2)),
        INSTRUCTION_TEST(C4E27D13CA                  , vcvtph2ps(ymm1, xmm2)),
        INSTRUCTION_TEST(C4E27D138C1A80000000        , vcvtph2ps(ymm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62727D48136A80              , vcvtph2ps(zmm13, k0, ymmword_ptr(rdx, -4096))),
        INSTRUCTION_TEST(62F27D48134C1A04            , vcvtph2ps(zmm1, k0, ymmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62127D4813EB                , vcvtph2ps(zmm13, k0, ymm27)),
        INSTRUCTION_TEST(62327D4813ACF023010000      , vcvtph2ps(zmm13, k0, ymmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62F27D4813CA                , vcvtph2ps(zmm1, k0, ymm2)),
        INSTRUCTION_TEST(62717D485B627F              , vcvtps2dq(zmm12, k0, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62E17D485B8200200000        , vcvtps2dq(zmm16, k0, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62E17D485B427F              , vcvtps2dq(zmm16, k0, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62E17D485B82C0DFFFFF        , vcvtps2dq(zmm16, k0, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62E17D485B4280              , vcvtps2dq(zmm16, k0, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62E17D485B01                , vcvtps2dq(zmm16, k0, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62A17D485B84F034120000      , vcvtps2dq(zmm16, k0, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62E17D485BC6                , vcvtps2dq(zmm16, k0, zmm6)),
        INSTRUCTION_TEST(62717D485BA200200000        , vcvtps2dq(zmm12, k0, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62717D485BA2C0DFFFFF        , vcvtps2dq(zmm12, k0, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62717D485B6280              , vcvtps2dq(zmm12, k0, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62717D485B21                , vcvtps2dq(zmm12, k0, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(C5F95B8C1A80000000          , vcvtps2dq(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(C5FD5BCA                    , vcvtps2dq(ymm1, ymm2)),
        INSTRUCTION_TEST(C5FD5B8C1A80000000          , vcvtps2dq(ymm1, ymmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(C5F95BCA                    , vcvtps2dq(xmm1, xmm2)),
        INSTRUCTION_TEST(62F17D485B4C1A02            , vcvtps2dq(zmm1, k0, zmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62117D485BE0                , vcvtps2dq(zmm12, k0, zmm24)),
        INSTRUCTION_TEST(62317D485BA4F023010000      , vcvtps2dq(zmm12, k0, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62F17D485BCA                , vcvtps2dq(zmm1, k0, zmm2)),
        INSTRUCTION_TEST(62717C485A5A7F              , vcvtps2pd(zmm11, k0, ymmword_ptr(rdx, 4064))),
        INSTRUCTION_TEST(62717C485A6A7F              , vcvtps2pd(zmm13, k0, ymmword_ptr(rdx, 4064))),
        INSTRUCTION_TEST(62717C485AAAE0EFFFFF        , vcvtps2pd(zmm13, k0, ymmword_ptr(rdx, -4128))),
        INSTRUCTION_TEST(62717C485A6A80              , vcvtps2pd(zmm13, k0, ymmword_ptr(rdx, -4096))),
        INSTRUCTION_TEST(62717C485A29                , vcvtps2pd(zmm13, k0, ymmword_ptr(rcx))),
        INSTRUCTION_TEST(62717C485AAA00100000        , vcvtps2pd(zmm13, k0, ymmword_ptr(rdx, 4096))),
        INSTRUCTION_TEST(62317C485AACF023010000      , vcvtps2pd(zmm13, k0, ymmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62717C485A9A00100000        , vcvtps2pd(zmm11, k0, ymmword_ptr(rdx, 4096))),
        INSTRUCTION_TEST(62717C485A9AE0EFFFFF        , vcvtps2pd(zmm11, k0, ymmword_ptr(rdx, -4128))),
        INSTRUCTION_TEST(62717C485AEE                , vcvtps2pd(zmm13, k0, ymm6)),
        INSTRUCTION_TEST(62717C485A19                , vcvtps2pd(zmm11, k0, ymmword_ptr(rcx))),
        INSTRUCTION_TEST(62317C485A9CF034120000      , vcvtps2pd(zmm11, k0, ymmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62117C485AD8                , vcvtps2pd(zmm11, k0, ymm24)),
        INSTRUCTION_TEST(62F17C485A4C1A04            , vcvtps2pd(zmm1, k0, ymmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F17C485ACA                , vcvtps2pd(zmm1, k0, ymm2)),
        INSTRUCTION_TEST(C5FC5A8C1A80000000          , vcvtps2pd(ymm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(C5FC5ACA                    , vcvtps2pd(ymm1, xmm2)),
        INSTRUCTION_TEST(C5F85ACA                    , vcvtps2pd(xmm1, xmm2)),
        INSTRUCTION_TEST(C5F85A8C1A80000000          , vcvtps2pd(xmm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62717C485A5A80              , vcvtps2pd(zmm11, k0, ymmword_ptr(rdx, -4096))),
        INSTRUCTION_TEST(62F37D481D31AB              , vcvtps2ph(ymmword_ptr(rcx), k0, zmm6, Imm(171))),
        INSTRUCTION_TEST(62E37D481D5A807B            , vcvtps2ph(ymmword_ptr(rdx, -4096),k0,  zmm19, Imm(123))),
        INSTRUCTION_TEST(62F37D481D72807B            , vcvtps2ph(ymmword_ptr(rdx, -4096),k0,  zmm6, Imm(123))),
        INSTRUCTION_TEST(62E37D481D9AE0EFFFFF7B      , vcvtps2ph(ymmword_ptr(rdx, -4128),k0,  zmm19, Imm(123))),
        INSTRUCTION_TEST(62E37D481D9A001000007B      , vcvtps2ph(ymmword_ptr(rdx, 4096),k0,  zmm19, Imm(123))),
        INSTRUCTION_TEST(62E37D481D5A7F7B            , vcvtps2ph(ymmword_ptr(rdx, 4064),k0,  zmm19, Imm(123))),
        INSTRUCTION_TEST(62F37D481D727F7B            , vcvtps2ph(ymmword_ptr(rdx, 4064),k0,  zmm6, Imm(123))),
        INSTRUCTION_TEST(62E37D481D19AB              , vcvtps2ph(ymmword_ptr(rcx), k0, zmm19, Imm(171))),
        INSTRUCTION_TEST(62F37D481DB2001000007B      , vcvtps2ph(ymmword_ptr(rdx, 4096), k0, zmm6, Imm(123))),
        INSTRUCTION_TEST(62F37D481DB2E0EFFFFF7B      , vcvtps2ph(ymmword_ptr(rdx, -4128), k0, zmm6, Imm(123))),
        INSTRUCTION_TEST(62E37D481D197B              , vcvtps2ph(ymmword_ptr(rcx), k0, zmm19, Imm(123))),
        INSTRUCTION_TEST(62F37D481D317B              , vcvtps2ph(ymmword_ptr(rcx), k0, zmm6, Imm(123))),
        INSTRUCTION_TEST(62B37D481DB4F0341200007B    , vcvtps2ph(ymmword_ptr(rax, r14, 8, 4660), k0, zmm6, Imm(123))),
        INSTRUCTION_TEST(62A37D481D9CF0230100007B    , vcvtps2ph(ymmword_ptr(rax, r14, 8, 291), k0, zmm19, Imm(123))),
        INSTRUCTION_TEST(62C37D481DF1AB              , vcvtps2ph(ymm9,k0,  zmm22, Imm(171))),
        INSTRUCTION_TEST(62C37D481DF17B              , vcvtps2ph(ymm9,k0,  zmm22, Imm(123))),
        INSTRUCTION_TEST(62537D481DF3AB              , vcvtps2ph(ymm11, k0, zmm14, Imm(171))),
        INSTRUCTION_TEST(62537D481DF37B              , vcvtps2ph(ymm11, k0, zmm14, Imm(123))),
        INSTRUCTION_TEST(62F37D481DD101              , vcvtps2ph(ymm1, k0, zmm2, Imm(1))),
        INSTRUCTION_TEST(C4E37D1D9C118000000001      , vcvtps2ph(xmmword_ptr(rcx, rdx, 1, 128), ymm3, Imm(1))),
        INSTRUCTION_TEST(C4E37D1DD101                , vcvtps2ph(xmm1, ymm2, Imm(1))),
        INSTRUCTION_TEST(C4E3791DD101                , vcvtps2ph(xmm1, xmm2, Imm(1))),
        INSTRUCTION_TEST(C4E3791D9C118000000001      , vcvtps2ph(qword_ptr(rcx, rdx, 1, 128), xmm3, Imm(1))),
        INSTRUCTION_TEST(62F37D481D5C110401          , vcvtps2ph(ymmword_ptr(rcx, rdx, 1, 128), k0, zmm3, Imm(1))),
        INSTRUCTION_TEST(62F17D487B4C1A04            , vcvtps2qq(zmm1,k0,  ymmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F17D487BCA                , vcvtps2qq(zmm1,k0,  ymm2)),
        INSTRUCTION_TEST(62F17D287B4C1A08            , vcvtps2qq(ymm1,k0,  xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F17D087BCA                , vcvtps2qq(xmm1,k0,  xmm2)),
        INSTRUCTION_TEST(62F17D087B4C1A10            , vcvtps2qq(xmm1,k0,  qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F17D287BCA                , vcvtps2qq(ymm1,k0,  xmm2)),
        INSTRUCTION_TEST(62717C4879627F              , vcvtps2udq(zmm12, k0, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62F17C4879AA00200000        , vcvtps2udq(zmm5, k0, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62F17C48796A7F              , vcvtps2udq(zmm5, k0, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62F17C4879AAC0DFFFFF        , vcvtps2udq(zmm5, k0, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62F17C48796A80              , vcvtps2udq(zmm5, k0, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62F17C487929                , vcvtps2udq(zmm5, k0, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62B17C4879ACF034120000      , vcvtps2udq(zmm5, k0, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62F17C4879EC                , vcvtps2udq(zmm5, k0, zmm4)),
        INSTRUCTION_TEST(62717C4879A200200000        , vcvtps2udq(zmm12, k0, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62717C4879A2C0DFFFFF        , vcvtps2udq(zmm12, k0, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62F17C2879CA                , vcvtps2udq(ymm1, k0, ymm2)),
        INSTRUCTION_TEST(62717C487921                , vcvtps2udq(zmm12, k0, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62F17C08794C1A08            , vcvtps2udq(xmm1, k0, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62717C48796280              , vcvtps2udq(zmm12, k0, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62F17C28794C1A04            , vcvtps2udq(ymm1, k0, ymmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F17C0879CA                , vcvtps2udq(xmm1, k0, xmm2)),
        INSTRUCTION_TEST(62F17C48794C1A02            , vcvtps2udq(zmm1, k0, zmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62117C4879E2                , vcvtps2udq(zmm12, k0, zmm26)),
        INSTRUCTION_TEST(62317C4879A4F023010000      , vcvtps2udq(zmm12, k0, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62F17C4879CA                , vcvtps2udq(zmm1, k0, zmm2)),
        INSTRUCTION_TEST(62F17D08794C1A10            , vcvtps2uqq(xmm1, k0, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F17D0879CA                , vcvtps2uqq(xmm1, k0, xmm2)),
        INSTRUCTION_TEST(62F17D2879CA                , vcvtps2uqq(ymm1, k0, xmm2)),
        INSTRUCTION_TEST(62F17D28794C1A08            , vcvtps2uqq(ymm1, k0, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F17D4879CA                , vcvtps2uqq(zmm1, k0, ymm2)),
        INSTRUCTION_TEST(62F17D48794C1A04            , vcvtps2uqq(zmm1, k0, ymmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F1FE48E6CA                , vcvtqq2pd(zmm1, k0, zmm2)),
        INSTRUCTION_TEST(62F1FE48E64C1A02            , vcvtqq2pd(zmm1, k0, zmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F1FE28E64C1A04            , vcvtqq2pd(ymm1, k0, ymmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F1FE28E6CA                , vcvtqq2pd(ymm1, k0, ymm2)),
        INSTRUCTION_TEST(62F1FE08E64C1A08            , vcvtqq2pd(xmm1, k0, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F1FE08E6CA                , vcvtqq2pd(xmm1, k0, xmm2)),
        INSTRUCTION_TEST(62F1FC085BCA                , vcvtqq2ps(xmm1, k0, xmm2)),
        INSTRUCTION_TEST(62F1FC085B4C1A08            , vcvtqq2ps(xmm1, k0, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F1FC285BCA                , vcvtqq2ps(xmm1, k0, ymm2)),
        INSTRUCTION_TEST(62F1FC285B4C1A04            , vcvtqq2ps(xmm1, k0, ymmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F1FC485BCA                , vcvtqq2ps(ymm1, k0, zmm2)),
        INSTRUCTION_TEST(62F1FC485B4C1A02            , vcvtqq2ps(ymm1, k0, zmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(C5FB2D8C1A80000000          , vcvtsd2si(ecx, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(C5FB2DCA                    , vcvtsd2si(ecx, xmm2)),
        INSTRUCTION_TEST(C4E1FB2D8C1A80000000        , vcvtsd2si(rcx, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(C4E1FB2DCA                  , vcvtsd2si(rcx, xmm2)),
        INSTRUCTION_TEST(62C1B7085ACC                , vcvtsd2ss(xmm17, k0, xmm9, xmm12)),
        INSTRUCTION_TEST(62E1B7085A8A00040000        , vcvtsd2ss(xmm17, k0, xmm9, qword_ptr(rdx, 1024))),
        INSTRUCTION_TEST(62E1B7085A4A7F              , vcvtsd2ss(xmm17, k0, xmm9, qword_ptr(rdx, 1016))),
        INSTRUCTION_TEST(62E1B7085A8AF8FBFFFF        , vcvtsd2ss(xmm17, k0, xmm9, qword_ptr(rdx, -1032))),
        INSTRUCTION_TEST(C5EB5A8C2B80000000          , vcvtsd2ss(xmm1, xmm2, qword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62E1B7085A09                , vcvtsd2ss(xmm17, k0, xmm9, qword_ptr(rcx))),
        INSTRUCTION_TEST(62A1B7085A8CF023010000      , vcvtsd2ss(xmm17, k0, xmm9, qword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(C5EB5ACB                    , vcvtsd2ss(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(62E1B7085A4A80              , vcvtsd2ss(xmm17, k0, xmm9, qword_ptr(rdx, -1024))),
        INSTRUCTION_TEST(62717F0879AA00040000        , vcvtsd2usi(r13d, qword_ptr(rdx, 1024))),
        INSTRUCTION_TEST(62517F0879EB                , vcvtsd2usi(r13d, xmm11)),
        INSTRUCTION_TEST(62117F0879EE                , vcvtsd2usi(r13d, xmm30)),
        INSTRUCTION_TEST(6231FF087984F023010000      , vcvtsd2usi(r8, qword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(6231FF087984F034120000      , vcvtsd2usi(r8, qword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(6271FF087901                , vcvtsd2usi(r8, qword_ptr(rcx))),
        INSTRUCTION_TEST(62F1FF0879CA                , vcvtsd2usi(rcx, xmm2)),
        INSTRUCTION_TEST(6271FF087982F8FBFFFF        , vcvtsd2usi(r8, qword_ptr(rdx, -1032))),
        INSTRUCTION_TEST(6271FF0879427F              , vcvtsd2usi(r8, qword_ptr(rdx, 1016))),
        INSTRUCTION_TEST(6271FF08798200040000        , vcvtsd2usi(r8, qword_ptr(rdx, 1024))),
        INSTRUCTION_TEST(6251FF0879C5                , vcvtsd2usi(r8, xmm13)),
        INSTRUCTION_TEST(62B1FF087984F023010000      , vcvtsd2usi(rax, qword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62B1FF087984F034120000      , vcvtsd2usi(rax, qword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62F1FF087901                , vcvtsd2usi(rax, qword_ptr(rcx))),
        INSTRUCTION_TEST(62F1FF08794280              , vcvtsd2usi(rax, qword_ptr(rdx, -1024))),
        INSTRUCTION_TEST(62F1FF087982F8FBFFFF        , vcvtsd2usi(rax, qword_ptr(rdx, -1032))),
        INSTRUCTION_TEST(62F1FF0879427F              , vcvtsd2usi(rax, qword_ptr(rdx, 1016))),
        INSTRUCTION_TEST(62F1FF08798200040000        , vcvtsd2usi(rax, qword_ptr(rdx, 1024))),
        INSTRUCTION_TEST(62D1FF0879C5                , vcvtsd2usi(rax, xmm13)),
        INSTRUCTION_TEST(62B1FF0879C2                , vcvtsd2usi(rax, xmm18)),
        INSTRUCTION_TEST(62F1FF08794C1A10            , vcvtsd2usi(rcx, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62717F08796A7F              , vcvtsd2usi(r13d, qword_ptr(rdx, 1016))),
        INSTRUCTION_TEST(6231FF0879C2                , vcvtsd2usi(r8, xmm18)),
        INSTRUCTION_TEST(62717F0879AAF8FBFFFF        , vcvtsd2usi(r13d, qword_ptr(rdx, -1032))),
        INSTRUCTION_TEST(6271FF08794280              , vcvtsd2usi(r8, qword_ptr(rdx, -1024))),
        INSTRUCTION_TEST(62717F087929                , vcvtsd2usi(r13d, qword_ptr(rcx))),
        INSTRUCTION_TEST(62717F08796A80              , vcvtsd2usi(r13d, qword_ptr(rdx, -1024))),
        INSTRUCTION_TEST(62B17F087984F023010000      , vcvtsd2usi(eax, qword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62F17F087901                , vcvtsd2usi(eax, qword_ptr(rcx))),
        INSTRUCTION_TEST(62F17F08794280              , vcvtsd2usi(eax, qword_ptr(rdx, -1024))),
        INSTRUCTION_TEST(62F17F087982F8FBFFFF        , vcvtsd2usi(eax, qword_ptr(rdx, -1032))),
        INSTRUCTION_TEST(62F17F0879427F              , vcvtsd2usi(eax, qword_ptr(rdx, 1016))),
        INSTRUCTION_TEST(62F17F08798200040000        , vcvtsd2usi(eax, qword_ptr(rdx, 1024))),
        INSTRUCTION_TEST(62D17F0879C3                , vcvtsd2usi(eax, xmm11)),
        INSTRUCTION_TEST(62917F0879C6                , vcvtsd2usi(eax, xmm30)),
        INSTRUCTION_TEST(62B17F0879ACF023010000      , vcvtsd2usi(ebp, qword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62B17F0879ACF034120000      , vcvtsd2usi(ebp, qword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62B17F087984F034120000      , vcvtsd2usi(eax, qword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62F17F08796A80              , vcvtsd2usi(ebp, qword_ptr(rdx, -1024))),
        INSTRUCTION_TEST(62F17F087929                , vcvtsd2usi(ebp, qword_ptr(rcx))),
        INSTRUCTION_TEST(62317F0879ACF034120000      , vcvtsd2usi(r13d, qword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62F17F0879CA                , vcvtsd2usi(ecx, xmm2)),
        INSTRUCTION_TEST(62F17F08794C1A10            , vcvtsd2usi(ecx, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62317F0879ACF023010000      , vcvtsd2usi(r13d, qword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62D17F0879EB                , vcvtsd2usi(ebp, xmm11)),
        INSTRUCTION_TEST(62F17F0879AA00040000        , vcvtsd2usi(ebp, qword_ptr(rdx, 1024))),
        INSTRUCTION_TEST(62F17F08796A7F              , vcvtsd2usi(ebp, qword_ptr(rdx, 1016))),
        INSTRUCTION_TEST(62F17F0879AAF8FBFFFF        , vcvtsd2usi(ebp, qword_ptr(rdx, -1032))),
        INSTRUCTION_TEST(62917F0879EE                , vcvtsd2usi(ebp, xmm30)),
        INSTRUCTION_TEST(C5B32A21                    , vcvtsi2sd(xmm4, xmm9, dword_ptr(rcx))),
        INSTRUCTION_TEST(C5B32AE5                    , vcvtsi2sd(xmm4, xmm9, ebp)),
        INSTRUCTION_TEST(C5B32AA200FEFFFF            , vcvtsi2sd(xmm4, xmm9, dword_ptr(rdx, -512))),
        INSTRUCTION_TEST(C5B32AA2FCFDFFFF            , vcvtsi2sd(xmm4, xmm9, dword_ptr(rdx, -516))),
        INSTRUCTION_TEST(C5B32AA2FC010000            , vcvtsi2sd(xmm4, xmm9, dword_ptr(rdx, 508))),
        INSTRUCTION_TEST(C5B32AA200020000            , vcvtsi2sd(xmm4, xmm9, dword_ptr(rdx, 512))),
        INSTRUCTION_TEST(C5B32AE0                    , vcvtsi2sd(xmm4, xmm9, eax)),
        INSTRUCTION_TEST(C4C1332AE5                  , vcvtsi2sd(xmm4, xmm9, r13d)),
        INSTRUCTION_TEST(C5AB2ABAFC010000            , vcvtsi2sd(xmm7, xmm10, dword_ptr(rdx, 508))),
        INSTRUCTION_TEST(C5AB2A39                    , vcvtsi2sd(xmm7, xmm10, dword_ptr(rcx))),
        INSTRUCTION_TEST(C5AB2ABA00FEFFFF            , vcvtsi2sd(xmm7, xmm10, dword_ptr(rdx, -512))),
        INSTRUCTION_TEST(C5AB2ABAFCFDFFFF            , vcvtsi2sd(xmm7, xmm10, dword_ptr(rdx, -516))),
        INSTRUCTION_TEST(C5AB2ABA00020000            , vcvtsi2sd(xmm7, xmm10, dword_ptr(rdx, 512))),
        INSTRUCTION_TEST(C5AB2AF8                    , vcvtsi2sd(xmm7, xmm10, eax)),
        INSTRUCTION_TEST(C5AB2AFD                    , vcvtsi2sd(xmm7, xmm10, ebp)),
        INSTRUCTION_TEST(C4C12B2AFD                  , vcvtsi2sd(xmm7, xmm10, r13d)),
        INSTRUCTION_TEST(C4A1332AA4F034120000        , vcvtsi2sd(xmm4, xmm9, dword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(C4A12B2ABCF023010000        , vcvtsi2sd(xmm7, xmm10, dword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62619F082AE8                , vcvtsi2sd(xmm29, xmm12, rax)),
        INSTRUCTION_TEST(62619F082AAAF8FBFFFF        , vcvtsi2sd(xmm29, xmm12, qword_ptr(rdx, -1032))),
        INSTRUCTION_TEST(62619F082AAA00040000        , vcvtsi2sd(xmm29, xmm12, qword_ptr(rdx, 1024))),
        INSTRUCTION_TEST(62419F082AE8                , vcvtsi2sd(xmm29, xmm12, r8)),
        INSTRUCTION_TEST(C5EB2A8C2B80000000          , vcvtsi2sd(xmm1, xmm2, dword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C5EB2ACB                    , vcvtsi2sd(xmm1, xmm2, ebx)),
        INSTRUCTION_TEST(C4E1EB2A8C2B80000000        , vcvtsi2sd(xmm1, xmm2, qword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E1EB2ACB                  , vcvtsi2sd(xmm1, xmm2, rbx)),
        INSTRUCTION_TEST(6261F7002A01                , vcvtsi2sd(xmm24, xmm17, qword_ptr(rcx))),
        INSTRUCTION_TEST(6261F7002A4280              , vcvtsi2sd(xmm24, xmm17, qword_ptr(rdx, -1024))),
        INSTRUCTION_TEST(6261F7002A82F8FBFFFF        , vcvtsi2sd(xmm24, xmm17, qword_ptr(rdx, -1032))),
        INSTRUCTION_TEST(6221F7002A84F034120000      , vcvtsi2sd(xmm24, xmm17, qword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(6261F7002A8200040000        , vcvtsi2sd(xmm24, xmm17, qword_ptr(rdx, 1024))),
        INSTRUCTION_TEST(6241F7002AC0                , vcvtsi2sd(xmm24, xmm17, r8)),
        INSTRUCTION_TEST(6261F7002AC0                , vcvtsi2sd(xmm24, xmm17, rax)),
        INSTRUCTION_TEST(62219F082AACF023010000      , vcvtsi2sd(xmm29, xmm12, qword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62619F082A29                , vcvtsi2sd(xmm29, xmm12, qword_ptr(rcx))),
        INSTRUCTION_TEST(62619F082A6A80              , vcvtsi2sd(xmm29, xmm12, qword_ptr(rdx, -1024))),
        INSTRUCTION_TEST(62619F082A6A7F              , vcvtsi2sd(xmm29, xmm12, qword_ptr(rdx, 1016))),
        INSTRUCTION_TEST(6261F7002A427F              , vcvtsi2sd(xmm24, xmm17, qword_ptr(rdx, 1016))),
        INSTRUCTION_TEST(62611E002ACD                , vcvtsi2ss(xmm25, xmm28, ebp)),
        INSTRUCTION_TEST(62611E002A09                , vcvtsi2ss(xmm25, xmm28, dword_ptr(rcx))),
        INSTRUCTION_TEST(62611E002A4A80              , vcvtsi2ss(xmm25, xmm28, dword_ptr(rdx, -512))),
        INSTRUCTION_TEST(62611E002A8AFCFDFFFF        , vcvtsi2ss(xmm25, xmm28, dword_ptr(rdx, -516))),
        INSTRUCTION_TEST(62611E002A4A7F              , vcvtsi2ss(xmm25, xmm28, dword_ptr(rdx, 508))),
        INSTRUCTION_TEST(62611E002A8A00020000        , vcvtsi2ss(xmm25, xmm28, dword_ptr(rdx, 512))),
        INSTRUCTION_TEST(62611E002AC8                , vcvtsi2ss(xmm25, xmm28, eax)),
        INSTRUCTION_TEST(62411E002ACD                , vcvtsi2ss(xmm25, xmm28, r13d)),
        INSTRUCTION_TEST(62E1AE082AC0                , vcvtsi2ss(xmm16, xmm10, rax)),
        INSTRUCTION_TEST(C4E1BA2A21                  , vcvtsi2ss(xmm4, xmm8, qword_ptr(rcx))),
        INSTRUCTION_TEST(C4E1BA2AA200FCFFFF          , vcvtsi2ss(xmm4, xmm8, qword_ptr(rdx, -1024))),
        INSTRUCTION_TEST(C4E1BA2AA2F8FBFFFF          , vcvtsi2ss(xmm4, xmm8, qword_ptr(rdx, -1032))),
        INSTRUCTION_TEST(C4E1BA2AA2F8030000          , vcvtsi2ss(xmm4, xmm8, qword_ptr(rdx, 1016))),
        INSTRUCTION_TEST(C4E1BA2AA200040000          , vcvtsi2ss(xmm4, xmm8, qword_ptr(rdx, 1024))),
        INSTRUCTION_TEST(C4C1BA2AE0                  , vcvtsi2ss(xmm4, xmm8, r8)),
        INSTRUCTION_TEST(C4E1BA2AE0                  , vcvtsi2ss(xmm4, xmm8, rax)),
        INSTRUCTION_TEST(C4A1BA2AA4F034120000        , vcvtsi2ss(xmm4, xmm8, qword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62C1AE082AC0                , vcvtsi2ss(xmm16, xmm10, r8)),
        INSTRUCTION_TEST(62211E002A8CF034120000      , vcvtsi2ss(xmm25, xmm28, dword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62E1AE082A427F              , vcvtsi2ss(xmm16, xmm10, qword_ptr(rdx, 1016))),
        INSTRUCTION_TEST(C5EA2A8C2B80000000          , vcvtsi2ss(xmm1, xmm2, dword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C5EA2ACB                    , vcvtsi2ss(xmm1, xmm2, ebx)),
        INSTRUCTION_TEST(C4E1EA2A8C2B80000000        , vcvtsi2ss(xmm1, xmm2, qword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E1EA2ACB                  , vcvtsi2ss(xmm1, xmm2, rbx)),
        INSTRUCTION_TEST(C4212A2ABCF023010000        , vcvtsi2ss(xmm15, xmm10, dword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(C52A2A39                    , vcvtsi2ss(xmm15, xmm10, dword_ptr(rcx))),
        INSTRUCTION_TEST(C52A2ABA00FEFFFF            , vcvtsi2ss(xmm15, xmm10, dword_ptr(rdx, -512))),
        INSTRUCTION_TEST(C52A2ABAFC010000            , vcvtsi2ss(xmm15, xmm10, dword_ptr(rdx, 508))),
        INSTRUCTION_TEST(C52A2ABAFCFDFFFF            , vcvtsi2ss(xmm15, xmm10, dword_ptr(rdx, -516))),
        INSTRUCTION_TEST(C52A2AF8                    , vcvtsi2ss(xmm15, xmm10, eax)),
        INSTRUCTION_TEST(C52A2AFD                    , vcvtsi2ss(xmm15, xmm10, ebp)),
        INSTRUCTION_TEST(C4412A2AFD                  , vcvtsi2ss(xmm15, xmm10, r13d)),
        INSTRUCTION_TEST(62A1AE082A84F023010000      , vcvtsi2ss(xmm16, xmm10, qword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62E1AE082A01                , vcvtsi2ss(xmm16, xmm10, qword_ptr(rcx))),
        INSTRUCTION_TEST(62E1AE082A4280              , vcvtsi2ss(xmm16, xmm10, qword_ptr(rdx, -1024))),
        INSTRUCTION_TEST(62E1AE082A8200040000        , vcvtsi2ss(xmm16, xmm10, qword_ptr(rdx, 1024))),
        INSTRUCTION_TEST(62E1AE082A82F8FBFFFF        , vcvtsi2ss(xmm16, xmm10, qword_ptr(rdx, -1032))),
        INSTRUCTION_TEST(C52A2ABA00020000            , vcvtsi2ss(xmm15, xmm10, dword_ptr(rdx, 512))),
        INSTRUCTION_TEST(62614E085AA2FCFDFFFF        , vcvtss2sd(xmm28, k0, xmm6, dword_ptr(rdx, -516))),
        INSTRUCTION_TEST(62614E085AA200020000        , vcvtss2sd(xmm28, k0, xmm6, dword_ptr(rdx, 512))),
        INSTRUCTION_TEST(62614E085A627F              , vcvtss2sd(xmm28, k0, xmm6, dword_ptr(rdx, 508))),
        INSTRUCTION_TEST(62614E085A6280              , vcvtss2sd(xmm28, k0, xmm6, dword_ptr(rdx, -512))),
        INSTRUCTION_TEST(62614E085AE6                , vcvtss2sd(xmm28, k0, xmm6, xmm6)),
        INSTRUCTION_TEST(62214E085AA4F023010000      , vcvtss2sd(xmm28, k0, xmm6, dword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(C5EA5ACB                    , vcvtss2sd(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(C5EA5A8C2B80000000          , vcvtss2sd(xmm1, xmm2, dword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62614E085A21                , vcvtss2sd(xmm28, k0, xmm6, dword_ptr(rcx))),
        INSTRUCTION_TEST(C5FA2D8C1A80000000          , vcvtss2si(ecx, dword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(C5FA2DCA                    , vcvtss2si(ecx, xmm2)),
        INSTRUCTION_TEST(C4E1FA2D8C1A80000000        , vcvtss2si(rcx, dword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(C4E1FA2DCA                  , vcvtss2si(rcx, xmm2)),
        INSTRUCTION_TEST(6271FE0879427F              , vcvtss2usi(r8, dword_ptr(rdx, 508))),
        INSTRUCTION_TEST(6271FE087982FCFDFFFF        , vcvtss2usi(r8, dword_ptr(rdx, -516))),
        INSTRUCTION_TEST(6271FE08794280              , vcvtss2usi(r8, dword_ptr(rdx, -512))),
        INSTRUCTION_TEST(62117E0879EC                , vcvtss2usi(r13d, xmm28)),
        INSTRUCTION_TEST(6231FE087984F034120000      , vcvtss2usi(r8, dword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(6231FE087984F023010000      , vcvtss2usi(r8, dword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(6271FE08798200020000        , vcvtss2usi(r8, dword_ptr(rdx, 512))),
        INSTRUCTION_TEST(62717E0879E9                , vcvtss2usi(r13d, xmm1)),
        INSTRUCTION_TEST(6271FE087901                , vcvtss2usi(r8, dword_ptr(rcx))),
        INSTRUCTION_TEST(6231FE0879C7                , vcvtss2usi(r8, xmm23)),
        INSTRUCTION_TEST(62F1FE087901                , vcvtss2usi(rax, dword_ptr(rcx))),
        INSTRUCTION_TEST(62B1FE087984F023010000      , vcvtss2usi(rax, dword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62B1FE087984F034120000      , vcvtss2usi(rax, dword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62F1FE08794280              , vcvtss2usi(rax, dword_ptr(rdx, -512))),
        INSTRUCTION_TEST(62F1FE087982FCFDFFFF        , vcvtss2usi(rax, dword_ptr(rdx, -516))),
        INSTRUCTION_TEST(62F1FE0879427F              , vcvtss2usi(rax, dword_ptr(rdx, 508))),
        INSTRUCTION_TEST(62F1FE08798200020000        , vcvtss2usi(rax, dword_ptr(rdx, 512))),
        INSTRUCTION_TEST(62F1FE0879CA                , vcvtss2usi(rcx, xmm2)),
        INSTRUCTION_TEST(62F1FE0879C3                , vcvtss2usi(rax, xmm3)),
        INSTRUCTION_TEST(62F1FE08794C1A20            , vcvtss2usi(rcx, dword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62717E0879AA00020000        , vcvtss2usi(r13d, dword_ptr(rdx, 512))),
        INSTRUCTION_TEST(6271FE0879C3                , vcvtss2usi(r8, xmm3)),
        INSTRUCTION_TEST(62717E08796A7F              , vcvtss2usi(r13d, dword_ptr(rdx, 508))),
        INSTRUCTION_TEST(62B1FE0879C7                , vcvtss2usi(rax, xmm23)),
        INSTRUCTION_TEST(62717E087929                , vcvtss2usi(r13d, dword_ptr(rcx))),
        INSTRUCTION_TEST(62717E0879AAFCFDFFFF        , vcvtss2usi(r13d, dword_ptr(rdx, -516))),
        INSTRUCTION_TEST(62B17E087984F034120000      , vcvtss2usi(eax, dword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62F17E087901                , vcvtss2usi(eax, dword_ptr(rcx))),
        INSTRUCTION_TEST(62F17E08794280              , vcvtss2usi(eax, dword_ptr(rdx, -512))),
        INSTRUCTION_TEST(62F17E087982FCFDFFFF        , vcvtss2usi(eax, dword_ptr(rdx, -516))),
        INSTRUCTION_TEST(62F17E0879427F              , vcvtss2usi(eax, dword_ptr(rdx, 508))),
        INSTRUCTION_TEST(62F17E08798200020000        , vcvtss2usi(eax, dword_ptr(rdx, 512))),
        INSTRUCTION_TEST(62F17E0879C1                , vcvtss2usi(eax, xmm1)),
        INSTRUCTION_TEST(62917E0879C4                , vcvtss2usi(eax, xmm28)),
        INSTRUCTION_TEST(62B17E0879ACF023010000      , vcvtss2usi(ebp, dword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62B17E0879ACF034120000      , vcvtss2usi(ebp, dword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62B17E087984F023010000      , vcvtss2usi(eax, dword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62F17E08796A80              , vcvtss2usi(ebp, dword_ptr(rdx, -512))),
        INSTRUCTION_TEST(62F17E0879AAFCFDFFFF        , vcvtss2usi(ebp, dword_ptr(rdx, -516))),
        INSTRUCTION_TEST(62F17E08796A7F              , vcvtss2usi(ebp, dword_ptr(rdx, 508))),
        INSTRUCTION_TEST(62F17E0879AA00020000        , vcvtss2usi(ebp, dword_ptr(rdx, 512))),
        INSTRUCTION_TEST(62F17E0879E9                , vcvtss2usi(ebp, xmm1)),
        INSTRUCTION_TEST(62917E0879EC                , vcvtss2usi(ebp, xmm28)),
        INSTRUCTION_TEST(62F17E08794C1A20            , vcvtss2usi(ecx, dword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F17E0879CA                , vcvtss2usi(ecx, xmm2)),
        INSTRUCTION_TEST(62317E0879ACF023010000      , vcvtss2usi(r13d, dword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62317E0879ACF034120000      , vcvtss2usi(r13d, dword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62F17E087929                , vcvtss2usi(ebp, dword_ptr(rcx))),
        INSTRUCTION_TEST(6261FD48E69A00200000        , vcvttpd2dq(ymm27, k0, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62F1FD48E6E1                , vcvttpd2dq(ymm4, k0, zmm1)),
        INSTRUCTION_TEST(62B1FD48E6A4F034120000      , vcvttpd2dq(ymm4, k0, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62F1FD48E6A200200000        , vcvttpd2dq(ymm4, k0, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62F1FD48E66280              , vcvttpd2dq(ymm4, k0, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62F1FD48E6A2C0DFFFFF        , vcvttpd2dq(ymm4, k0, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62F1FD48E6627F              , vcvttpd2dq(ymm4, k0, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(6261FD48E65A7F              , vcvttpd2dq(ymm27, k0, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62F1FD48E621                , vcvttpd2dq(ymm4, k0, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(6261FD48E69AC0DFFFFF        , vcvttpd2dq(ymm27, k0, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62F1FD48E64C1A02            , vcvttpd2dq(ymm1, k0, zmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(6261FD48E619                , vcvttpd2dq(ymm27, k0, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(6221FD48E69CF023010000      , vcvttpd2dq(ymm27, k0, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(6241FD48E6D9                , vcvttpd2dq(ymm27, k0, zmm9)),
        INSTRUCTION_TEST(62F1FD48E6CA                , vcvttpd2dq(ymm1, k0, zmm2)),
        INSTRUCTION_TEST(C5FDE68C1A80000000          , vcvttpd2dq(xmm1, ymmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(C5FDE6CA                    , vcvttpd2dq(xmm1, ymm2)),
        INSTRUCTION_TEST(C5F9E68C1A80000000          , vcvttpd2dq(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(C5F9E6CA                    , vcvttpd2dq(xmm1, xmm2)),
        INSTRUCTION_TEST(6261FD48E65A80              , vcvttpd2dq(ymm27, k0, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62F1FD487A4C1A02            , vcvttpd2qq(zmm1, k0, zmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F1FD287A4C1A04            , vcvttpd2qq(ymm1, k0, ymmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F1FD487ACA                , vcvttpd2qq(zmm1, k0, zmm2)),
        INSTRUCTION_TEST(62F1FD087A4C1A08            , vcvttpd2qq(xmm1, k0, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F1FD287ACA                , vcvttpd2qq(ymm1, k0, ymm2)),
        INSTRUCTION_TEST(62F1FD087ACA                , vcvttpd2qq(xmm1, k0, xmm2)),
        INSTRUCTION_TEST(62E1FC487882C0DFFFFF        , vcvttpd2udq(ymm16, k0, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62E1FC4878A200200000        , vcvttpd2udq(ymm20, k0, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62E1FC4878627F              , vcvttpd2udq(ymm20, k0, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62E1FC4878A2C0DFFFFF        , vcvttpd2udq(ymm20, k0, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62E1FC48786280              , vcvttpd2udq(ymm20, k0, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62E1FC487821                , vcvttpd2udq(ymm20, k0, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62A1FC4878A4F034120000      , vcvttpd2udq(ymm20, k0, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62C1FC4878E6                , vcvttpd2udq(ymm20, k0, zmm14)),
        INSTRUCTION_TEST(62E1FC48788200200000        , vcvttpd2udq(ymm16, k0, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62E1FC4878427F              , vcvttpd2udq(ymm16, k0, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62E1FC48784280              , vcvttpd2udq(ymm16, k0, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62E1FC487801                , vcvttpd2udq(ymm16, k0, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62A1FC487884F023010000      , vcvttpd2udq(ymm16, k0, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62A1FC4878C4                , vcvttpd2udq(ymm16, k0, zmm20)),
        INSTRUCTION_TEST(62F1FC48784C1A02            , vcvttpd2udq(ymm1, k0, zmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F1FC4878CA                , vcvttpd2udq(ymm1, k0, zmm2)),
        INSTRUCTION_TEST(62F1FC28784C1A04            , vcvttpd2udq(xmm1, k0, ymmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F1FC2878CA                , vcvttpd2udq(xmm1, k0, ymm2)),
        INSTRUCTION_TEST(62F1FC08784C1A08            , vcvttpd2udq(xmm1, k0, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F1FC0878CA                , vcvttpd2udq(xmm1, k0, xmm2)),
        INSTRUCTION_TEST(62F1FD4878CA                , vcvttpd2uqq(zmm1, k0, zmm2)),
        INSTRUCTION_TEST(62F1FD28784C1A04            , vcvttpd2uqq(ymm1, k0, ymmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F1FD48784C1A02            , vcvttpd2uqq(zmm1, k0, zmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F1FD2878CA                , vcvttpd2uqq(ymm1, k0, ymm2)),
        INSTRUCTION_TEST(62F1FD0878CA                , vcvttpd2uqq(xmm1, k0, xmm2)),
        INSTRUCTION_TEST(62F1FD08784C1A08            , vcvttpd2uqq(xmm1, k0, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62E17E485BBA00200000        , vcvttps2dq(zmm23, k0, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62E17E485B7A7F              , vcvttps2dq(zmm23, k0, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62E17E485BBAC0DFFFFF        , vcvttps2dq(zmm23, k0, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62E17E485B7A80              , vcvttps2dq(zmm23, k0, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62E17E485B39                , vcvttps2dq(zmm23, k0, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62A17E485BBCF034120000      , vcvttps2dq(zmm23, k0, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62817E485BFC                , vcvttps2dq(zmm23, k0, zmm28)),
        INSTRUCTION_TEST(62717E485BB200200000        , vcvttps2dq(zmm14, k0, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62717E485B727F              , vcvttps2dq(zmm14, k0, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62717E485BB2C0DFFFFF        , vcvttps2dq(zmm14, k0, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62717E485B7280              , vcvttps2dq(zmm14, k0, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62317E485BB4F023010000      , vcvttps2dq(zmm14, k0, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62117E485BF1                , vcvttps2dq(zmm14, k0, zmm25)),
        INSTRUCTION_TEST(62F17E485B4C1A02            , vcvttps2dq(zmm1, k0, zmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F17E485BCA                , vcvttps2dq(zmm1, k0, zmm2)),
        INSTRUCTION_TEST(C5FE5B8C1A80000000          , vcvttps2dq(ymm1, ymmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(C5FE5BCA                    , vcvttps2dq(ymm1, ymm2)),
        INSTRUCTION_TEST(C5FA5B8C1A80000000          , vcvttps2dq(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(C5FA5BCA                    , vcvttps2dq(xmm1, xmm2)),
        INSTRUCTION_TEST(62717E485B31                , vcvttps2dq(zmm14, k0, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62F17D487A4C1A04            , vcvttps2qq(zmm1, k0, ymmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F17D287A4C1A08            , vcvttps2qq(ymm1, k0, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F17D487ACA                , vcvttps2qq(zmm1, k0, ymm2)),
        INSTRUCTION_TEST(62F17D087ACA                , vcvttps2qq(xmm1, k0, xmm2)),
        INSTRUCTION_TEST(62F17D087A4C1A10            , vcvttps2qq(xmm1, k0, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F17D287ACA                , vcvttps2qq(ymm1, k0, xmm2)),
        INSTRUCTION_TEST(62F17C4878AA00200000        , vcvttps2udq(zmm5, k0, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62F17C48786A7F              , vcvttps2udq(zmm5, k0, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62F17C4878AAC0DFFFFF        , vcvttps2udq(zmm5, k0, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62F17C48786A80              , vcvttps2udq(zmm5, k0, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62F17C487829                , vcvttps2udq(zmm5, k0, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62B17C4878ACF034120000      , vcvttps2udq(zmm5, k0, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62B17C4878EC                , vcvttps2udq(zmm5, k0, zmm20)),
        INSTRUCTION_TEST(62717C4878B200200000        , vcvttps2udq(zmm14, k0, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62717C4878727F              , vcvttps2udq(zmm14, k0, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62717C4878B2C0DFFFFF        , vcvttps2udq(zmm14, k0, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62717C48787280              , vcvttps2udq(zmm14, k0, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62717C487831                , vcvttps2udq(zmm14, k0, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62317C4878B4F023010000      , vcvttps2udq(zmm14, k0, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62117C4878F4                , vcvttps2udq(zmm14, k0, zmm28)),
        INSTRUCTION_TEST(62F17C48784C1A02            , vcvttps2udq(zmm1, k0, zmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F17C4878CA                , vcvttps2udq(zmm1, k0, zmm2)),
        INSTRUCTION_TEST(62F17C28784C1A04            , vcvttps2udq(ymm1, k0, ymmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F17C2878CA                , vcvttps2udq(ymm1, k0, ymm2)),
        INSTRUCTION_TEST(62F17C08784C1A08            , vcvttps2udq(xmm1, k0, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F17C0878CA                , vcvttps2udq(xmm1, k0, xmm2)),
        INSTRUCTION_TEST(62F17D4878CA                , vcvttps2uqq(zmm1, k0, ymm2)),
        INSTRUCTION_TEST(62F17D28784C1A08            , vcvttps2uqq(ymm1, k0, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F17D48784C1A04            , vcvttps2uqq(zmm1, k0, ymmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F17D0878CA                , vcvttps2uqq(xmm1, k0, xmm2)),
        INSTRUCTION_TEST(62F17D08784C1A10            , vcvttps2uqq(xmm1, k0, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F17D2878CA                , vcvttps2uqq(ymm1, k0, xmm2)),
        INSTRUCTION_TEST(C5FB2C8C1A80000000          , vcvttsd2si(ecx, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(C5FB2CCA                    , vcvttsd2si(ecx, xmm2)),
        INSTRUCTION_TEST(C4E1FB2C8C1A80000000        , vcvttsd2si(rcx, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(C4E1FB2CCA                  , vcvttsd2si(rcx, xmm2)),
        INSTRUCTION_TEST(6271FF0878427F              , vcvttsd2usi(r8, qword_ptr(rdx, 1016))),
        INSTRUCTION_TEST(6271FF087882F8FBFFFF        , vcvttsd2usi(r8, qword_ptr(rdx, -1032))),
        INSTRUCTION_TEST(6271FF08784280              , vcvttsd2usi(r8, qword_ptr(rdx, -1024))),
        INSTRUCTION_TEST(6271FF087801                , vcvttsd2usi(r8, qword_ptr(rcx))),
        INSTRUCTION_TEST(62317F0878EF                , vcvttsd2usi(r13d, xmm23)),
        INSTRUCTION_TEST(6231FF087884F023010000      , vcvttsd2usi(r8, qword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(6271FF08788200040000        , vcvttsd2usi(r8, qword_ptr(rdx, 1024))),
        INSTRUCTION_TEST(62317F0878ED                , vcvttsd2usi(r13d, xmm21)),
        INSTRUCTION_TEST(62717F0878AA00040000        , vcvttsd2usi(r13d, qword_ptr(rdx, 1024))),
        INSTRUCTION_TEST(6231FF087884F034120000      , vcvttsd2usi(r8, qword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(6231FF0878C3                , vcvttsd2usi(r8, xmm19)),
        INSTRUCTION_TEST(62B1FF087884F034120000      , vcvttsd2usi(rax, qword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62B1FF087884F023010000      , vcvttsd2usi(rax, qword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62F1FF087801                , vcvttsd2usi(rax, qword_ptr(rcx))),
        INSTRUCTION_TEST(62F1FF08784280              , vcvttsd2usi(rax, qword_ptr(rdx, -1024))),
        INSTRUCTION_TEST(62F1FF087882F8FBFFFF        , vcvttsd2usi(rax, qword_ptr(rdx, -1032))),
        INSTRUCTION_TEST(62F1FF0878427F              , vcvttsd2usi(rax, qword_ptr(rdx, 1016))),
        INSTRUCTION_TEST(62F1FF08788200040000        , vcvttsd2usi(rax, qword_ptr(rdx, 1024))),
        INSTRUCTION_TEST(62B1FF0878C3                , vcvttsd2usi(rax, xmm19)),
        INSTRUCTION_TEST(62F1FF0878CA                , vcvttsd2usi(rcx, xmm2)),
        INSTRUCTION_TEST(62F1FF08784C1A10            , vcvttsd2usi(rcx, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62717F08786A7F              , vcvttsd2usi(r13d, qword_ptr(rdx, 1016))),
        INSTRUCTION_TEST(6271FF0878C7                , vcvttsd2usi(r8, xmm7)),
        INSTRUCTION_TEST(62717F0878AAF8FBFFFF        , vcvttsd2usi(r13d, qword_ptr(rdx, -1032))),
        INSTRUCTION_TEST(62F1FF0878C7                , vcvttsd2usi(rax, xmm7)),
        INSTRUCTION_TEST(62717F087829                , vcvttsd2usi(r13d, qword_ptr(rcx))),
        INSTRUCTION_TEST(62B17F087884F023010000      , vcvttsd2usi(eax, qword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62B17F087884F034120000      , vcvttsd2usi(eax, qword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62F17F087801                , vcvttsd2usi(eax, qword_ptr(rcx))),
        INSTRUCTION_TEST(62F17F08784280              , vcvttsd2usi(eax, qword_ptr(rdx, -1024))),
        INSTRUCTION_TEST(62F17F087882F8FBFFFF        , vcvttsd2usi(eax, qword_ptr(rdx, -1032))),
        INSTRUCTION_TEST(62F17F0878427F              , vcvttsd2usi(eax, qword_ptr(rdx, 1016))),
        INSTRUCTION_TEST(62F17F08788200040000        , vcvttsd2usi(eax, qword_ptr(rdx, 1024))),
        INSTRUCTION_TEST(62B17F0878C5                , vcvttsd2usi(eax, xmm21)),
        INSTRUCTION_TEST(62B17F0878C7                , vcvttsd2usi(eax, xmm23)),
        INSTRUCTION_TEST(62B17F0878ACF023010000      , vcvttsd2usi(ebp, qword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62B17F0878ACF034120000      , vcvttsd2usi(ebp, qword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62F17F08786A80              , vcvttsd2usi(ebp, qword_ptr(rdx, -1024))),
        INSTRUCTION_TEST(62F17F0878AAF8FBFFFF        , vcvttsd2usi(ebp, qword_ptr(rdx, -1032))),
        INSTRUCTION_TEST(62F17F08786A7F              , vcvttsd2usi(ebp, qword_ptr(rdx, 1016))),
        INSTRUCTION_TEST(62F17F0878AA00040000        , vcvttsd2usi(ebp, qword_ptr(rdx, 1024))),
        INSTRUCTION_TEST(62B17F0878ED                , vcvttsd2usi(ebp, xmm21)),
        INSTRUCTION_TEST(62B17F0878EF                , vcvttsd2usi(ebp, xmm23)),
        INSTRUCTION_TEST(62F17F08784C1A10            , vcvttsd2usi(ecx, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F17F0878CA                , vcvttsd2usi(ecx, xmm2)),
        INSTRUCTION_TEST(62317F0878ACF023010000      , vcvttsd2usi(r13d, qword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62317F0878ACF034120000      , vcvttsd2usi(r13d, qword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62F17F087829                , vcvttsd2usi(ebp, qword_ptr(rcx))),
        INSTRUCTION_TEST(62717F08786A80              , vcvttsd2usi(r13d, qword_ptr(rdx, -1024))),
        INSTRUCTION_TEST(C4E1FA2C8C1A80000000        , vcvttss2si(rcx, dword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(C4E1FA2CCA                  , vcvttss2si(rcx, xmm2)),
        INSTRUCTION_TEST(C5FA2CCA                    , vcvttss2si(ecx, xmm2)),
        INSTRUCTION_TEST(C5FA2C8C1A80000000          , vcvttss2si(ecx, dword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F1FE0878CA                , vcvttss2usi(rcx, xmm2)),
        INSTRUCTION_TEST(62717E0878AA00020000        , vcvttss2usi(r13d, dword_ptr(rdx, 512))),
        INSTRUCTION_TEST(62317E0878EA                , vcvttss2usi(r13d, xmm18)),
        INSTRUCTION_TEST(62317E0878ED                , vcvttss2usi(r13d, xmm21)),
        INSTRUCTION_TEST(6231FE087884F023010000      , vcvttss2usi(r8, dword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(6231FE087884F034120000      , vcvttss2usi(r8, dword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(6271FE087801                , vcvttss2usi(r8, dword_ptr(rcx))),
        INSTRUCTION_TEST(6271FE08784280              , vcvttss2usi(r8, dword_ptr(rdx, -512))),
        INSTRUCTION_TEST(6271FE087882FCFDFFFF        , vcvttss2usi(r8, dword_ptr(rdx, -516))),
        INSTRUCTION_TEST(6271FE0878427F              , vcvttss2usi(r8, dword_ptr(rdx, 508))),
        INSTRUCTION_TEST(6271FE08788200020000        , vcvttss2usi(r8, dword_ptr(rdx, 512))),
        INSTRUCTION_TEST(6211FE0878C3                , vcvttss2usi(r8, xmm27)),
        INSTRUCTION_TEST(62B1FE087884F023010000      , vcvttss2usi(rax, dword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62B1FE087884F034120000      , vcvttss2usi(rax, dword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62F1FE087801                , vcvttss2usi(rax, dword_ptr(rcx))),
        INSTRUCTION_TEST(62F1FE08784280              , vcvttss2usi(rax, dword_ptr(rdx, -512))),
        INSTRUCTION_TEST(62F1FE087882FCFDFFFF        , vcvttss2usi(rax, dword_ptr(rdx, -516))),
        INSTRUCTION_TEST(62F1FE0878427F              , vcvttss2usi(rax, dword_ptr(rdx, 508))),
        INSTRUCTION_TEST(62F1FE08788200020000        , vcvttss2usi(rax, dword_ptr(rdx, 512))),
        INSTRUCTION_TEST(6291FE0878C3                , vcvttss2usi(rax, xmm27)),
        INSTRUCTION_TEST(62F1FE0878C7                , vcvttss2usi(rax, xmm7)),
        INSTRUCTION_TEST(62F1FE08784C1A20            , vcvttss2usi(rcx, dword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(6271FE0878C7                , vcvttss2usi(r8, xmm7)),
        INSTRUCTION_TEST(62717E08786A7F              , vcvttss2usi(r13d, dword_ptr(rdx, 508))),
        INSTRUCTION_TEST(62717E087829                , vcvttss2usi(r13d, dword_ptr(rcx))),
        INSTRUCTION_TEST(62717E08786A80              , vcvttss2usi(r13d, dword_ptr(rdx, -512))),
        INSTRUCTION_TEST(62717E0878AAFCFDFFFF        , vcvttss2usi(r13d, dword_ptr(rdx, -516))),
        INSTRUCTION_TEST(62B17E087884F023010000      , vcvttss2usi(eax, dword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62B17E087884F034120000      , vcvttss2usi(eax, dword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62F17E08784280              , vcvttss2usi(eax, dword_ptr(rdx, -512))),
        INSTRUCTION_TEST(62F17E087882FCFDFFFF        , vcvttss2usi(eax, dword_ptr(rdx, -516))),
        INSTRUCTION_TEST(62F17E0878427F              , vcvttss2usi(eax, dword_ptr(rdx, 508))),
        INSTRUCTION_TEST(62F17E08788200020000        , vcvttss2usi(eax, dword_ptr(rdx, 512))),
        INSTRUCTION_TEST(62B17E0878C2                , vcvttss2usi(eax, xmm18)),
        INSTRUCTION_TEST(62B17E0878C5                , vcvttss2usi(eax, xmm21)),
        INSTRUCTION_TEST(62B17E0878ACF023010000      , vcvttss2usi(ebp, dword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62B17E0878ACF034120000      , vcvttss2usi(ebp, dword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62F17E087801                , vcvttss2usi(eax, dword_ptr(rcx))),
        INSTRUCTION_TEST(62F17E08786A80              , vcvttss2usi(ebp, dword_ptr(rdx, -512))),
        INSTRUCTION_TEST(62F17E087829                , vcvttss2usi(ebp, dword_ptr(rcx))),
        INSTRUCTION_TEST(62317E0878ACF034120000      , vcvttss2usi(r13d, dword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62317E0878ACF023010000      , vcvttss2usi(r13d, dword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62F17E08784C1A20            , vcvttss2usi(ecx, dword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F17E0878CA                , vcvttss2usi(ecx, xmm2)),
        INSTRUCTION_TEST(62B17E0878EA                , vcvttss2usi(ebp, xmm18)),
        INSTRUCTION_TEST(62F17E0878AA00020000        , vcvttss2usi(ebp, dword_ptr(rdx, 512))),
        INSTRUCTION_TEST(62F17E08786A7F              , vcvttss2usi(ebp, dword_ptr(rdx, 508))),
        INSTRUCTION_TEST(62F17E0878AAFCFDFFFF        , vcvttss2usi(ebp, dword_ptr(rdx, -516))),
        INSTRUCTION_TEST(62B17E0878ED                , vcvttss2usi(ebp, xmm21)),
        INSTRUCTION_TEST(62E17E487A9200100000        , vcvtudq2pd(zmm18, k0, ymmword_ptr(rdx, 4096))),
        INSTRUCTION_TEST(62A17E487A94F034120000      , vcvtudq2pd(zmm18, k0, ymmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62E17E487A527F              , vcvtudq2pd(zmm18, k0, ymmword_ptr(rdx, 4064))),
        INSTRUCTION_TEST(62E17E487A92E0EFFFFF        , vcvtudq2pd(zmm18, k0, ymmword_ptr(rdx, -4128))),
        INSTRUCTION_TEST(62E17E487A5280              , vcvtudq2pd(zmm18, k0, ymmword_ptr(rdx, -4096))),
        INSTRUCTION_TEST(62E17E487A11                , vcvtudq2pd(zmm18, k0, ymmword_ptr(rcx))),
        INSTRUCTION_TEST(62A17E487A94F023010000      , vcvtudq2pd(zmm18, k0, ymmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62F17E287A4C1A08            , vcvtudq2pd(ymm1, k0, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62C17E487AD6                , vcvtudq2pd(zmm18, k0, ymm14)),
        INSTRUCTION_TEST(62F17E487A4C1A04            , vcvtudq2pd(zmm1, k0, ymmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F17E487ACA                , vcvtudq2pd(zmm1, k0, ymm2)),
        INSTRUCTION_TEST(62F17E287ACA                , vcvtudq2pd(ymm1, k0, xmm2)),
        INSTRUCTION_TEST(62F17E087ACA                , vcvtudq2pd(xmm1, k0, xmm2)),
        INSTRUCTION_TEST(62F17E087A4C1A10            , vcvtudq2pd(xmm1, k0, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62817E487AD4                , vcvtudq2pd(zmm18, k0, ymm28)),
        INSTRUCTION_TEST(62B17F487ABCF034120000      , vcvtudq2ps(zmm7, k0, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62E17F487A9200200000        , vcvtudq2ps(zmm18, k0, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62F17F487A39                , vcvtudq2ps(zmm7, k0, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62E17F487A527F              , vcvtudq2ps(zmm18, k0, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62F17F487ABAC0DFFFFF        , vcvtudq2ps(zmm7, k0, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62F17F487A7A7F              , vcvtudq2ps(zmm7, k0, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62F17F487ABA00200000        , vcvtudq2ps(zmm7, k0, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62F17F487A7A80              , vcvtudq2ps(zmm7, k0, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62E17F487A92C0DFFFFF        , vcvtudq2ps(zmm18, k0, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62917F487AF8                , vcvtudq2ps(zmm7, k0, zmm24)),
        INSTRUCTION_TEST(62E17F487A11                , vcvtudq2ps(zmm18, k0, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62F17F087ACA                , vcvtudq2ps(xmm1, k0, xmm2)),
        INSTRUCTION_TEST(62F17F087A4C1A08            , vcvtudq2ps(xmm1, k0, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F17F287ACA                , vcvtudq2ps(ymm1, k0, ymm2)),
        INSTRUCTION_TEST(62F17F287A4C1A04            , vcvtudq2ps(ymm1, k0, ymmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62E17F487A5280              , vcvtudq2ps(zmm18, k0, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62F17F487A4C1A02            , vcvtudq2ps(zmm1, k0, zmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F17F487ACA                , vcvtudq2ps(zmm1, k0, zmm2)),
        INSTRUCTION_TEST(62E17F487AD7                , vcvtudq2ps(zmm18, k0, zmm7)),
        INSTRUCTION_TEST(62A17F487A94F023010000      , vcvtudq2ps(zmm18, k0, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62F1FE087ACA                , vcvtuqq2pd(xmm1, k0, xmm2)),
        INSTRUCTION_TEST(62F1FE087A4C1A08            , vcvtuqq2pd(xmm1, k0, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F1FE287ACA                , vcvtuqq2pd(ymm1, k0, ymm2)),
        INSTRUCTION_TEST(62F1FE287A4C1A04            , vcvtuqq2pd(ymm1, k0, ymmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F1FE487ACA                , vcvtuqq2pd(zmm1, k0, zmm2)),
        INSTRUCTION_TEST(62F1FE487A4C1A02            , vcvtuqq2pd(zmm1, k0, zmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F1FF487A4C1A02            , vcvtuqq2ps(ymm1, k0, zmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F1FF487ACA                , vcvtuqq2ps(ymm1, k0, zmm2)),
        INSTRUCTION_TEST(62F1FF287A4C1A04            , vcvtuqq2ps(xmm1, k0, ymmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F1FF087A4C1A08            , vcvtuqq2ps(xmm1, k0, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F1FF087ACA                , vcvtuqq2ps(xmm1, k0, xmm2)),
        INSTRUCTION_TEST(62F1FF287ACA                , vcvtuqq2ps(xmm1, k0, ymm2)),
        INSTRUCTION_TEST(62C177087BDD                , vcvtusi2sd(xmm19, xmm1, r13d)),
        INSTRUCTION_TEST(62E177087B5A80              , vcvtusi2sd(xmm19, xmm1, dword_ptr(rdx, -512))),
        INSTRUCTION_TEST(62E177087B9AFCFDFFFF        , vcvtusi2sd(xmm19, xmm1, dword_ptr(rdx, -516))),
        INSTRUCTION_TEST(62E177087B5A7F              , vcvtusi2sd(xmm19, xmm1, dword_ptr(rdx, 508))),
        INSTRUCTION_TEST(62E177087B9A00020000        , vcvtusi2sd(xmm19, xmm1, dword_ptr(rdx, 512))),
        INSTRUCTION_TEST(62E177087BD8                , vcvtusi2sd(xmm19, xmm1, eax)),
        INSTRUCTION_TEST(62E177087BDD                , vcvtusi2sd(xmm19, xmm1, ebp)),
        INSTRUCTION_TEST(622177007B94F034120000      , vcvtusi2sd(xmm26, xmm17, dword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(624177007BD5                , vcvtusi2sd(xmm26, xmm17, r13d)),
        INSTRUCTION_TEST(626177007B5280              , vcvtusi2sd(xmm26, xmm17, dword_ptr(rdx, -512))),
        INSTRUCTION_TEST(626177007B92FCFDFFFF        , vcvtusi2sd(xmm26, xmm17, dword_ptr(rdx, -516))),
        INSTRUCTION_TEST(626177007B527F              , vcvtusi2sd(xmm26, xmm17, dword_ptr(rdx, 508))),
        INSTRUCTION_TEST(626177007B9200020000        , vcvtusi2sd(xmm26, xmm17, dword_ptr(rdx, 512))),
        INSTRUCTION_TEST(626177007BD0                , vcvtusi2sd(xmm26, xmm17, eax)),
        INSTRUCTION_TEST(626177007BD5                , vcvtusi2sd(xmm26, xmm17, ebp)),
        INSTRUCTION_TEST(62E177087B19                , vcvtusi2sd(xmm19, xmm1, dword_ptr(rcx))),
        INSTRUCTION_TEST(626177007B11                , vcvtusi2sd(xmm26, xmm17, dword_ptr(rcx))),
        INSTRUCTION_TEST(6271AF007BF0                , vcvtusi2sd(xmm14, xmm26, rax)),
        INSTRUCTION_TEST(62A177087B9CF023010000      , vcvtusi2sd(xmm19, xmm1, dword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(6271AF007BB200040000        , vcvtusi2sd(xmm14, xmm26, qword_ptr(rdx, 1024))),
        INSTRUCTION_TEST(6251AF007BF0                , vcvtusi2sd(xmm14, xmm26, r8)),
        INSTRUCTION_TEST(62F16F087BCB                , vcvtusi2sd(xmm1, xmm2, ebx)),
        INSTRUCTION_TEST(62F1EF087B4C2B10            , vcvtusi2sd(xmm1, xmm2, qword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F1EF087BCB                , vcvtusi2sd(xmm1, xmm2, rbx)),
        INSTRUCTION_TEST(6231CF087B9CF034120000      , vcvtusi2sd(xmm11, xmm6, qword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(6271CF087B19                , vcvtusi2sd(xmm11, xmm6, qword_ptr(rcx))),
        INSTRUCTION_TEST(6271CF087B5A80              , vcvtusi2sd(xmm11, xmm6, qword_ptr(rdx, -1024))),
        INSTRUCTION_TEST(6271CF087B9AF8FBFFFF        , vcvtusi2sd(xmm11, xmm6, qword_ptr(rdx, -1032))),
        INSTRUCTION_TEST(6271CF087B5A7F              , vcvtusi2sd(xmm11, xmm6, qword_ptr(rdx, 1016))),
        INSTRUCTION_TEST(62F16F087B4C2B20            , vcvtusi2sd(xmm1, xmm2, dword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(6251CF087BD8                , vcvtusi2sd(xmm11, xmm6, r8)),
        INSTRUCTION_TEST(6271CF087BD8                , vcvtusi2sd(xmm11, xmm6, rax)),
        INSTRUCTION_TEST(6231AF007BB4F023010000      , vcvtusi2sd(xmm14, xmm26, qword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(6271AF007B31                , vcvtusi2sd(xmm14, xmm26, qword_ptr(rcx))),
        INSTRUCTION_TEST(6271AF007B7280              , vcvtusi2sd(xmm14, xmm26, qword_ptr(rdx, -1024))),
        INSTRUCTION_TEST(6271AF007BB2F8FBFFFF        , vcvtusi2sd(xmm14, xmm26, qword_ptr(rdx, -1032))),
        INSTRUCTION_TEST(6271AF007B727F              , vcvtusi2sd(xmm14, xmm26, qword_ptr(rdx, 1016))),
        INSTRUCTION_TEST(6271CF087B9A00040000        , vcvtusi2sd(xmm11, xmm6, qword_ptr(rdx, 1024))),
        INSTRUCTION_TEST(62F12E007B29                , vcvtusi2ss(xmm5, xmm26, dword_ptr(rcx))),
        INSTRUCTION_TEST(62F12E007B6A80              , vcvtusi2ss(xmm5, xmm26, dword_ptr(rdx, -512))),
        INSTRUCTION_TEST(62F12E007BAAFCFDFFFF        , vcvtusi2ss(xmm5, xmm26, dword_ptr(rdx, -516))),
        INSTRUCTION_TEST(62F12E007B6A7F              , vcvtusi2ss(xmm5, xmm26, dword_ptr(rdx, 508))),
        INSTRUCTION_TEST(62F12E007BAA00020000        , vcvtusi2ss(xmm5, xmm26, dword_ptr(rdx, 512))),
        INSTRUCTION_TEST(62F12E007BE8                , vcvtusi2ss(xmm5, xmm26, eax)),
        INSTRUCTION_TEST(62F12E007BED                , vcvtusi2ss(xmm5, xmm26, ebp)),
        INSTRUCTION_TEST(62D12E007BED                , vcvtusi2ss(xmm5, xmm26, r13d)),
        INSTRUCTION_TEST(627146087B8AFCFDFFFF        , vcvtusi2ss(xmm9, xmm7, dword_ptr(rdx, -516))),
        INSTRUCTION_TEST(627146087B09                , vcvtusi2ss(xmm9, xmm7, dword_ptr(rcx))),
        INSTRUCTION_TEST(627146087B4A80              , vcvtusi2ss(xmm9, xmm7, dword_ptr(rdx, -512))),
        INSTRUCTION_TEST(627146087B4A7F              , vcvtusi2ss(xmm9, xmm7, dword_ptr(rdx, 508))),
        INSTRUCTION_TEST(627146087B8A00020000        , vcvtusi2ss(xmm9, xmm7, dword_ptr(rdx, 512))),
        INSTRUCTION_TEST(627146087BC8                , vcvtusi2ss(xmm9, xmm7, eax)),
        INSTRUCTION_TEST(627146087BCD                , vcvtusi2ss(xmm9, xmm7, ebp)),
        INSTRUCTION_TEST(625146087BCD                , vcvtusi2ss(xmm9, xmm7, r13d)),
        INSTRUCTION_TEST(62B12E007BACF023010000      , vcvtusi2ss(xmm5, xmm26, dword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(623146087B8CF034120000      , vcvtusi2ss(xmm9, xmm7, dword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62E1AE087BE8                , vcvtusi2ss(xmm21, xmm10, rax)),
        INSTRUCTION_TEST(62C1AE087BE8                , vcvtusi2ss(xmm21, xmm10, r8)),
        INSTRUCTION_TEST(62E1AE087BAA00040000        , vcvtusi2ss(xmm21, xmm10, qword_ptr(rdx, 1024))),
        INSTRUCTION_TEST(62F16E087B4C2B20            , vcvtusi2ss(xmm1, xmm2, dword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F1EE087B4C2B10            , vcvtusi2ss(xmm1, xmm2, qword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F1EE087BCB                , vcvtusi2ss(xmm1, xmm2, rbx)),
        INSTRUCTION_TEST(6231CE007BB4F023010000      , vcvtusi2ss(xmm14, xmm22, qword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(6271CE007B31                , vcvtusi2ss(xmm14, xmm22, qword_ptr(rcx))),
        INSTRUCTION_TEST(6271CE007B7280              , vcvtusi2ss(xmm14, xmm22, qword_ptr(rdx, -1024))),
        INSTRUCTION_TEST(6271CE007BB2F8FBFFFF        , vcvtusi2ss(xmm14, xmm22, qword_ptr(rdx, -1032))),
        INSTRUCTION_TEST(6271CE007B727F              , vcvtusi2ss(xmm14, xmm22, qword_ptr(rdx, 1016))),
        INSTRUCTION_TEST(62F16E087BCB                , vcvtusi2ss(xmm1, xmm2, ebx)),
        INSTRUCTION_TEST(6251CE007BF0                , vcvtusi2ss(xmm14, xmm22, r8)),
        INSTRUCTION_TEST(6271CE007BF0                , vcvtusi2ss(xmm14, xmm22, rax)),
        INSTRUCTION_TEST(62A1AE087BACF034120000      , vcvtusi2ss(xmm21, xmm10, qword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62E1AE087B29                , vcvtusi2ss(xmm21, xmm10, qword_ptr(rcx))),
        INSTRUCTION_TEST(62E1AE087B6A80              , vcvtusi2ss(xmm21, xmm10, qword_ptr(rdx, -1024))),
        INSTRUCTION_TEST(62E1AE087BAAF8FBFFFF        , vcvtusi2ss(xmm21, xmm10, qword_ptr(rdx, -1032))),
        INSTRUCTION_TEST(62E1AE087B6A7F              , vcvtusi2ss(xmm21, xmm10, qword_ptr(rdx, 1016))),
        INSTRUCTION_TEST(6271CE007BB200040000        , vcvtusi2ss(xmm14, xmm22, qword_ptr(rdx, 1024))),
        INSTRUCTION_TEST(62F36D48424C2B0201          , vdbpsadbw(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F36D4842CB01              , vdbpsadbw(zmm1, k0, zmm2, zmm3, Imm(1))),
        INSTRUCTION_TEST(62F36D28424C2B0401          , vdbpsadbw(ymm1, k0, ymm2, ymmword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F36D08424C2B0801          , vdbpsadbw(xmm1, k0, xmm2, xmmword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F36D0842CB01              , vdbpsadbw(xmm1, k0, xmm2, xmm3, Imm(1))),
        INSTRUCTION_TEST(62F36D2842CB01              , vdbpsadbw(ymm1, k0, ymm2, ymm3, Imm(1))),
        INSTRUCTION_TEST(62E1DD485E9200200000        , vdivpd(zmm18, k0, zmm4, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62E1CD485E9200200000        , vdivpd(zmm18, k0, zmm6, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62E1CD485E527F              , vdivpd(zmm18, k0, zmm6, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62E1CD485E92C0DFFFFF        , vdivpd(zmm18, k0, zmm6, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62E1CD485E5280              , vdivpd(zmm18, k0, zmm6, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62E1CD485E11                , vdivpd(zmm18, k0, zmm6, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62A1CD485E94F023010000      , vdivpd(zmm18, k0, zmm6, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62C1CD485ED3                , vdivpd(zmm18, k0, zmm6, zmm11)),
        INSTRUCTION_TEST(62E1DD485E92C0DFFFFF        , vdivpd(zmm18, k0, zmm4, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62E1DD485E527F              , vdivpd(zmm18, k0, zmm4, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62E1DD485E11                , vdivpd(zmm18, k0, zmm4, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62E1DD485E5280              , vdivpd(zmm18, k0, zmm4, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(C5E95E8C2B80000000          , vdivpd(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C5ED5ECB                    , vdivpd(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(C5ED5E8C2B80000000          , vdivpd(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C5E95ECB                    , vdivpd(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(62F1ED485E4C2B02            , vdivpd(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62A1DD485ED3                , vdivpd(zmm18, k0, zmm4, zmm19)),
        INSTRUCTION_TEST(62A1DD485E94F034120000      , vdivpd(zmm18, k0, zmm4, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62F1ED485ECB                , vdivpd(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(62F114405EB200200000        , vdivps(zmm6, k0, zmm29, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62F114405E727F              , vdivps(zmm6, k0, zmm29, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62F114405EB2C0DFFFFF        , vdivps(zmm6, k0, zmm29, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62F114405E7280              , vdivps(zmm6, k0, zmm29, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62F114405E31                , vdivps(zmm6, k0, zmm29, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62B114405EB4F034120000      , vdivps(zmm6, k0, zmm29, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62F114405EF4                , vdivps(zmm6, k0, zmm29, zmm4)),
        INSTRUCTION_TEST(62E144405EBA00200000        , vdivps(zmm23, k0, zmm23, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62E144405E7A7F              , vdivps(zmm23, k0, zmm23, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62E144405EBAC0DFFFFF        , vdivps(zmm23, k0, zmm23, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62E144405E39                , vdivps(zmm23, k0, zmm23, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62A144405EBCF023010000      , vdivps(zmm23, k0, zmm23, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(628144405EFC                , vdivps(zmm23, k0, zmm23, zmm28)),
        INSTRUCTION_TEST(62F16C485E4C2B02            , vdivps(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F16C485ECB                , vdivps(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(C5EC5E8C2B80000000          , vdivps(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C5EC5ECB                    , vdivps(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(C5E85E8C2B80000000          , vdivps(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C5E85ECB                    , vdivps(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(62E144405E7A80              , vdivps(zmm23, k0, zmm23, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(6231DF085EFE                , vdivsd(xmm15, k0, xmm4, xmm22)),
        INSTRUCTION_TEST(626197085EAA00040000        , vdivsd(xmm29, k0, xmm13, qword_ptr(rdx, 1024))),
        INSTRUCTION_TEST(626197085E6A7F              , vdivsd(xmm29, k0, xmm13, qword_ptr(rdx, 1016))),
        INSTRUCTION_TEST(626197085E6A80              , vdivsd(xmm29, k0, xmm13, qword_ptr(rdx, -1024))),
        INSTRUCTION_TEST(626197085E29                , vdivsd(xmm29, k0, xmm13, qword_ptr(rcx))),
        INSTRUCTION_TEST(622197085EACF023010000      , vdivsd(xmm29, k0, xmm13, qword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(C55B5EBA00040000            , vdivsd(xmm15, xmm4, qword_ptr(rdx, 1024))),
        INSTRUCTION_TEST(622197085EEE                , vdivsd(xmm29, k0, xmm13, xmm22)),
        INSTRUCTION_TEST(C55B5EBAF8FBFFFF            , vdivsd(xmm15, xmm4, qword_ptr(rdx, -1032))),
        INSTRUCTION_TEST(C55B5EBAF8030000            , vdivsd(xmm15, xmm4, qword_ptr(rdx, 1016))),
        INSTRUCTION_TEST(C5EB5E8C2B80000000          , vdivsd(xmm1, xmm2, qword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C5EB5ECB                    , vdivsd(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(626197085EAAF8FBFFFF        , vdivsd(xmm29, k0, xmm13, qword_ptr(rdx, -1032))),
        INSTRUCTION_TEST(C4215B5EBCF034120000        , vdivsd(xmm15, xmm4, qword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(C55B5E39                    , vdivsd(xmm15, xmm4, qword_ptr(rcx))),
        INSTRUCTION_TEST(C55B5EBA00FCFFFF            , vdivsd(xmm15, xmm4, qword_ptr(rdx, -1024))),
        INSTRUCTION_TEST(62216E085EDC                , vdivss(xmm27, k0, xmm2, xmm20)),
        INSTRUCTION_TEST(62616E085E9A00020000        , vdivss(xmm27, k0, xmm2, dword_ptr(rdx, 512))),
        INSTRUCTION_TEST(62616E085E9AFCFDFFFF        , vdivss(xmm27, k0, xmm2, dword_ptr(rdx, -516))),
        INSTRUCTION_TEST(62616E085E5A80              , vdivss(xmm27, k0, xmm2, dword_ptr(rdx, -512))),
        INSTRUCTION_TEST(62616E085E19                , vdivss(xmm27, k0, xmm2, dword_ptr(rcx))),
        INSTRUCTION_TEST(62216E085E9CF034120000      , vdivss(xmm27, k0, xmm2, dword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62A14E085EE9                , vdivss(xmm21, k0, xmm6, xmm17)),
        INSTRUCTION_TEST(62616E085E5A7F              , vdivss(xmm27, k0, xmm2, dword_ptr(rdx, 508))),
        INSTRUCTION_TEST(62E14E085E6A7F              , vdivss(xmm21, k0, xmm6, dword_ptr(rdx, 508))),
        INSTRUCTION_TEST(62E14E085EAAFCFDFFFF        , vdivss(xmm21, k0, xmm6, dword_ptr(rdx, -516))),
        INSTRUCTION_TEST(62E14E085E6A80              , vdivss(xmm21, k0, xmm6, dword_ptr(rdx, -512))),
        INSTRUCTION_TEST(62E14E085E29                , vdivss(xmm21, k0, xmm6, dword_ptr(rcx))),
        INSTRUCTION_TEST(62A14E085EACF023010000      , vdivss(xmm21, k0, xmm6, dword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(C5EA5ECB                    , vdivss(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(C5EA5E8C2B80000000          , vdivss(xmm1, xmm2, dword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62E14E085EAA00020000        , vdivss(xmm21, k0, xmm6, dword_ptr(rdx, 512))),
        INSTRUCTION_TEST(62F26E48524C2B02            , vdpbf16ps(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F26E4852CB                , vdpbf16ps(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(62F26E28524C2B04            , vdpbf16ps(ymm1, k0, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F26E08524C2B08            , vdpbf16ps(xmm1, k0, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F26E0852CB                , vdpbf16ps(xmm1, k0, xmm2, xmm3)),
        INSTRUCTION_TEST(62F26E2852CB                , vdpbf16ps(ymm1, k0, ymm2, ymm3)),
        INSTRUCTION_TEST(C4E36941CB01                , vdppd(xmm1, xmm2, xmm3, Imm(1))),
        INSTRUCTION_TEST(C4E369418C2B8000000001      , vdppd(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(C4E36940CB01                , vdpps(xmm1, xmm2, xmm3, Imm(1))),
        INSTRUCTION_TEST(C4E369408C2B8000000001      , vdpps(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(C4E36D40CB01                , vdpps(ymm1, ymm2, ymm3, Imm(1))),
        INSTRUCTION_TEST(C4E36D408C2B8000000001      , vdpps(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(0F00A41180000000            , verr(word_ptr(rcx, rdx, 1, 128))),
        INSTRUCTION_TEST(0F00E1                      , verr(cx)),
        INSTRUCTION_TEST(0F00AC1180000000            , verw(word_ptr(rcx, rdx, 1, 128))),
        INSTRUCTION_TEST(0F00E9                      , verw(cx)),
        INSTRUCTION_TEST(62F2FD48C8CA                , vexp2pd(zmm1, k0, zmm2)),
        INSTRUCTION_TEST(62F2FD48C84C1A02            , vexp2pd(zmm1, k0, zmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F27D48C8CA                , vexp2ps(zmm1, k0, zmm2)),
        INSTRUCTION_TEST(62F27D48C84C1A02            , vexp2ps(zmm1, k0, zmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F2FD4888A200040000        , vexpandpd(zmm4, k0, zmmword_ptr(rdx, 1024))),
        INSTRUCTION_TEST(62F2FD4888627F              , vexpandpd(zmm4, k0, zmmword_ptr(rdx, 1016))),
        INSTRUCTION_TEST(62F2FD4888A2F8FBFFFF        , vexpandpd(zmm4, k0, zmmword_ptr(rdx, -1032))),
        INSTRUCTION_TEST(62F2FD48886280              , vexpandpd(zmm4, k0, zmmword_ptr(rdx, -1024))),
        INSTRUCTION_TEST(62F2FD488821                , vexpandpd(zmm4, k0, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62B2FD4888A4F034120000      , vexpandpd(zmm4, k0, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(6262FD48888200040000        , vexpandpd(zmm24, k0, zmmword_ptr(rdx, 1024))),
        INSTRUCTION_TEST(6262FD488882F8FBFFFF        , vexpandpd(zmm24, k0, zmmword_ptr(rdx, -1032))),
        INSTRUCTION_TEST(6262FD48884280              , vexpandpd(zmm24, k0, zmmword_ptr(rdx, -1024))),
        INSTRUCTION_TEST(6262FD4888427F              , vexpandpd(zmm24, k0, zmmword_ptr(rdx, 1016))),
        INSTRUCTION_TEST(6222FD488884F023010000      , vexpandpd(zmm24, k0, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(6262FD488801                , vexpandpd(zmm24, k0, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62F2FD08884C1A10            , vexpandpd(xmm1, k0, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F2FD2888CA                , vexpandpd(ymm1, k0, ymm2)),
        INSTRUCTION_TEST(62F2FD28884C1A10            , vexpandpd(ymm1, k0, ymmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F2FD0888CA                , vexpandpd(xmm1, k0, xmm2)),
        INSTRUCTION_TEST(62F2FD48884C1A10            , vexpandpd(zmm1, k0, zmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62C2FD4888D1                , vexpandpd(zmm18, k0, zmm9)),
        INSTRUCTION_TEST(62C2FD4888FF                , vexpandpd(zmm23, k0, zmm15)),
        INSTRUCTION_TEST(62F2FD4888CA                , vexpandpd(zmm1, k0, zmm2)),
        INSTRUCTION_TEST(62627D48886280              , vexpandps(zmm28, k0, zmmword_ptr(rdx, -512))),
        INSTRUCTION_TEST(62F27D4888A2FCFDFFFF        , vexpandps(zmm4, k0, zmmword_ptr(rdx, -516))),
        INSTRUCTION_TEST(62F27D4888A200020000        , vexpandps(zmm4, k0, zmmword_ptr(rdx, 512))),
        INSTRUCTION_TEST(62F27D488821                , vexpandps(zmm4, k0, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62B27D4888A4F023010000      , vexpandps(zmm4, k0, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62627D4888A200020000        , vexpandps(zmm28, k0, zmmword_ptr(rdx, 512))),
        INSTRUCTION_TEST(62627D4888627F              , vexpandps(zmm28, k0, zmmword_ptr(rdx, 508))),
        INSTRUCTION_TEST(62627D4888A2FCFDFFFF        , vexpandps(zmm28, k0, zmmword_ptr(rdx, -516))),
        INSTRUCTION_TEST(62627D488821                , vexpandps(zmm28, k0, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62F27D4888627F              , vexpandps(zmm4, k0, zmmword_ptr(rdx, 508))),
        INSTRUCTION_TEST(62A27D4888CA                , vexpandps(zmm17, k0, zmm18)),
        INSTRUCTION_TEST(62227D4888A4F034120000      , vexpandps(zmm28, k0, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62F27D0888CA                , vexpandps(xmm1, k0, xmm2)),
        INSTRUCTION_TEST(62F27D08884C1A20            , vexpandps(xmm1, k0, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F27D2888CA                , vexpandps(ymm1, k0, ymm2)),
        INSTRUCTION_TEST(62F27D48886280              , vexpandps(zmm4, k0, zmmword_ptr(rdx, -512))),
        INSTRUCTION_TEST(62F27D4888CA                , vexpandps(zmm1, k0, zmm2)),
        INSTRUCTION_TEST(62F27D48884C1A20            , vexpandps(zmm1, k0, zmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F27D28884C1A20            , vexpandps(ymm1, k0, ymmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62527D4888F1                , vexpandps(zmm14, k0, zmm9)),
        INSTRUCTION_TEST(C4E37D19D101                , vextractf128(xmm1, ymm2, Imm(1))),
        INSTRUCTION_TEST(C4E37D199C118000000001      , vextractf128(xmmword_ptr(rcx, rdx, 1, 128), ymm3, Imm(1))),
        INSTRUCTION_TEST(62E37D4819B2000800007B      , vextractf32x4(xmmword_ptr(rdx, 2048), k0, zmm22, Imm(123))),
        INSTRUCTION_TEST(62E37D4819A2000800007B      , vextractf32x4(xmmword_ptr(rdx, 2048), k0, zmm20, Imm(123))),
        INSTRUCTION_TEST(62E37D4819727F7B            , vextractf32x4(xmmword_ptr(rdx, 2032), k0, zmm22, Imm(123))),
        INSTRUCTION_TEST(62E37D4819627F7B            , vextractf32x4(xmmword_ptr(rdx, 2032), k0, zmm20, Imm(123))),
        INSTRUCTION_TEST(62E37D4819B2F0F7FFFF7B      , vextractf32x4(xmmword_ptr(rdx, -2064), k0, zmm22, Imm(123))),
        INSTRUCTION_TEST(62E37D4819A2F0F7FFFF7B      , vextractf32x4(xmmword_ptr(rdx, -2064), k0, zmm20, Imm(123))),
        INSTRUCTION_TEST(62E37D481962807B            , vextractf32x4(xmmword_ptr(rdx, -2048), k0, zmm20, Imm(123))),
        INSTRUCTION_TEST(62E37D481931AB              , vextractf32x4(xmmword_ptr(rcx), k0, zmm22, Imm(171))),
        INSTRUCTION_TEST(62E37D4819317B              , vextractf32x4(xmmword_ptr(rcx), k0, zmm22, Imm(123))),
        INSTRUCTION_TEST(62E37D481921AB              , vextractf32x4(xmmword_ptr(rcx), k0, zmm20, Imm(171))),
        INSTRUCTION_TEST(62E37D481972807B            , vextractf32x4(xmmword_ptr(rdx, -2048), k0, zmm22, Imm(123))),
        INSTRUCTION_TEST(62F37D48195C110801          , vextractf32x4(xmmword_ptr(rcx, rdx, 1, 128), k0, zmm3, Imm(1))),
        INSTRUCTION_TEST(62F37D28195C110801          , vextractf32x4(xmmword_ptr(rcx, rdx, 1, 128), k0, ymm3, Imm(1))),
        INSTRUCTION_TEST(62A37D4819B4F0341200007B    , vextractf32x4(xmmword_ptr(rax, r14, 8, 4660), k0, zmm22, Imm(123))),
        INSTRUCTION_TEST(62A37D4819A4F0230100007B    , vextractf32x4(xmmword_ptr(rax, r14, 8, 291), k0, zmm20, Imm(123))),
        INSTRUCTION_TEST(62337D4819D5AB              , vextractf32x4(xmm21, k0, zmm10, Imm(171))),
        INSTRUCTION_TEST(62337D4819D57B              , vextractf32x4(xmm21, k0, zmm10, Imm(123))),
        INSTRUCTION_TEST(62C37D4819EFAB              , vextractf32x4(xmm15, k0, zmm21, Imm(171))),
        INSTRUCTION_TEST(62C37D4819EF7B              , vextractf32x4(xmm15, k0, zmm21, Imm(123))),
        INSTRUCTION_TEST(62F37D4819D101              , vextractf32x4(xmm1, k0, zmm2, Imm(1))),
        INSTRUCTION_TEST(62F37D2819D101              , vextractf32x4(xmm1, k0, ymm2, Imm(1))),
        INSTRUCTION_TEST(62E37D4819217B              , vextractf32x4(xmmword_ptr(rcx), k0, zmm20, Imm(123))),
        INSTRUCTION_TEST(62F37D481BD101              , vextractf32x8(ymm1, k0, zmm2, Imm(1))),
        INSTRUCTION_TEST(62F37D481B5C110401          , vextractf32x8(ymmword_ptr(rcx, rdx, 1, 128), k0, zmm3, Imm(1))),
        INSTRUCTION_TEST(62F3FD48195C110801          , vextractf64x2(xmmword_ptr(rcx, rdx, 1, 128), k0, zmm3, Imm(1))),
        INSTRUCTION_TEST(62F3FD28195C110801          , vextractf64x2(xmmword_ptr(rcx, rdx, 1, 128), k0, ymm3, Imm(1))),
        INSTRUCTION_TEST(62F3FD2819D101              , vextractf64x2(xmm1, k0, ymm2, Imm(1))),
        INSTRUCTION_TEST(62F3FD4819D101              , vextractf64x2(xmm1, k0, zmm2, Imm(1))),
        INSTRUCTION_TEST(62F3FD481B29AB              , vextractf64x4(ymmword_ptr(rcx), k0, zmm5, Imm(171))),
        INSTRUCTION_TEST(62F3FD481B6A7F7B            , vextractf64x4(ymmword_ptr(rdx, 4064), k0, zmm5, Imm(123))),
        INSTRUCTION_TEST(6273FD481B627F7B            , vextractf64x4(ymmword_ptr(rdx, 4064), k0, zmm12, Imm(123))),
        INSTRUCTION_TEST(62F3FD481BAAE0EFFFFF7B      , vextractf64x4(ymmword_ptr(rdx, -4128), k0, zmm5, Imm(123))),
        INSTRUCTION_TEST(6273FD481BA2001000007B      , vextractf64x4(ymmword_ptr(rdx, 4096), k0, zmm12, Imm(123))),
        INSTRUCTION_TEST(6273FD481BA2E0EFFFFF7B      , vextractf64x4(ymmword_ptr(rdx, -4128), k0, zmm12, Imm(123))),
        INSTRUCTION_TEST(62F3FD481B6A807B            , vextractf64x4(ymmword_ptr(rdx, -4096), k0, zmm5, Imm(123))),
        INSTRUCTION_TEST(62F3FD481BAA001000007B      , vextractf64x4(ymmword_ptr(rdx, 4096), k0, zmm5, Imm(123))),
        INSTRUCTION_TEST(6273FD481B62807B            , vextractf64x4(ymmword_ptr(rdx, -4096), k0, zmm12, Imm(123))),
        INSTRUCTION_TEST(62F3FD481B297B              , vextractf64x4(ymmword_ptr(rcx), k0, zmm5, Imm(123))),
        INSTRUCTION_TEST(6233FD481BA4F0341200007B    , vextractf64x4(ymmword_ptr(rax, r14, 8, 4660), k0, zmm12, Imm(123))),
        INSTRUCTION_TEST(6273FD481B217B              , vextractf64x4(ymmword_ptr(rcx), k0, zmm12, Imm(123))),
        INSTRUCTION_TEST(6273FD481B21AB              , vextractf64x4(ymmword_ptr(rcx), k0, zmm12, Imm(171))),
        INSTRUCTION_TEST(62F3FD481BD101              , vextractf64x4(ymm1, k0, zmm2, Imm(1))),
        INSTRUCTION_TEST(6243FD481BC37B              , vextractf64x4(ymm11, k0, zmm24, Imm(123))),
        INSTRUCTION_TEST(62D3FD481BEE7B              , vextractf64x4(ymm14, k0, zmm5, Imm(123))),
        INSTRUCTION_TEST(62D3FD481BEEAB              , vextractf64x4(ymm14, k0, zmm5, Imm(171))),
        INSTRUCTION_TEST(62B3FD481BACF0230100007B    , vextractf64x4(ymmword_ptr(rax, r14, 8, 291), k0, zmm5, Imm(123))),
        INSTRUCTION_TEST(62F3FD481B5C110401          , vextractf64x4(ymmword_ptr(rcx, rdx, 1, 128), k0, zmm3, Imm(1))),
        INSTRUCTION_TEST(6243FD481BC3AB              , vextractf64x4(ymm11, k0, zmm24, Imm(171))),
        INSTRUCTION_TEST(C4E37D39D101                , vextracti128(xmm1, ymm2, Imm(1))),
        INSTRUCTION_TEST(C4E37D399C118000000001      , vextracti128(xmmword_ptr(rcx, rdx, 1, 128), ymm3, Imm(1))),
        INSTRUCTION_TEST(62637D4839297B              , vextracti32x4(xmmword_ptr(rcx), k0, zmm29, Imm(123))),
        INSTRUCTION_TEST(62637D483929AB              , vextracti32x4(xmmword_ptr(rcx), k0, zmm29, Imm(171))),
        INSTRUCTION_TEST(62F37D48394A807B            , vextracti32x4(xmmword_ptr(rdx, -2048), k0, zmm1, Imm(123))),
        INSTRUCTION_TEST(62637D48396A807B            , vextracti32x4(xmmword_ptr(rdx, -2048), k0, zmm29, Imm(123))),
        INSTRUCTION_TEST(62637D4839AAF0F7FFFF7B      , vextracti32x4(xmmword_ptr(rdx, -2064), k0, zmm29, Imm(123))),
        INSTRUCTION_TEST(62F37D48394A7F7B            , vextracti32x4(xmmword_ptr(rdx, 2032), k0, zmm1, Imm(123))),
        INSTRUCTION_TEST(62637D48396A7F7B            , vextracti32x4(xmmword_ptr(rdx, 2032), k0, zmm29, Imm(123))),
        INSTRUCTION_TEST(62F37D48398A000800007B      , vextracti32x4(xmmword_ptr(rdx, 2048), k0, zmm1, Imm(123))),
        INSTRUCTION_TEST(62637D4839AA000800007B      , vextracti32x4(xmmword_ptr(rdx, 2048), k0, zmm29, Imm(123))),
        INSTRUCTION_TEST(62F37D48398AF0F7FFFF7B      , vextracti32x4(xmmword_ptr(rdx, -2064), k0, zmm1, Imm(123))),
        INSTRUCTION_TEST(62F37D483909AB              , vextracti32x4(xmmword_ptr(rcx), k0, zmm1, Imm(171))),
        INSTRUCTION_TEST(62F37D48395C110801          , vextracti32x4(xmmword_ptr(rcx, rdx, 1, 128), k0, zmm3, Imm(1))),
        INSTRUCTION_TEST(62F37D28395C110801          , vextracti32x4(xmmword_ptr(rcx, rdx, 1, 128), k0, ymm3, Imm(1))),
        INSTRUCTION_TEST(62B37D48398CF0341200007B    , vextracti32x4(xmmword_ptr(rax, r14, 8, 4660), k0, zmm1, Imm(123))),
        INSTRUCTION_TEST(62137D4839E5AB              , vextracti32x4(xmm29, k0, zmm12, Imm(171))),
        INSTRUCTION_TEST(62137D4839E57B              , vextracti32x4(xmm29, k0, zmm12, Imm(123))),
        INSTRUCTION_TEST(62C37D4839C5AB              , vextracti32x4(xmm13, k0, zmm16, Imm(171))),
        INSTRUCTION_TEST(62C37D4839C57B              , vextracti32x4(xmm13, k0, zmm16, Imm(123))),
        INSTRUCTION_TEST(62F37D4839D101              , vextracti32x4(xmm1, k0, zmm2, Imm(1))),
        INSTRUCTION_TEST(62F37D2839D101              , vextracti32x4(xmm1, k0, ymm2, Imm(1))),
        INSTRUCTION_TEST(62F37D4839097B              , vextracti32x4(xmmword_ptr(rcx), k0, zmm1, Imm(123))),
        INSTRUCTION_TEST(62237D4839ACF0230100007B    , vextracti32x4(xmmword_ptr(rax, r14, 8, 291), k0, zmm29, Imm(123))),
        INSTRUCTION_TEST(62F37D483B5C110401          , vextracti32x8(ymmword_ptr(rcx, rdx, 1, 128), k0, zmm3, Imm(1))),
        INSTRUCTION_TEST(62F37D483BD101              , vextracti32x8(ymm1, k0, zmm2, Imm(1))),
        INSTRUCTION_TEST(62F3FD48395C110801          , vextracti64x2(xmmword_ptr(rcx, rdx, 1, 128), k0, zmm3, Imm(1))),
        INSTRUCTION_TEST(62F3FD28395C110801          , vextracti64x2(xmmword_ptr(rcx, rdx, 1, 128), k0, ymm3, Imm(1))),
        INSTRUCTION_TEST(62F3FD2839D101              , vextracti64x2(xmm1, k0, ymm2, Imm(1))),
        INSTRUCTION_TEST(62F3FD4839D101              , vextracti64x2(xmm1, k0, zmm2, Imm(1))),
        INSTRUCTION_TEST(62F3FD483B92001000007B      , vextracti64x4(ymmword_ptr(rdx, 4096), k0, zmm2, Imm(123))),
        INSTRUCTION_TEST(6263FD483B727F7B            , vextracti64x4(ymmword_ptr(rdx, 4064), k0, zmm30, Imm(123))),
        INSTRUCTION_TEST(62F3FD483B527F7B            , vextracti64x4(ymmword_ptr(rdx, 4064), k0, zmm2, Imm(123))),
        INSTRUCTION_TEST(6263FD483BB2E0EFFFFF7B      , vextracti64x4(ymmword_ptr(rdx, -4128), k0, zmm30, Imm(123))),
        INSTRUCTION_TEST(62F3FD483B92E0EFFFFF7B      , vextracti64x4(ymmword_ptr(rdx, -4128), k0, zmm2, Imm(123))),
        INSTRUCTION_TEST(6263FD483B72807B            , vextracti64x4(ymmword_ptr(rdx, -4096), k0, zmm30, Imm(123))),
        INSTRUCTION_TEST(62F3FD483B52807B            , vextracti64x4(ymmword_ptr(rdx, -4096), k0, zmm2, Imm(123))),
        INSTRUCTION_TEST(6263FD483B31AB              , vextracti64x4(ymmword_ptr(rcx), k0, zmm30, Imm(171))),
        INSTRUCTION_TEST(6263FD483B317B              , vextracti64x4(ymmword_ptr(rcx), k0, zmm30, Imm(123))),
        INSTRUCTION_TEST(6263FD483BB2001000007B      , vextracti64x4(ymmword_ptr(rdx, 4096), k0, zmm30, Imm(123))),
        INSTRUCTION_TEST(62F3FD483B117B              , vextracti64x4(ymmword_ptr(rcx), k0, zmm2, Imm(123))),
        INSTRUCTION_TEST(62F3FD483B5C110401          , vextracti64x4(ymmword_ptr(rcx, rdx, 1, 128), k0, zmm3, Imm(1))),
        INSTRUCTION_TEST(62B3FD483B94F0341200007B    , vextracti64x4(ymmword_ptr(rax, r14, 8, 4660), k0, zmm2, Imm(123))),
        INSTRUCTION_TEST(6223FD483BB4F0230100007B    , vextracti64x4(ymmword_ptr(rax, r14, 8, 291), k0, zmm30, Imm(123))),
        INSTRUCTION_TEST(62E3FD483BF5AB              , vextracti64x4(ymm5, k0, zmm22, Imm(171))),
        INSTRUCTION_TEST(62E3FD483BF57B              , vextracti64x4(ymm5, k0, zmm22, Imm(123))),
        INSTRUCTION_TEST(62C3FD483BC5AB              , vextracti64x4(ymm13, k0, zmm16, Imm(171))),
        INSTRUCTION_TEST(62C3FD483BC57B              , vextracti64x4(ymm13, k0, zmm16, Imm(123))),
        INSTRUCTION_TEST(62F3FD483BD101              , vextracti64x4(ymm1, k0, zmm2, Imm(1))),
        INSTRUCTION_TEST(62F3FD483B11AB              , vextracti64x4(ymmword_ptr(rcx), k0, zmm2, Imm(171))),
        INSTRUCTION_TEST(62437D0817C07B              , vextractps(r8d, xmm24, Imm(123))),
        INSTRUCTION_TEST(C4E37917D101                , vextractps(ecx, xmm2, Imm(1))),
        INSTRUCTION_TEST(62637D0817C0AB              , vextractps(eax, xmm24, Imm(171))),
        INSTRUCTION_TEST(62637D0817C07B              , vextractps(eax, xmm24, Imm(123))),
        INSTRUCTION_TEST(62637D081782000200007B      , vextractps(dword_ptr(rdx, 512), xmm24, Imm(123))),
        INSTRUCTION_TEST(62637D0817427F7B            , vextractps(dword_ptr(rdx, 508), xmm24, Imm(123))),
        INSTRUCTION_TEST(62637D081742807B            , vextractps(dword_ptr(rdx, -512), xmm24, Imm(123))),
        INSTRUCTION_TEST(62637D0817017B              , vextractps(dword_ptr(rcx), xmm24, Imm(123))),
        INSTRUCTION_TEST(C4E379179C118000000001      , vextractps(dword_ptr(rcx, rdx, 1, 128), xmm3, Imm(1))),
        INSTRUCTION_TEST(62237D081784F0230100007B    , vextractps(dword_ptr(rax, r14, 8, 291), xmm24, Imm(123))),
        INSTRUCTION_TEST(62637D081782FCFDFFFF7B      , vextractps(dword_ptr(rdx, -516), xmm24, Imm(123))),
        INSTRUCTION_TEST(62E28548987A7F              , vfmadd132pd(zmm23, k0, zmm15, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(6262FD40989200200000        , vfmadd132pd(zmm26, k0, zmm16, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(6262FD4098527F              , vfmadd132pd(zmm26, k0, zmm16, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(6262FD409892C0DFFFFF        , vfmadd132pd(zmm26, k0, zmm16, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(6262FD40985280              , vfmadd132pd(zmm26, k0, zmm16, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(6262FD409811                , vfmadd132pd(zmm26, k0, zmm16, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(6222FD409894F023010000      , vfmadd132pd(zmm26, k0, zmm16, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(6222FD4098D5                , vfmadd132pd(zmm26, k0, zmm16, zmm21)),
        INSTRUCTION_TEST(62E2854898BA00200000        , vfmadd132pd(zmm23, k0, zmm15, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62E2854898BAC0DFFFFF        , vfmadd132pd(zmm23, k0, zmm15, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62A2854898BCF034120000      , vfmadd132pd(zmm23, k0, zmm15, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62E285489839                , vfmadd132pd(zmm23, k0, zmm15, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(C4E2E998CB                  , vfmadd132pd(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(C4E2E9988C2B80000000        , vfmadd132pd(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E2ED98CB                  , vfmadd132pd(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(C4E2ED988C2B80000000        , vfmadd132pd(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62E28548987A80              , vfmadd132pd(zmm23, k0, zmm15, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62F2ED48984C2B02            , vfmadd132pd(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62C2854898FB                , vfmadd132pd(zmm23, k0, zmm15, zmm11)),
        INSTRUCTION_TEST(62F2ED4898CB                , vfmadd132pd(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(62E23540989A00200000        , vfmadd132ps(zmm19, k0, zmm25, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62F25D40988AC0DFFFFF        , vfmadd132ps(zmm1, k0, zmm20, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62E23540989AC0DFFFFF        , vfmadd132ps(zmm19, k0, zmm25, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62E23540985A80              , vfmadd132ps(zmm19, k0, zmm25, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62E235409819                , vfmadd132ps(zmm19, k0, zmm25, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62A23540989CF034120000      , vfmadd132ps(zmm19, k0, zmm25, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62A2354098D9                , vfmadd132ps(zmm19, k0, zmm25, zmm17)),
        INSTRUCTION_TEST(62F25D40988A00200000        , vfmadd132ps(zmm1, k0, zmm20, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62F25D40984A7F              , vfmadd132ps(zmm1, k0, zmm20, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62E23540985A7F              , vfmadd132ps(zmm19, k0, zmm25, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62F25D40984A80              , vfmadd132ps(zmm1, k0, zmm20, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62B25D40988CF023010000      , vfmadd132ps(zmm1, k0, zmm20, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62925D4098C9                , vfmadd132ps(zmm1, k0, zmm20, zmm25)),
        INSTRUCTION_TEST(62F26D48984C2B02            , vfmadd132ps(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F26D4898CB                , vfmadd132ps(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(C4E26D988C2B80000000        , vfmadd132ps(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E26D98CB                  , vfmadd132ps(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(C4E269988C2B80000000        , vfmadd132ps(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E26998CB                  , vfmadd132ps(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(62F25D409809                , vfmadd132ps(zmm1, k0, zmm20, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(6262F5009921                , vfmadd132sd(xmm28, k0, xmm17, qword_ptr(rcx))),
        INSTRUCTION_TEST(6262F500996280              , vfmadd132sd(xmm28, k0, xmm17, qword_ptr(rdx, -1024))),
        INSTRUCTION_TEST(6262F50099A2F8FBFFFF        , vfmadd132sd(xmm28, k0, xmm17, qword_ptr(rdx, -1032))),
        INSTRUCTION_TEST(6222F50099A4F023010000      , vfmadd132sd(xmm28, k0, xmm17, qword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(6262F50099A200040000        , vfmadd132sd(xmm28, k0, xmm17, qword_ptr(rdx, 1024))),
        INSTRUCTION_TEST(6262F50099E3                , vfmadd132sd(xmm28, k0, xmm17, xmm3)),
        INSTRUCTION_TEST(6262F50099627F              , vfmadd132sd(xmm28, k0, xmm17, qword_ptr(rdx, 1016))),
        INSTRUCTION_TEST(62C2CD0099D3                , vfmadd132sd(xmm18, k0, xmm22, xmm11)),
        INSTRUCTION_TEST(62A2CD009994F034120000      , vfmadd132sd(xmm18, k0, xmm22, qword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62E2CD0099527F              , vfmadd132sd(xmm18, k0, xmm22, qword_ptr(rdx, 1016))),
        INSTRUCTION_TEST(62E2CD009992F8FBFFFF        , vfmadd132sd(xmm18, k0, xmm22, qword_ptr(rdx, -1032))),
        INSTRUCTION_TEST(62E2CD00995280              , vfmadd132sd(xmm18, k0, xmm22, qword_ptr(rdx, -1024))),
        INSTRUCTION_TEST(62E2CD009911                , vfmadd132sd(xmm18, k0, xmm22, qword_ptr(rcx))),
        INSTRUCTION_TEST(C4E2E999CB                  , vfmadd132sd(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(C4E2E9998C2B80000000        , vfmadd132sd(xmm1, xmm2, qword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62E2CD00999200040000        , vfmadd132sd(xmm18, k0, xmm22, qword_ptr(rdx, 1024))),
        INSTRUCTION_TEST(6222750099F6                , vfmadd132ss(xmm30, k0, xmm17, xmm22)),
        INSTRUCTION_TEST(6262750099B200020000        , vfmadd132ss(xmm30, k0, xmm17, dword_ptr(rdx, 512))),
        INSTRUCTION_TEST(6262750099727F              , vfmadd132ss(xmm30, k0, xmm17, dword_ptr(rdx, 508))),
        INSTRUCTION_TEST(6262750099B2FCFDFFFF        , vfmadd132ss(xmm30, k0, xmm17, dword_ptr(rdx, -516))),
        INSTRUCTION_TEST(62627500997280              , vfmadd132ss(xmm30, k0, xmm17, dword_ptr(rdx, -512))),
        INSTRUCTION_TEST(626275009931                , vfmadd132ss(xmm30, k0, xmm17, dword_ptr(rcx))),
        INSTRUCTION_TEST(6222750099B4F023010000      , vfmadd132ss(xmm30, k0, xmm17, dword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62625D0099D7                , vfmadd132ss(xmm26, k0, xmm20, xmm7)),
        INSTRUCTION_TEST(62625D0099527F              , vfmadd132ss(xmm26, k0, xmm20, dword_ptr(rdx, 508))),
        INSTRUCTION_TEST(62625D009992FCFDFFFF        , vfmadd132ss(xmm26, k0, xmm20, dword_ptr(rdx, -516))),
        INSTRUCTION_TEST(62625D00995280              , vfmadd132ss(xmm26, k0, xmm20, dword_ptr(rdx, -512))),
        INSTRUCTION_TEST(62625D009911                , vfmadd132ss(xmm26, k0, xmm20, dword_ptr(rcx))),
        INSTRUCTION_TEST(62225D009994F034120000      , vfmadd132ss(xmm26, k0, xmm20, dword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(C4E26999CB                  , vfmadd132ss(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(C4E269998C2B80000000        , vfmadd132ss(xmm1, xmm2, dword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62625D00999200020000        , vfmadd132ss(xmm26, k0, xmm20, dword_ptr(rdx, 512))),
        INSTRUCTION_TEST(62E2FD40A89200200000        , vfmadd213pd(zmm18, k0, zmm16, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62F2F540A8F1                , vfmadd213pd(zmm6, k0, zmm17, zmm1)),
        INSTRUCTION_TEST(62B2F540A8B4F034120000      , vfmadd213pd(zmm6, k0, zmm17, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62F2F540A8727F              , vfmadd213pd(zmm6, k0, zmm17, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62F2F540A87280              , vfmadd213pd(zmm6, k0, zmm17, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62F2F540A8B2C0DFFFFF        , vfmadd213pd(zmm6, k0, zmm17, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62F2F540A8B200200000        , vfmadd213pd(zmm6, k0, zmm17, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62F2F540A831                , vfmadd213pd(zmm6, k0, zmm17, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62E2FD40A892C0DFFFFF        , vfmadd213pd(zmm18, k0, zmm16, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62E2FD40A8527F              , vfmadd213pd(zmm18, k0, zmm16, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62E2FD40A811                , vfmadd213pd(zmm18, k0, zmm16, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62A2FD40A894F023010000      , vfmadd213pd(zmm18, k0, zmm16, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(6282FD40A8D1                , vfmadd213pd(zmm18, k0, zmm16, zmm25)),
        INSTRUCTION_TEST(62F2ED48A84C2B02            , vfmadd213pd(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F2ED48A8CB                , vfmadd213pd(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(C4E2EDA88C2B80000000        , vfmadd213pd(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E2EDA8CB                  , vfmadd213pd(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(C4E2E9A88C2B80000000        , vfmadd213pd(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E2E9A8CB                  , vfmadd213pd(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(62E2FD40A85280              , vfmadd213pd(zmm18, k0, zmm16, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62E26540A8627F              , vfmadd213ps(zmm20, k0, zmm19, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62E26540A8A200200000        , vfmadd213ps(zmm20, k0, zmm19, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62923D40A8E8                , vfmadd213ps(zmm5, k0, zmm24, zmm24)),
        INSTRUCTION_TEST(62B23D40A8ACF034120000      , vfmadd213ps(zmm5, k0, zmm24, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62F23D40A8AA00200000        , vfmadd213ps(zmm5, k0, zmm24, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62F23D40A86A80              , vfmadd213ps(zmm5, k0, zmm24, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62F23D40A8AAC0DFFFFF        , vfmadd213ps(zmm5, k0, zmm24, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62F23D40A86A7F              , vfmadd213ps(zmm5, k0, zmm24, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62F23D40A829                , vfmadd213ps(zmm5, k0, zmm24, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62E26540A8A2C0DFFFFF        , vfmadd213ps(zmm20, k0, zmm19, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62E26540A821                , vfmadd213ps(zmm20, k0, zmm19, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62A26540A8A4F023010000      , vfmadd213ps(zmm20, k0, zmm19, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62C26540A8E6                , vfmadd213ps(zmm20, k0, zmm19, zmm14)),
        INSTRUCTION_TEST(62F26D48A84C2B02            , vfmadd213ps(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F26D48A8CB                , vfmadd213ps(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(C4E26DA88C2B80000000        , vfmadd213ps(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E26DA8CB                  , vfmadd213ps(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(C4E269A88C2B80000000        , vfmadd213ps(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E269A8CB                  , vfmadd213ps(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(62E26540A86280              , vfmadd213ps(zmm20, k0, zmm19, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62028D08A9F4                , vfmadd213sd(xmm30, k0, xmm14, xmm28)),
        INSTRUCTION_TEST(62628D08A9727F              , vfmadd213sd(xmm30, k0, xmm14, qword_ptr(rdx, 1016))),
        INSTRUCTION_TEST(62628D08A9B2F8FBFFFF        , vfmadd213sd(xmm30, k0, xmm14, qword_ptr(rdx, -1032))),
        INSTRUCTION_TEST(62628D08A9B200040000        , vfmadd213sd(xmm30, k0, xmm14, qword_ptr(rdx, 1024))),
        INSTRUCTION_TEST(62628D08A931                , vfmadd213sd(xmm30, k0, xmm14, qword_ptr(rcx))),
        INSTRUCTION_TEST(62228D08A9B4F034120000      , vfmadd213sd(xmm30, k0, xmm14, qword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(6242AD00A9C5                , vfmadd213sd(xmm24, k0, xmm26, xmm13)),
        INSTRUCTION_TEST(62628D08A97280              , vfmadd213sd(xmm30, k0, xmm14, qword_ptr(rdx, -1024))),
        INSTRUCTION_TEST(6262AD00A9427F              , vfmadd213sd(xmm24, k0, xmm26, qword_ptr(rdx, 1016))),
        INSTRUCTION_TEST(6262AD00A982F8FBFFFF        , vfmadd213sd(xmm24, k0, xmm26, qword_ptr(rdx, -1032))),
        INSTRUCTION_TEST(6262AD00A94280              , vfmadd213sd(xmm24, k0, xmm26, qword_ptr(rdx, -1024))),
        INSTRUCTION_TEST(6262AD00A901                , vfmadd213sd(xmm24, k0, xmm26, qword_ptr(rcx))),
        INSTRUCTION_TEST(6222AD00A984F023010000      , vfmadd213sd(xmm24, k0, xmm26, qword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(C4E2E9A9CB                  , vfmadd213sd(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(C4E2E9A98C2B80000000        , vfmadd213sd(xmm1, xmm2, qword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(6262AD00A98200040000        , vfmadd213sd(xmm24, k0, xmm26, qword_ptr(rdx, 1024))),
        INSTRUCTION_TEST(62224D00A9F0                , vfmadd213ss(xmm30, k0, xmm22, xmm16)),
        INSTRUCTION_TEST(62624D00A9B200020000        , vfmadd213ss(xmm30, k0, xmm22, dword_ptr(rdx, 512))),
        INSTRUCTION_TEST(62624D00A9727F              , vfmadd213ss(xmm30, k0, xmm22, dword_ptr(rdx, 508))),
        INSTRUCTION_TEST(62624D00A9B2FCFDFFFF        , vfmadd213ss(xmm30, k0, xmm22, dword_ptr(rdx, -516))),
        INSTRUCTION_TEST(62624D00A97280              , vfmadd213ss(xmm30, k0, xmm22, dword_ptr(rdx, -512))),
        INSTRUCTION_TEST(62624D00A931                , vfmadd213ss(xmm30, k0, xmm22, dword_ptr(rcx))),
        INSTRUCTION_TEST(62224D00A9B4F023010000      , vfmadd213ss(xmm30, k0, xmm22, dword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62823500A9D1                , vfmadd213ss(xmm18, k0, xmm25, xmm25)),
        INSTRUCTION_TEST(62E23500A9527F              , vfmadd213ss(xmm18, k0, xmm25, dword_ptr(rdx, 508))),
        INSTRUCTION_TEST(62E23500A992FCFDFFFF        , vfmadd213ss(xmm18, k0, xmm25, dword_ptr(rdx, -516))),
        INSTRUCTION_TEST(62E23500A95280              , vfmadd213ss(xmm18, k0, xmm25, dword_ptr(rdx, -512))),
        INSTRUCTION_TEST(62E23500A911                , vfmadd213ss(xmm18, k0, xmm25, dword_ptr(rcx))),
        INSTRUCTION_TEST(62A23500A994F034120000      , vfmadd213ss(xmm18, k0, xmm25, dword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(C4E269A9CB                  , vfmadd213ss(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(C4E269A98C2B80000000        , vfmadd213ss(xmm1, xmm2, dword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62E23500A99200020000        , vfmadd213ss(xmm18, k0, xmm25, dword_ptr(rdx, 512))),
        INSTRUCTION_TEST(6262CD48B8727F              , vfmadd231pd(zmm30, k0, zmm6, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(6262CD48B8B200200000        , vfmadd231pd(zmm30, k0, zmm6, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62B29548B8EB                , vfmadd231pd(zmm5, k0, zmm13, zmm19)),
        INSTRUCTION_TEST(62B29548B8ACF034120000      , vfmadd231pd(zmm5, k0, zmm13, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62F29548B86A7F              , vfmadd231pd(zmm5, k0, zmm13, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62F29548B86A80              , vfmadd231pd(zmm5, k0, zmm13, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62F29548B8AAC0DFFFFF        , vfmadd231pd(zmm5, k0, zmm13, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62F29548B829                , vfmadd231pd(zmm5, k0, zmm13, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(6262CD48B8B2C0DFFFFF        , vfmadd231pd(zmm30, k0, zmm6, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62F29548B8AA00200000        , vfmadd231pd(zmm5, k0, zmm13, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(6262CD48B831                , vfmadd231pd(zmm30, k0, zmm6, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(6222CD48B8B4F023010000      , vfmadd231pd(zmm30, k0, zmm6, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(6242CD48B8F1                , vfmadd231pd(zmm30, k0, zmm6, zmm9)),
        INSTRUCTION_TEST(62F2ED48B84C2B02            , vfmadd231pd(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F2ED48B8CB                , vfmadd231pd(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(C4E2EDB88C2B80000000        , vfmadd231pd(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E2EDB8CB                  , vfmadd231pd(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(C4E2E9B88C2B80000000        , vfmadd231pd(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E2E9B8CB                  , vfmadd231pd(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(6262CD48B87280              , vfmadd231pd(zmm30, k0, zmm6, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62F27548B89200200000        , vfmadd231ps(zmm2, k0, zmm1, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62021D40B8D9                , vfmadd231ps(zmm27, k0, zmm28, zmm25)),
        INSTRUCTION_TEST(62221D40B89CF023010000      , vfmadd231ps(zmm27, k0, zmm28, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62621D40B89A00200000        , vfmadd231ps(zmm27, k0, zmm28, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62621D40B89AC0DFFFFF        , vfmadd231ps(zmm27, k0, zmm28, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62621D40B85A7F              , vfmadd231ps(zmm27, k0, zmm28, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62F27548B8527F              , vfmadd231ps(zmm2, k0, zmm1, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62621D40B819                , vfmadd231ps(zmm27, k0, zmm28, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62F27548B892C0DFFFFF        , vfmadd231ps(zmm2, k0, zmm1, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62621D40B85A80              , vfmadd231ps(zmm27, k0, zmm28, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62F27548B811                , vfmadd231ps(zmm2, k0, zmm1, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62F27548B85280              , vfmadd231ps(zmm2, k0, zmm1, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(C4E269B88C2B80000000        , vfmadd231ps(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E26DB8CB                  , vfmadd231ps(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(C4E26DB88C2B80000000        , vfmadd231ps(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E269B8CB                  , vfmadd231ps(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(62F26D48B84C2B02            , vfmadd231ps(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62B27548B8D2                , vfmadd231ps(zmm2, k0, zmm1, zmm18)),
        INSTRUCTION_TEST(62B27548B894F034120000      , vfmadd231ps(zmm2, k0, zmm1, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62F26D48B8CB                , vfmadd231ps(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(C4C289B9F6                  , vfmadd231sd(xmm6, xmm14, xmm14)),
        INSTRUCTION_TEST(C4E289B9B200040000          , vfmadd231sd(xmm6, xmm14, qword_ptr(rdx, 1024))),
        INSTRUCTION_TEST(C4E289B9B2F8030000          , vfmadd231sd(xmm6, xmm14, qword_ptr(rdx, 1016))),
        INSTRUCTION_TEST(C4E289B9B2F8FBFFFF          , vfmadd231sd(xmm6, xmm14, qword_ptr(rdx, -1032))),
        INSTRUCTION_TEST(C4E289B9B200FCFFFF          , vfmadd231sd(xmm6, xmm14, qword_ptr(rdx, -1024))),
        INSTRUCTION_TEST(C4E289B931                  , vfmadd231sd(xmm6, xmm14, qword_ptr(rcx))),
        INSTRUCTION_TEST(C4A289B9B4F034120000        , vfmadd231sd(xmm6, xmm14, qword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(C4C2F1B9E6                  , vfmadd231sd(xmm4, xmm1, xmm14)),
        INSTRUCTION_TEST(C4E2F1B9A2F8030000          , vfmadd231sd(xmm4, xmm1, qword_ptr(rdx, 1016))),
        INSTRUCTION_TEST(C4E2F1B9A2F8FBFFFF          , vfmadd231sd(xmm4, xmm1, qword_ptr(rdx, -1032))),
        INSTRUCTION_TEST(C4E2F1B9A200FCFFFF          , vfmadd231sd(xmm4, xmm1, qword_ptr(rdx, -1024))),
        INSTRUCTION_TEST(C4E2F1B921                  , vfmadd231sd(xmm4, xmm1, qword_ptr(rcx))),
        INSTRUCTION_TEST(C4A2F1B9A4F023010000        , vfmadd231sd(xmm4, xmm1, qword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(C4E2E9B9CB                  , vfmadd231sd(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(C4E2E9B98C2B80000000        , vfmadd231sd(xmm1, xmm2, qword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E2F1B9A200040000          , vfmadd231sd(xmm4, xmm1, qword_ptr(rdx, 1024))),
        INSTRUCTION_TEST(62220508B9ACF023010000      , vfmadd231ss(xmm29, k0, xmm15, dword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62620508B9AA00020000        , vfmadd231ss(xmm29, k0, xmm15, dword_ptr(rdx, 512))),
        INSTRUCTION_TEST(62420508B9EA                , vfmadd231ss(xmm29, k0, xmm15, xmm10)),
        INSTRUCTION_TEST(62620508B96A7F              , vfmadd231ss(xmm29, k0, xmm15, dword_ptr(rdx, 508))),
        INSTRUCTION_TEST(62620508B9AAFCFDFFFF        , vfmadd231ss(xmm29, k0, xmm15, dword_ptr(rdx, -516))),
        INSTRUCTION_TEST(62620508B96A80              , vfmadd231ss(xmm29, k0, xmm15, dword_ptr(rdx, -512))),
        INSTRUCTION_TEST(62620508B929                , vfmadd231ss(xmm29, k0, xmm15, dword_ptr(rcx))),
        INSTRUCTION_TEST(62220D08B9D3                , vfmadd231ss(xmm26, k0, xmm14, xmm19)),
        INSTRUCTION_TEST(C4E269B9CB                  , vfmadd231ss(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(62620D08B9527F              , vfmadd231ss(xmm26, k0, xmm14, dword_ptr(rdx, 508))),
        INSTRUCTION_TEST(62620D08B992FCFDFFFF        , vfmadd231ss(xmm26, k0, xmm14, dword_ptr(rdx, -516))),
        INSTRUCTION_TEST(62620D08B95280              , vfmadd231ss(xmm26, k0, xmm14, dword_ptr(rdx, -512))),
        INSTRUCTION_TEST(62620D08B911                , vfmadd231ss(xmm26, k0, xmm14, dword_ptr(rcx))),
        INSTRUCTION_TEST(62220D08B994F034120000      , vfmadd231ss(xmm26, k0, xmm14, dword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(C4E269B98C2B80000000        , vfmadd231ss(xmm1, xmm2, dword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62620D08B99200020000        , vfmadd231ss(xmm26, k0, xmm14, dword_ptr(rdx, 512))),
        INSTRUCTION_TEST(C4E36969CB40                , vfmaddpd(xmm1, xmm2, xmm3, xmm4)),
        INSTRUCTION_TEST(C4E3E9698C358000000030      , vfmaddpd(xmm1, xmm2, xmm3, xmmword_ptr(rbp, rsi, 1, 128))),
        INSTRUCTION_TEST(C4E369698C2B8000000060      , vfmaddpd(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128), xmm6)),
        INSTRUCTION_TEST(C4E36D69CB40                , vfmaddpd(ymm1, ymm2, ymm3, ymm4)),
        INSTRUCTION_TEST(C4E3ED698C358000000030      , vfmaddpd(ymm1, ymm2, ymm3, ymmword_ptr(rbp, rsi, 1, 128))),
        INSTRUCTION_TEST(C4E36D698C2B8000000060      , vfmaddpd(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128), ymm6)),
        INSTRUCTION_TEST(C4E36D688C2B8000000060      , vfmaddps(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128), ymm6)),
        INSTRUCTION_TEST(C4E3ED688C358000000030      , vfmaddps(ymm1, ymm2, ymm3, ymmword_ptr(rbp, rsi, 1, 128))),
        INSTRUCTION_TEST(C4E36D68CB40                , vfmaddps(ymm1, ymm2, ymm3, ymm4)),
        INSTRUCTION_TEST(C4E3E9688C358000000030      , vfmaddps(xmm1, xmm2, xmm3, xmmword_ptr(rbp, rsi, 1, 128))),
        INSTRUCTION_TEST(C4E36968CB40                , vfmaddps(xmm1, xmm2, xmm3, xmm4)),
        INSTRUCTION_TEST(C4E369688C2B8000000060      , vfmaddps(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128), xmm6)),
        INSTRUCTION_TEST(C4E3696B8C2B8000000060      , vfmaddsd(xmm1, xmm2, qword_ptr(rbx, rbp, 1, 128), xmm6)),
        INSTRUCTION_TEST(C4E3E96B8C358000000030      , vfmaddsd(xmm1, xmm2, xmm3, qword_ptr(rbp, rsi, 1, 128))),
        INSTRUCTION_TEST(C4E3696BCB40                , vfmaddsd(xmm1, xmm2, xmm3, xmm4)),
        INSTRUCTION_TEST(C4E3E96A8C358000000030      , vfmaddss(xmm1, xmm2, xmm3, dword_ptr(rbp, rsi, 1, 128))),
        INSTRUCTION_TEST(C4E3696ACB40                , vfmaddss(xmm1, xmm2, xmm3, xmm4)),
        INSTRUCTION_TEST(C4E3696A8C2B8000000060      , vfmaddss(xmm1, xmm2, dword_ptr(rbx, rbp, 1, 128), xmm6)),
        INSTRUCTION_TEST(62E2CD4896A200200000        , vfmaddsub132pd(zmm20, k0, zmm6, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62E2CD4896627F              , vfmaddsub132pd(zmm20, k0, zmm6, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62E2CD4896A2C0DFFFFF        , vfmaddsub132pd(zmm20, k0, zmm6, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62E2CD48966280              , vfmaddsub132pd(zmm20, k0, zmm6, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62E2CD489621                , vfmaddsub132pd(zmm20, k0, zmm6, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62A2CD4896A4F034120000      , vfmaddsub132pd(zmm20, k0, zmm6, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62E2CD4896E2                , vfmaddsub132pd(zmm20, k0, zmm6, zmm2)),
        INSTRUCTION_TEST(62E2B54096A200200000        , vfmaddsub132pd(zmm20, k0, zmm25, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62E2B54096A2C0DFFFFF        , vfmaddsub132pd(zmm20, k0, zmm25, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62E2B54096627F              , vfmaddsub132pd(zmm20, k0, zmm25, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62E2B5409621                , vfmaddsub132pd(zmm20, k0, zmm25, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(C4E2E996CB                  , vfmaddsub132pd(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(C4E2E9968C2B80000000        , vfmaddsub132pd(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E2ED96CB                  , vfmaddsub132pd(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(C4E2ED968C2B80000000        , vfmaddsub132pd(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62E2B540966280              , vfmaddsub132pd(zmm20, k0, zmm25, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62F2ED48964C2B02            , vfmaddsub132pd(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62A2B54096E5                , vfmaddsub132pd(zmm20, k0, zmm25, zmm21)),
        INSTRUCTION_TEST(62A2B54096A4F023010000      , vfmaddsub132pd(zmm20, k0, zmm25, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62F2ED4896CB                , vfmaddsub132pd(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(627235489692C0DFFFFF        , vfmaddsub132ps(zmm10, k0, zmm9, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62624548966A7F              , vfmaddsub132ps(zmm29, k0, zmm7, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(6262454896AAC0DFFFFF        , vfmaddsub132ps(zmm29, k0, zmm7, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62624548966A80              , vfmaddsub132ps(zmm29, k0, zmm7, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(626245489629                , vfmaddsub132ps(zmm29, k0, zmm7, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(6222454896ACF034120000      , vfmaddsub132ps(zmm29, k0, zmm7, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(6242454896EC                , vfmaddsub132ps(zmm29, k0, zmm7, zmm12)),
        INSTRUCTION_TEST(62723548969200200000        , vfmaddsub132ps(zmm10, k0, zmm9, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(6272354896527F              , vfmaddsub132ps(zmm10, k0, zmm9, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(6262454896AA00200000        , vfmaddsub132ps(zmm29, k0, zmm7, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62723548965280              , vfmaddsub132ps(zmm10, k0, zmm9, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(623235489694F023010000      , vfmaddsub132ps(zmm10, k0, zmm9, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(6232354896D4                , vfmaddsub132ps(zmm10, k0, zmm9, zmm20)),
        INSTRUCTION_TEST(62F26D48964C2B02            , vfmaddsub132ps(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F26D4896CB                , vfmaddsub132ps(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(C4E26D968C2B80000000        , vfmaddsub132ps(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E26D96CB                  , vfmaddsub132ps(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(C4E269968C2B80000000        , vfmaddsub132ps(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E26996CB                  , vfmaddsub132ps(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(627235489611                , vfmaddsub132ps(zmm10, k0, zmm9, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(6262CD48A69200200000        , vfmaddsub213pd(zmm26, k0, zmm6, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(6242DD40A6DF                , vfmaddsub213pd(zmm27, k0, zmm20, zmm15)),
        INSTRUCTION_TEST(6262DD40A69A00200000        , vfmaddsub213pd(zmm27, k0, zmm20, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(6262CD48A6527F              , vfmaddsub213pd(zmm26, k0, zmm6, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(6262DD40A65A80              , vfmaddsub213pd(zmm27, k0, zmm20, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(6262DD40A69AC0DFFFFF        , vfmaddsub213pd(zmm27, k0, zmm20, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(6262DD40A65A7F              , vfmaddsub213pd(zmm27, k0, zmm20, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(6262DD40A619                , vfmaddsub213pd(zmm27, k0, zmm20, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(6262CD48A692C0DFFFFF        , vfmaddsub213pd(zmm26, k0, zmm6, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(6222DD40A69CF034120000      , vfmaddsub213pd(zmm27, k0, zmm20, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(6262CD48A611                , vfmaddsub213pd(zmm26, k0, zmm6, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(C4E2E9A6CB                  , vfmaddsub213pd(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(C4E2E9A68C2B80000000        , vfmaddsub213pd(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E2EDA6CB                  , vfmaddsub213pd(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(C4E2EDA68C2B80000000        , vfmaddsub213pd(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(6262CD48A65280              , vfmaddsub213pd(zmm26, k0, zmm6, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62F2ED48A64C2B02            , vfmaddsub213pd(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(6242CD48A6D2                , vfmaddsub213pd(zmm26, k0, zmm6, zmm10)),
        INSTRUCTION_TEST(6222CD48A694F023010000      , vfmaddsub213pd(zmm26, k0, zmm6, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62F2ED48A6CB                , vfmaddsub213pd(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(62E23D40A64A7F              , vfmaddsub213ps(zmm17, k0, zmm24, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62E21548A6B200200000        , vfmaddsub213ps(zmm22, k0, zmm13, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62E21548A6727F              , vfmaddsub213ps(zmm22, k0, zmm13, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62E21548A6B2C0DFFFFF        , vfmaddsub213ps(zmm22, k0, zmm13, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62E21548A67280              , vfmaddsub213ps(zmm22, k0, zmm13, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62E21548A631                , vfmaddsub213ps(zmm22, k0, zmm13, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62A21548A6B4F034120000      , vfmaddsub213ps(zmm22, k0, zmm13, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62C21548A6F2                , vfmaddsub213ps(zmm22, k0, zmm13, zmm10)),
        INSTRUCTION_TEST(62E23D40A68A00200000        , vfmaddsub213ps(zmm17, k0, zmm24, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62E23D40A68AC0DFFFFF        , vfmaddsub213ps(zmm17, k0, zmm24, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62E23D40A64A80              , vfmaddsub213ps(zmm17, k0, zmm24, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62E23D40A609                , vfmaddsub213ps(zmm17, k0, zmm24, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62A23D40A68CF023010000      , vfmaddsub213ps(zmm17, k0, zmm24, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62823D40A6CC                , vfmaddsub213ps(zmm17, k0, zmm24, zmm28)),
        INSTRUCTION_TEST(62F26D48A64C2B02            , vfmaddsub213ps(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F26D48A6CB                , vfmaddsub213ps(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(C4E26DA68C2B80000000        , vfmaddsub213ps(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E26DA6CB                  , vfmaddsub213ps(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(C4E269A68C2B80000000        , vfmaddsub213ps(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E269A6CB                  , vfmaddsub213ps(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(62F2B540B6A200200000        , vfmaddsub231pd(zmm4, k0, zmm25, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(6212A540B6C8                , vfmaddsub231pd(zmm9, k0, zmm27, zmm24)),
        INSTRUCTION_TEST(6232A540B68CF023010000      , vfmaddsub231pd(zmm9, k0, zmm27, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(6272A540B609                , vfmaddsub231pd(zmm9, k0, zmm27, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(6272A540B68A00200000        , vfmaddsub231pd(zmm9, k0, zmm27, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(6272A540B68AC0DFFFFF        , vfmaddsub231pd(zmm9, k0, zmm27, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(6272A540B64A7F              , vfmaddsub231pd(zmm9, k0, zmm27, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62F2B540B6627F              , vfmaddsub231pd(zmm4, k0, zmm25, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(6272A540B64A80              , vfmaddsub231pd(zmm9, k0, zmm27, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62F2B540B6A2C0DFFFFF        , vfmaddsub231pd(zmm4, k0, zmm25, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(6292B540B6E4                , vfmaddsub231pd(zmm4, k0, zmm25, zmm28)),
        INSTRUCTION_TEST(62F2B540B621                , vfmaddsub231pd(zmm4, k0, zmm25, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62B2B540B6A4F034120000      , vfmaddsub231pd(zmm4, k0, zmm25, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62F2ED48B64C2B02            , vfmaddsub231pd(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F2ED48B6CB                , vfmaddsub231pd(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(C4E2EDB68C2B80000000        , vfmaddsub231pd(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E2EDB6CB                  , vfmaddsub231pd(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(C4E2E9B68C2B80000000        , vfmaddsub231pd(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E2E9B6CB                  , vfmaddsub231pd(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(62F2B540B66280              , vfmaddsub231pd(zmm4, k0, zmm25, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62725540B6BA00200000        , vfmaddsub231ps(zmm15, k0, zmm21, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62D24D40B6FA                , vfmaddsub231ps(zmm7, k0, zmm22, zmm10)),
        INSTRUCTION_TEST(62B24D40B6BCF034120000      , vfmaddsub231ps(zmm7, k0, zmm22, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62F24D40B67A7F              , vfmaddsub231ps(zmm7, k0, zmm22, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62F24D40B67A80              , vfmaddsub231ps(zmm7, k0, zmm22, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62F24D40B6BAC0DFFFFF        , vfmaddsub231ps(zmm7, k0, zmm22, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62725540B67A7F              , vfmaddsub231ps(zmm15, k0, zmm21, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62F24D40B6BA00200000        , vfmaddsub231ps(zmm7, k0, zmm22, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62F24D40B639                , vfmaddsub231ps(zmm7, k0, zmm22, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62725540B6BAC0DFFFFF        , vfmaddsub231ps(zmm15, k0, zmm21, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62325540B6FB                , vfmaddsub231ps(zmm15, k0, zmm21, zmm19)),
        INSTRUCTION_TEST(62725540B639                , vfmaddsub231ps(zmm15, k0, zmm21, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62325540B6BCF023010000      , vfmaddsub231ps(zmm15, k0, zmm21, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62F26D48B64C2B02            , vfmaddsub231ps(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F26D48B6CB                , vfmaddsub231ps(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(C4E26DB68C2B80000000        , vfmaddsub231ps(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E26DB6CB                  , vfmaddsub231ps(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(C4E269B68C2B80000000        , vfmaddsub231ps(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E269B6CB                  , vfmaddsub231ps(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(62725540B67A80              , vfmaddsub231ps(zmm15, k0, zmm21, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(C4E3ED5D8C358000000030      , vfmaddsubpd(ymm1, ymm2, ymm3, ymmword_ptr(rbp, rsi, 1, 128))),
        INSTRUCTION_TEST(C4E36D5D8C2B8000000060      , vfmaddsubpd(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128), ymm6)),
        INSTRUCTION_TEST(C4E36D5DCB40                , vfmaddsubpd(ymm1, ymm2, ymm3, ymm4)),
        INSTRUCTION_TEST(C4E3695D8C2B8000000060      , vfmaddsubpd(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128), xmm6)),
        INSTRUCTION_TEST(C4E3E95D8C358000000030      , vfmaddsubpd(xmm1, xmm2, xmm3, xmmword_ptr(rbp, rsi, 1, 128))),
        INSTRUCTION_TEST(C4E3695DCB40                , vfmaddsubpd(xmm1, xmm2, xmm3, xmm4)),
        INSTRUCTION_TEST(C4E3695CCB40                , vfmaddsubps(xmm1, xmm2, xmm3, xmm4)),
        INSTRUCTION_TEST(C4E3E95C8C358000000030      , vfmaddsubps(xmm1, xmm2, xmm3, xmmword_ptr(rbp, rsi, 1, 128))),
        INSTRUCTION_TEST(C4E3695C8C2B8000000060      , vfmaddsubps(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128), xmm6)),
        INSTRUCTION_TEST(C4E36D5CCB40                , vfmaddsubps(ymm1, ymm2, ymm3, ymm4)),
        INSTRUCTION_TEST(C4E3ED5C8C358000000030      , vfmaddsubps(ymm1, ymm2, ymm3, ymmword_ptr(rbp, rsi, 1, 128))),
        INSTRUCTION_TEST(C4E36D5C8C2B8000000060      , vfmaddsubps(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128), ymm6)),
        INSTRUCTION_TEST(622295409ACB                , vfmsub132pd(zmm25, k0, zmm29, zmm19)),
        INSTRUCTION_TEST(622295409A8CF034120000      , vfmsub132pd(zmm25, k0, zmm29, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(626295409A09                , vfmsub132pd(zmm25, k0, zmm29, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(626295409A8A00200000        , vfmsub132pd(zmm25, k0, zmm29, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(626295409A8AC0DFFFFF        , vfmsub132pd(zmm25, k0, zmm29, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(626295409A4A7F              , vfmsub132pd(zmm25, k0, zmm29, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62E29D489AB200200000        , vfmsub132pd(zmm22, k0, zmm12, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(626295409A4A80              , vfmsub132pd(zmm25, k0, zmm29, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62E29D489A727F              , vfmsub132pd(zmm22, k0, zmm12, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62E29D489AB2C0DFFFFF        , vfmsub132pd(zmm22, k0, zmm12, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(C4E2E99ACB                  , vfmsub132pd(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(62E29D489A31                , vfmsub132pd(zmm22, k0, zmm12, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62A29D489AB4F023010000      , vfmsub132pd(zmm22, k0, zmm12, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62829D489AF3                , vfmsub132pd(zmm22, k0, zmm12, zmm27)),
        INSTRUCTION_TEST(62F2ED489A4C2B02            , vfmsub132pd(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F2ED489ACB                , vfmsub132pd(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(C4E2ED9A8C2B80000000        , vfmsub132pd(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E2ED9ACB                  , vfmsub132pd(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(C4E2E99A8C2B80000000        , vfmsub132pd(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62E29D489A7280              , vfmsub132pd(zmm22, k0, zmm12, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62F26D489A4C2B02            , vfmsub132ps(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62224D489ADA                , vfmsub132ps(zmm27, k0, zmm6, zmm18)),
        INSTRUCTION_TEST(62224D489A9CF034120000      , vfmsub132ps(zmm27, k0, zmm6, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62624D489A19                , vfmsub132ps(zmm27, k0, zmm6, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62F26D489ACB                , vfmsub132ps(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(62624D489A9AC0DFFFFF        , vfmsub132ps(zmm27, k0, zmm6, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62624D489A5A7F              , vfmsub132ps(zmm27, k0, zmm6, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62624D489A9A00200000        , vfmsub132ps(zmm27, k0, zmm6, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62624D489A5A80              , vfmsub132ps(zmm27, k0, zmm6, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62F215489A8A00200000        , vfmsub132ps(zmm1, k0, zmm13, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62F215489A4A80              , vfmsub132ps(zmm1, k0, zmm13, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62F215489A8AC0DFFFFF        , vfmsub132ps(zmm1, k0, zmm13, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62F215489A4A7F              , vfmsub132ps(zmm1, k0, zmm13, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(C4E2699A8C2B80000000        , vfmsub132ps(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E26D9ACB                  , vfmsub132ps(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(C4E26D9A8C2B80000000        , vfmsub132ps(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E2699ACB                  , vfmsub132ps(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(62B215489A8CF023010000      , vfmsub132ps(zmm1, k0, zmm13, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62F215489A09                , vfmsub132ps(zmm1, k0, zmm13, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62B215489AC8                , vfmsub132ps(zmm1, k0, zmm13, zmm16)),
        INSTRUCTION_TEST(6282DD009BFC                , vfmsub132sd(xmm23, k0, xmm20, xmm28)),
        INSTRUCTION_TEST(62E2DD009BBA00040000        , vfmsub132sd(xmm23, k0, xmm20, qword_ptr(rdx, 1024))),
        INSTRUCTION_TEST(62E2DD009B7A7F              , vfmsub132sd(xmm23, k0, xmm20, qword_ptr(rdx, 1016))),
        INSTRUCTION_TEST(62E2DD009BBAF8FBFFFF        , vfmsub132sd(xmm23, k0, xmm20, qword_ptr(rdx, -1032))),
        INSTRUCTION_TEST(62E2DD009B7A80              , vfmsub132sd(xmm23, k0, xmm20, qword_ptr(rdx, -1024))),
        INSTRUCTION_TEST(62E2DD009B39                , vfmsub132sd(xmm23, k0, xmm20, qword_ptr(rcx))),
        INSTRUCTION_TEST(62A2DD009BBCF034120000      , vfmsub132sd(xmm23, k0, xmm20, qword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(6212BD089BE3                , vfmsub132sd(xmm12, k0, xmm8, xmm27)),
        INSTRUCTION_TEST(C462B99BA2F8030000          , vfmsub132sd(xmm12, xmm8, qword_ptr(rdx, 1016))),
        INSTRUCTION_TEST(C462B99BA2F8FBFFFF          , vfmsub132sd(xmm12, xmm8, qword_ptr(rdx, -1032))),
        INSTRUCTION_TEST(C462B99BA200FCFFFF          , vfmsub132sd(xmm12, xmm8, qword_ptr(rdx, -1024))),
        INSTRUCTION_TEST(C462B99B21                  , vfmsub132sd(xmm12, xmm8, qword_ptr(rcx))),
        INSTRUCTION_TEST(C422B99BA4F023010000        , vfmsub132sd(xmm12, xmm8, qword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(C4E2E99BCB                  , vfmsub132sd(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(C4E2E99B8C2B80000000        , vfmsub132sd(xmm1, xmm2, qword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C462B99BA200040000          , vfmsub132sd(xmm12, xmm8, qword_ptr(rdx, 1024))),
        INSTRUCTION_TEST(62224D009BB4F023010000      , vfmsub132ss(xmm30, k0, xmm22, dword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62624D009BB200020000        , vfmsub132ss(xmm30, k0, xmm22, dword_ptr(rdx, 512))),
        INSTRUCTION_TEST(62624D009B727F              , vfmsub132ss(xmm30, k0, xmm22, dword_ptr(rdx, 508))),
        INSTRUCTION_TEST(62624D009BB2FCFDFFFF        , vfmsub132ss(xmm30, k0, xmm22, dword_ptr(rdx, -516))),
        INSTRUCTION_TEST(62624D009B7280              , vfmsub132ss(xmm30, k0, xmm22, dword_ptr(rdx, -512))),
        INSTRUCTION_TEST(62624D009B31                , vfmsub132ss(xmm30, k0, xmm22, dword_ptr(rcx))),
        INSTRUCTION_TEST(620235089BD1                , vfmsub132ss(xmm26, k0, xmm9, xmm25)),
        INSTRUCTION_TEST(62024D009BF3                , vfmsub132ss(xmm30, k0, xmm22, xmm27)),
        INSTRUCTION_TEST(626235089B527F              , vfmsub132ss(xmm26, k0, xmm9, dword_ptr(rdx, 508))),
        INSTRUCTION_TEST(626235089B92FCFDFFFF        , vfmsub132ss(xmm26, k0, xmm9, dword_ptr(rdx, -516))),
        INSTRUCTION_TEST(626235089B5280              , vfmsub132ss(xmm26, k0, xmm9, dword_ptr(rdx, -512))),
        INSTRUCTION_TEST(626235089B11                , vfmsub132ss(xmm26, k0, xmm9, dword_ptr(rcx))),
        INSTRUCTION_TEST(622235089B94F034120000      , vfmsub132ss(xmm26, k0, xmm9, dword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(C4E2699BCB                  , vfmsub132ss(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(C4E2699B8C2B80000000        , vfmsub132ss(xmm1, xmm2, dword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(626235089B9200020000        , vfmsub132ss(xmm26, k0, xmm9, dword_ptr(rdx, 512))),
        INSTRUCTION_TEST(62B2ED40AAFB                , vfmsub213pd(zmm7, k0, zmm18, zmm19)),
        INSTRUCTION_TEST(62B2ED40AABCF034120000      , vfmsub213pd(zmm7, k0, zmm18, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62F2ED40AA39                , vfmsub213pd(zmm7, k0, zmm18, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62F2ED40AABA00200000        , vfmsub213pd(zmm7, k0, zmm18, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62F2ED40AABAC0DFFFFF        , vfmsub213pd(zmm7, k0, zmm18, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62F2ED40AA7A7F              , vfmsub213pd(zmm7, k0, zmm18, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62F2AD48AAAA00200000        , vfmsub213pd(zmm5, k0, zmm10, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62F2ED40AA7A80              , vfmsub213pd(zmm7, k0, zmm18, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62F2AD48AA6A7F              , vfmsub213pd(zmm5, k0, zmm10, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62F2AD48AAAAC0DFFFFF        , vfmsub213pd(zmm5, k0, zmm10, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(C4E2E9AACB                  , vfmsub213pd(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(62F2AD48AA29                , vfmsub213pd(zmm5, k0, zmm10, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62B2AD48AAACF023010000      , vfmsub213pd(zmm5, k0, zmm10, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62F2AD48AAEC                , vfmsub213pd(zmm5, k0, zmm10, zmm4)),
        INSTRUCTION_TEST(62F2ED48AA4C2B02            , vfmsub213pd(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F2ED48AACB                , vfmsub213pd(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(C4E2EDAA8C2B80000000        , vfmsub213pd(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E2EDAACB                  , vfmsub213pd(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(C4E2E9AA8C2B80000000        , vfmsub213pd(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F2AD48AA6A80              , vfmsub213pd(zmm5, k0, zmm10, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62E24D40AAB200200000        , vfmsub213ps(zmm22, k0, zmm22, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62327D40AA8CF034120000      , vfmsub213ps(zmm9, k0, zmm16, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62727D40AACE                , vfmsub213ps(zmm9, k0, zmm16, zmm6)),
        INSTRUCTION_TEST(62E24D40AA727F              , vfmsub213ps(zmm22, k0, zmm22, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62727D40AA8AC0DFFFFF        , vfmsub213ps(zmm9, k0, zmm16, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62727D40AA4A7F              , vfmsub213ps(zmm9, k0, zmm16, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62727D40AA8A00200000        , vfmsub213ps(zmm9, k0, zmm16, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62727D40AA09                , vfmsub213ps(zmm9, k0, zmm16, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62E24D40AAB2C0DFFFFF        , vfmsub213ps(zmm22, k0, zmm22, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62727D40AA4A80              , vfmsub213ps(zmm9, k0, zmm16, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62E24D40AA31                , vfmsub213ps(zmm22, k0, zmm22, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(C4E269AACB                  , vfmsub213ps(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(C4E26DAACB                  , vfmsub213ps(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(62E24D40AA7280              , vfmsub213ps(zmm22, k0, zmm22, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(C4E26DAA8C2B80000000        , vfmsub213ps(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E269AA8C2B80000000        , vfmsub213ps(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F26D48AA4C2B02            , vfmsub213ps(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62C24D40AAF2                , vfmsub213ps(zmm22, k0, zmm22, zmm10)),
        INSTRUCTION_TEST(62A24D40AAB4F023010000      , vfmsub213ps(zmm22, k0, zmm22, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62F26D48AACB                , vfmsub213ps(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(C4A2B9ABB4F023010000        , vfmsub213sd(xmm6, xmm8, qword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(C4E2B9ABB200040000          , vfmsub213sd(xmm6, xmm8, qword_ptr(rdx, 1024))),
        INSTRUCTION_TEST(C4E2B9ABB2F8030000          , vfmsub213sd(xmm6, xmm8, qword_ptr(rdx, 1016))),
        INSTRUCTION_TEST(C4E2B9ABB2F8FBFFFF          , vfmsub213sd(xmm6, xmm8, qword_ptr(rdx, -1032))),
        INSTRUCTION_TEST(C4E2B9ABB200FCFFFF          , vfmsub213sd(xmm6, xmm8, qword_ptr(rdx, -1024))),
        INSTRUCTION_TEST(C4E2B9AB31                  , vfmsub213sd(xmm6, xmm8, qword_ptr(rcx))),
        INSTRUCTION_TEST(6232ED00ABE4                , vfmsub213sd(xmm12, k0, xmm18, xmm20)),
        INSTRUCTION_TEST(C4C2B9ABF4                  , vfmsub213sd(xmm6, xmm8, xmm12)),
        INSTRUCTION_TEST(6272ED00AB627F              , vfmsub213sd(xmm12, k0, xmm18, qword_ptr(rdx, 1016))),
        INSTRUCTION_TEST(6272ED00ABA2F8FBFFFF        , vfmsub213sd(xmm12, k0, xmm18, qword_ptr(rdx, -1032))),
        INSTRUCTION_TEST(6272ED00AB6280              , vfmsub213sd(xmm12, k0, xmm18, qword_ptr(rdx, -1024))),
        INSTRUCTION_TEST(6272ED00AB21                , vfmsub213sd(xmm12, k0, xmm18, qword_ptr(rcx))),
        INSTRUCTION_TEST(6232ED00ABA4F034120000      , vfmsub213sd(xmm12, k0, xmm18, qword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(C4E2E9ABCB                  , vfmsub213sd(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(C4E2E9AB8C2B80000000        , vfmsub213sd(xmm1, xmm2, qword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(6272ED00ABA200040000        , vfmsub213sd(xmm12, k0, xmm18, qword_ptr(rdx, 1024))),
        INSTRUCTION_TEST(62621508ABB200020000        , vfmsub213ss(xmm30, k0, xmm13, dword_ptr(rdx, 512))),
        INSTRUCTION_TEST(62621508AB727F              , vfmsub213ss(xmm30, k0, xmm13, dword_ptr(rdx, 508))),
        INSTRUCTION_TEST(62621508ABB2FCFDFFFF        , vfmsub213ss(xmm30, k0, xmm13, dword_ptr(rdx, -516))),
        INSTRUCTION_TEST(62621508AB7280              , vfmsub213ss(xmm30, k0, xmm13, dword_ptr(rdx, -512))),
        INSTRUCTION_TEST(62621508AB31                , vfmsub213ss(xmm30, k0, xmm13, dword_ptr(rcx))),
        INSTRUCTION_TEST(62221508ABB4F023010000      , vfmsub213ss(xmm30, k0, xmm13, dword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62021508ABF2                , vfmsub213ss(xmm30, k0, xmm13, xmm26)),
        INSTRUCTION_TEST(C4C271ABD1                  , vfmsub213ss(xmm2, xmm1, xmm9)),
        INSTRUCTION_TEST(C4E271AB92FC010000          , vfmsub213ss(xmm2, xmm1, dword_ptr(rdx, 508))),
        INSTRUCTION_TEST(C4E271AB92FCFDFFFF          , vfmsub213ss(xmm2, xmm1, dword_ptr(rdx, -516))),
        INSTRUCTION_TEST(C4E271AB9200FEFFFF          , vfmsub213ss(xmm2, xmm1, dword_ptr(rdx, -512))),
        INSTRUCTION_TEST(C4E271AB11                  , vfmsub213ss(xmm2, xmm1, dword_ptr(rcx))),
        INSTRUCTION_TEST(C4A271AB94F034120000        , vfmsub213ss(xmm2, xmm1, dword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(C4E269ABCB                  , vfmsub213ss(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(C4E269AB8C2B80000000        , vfmsub213ss(xmm1, xmm2, dword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E271AB9200020000          , vfmsub213ss(xmm2, xmm1, dword_ptr(rdx, 512))),
        INSTRUCTION_TEST(62E2ED48BA8A00200000        , vfmsub231pd(zmm17, k0, zmm2, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62D29D48BAEB                , vfmsub231pd(zmm5, k0, zmm12, zmm11)),
        INSTRUCTION_TEST(62B29D48BAACF023010000      , vfmsub231pd(zmm5, k0, zmm12, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62F29D48BA29                , vfmsub231pd(zmm5, k0, zmm12, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62F29D48BAAA00200000        , vfmsub231pd(zmm5, k0, zmm12, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62F29D48BAAAC0DFFFFF        , vfmsub231pd(zmm5, k0, zmm12, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62F29D48BA6A7F              , vfmsub231pd(zmm5, k0, zmm12, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62E2ED48BA4A7F              , vfmsub231pd(zmm17, k0, zmm2, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62F29D48BA6A80              , vfmsub231pd(zmm5, k0, zmm12, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62E2ED48BA8AC0DFFFFF        , vfmsub231pd(zmm17, k0, zmm2, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(C4E2E9BACB                  , vfmsub231pd(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(62E2ED48BA09                , vfmsub231pd(zmm17, k0, zmm2, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62A2ED48BA8CF034120000      , vfmsub231pd(zmm17, k0, zmm2, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62C2ED48BACE                , vfmsub231pd(zmm17, k0, zmm2, zmm14)),
        INSTRUCTION_TEST(62F2ED48BA4C2B02            , vfmsub231pd(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F2ED48BACB                , vfmsub231pd(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(C4E2EDBA8C2B80000000        , vfmsub231pd(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E2EDBACB                  , vfmsub231pd(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(C4E2E9BA8C2B80000000        , vfmsub231pd(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62E2ED48BA4A80              , vfmsub231pd(zmm17, k0, zmm2, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62F25D48BAAA00200000        , vfmsub231ps(zmm5, k0, zmm4, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62925540BAF3                , vfmsub231ps(zmm6, k0, zmm21, zmm27)),
        INSTRUCTION_TEST(62B25540BAB4F023010000      , vfmsub231ps(zmm6, k0, zmm21, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62F25540BA31                , vfmsub231ps(zmm6, k0, zmm21, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62F25540BAB200200000        , vfmsub231ps(zmm6, k0, zmm21, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62F25540BAB2C0DFFFFF        , vfmsub231ps(zmm6, k0, zmm21, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62F25540BA727F              , vfmsub231ps(zmm6, k0, zmm21, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62F25D48BA6A7F              , vfmsub231ps(zmm5, k0, zmm4, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62F25540BA7280              , vfmsub231ps(zmm6, k0, zmm21, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62F25D48BAAAC0DFFFFF        , vfmsub231ps(zmm5, k0, zmm4, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62F25D48BA6A80              , vfmsub231ps(zmm5, k0, zmm4, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62F25D48BA29                , vfmsub231ps(zmm5, k0, zmm4, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(C4E269BA8C2B80000000        , vfmsub231ps(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E26DBACB                  , vfmsub231ps(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(C4E26DBA8C2B80000000        , vfmsub231ps(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E269BACB                  , vfmsub231ps(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(62F26D48BA4C2B02            , vfmsub231ps(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62B25D48BAEC                , vfmsub231ps(zmm5, k0, zmm4, zmm20)),
        INSTRUCTION_TEST(62B25D48BAACF034120000      , vfmsub231ps(zmm5, k0, zmm4, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62F26D48BACB                , vfmsub231ps(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(C4A2E9BBBCF034120000        , vfmsub231sd(xmm7, xmm2, qword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(C4E2E9BBBA00040000          , vfmsub231sd(xmm7, xmm2, qword_ptr(rdx, 1024))),
        INSTRUCTION_TEST(C4E2E9BBBAF8030000          , vfmsub231sd(xmm7, xmm2, qword_ptr(rdx, 1016))),
        INSTRUCTION_TEST(C4E2E9BBBAF8FBFFFF          , vfmsub231sd(xmm7, xmm2, qword_ptr(rdx, -1032))),
        INSTRUCTION_TEST(C4E2E9BBBA00FCFFFF          , vfmsub231sd(xmm7, xmm2, qword_ptr(rdx, -1024))),
        INSTRUCTION_TEST(C4E2E9BB39                  , vfmsub231sd(xmm7, xmm2, qword_ptr(rcx))),
        INSTRUCTION_TEST(C4E2E9BBFE                  , vfmsub231sd(xmm7, xmm2, xmm6)),
        INSTRUCTION_TEST(C4E2A1BBDE                  , vfmsub231sd(xmm3, xmm11, xmm6)),
        INSTRUCTION_TEST(C4E2A1BB9AF8FBFFFF          , vfmsub231sd(xmm3, xmm11, qword_ptr(rdx, -1032))),
        INSTRUCTION_TEST(C4E2A1BB9AF8030000          , vfmsub231sd(xmm3, xmm11, qword_ptr(rdx, 1016))),
        INSTRUCTION_TEST(C4E2A1BB9A00FCFFFF          , vfmsub231sd(xmm3, xmm11, qword_ptr(rdx, -1024))),
        INSTRUCTION_TEST(C4E2A1BB19                  , vfmsub231sd(xmm3, xmm11, qword_ptr(rcx))),
        INSTRUCTION_TEST(C4A2A1BB9CF023010000        , vfmsub231sd(xmm3, xmm11, qword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(C4E2E9BBCB                  , vfmsub231sd(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(C4E2E9BB8C2B80000000        , vfmsub231sd(xmm1, xmm2, qword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E2A1BB9A00040000          , vfmsub231sd(xmm3, xmm11, qword_ptr(rdx, 1024))),
        INSTRUCTION_TEST(62626508BBED                , vfmsub231ss(xmm29, k0, xmm3, xmm5)),
        INSTRUCTION_TEST(62626508BBAA00020000        , vfmsub231ss(xmm29, k0, xmm3, dword_ptr(rdx, 512))),
        INSTRUCTION_TEST(62626508BB6A7F              , vfmsub231ss(xmm29, k0, xmm3, dword_ptr(rdx, 508))),
        INSTRUCTION_TEST(62626508BBAAFCFDFFFF        , vfmsub231ss(xmm29, k0, xmm3, dword_ptr(rdx, -516))),
        INSTRUCTION_TEST(62626508BB6A80              , vfmsub231ss(xmm29, k0, xmm3, dword_ptr(rdx, -512))),
        INSTRUCTION_TEST(62626508BB29                , vfmsub231ss(xmm29, k0, xmm3, dword_ptr(rcx))),
        INSTRUCTION_TEST(62226508BBACF023010000      , vfmsub231ss(xmm29, k0, xmm3, dword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62A26D00BBD2                , vfmsub231ss(xmm18, k0, xmm18, xmm18)),
        INSTRUCTION_TEST(62E26D00BB527F              , vfmsub231ss(xmm18, k0, xmm18, dword_ptr(rdx, 508))),
        INSTRUCTION_TEST(62E26D00BB92FCFDFFFF        , vfmsub231ss(xmm18, k0, xmm18, dword_ptr(rdx, -516))),
        INSTRUCTION_TEST(62E26D00BB5280              , vfmsub231ss(xmm18, k0, xmm18, dword_ptr(rdx, -512))),
        INSTRUCTION_TEST(62E26D00BB11                , vfmsub231ss(xmm18, k0, xmm18, dword_ptr(rcx))),
        INSTRUCTION_TEST(62A26D00BB94F034120000      , vfmsub231ss(xmm18, k0, xmm18, dword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(C4E269BBCB                  , vfmsub231ss(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(C4E269BB8C2B80000000        , vfmsub231ss(xmm1, xmm2, dword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62E26D00BB9200020000        , vfmsub231ss(xmm18, k0, xmm18, dword_ptr(rdx, 512))),
        INSTRUCTION_TEST(62E2DD4897EF                , vfmsubadd132pd(zmm21, k0, zmm4, zmm7)),
        INSTRUCTION_TEST(62A2DD4897ACF034120000      , vfmsubadd132pd(zmm21, k0, zmm4, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62E2DD489729                , vfmsubadd132pd(zmm21, k0, zmm4, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62E2DD4897AA00200000        , vfmsubadd132pd(zmm21, k0, zmm4, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62E2DD4897AAC0DFFFFF        , vfmsubadd132pd(zmm21, k0, zmm4, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62E2DD48976A7F              , vfmsubadd132pd(zmm21, k0, zmm4, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62E29D40976A7F              , vfmsubadd132pd(zmm21, k0, zmm28, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62E2DD48976A80              , vfmsubadd132pd(zmm21, k0, zmm4, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62E29D4097AAC0DFFFFF        , vfmsubadd132pd(zmm21, k0, zmm28, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62E29D4097AA00200000        , vfmsubadd132pd(zmm21, k0, zmm28, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62E29D409729                , vfmsubadd132pd(zmm21, k0, zmm28, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(C4E2E9978C2B80000000        , vfmsubadd132pd(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E2ED97CB                  , vfmsubadd132pd(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(C4E2ED978C2B80000000        , vfmsubadd132pd(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F2ED4897CB                , vfmsubadd132pd(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(C4E2E997CB                  , vfmsubadd132pd(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(62A29D4097EA                , vfmsubadd132pd(zmm21, k0, zmm28, zmm18)),
        INSTRUCTION_TEST(62E29D40976A80              , vfmsubadd132pd(zmm21, k0, zmm28, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62A29D4097ACF023010000      , vfmsubadd132pd(zmm21, k0, zmm28, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62F2ED48974C2B02            , vfmsubadd132pd(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F2154897527F              , vfmsubadd132ps(zmm2, k0, zmm13, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62F23D4097B200200000        , vfmsubadd132ps(zmm6, k0, zmm24, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62F23D4097727F              , vfmsubadd132ps(zmm6, k0, zmm24, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62F23D4097B2C0DFFFFF        , vfmsubadd132ps(zmm6, k0, zmm24, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62F23D40977280              , vfmsubadd132ps(zmm6, k0, zmm24, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62F23D409731                , vfmsubadd132ps(zmm6, k0, zmm24, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62B23D4097B4F034120000      , vfmsubadd132ps(zmm6, k0, zmm24, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62B23D4097F0                , vfmsubadd132ps(zmm6, k0, zmm24, zmm16)),
        INSTRUCTION_TEST(62F21548979200200000        , vfmsubadd132ps(zmm2, k0, zmm13, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62F215489792C0DFFFFF        , vfmsubadd132ps(zmm2, k0, zmm13, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62F21548975280              , vfmsubadd132ps(zmm2, k0, zmm13, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62F215489711                , vfmsubadd132ps(zmm2, k0, zmm13, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62B215489794F023010000      , vfmsubadd132ps(zmm2, k0, zmm13, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62B2154897D5                , vfmsubadd132ps(zmm2, k0, zmm13, zmm21)),
        INSTRUCTION_TEST(62F26D48974C2B02            , vfmsubadd132ps(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F26D4897CB                , vfmsubadd132ps(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(C4E26D978C2B80000000        , vfmsubadd132ps(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E26D97CB                  , vfmsubadd132ps(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(C4E269978C2B80000000        , vfmsubadd132ps(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E26997CB                  , vfmsubadd132ps(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(6272CD40A79A00200000        , vfmsubadd213pd(zmm11, k0, zmm22, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62F2C540A7D2                , vfmsubadd213pd(zmm2, k0, zmm23, zmm2)),
        INSTRUCTION_TEST(62B2C540A794F023010000      , vfmsubadd213pd(zmm2, k0, zmm23, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62F2C540A711                , vfmsubadd213pd(zmm2, k0, zmm23, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62F2C540A79200200000        , vfmsubadd213pd(zmm2, k0, zmm23, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62F2C540A792C0DFFFFF        , vfmsubadd213pd(zmm2, k0, zmm23, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62F2C540A7527F              , vfmsubadd213pd(zmm2, k0, zmm23, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(6272CD40A75A7F              , vfmsubadd213pd(zmm11, k0, zmm22, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62F2C540A75280              , vfmsubadd213pd(zmm2, k0, zmm23, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(6272CD40A79AC0DFFFFF        , vfmsubadd213pd(zmm11, k0, zmm22, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(6272CD40A75A80              , vfmsubadd213pd(zmm11, k0, zmm22, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(6272CD40A719                , vfmsubadd213pd(zmm11, k0, zmm22, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(C4E2E9A78C2B80000000        , vfmsubadd213pd(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E2EDA7CB                  , vfmsubadd213pd(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(C4E2EDA78C2B80000000        , vfmsubadd213pd(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E2E9A7CB                  , vfmsubadd213pd(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(62F2ED48A74C2B02            , vfmsubadd213pd(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(6252CD40A7DB                , vfmsubadd213pd(zmm11, k0, zmm22, zmm11)),
        INSTRUCTION_TEST(6232CD40A79CF034120000      , vfmsubadd213pd(zmm11, k0, zmm22, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62F2ED48A7CB                , vfmsubadd213pd(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(62723548A7A200200000        , vfmsubadd213ps(zmm12, k0, zmm9, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62E21D48A7B200200000        , vfmsubadd213ps(zmm22, k0, zmm12, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62E21D48A7727F              , vfmsubadd213ps(zmm22, k0, zmm12, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62E21D48A7B2C0DFFFFF        , vfmsubadd213ps(zmm22, k0, zmm12, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62E21D48A77280              , vfmsubadd213ps(zmm22, k0, zmm12, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62E21D48A731                , vfmsubadd213ps(zmm22, k0, zmm12, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62A21D48A7B4F023010000      , vfmsubadd213ps(zmm22, k0, zmm12, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62C21D48A7F6                , vfmsubadd213ps(zmm22, k0, zmm12, zmm14)),
        INSTRUCTION_TEST(62723548A7627F              , vfmsubadd213ps(zmm12, k0, zmm9, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62723548A7A2C0DFFFFF        , vfmsubadd213ps(zmm12, k0, zmm9, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(C4E269A7CB                  , vfmsubadd213ps(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(62723548A721                , vfmsubadd213ps(zmm12, k0, zmm9, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62323548A7A4F034120000      , vfmsubadd213ps(zmm12, k0, zmm9, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62523548A7E2                , vfmsubadd213ps(zmm12, k0, zmm9, zmm10)),
        INSTRUCTION_TEST(62F26D48A74C2B02            , vfmsubadd213ps(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F26D48A7CB                , vfmsubadd213ps(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(C4E26DA78C2B80000000        , vfmsubadd213ps(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E26DA7CB                  , vfmsubadd213ps(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(C4E269A78C2B80000000        , vfmsubadd213ps(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62723548A76280              , vfmsubadd213ps(zmm12, k0, zmm9, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62E2A540B78200200000        , vfmsubadd231pd(zmm16, k0, zmm27, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62C2ED48B7E9                , vfmsubadd231pd(zmm21, k0, zmm2, zmm9)),
        INSTRUCTION_TEST(62A2ED48B7ACF034120000      , vfmsubadd231pd(zmm21, k0, zmm2, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62E2ED48B729                , vfmsubadd231pd(zmm21, k0, zmm2, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62E2A540B7427F              , vfmsubadd231pd(zmm16, k0, zmm27, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62E2ED48B7AAC0DFFFFF        , vfmsubadd231pd(zmm21, k0, zmm2, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62E2ED48B76A7F              , vfmsubadd231pd(zmm21, k0, zmm2, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62E2ED48B7AA00200000        , vfmsubadd231pd(zmm21, k0, zmm2, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62E2ED48B76A80              , vfmsubadd231pd(zmm21, k0, zmm2, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62E2A540B782C0DFFFFF        , vfmsubadd231pd(zmm16, k0, zmm27, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62F2ED48B7CB                , vfmsubadd231pd(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(62E2A540B701                , vfmsubadd231pd(zmm16, k0, zmm27, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62A2A540B784F023010000      , vfmsubadd231pd(zmm16, k0, zmm27, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62A2A540B7C5                , vfmsubadd231pd(zmm16, k0, zmm27, zmm21)),
        INSTRUCTION_TEST(62F2ED48B74C2B02            , vfmsubadd231pd(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E2EDB78C2B80000000        , vfmsubadd231pd(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E2EDB7CB                  , vfmsubadd231pd(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(C4E2E9B78C2B80000000        , vfmsubadd231pd(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E2E9B7CB                  , vfmsubadd231pd(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(62E2A540B74280              , vfmsubadd231pd(zmm16, k0, zmm27, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62625540B79A00200000        , vfmsubadd231ps(zmm27, k0, zmm21, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62326540B784F023010000      , vfmsubadd231ps(zmm8, k0, zmm19, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62726540B7C1                , vfmsubadd231ps(zmm8, k0, zmm19, zmm1)),
        INSTRUCTION_TEST(62726540B7427F              , vfmsubadd231ps(zmm8, k0, zmm19, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62726540B74280              , vfmsubadd231ps(zmm8, k0, zmm19, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62726540B782C0DFFFFF        , vfmsubadd231ps(zmm8, k0, zmm19, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62625540B75A7F              , vfmsubadd231ps(zmm27, k0, zmm21, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62726540B78200200000        , vfmsubadd231ps(zmm8, k0, zmm19, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62726540B701                , vfmsubadd231ps(zmm8, k0, zmm19, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62625540B79AC0DFFFFF        , vfmsubadd231ps(zmm27, k0, zmm21, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62025540B7D8                , vfmsubadd231ps(zmm27, k0, zmm21, zmm24)),
        INSTRUCTION_TEST(62625540B719                , vfmsubadd231ps(zmm27, k0, zmm21, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62225540B79CF034120000      , vfmsubadd231ps(zmm27, k0, zmm21, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62F26D48B74C2B02            , vfmsubadd231ps(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F26D48B7CB                , vfmsubadd231ps(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(C4E26DB78C2B80000000        , vfmsubadd231ps(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E26DB7CB                  , vfmsubadd231ps(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(C4E269B78C2B80000000        , vfmsubadd231ps(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E269B7CB                  , vfmsubadd231ps(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(62625540B75A80              , vfmsubadd231ps(zmm27, k0, zmm21, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(C4E36D5F8C2B8000000060      , vfmsubaddpd(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128), ymm6)),
        INSTRUCTION_TEST(C4E36D5FCB40                , vfmsubaddpd(ymm1, ymm2, ymm3, ymm4)),
        INSTRUCTION_TEST(C4E3ED5F8C358000000030      , vfmsubaddpd(ymm1, ymm2, ymm3, ymmword_ptr(rbp, rsi, 1, 128))),
        INSTRUCTION_TEST(C4E3E95F8C358000000030      , vfmsubaddpd(xmm1, xmm2, xmm3, xmmword_ptr(rbp, rsi, 1, 128))),
        INSTRUCTION_TEST(C4E3695F8C2B8000000060      , vfmsubaddpd(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128), xmm6)),
        INSTRUCTION_TEST(C4E3695FCB40                , vfmsubaddpd(xmm1, xmm2, xmm3, xmm4)),
        INSTRUCTION_TEST(C4E3695ECB40                , vfmsubaddps(xmm1, xmm2, xmm3, xmm4)),
        INSTRUCTION_TEST(C4E3E95E8C358000000030      , vfmsubaddps(xmm1, xmm2, xmm3, xmmword_ptr(rbp, rsi, 1, 128))),
        INSTRUCTION_TEST(C4E3695E8C2B8000000060      , vfmsubaddps(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128), xmm6)),
        INSTRUCTION_TEST(C4E36D5ECB40                , vfmsubaddps(ymm1, ymm2, ymm3, ymm4)),
        INSTRUCTION_TEST(C4E3ED5E8C358000000030      , vfmsubaddps(ymm1, ymm2, ymm3, ymmword_ptr(rbp, rsi, 1, 128))),
        INSTRUCTION_TEST(C4E36D5E8C2B8000000060      , vfmsubaddps(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128), ymm6)),
        INSTRUCTION_TEST(C4E3696DCB40                , vfmsubpd(xmm1, xmm2, xmm3, xmm4)),
        INSTRUCTION_TEST(C4E3E96D8C358000000030      , vfmsubpd(xmm1, xmm2, xmm3, xmmword_ptr(rbp, rsi, 1, 128))),
        INSTRUCTION_TEST(C4E3696D8C2B8000000060      , vfmsubpd(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128), xmm6)),
        INSTRUCTION_TEST(C4E36D6DCB40                , vfmsubpd(ymm1, ymm2, ymm3, ymm4)),
        INSTRUCTION_TEST(C4E3ED6D8C358000000030      , vfmsubpd(ymm1, ymm2, ymm3, ymmword_ptr(rbp, rsi, 1, 128))),
        INSTRUCTION_TEST(C4E36D6D8C2B8000000060      , vfmsubpd(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128), ymm6)),
        INSTRUCTION_TEST(C4E36D6C8C2B8000000060      , vfmsubps(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128), ymm6)),
        INSTRUCTION_TEST(C4E36D6CCB40                , vfmsubps(ymm1, ymm2, ymm3, ymm4)),
        INSTRUCTION_TEST(C4E3ED6C8C358000000030      , vfmsubps(ymm1, ymm2, ymm3, ymmword_ptr(rbp, rsi, 1, 128))),
        INSTRUCTION_TEST(C4E3696C8C2B8000000060      , vfmsubps(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128), xmm6)),
        INSTRUCTION_TEST(C4E3696CCB40                , vfmsubps(xmm1, xmm2, xmm3, xmm4)),
        INSTRUCTION_TEST(C4E3E96C8C358000000030      , vfmsubps(xmm1, xmm2, xmm3, xmmword_ptr(rbp, rsi, 1, 128))),
        INSTRUCTION_TEST(C4E3696F8C2B8000000060      , vfmsubsd(xmm1, xmm2, qword_ptr(rbx, rbp, 1, 128), xmm6)),
        INSTRUCTION_TEST(C4E3E96F8C358000000030      , vfmsubsd(xmm1, xmm2, xmm3, qword_ptr(rbp, rsi, 1, 128))),
        INSTRUCTION_TEST(C4E3696FCB40                , vfmsubsd(xmm1, xmm2, xmm3, xmm4)),
        INSTRUCTION_TEST(C4E3696E8C2B8000000060      , vfmsubss(xmm1, xmm2, dword_ptr(rbx, rbp, 1, 128), xmm6)),
        INSTRUCTION_TEST(C4E3E96E8C358000000030      , vfmsubss(xmm1, xmm2, xmm3, dword_ptr(rbp, rsi, 1, 128))),
        INSTRUCTION_TEST(C4E3696ECB40                , vfmsubss(xmm1, xmm2, xmm3, xmm4)),
        INSTRUCTION_TEST(627295409C9A00200000        , vfnmadd132pd(zmm11, k0, zmm29, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62328D489CA4F023010000      , vfnmadd132pd(zmm12, k0, zmm14, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62728D489CE1                , vfnmadd132pd(zmm12, k0, zmm14, zmm1)),
        INSTRUCTION_TEST(62728D489C627F              , vfnmadd132pd(zmm12, k0, zmm14, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62728D489CA200200000        , vfnmadd132pd(zmm12, k0, zmm14, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62728D489CA2C0DFFFFF        , vfnmadd132pd(zmm12, k0, zmm14, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(627295409C5A7F              , vfnmadd132pd(zmm11, k0, zmm29, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62728D489C21                , vfnmadd132pd(zmm12, k0, zmm14, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(627295409C9AC0DFFFFF        , vfnmadd132pd(zmm11, k0, zmm29, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62728D489C6280              , vfnmadd132pd(zmm12, k0, zmm14, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(627295409C19                , vfnmadd132pd(zmm11, k0, zmm29, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(C4E2E99CCB                  , vfnmadd132pd(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(C4E2E99C8C2B80000000        , vfnmadd132pd(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E2ED9CCB                  , vfnmadd132pd(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(C4E2ED9C8C2B80000000        , vfnmadd132pd(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(627295409C5A80              , vfnmadd132pd(zmm11, k0, zmm29, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62F2ED489C4C2B02            , vfnmadd132pd(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(623295409CDB                , vfnmadd132pd(zmm11, k0, zmm29, zmm19)),
        INSTRUCTION_TEST(623295409C9CF034120000      , vfnmadd132pd(zmm11, k0, zmm29, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62F2ED489CCB                , vfnmadd132pd(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(62E265489C4A7F              , vfnmadd132ps(zmm17, k0, zmm3, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62E27D409CAA00200000        , vfnmadd132ps(zmm21, k0, zmm16, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62E27D409C6A7F              , vfnmadd132ps(zmm21, k0, zmm16, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62E27D409CAAC0DFFFFF        , vfnmadd132ps(zmm21, k0, zmm16, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62E27D409C6A80              , vfnmadd132ps(zmm21, k0, zmm16, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62E27D409C29                , vfnmadd132ps(zmm21, k0, zmm16, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62A27D409CACF023010000      , vfnmadd132ps(zmm21, k0, zmm16, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62C27D409CEA                , vfnmadd132ps(zmm21, k0, zmm16, zmm10)),
        INSTRUCTION_TEST(62E265489C8A00200000        , vfnmadd132ps(zmm17, k0, zmm3, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62E265489C8AC0DFFFFF        , vfnmadd132ps(zmm17, k0, zmm3, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62E265489C09                , vfnmadd132ps(zmm17, k0, zmm3, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(C4E2699CCB                  , vfnmadd132ps(xmm1,xmm2, xmm3)),
        INSTRUCTION_TEST(62A265489C8CF034120000      , vfnmadd132ps(zmm17, k0, zmm3, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62C265489CCF                , vfnmadd132ps(zmm17, k0, zmm3, zmm15)),
        INSTRUCTION_TEST(62F26D489C4C2B02            , vfnmadd132ps(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F26D489CCB                , vfnmadd132ps(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(C4E26D9C8C2B80000000        , vfnmadd132ps(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E26D9CCB                  , vfnmadd132ps(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(C4E2699C8C2B80000000        , vfnmadd132ps(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62E265489C4A80              , vfnmadd132ps(zmm17, k0, zmm3, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62F2BD009D11                , vfnmadd132sd(xmm2, k0, xmm24, qword_ptr(rcx))),
        INSTRUCTION_TEST(62F2BD009D5280              , vfnmadd132sd(xmm2, k0, xmm24, qword_ptr(rdx, -1024))),
        INSTRUCTION_TEST(62F2BD009D92F8FBFFFF        , vfnmadd132sd(xmm2, k0, xmm24, qword_ptr(rdx, -1032))),
        INSTRUCTION_TEST(62B2BD009D94F034120000      , vfnmadd132sd(xmm2, k0, xmm24, qword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62F2BD009D9200040000        , vfnmadd132sd(xmm2, k0, xmm24, qword_ptr(rdx, 1024))),
        INSTRUCTION_TEST(62F2BD009DD1                , vfnmadd132sd(xmm2, k0, xmm24, xmm1)),
        INSTRUCTION_TEST(62F2BD009D527F              , vfnmadd132sd(xmm2, k0, xmm24, qword_ptr(rdx, 1016))),
        INSTRUCTION_TEST(62C2E5089DDB                , vfnmadd132sd(xmm19, k0, xmm3, xmm11)),
        INSTRUCTION_TEST(62E2E5089D9AF8FBFFFF        , vfnmadd132sd(xmm19, k0, xmm3, qword_ptr(rdx, -1032))),
        INSTRUCTION_TEST(62E2E5089D5A7F              , vfnmadd132sd(xmm19, k0, xmm3, qword_ptr(rdx, 1016))),
        INSTRUCTION_TEST(62E2E5089D5A80              , vfnmadd132sd(xmm19, k0, xmm3, qword_ptr(rdx, -1024))),
        INSTRUCTION_TEST(62E2E5089D19                , vfnmadd132sd(xmm19, k0, xmm3, qword_ptr(rcx))),
        INSTRUCTION_TEST(62A2E5089D9CF023010000      , vfnmadd132sd(xmm19, k0, xmm3, qword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(C4E2E99DCB                  , vfnmadd132sd(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(C4E2E99D8C2B80000000        , vfnmadd132sd(xmm1, xmm2, qword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62E2E5089D9A00040000        , vfnmadd132sd(xmm19, k0, xmm3, qword_ptr(rdx, 1024))),
        INSTRUCTION_TEST(622275089DCF                , vfnmadd132ss(xmm25, k0, xmm1, xmm23)),
        INSTRUCTION_TEST(626275089D8A00020000        , vfnmadd132ss(xmm25, k0, xmm1, dword_ptr(rdx, 512))),
        INSTRUCTION_TEST(626275089D4A7F              , vfnmadd132ss(xmm25, k0, xmm1, dword_ptr(rdx, 508))),
        INSTRUCTION_TEST(626275089D8AFCFDFFFF        , vfnmadd132ss(xmm25, k0, xmm1, dword_ptr(rdx, -516))),
        INSTRUCTION_TEST(626275089D4A80              , vfnmadd132ss(xmm25, k0, xmm1, dword_ptr(rdx, -512))),
        INSTRUCTION_TEST(626275089D09                , vfnmadd132ss(xmm25, k0, xmm1, dword_ptr(rcx))),
        INSTRUCTION_TEST(622275089D8CF023010000      , vfnmadd132ss(xmm25, k0, xmm1, dword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(625265009DED                , vfnmadd132ss(xmm13, k0, xmm19, xmm13)),
        INSTRUCTION_TEST(627265009D6A7F              , vfnmadd132ss(xmm13, k0, xmm19, dword_ptr(rdx, 508))),
        INSTRUCTION_TEST(627265009DAAFCFDFFFF        , vfnmadd132ss(xmm13, k0, xmm19, dword_ptr(rdx, -516))),
        INSTRUCTION_TEST(627265009D6A80              , vfnmadd132ss(xmm13, k0, xmm19, dword_ptr(rdx, -512))),
        INSTRUCTION_TEST(627265009D29                , vfnmadd132ss(xmm13, k0, xmm19, dword_ptr(rcx))),
        INSTRUCTION_TEST(623265009DACF034120000      , vfnmadd132ss(xmm13, k0, xmm19, dword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(C4E2699DCB                  , vfnmadd132ss(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(C4E2699D8C2B80000000        , vfnmadd132ss(xmm1, xmm2, dword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(627265009DAA00020000        , vfnmadd132ss(xmm13, k0, xmm19, dword_ptr(rdx, 512))),
        INSTRUCTION_TEST(6272F548ACBA00200000        , vfnmadd213pd(zmm15, k0, zmm1, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62C2FD40ACC9                , vfnmadd213pd(zmm17, k0, zmm16, zmm9)),
        INSTRUCTION_TEST(62A2FD40AC8CF023010000      , vfnmadd213pd(zmm17, k0, zmm16, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62E2FD40AC4A7F              , vfnmadd213pd(zmm17, k0, zmm16, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62E2FD40AC4A80              , vfnmadd213pd(zmm17, k0, zmm16, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62E2FD40AC8AC0DFFFFF        , vfnmadd213pd(zmm17, k0, zmm16, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(6272F548AC7A7F              , vfnmadd213pd(zmm15, k0, zmm1, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62E2FD40AC09                , vfnmadd213pd(zmm17, k0, zmm16, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(6272F548ACBAC0DFFFFF        , vfnmadd213pd(zmm15, k0, zmm1, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62E2FD40AC8A00200000        , vfnmadd213pd(zmm17, k0, zmm16, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(6272F548AC39                , vfnmadd213pd(zmm15, k0, zmm1, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(C4E2E9ACCB                  , vfnmadd213pd(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(C4E2E9AC8C2B80000000        , vfnmadd213pd(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E2EDACCB                  , vfnmadd213pd(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(C4E2EDAC8C2B80000000        , vfnmadd213pd(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(6272F548AC7A80              , vfnmadd213pd(zmm15, k0, zmm1, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62F2ED48AC4C2B02            , vfnmadd213pd(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(6232F548ACFE                , vfnmadd213pd(zmm15, k0, zmm1, zmm22)),
        INSTRUCTION_TEST(6232F548ACBCF034120000      , vfnmadd213pd(zmm15, k0, zmm1, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62F2ED48ACCB                , vfnmadd213pd(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(62725D48AC5A7F              , vfnmadd213ps(zmm11, k0, zmm4, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62622D48AC9200200000        , vfnmadd213ps(zmm26, k0, zmm10, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62622D48AC527F              , vfnmadd213ps(zmm26, k0, zmm10, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62622D48AC92C0DFFFFF        , vfnmadd213ps(zmm26, k0, zmm10, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62622D48AC5280              , vfnmadd213ps(zmm26, k0, zmm10, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62622D48AC11                , vfnmadd213ps(zmm26, k0, zmm10, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62622D48ACD6                , vfnmadd213ps(zmm26, k0, zmm10, zmm6)),
        INSTRUCTION_TEST(62222D48AC94F023010000      , vfnmadd213ps(zmm26, k0, zmm10, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62725D48AC9A00200000        , vfnmadd213ps(zmm11, k0, zmm4, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62725D48AC9AC0DFFFFF        , vfnmadd213ps(zmm11, k0, zmm4, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62725D48AC19                , vfnmadd213ps(zmm11, k0, zmm4, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62725D48ACDC                , vfnmadd213ps(zmm11, k0, zmm4, zmm4)),
        INSTRUCTION_TEST(62325D48AC9CF034120000      , vfnmadd213ps(zmm11, k0, zmm4, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62F26D48AC4C2B02            , vfnmadd213ps(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F26D48ACCB                , vfnmadd213ps(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(C4E26DAC8C2B80000000        , vfnmadd213ps(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E26DACCB                  , vfnmadd213ps(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(C4E269AC8C2B80000000        , vfnmadd213ps(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E269ACCB                  , vfnmadd213ps(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(62725D48AC5A80              , vfnmadd213ps(zmm11, k0, zmm4, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62228D08ADB4F034120000      , vfnmadd213sd(xmm30, k0, xmm14, qword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62628D08ADB200040000        , vfnmadd213sd(xmm30, k0, xmm14, qword_ptr(rdx, 1024))),
        INSTRUCTION_TEST(62628D08AD727F              , vfnmadd213sd(xmm30, k0, xmm14, qword_ptr(rdx, 1016))),
        INSTRUCTION_TEST(62628D08ADB2F8FBFFFF        , vfnmadd213sd(xmm30, k0, xmm14, qword_ptr(rdx, -1032))),
        INSTRUCTION_TEST(62628D08AD7280              , vfnmadd213sd(xmm30, k0, xmm14, qword_ptr(rdx, -1024))),
        INSTRUCTION_TEST(62628D08AD31                , vfnmadd213sd(xmm30, k0, xmm14, qword_ptr(rcx))),
        INSTRUCTION_TEST(6272A500ADEA                , vfnmadd213sd(xmm13, k0, xmm27, xmm2)),
        INSTRUCTION_TEST(62428D08ADF3                , vfnmadd213sd(xmm30, k0, xmm14, xmm11)),
        INSTRUCTION_TEST(6272A500AD6A7F              , vfnmadd213sd(xmm13, k0, xmm27, qword_ptr(rdx, 1016))),
        INSTRUCTION_TEST(6272A500ADAAF8FBFFFF        , vfnmadd213sd(xmm13, k0, xmm27, qword_ptr(rdx, -1032))),
        INSTRUCTION_TEST(6272A500AD6A80              , vfnmadd213sd(xmm13, k0, xmm27, qword_ptr(rdx, -1024))),
        INSTRUCTION_TEST(6272A500AD29                , vfnmadd213sd(xmm13, k0, xmm27, qword_ptr(rcx))),
        INSTRUCTION_TEST(6232A500ADACF023010000      , vfnmadd213sd(xmm13, k0, xmm27, qword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(C4E2E9ADCB                  , vfnmadd213sd(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(C4E2E9AD8C2B80000000        , vfnmadd213sd(xmm1, xmm2, qword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(6272A500ADAA00040000        , vfnmadd213sd(xmm13, k0, xmm27, qword_ptr(rdx, 1024))),
        INSTRUCTION_TEST(62024508ADEC                , vfnmadd213ss(xmm29, k0, xmm7, xmm28)),
        INSTRUCTION_TEST(62624508ADAA00020000        , vfnmadd213ss(xmm29, k0, xmm7, dword_ptr(rdx, 512))),
        INSTRUCTION_TEST(62624508AD6A7F              , vfnmadd213ss(xmm29, k0, xmm7, dword_ptr(rdx, 508))),
        INSTRUCTION_TEST(62624508ADAAFCFDFFFF        , vfnmadd213ss(xmm29, k0, xmm7, dword_ptr(rdx, -516))),
        INSTRUCTION_TEST(62624508AD6A80              , vfnmadd213ss(xmm29, k0, xmm7, dword_ptr(rdx, -512))),
        INSTRUCTION_TEST(62624508AD29                , vfnmadd213ss(xmm29, k0, xmm7, dword_ptr(rcx))),
        INSTRUCTION_TEST(62224508ADACF023010000      , vfnmadd213ss(xmm29, k0, xmm7, dword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62226508ADE4                , vfnmadd213ss(xmm28, k0, xmm3, xmm20)),
        INSTRUCTION_TEST(62626508AD627F              , vfnmadd213ss(xmm28, k0, xmm3, dword_ptr(rdx, 508))),
        INSTRUCTION_TEST(62626508ADA2FCFDFFFF        , vfnmadd213ss(xmm28, k0, xmm3, dword_ptr(rdx, -516))),
        INSTRUCTION_TEST(62626508AD6280              , vfnmadd213ss(xmm28, k0, xmm3, dword_ptr(rdx, -512))),
        INSTRUCTION_TEST(62626508AD21                , vfnmadd213ss(xmm28, k0, xmm3, dword_ptr(rcx))),
        INSTRUCTION_TEST(62226508ADA4F034120000      , vfnmadd213ss(xmm28, k0, xmm3, dword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(C4E269ADCB                  , vfnmadd213ss(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(C4E269AD8C2B80000000        , vfnmadd213ss(xmm1, xmm2, dword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62626508ADA200020000        , vfnmadd213ss(xmm28, k0, xmm3, dword_ptr(rdx, 512))),
        INSTRUCTION_TEST(6272A548BC627F              , vfnmadd231pd(zmm12, k0, zmm11, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(6272A548BCA200200000        , vfnmadd231pd(zmm12, k0, zmm11, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(6232D548BCFE                , vfnmadd231pd(zmm15, k0, zmm5, zmm22)),
        INSTRUCTION_TEST(6232D548BCBCF034120000      , vfnmadd231pd(zmm15, k0, zmm5, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(6272D548BC7A7F              , vfnmadd231pd(zmm15, k0, zmm5, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(6272D548BC7A80              , vfnmadd231pd(zmm15, k0, zmm5, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(6272D548BCBA00200000        , vfnmadd231pd(zmm15, k0, zmm5, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(6272D548BCBAC0DFFFFF        , vfnmadd231pd(zmm15, k0, zmm5, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(6272D548BC39                , vfnmadd231pd(zmm15, k0, zmm5, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(6272A548BCA2C0DFFFFF        , vfnmadd231pd(zmm12, k0, zmm11, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(C4E2EDBCCB                  , vfnmadd231pd(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(6272A548BC21                , vfnmadd231pd(zmm12, k0, zmm11, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(6232A548BCA4F023010000      , vfnmadd231pd(zmm12, k0, zmm11, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(6232A548BCE0                , vfnmadd231pd(zmm12, k0, zmm11, zmm16)),
        INSTRUCTION_TEST(62F2ED48BC4C2B02            , vfnmadd231pd(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F2ED48BCCB                , vfnmadd231pd(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(C4E2EDBC8C2B80000000        , vfnmadd231pd(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E2E9BC8C2B80000000        , vfnmadd231pd(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E2E9BCCB                  , vfnmadd231pd(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(6272A548BC6280              , vfnmadd231pd(zmm12, k0, zmm11, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62721D40BCAA00200000        , vfnmadd231ps(zmm13, k0, zmm28, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62124548BCF0                , vfnmadd231ps(zmm14, k0, zmm7, zmm24)),
        INSTRUCTION_TEST(62324548BCB4F023010000      , vfnmadd231ps(zmm14, k0, zmm7, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62721D40BC6A7F              , vfnmadd231ps(zmm13, k0, zmm28, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62724548BC7280              , vfnmadd231ps(zmm14, k0, zmm7, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62724548BCB2C0DFFFFF        , vfnmadd231ps(zmm14, k0, zmm7, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62724548BC727F              , vfnmadd231ps(zmm14, k0, zmm7, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62724548BC31                , vfnmadd231ps(zmm14, k0, zmm7, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62721D40BCAAC0DFFFFF        , vfnmadd231ps(zmm13, k0, zmm28, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62724548BCB200200000        , vfnmadd231ps(zmm14, k0, zmm7, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62721D40BC29                , vfnmadd231ps(zmm13, k0, zmm28, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(C4E269BCCB                  , vfnmadd231ps(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(C4E269BC8C2B80000000        , vfnmadd231ps(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E26DBCCB                  , vfnmadd231ps(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(C4E26DBC8C2B80000000        , vfnmadd231ps(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62721D40BC6A80              , vfnmadd231ps(zmm13, k0, zmm28, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62F26D48BC4C2B02            , vfnmadd231ps(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62521D40BCE9                , vfnmadd231ps(zmm13, k0, zmm28, zmm9)),
        INSTRUCTION_TEST(62321D40BCACF034120000      , vfnmadd231ps(zmm13, k0, zmm28, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62F26D48BCCB                , vfnmadd231ps(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(62B2A508BDF2                , vfnmadd231sd(xmm6, k0, xmm11, xmm18)),
        INSTRUCTION_TEST(C4E2A1BDB2F8030000          , vfnmadd231sd(xmm6, xmm11, qword_ptr(rdx, 1016))),
        INSTRUCTION_TEST(C4E2A1BDB2F8FBFFFF          , vfnmadd231sd(xmm6, xmm11, qword_ptr(rdx, -1032))),
        INSTRUCTION_TEST(C4E2A1BDB200FCFFFF          , vfnmadd231sd(xmm6, xmm11, qword_ptr(rdx, -1024))),
        INSTRUCTION_TEST(C4E2A1BD31                  , vfnmadd231sd(xmm6, xmm11, qword_ptr(rcx))),
        INSTRUCTION_TEST(C4A2A1BDB4F023010000        , vfnmadd231sd(xmm6, xmm11, qword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(C462D1BDF5                  , vfnmadd231sd(xmm14, xmm5, xmm5)),
        INSTRUCTION_TEST(C4E2A1BDB200040000          , vfnmadd231sd(xmm6, xmm11, qword_ptr(rdx, 1024))),
        INSTRUCTION_TEST(C462D1BDB2F8030000          , vfnmadd231sd(xmm14, xmm5, qword_ptr(rdx, 1016))),
        INSTRUCTION_TEST(C462D1BDB2F8FBFFFF          , vfnmadd231sd(xmm14, xmm5, qword_ptr(rdx, -1032))),
        INSTRUCTION_TEST(C462D1BDB200FCFFFF          , vfnmadd231sd(xmm14, xmm5, qword_ptr(rdx, -1024))),
        INSTRUCTION_TEST(C462D1BD31                  , vfnmadd231sd(xmm14, xmm5, qword_ptr(rcx))),
        INSTRUCTION_TEST(C422D1BDB4F034120000        , vfnmadd231sd(xmm14, xmm5, qword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(C4E2E9BDCB                  , vfnmadd231sd(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(C4E2E9BD8C2B80000000        , vfnmadd231sd(xmm1, xmm2, qword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C462D1BDB200040000          , vfnmadd231sd(xmm14, xmm5, qword_ptr(rdx, 1024))),
        INSTRUCTION_TEST(62226D08BDD9                , vfnmadd231ss(xmm27, k0, xmm2, xmm17)),
        INSTRUCTION_TEST(62626D08BD9A00020000        , vfnmadd231ss(xmm27, k0, xmm2, dword_ptr(rdx, 512))),
        INSTRUCTION_TEST(62626D08BD5A7F              , vfnmadd231ss(xmm27, k0, xmm2, dword_ptr(rdx, 508))),
        INSTRUCTION_TEST(62626D08BD9AFCFDFFFF        , vfnmadd231ss(xmm27, k0, xmm2, dword_ptr(rdx, -516))),
        INSTRUCTION_TEST(62626D08BD5A80              , vfnmadd231ss(xmm27, k0, xmm2, dword_ptr(rdx, -512))),
        INSTRUCTION_TEST(62626D08BD19                , vfnmadd231ss(xmm27, k0, xmm2, dword_ptr(rcx))),
        INSTRUCTION_TEST(62226D08BD9CF023010000      , vfnmadd231ss(xmm27, k0, xmm2, dword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62027D00BDC9                , vfnmadd231ss(xmm25, k0, xmm16, xmm25)),
        INSTRUCTION_TEST(62627D00BD4A80              , vfnmadd231ss(xmm25, k0, xmm16, dword_ptr(rdx, -512))),
        INSTRUCTION_TEST(62627D00BD4A7F              , vfnmadd231ss(xmm25, k0, xmm16, dword_ptr(rdx, 508))),
        INSTRUCTION_TEST(62627D00BD8AFCFDFFFF        , vfnmadd231ss(xmm25, k0, xmm16, dword_ptr(rdx, -516))),
        INSTRUCTION_TEST(62627D00BD09                , vfnmadd231ss(xmm25, k0, xmm16, dword_ptr(rcx))),
        INSTRUCTION_TEST(62227D00BD8CF034120000      , vfnmadd231ss(xmm25, k0, xmm16, dword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(C4E269BDCB                  , vfnmadd231ss(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(C4E269BD8C2B80000000        , vfnmadd231ss(xmm1, xmm2, dword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62627D00BD8A00020000        , vfnmadd231ss(xmm25, k0, xmm16, dword_ptr(rdx, 512))),
        INSTRUCTION_TEST(C4E3ED798C358000000030      , vfnmaddpd(ymm1, ymm2, ymm3, ymmword_ptr(rbp, rsi, 1, 128))),
        INSTRUCTION_TEST(C4E36D79CB40                , vfnmaddpd(ymm1, ymm2, ymm3, ymm4)),
        INSTRUCTION_TEST(C4E36D798C2B8000000060      , vfnmaddpd(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128), ymm6)),
        INSTRUCTION_TEST(C4E3E9798C358000000030      , vfnmaddpd(xmm1, xmm2, xmm3, xmmword_ptr(rbp, rsi, 1, 128))),
        INSTRUCTION_TEST(C4E36979CB40                , vfnmaddpd(xmm1, xmm2, xmm3, xmm4)),
        INSTRUCTION_TEST(C4E369798C2B8000000060      , vfnmaddpd(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128), xmm6)),
        INSTRUCTION_TEST(C4E36978CB40                , vfnmaddps(xmm1, xmm2, xmm3, xmm4)),
        INSTRUCTION_TEST(C4E3E9788C358000000030      , vfnmaddps(xmm1, xmm2, xmm3, xmmword_ptr(rbp, rsi, 1, 128))),
        INSTRUCTION_TEST(C4E369788C2B8000000060      , vfnmaddps(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128), xmm6)),
        INSTRUCTION_TEST(C4E36D78CB40                , vfnmaddps(ymm1, ymm2, ymm3, ymm4)),
        INSTRUCTION_TEST(C4E3ED788C358000000030      , vfnmaddps(ymm1, ymm2, ymm3, ymmword_ptr(rbp, rsi, 1, 128))),
        INSTRUCTION_TEST(C4E36D788C2B8000000060      , vfnmaddps(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128), ymm6)),
        INSTRUCTION_TEST(C4E3697B8C2B8000000060      , vfnmaddsd(xmm1, xmm2, qword_ptr(rbx, rbp, 1, 128), xmm6)),
        INSTRUCTION_TEST(C4E3E97B8C358000000030      , vfnmaddsd(xmm1, xmm2, xmm3, qword_ptr(rbp, rsi, 1, 128))),
        INSTRUCTION_TEST(C4E3697BCB40                , vfnmaddsd(xmm1, xmm2, xmm3, xmm4)),
        INSTRUCTION_TEST(C4E3697ACB40                , vfnmaddss(xmm1, xmm2, xmm3, xmm4)),
        INSTRUCTION_TEST(C4E3697A8C2B8000000060      , vfnmaddss(xmm1, xmm2, dword_ptr(rbx, rbp, 1, 128), xmm6)),
        INSTRUCTION_TEST(C4E3E97A8C358000000030      , vfnmaddss(xmm1, xmm2, xmm3, dword_ptr(rbp, rsi, 1, 128))),
        INSTRUCTION_TEST(6262BD489E5A7F              , vfnmsub132pd(zmm27, k0, zmm8, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(6262D5489E627F              , vfnmsub132pd(zmm28, k0, zmm5, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(6262D5489EA2C0DFFFFF        , vfnmsub132pd(zmm28, k0, zmm5, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(6262D5489E6280              , vfnmsub132pd(zmm28, k0, zmm5, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(6262D5489E21                , vfnmsub132pd(zmm28, k0, zmm5, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(6262D5489EE6                , vfnmsub132pd(zmm28, k0, zmm5, zmm6)),
        INSTRUCTION_TEST(6222D5489EA4F023010000      , vfnmsub132pd(zmm28, k0, zmm5, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(6262BD489E9A00200000        , vfnmsub132pd(zmm27, k0, zmm8, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(6262D5489EA200200000        , vfnmsub132pd(zmm28, k0, zmm5, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(6262BD489E9AC0DFFFFF        , vfnmsub132pd(zmm27, k0, zmm8, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(6222BD489E9CF034120000      , vfnmsub132pd(zmm27, k0, zmm8, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(6262BD489E19                , vfnmsub132pd(zmm27, k0, zmm8, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(6242BD489ED9                , vfnmsub132pd(zmm27, k0, zmm8, zmm9)),
        INSTRUCTION_TEST(62F2ED489E4C2B02            , vfnmsub132pd(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F2ED489ECB                , vfnmsub132pd(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(C4E2ED9E8C2B80000000        , vfnmsub132pd(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E2ED9ECB                  , vfnmsub132pd(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(C4E2E99E8C2B80000000        , vfnmsub132pd(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E2E99ECB                  , vfnmsub132pd(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(6262BD489E5A80              , vfnmsub132pd(zmm27, k0, zmm8, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(626225489E9200200000        , vfnmsub132ps(zmm26, k0, zmm11, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62F26D409EE6                , vfnmsub132ps(zmm4, k0, zmm18, zmm6)),
        INSTRUCTION_TEST(62B26D409EA4F023010000      , vfnmsub132ps(zmm4, k0, zmm18, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62F26D409E21                , vfnmsub132ps(zmm4, k0, zmm18, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62F26D409E627F              , vfnmsub132ps(zmm4, k0, zmm18, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62F26D409E6280              , vfnmsub132ps(zmm4, k0, zmm18, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62F26D409EA2C0DFFFFF        , vfnmsub132ps(zmm4, k0, zmm18, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(626225489E527F              , vfnmsub132ps(zmm26, k0, zmm11, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62F26D409EA200200000        , vfnmsub132ps(zmm4, k0, zmm18, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(626225489E92C0DFFFFF        , vfnmsub132ps(zmm26, k0, zmm11, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(C4E26D9E8C2B80000000        , vfnmsub132ps(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(626225489E11                , vfnmsub132ps(zmm26, k0, zmm11, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(622225489E94F034120000      , vfnmsub132ps(zmm26, k0, zmm11, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(624225489ED1                , vfnmsub132ps(zmm26, k0, zmm11, zmm9)),
        INSTRUCTION_TEST(62F26D489E4C2B02            , vfnmsub132ps(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F26D489ECB                , vfnmsub132ps(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(C4E26D9ECB                  , vfnmsub132ps(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(C4E2699E8C2B80000000        , vfnmsub132ps(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E2699ECB                  , vfnmsub132ps(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(626225489E5280              , vfnmsub132ps(zmm26, k0, zmm11, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(C4A2B99F9CF034120000        , vfnmsub132sd(xmm3, xmm8, qword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(C4E2B99F19                  , vfnmsub132sd(xmm3, xmm8, qword_ptr(rcx))),
        INSTRUCTION_TEST(C4E2B99F9A00FCFFFF          , vfnmsub132sd(xmm3, xmm8, qword_ptr(rdx, -1024))),
        INSTRUCTION_TEST(62B2BD089FDB                , vfnmsub132sd(xmm3, k0, xmm8, xmm19)),
        INSTRUCTION_TEST(C4E2B99F9AF8030000          , vfnmsub132sd(xmm3, xmm8, qword_ptr(rdx, 1016))),
        INSTRUCTION_TEST(C4E2B99F9A00040000          , vfnmsub132sd(xmm3, xmm8, qword_ptr(rdx, 1024))),
        INSTRUCTION_TEST(C4E2B99F9AF8FBFFFF          , vfnmsub132sd(xmm3, xmm8, qword_ptr(rdx, -1032))),
        INSTRUCTION_TEST(6242A5089FD5                , vfnmsub132sd(xmm26, k0, xmm11, xmm13)),
        INSTRUCTION_TEST(6262A5089F527F              , vfnmsub132sd(xmm26, k0, xmm11, qword_ptr(rdx, 1016))),
        INSTRUCTION_TEST(6262A5089F92F8FBFFFF        , vfnmsub132sd(xmm26, k0, xmm11, qword_ptr(rdx, -1032))),
        INSTRUCTION_TEST(6262A5089F5280              , vfnmsub132sd(xmm26, k0, xmm11, qword_ptr(rdx, -1024))),
        INSTRUCTION_TEST(6262A5089F11                , vfnmsub132sd(xmm26, k0, xmm11, qword_ptr(rcx))),
        INSTRUCTION_TEST(6222A5089F94F023010000      , vfnmsub132sd(xmm26, k0, xmm11, qword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(C4E2E99FCB                  , vfnmsub132sd(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(C4E2E99F8C2B80000000        , vfnmsub132sd(xmm1, xmm2, qword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(6262A5089F9200040000        , vfnmsub132sd(xmm26, k0, xmm11, qword_ptr(rdx, 1024))),
        INSTRUCTION_TEST(62824D009FF8                , vfnmsub132ss(xmm23, k0, xmm22, xmm24)),
        INSTRUCTION_TEST(62E24D009FBA00020000        , vfnmsub132ss(xmm23, k0, xmm22, dword_ptr(rdx, 512))),
        INSTRUCTION_TEST(62E24D009F7A7F              , vfnmsub132ss(xmm23, k0, xmm22, dword_ptr(rdx, 508))),
        INSTRUCTION_TEST(62E24D009FBAFCFDFFFF        , vfnmsub132ss(xmm23, k0, xmm22, dword_ptr(rdx, -516))),
        INSTRUCTION_TEST(62E24D009F7A80              , vfnmsub132ss(xmm23, k0, xmm22, dword_ptr(rdx, -512))),
        INSTRUCTION_TEST(62E24D009F39                , vfnmsub132ss(xmm23, k0, xmm22, dword_ptr(rcx))),
        INSTRUCTION_TEST(62E275009FDE                , vfnmsub132ss(xmm19, k0, xmm17, xmm6)),
        INSTRUCTION_TEST(62A24D009FBCF023010000      , vfnmsub132ss(xmm23, k0, xmm22, dword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62E275009F5A7F              , vfnmsub132ss(xmm19, k0, xmm17, dword_ptr(rdx, 508))),
        INSTRUCTION_TEST(C4E2699F8C2B80000000        , vfnmsub132ss(xmm1, xmm2, dword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E2699FCB                  , vfnmsub132ss(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(62A275009F9CF034120000      , vfnmsub132ss(xmm19, k0, xmm17, dword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62E275009F9A00020000        , vfnmsub132ss(xmm19, k0, xmm17, dword_ptr(rdx, 512))),
        INSTRUCTION_TEST(62E275009F5A80              , vfnmsub132ss(xmm19, k0, xmm17, dword_ptr(rdx, -512))),
        INSTRUCTION_TEST(62E275009F9AFCFDFFFF        , vfnmsub132ss(xmm19, k0, xmm17, dword_ptr(rdx, -516))),
        INSTRUCTION_TEST(62E275009F19                , vfnmsub132ss(xmm19, k0, xmm17, dword_ptr(rcx))),
        INSTRUCTION_TEST(62F2BD48AEBA00200000        , vfnmsub213pd(zmm7, k0, zmm8, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62E2ED40AE7A7F              , vfnmsub213pd(zmm23, k0, zmm18, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62F2BD48AEBAC0DFFFFF        , vfnmsub213pd(zmm7, k0, zmm8, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62F2BD48AE7A80              , vfnmsub213pd(zmm7, k0, zmm8, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62F2BD48AE39                , vfnmsub213pd(zmm7, k0, zmm8, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62B2BD48AEBCF034120000      , vfnmsub213pd(zmm7, k0, zmm8, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62F2BD48AEFF                , vfnmsub213pd(zmm7, k0, zmm8, zmm7)),
        INSTRUCTION_TEST(62E2ED40AEBA00200000        , vfnmsub213pd(zmm23, k0, zmm18, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62F2BD48AE7A7F              , vfnmsub213pd(zmm7, k0, zmm8, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62E2ED40AEBAC0DFFFFF        , vfnmsub213pd(zmm23, k0, zmm18, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62E2ED40AE39                , vfnmsub213pd(zmm23, k0, zmm18, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62A2ED40AEBCF023010000      , vfnmsub213pd(zmm23, k0, zmm18, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62C2ED40AEFB                , vfnmsub213pd(zmm23, k0, zmm18, zmm11)),
        INSTRUCTION_TEST(62F2ED48AE4C2B02            , vfnmsub213pd(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F2ED48AECB                , vfnmsub213pd(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(C4E2EDAE8C2B80000000        , vfnmsub213pd(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E2EDAECB                  , vfnmsub213pd(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(C4E2E9AE8C2B80000000        , vfnmsub213pd(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E2E9AECB                  , vfnmsub213pd(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(62E2ED40AE7A80              , vfnmsub213pd(zmm23, k0, zmm18, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62E21548AEAA00200000        , vfnmsub213ps(zmm21, k0, zmm13, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62423D40AEEE                , vfnmsub213ps(zmm29, k0, zmm24, zmm14)),
        INSTRUCTION_TEST(62223D40AEACF034120000      , vfnmsub213ps(zmm29, k0, zmm24, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62623D40AEAA00200000        , vfnmsub213ps(zmm29, k0, zmm24, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62623D40AE6A80              , vfnmsub213ps(zmm29, k0, zmm24, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62623D40AEAAC0DFFFFF        , vfnmsub213ps(zmm29, k0, zmm24, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62623D40AE6A7F              , vfnmsub213ps(zmm29, k0, zmm24, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62623D40AE29                , vfnmsub213ps(zmm29, k0, zmm24, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62E21548AEAAC0DFFFFF        , vfnmsub213ps(zmm21, k0, zmm13, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62E21548AE6A7F              , vfnmsub213ps(zmm21, k0, zmm13, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62E21548AE29                , vfnmsub213ps(zmm21, k0, zmm13, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62E21548AE6A80              , vfnmsub213ps(zmm21, k0, zmm13, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(C4E269AE8C2B80000000        , vfnmsub213ps(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E26DAECB                  , vfnmsub213ps(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(C4E26DAE8C2B80000000        , vfnmsub213ps(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E269AECB                  , vfnmsub213ps(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(62F26D48AE4C2B02            , vfnmsub213ps(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62E21548AEEA                , vfnmsub213ps(zmm21, k0, zmm13, zmm2)),
        INSTRUCTION_TEST(62A21548AEACF023010000      , vfnmsub213ps(zmm21, k0, zmm13, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62F26D48AECB                , vfnmsub213ps(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(6222C500AFACF023010000      , vfnmsub213sd(xmm29, k0, xmm23, qword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(6202C500AFEC                , vfnmsub213sd(xmm29, k0, xmm23, xmm28)),
        INSTRUCTION_TEST(6262C500AFAA00040000        , vfnmsub213sd(xmm29, k0, xmm23, qword_ptr(rdx, 1024))),
        INSTRUCTION_TEST(6262C500AF6A7F              , vfnmsub213sd(xmm29, k0, xmm23, qword_ptr(rdx, 1016))),
        INSTRUCTION_TEST(6262C500AFAAF8FBFFFF        , vfnmsub213sd(xmm29, k0, xmm23, qword_ptr(rdx, -1032))),
        INSTRUCTION_TEST(6262C500AF6A80              , vfnmsub213sd(xmm29, k0, xmm23, qword_ptr(rdx, -1024))),
        INSTRUCTION_TEST(6262C500AF29                , vfnmsub213sd(xmm29, k0, xmm23, qword_ptr(rcx))),
        INSTRUCTION_TEST(6222BD08AFCF                , vfnmsub213sd(xmm25, k0, xmm8, xmm23)),
        INSTRUCTION_TEST(6262BD08AF4A7F              , vfnmsub213sd(xmm25, k0, xmm8, qword_ptr(rdx, 1016))),
        INSTRUCTION_TEST(6262BD08AF8AF8FBFFFF        , vfnmsub213sd(xmm25, k0, xmm8, qword_ptr(rdx, -1032))),
        INSTRUCTION_TEST(6262BD08AF4A80              , vfnmsub213sd(xmm25, k0, xmm8, qword_ptr(rdx, -1024))),
        INSTRUCTION_TEST(6262BD08AF09                , vfnmsub213sd(xmm25, k0, xmm8, qword_ptr(rcx))),
        INSTRUCTION_TEST(6222BD08AF8CF034120000      , vfnmsub213sd(xmm25, k0, xmm8, qword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(C4E2E9AFCB                  , vfnmsub213sd(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(C4E2E9AF8C2B80000000        , vfnmsub213sd(xmm1, xmm2, qword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(6262BD08AF8A00040000        , vfnmsub213sd(xmm25, k0, xmm8, qword_ptr(rdx, 1024))),
        INSTRUCTION_TEST(62625D00AF01                , vfnmsub213ss(xmm24, k0, xmm20, dword_ptr(rcx))),
        INSTRUCTION_TEST(62425D00AFC7                , vfnmsub213ss(xmm24, k0, xmm20, xmm15)),
        INSTRUCTION_TEST(62625D00AF8200020000        , vfnmsub213ss(xmm24, k0, xmm20, dword_ptr(rdx, 512))),
        INSTRUCTION_TEST(62625D00AF427F              , vfnmsub213ss(xmm24, k0, xmm20, dword_ptr(rdx, 508))),
        INSTRUCTION_TEST(62625D00AF82FCFDFFFF        , vfnmsub213ss(xmm24, k0, xmm20, dword_ptr(rdx, -516))),
        INSTRUCTION_TEST(62625D00AF4280              , vfnmsub213ss(xmm24, k0, xmm20, dword_ptr(rdx, -512))),
        INSTRUCTION_TEST(62521D00AFF4                , vfnmsub213ss(xmm14, k0, xmm28, xmm12)),
        INSTRUCTION_TEST(62225D00AF84F034120000      , vfnmsub213ss(xmm24, k0, xmm20, dword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62721D00AF727F              , vfnmsub213ss(xmm14, k0, xmm28, dword_ptr(rdx, 508))),
        INSTRUCTION_TEST(C4E269AF8C2B80000000        , vfnmsub213ss(xmm1, xmm2, dword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E269AFCB                  , vfnmsub213ss(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(62321D00AFB4F023010000      , vfnmsub213ss(xmm14, k0, xmm28, dword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62721D00AFB200020000        , vfnmsub213ss(xmm14, k0, xmm28, dword_ptr(rdx, 512))),
        INSTRUCTION_TEST(62721D00AF7280              , vfnmsub213ss(xmm14, k0, xmm28, dword_ptr(rdx, -512))),
        INSTRUCTION_TEST(62721D00AFB2FCFDFFFF        , vfnmsub213ss(xmm14, k0, xmm28, dword_ptr(rdx, -516))),
        INSTRUCTION_TEST(62721D00AF31                , vfnmsub213ss(xmm14, k0, xmm28, dword_ptr(rcx))),
        INSTRUCTION_TEST(6262DD48BEAA00200000        , vfnmsub231pd(zmm29, k0, zmm4, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62F2ED48BECB                , vfnmsub231pd(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(6262DD48BEAAC0DFFFFF        , vfnmsub231pd(zmm29, k0, zmm4, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(6262DD48BE6A80              , vfnmsub231pd(zmm29, k0, zmm4, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(6262DD48BE29                , vfnmsub231pd(zmm29, k0, zmm4, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(6222DD48BEACF023010000      , vfnmsub231pd(zmm29, k0, zmm4, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(6202DD48BEE8                , vfnmsub231pd(zmm29, k0, zmm4, zmm24)),
        INSTRUCTION_TEST(62F2ED48BE4C2B02            , vfnmsub231pd(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(6262DD48BE6A7F              , vfnmsub231pd(zmm29, k0, zmm4, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62F2AD48BE8A00200000        , vfnmsub231pd(zmm1, k0, zmm10, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62F2AD48BE8AC0DFFFFF        , vfnmsub231pd(zmm1, k0, zmm10, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62F2AD48BE4A80              , vfnmsub231pd(zmm1, k0, zmm10, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62F2AD48BE09                , vfnmsub231pd(zmm1, k0, zmm10, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62B2AD48BE8CF034120000      , vfnmsub231pd(zmm1, k0, zmm10, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62D2AD48BECC                , vfnmsub231pd(zmm1, k0, zmm10, zmm12)),
        INSTRUCTION_TEST(C4E2EDBE8C2B80000000        , vfnmsub231pd(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E2EDBECB                  , vfnmsub231pd(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(C4E2E9BE8C2B80000000        , vfnmsub231pd(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E2E9BECB                  , vfnmsub231pd(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(62F2AD48BE4A7F              , vfnmsub231pd(zmm1, k0, zmm10, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62E22D40BE8A00200000        , vfnmsub231ps(zmm17, k0, zmm26, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62524D48BEC5                , vfnmsub231ps(zmm8, k0, zmm6, zmm13)),
        INSTRUCTION_TEST(62324D48BE84F023010000      , vfnmsub231ps(zmm8, k0, zmm6, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62E22D40BE4A7F              , vfnmsub231ps(zmm17, k0, zmm26, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62724D48BE4280              , vfnmsub231ps(zmm8, k0, zmm6, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62724D48BE8200200000        , vfnmsub231ps(zmm8, k0, zmm6, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62724D48BE427F              , vfnmsub231ps(zmm8, k0, zmm6, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62724D48BE01                , vfnmsub231ps(zmm8, k0, zmm6, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62E22D40BE8AC0DFFFFF        , vfnmsub231ps(zmm17, k0, zmm26, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62724D48BE82C0DFFFFF        , vfnmsub231ps(zmm8, k0, zmm6, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62E22D40BE09                , vfnmsub231ps(zmm17, k0, zmm26, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62E22D40BE4A80              , vfnmsub231ps(zmm17, k0, zmm26, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(C4E269BE8C2B80000000        , vfnmsub231ps(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E26DBECB                  , vfnmsub231ps(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(C4E26DBE8C2B80000000        , vfnmsub231ps(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E269BECB                  , vfnmsub231ps(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(62F26D48BE4C2B02            , vfnmsub231ps(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62A22D40BECB                , vfnmsub231ps(zmm17, k0, zmm26, zmm19)),
        INSTRUCTION_TEST(62A22D40BE8CF034120000      , vfnmsub231ps(zmm17, k0, zmm26, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62F26D48BECB                , vfnmsub231ps(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(62B2DD00BFB4F023010000      , vfnmsub231sd(xmm6, k0, xmm20, qword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62D2DD00BFF6                , vfnmsub231sd(xmm6, k0, xmm20, xmm14)),
        INSTRUCTION_TEST(62F2DD00BFB200040000        , vfnmsub231sd(xmm6, k0, xmm20, qword_ptr(rdx, 1024))),
        INSTRUCTION_TEST(62F2DD00BF727F              , vfnmsub231sd(xmm6, k0, xmm20, qword_ptr(rdx, 1016))),
        INSTRUCTION_TEST(62F2DD00BF7280              , vfnmsub231sd(xmm6, k0, xmm20, qword_ptr(rdx, -1024))),
        INSTRUCTION_TEST(62F2DD00BF31                , vfnmsub231sd(xmm6, k0, xmm20, qword_ptr(rcx))),
        INSTRUCTION_TEST(6282CD08BFDA                , vfnmsub231sd(xmm19, k0, xmm6, xmm26)),
        INSTRUCTION_TEST(62F2DD00BFB2F8FBFFFF        , vfnmsub231sd(xmm6, k0, xmm20, qword_ptr(rdx, -1032))),
        INSTRUCTION_TEST(62E2CD08BF5A7F              , vfnmsub231sd(xmm19, k0, xmm6, qword_ptr(rdx, 1016))),
        INSTRUCTION_TEST(62E2CD08BF9AF8FBFFFF        , vfnmsub231sd(xmm19, k0, xmm6, qword_ptr(rdx, -1032))),
        INSTRUCTION_TEST(62E2CD08BF5A80              , vfnmsub231sd(xmm19, k0, xmm6, qword_ptr(rdx, -1024))),
        INSTRUCTION_TEST(62E2CD08BF19                , vfnmsub231sd(xmm19, k0, xmm6, qword_ptr(rcx))),
        INSTRUCTION_TEST(62A2CD08BF9CF034120000      , vfnmsub231sd(xmm19, k0, xmm6, qword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(C4E2E9BFCB                  , vfnmsub231sd(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(C4E2E9BF8C2B80000000        , vfnmsub231sd(xmm1, xmm2, qword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62E2CD08BF9A00040000        , vfnmsub231sd(xmm19, k0, xmm6, qword_ptr(rdx, 1024))),
        INSTRUCTION_TEST(62421508BFD2                , vfnmsub231ss(xmm26, k0, xmm13, xmm10)),
        INSTRUCTION_TEST(62621508BF9200020000        , vfnmsub231ss(xmm26, k0, xmm13, dword_ptr(rdx, 512))),
        INSTRUCTION_TEST(62621508BF527F              , vfnmsub231ss(xmm26, k0, xmm13, dword_ptr(rdx, 508))),
        INSTRUCTION_TEST(62621508BF92FCFDFFFF        , vfnmsub231ss(xmm26, k0, xmm13, dword_ptr(rdx, -516))),
        INSTRUCTION_TEST(62621508BF5280              , vfnmsub231ss(xmm26, k0, xmm13, dword_ptr(rdx, -512))),
        INSTRUCTION_TEST(62621508BF11                , vfnmsub231ss(xmm26, k0, xmm13, dword_ptr(rcx))),
        INSTRUCTION_TEST(62221508BF94F023010000      , vfnmsub231ss(xmm26, k0, xmm13, dword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62726D00BFE9                , vfnmsub231ss(xmm13, k0, xmm18, xmm1)),
        INSTRUCTION_TEST(62726D00BFAAFCFDFFFF        , vfnmsub231ss(xmm13, k0, xmm18, dword_ptr(rdx, -516))),
        INSTRUCTION_TEST(62726D00BF6A7F              , vfnmsub231ss(xmm13, k0, xmm18, dword_ptr(rdx, 508))),
        INSTRUCTION_TEST(62726D00BF6A80              , vfnmsub231ss(xmm13, k0, xmm18, dword_ptr(rdx, -512))),
        INSTRUCTION_TEST(62726D00BF29                , vfnmsub231ss(xmm13, k0, xmm18, dword_ptr(rcx))),
        INSTRUCTION_TEST(62326D00BFACF034120000      , vfnmsub231ss(xmm13, k0, xmm18, dword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(C4E269BFCB                  , vfnmsub231ss(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(C4E269BF8C2B80000000        , vfnmsub231ss(xmm1, xmm2, dword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62726D00BFAA00020000        , vfnmsub231ss(xmm13, k0, xmm18, dword_ptr(rdx, 512))),
        INSTRUCTION_TEST(C4E36D7D8C2B8000000060      , vfnmsubpd(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128), ymm6)),
        INSTRUCTION_TEST(C4E3ED7D8C358000000030      , vfnmsubpd(ymm1, ymm2, ymm3, ymmword_ptr(rbp, rsi, 1, 128))),
        INSTRUCTION_TEST(C4E36D7DCB40                , vfnmsubpd(ymm1, ymm2, ymm3, ymm4)),
        INSTRUCTION_TEST(C4E3E97D8C358000000030      , vfnmsubpd(xmm1, xmm2, xmm3, xmmword_ptr(rbp, rsi, 1, 128))),
        INSTRUCTION_TEST(C4E3697DCB40                , vfnmsubpd(xmm1, xmm2, xmm3, xmm4)),
        INSTRUCTION_TEST(C4E3697D8C2B8000000060      , vfnmsubpd(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128), xmm6)),
        INSTRUCTION_TEST(C4E3697CCB40                , vfnmsubps(xmm1, xmm2, xmm3, xmm4)),
        INSTRUCTION_TEST(C4E3E97C8C358000000030      , vfnmsubps(xmm1, xmm2, xmm3, xmmword_ptr(rbp, rsi, 1, 128))),
        INSTRUCTION_TEST(C4E3697C8C2B8000000060      , vfnmsubps(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128), xmm6)),
        INSTRUCTION_TEST(C4E36D7CCB40                , vfnmsubps(ymm1, ymm2, ymm3, ymm4)),
        INSTRUCTION_TEST(C4E3ED7C8C358000000030      , vfnmsubps(ymm1, ymm2, ymm3, ymmword_ptr(rbp, rsi, 1, 128))),
        INSTRUCTION_TEST(C4E36D7C8C2B8000000060      , vfnmsubps(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128), ymm6)),
        INSTRUCTION_TEST(C4E3697F8C2B8000000060      , vfnmsubsd(xmm1, xmm2, qword_ptr(rbx, rbp, 1, 128), xmm6)),
        INSTRUCTION_TEST(C4E3E97F8C358000000030      , vfnmsubsd(xmm1, xmm2, xmm3, qword_ptr(rbp, rsi, 1, 128))),
        INSTRUCTION_TEST(C4E3697FCB40                , vfnmsubsd(xmm1, xmm2, xmm3, xmm4)),
        INSTRUCTION_TEST(C4E3697ECB40                , vfnmsubss(xmm1, xmm2, xmm3, xmm4)),
        INSTRUCTION_TEST(C4E3697E8C2B8000000060      , vfnmsubss(xmm1, xmm2, dword_ptr(rbx, rbp, 1, 128), xmm6)),
        INSTRUCTION_TEST(C4E3E97E8C358000000030      , vfnmsubss(xmm1, xmm2, xmm3, dword_ptr(rbp, rsi, 1, 128))),
        INSTRUCTION_TEST(62F3FD0866CA01              , vfpclasspd(k1, k0, xmm2, Imm(1))),
        INSTRUCTION_TEST(62F3FD08664C1A0801          , vfpclasspd(k1, k0, xmmword_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F3FD2866CA01              , vfpclasspd(k1, k0, ymm2, Imm(1))),
        INSTRUCTION_TEST(62F3FD28664C1A0401          , vfpclasspd(k1, k0, ymmword_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F3FD4866CA01              , vfpclasspd(k1, k0, zmm2, Imm(1))),
        INSTRUCTION_TEST(62F3FD48664C1A0201          , vfpclasspd(k1, k0, zmmword_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F37D4866CA01              , vfpclassps(k1, k0, zmm2, Imm(1))),
        INSTRUCTION_TEST(62F37D48664C1A0201          , vfpclassps(k1, k0, zmmword_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F37D28664C1A0401          , vfpclassps(k1, k0, ymmword_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F37D0866CA01              , vfpclassps(k1, k0, xmm2, Imm(1))),
        INSTRUCTION_TEST(62F37D08664C1A0801          , vfpclassps(k1, k0, xmmword_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F37D2866CA01              , vfpclassps(k1, k0, ymm2, Imm(1))),
        INSTRUCTION_TEST(62F3FD08674C1A1001          , vfpclasssd(k1, k0, qword_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F3FD0867CA01              , vfpclasssd(k1, k0, xmm2, Imm(1))),
        INSTRUCTION_TEST(62F37D08674C1A2001          , vfpclassss(k1, k0, dword_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F37D0867CA01              , vfpclassss(k1, k0, xmm2, Imm(1))),
        INSTRUCTION_TEST(8FE97C818C1A80000000        , vfrczpd(ymm1, ymmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(8FE97C81CA                  , vfrczpd(ymm1, ymm2)),
        INSTRUCTION_TEST(8FE97881CA                  , vfrczpd(xmm1, xmm2)),
        INSTRUCTION_TEST(8FE978818C1A80000000        , vfrczpd(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(8FE97880CA                  , vfrczps(xmm1, xmm2)),
        INSTRUCTION_TEST(8FE978808C1A80000000        , vfrczps(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(8FE97C80CA                  , vfrczps(ymm1, ymm2)),
        INSTRUCTION_TEST(8FE97C808C1A80000000        , vfrczps(ymm1, ymmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(8FE978838C1A80000000        , vfrczsd(xmm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(8FE97883CA                  , vfrczsd(xmm1, xmm2)),
        INSTRUCTION_TEST(8FE97882CA                  , vfrczss(xmm1, xmm2)),
        INSTRUCTION_TEST(8FE978828C1A80000000        , vfrczss(xmm1, dword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F2FD0842CA                , vgetexppd(xmm1, k0, xmm2)),
        INSTRUCTION_TEST(62F2FD4842B200200000        , vgetexppd(zmm6, k0, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62F2FD4842727F              , vgetexppd(zmm6, k0, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62F2FD4842B2C0DFFFFF        , vgetexppd(zmm6, k0, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62F2FD48427280              , vgetexppd(zmm6, k0, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62F2FD484231                , vgetexppd(zmm6, k0, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(6292FD4842F1                , vgetexppd(zmm6, k0, zmm25)),
        INSTRUCTION_TEST(6272FD4842B200200000        , vgetexppd(zmm14, k0, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(6272FD4842727F              , vgetexppd(zmm14, k0, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(6272FD4842B2C0DFFFFF        , vgetexppd(zmm14, k0, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62B2FD4842B4F034120000      , vgetexppd(zmm6, k0, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(6272FD484231                , vgetexppd(zmm14, k0, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(6232FD4842B4F023010000      , vgetexppd(zmm14, k0, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(6212FD4842F1                , vgetexppd(zmm14, k0, zmm25)),
        INSTRUCTION_TEST(62F2FD48424C1A02            , vgetexppd(zmm1, k0, zmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F2FD4842CA                , vgetexppd(zmm1, k0, zmm2)),
        INSTRUCTION_TEST(62F2FD28424C1A04            , vgetexppd(ymm1, k0, ymmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F2FD2842CA                , vgetexppd(ymm1, k0, ymm2)),
        INSTRUCTION_TEST(62F2FD08424C1A08            , vgetexppd(xmm1, k0, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(6272FD48427280              , vgetexppd(zmm14, k0, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62F27D48424C1A02            , vgetexpps(zmm1, k0, zmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62B27D4842F8                , vgetexpps(zmm7, k0, zmm16)),
        INSTRUCTION_TEST(62B27D4842BCF034120000      , vgetexpps(zmm7, k0, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62F27D484239                , vgetexpps(zmm7, k0, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62F27D4842BA00200000        , vgetexpps(zmm7, k0, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62F27D4842BAC0DFFFFF        , vgetexpps(zmm7, k0, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62F27D48427A7F              , vgetexpps(zmm7, k0, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62F27D48428A00200000        , vgetexpps(zmm1, k0, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62F27D48427A80              , vgetexpps(zmm7, k0, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62F27D48424A7F              , vgetexpps(zmm1, k0, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62F27D28424C1A04            , vgetexpps(ymm1, k0, ymmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F27D48424A80              , vgetexpps(zmm1, k0, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62F27D48428AC0DFFFFF        , vgetexpps(zmm1, k0, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62F27D08424C1A08            , vgetexpps(xmm1, k0, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F27D2842CA                , vgetexpps(ymm1, k0, ymm2)),
        INSTRUCTION_TEST(62F27D0842CA                , vgetexpps(xmm1, k0, xmm2)),
        INSTRUCTION_TEST(62F27D4842CE                , vgetexpps(zmm1, k0, zmm6)),
        INSTRUCTION_TEST(62B27D48428CF023010000      , vgetexpps(zmm1, k0, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62F27D484209                , vgetexpps(zmm1, k0, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62F27D4842CA                , vgetexpps(zmm1, k0, zmm2)),
        INSTRUCTION_TEST(62B2C5084394F023010000      , vgetexpsd(xmm2, k0, xmm7, qword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62F2C50843D2                , vgetexpsd(xmm2, k0, xmm7, xmm2)),
        INSTRUCTION_TEST(62F2C508439200040000        , vgetexpsd(xmm2, k0, xmm7, qword_ptr(rdx, 1024))),
        INSTRUCTION_TEST(62F2C50843527F              , vgetexpsd(xmm2, k0, xmm7, qword_ptr(rdx, 1016))),
        INSTRUCTION_TEST(62F2C5084392F8FBFFFF        , vgetexpsd(xmm2, k0, xmm7, qword_ptr(rdx, -1032))),
        INSTRUCTION_TEST(62F2C508435280              , vgetexpsd(xmm2, k0, xmm7, qword_ptr(rdx, -1024))),
        INSTRUCTION_TEST(62F2C5084311                , vgetexpsd(xmm2, k0, xmm7, qword_ptr(rcx))),
        INSTRUCTION_TEST(62728D0843D5                , vgetexpsd(xmm10, k0, xmm14, xmm5)),
        INSTRUCTION_TEST(62F2ED0843CB                , vgetexpsd(xmm1, k0, xmm2, xmm3)),
        INSTRUCTION_TEST(62728D0843527F              , vgetexpsd(xmm10, k0, xmm14, qword_ptr(rdx, 1016))),
        INSTRUCTION_TEST(62728D084392F8FBFFFF        , vgetexpsd(xmm10, k0, xmm14, qword_ptr(rdx, -1032))),
        INSTRUCTION_TEST(62728D08435280              , vgetexpsd(xmm10, k0, xmm14, qword_ptr(rdx, -1024))),
        INSTRUCTION_TEST(62728D084311                , vgetexpsd(xmm10, k0, xmm14, qword_ptr(rcx))),
        INSTRUCTION_TEST(62328D084394F034120000      , vgetexpsd(xmm10, k0, xmm14, qword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62F2ED08434C2B10            , vgetexpsd(xmm1, k0, xmm2, qword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62728D08439200040000        , vgetexpsd(xmm10, k0, xmm14, qword_ptr(rdx, 1024))),
        INSTRUCTION_TEST(6282750843E2                , vgetexpss(xmm20, k0, xmm1, xmm26)),
        INSTRUCTION_TEST(62E2750843A200020000        , vgetexpss(xmm20, k0, xmm1, dword_ptr(rdx, 512))),
        INSTRUCTION_TEST(62E2750843627F              , vgetexpss(xmm20, k0, xmm1, dword_ptr(rdx, 508))),
        INSTRUCTION_TEST(62E2750843A2FCFDFFFF        , vgetexpss(xmm20, k0, xmm1, dword_ptr(rdx, -516))),
        INSTRUCTION_TEST(62E27508436280              , vgetexpss(xmm20, k0, xmm1, dword_ptr(rdx, -512))),
        INSTRUCTION_TEST(62E275084321                , vgetexpss(xmm20, k0, xmm1, dword_ptr(rcx))),
        INSTRUCTION_TEST(62A2550043C7                , vgetexpss(xmm16, k0, xmm21, xmm23)),
        INSTRUCTION_TEST(62A2750843A4F023010000      , vgetexpss(xmm20, k0, xmm1, dword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62E2550043427F              , vgetexpss(xmm16, k0, xmm21, dword_ptr(rdx, 508))),
        INSTRUCTION_TEST(62E255004382FCFDFFFF        , vgetexpss(xmm16, k0, xmm21, dword_ptr(rdx, -516))),
        INSTRUCTION_TEST(62E25500434280              , vgetexpss(xmm16, k0, xmm21, dword_ptr(rdx, -512))),
        INSTRUCTION_TEST(62E255004301                , vgetexpss(xmm16, k0, xmm21, dword_ptr(rcx))),
        INSTRUCTION_TEST(62A255004384F034120000      , vgetexpss(xmm16, k0, xmm21, dword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62F26D0843CB                , vgetexpss(xmm1, k0, xmm2, xmm3)),
        INSTRUCTION_TEST(62F26D08434C2B20            , vgetexpss(xmm1, k0, xmm2, dword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62E25500438200020000        , vgetexpss(xmm16, k0, xmm21, dword_ptr(rdx, 512))),
        INSTRUCTION_TEST(62F3FD482692002000007B      , vgetmantpd(zmm2, k0, zmmword_ptr(rdx, 8192), Imm(123))),
        INSTRUCTION_TEST(6253FD4826CA7B              , vgetmantpd(zmm9, k0, zmm10, Imm(123))),
        INSTRUCTION_TEST(6253FD4826CAAB              , vgetmantpd(zmm9, k0, zmm10, Imm(171))),
        INSTRUCTION_TEST(6233FD48268CF0341200007B    , vgetmantpd(zmm9, k0, zmmword_ptr(rax, r14, 8, 4660), Imm(123))),
        INSTRUCTION_TEST(6273FD48264A7F7B            , vgetmantpd(zmm9, k0, zmmword_ptr(rdx, 8128), Imm(123))),
        INSTRUCTION_TEST(6273FD48264A807B            , vgetmantpd(zmm9, k0, zmmword_ptr(rdx, -8192), Imm(123))),
        INSTRUCTION_TEST(6273FD48268AC0DFFFFF7B      , vgetmantpd(zmm9, k0, zmmword_ptr(rdx, -8256), Imm(123))),
        INSTRUCTION_TEST(6273FD48268A002000007B      , vgetmantpd(zmm9, k0, zmmword_ptr(rdx, 8192), Imm(123))),
        INSTRUCTION_TEST(62F3FD4826527F7B            , vgetmantpd(zmm2, k0, zmmword_ptr(rdx, 8128), Imm(123))),
        INSTRUCTION_TEST(6273FD4826097B              , vgetmantpd(zmm9, k0, zmmword_ptr(rcx), Imm(123))),
        INSTRUCTION_TEST(62F3FD482692C0DFFFFF7B      , vgetmantpd(zmm2, k0, zmmword_ptr(rdx, -8256), Imm(123))),
        INSTRUCTION_TEST(62B3FD482694F0230100007B    , vgetmantpd(zmm2, k0, zmmword_ptr(rax, r14, 8, 291), Imm(123))),
        INSTRUCTION_TEST(62F3FD4826117B              , vgetmantpd(zmm2, k0, zmmword_ptr(rcx), Imm(123))),
        INSTRUCTION_TEST(62F3FD482652807B            , vgetmantpd(zmm2, k0, zmmword_ptr(rdx, -8192), Imm(123))),
        INSTRUCTION_TEST(6293FD4826D2AB              , vgetmantpd(zmm2, k0, zmm26, Imm(171))),
        INSTRUCTION_TEST(6293FD4826D27B              , vgetmantpd(zmm2, k0, zmm26, Imm(123))),
        INSTRUCTION_TEST(62F3FD48264C1A0201          , vgetmantpd(zmm1, k0, zmmword_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F3FD4826CA01              , vgetmantpd(zmm1, k0, zmm2, Imm(1))),
        INSTRUCTION_TEST(62F3FD28264C1A0401          , vgetmantpd(ymm1, k0, ymmword_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F3FD2826CA01              , vgetmantpd(ymm1, k0, ymm2, Imm(1))),
        INSTRUCTION_TEST(62F3FD08264C1A0801          , vgetmantpd(xmm1, k0, xmmword_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F3FD0826CA01              , vgetmantpd(xmm1, k0, xmm2, Imm(1))),
        INSTRUCTION_TEST(62E37D4826B2002000007B      , vgetmantps(zmm22, k0, zmmword_ptr(rdx, 8192), Imm(123))),
        INSTRUCTION_TEST(62037D4826E97B              , vgetmantps(zmm29, k0, zmm25, Imm(123))),
        INSTRUCTION_TEST(62037D4826E9AB              , vgetmantps(zmm29, k0, zmm25, Imm(171))),
        INSTRUCTION_TEST(62237D4826ACF0341200007B    , vgetmantps(zmm29, k0, zmmword_ptr(rax, r14, 8, 4660), Imm(123))),
        INSTRUCTION_TEST(62637D4826AA002000007B      , vgetmantps(zmm29, k0, zmmword_ptr(rdx, 8192), Imm(123))),
        INSTRUCTION_TEST(62637D48266A807B            , vgetmantps(zmm29, k0, zmmword_ptr(rdx, -8192), Imm(123))),
        INSTRUCTION_TEST(62637D4826AAC0DFFFFF7B      , vgetmantps(zmm29, k0, zmmword_ptr(rdx, -8256), Imm(123))),
        INSTRUCTION_TEST(62637D48266A7F7B            , vgetmantps(zmm29, k0, zmmword_ptr(rdx, 8128), Imm(123))),
        INSTRUCTION_TEST(62E37D4826727F7B            , vgetmantps(zmm22, k0, zmmword_ptr(rdx, 8128), Imm(123))),
        INSTRUCTION_TEST(62637D4826297B              , vgetmantps(zmm29, k0, zmmword_ptr(rcx), Imm(123))),
        INSTRUCTION_TEST(62E37D4826B2C0DFFFFF7B      , vgetmantps(zmm22, k0, zmmword_ptr(rdx, -8256), Imm(123))),
        INSTRUCTION_TEST(62E37D482672807B            , vgetmantps(zmm22, k0, zmmword_ptr(rdx, -8192), Imm(123))),
        INSTRUCTION_TEST(62E37D4826317B              , vgetmantps(zmm22, k0, zmmword_ptr(rcx), Imm(123))),
        INSTRUCTION_TEST(62F37D08264C1A0801          , vgetmantps(xmm1, k0, xmmword_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F37D2826CA01              , vgetmantps(ymm1, k0, ymm2, Imm(1))),
        INSTRUCTION_TEST(62F37D28264C1A0401          , vgetmantps(ymm1, k0, ymmword_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F37D4826CA01              , vgetmantps(zmm1, k0, zmm2, Imm(1))),
        INSTRUCTION_TEST(62F37D0826CA01              , vgetmantps(xmm1, k0, xmm2, Imm(1))),
        INSTRUCTION_TEST(62837D4826F47B              , vgetmantps(zmm22, k0, zmm28, Imm(123))),
        INSTRUCTION_TEST(62837D4826F4AB              , vgetmantps(zmm22, k0, zmm28, Imm(171))),
        INSTRUCTION_TEST(62A37D4826B4F0230100007B    , vgetmantps(zmm22, k0, zmmword_ptr(rax, r14, 8, 291), Imm(123))),
        INSTRUCTION_TEST(62F37D48264C1A0201          , vgetmantps(zmm1, k0, zmmword_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62B3A500279CF0230100007B    , vgetmantsd(xmm3, k0, xmm27, qword_ptr(rax, r14, 8, 291), Imm(123))),
        INSTRUCTION_TEST(62D3A50027D87B              , vgetmantsd(xmm3, k0, xmm27, xmm8, Imm(123))),
        INSTRUCTION_TEST(62F3A500279A000400007B      , vgetmantsd(xmm3, k0, xmm27, qword_ptr(rdx, 1024), Imm(123))),
        INSTRUCTION_TEST(62F3A500275A7F7B            , vgetmantsd(xmm3, k0, xmm27, qword_ptr(rdx, 1016), Imm(123))),
        INSTRUCTION_TEST(62F3A500279AF8FBFFFF7B      , vgetmantsd(xmm3, k0, xmm27, qword_ptr(rdx, -1032), Imm(123))),
        INSTRUCTION_TEST(62F3A500275A807B            , vgetmantsd(xmm3, k0, xmm27, qword_ptr(rdx, -1024), Imm(123))),
        INSTRUCTION_TEST(62F3A50027197B              , vgetmantsd(xmm3, k0, xmm27, qword_ptr(rcx), Imm(123))),
        INSTRUCTION_TEST(62D3A50027D8AB              , vgetmantsd(xmm3, k0, xmm27, xmm8, Imm(171))),
        INSTRUCTION_TEST(62A3E50827E1AB              , vgetmantsd(xmm20, k0, xmm3, xmm17, Imm(171))),
        INSTRUCTION_TEST(62E3E50827627F7B            , vgetmantsd(xmm20, k0, xmm3, qword_ptr(rdx, 1016), Imm(123))),
        INSTRUCTION_TEST(62E3E50827A2000400007B      , vgetmantsd(xmm20, k0, xmm3, qword_ptr(rdx, 1024), Imm(123))),
        INSTRUCTION_TEST(62E3E50827A2F8FBFFFF7B      , vgetmantsd(xmm20, k0, xmm3, qword_ptr(rdx, -1032), Imm(123))),
        INSTRUCTION_TEST(62E3E5082762807B            , vgetmantsd(xmm20, k0, xmm3, qword_ptr(rdx, -1024), Imm(123))),
        INSTRUCTION_TEST(62E3E50827217B              , vgetmantsd(xmm20, k0, xmm3, qword_ptr(rcx), Imm(123))),
        INSTRUCTION_TEST(62A3E50827A4F0341200007B    , vgetmantsd(xmm20, k0, xmm3, qword_ptr(rax, r14, 8, 4660), Imm(123))),
        INSTRUCTION_TEST(62F3ED0827CB01              , vgetmantsd(xmm1, k0, xmm2, xmm3, Imm(1))),
        INSTRUCTION_TEST(62F3ED08274C2B1001          , vgetmantsd(xmm1, k0, xmm2, qword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62A3E50827E17B              , vgetmantsd(xmm20, k0, xmm3, xmm17, Imm(123))),
        INSTRUCTION_TEST(62D36D0827DCAB              , vgetmantss(xmm3, k0, xmm2, xmm12, Imm(171))),
        INSTRUCTION_TEST(62D36D0827DC7B              , vgetmantss(xmm3, k0, xmm2, xmm12, Imm(123))),
        INSTRUCTION_TEST(62F36D08279A000200007B      , vgetmantss(xmm3, k0, xmm2, dword_ptr(rdx, 512), Imm(123))),
        INSTRUCTION_TEST(62F36D08275A7F7B            , vgetmantss(xmm3, k0, xmm2, dword_ptr(rdx, 508), Imm(123))),
        INSTRUCTION_TEST(62F36D08279AFCFDFFFF7B      , vgetmantss(xmm3, k0, xmm2, dword_ptr(rdx, -516), Imm(123))),
        INSTRUCTION_TEST(62F36D08275A807B            , vgetmantss(xmm3, k0, xmm2, dword_ptr(rdx, -512), Imm(123))),
        INSTRUCTION_TEST(62F36D0827197B              , vgetmantss(xmm3, k0, xmm2, dword_ptr(rcx), Imm(123))),
        INSTRUCTION_TEST(62B36D08279CF0230100007B    , vgetmantss(xmm3, k0, xmm2, dword_ptr(rax, r14, 8, 291), Imm(123))),
        INSTRUCTION_TEST(62731D0827EEAB              , vgetmantss(xmm13, k0, xmm12, xmm6, Imm(171))),
        INSTRUCTION_TEST(62731D0827AA000200007B      , vgetmantss(xmm13, k0, xmm12, dword_ptr(rdx, 512), Imm(123))),
        INSTRUCTION_TEST(62731D08276A7F7B            , vgetmantss(xmm13, k0, xmm12, dword_ptr(rdx, 508), Imm(123))),
        INSTRUCTION_TEST(62731D0827AAFCFDFFFF7B      , vgetmantss(xmm13, k0, xmm12, dword_ptr(rdx, -516), Imm(123))),
        INSTRUCTION_TEST(62731D08276A807B            , vgetmantss(xmm13, k0, xmm12, dword_ptr(rdx, -512), Imm(123))),
        INSTRUCTION_TEST(62731D0827297B              , vgetmantss(xmm13, k0, xmm12, dword_ptr(rcx), Imm(123))),
        INSTRUCTION_TEST(62331D0827ACF0341200007B    , vgetmantss(xmm13, k0, xmm12, dword_ptr(rax, r14, 8, 4660), Imm(123))),
        INSTRUCTION_TEST(62F36D0827CB01              , vgetmantss(xmm1, k0, xmm2, xmm3, Imm(1))),
        INSTRUCTION_TEST(62F36D08274C2B2001          , vgetmantss(xmm1, k0, xmm2, dword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62731D0827EE7B              , vgetmantss(xmm13, k0, xmm12, xmm6, Imm(123))),
        INSTRUCTION_TEST(62F3ED48CF4C11010F          , vgf2p8affineinvqb(zmm1, k0, zmm2, zmmword_ptr(rcx, rdx, 1, 64), 15)),
        INSTRUCTION_TEST(62F3ED48CF4C2B0201          , vgf2p8affineinvqb(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F3ED48CFCB01              , vgf2p8affineinvqb(zmm1, k0, zmm2, zmm3, Imm(1))),
        INSTRUCTION_TEST(C4E3E9CFCB01                , vgf2p8affineinvqb(xmm1, xmm2, xmm3, Imm(1))),
        INSTRUCTION_TEST(C4E3EDCFCB01                , vgf2p8affineinvqb(ymm1, ymm2, ymm3, Imm(1))),
        INSTRUCTION_TEST(C4E3E9CF8C2B8000000001      , vgf2p8affineinvqb(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(C4E3EDCF8C2B8000000001      , vgf2p8affineinvqb(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F3ED48CE4C2B0201          , vgf2p8affineqb(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F3ED48CE4C11010F          , vgf2p8affineqb(zmm1, k0, zmm2, zmmword_ptr(rcx, rdx, 1, 64), 15)),
        INSTRUCTION_TEST(62F3ED48CECB01              , vgf2p8affineqb(zmm1, k0, zmm2, zmm3, Imm(1))),
        INSTRUCTION_TEST(C4E3E9CECB01                , vgf2p8affineqb(xmm1, xmm2, xmm3, Imm(1))),
        INSTRUCTION_TEST(C4E3EDCECB01                , vgf2p8affineqb(ymm1, ymm2, ymm3, Imm(1))),
        INSTRUCTION_TEST(C4E3E9CE8C2B8000000001      , vgf2p8affineqb(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(C4E3EDCE8C2B8000000001      , vgf2p8affineqb(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F26D48CF4C2B02            , vgf2p8mulb(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F26D48CFCB                , vgf2p8mulb(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(C4E26DCF8C2B80000000        , vgf2p8mulb(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F26D48CF4C1101            , vgf2p8mulb(zmm1, k0, zmm2, zmmword_ptr(rcx, rdx, 1, 64))),
        INSTRUCTION_TEST(C4E269CF8C2B80000000        , vgf2p8mulb(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E269CFCB                  , vgf2p8mulb(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(C4E26DCFCB                  , vgf2p8mulb(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(C5E97CCB                    , vhaddpd(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(C5E97C8C2B80000000          , vhaddpd(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C5ED7CCB                    , vhaddpd(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(C5ED7C8C2B80000000          , vhaddpd(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C5EB7CCB                    , vhaddps(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(C5EB7C8C2B80000000          , vhaddps(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C5EF7CCB                    , vhaddps(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(C5EF7C8C2B80000000          , vhaddps(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C5ED7D8C2B80000000          , vhsubpd(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C5ED7DCB                    , vhsubpd(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(C5E97DCB                    , vhsubpd(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(C5E97D8C2B80000000          , vhsubpd(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C5EB7DCB                    , vhsubps(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(C5EB7D8C2B80000000          , vhsubps(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C5EF7DCB                    , vhsubps(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(C5EF7D8C2B80000000          , vhsubps(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E36D18CB01                , vinsertf128(ymm1, ymm2, xmm3, Imm(1))),
        INSTRUCTION_TEST(C4E36D188C2B8000000001      , vinsertf128(ymm1, ymm2, xmmword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62732D40189A000800007B      , vinsertf32x4(zmm11, k0, zmm26, xmmword_ptr(rdx, 2048), Imm(123))),
        INSTRUCTION_TEST(62737D401882000800007B      , vinsertf32x4(zmm8, k0, zmm16, xmmword_ptr(rdx, 2048), Imm(123))),
        INSTRUCTION_TEST(62737D4018427F7B            , vinsertf32x4(zmm8, k0, zmm16, xmmword_ptr(rdx, 2032), Imm(123))),
        INSTRUCTION_TEST(62737D401882F0F7FFFF7B      , vinsertf32x4(zmm8, k0, zmm16, xmmword_ptr(rdx, -2064), Imm(123))),
        INSTRUCTION_TEST(62737D401842807B            , vinsertf32x4(zmm8, k0, zmm16, xmmword_ptr(rdx, -2048), Imm(123))),
        INSTRUCTION_TEST(62737D4018017B              , vinsertf32x4(zmm8, k0, zmm16, xmmword_ptr(rcx), Imm(123))),
        INSTRUCTION_TEST(62337D401884F0341200007B    , vinsertf32x4(zmm8, k0, zmm16, xmmword_ptr(rax, r14, 8, 4660), Imm(123))),
        INSTRUCTION_TEST(62337D4018C67B              , vinsertf32x4(zmm8, k0, zmm16, xmm22, Imm(123))),
        INSTRUCTION_TEST(62732D40185A7F7B            , vinsertf32x4(zmm11, k0, zmm26, xmmword_ptr(rdx, 2032), Imm(123))),
        INSTRUCTION_TEST(62337D4018C6AB              , vinsertf32x4(zmm8, k0, zmm16, xmm22, Imm(171))),
        INSTRUCTION_TEST(62732D40185A807B            , vinsertf32x4(zmm11, k0, zmm26, xmmword_ptr(rdx, -2048), Imm(123))),
        INSTRUCTION_TEST(62732D4018197B              , vinsertf32x4(zmm11, k0, zmm26, xmmword_ptr(rcx), Imm(123))),
        INSTRUCTION_TEST(62732D4018DBAB              , vinsertf32x4(zmm11, k0, zmm26, xmm3, Imm(171))),
        INSTRUCTION_TEST(62732D40189AF0F7FFFF7B      , vinsertf32x4(zmm11, k0, zmm26, xmmword_ptr(rdx, -2064), Imm(123))),
        INSTRUCTION_TEST(62732D4018DB7B              , vinsertf32x4(zmm11, k0, zmm26, xmm3, Imm(123))),
        INSTRUCTION_TEST(62332D40189CF0230100007B    , vinsertf32x4(zmm11, k0, zmm26, xmmword_ptr(rax, r14, 8, 291), Imm(123))),
        INSTRUCTION_TEST(62F36D48184C2B0801          , vinsertf32x4(zmm1, k0, zmm2, xmmword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F36D4818CB01              , vinsertf32x4(zmm1, k0, zmm2, xmm3, Imm(1))),
        INSTRUCTION_TEST(62F36D28184C2B0801          , vinsertf32x4(ymm1, k0, ymm2, xmmword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F36D2818CB01              , vinsertf32x4(ymm1, k0, ymm2, xmm3, Imm(1))),
        INSTRUCTION_TEST(62F36D481ACB01              , vinsertf32x8(zmm1, k0, zmm2, ymm3, Imm(1))),
        INSTRUCTION_TEST(62F36D481A4C2B0401          , vinsertf32x8(zmm1, k0, zmm2, ymmword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F3ED48184C2B0801          , vinsertf64x2(zmm1, k0, zmm2, xmmword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F3ED4818CB01              , vinsertf64x2(zmm1, k0, zmm2, xmm3, Imm(1))),
        INSTRUCTION_TEST(62F3ED2818CB01              , vinsertf64x2(ymm1, k0, ymm2, xmm3, Imm(1))),
        INSTRUCTION_TEST(62F3ED28184C2B0801          , vinsertf64x2(ymm1, k0, ymm2, xmmword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62C3B5481AF47B              , vinsertf64x4(zmm22, k0, zmm9, ymm12, Imm(123))),
        INSTRUCTION_TEST(62E3B5481A727F7B            , vinsertf64x4(zmm22, k0, zmm9, ymmword_ptr(rdx, 4064), Imm(123))),
        INSTRUCTION_TEST(62E3B5481AB2E0EFFFFF7B      , vinsertf64x4(zmm22, k0, zmm9, ymmword_ptr(rdx, -4128), Imm(123))),
        INSTRUCTION_TEST(62E3B5481A72807B            , vinsertf64x4(zmm22, k0, zmm9, ymmword_ptr(rdx, -4096), Imm(123))),
        INSTRUCTION_TEST(62E3B5481A317B              , vinsertf64x4(zmm22, k0, zmm9, ymmword_ptr(rcx), Imm(123))),
        INSTRUCTION_TEST(62A3B5481AB4F0341200007B    , vinsertf64x4(zmm22, k0, zmm9, ymmword_ptr(rax, r14, 8, 4660), Imm(123))),
        INSTRUCTION_TEST(62C3B5481AF4AB              , vinsertf64x4(zmm22, k0, zmm9, ymm12, Imm(171))),
        INSTRUCTION_TEST(62E3B5481AB2001000007B      , vinsertf64x4(zmm22, k0, zmm9, ymmword_ptr(rdx, 4096), Imm(123))),
        INSTRUCTION_TEST(62F3D5481A8A001000007B      , vinsertf64x4(zmm1, k0, zmm5, ymmword_ptr(rdx, 4096), Imm(123))),
        INSTRUCTION_TEST(62F3D5481A4A807B            , vinsertf64x4(zmm1, k0, zmm5, ymmword_ptr(rdx, -4096), Imm(123))),
        INSTRUCTION_TEST(62F3D5481A8AE0EFFFFF7B      , vinsertf64x4(zmm1, k0, zmm5, ymmword_ptr(rdx, -4128), Imm(123))),
        INSTRUCTION_TEST(62F3D5481A097B              , vinsertf64x4(zmm1, k0, zmm5, ymmword_ptr(rcx), Imm(123))),
        INSTRUCTION_TEST(62B3D5481A8CF0230100007B    , vinsertf64x4(zmm1, k0, zmm5, ymmword_ptr(rax, r14, 8, 291), Imm(123))),
        INSTRUCTION_TEST(62F3D5481ACFAB              , vinsertf64x4(zmm1, k0, zmm5, ymm7, Imm(171))),
        INSTRUCTION_TEST(62F3D5481ACF7B              , vinsertf64x4(zmm1, k0, zmm5, ymm7, Imm(123))),
        INSTRUCTION_TEST(62F3ED481A4C2B0401          , vinsertf64x4(zmm1, k0, zmm2, ymmword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F3ED481ACB01              , vinsertf64x4(zmm1, k0, zmm2, ymm3, Imm(1))),
        INSTRUCTION_TEST(62F3D5481A4A7F7B            , vinsertf64x4(zmm1, k0, zmm5, ymmword_ptr(rdx, 4064), Imm(123))),
        INSTRUCTION_TEST(C4E36D388C2B8000000001      , vinserti128(ymm1, ymm2, xmmword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(C4E36D38CB01                , vinserti128(ymm1, ymm2, xmm3, Imm(1))),
        INSTRUCTION_TEST(62E31548388A000800007B      , vinserti32x4(zmm17, k0, zmm13, xmmword_ptr(rdx, 2048), Imm(123))),
        INSTRUCTION_TEST(62F35D48387A7F7B            , vinserti32x4(zmm7, k0, zmm4, xmmword_ptr(rdx, 2032), Imm(123))),
        INSTRUCTION_TEST(62F35D4838BAF0F7FFFF7B      , vinserti32x4(zmm7, k0, zmm4, xmmword_ptr(rdx, -2064), Imm(123))),
        INSTRUCTION_TEST(62F35D48387A807B            , vinserti32x4(zmm7, k0, zmm4, xmmword_ptr(rdx, -2048), Imm(123))),
        INSTRUCTION_TEST(62F35D4838397B              , vinserti32x4(zmm7, k0, zmm4, xmmword_ptr(rcx), Imm(123))),
        INSTRUCTION_TEST(62F35D4838BA000800007B      , vinserti32x4(zmm7, k0, zmm4, xmmword_ptr(rdx, 2048), Imm(123))),
        INSTRUCTION_TEST(62B35D4838BCF0341200007B    , vinserti32x4(zmm7, k0, zmm4, xmmword_ptr(rax, r14, 8, 4660), Imm(123))),
        INSTRUCTION_TEST(62935D4838F8AB              , vinserti32x4(zmm7, k0, zmm4, xmm24, Imm(171))),
        INSTRUCTION_TEST(62935D4838F87B              , vinserti32x4(zmm7, k0, zmm4, xmm24, Imm(123))),
        INSTRUCTION_TEST(62E31548384A7F7B            , vinserti32x4(zmm17, k0, zmm13, xmmword_ptr(rdx, 2032), Imm(123))),
        INSTRUCTION_TEST(62C3154838CA7B              , vinserti32x4(zmm17, k0, zmm13, xmm10, Imm(123))),
        INSTRUCTION_TEST(62E31548384A807B            , vinserti32x4(zmm17, k0, zmm13, xmmword_ptr(rdx, -2048), Imm(123))),
        INSTRUCTION_TEST(62E3154838097B              , vinserti32x4(zmm17, k0, zmm13, xmmword_ptr(rcx), Imm(123))),
        INSTRUCTION_TEST(62A31548388CF0230100007B    , vinserti32x4(zmm17, k0, zmm13, xmmword_ptr(rax, r14, 8, 291), Imm(123))),
        INSTRUCTION_TEST(62C3154838CAAB              , vinserti32x4(zmm17, k0, zmm13, xmm10, Imm(171))),
        INSTRUCTION_TEST(62F36D48384C2B0801          , vinserti32x4(zmm1, k0, zmm2, xmmword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F36D4838CB01              , vinserti32x4(zmm1, k0, zmm2, xmm3, Imm(1))),
        INSTRUCTION_TEST(62F36D28384C2B0801          , vinserti32x4(ymm1, k0, ymm2, xmmword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F36D2838CB01              , vinserti32x4(ymm1, k0, ymm2, xmm3, Imm(1))),
        INSTRUCTION_TEST(62E31548388AF0F7FFFF7B      , vinserti32x4(zmm17, k0, zmm13, xmmword_ptr(rdx, -2064), Imm(123))),
        INSTRUCTION_TEST(62F36D483ACB01              , vinserti32x8(zmm1, k0, zmm2, ymm3, Imm(1))),
        INSTRUCTION_TEST(62F36D483A4C2B0401          , vinserti32x8(zmm1, k0, zmm2, ymmword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F3ED48384C2B0801          , vinserti64x2(zmm1, k0, zmm2, xmmword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F3ED4838CB01              , vinserti64x2(zmm1, k0, zmm2, xmm3, Imm(1))),
        INSTRUCTION_TEST(62F3ED2838CB01              , vinserti64x2(ymm1, k0, ymm2, xmm3, Imm(1))),
        INSTRUCTION_TEST(62F3ED28384C2B0801          , vinserti64x2(ymm1, k0, ymm2, xmmword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F3B5403AE47B              , vinserti64x4(zmm4, k0, zmm25, ymm4, Imm(123))),
        INSTRUCTION_TEST(62F3B5403AA2001000007B      , vinserti64x4(zmm4, k0, zmm25, ymmword_ptr(rdx, 4096), Imm(123))),
        INSTRUCTION_TEST(62F3B5403A627F7B            , vinserti64x4(zmm4, k0, zmm25, ymmword_ptr(rdx, 4064), Imm(123))),
        INSTRUCTION_TEST(62F3B5403AA2E0EFFFFF7B      , vinserti64x4(zmm4, k0, zmm25, ymmword_ptr(rdx, -4128), Imm(123))),
        INSTRUCTION_TEST(62F3B5403A62807B            , vinserti64x4(zmm4, k0, zmm25, ymmword_ptr(rdx, -4096), Imm(123))),
        INSTRUCTION_TEST(62F3B5403A217B              , vinserti64x4(zmm4, k0, zmm25, ymmword_ptr(rcx), Imm(123))),
        INSTRUCTION_TEST(62B3B5403AA4F0230100007B    , vinserti64x4(zmm4, k0, zmm25, ymmword_ptr(rax, r14, 8, 291), Imm(123))),
        INSTRUCTION_TEST(62F3B5403AE4AB              , vinserti64x4(zmm4, k0, zmm25, ymm4, Imm(171))),
        INSTRUCTION_TEST(62E3D5483A92001000007B      , vinserti64x4(zmm18, k0, zmm5, ymmword_ptr(rdx, 4096), Imm(123))),
        INSTRUCTION_TEST(62E3D5483A117B              , vinserti64x4(zmm18, k0, zmm5, ymmword_ptr(rcx), Imm(123))),
        INSTRUCTION_TEST(62E3D5483A92E0EFFFFF7B      , vinserti64x4(zmm18, k0, zmm5, ymmword_ptr(rdx, -4128), Imm(123))),
        INSTRUCTION_TEST(62E3D5483A52807B            , vinserti64x4(zmm18, k0, zmm5, ymmword_ptr(rdx, -4096), Imm(123))),
        INSTRUCTION_TEST(62A3D5483A94F0341200007B    , vinserti64x4(zmm18, k0, zmm5, ymmword_ptr(rax, r14, 8, 4660), Imm(123))),
        INSTRUCTION_TEST(62A3D5483AD7AB              , vinserti64x4(zmm18, k0, zmm5, ymm23, Imm(171))),
        INSTRUCTION_TEST(62A3D5483AD77B              , vinserti64x4(zmm18, k0, zmm5, ymm23, Imm(123))),
        INSTRUCTION_TEST(62F3ED483A4C2B0401          , vinserti64x4(zmm1, k0, zmm2, ymmword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F3ED483ACB01              , vinserti64x4(zmm1, k0, zmm2, ymm3, Imm(1))),
        INSTRUCTION_TEST(62E3D5483A527F7B            , vinserti64x4(zmm18, k0, zmm5, ymmword_ptr(rdx, 4064), Imm(123))),
        INSTRUCTION_TEST(6263750021317B              , vinsertps(xmm30, xmm17, dword_ptr(rcx), Imm(123))),
        INSTRUCTION_TEST(626375002172807B            , vinsertps(xmm30, xmm17, dword_ptr(rdx, -512), Imm(123))),
        INSTRUCTION_TEST(6263750021B2FCFDFFFF7B      , vinsertps(xmm30, xmm17, dword_ptr(rdx, -516), Imm(123))),
        INSTRUCTION_TEST(6263750021F57B              , vinsertps(xmm30, xmm17, xmm5, Imm(123))),
        INSTRUCTION_TEST(6263750021F5AB              , vinsertps(xmm30, xmm17, xmm5, Imm(171))),
        INSTRUCTION_TEST(6263750021B2000200007B      , vinsertps(xmm30, xmm17, dword_ptr(rdx, 512), Imm(123))),
        INSTRUCTION_TEST(6223750021B4F0230100007B    , vinsertps(xmm30, xmm17, dword_ptr(rax, r14, 8, 291), Imm(123))),
        INSTRUCTION_TEST(6263750021727F7B            , vinsertps(xmm30, xmm17, dword_ptr(rdx, 508), Imm(123))),
        INSTRUCTION_TEST(62034D0821C4AB              , vinsertps(xmm24, xmm6, xmm28, Imm(171))),
        INSTRUCTION_TEST(62234D082184F0341200007B    , vinsertps(xmm24, xmm6, dword_ptr(rax, r14, 8, 4660), Imm(123))),
        INSTRUCTION_TEST(62634D082182000200007B      , vinsertps(xmm24, xmm6, dword_ptr(rdx, 512), Imm(123))),
        INSTRUCTION_TEST(62634D0821427F7B            , vinsertps(xmm24, xmm6, dword_ptr(rdx, 508), Imm(123))),
        INSTRUCTION_TEST(62634D082182FCFDFFFF7B      , vinsertps(xmm24, xmm6, dword_ptr(rdx, -516), Imm(123))),
        INSTRUCTION_TEST(62634D082142807B            , vinsertps(xmm24, xmm6, dword_ptr(rdx, -512), Imm(123))),
        INSTRUCTION_TEST(62634D0821017B              , vinsertps(xmm24, xmm6, dword_ptr(rcx), Imm(123))),
        INSTRUCTION_TEST(C4E36921CB01                , vinsertps(xmm1, xmm2, xmm3, Imm(1))),
        INSTRUCTION_TEST(C4E369218C2B8000000001      , vinsertps(xmm1, xmm2, dword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62034D0821C47B              , vinsertps(xmm24, xmm6, xmm28, Imm(123))),
        INSTRUCTION_TEST(C5FBF08C1A80000000          , vlddqu(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(C5FFF08C1A80000000          , vlddqu(ymm1, ymmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(C5F8AE941180000000          , vldmxcsr(dword_ptr(rcx, rdx, 1, 128))),
        INSTRUCTION_TEST(C5F9F7CA                    , vmaskmovdqu(xmm1, xmm2)),
        INSTRUCTION_TEST(C4E2652FA41180000000        , vmaskmovpd(ymmword_ptr(rcx, rdx, 1, 128), ymm3, ymm4)),
        INSTRUCTION_TEST(C4E26D2D8C2B80000000        , vmaskmovpd(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E2692D8C2B80000000        , vmaskmovpd(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E2612FA41180000000        , vmaskmovpd(xmmword_ptr(rcx, rdx, 1, 128), xmm3, xmm4)),
        INSTRUCTION_TEST(C4E2692C8C2B80000000        , vmaskmovps(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E2612EA41180000000        , vmaskmovps(xmmword_ptr(rcx, rdx, 1, 128), xmm3, xmm4)),
        INSTRUCTION_TEST(C4E26D2C8C2B80000000        , vmaskmovps(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E2652EA41180000000        , vmaskmovps(ymmword_ptr(rcx, rdx, 1, 128), ymm3, ymm4)),
        INSTRUCTION_TEST(62619D405F727F              , vmaxpd(zmm30, k0, zmm28, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62719D485F8A00200000        , vmaxpd(zmm9, k0, zmm12, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62719D485F4A7F              , vmaxpd(zmm9, k0, zmm12, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62719D485F8AC0DFFFFF        , vmaxpd(zmm9, k0, zmm12, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62719D485F4A80              , vmaxpd(zmm9, k0, zmm12, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62319D485F8CF034120000      , vmaxpd(zmm9, k0, zmm12, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62319D485FCD                , vmaxpd(zmm9, k0, zmm12, zmm21)),
        INSTRUCTION_TEST(62619D405FB200200000        , vmaxpd(zmm30, k0, zmm28, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62619D405FB2C0DFFFFF        , vmaxpd(zmm30, k0, zmm28, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62719D485F09                , vmaxpd(zmm9, k0, zmm12, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62619D405F31                , vmaxpd(zmm30, k0, zmm28, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(C5E95F8C2B80000000          , vmaxpd(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C5ED5FCB                    , vmaxpd(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(C5ED5F8C2B80000000          , vmaxpd(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62619D405F7280              , vmaxpd(zmm30, k0, zmm28, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(C5E95FCB                    , vmaxpd(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(62F1ED485F4C2B02            , vmaxpd(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62219D405FF4                , vmaxpd(zmm30, k0, zmm28, zmm20)),
        INSTRUCTION_TEST(62219D405FB4F023010000      , vmaxpd(zmm30, k0, zmm28, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62F1ED485FCB                , vmaxpd(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(62E13C405F4A7F              , vmaxps(zmm17, k0, zmm24, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62614C485F8A00200000        , vmaxps(zmm25, k0, zmm6, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62614C485F4A7F              , vmaxps(zmm25, k0, zmm6, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62614C485F8AC0DFFFFF        , vmaxps(zmm25, k0, zmm6, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62614C485F4A80              , vmaxps(zmm25, k0, zmm6, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62614C485F09                , vmaxps(zmm25, k0, zmm6, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62214C485F8CF023010000      , vmaxps(zmm25, k0, zmm6, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62214C485FCC                , vmaxps(zmm25, k0, zmm6, zmm20)),
        INSTRUCTION_TEST(62E13C405F8A00200000        , vmaxps(zmm17, k0, zmm24, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62E13C405F8AC0DFFFFF        , vmaxps(zmm17, k0, zmm24, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62E13C405F09                , vmaxps(zmm17, k0, zmm24, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(C5E85FCB                    , vmaxps(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(62A13C405F8CF034120000      , vmaxps(zmm17, k0, zmm24, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62813C405FC8                , vmaxps(zmm17, k0, zmm24, zmm24)),
        INSTRUCTION_TEST(62F16C485F4C2B02            , vmaxps(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F16C485FCB                , vmaxps(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(C5EC5F8C2B80000000          , vmaxps(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C5EC5FCB                    , vmaxps(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(C5E85F8C2B80000000          , vmaxps(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62E13C405F4A80              , vmaxps(zmm17, k0, zmm24, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62E1D7005F39                , vmaxsd(xmm23, k0, xmm21, qword_ptr(rcx))),
        INSTRUCTION_TEST(62E1D7005F7A80              , vmaxsd(xmm23, k0, xmm21, qword_ptr(rdx, -1024))),
        INSTRUCTION_TEST(62E1D7005FBAF8FBFFFF        , vmaxsd(xmm23, k0, xmm21, qword_ptr(rdx, -1032))),
        INSTRUCTION_TEST(62A1D7005FBCF034120000      , vmaxsd(xmm23, k0, xmm21, qword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62E1D7005FBA00040000        , vmaxsd(xmm23, k0, xmm21, qword_ptr(rdx, 1024))),
        INSTRUCTION_TEST(62A1D7005FF8                , vmaxsd(xmm23, k0, xmm21, xmm16)),
        INSTRUCTION_TEST(62E1D7005F7A7F              , vmaxsd(xmm23, k0, xmm21, qword_ptr(rdx, 1016))),
        INSTRUCTION_TEST(6281E7005FE1                , vmaxsd(xmm20, k0, xmm19, xmm25)),
        INSTRUCTION_TEST(62E1E7005F627F              , vmaxsd(xmm20, k0, xmm19, qword_ptr(rdx, 1016))),
        INSTRUCTION_TEST(62E1E7005FA2F8FBFFFF        , vmaxsd(xmm20, k0, xmm19, qword_ptr(rdx, -1032))),
        INSTRUCTION_TEST(62E1E7005F6280              , vmaxsd(xmm20, k0, xmm19, qword_ptr(rdx, -1024))),
        INSTRUCTION_TEST(62E1E7005F21                , vmaxsd(xmm20, k0, xmm19, qword_ptr(rcx))),
        INSTRUCTION_TEST(62A1E7005FA4F023010000      , vmaxsd(xmm20, k0, xmm19, qword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(C5EB5FCB                    , vmaxsd(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(C5EB5F8C2B80000000          , vmaxsd(xmm1, xmm2, qword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62E1E7005FA200040000        , vmaxsd(xmm20, k0, xmm19, qword_ptr(rdx, 1024))),
        INSTRUCTION_TEST(C55A5F01                    , vmaxss(xmm8, xmm4, dword_ptr(rcx))),
        INSTRUCTION_TEST(C55A5FC6                    , vmaxss(xmm8, xmm4, xmm6)),
        INSTRUCTION_TEST(C55A5F8200020000            , vmaxss(xmm8, xmm4, dword_ptr(rdx, 512))),
        INSTRUCTION_TEST(C55A5F82FC010000            , vmaxss(xmm8, xmm4, dword_ptr(rdx, 508))),
        INSTRUCTION_TEST(C55A5F82FCFDFFFF            , vmaxss(xmm8, xmm4, dword_ptr(rdx, -516))),
        INSTRUCTION_TEST(C55A5F8200FEFFFF            , vmaxss(xmm8, xmm4, dword_ptr(rdx, -512))),
        INSTRUCTION_TEST(C4215A5F84F023010000        , vmaxss(xmm8, xmm4, dword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62A176005FF0                , vmaxss(xmm22, k0, xmm17, xmm16)),
        INSTRUCTION_TEST(62E176005FB200020000        , vmaxss(xmm22, k0, xmm17, dword_ptr(rdx, 512))),
        INSTRUCTION_TEST(62E176005F727F              , vmaxss(xmm22, k0, xmm17, dword_ptr(rdx, 508))),
        INSTRUCTION_TEST(62E176005FB2FCFDFFFF        , vmaxss(xmm22, k0, xmm17, dword_ptr(rdx, -516))),
        INSTRUCTION_TEST(62E176005F7280              , vmaxss(xmm22, k0, xmm17, dword_ptr(rdx, -512))),
        INSTRUCTION_TEST(62E176005F31                , vmaxss(xmm22, k0, xmm17, dword_ptr(rcx))),
        INSTRUCTION_TEST(62A176005FB4F034120000      , vmaxss(xmm22, k0, xmm17, dword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(C5EA5FCB                    , vmaxss(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(C5EA5F8C2B80000000          , vmaxss(xmm1, xmm2, dword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(0F01C1                      , vmcall()),
        INSTRUCTION_TEST(660FC7B41180000000          , vmclear(qword_ptr(rcx, rdx, 1, 128))),
        INSTRUCTION_TEST(0F01D4                      , vmfunc()),
        INSTRUCTION_TEST(62B1CD485DF6                , vminpd(zmm6, k0, zmm6, zmm22)),
        INSTRUCTION_TEST(62B1CD485DB4F023010000      , vminpd(zmm6, k0, zmm6, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62F1CD485D31                , vminpd(zmm6, k0, zmm6, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(6271B5405D527F              , vminpd(zmm10, k0, zmm25, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62F1CD485DB2C0DFFFFF        , vminpd(zmm6, k0, zmm6, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62F1CD485D727F              , vminpd(zmm6, k0, zmm6, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62F1CD485DB200200000        , vminpd(zmm6, k0, zmm6, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62F1CD485D7280              , vminpd(zmm6, k0, zmm6, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(6271B5405D92C0DFFFFF        , vminpd(zmm10, k0, zmm25, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(6271B5405D9200200000        , vminpd(zmm10, k0, zmm25, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(6271B5405D11                , vminpd(zmm10, k0, zmm25, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(C5E95D8C2B80000000          , vminpd(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C5ED5DCB                    , vminpd(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(C5ED5D8C2B80000000          , vminpd(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F1ED485DCB                , vminpd(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(C5E95DCB                    , vminpd(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(6211B5405DD3                , vminpd(zmm10, k0, zmm25, zmm27)),
        INSTRUCTION_TEST(6271B5405D5280              , vminpd(zmm10, k0, zmm25, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(6231B5405D94F034120000      , vminpd(zmm10, k0, zmm25, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62F1ED485D4C2B02            , vminpd(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F10C485D5A7F              , vminps(zmm3, k0, zmm14, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62F164485D9A00200000        , vminps(zmm3, k0, zmm3, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62F164485D5A7F              , vminps(zmm3, k0, zmm3, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62F164485D9AC0DFFFFF        , vminps(zmm3, k0, zmm3, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62F164485D5A80              , vminps(zmm3, k0, zmm3, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62F164485D19                , vminps(zmm3, k0, zmm3, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62B164485D9CF023010000      , vminps(zmm3, k0, zmm3, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62F164485DDF                , vminps(zmm3, k0, zmm3, zmm7)),
        INSTRUCTION_TEST(62F10C485D9A00200000        , vminps(zmm3, k0, zmm14, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62F10C485D9AC0DFFFFF        , vminps(zmm3, k0, zmm14, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62F10C485D19                , vminps(zmm3, k0, zmm14, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62B10C485D9CF034120000      , vminps(zmm3, k0, zmm14, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62B10C485DD8                , vminps(zmm3, k0, zmm14, zmm16)),
        INSTRUCTION_TEST(62F16C485D4C2B02            , vminps(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F16C485DCB                , vminps(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(C5EC5D8C2B80000000          , vminps(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C5EC5DCB                    , vminps(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(C5E85D8C2B80000000          , vminps(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C5E85DCB                    , vminps(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(62F10C485D5A80              , vminps(zmm3, k0, zmm14, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62B1B7005DACF023010000      , vminsd(xmm5, k0, xmm25, qword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62F1B7005DAA00040000        , vminsd(xmm5, k0, xmm25, qword_ptr(rdx, 1024))),
        INSTRUCTION_TEST(62F1B7005D6A7F              , vminsd(xmm5, k0, xmm25, qword_ptr(rdx, 1016))),
        INSTRUCTION_TEST(62F1B7005DAAF8FBFFFF        , vminsd(xmm5, k0, xmm25, qword_ptr(rdx, -1032))),
        INSTRUCTION_TEST(62F1B7005D6A80              , vminsd(xmm5, k0, xmm25, qword_ptr(rdx, -1024))),
        INSTRUCTION_TEST(6291B7005DEA                , vminsd(xmm5, k0, xmm25, xmm26)),
        INSTRUCTION_TEST(6211F7085DD4                , vminsd(xmm10, k0, xmm1, xmm28)),
        INSTRUCTION_TEST(62F1B7005D29                , vminsd(xmm5, k0, xmm25, qword_ptr(rcx))),
        INSTRUCTION_TEST(C5735D92F8030000            , vminsd(xmm10, xmm1, qword_ptr(rdx, 1016))),
        INSTRUCTION_TEST(C5735D92F8FBFFFF            , vminsd(xmm10, xmm1, qword_ptr(rdx, -1032))),
        INSTRUCTION_TEST(C5735D9200FCFFFF            , vminsd(xmm10, xmm1, qword_ptr(rdx, -1024))),
        INSTRUCTION_TEST(C5735D11                    , vminsd(xmm10, xmm1, qword_ptr(rcx))),
        INSTRUCTION_TEST(C421735D94F034120000        , vminsd(xmm10, xmm1, qword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(C5EB5DCB                    , vminsd(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(C5EB5D8C2B80000000          , vminsd(xmm1, xmm2, qword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C5735D9200040000            , vminsd(xmm10, xmm1, qword_ptr(rdx, 1024))),
        INSTRUCTION_TEST(624156085DE0                , vminss(xmm28, k0, xmm5, xmm8)),
        INSTRUCTION_TEST(626156085DA200020000        , vminss(xmm28, k0, xmm5, dword_ptr(rdx, 512))),
        INSTRUCTION_TEST(626156085DA2FCFDFFFF        , vminss(xmm28, k0, xmm5, dword_ptr(rdx, -516))),
        INSTRUCTION_TEST(626156085D6280              , vminss(xmm28, k0, xmm5, dword_ptr(rdx, -512))),
        INSTRUCTION_TEST(626156085D21                , vminss(xmm28, k0, xmm5, dword_ptr(rcx))),
        INSTRUCTION_TEST(622156085DA4F034120000      , vminss(xmm28, k0, xmm5, dword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(623176005DD3                , vminss(xmm10, k0, xmm17, xmm19)),
        INSTRUCTION_TEST(626156085D627F              , vminss(xmm28, k0, xmm5, dword_ptr(rdx, 508))),
        INSTRUCTION_TEST(627176005D527F              , vminss(xmm10, k0, xmm17, dword_ptr(rdx, 508))),
        INSTRUCTION_TEST(627176005D92FCFDFFFF        , vminss(xmm10, k0, xmm17, dword_ptr(rdx, -516))),
        INSTRUCTION_TEST(627176005D5280              , vminss(xmm10, k0, xmm17, dword_ptr(rdx, -512))),
        INSTRUCTION_TEST(627176005D11                , vminss(xmm10, k0, xmm17, dword_ptr(rcx))),
        INSTRUCTION_TEST(623176005D94F023010000      , vminss(xmm10, k0, xmm17, dword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(C5EA5DCB                    , vminss(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(C5EA5D8C2B80000000          , vminss(xmm1, xmm2, dword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(627176005D9200020000        , vminss(xmm10, k0, xmm17, dword_ptr(rdx, 512))),
        INSTRUCTION_TEST(0F01C2                      , vmlaunch()),
        INSTRUCTION_TEST(0F01D9                      , vmmcall()),
        INSTRUCTION_TEST(62F1FD4828BA00200000        , vmovapd(zmm7, k0, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62A1FD482994F023010000      , vmovapd(zmmword_ptr(rax, r14, 8, 291), k0, zmm18)),
        INSTRUCTION_TEST(6221FD48299CF034120000      , vmovapd(zmmword_ptr(rax, r14, 8, 4660), k0, zmm27)),
        INSTRUCTION_TEST(62F1FD48287A7F              , vmovapd(zmm7, k0, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62F1FD48295C1102            , vmovapd(zmmword_ptr(rcx, rdx, 1, 128), k0, zmm3)),
        INSTRUCTION_TEST(6261FD482919                , vmovapd(zmmword_ptr(rcx), k0, zmm27)),
        INSTRUCTION_TEST(62E1FD482911                , vmovapd(zmmword_ptr(rcx), k0, zmm18)),
        INSTRUCTION_TEST(6261FD48295A7F              , vmovapd(zmmword_ptr(rdx, 8128), k0, zmm27)),
        INSTRUCTION_TEST(62E1FD48295280              , vmovapd(zmmword_ptr(rdx, -8192), k0, zmm18)),
        INSTRUCTION_TEST(6261FD48299AC0DFFFFF        , vmovapd(zmmword_ptr(rdx, -8256), k0, zmm27)),
        INSTRUCTION_TEST(62E1FD482992C0DFFFFF        , vmovapd(zmmword_ptr(rdx, -8256), k0, zmm18)),
        INSTRUCTION_TEST(62E1FD4829527F              , vmovapd(zmmword_ptr(rdx, 8128), k0, zmm18)),
        INSTRUCTION_TEST(6261FD48299A00200000        , vmovapd(zmmword_ptr(rdx, 8192), k0, zmm27)),
        INSTRUCTION_TEST(62E1FD48299200200000        , vmovapd(zmmword_ptr(rdx, 8192), k0, zmm18)),
        INSTRUCTION_TEST(62F1FD4828BAC0DFFFFF        , vmovapd(zmm7, k0, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(6261FD48295A80              , vmovapd(zmmword_ptr(rdx, -8192), k0, zmm27)),
        INSTRUCTION_TEST(62F1FD482839                , vmovapd(zmm7, k0, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62F1FD48287A80              , vmovapd(zmm7, k0, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62B1FD4828BCF023010000      , vmovapd(zmm7, k0, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(C5F9288C1A80000000          , vmovapd(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(C5F9299C1180000000          , vmovapd(xmmword_ptr(rcx, rdx, 1, 128), xmm3)),
        INSTRUCTION_TEST(C5FD28CA                    , vmovapd(ymm1, ymm2)),
        INSTRUCTION_TEST(C5FD288C1A80000000          , vmovapd(ymm1, ymmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(C5FD299C1180000000          , vmovapd(ymmword_ptr(rcx, rdx, 1, 128), ymm3)),
        INSTRUCTION_TEST(62F1FD4828CA                , vmovapd(zmm1, k0, zmm2)),
        INSTRUCTION_TEST(62F1FD48284C1A02            , vmovapd(zmm1, k0, zmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(C5F928CA                    , vmovapd(xmm1, xmm2)),
        INSTRUCTION_TEST(62A1FD4828BCF034120000      , vmovapd(zmm23, k0, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62E1FD482839                , vmovapd(zmm23, k0, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62E1FD48287A80              , vmovapd(zmm23, k0, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62E1FD4828BAC0DFFFFF        , vmovapd(zmm23, k0, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62E1FD48287A7F              , vmovapd(zmm23, k0, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62E1FD4828BA00200000        , vmovapd(zmm23, k0, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62D1FD4828FE                , vmovapd(zmm7, k0, zmm14)),
        INSTRUCTION_TEST(62C1FD4828F9                , vmovapd(zmm23, k0, zmm9)),
        INSTRUCTION_TEST(62F17C48286A7F              , vmovaps(zmm5, k0, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62F17C4828AA00200000        , vmovaps(zmm5, k0, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62317C48298CF023010000      , vmovaps(zmmword_ptr(rax, r14, 8, 291), k0, zmm9)),
        INSTRUCTION_TEST(62A17C48298CF034120000      , vmovaps(zmmword_ptr(rax, r14, 8, 4660), k0, zmm17)),
        INSTRUCTION_TEST(62F17C48295C1102            , vmovaps(zmmword_ptr(rcx, rdx, 1, 128), k0, zmm3)),
        INSTRUCTION_TEST(62E17C482909                , vmovaps(zmmword_ptr(rcx), k0, zmm17)),
        INSTRUCTION_TEST(62717C482909                , vmovaps(zmmword_ptr(rcx), k0, zmm9)),
        INSTRUCTION_TEST(62717C48298AC0DFFFFF        , vmovaps(zmmword_ptr(rdx, -8256), k0, zmm9)),
        INSTRUCTION_TEST(62717C48294A80              , vmovaps(zmmword_ptr(rdx, -8192), k0, zmm9)),
        INSTRUCTION_TEST(62E17C48298AC0DFFFFF        , vmovaps(zmmword_ptr(rdx, -8256), k0, zmm17)),
        INSTRUCTION_TEST(62E17C48294A7F              , vmovaps(zmmword_ptr(rdx, 8128), k0, zmm17)),
        INSTRUCTION_TEST(62717C48294A7F              , vmovaps(zmmword_ptr(rdx, 8128), k0, zmm9)),
        INSTRUCTION_TEST(62E17C48298A00200000        , vmovaps(zmmword_ptr(rdx, 8192), k0, zmm17)),
        INSTRUCTION_TEST(62717C48298A00200000        , vmovaps(zmmword_ptr(rdx, 8192), k0, zmm9)),
        INSTRUCTION_TEST(62F17C4828AAC0DFFFFF        , vmovaps(zmm5, k0, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62E17C48294A80              , vmovaps(zmmword_ptr(rdx, -8192), k0, zmm17)),
        INSTRUCTION_TEST(62F17C48286A80              , vmovaps(zmm5, k0, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62617C482892C0DFFFFF        , vmovaps(zmm26, k0, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(C5FC299C1180000000          , vmovaps(ymmword_ptr(rcx, rdx, 1, 128), ymm3)),
        INSTRUCTION_TEST(C5F828CA                    , vmovaps(xmm1, xmm2)),
        INSTRUCTION_TEST(C5F8288C1A80000000          , vmovaps(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(C5F8299C1180000000          , vmovaps(xmmword_ptr(rcx, rdx, 1, 128), xmm3)),
        INSTRUCTION_TEST(C5FC28CA                    , vmovaps(ymm1, ymm2)),
        INSTRUCTION_TEST(C5FC288C1A80000000          , vmovaps(ymm1, ymmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F17C482829                , vmovaps(zmm5, k0, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62F17C4828CA                , vmovaps(zmm1, k0, zmm2)),
        INSTRUCTION_TEST(62F17C48284C1A02            , vmovaps(zmm1, k0, zmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62B17C4828ACF023010000      , vmovaps(zmm5, k0, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62217C482894F034120000      , vmovaps(zmm26, k0, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62617C482811                , vmovaps(zmm26, k0, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62617C48285280              , vmovaps(zmm26, k0, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62617C4828527F              , vmovaps(zmm26, k0, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62617C48289200200000        , vmovaps(zmm26, k0, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62D17C4828E9                , vmovaps(zmm5, k0, zmm9)),
        INSTRUCTION_TEST(62417C4828D3                , vmovaps(zmm26, k0, zmm11)),
        INSTRUCTION_TEST(62617D086E11                , vmovd(xmm26, dword_ptr(rcx))),
        INSTRUCTION_TEST(C5796EB2FC010000            , vmovd(xmm14, dword_ptr(rdx, 508))),
        INSTRUCTION_TEST(C5796EB200020000            , vmovd(xmm14, dword_ptr(rdx, 512))),
        INSTRUCTION_TEST(C5796EF5                    , vmovd(xmm14, ebp)),
        INSTRUCTION_TEST(C441796EF5                  , vmovd(xmm14, r13d)),
        INSTRUCTION_TEST(62217D086E94F023010000      , vmovd(xmm26, dword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(C5796EB2FCFDFFFF            , vmovd(xmm14, dword_ptr(rdx, -516))),
        INSTRUCTION_TEST(62617D086E5280              , vmovd(xmm26, dword_ptr(rdx, -512))),
        INSTRUCTION_TEST(62617D086E527F              , vmovd(xmm26, dword_ptr(rdx, 508))),
        INSTRUCTION_TEST(62617D086E9200020000        , vmovd(xmm26, dword_ptr(rdx, 512))),
        INSTRUCTION_TEST(62617D086ED0                , vmovd(xmm26, eax)),
        INSTRUCTION_TEST(62617D086ED5                , vmovd(xmm26, ebp)),
        INSTRUCTION_TEST(62417D086ED5                , vmovd(xmm26, r13d)),
        INSTRUCTION_TEST(C5796EB200FEFFFF            , vmovd(xmm14, dword_ptr(rdx, -512))),
        INSTRUCTION_TEST(62617D086E92FCFDFFFF        , vmovd(xmm26, dword_ptr(rdx, -516))),
        INSTRUCTION_TEST(C5796E31                    , vmovd(xmm14, dword_ptr(rcx))),
        INSTRUCTION_TEST(C5796EF0                    , vmovd(xmm14, eax)),
        INSTRUCTION_TEST(C5F96ECA                    , vmovd(xmm1, edx)),
        INSTRUCTION_TEST(C4A1797EACF023010000        , vmovd(dword_ptr(rax, r14, 8, 291), xmm5)),
        INSTRUCTION_TEST(C421797EB4F034120000        , vmovd(dword_ptr(rax, r14, 8, 4660), xmm14)),
        INSTRUCTION_TEST(C5F97E9C1180000000          , vmovd(dword_ptr(rcx, rdx, 1, 128), xmm3)),
        INSTRUCTION_TEST(C5797E31                    , vmovd(dword_ptr(rcx), xmm14)),
        INSTRUCTION_TEST(C5F97E29                    , vmovd(dword_ptr(rcx), xmm5)),
        INSTRUCTION_TEST(C5797EB200FEFFFF            , vmovd(dword_ptr(rdx, -512), xmm14)),
        INSTRUCTION_TEST(C5F97EAA00FEFFFF            , vmovd(dword_ptr(rdx, -512), xmm5)),
        INSTRUCTION_TEST(C5797EB2FCFDFFFF            , vmovd(dword_ptr(rdx, -516), xmm14)),
        INSTRUCTION_TEST(C421796EB4F034120000        , vmovd(xmm14, dword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(C5797EB2FC010000            , vmovd(dword_ptr(rdx, 508), xmm14)),
        INSTRUCTION_TEST(C5F97EAAFC010000            , vmovd(dword_ptr(rdx, 508), xmm5)),
        INSTRUCTION_TEST(C5F97EAAFCFDFFFF            , vmovd(dword_ptr(rdx, -516), xmm5)),
        INSTRUCTION_TEST(C5797EB200020000            , vmovd(dword_ptr(rdx, 512), xmm14)),
        INSTRUCTION_TEST(C5F97EAA00020000            , vmovd(dword_ptr(rdx, 512), xmm5)),
        INSTRUCTION_TEST(C5F97ED1                    , vmovd(ecx, xmm2)),
        INSTRUCTION_TEST(C5F96E8C1A80000000          , vmovd(xmm1, dword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F1FF48124C1A02            , vmovddup(zmm1, k0, zmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(6291FF4812ED                , vmovddup(zmm5, k0, zmm29)),
        INSTRUCTION_TEST(62B1FF4812ACF023010000      , vmovddup(zmm5, k0, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62F1FF4812AAC0DFFFFF        , vmovddup(zmm5, k0, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62F1FF48126A80              , vmovddup(zmm5, k0, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62F1FF48128A00200000        , vmovddup(zmm1, k0, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62F1FF48126A7F              , vmovddup(zmm5, k0, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62F1FF4812AA00200000        , vmovddup(zmm5, k0, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62F1FF481229                , vmovddup(zmm5, k0, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62F1FF48124A7F              , vmovddup(zmm1, k0, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62F1FF481209                , vmovddup(zmm1, k0, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62F1FF48124A80              , vmovddup(zmm1, k0, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62B1FF48128CF034120000      , vmovddup(zmm1, k0, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62F1FF4812CA                , vmovddup(zmm1, k0, zmm2)),
        INSTRUCTION_TEST(62D1FF4812CA                , vmovddup(zmm1, k0, zmm10)),
        INSTRUCTION_TEST(C5FF128C1A80000000          , vmovddup(ymm1, ymmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(C5FB12CA                    , vmovddup(xmm1, xmm2)),
        INSTRUCTION_TEST(C5FB128C1A80000000          , vmovddup(xmm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F1FF48128AC0DFFFFF        , vmovddup(zmm1, k0, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(C5FF12CA                    , vmovddup(ymm1, ymm2)),
        INSTRUCTION_TEST(C5FD6FCA                    , vmovdqa(ymm1, ymm2)),
        INSTRUCTION_TEST(C5FD7F9C1180000000          , vmovdqa(ymmword_ptr(rcx, rdx, 1, 128), ymm3)),
        INSTRUCTION_TEST(C5FD6F8C1A80000000          , vmovdqa(ymm1, ymmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(C5F97F9C1180000000          , vmovdqa(xmmword_ptr(rcx, rdx, 1, 128), xmm3)),
        INSTRUCTION_TEST(C5F96FCA                    , vmovdqa(xmm1, xmm2)),
        INSTRUCTION_TEST(C5F96F8C1A80000000          , vmovdqa(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F17D487F9A00200000        , vmovdqa32(zmmword_ptr(rdx, 8192), k0, zmm3)),
        INSTRUCTION_TEST(62E17D487F9200200000        , vmovdqa32(zmmword_ptr(rdx, 8192), k0, zmm18)),
        INSTRUCTION_TEST(62F17D487F5A7F              , vmovdqa32(zmmword_ptr(rdx, 8128), k0, zmm3)),
        INSTRUCTION_TEST(62E17D487F527F              , vmovdqa32(zmmword_ptr(rdx, 8128), k0, zmm18)),
        INSTRUCTION_TEST(62F17D487F9AC0DFFFFF        , vmovdqa32(zmmword_ptr(rdx, -8256), k0, zmm3)),
        INSTRUCTION_TEST(62E17D487F92C0DFFFFF        , vmovdqa32(zmmword_ptr(rdx, -8256), k0, zmm18)),
        INSTRUCTION_TEST(62F17D487F5A80              , vmovdqa32(zmmword_ptr(rdx, -8192), k0, zmm3)),
        INSTRUCTION_TEST(62E17D487F5280              , vmovdqa32(zmmword_ptr(rdx, -8192), k0, zmm18)),
        INSTRUCTION_TEST(62F17D487F19                , vmovdqa32(zmmword_ptr(rcx), k0, zmm3)),
        INSTRUCTION_TEST(62E17D487F11                , vmovdqa32(zmmword_ptr(rcx), k0, zmm18)),
        INSTRUCTION_TEST(62F17D487F5C1102            , vmovdqa32(zmmword_ptr(rcx, rdx, 1, 128), k0, zmm3)),
        INSTRUCTION_TEST(62B17D487F9CF034120000      , vmovdqa32(zmmword_ptr(rax, r14, 8, 4660), k0, zmm3)),
        INSTRUCTION_TEST(62E17D486FB200200000        , vmovdqa32(zmm22, k0, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62E17D486F727F              , vmovdqa32(zmm22, k0, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62A17D487F94F023010000      , vmovdqa32(zmmword_ptr(rax, r14, 8, 291), k0, zmm18)),
        INSTRUCTION_TEST(62E17D486F7280              , vmovdqa32(zmm22, k0, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62F17D086FCA                , vmovdqa32(xmm1, k0, xmm2)),
        INSTRUCTION_TEST(62F17D086F4C1A08            , vmovdqa32(xmm1, k0, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F17D087F5C1108            , vmovdqa32(xmmword_ptr(rcx, rdx, 1, 128), k0, xmm3)),
        INSTRUCTION_TEST(62E17D486FB2C0DFFFFF        , vmovdqa32(zmm22, k0, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62F17D286F4C1A04            , vmovdqa32(ymm1, k0, ymmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F17D287F5C1104            , vmovdqa32(ymmword_ptr(rcx, rdx, 1, 128), k0, ymm3)),
        INSTRUCTION_TEST(62F17D286FCA                , vmovdqa32(ymm1, k0, ymm2)),
        INSTRUCTION_TEST(62F17D486F4C1A02            , vmovdqa32(zmm1, k0, zmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62A17D486FF2                , vmovdqa32(zmm22, k0, zmm18)),
        INSTRUCTION_TEST(62817D486FF5                , vmovdqa32(zmm22, k0, zmm29)),
        INSTRUCTION_TEST(62A17D486FB4F023010000      , vmovdqa32(zmm22, k0, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62A17D486FB4F034120000      , vmovdqa32(zmm22, k0, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62E17D486F31                , vmovdqa32(zmm22, k0, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62F17D486FCA                , vmovdqa32(zmm1, k0, zmm2)),
        INSTRUCTION_TEST(62E1FD487F11                , vmovdqa64(zmmword_ptr(rcx), k0, zmm18)),
        INSTRUCTION_TEST(6271FD486F8AC0DFFFFF        , vmovdqa64(zmm9, k0, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(6271FD486F4A7F              , vmovdqa64(zmm9, k0, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(6271FD486F8A00200000        , vmovdqa64(zmm9, k0, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62A1FD487F94F034120000      , vmovdqa64(zmmword_ptr(rax, r14, 8, 4660), k0, zmm18)),
        INSTRUCTION_TEST(62F1FD487F5C1102            , vmovdqa64(zmmword_ptr(rcx, rdx, 1, 128), k0, zmm3)),
        INSTRUCTION_TEST(62E1FD487F19                , vmovdqa64(zmmword_ptr(rcx), k0, zmm19)),
        INSTRUCTION_TEST(6271FD486F4A80              , vmovdqa64(zmm9, k0, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62E1FD487F5A80              , vmovdqa64(zmmword_ptr(rdx, -8192), k0, zmm19)),
        INSTRUCTION_TEST(62E1FD487F92C0DFFFFF        , vmovdqa64(zmmword_ptr(rdx, -8256), k0, zmm18)),
        INSTRUCTION_TEST(62E1FD487F9AC0DFFFFF        , vmovdqa64(zmmword_ptr(rdx, -8256), k0, zmm19)),
        INSTRUCTION_TEST(62E1FD487F527F              , vmovdqa64(zmmword_ptr(rdx, 8128), k0, zmm18)),
        INSTRUCTION_TEST(62E1FD487F5A7F              , vmovdqa64(zmmword_ptr(rdx, 8128), k0, zmm19)),
        INSTRUCTION_TEST(62E1FD487F9200200000        , vmovdqa64(zmmword_ptr(rdx, 8192), k0, zmm18)),
        INSTRUCTION_TEST(62E1FD487F9A00200000        , vmovdqa64(zmmword_ptr(rdx, 8192), k0, zmm19)),
        INSTRUCTION_TEST(62E1FD487F5280              , vmovdqa64(zmmword_ptr(rdx, -8192), k0, zmm18)),
        INSTRUCTION_TEST(6271FD486F09                , vmovdqa64(zmm9, k0, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62A1FD487F9CF023010000      , vmovdqa64(zmmword_ptr(rax, r14, 8, 291), k0, zmm19)),
        INSTRUCTION_TEST(6231FD486F8CF034120000      , vmovdqa64(zmm9, k0, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62F1FD086FCA                , vmovdqa64(xmm1, k0, xmm2)),
        INSTRUCTION_TEST(62F1FD086F4C1A08            , vmovdqa64(xmm1, k0, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F1FD087F5C1108            , vmovdqa64(xmmword_ptr(rcx, rdx, 1, 128), k0, xmm3)),
        INSTRUCTION_TEST(62F1FD286FCA                , vmovdqa64(ymm1, k0, ymm2)),
        INSTRUCTION_TEST(62F1FD286F4C1A04            , vmovdqa64(ymm1, k0, ymmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F1FD287F5C1104            , vmovdqa64(ymmword_ptr(rcx, rdx, 1, 128), k0, ymm3)),
        INSTRUCTION_TEST(62F1FD486FCA                , vmovdqa64(zmm1, k0, zmm2)),
        INSTRUCTION_TEST(62F1FD486F4C1A02            , vmovdqa64(zmm1, k0, zmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(6271FD486FCF                , vmovdqa64(zmm9, k0, zmm7)),
        INSTRUCTION_TEST(62A1FD486FB4F023010000      , vmovdqa64(zmm22, k0, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62E1FD486F31                , vmovdqa64(zmm22, k0, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62E1FD486F7280              , vmovdqa64(zmm22, k0, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62C1FD486FF4                , vmovdqa64(zmm22, k0, zmm12)),
        INSTRUCTION_TEST(62E1FD486FB2C0DFFFFF        , vmovdqa64(zmm22, k0, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62E1FD486F727F              , vmovdqa64(zmm22, k0, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62E1FD486FB200200000        , vmovdqa64(zmm22, k0, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(C5FE7F9C1180000000          , vmovdqu(ymmword_ptr(rcx, rdx, 1, 128), ymm3)),
        INSTRUCTION_TEST(C5FE6F8C1A80000000          , vmovdqu(ymm1, ymmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(C5FE6FCA                    , vmovdqu(ymm1, ymm2)),
        INSTRUCTION_TEST(C5FA6F8C1A80000000          , vmovdqu(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(C5FA6FCA                    , vmovdqu(xmm1, xmm2)),
        INSTRUCTION_TEST(C5FA7F9C1180000000          , vmovdqu(xmmword_ptr(rcx, rdx, 1, 128), xmm3)),
        INSTRUCTION_TEST(62F1FF486FCA                , vmovdqu16(zmm1, k0, zmm2)),
        INSTRUCTION_TEST(62F1FF487F5C1102            , vmovdqu16(zmmword_ptr(rcx, rdx, 1, 128), k0, zmm3)),
        INSTRUCTION_TEST(62F1FF486F4C1A02            , vmovdqu16(zmm1, k0, zmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F1FF286F4C1A04            , vmovdqu16(ymm1, k0, ymmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F1FF287F5C1104            , vmovdqu16(ymmword_ptr(rcx, rdx, 1, 128), k0, ymm3)),
        INSTRUCTION_TEST(62F1FF087F5C1108            , vmovdqu16(xmmword_ptr(rcx, rdx, 1, 128), k0, xmm3)),
        INSTRUCTION_TEST(62F1FF086F4C1A08            , vmovdqu16(xmm1, k0, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F1FF086FCA                , vmovdqu16(xmm1, k0, xmm2)),
        INSTRUCTION_TEST(62F1FF286FCA                , vmovdqu16(ymm1, k0, ymm2)),
        INSTRUCTION_TEST(62F17E486FAA00200000        , vmovdqu32(zmm5, k0, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62A17E487FB4F023010000      , vmovdqu32(zmmword_ptr(rax, r14, 8, 291), k0, zmm22)),
        INSTRUCTION_TEST(62B17E487F94F034120000      , vmovdqu32(zmmword_ptr(rax, r14, 8, 4660), k0, zmm2)),
        INSTRUCTION_TEST(62F17E487F5C1102            , vmovdqu32(zmmword_ptr(rcx, rdx, 1, 128), k0, zmm3)),
        INSTRUCTION_TEST(62F17E487F11                , vmovdqu32(zmmword_ptr(rcx), k0, zmm2)),
        INSTRUCTION_TEST(62E17E487F31                , vmovdqu32(zmmword_ptr(rcx), k0, zmm22)),
        INSTRUCTION_TEST(62F17E487F5280              , vmovdqu32(zmmword_ptr(rdx, -8192), k0, zmm2)),
        INSTRUCTION_TEST(62F17E487F92C0DFFFFF        , vmovdqu32(zmmword_ptr(rdx, -8256), k0, zmm2)),
        INSTRUCTION_TEST(62E17E487FB2C0DFFFFF        , vmovdqu32(zmmword_ptr(rdx, -8256), k0, zmm22)),
        INSTRUCTION_TEST(62F17E487F527F              , vmovdqu32(zmmword_ptr(rdx, 8128), k0, zmm2)),
        INSTRUCTION_TEST(62E17E487F727F              , vmovdqu32(zmmword_ptr(rdx, 8128), k0, zmm22)),
        INSTRUCTION_TEST(62F17E487F9200200000        , vmovdqu32(zmmword_ptr(rdx, 8192), k0, zmm2)),
        INSTRUCTION_TEST(62E17E487FB200200000        , vmovdqu32(zmmword_ptr(rdx, 8192), k0, zmm22)),
        INSTRUCTION_TEST(62E17E487F7280              , vmovdqu32(zmmword_ptr(rdx, -8192), k0, zmm22)),
        INSTRUCTION_TEST(62F17E486F6A7F              , vmovdqu32(zmm5, k0, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62B17E486FACF034120000      , vmovdqu32(zmm5, k0, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62F17E486F6A80              , vmovdqu32(zmm5, k0, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62F17E486FAAC0DFFFFF        , vmovdqu32(zmm5, k0, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62F17E086FCA                , vmovdqu32(xmm1, k0, xmm2)),
        INSTRUCTION_TEST(62F17E086F4C1A08            , vmovdqu32(xmm1, k0, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F17E087F5C1108            , vmovdqu32(xmmword_ptr(rcx, rdx, 1, 128), k0, xmm3)),
        INSTRUCTION_TEST(62F17E286F4C1A04            , vmovdqu32(ymm1, k0, ymmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F17E287F5C1104            , vmovdqu32(ymmword_ptr(rcx, rdx, 1, 128), k0, ymm3)),
        INSTRUCTION_TEST(62F17E286FCA                , vmovdqu32(ymm1, k0, ymm2)),
        INSTRUCTION_TEST(62F17E486F4C1A02            , vmovdqu32(zmm1, k0, zmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62B17E486FEE                , vmovdqu32(zmm5, k0, zmm22)),
        INSTRUCTION_TEST(62917E486FE8                , vmovdqu32(zmm5, k0, zmm24)),
        INSTRUCTION_TEST(62B17E486FACF023010000      , vmovdqu32(zmm5, k0, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62F17E486F29                , vmovdqu32(zmm5, k0, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62F17E486FCA                , vmovdqu32(zmm1, k0, zmm2)),
        INSTRUCTION_TEST(6261FE487F01                , vmovdqu64(zmmword_ptr(rcx), k0, zmm24)),
        INSTRUCTION_TEST(62F1FE486F727F              , vmovdqu64(zmm6, k0, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62F1FE486FB200200000        , vmovdqu64(zmm6, k0, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(6221FE487F84F023010000      , vmovdqu64(zmmword_ptr(rax, r14, 8, 291), k0, zmm24)),
        INSTRUCTION_TEST(62F1FE487FBA00200000        , vmovdqu64(zmmword_ptr(rdx, 8192), k0, zmm7)),
        INSTRUCTION_TEST(62F1FE487F5C1102            , vmovdqu64(zmmword_ptr(rcx, rdx, 1, 128), k0, zmm3)),
        INSTRUCTION_TEST(62F1FE487F39                , vmovdqu64(zmmword_ptr(rcx), k0, zmm7)),
        INSTRUCTION_TEST(6261FE487F8200200000        , vmovdqu64(zmmword_ptr(rdx, 8192), k0, zmm24)),
        INSTRUCTION_TEST(62F1FE487F7A80              , vmovdqu64(zmmword_ptr(rdx, -8192), k0, zmm7)),
        INSTRUCTION_TEST(6261FE487F82C0DFFFFF        , vmovdqu64(zmmword_ptr(rdx, -8256), k0, zmm24)),
        INSTRUCTION_TEST(62F1FE487FBAC0DFFFFF        , vmovdqu64(zmmword_ptr(rdx, -8256), k0, zmm7)),
        INSTRUCTION_TEST(6261FE487F427F              , vmovdqu64(zmmword_ptr(rdx, 8128), k0, zmm24)),
        INSTRUCTION_TEST(62F1FE487F7A7F              , vmovdqu64(zmmword_ptr(rdx, 8128), k0, zmm7)),
        INSTRUCTION_TEST(62F1FE486FB2C0DFFFFF        , vmovdqu64(zmm6, k0, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(6261FE487F4280              , vmovdqu64(zmmword_ptr(rdx, -8192), k0, zmm24)),
        INSTRUCTION_TEST(62F1FE486F31                , vmovdqu64(zmm6, k0, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62B1FE487FBCF034120000      , vmovdqu64(zmmword_ptr(rax, r14, 8, 4660), k0, zmm7)),
        INSTRUCTION_TEST(62D1FE486FF7                , vmovdqu64(zmm6, k0, zmm15)),
        INSTRUCTION_TEST(62B1FE486FB4F023010000      , vmovdqu64(zmm6, k0, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62F1FE086FCA                , vmovdqu64(xmm1, k0, xmm2)),
        INSTRUCTION_TEST(62F1FE086F4C1A08            , vmovdqu64(xmm1, k0, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F1FE087F5C1108            , vmovdqu64(xmmword_ptr(rcx, rdx, 1, 128), k0, xmm3)),
        INSTRUCTION_TEST(62F1FE286FCA                , vmovdqu64(ymm1, k0, ymm2)),
        INSTRUCTION_TEST(62F1FE286F4C1A04            , vmovdqu64(ymm1, k0, ymmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F1FE287F5C1104            , vmovdqu64(ymmword_ptr(rcx, rdx, 1, 128), k0, ymm3)),
        INSTRUCTION_TEST(62F1FE486FCA                , vmovdqu64(zmm1, k0, zmm2)),
        INSTRUCTION_TEST(62F1FE486F7280              , vmovdqu64(zmm6, k0, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(6221FE486FA4F034120000      , vmovdqu64(zmm28, k0, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(6261FE486F21                , vmovdqu64(zmm28, k0, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(6261FE486F6280              , vmovdqu64(zmm28, k0, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(6261FE486F627F              , vmovdqu64(zmm28, k0, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(6261FE486FE5                , vmovdqu64(zmm28, k0, zmm5)),
        INSTRUCTION_TEST(6261FE486FA2C0DFFFFF        , vmovdqu64(zmm28, k0, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(6261FE486FA200200000        , vmovdqu64(zmm28, k0, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62F1FE486F4C1A02            , vmovdqu64(zmm1, k0, zmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F17F487F5C1102            , vmovdqu8(zmmword_ptr(rcx, rdx, 1, 128), k0, zmm3)),
        INSTRUCTION_TEST(62F17F486FCA                , vmovdqu8(zmm1, k0, zmm2)),
        INSTRUCTION_TEST(62F17F287F5C1104            , vmovdqu8(ymmword_ptr(rcx, rdx, 1, 128), k0, ymm3)),
        INSTRUCTION_TEST(62F17F286F4C1A04            , vmovdqu8(ymm1, k0, ymmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F17F486F4C1A02            , vmovdqu8(zmm1, k0, zmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F17F087F5C1108            , vmovdqu8(xmmword_ptr(rcx, rdx, 1, 128), k0, xmm3)),
        INSTRUCTION_TEST(62F17F086F4C1A08            , vmovdqu8(xmm1, k0, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F17F086FCA                , vmovdqu8(xmm1, k0, xmm2)),
        INSTRUCTION_TEST(62F17F286FCA                , vmovdqu8(ymm1, k0, ymm2)),
        INSTRUCTION_TEST(C5E812CB                    , vmovhlps(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(62A1740812D7                , vmovhlps(xmm18, xmm1, xmm23)),
        INSTRUCTION_TEST(62817C0012F1                , vmovhlps(xmm22, xmm16, xmm25)),
        INSTRUCTION_TEST(62E19D001619                , vmovhpd(xmm19, xmm28, qword_ptr(rcx))),
        INSTRUCTION_TEST(6261F508168A00040000        , vmovhpd(xmm25, xmm1, qword_ptr(rdx, 1024))),
        INSTRUCTION_TEST(62E19D00169AF8FBFFFF        , vmovhpd(xmm19, xmm28, qword_ptr(rdx, -1032))),
        INSTRUCTION_TEST(62E19D00165A7F              , vmovhpd(xmm19, xmm28, qword_ptr(rdx, 1016))),
        INSTRUCTION_TEST(62E19D00169A00040000        , vmovhpd(xmm19, xmm28, qword_ptr(rdx, 1024))),
        INSTRUCTION_TEST(6261F508164A80              , vmovhpd(xmm25, xmm1, qword_ptr(rdx, -1024))),
        INSTRUCTION_TEST(6261F5081609                , vmovhpd(xmm25, xmm1, qword_ptr(rcx))),
        INSTRUCTION_TEST(6261F508168AF8FBFFFF        , vmovhpd(xmm25, xmm1, qword_ptr(rdx, -1032))),
        INSTRUCTION_TEST(6261F508164A7F              , vmovhpd(xmm25, xmm1, qword_ptr(rdx, 1016))),
        INSTRUCTION_TEST(62A19D00169CF023010000      , vmovhpd(xmm19, xmm28, qword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(6221F508168CF034120000      , vmovhpd(xmm25, xmm1, qword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(C5E9168C2B80000000          , vmovhpd(xmm1, xmm2, qword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62E19D00165A80              , vmovhpd(xmm19, xmm28, qword_ptr(rdx, -1024))),
        INSTRUCTION_TEST(C57917BA00040000            , vmovhpd(qword_ptr(rdx, 1024), xmm15)),
        INSTRUCTION_TEST(C4217917BCF023010000        , vmovhpd(qword_ptr(rax, r14, 8, 291), xmm15)),
        INSTRUCTION_TEST(C42179178CF034120000        , vmovhpd(qword_ptr(rax, r14, 8, 4660), xmm9)),
        INSTRUCTION_TEST(C5F9179C1180000000          , vmovhpd(qword_ptr(rcx, rdx, 1, 128), xmm3)),
        INSTRUCTION_TEST(C579178A00040000            , vmovhpd(qword_ptr(rdx, 1024), xmm9)),
        INSTRUCTION_TEST(C5791709                    , vmovhpd(qword_ptr(rcx), xmm9)),
        INSTRUCTION_TEST(C5791739                    , vmovhpd(qword_ptr(rcx), xmm15)),
        INSTRUCTION_TEST(C579178A00FCFFFF            , vmovhpd(qword_ptr(rdx, -1024), xmm9)),
        INSTRUCTION_TEST(C57917BAF8FBFFFF            , vmovhpd(qword_ptr(rdx, -1032), xmm15)),
        INSTRUCTION_TEST(C579178AF8FBFFFF            , vmovhpd(qword_ptr(rdx, -1032), xmm9)),
        INSTRUCTION_TEST(C57917BAF8030000            , vmovhpd(qword_ptr(rdx, 1016), xmm15)),
        INSTRUCTION_TEST(C579178AF8030000            , vmovhpd(qword_ptr(rdx, 1016), xmm9)),
        INSTRUCTION_TEST(C57917BA00FCFFFF            , vmovhpd(qword_ptr(rdx, -1024), xmm15)),
        INSTRUCTION_TEST(62E174001621                , vmovhps(xmm20, xmm17, qword_ptr(rcx))),
        INSTRUCTION_TEST(62E17400166280              , vmovhps(xmm20, xmm17, qword_ptr(rdx, -1024))),
        INSTRUCTION_TEST(62E1740016A2F8FBFFFF        , vmovhps(xmm20, xmm17, qword_ptr(rdx, -1032))),
        INSTRUCTION_TEST(62E1740016627F              , vmovhps(xmm20, xmm17, qword_ptr(rdx, 1016))),
        INSTRUCTION_TEST(6261740016627F              , vmovhps(xmm28, xmm17, qword_ptr(rdx, 1016))),
        INSTRUCTION_TEST(6221740016A4F034120000      , vmovhps(xmm28, xmm17, qword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(626174001621                , vmovhps(xmm28, xmm17, qword_ptr(rcx))),
        INSTRUCTION_TEST(62617400166280              , vmovhps(xmm28, xmm17, qword_ptr(rdx, -1024))),
        INSTRUCTION_TEST(6261740016A2F8FBFFFF        , vmovhps(xmm28, xmm17, qword_ptr(rdx, -1032))),
        INSTRUCTION_TEST(62A1740016A4F023010000      , vmovhps(xmm20, xmm17, qword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62E1740016A200040000        , vmovhps(xmm20, xmm17, qword_ptr(rdx, 1024))),
        INSTRUCTION_TEST(C5E8168C2B80000000          , vmovhps(xmm1, xmm2, qword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62E17C08175280              , vmovhps(qword_ptr(rdx, -1024), xmm18)),
        INSTRUCTION_TEST(C57817AA00040000            , vmovhps(qword_ptr(rdx, 1024), xmm13)),
        INSTRUCTION_TEST(62E17C0817527F              , vmovhps(qword_ptr(rdx, 1016), xmm18)),
        INSTRUCTION_TEST(C57817AAF8030000            , vmovhps(qword_ptr(rdx, 1016), xmm13)),
        INSTRUCTION_TEST(62E17C081792F8FBFFFF        , vmovhps(qword_ptr(rdx, -1032), xmm18)),
        INSTRUCTION_TEST(C57817AAF8FBFFFF            , vmovhps(qword_ptr(rdx, -1032), xmm13)),
        INSTRUCTION_TEST(C57817AA00FCFFFF            , vmovhps(qword_ptr(rdx, -1024), xmm13)),
        INSTRUCTION_TEST(62E17C081711                , vmovhps(qword_ptr(rcx), xmm18)),
        INSTRUCTION_TEST(C5781729                    , vmovhps(qword_ptr(rcx), xmm13)),
        INSTRUCTION_TEST(C5F8179C1180000000          , vmovhps(qword_ptr(rcx, rdx, 1, 128), xmm3)),
        INSTRUCTION_TEST(C4217817ACF034120000        , vmovhps(qword_ptr(rax, r14, 8, 4660), xmm13)),
        INSTRUCTION_TEST(62A17C081794F023010000      , vmovhps(qword_ptr(rax, r14, 8, 291), xmm18)),
        INSTRUCTION_TEST(62E17C08179200040000        , vmovhps(qword_ptr(rdx, 1024), xmm18)),
        INSTRUCTION_TEST(6261740016A200040000        , vmovhps(xmm28, xmm17, qword_ptr(rdx, 1024))),
        INSTRUCTION_TEST(C59816F6                    , vmovlhps(xmm6, xmm12, xmm6)),
        INSTRUCTION_TEST(6211140816EC                , vmovlhps(xmm13, xmm13, xmm28)),
        INSTRUCTION_TEST(C5E816CB                    , vmovlhps(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(6261CD0812AA00040000        , vmovlpd(xmm29, xmm6, qword_ptr(rdx, 1024))),
        INSTRUCTION_TEST(6261CD08126A7F              , vmovlpd(xmm29, xmm6, qword_ptr(rdx, 1016))),
        INSTRUCTION_TEST(6261CD0812AAF8FBFFFF        , vmovlpd(xmm29, xmm6, qword_ptr(rdx, -1032))),
        INSTRUCTION_TEST(6261CD08126A80              , vmovlpd(xmm29, xmm6, qword_ptr(rdx, -1024))),
        INSTRUCTION_TEST(6261CD081229                , vmovlpd(xmm29, xmm6, qword_ptr(rcx))),
        INSTRUCTION_TEST(6221CD0812ACF023010000      , vmovlpd(xmm29, xmm6, qword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62E1DD00128A00040000        , vmovlpd(xmm17, xmm20, qword_ptr(rdx, 1024))),
        INSTRUCTION_TEST(62E1DD00128AF8FBFFFF        , vmovlpd(xmm17, xmm20, qword_ptr(rdx, -1032))),
        INSTRUCTION_TEST(62E1DD00124A80              , vmovlpd(xmm17, xmm20, qword_ptr(rdx, -1024))),
        INSTRUCTION_TEST(62E1DD001209                , vmovlpd(xmm17, xmm20, qword_ptr(rcx))),
        INSTRUCTION_TEST(62A1DD00128CF034120000      , vmovlpd(xmm17, xmm20, qword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(C5E9128C2B80000000          , vmovlpd(xmm1, xmm2, qword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62E1DD00124A7F              , vmovlpd(xmm17, xmm20, qword_ptr(rdx, 1016))),
        INSTRUCTION_TEST(C57913BA00040000            , vmovlpd(qword_ptr(rdx, 1024), xmm15)),
        INSTRUCTION_TEST(C5F913BA00040000            , vmovlpd(qword_ptr(rdx, 1024), xmm7)),
        INSTRUCTION_TEST(C4A17913BCF034120000        , vmovlpd(qword_ptr(rax, r14, 8, 4660), xmm7)),
        INSTRUCTION_TEST(C5F9139C1180000000          , vmovlpd(qword_ptr(rcx, rdx, 1, 128), xmm3)),
        INSTRUCTION_TEST(C5791339                    , vmovlpd(qword_ptr(rcx), xmm15)),
        INSTRUCTION_TEST(C5F91339                    , vmovlpd(qword_ptr(rcx), xmm7)),
        INSTRUCTION_TEST(C57913BA00FCFFFF            , vmovlpd(qword_ptr(rdx, -1024), xmm15)),
        INSTRUCTION_TEST(C4217913BCF023010000        , vmovlpd(qword_ptr(rax, r14, 8, 291), xmm15)),
        INSTRUCTION_TEST(C57913BAF8FBFFFF            , vmovlpd(qword_ptr(rdx, -1032), xmm15)),
        INSTRUCTION_TEST(C5F913BAF8FBFFFF            , vmovlpd(qword_ptr(rdx, -1032), xmm7)),
        INSTRUCTION_TEST(C57913BAF8030000            , vmovlpd(qword_ptr(rdx, 1016), xmm15)),
        INSTRUCTION_TEST(C5F913BAF8030000            , vmovlpd(qword_ptr(rdx, 1016), xmm7)),
        INSTRUCTION_TEST(C5F913BA00FCFFFF            , vmovlpd(qword_ptr(rdx, -1024), xmm7)),
        INSTRUCTION_TEST(C42120129CF034120000        , vmovlps(xmm11, xmm11, qword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(C5201219                    , vmovlps(xmm11, xmm11, qword_ptr(rcx))),
        INSTRUCTION_TEST(C520129A00FCFFFF            , vmovlps(xmm11, xmm11, qword_ptr(rdx, -1024))),
        INSTRUCTION_TEST(C520129AF8FBFFFF            , vmovlps(xmm11, xmm11, qword_ptr(rdx, -1032))),
        INSTRUCTION_TEST(C520129AF8030000            , vmovlps(xmm11, xmm11, qword_ptr(rdx, 1016))),
        INSTRUCTION_TEST(C520129A00040000            , vmovlps(xmm11, xmm11, qword_ptr(rdx, 1024))),
        INSTRUCTION_TEST(62F15C00127A80              , vmovlps(xmm7, xmm20, qword_ptr(rdx, -1024))),
        INSTRUCTION_TEST(62F15C001239                , vmovlps(xmm7, xmm20, qword_ptr(rcx))),
        INSTRUCTION_TEST(62F15C0012BAF8FBFFFF        , vmovlps(xmm7, xmm20, qword_ptr(rdx, -1032))),
        INSTRUCTION_TEST(62F15C00127A7F              , vmovlps(xmm7, xmm20, qword_ptr(rdx, 1016))),
        INSTRUCTION_TEST(62F15C0012BA00040000        , vmovlps(xmm7, xmm20, qword_ptr(rdx, 1024))),
        INSTRUCTION_TEST(62B15C0012BCF023010000      , vmovlps(xmm7, xmm20, qword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62617C08139A00040000        , vmovlps(qword_ptr(rdx, 1024), xmm27)),
        INSTRUCTION_TEST(62E17C08136A7F              , vmovlps(qword_ptr(rdx, 1016), xmm21)),
        INSTRUCTION_TEST(62E17C0813AA00040000        , vmovlps(qword_ptr(rdx, 1024), xmm21)),
        INSTRUCTION_TEST(C5E8128C2B80000000          , vmovlps(xmm1, xmm2, qword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62A17C0813ACF034120000      , vmovlps(qword_ptr(rax, r14, 8, 4660), xmm21)),
        INSTRUCTION_TEST(C5F8139C1180000000          , vmovlps(qword_ptr(rcx, rdx, 1, 128), xmm3)),
        INSTRUCTION_TEST(62E17C081329                , vmovlps(qword_ptr(rcx), xmm21)),
        INSTRUCTION_TEST(62617C081319                , vmovlps(qword_ptr(rcx), xmm27)),
        INSTRUCTION_TEST(62217C08139CF023010000      , vmovlps(qword_ptr(rax, r14, 8, 291), xmm27)),
        INSTRUCTION_TEST(62617C08135A80              , vmovlps(qword_ptr(rdx, -1024), xmm27)),
        INSTRUCTION_TEST(62E17C0813AAF8FBFFFF        , vmovlps(qword_ptr(rdx, -1032), xmm21)),
        INSTRUCTION_TEST(62617C08139AF8FBFFFF        , vmovlps(qword_ptr(rdx, -1032), xmm27)),
        INSTRUCTION_TEST(62617C08135A7F              , vmovlps(qword_ptr(rdx, 1016), xmm27)),
        INSTRUCTION_TEST(62E17C08136A80              , vmovlps(qword_ptr(rdx, -1024), xmm21)),
        INSTRUCTION_TEST(C5F950CA                    , vmovmskpd(ecx, xmm2)),
        INSTRUCTION_TEST(C5FD50CA                    , vmovmskpd(ecx, ymm2)),
        INSTRUCTION_TEST(C5F850CA                    , vmovmskps(ecx, xmm2)),
        INSTRUCTION_TEST(C5FC50CA                    , vmovmskps(ecx, ymm2)),
        INSTRUCTION_TEST(62617D48E78200200000        , vmovntdq(zmmword_ptr(rdx, 8192), zmm24)),
        INSTRUCTION_TEST(62E17D48E7A2C0DFFFFF        , vmovntdq(zmmword_ptr(rdx, -8256), zmm20)),
        INSTRUCTION_TEST(62E17D48E7A200200000        , vmovntdq(zmmword_ptr(rdx, 8192), zmm20)),
        INSTRUCTION_TEST(62617D48E7427F              , vmovntdq(zmmword_ptr(rdx, 8128), zmm24)),
        INSTRUCTION_TEST(62E17D48E7627F              , vmovntdq(zmmword_ptr(rdx, 8128), zmm20)),
        INSTRUCTION_TEST(62617D48E782C0DFFFFF        , vmovntdq(zmmword_ptr(rdx, -8256), zmm24)),
        INSTRUCTION_TEST(62617D48E74280              , vmovntdq(zmmword_ptr(rdx, -8192), zmm24)),
        INSTRUCTION_TEST(C5F9E79C1180000000          , vmovntdq(xmmword_ptr(rcx, rdx, 1, 128), xmm3)),
        INSTRUCTION_TEST(62617D48E701                , vmovntdq(zmmword_ptr(rcx), zmm24)),
        INSTRUCTION_TEST(62E17D48E721                , vmovntdq(zmmword_ptr(rcx), zmm20)),
        INSTRUCTION_TEST(62F17D48E75C1102            , vmovntdq(zmmword_ptr(rcx, rdx, 1, 128), zmm3)),
        INSTRUCTION_TEST(62A17D48E7A4F034120000      , vmovntdq(zmmword_ptr(rax, r14, 8, 4660), zmm20)),
        INSTRUCTION_TEST(62217D48E784F023010000      , vmovntdq(zmmword_ptr(rax, r14, 8, 291), zmm24)),
        INSTRUCTION_TEST(C5FDE79C1180000000          , vmovntdq(ymmword_ptr(rcx, rdx, 1, 128), ymm3)),
        INSTRUCTION_TEST(62E17D48E76280              , vmovntdq(zmmword_ptr(rdx, -8192), zmm20)),
        INSTRUCTION_TEST(62E27D482A09                , vmovntdqa(zmm17, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62E27D482A4A80              , vmovntdqa(zmm17, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62A27D482A8CF023010000      , vmovntdqa(zmm17, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62E27D482A4A7F              , vmovntdqa(zmm17, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62E27D482A8A00200000        , vmovntdqa(zmm17, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62E27D482A8AC0DFFFFF        , vmovntdqa(zmm17, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62727D482A9200200000        , vmovntdqa(zmm10, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62727D482A92C0DFFFFF        , vmovntdqa(zmm10, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62727D482A5280              , vmovntdqa(zmm10, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62727D482A11                , vmovntdqa(zmm10, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62327D482A94F034120000      , vmovntdqa(zmm10, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62F27D482A4C1A02            , vmovntdqa(zmm1, zmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(C4E27D2A8C1A80000000        , vmovntdqa(ymm1, ymmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(C4E2792A8C1A80000000        , vmovntdqa(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62727D482A527F              , vmovntdqa(zmm10, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62E1FD482B6A7F              , vmovntpd(zmmword_ptr(rdx, 8128), zmm21)),
        INSTRUCTION_TEST(62E1FD482BAA00200000        , vmovntpd(zmmword_ptr(rdx, 8192), zmm21)),
        INSTRUCTION_TEST(62E1FD482B8A00200000        , vmovntpd(zmmword_ptr(rdx, 8192), zmm17)),
        INSTRUCTION_TEST(62E1FD482B4A7F              , vmovntpd(zmmword_ptr(rdx, 8128), zmm17)),
        INSTRUCTION_TEST(62E1FD482BAAC0DFFFFF        , vmovntpd(zmmword_ptr(rdx, -8256), zmm21)),
        INSTRUCTION_TEST(62E1FD482B8AC0DFFFFF        , vmovntpd(zmmword_ptr(rdx, -8256), zmm17)),
        INSTRUCTION_TEST(62E1FD482B6A80              , vmovntpd(zmmword_ptr(rdx, -8192), zmm21)),
        INSTRUCTION_TEST(62E1FD482B4A80              , vmovntpd(zmmword_ptr(rdx, -8192), zmm17)),
        INSTRUCTION_TEST(62E1FD482B29                , vmovntpd(zmmword_ptr(rcx), zmm21)),
        INSTRUCTION_TEST(62E1FD482B09                , vmovntpd(zmmword_ptr(rcx), zmm17)),
        INSTRUCTION_TEST(62F1FD482B5C1102            , vmovntpd(zmmword_ptr(rcx, rdx, 1, 128), zmm3)),
        INSTRUCTION_TEST(62A1FD482BACF034120000      , vmovntpd(zmmword_ptr(rax, r14, 8, 4660), zmm21)),
        INSTRUCTION_TEST(62A1FD482B8CF023010000      , vmovntpd(zmmword_ptr(rax, r14, 8, 291), zmm17)),
        INSTRUCTION_TEST(C5FD2B9C1180000000          , vmovntpd(ymmword_ptr(rcx, rdx, 1, 128), ymm3)),
        INSTRUCTION_TEST(C5F92B9C1180000000          , vmovntpd(xmmword_ptr(rcx, rdx, 1, 128), xmm3)),
        INSTRUCTION_TEST(62E17C482BBA00200000        , vmovntps(zmmword_ptr(rdx, 8192), zmm23)),
        INSTRUCTION_TEST(62F17C482B6A7F              , vmovntps(zmmword_ptr(rdx, 8128), zmm5)),
        INSTRUCTION_TEST(62E17C482B7A7F              , vmovntps(zmmword_ptr(rdx, 8128), zmm23)),
        INSTRUCTION_TEST(62F17C482BAAC0DFFFFF        , vmovntps(zmmword_ptr(rdx, -8256), zmm5)),
        INSTRUCTION_TEST(62E17C482BBAC0DFFFFF        , vmovntps(zmmword_ptr(rdx, -8256), zmm23)),
        INSTRUCTION_TEST(62F17C482B6A80              , vmovntps(zmmword_ptr(rdx, -8192), zmm5)),
        INSTRUCTION_TEST(62E17C482B7A80              , vmovntps(zmmword_ptr(rdx, -8192), zmm23)),
        INSTRUCTION_TEST(62F17C482BAA00200000        , vmovntps(zmmword_ptr(rdx, 8192), zmm5)),
        INSTRUCTION_TEST(62E17C482B39                , vmovntps(zmmword_ptr(rcx), zmm23)),
        INSTRUCTION_TEST(62F17C482B5C1102            , vmovntps(zmmword_ptr(rcx, rdx, 1, 128), zmm3)),
        INSTRUCTION_TEST(62A17C482BBCF034120000      , vmovntps(zmmword_ptr(rax, r14, 8, 4660), zmm23)),
        INSTRUCTION_TEST(62B17C482BACF023010000      , vmovntps(zmmword_ptr(rax, r14, 8, 291), zmm5)),
        INSTRUCTION_TEST(C5FC2B9C1180000000          , vmovntps(ymmword_ptr(rcx, rdx, 1, 128), ymm3)),
        INSTRUCTION_TEST(C5F82B9C1180000000          , vmovntps(xmmword_ptr(rcx, rdx, 1, 128), xmm3)),
        INSTRUCTION_TEST(62F17C482B29                , vmovntps(zmmword_ptr(rcx), zmm5)),
        INSTRUCTION_TEST(6261FD086EC0                , vmovq(xmm24, rax)),
        INSTRUCTION_TEST(6241FD086EC0                , vmovq(xmm24, r8)),
        INSTRUCTION_TEST(C5FA7ECA                    , vmovq(xmm1, xmm2)),
        INSTRUCTION_TEST(C4E1F97ED1                  , vmovq(rcx, xmm2)),
        INSTRUCTION_TEST(C5FA7E8C1A80000000          , vmovq(xmm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(C4E1F96ECA                  , vmovq(xmm1, rdx)),
        INSTRUCTION_TEST(C5F9D69C1180000000          , vmovq(qword_ptr(rcx, rdx, 1, 128), xmm3)),
        INSTRUCTION_TEST(C5FB10AAF8030000            , vmovsd(xmm5, qword_ptr(rdx, 1016))),
        INSTRUCTION_TEST(C5FB10AAF8FBFFFF            , vmovsd(xmm5, qword_ptr(rdx, -1032))),
        INSTRUCTION_TEST(C5FB10AA00FCFFFF            , vmovsd(xmm5, qword_ptr(rdx, -1024))),
        INSTRUCTION_TEST(C5FB1029                    , vmovsd(xmm5, qword_ptr(rcx))),
        INSTRUCTION_TEST(C4A17B10ACF034120000        , vmovsd(xmm5, qword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(6221E70810DB                , vmovsd(xmm27, k0, xmm3, xmm19)),
        INSTRUCTION_TEST(C5EB10CB                    , vmovsd(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(6261FF08104A7F              , vmovsd(xmm25, k0, qword_ptr(rdx, 1016))),
        INSTRUCTION_TEST(6261FF08108AF8FBFFFF        , vmovsd(xmm25, k0, qword_ptr(rdx, -1032))),
        INSTRUCTION_TEST(6261FF08104A80              , vmovsd(xmm25, k0, qword_ptr(rdx, -1024))),
        INSTRUCTION_TEST(6261FF081009                , vmovsd(xmm25, k0, qword_ptr(rcx))),
        INSTRUCTION_TEST(6221FF08108CF023010000      , vmovsd(xmm25, k0, qword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62C1970010EC                , vmovsd(xmm21, k0, xmm29, xmm12)),
        INSTRUCTION_TEST(6261FF08108A00040000        , vmovsd(xmm25, k0, qword_ptr(rdx, 1024))),
        INSTRUCTION_TEST(C5FB108C1A80000000          , vmovsd(xmm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62E1FF08118A00040000        , vmovsd(qword_ptr(rdx, 1024), k0, xmm17)),
        INSTRUCTION_TEST(6261FF08114A7F              , vmovsd(qword_ptr(rdx, 1016), k0, xmm25)),
        INSTRUCTION_TEST(62E1FF08114A7F              , vmovsd(qword_ptr(rdx, 1016), k0, xmm17)),
        INSTRUCTION_TEST(6261FF08118AF8FBFFFF        , vmovsd(qword_ptr(rdx, -1032), k0, xmm25)),
        INSTRUCTION_TEST(62E1FF08118AF8FBFFFF        , vmovsd(qword_ptr(rdx, -1032), k0, xmm17)),
        INSTRUCTION_TEST(6261FF08114A80              , vmovsd(qword_ptr(rdx, -1024), k0, xmm25)),
        INSTRUCTION_TEST(62E1FF08114A80              , vmovsd(qword_ptr(rdx, -1024), k0, xmm17)),
        INSTRUCTION_TEST(6261FF081109                , vmovsd(qword_ptr(rcx), k0, xmm25)),
        INSTRUCTION_TEST(62E1FF081109                , vmovsd(qword_ptr(rcx), k0, xmm17)),
        INSTRUCTION_TEST(C5FB119C1180000000          , vmovsd(qword_ptr(rcx, rdx, 1, 128), xmm3)),
        INSTRUCTION_TEST(62A1FF08118CF034120000      , vmovsd(qword_ptr(rax, r14, 8, 4660), k0, xmm17)),
        INSTRUCTION_TEST(6221FF08118CF023010000      , vmovsd(qword_ptr(rax, r14, 8, 291), k0, xmm25)),
        INSTRUCTION_TEST(C5FB10AA00040000            , vmovsd(xmm5, qword_ptr(rdx, 1024))),
        INSTRUCTION_TEST(6261FF08118A00040000        , vmovsd(qword_ptr(rdx, 1024), k0, xmm25)),
        INSTRUCTION_TEST(62A17E4816D4                , vmovshdup(zmm18, k0, zmm20)),
        INSTRUCTION_TEST(62A17E481694F034120000      , vmovshdup(zmm18, k0, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62E17E48168200200000        , vmovshdup(zmm16, k0, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62E17E481611                , vmovshdup(zmm18, k0, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62E17E4816427F              , vmovshdup(zmm16, k0, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62E17E481692C0DFFFFF        , vmovshdup(zmm18, k0, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62E17E4816527F              , vmovshdup(zmm18, k0, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62E17E48169200200000        , vmovshdup(zmm18, k0, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62E17E48165280              , vmovshdup(zmm18, k0, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62E17E481682C0DFFFFF        , vmovshdup(zmm16, k0, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(C5FA16CA                    , vmovshdup(xmm1, xmm2)),
        INSTRUCTION_TEST(62E17E481601                , vmovshdup(zmm16, k0, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(C5FA168C1A80000000          , vmovshdup(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(C5FE16CA                    , vmovshdup(ymm1, ymm2)),
        INSTRUCTION_TEST(C5FE168C1A80000000          , vmovshdup(ymm1, ymmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62E17E48164280              , vmovshdup(zmm16, k0, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62F17E48164C1A02            , vmovshdup(zmm1, k0, zmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62817E4816C3                , vmovshdup(zmm16, k0, zmm27)),
        INSTRUCTION_TEST(62A17E481684F023010000      , vmovshdup(zmm16, k0, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62F17E4816CA                , vmovshdup(zmm1, k0, zmm2)),
        INSTRUCTION_TEST(62717E48126A7F              , vmovsldup(zmm13, k0, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62717E48128A00200000        , vmovsldup(zmm9, k0, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62717E48124A7F              , vmovsldup(zmm9, k0, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62717E48128AC0DFFFFF        , vmovsldup(zmm9, k0, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62717E48124A80              , vmovsldup(zmm9, k0, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62717E481209                , vmovsldup(zmm9, k0, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62717E4812CD                , vmovsldup(zmm9, k0, zmm5)),
        INSTRUCTION_TEST(62317E48128CF034120000      , vmovsldup(zmm9, k0, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62717E4812AAC0DFFFFF        , vmovsldup(zmm13, k0, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62717E4812AA00200000        , vmovsldup(zmm13, k0, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62717E481229                , vmovsldup(zmm13, k0, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62317E4812ACF023010000      , vmovsldup(zmm13, k0, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62517E4812EE                , vmovsldup(zmm13, k0, zmm14)),
        INSTRUCTION_TEST(62F17E48124C1A02            , vmovsldup(zmm1, k0, zmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F17E4812CA                , vmovsldup(zmm1, k0, zmm2)),
        INSTRUCTION_TEST(C5FE128C1A80000000          , vmovsldup(ymm1, ymmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(C5FE12CA                    , vmovsldup(ymm1, ymm2)),
        INSTRUCTION_TEST(C5FA128C1A80000000          , vmovsldup(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(C5FA12CA                    , vmovsldup(xmm1, xmm2)),
        INSTRUCTION_TEST(62717E48126A80              , vmovsldup(zmm13, k0, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(C5FA108AFCFDFFFF            , vmovss(xmm1, dword_ptr(rdx, -516))),
        INSTRUCTION_TEST(C5FA108AFC010000            , vmovss(xmm1, dword_ptr(rdx, 508))),
        INSTRUCTION_TEST(C5FA108A00020000            , vmovss(xmm1, dword_ptr(rdx, 512))),
        INSTRUCTION_TEST(C5FA108C1A80000000          , vmovss(xmm1, dword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(C5EA10CB                    , vmovss(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(62C1260010D6                , vmovss(xmm18, k0, xmm27, xmm14)),
        INSTRUCTION_TEST(C5FA1092FC010000            , vmovss(xmm2, dword_ptr(rdx, 508))),
        INSTRUCTION_TEST(C5FA1011                    , vmovss(xmm2, dword_ptr(rcx))),
        INSTRUCTION_TEST(C5FA109200FEFFFF            , vmovss(xmm2, dword_ptr(rdx, -512))),
        INSTRUCTION_TEST(C5FA1092FCFDFFFF            , vmovss(xmm2, dword_ptr(rdx, -516))),
        INSTRUCTION_TEST(C5FA109200020000            , vmovss(xmm2, dword_ptr(rdx, 512))),
        INSTRUCTION_TEST(C5FA108A00FEFFFF            , vmovss(xmm1, dword_ptr(rdx, -512))),
        INSTRUCTION_TEST(C4A17A1094F023010000        , vmovss(xmm2, dword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(C5FA1009                    , vmovss(xmm1, dword_ptr(rcx))),
        INSTRUCTION_TEST(C57A11AAFCFDFFFF            , vmovss(dword_ptr(rdx, -516), xmm13)),
        INSTRUCTION_TEST(C5FA11AA00020000            , vmovss(dword_ptr(rdx, 512), xmm5)),
        INSTRUCTION_TEST(C4A17A108CF034120000        , vmovss(xmm1, dword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(C4A17A11ACF023010000        , vmovss(dword_ptr(rax, r14, 8, 291), xmm5)),
        INSTRUCTION_TEST(C4217A11ACF034120000        , vmovss(dword_ptr(rax, r14, 8, 4660), xmm13)),
        INSTRUCTION_TEST(C5FA119C1180000000          , vmovss(dword_ptr(rcx, rdx, 1, 128), xmm3)),
        INSTRUCTION_TEST(C5FA1129                    , vmovss(dword_ptr(rcx), xmm5)),
        INSTRUCTION_TEST(C57A11AA00FEFFFF            , vmovss(dword_ptr(rdx, -512), xmm13)),
        INSTRUCTION_TEST(C57A1129                    , vmovss(dword_ptr(rcx), xmm13)),
        INSTRUCTION_TEST(C5FA11AAFCFDFFFF            , vmovss(dword_ptr(rdx, -516), xmm5)),
        INSTRUCTION_TEST(C57A11AAFC010000            , vmovss(dword_ptr(rdx, 508), xmm13)),
        INSTRUCTION_TEST(C5FA11AAFC010000            , vmovss(dword_ptr(rdx, 508), xmm5)),
        INSTRUCTION_TEST(C57A11AA00020000            , vmovss(dword_ptr(rdx, 512), xmm13)),
        INSTRUCTION_TEST(C5FA11AA00FEFFFF            , vmovss(dword_ptr(rdx, -512), xmm5)),
        INSTRUCTION_TEST(6261FD48109AC0DFFFFF        , vmovupd(zmm27, k0, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(6271FD481111                , vmovupd(zmmword_ptr(rcx), k0, zmm10)),
        INSTRUCTION_TEST(6261FD48105A7F              , vmovupd(zmm27, k0, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(6261FD48109A00200000        , vmovupd(zmm27, k0, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(6231FD481194F023010000      , vmovupd(zmmword_ptr(rax, r14, 8, 291), k0, zmm10)),
        INSTRUCTION_TEST(62F1FD48115C1102            , vmovupd(zmmword_ptr(rcx, rdx, 1, 128), k0, zmm3)),
        INSTRUCTION_TEST(6271FD481101                , vmovupd(zmmword_ptr(rcx), k0, zmm8)),
        INSTRUCTION_TEST(6271FD48119200200000        , vmovupd(zmmword_ptr(rdx, 8192), k0, zmm10)),
        INSTRUCTION_TEST(6271FD48114280              , vmovupd(zmmword_ptr(rdx, -8192), k0, zmm8)),
        INSTRUCTION_TEST(6271FD481192C0DFFFFF        , vmovupd(zmmword_ptr(rdx, -8256), k0, zmm10)),
        INSTRUCTION_TEST(6271FD481182C0DFFFFF        , vmovupd(zmmword_ptr(rdx, -8256), k0, zmm8)),
        INSTRUCTION_TEST(6271FD4811527F              , vmovupd(zmmword_ptr(rdx, 8128), k0, zmm10)),
        INSTRUCTION_TEST(6271FD4811427F              , vmovupd(zmmword_ptr(rdx, 8128), k0, zmm8)),
        INSTRUCTION_TEST(6271FD48118200200000        , vmovupd(zmmword_ptr(rdx, 8192), k0, zmm8)),
        INSTRUCTION_TEST(6261FD48105A80              , vmovupd(zmm27, k0, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(6271FD48115280              , vmovupd(zmmword_ptr(rdx, -8192), k0, zmm10)),
        INSTRUCTION_TEST(6261FD481019                , vmovupd(zmm27, k0, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(6231FD481184F034120000      , vmovupd(zmmword_ptr(rax, r14, 8, 4660), k0, zmm8)),
        INSTRUCTION_TEST(6241FD4810D9                , vmovupd(zmm27, k0, zmm9)),
        INSTRUCTION_TEST(6221FD48109CF023010000      , vmovupd(zmm27, k0, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(C5F910CA                    , vmovupd(xmm1, xmm2)),
        INSTRUCTION_TEST(C5F9119C1180000000          , vmovupd(xmmword_ptr(rcx, rdx, 1, 128), xmm3)),
        INSTRUCTION_TEST(C5FD10CA                    , vmovupd(ymm1, ymm2)),
        INSTRUCTION_TEST(C5FD108C1A80000000          , vmovupd(ymm1, ymmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(C5FD119C1180000000          , vmovupd(ymmword_ptr(rcx, rdx, 1, 128), ymm3)),
        INSTRUCTION_TEST(62F1FD4810CA                , vmovupd(zmm1, k0, zmm2)),
        INSTRUCTION_TEST(62F1FD48104C1A02            , vmovupd(zmm1, k0, zmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(C5F9108C1A80000000          , vmovupd(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62A1FD481084F034120000      , vmovupd(zmm16, k0, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62E1FD4810C6                , vmovupd(zmm16, k0, zmm6)),
        INSTRUCTION_TEST(62E1FD4810427F              , vmovupd(zmm16, k0, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62E1FD481082C0DFFFFF        , vmovupd(zmm16, k0, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62E1FD48108200200000        , vmovupd(zmm16, k0, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62E1FD481001                , vmovupd(zmm16, k0, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62E1FD48104280              , vmovupd(zmm16, k0, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62E17C4810B200200000        , vmovups(zmm22, k0, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62F17C4811A200200000        , vmovups(zmmword_ptr(rdx, 8192), k0, zmm4)),
        INSTRUCTION_TEST(62617C48118200200000        , vmovups(zmmword_ptr(rdx, 8192), k0, zmm24)),
        INSTRUCTION_TEST(62F17C4811627F              , vmovups(zmmword_ptr(rdx, 8128), k0, zmm4)),
        INSTRUCTION_TEST(62617C4811427F              , vmovups(zmmword_ptr(rdx, 8128), k0, zmm24)),
        INSTRUCTION_TEST(62617C481182C0DFFFFF        , vmovups(zmmword_ptr(rdx, -8256), k0, zmm24)),
        INSTRUCTION_TEST(62F17C48116280              , vmovups(zmmword_ptr(rdx, -8192), k0, zmm4)),
        INSTRUCTION_TEST(62617C48114280              , vmovups(zmmword_ptr(rdx, -8192), k0, zmm24)),
        INSTRUCTION_TEST(62F17C481121                , vmovups(zmmword_ptr(rcx), k0, zmm4)),
        INSTRUCTION_TEST(62617C481101                , vmovups(zmmword_ptr(rcx), k0, zmm24)),
        INSTRUCTION_TEST(62F17C48115C1102            , vmovups(zmmword_ptr(rcx, rdx, 1, 128), k0, zmm3)),
        INSTRUCTION_TEST(62B17C4811A4F034120000      , vmovups(zmmword_ptr(rax, r14, 8, 4660), k0, zmm4)),
        INSTRUCTION_TEST(62217C481184F023010000      , vmovups(zmmword_ptr(rax, r14, 8, 291), k0, zmm24)),
        INSTRUCTION_TEST(62E17C4810727F              , vmovups(zmm22, k0, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62F17C4811A2C0DFFFFF        , vmovups(zmmword_ptr(rdx, -8256), k0, zmm4)),
        INSTRUCTION_TEST(62E17C48107280              , vmovups(zmm22, k0, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62E17C4810B2C0DFFFFF        , vmovups(zmm22, k0, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(C5F810CA                    , vmovups(xmm1, xmm2)),
        INSTRUCTION_TEST(C5F8108C1A80000000          , vmovups(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(C5F8119C1180000000          , vmovups(xmmword_ptr(rcx, rdx, 1, 128), xmm3)),
        INSTRUCTION_TEST(C5FC10CA                    , vmovups(ymm1, ymm2)),
        INSTRUCTION_TEST(C5FC108C1A80000000          , vmovups(ymm1, ymmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(C5FC119C1180000000          , vmovups(ymmword_ptr(rcx, rdx, 1, 128), ymm3)),
        INSTRUCTION_TEST(62F17C48104C1A02            , vmovups(zmm1, k0, zmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62A17C4810F6                , vmovups(zmm22, k0, zmm22)),
        INSTRUCTION_TEST(62E17C4810F4                , vmovups(zmm22, k0, zmm4)),
        INSTRUCTION_TEST(62A17C4810B4F023010000      , vmovups(zmm22, k0, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62A17C4810B4F034120000      , vmovups(zmm22, k0, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62E17C481031                , vmovups(zmm22, k0, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62F17C4810CA                , vmovups(zmm1, k0, zmm2)),
        INSTRUCTION_TEST(C4E36D428C2B8000000001      , vmpsadbw(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(C4E36D42CB01                , vmpsadbw(ymm1, ymm2, ymm3, Imm(1))),
        INSTRUCTION_TEST(C4E369428C2B8000000001      , vmpsadbw(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(C4E36942CB01                , vmpsadbw(xmm1, xmm2, xmm3, Imm(1))),
        INSTRUCTION_TEST(0FC7B41180000000            , vmptrld(qword_ptr(rcx, rdx, 1, 128))),
        INSTRUCTION_TEST(0FC7BC1180000000            , vmptrst(qword_ptr(rcx, rdx, 1, 128))),
        INSTRUCTION_TEST(0F789C1180000000            , vmread(qword_ptr(rcx, rdx, 1, 128), rbx)),
        INSTRUCTION_TEST(0F78D1                      , vmread(rcx, rdx)),
        INSTRUCTION_TEST(0F01C3                      , vmresume()),
        INSTRUCTION_TEST(6261DD4859427F              , vmulpd(zmm24, k0, zmm4, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(6261DD485982C0DFFFFF        , vmulpd(zmm24, k0, zmm4, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(6261DD48598200200000        , vmulpd(zmm24, k0, zmm4, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(6261DD48594280              , vmulpd(zmm24, k0, zmm4, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(6261DD485901                , vmulpd(zmm24, k0, zmm4, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(6221DD485984F023010000      , vmulpd(zmm24, k0, zmm4, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(6221DD4859C7                , vmulpd(zmm24, k0, zmm4, zmm23)),
        INSTRUCTION_TEST(6261B540598200200000        , vmulpd(zmm24, k0, zmm25, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(6261B54059427F              , vmulpd(zmm24, k0, zmm25, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(6261B5405982C0DFFFFF        , vmulpd(zmm24, k0, zmm25, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(C5ED59CB                    , vmulpd(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(6261B5405901                , vmulpd(zmm24, k0, zmm25, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(6221B5405984F034120000      , vmulpd(zmm24, k0, zmm25, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(6261B54059C5                , vmulpd(zmm24, k0, zmm25, zmm5)),
        INSTRUCTION_TEST(62F1ED48594C2B02            , vmulpd(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F1ED4859CB                , vmulpd(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(C5ED598C2B80000000          , vmulpd(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C5E9598C2B80000000          , vmulpd(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C5E959CB                    , vmulpd(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(6261B540594280              , vmulpd(zmm24, k0, zmm25, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62F14C48599A00200000        , vmulps(zmm3, k0, zmm6, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62F1244059F3                , vmulps(zmm6, k0, zmm27, zmm3)),
        INSTRUCTION_TEST(62B1244059B4F034120000      , vmulps(zmm6, k0, zmm27, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62F124405931                , vmulps(zmm6, k0, zmm27, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62F1244059727F              , vmulps(zmm6, k0, zmm27, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62F12440597280              , vmulps(zmm6, k0, zmm27, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62F1244059B2C0DFFFFF        , vmulps(zmm6, k0, zmm27, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62F14C48595A7F              , vmulps(zmm3, k0, zmm6, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62F1244059B200200000        , vmulps(zmm6, k0, zmm27, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62F14C48599AC0DFFFFF        , vmulps(zmm3, k0, zmm6, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(C5EC598C2B80000000          , vmulps(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F14C485919                , vmulps(zmm3, k0, zmm6, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(C5E859CB                    , vmulps(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(C5E8598C2B80000000          , vmulps(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C5EC59CB                    , vmulps(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(62F14C48595A80              , vmulps(zmm3, k0, zmm6, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62F16C48594C2B02            , vmulps(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62914C4859D8                , vmulps(zmm3, k0, zmm6, zmm24)),
        INSTRUCTION_TEST(62B14C48599CF023010000      , vmulps(zmm3, k0, zmm6, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62F16C4859CB                , vmulps(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(62A1970059BCF034120000      , vmulsd(xmm23, k0, xmm29, qword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62A1970059FE                , vmulsd(xmm23, k0, xmm29, xmm22)),
        INSTRUCTION_TEST(62E1970059BA00040000        , vmulsd(xmm23, k0, xmm29, qword_ptr(rdx, 1024))),
        INSTRUCTION_TEST(62E19700597A7F              , vmulsd(xmm23, k0, xmm29, qword_ptr(rdx, 1016))),
        INSTRUCTION_TEST(62E1970059BAF8FBFFFF        , vmulsd(xmm23, k0, xmm29, qword_ptr(rdx, -1032))),
        INSTRUCTION_TEST(62E19700597A80              , vmulsd(xmm23, k0, xmm29, qword_ptr(rdx, -1024))),
        INSTRUCTION_TEST(62E197005939                , vmulsd(xmm23, k0, xmm29, qword_ptr(rcx))),
        INSTRUCTION_TEST(6231DF0859EA                , vmulsd(xmm13, k0, xmm4, xmm18)),
        INSTRUCTION_TEST(C55B59AAF8030000            , vmulsd(xmm13, xmm4, qword_ptr(rdx, 1016))),
        INSTRUCTION_TEST(C55B59AAF8FBFFFF            , vmulsd(xmm13, xmm4, qword_ptr(rdx, -1032))),
        INSTRUCTION_TEST(C55B59AA00FCFFFF            , vmulsd(xmm13, xmm4, qword_ptr(rdx, -1024))),
        INSTRUCTION_TEST(C55B5929                    , vmulsd(xmm13, xmm4, qword_ptr(rcx))),
        INSTRUCTION_TEST(C4215B59ACF023010000        , vmulsd(xmm13, xmm4, qword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(C5EB59CB                    , vmulsd(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(C5EB598C2B80000000          , vmulsd(xmm1, xmm2, qword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C55B59AA00040000            , vmulsd(xmm13, xmm4, qword_ptr(rdx, 1024))),
        INSTRUCTION_TEST(62212E00598CF034120000      , vmulss(xmm25, k0, xmm26, dword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62612E00598A00020000        , vmulss(xmm25, k0, xmm26, dword_ptr(rdx, 512))),
        INSTRUCTION_TEST(62612E00594A7F              , vmulss(xmm25, k0, xmm26, dword_ptr(rdx, 508))),
        INSTRUCTION_TEST(62612E00598AFCFDFFFF        , vmulss(xmm25, k0, xmm26, dword_ptr(rdx, -516))),
        INSTRUCTION_TEST(62612E00594A80              , vmulss(xmm25, k0, xmm26, dword_ptr(rdx, -512))),
        INSTRUCTION_TEST(62412E0059C8                , vmulss(xmm25, k0, xmm26, xmm8)),
        INSTRUCTION_TEST(62C12E0859F6                , vmulss(xmm22, k0, xmm10, xmm14)),
        INSTRUCTION_TEST(62612E005909                , vmulss(xmm25, k0, xmm26, dword_ptr(rcx))),
        INSTRUCTION_TEST(62E12E0859727F              , vmulss(xmm22, k0, xmm10, dword_ptr(rdx, 508))),
        INSTRUCTION_TEST(62E12E0859B2FCFDFFFF        , vmulss(xmm22, k0, xmm10, dword_ptr(rdx, -516))),
        INSTRUCTION_TEST(62E12E08597280              , vmulss(xmm22, k0, xmm10, dword_ptr(rdx, -512))),
        INSTRUCTION_TEST(62E12E085931                , vmulss(xmm22, k0, xmm10, dword_ptr(rcx))),
        INSTRUCTION_TEST(62A12E0859B4F023010000      , vmulss(xmm22, k0, xmm10, dword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(C5EA59CB                    , vmulss(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(C5EA598C2B80000000          , vmulss(xmm1, xmm2, dword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62E12E0859B200020000        , vmulss(xmm22, k0, xmm10, dword_ptr(rdx, 512))),
        INSTRUCTION_TEST(0F79CA                      , vmwrite(rcx, rdx)),
        INSTRUCTION_TEST(0F798C1A80000000            , vmwrite(rcx, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(F30FC7B41180000000          , vmxon(qword_ptr(rcx, rdx, 1, 128))),
        INSTRUCTION_TEST(C5E956CB                    , vorpd(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(C5E9568C2B80000000          , vorpd(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C5ED56CB                    , vorpd(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(C5ED568C2B80000000          , vorpd(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F1ED4856CB                , vorpd(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(62F1ED48564C2B02            , vorpd(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F16C48564C2B02            , vorps(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F16C4856CB                , vorps(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(C5EC568C2B80000000          , vorps(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C5E8568C2B80000000          , vorps(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C5E856CB                    , vorps(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(C5EC56CB                    , vorps(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(C4E2791CCA                  , vpabsb(xmm1, xmm2)),
        INSTRUCTION_TEST(C4E2791C8C1A80000000        , vpabsb(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(C4E27D1CCA                  , vpabsb(ymm1, ymm2)),
        INSTRUCTION_TEST(C4E27D1C8C1A80000000        , vpabsb(ymm1, ymmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F27D481CCA                , vpabsb(zmm1, k0, zmm2)),
        INSTRUCTION_TEST(62F27D481C4C1A02            , vpabsb(zmm1, k0, zmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62727D481EBA00200000        , vpabsd(zmm15, k0, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62927D481EE5                , vpabsd(zmm4, k0, zmm29)),
        INSTRUCTION_TEST(62B27D481EA4F034120000      , vpabsd(zmm4, k0, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62F27D481E627F              , vpabsd(zmm4, k0, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62F27D481E6280              , vpabsd(zmm4, k0, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62F27D481EA2C0DFFFFF        , vpabsd(zmm4, k0, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62F27D481EA200200000        , vpabsd(zmm4, k0, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62F27D481E21                , vpabsd(zmm4, k0, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62727D481EBAC0DFFFFF        , vpabsd(zmm15, k0, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62727D481E7A7F              , vpabsd(zmm15, k0, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62727D481E39                , vpabsd(zmm15, k0, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(C4E2791ECA                  , vpabsd(xmm1, xmm2)),
        INSTRUCTION_TEST(C4E2791E8C1A80000000        , vpabsd(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(C4E27D1ECA                  , vpabsd(ymm1, ymm2)),
        INSTRUCTION_TEST(C4E27D1E8C1A80000000        , vpabsd(ymm1, ymmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62727D481E7A80              , vpabsd(zmm15, k0, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62F27D481E4C1A02            , vpabsd(zmm1, k0, zmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62527D481EFE                , vpabsd(zmm15, k0, zmm14)),
        INSTRUCTION_TEST(62327D481EBCF023010000      , vpabsd(zmm15, k0, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62F27D481ECA                , vpabsd(zmm1, k0, zmm2)),
        INSTRUCTION_TEST(62E2FD481F8AC0DFFFFF        , vpabsq(zmm17, k0, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62F2FD481F6A7F              , vpabsq(zmm5, k0, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62F2FD481FAAC0DFFFFF        , vpabsq(zmm5, k0, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62F2FD481F6A80              , vpabsq(zmm5, k0, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62F2FD481F29                , vpabsq(zmm5, k0, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62B2FD481FACF023010000      , vpabsq(zmm5, k0, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(6292FD481FE8                , vpabsq(zmm5, k0, zmm24)),
        INSTRUCTION_TEST(62E2FD481F8A00200000        , vpabsq(zmm17, k0, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62E2FD481F4A7F              , vpabsq(zmm17, k0, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62F2FD481FAA00200000        , vpabsq(zmm5, k0, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62E2FD481F4A80              , vpabsq(zmm17, k0, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62A2FD481F8CF034120000      , vpabsq(zmm17, k0, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62E2FD481FCB                , vpabsq(zmm17, k0, zmm3)),
        INSTRUCTION_TEST(62F2FD481F4C1A02            , vpabsq(zmm1, k0, zmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F2FD481FCA                , vpabsq(zmm1, k0, zmm2)),
        INSTRUCTION_TEST(62F2FD281F4C1A04            , vpabsq(ymm1, k0, ymmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F2FD281FCA                , vpabsq(ymm1, k0, ymm2)),
        INSTRUCTION_TEST(62F2FD081F4C1A08            , vpabsq(xmm1, k0, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F2FD081FCA                , vpabsq(xmm1, k0, xmm2)),
        INSTRUCTION_TEST(62E2FD481F09                , vpabsq(zmm17, k0, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(C4E2791DCA                  , vpabsw(xmm1, xmm2)),
        INSTRUCTION_TEST(C4E2791D8C1A80000000        , vpabsw(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(C4E27D1DCA                  , vpabsw(ymm1, ymm2)),
        INSTRUCTION_TEST(C4E27D1D8C1A80000000        , vpabsw(ymm1, ymmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F27D481DCA                , vpabsw(zmm1, k0, zmm2)),
        INSTRUCTION_TEST(62F27D481D4C1A02            , vpabsw(zmm1, k0, zmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F16D486B4C2B02            , vpackssdw(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F16D486BCB                , vpackssdw(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(C5ED6B8C2B80000000          , vpackssdw(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C5E96B8C2B80000000          , vpackssdw(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C5E96BCB                    , vpackssdw(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(C5ED6BCB                    , vpackssdw(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(C5E963CB                    , vpacksswb(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(C5E9638C2B80000000          , vpacksswb(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C5ED63CB                    , vpacksswb(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(C5ED638C2B80000000          , vpacksswb(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F16D4863CB                , vpacksswb(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(62F16D48634C2B02            , vpacksswb(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E2692BCB                  , vpackusdw(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(C4E2692B8C2B80000000        , vpackusdw(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E26D2BCB                  , vpackusdw(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(C4E26D2B8C2B80000000        , vpackusdw(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F26D482BCB                , vpackusdw(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(62F26D482B4C2B02            , vpackusdw(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F16D48674C2B02            , vpackuswb(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F16D4867CB                , vpackuswb(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(C5ED678C2B80000000          , vpackuswb(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C5E9678C2B80000000          , vpackuswb(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C5E967CB                    , vpackuswb(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(C5ED67CB                    , vpackuswb(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(62F16D48FCCB                , vpaddb(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(C5EDFC8C2B80000000          , vpaddb(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F16D48FC4C2B02            , vpaddb(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C5E9FC8C2B80000000          , vpaddb(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C5E9FCCB                    , vpaddb(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(C5EDFCCB                    , vpaddb(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(62615548FE9200200000        , vpaddd(zmm26, k0, zmm5, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62615548FE527F              , vpaddd(zmm26, k0, zmm5, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62615548FE92C0DFFFFF        , vpaddd(zmm26, k0, zmm5, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62615548FE5280              , vpaddd(zmm26, k0, zmm5, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62615548FE11                , vpaddd(zmm26, k0, zmm5, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62215548FE94F023010000      , vpaddd(zmm26, k0, zmm5, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62215548FED4                , vpaddd(zmm26, k0, zmm5, zmm20)),
        INSTRUCTION_TEST(62713D48FE9200200000        , vpaddd(zmm10, k0, zmm8, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62713D48FE92C0DFFFFF        , vpaddd(zmm10, k0, zmm8, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62713D48FE527F              , vpaddd(zmm10, k0, zmm8, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62713D48FE11                , vpaddd(zmm10, k0, zmm8, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(C5E9FECB                    , vpaddd(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(C5E9FE8C2B80000000          , vpaddd(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C5EDFECB                    , vpaddd(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(C5EDFE8C2B80000000          , vpaddd(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62713D48FE5280              , vpaddd(zmm10, k0, zmm8, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62F16D48FE4C2B02            , vpaddd(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62513D48FED3                , vpaddd(zmm10, k0, zmm8, zmm11)),
        INSTRUCTION_TEST(62313D48FE94F034120000      , vpaddd(zmm10, k0, zmm8, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62F16D48FECB                , vpaddd(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(6271B540D48200200000        , vpaddq(zmm8, k0, zmm25, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(6271B540D4427F              , vpaddq(zmm8, k0, zmm25, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(6271B540D482C0DFFFFF        , vpaddq(zmm8, k0, zmm25, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(6271B540D44280              , vpaddq(zmm8, k0, zmm25, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(6231B540D484F023010000      , vpaddq(zmm8, k0, zmm25, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(6251B540D4C6                , vpaddq(zmm8, k0, zmm25, zmm14)),
        INSTRUCTION_TEST(62F1DD48D4B200200000        , vpaddq(zmm6, k0, zmm4, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62F1DD48D4727F              , vpaddq(zmm6, k0, zmm4, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62F1DD48D4B2C0DFFFFF        , vpaddq(zmm6, k0, zmm4, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(6271B540D401                , vpaddq(zmm8, k0, zmm25, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62F1DD48D431                , vpaddq(zmm6, k0, zmm4, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62B1DD48D4B4F034120000      , vpaddq(zmm6, k0, zmm4, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(6291DD48D4F2                , vpaddq(zmm6, k0, zmm4, zmm26)),
        INSTRUCTION_TEST(62F1ED48D44C2B02            , vpaddq(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F1ED48D4CB                , vpaddq(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(C5EDD48C2B80000000          , vpaddq(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C5EDD4CB                    , vpaddq(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(C5E9D48C2B80000000          , vpaddq(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C5E9D4CB                    , vpaddq(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(62F1DD48D47280              , vpaddq(zmm6, k0, zmm4, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62F16D48EC4C2B02            , vpaddsb(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F16D48ECCB                , vpaddsb(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(C5EDEC8C2B80000000          , vpaddsb(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C5EDECCB                    , vpaddsb(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(C5E9ECCB                    , vpaddsb(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(C5E9EC8C2B80000000          , vpaddsb(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C5E9EDCB                    , vpaddsw(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(C5E9ED8C2B80000000          , vpaddsw(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C5EDEDCB                    , vpaddsw(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(C5EDED8C2B80000000          , vpaddsw(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F16D48EDCB                , vpaddsw(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(62F16D48ED4C2B02            , vpaddsw(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F16D48DCCB                , vpaddusb(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(62F16D48DC4C2B02            , vpaddusb(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C5EDDC8C2B80000000          , vpaddusb(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C5E9DCCB                    , vpaddusb(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(C5E9DC8C2B80000000          , vpaddusb(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C5EDDCCB                    , vpaddusb(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(C5E9DDCB                    , vpaddusw(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(C5E9DD8C2B80000000          , vpaddusw(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C5EDDDCB                    , vpaddusw(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(C5EDDD8C2B80000000          , vpaddusw(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F16D48DDCB                , vpaddusw(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(62F16D48DD4C2B02            , vpaddusw(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C5E9FD8C2B80000000          , vpaddw(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C5EDFDCB                    , vpaddw(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(C5EDFD8C2B80000000          , vpaddw(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F16D48FDCB                , vpaddw(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(62F16D48FD4C2B02            , vpaddw(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C5E9FDCB                    , vpaddw(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(C4E36D0F8C2B8000000001      , vpalignr(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F36D480FCB01              , vpalignr(zmm1, k0, zmm2, zmm3, Imm(1))),
        INSTRUCTION_TEST(62F36D480F4C2B0201          , vpalignr(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(C4E36D0FCB01                , vpalignr(ymm1, ymm2, ymm3, Imm(1))),
        INSTRUCTION_TEST(C4E3690FCB01                , vpalignr(xmm1, xmm2, xmm3, Imm(1))),
        INSTRUCTION_TEST(C4E3690F8C2B8000000001      , vpalignr(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(C5E9DBCB                    , vpand(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(C5E9DB8C2B80000000          , vpand(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C5EDDBCB                    , vpand(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(C5EDDB8C2B80000000          , vpand(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62E14D40DB9A00200000        , vpandd(zmm19, k0, zmm22, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62E14D40DB5A7F              , vpandd(zmm19, k0, zmm22, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62E14D40DB9AC0DFFFFF        , vpandd(zmm19, k0, zmm22, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62E14D40DB5A80              , vpandd(zmm19, k0, zmm22, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62E14D40DB19                , vpandd(zmm19, k0, zmm22, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62A14D40DB9CF023010000      , vpandd(zmm19, k0, zmm22, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62814D40DBD9                , vpandd(zmm19, k0, zmm22, zmm25)),
        INSTRUCTION_TEST(62F11540DB8A00200000        , vpandd(zmm1, k0, zmm29, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62F11540DB8AC0DFFFFF        , vpandd(zmm1, k0, zmm29, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62F11540DB4A7F              , vpandd(zmm1, k0, zmm29, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62F11540DB09                , vpandd(zmm1, k0, zmm29, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62B11540DB8CF034120000      , vpandd(zmm1, k0, zmm29, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62B11540DBC8                , vpandd(zmm1, k0, zmm29, zmm16)),
        INSTRUCTION_TEST(62F16D48DB4C2B02            , vpandd(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F16D48DBCB                , vpandd(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(62F16D28DB4C2B04            , vpandd(ymm1, k0, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F16D28DBCB                , vpandd(ymm1, k0, ymm2, ymm3)),
        INSTRUCTION_TEST(62F16D08DB4C2B08            , vpandd(xmm1, k0, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F16D08DBCB                , vpandd(xmm1, k0, xmm2, xmm3)),
        INSTRUCTION_TEST(62F11540DB4A80              , vpandd(zmm1, k0, zmm29, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(C5EDDF8C2B80000000          , vpandn(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C5EDDFCB                    , vpandn(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(C5E9DF8C2B80000000          , vpandn(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C5E9DFCB                    , vpandn(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(62611D40DFB200200000        , vpandnd(zmm30, k0, zmm28, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62611D40DF727F              , vpandnd(zmm30, k0, zmm28, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62611D40DFB2C0DFFFFF        , vpandnd(zmm30, k0, zmm28, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62611D40DF7280              , vpandnd(zmm30, k0, zmm28, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62611D40DF31                , vpandnd(zmm30, k0, zmm28, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62211D40DFB4F023010000      , vpandnd(zmm30, k0, zmm28, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62411D40DFF7                , vpandnd(zmm30, k0, zmm28, zmm15)),
        INSTRUCTION_TEST(62E17540DFB200200000        , vpandnd(zmm22, k0, zmm17, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62E17540DF727F              , vpandnd(zmm22, k0, zmm17, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62E17540DFB2C0DFFFFF        , vpandnd(zmm22, k0, zmm17, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62A17540DFB4F034120000      , vpandnd(zmm22, k0, zmm17, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62E17540DF31                , vpandnd(zmm22, k0, zmm17, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62A17540DFF0                , vpandnd(zmm22, k0, zmm17, zmm16)),
        INSTRUCTION_TEST(62F16D48DF4C2B02            , vpandnd(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F16D48DFCB                , vpandnd(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(62F16D28DF4C2B04            , vpandnd(ymm1, k0, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F16D28DFCB                , vpandnd(ymm1, k0, ymm2, ymm3)),
        INSTRUCTION_TEST(62F16D08DF4C2B08            , vpandnd(xmm1, k0, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F16D08DFCB                , vpandnd(xmm1, k0, xmm2, xmm3)),
        INSTRUCTION_TEST(62E17540DF7280              , vpandnd(zmm22, k0, zmm17, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62E1C548DFA200200000        , vpandnq(zmm20, k0, zmm7, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(6261DD48DFCA                , vpandnq(zmm25, k0, zmm4, zmm2)),
        INSTRUCTION_TEST(6221DD48DF8CF034120000      , vpandnq(zmm25, k0, zmm4, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(6261DD48DF8A00200000        , vpandnq(zmm25, k0, zmm4, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(6261DD48DF4A7F              , vpandnq(zmm25, k0, zmm4, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(6261DD48DF4A80              , vpandnq(zmm25, k0, zmm4, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(6261DD48DF8AC0DFFFFF        , vpandnq(zmm25, k0, zmm4, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62E1C548DF627F              , vpandnq(zmm20, k0, zmm7, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(6261DD48DF09                , vpandnq(zmm25, k0, zmm4, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62E1C548DFA2C0DFFFFF        , vpandnq(zmm20, k0, zmm7, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62F1ED48DFCB                , vpandnq(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(62E1C548DF21                , vpandnq(zmm20, k0, zmm7, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62A1C548DFA4F023010000      , vpandnq(zmm20, k0, zmm7, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62A1C548DFE3                , vpandnq(zmm20, k0, zmm7, zmm19)),
        INSTRUCTION_TEST(62F1ED48DF4C2B02            , vpandnq(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F1ED28DF4C2B04            , vpandnq(ymm1, k0, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F1ED28DFCB                , vpandnq(ymm1, k0, ymm2, ymm3)),
        INSTRUCTION_TEST(62F1ED08DF4C2B08            , vpandnq(xmm1, k0, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F1ED08DFCB                , vpandnq(xmm1, k0, xmm2, xmm3)),
        INSTRUCTION_TEST(62E1C548DF6280              , vpandnq(zmm20, k0, zmm7, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62E1DD48DB8A00200000        , vpandq(zmm17, k0, zmm4, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62419D48DBD9                , vpandq(zmm27, k0, zmm12, zmm9)),
        INSTRUCTION_TEST(62219D48DB9CF034120000      , vpandq(zmm27, k0, zmm12, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62619D48DB5A7F              , vpandq(zmm27, k0, zmm12, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62619D48DB5A80              , vpandq(zmm27, k0, zmm12, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62619D48DB9AC0DFFFFF        , vpandq(zmm27, k0, zmm12, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62E1DD48DB4A7F              , vpandq(zmm17, k0, zmm4, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62619D48DB9A00200000        , vpandq(zmm27, k0, zmm12, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62619D48DB19                , vpandq(zmm27, k0, zmm12, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62E1DD48DB8AC0DFFFFF        , vpandq(zmm17, k0, zmm4, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(6281DD48DBCA                , vpandq(zmm17, k0, zmm4, zmm26)),
        INSTRUCTION_TEST(62E1DD48DB09                , vpandq(zmm17, k0, zmm4, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62A1DD48DB8CF023010000      , vpandq(zmm17, k0, zmm4, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62F1ED48DB4C2B02            , vpandq(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F1ED48DBCB                , vpandq(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(62F1ED28DB4C2B04            , vpandq(ymm1, k0, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F1ED28DBCB                , vpandq(ymm1, k0, ymm2, ymm3)),
        INSTRUCTION_TEST(62F1ED08DB4C2B08            , vpandq(xmm1, k0, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F1ED08DBCB                , vpandq(xmm1, k0, xmm2, xmm3)),
        INSTRUCTION_TEST(62E1DD48DB4A80              , vpandq(zmm17, k0, zmm4, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62F16D48E04C2B02            , vpavgb(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F16D48E0CB                , vpavgb(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(C5EDE08C2B80000000          , vpavgb(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C5EDE0CB                    , vpavgb(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(C5E9E0CB                    , vpavgb(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(C5E9E08C2B80000000          , vpavgb(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C5E9E3CB                    , vpavgw(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(C5E9E38C2B80000000          , vpavgw(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C5EDE3CB                    , vpavgw(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(C5EDE38C2B80000000          , vpavgw(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F16D48E3CB                , vpavgw(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(62F16D48E34C2B02            , vpavgw(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E36D02CB01                , vpblendd(ymm1, ymm2, ymm3, Imm(1))),
        INSTRUCTION_TEST(C4E36D028C2B8000000001      , vpblendd(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(C4E36902CB01                , vpblendd(xmm1, xmm2, xmm3, Imm(1))),
        INSTRUCTION_TEST(C4E369028C2B8000000001      , vpblendd(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(C4E3694CCB40                , vpblendvb(xmm1, xmm2, xmm3, xmm4)),
        INSTRUCTION_TEST(C4E3694C8C2B8000000060      , vpblendvb(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128), xmm6)),
        INSTRUCTION_TEST(C4E36D4CCB40                , vpblendvb(ymm1, ymm2, ymm3, ymm4)),
        INSTRUCTION_TEST(C4E36D4C8C2B8000000060      , vpblendvb(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128), ymm6)),
        INSTRUCTION_TEST(C4E3690ECB01                , vpblendw(xmm1, xmm2, xmm3, Imm(1))),
        INSTRUCTION_TEST(C4E3690E8C2B8000000001      , vpblendw(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(C4E36D0ECB01                , vpblendw(ymm1, ymm2, ymm3, Imm(1))),
        INSTRUCTION_TEST(C4E36D0E8C2B8000000001      , vpblendw(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F27D48788C1A80000000      , vpbroadcastb(zmm1, k0, byte_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F27D4878CA                , vpbroadcastb(zmm1, k0, xmm2)),
        INSTRUCTION_TEST(62F27D487ACA                , vpbroadcastb(zmm1, k0, edx)),
        INSTRUCTION_TEST(C4E27D78CA                  , vpbroadcastb(ymm1, xmm2)),
        INSTRUCTION_TEST(62F27D087ACA                , vpbroadcastb(xmm1, k0, edx)),
        INSTRUCTION_TEST(C4E27D788C1A80000000        , vpbroadcastb(ymm1, byte_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(C4E27978CA                  , vpbroadcastb(xmm1, xmm2)),
        INSTRUCTION_TEST(C4E279788C1A80000000        , vpbroadcastb(xmm1, byte_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F27D287ACA                , vpbroadcastb(ymm1, k0, edx)),
        INSTRUCTION_TEST(62E27D48585A7F              , vpbroadcastd(zmm19, k0, dword_ptr(rdx, 508))),
        INSTRUCTION_TEST(62E27D48589A00020000        , vpbroadcastd(zmm19, k0, dword_ptr(rdx, 512))),
        INSTRUCTION_TEST(62627D485811                , vpbroadcastd(zmm26, k0, dword_ptr(rcx))),
        INSTRUCTION_TEST(62627D48585280              , vpbroadcastd(zmm26, k0, dword_ptr(rdx, -512))),
        INSTRUCTION_TEST(62627D485892FCFDFFFF        , vpbroadcastd(zmm26, k0, dword_ptr(rdx, -516))),
        INSTRUCTION_TEST(62627D4858527F              , vpbroadcastd(zmm26, k0, dword_ptr(rdx, 508))),
        INSTRUCTION_TEST(62227D485894F023010000      , vpbroadcastd(zmm26, k0, dword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62F27D4858D9                , vpbroadcastd(zmm3, k0, xmm1)),
        INSTRUCTION_TEST(62727D487CC0                , vpbroadcastd(zmm8, k0, eax)),
        INSTRUCTION_TEST(62727D487CC5                , vpbroadcastd(zmm8, k0, ebp)),
        INSTRUCTION_TEST(62527D487CC5                , vpbroadcastd(zmm8, k0, r13d)),
        INSTRUCTION_TEST(62E27D48589AFCFDFFFF        , vpbroadcastd(zmm19, k0, dword_ptr(rdx, -516))),
        INSTRUCTION_TEST(62627D48589200020000        , vpbroadcastd(zmm26, k0, dword_ptr(rdx, 512))),
        INSTRUCTION_TEST(62E27D48585A80              , vpbroadcastd(zmm19, k0, dword_ptr(rdx, -512))),
        INSTRUCTION_TEST(62F27D48584C1A20            , vpbroadcastd(zmm1, k0, dword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62A27D48589CF034120000      , vpbroadcastd(zmm19, k0, dword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62E27D485819                , vpbroadcastd(zmm19, k0, dword_ptr(rcx))),
        INSTRUCTION_TEST(62F27D087CCA                , vpbroadcastd(xmm1, k0, edx)),
        INSTRUCTION_TEST(C4E27958CA                  , vpbroadcastd(xmm1, xmm2)),
        INSTRUCTION_TEST(C4E27D588C1A80000000        , vpbroadcastd(ymm1, dword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F27D287CCA                , vpbroadcastd(ymm1, k0, edx)),
        INSTRUCTION_TEST(C4E27D58CA                  , vpbroadcastd(ymm1, xmm2)),
        INSTRUCTION_TEST(C4E279588C1A80000000        , vpbroadcastd(xmm1, dword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F27D487CCA                , vpbroadcastd(zmm1, k0, edx)),
        INSTRUCTION_TEST(62F27D4858CA                , vpbroadcastd(zmm1, k0, xmm2)),
        INSTRUCTION_TEST(62327D4858D6                , vpbroadcastd(zmm10, k0, xmm22)),
        INSTRUCTION_TEST(62727D487CD8                , vpbroadcastd(zmm11, k0, eax)),
        INSTRUCTION_TEST(62727D487CDD                , vpbroadcastd(zmm11, k0, ebp)),
        INSTRUCTION_TEST(62527D487CDD                , vpbroadcastd(zmm11, k0, r13d)),
        INSTRUCTION_TEST(6272FD4859727F              , vpbroadcastq(zmm14, k0, qword_ptr(rdx, 1016))),
        INSTRUCTION_TEST(6272FD4859B200040000        , vpbroadcastq(zmm14, k0, qword_ptr(rdx, 1024))),
        INSTRUCTION_TEST(6222FD48598CF023010000      , vpbroadcastq(zmm25, k0, qword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(6262FD485909                , vpbroadcastq(zmm25, k0, qword_ptr(rcx))),
        INSTRUCTION_TEST(6272FD4859C4                , vpbroadcastq(zmm8, k0, xmm4)),
        INSTRUCTION_TEST(6262FD48598A00040000        , vpbroadcastq(zmm25, k0, qword_ptr(rdx, 1024))),
        INSTRUCTION_TEST(6262FD48594A7F              , vpbroadcastq(zmm25, k0, qword_ptr(rdx, 1016))),
        INSTRUCTION_TEST(6272FD4859B2F8FBFFFF        , vpbroadcastq(zmm14, k0, qword_ptr(rdx, -1032))),
        INSTRUCTION_TEST(62F2FD4859DD                , vpbroadcastq(zmm3, k0, xmm5)),
        INSTRUCTION_TEST(62D2FD487CE8                , vpbroadcastq(zmm5, k0, r8)),
        INSTRUCTION_TEST(62F2FD487CE8                , vpbroadcastq(zmm5, k0, rax)),
        INSTRUCTION_TEST(6262FD48598AF8FBFFFF        , vpbroadcastq(zmm25, k0, qword_ptr(rdx, -1032))),
        INSTRUCTION_TEST(6272FD48597280              , vpbroadcastq(zmm14, k0, qword_ptr(rdx, -1024))),
        INSTRUCTION_TEST(6262FD48594A80              , vpbroadcastq(zmm25, k0, qword_ptr(rdx, -1024))),
        INSTRUCTION_TEST(6232FD4859B4F034120000      , vpbroadcastq(zmm14, k0, qword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(6272FD485931                , vpbroadcastq(zmm14, k0, qword_ptr(rcx))),
        INSTRUCTION_TEST(62F2FD087CCA                , vpbroadcastq(xmm1, k0, rdx)),
        INSTRUCTION_TEST(C4E27959CA                  , vpbroadcastq(xmm1, xmm2)),
        INSTRUCTION_TEST(C4E27D598C1A80000000        , vpbroadcastq(ymm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F2FD287CCA                , vpbroadcastq(ymm1, k0, rdx)),
        INSTRUCTION_TEST(C4E279598C1A80000000        , vpbroadcastq(xmm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F2FD48594C1A10            , vpbroadcastq(zmm1, k0, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62D2FD487CC8                , vpbroadcastq(zmm1, k0, r8)),
        INSTRUCTION_TEST(62F2FD487CC8                , vpbroadcastq(zmm1, k0, rax)),
        INSTRUCTION_TEST(62F2FD487CCA                , vpbroadcastq(zmm1, k0, rdx)),
        INSTRUCTION_TEST(62F2FD4859CA                , vpbroadcastq(zmm1, k0, xmm2)),
        INSTRUCTION_TEST(C4E27D59CA                  , vpbroadcastq(ymm1, xmm2)),
        INSTRUCTION_TEST(62F27D4879CA                , vpbroadcastw(zmm1, k0, xmm2)),
        INSTRUCTION_TEST(62F27D48794C1A40            , vpbroadcastw(zmm1, k0, word_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F27D487BCA                , vpbroadcastw(zmm1, k0, edx)),
        INSTRUCTION_TEST(C4E27D79CA                  , vpbroadcastw(ymm1, xmm2)),
        INSTRUCTION_TEST(C4E27979CA                  , vpbroadcastw(xmm1, xmm2)),
        INSTRUCTION_TEST(62F27D287BCA                , vpbroadcastw(ymm1, k0, edx)),
        INSTRUCTION_TEST(C4E279798C1A80000000        , vpbroadcastw(xmm1, word_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F27D087BCA                , vpbroadcastw(xmm1, k0, edx)),
        INSTRUCTION_TEST(C4E27D798C1A80000000        , vpbroadcastw(ymm1, word_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F36D48444C2B0201          , vpclmulqdq(zmm1, zmm2, zmmword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F36D4844CB01              , vpclmulqdq(zmm1, zmm2, zmm3, Imm(1))),
        INSTRUCTION_TEST(C4E36D448C2B8000000001      , vpclmulqdq(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(C4E369448C2B8000000001      , vpclmulqdq(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(C4E36944CB01                , vpclmulqdq(xmm1, xmm2, xmm3, Imm(1))),
        INSTRUCTION_TEST(C4E36D44CB01                , vpclmulqdq(ymm1, ymm2, ymm3, Imm(1))),
        INSTRUCTION_TEST(8FE868A2CB40                , vpcmov(xmm1, xmm2, xmm3, xmm4)),
        INSTRUCTION_TEST(8FE8E8A28C358000000030      , vpcmov(xmm1, xmm2, xmm3, xmmword_ptr(rbp, rsi, 1, 128))),
        INSTRUCTION_TEST(8FE868A28C2B8000000060      , vpcmov(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128), xmm6)),
        INSTRUCTION_TEST(8FE86CA2CB40                , vpcmov(ymm1, ymm2, ymm3, ymm4)),
        INSTRUCTION_TEST(8FE8ECA28C358000000030      , vpcmov(ymm1, ymm2, ymm3, ymmword_ptr(rbp, rsi, 1, 128))),
        INSTRUCTION_TEST(8FE86CA28C2B8000000060      , vpcmov(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128), ymm6)),
        INSTRUCTION_TEST(62F36D283F4C2B0401          , vpcmpb(k1, k0, ymm2, ymmword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F36D483F4C2B0201          , vpcmpb(k1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F36D483FCB01              , vpcmpb(k1, k0, zmm2, zmm3, Imm(1))),
        INSTRUCTION_TEST(62F36D083FCB01              , vpcmpb(k1, k0, xmm2, xmm3, Imm(1))),
        INSTRUCTION_TEST(62F36D083F4C2B0801          , vpcmpb(k1, k0, xmm2, xmmword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F36D283FCB01              , vpcmpb(k1, k0, ymm2, ymm3, Imm(1))),
        INSTRUCTION_TEST(62D335401FEAAB              , vpcmpd(k5, k0, zmm25, zmm10, Imm(171))),
        INSTRUCTION_TEST(62F335401F92002000007B      , vpcmpd(k2, k0, zmm25, zmmword_ptr(rdx, 8192), Imm(123))),
        INSTRUCTION_TEST(62D335401FEA7B              , vpcmpd(k5, k0, zmm25, zmm10, Imm(123))),
        INSTRUCTION_TEST(62B335401FACF0230100007B    , vpcmpd(k5, k0, zmm25, zmmword_ptr(rax, r14, 8, 291), Imm(123))),
        INSTRUCTION_TEST(62F335401F527F7B            , vpcmpd(k2, k0, zmm25, zmmword_ptr(rdx, 8128), Imm(123))),
        INSTRUCTION_TEST(62F335401F6A807B            , vpcmpd(k5, k0, zmm25, zmmword_ptr(rdx, -8192), Imm(123))),
        INSTRUCTION_TEST(62F335401FAAC0DFFFFF7B      , vpcmpd(k5, k0, zmm25, zmmword_ptr(rdx, -8256), Imm(123))),
        INSTRUCTION_TEST(62F335401FAA002000007B      , vpcmpd(k5, k0, zmm25, zmmword_ptr(rdx, 8192), Imm(123))),
        INSTRUCTION_TEST(62F335401F6A7F7B            , vpcmpd(k5, k0, zmm25, zmmword_ptr(rdx, 8128), Imm(123))),
        INSTRUCTION_TEST(62F335401F297B              , vpcmpd(k5, k0, zmm25, zmmword_ptr(rcx), Imm(123))),
        INSTRUCTION_TEST(62F335401F92C0DFFFFF7B      , vpcmpd(k2, k0, zmm25, zmmword_ptr(rdx, -8256), Imm(123))),
        INSTRUCTION_TEST(62F36D081F4C2B0801          , vpcmpd(k1, k0, xmm2, xmmword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F335401F117B              , vpcmpd(k2, k0, zmm25, zmmword_ptr(rcx), Imm(123))),
        INSTRUCTION_TEST(62B335401F94F0341200007B    , vpcmpd(k2, k0, zmm25, zmmword_ptr(rax, r14, 8, 4660), Imm(123))),
        INSTRUCTION_TEST(62B335401FD6AB              , vpcmpd(k2, k0, zmm25, zmm22, Imm(171))),
        INSTRUCTION_TEST(62B335401FD67B              , vpcmpd(k2, k0, zmm25, zmm22, Imm(123))),
        INSTRUCTION_TEST(62F36D481F4C2B0201          , vpcmpd(k1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F36D481FCB01              , vpcmpd(k1, k0, zmm2, zmm3, Imm(1))),
        INSTRUCTION_TEST(62F36D281F4C2B0401          , vpcmpd(k1, k0, ymm2, ymmword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F36D281FCB01              , vpcmpd(k1, k0, ymm2, ymm3, Imm(1))),
        INSTRUCTION_TEST(62F36D081FCB01              , vpcmpd(k1, k0, xmm2, xmm3, Imm(1))),
        INSTRUCTION_TEST(62F335401F52807B            , vpcmpd(k2, k0, zmm25, zmmword_ptr(rdx, -8192), Imm(123))),
        INSTRUCTION_TEST(C5ED748C2B80000000          , vpcmpeqb(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C5ED74CB                    , vpcmpeqb(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(C5E9748C2B80000000          , vpcmpeqb(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C5E974CB                    , vpcmpeqb(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(62F16D48744C2B02            , vpcmpeqb(k1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F16D28744C2B04            , vpcmpeqb(k1, k0, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F16D2874CB                , vpcmpeqb(k1, k0, ymm2, ymm3)),
        INSTRUCTION_TEST(62F16D08744C2B08            , vpcmpeqb(k1, k0, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F16D0874CB                , vpcmpeqb(k1, k0, xmm2, xmm3)),
        INSTRUCTION_TEST(62F16D4874CB                , vpcmpeqb(k1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(62B13D4876ACF034120000      , vpcmpeqd(k5, k0, zmm8, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62F13D487629                , vpcmpeqd(k5, k0, zmm8, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62F13D48766A80              , vpcmpeqd(k5, k0, zmm8, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62F13D4876AAC0DFFFFF        , vpcmpeqd(k5, k0, zmm8, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62F13D48766A7F              , vpcmpeqd(k5, k0, zmm8, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(C5E976CB                    , vpcmpeqd(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(C5E9768C2B80000000          , vpcmpeqd(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C5ED76CB                    , vpcmpeqd(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(C5ED768C2B80000000          , vpcmpeqd(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62B13D4876ED                , vpcmpeqd(k5, k0, zmm8, zmm21)),
        INSTRUCTION_TEST(62F13D4876AA00200000        , vpcmpeqd(k5, k0, zmm8, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62F16D4876AA00200000        , vpcmpeqd(k5, k0, zmm2, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62B16D4876ACF023010000      , vpcmpeqd(k5, k0, zmm2, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62F16D4876AAC0DFFFFF        , vpcmpeqd(k5, k0, zmm2, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62F16D08764C2B08            , vpcmpeqd(k1, k0, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F16D2876CB                , vpcmpeqd(k1, k0, ymm2, ymm3)),
        INSTRUCTION_TEST(62F16D28764C2B04            , vpcmpeqd(k1, k0, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F16D4876CB                , vpcmpeqd(k1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(62F16D0876CB                , vpcmpeqd(k1, k0, xmm2, xmm3)),
        INSTRUCTION_TEST(62D16D4876EA                , vpcmpeqd(k5, k0, zmm2, zmm10)),
        INSTRUCTION_TEST(62F16D48766A7F              , vpcmpeqd(k5, k0, zmm2, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62F16D487629                , vpcmpeqd(k5, k0, zmm2, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62F16D48766A80              , vpcmpeqd(k5, k0, zmm2, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62F16D48764C2B02            , vpcmpeqd(k1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62D28D4829E1                , vpcmpeqq(k4, k0, zmm14, zmm9)),
        INSTRUCTION_TEST(C4E26D29CB                  , vpcmpeqq(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(C4E269298C2B80000000        , vpcmpeqq(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E26929CB                  , vpcmpeqq(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(62F28D4829A200200000        , vpcmpeqq(k4, k0, zmm14, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62F28D4829627F              , vpcmpeqq(k4, k0, zmm14, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62F28D4829A2C0DFFFFF        , vpcmpeqq(k4, k0, zmm14, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62F28D48296280              , vpcmpeqq(k4, k0, zmm14, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62F28D482921                , vpcmpeqq(k4, k0, zmm14, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62B28D4829A4F034120000      , vpcmpeqq(k4, k0, zmm14, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62F2CD40299A00200000        , vpcmpeqq(k3, k0, zmm22, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(C4E26D298C2B80000000        , vpcmpeqq(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F2CD40299AC0DFFFFF        , vpcmpeqq(k3, k0, zmm22, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62F2CD40295A7F              , vpcmpeqq(k3, k0, zmm22, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62F2ED08294C2B08            , vpcmpeqq(k1, k0, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F2ED2829CB                , vpcmpeqq(k1, k0, ymm2, ymm3)),
        INSTRUCTION_TEST(62F2ED28294C2B04            , vpcmpeqq(k1, k0, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F2ED4829CB                , vpcmpeqq(k1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(62F2ED0829CB                , vpcmpeqq(k1, k0, xmm2, xmm3)),
        INSTRUCTION_TEST(62F2CD4029DA                , vpcmpeqq(k3, k0, zmm22, zmm2)),
        INSTRUCTION_TEST(62B2CD40299CF023010000      , vpcmpeqq(k3, k0, zmm22, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62F2CD402919                , vpcmpeqq(k3, k0, zmm22, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62F2CD40295A80              , vpcmpeqq(k3, k0, zmm22, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62F2ED48294C2B02            , vpcmpeqq(k1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C5ED75CB                    , vpcmpeqw(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(62F16D0875CB                , vpcmpeqw(k1, k0, xmm2, xmm3)),
        INSTRUCTION_TEST(62F16D08754C2B08            , vpcmpeqw(k1, k0, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F16D2875CB                , vpcmpeqw(k1, k0, ymm2, ymm3)),
        INSTRUCTION_TEST(62F16D28754C2B04            , vpcmpeqw(k1, k0, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F16D4875CB                , vpcmpeqw(k1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(62F16D48754C2B02            , vpcmpeqw(k1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C5E975CB                    , vpcmpeqw(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(C5E9758C2B80000000          , vpcmpeqw(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C5ED758C2B80000000          , vpcmpeqw(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E379618C1A8000000001      , vpcmpestri(xmm1, xmmword_ptr(rdx, rbx, 1, 128), 1)),
        INSTRUCTION_TEST(C4E37961CA01                , vpcmpestri(xmm1, xmm2, 1)),
        INSTRUCTION_TEST(C4E37960CA01                , vpcmpestrm(xmm1, xmm2, 1)),
        INSTRUCTION_TEST(C4E379608C1A8000000001      , vpcmpestrm(xmm1, xmmword_ptr(rdx, rbx, 1, 128), 1)),
        INSTRUCTION_TEST(C5ED64CB                    , vpcmpgtb(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(C5ED648C2B80000000          , vpcmpgtb(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C5E9648C2B80000000          , vpcmpgtb(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C5E964CB                    , vpcmpgtb(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(62F16D48644C2B02            , vpcmpgtb(k1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F16D28644C2B04            , vpcmpgtb(k1, k0, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F16D2864CB                , vpcmpgtb(k1, k0, ymm2, ymm3)),
        INSTRUCTION_TEST(62F16D08644C2B08            , vpcmpgtb(k1, k0, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F16D0864CB                , vpcmpgtb(k1, k0, xmm2, xmm3)),
        INSTRUCTION_TEST(62F16D4864CB                , vpcmpgtb(k1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(62F15540666A80              , vpcmpgtd(k5, k0, zmm21, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62B1554066ACF023010000      , vpcmpgtd(k5, k0, zmm21, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62F155406629                , vpcmpgtd(k5, k0, zmm21, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62F1554066AAC0DFFFFF        , vpcmpgtd(k5, k0, zmm21, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62D1554066E8                , vpcmpgtd(k5, k0, zmm21, zmm8)),
        INSTRUCTION_TEST(62F1554066AA00200000        , vpcmpgtd(k5, k0, zmm21, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(C5E9668C2B80000000          , vpcmpgtd(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C5ED66CB                    , vpcmpgtd(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(C5ED668C2B80000000          , vpcmpgtd(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F15540666A7F              , vpcmpgtd(k5, k0, zmm21, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62F13D4866A200200000        , vpcmpgtd(k4, k0, zmm8, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(C5E966CB                    , vpcmpgtd(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(62F13D4866A2C0DFFFFF        , vpcmpgtd(k4, k0, zmm8, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62F16D08664C2B08            , vpcmpgtd(k1, k0, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F16D2866CB                , vpcmpgtd(k1, k0, ymm2, ymm3)),
        INSTRUCTION_TEST(62F13D4866627F              , vpcmpgtd(k4, k0, zmm8, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62F16D28664C2B04            , vpcmpgtd(k1, k0, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F16D4866CB                , vpcmpgtd(k1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(62F16D0866CB                , vpcmpgtd(k1, k0, xmm2, xmm3)),
        INSTRUCTION_TEST(62D13D4866E0                , vpcmpgtd(k4, k0, zmm8, zmm8)),
        INSTRUCTION_TEST(62B13D4866A4F034120000      , vpcmpgtd(k4, k0, zmm8, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62F13D486621                , vpcmpgtd(k4, k0, zmm8, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62F13D48666280              , vpcmpgtd(k4, k0, zmm8, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62F16D48664C2B02            , vpcmpgtd(k1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62B29D48379CF034120000      , vpcmpgtq(k3, k0, zmm12, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62F29D483719                , vpcmpgtq(k3, k0, zmm12, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62F29D48375A80              , vpcmpgtq(k3, k0, zmm12, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62F29D48379AC0DFFFFF        , vpcmpgtq(k3, k0, zmm12, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62F29D48375A7F              , vpcmpgtq(k3, k0, zmm12, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(C4E26937CB                  , vpcmpgtq(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(C4E269378C2B80000000        , vpcmpgtq(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E26D37CB                  , vpcmpgtq(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(C4E26D378C2B80000000        , vpcmpgtq(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62929D4837DA                , vpcmpgtq(k3, k0, zmm12, zmm26)),
        INSTRUCTION_TEST(62F29D48379A00200000        , vpcmpgtq(k3, k0, zmm12, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62F2DD40379200200000        , vpcmpgtq(k2, k0, zmm20, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62B2DD403794F023010000      , vpcmpgtq(k2, k0, zmm20, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62F2DD403792C0DFFFFF        , vpcmpgtq(k2, k0, zmm20, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62F2DD40375280              , vpcmpgtq(k2, k0, zmm20, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62F2DD403711                , vpcmpgtq(k2, k0, zmm20, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62B2DD4037D1                , vpcmpgtq(k2, k0, zmm20, zmm17)),
        INSTRUCTION_TEST(62F2ED48374C2B02            , vpcmpgtq(k1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F2ED4837CB                , vpcmpgtq(k1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(62F2ED28374C2B04            , vpcmpgtq(k1, k0, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F2ED2837CB                , vpcmpgtq(k1, k0, ymm2, ymm3)),
        INSTRUCTION_TEST(62F2ED08374C2B08            , vpcmpgtq(k1, k0, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F2ED0837CB                , vpcmpgtq(k1, k0, xmm2, xmm3)),
        INSTRUCTION_TEST(62F2DD4037527F              , vpcmpgtq(k2, k0, zmm20, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(C5E965CB                    , vpcmpgtw(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(C5ED658C2B80000000          , vpcmpgtw(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C5E9658C2B80000000          , vpcmpgtw(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F16D48654C2B02            , vpcmpgtw(k1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C5ED65CB                    , vpcmpgtw(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(62F16D28654C2B04            , vpcmpgtw(k1, k0, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F16D2865CB                , vpcmpgtw(k1, k0, ymm2, ymm3)),
        INSTRUCTION_TEST(62F16D08654C2B08            , vpcmpgtw(k1, k0, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F16D0865CB                , vpcmpgtw(k1, k0, xmm2, xmm3)),
        INSTRUCTION_TEST(62F16D4865CB                , vpcmpgtw(k1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(C4E37963CA01                , vpcmpistri(xmm1, xmm2, 1)),
        INSTRUCTION_TEST(C4E379638C1A8000000001      , vpcmpistri(xmm1, xmmword_ptr(rdx, rbx, 1, 128), 1)),
        INSTRUCTION_TEST(C4E37962CA01                , vpcmpistrm(xmm1, xmm2, 1)),
        INSTRUCTION_TEST(C4E379628C1A8000000001      , vpcmpistrm(xmm1, xmmword_ptr(rdx, rbx, 1, 128), 1)),
        INSTRUCTION_TEST(62F3B5401F9A002000007B      , vpcmpq(k3, k0, zmm25, zmmword_ptr(rdx, 8192), Imm(123))),
        INSTRUCTION_TEST(62939D401FECAB              , vpcmpq(k5, k0, zmm28, zmm28, Imm(171))),
        INSTRUCTION_TEST(62B39D401FACF0230100007B    , vpcmpq(k5, k0, zmm28, zmmword_ptr(rax, r14, 8, 291), Imm(123))),
        INSTRUCTION_TEST(62F39D401F6A7F7B            , vpcmpq(k5, k0, zmm28, zmmword_ptr(rdx, 8128), Imm(123))),
        INSTRUCTION_TEST(62F39D401F6A807B            , vpcmpq(k5, k0, zmm28, zmmword_ptr(rdx, -8192), Imm(123))),
        INSTRUCTION_TEST(62F39D401FAAC0DFFFFF7B      , vpcmpq(k5, k0, zmm28, zmmword_ptr(rdx, -8256), Imm(123))),
        INSTRUCTION_TEST(62F3B5401F5A7F7B            , vpcmpq(k3, k0, zmm25, zmmword_ptr(rdx, 8128), Imm(123))),
        INSTRUCTION_TEST(62F39D401FAA002000007B      , vpcmpq(k5, k0, zmm28, zmmword_ptr(rdx, 8192), Imm(123))),
        INSTRUCTION_TEST(62F39D401F297B              , vpcmpq(k5, k0, zmm28, zmmword_ptr(rcx), Imm(123))),
        INSTRUCTION_TEST(62F3B5401F9AC0DFFFFF7B      , vpcmpq(k3, k0, zmm25, zmmword_ptr(rdx, -8256), Imm(123))),
        INSTRUCTION_TEST(62939D401FEC7B              , vpcmpq(k5, k0, zmm28, zmm28, Imm(123))),
        INSTRUCTION_TEST(62F3B5401F197B              , vpcmpq(k3, k0, zmm25, zmmword_ptr(rcx), Imm(123))),
        INSTRUCTION_TEST(62B3B5401F9CF0341200007B    , vpcmpq(k3, k0, zmm25, zmmword_ptr(rax, r14, 8, 4660), Imm(123))),
        INSTRUCTION_TEST(62D3B5401FDFAB              , vpcmpq(k3, k0, zmm25, zmm15, Imm(171))),
        INSTRUCTION_TEST(62D3B5401FDF7B              , vpcmpq(k3, k0, zmm25, zmm15, Imm(123))),
        INSTRUCTION_TEST(62F3ED481F4C2B0201          , vpcmpq(k1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F3ED481FCB01              , vpcmpq(k1, k0, zmm2, zmm3, Imm(1))),
        INSTRUCTION_TEST(62F3ED281F4C2B0401          , vpcmpq(k1, k0, ymm2, ymmword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F3ED281FCB01              , vpcmpq(k1, k0, ymm2, ymm3, Imm(1))),
        INSTRUCTION_TEST(62F3ED081F4C2B0801          , vpcmpq(k1, k0, xmm2, xmmword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F3ED081FCB01              , vpcmpq(k1, k0, xmm2, xmm3, Imm(1))),
        INSTRUCTION_TEST(62F3B5401F5A807B            , vpcmpq(k3, k0, zmm25, zmmword_ptr(rdx, -8192), Imm(123))),
        INSTRUCTION_TEST(62F36D483E4C2B0201          , vpcmpub(k1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F36D483ECB01              , vpcmpub(k1, k0, zmm2, zmm3, Imm(1))),
        INSTRUCTION_TEST(62F36D283E4C2B0401          , vpcmpub(k1, k0, ymm2, ymmword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F36D283ECB01              , vpcmpub(k1, k0, ymm2, ymm3, Imm(1))),
        INSTRUCTION_TEST(62F36D083ECB01              , vpcmpub(k1, k0, xmm2, xmm3, Imm(1))),
        INSTRUCTION_TEST(62F36D083E4C2B0801          , vpcmpub(k1, k0, xmm2, xmmword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F335401E527F7B            , vpcmpud(k2, k0, zmm25, zmmword_ptr(rdx, 8128), Imm(123))),
        INSTRUCTION_TEST(62F315401E9A002000007B      , vpcmpud(k3, k0, zmm29, zmmword_ptr(rdx, 8192), Imm(123))),
        INSTRUCTION_TEST(62F315401E5A7F7B            , vpcmpud(k3, k0, zmm29, zmmword_ptr(rdx, 8128), Imm(123))),
        INSTRUCTION_TEST(62F315401E9AC0DFFFFF7B      , vpcmpud(k3, k0, zmm29, zmmword_ptr(rdx, -8256), Imm(123))),
        INSTRUCTION_TEST(62F315401E5A807B            , vpcmpud(k3, k0, zmm29, zmmword_ptr(rdx, -8192), Imm(123))),
        INSTRUCTION_TEST(62F315401E197B              , vpcmpud(k3, k0, zmm29, zmmword_ptr(rcx), Imm(123))),
        INSTRUCTION_TEST(62B315401E9CF0341200007B    , vpcmpud(k3, k0, zmm29, zmmword_ptr(rax, r14, 8, 4660), Imm(123))),
        INSTRUCTION_TEST(62F315401EDDAB              , vpcmpud(k3, k0, zmm29, zmm5, Imm(171))),
        INSTRUCTION_TEST(62F315401EDD7B              , vpcmpud(k3, k0, zmm29, zmm5, Imm(123))),
        INSTRUCTION_TEST(62F335401E92002000007B      , vpcmpud(k2, k0, zmm25, zmmword_ptr(rdx, 8192), Imm(123))),
        INSTRUCTION_TEST(62F335401E92C0DFFFFF7B      , vpcmpud(k2, k0, zmm25, zmmword_ptr(rdx, -8256), Imm(123))),
        INSTRUCTION_TEST(62F36D481E4C2B0201          , vpcmpud(k1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F335401E117B              , vpcmpud(k2, k0, zmm25, zmmword_ptr(rcx), Imm(123))),
        INSTRUCTION_TEST(62B335401E94F0230100007B    , vpcmpud(k2, k0, zmm25, zmmword_ptr(rax, r14, 8, 291), Imm(123))),
        INSTRUCTION_TEST(62F335401ED7AB              , vpcmpud(k2, k0, zmm25, zmm7, Imm(171))),
        INSTRUCTION_TEST(62F335401ED77B              , vpcmpud(k2, k0, zmm25, zmm7, Imm(123))),
        INSTRUCTION_TEST(62F335401E52807B            , vpcmpud(k2, k0, zmm25, zmmword_ptr(rdx, -8192), Imm(123))),
        INSTRUCTION_TEST(62F36D481ECB01              , vpcmpud(k1, k0, zmm2, zmm3, Imm(1))),
        INSTRUCTION_TEST(62F36D281E4C2B0401          , vpcmpud(k1, k0, ymm2, ymmword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F36D281ECB01              , vpcmpud(k1, k0, ymm2, ymm3, Imm(1))),
        INSTRUCTION_TEST(62F36D081E4C2B0801          , vpcmpud(k1, k0, xmm2, xmmword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F36D081ECB01              , vpcmpud(k1, k0, xmm2, xmm3, Imm(1))),
        INSTRUCTION_TEST(62F38D481E92002000007B      , vpcmpuq(k2, k0, zmm14, zmmword_ptr(rdx, 8192), Imm(123))),
        INSTRUCTION_TEST(62D38D481ED87B              , vpcmpuq(k3, k0, zmm14, zmm8, Imm(123))),
        INSTRUCTION_TEST(62D38D481ED8AB              , vpcmpuq(k3, k0, zmm14, zmm8, Imm(171))),
        INSTRUCTION_TEST(62B38D481E9CF0230100007B    , vpcmpuq(k3, k0, zmm14, zmmword_ptr(rax, r14, 8, 291), Imm(123))),
        INSTRUCTION_TEST(62F38D481E9A002000007B      , vpcmpuq(k3, k0, zmm14, zmmword_ptr(rdx, 8192), Imm(123))),
        INSTRUCTION_TEST(62F38D481E5A807B            , vpcmpuq(k3, k0, zmm14, zmmword_ptr(rdx, -8192), Imm(123))),
        INSTRUCTION_TEST(62F38D481E9AC0DFFFFF7B      , vpcmpuq(k3, k0, zmm14, zmmword_ptr(rdx, -8256), Imm(123))),
        INSTRUCTION_TEST(62F38D481E5A7F7B            , vpcmpuq(k3, k0, zmm14, zmmword_ptr(rdx, 8128), Imm(123))),
        INSTRUCTION_TEST(62F38D481E527F7B            , vpcmpuq(k2, k0, zmm14, zmmword_ptr(rdx, 8128), Imm(123))),
        INSTRUCTION_TEST(62F38D481E197B              , vpcmpuq(k3, k0, zmm14, zmmword_ptr(rcx), Imm(123))),
        INSTRUCTION_TEST(62F38D481E92C0DFFFFF7B      , vpcmpuq(k2, k0, zmm14, zmmword_ptr(rdx, -8256), Imm(123))),
        INSTRUCTION_TEST(62F38D481E117B              , vpcmpuq(k2, k0, zmm14, zmmword_ptr(rcx), Imm(123))),
        INSTRUCTION_TEST(62F3ED081E4C2B0801          , vpcmpuq(k1, k0, xmm2, xmmword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62B38D481E94F0341200007B    , vpcmpuq(k2, k0, zmm14, zmmword_ptr(rax, r14, 8, 4660), Imm(123))),
        INSTRUCTION_TEST(62D38D481ED4AB              , vpcmpuq(k2, k0, zmm14, zmm12, Imm(171))),
        INSTRUCTION_TEST(62D38D481ED47B              , vpcmpuq(k2, k0, zmm14, zmm12, Imm(123))),
        INSTRUCTION_TEST(62F3ED481E4C2B0201          , vpcmpuq(k1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F3ED481ECB01              , vpcmpuq(k1, k0, zmm2, zmm3, Imm(1))),
        INSTRUCTION_TEST(62F3ED281E4C2B0401          , vpcmpuq(k1, k0, ymm2, ymmword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F3ED281ECB01              , vpcmpuq(k1, k0, ymm2, ymm3, Imm(1))),
        INSTRUCTION_TEST(62F38D481E52807B            , vpcmpuq(k2, k0, zmm14, zmmword_ptr(rdx, -8192), Imm(123))),
        INSTRUCTION_TEST(62F3ED081ECB01              , vpcmpuq(k1, k0, xmm2, xmm3, Imm(1))),
        INSTRUCTION_TEST(62F3ED083ECB01              , vpcmpuw(k1, k0, xmm2, xmm3, Imm(1))),
        INSTRUCTION_TEST(62F3ED083E4C2B0801          , vpcmpuw(k1, k0, xmm2, xmmword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F3ED283ECB01              , vpcmpuw(k1, k0, ymm2, ymm3, Imm(1))),
        INSTRUCTION_TEST(62F3ED283E4C2B0401          , vpcmpuw(k1, k0, ymm2, ymmword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F3ED483ECB01              , vpcmpuw(k1, k0, zmm2, zmm3, Imm(1))),
        INSTRUCTION_TEST(62F3ED483E4C2B0201          , vpcmpuw(k1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F3ED483F4C2B0201          , vpcmpw(k1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F3ED483FCB01              , vpcmpw(k1, k0, zmm2, zmm3, Imm(1))),
        INSTRUCTION_TEST(62F3ED283F4C2B0401          , vpcmpw(k1, k0, ymm2, ymmword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F3ED083F4C2B0801          , vpcmpw(k1, k0, xmm2, xmmword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F3ED083FCB01              , vpcmpw(k1, k0, xmm2, xmm3, Imm(1))),
        INSTRUCTION_TEST(62F3ED283FCB01              , vpcmpw(k1, k0, ymm2, ymm3, Imm(1))),
        INSTRUCTION_TEST(8FE868CCCB01                , vpcomb(xmm1, xmm2, xmm3, Imm(1))),
        INSTRUCTION_TEST(8FE868CC8C2B8000000001      , vpcomb(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(8FE868CECB01                , vpcomd(xmm1, xmm2, xmm3, Imm(1))),
        INSTRUCTION_TEST(8FE868CE8C2B8000000001      , vpcomd(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F27D4863D1                , vpcompressb(zmm1, k0, zmm2)),
        INSTRUCTION_TEST(62F27D48639C1180000000      , vpcompressb(zmmword_ptr(rcx, rdx, 1, 128), k0, zmm3)),
        INSTRUCTION_TEST(62F27D28639C1180000000      , vpcompressb(ymmword_ptr(rcx, rdx, 1, 128), k0, ymm3)),
        INSTRUCTION_TEST(62F27D08639C1180000000      , vpcompressb(xmmword_ptr(rcx, rdx, 1, 128), k0, xmm3)),
        INSTRUCTION_TEST(62F27D0863D1                , vpcompressb(xmm1, k0, xmm2)),
        INSTRUCTION_TEST(62F27D2863D1                , vpcompressb(ymm1, k0, ymm2)),
        INSTRUCTION_TEST(62E27D488BA2FCFDFFFF        , vpcompressd(zmmword_ptr(rdx, -516), k0, zmm20)),
        INSTRUCTION_TEST(62E27D488B6280              , vpcompressd(zmmword_ptr(rdx, -512), k0, zmm20)),
        INSTRUCTION_TEST(62E27D488B7A80              , vpcompressd(zmmword_ptr(rdx, -512), k0, zmm23)),
        INSTRUCTION_TEST(62E27D488BBAFCFDFFFF        , vpcompressd(zmmword_ptr(rdx, -516), k0, zmm23)),
        INSTRUCTION_TEST(62A27D488BA4F034120000      , vpcompressd(zmmword_ptr(rax, r14, 8, 4660), k0, zmm20)),
        INSTRUCTION_TEST(62E27D488B7A7F              , vpcompressd(zmmword_ptr(rdx, 508), k0, zmm23)),
        INSTRUCTION_TEST(62E27D488BA200020000        , vpcompressd(zmmword_ptr(rdx, 512), k0, zmm20)),
        INSTRUCTION_TEST(62E27D488BBA00020000        , vpcompressd(zmmword_ptr(rdx, 512), k0, zmm23)),
        INSTRUCTION_TEST(62E27D488B627F              , vpcompressd(zmmword_ptr(rdx, 508), k0, zmm20)),
        INSTRUCTION_TEST(62E27D488B39                , vpcompressd(zmmword_ptr(rcx), k0, zmm23)),
        INSTRUCTION_TEST(62A27D488BD0                , vpcompressd(zmm16, k0, zmm18)),
        INSTRUCTION_TEST(62F27D488B5C1120            , vpcompressd(zmmword_ptr(rcx, rdx, 1, 128), k0, zmm3)),
        INSTRUCTION_TEST(62A27D488BBCF023010000      , vpcompressd(zmmword_ptr(rax, r14, 8, 291), k0, zmm23)),
        INSTRUCTION_TEST(62127D488BE0                , vpcompressd(zmm24, k0, zmm12)),
        INSTRUCTION_TEST(62F27D488BD1                , vpcompressd(zmm1, k0, zmm2)),
        INSTRUCTION_TEST(62F27D288B5C1120            , vpcompressd(ymmword_ptr(rcx, rdx, 1, 128), k0, ymm3)),
        INSTRUCTION_TEST(62F27D288BD1                , vpcompressd(ymm1, k0, ymm2)),
        INSTRUCTION_TEST(62F27D088B5C1120            , vpcompressd(xmmword_ptr(rcx, rdx, 1, 128), k0, xmm3)),
        INSTRUCTION_TEST(62F27D088BD1                , vpcompressd(xmm1, k0, xmm2)),
        INSTRUCTION_TEST(62E27D488B21                , vpcompressd(zmmword_ptr(rcx), k0, zmm20)),
        INSTRUCTION_TEST(6272FD488B9AF8FBFFFF        , vpcompressq(zmmword_ptr(rdx, -1032), k0, zmm11)),
        INSTRUCTION_TEST(6272FD488B5A80              , vpcompressq(zmmword_ptr(rdx, -1024), k0, zmm11)),
        INSTRUCTION_TEST(62E2FD488B5A80              , vpcompressq(zmmword_ptr(rdx, -1024), k0, zmm19)),
        INSTRUCTION_TEST(62E2FD488B9AF8FBFFFF        , vpcompressq(zmmword_ptr(rdx, -1032), k0, zmm19)),
        INSTRUCTION_TEST(62E2FD488B19                , vpcompressq(zmmword_ptr(rcx), k0, zmm19)),
        INSTRUCTION_TEST(62E2FD488B5A7F              , vpcompressq(zmmword_ptr(rdx, 1016), k0, zmm19)),
        INSTRUCTION_TEST(6272FD488B9A00040000        , vpcompressq(zmmword_ptr(rdx, 1024), k0, zmm11)),
        INSTRUCTION_TEST(62E2FD488B9A00040000        , vpcompressq(zmmword_ptr(rdx, 1024), k0, zmm19)),
        INSTRUCTION_TEST(6272FD488B5A7F              , vpcompressq(zmmword_ptr(rdx, 1016), k0, zmm11)),
        INSTRUCTION_TEST(6272FD488B19                , vpcompressq(zmmword_ptr(rcx), k0, zmm11)),
        INSTRUCTION_TEST(62F2FD088BD1                , vpcompressq(xmm1, k0, xmm2)),
        INSTRUCTION_TEST(62A2FD488B9CF034120000      , vpcompressq(zmmword_ptr(rax, r14, 8, 4660), k0, zmm19)),
        INSTRUCTION_TEST(6232FD488B9CF023010000      , vpcompressq(zmmword_ptr(rax, r14, 8, 291), k0, zmm11)),
        INSTRUCTION_TEST(62A2FD488BD9                , vpcompressq(zmm17, k0, zmm19)),
        INSTRUCTION_TEST(6252FD488BC4                , vpcompressq(zmm12, k0, zmm8)),
        INSTRUCTION_TEST(62F2FD488BD1                , vpcompressq(zmm1, k0, zmm2)),
        INSTRUCTION_TEST(62F2FD288B5C1110            , vpcompressq(ymmword_ptr(rcx, rdx, 1, 128), k0, ymm3)),
        INSTRUCTION_TEST(62F2FD288BD1                , vpcompressq(ymm1, k0, ymm2)),
        INSTRUCTION_TEST(62F2FD088B5C1110            , vpcompressq(xmmword_ptr(rcx, rdx, 1, 128), k0, xmm3)),
        INSTRUCTION_TEST(62F2FD488B5C1110            , vpcompressq(zmmword_ptr(rcx, rdx, 1, 128), k0, zmm3)),
        INSTRUCTION_TEST(62F2FD4863D1                , vpcompressw(zmm1, k0, zmm2)),
        INSTRUCTION_TEST(62F2FD28635C1140            , vpcompressw(ymmword_ptr(rcx, rdx, 1, 128), k0, ymm3)),
        INSTRUCTION_TEST(62F2FD48635C1140            , vpcompressw(zmmword_ptr(rcx, rdx, 1, 128), k0, zmm3)),
        INSTRUCTION_TEST(62F2FD08635C1140            , vpcompressw(xmmword_ptr(rcx, rdx, 1, 128), k0, xmm3)),
        INSTRUCTION_TEST(62F2FD0863D1                , vpcompressw(xmm1, k0, xmm2)),
        INSTRUCTION_TEST(62F2FD2863D1                , vpcompressw(ymm1, k0, ymm2)),
        INSTRUCTION_TEST(8FE868CFCB01                , vpcomq(xmm1, xmm2, xmm3, Imm(1))),
        INSTRUCTION_TEST(8FE868CF8C2B8000000001      , vpcomq(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(8FE868ECCB01                , vpcomub(xmm1, xmm2, xmm3, Imm(1))),
        INSTRUCTION_TEST(8FE868EC8C2B8000000001      , vpcomub(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(8FE868EE8C2B8000000001      , vpcomud(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(8FE868EECB01                , vpcomud(xmm1, xmm2, xmm3, Imm(1))),
        INSTRUCTION_TEST(8FE868EFCB01                , vpcomuq(xmm1, xmm2, xmm3, Imm(1))),
        INSTRUCTION_TEST(8FE868EF8C2B8000000001      , vpcomuq(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(8FE868EDCB01                , vpcomuw(xmm1, xmm2, xmm3, Imm(1))),
        INSTRUCTION_TEST(8FE868ED8C2B8000000001      , vpcomuw(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(8FE868CDCB01                , vpcomw(xmm1, xmm2, xmm3, Imm(1))),
        INSTRUCTION_TEST(8FE868CD8C2B8000000001      , vpcomw(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F27D48C44C1A02            , vpconflictd(zmm1, k0, zmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F27D28C44C1A04            , vpconflictd(ymm1, k0, ymmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F27D48C4CA                , vpconflictd(zmm1, k0, zmm2)),
        INSTRUCTION_TEST(62F27D08C44C1A08            , vpconflictd(xmm1, k0, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F27D08C4CA                , vpconflictd(xmm1, k0, xmm2)),
        INSTRUCTION_TEST(62F27D28C4CA                , vpconflictd(ymm1, k0, ymm2)),
        INSTRUCTION_TEST(62F2FD08C4CA                , vpconflictq(xmm1, k0, xmm2)),
        INSTRUCTION_TEST(62F2FD08C44C1A08            , vpconflictq(xmm1, k0, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F2FD28C4CA                , vpconflictq(ymm1, k0, ymm2)),
        INSTRUCTION_TEST(62F2FD28C44C1A04            , vpconflictq(ymm1, k0, ymmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F2FD48C4CA                , vpconflictq(zmm1, k0, zmm2)),
        INSTRUCTION_TEST(62F2FD48C44C1A02            , vpconflictq(zmm1, k0, zmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F26D48504C2B02            , vpdpbusd(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F26D4850CB                , vpdpbusd(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(62F2552850F4                , vpdpbusd(ymm6, k0, ymm5, ymm4)),
        INSTRUCTION_TEST(62F26D0850CB                , vpdpbusd(xmm1, k0, xmm2, xmm3)),
        INSTRUCTION_TEST(62F26D2850CB                , vpdpbusd(ymm1, k0, ymm2, ymm3)),
        INSTRUCTION_TEST(62F26D08504C2B08            , vpdpbusd(xmm1, k0, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F26D28504C2B04            , vpdpbusd(ymm1, k0, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F26D4851CB                , vpdpbusds(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(62F26D48514C2B02            , vpdpbusds(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F26D28514C2B04            , vpdpbusds(ymm1, k0, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F26D0851CB                , vpdpbusds(xmm1, k0, xmm2, xmm3)),
        INSTRUCTION_TEST(62F26D08514C2B08            , vpdpbusds(xmm1, k0, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F26D2851CB                , vpdpbusds(ymm1, k0, ymm2, ymm3)),
        INSTRUCTION_TEST(62F26D0852CB                , vpdpwssd(xmm1, k0, xmm2, xmm3)),
        INSTRUCTION_TEST(62F26D08524C2B08            , vpdpwssd(xmm1, k0, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F26D2852CB                , vpdpwssd(ymm1, k0, ymm2, ymm3)),
        INSTRUCTION_TEST(62F26D28524C2B04            , vpdpwssd(ymm1, k0, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F26D4852CB                , vpdpwssd(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(62F26D48524C2B02            , vpdpwssd(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F26D4853CB                , vpdpwssds(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(62F26D0853CB                , vpdpwssds(xmm1, k0, xmm2, xmm3)),
        INSTRUCTION_TEST(62F26D08534C2B08            , vpdpwssds(xmm1, k0, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F26D2853CB                , vpdpwssds(ymm1, k0, ymm2, ymm3)),
        INSTRUCTION_TEST(62F26D28534C2B04            , vpdpwssds(ymm1, k0, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F26D48534C2B02            , vpdpwssds(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E36D068C2B8000000001      , vperm2f128(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(C4E36D06CB01                , vperm2f128(ymm1, ymm2, ymm3, Imm(1))),
        INSTRUCTION_TEST(C4E36D46CB01                , vperm2i128(ymm1, ymm2, ymm3, Imm(1))),
        INSTRUCTION_TEST(C4E36D468C2B8000000001      , vperm2i128(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F26D488D4C2B02            , vpermb(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F26D088DCB                , vpermb(xmm1, k0, xmm2, xmm3)),
        INSTRUCTION_TEST(62F26D088D4C2B08            , vpermb(xmm1, k0, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F26D288DCB                , vpermb(ymm1, k0, ymm2, ymm3)),
        INSTRUCTION_TEST(62F26D288D4C2B04            , vpermb(ymm1, k0, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F26D488DCB                , vpermb(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(62422D4836C8                , vpermd(zmm25, k0, zmm10, zmm8)),
        INSTRUCTION_TEST(62222D48368CF034120000      , vpermd(zmm25, k0, zmm10, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62A21D4036B4F023010000      , vpermd(zmm22, k0, zmm28, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62622D48364A80              , vpermd(zmm25, k0, zmm10, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62622D48368AC0DFFFFF        , vpermd(zmm25, k0, zmm10, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62E21D4036B200200000        , vpermd(zmm22, k0, zmm28, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62622D483609                , vpermd(zmm25, k0, zmm10, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62E21D4036727F              , vpermd(zmm22, k0, zmm28, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62C21D4036F1                , vpermd(zmm22, k0, zmm28, zmm9)),
        INSTRUCTION_TEST(62E21D40367280              , vpermd(zmm22, k0, zmm28, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62E21D403631                , vpermd(zmm22, k0, zmm28, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62622D48364A7F              , vpermd(zmm25, k0, zmm10, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62F26D48364C2B02            , vpermd(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F26D4836CB                , vpermd(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(C4E26D368C2B80000000        , vpermd(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E26D36CB                  , vpermd(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(62E21D4036B2C0DFFFFF        , vpermd(zmm22, k0, zmm28, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62622D48368A00200000        , vpermd(zmm25, k0, zmm10, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62F26D28754C2B04            , vpermi2b(ymm1, k0, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F26D48754C2B02            , vpermi2b(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F26D4875CB                , vpermi2b(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(62F26D0875CB                , vpermi2b(xmm1, k0, xmm2, xmm3)),
        INSTRUCTION_TEST(62F26D2875CB                , vpermi2b(ymm1, k0, ymm2, ymm3)),
        INSTRUCTION_TEST(62F26D08754C2B08            , vpermi2b(xmm1, k0, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62726548768A00200000        , vpermi2d(zmm9, k0, zmm3, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62726548764A7F              , vpermi2d(zmm9, k0, zmm3, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62726548768AC0DFFFFF        , vpermi2d(zmm9, k0, zmm3, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62726548764A80              , vpermi2d(zmm9, k0, zmm3, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(627265487609                , vpermi2d(zmm9, k0, zmm3, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62326548768CF034120000      , vpermi2d(zmm9, k0, zmm3, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(6252654876C9                , vpermi2d(zmm9, k0, zmm3, zmm9)),
        INSTRUCTION_TEST(62721D40769200200000        , vpermi2d(zmm10, k0, zmm28, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62721D4076527F              , vpermi2d(zmm10, k0, zmm28, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62721D407692C0DFFFFF        , vpermi2d(zmm10, k0, zmm28, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62721D407611                , vpermi2d(zmm10, k0, zmm28, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62721D4076D4                , vpermi2d(zmm10, k0, zmm28, zmm4)),
        INSTRUCTION_TEST(62321D407694F023010000      , vpermi2d(zmm10, k0, zmm28, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62F26D48764C2B02            , vpermi2d(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F26D4876CB                , vpermi2d(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(62F26D28764C2B04            , vpermi2d(ymm1, k0, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F26D2876CB                , vpermi2d(ymm1, k0, ymm2, ymm3)),
        INSTRUCTION_TEST(62F26D08764C2B08            , vpermi2d(xmm1, k0, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F26D0876CB                , vpermi2d(xmm1, k0, xmm2, xmm3)),
        INSTRUCTION_TEST(62721D40765280              , vpermi2d(zmm10, k0, zmm28, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62A2A54077ED                , vpermi2pd(zmm21, k0, zmm27, zmm21)),
        INSTRUCTION_TEST(62A2A54077ACF034120000      , vpermi2pd(zmm21, k0, zmm27, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62E2A5407729                , vpermi2pd(zmm21, k0, zmm27, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62E2A540776A80              , vpermi2pd(zmm21, k0, zmm27, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62E2A540776A7F              , vpermi2pd(zmm21, k0, zmm27, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62E2A54077AA00200000        , vpermi2pd(zmm21, k0, zmm27, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62E2D54877A200200000        , vpermi2pd(zmm20, k0, zmm5, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62E2A54077AAC0DFFFFF        , vpermi2pd(zmm21, k0, zmm27, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62E2D54877627F              , vpermi2pd(zmm20, k0, zmm5, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62E2D54877A2C0DFFFFF        , vpermi2pd(zmm20, k0, zmm5, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62F2ED08774C2B08            , vpermi2pd(xmm1, k0, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62E2D5487721                , vpermi2pd(zmm20, k0, zmm5, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62A2D54877A4F023010000      , vpermi2pd(zmm20, k0, zmm5, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62A2D54877E4                , vpermi2pd(zmm20, k0, zmm5, zmm20)),
        INSTRUCTION_TEST(62F2ED48774C2B02            , vpermi2pd(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F2ED4877CB                , vpermi2pd(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(62F2ED0877CB                , vpermi2pd(xmm1, k0, xmm2, xmm3)),
        INSTRUCTION_TEST(62F2ED28774C2B04            , vpermi2pd(ymm1, k0, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F2ED2877CB                , vpermi2pd(ymm1, k0, ymm2, ymm3)),
        INSTRUCTION_TEST(62E2D548776280              , vpermi2pd(zmm20, k0, zmm5, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62624540778200200000        , vpermi2ps(zmm24, k0, zmm23, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62222D407794F034120000      , vpermi2ps(zmm26, k0, zmm26, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62622D4077D1                , vpermi2ps(zmm26, k0, zmm26, zmm1)),
        INSTRUCTION_TEST(62622D407711                , vpermi2ps(zmm26, k0, zmm26, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62622D40779200200000        , vpermi2ps(zmm26, k0, zmm26, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62622D407792C0DFFFFF        , vpermi2ps(zmm26, k0, zmm26, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62622D4077527F              , vpermi2ps(zmm26, k0, zmm26, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(6262454077427F              , vpermi2ps(zmm24, k0, zmm23, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62622D40775280              , vpermi2ps(zmm26, k0, zmm26, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(626245407782C0DFFFFF        , vpermi2ps(zmm24, k0, zmm23, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62624540774280              , vpermi2ps(zmm24, k0, zmm23, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(626245407701                , vpermi2ps(zmm24, k0, zmm23, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(622245407784F023010000      , vpermi2ps(zmm24, k0, zmm23, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(6242454077C0                , vpermi2ps(zmm24, k0, zmm23, zmm8)),
        INSTRUCTION_TEST(62F26D48774C2B02            , vpermi2ps(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F26D4877CB                , vpermi2ps(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(62F26D28774C2B04            , vpermi2ps(ymm1, k0, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F26D2877CB                , vpermi2ps(ymm1, k0, ymm2, ymm3)),
        INSTRUCTION_TEST(62F26D08774C2B08            , vpermi2ps(xmm1, k0, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F26D0877CB                , vpermi2ps(xmm1, k0, xmm2, xmm3)),
        INSTRUCTION_TEST(62E29D40769200200000        , vpermi2q(zmm18, k0, zmm28, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(6282CD4076E9                , vpermi2q(zmm21, k0, zmm22, zmm25)),
        INSTRUCTION_TEST(62A2CD4076ACF034120000      , vpermi2q(zmm21, k0, zmm22, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62E2CD407629                , vpermi2q(zmm21, k0, zmm22, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62E2CD40766A7F              , vpermi2q(zmm21, k0, zmm22, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62E2CD4076AAC0DFFFFF        , vpermi2q(zmm21, k0, zmm22, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62E29D4076527F              , vpermi2q(zmm18, k0, zmm28, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62E2CD4076AA00200000        , vpermi2q(zmm21, k0, zmm22, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62E2CD40766A80              , vpermi2q(zmm21, k0, zmm22, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62E29D407692C0DFFFFF        , vpermi2q(zmm18, k0, zmm28, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62F2ED2876CB                , vpermi2q(ymm1, k0, ymm2, ymm3)),
        INSTRUCTION_TEST(62E29D407611                , vpermi2q(zmm18, k0, zmm28, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62A29D407694F023010000      , vpermi2q(zmm18, k0, zmm28, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62829D4076D4                , vpermi2q(zmm18, k0, zmm28, zmm28)),
        INSTRUCTION_TEST(62F2ED48764C2B02            , vpermi2q(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F2ED4876CB                , vpermi2q(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(62F2ED28764C2B04            , vpermi2q(ymm1, k0, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F2ED08764C2B08            , vpermi2q(xmm1, k0, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F2ED0876CB                , vpermi2q(xmm1, k0, xmm2, xmm3)),
        INSTRUCTION_TEST(62E29D40765280              , vpermi2q(zmm18, k0, zmm28, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62F2ED0875CB                , vpermi2w(xmm1, k0, xmm2, xmm3)),
        INSTRUCTION_TEST(62F2ED08754C2B08            , vpermi2w(xmm1, k0, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F2ED2875CB                , vpermi2w(ymm1, k0, ymm2, ymm3)),
        INSTRUCTION_TEST(62F2ED28754C2B04            , vpermi2w(ymm1, k0, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F2ED4875CB                , vpermi2w(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(62F2ED48754C2B02            , vpermi2w(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62E3FD4805DCAB              , vpermilpd(zmm19, k0, zmm4, Imm(171))),
        INSTRUCTION_TEST(62A3FD48059CF0230100007B    , vpermilpd(zmm19, k0, zmmword_ptr(rax, r14, 8, 291), Imm(123))),
        INSTRUCTION_TEST(62E3FD4805DC7B              , vpermilpd(zmm19, k0, zmm4, Imm(123))),
        INSTRUCTION_TEST(62E3FD4805197B              , vpermilpd(zmm19, k0, zmmword_ptr(rcx), Imm(123))),
        INSTRUCTION_TEST(6273FD48057A7F7B            , vpermilpd(zmm15, k0, zmmword_ptr(rdx, 8128), Imm(123))),
        INSTRUCTION_TEST(6273FD4805BAC0DFFFFF7B      , vpermilpd(zmm15, k0, zmmword_ptr(rdx, -8256), Imm(123))),
        INSTRUCTION_TEST(6273FD48057A807B            , vpermilpd(zmm15, k0, zmmword_ptr(rdx, -8192), Imm(123))),
        INSTRUCTION_TEST(6273FD4805BA002000007B      , vpermilpd(zmm15, k0, zmmword_ptr(rdx, 8192), Imm(123))),
        INSTRUCTION_TEST(62E3FD48055A807B            , vpermilpd(zmm19, k0, zmmword_ptr(rdx, -8192), Imm(123))),
        INSTRUCTION_TEST(62A2AD400DBCF034120000      , vpermilpd(zmm23, k0, zmm26, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62E3FD48059A002000007B      , vpermilpd(zmm19, k0, zmmword_ptr(rdx, 8192), Imm(123))),
        INSTRUCTION_TEST(62C2AD400DF8                , vpermilpd(zmm23, k0, zmm26, zmm8)),
        INSTRUCTION_TEST(62E2AD400D39                , vpermilpd(zmm23, k0, zmm26, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62E2AD400D7A80              , vpermilpd(zmm23, k0, zmm26, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62E2AD400DBAC0DFFFFF        , vpermilpd(zmm23, k0, zmm26, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62E2AD400D7A7F              , vpermilpd(zmm23, k0, zmm26, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62E2AD400DBA00200000        , vpermilpd(zmm23, k0, zmm26, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(6273FD4805397B              , vpermilpd(zmm15, k0, zmmword_ptr(rcx), Imm(123))),
        INSTRUCTION_TEST(62E3FD48059AC0DFFFFF7B      , vpermilpd(zmm19, k0, zmmword_ptr(rdx, -8256), Imm(123))),
        INSTRUCTION_TEST(6233FD4805BCF0341200007B    , vpermilpd(zmm15, k0, zmmword_ptr(rax, r14, 8, 4660), Imm(123))),
        INSTRUCTION_TEST(62E3FD48055A7F7B            , vpermilpd(zmm19, k0, zmmword_ptr(rdx, 8128), Imm(123))),
        INSTRUCTION_TEST(6253FD4805F87B              , vpermilpd(zmm15, k0, zmm8, Imm(123))),
        INSTRUCTION_TEST(C4E37905CA01                , vpermilpd(xmm1, xmm2, Imm(1))),
        INSTRUCTION_TEST(C4E2690DCB                  , vpermilpd(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(C4E2690D8C2B80000000        , vpermilpd(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E379058C1A8000000001      , vpermilpd(xmm1, xmmword_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(C4E37D05CA01                , vpermilpd(ymm1, ymm2, Imm(1))),
        INSTRUCTION_TEST(C4E26D0DCB                  , vpermilpd(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(C4E26D0D8C2B80000000        , vpermilpd(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E37D058C1A8000000001      , vpermilpd(ymm1, ymmword_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F3FD4805CA01              , vpermilpd(zmm1, k0, zmm2, Imm(1))),
        INSTRUCTION_TEST(6253FD4805F8AB              , vpermilpd(zmm15, k0, zmm8, Imm(171))),
        INSTRUCTION_TEST(62F2ED480D4C2B02            , vpermilpd(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F2ED480DCB                , vpermilpd(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(62F3FD48054C1A0201          , vpermilpd(zmm1, k0, zmmword_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F2AD400D4A7F              , vpermilpd(zmm1, k0, zmm26, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62F2AD400D8AC0DFFFFF        , vpermilpd(zmm1, k0, zmm26, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62F2AD400D8A00200000        , vpermilpd(zmm1, k0, zmm26, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62F2AD400D09                , vpermilpd(zmm1, k0, zmm26, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62B2AD400D8CF023010000      , vpermilpd(zmm1, k0, zmm26, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62B2AD400DCD                , vpermilpd(zmm1, k0, zmm26, zmm21)),
        INSTRUCTION_TEST(62F2AD400D4A80              , vpermilpd(zmm1, k0, zmm26, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62637D4804097B              , vpermilps(zmm25, k0, zmmword_ptr(rcx), Imm(123))),
        INSTRUCTION_TEST(62F37D480452807B            , vpermilps(zmm2, k0, zmmword_ptr(rdx, -8192), Imm(123))),
        INSTRUCTION_TEST(62F37D4804527F7B            , vpermilps(zmm2, k0, zmmword_ptr(rdx, 8128), Imm(123))),
        INSTRUCTION_TEST(62F37D480492002000007B      , vpermilps(zmm2, k0, zmmword_ptr(rdx, 8192), Imm(123))),
        INSTRUCTION_TEST(62237D48048CF0341200007B    , vpermilps(zmm25, k0, zmmword_ptr(rax, r14, 8, 4660), Imm(123))),
        INSTRUCTION_TEST(62637D4804CF7B              , vpermilps(zmm25, k0, zmm7, Imm(123))),
        INSTRUCTION_TEST(62637D4804CFAB              , vpermilps(zmm25, k0, zmm7, Imm(171))),
        INSTRUCTION_TEST(62637D48044A807B            , vpermilps(zmm25, k0, zmmword_ptr(rdx, -8192), Imm(123))),
        INSTRUCTION_TEST(62B22D400C9CF034120000      , vpermilps(zmm3, k0, zmm26, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62637D48044A7F7B            , vpermilps(zmm25, k0, zmmword_ptr(rdx, 8128), Imm(123))),
        INSTRUCTION_TEST(62637D48048A002000007B      , vpermilps(zmm25, k0, zmmword_ptr(rdx, 8192), Imm(123))),
        INSTRUCTION_TEST(62B22D400CD9                , vpermilps(zmm3, k0, zmm26, zmm17)),
        INSTRUCTION_TEST(62F37D4804117B              , vpermilps(zmm2, k0, zmmword_ptr(rcx), Imm(123))),
        INSTRUCTION_TEST(62F22D400C19                , vpermilps(zmm3, k0, zmm26, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62F22D400C5A80              , vpermilps(zmm3, k0, zmm26, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62F22D400C9AC0DFFFFF        , vpermilps(zmm3, k0, zmm26, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62F22D400C5A7F              , vpermilps(zmm3, k0, zmm26, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62F22D400C9A00200000        , vpermilps(zmm3, k0, zmm26, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62637D48048AC0DFFFFF7B      , vpermilps(zmm25, k0, zmmword_ptr(rdx, -8256), Imm(123))),
        INSTRUCTION_TEST(62B37D480494F0230100007B    , vpermilps(zmm2, k0, zmmword_ptr(rax, r14, 8, 291), Imm(123))),
        INSTRUCTION_TEST(62F37D480492C0DFFFFF7B      , vpermilps(zmm2, k0, zmmword_ptr(rdx, -8256), Imm(123))),
        INSTRUCTION_TEST(C4E37904CA01                , vpermilps(xmm1, xmm2, Imm(1))),
        INSTRUCTION_TEST(62B37D4804D6AB              , vpermilps(zmm2, k0, zmm22, Imm(171))),
        INSTRUCTION_TEST(C4E2690CCB                  , vpermilps(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(C4E2690C8C2B80000000        , vpermilps(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E379048C1A8000000001      , vpermilps(xmm1, xmmword_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(C4E37D04CA01                , vpermilps(ymm1, ymm2, Imm(1))),
        INSTRUCTION_TEST(C4E26D0CCB                  , vpermilps(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(C4E26D0C8C2B80000000        , vpermilps(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E37D048C1A8000000001      , vpermilps(ymm1, ymmword_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F37D4804CA01              , vpermilps(zmm1, k0, zmm2, Imm(1))),
        INSTRUCTION_TEST(62B37D4804D67B              , vpermilps(zmm2, k0, zmm22, Imm(123))),
        INSTRUCTION_TEST(62F26D480C4C2B02            , vpermilps(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F26D480CCB                , vpermilps(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(62725D400C6A7F              , vpermilps(zmm13, k0, zmm20, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62725D400CAAC0DFFFFF        , vpermilps(zmm13, k0, zmm20, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62725D400C6A80              , vpermilps(zmm13, k0, zmm20, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62725D400CAA00200000        , vpermilps(zmm13, k0, zmm20, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62725D400CEA                , vpermilps(zmm13, k0, zmm20, zmm2)),
        INSTRUCTION_TEST(62325D400CACF023010000      , vpermilps(zmm13, k0, zmm20, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62F37D48044C1A0201          , vpermilps(zmm1, k0, zmmword_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62725D400C29                , vpermilps(zmm13, k0, zmm20, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(6263FD4801A2C0DFFFFF7B      , vpermpd(zmm28, k0, zmmword_ptr(rdx, -8256), Imm(123))),
        INSTRUCTION_TEST(62629D48169A00200000        , vpermpd(zmm27, k0, zmm12, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(6243FD4801E77B              , vpermpd(zmm28, k0, zmm15, Imm(123))),
        INSTRUCTION_TEST(6243FD4801E7AB              , vpermpd(zmm28, k0, zmm15, Imm(171))),
        INSTRUCTION_TEST(6223FD4801A4F0341200007B    , vpermpd(zmm28, k0, zmmword_ptr(rax, r14, 8, 4660), Imm(123))),
        INSTRUCTION_TEST(6263FD4801217B              , vpermpd(zmm28, k0, zmmword_ptr(rcx), Imm(123))),
        INSTRUCTION_TEST(6263FD480162807B            , vpermpd(zmm28, k0, zmmword_ptr(rdx, -8192), Imm(123))),
        INSTRUCTION_TEST(6263FD4801627F7B            , vpermpd(zmm28, k0, zmmword_ptr(rdx, 8128), Imm(123))),
        INSTRUCTION_TEST(62F3FD4801197B              , vpermpd(zmm3, k0, zmmword_ptr(rcx), Imm(123))),
        INSTRUCTION_TEST(6293FD4801DB7B              , vpermpd(zmm3, k0, zmm27, Imm(123))),
        INSTRUCTION_TEST(6293FD4801DBAB              , vpermpd(zmm3, k0, zmm27, Imm(171))),
        INSTRUCTION_TEST(62B3FD48019CF0230100007B    , vpermpd(zmm3, k0, zmmword_ptr(rax, r14, 8, 291), Imm(123))),
        INSTRUCTION_TEST(62F3FD48015A807B            , vpermpd(zmm3, k0, zmmword_ptr(rdx, -8192), Imm(123))),
        INSTRUCTION_TEST(62F3FD48019AC0DFFFFF7B      , vpermpd(zmm3, k0, zmmword_ptr(rdx, -8256), Imm(123))),
        INSTRUCTION_TEST(62F3FD48015A7F7B            , vpermpd(zmm3, k0, zmmword_ptr(rdx, 8128), Imm(123))),
        INSTRUCTION_TEST(62F3FD48019A002000007B      , vpermpd(zmm3, k0, zmmword_ptr(rdx, 8192), Imm(123))),
        INSTRUCTION_TEST(6263FD4801A2002000007B      , vpermpd(zmm28, k0, zmmword_ptr(rdx, 8192), Imm(123))),
        INSTRUCTION_TEST(62629D48165A7F              , vpermpd(zmm27, k0, zmm12, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62629D48169AC0DFFFFF        , vpermpd(zmm27, k0, zmm12, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62629D481619                , vpermpd(zmm27, k0, zmm12, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62629D48165A80              , vpermpd(zmm27, k0, zmm12, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(C4E3FD01CA01                , vpermpd(ymm1, ymm2, Imm(1))),
        INSTRUCTION_TEST(62F2ED2816CB                , vpermpd(ymm1, k0, ymm2, ymm3)),
        INSTRUCTION_TEST(62F2ED28164C2B04            , vpermpd(ymm1, k0, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E3FD018C1A8000000001      , vpermpd(ymm1, ymmword_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F2ED4816CB                , vpermpd(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(62F2ED48164C2B02            , vpermpd(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F3FD48014C1A0201          , vpermpd(zmm1, k0, zmmword_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F3FD4801CA01              , vpermpd(zmm1, k0, zmm2, Imm(1))),
        INSTRUCTION_TEST(622295401694F023010000      , vpermpd(zmm26, k0, zmm29, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(6222954016D2                , vpermpd(zmm26, k0, zmm29, zmm18)),
        INSTRUCTION_TEST(62229D48169CF034120000      , vpermpd(zmm27, k0, zmm12, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62629540169200200000        , vpermpd(zmm26, k0, zmm29, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(6262954016527F              , vpermpd(zmm26, k0, zmm29, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62629D4816DA                , vpermpd(zmm27, k0, zmm12, zmm2)),
        INSTRUCTION_TEST(62629540165280              , vpermpd(zmm26, k0, zmm29, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(626295401611                , vpermpd(zmm26, k0, zmm29, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(626295401692C0DFFFFF        , vpermpd(zmm26, k0, zmm29, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62E24D4816B200200000        , vpermps(zmm22, k0, zmm6, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62F26D4816A200200000        , vpermps(zmm4, k0, zmm2, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62F26D4816627F              , vpermps(zmm4, k0, zmm2, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62F26D4816A2C0DFFFFF        , vpermps(zmm4, k0, zmm2, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62F26D48166280              , vpermps(zmm4, k0, zmm2, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62F26D481621                , vpermps(zmm4, k0, zmm2, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62B26D4816A4F023010000      , vpermps(zmm4, k0, zmm2, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62926D4816E0                , vpermps(zmm4, k0, zmm2, zmm24)),
        INSTRUCTION_TEST(62E24D4816727F              , vpermps(zmm22, k0, zmm6, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62E24D4816B2C0DFFFFF        , vpermps(zmm22, k0, zmm6, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62E24D48167280              , vpermps(zmm22, k0, zmm6, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62E24D481631                , vpermps(zmm22, k0, zmm6, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62A24D4816B4F034120000      , vpermps(zmm22, k0, zmm6, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62A24D4816F5                , vpermps(zmm22, k0, zmm6, zmm21)),
        INSTRUCTION_TEST(62F26D48164C2B02            , vpermps(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F26D4816CB                , vpermps(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(C4E26D168C2B80000000        , vpermps(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E26D16CB                  , vpermps(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(62E2C54036AA00200000        , vpermq(zmm21, k0, zmm23, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(6223FD4800C77B              , vpermq(zmm24, k0, zmm23, Imm(123))),
        INSTRUCTION_TEST(6223FD4800C7AB              , vpermq(zmm24, k0, zmm23, Imm(171))),
        INSTRUCTION_TEST(6223FD480084F0341200007B    , vpermq(zmm24, k0, zmmword_ptr(rax, r14, 8, 4660), Imm(123))),
        INSTRUCTION_TEST(6263FD4800017B              , vpermq(zmm24, k0, zmmword_ptr(rcx), Imm(123))),
        INSTRUCTION_TEST(6263FD480042807B            , vpermq(zmm24, k0, zmmword_ptr(rdx, -8192), Imm(123))),
        INSTRUCTION_TEST(6263FD480082C0DFFFFF7B      , vpermq(zmm24, k0, zmmword_ptr(rdx, -8256), Imm(123))),
        INSTRUCTION_TEST(6263FD4800427F7B            , vpermq(zmm24, k0, zmmword_ptr(rdx, 8128), Imm(123))),
        INSTRUCTION_TEST(6263FD4800CCAB              , vpermq(zmm25, k0, zmm4, Imm(171))),
        INSTRUCTION_TEST(6223FD48008CF0230100007B    , vpermq(zmm25, k0, zmmword_ptr(rax, r14, 8, 291), Imm(123))),
        INSTRUCTION_TEST(6263FD4800CC7B              , vpermq(zmm25, k0, zmm4, Imm(123))),
        INSTRUCTION_TEST(62E2C540366A7F              , vpermq(zmm21, k0, zmm23, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(6263FD48008A002000007B      , vpermq(zmm25, k0, zmmword_ptr(rdx, 8192), Imm(123))),
        INSTRUCTION_TEST(6263FD4800097B              , vpermq(zmm25, k0, zmmword_ptr(rcx), Imm(123))),
        INSTRUCTION_TEST(6263FD48004A807B            , vpermq(zmm25, k0, zmmword_ptr(rdx, -8192), Imm(123))),
        INSTRUCTION_TEST(6263FD48008AC0DFFFFF7B      , vpermq(zmm25, k0, zmmword_ptr(rdx, -8256), Imm(123))),
        INSTRUCTION_TEST(6263FD48004A7F7B            , vpermq(zmm25, k0, zmmword_ptr(rdx, 8128), Imm(123))),
        INSTRUCTION_TEST(6263FD480082002000007B      , vpermq(zmm24, k0, zmmword_ptr(rdx, 8192), Imm(123))),
        INSTRUCTION_TEST(62E2C54036AAC0DFFFFF        , vpermq(zmm21, k0, zmm23, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62F3FD4800CA01              , vpermq(zmm1, k0, zmm2, Imm(1))),
        INSTRUCTION_TEST(62E2C5403629                , vpermq(zmm21, k0, zmm23, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(C4E3FD00CA01                , vpermq(ymm1, ymm2, Imm(1))),
        INSTRUCTION_TEST(62F2ED2836CB                , vpermq(ymm1, k0, ymm2, ymm3)),
        INSTRUCTION_TEST(62F2ED28364C2B04            , vpermq(ymm1, k0, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62E2C540366A80              , vpermq(zmm21, k0, zmm23, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62F2ED4836CB                , vpermq(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(62F2ED48364C2B02            , vpermq(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F3FD48004C1A0201          , vpermq(zmm1, k0, zmmword_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62A2CD4036D3                , vpermq(zmm18, k0, zmm22, zmm19)),
        INSTRUCTION_TEST(C4E3FD008C1A8000000001      , vpermq(ymm1, ymmword_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62E2CD403611                , vpermq(zmm18, k0, zmm22, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62E2CD40365280              , vpermq(zmm18, k0, zmm22, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62E2CD403692C0DFFFFF        , vpermq(zmm18, k0, zmm22, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62E2CD4036527F              , vpermq(zmm18, k0, zmm22, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62E2CD40369200200000        , vpermq(zmm18, k0, zmm22, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62A2C54036EE                , vpermq(zmm21, k0, zmm23, zmm22)),
        INSTRUCTION_TEST(62A2C54036ACF023010000      , vpermq(zmm21, k0, zmm23, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62A2CD403694F034120000      , vpermq(zmm18, k0, zmm22, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62F26D487D4C2B02            , vpermt2b(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F26D487DCB                , vpermt2b(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(62F26D287D4C2B04            , vpermt2b(ymm1, k0, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F26D087DCB                , vpermt2b(xmm1, k0, xmm2, xmm3)),
        INSTRUCTION_TEST(62F26D087D4C2B08            , vpermt2b(xmm1, k0, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F26D287DCB                , vpermt2b(ymm1, k0, ymm2, ymm3)),
        INSTRUCTION_TEST(628205487ED9                , vpermt2d(zmm19, k0, zmm15, zmm25)),
        INSTRUCTION_TEST(62724D487EAA00200000        , vpermt2d(zmm13, k0, zmm6, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62E205487E9A00200000        , vpermt2d(zmm19, k0, zmm15, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62724D487E6A7F              , vpermt2d(zmm13, k0, zmm6, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62E205487E5A80              , vpermt2d(zmm19, k0, zmm15, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62E205487E9AC0DFFFFF        , vpermt2d(zmm19, k0, zmm15, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62E205487E5A7F              , vpermt2d(zmm19, k0, zmm15, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62E205487E19                , vpermt2d(zmm19, k0, zmm15, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62724D487EAAC0DFFFFF        , vpermt2d(zmm13, k0, zmm6, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62A205487E9CF034120000      , vpermt2d(zmm19, k0, zmm15, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62724D487E29                , vpermt2d(zmm13, k0, zmm6, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62F26D087ECB                , vpermt2d(xmm1, k0, xmm2, xmm3)),
        INSTRUCTION_TEST(62F26D087E4C2B08            , vpermt2d(xmm1, k0, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F26D287ECB                , vpermt2d(ymm1, k0, ymm2, ymm3)),
        INSTRUCTION_TEST(62F26D287E4C2B04            , vpermt2d(ymm1, k0, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62724D487E6A80              , vpermt2d(zmm13, k0, zmm6, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62F26D487E4C2B02            , vpermt2d(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62324D487EACF023010000      , vpermt2d(zmm13, k0, zmm6, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62724D487EEE                , vpermt2d(zmm13, k0, zmm6, zmm6)),
        INSTRUCTION_TEST(62F26D487ECB                , vpermt2d(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(6272CD407F627F              , vpermt2pd(zmm12, k0, zmm22, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62F2A5407F9A00200000        , vpermt2pd(zmm3, k0, zmm27, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62F2A5407F5A7F              , vpermt2pd(zmm3, k0, zmm27, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62F2A5407F9AC0DFFFFF        , vpermt2pd(zmm3, k0, zmm27, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62F2A5407F5A80              , vpermt2pd(zmm3, k0, zmm27, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62F2A5407F19                , vpermt2pd(zmm3, k0, zmm27, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62B2A5407F9CF023010000      , vpermt2pd(zmm3, k0, zmm27, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62D2A5407FDB                , vpermt2pd(zmm3, k0, zmm27, zmm11)),
        INSTRUCTION_TEST(6272CD407FA200200000        , vpermt2pd(zmm12, k0, zmm22, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(6272CD407FA2C0DFFFFF        , vpermt2pd(zmm12, k0, zmm22, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(6272CD407F6280              , vpermt2pd(zmm12, k0, zmm22, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(6272CD407F21                , vpermt2pd(zmm12, k0, zmm22, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(6232CD407FA4F034120000      , vpermt2pd(zmm12, k0, zmm22, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(6212CD407FE1                , vpermt2pd(zmm12, k0, zmm22, zmm25)),
        INSTRUCTION_TEST(62F2ED487F4C2B02            , vpermt2pd(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F2ED487FCB                , vpermt2pd(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(62F2ED287F4C2B04            , vpermt2pd(ymm1, k0, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F2ED287FCB                , vpermt2pd(ymm1, k0, ymm2, ymm3)),
        INSTRUCTION_TEST(62F2ED087F4C2B08            , vpermt2pd(xmm1, k0, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F2ED087FCB                , vpermt2pd(xmm1, k0, xmm2, xmm3)),
        INSTRUCTION_TEST(62721D407F9A00200000        , vpermt2ps(zmm11, k0, zmm28, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62F205487FFF                , vpermt2ps(zmm7, k0, zmm15, zmm7)),
        INSTRUCTION_TEST(62B205487FBCF034120000      , vpermt2ps(zmm7, k0, zmm15, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62F205487F39                , vpermt2ps(zmm7, k0, zmm15, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62F205487FBA00200000        , vpermt2ps(zmm7, k0, zmm15, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62F205487FBAC0DFFFFF        , vpermt2ps(zmm7, k0, zmm15, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62F205487F7A7F              , vpermt2ps(zmm7, k0, zmm15, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62721D407F5A7F              , vpermt2ps(zmm11, k0, zmm28, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62F205487F7A80              , vpermt2ps(zmm7, k0, zmm15, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62721D407F9AC0DFFFFF        , vpermt2ps(zmm11, k0, zmm28, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62721D407F5A80              , vpermt2ps(zmm11, k0, zmm28, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62721D407F19                , vpermt2ps(zmm11, k0, zmm28, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62721D407FDA                , vpermt2ps(zmm11, k0, zmm28, zmm2)),
        INSTRUCTION_TEST(62321D407F9CF023010000      , vpermt2ps(zmm11, k0, zmm28, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62F26D487F4C2B02            , vpermt2ps(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F26D487FCB                , vpermt2ps(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(62F26D287F4C2B04            , vpermt2ps(ymm1, k0, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F26D287FCB                , vpermt2ps(ymm1, k0, ymm2, ymm3)),
        INSTRUCTION_TEST(62F26D087F4C2B08            , vpermt2ps(xmm1, k0, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F26D087FCB                , vpermt2ps(xmm1, k0, xmm2, xmm3)),
        INSTRUCTION_TEST(62F2D5407E9200200000        , vpermt2q(zmm2, k0, zmm21, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62A29D487EE8                , vpermt2q(zmm21, k0, zmm12, zmm16)),
        INSTRUCTION_TEST(62A29D487EACF023010000      , vpermt2q(zmm21, k0, zmm12, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62E29D487E29                , vpermt2q(zmm21, k0, zmm12, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62E29D487EAA00200000        , vpermt2q(zmm21, k0, zmm12, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62E29D487EAAC0DFFFFF        , vpermt2q(zmm21, k0, zmm12, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62E29D487E6A7F              , vpermt2q(zmm21, k0, zmm12, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62F2D5407E527F              , vpermt2q(zmm2, k0, zmm21, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62E29D487E6A80              , vpermt2q(zmm21, k0, zmm12, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62F2D5407E92C0DFFFFF        , vpermt2q(zmm2, k0, zmm21, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62F2D5407E5280              , vpermt2q(zmm2, k0, zmm21, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62F2D5407E11                , vpermt2q(zmm2, k0, zmm21, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62F2ED087E4C2B08            , vpermt2q(xmm1, k0, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F2ED287ECB                , vpermt2q(ymm1, k0, ymm2, ymm3)),
        INSTRUCTION_TEST(62F2ED287E4C2B04            , vpermt2q(ymm1, k0, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F2ED087ECB                , vpermt2q(xmm1, k0, xmm2, xmm3)),
        INSTRUCTION_TEST(62F2ED487E4C2B02            , vpermt2q(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F2D5407ED4                , vpermt2q(zmm2, k0, zmm21, zmm4)),
        INSTRUCTION_TEST(62B2D5407E94F034120000      , vpermt2q(zmm2, k0, zmm21, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62F2ED487ECB                , vpermt2q(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(62F2ED087DCB                , vpermt2w(xmm1, k0, xmm2, xmm3)),
        INSTRUCTION_TEST(62F2ED087D4C2B08            , vpermt2w(xmm1, k0, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F2ED287DCB                , vpermt2w(ymm1, k0, ymm2, ymm3)),
        INSTRUCTION_TEST(62F2ED287D4C2B04            , vpermt2w(ymm1, k0, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F2ED487DCB                , vpermt2w(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(62F2ED487D4C2B02            , vpermt2w(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F2ED488DCB                , vpermw(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(62F2ED488D4C2B02            , vpermw(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F2ED288D4C2B04            , vpermw(ymm1, k0, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F2ED088D4C2B08            , vpermw(xmm1, k0, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F2ED288DCB                , vpermw(ymm1, k0, ymm2, ymm3)),
        INSTRUCTION_TEST(62F2ED088DCB                , vpermw(xmm1, k0, xmm2, xmm3)),
        INSTRUCTION_TEST(62F27D0862CA                , vpexpandb(xmm1, k0, xmm2)),
        INSTRUCTION_TEST(62F27D08628C1A80000000      , vpexpandb(xmm1, k0, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F27D2862CA                , vpexpandb(ymm1, k0, ymm2)),
        INSTRUCTION_TEST(62F27D28628C1A80000000      , vpexpandb(ymm1, k0, ymmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F27D4862CA                , vpexpandb(zmm1, k0, zmm2)),
        INSTRUCTION_TEST(62F27D48628C1A80000000      , vpexpandb(zmm1, k0, zmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62A27D488994F034120000      , vpexpandd(zmm18, k0, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62E27D488911                , vpexpandd(zmm18, k0, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62E27D48895280              , vpexpandd(zmm18, k0, zmmword_ptr(rdx, -512))),
        INSTRUCTION_TEST(62827D4889D9                , vpexpandd(zmm19, k0, zmm25)),
        INSTRUCTION_TEST(62E27D4889527F              , vpexpandd(zmm18, k0, zmmword_ptr(rdx, 508))),
        INSTRUCTION_TEST(62E27D48899200020000        , vpexpandd(zmm18, k0, zmmword_ptr(rdx, 512))),
        INSTRUCTION_TEST(62127D4889E4                , vpexpandd(zmm12, k0, zmm28)),
        INSTRUCTION_TEST(62E27D488992FCFDFFFF        , vpexpandd(zmm18, k0, zmmword_ptr(rdx, -516))),
        INSTRUCTION_TEST(62F27D48894C1A20            , vpexpandd(zmm1, k0, zmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F27D48898A00020000        , vpexpandd(zmm1, k0, zmmword_ptr(rdx, 512))),
        INSTRUCTION_TEST(62F27D0889CA                , vpexpandd(xmm1, k0, xmm2)),
        INSTRUCTION_TEST(62F27D48898AFCFDFFFF        , vpexpandd(zmm1, k0, zmmword_ptr(rdx, -516))),
        INSTRUCTION_TEST(62F27D48894A80              , vpexpandd(zmm1, k0, zmmword_ptr(rdx, -512))),
        INSTRUCTION_TEST(62F27D488909                , vpexpandd(zmm1, k0, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62B27D48898CF023010000      , vpexpandd(zmm1, k0, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62F27D4889CA                , vpexpandd(zmm1, k0, zmm2)),
        INSTRUCTION_TEST(62F27D28894C1A20            , vpexpandd(ymm1, k0, ymmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F27D2889CA                , vpexpandd(ymm1, k0, ymm2)),
        INSTRUCTION_TEST(62F27D08894C1A20            , vpexpandd(xmm1, k0, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F27D48894A7F              , vpexpandd(zmm1, k0, zmmword_ptr(rdx, 508))),
        INSTRUCTION_TEST(6252FD4889F8                , vpexpandq(zmm15, k0, zmm8)),
        INSTRUCTION_TEST(6282FD4889CA                , vpexpandq(zmm17, k0, zmm26)),
        INSTRUCTION_TEST(6222FD488984F023010000      , vpexpandq(zmm24, k0, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(6262FD488901                , vpexpandq(zmm24, k0, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62F2FD48894C1A10            , vpexpandq(zmm1, k0, zmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(6262FD488982F8FBFFFF        , vpexpandq(zmm24, k0, zmmword_ptr(rdx, -1032))),
        INSTRUCTION_TEST(6262FD4889427F              , vpexpandq(zmm24, k0, zmmword_ptr(rdx, 1016))),
        INSTRUCTION_TEST(6262FD48898200040000        , vpexpandq(zmm24, k0, zmmword_ptr(rdx, 1024))),
        INSTRUCTION_TEST(6262FD48894280              , vpexpandq(zmm24, k0, zmmword_ptr(rdx, -1024))),
        INSTRUCTION_TEST(62F2FD48894A7F              , vpexpandq(zmm1, k0, zmmword_ptr(rdx, 1016))),
        INSTRUCTION_TEST(62F2FD48898AF8FBFFFF        , vpexpandq(zmm1, k0, zmmword_ptr(rdx, -1032))),
        INSTRUCTION_TEST(62F2FD0889CA                , vpexpandq(xmm1, k0, xmm2)),
        INSTRUCTION_TEST(62F2FD08894C1A10            , vpexpandq(xmm1, k0, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F2FD2889CA                , vpexpandq(ymm1, k0, ymm2)),
        INSTRUCTION_TEST(62F2FD28894C1A10            , vpexpandq(ymm1, k0, ymmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F2FD48898A00040000        , vpexpandq(zmm1, k0, zmmword_ptr(rdx, 1024))),
        INSTRUCTION_TEST(62B2FD48898CF034120000      , vpexpandq(zmm1, k0, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62F2FD488909                , vpexpandq(zmm1, k0, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62F2FD48894A80              , vpexpandq(zmm1, k0, zmmword_ptr(rdx, -1024))),
        INSTRUCTION_TEST(62F2FD4889CA                , vpexpandq(zmm1, k0, zmm2)),
        INSTRUCTION_TEST(62F2FD0862CA                , vpexpandw(xmm1, k0, xmm2)),
        INSTRUCTION_TEST(62F2FD08624C1A40            , vpexpandw(xmm1, k0, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F2FD2862CA                , vpexpandw(ymm1, k0, ymm2)),
        INSTRUCTION_TEST(62F2FD28624C1A40            , vpexpandw(ymm1, k0, ymmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F2FD4862CA                , vpexpandw(zmm1, k0, zmm2)),
        INSTRUCTION_TEST(62F2FD48624C1A40            , vpexpandw(zmm1, k0, zmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(C4E379149C118000000001      , vpextrb(byte_ptr(rcx, rdx, 1, 128), xmm3, Imm(1))),
        INSTRUCTION_TEST(C4E37914D101                , vpextrb(ecx, xmm2, Imm(1))),
        INSTRUCTION_TEST(C4E37916D101                , vpextrd(ecx, xmm2, Imm(1))),
        INSTRUCTION_TEST(C4E379169C118000000001      , vpextrd(dword_ptr(rcx, rdx, 1, 128), xmm3, Imm(1))),
        INSTRUCTION_TEST(C4E3F9169C118000000001      , vpextrq(qword_ptr(rcx, rdx, 1, 128), xmm3, Imm(1))),
        INSTRUCTION_TEST(C4E3F916D101                , vpextrq(rcx, xmm2, Imm(1))),
        INSTRUCTION_TEST(C5F9C5CA01                  , vpextrw(ecx, xmm2, Imm(1))),
        INSTRUCTION_TEST(C4E379159C118000000001      , vpextrw(word_ptr(rcx, rdx, 1, 128), xmm3, Imm(1))),
        INSTRUCTION_TEST(8FE978C2CA                  , vphaddbd(xmm1, xmm2)),
        INSTRUCTION_TEST(8FE978C28C1A80000000        , vphaddbd(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(8FE978C3CA                  , vphaddbq(xmm1, xmm2)),
        INSTRUCTION_TEST(8FE978C38C1A80000000        , vphaddbq(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(8FE978C1CA                  , vphaddbw(xmm1, xmm2)),
        INSTRUCTION_TEST(8FE978C18C1A80000000        , vphaddbw(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(C4E26902CB                  , vphaddd(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(C4E269028C2B80000000        , vphaddd(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E26D02CB                  , vphaddd(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(C4E26D028C2B80000000        , vphaddd(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(8FE978CBCA                  , vphadddq(xmm1, xmm2)),
        INSTRUCTION_TEST(8FE978CB8C1A80000000        , vphadddq(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(C4E26D038C2B80000000        , vphaddsw(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E26D03CB                  , vphaddsw(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(C4E26903CB                  , vphaddsw(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(C4E269038C2B80000000        , vphaddsw(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(8FE978D2CA                  , vphaddubd(xmm1, xmm2)),
        INSTRUCTION_TEST(8FE978D28C1A80000000        , vphaddubd(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(8FE978D3CA                  , vphaddubq(xmm1, xmm2)),
        INSTRUCTION_TEST(8FE978D38C1A80000000        , vphaddubq(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(8FE978D1CA                  , vphaddubw(xmm1, xmm2)),
        INSTRUCTION_TEST(8FE978D18C1A80000000        , vphaddubw(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(8FE978DB8C1A80000000        , vphaddudq(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(8FE978DBCA                  , vphaddudq(xmm1, xmm2)),
        INSTRUCTION_TEST(8FE978D6CA                  , vphadduwd(xmm1, xmm2)),
        INSTRUCTION_TEST(8FE978D68C1A80000000        , vphadduwd(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(8FE978D7CA                  , vphadduwq(xmm1, xmm2)),
        INSTRUCTION_TEST(8FE978D78C1A80000000        , vphadduwq(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(C4E26D018C2B80000000        , vphaddw(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E26901CB                  , vphaddw(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(C4E269018C2B80000000        , vphaddw(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E26D01CB                  , vphaddw(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(8FE978C68C1A80000000        , vphaddwd(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(8FE978C6CA                  , vphaddwd(xmm1, xmm2)),
        INSTRUCTION_TEST(8FE978C7CA                  , vphaddwq(xmm1, xmm2)),
        INSTRUCTION_TEST(8FE978C78C1A80000000        , vphaddwq(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(C4E27941CA                  , vphminposuw(xmm1, xmm2)),
        INSTRUCTION_TEST(C4E279418C1A80000000        , vphminposuw(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(8FE978E1CA                  , vphsubbw(xmm1, xmm2)),
        INSTRUCTION_TEST(8FE978E18C1A80000000        , vphsubbw(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(C4E26D068C2B80000000        , vphsubd(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E26D06CB                  , vphsubd(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(C4E26906CB                  , vphsubd(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(C4E269068C2B80000000        , vphsubd(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(8FE978E3CA                  , vphsubdq(xmm1, xmm2)),
        INSTRUCTION_TEST(8FE978E38C1A80000000        , vphsubdq(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(C4E26D078C2B80000000        , vphsubsw(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E26907CB                  , vphsubsw(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(C4E269078C2B80000000        , vphsubsw(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E26D07CB                  , vphsubsw(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(C4E26D058C2B80000000        , vphsubw(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E26D05CB                  , vphsubw(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(C4E26905CB                  , vphsubw(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(C4E269058C2B80000000        , vphsubw(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(8FE978E2CA                  , vphsubwd(xmm1, xmm2)),
        INSTRUCTION_TEST(8FE978E28C1A80000000        , vphsubwd(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(C4E369208C2B8000000001      , vpinsrb(xmm1, xmm2, byte_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(C4E36920CB01                , vpinsrb(xmm1, xmm2, ebx, Imm(1))),
        INSTRUCTION_TEST(C4E369228C2B8000000001      , vpinsrd(xmm1, xmm2, dword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(C4E36922CB01                , vpinsrd(xmm1, xmm2, ebx, Imm(1))),
        INSTRUCTION_TEST(C4E3E9228C2B8000000001      , vpinsrq(xmm1, xmm2, qword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(C4E3E922CB01                , vpinsrq(xmm1, xmm2, rbx, Imm(1))),
        INSTRUCTION_TEST(C5E9C4CB01                  , vpinsrw(xmm1, xmm2, ebx, Imm(1))),
        INSTRUCTION_TEST(C5E9C48C2B8000000001        , vpinsrw(xmm1, xmm2, word_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F27D48444C1A02            , vplzcntd(zmm1, k0, zmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F27D28444C1A04            , vplzcntd(ymm1, k0, ymmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F27D4844CA                , vplzcntd(zmm1, k0, zmm2)),
        INSTRUCTION_TEST(62F27D08444C1A08            , vplzcntd(xmm1, k0, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F27D0844CA                , vplzcntd(xmm1, k0, xmm2)),
        INSTRUCTION_TEST(62F27D2844CA                , vplzcntd(ymm1, k0, ymm2)),
        INSTRUCTION_TEST(62F2FD48444C1A02            , vplzcntq(zmm1, k0, zmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F2FD4844CA                , vplzcntq(zmm1, k0, zmm2)),
        INSTRUCTION_TEST(62F2FD28444C1A04            , vplzcntq(ymm1, k0, ymmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F2FD08444C1A08            , vplzcntq(xmm1, k0, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F2FD0844CA                , vplzcntq(xmm1, k0, xmm2)),
        INSTRUCTION_TEST(62F2FD2844CA                , vplzcntq(ymm1, k0, ymm2)),
        INSTRUCTION_TEST(8FE8689ECB40                , vpmacsdd(xmm1, xmm2, xmm3, xmm4)),
        INSTRUCTION_TEST(8FE8689E8C2B8000000060      , vpmacsdd(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128), xmm6)),
        INSTRUCTION_TEST(8FE8689FCB40                , vpmacsdqh(xmm1, xmm2, xmm3, xmm4)),
        INSTRUCTION_TEST(8FE8689F8C2B8000000060      , vpmacsdqh(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128), xmm6)),
        INSTRUCTION_TEST(8FE86897CB40                , vpmacsdql(xmm1, xmm2, xmm3, xmm4)),
        INSTRUCTION_TEST(8FE868978C2B8000000060      , vpmacsdql(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128), xmm6)),
        INSTRUCTION_TEST(8FE8688ECB40                , vpmacssdd(xmm1, xmm2, xmm3, xmm4)),
        INSTRUCTION_TEST(8FE8688E8C2B8000000060      , vpmacssdd(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128), xmm6)),
        INSTRUCTION_TEST(8FE8688FCB40                , vpmacssdqh(xmm1, xmm2, xmm3, xmm4)),
        INSTRUCTION_TEST(8FE8688F8C2B8000000060      , vpmacssdqh(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128), xmm6)),
        INSTRUCTION_TEST(8FE868878C2B8000000060      , vpmacssdql(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128), xmm6)),
        INSTRUCTION_TEST(8FE86887CB40                , vpmacssdql(xmm1, xmm2, xmm3, xmm4)),
        INSTRUCTION_TEST(8FE86886CB40                , vpmacsswd(xmm1, xmm2, xmm3, xmm4)),
        INSTRUCTION_TEST(8FE868868C2B8000000060      , vpmacsswd(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128), xmm6)),
        INSTRUCTION_TEST(8FE868858C2B8000000060      , vpmacssww(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128), xmm6)),
        INSTRUCTION_TEST(8FE86885CB40                , vpmacssww(xmm1, xmm2, xmm3, xmm4)),
        INSTRUCTION_TEST(8FE86896CB40                , vpmacswd(xmm1, xmm2, xmm3, xmm4)),
        INSTRUCTION_TEST(8FE868968C2B8000000060      , vpmacswd(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128), xmm6)),
        INSTRUCTION_TEST(8FE86895CB40                , vpmacsww(xmm1, xmm2, xmm3, xmm4)),
        INSTRUCTION_TEST(8FE868958C2B8000000060      , vpmacsww(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128), xmm6)),
        INSTRUCTION_TEST(8FE868A6CB40                , vpmadcsswd(xmm1, xmm2, xmm3, xmm4)),
        INSTRUCTION_TEST(8FE868A68C2B8000000060      , vpmadcsswd(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128), xmm6)),
        INSTRUCTION_TEST(8FE868B6CB40                , vpmadcswd(xmm1, xmm2, xmm3, xmm4)),
        INSTRUCTION_TEST(8FE868B68C2B8000000060      , vpmadcswd(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128), xmm6)),
        INSTRUCTION_TEST(62F2ED48B54C2B02            , vpmadd52huq(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F2ED48B5CB                , vpmadd52huq(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(62F2ED28B54C2B04            , vpmadd52huq(ymm1, k0, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F2ED08B5CB                , vpmadd52huq(xmm1, k0, xmm2, xmm3)),
        INSTRUCTION_TEST(62F2ED08B54C2B08            , vpmadd52huq(xmm1, k0, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F2ED28B5CB                , vpmadd52huq(ymm1, k0, ymm2, ymm3)),
        INSTRUCTION_TEST(62F2ED48B4CB                , vpmadd52luq(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(62F2ED48B44C2B02            , vpmadd52luq(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F2ED28B44C2B04            , vpmadd52luq(ymm1, k0, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F2ED28B4CB                , vpmadd52luq(ymm1, k0, ymm2, ymm3)),
        INSTRUCTION_TEST(62F2ED08B44C2B08            , vpmadd52luq(xmm1, k0, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F2ED08B4CB                , vpmadd52luq(xmm1, k0, xmm2, xmm3)),
        INSTRUCTION_TEST(C4E26904CB                  , vpmaddubsw(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(C4E269048C2B80000000        , vpmaddubsw(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E26D04CB                  , vpmaddubsw(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(C4E26D048C2B80000000        , vpmaddubsw(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F26D4804CB                , vpmaddubsw(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(62F26D48044C2B02            , vpmaddubsw(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C5E9F5CB                    , vpmaddwd(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(C5E9F58C2B80000000          , vpmaddwd(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C5EDF5CB                    , vpmaddwd(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(C5EDF58C2B80000000          , vpmaddwd(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F16D48F5CB                , vpmaddwd(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(62F16D48F54C2B02            , vpmaddwd(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E2658EA41180000000        , vpmaskmovd(ymmword_ptr(rcx, rdx, 1, 128), ymm3, ymm4)),
        INSTRUCTION_TEST(C4E26D8C8C2B80000000        , vpmaskmovd(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E2698C8C2B80000000        , vpmaskmovd(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E2618EA41180000000        , vpmaskmovd(xmmword_ptr(rcx, rdx, 1, 128), xmm3, xmm4)),
        INSTRUCTION_TEST(C4E2E98C8C2B80000000        , vpmaskmovq(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E2E18EA41180000000        , vpmaskmovq(xmmword_ptr(rcx, rdx, 1, 128), xmm3, xmm4)),
        INSTRUCTION_TEST(C4E2ED8C8C2B80000000        , vpmaskmovq(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E2E58EA41180000000        , vpmaskmovq(ymmword_ptr(rcx, rdx, 1, 128), ymm3, ymm4)),
        INSTRUCTION_TEST(62F26D483CCB                , vpmaxsb(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(C4E2693CCB                  , vpmaxsb(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(C4E2693C8C2B80000000        , vpmaxsb(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E26D3CCB                  , vpmaxsb(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(C4E26D3C8C2B80000000        , vpmaxsb(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F26D483C4C2B02            , vpmaxsb(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(626215483DB200200000        , vpmaxsd(zmm30, k0, zmm13, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62B23D483DF0                , vpmaxsd(zmm6, k0, zmm8, zmm16)),
        INSTRUCTION_TEST(62B23D483DB4F023010000      , vpmaxsd(zmm6, k0, zmm8, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62F26D483D4C2B02            , vpmaxsd(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F23D483D7280              , vpmaxsd(zmm6, k0, zmm8, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62F23D483DB2C0DFFFFF        , vpmaxsd(zmm6, k0, zmm8, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(626215483D727F              , vpmaxsd(zmm30, k0, zmm13, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62F23D483D31                , vpmaxsd(zmm6, k0, zmm8, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(626215483DB2C0DFFFFF        , vpmaxsd(zmm30, k0, zmm13, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(C4E26D3D8C2B80000000        , vpmaxsd(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(626215483D31                , vpmaxsd(zmm30, k0, zmm13, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(622215483DB4F034120000      , vpmaxsd(zmm30, k0, zmm13, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(624215483DF1                , vpmaxsd(zmm30, k0, zmm13, zmm9)),
        INSTRUCTION_TEST(62F26D483DCB                , vpmaxsd(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(62F23D483D727F              , vpmaxsd(zmm6, k0, zmm8, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(C4E26D3DCB                  , vpmaxsd(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(C4E2693D8C2B80000000        , vpmaxsd(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E2693DCB                  , vpmaxsd(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(626215483D7280              , vpmaxsd(zmm30, k0, zmm13, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62F23D483DB200200000        , vpmaxsd(zmm6, k0, zmm8, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62F2CD483DF1                , vpmaxsq(zmm6, k0, zmm6, zmm1)),
        INSTRUCTION_TEST(62F2C5483D9A00200000        , vpmaxsq(zmm3, k0, zmm7, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62B2CD483DB4F023010000      , vpmaxsq(zmm6, k0, zmm6, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62F2C5483D5A7F              , vpmaxsq(zmm3, k0, zmm7, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62F2CD483DB200200000        , vpmaxsq(zmm6, k0, zmm6, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62F2CD483DB2C0DFFFFF        , vpmaxsq(zmm6, k0, zmm6, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62F2CD483D727F              , vpmaxsq(zmm6, k0, zmm6, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62F2CD483D31                , vpmaxsq(zmm6, k0, zmm6, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62F2C5483D9AC0DFFFFF        , vpmaxsq(zmm3, k0, zmm7, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62F2CD483D7280              , vpmaxsq(zmm6, k0, zmm6, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62F2C5483D19                , vpmaxsq(zmm3, k0, zmm7, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62B2C5483D9CF034120000      , vpmaxsq(zmm3, k0, zmm7, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62F2C5483DDD                , vpmaxsq(zmm3, k0, zmm7, zmm5)),
        INSTRUCTION_TEST(62F2ED483D4C2B02            , vpmaxsq(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F2ED483DCB                , vpmaxsq(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(62F2ED283D4C2B04            , vpmaxsq(ymm1, k0, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F2ED283DCB                , vpmaxsq(ymm1, k0, ymm2, ymm3)),
        INSTRUCTION_TEST(62F2ED083D4C2B08            , vpmaxsq(xmm1, k0, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F2ED083DCB                , vpmaxsq(xmm1, k0, xmm2, xmm3)),
        INSTRUCTION_TEST(62F2C5483D5A80              , vpmaxsq(zmm3, k0, zmm7, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62F16D48EECB                , vpmaxsw(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(C5EDEE8C2B80000000          , vpmaxsw(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F16D48EE4C2B02            , vpmaxsw(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C5E9EE8C2B80000000          , vpmaxsw(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C5E9EECB                    , vpmaxsw(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(C5EDEECB                    , vpmaxsw(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(62F16D48DECB                , vpmaxub(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(C5EDDE8C2B80000000          , vpmaxub(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F16D48DE4C2B02            , vpmaxub(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C5E9DE8C2B80000000          , vpmaxub(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C5E9DECB                    , vpmaxub(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(C5EDDECB                    , vpmaxub(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(62E245483F4A7F              , vpmaxud(zmm17, k0, zmm7, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62E225403F727F              , vpmaxud(zmm22, k0, zmm27, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62E225403FB2C0DFFFFF        , vpmaxud(zmm22, k0, zmm27, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62E225403F7280              , vpmaxud(zmm22, k0, zmm27, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62E225403F31                , vpmaxud(zmm22, k0, zmm27, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62A225403FB4F034120000      , vpmaxud(zmm22, k0, zmm27, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62C225403FF4                , vpmaxud(zmm22, k0, zmm27, zmm12)),
        INSTRUCTION_TEST(62E245483F8A00200000        , vpmaxud(zmm17, k0, zmm7, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62E245483F8AC0DFFFFF        , vpmaxud(zmm17, k0, zmm7, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62E225403FB200200000        , vpmaxud(zmm22, k0, zmm27, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62E245483F09                , vpmaxud(zmm17, k0, zmm7, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62A245483F8CF023010000      , vpmaxud(zmm17, k0, zmm7, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(628245483FC9                , vpmaxud(zmm17, k0, zmm7, zmm25)),
        INSTRUCTION_TEST(62F26D483F4C2B02            , vpmaxud(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F26D483FCB                , vpmaxud(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(C4E26D3F8C2B80000000        , vpmaxud(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E26D3FCB                  , vpmaxud(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(C4E2693F8C2B80000000        , vpmaxud(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E2693FCB                  , vpmaxud(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(62E245483F4A80              , vpmaxud(zmm17, k0, zmm7, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62F2B5483F8A00200000        , vpmaxuq(zmm1, k0, zmm9, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(6202CD403FF1                , vpmaxuq(zmm30, k0, zmm22, zmm25)),
        INSTRUCTION_TEST(6222CD403FB4F023010000      , vpmaxuq(zmm30, k0, zmm22, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(6262CD403FB200200000        , vpmaxuq(zmm30, k0, zmm22, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(6262CD403FB2C0DFFFFF        , vpmaxuq(zmm30, k0, zmm22, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(6262CD403F727F              , vpmaxuq(zmm30, k0, zmm22, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62F2B5483F4A7F              , vpmaxuq(zmm1, k0, zmm9, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(6262CD403F7280              , vpmaxuq(zmm30, k0, zmm22, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62F2B5483F8AC0DFFFFF        , vpmaxuq(zmm1, k0, zmm9, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(6262CD403F31                , vpmaxuq(zmm30, k0, zmm22, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62F2B5483F09                , vpmaxuq(zmm1, k0, zmm9, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62B2B5483F8CF034120000      , vpmaxuq(zmm1, k0, zmm9, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62D2B5483FCC                , vpmaxuq(zmm1, k0, zmm9, zmm12)),
        INSTRUCTION_TEST(62F2ED483F4C2B02            , vpmaxuq(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F2ED483FCB                , vpmaxuq(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(62F2ED283F4C2B04            , vpmaxuq(ymm1, k0, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F2ED283FCB                , vpmaxuq(ymm1, k0, ymm2, ymm3)),
        INSTRUCTION_TEST(62F2ED083F4C2B08            , vpmaxuq(xmm1, k0, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F2ED083FCB                , vpmaxuq(xmm1, k0, xmm2, xmm3)),
        INSTRUCTION_TEST(62F2B5483F4A80              , vpmaxuq(zmm1, k0, zmm9, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62F26D483ECB                , vpmaxuw(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(C4E26D3E8C2B80000000        , vpmaxuw(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F26D483E4C2B02            , vpmaxuw(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E2693E8C2B80000000        , vpmaxuw(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E2693ECB                  , vpmaxuw(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(C4E26D3ECB                  , vpmaxuw(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(C4E26938CB                  , vpminsb(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(C4E269388C2B80000000        , vpminsb(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E26D38CB                  , vpminsb(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(C4E26D388C2B80000000        , vpminsb(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F26D4838CB                , vpminsb(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(62F26D48384C2B02            , vpminsb(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62E27540399A00200000        , vpminsd(zmm19, k0, zmm17, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62927D4039D0                , vpminsd(zmm2, k0, zmm16, zmm24)),
        INSTRUCTION_TEST(62B27D403994F023010000      , vpminsd(zmm2, k0, zmm16, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62F27D403911                , vpminsd(zmm2, k0, zmm16, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62F27D40399200200000        , vpminsd(zmm2, k0, zmm16, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62F27D403992C0DFFFFF        , vpminsd(zmm2, k0, zmm16, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62F27D4039527F              , vpminsd(zmm2, k0, zmm16, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62E27540395A7F              , vpminsd(zmm19, k0, zmm17, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62F27D40395280              , vpminsd(zmm2, k0, zmm16, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62E27540399AC0DFFFFF        , vpminsd(zmm19, k0, zmm17, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(C4E26D398C2B80000000        , vpminsd(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62E275403919                , vpminsd(zmm19, k0, zmm17, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62A27540399CF034120000      , vpminsd(zmm19, k0, zmm17, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62E2754039DA                , vpminsd(zmm19, k0, zmm17, zmm2)),
        INSTRUCTION_TEST(62F26D48394C2B02            , vpminsd(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F26D4839CB                , vpminsd(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(62E27540395A80              , vpminsd(zmm19, k0, zmm17, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(C4E26D39CB                  , vpminsd(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(C4E269398C2B80000000        , vpminsd(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E26939CB                  , vpminsd(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(62E2FD4039A200200000        , vpminsq(zmm20, k0, zmm16, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62A2F54839EE                , vpminsq(zmm21, k0, zmm1, zmm22)),
        INSTRUCTION_TEST(62A2F54839ACF034120000      , vpminsq(zmm21, k0, zmm1, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62E2F5483929                , vpminsq(zmm21, k0, zmm1, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62E2F54839AA00200000        , vpminsq(zmm21, k0, zmm1, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62E2F54839AAC0DFFFFF        , vpminsq(zmm21, k0, zmm1, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62E2F548396A7F              , vpminsq(zmm21, k0, zmm1, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62E2FD4039627F              , vpminsq(zmm20, k0, zmm16, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62E2F548396A80              , vpminsq(zmm21, k0, zmm1, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62E2FD4039A2C0DFFFFF        , vpminsq(zmm20, k0, zmm16, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62A2FD4039E1                , vpminsq(zmm20, k0, zmm16, zmm17)),
        INSTRUCTION_TEST(62E2FD403921                , vpminsq(zmm20, k0, zmm16, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62A2FD4039A4F023010000      , vpminsq(zmm20, k0, zmm16, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62F2ED48394C2B02            , vpminsq(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F2ED4839CB                , vpminsq(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(62F2ED28394C2B04            , vpminsq(ymm1, k0, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F2ED2839CB                , vpminsq(ymm1, k0, ymm2, ymm3)),
        INSTRUCTION_TEST(62F2ED08394C2B08            , vpminsq(xmm1, k0, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F2ED0839CB                , vpminsq(xmm1, k0, xmm2, xmm3)),
        INSTRUCTION_TEST(62E2FD40396280              , vpminsq(zmm20, k0, zmm16, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62F16D48EACB                , vpminsw(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(C5EDEA8C2B80000000          , vpminsw(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F16D48EA4C2B02            , vpminsw(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C5E9EA8C2B80000000          , vpminsw(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C5EDEACB                    , vpminsw(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(C5E9EACB                    , vpminsw(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(C5E9DACB                    , vpminub(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(C5E9DA8C2B80000000          , vpminub(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C5EDDACB                    , vpminub(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(C5EDDA8C2B80000000          , vpminub(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F16D48DACB                , vpminub(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(62F16D48DA4C2B02            , vpminub(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F24D403BA200200000        , vpminud(zmm4, k0, zmm22, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62F245403B5A7F              , vpminud(zmm3, k0, zmm23, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62F245403B9A00200000        , vpminud(zmm3, k0, zmm23, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62F24D403BE3                , vpminud(zmm4, k0, zmm22, zmm3)),
        INSTRUCTION_TEST(62F24D403BA2C0DFFFFF        , vpminud(zmm4, k0, zmm22, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62F24D403B21                , vpminud(zmm4, k0, zmm22, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62F24D403B6280              , vpminud(zmm4, k0, zmm22, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62F245403B9AC0DFFFFF        , vpminud(zmm3, k0, zmm23, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62B24D403BA4F034120000      , vpminud(zmm4, k0, zmm22, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62F24D403B627F              , vpminud(zmm4, k0, zmm22, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62F245403B5A80              , vpminud(zmm3, k0, zmm23, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62B245403B9CF023010000      , vpminud(zmm3, k0, zmm23, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62B245403BDC                , vpminud(zmm3, k0, zmm23, zmm20)),
        INSTRUCTION_TEST(62F26D483B4C2B02            , vpminud(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F26D483BCB                , vpminud(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(C4E26D3B8C2B80000000        , vpminud(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E26D3BCB                  , vpminud(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(C4E2693B8C2B80000000        , vpminud(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E2693BCB                  , vpminud(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(62F245403B19                , vpminud(zmm3, k0, zmm23, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(6272AD403B9A00200000        , vpminuq(zmm11, k0, zmm26, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(6202AD403BCB                , vpminuq(zmm25, k0, zmm26, zmm27)),
        INSTRUCTION_TEST(6222AD403B8CF034120000      , vpminuq(zmm25, k0, zmm26, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(6272AD403B5A7F              , vpminuq(zmm11, k0, zmm26, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(6262AD403B4A80              , vpminuq(zmm25, k0, zmm26, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(6262AD403B8AC0DFFFFF        , vpminuq(zmm25, k0, zmm26, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(6262AD403B4A7F              , vpminuq(zmm25, k0, zmm26, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(6262AD403B8A00200000        , vpminuq(zmm25, k0, zmm26, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(6272AD403B9AC0DFFFFF        , vpminuq(zmm11, k0, zmm26, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(6262AD403B09                , vpminuq(zmm25, k0, zmm26, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(6272AD403B19                , vpminuq(zmm11, k0, zmm26, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(6272AD403BDF                , vpminuq(zmm11, k0, zmm26, zmm7)),
        INSTRUCTION_TEST(6232AD403B9CF023010000      , vpminuq(zmm11, k0, zmm26, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62F2ED483B4C2B02            , vpminuq(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F2ED483BCB                , vpminuq(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(62F2ED283B4C2B04            , vpminuq(ymm1, k0, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F2ED283BCB                , vpminuq(ymm1, k0, ymm2, ymm3)),
        INSTRUCTION_TEST(62F2ED083B4C2B08            , vpminuq(xmm1, k0, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F2ED083BCB                , vpminuq(xmm1, k0, xmm2, xmm3)),
        INSTRUCTION_TEST(6272AD403B5A80              , vpminuq(zmm11, k0, zmm26, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62F26D483ACB                , vpminuw(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(C4E26D3A8C2B80000000        , vpminuw(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F26D483A4C2B02            , vpminuw(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E2693A8C2B80000000        , vpminuw(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E2693ACB                  , vpminuw(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(C4E26D3ACB                  , vpminuw(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(62F27E0829CA                , vpmovb2m(k1, xmm2)),
        INSTRUCTION_TEST(62F27E2829CA                , vpmovb2m(k1, ymm2)),
        INSTRUCTION_TEST(62F27E4829CA                , vpmovb2m(k1, zmm2)),
        INSTRUCTION_TEST(62F27E0839CA                , vpmovd2m(k1, xmm2)),
        INSTRUCTION_TEST(62F27E2839CA                , vpmovd2m(k1, ymm2)),
        INSTRUCTION_TEST(62F27E4839CA                , vpmovd2m(k1, zmm2)),
        INSTRUCTION_TEST(62727E48316280              , vpmovdb(xmmword_ptr(rdx, -2048), k0, zmm12)),
        INSTRUCTION_TEST(62627E48316A80              , vpmovdb(xmmword_ptr(rdx, -2048), k0, zmm29)),
        INSTRUCTION_TEST(62727E4831A2F0F7FFFF        , vpmovdb(xmmword_ptr(rdx, -2064), k0, zmm12)),
        INSTRUCTION_TEST(62627E4831AAF0F7FFFF        , vpmovdb(xmmword_ptr(rdx, -2064), k0, zmm29)),
        INSTRUCTION_TEST(62627E4831AA00080000        , vpmovdb(xmmword_ptr(rdx, 2048), k0, zmm29)),
        INSTRUCTION_TEST(62627E48316A7F              , vpmovdb(xmmword_ptr(rdx, 2032), k0, zmm29)),
        INSTRUCTION_TEST(62727E4831A200080000        , vpmovdb(xmmword_ptr(rdx, 2048), k0, zmm12)),
        INSTRUCTION_TEST(62627E483129                , vpmovdb(xmmword_ptr(rcx), k0, zmm29)),
        INSTRUCTION_TEST(62727E4831627F              , vpmovdb(xmmword_ptr(rdx, 2032), k0, zmm12)),
        INSTRUCTION_TEST(62727E483121                , vpmovdb(xmmword_ptr(rcx), k0, zmm12)),
        INSTRUCTION_TEST(62F27E2831D1                , vpmovdb(xmm1, k0, ymm2)),
        INSTRUCTION_TEST(62227E4831ACF034120000      , vpmovdb(xmmword_ptr(rax, r14, 8, 4660), k0, zmm29)),
        INSTRUCTION_TEST(62327E4831A4F023010000      , vpmovdb(xmmword_ptr(rax, r14, 8, 291), k0, zmm12)),
        INSTRUCTION_TEST(62F27E4831EA                , vpmovdb(xmm2, k0, zmm5)),
        INSTRUCTION_TEST(62F27E4831D1                , vpmovdb(xmm1, k0, zmm2)),
        INSTRUCTION_TEST(62727E4831D9                , vpmovdb(xmm1, k0, zmm11)),
        INSTRUCTION_TEST(62F27E0831D1                , vpmovdb(xmm1, k0, xmm2)),
        INSTRUCTION_TEST(62F27E28315C1110            , vpmovdb(qword_ptr(rcx, rdx, 1, 128), k0, ymm3)),
        INSTRUCTION_TEST(62F27E08315C1120            , vpmovdb(dword_ptr(rcx, rdx, 1, 128), k0, xmm3)),
        INSTRUCTION_TEST(62F27E48315C1108            , vpmovdb(xmmword_ptr(rcx, rdx, 1, 128), k0, zmm3)),
        INSTRUCTION_TEST(62F27E48336A80              , vpmovdw(ymmword_ptr(rdx, -4096), k0, zmm5)),
        INSTRUCTION_TEST(62F27E48337A80              , vpmovdw(ymmword_ptr(rdx, -4096), k0, zmm7)),
        INSTRUCTION_TEST(62F27E4833AAE0EFFFFF        , vpmovdw(ymmword_ptr(rdx, -4128), k0, zmm5)),
        INSTRUCTION_TEST(62F27E4833BAE0EFFFFF        , vpmovdw(ymmword_ptr(rdx, -4128), k0, zmm7)),
        INSTRUCTION_TEST(62F27E4833BA00100000        , vpmovdw(ymmword_ptr(rdx, 4096), k0, zmm7)),
        INSTRUCTION_TEST(62F27E48337A7F              , vpmovdw(ymmword_ptr(rdx, 4064), k0, zmm7)),
        INSTRUCTION_TEST(62F27E4833AA00100000        , vpmovdw(ymmword_ptr(rdx, 4096), k0, zmm5)),
        INSTRUCTION_TEST(62F27E483339                , vpmovdw(ymmword_ptr(rcx), k0, zmm7)),
        INSTRUCTION_TEST(62F27E48336A7F              , vpmovdw(ymmword_ptr(rdx, 4064), k0, zmm5)),
        INSTRUCTION_TEST(62F27E483329                , vpmovdw(ymmword_ptr(rcx), k0, zmm5)),
        INSTRUCTION_TEST(62F27E2833D1                , vpmovdw(xmm1, k0, ymm2)),
        INSTRUCTION_TEST(62B27E4833ACF034120000      , vpmovdw(ymmword_ptr(rax, r14, 8, 4660), k0, zmm5)),
        INSTRUCTION_TEST(62B27E4833BCF023010000      , vpmovdw(ymmword_ptr(rax, r14, 8, 291), k0, zmm7)),
        INSTRUCTION_TEST(62527E4833D1                , vpmovdw(ymm9, k0, zmm10)),
        INSTRUCTION_TEST(62227E4833EE                , vpmovdw(ymm22, k0, zmm29)),
        INSTRUCTION_TEST(62F27E4833D1                , vpmovdw(ymm1, k0, zmm2)),
        INSTRUCTION_TEST(62F27E28335C1108            , vpmovdw(xmmword_ptr(rcx, rdx, 1, 128), k0, ymm3)),
        INSTRUCTION_TEST(62F27E48335C1104            , vpmovdw(ymmword_ptr(rcx, rdx, 1, 128), k0, zmm3)),
        INSTRUCTION_TEST(62F27E0833D1                , vpmovdw(xmm1, k0, xmm2)),
        INSTRUCTION_TEST(62F27E08335C1110            , vpmovdw(qword_ptr(rcx, rdx, 1, 128), k0, xmm3)),
        INSTRUCTION_TEST(62F27E0828CA                , vpmovm2b(xmm1, k2)),
        INSTRUCTION_TEST(62F27E2828CA                , vpmovm2b(ymm1, k2)),
        INSTRUCTION_TEST(62F27E4828CA                , vpmovm2b(zmm1, k2)),
        INSTRUCTION_TEST(62F27E0838CA                , vpmovm2d(xmm1, k2)),
        INSTRUCTION_TEST(62F27E2838CA                , vpmovm2d(ymm1, k2)),
        INSTRUCTION_TEST(62F27E4838CA                , vpmovm2d(zmm1, k2)),
        INSTRUCTION_TEST(62F2FE2838CA                , vpmovm2q(ymm1, k2)),
        INSTRUCTION_TEST(62F2FE4838CA                , vpmovm2q(zmm1, k2)),
        INSTRUCTION_TEST(62F2FE0838CA                , vpmovm2q(xmm1, k2)),
        INSTRUCTION_TEST(62F2FE0828CA                , vpmovm2w(xmm1, k2)),
        INSTRUCTION_TEST(62F2FE2828CA                , vpmovm2w(ymm1, k2)),
        INSTRUCTION_TEST(62F2FE4828CA                , vpmovm2w(zmm1, k2)),
        INSTRUCTION_TEST(C5F9D7CA                    , vpmovmskb(ecx, xmm2)),
        INSTRUCTION_TEST(C5FDD7CA                    , vpmovmskb(ecx, ymm2)),
        INSTRUCTION_TEST(62F2FE0839CA                , vpmovq2m(k1, xmm2)),
        INSTRUCTION_TEST(62F2FE2839CA                , vpmovq2m(k1, ymm2)),
        INSTRUCTION_TEST(62F2FE4839CA                , vpmovq2m(k1, zmm2)),
        INSTRUCTION_TEST(62F27E48325A7F              , vpmovqb(qword_ptr(rdx, 1016), k0, zmm3)),
        INSTRUCTION_TEST(62727E4832B200040000        , vpmovqb(qword_ptr(rdx, 1024), k0, zmm14)),
        INSTRUCTION_TEST(62F27E48329A00040000        , vpmovqb(qword_ptr(rdx, 1024), k0, zmm3)),
        INSTRUCTION_TEST(62F27E08325C1140            , vpmovqb(word_ptr(rcx, rdx, 1, 128), k0, xmm3)),
        INSTRUCTION_TEST(62327E4832F7                , vpmovqb(xmm23, k0, zmm14)),
        INSTRUCTION_TEST(62F27E0832D1                , vpmovqb(xmm1, k0, xmm2)),
        INSTRUCTION_TEST(62F27E2832D1                , vpmovqb(xmm1, k0, ymm2)),
        INSTRUCTION_TEST(62F27E4832D1                , vpmovqb(xmm1, k0, zmm2)),
        INSTRUCTION_TEST(62F27E4832D3                , vpmovqb(xmm3, k0, zmm2)),
        INSTRUCTION_TEST(62727E4832727F              , vpmovqb(qword_ptr(rdx, 1016), k0, zmm14)),
        INSTRUCTION_TEST(62727E483231                , vpmovqb(qword_ptr(rcx), k0, zmm14)),
        INSTRUCTION_TEST(62727E4832B2F8FBFFFF        , vpmovqb(qword_ptr(rdx, -1032), k0, zmm14)),
        INSTRUCTION_TEST(62F27E28325C1120            , vpmovqb(dword_ptr(rcx, rdx, 1, 128), k0, ymm3)),
        INSTRUCTION_TEST(62B27E48329CF023010000      , vpmovqb(qword_ptr(rax, r14, 8, 291), k0, zmm3)),
        INSTRUCTION_TEST(62327E4832B4F034120000      , vpmovqb(qword_ptr(rax, r14, 8, 4660), k0, zmm14)),
        INSTRUCTION_TEST(62F27E48329AF8FBFFFF        , vpmovqb(qword_ptr(rdx, -1032), k0, zmm3)),
        INSTRUCTION_TEST(62F27E483219                , vpmovqb(qword_ptr(rcx), k0, zmm3)),
        INSTRUCTION_TEST(62727E48327280              , vpmovqb(qword_ptr(rdx, -1024), k0, zmm14)),
        INSTRUCTION_TEST(62F27E48325A80              , vpmovqb(qword_ptr(rdx, -1024), k0, zmm3)),
        INSTRUCTION_TEST(62F27E48325C1110            , vpmovqb(qword_ptr(rcx, rdx, 1, 128), k0, zmm3)),
        INSTRUCTION_TEST(62627E4835A200100000        , vpmovqd(ymmword_ptr(rdx, 4096), k0, zmm28)),
        INSTRUCTION_TEST(62727E48359200100000        , vpmovqd(ymmword_ptr(rdx, 4096), k0, zmm10)),
        INSTRUCTION_TEST(62627E4835627F              , vpmovqd(ymmword_ptr(rdx, 4064), k0, zmm28)),
        INSTRUCTION_TEST(62727E4835527F              , vpmovqd(ymmword_ptr(rdx, 4064), k0, zmm10)),
        INSTRUCTION_TEST(62627E4835A2E0EFFFFF        , vpmovqd(ymmword_ptr(rdx, -4128), k0, zmm28)),
        INSTRUCTION_TEST(62727E483592E0EFFFFF        , vpmovqd(ymmword_ptr(rdx, -4128), k0, zmm10)),
        INSTRUCTION_TEST(62627E48356280              , vpmovqd(ymmword_ptr(rdx, -4096), k0, zmm28)),
        INSTRUCTION_TEST(62627E483521                , vpmovqd(ymmword_ptr(rcx), k0, zmm28)),
        INSTRUCTION_TEST(62F27E48355C1104            , vpmovqd(ymmword_ptr(rcx, rdx, 1, 128), k0, zmm3)),
        INSTRUCTION_TEST(62727E48355280              , vpmovqd(ymmword_ptr(rdx, -4096), k0, zmm10)),
        INSTRUCTION_TEST(62727E483511                , vpmovqd(ymmword_ptr(rcx), k0, zmm10)),
        INSTRUCTION_TEST(62227E4835A4F023010000      , vpmovqd(ymmword_ptr(rax, r14, 8, 291), k0, zmm28)),
        INSTRUCTION_TEST(62627E4835CE                , vpmovqd(ymm6, k0, zmm25)),
        INSTRUCTION_TEST(62027E4835EC                , vpmovqd(ymm28, k0, zmm29)),
        INSTRUCTION_TEST(62F27E4835D1                , vpmovqd(ymm1, k0, zmm2)),
        INSTRUCTION_TEST(62F27E28355C1108            , vpmovqd(xmmword_ptr(rcx, rdx, 1, 128), k0, ymm3)),
        INSTRUCTION_TEST(62F27E2835D1                , vpmovqd(xmm1, k0, ymm2)),
        INSTRUCTION_TEST(62F27E0835D1                , vpmovqd(xmm1, k0, xmm2)),
        INSTRUCTION_TEST(62F27E08355C1110            , vpmovqd(qword_ptr(rcx, rdx, 1, 128), k0, xmm3)),
        INSTRUCTION_TEST(62327E483594F034120000      , vpmovqd(ymmword_ptr(rax, r14, 8, 4660), k0, zmm10)),
        INSTRUCTION_TEST(62F27E483439                , vpmovqw(xmmword_ptr(rcx), k0, zmm7)),
        INSTRUCTION_TEST(62E27E48347280              , vpmovqw(xmmword_ptr(rdx, -2048), k0, zmm22)),
        INSTRUCTION_TEST(62F27E48347A80              , vpmovqw(xmmword_ptr(rdx, -2048), k0, zmm7)),
        INSTRUCTION_TEST(62E27E4834B2F0F7FFFF        , vpmovqw(xmmword_ptr(rdx, -2064), k0, zmm22)),
        INSTRUCTION_TEST(62F27E4834BA00080000        , vpmovqw(xmmword_ptr(rdx, 2048), k0, zmm7)),
        INSTRUCTION_TEST(62E27E4834727F              , vpmovqw(xmmword_ptr(rdx, 2032), k0, zmm22)),
        INSTRUCTION_TEST(62F27E48347A7F              , vpmovqw(xmmword_ptr(rdx, 2032), k0, zmm7)),
        INSTRUCTION_TEST(62E27E4834B200080000        , vpmovqw(xmmword_ptr(rdx, 2048), k0, zmm22)),
        INSTRUCTION_TEST(62F27E4834BAF0F7FFFF        , vpmovqw(xmmword_ptr(rdx, -2064), k0, zmm7)),
        INSTRUCTION_TEST(62E27E483431                , vpmovqw(xmmword_ptr(rcx), k0, zmm22)),
        INSTRUCTION_TEST(62F27E48345C1108            , vpmovqw(xmmword_ptr(rcx, rdx, 1, 128), k0, zmm3)),
        INSTRUCTION_TEST(62B27E4834BCF023010000      , vpmovqw(xmmword_ptr(rax, r14, 8, 291), k0, zmm7)),
        INSTRUCTION_TEST(62F27E08345C1120            , vpmovqw(dword_ptr(rcx, rdx, 1, 128), k0, xmm3)),
        INSTRUCTION_TEST(62F27E28345C1110            , vpmovqw(qword_ptr(rcx, rdx, 1, 128), k0, ymm3)),
        INSTRUCTION_TEST(62F27E0834D1                , vpmovqw(xmm1, k0, xmm2)),
        INSTRUCTION_TEST(62A27E4834B4F034120000      , vpmovqw(xmmword_ptr(rax, r14, 8, 4660), k0, zmm22)),
        INSTRUCTION_TEST(62F27E4834D1                , vpmovqw(xmm1, k0, zmm2)),
        INSTRUCTION_TEST(62A27E4834DD                , vpmovqw(xmm21, k0, zmm19)),
        INSTRUCTION_TEST(62E27E4834D6                , vpmovqw(xmm6, k0, zmm18)),
        INSTRUCTION_TEST(62F27E2834D1                , vpmovqw(xmm1, k0, ymm2)),
        INSTRUCTION_TEST(62F27E4821B200080000        , vpmovsdb(xmmword_ptr(rdx, 2048), k0, zmm6)),
        INSTRUCTION_TEST(62627E482121                , vpmovsdb(xmmword_ptr(rcx), k0, zmm28)),
        INSTRUCTION_TEST(62F27E4821727F              , vpmovsdb(xmmword_ptr(rdx, 2032), k0, zmm6)),
        INSTRUCTION_TEST(62627E4821627F              , vpmovsdb(xmmword_ptr(rdx, 2032), k0, zmm28)),
        INSTRUCTION_TEST(62F27E4821B2F0F7FFFF        , vpmovsdb(xmmword_ptr(rdx, -2064), k0, zmm6)),
        INSTRUCTION_TEST(62627E4821A2F0F7FFFF        , vpmovsdb(xmmword_ptr(rdx, -2064), k0, zmm28)),
        INSTRUCTION_TEST(62F27E48217280              , vpmovsdb(xmmword_ptr(rdx, -2048), k0, zmm6)),
        INSTRUCTION_TEST(62627E48216280              , vpmovsdb(xmmword_ptr(rdx, -2048), k0, zmm28)),
        INSTRUCTION_TEST(62F27E482131                , vpmovsdb(xmmword_ptr(rcx), k0, zmm6)),
        INSTRUCTION_TEST(62627E4821A200080000        , vpmovsdb(xmmword_ptr(rdx, 2048), k0, zmm28)),
        INSTRUCTION_TEST(62F27E48215C1108            , vpmovsdb(xmmword_ptr(rcx, rdx, 1, 128), k0, zmm3)),
        INSTRUCTION_TEST(62B27E4821B4F023010000      , vpmovsdb(xmmword_ptr(rax, r14, 8, 291), k0, zmm6)),
        INSTRUCTION_TEST(62927E4821CB                , vpmovsdb(xmm27, k0, zmm1)),
        INSTRUCTION_TEST(62B27E4821D5                , vpmovsdb(xmm21, k0, zmm2)),
        INSTRUCTION_TEST(62F27E4821D1                , vpmovsdb(xmm1, k0, zmm2)),
        INSTRUCTION_TEST(62F27E2821D1                , vpmovsdb(xmm1, k0, ymm2)),
        INSTRUCTION_TEST(62F27E0821D1                , vpmovsdb(xmm1, k0, xmm2)),
        INSTRUCTION_TEST(62F27E28215C1110            , vpmovsdb(qword_ptr(rcx, rdx, 1, 128), k0, ymm3)),
        INSTRUCTION_TEST(62F27E08215C1120            , vpmovsdb(dword_ptr(rcx, rdx, 1, 128), k0, xmm3)),
        INSTRUCTION_TEST(62227E4821A4F034120000      , vpmovsdb(xmmword_ptr(rax, r14, 8, 4660), k0, zmm28)),
        INSTRUCTION_TEST(62E27E48239200100000        , vpmovsdw(ymmword_ptr(rdx, 4096), k0, zmm18)),
        INSTRUCTION_TEST(62E27E48235280              , vpmovsdw(ymmword_ptr(rdx, -4096), k0, zmm18)),
        INSTRUCTION_TEST(62727E4823B2E0EFFFFF        , vpmovsdw(ymmword_ptr(rdx, -4128), k0, zmm14)),
        INSTRUCTION_TEST(62E27E482311                , vpmovsdw(ymmword_ptr(rcx), k0, zmm18)),
        INSTRUCTION_TEST(62727E4823727F              , vpmovsdw(ymmword_ptr(rdx, 4064), k0, zmm14)),
        INSTRUCTION_TEST(62E27E4823527F              , vpmovsdw(ymmword_ptr(rdx, 4064), k0, zmm18)),
        INSTRUCTION_TEST(62727E4823B200100000        , vpmovsdw(ymmword_ptr(rdx, 4096), k0, zmm14)),
        INSTRUCTION_TEST(62E27E482392E0EFFFFF        , vpmovsdw(ymmword_ptr(rdx, -4128), k0, zmm18)),
        INSTRUCTION_TEST(62727E482331                , vpmovsdw(ymmword_ptr(rcx), k0, zmm14)),
        INSTRUCTION_TEST(62727E48237280              , vpmovsdw(ymmword_ptr(rdx, -4096), k0, zmm14)),
        INSTRUCTION_TEST(62A27E482394F034120000      , vpmovsdw(ymmword_ptr(rax, r14, 8, 4660), k0, zmm18)),
        INSTRUCTION_TEST(62F27E08235C1110            , vpmovsdw(qword_ptr(rcx, rdx, 1, 128), k0, xmm3)),
        INSTRUCTION_TEST(62F27E0823D1                , vpmovsdw(xmm1, k0, xmm2)),
        INSTRUCTION_TEST(62F27E2823D1                , vpmovsdw(xmm1, k0, ymm2)),
        INSTRUCTION_TEST(62F27E28235C1108            , vpmovsdw(xmmword_ptr(rcx, rdx, 1, 128), k0, ymm3)),
        INSTRUCTION_TEST(62F27E48235C1104            , vpmovsdw(ymmword_ptr(rcx, rdx, 1, 128), k0, zmm3)),
        INSTRUCTION_TEST(62027E4823C0                , vpmovsdw(ymm24, k0, zmm24)),
        INSTRUCTION_TEST(62127E4823F1                , vpmovsdw(ymm25, k0, zmm14)),
        INSTRUCTION_TEST(62327E4823B4F023010000      , vpmovsdw(ymmword_ptr(rax, r14, 8, 291), k0, zmm14)),
        INSTRUCTION_TEST(62F27E4823D1                , vpmovsdw(ymm1, k0, zmm2)),
        INSTRUCTION_TEST(62E27E4822527F              , vpmovsqb(qword_ptr(rdx, 1016), k0, zmm18)),
        INSTRUCTION_TEST(62027E4822EE                , vpmovsqb(xmm30, k0, zmm29)),
        INSTRUCTION_TEST(62227E4822CC                , vpmovsqb(xmm20, k0, zmm25)),
        INSTRUCTION_TEST(62F27E4822D1                , vpmovsqb(xmm1, k0, zmm2)),
        INSTRUCTION_TEST(62F27E2822D1                , vpmovsqb(xmm1, k0, ymm2)),
        INSTRUCTION_TEST(62F27E0822D1                , vpmovsqb(xmm1, k0, xmm2)),
        INSTRUCTION_TEST(62F27E08225C1140            , vpmovsqb(word_ptr(rcx, rdx, 1, 128), k0, xmm3)),
        INSTRUCTION_TEST(62E27E48229200040000        , vpmovsqb(qword_ptr(rdx, 1024), k0, zmm18)),
        INSTRUCTION_TEST(62E27E48228200040000        , vpmovsqb(qword_ptr(rdx, 1024), k0, zmm16)),
        INSTRUCTION_TEST(62E27E4822427F              , vpmovsqb(qword_ptr(rdx, 1016), k0, zmm16)),
        INSTRUCTION_TEST(62E27E482292F8FBFFFF        , vpmovsqb(qword_ptr(rdx, -1032), k0, zmm18)),
        INSTRUCTION_TEST(62E27E482282F8FBFFFF        , vpmovsqb(qword_ptr(rdx, -1032), k0, zmm16)),
        INSTRUCTION_TEST(62E27E48225280              , vpmovsqb(qword_ptr(rdx, -1024), k0, zmm18)),
        INSTRUCTION_TEST(62E27E48224280              , vpmovsqb(qword_ptr(rdx, -1024), k0, zmm16)),
        INSTRUCTION_TEST(62E27E482211                , vpmovsqb(qword_ptr(rcx), k0, zmm18)),
        INSTRUCTION_TEST(62E27E482201                , vpmovsqb(qword_ptr(rcx), k0, zmm16)),
        INSTRUCTION_TEST(62F27E48225C1110            , vpmovsqb(qword_ptr(rcx, rdx, 1, 128), k0, zmm3)),
        INSTRUCTION_TEST(62A27E482294F034120000      , vpmovsqb(qword_ptr(rax, r14, 8, 4660), k0, zmm18)),
        INSTRUCTION_TEST(62A27E482284F023010000      , vpmovsqb(qword_ptr(rax, r14, 8, 291), k0, zmm16)),
        INSTRUCTION_TEST(62F27E28225C1120            , vpmovsqb(dword_ptr(rcx, rdx, 1, 128), k0, ymm3)),
        INSTRUCTION_TEST(62E27E48255280              , vpmovsqd(ymmword_ptr(rdx, -4096), k0, zmm18)),
        INSTRUCTION_TEST(62727E48254A80              , vpmovsqd(ymmword_ptr(rdx, -4096), k0, zmm9)),
        INSTRUCTION_TEST(62E27E482592E0EFFFFF        , vpmovsqd(ymmword_ptr(rdx, -4128), k0, zmm18)),
        INSTRUCTION_TEST(62727E48258AE0EFFFFF        , vpmovsqd(ymmword_ptr(rdx, -4128), k0, zmm9)),
        INSTRUCTION_TEST(62727E48258A00100000        , vpmovsqd(ymmword_ptr(rdx, 4096), k0, zmm9)),
        INSTRUCTION_TEST(62727E48254A7F              , vpmovsqd(ymmword_ptr(rdx, 4064), k0, zmm9)),
        INSTRUCTION_TEST(62E27E48259200100000        , vpmovsqd(ymmword_ptr(rdx, 4096), k0, zmm18)),
        INSTRUCTION_TEST(62727E482509                , vpmovsqd(ymmword_ptr(rcx), k0, zmm9)),
        INSTRUCTION_TEST(62E27E4825527F              , vpmovsqd(ymmword_ptr(rdx, 4064), k0, zmm18)),
        INSTRUCTION_TEST(62E27E482511                , vpmovsqd(ymmword_ptr(rcx), k0, zmm18)),
        INSTRUCTION_TEST(62D27E4825D7                , vpmovsqd(ymm15, k0, zmm2)),
        INSTRUCTION_TEST(62A27E482594F034120000      , vpmovsqd(ymmword_ptr(rax, r14, 8, 4660), k0, zmm18)),
        INSTRUCTION_TEST(62327E48258CF023010000      , vpmovsqd(ymmword_ptr(rax, r14, 8, 291), k0, zmm9)),
        INSTRUCTION_TEST(62327E4825CE                , vpmovsqd(ymm22, k0, zmm9)),
        INSTRUCTION_TEST(62F27E4825D1                , vpmovsqd(ymm1, k0, zmm2)),
        INSTRUCTION_TEST(62F27E28255C1108            , vpmovsqd(xmmword_ptr(rcx, rdx, 1, 128), k0, ymm3)),
        INSTRUCTION_TEST(62F27E2825D1                , vpmovsqd(xmm1, k0, ymm2)),
        INSTRUCTION_TEST(62F27E0825D1                , vpmovsqd(xmm1, k0, xmm2)),
        INSTRUCTION_TEST(62F27E08255C1110            , vpmovsqd(qword_ptr(rcx, rdx, 1, 128), k0, xmm3)),
        INSTRUCTION_TEST(62F27E48255C1104            , vpmovsqd(ymmword_ptr(rcx, rdx, 1, 128), k0, zmm3)),
        INSTRUCTION_TEST(62F27E48244A80              , vpmovsqw(xmmword_ptr(rdx, -2048), k0, zmm1)),
        INSTRUCTION_TEST(62627E48245280              , vpmovsqw(xmmword_ptr(rdx, -2048), k0, zmm26)),
        INSTRUCTION_TEST(62F27E48248AF0F7FFFF        , vpmovsqw(xmmword_ptr(rdx, -2064), k0, zmm1)),
        INSTRUCTION_TEST(62F27E48248A00080000        , vpmovsqw(xmmword_ptr(rdx, 2048), k0, zmm1)),
        INSTRUCTION_TEST(62F27E48244A7F              , vpmovsqw(xmmword_ptr(rdx, 2032), k0, zmm1)),
        INSTRUCTION_TEST(62627E4824527F              , vpmovsqw(xmmword_ptr(rdx, 2032), k0, zmm26)),
        INSTRUCTION_TEST(62627E482411                , vpmovsqw(xmmword_ptr(rcx), k0, zmm26)),
        INSTRUCTION_TEST(62627E48249200080000        , vpmovsqw(xmmword_ptr(rdx, 2048), k0, zmm26)),
        INSTRUCTION_TEST(62627E482492F0F7FFFF        , vpmovsqw(xmmword_ptr(rdx, -2064), k0, zmm26)),
        INSTRUCTION_TEST(62F27E482409                , vpmovsqw(xmmword_ptr(rcx), k0, zmm1)),
        INSTRUCTION_TEST(62827E4824DB                , vpmovsqw(xmm27, k0, zmm19)),
        INSTRUCTION_TEST(62227E482494F034120000      , vpmovsqw(xmmword_ptr(rax, r14, 8, 4660), k0, zmm26)),
        INSTRUCTION_TEST(62B27E48248CF023010000      , vpmovsqw(xmmword_ptr(rax, r14, 8, 291), k0, zmm1)),
        INSTRUCTION_TEST(62B27E4824E0                , vpmovsqw(xmm16, k0, zmm4)),
        INSTRUCTION_TEST(62F27E4824D1                , vpmovsqw(xmm1, k0, zmm2)),
        INSTRUCTION_TEST(62F27E2824D1                , vpmovsqw(xmm1, k0, ymm2)),
        INSTRUCTION_TEST(62F27E0824D1                , vpmovsqw(xmm1, k0, xmm2)),
        INSTRUCTION_TEST(62F27E28245C1110            , vpmovsqw(qword_ptr(rcx, rdx, 1, 128), k0, ymm3)),
        INSTRUCTION_TEST(62F27E08245C1120            , vpmovsqw(dword_ptr(rcx, rdx, 1, 128), k0, xmm3)),
        INSTRUCTION_TEST(62F27E48245C1108            , vpmovsqw(xmmword_ptr(rcx, rdx, 1, 128), k0, zmm3)),
        INSTRUCTION_TEST(62F27E4820D1                , vpmovswb(ymm1, k0, zmm2)),
        INSTRUCTION_TEST(62F27E28205C1108            , vpmovswb(xmmword_ptr(rcx, rdx, 1, 128), k0, ymm3)),
        INSTRUCTION_TEST(62F27E48205C1104            , vpmovswb(ymmword_ptr(rcx, rdx, 1, 128), k0, zmm3)),
        INSTRUCTION_TEST(62F27E0820D1                , vpmovswb(xmm1, k0, xmm2)),
        INSTRUCTION_TEST(62F27E08205C1110            , vpmovswb(qword_ptr(rcx, rdx, 1, 128), k0, xmm3)),
        INSTRUCTION_TEST(62F27E2820D1                , vpmovswb(xmm1, k0, ymm2)),
        INSTRUCTION_TEST(62627D48219A00080000        , vpmovsxbd(zmm27, k0, xmmword_ptr(rdx, 2048))),
        INSTRUCTION_TEST(62627D48215A7F              , vpmovsxbd(zmm27, k0, xmmword_ptr(rdx, 2032))),
        INSTRUCTION_TEST(62627D48219AF0F7FFFF        , vpmovsxbd(zmm27, k0, xmmword_ptr(rdx, -2064))),
        INSTRUCTION_TEST(62627D48215A80              , vpmovsxbd(zmm27, k0, xmmword_ptr(rdx, -2048))),
        INSTRUCTION_TEST(62627D482119                , vpmovsxbd(zmm27, k0, xmmword_ptr(rcx))),
        INSTRUCTION_TEST(62627D4821DF                , vpmovsxbd(zmm27, k0, xmm7)),
        INSTRUCTION_TEST(62227D48219CF023010000      , vpmovsxbd(zmm27, k0, xmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62627D48214A7F              , vpmovsxbd(zmm25, k0, xmmword_ptr(rdx, 2032))),
        INSTRUCTION_TEST(62627D48218AF0F7FFFF        , vpmovsxbd(zmm25, k0, xmmword_ptr(rdx, -2064))),
        INSTRUCTION_TEST(62627D48218A00080000        , vpmovsxbd(zmm25, k0, xmmword_ptr(rdx, 2048))),
        INSTRUCTION_TEST(62627D482109                , vpmovsxbd(zmm25, k0, xmmword_ptr(rcx))),
        INSTRUCTION_TEST(62627D48214A80              , vpmovsxbd(zmm25, k0, xmmword_ptr(rdx, -2048))),
        INSTRUCTION_TEST(C4E27921CA                  , vpmovsxbd(xmm1, xmm2)),
        INSTRUCTION_TEST(C4E27D218C1A80000000        , vpmovsxbd(ymm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(C4E27D21CA                  , vpmovsxbd(ymm1, xmm2)),
        INSTRUCTION_TEST(C4E279218C1A80000000        , vpmovsxbd(xmm1, dword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F27D48214C1A08            , vpmovsxbd(zmm1, k0, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62427D4821CA                , vpmovsxbd(zmm25, k0, xmm10)),
        INSTRUCTION_TEST(62227D48218CF034120000      , vpmovsxbd(zmm25, k0, xmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62F27D4821CA                , vpmovsxbd(zmm1, k0, xmm2)),
        INSTRUCTION_TEST(62727D48225A7F              , vpmovsxbq(zmm11, k0, qword_ptr(rdx, 1016))),
        INSTRUCTION_TEST(62727D4822727F              , vpmovsxbq(zmm14, k0, qword_ptr(rdx, 1016))),
        INSTRUCTION_TEST(62727D4822B2F8FBFFFF        , vpmovsxbq(zmm14, k0, qword_ptr(rdx, -1032))),
        INSTRUCTION_TEST(62727D4822B200040000        , vpmovsxbq(zmm14, k0, qword_ptr(rdx, 1024))),
        INSTRUCTION_TEST(62727D482231                , vpmovsxbq(zmm14, k0, qword_ptr(rcx))),
        INSTRUCTION_TEST(62527D4822F6                , vpmovsxbq(zmm14, k0, xmm14)),
        INSTRUCTION_TEST(62327D4822B4F034120000      , vpmovsxbq(zmm14, k0, qword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62727D48229A00040000        , vpmovsxbq(zmm11, k0, qword_ptr(rdx, 1024))),
        INSTRUCTION_TEST(62727D48229AF8FBFFFF        , vpmovsxbq(zmm11, k0, qword_ptr(rdx, -1032))),
        INSTRUCTION_TEST(62727D48227280              , vpmovsxbq(zmm14, k0, qword_ptr(rdx, -1024))),
        INSTRUCTION_TEST(62727D482219                , vpmovsxbq(zmm11, k0, qword_ptr(rcx))),
        INSTRUCTION_TEST(62527D4822DB                , vpmovsxbq(zmm11, k0, xmm11)),
        INSTRUCTION_TEST(62327D48229CF023010000      , vpmovsxbq(zmm11, k0, qword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62F27D4822CA                , vpmovsxbq(zmm1, k0, xmm2)),
        INSTRUCTION_TEST(62F27D48224C1A10            , vpmovsxbq(zmm1, k0, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(C4E27D22CA                  , vpmovsxbq(ymm1, xmm2)),
        INSTRUCTION_TEST(C4E27D228C1A80000000        , vpmovsxbq(ymm1, dword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(C4E27922CA                  , vpmovsxbq(xmm1, xmm2)),
        INSTRUCTION_TEST(C4E279228C1A80000000        , vpmovsxbq(xmm1, word_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62727D48225A80              , vpmovsxbq(zmm11, k0, qword_ptr(rdx, -1024))),
        INSTRUCTION_TEST(62F27D48204C1A04            , vpmovsxbw(zmm1, k0, ymmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F27D4820CA                , vpmovsxbw(zmm1, k0, ymm2)),
        INSTRUCTION_TEST(C4E27D208C1A80000000        , vpmovsxbw(ymm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(C4E27D20CA                  , vpmovsxbw(ymm1, xmm2)),
        INSTRUCTION_TEST(C4E27920CA                  , vpmovsxbw(xmm1, xmm2)),
        INSTRUCTION_TEST(C4E279208C1A80000000        , vpmovsxbw(xmm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62627D4825527F              , vpmovsxdq(zmm26, k0, ymmword_ptr(rdx, 4064))),
        INSTRUCTION_TEST(62627D482592E0EFFFFF        , vpmovsxdq(zmm26, k0, ymmword_ptr(rdx, -4128))),
        INSTRUCTION_TEST(62627D48255280              , vpmovsxdq(zmm26, k0, ymmword_ptr(rdx, -4096))),
        INSTRUCTION_TEST(62627D482511                , vpmovsxdq(zmm26, k0, ymmword_ptr(rcx))),
        INSTRUCTION_TEST(62227D482594F023010000      , vpmovsxdq(zmm26, k0, ymmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62027D4825D5                , vpmovsxdq(zmm26, k0, ymm29)),
        INSTRUCTION_TEST(62627D48259200100000        , vpmovsxdq(zmm26, k0, ymmword_ptr(rdx, 4096))),
        INSTRUCTION_TEST(62727D48259A00100000        , vpmovsxdq(zmm11, k0, ymmword_ptr(rdx, 4096))),
        INSTRUCTION_TEST(62727D48255A7F              , vpmovsxdq(zmm11, k0, ymmword_ptr(rdx, 4064))),
        INSTRUCTION_TEST(62727D48259AE0EFFFFF        , vpmovsxdq(zmm11, k0, ymmword_ptr(rdx, -4128))),
        INSTRUCTION_TEST(62527D4825D9                , vpmovsxdq(zmm11, k0, ymm9)),
        INSTRUCTION_TEST(62727D482519                , vpmovsxdq(zmm11, k0, ymmword_ptr(rcx))),
        INSTRUCTION_TEST(62327D48259CF034120000      , vpmovsxdq(zmm11, k0, ymmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62F27D48254C1A04            , vpmovsxdq(zmm1, k0, ymmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F27D4825CA                , vpmovsxdq(zmm1, k0, ymm2)),
        INSTRUCTION_TEST(C4E27D258C1A80000000        , vpmovsxdq(ymm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(C4E27D25CA                  , vpmovsxdq(ymm1, xmm2)),
        INSTRUCTION_TEST(C4E27925CA                  , vpmovsxdq(xmm1, xmm2)),
        INSTRUCTION_TEST(C4E279258C1A80000000        , vpmovsxdq(xmm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62727D48255A80              , vpmovsxdq(zmm11, k0, ymmword_ptr(rdx, -4096))),
        INSTRUCTION_TEST(62727D4823A200100000        , vpmovsxwd(zmm12, k0, ymmword_ptr(rdx, 4096))),
        INSTRUCTION_TEST(62C27D4823FB                , vpmovsxwd(zmm23, k0, ymm11)),
        INSTRUCTION_TEST(62A27D4823BCF023010000      , vpmovsxwd(zmm23, k0, ymmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62E27D482339                , vpmovsxwd(zmm23, k0, ymmword_ptr(rcx))),
        INSTRUCTION_TEST(62E27D4823BA00100000        , vpmovsxwd(zmm23, k0, ymmword_ptr(rdx, 4096))),
        INSTRUCTION_TEST(62E27D4823BAE0EFFFFF        , vpmovsxwd(zmm23, k0, ymmword_ptr(rdx, -4128))),
        INSTRUCTION_TEST(62E27D48237A7F              , vpmovsxwd(zmm23, k0, ymmword_ptr(rdx, 4064))),
        INSTRUCTION_TEST(62727D4823627F              , vpmovsxwd(zmm12, k0, ymmword_ptr(rdx, 4064))),
        INSTRUCTION_TEST(62E27D48237A80              , vpmovsxwd(zmm23, k0, ymmword_ptr(rdx, -4096))),
        INSTRUCTION_TEST(62727D4823A2E0EFFFFF        , vpmovsxwd(zmm12, k0, ymmword_ptr(rdx, -4128))),
        INSTRUCTION_TEST(62727D482321                , vpmovsxwd(zmm12, k0, ymmword_ptr(rcx))),
        INSTRUCTION_TEST(62727D48236280              , vpmovsxwd(zmm12, k0, ymmword_ptr(rdx, -4096))),
        INSTRUCTION_TEST(C4E27923CA                  , vpmovsxwd(xmm1, xmm2)),
        INSTRUCTION_TEST(C4E27D23CA                  , vpmovsxwd(ymm1, xmm2)),
        INSTRUCTION_TEST(C4E27D238C1A80000000        , vpmovsxwd(ymm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(C4E279238C1A80000000        , vpmovsxwd(xmm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F27D48234C1A04            , vpmovsxwd(zmm1, k0, ymmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62527D4823E3                , vpmovsxwd(zmm12, k0, ymm11)),
        INSTRUCTION_TEST(62327D4823A4F034120000      , vpmovsxwd(zmm12, k0, ymmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62F27D4823CA                , vpmovsxwd(zmm1, k0, ymm2)),
        INSTRUCTION_TEST(62727D48249A00080000        , vpmovsxwq(zmm11, k0, xmmword_ptr(rdx, 2048))),
        INSTRUCTION_TEST(62627D48248A00080000        , vpmovsxwq(zmm25, k0, xmmword_ptr(rdx, 2048))),
        INSTRUCTION_TEST(62627D48244A7F              , vpmovsxwq(zmm25, k0, xmmword_ptr(rdx, 2032))),
        INSTRUCTION_TEST(62627D48248AF0F7FFFF        , vpmovsxwq(zmm25, k0, xmmword_ptr(rdx, -2064))),
        INSTRUCTION_TEST(62627D48244A80              , vpmovsxwq(zmm25, k0, xmmword_ptr(rdx, -2048))),
        INSTRUCTION_TEST(62627D482409                , vpmovsxwq(zmm25, k0, xmmword_ptr(rcx))),
        INSTRUCTION_TEST(62227D48248CF023010000      , vpmovsxwq(zmm25, k0, xmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62027D4824C9                , vpmovsxwq(zmm25, k0, xmm25)),
        INSTRUCTION_TEST(62727D48245A7F              , vpmovsxwq(zmm11, k0, xmmword_ptr(rdx, 2032))),
        INSTRUCTION_TEST(62727D48249AF0F7FFFF        , vpmovsxwq(zmm11, k0, xmmword_ptr(rdx, -2064))),
        INSTRUCTION_TEST(C4E279248C1A80000000        , vpmovsxwq(xmm1, dword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62727D482419                , vpmovsxwq(zmm11, k0, xmmword_ptr(rcx))),
        INSTRUCTION_TEST(62327D48249CF034120000      , vpmovsxwq(zmm11, k0, xmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62527D4824DE                , vpmovsxwq(zmm11, k0, xmm14)),
        INSTRUCTION_TEST(62F27D48244C1A08            , vpmovsxwq(zmm1, k0, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F27D4824CA                , vpmovsxwq(zmm1, k0, xmm2)),
        INSTRUCTION_TEST(C4E27D24CA                  , vpmovsxwq(ymm1, xmm2)),
        INSTRUCTION_TEST(C4E27D248C1A80000000        , vpmovsxwq(ymm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(C4E27924CA                  , vpmovsxwq(xmm1, xmm2)),
        INSTRUCTION_TEST(62727D48245A80              , vpmovsxwq(zmm11, k0, xmmword_ptr(rdx, -2048))),
        INSTRUCTION_TEST(62E27E48117A80              , vpmovusdb(xmmword_ptr(rdx, -2048), k0, zmm23)),
        INSTRUCTION_TEST(62627E48117280              , vpmovusdb(xmmword_ptr(rdx, -2048), k0, zmm30)),
        INSTRUCTION_TEST(62E27E4811BAF0F7FFFF        , vpmovusdb(xmmword_ptr(rdx, -2064), k0, zmm23)),
        INSTRUCTION_TEST(62627E4811B2F0F7FFFF        , vpmovusdb(xmmword_ptr(rdx, -2064), k0, zmm30)),
        INSTRUCTION_TEST(62627E481131                , vpmovusdb(xmmword_ptr(rcx), k0, zmm30)),
        INSTRUCTION_TEST(62627E4811727F              , vpmovusdb(xmmword_ptr(rdx, 2032), k0, zmm30)),
        INSTRUCTION_TEST(62E27E4811BA00080000        , vpmovusdb(xmmword_ptr(rdx, 2048), k0, zmm23)),
        INSTRUCTION_TEST(62627E4811B200080000        , vpmovusdb(xmmword_ptr(rdx, 2048), k0, zmm30)),
        INSTRUCTION_TEST(62E27E48117A7F              , vpmovusdb(xmmword_ptr(rdx, 2032), k0, zmm23)),
        INSTRUCTION_TEST(62E27E481139                , vpmovusdb(xmmword_ptr(rcx), k0, zmm23)),
        INSTRUCTION_TEST(62A27E4811BCF023010000      , vpmovusdb(xmmword_ptr(rax, r14, 8, 291), k0, zmm23)),
        INSTRUCTION_TEST(62227E4811B4F034120000      , vpmovusdb(xmmword_ptr(rax, r14, 8, 4660), k0, zmm30)),
        INSTRUCTION_TEST(62E27E4811DB                , vpmovusdb(xmm3, k0, zmm19)),
        INSTRUCTION_TEST(62B27E4811D4                , vpmovusdb(xmm20, k0, zmm2)),
        INSTRUCTION_TEST(62F27E4811D1                , vpmovusdb(xmm1, k0, zmm2)),
        INSTRUCTION_TEST(62F27E2811D1                , vpmovusdb(xmm1, k0, ymm2)),
        INSTRUCTION_TEST(62F27E0811D1                , vpmovusdb(xmm1, k0, xmm2)),
        INSTRUCTION_TEST(62F27E28115C1110            , vpmovusdb(qword_ptr(rcx, rdx, 1, 128), k0, ymm3)),
        INSTRUCTION_TEST(62F27E08115C1120            , vpmovusdb(dword_ptr(rcx, rdx, 1, 128), k0, xmm3)),
        INSTRUCTION_TEST(62F27E48115C1108            , vpmovusdb(xmmword_ptr(rcx, rdx, 1, 128), k0, zmm3)),
        INSTRUCTION_TEST(62F27E4813AA00100000        , vpmovusdw(ymmword_ptr(rdx, 4096), k0, zmm5)),
        INSTRUCTION_TEST(62F27E48136A7F              , vpmovusdw(ymmword_ptr(rdx, 4064), k0, zmm5)),
        INSTRUCTION_TEST(62F27E4813AAE0EFFFFF        , vpmovusdw(ymmword_ptr(rdx, -4128), k0, zmm5)),
        INSTRUCTION_TEST(62F27E48136A80              , vpmovusdw(ymmword_ptr(rdx, -4096), k0, zmm5)),
        INSTRUCTION_TEST(62F27E481329                , vpmovusdw(ymmword_ptr(rcx), k0, zmm5)),
        INSTRUCTION_TEST(62F27E48135C1104            , vpmovusdw(ymmword_ptr(rcx, rdx, 1, 128), k0, zmm3)),
        INSTRUCTION_TEST(62B27E4813ACF034120000      , vpmovusdw(ymmword_ptr(rax, r14, 8, 4660), k0, zmm5)),
        INSTRUCTION_TEST(62B27E4813ACF023010000      , vpmovusdw(ymmword_ptr(rax, r14, 8, 291), k0, zmm5)),
        INSTRUCTION_TEST(62C27E4813CF                , vpmovusdw(ymm15, k0, zmm17)),
        INSTRUCTION_TEST(62F27E4813D1                , vpmovusdw(ymm1, k0, zmm2)),
        INSTRUCTION_TEST(62F27E28135C1108            , vpmovusdw(xmmword_ptr(rcx, rdx, 1, 128), k0, ymm3)),
        INSTRUCTION_TEST(62F27E2813D1                , vpmovusdw(xmm1, k0, ymm2)),
        INSTRUCTION_TEST(62F27E0813D1                , vpmovusdw(xmm1, k0, xmm2)),
        INSTRUCTION_TEST(62F27E08135C1110            , vpmovusdw(qword_ptr(rcx, rdx, 1, 128), k0, xmm3)),
        INSTRUCTION_TEST(62D27E4813F8                , vpmovusdw(ymm8, k0, zmm7)),
        INSTRUCTION_TEST(62727E4812AA00040000        , vpmovusqb(qword_ptr(rdx, 1024), k0, zmm13)),
        INSTRUCTION_TEST(62627E4812A200040000        , vpmovusqb(qword_ptr(rdx, 1024), k0, zmm28)),
        INSTRUCTION_TEST(62F27E08125C1140            , vpmovusqb(word_ptr(rcx, rdx, 1, 128), k0, xmm3)),
        INSTRUCTION_TEST(62F27E0812D1                , vpmovusqb(xmm1, k0, xmm2)),
        INSTRUCTION_TEST(62027E4812E0                , vpmovusqb(xmm24, k0, zmm28)),
        INSTRUCTION_TEST(62F27E4812D1                , vpmovusqb(xmm1, k0, zmm2)),
        INSTRUCTION_TEST(62D27E4812D0                , vpmovusqb(xmm8, k0, zmm2)),
        INSTRUCTION_TEST(62627E4812627F              , vpmovusqb(qword_ptr(rdx, 1016), k0, zmm28)),
        INSTRUCTION_TEST(62F27E2812D1                , vpmovusqb(xmm1, k0, ymm2)),
        INSTRUCTION_TEST(62627E4812A2F8FBFFFF        , vpmovusqb(qword_ptr(rdx, -1032), k0, zmm28)),
        INSTRUCTION_TEST(62727E48126A80              , vpmovusqb(qword_ptr(rdx, -1024), k0, zmm13)),
        INSTRUCTION_TEST(62727E4812AAF8FBFFFF        , vpmovusqb(qword_ptr(rdx, -1032), k0, zmm13)),
        INSTRUCTION_TEST(62227E4812A4F023010000      , vpmovusqb(qword_ptr(rax, r14, 8, 291), k0, zmm28)),
        INSTRUCTION_TEST(62327E4812ACF034120000      , vpmovusqb(qword_ptr(rax, r14, 8, 4660), k0, zmm13)),
        INSTRUCTION_TEST(62F27E48125C1110            , vpmovusqb(qword_ptr(rcx, rdx, 1, 128), k0, zmm3)),
        INSTRUCTION_TEST(62F27E28125C1120            , vpmovusqb(dword_ptr(rcx, rdx, 1, 128), k0, ymm3)),
        INSTRUCTION_TEST(62627E481221                , vpmovusqb(qword_ptr(rcx), k0, zmm28)),
        INSTRUCTION_TEST(62727E48126A7F              , vpmovusqb(qword_ptr(rdx, 1016), k0, zmm13)),
        INSTRUCTION_TEST(62627E48126280              , vpmovusqb(qword_ptr(rdx, -1024), k0, zmm28)),
        INSTRUCTION_TEST(62727E481229                , vpmovusqb(qword_ptr(rcx), k0, zmm13)),
        INSTRUCTION_TEST(62E27E48154A80              , vpmovusqd(ymmword_ptr(rdx, -4096), k0, zmm17)),
        INSTRUCTION_TEST(62E27E4815B200100000        , vpmovusqd(ymmword_ptr(rdx, 4096), k0, zmm22)),
        INSTRUCTION_TEST(62E27E48158A00100000        , vpmovusqd(ymmword_ptr(rdx, 4096), k0, zmm17)),
        INSTRUCTION_TEST(62E27E4815727F              , vpmovusqd(ymmword_ptr(rdx, 4064), k0, zmm22)),
        INSTRUCTION_TEST(62E27E48154A7F              , vpmovusqd(ymmword_ptr(rdx, 4064), k0, zmm17)),
        INSTRUCTION_TEST(62E27E4815B2E0EFFFFF        , vpmovusqd(ymmword_ptr(rdx, -4128), k0, zmm22)),
        INSTRUCTION_TEST(62E27E48158AE0EFFFFF        , vpmovusqd(ymmword_ptr(rdx, -4128), k0, zmm17)),
        INSTRUCTION_TEST(62E27E48157280              , vpmovusqd(ymmword_ptr(rdx, -4096), k0, zmm22)),
        INSTRUCTION_TEST(62E27E481531                , vpmovusqd(ymmword_ptr(rcx), k0, zmm22)),
        INSTRUCTION_TEST(62E27E481509                , vpmovusqd(ymmword_ptr(rcx), k0, zmm17)),
        INSTRUCTION_TEST(62F27E08155C1110            , vpmovusqd(qword_ptr(rcx, rdx, 1, 128), k0, xmm3)),
        INSTRUCTION_TEST(62A27E48158CF034120000      , vpmovusqd(ymmword_ptr(rax, r14, 8, 4660), k0, zmm17)),
        INSTRUCTION_TEST(62A27E4815B4F023010000      , vpmovusqd(ymmword_ptr(rax, r14, 8, 291), k0, zmm22)),
        INSTRUCTION_TEST(62D27E4815E0                , vpmovusqd(ymm8, k0, zmm4)),
        INSTRUCTION_TEST(62627E4815D2                , vpmovusqd(ymm2, k0, zmm26)),
        INSTRUCTION_TEST(62F27E4815D1                , vpmovusqd(ymm1, k0, zmm2)),
        INSTRUCTION_TEST(62F27E28155C1108            , vpmovusqd(xmmword_ptr(rcx, rdx, 1, 128), k0, ymm3)),
        INSTRUCTION_TEST(62F27E2815D1                , vpmovusqd(xmm1, k0, ymm2)),
        INSTRUCTION_TEST(62F27E0815D1                , vpmovusqd(xmm1, k0, xmm2)),
        INSTRUCTION_TEST(62F27E48155C1104            , vpmovusqd(ymmword_ptr(rcx, rdx, 1, 128), k0, zmm3)),
        INSTRUCTION_TEST(62627E48144A80              , vpmovusqw(xmmword_ptr(rdx, -2048), k0, zmm25)),
        INSTRUCTION_TEST(62F27E48147280              , vpmovusqw(xmmword_ptr(rdx, -2048), k0, zmm6)),
        INSTRUCTION_TEST(62627E48148AF0F7FFFF        , vpmovusqw(xmmword_ptr(rdx, -2064), k0, zmm25)),
        INSTRUCTION_TEST(62F27E4814B200080000        , vpmovusqw(xmmword_ptr(rdx, 2048), k0, zmm6)),
        INSTRUCTION_TEST(62F27E481431                , vpmovusqw(xmmword_ptr(rcx), k0, zmm6)),
        INSTRUCTION_TEST(62627E48144A7F              , vpmovusqw(xmmword_ptr(rdx, 2032), k0, zmm25)),
        INSTRUCTION_TEST(62F27E4814727F              , vpmovusqw(xmmword_ptr(rdx, 2032), k0, zmm6)),
        INSTRUCTION_TEST(62627E48148A00080000        , vpmovusqw(xmmword_ptr(rdx, 2048), k0, zmm25)),
        INSTRUCTION_TEST(62F27E4814B2F0F7FFFF        , vpmovusqw(xmmword_ptr(rdx, -2064), k0, zmm6)),
        INSTRUCTION_TEST(62627E481409                , vpmovusqw(xmmword_ptr(rcx), k0, zmm25)),
        INSTRUCTION_TEST(62F27E4814D1                , vpmovusqw(xmm1, k0, zmm2)),
        INSTRUCTION_TEST(62B27E4814B4F034120000      , vpmovusqw(xmmword_ptr(rax, r14, 8, 4660), k0, zmm6)),
        INSTRUCTION_TEST(62227E48148CF023010000      , vpmovusqw(xmmword_ptr(rax, r14, 8, 291), k0, zmm25)),
        INSTRUCTION_TEST(62927E4814CD                , vpmovusqw(xmm29, k0, zmm1)),
        INSTRUCTION_TEST(62127E4814D4                , vpmovusqw(xmm28, k0, zmm10)),
        INSTRUCTION_TEST(62F27E2814D1                , vpmovusqw(xmm1, k0, ymm2)),
        INSTRUCTION_TEST(62F27E0814D1                , vpmovusqw(xmm1, k0, xmm2)),
        INSTRUCTION_TEST(62F27E28145C1110            , vpmovusqw(qword_ptr(rcx, rdx, 1, 128), k0, ymm3)),
        INSTRUCTION_TEST(62F27E08145C1120            , vpmovusqw(dword_ptr(rcx, rdx, 1, 128), k0, xmm3)),
        INSTRUCTION_TEST(62F27E48145C1108            , vpmovusqw(xmmword_ptr(rcx, rdx, 1, 128), k0, zmm3)),
        INSTRUCTION_TEST(62F27E48105C1104            , vpmovuswb(ymmword_ptr(rcx, rdx, 1, 128), k0, zmm3)),
        INSTRUCTION_TEST(62F27E28105C1108            , vpmovuswb(xmmword_ptr(rcx, rdx, 1, 128), k0, ymm3)),
        INSTRUCTION_TEST(62F27E4810D1                , vpmovuswb(ymm1, k0, zmm2)),
        INSTRUCTION_TEST(62F27E0810D1                , vpmovuswb(xmm1, k0, xmm2)),
        INSTRUCTION_TEST(62F27E08105C1110            , vpmovuswb(qword_ptr(rcx, rdx, 1, 128), k0, xmm3)),
        INSTRUCTION_TEST(62F27E2810D1                , vpmovuswb(xmm1, k0, ymm2)),
        INSTRUCTION_TEST(62F2FE0829CA                , vpmovw2m(k1, xmm2)),
        INSTRUCTION_TEST(62F2FE2829CA                , vpmovw2m(k1, ymm2)),
        INSTRUCTION_TEST(62F2FE4829CA                , vpmovw2m(k1, zmm2)),
        INSTRUCTION_TEST(62F27E08305C1110            , vpmovwb(qword_ptr(rcx, rdx, 1, 128), k0, xmm3)),
        INSTRUCTION_TEST(62F27E0830D1                , vpmovwb(xmm1, k0, xmm2)),
        INSTRUCTION_TEST(62F27E2830D1                , vpmovwb(xmm1, k0, ymm2)),
        INSTRUCTION_TEST(62F27E28305C1108            , vpmovwb(xmmword_ptr(rcx, rdx, 1, 128), k0, ymm3)),
        INSTRUCTION_TEST(62F27E4830D1                , vpmovwb(ymm1, k0, zmm2)),
        INSTRUCTION_TEST(62F27E48305C1104            , vpmovwb(ymmword_ptr(rcx, rdx, 1, 128), k0, zmm3)),
        INSTRUCTION_TEST(62027D4831C9                , vpmovzxbd(zmm25, k0, xmm25)),
        INSTRUCTION_TEST(62E27D48319200080000        , vpmovzxbd(zmm18, k0, xmmword_ptr(rdx, 2048))),
        INSTRUCTION_TEST(62227D48318CF034120000      , vpmovzxbd(zmm25, k0, xmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62E27D4831527F              , vpmovzxbd(zmm18, k0, xmmword_ptr(rdx, 2032))),
        INSTRUCTION_TEST(62627D48318AF0F7FFFF        , vpmovzxbd(zmm25, k0, xmmword_ptr(rdx, -2064))),
        INSTRUCTION_TEST(62627D48314A7F              , vpmovzxbd(zmm25, k0, xmmword_ptr(rdx, 2032))),
        INSTRUCTION_TEST(62627D48318A00080000        , vpmovzxbd(zmm25, k0, xmmword_ptr(rdx, 2048))),
        INSTRUCTION_TEST(62627D483109                , vpmovzxbd(zmm25, k0, xmmword_ptr(rcx))),
        INSTRUCTION_TEST(62E27D483192F0F7FFFF        , vpmovzxbd(zmm18, k0, xmmword_ptr(rdx, -2064))),
        INSTRUCTION_TEST(62627D48314A80              , vpmovzxbd(zmm25, k0, xmmword_ptr(rdx, -2048))),
        INSTRUCTION_TEST(62E27D483111                , vpmovzxbd(zmm18, k0, xmmword_ptr(rcx))),
        INSTRUCTION_TEST(C4E279318C1A80000000        , vpmovzxbd(xmm1, dword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(C4E27931CA                  , vpmovzxbd(xmm1, xmm2)),
        INSTRUCTION_TEST(C4E27D318C1A80000000        , vpmovzxbd(ymm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(C4E27D31CA                  , vpmovzxbd(ymm1, xmm2)),
        INSTRUCTION_TEST(62E27D48315280              , vpmovzxbd(zmm18, k0, xmmword_ptr(rdx, -2048))),
        INSTRUCTION_TEST(62F27D4831CA                , vpmovzxbd(zmm1, k0, xmm2)),
        INSTRUCTION_TEST(62F27D48314C1A08            , vpmovzxbd(zmm1, k0, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62827D4831D1                , vpmovzxbd(zmm18, k0, xmm25)),
        INSTRUCTION_TEST(62A27D483194F023010000      , vpmovzxbd(zmm18, k0, xmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62D27D4832EF                , vpmovzxbq(zmm5, k0, xmm15)),
        INSTRUCTION_TEST(62F27D4832AA00040000        , vpmovzxbq(zmm5, k0, qword_ptr(rdx, 1024))),
        INSTRUCTION_TEST(62F27D48326A7F              , vpmovzxbq(zmm5, k0, qword_ptr(rdx, 1016))),
        INSTRUCTION_TEST(62F27D4832AAF8FBFFFF        , vpmovzxbq(zmm5, k0, qword_ptr(rdx, -1032))),
        INSTRUCTION_TEST(62F27D483229                , vpmovzxbq(zmm5, k0, qword_ptr(rcx))),
        INSTRUCTION_TEST(62B27D4832ACF023010000      , vpmovzxbq(zmm5, k0, qword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62A27D4832FC                , vpmovzxbq(zmm23, k0, xmm20)),
        INSTRUCTION_TEST(62E27D4832BA00040000        , vpmovzxbq(zmm23, k0, qword_ptr(rdx, 1024))),
        INSTRUCTION_TEST(62E27D48327A7F              , vpmovzxbq(zmm23, k0, qword_ptr(rdx, 1016))),
        INSTRUCTION_TEST(62F27D48326A80              , vpmovzxbq(zmm5, k0, qword_ptr(rdx, -1024))),
        INSTRUCTION_TEST(62E27D48327A80              , vpmovzxbq(zmm23, k0, qword_ptr(rdx, -1024))),
        INSTRUCTION_TEST(62E27D483239                , vpmovzxbq(zmm23, k0, qword_ptr(rcx))),
        INSTRUCTION_TEST(62A27D4832BCF034120000      , vpmovzxbq(zmm23, k0, qword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62F27D4832CA                , vpmovzxbq(zmm1, k0, xmm2)),
        INSTRUCTION_TEST(62F27D48324C1A10            , vpmovzxbq(zmm1, k0, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(C4E27D32CA                  , vpmovzxbq(ymm1, xmm2)),
        INSTRUCTION_TEST(C4E27D328C1A80000000        , vpmovzxbq(ymm1, dword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(C4E27932CA                  , vpmovzxbq(xmm1, xmm2)),
        INSTRUCTION_TEST(C4E279328C1A80000000        , vpmovzxbq(xmm1, word_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62E27D4832BAF8FBFFFF        , vpmovzxbq(zmm23, k0, qword_ptr(rdx, -1032))),
        INSTRUCTION_TEST(62F27D48304C1A04            , vpmovzxbw(zmm1, k0, ymmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F27D4830CA                , vpmovzxbw(zmm1, k0, ymm2)),
        INSTRUCTION_TEST(C4E27D308C1A80000000        , vpmovzxbw(ymm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(C4E27D30CA                  , vpmovzxbw(ymm1, xmm2)),
        INSTRUCTION_TEST(C4E279308C1A80000000        , vpmovzxbw(xmm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(C4E27930CA                  , vpmovzxbw(xmm1, xmm2)),
        INSTRUCTION_TEST(62E27D4835627F              , vpmovzxdq(zmm20, k0, ymmword_ptr(rdx, 4064))),
        INSTRUCTION_TEST(62727D48358A00100000        , vpmovzxdq(zmm9, k0, ymmword_ptr(rdx, 4096))),
        INSTRUCTION_TEST(62727D48354A7F              , vpmovzxdq(zmm9, k0, ymmword_ptr(rdx, 4064))),
        INSTRUCTION_TEST(62727D48358AE0EFFFFF        , vpmovzxdq(zmm9, k0, ymmword_ptr(rdx, -4128))),
        INSTRUCTION_TEST(62727D48354A80              , vpmovzxdq(zmm9, k0, ymmword_ptr(rdx, -4096))),
        INSTRUCTION_TEST(62727D483509                , vpmovzxdq(zmm9, k0, ymmword_ptr(rcx))),
        INSTRUCTION_TEST(62327D48358CF034120000      , vpmovzxdq(zmm9, k0, ymmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62327D4835CE                , vpmovzxdq(zmm9, k0, ymm22)),
        INSTRUCTION_TEST(62E27D4835A200100000        , vpmovzxdq(zmm20, k0, ymmword_ptr(rdx, 4096))),
        INSTRUCTION_TEST(62E27D4835A2E0EFFFFF        , vpmovzxdq(zmm20, k0, ymmword_ptr(rdx, -4128))),
        INSTRUCTION_TEST(62E27D4835E4                , vpmovzxdq(zmm20, k0, ymm4)),
        INSTRUCTION_TEST(62E27D483521                , vpmovzxdq(zmm20, k0, ymmword_ptr(rcx))),
        INSTRUCTION_TEST(C4E279358C1A80000000        , vpmovzxdq(xmm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(C4E27935CA                  , vpmovzxdq(xmm1, xmm2)),
        INSTRUCTION_TEST(C4E27D35CA                  , vpmovzxdq(ymm1, xmm2)),
        INSTRUCTION_TEST(C4E27D358C1A80000000        , vpmovzxdq(ymm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62E27D48356280              , vpmovzxdq(zmm20, k0, ymmword_ptr(rdx, -4096))),
        INSTRUCTION_TEST(62F27D48354C1A04            , vpmovzxdq(zmm1, k0, ymmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62A27D4835A4F023010000      , vpmovzxdq(zmm20, k0, ymmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62F27D4835CA                , vpmovzxdq(zmm1, k0, ymm2)),
        INSTRUCTION_TEST(62727D48338200100000        , vpmovzxwd(zmm8, k0, ymmword_ptr(rdx, 4096))),
        INSTRUCTION_TEST(62727D4833427F              , vpmovzxwd(zmm8, k0, ymmword_ptr(rdx, 4064))),
        INSTRUCTION_TEST(62727D483382E0EFFFFF        , vpmovzxwd(zmm8, k0, ymmword_ptr(rdx, -4128))),
        INSTRUCTION_TEST(62727D48334280              , vpmovzxwd(zmm8, k0, ymmword_ptr(rdx, -4096))),
        INSTRUCTION_TEST(62727D483301                , vpmovzxwd(zmm8, k0, ymmword_ptr(rcx))),
        INSTRUCTION_TEST(62727D4833C6                , vpmovzxwd(zmm8, k0, ymm6)),
        INSTRUCTION_TEST(62327D483384F023010000      , vpmovzxwd(zmm8, k0, ymmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62F27D4833B200100000        , vpmovzxwd(zmm6, k0, ymmword_ptr(rdx, 4096))),
        INSTRUCTION_TEST(62F27D4833727F              , vpmovzxwd(zmm6, k0, ymmword_ptr(rdx, 4064))),
        INSTRUCTION_TEST(62F27D4833B2E0EFFFFF        , vpmovzxwd(zmm6, k0, ymmword_ptr(rdx, -4128))),
        INSTRUCTION_TEST(62F27D48337280              , vpmovzxwd(zmm6, k0, ymmword_ptr(rdx, -4096))),
        INSTRUCTION_TEST(62B27D4833B4F034120000      , vpmovzxwd(zmm6, k0, ymmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62F27D48334C1A04            , vpmovzxwd(zmm1, k0, ymmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F27D4833CA                , vpmovzxwd(zmm1, k0, ymm2)),
        INSTRUCTION_TEST(C4E27D338C1A80000000        , vpmovzxwd(ymm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(C4E27D33CA                  , vpmovzxwd(ymm1, xmm2)),
        INSTRUCTION_TEST(C4E27933CA                  , vpmovzxwd(xmm1, xmm2)),
        INSTRUCTION_TEST(C4E279338C1A80000000        , vpmovzxwd(xmm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F27D483331                , vpmovzxwd(zmm6, k0, ymmword_ptr(rcx))),
        INSTRUCTION_TEST(62927D4833F5                , vpmovzxwd(zmm6, k0, ymm29)),
        INSTRUCTION_TEST(62D27D4834EF                , vpmovzxwq(zmm5, k0, xmm15)),
        INSTRUCTION_TEST(62F27D483429                , vpmovzxwq(zmm5, k0, xmmword_ptr(rcx))),
        INSTRUCTION_TEST(62B27D4834ACF023010000      , vpmovzxwq(zmm5, k0, xmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62F27D4834CA                , vpmovzxwq(zmm1, k0, xmm2)),
        INSTRUCTION_TEST(62F27D4834AAF0F7FFFF        , vpmovzxwq(zmm5, k0, xmmword_ptr(rdx, -2064))),
        INSTRUCTION_TEST(62F27D48346A7F              , vpmovzxwq(zmm5, k0, xmmword_ptr(rdx, 2032))),
        INSTRUCTION_TEST(62F27D4834AA00080000        , vpmovzxwq(zmm5, k0, xmmword_ptr(rdx, 2048))),
        INSTRUCTION_TEST(62727D4834A200080000        , vpmovzxwq(zmm12, k0, xmmword_ptr(rdx, 2048))),
        INSTRUCTION_TEST(62F27D48346A80              , vpmovzxwq(zmm5, k0, xmmword_ptr(rdx, -2048))),
        INSTRUCTION_TEST(62727D4834627F              , vpmovzxwq(zmm12, k0, xmmword_ptr(rdx, 2032))),
        INSTRUCTION_TEST(C4E27934CA                  , vpmovzxwq(xmm1, xmm2)),
        INSTRUCTION_TEST(62727D48346280              , vpmovzxwq(zmm12, k0, xmmword_ptr(rdx, -2048))),
        INSTRUCTION_TEST(62727D483421                , vpmovzxwq(zmm12, k0, xmmword_ptr(rcx))),
        INSTRUCTION_TEST(62327D4834A4F034120000      , vpmovzxwq(zmm12, k0, xmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62327D4834E2                , vpmovzxwq(zmm12, k0, xmm18)),
        INSTRUCTION_TEST(62F27D48344C1A08            , vpmovzxwq(zmm1, k0, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(C4E27D34CA                  , vpmovzxwq(ymm1, xmm2)),
        INSTRUCTION_TEST(C4E27D348C1A80000000        , vpmovzxwq(ymm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62727D4834A2F0F7FFFF        , vpmovzxwq(zmm12, k0, xmmword_ptr(rdx, -2064))),
        INSTRUCTION_TEST(C4E279348C1A80000000        , vpmovzxwq(xmm1, dword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(6262B54828AA00200000        , vpmuldq(zmm29, k0, zmm9, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62D2CD4028EC                , vpmuldq(zmm5, k0, zmm22, zmm12)),
        INSTRUCTION_TEST(62B2CD4028ACF034120000      , vpmuldq(zmm5, k0, zmm22, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62F2CD402829                , vpmuldq(zmm5, k0, zmm22, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62F2CD4028AA00200000        , vpmuldq(zmm5, k0, zmm22, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62F2CD4028AAC0DFFFFF        , vpmuldq(zmm5, k0, zmm22, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62F2CD40286A7F              , vpmuldq(zmm5, k0, zmm22, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(6262B548286A7F              , vpmuldq(zmm29, k0, zmm9, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62F2CD40286A80              , vpmuldq(zmm5, k0, zmm22, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(6262B54828AAC0DFFFFF        , vpmuldq(zmm29, k0, zmm9, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(6242B54828E9                , vpmuldq(zmm29, k0, zmm9, zmm9)),
        INSTRUCTION_TEST(6262B5482829                , vpmuldq(zmm29, k0, zmm9, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(6222B54828ACF023010000      , vpmuldq(zmm29, k0, zmm9, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62F2ED48284C2B02            , vpmuldq(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F2ED4828CB                , vpmuldq(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(C4E26D288C2B80000000        , vpmuldq(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E26D28CB                  , vpmuldq(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(C4E269288C2B80000000        , vpmuldq(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E26928CB                  , vpmuldq(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(6262B548286A80              , vpmuldq(zmm29, k0, zmm9, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62F26D480BCB                , vpmulhrsw(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(C4E26D0B8C2B80000000        , vpmulhrsw(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F26D480B4C2B02            , vpmulhrsw(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E2690B8C2B80000000        , vpmulhrsw(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E2690BCB                  , vpmulhrsw(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(C4E26D0BCB                  , vpmulhrsw(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(C5E9E4CB                    , vpmulhuw(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(C5E9E48C2B80000000          , vpmulhuw(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C5EDE4CB                    , vpmulhuw(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(C5EDE48C2B80000000          , vpmulhuw(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F16D48E4CB                , vpmulhuw(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(62F16D48E44C2B02            , vpmulhuw(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F16D48E54C2B02            , vpmulhw(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F16D48E5CB                , vpmulhw(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(C5EDE58C2B80000000          , vpmulhw(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C5E9E58C2B80000000          , vpmulhw(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C5E9E5CB                    , vpmulhw(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(C5EDE5CB                    , vpmulhw(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(62723D4040A2C0DFFFFF        , vpmulld(zmm12, k0, zmm24, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(6272654840627F              , vpmulld(zmm12, k0, zmm3, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(6272654840A2C0DFFFFF        , vpmulld(zmm12, k0, zmm3, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62726548406280              , vpmulld(zmm12, k0, zmm3, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(6272654840A200200000        , vpmulld(zmm12, k0, zmm3, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(627265484021                , vpmulld(zmm12, k0, zmm3, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(6272654840E2                , vpmulld(zmm12, k0, zmm3, zmm2)),
        INSTRUCTION_TEST(62723D4040A200200000        , vpmulld(zmm12, k0, zmm24, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62723D4040627F              , vpmulld(zmm12, k0, zmm24, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62723D40406280              , vpmulld(zmm12, k0, zmm24, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(C4E26D408C2B80000000        , vpmulld(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62723D4040E2                , vpmulld(zmm12, k0, zmm24, zmm2)),
        INSTRUCTION_TEST(6232654840A4F023010000      , vpmulld(zmm12, k0, zmm3, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62323D4040A4F034120000      , vpmulld(zmm12, k0, zmm24, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62F26D48404C2B02            , vpmulld(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F26D4840CB                , vpmulld(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(C4E26D40CB                  , vpmulld(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(C4E269408C2B80000000        , vpmulld(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E26940CB                  , vpmulld(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(62723D404021                , vpmulld(zmm12, k0, zmm24, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62F2ED4840CB                , vpmullq(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(62F2ED48404C2B02            , vpmullq(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F2ED28404C2B04            , vpmullq(ymm1, k0, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F2ED08404C2B08            , vpmullq(xmm1, k0, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F2ED0840CB                , vpmullq(xmm1, k0, xmm2, xmm3)),
        INSTRUCTION_TEST(62F2ED2840CB                , vpmullq(ymm1, k0, ymm2, ymm3)),
        INSTRUCTION_TEST(C5E9D5CB                    , vpmullw(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(C5E9D58C2B80000000          , vpmullw(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C5EDD5CB                    , vpmullw(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(C5EDD58C2B80000000          , vpmullw(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F16D48D5CB                , vpmullw(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(62F16D48D54C2B02            , vpmullw(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F2ED48834C2B02            , vpmultishiftqb(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F2ED4883CB                , vpmultishiftqb(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(62F2ED28834C2B04            , vpmultishiftqb(ymm1, k0, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F2ED08834C2B08            , vpmultishiftqb(xmm1, k0, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F2ED0883CB                , vpmultishiftqb(xmm1, k0, xmm2, xmm3)),
        INSTRUCTION_TEST(62F2ED2883CB                , vpmultishiftqb(ymm1, k0, ymm2, ymm3)),
        INSTRUCTION_TEST(62E1D548F47A7F              , vpmuludq(zmm23, k0, zmm5, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(6271ED48F48200200000        , vpmuludq(zmm8, k0, zmm2, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(6271ED48F4427F              , vpmuludq(zmm8, k0, zmm2, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(6271ED48F482C0DFFFFF        , vpmuludq(zmm8, k0, zmm2, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(6271ED48F44280              , vpmuludq(zmm8, k0, zmm2, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(6271ED48F401                , vpmuludq(zmm8, k0, zmm2, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(6231ED48F484F034120000      , vpmuludq(zmm8, k0, zmm2, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(6251ED48F4C2                , vpmuludq(zmm8, k0, zmm2, zmm10)),
        INSTRUCTION_TEST(62E1D548F4BA00200000        , vpmuludq(zmm23, k0, zmm5, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62E1D548F4BAC0DFFFFF        , vpmuludq(zmm23, k0, zmm5, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(C5EDF48C2B80000000          , vpmuludq(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62E1D548F439                , vpmuludq(zmm23, k0, zmm5, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62A1D548F4BCF023010000      , vpmuludq(zmm23, k0, zmm5, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62C1D548F4F9                , vpmuludq(zmm23, k0, zmm5, zmm9)),
        INSTRUCTION_TEST(62F1ED48F44C2B02            , vpmuludq(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F1ED48F4CB                , vpmuludq(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(C5EDF4CB                    , vpmuludq(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(C5E9F48C2B80000000          , vpmuludq(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C5E9F4CB                    , vpmuludq(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(62E1D548F47A80              , vpmuludq(zmm23, k0, zmm5, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62F27D4854CA                , vpopcntb(zmm1, k0, zmm2)),
        INSTRUCTION_TEST(62F27D48544C1A02            , vpopcntb(zmm1, k0, zmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F27D28544C1A04            , vpopcntb(ymm1, k0, ymmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F27D0854CA                , vpopcntb(xmm1, k0, xmm2)),
        INSTRUCTION_TEST(62F27D08544C1A08            , vpopcntb(xmm1, k0, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F27D2854CA                , vpopcntb(ymm1, k0, ymm2)),
        INSTRUCTION_TEST(62F27D0855CA                , vpopcntd(xmm1, k0, xmm2)),
        INSTRUCTION_TEST(62F27D08554C1A08            , vpopcntd(xmm1, k0, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F27D2855CA                , vpopcntd(ymm1, k0, ymm2)),
        INSTRUCTION_TEST(62F27D28554C1A04            , vpopcntd(ymm1, k0, ymmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F27D4855CA                , vpopcntd(zmm1, k0, zmm2)),
        INSTRUCTION_TEST(62F27D48554C1A02            , vpopcntd(zmm1, k0, zmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F2FD08554C1A08            , vpopcntq(xmm1, k0, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F2FD2855CA                , vpopcntq(ymm1, k0, ymm2)),
        INSTRUCTION_TEST(62F2FD28554C1A04            , vpopcntq(ymm1, k0, ymmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F2FD4855CA                , vpopcntq(zmm1, k0, zmm2)),
        INSTRUCTION_TEST(62F2FD48554C1A02            , vpopcntq(zmm1, k0, zmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F2FD0855CA                , vpopcntq(xmm1, k0, xmm2)),
        INSTRUCTION_TEST(62F2FD28544C1A04            , vpopcntw(ymm1, k0, ymmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F2FD4854CA                , vpopcntw(zmm1, k0, zmm2)),
        INSTRUCTION_TEST(62F2FD48544C1A02            , vpopcntw(zmm1, k0, zmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F2FD2854CA                , vpopcntw(ymm1, k0, ymm2)),
        INSTRUCTION_TEST(62F2FD0854CA                , vpopcntw(xmm1, k0, xmm2)),
        INSTRUCTION_TEST(62F2FD08544C1A08            , vpopcntw(xmm1, k0, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(C5E9EBCB                    , vpor(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(C5E9EB8C2B80000000          , vpor(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C5EDEBCB                    , vpor(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(C5EDEB8C2B80000000          , vpor(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62E16D48EB7A7F              , vpord(zmm23, k0, zmm2, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62612D48EB6A7F              , vpord(zmm29, k0, zmm10, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62612D48EBAAC0DFFFFF        , vpord(zmm29, k0, zmm10, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62612D48EBAA00200000        , vpord(zmm29, k0, zmm10, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62612D48EB29                , vpord(zmm29, k0, zmm10, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62212D48EBACF034120000      , vpord(zmm29, k0, zmm10, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62412D48EBEF                , vpord(zmm29, k0, zmm10, zmm15)),
        INSTRUCTION_TEST(62E16D48EBBA00200000        , vpord(zmm23, k0, zmm2, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62E16D48EBBAC0DFFFFF        , vpord(zmm23, k0, zmm2, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62612D48EB6A80              , vpord(zmm29, k0, zmm10, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62E16D48EB39                , vpord(zmm23, k0, zmm2, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62A16D48EBBCF023010000      , vpord(zmm23, k0, zmm2, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62A16D48EBFC                , vpord(zmm23, k0, zmm2, zmm20)),
        INSTRUCTION_TEST(62F16D48EB4C2B02            , vpord(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F16D48EBCB                , vpord(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(62F16D28EB4C2B04            , vpord(ymm1, k0, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F16D28EBCB                , vpord(ymm1, k0, ymm2, ymm3)),
        INSTRUCTION_TEST(62F16D08EB4C2B08            , vpord(xmm1, k0, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F16D08EBCB                , vpord(xmm1, k0, xmm2, xmm3)),
        INSTRUCTION_TEST(62E16D48EB7A80              , vpord(zmm23, k0, zmm2, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62F1ED48EB4C2B02            , vporq(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(6291C540EBDA                , vporq(zmm3, k0, zmm23, zmm26)),
        INSTRUCTION_TEST(62B1C540EB9CF034120000      , vporq(zmm3, k0, zmm23, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62F1C540EB9A00200000        , vporq(zmm3, k0, zmm23, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62F1C540EB5A80              , vporq(zmm3, k0, zmm23, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62F1C540EB9AC0DFFFFF        , vporq(zmm3, k0, zmm23, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62F1C540EB5A7F              , vporq(zmm3, k0, zmm23, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62F1C540EB19                , vporq(zmm3, k0, zmm23, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62F1AD48EB8A00200000        , vporq(zmm1, k0, zmm10, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62F1ED48EBCB                , vporq(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(62F1AD48EB8AC0DFFFFF        , vporq(zmm1, k0, zmm10, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62F1AD48EB4A80              , vporq(zmm1, k0, zmm10, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62F1AD48EB09                , vporq(zmm1, k0, zmm10, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62B1AD48EB8CF023010000      , vporq(zmm1, k0, zmm10, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62F1AD48EBCE                , vporq(zmm1, k0, zmm10, zmm6)),
        INSTRUCTION_TEST(62F1ED28EB4C2B04            , vporq(ymm1, k0, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F1ED28EBCB                , vporq(ymm1, k0, ymm2, ymm3)),
        INSTRUCTION_TEST(62F1ED08EB4C2B08            , vporq(xmm1, k0, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F1ED08EBCB                , vporq(xmm1, k0, xmm2, xmm3)),
        INSTRUCTION_TEST(62F1AD48EB4A7F              , vporq(zmm1, k0, zmm10, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(8FE868A38C2B8000000060      , vpperm(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128), xmm6)),
        INSTRUCTION_TEST(8FE8E8A38C358000000030      , vpperm(xmm1, xmm2, xmm3, xmmword_ptr(rbp, rsi, 1, 128))),
        INSTRUCTION_TEST(8FE868A3CB40                , vpperm(xmm1, xmm2, xmm3, xmm4)),
        INSTRUCTION_TEST(62F16540728A002000007B      , vprold(zmm19, k0, zmmword_ptr(rdx, 8192), Imm(123))),
        INSTRUCTION_TEST(62F16540724A7F7B            , vprold(zmm19, k0, zmmword_ptr(rdx, 8128), Imm(123))),
        INSTRUCTION_TEST(62F16540728AC0DFFFFF7B      , vprold(zmm19, k0, zmmword_ptr(rdx, -8256), Imm(123))),
        INSTRUCTION_TEST(62F16540724A807B            , vprold(zmm19, k0, zmmword_ptr(rdx, -8192), Imm(123))),
        INSTRUCTION_TEST(62F1654072097B              , vprold(zmm19, k0, zmmword_ptr(rcx), Imm(123))),
        INSTRUCTION_TEST(62B16540728CF0341200007B    , vprold(zmm19, k0, zmmword_ptr(rax, r14, 8, 4660), Imm(123))),
        INSTRUCTION_TEST(6291654072CBAB              , vprold(zmm19, k0, zmm27, Imm(171))),
        INSTRUCTION_TEST(6291654072CB7B              , vprold(zmm19, k0, zmm27, Imm(123))),
        INSTRUCTION_TEST(62F12D48728A002000007B      , vprold(zmm10, k0, zmmword_ptr(rdx, 8192), Imm(123))),
        INSTRUCTION_TEST(62F12D48724A7F7B            , vprold(zmm10, k0, zmmword_ptr(rdx, 8128), Imm(123))),
        INSTRUCTION_TEST(62F12D48728AC0DFFFFF7B      , vprold(zmm10, k0, zmmword_ptr(rdx, -8256), Imm(123))),
        INSTRUCTION_TEST(62F12D4872097B              , vprold(zmm10, k0, zmmword_ptr(rcx), Imm(123))),
        INSTRUCTION_TEST(62B12D48728CF0230100007B    , vprold(zmm10, k0, zmmword_ptr(rax, r14, 8, 291), Imm(123))),
        INSTRUCTION_TEST(62F1750872CA01              , vprold(xmm1, k0, xmm2, Imm(1))),
        INSTRUCTION_TEST(62F17508724C1A0801          , vprold(xmm1, k0, xmmword_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F1752872CA01              , vprold(ymm1, k0, ymm2, Imm(1))),
        INSTRUCTION_TEST(62F17528724C1A0401          , vprold(ymm1, k0, ymmword_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F12D48724A807B            , vprold(zmm10, k0, zmmword_ptr(rdx, -8192), Imm(123))),
        INSTRUCTION_TEST(62F17548724C1A0201          , vprold(zmm1, k0, zmmword_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F12D4872CD7B              , vprold(zmm10, k0, zmm5, Imm(123))),
        INSTRUCTION_TEST(62F12D4872CDAB              , vprold(zmm10, k0, zmm5, Imm(171))),
        INSTRUCTION_TEST(62F1754872CA01              , vprold(zmm1, k0, zmm2, Imm(1))),
        INSTRUCTION_TEST(62F1E540728A002000007B      , vprolq(zmm19, k0, zmmword_ptr(rdx, 8192), Imm(123))),
        INSTRUCTION_TEST(62F1F540728AC0DFFFFF7B      , vprolq(zmm17, k0, zmmword_ptr(rdx, -8256), Imm(123))),
        INSTRUCTION_TEST(62F1E540728AC0DFFFFF7B      , vprolq(zmm19, k0, zmmword_ptr(rdx, -8256), Imm(123))),
        INSTRUCTION_TEST(62F1E540724A807B            , vprolq(zmm19, k0, zmmword_ptr(rdx, -8192), Imm(123))),
        INSTRUCTION_TEST(62F1E54072097B              , vprolq(zmm19, k0, zmmword_ptr(rcx), Imm(123))),
        INSTRUCTION_TEST(62B1E540728CF0230100007B    , vprolq(zmm19, k0, zmmword_ptr(rax, r14, 8, 291), Imm(123))),
        INSTRUCTION_TEST(62B1E54072CAAB              , vprolq(zmm19, k0, zmm18, Imm(171))),
        INSTRUCTION_TEST(62B1E54072CA7B              , vprolq(zmm19, k0, zmm18, Imm(123))),
        INSTRUCTION_TEST(62F1F540728A002000007B      , vprolq(zmm17, k0, zmmword_ptr(rdx, 8192), Imm(123))),
        INSTRUCTION_TEST(62F1F540724A7F7B            , vprolq(zmm17, k0, zmmword_ptr(rdx, 8128), Imm(123))),
        INSTRUCTION_TEST(62F1E540724A7F7B            , vprolq(zmm19, k0, zmmword_ptr(rdx, 8128), Imm(123))),
        INSTRUCTION_TEST(62F1F540724A807B            , vprolq(zmm17, k0, zmmword_ptr(rdx, -8192), Imm(123))),
        INSTRUCTION_TEST(62B1F540728CF0341200007B    , vprolq(zmm17, k0, zmmword_ptr(rax, r14, 8, 4660), Imm(123))),
        INSTRUCTION_TEST(62D1F54072CAAB              , vprolq(zmm17, k0, zmm10, Imm(171))),
        INSTRUCTION_TEST(62D1F54072CA7B              , vprolq(zmm17, k0, zmm10, Imm(123))),
        INSTRUCTION_TEST(62F1F548724C1A0201          , vprolq(zmm1, k0, zmmword_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F1F54872CA01              , vprolq(zmm1, k0, zmm2, Imm(1))),
        INSTRUCTION_TEST(62F1F528724C1A0401          , vprolq(ymm1, k0, ymmword_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F1F52872CA01              , vprolq(ymm1, k0, ymm2, Imm(1))),
        INSTRUCTION_TEST(62F1F508724C1A0801          , vprolq(xmm1, k0, xmmword_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F1F50872CA01              , vprolq(xmm1, k0, xmm2, Imm(1))),
        INSTRUCTION_TEST(62F1F54072097B              , vprolq(zmm17, k0, zmmword_ptr(rcx), Imm(123))),
        INSTRUCTION_TEST(6262754815A200200000        , vprolvd(zmm28, k0, zmm1, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62B2154015F0                , vprolvd(zmm6, k0, zmm29, zmm16)),
        INSTRUCTION_TEST(62B2154015B4F023010000      , vprolvd(zmm6, k0, zmm29, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(6262754815627F              , vprolvd(zmm28, k0, zmm1, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62F21540157280              , vprolvd(zmm6, k0, zmm29, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62F2154015727F              , vprolvd(zmm6, k0, zmm29, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62F2154015B200200000        , vprolvd(zmm6, k0, zmm29, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62F215401531                , vprolvd(zmm6, k0, zmm29, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(6262754815A2C0DFFFFF        , vprolvd(zmm28, k0, zmm1, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62F2154015B2C0DFFFFF        , vprolvd(zmm6, k0, zmm29, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(626275481521                , vprolvd(zmm28, k0, zmm1, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62F26D08154C2B08            , vprolvd(xmm1, k0, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62627548156280              , vprolvd(zmm28, k0, zmm1, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62F26D2815CB                , vprolvd(ymm1, k0, ymm2, ymm3)),
        INSTRUCTION_TEST(62F26D28154C2B04            , vprolvd(ymm1, k0, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F26D0815CB                , vprolvd(xmm1, k0, xmm2, xmm3)),
        INSTRUCTION_TEST(62F26D48154C2B02            , vprolvd(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(6242754815E2                , vprolvd(zmm28, k0, zmm1, zmm10)),
        INSTRUCTION_TEST(6222754815A4F034120000      , vprolvd(zmm28, k0, zmm1, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62F26D4815CB                , vprolvd(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(6262F548158A00200000        , vprolvq(zmm25, k0, zmm1, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(6262DD48159A00200000        , vprolvq(zmm27, k0, zmm4, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(6262DD48155A7F              , vprolvq(zmm27, k0, zmm4, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(6262DD48159AC0DFFFFF        , vprolvq(zmm27, k0, zmm4, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(6262DD48155A80              , vprolvq(zmm27, k0, zmm4, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(6262DD481519                , vprolvq(zmm27, k0, zmm4, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(6262DD4815DE                , vprolvq(zmm27, k0, zmm4, zmm6)),
        INSTRUCTION_TEST(6222DD48159CF023010000      , vprolvq(zmm27, k0, zmm4, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(6262F548154A7F              , vprolvq(zmm25, k0, zmm1, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(6262F548158AC0DFFFFF        , vprolvq(zmm25, k0, zmm1, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(6262F5481509                , vprolvq(zmm25, k0, zmm1, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(6222F548158CF034120000      , vprolvq(zmm25, k0, zmm1, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(6222F54815CE                , vprolvq(zmm25, k0, zmm1, zmm22)),
        INSTRUCTION_TEST(62F2ED48154C2B02            , vprolvq(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F2ED4815CB                , vprolvq(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(62F2ED28154C2B04            , vprolvq(ymm1, k0, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F2ED2815CB                , vprolvq(ymm1, k0, ymm2, ymm3)),
        INSTRUCTION_TEST(62F2ED08154C2B08            , vprolvq(xmm1, k0, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F2ED0815CB                , vprolvq(xmm1, k0, xmm2, xmm3)),
        INSTRUCTION_TEST(6262F548154A80              , vprolvq(zmm25, k0, zmm1, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62F1454872427F7B            , vprord(zmm7, k0, zmmword_ptr(rdx, 8128), Imm(123))),
        INSTRUCTION_TEST(62F145487282002000007B      , vprord(zmm7, k0, zmmword_ptr(rdx, 8192), Imm(123))),
        INSTRUCTION_TEST(62D1354872C27B              , vprord(zmm9, k0, zmm10, Imm(123))),
        INSTRUCTION_TEST(62D1354872C2AB              , vprord(zmm9, k0, zmm10, Imm(171))),
        INSTRUCTION_TEST(62B135487284F0230100007B    , vprord(zmm9, k0, zmmword_ptr(rax, r14, 8, 291), Imm(123))),
        INSTRUCTION_TEST(62F135487282002000007B      , vprord(zmm9, k0, zmmword_ptr(rdx, 8192), Imm(123))),
        INSTRUCTION_TEST(62F135487242807B            , vprord(zmm9, k0, zmmword_ptr(rdx, -8192), Imm(123))),
        INSTRUCTION_TEST(62F135487282C0DFFFFF7B      , vprord(zmm9, k0, zmmword_ptr(rdx, -8256), Imm(123))),
        INSTRUCTION_TEST(62F1354872427F7B            , vprord(zmm9, k0, zmmword_ptr(rdx, 8128), Imm(123))),
        INSTRUCTION_TEST(62F1354872017B              , vprord(zmm9, k0, zmmword_ptr(rcx), Imm(123))),
        INSTRUCTION_TEST(62F145487282C0DFFFFF7B      , vprord(zmm7, k0, zmmword_ptr(rdx, -8256), Imm(123))),
        INSTRUCTION_TEST(62F1750872C201              , vprord(xmm1, k0, xmm2, Imm(1))),
        INSTRUCTION_TEST(62F1454872017B              , vprord(zmm7, k0, zmmword_ptr(rcx), Imm(123))),
        INSTRUCTION_TEST(62B145487284F0341200007B    , vprord(zmm7, k0, zmmword_ptr(rax, r14, 8, 4660), Imm(123))),
        INSTRUCTION_TEST(62D1454872C7AB              , vprord(zmm7, k0, zmm15, Imm(171))),
        INSTRUCTION_TEST(62D1454872C77B              , vprord(zmm7, k0, zmm15, Imm(123))),
        INSTRUCTION_TEST(62F1754872441A0201          , vprord(zmm1, k0, zmmword_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F1754872C201              , vprord(zmm1, k0, zmm2, Imm(1))),
        INSTRUCTION_TEST(62F1752872441A0401          , vprord(ymm1, k0, ymmword_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F1752872C201              , vprord(ymm1, k0, ymm2, Imm(1))),
        INSTRUCTION_TEST(62F1750872441A0801          , vprord(xmm1, k0, xmmword_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F145487242807B            , vprord(zmm7, k0, zmmword_ptr(rdx, -8192), Imm(123))),
        INSTRUCTION_TEST(62F1FD407282002000007B      , vprorq(zmm16, k0, zmmword_ptr(rdx, 8192), Imm(123))),
        INSTRUCTION_TEST(6291D54072C17B              , vprorq(zmm21, k0, zmm25, Imm(123))),
        INSTRUCTION_TEST(6291D54072C1AB              , vprorq(zmm21, k0, zmm25, Imm(171))),
        INSTRUCTION_TEST(62B1D5407284F0230100007B    , vprorq(zmm21, k0, zmmword_ptr(rax, r14, 8, 291), Imm(123))),
        INSTRUCTION_TEST(62F1FD4072427F7B            , vprorq(zmm16, k0, zmmword_ptr(rdx, 8128), Imm(123))),
        INSTRUCTION_TEST(62F1D5407242807B            , vprorq(zmm21, k0, zmmword_ptr(rdx, -8192), Imm(123))),
        INSTRUCTION_TEST(62F1D5407282C0DFFFFF7B      , vprorq(zmm21, k0, zmmword_ptr(rdx, -8256), Imm(123))),
        INSTRUCTION_TEST(62F1D54072427F7B            , vprorq(zmm21, k0, zmmword_ptr(rdx, 8128), Imm(123))),
        INSTRUCTION_TEST(62F1D54072017B              , vprorq(zmm21, k0, zmmword_ptr(rcx), Imm(123))),
        INSTRUCTION_TEST(62F1FD407282C0DFFFFF7B      , vprorq(zmm16, k0, zmmword_ptr(rdx, -8256), Imm(123))),
        INSTRUCTION_TEST(62F1D5407282002000007B      , vprorq(zmm21, k0, zmmword_ptr(rdx, 8192), Imm(123))),
        INSTRUCTION_TEST(62F1FD4072017B              , vprorq(zmm16, k0, zmmword_ptr(rcx), Imm(123))),
        INSTRUCTION_TEST(62F1F50872C201              , vprorq(xmm1, k0, xmm2, Imm(1))),
        INSTRUCTION_TEST(62F1F50872441A0801          , vprorq(xmm1, k0, xmmword_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F1F52872C201              , vprorq(ymm1, k0, ymm2, Imm(1))),
        INSTRUCTION_TEST(62F1F52872441A0401          , vprorq(ymm1, k0, ymmword_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F1FD407242807B            , vprorq(zmm16, k0, zmmword_ptr(rdx, -8192), Imm(123))),
        INSTRUCTION_TEST(62F1F54872441A0201          , vprorq(zmm1, k0, zmmword_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(6291FD4072C37B              , vprorq(zmm16, k0, zmm27, Imm(123))),
        INSTRUCTION_TEST(6291FD4072C3AB              , vprorq(zmm16, k0, zmm27, Imm(171))),
        INSTRUCTION_TEST(62B1FD407284F0341200007B    , vprorq(zmm16, k0, zmmword_ptr(rax, r14, 8, 4660), Imm(123))),
        INSTRUCTION_TEST(62F1F54872C201              , vprorq(zmm1, k0, zmm2, Imm(1))),
        INSTRUCTION_TEST(62725D40149A00200000        , vprorvd(zmm11, k0, zmm20, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62725D40145A7F              , vprorvd(zmm11, k0, zmm20, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62725D40149AC0DFFFFF        , vprorvd(zmm11, k0, zmm20, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62725D40145A80              , vprorvd(zmm11, k0, zmm20, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62725D401419                , vprorvd(zmm11, k0, zmm20, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62325D40149CF023010000      , vprorvd(zmm11, k0, zmm20, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62125D4014DC                , vprorvd(zmm11, k0, zmm20, zmm28)),
        INSTRUCTION_TEST(62F23D48148A00200000        , vprorvd(zmm1, k0, zmm8, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62F23D48148AC0DFFFFF        , vprorvd(zmm1, k0, zmm8, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62F23D48144A7F              , vprorvd(zmm1, k0, zmm8, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62F23D481409                , vprorvd(zmm1, k0, zmm8, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62B23D48148CF034120000      , vprorvd(zmm1, k0, zmm8, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62D23D4814CD                , vprorvd(zmm1, k0, zmm8, zmm13)),
        INSTRUCTION_TEST(62F26D48144C2B02            , vprorvd(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F26D4814CB                , vprorvd(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(62F26D28144C2B04            , vprorvd(ymm1, k0, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F26D2814CB                , vprorvd(ymm1, k0, ymm2, ymm3)),
        INSTRUCTION_TEST(62F26D08144C2B08            , vprorvd(xmm1, k0, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F26D0814CB                , vprorvd(xmm1, k0, xmm2, xmm3)),
        INSTRUCTION_TEST(62F23D48144A80              , vprorvd(zmm1, k0, zmm8, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(6272BD4014B200200000        , vprorvq(zmm14, k0, zmm24, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(6282C54014E8                , vprorvq(zmm21, k0, zmm23, zmm24)),
        INSTRUCTION_TEST(62A2C54014ACF034120000      , vprorvq(zmm21, k0, zmm23, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62E2C54014AA00200000        , vprorvq(zmm21, k0, zmm23, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62E2C540146A80              , vprorvq(zmm21, k0, zmm23, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62E2C54014AAC0DFFFFF        , vprorvq(zmm21, k0, zmm23, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62E2C540146A7F              , vprorvq(zmm21, k0, zmm23, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62E2C5401429                , vprorvq(zmm21, k0, zmm23, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(6272BD4014727F              , vprorvq(zmm14, k0, zmm24, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(6272BD4014B2C0DFFFFF        , vprorvq(zmm14, k0, zmm24, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(6272BD401431                , vprorvq(zmm14, k0, zmm24, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62F2ED0814CB                , vprorvq(xmm1, k0, xmm2, xmm3)),
        INSTRUCTION_TEST(62F2ED08144C2B08            , vprorvq(xmm1, k0, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F2ED2814CB                , vprorvq(ymm1, k0, ymm2, ymm3)),
        INSTRUCTION_TEST(62F2ED28144C2B04            , vprorvq(ymm1, k0, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(6272BD40147280              , vprorvq(zmm14, k0, zmm24, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62F2ED48144C2B02            , vprorvq(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(6232BD4014B4F023010000      , vprorvq(zmm14, k0, zmm24, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(6272BD4014F1                , vprorvq(zmm14, k0, zmm24, zmm1)),
        INSTRUCTION_TEST(62F2ED4814CB                , vprorvq(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(8FE96090CA                  , vprotb(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(8FE9E8908C2B80000000        , vprotb(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(8FE878C08C1A8000000001      , vprotb(xmm1, xmmword_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(8FE958908C1A80000000        , vprotb(xmm1, xmmword_ptr(rdx, rbx, 1, 128), xmm4)),
        INSTRUCTION_TEST(8FE878C0CA01                , vprotb(xmm1, xmm2, Imm(1))),
        INSTRUCTION_TEST(8FE878C2CA01                , vprotd(xmm1, xmm2, Imm(1))),
        INSTRUCTION_TEST(8FE96092CA                  , vprotd(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(8FE9E8928C2B80000000        , vprotd(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(8FE878C28C1A8000000001      , vprotd(xmm1, xmmword_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(8FE958928C1A80000000        , vprotd(xmm1, xmmword_ptr(rdx, rbx, 1, 128), xmm4)),
        INSTRUCTION_TEST(8FE958938C1A80000000        , vprotq(xmm1, xmmword_ptr(rdx, rbx, 1, 128), xmm4)),
        INSTRUCTION_TEST(8FE878C38C1A8000000001      , vprotq(xmm1, xmmword_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(8FE96093CA                  , vprotq(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(8FE878C3CA01                , vprotq(xmm1, xmm2, Imm(1))),
        INSTRUCTION_TEST(8FE9E8938C2B80000000        , vprotq(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(8FE878C1CA01                , vprotw(xmm1, xmm2, Imm(1))),
        INSTRUCTION_TEST(8FE96091CA                  , vprotw(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(8FE9E8918C2B80000000        , vprotw(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(8FE878C18C1A8000000001      , vprotw(xmm1, xmmword_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(8FE958918C1A80000000        , vprotw(xmm1, xmmword_ptr(rdx, rbx, 1, 128), xmm4)),
        INSTRUCTION_TEST(62F16D48F64C2B02            , vpsadbw(zmm1, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F16D48F6CB                , vpsadbw(zmm1, zmm2, zmm3)),
        INSTRUCTION_TEST(C5EDF68C2B80000000          , vpsadbw(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C5EDF6CB                    , vpsadbw(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(C5E9F6CB                    , vpsadbw(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(C5E9F68C2B80000000          , vpsadbw(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(8FE96098CA                  , vpshab(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(8FE9E8988C2B80000000        , vpshab(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(8FE958988C1A80000000        , vpshab(xmm1, xmmword_ptr(rdx, rbx, 1, 128), xmm4)),
        INSTRUCTION_TEST(8FE9589A8C1A80000000        , vpshad(xmm1, xmmword_ptr(rdx, rbx, 1, 128), xmm4)),
        INSTRUCTION_TEST(8FE9609ACA                  , vpshad(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(8FE9E89A8C2B80000000        , vpshad(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(8FE9589B8C1A80000000        , vpshaq(xmm1, xmmword_ptr(rdx, rbx, 1, 128), xmm4)),
        INSTRUCTION_TEST(8FE9E89B8C2B80000000        , vpshaq(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(8FE9609BCA                  , vpshaq(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(8FE96099CA                  , vpshaw(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(8FE9E8998C2B80000000        , vpshaw(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(8FE958998C1A80000000        , vpshaw(xmm1, xmmword_ptr(rdx, rbx, 1, 128), xmm4)),
        INSTRUCTION_TEST(8FE96094CA                  , vpshlb(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(8FE9E8948C2B80000000        , vpshlb(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(8FE958948C1A80000000        , vpshlb(xmm1, xmmword_ptr(rdx, rbx, 1, 128), xmm4)),
        INSTRUCTION_TEST(8FE9E8968C2B80000000        , vpshld(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(8FE958968C1A80000000        , vpshld(xmm1, xmmword_ptr(rdx, rbx, 1, 128), xmm4)),
        INSTRUCTION_TEST(8FE96096CA                  , vpshld(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(62F36D0871CB01              , vpshldd(xmm1, k0, xmm2, xmm3, Imm(1))),
        INSTRUCTION_TEST(62F36D08714C2B0801          , vpshldd(xmm1, k0, xmm2, xmmword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F36D2871CB01              , vpshldd(ymm1, k0, ymm2, ymm3, Imm(1))),
        INSTRUCTION_TEST(62F36D28714C2B0401          , vpshldd(ymm1, k0, ymm2, ymmword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F36D4871CB01              , vpshldd(zmm1, k0, zmm2, zmm3, Imm(1))),
        INSTRUCTION_TEST(62F36D48714C2B0201          , vpshldd(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F26D4871CB                , vpshldvd(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(62F26D48714C2B02            , vpshldvd(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F26D28714C2B04            , vpshldvd(ymm1, k0, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F26D08714C2B08            , vpshldvd(xmm1, k0, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F26D0871CB                , vpshldvd(xmm1, k0, xmm2, xmm3)),
        INSTRUCTION_TEST(62F26D2871CB                , vpshldvd(ymm1, k0, ymm2, ymm3)),
        INSTRUCTION_TEST(62F2ED0871CB                , vpshldvq(xmm1, k0, xmm2, xmm3)),
        INSTRUCTION_TEST(62F2ED08714C2B08            , vpshldvq(xmm1, k0, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F2ED2871CB                , vpshldvq(ymm1, k0, ymm2, ymm3)),
        INSTRUCTION_TEST(62F2ED28714C2B04            , vpshldvq(ymm1, k0, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F2ED4871CB                , vpshldvq(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(62F2ED48714C2B02            , vpshldvq(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F2ED48704C2B02            , vpshldvw(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F2ED0870CB                , vpshldvw(xmm1, k0, xmm2, xmm3)),
        INSTRUCTION_TEST(62F2ED08704C2B08            , vpshldvw(xmm1, k0, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F2ED2870CB                , vpshldvw(ymm1, k0, ymm2, ymm3)),
        INSTRUCTION_TEST(62F2ED28704C2B04            , vpshldvw(ymm1, k0, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F2ED4870CB                , vpshldvw(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(8FE958978C1A80000000        , vpshlq(xmm1, xmmword_ptr(rdx, rbx, 1, 128), xmm4)),
        INSTRUCTION_TEST(8FE9E8978C2B80000000        , vpshlq(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(8FE96097CA                  , vpshlq(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(8FE96095CA                  , vpshlw(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(8FE9E8958C2B80000000        , vpshlw(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(8FE958958C1A80000000        , vpshlw(xmm1, xmmword_ptr(rdx, rbx, 1, 128), xmm4)),
        INSTRUCTION_TEST(62F36D4873CB01              , vpshrdd(zmm1, k0, zmm2, zmm3, Imm(1))),
        INSTRUCTION_TEST(62F36D28734C2B0401          , vpshrdd(ymm1, k0, ymm2, ymmword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F36D48734C2B0201          , vpshrdd(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F36D08734C2B0801          , vpshrdd(xmm1, k0, xmm2, xmmword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F36D0873CB01              , vpshrdd(xmm1, k0, xmm2, xmm3, Imm(1))),
        INSTRUCTION_TEST(62F36D2873CB01              , vpshrdd(ymm1, k0, ymm2, ymm3, Imm(1))),
        INSTRUCTION_TEST(62F26D0873CB                , vpshrdvd(xmm1, k0, xmm2, xmm3)),
        INSTRUCTION_TEST(62F26D08734C2B08            , vpshrdvd(xmm1, k0, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F26D2873CB                , vpshrdvd(ymm1, k0, ymm2, ymm3)),
        INSTRUCTION_TEST(62F26D28734C2B04            , vpshrdvd(ymm1, k0, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F26D4873CB                , vpshrdvd(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(62F26D48734C2B02            , vpshrdvd(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F2ED4873CB                , vpshrdvq(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(62F2ED48734C2B02            , vpshrdvq(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F2ED28734C2B04            , vpshrdvq(ymm1, k0, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F2ED08734C2B08            , vpshrdvq(xmm1, k0, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F2ED0873CB                , vpshrdvq(xmm1, k0, xmm2, xmm3)),
        INSTRUCTION_TEST(62F2ED2873CB                , vpshrdvq(ymm1, k0, ymm2, ymm3)),
        INSTRUCTION_TEST(62F2ED0872CB                , vpshrdvw(xmm1, k0, xmm2, xmm3)),
        INSTRUCTION_TEST(62F2ED08724C2B08            , vpshrdvw(xmm1, k0, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F2ED2872CB                , vpshrdvw(ymm1, k0, ymm2, ymm3)),
        INSTRUCTION_TEST(62F2ED28724C2B04            , vpshrdvw(ymm1, k0, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F2ED4872CB                , vpshrdvw(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(62F2ED48724C2B02            , vpshrdvw(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F3ED0872CB01              , vpshrdw(xmm1, k0, xmm2, xmm3, Imm(1))),
        INSTRUCTION_TEST(62F3ED08724C2B0801          , vpshrdw(xmm1, k0, xmm2, xmmword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F3ED2872CB01              , vpshrdw(ymm1, k0, ymm2, ymm3, Imm(1))),
        INSTRUCTION_TEST(62F3ED28724C2B0401          , vpshrdw(ymm1, k0, ymm2, ymmword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F3ED4872CB01              , vpshrdw(zmm1, k0, zmm2, zmm3, Imm(1))),
        INSTRUCTION_TEST(62F3ED48724C2B0201          , vpshrdw(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F26D48004C2B02            , vpshufb(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F26D4800CB                , vpshufb(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(C4E26D008C2B80000000        , vpshufb(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E26D00CB                  , vpshufb(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(C4E269008C2B80000000        , vpshufb(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E26900CB                  , vpshufb(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(62F26D088FCB                , vpshufbitqmb(k1, k0, xmm2, xmm3)),
        INSTRUCTION_TEST(62F26D088F4C2B08            , vpshufbitqmb(k1, k0, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F26D288FCB                , vpshufbitqmb(k1, k0, ymm2, ymm3)),
        INSTRUCTION_TEST(62F26D288F4C2B04            , vpshufbitqmb(k1, k0, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F26D488FCB                , vpshufbitqmb(k1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(62F26D488F4C2B02            , vpshufbitqmb(k1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62E17D48709A002000007B      , vpshufd(zmm19, k0, zmmword_ptr(rdx, 8192), Imm(123))),
        INSTRUCTION_TEST(62D17D4870FC7B              , vpshufd(zmm7, k0, zmm12, Imm(123))),
        INSTRUCTION_TEST(62D17D4870FCAB              , vpshufd(zmm7, k0, zmm12, Imm(171))),
        INSTRUCTION_TEST(62B17D4870BCF0341200007B    , vpshufd(zmm7, k0, zmmword_ptr(rax, r14, 8, 4660), Imm(123))),
        INSTRUCTION_TEST(62F17D4870BA002000007B      , vpshufd(zmm7, k0, zmmword_ptr(rdx, 8192), Imm(123))),
        INSTRUCTION_TEST(62F17D48707A807B            , vpshufd(zmm7, k0, zmmword_ptr(rdx, -8192), Imm(123))),
        INSTRUCTION_TEST(62F17D4870BAC0DFFFFF7B      , vpshufd(zmm7, k0, zmmword_ptr(rdx, -8256), Imm(123))),
        INSTRUCTION_TEST(62F17D48707A7F7B            , vpshufd(zmm7, k0, zmmword_ptr(rdx, 8128), Imm(123))),
        INSTRUCTION_TEST(62E17D48705A7F7B            , vpshufd(zmm19, k0, zmmword_ptr(rdx, 8128), Imm(123))),
        INSTRUCTION_TEST(62F17D4870397B              , vpshufd(zmm7, k0, zmmword_ptr(rcx), Imm(123))),
        INSTRUCTION_TEST(62E17D48705A807B            , vpshufd(zmm19, k0, zmmword_ptr(rdx, -8192), Imm(123))),
        INSTRUCTION_TEST(62817D4870D9AB              , vpshufd(zmm19, k0, zmm25, Imm(171))),
        INSTRUCTION_TEST(62E17D4870197B              , vpshufd(zmm19, k0, zmmword_ptr(rcx), Imm(123))),
        INSTRUCTION_TEST(62A17D48709CF0230100007B    , vpshufd(zmm19, k0, zmmword_ptr(rax, r14, 8, 291), Imm(123))),
        INSTRUCTION_TEST(62817D4870D97B              , vpshufd(zmm19, k0, zmm25, Imm(123))),
        INSTRUCTION_TEST(62F17D48704C1A0201          , vpshufd(zmm1, k0, zmmword_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F17D4870CA01              , vpshufd(zmm1, k0, zmm2, Imm(1))),
        INSTRUCTION_TEST(C5FD708C1A8000000001        , vpshufd(ymm1, ymmword_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(C5FD70CA01                  , vpshufd(ymm1, ymm2, Imm(1))),
        INSTRUCTION_TEST(C5F9708C1A8000000001        , vpshufd(xmm1, xmmword_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(C5F970CA01                  , vpshufd(xmm1, xmm2, Imm(1))),
        INSTRUCTION_TEST(62E17D48709AC0DFFFFF7B      , vpshufd(zmm19, k0, zmmword_ptr(rdx, -8256), Imm(123))),
        INSTRUCTION_TEST(62F17E4870CA01              , vpshufhw(zmm1, k0, zmm2, Imm(1))),
        INSTRUCTION_TEST(62F17E48704C1A0201          , vpshufhw(zmm1, k0, zmmword_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(C5FE708C1A8000000001        , vpshufhw(ymm1, ymmword_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(C5FA70CA01                  , vpshufhw(xmm1, xmm2, Imm(1))),
        INSTRUCTION_TEST(C5FA708C1A8000000001        , vpshufhw(xmm1, xmmword_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(C5FE70CA01                  , vpshufhw(ymm1, ymm2, Imm(1))),
        INSTRUCTION_TEST(62F17F48704C1A0201          , vpshuflw(zmm1, k0, zmmword_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F17F4870CA01              , vpshuflw(zmm1, k0, zmm2, Imm(1))),
        INSTRUCTION_TEST(C5FF708C1A8000000001        , vpshuflw(ymm1, ymmword_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(C5FB70CA01                  , vpshuflw(xmm1, xmm2, Imm(1))),
        INSTRUCTION_TEST(C5FB708C1A8000000001        , vpshuflw(xmm1, xmmword_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(C5FF70CA01                  , vpshuflw(ymm1, ymm2, Imm(1))),
        INSTRUCTION_TEST(C4E26908CB                  , vpsignb(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(C4E269088C2B80000000        , vpsignb(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E26D08CB                  , vpsignb(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(C4E26D088C2B80000000        , vpsignb(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E2690ACB                  , vpsignd(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(C4E2690A8C2B80000000        , vpsignd(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E26D0ACB                  , vpsignd(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(C4E26D0A8C2B80000000        , vpsignd(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E26D098C2B80000000        , vpsignw(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E26D09CB                  , vpsignw(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(C4E269098C2B80000000        , vpsignw(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E26909CB                  , vpsignw(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(62F1154072317B              , vpslld(zmm29, k0, zmmword_ptr(rcx), Imm(123))),
        INSTRUCTION_TEST(62F125407272807B            , vpslld(zmm27, k0, zmmword_ptr(rdx, -8192), Imm(123))),
        INSTRUCTION_TEST(62F1254072B2C0DFFFFF7B      , vpslld(zmm27, k0, zmmword_ptr(rdx, -8256), Imm(123))),
        INSTRUCTION_TEST(62F1254072727F7B            , vpslld(zmm27, k0, zmmword_ptr(rdx, 8128), Imm(123))),
        INSTRUCTION_TEST(62F1254072B2002000007B      , vpslld(zmm27, k0, zmmword_ptr(rdx, 8192), Imm(123))),
        INSTRUCTION_TEST(6291154072F07B              , vpslld(zmm29, k0, zmm24, Imm(123))),
        INSTRUCTION_TEST(6291154072F0AB              , vpslld(zmm29, k0, zmm24, Imm(171))),
        INSTRUCTION_TEST(62B1154072B4F0230100007B    , vpslld(zmm29, k0, zmmword_ptr(rax, r14, 8, 291), Imm(123))),
        INSTRUCTION_TEST(62F115407272807B            , vpslld(zmm29, k0, zmmword_ptr(rdx, -8192), Imm(123))),
        INSTRUCTION_TEST(62F1254072317B              , vpslld(zmm27, k0, zmmword_ptr(rcx), Imm(123))),
        INSTRUCTION_TEST(62F1154072727F7B            , vpslld(zmm29, k0, zmmword_ptr(rdx, 8128), Imm(123))),
        INSTRUCTION_TEST(62F1154072B2002000007B      , vpslld(zmm29, k0, zmmword_ptr(rdx, 8192), Imm(123))),
        INSTRUCTION_TEST(62D14D40F2FE                , vpslld(zmm7, k0, zmm22, xmm14)),
        INSTRUCTION_TEST(62B14D40F2BCF034120000      , vpslld(zmm7, k0, zmm22, xmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62F14D40F239                , vpslld(zmm7, k0, zmm22, xmmword_ptr(rcx))),
        INSTRUCTION_TEST(62F14D40F27A80              , vpslld(zmm7, k0, zmm22, xmmword_ptr(rdx, -2048))),
        INSTRUCTION_TEST(62F14D40F2BAF0F7FFFF        , vpslld(zmm7, k0, zmm22, xmmword_ptr(rdx, -2064))),
        INSTRUCTION_TEST(62F14D40F2BA00080000        , vpslld(zmm7, k0, zmm22, xmmword_ptr(rdx, 2048))),
        INSTRUCTION_TEST(62F1154072B2C0DFFFFF7B      , vpslld(zmm29, k0, zmmword_ptr(rdx, -8256), Imm(123))),
        INSTRUCTION_TEST(62B1254072B4F0341200007B    , vpslld(zmm27, k0, zmmword_ptr(rax, r14, 8, 4660), Imm(123))),
        INSTRUCTION_TEST(62F14D40F27A7F              , vpslld(zmm7, k0, zmm22, xmmword_ptr(rdx, 2032))),
        INSTRUCTION_TEST(62F1254072F57B              , vpslld(zmm27, k0, zmm5, Imm(123))),
        INSTRUCTION_TEST(62F1254072F5AB              , vpslld(zmm27, k0, zmm5, Imm(171))),
        INSTRUCTION_TEST(C5F172F201                  , vpslld(xmm1, xmm2, Imm(1))),
        INSTRUCTION_TEST(C5E9F2CB                    , vpslld(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(C5E9F28C2B80000000          , vpslld(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F1750872741A0801          , vpslld(xmm1, k0, xmmword_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(C5F572F201                  , vpslld(ymm1, ymm2, Imm(1))),
        INSTRUCTION_TEST(C5EDF2CB                    , vpslld(ymm1, ymm2, xmm3)),
        INSTRUCTION_TEST(C5EDF28C2B80000000          , vpslld(ymm1, ymm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F1752872741A0401          , vpslld(ymm1, k0, ymmword_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F1754872F201              , vpslld(zmm1, k0, zmm2, Imm(1))),
        INSTRUCTION_TEST(62F16D48F24C2B08            , vpslld(zmm1, k0, zmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F1754872741A0201          , vpslld(zmm1, k0, zmmword_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62E14D48F2CB                , vpslld(zmm17, k0, zmm6, xmm3)),
        INSTRUCTION_TEST(62A14D48F28CF023010000      , vpslld(zmm17, k0, zmm6, xmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62E14D48F209                , vpslld(zmm17, k0, zmm6, xmmword_ptr(rcx))),
        INSTRUCTION_TEST(62E14D48F24A80              , vpslld(zmm17, k0, zmm6, xmmword_ptr(rdx, -2048))),
        INSTRUCTION_TEST(62E14D48F28AF0F7FFFF        , vpslld(zmm17, k0, zmm6, xmmword_ptr(rdx, -2064))),
        INSTRUCTION_TEST(62E14D48F24A7F              , vpslld(zmm17, k0, zmm6, xmmword_ptr(rdx, 2032))),
        INSTRUCTION_TEST(62E14D48F28A00080000        , vpslld(zmm17, k0, zmm6, xmmword_ptr(rdx, 2048))),
        INSTRUCTION_TEST(62F16D48F2CB                , vpslld(zmm1, k0, zmm2, xmm3)),
        INSTRUCTION_TEST(62F1754873FA01              , vpslldq(zmm1, zmm2, Imm(1))),
        INSTRUCTION_TEST(62F17548737C1A0201          , vpslldq(zmm1, zmmword_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F17528737C1A0401          , vpslldq(ymm1, ymmword_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(C5F573FA01                  , vpslldq(ymm1, ymm2, Imm(1))),
        INSTRUCTION_TEST(62F17508737C1A0801          , vpslldq(xmm1, xmmword_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(C5F173FA01                  , vpslldq(xmm1, xmm2, Imm(1))),
        INSTRUCTION_TEST(6261ED48F3AAF0F7FFFF        , vpsllq(zmm29, k0, zmm2, xmmword_ptr(rdx, -2064))),
        INSTRUCTION_TEST(6261ED48F36A7F              , vpsllq(zmm29, k0, zmm2, xmmword_ptr(rdx, 2032))),
        INSTRUCTION_TEST(6261ED48F3AA00080000        , vpsllq(zmm29, k0, zmm2, xmmword_ptr(rdx, 2048))),
        INSTRUCTION_TEST(62F1DD4873F67B              , vpsllq(zmm4, k0, zmm6, Imm(123))),
        INSTRUCTION_TEST(62F1DD4873F6AB              , vpsllq(zmm4, k0, zmm6, Imm(171))),
        INSTRUCTION_TEST(62B1DD4873B4F0341200007B    , vpsllq(zmm4, k0, zmmword_ptr(rax, r14, 8, 4660), Imm(123))),
        INSTRUCTION_TEST(62F1DD4873317B              , vpsllq(zmm4, k0, zmmword_ptr(rcx), Imm(123))),
        INSTRUCTION_TEST(62F1DD487372807B            , vpsllq(zmm4, k0, zmmword_ptr(rdx, -8192), Imm(123))),
        INSTRUCTION_TEST(62F1DD4873B2C0DFFFFF7B      , vpsllq(zmm4, k0, zmmword_ptr(rdx, -8256), Imm(123))),
        INSTRUCTION_TEST(62F1DD4873727F7B            , vpsllq(zmm4, k0, zmmword_ptr(rdx, 8128), Imm(123))),
        INSTRUCTION_TEST(62F1DD4873B2002000007B      , vpsllq(zmm4, k0, zmmword_ptr(rdx, 8192), Imm(123))),
        INSTRUCTION_TEST(62D1CD4873F37B              , vpsllq(zmm6, k0, zmm11, Imm(123))),
        INSTRUCTION_TEST(62D1CD4873F3AB              , vpsllq(zmm6, k0, zmm11, Imm(171))),
        INSTRUCTION_TEST(62B1CD4873B4F0230100007B    , vpsllq(zmm6, k0, zmmword_ptr(rax, r14, 8, 291), Imm(123))),
        INSTRUCTION_TEST(62F1CD4873317B              , vpsllq(zmm6, k0, zmmword_ptr(rcx), Imm(123))),
        INSTRUCTION_TEST(62F1CD487372807B            , vpsllq(zmm6, k0, zmmword_ptr(rdx, -8192), Imm(123))),
        INSTRUCTION_TEST(62F1CD4873B2C0DFFFFF7B      , vpsllq(zmm6, k0, zmmword_ptr(rdx, -8256), Imm(123))),
        INSTRUCTION_TEST(62F1CD4873727F7B            , vpsllq(zmm6, k0, zmmword_ptr(rdx, 8128), Imm(123))),
        INSTRUCTION_TEST(62F1CD4873B2002000007B      , vpsllq(zmm6, k0, zmmword_ptr(rdx, 8192), Imm(123))),
        INSTRUCTION_TEST(6261ED48F36A80              , vpsllq(zmm29, k0, zmm2, xmmword_ptr(rdx, -2048))),
        INSTRUCTION_TEST(6261ED48F329                , vpsllq(zmm29, k0, zmm2, xmmword_ptr(rcx))),
        INSTRUCTION_TEST(6221F540F3D7                , vpsllq(zmm26, k0, zmm17, xmm23)),
        INSTRUCTION_TEST(6201ED48F3EB                , vpsllq(zmm29, k0, zmm2, xmm27)),
        INSTRUCTION_TEST(C5F173F201                  , vpsllq(xmm1, xmm2, Imm(1))),
        INSTRUCTION_TEST(C5E9F3CB                    , vpsllq(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(C5E9F38C2B80000000          , vpsllq(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F1F50873741A0801          , vpsllq(xmm1, k0, xmmword_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(C5F573F201                  , vpsllq(ymm1, ymm2, Imm(1))),
        INSTRUCTION_TEST(6221ED48F3ACF034120000      , vpsllq(zmm29, k0, zmm2, xmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(C5EDF38C2B80000000          , vpsllq(ymm1, ymm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F1F52873741A0401          , vpsllq(ymm1, k0, ymmword_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F1F54873F201              , vpsllq(zmm1, k0, zmm2, Imm(1))),
        INSTRUCTION_TEST(C5EDF3CB                    , vpsllq(ymm1, ymm2, xmm3)),
        INSTRUCTION_TEST(62F1ED48F34C2B08            , vpsllq(zmm1, k0, zmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F1F54873741A0201          , vpsllq(zmm1, k0, zmmword_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(6221F540F394F023010000      , vpsllq(zmm26, k0, zmm17, xmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(6261F540F311                , vpsllq(zmm26, k0, zmm17, xmmword_ptr(rcx))),
        INSTRUCTION_TEST(6261F540F35280              , vpsllq(zmm26, k0, zmm17, xmmword_ptr(rdx, -2048))),
        INSTRUCTION_TEST(6261F540F392F0F7FFFF        , vpsllq(zmm26, k0, zmm17, xmmword_ptr(rdx, -2064))),
        INSTRUCTION_TEST(6261F540F3527F              , vpsllq(zmm26, k0, zmm17, xmmword_ptr(rdx, 2032))),
        INSTRUCTION_TEST(6261F540F39200080000        , vpsllq(zmm26, k0, zmm17, xmmword_ptr(rdx, 2048))),
        INSTRUCTION_TEST(62F1ED48F3CB                , vpsllq(zmm1, k0, zmm2, xmm3)),
        INSTRUCTION_TEST(62721D48479200200000        , vpsllvd(zmm10, k0, zmm12, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(6242654847EF                , vpsllvd(zmm29, k0, zmm3, zmm15)),
        INSTRUCTION_TEST(6222654847ACF034120000      , vpsllvd(zmm29, k0, zmm3, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(626265484729                , vpsllvd(zmm29, k0, zmm3, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(6262654847AA00200000        , vpsllvd(zmm29, k0, zmm3, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(6262654847AAC0DFFFFF        , vpsllvd(zmm29, k0, zmm3, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62626548476A7F              , vpsllvd(zmm29, k0, zmm3, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62721D4847527F              , vpsllvd(zmm10, k0, zmm12, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62626548476A80              , vpsllvd(zmm29, k0, zmm3, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62721D484792C0DFFFFF        , vpsllvd(zmm10, k0, zmm12, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62F26D4847CB                , vpsllvd(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(62721D484711                , vpsllvd(zmm10, k0, zmm12, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62321D484794F023010000      , vpsllvd(zmm10, k0, zmm12, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62321D4847D6                , vpsllvd(zmm10, k0, zmm12, zmm22)),
        INSTRUCTION_TEST(62F26D48474C2B02            , vpsllvd(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E26D478C2B80000000        , vpsllvd(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E26D47CB                  , vpsllvd(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(C4E269478C2B80000000        , vpsllvd(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E26947CB                  , vpsllvd(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(62721D48475280              , vpsllvd(zmm10, k0, zmm12, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62E2D548478200200000        , vpsllvq(zmm16, k0, zmm5, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62D2954047E5                , vpsllvq(zmm4, k0, zmm29, zmm13)),
        INSTRUCTION_TEST(62B2954047A4F034120000      , vpsllvq(zmm4, k0, zmm29, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62F2954047627F              , vpsllvq(zmm4, k0, zmm29, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62F29540476280              , vpsllvq(zmm4, k0, zmm29, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62F2954047A2C0DFFFFF        , vpsllvq(zmm4, k0, zmm29, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62E2D54847427F              , vpsllvq(zmm16, k0, zmm5, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62F2954047A200200000        , vpsllvq(zmm4, k0, zmm29, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62F295404721                , vpsllvq(zmm4, k0, zmm29, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62E2D5484782C0DFFFFF        , vpsllvq(zmm16, k0, zmm5, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(6282D54847C2                , vpsllvq(zmm16, k0, zmm5, zmm26)),
        INSTRUCTION_TEST(62E2D5484701                , vpsllvq(zmm16, k0, zmm5, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62A2D5484784F023010000      , vpsllvq(zmm16, k0, zmm5, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62F2ED48474C2B02            , vpsllvq(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F2ED4847CB                , vpsllvq(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(C4E2ED478C2B80000000        , vpsllvq(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E2ED47CB                  , vpsllvq(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(C4E2E9478C2B80000000        , vpsllvq(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E2E947CB                  , vpsllvq(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(62E2D548474280              , vpsllvq(zmm16, k0, zmm5, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62F2ED48124C2B02            , vpsllvw(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F2ED28124C2B04            , vpsllvw(ymm1, k0, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F2ED4812CB                , vpsllvw(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(62F2ED08124C2B08            , vpsllvw(xmm1, k0, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F2ED0812CB                , vpsllvw(xmm1, k0, xmm2, xmm3)),
        INSTRUCTION_TEST(62F2ED2812CB                , vpsllvw(ymm1, k0, ymm2, ymm3)),
        INSTRUCTION_TEST(62F1754871741A0201          , vpsllw(zmm1, k0, zmmword_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F16D48F14C2B08            , vpsllw(zmm1, k0, zmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F16D48F1CB                , vpsllw(zmm1, k0, zmm2, xmm3)),
        INSTRUCTION_TEST(62F1752871741A0401          , vpsllw(ymm1, k0, ymmword_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(C5EDF18C2B80000000          , vpsllw(ymm1, ymm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F1754871F201              , vpsllw(zmm1, k0, zmm2, Imm(1))),
        INSTRUCTION_TEST(C5F571F201                  , vpsllw(ymm1, ymm2, Imm(1))),
        INSTRUCTION_TEST(62F1750871741A0801          , vpsllw(xmm1, k0, xmmword_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(C5E9F18C2B80000000          , vpsllw(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C5E9F1CB                    , vpsllw(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(C5F171F201                  , vpsllw(xmm1, xmm2, Imm(1))),
        INSTRUCTION_TEST(C5EDF1CB                    , vpsllw(ymm1, ymm2, xmm3)),
        INSTRUCTION_TEST(62F125407262807B            , vpsrad(zmm27, k0, zmmword_ptr(rdx, -8192), Imm(123))),
        INSTRUCTION_TEST(62E12D48E25A7F              , vpsrad(zmm19, k0, zmm10, xmmword_ptr(rdx, 2032))),
        INSTRUCTION_TEST(62E12D48E29A00080000        , vpsrad(zmm19, k0, zmm10, xmmword_ptr(rdx, 2048))),
        INSTRUCTION_TEST(62E12D48E29AF0F7FFFF        , vpsrad(zmm19, k0, zmm10, xmmword_ptr(rdx, -2064))),
        INSTRUCTION_TEST(62F1254072E57B              , vpsrad(zmm27, k0, zmm5, Imm(123))),
        INSTRUCTION_TEST(62F1254072E5AB              , vpsrad(zmm27, k0, zmm5, Imm(171))),
        INSTRUCTION_TEST(62B1254072A4F0230100007B    , vpsrad(zmm27, k0, zmmword_ptr(rax, r14, 8, 291), Imm(123))),
        INSTRUCTION_TEST(62F1254072217B              , vpsrad(zmm27, k0, zmmword_ptr(rcx), Imm(123))),
        INSTRUCTION_TEST(62F1254072A2C0DFFFFF7B      , vpsrad(zmm27, k0, zmmword_ptr(rdx, -8256), Imm(123))),
        INSTRUCTION_TEST(62F13D487262807B            , vpsrad(zmm8, k0, zmmword_ptr(rdx, -8192), Imm(123))),
        INSTRUCTION_TEST(62F1254072A2002000007B      , vpsrad(zmm27, k0, zmmword_ptr(rdx, 8192), Imm(123))),
        INSTRUCTION_TEST(62B13D4872E47B              , vpsrad(zmm8, k0, zmm20, Imm(123))),
        INSTRUCTION_TEST(62B13D4872E4AB              , vpsrad(zmm8, k0, zmm20, Imm(171))),
        INSTRUCTION_TEST(62B13D4872A4F0341200007B    , vpsrad(zmm8, k0, zmmword_ptr(rax, r14, 8, 4660), Imm(123))),
        INSTRUCTION_TEST(62F13D4872217B              , vpsrad(zmm8, k0, zmmword_ptr(rcx), Imm(123))),
        INSTRUCTION_TEST(62F13D4872A2C0DFFFFF7B      , vpsrad(zmm8, k0, zmmword_ptr(rdx, -8256), Imm(123))),
        INSTRUCTION_TEST(62F13D4872627F7B            , vpsrad(zmm8, k0, zmmword_ptr(rdx, 8128), Imm(123))),
        INSTRUCTION_TEST(62E12D48E25A80              , vpsrad(zmm19, k0, zmm10, xmmword_ptr(rdx, -2048))),
        INSTRUCTION_TEST(62F1254072627F7B            , vpsrad(zmm27, k0, zmmword_ptr(rdx, 8128), Imm(123))),
        INSTRUCTION_TEST(62E12D48E219                , vpsrad(zmm19, k0, zmm10, xmmword_ptr(rcx))),
        INSTRUCTION_TEST(62F13D4872A2002000007B      , vpsrad(zmm8, k0, zmmword_ptr(rdx, 8192), Imm(123))),
        INSTRUCTION_TEST(62812D48E2D9                , vpsrad(zmm19, k0, zmm10, xmm25)),
        INSTRUCTION_TEST(62A12D48E29CF034120000      , vpsrad(zmm19, k0, zmm10, xmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(C5F172E201                  , vpsrad(xmm1, xmm2, Imm(1))),
        INSTRUCTION_TEST(C5E9E2CB                    , vpsrad(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(C5E9E28C2B80000000          , vpsrad(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F1750872641A0801          , vpsrad(xmm1, k0, xmmword_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(C5F572E201                  , vpsrad(ymm1, ymm2, Imm(1))),
        INSTRUCTION_TEST(C5EDE28C2B80000000          , vpsrad(ymm1, ymm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F1752872641A0401          , vpsrad(ymm1, k0, ymmword_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F1754872E201              , vpsrad(zmm1, k0, zmm2, Imm(1))),
        INSTRUCTION_TEST(C5EDE2CB                    , vpsrad(ymm1, ymm2, xmm3)),
        INSTRUCTION_TEST(62F16D48E24C2B08            , vpsrad(zmm1, k0, zmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62E10D48E29200080000        , vpsrad(zmm18, k0, zmm14, xmmword_ptr(rdx, 2048))),
        INSTRUCTION_TEST(62F16D48E2CB                , vpsrad(zmm1, k0, zmm2, xmm3)),
        INSTRUCTION_TEST(62E10D48E292F0F7FFFF        , vpsrad(zmm18, k0, zmm14, xmmword_ptr(rdx, -2064))),
        INSTRUCTION_TEST(62E10D48E25280              , vpsrad(zmm18, k0, zmm14, xmmword_ptr(rdx, -2048))),
        INSTRUCTION_TEST(62E10D48E2527F              , vpsrad(zmm18, k0, zmm14, xmmword_ptr(rdx, 2032))),
        INSTRUCTION_TEST(62A10D48E294F023010000      , vpsrad(zmm18, k0, zmm14, xmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62E10D48E2D6                , vpsrad(zmm18, k0, zmm14, xmm6)),
        INSTRUCTION_TEST(62F1754872641A0201          , vpsrad(zmm1, k0, zmmword_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62E10D48E211                , vpsrad(zmm18, k0, zmm14, xmmword_ptr(rcx))),
        INSTRUCTION_TEST(62F195407262807B            , vpsraq(zmm29, k0, zmmword_ptr(rdx, -8192), Imm(123))),
        INSTRUCTION_TEST(62E1C540E2BAF0F7FFFF        , vpsraq(zmm23, k0, zmm23, xmmword_ptr(rdx, -2064))),
        INSTRUCTION_TEST(62E1C540E2BA00080000        , vpsraq(zmm23, k0, zmm23, xmmword_ptr(rdx, 2048))),
        INSTRUCTION_TEST(62F1954072E57B              , vpsraq(zmm29, k0, zmm5, Imm(123))),
        INSTRUCTION_TEST(62F1954072E5AB              , vpsraq(zmm29, k0, zmm5, Imm(171))),
        INSTRUCTION_TEST(62B1954072A4F0230100007B    , vpsraq(zmm29, k0, zmmword_ptr(rax, r14, 8, 291), Imm(123))),
        INSTRUCTION_TEST(62F1954072217B              , vpsraq(zmm29, k0, zmmword_ptr(rcx), Imm(123))),
        INSTRUCTION_TEST(62F1954072A2C0DFFFFF7B      , vpsraq(zmm29, k0, zmmword_ptr(rdx, -8256), Imm(123))),
        INSTRUCTION_TEST(62F1E54872A2002000007B      , vpsraq(zmm3, k0, zmmword_ptr(rdx, 8192), Imm(123))),
        INSTRUCTION_TEST(62F1954072A2002000007B      , vpsraq(zmm29, k0, zmmword_ptr(rdx, 8192), Imm(123))),
        INSTRUCTION_TEST(62B1E54872E27B              , vpsraq(zmm3, k0, zmm18, Imm(123))),
        INSTRUCTION_TEST(62B1E54872E2AB              , vpsraq(zmm3, k0, zmm18, Imm(171))),
        INSTRUCTION_TEST(62B1E54872A4F0341200007B    , vpsraq(zmm3, k0, zmmword_ptr(rax, r14, 8, 4660), Imm(123))),
        INSTRUCTION_TEST(62F1E54872217B              , vpsraq(zmm3, k0, zmmword_ptr(rcx), Imm(123))),
        INSTRUCTION_TEST(62F1E5487262807B            , vpsraq(zmm3, k0, zmmword_ptr(rdx, -8192), Imm(123))),
        INSTRUCTION_TEST(62F1E54872A2C0DFFFFF7B      , vpsraq(zmm3, k0, zmmword_ptr(rdx, -8256), Imm(123))),
        INSTRUCTION_TEST(62F1E54872627F7B            , vpsraq(zmm3, k0, zmmword_ptr(rdx, 8128), Imm(123))),
        INSTRUCTION_TEST(62E1C540E27A80              , vpsraq(zmm23, k0, zmm23, xmmword_ptr(rdx, -2048))),
        INSTRUCTION_TEST(62F1954072627F7B            , vpsraq(zmm29, k0, zmmword_ptr(rdx, 8128), Imm(123))),
        INSTRUCTION_TEST(62E1C540E239                , vpsraq(zmm23, k0, zmm23, xmmword_ptr(rcx))),
        INSTRUCTION_TEST(62E1C540E27A7F              , vpsraq(zmm23, k0, zmm23, xmmword_ptr(rdx, 2032))),
        INSTRUCTION_TEST(62A1C540E2FA                , vpsraq(zmm23, k0, zmm23, xmm18)),
        INSTRUCTION_TEST(62A1C540E2BCF034120000      , vpsraq(zmm23, k0, zmm23, xmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62F1F50872E201              , vpsraq(xmm1, k0, xmm2, Imm(1))),
        INSTRUCTION_TEST(62F1ED08E2CB                , vpsraq(xmm1, k0, xmm2, xmm3)),
        INSTRUCTION_TEST(62F1ED08E24C2B08            , vpsraq(xmm1, k0, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F1F50872641A0801          , vpsraq(xmm1, k0, xmmword_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F1ED28E2CB                , vpsraq(ymm1, k0, ymm2, xmm3)),
        INSTRUCTION_TEST(62F1ED28E24C2B08            , vpsraq(ymm1, k0, ymm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F1F52872641A0401          , vpsraq(ymm1, k0, ymmword_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F1F54872E201              , vpsraq(zmm1, k0, zmm2, Imm(1))),
        INSTRUCTION_TEST(62F1F52872E201              , vpsraq(ymm1, k0, ymm2, Imm(1))),
        INSTRUCTION_TEST(62F1ED48E24C2B08            , vpsraq(zmm1, k0, zmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F1F54872641A0201          , vpsraq(zmm1, k0, zmmword_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62A1F540E2EE                , vpsraq(zmm21, k0, zmm17, xmm22)),
        INSTRUCTION_TEST(62A1F540E2ACF023010000      , vpsraq(zmm21, k0, zmm17, xmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62E1F540E229                , vpsraq(zmm21, k0, zmm17, xmmword_ptr(rcx))),
        INSTRUCTION_TEST(62E1F540E26A80              , vpsraq(zmm21, k0, zmm17, xmmword_ptr(rdx, -2048))),
        INSTRUCTION_TEST(62E1F540E2AAF0F7FFFF        , vpsraq(zmm21, k0, zmm17, xmmword_ptr(rdx, -2064))),
        INSTRUCTION_TEST(62E1F540E26A7F              , vpsraq(zmm21, k0, zmm17, xmmword_ptr(rdx, 2032))),
        INSTRUCTION_TEST(62F1ED48E2CB                , vpsraq(zmm1, k0, zmm2, xmm3)),
        INSTRUCTION_TEST(62E1F540E2AA00080000        , vpsraq(zmm21, k0, zmm17, xmmword_ptr(rdx, 2048))),
        INSTRUCTION_TEST(62E25548468A00200000        , vpsravd(zmm17, k0, zmm5, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(6222154046E9                , vpsravd(zmm29, k0, zmm29, zmm17)),
        INSTRUCTION_TEST(6222154046ACF023010000      , vpsravd(zmm29, k0, zmm29, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(626215404629                , vpsravd(zmm29, k0, zmm29, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(6262154046AA00200000        , vpsravd(zmm29, k0, zmm29, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(6262154046AAC0DFFFFF        , vpsravd(zmm29, k0, zmm29, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62621540466A7F              , vpsravd(zmm29, k0, zmm29, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62E25548464A7F              , vpsravd(zmm17, k0, zmm5, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62621540466A80              , vpsravd(zmm29, k0, zmm29, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62E25548468AC0DFFFFF        , vpsravd(zmm17, k0, zmm5, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(C4E26D468C2B80000000        , vpsravd(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62E255484609                , vpsravd(zmm17, k0, zmm5, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62A25548468CF034120000      , vpsravd(zmm17, k0, zmm5, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62E2554846CE                , vpsravd(zmm17, k0, zmm5, zmm6)),
        INSTRUCTION_TEST(62F26D48464C2B02            , vpsravd(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F26D4846CB                , vpsravd(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(C4E26D46CB                  , vpsravd(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(C4E269468C2B80000000        , vpsravd(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E26946CB                  , vpsravd(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(62E25548464A80              , vpsravd(zmm17, k0, zmm5, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62E2DD4046B200200000        , vpsravq(zmm22, k0, zmm20, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(6292BD4846DA                , vpsravq(zmm3, k0, zmm8, zmm26)),
        INSTRUCTION_TEST(62B2BD48469CF034120000      , vpsravq(zmm3, k0, zmm8, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62F2BD48469A00200000        , vpsravq(zmm3, k0, zmm8, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62F2BD48465A80              , vpsravq(zmm3, k0, zmm8, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62F2BD48469AC0DFFFFF        , vpsravq(zmm3, k0, zmm8, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62F2BD48465A7F              , vpsravq(zmm3, k0, zmm8, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62F2BD484619                , vpsravq(zmm3, k0, zmm8, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62E2DD4046B2C0DFFFFF        , vpsravq(zmm22, k0, zmm20, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62E2DD4046727F              , vpsravq(zmm22, k0, zmm20, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62E2DD404631                , vpsravq(zmm22, k0, zmm20, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62A2DD4046B4F023010000      , vpsravq(zmm22, k0, zmm20, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62E2DD4046F2                , vpsravq(zmm22, k0, zmm20, zmm2)),
        INSTRUCTION_TEST(62F2ED48464C2B02            , vpsravq(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F2ED4846CB                , vpsravq(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(62F2ED28464C2B04            , vpsravq(ymm1, k0, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F2ED2846CB                , vpsravq(ymm1, k0, ymm2, ymm3)),
        INSTRUCTION_TEST(62F2ED08464C2B08            , vpsravq(xmm1, k0, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F2ED0846CB                , vpsravq(xmm1, k0, xmm2, xmm3)),
        INSTRUCTION_TEST(62E2DD40467280              , vpsravq(zmm22, k0, zmm20, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62F2ED48114C2B02            , vpsravw(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F2ED4811CB                , vpsravw(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(62F2ED28114C2B04            , vpsravw(ymm1, k0, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F2ED0811CB                , vpsravw(xmm1, k0, xmm2, xmm3)),
        INSTRUCTION_TEST(62F2ED08114C2B08            , vpsravw(xmm1, k0, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F2ED2811CB                , vpsravw(ymm1, k0, ymm2, ymm3)),
        INSTRUCTION_TEST(62F1752871641A0401          , vpsraw(ymm1, k0, ymmword_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F16D48E14C2B08            , vpsraw(zmm1, k0, zmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F16D48E1CB                , vpsraw(zmm1, k0, zmm2, xmm3)),
        INSTRUCTION_TEST(62F1754871641A0201          , vpsraw(zmm1, k0, zmmword_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(C5EDE18C2B80000000          , vpsraw(ymm1, ymm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F1754871E201              , vpsraw(zmm1, k0, zmm2, Imm(1))),
        INSTRUCTION_TEST(C5F571E201                  , vpsraw(ymm1, ymm2, Imm(1))),
        INSTRUCTION_TEST(62F1750871641A0801          , vpsraw(xmm1, k0, xmmword_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(C5E9E18C2B80000000          , vpsraw(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C5E9E1CB                    , vpsraw(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(C5F171E201                  , vpsraw(xmm1, xmm2, Imm(1))),
        INSTRUCTION_TEST(C5EDE1CB                    , vpsraw(ymm1, ymm2, xmm3)),
        INSTRUCTION_TEST(62F1254872117B              , vpsrld(zmm11, k0, zmmword_ptr(rcx), Imm(123))),
        INSTRUCTION_TEST(62F125487252807B            , vpsrld(zmm11, k0, zmmword_ptr(rdx, -8192), Imm(123))),
        INSTRUCTION_TEST(62F125487292C0DFFFFF7B      , vpsrld(zmm11, k0, zmmword_ptr(rdx, -8256), Imm(123))),
        INSTRUCTION_TEST(62F1254872527F7B            , vpsrld(zmm11, k0, zmmword_ptr(rdx, 8128), Imm(123))),
        INSTRUCTION_TEST(62F125487292002000007B      , vpsrld(zmm11, k0, zmmword_ptr(rdx, 8192), Imm(123))),
        INSTRUCTION_TEST(62216540D2CF                , vpsrld(zmm25, k0, zmm19, xmm23)),
        INSTRUCTION_TEST(62216540D28CF034120000      , vpsrld(zmm25, k0, zmm19, xmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62616540D209                , vpsrld(zmm25, k0, zmm19, xmmword_ptr(rcx))),
        INSTRUCTION_TEST(62616540D24A80              , vpsrld(zmm25, k0, zmm19, xmmword_ptr(rdx, -2048))),
        INSTRUCTION_TEST(62616540D28A00080000        , vpsrld(zmm25, k0, zmm19, xmmword_ptr(rdx, 2048))),
        INSTRUCTION_TEST(62616540D24A7F              , vpsrld(zmm25, k0, zmm19, xmmword_ptr(rdx, 2032))),
        INSTRUCTION_TEST(62F1654872D57B              , vpsrld(zmm3, k0, zmm5, Imm(123))),
        INSTRUCTION_TEST(62F1654872D5AB              , vpsrld(zmm3, k0, zmm5, Imm(171))),
        INSTRUCTION_TEST(62B165487294F0230100007B    , vpsrld(zmm3, k0, zmmword_ptr(rax, r14, 8, 291), Imm(123))),
        INSTRUCTION_TEST(62F1654872117B              , vpsrld(zmm3, k0, zmmword_ptr(rcx), Imm(123))),
        INSTRUCTION_TEST(62F165487252807B            , vpsrld(zmm3, k0, zmmword_ptr(rdx, -8192), Imm(123))),
        INSTRUCTION_TEST(62F165487292C0DFFFFF7B      , vpsrld(zmm3, k0, zmmword_ptr(rdx, -8256), Imm(123))),
        INSTRUCTION_TEST(62F1654872527F7B            , vpsrld(zmm3, k0, zmmword_ptr(rdx, 8128), Imm(123))),
        INSTRUCTION_TEST(62F165487292002000007B      , vpsrld(zmm3, k0, zmmword_ptr(rdx, 8192), Imm(123))),
        INSTRUCTION_TEST(62616540D28AF0F7FFFF        , vpsrld(zmm25, k0, zmm19, xmmword_ptr(rdx, -2064))),
        INSTRUCTION_TEST(62B125487294F0341200007B    , vpsrld(zmm11, k0, zmmword_ptr(rax, r14, 8, 4660), Imm(123))),
        INSTRUCTION_TEST(62D1254872D17B              , vpsrld(zmm11, k0, zmm9, Imm(123))),
        INSTRUCTION_TEST(62F1754872541A0201          , vpsrld(zmm1, k0, zmmword_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(C5F172D201                  , vpsrld(xmm1, xmm2, Imm(1))),
        INSTRUCTION_TEST(C5E9D2CB                    , vpsrld(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(C5E9D28C2B80000000          , vpsrld(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F1750872541A0801          , vpsrld(xmm1, k0, xmmword_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(C5F572D201                  , vpsrld(ymm1, ymm2, Imm(1))),
        INSTRUCTION_TEST(C5EDD2CB                    , vpsrld(ymm1, ymm2, xmm3)),
        INSTRUCTION_TEST(C5EDD28C2B80000000          , vpsrld(ymm1, ymm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62917540D2C9                , vpsrld(zmm1, k0, zmm17, xmm25)),
        INSTRUCTION_TEST(62F1752872541A0401          , vpsrld(ymm1, k0, ymmword_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F17540D209                , vpsrld(zmm1, k0, zmm17, xmmword_ptr(rcx))),
        INSTRUCTION_TEST(62F17540D24A80              , vpsrld(zmm1, k0, zmm17, xmmword_ptr(rdx, -2048))),
        INSTRUCTION_TEST(62F17540D28AF0F7FFFF        , vpsrld(zmm1, k0, zmm17, xmmword_ptr(rdx, -2064))),
        INSTRUCTION_TEST(62F17540D24A7F              , vpsrld(zmm1, k0, zmm17, xmmword_ptr(rdx, 2032))),
        INSTRUCTION_TEST(62F17540D28A00080000        , vpsrld(zmm1, k0, zmm17, xmmword_ptr(rdx, 2048))),
        INSTRUCTION_TEST(62F1754872D201              , vpsrld(zmm1, k0, zmm2, Imm(1))),
        INSTRUCTION_TEST(62F16D48D2CB                , vpsrld(zmm1, k0, zmm2, xmm3)),
        INSTRUCTION_TEST(62F16D48D24C2B08            , vpsrld(zmm1, k0, zmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62B17540D28CF023010000      , vpsrld(zmm1, k0, zmm17, xmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62D1254872D1AB              , vpsrld(zmm11, k0, zmm9, Imm(171))),
        INSTRUCTION_TEST(62F17528735C1A0401          , vpsrldq(ymm1, ymmword_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F17548735C1A0201          , vpsrldq(zmm1, zmmword_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F1754873DA01              , vpsrldq(zmm1, zmm2, Imm(1))),
        INSTRUCTION_TEST(C5F573DA01                  , vpsrldq(ymm1, ymm2, Imm(1))),
        INSTRUCTION_TEST(C5F173DA01                  , vpsrldq(xmm1, xmm2, Imm(1))),
        INSTRUCTION_TEST(62F17508735C1A0801          , vpsrldq(xmm1, xmmword_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62E1AD48D35A80              , vpsrlq(zmm19, k0, zmm10, xmmword_ptr(rdx, -2048))),
        INSTRUCTION_TEST(62E1AD48D39AF0F7FFFF        , vpsrlq(zmm19, k0, zmm10, xmmword_ptr(rdx, -2064))),
        INSTRUCTION_TEST(62E1AD48D35A7F              , vpsrlq(zmm19, k0, zmm10, xmmword_ptr(rdx, 2032))),
        INSTRUCTION_TEST(62E1AD48D39A00080000        , vpsrlq(zmm19, k0, zmm10, xmmword_ptr(rdx, 2048))),
        INSTRUCTION_TEST(62D1CD4873D67B              , vpsrlq(zmm6, k0, zmm14, Imm(123))),
        INSTRUCTION_TEST(62D1CD4873D6AB              , vpsrlq(zmm6, k0, zmm14, Imm(171))),
        INSTRUCTION_TEST(62B1CD487394F0341200007B    , vpsrlq(zmm6, k0, zmmword_ptr(rax, r14, 8, 4660), Imm(123))),
        INSTRUCTION_TEST(62F1CD4873117B              , vpsrlq(zmm6, k0, zmmword_ptr(rcx), Imm(123))),
        INSTRUCTION_TEST(62F1CD487352807B            , vpsrlq(zmm6, k0, zmmword_ptr(rdx, -8192), Imm(123))),
        INSTRUCTION_TEST(62F1CD487392C0DFFFFF7B      , vpsrlq(zmm6, k0, zmmword_ptr(rdx, -8256), Imm(123))),
        INSTRUCTION_TEST(62F1CD487392002000007B      , vpsrlq(zmm6, k0, zmmword_ptr(rdx, 8192), Imm(123))),
        INSTRUCTION_TEST(6231A548D38CF023010000      , vpsrlq(zmm9, k0, zmm11, xmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(6271A548D3CB                , vpsrlq(zmm9, k0, zmm11, xmm3)),
        INSTRUCTION_TEST(6271A548D309                , vpsrlq(zmm9, k0, zmm11, xmmword_ptr(rcx))),
        INSTRUCTION_TEST(6271A548D34A80              , vpsrlq(zmm9, k0, zmm11, xmmword_ptr(rdx, -2048))),
        INSTRUCTION_TEST(6271A548D38AF0F7FFFF        , vpsrlq(zmm9, k0, zmm11, xmmword_ptr(rdx, -2064))),
        INSTRUCTION_TEST(6271A548D34A7F              , vpsrlq(zmm9, k0, zmm11, xmmword_ptr(rdx, 2032))),
        INSTRUCTION_TEST(62E1AD48D319                , vpsrlq(zmm19, k0, zmm10, xmmword_ptr(rcx))),
        INSTRUCTION_TEST(6271A548D38A00080000        , vpsrlq(zmm9, k0, zmm11, xmmword_ptr(rdx, 2048))),
        INSTRUCTION_TEST(62A1AD48D39CF034120000      , vpsrlq(zmm19, k0, zmm10, xmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62F1CD4873527F7B            , vpsrlq(zmm6, k0, zmmword_ptr(rdx, 8128), Imm(123))),
        INSTRUCTION_TEST(62F1ED407392002000007B      , vpsrlq(zmm18, k0, zmmword_ptr(rdx, 8192), Imm(123))),
        INSTRUCTION_TEST(62A1AD48D3DD                , vpsrlq(zmm19, k0, zmm10, xmm21)),
        INSTRUCTION_TEST(C5F173D201                  , vpsrlq(xmm1, xmm2, Imm(1))),
        INSTRUCTION_TEST(C5E9D3CB                    , vpsrlq(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(C5E9D38C2B80000000          , vpsrlq(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F1F50873541A0801          , vpsrlq(xmm1, k0, xmmword_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(C5F573D201                  , vpsrlq(ymm1, ymm2, Imm(1))),
        INSTRUCTION_TEST(C5EDD38C2B80000000          , vpsrlq(ymm1, ymm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F1F52873541A0401          , vpsrlq(ymm1, k0, ymmword_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F1F54873D201              , vpsrlq(zmm1, k0, zmm2, Imm(1))),
        INSTRUCTION_TEST(C5EDD3CB                    , vpsrlq(ymm1, ymm2, xmm3)),
        INSTRUCTION_TEST(62F1ED48D34C2B08            , vpsrlq(zmm1, k0, zmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F1ED48D3CB                , vpsrlq(zmm1, k0, zmm2, xmm3)),
        INSTRUCTION_TEST(62F1ED407392C0DFFFFF7B      , vpsrlq(zmm18, k0, zmmword_ptr(rdx, -8256), Imm(123))),
        INSTRUCTION_TEST(62F1ED407352807B            , vpsrlq(zmm18, k0, zmmword_ptr(rdx, -8192), Imm(123))),
        INSTRUCTION_TEST(62F1ED4073117B              , vpsrlq(zmm18, k0, zmmword_ptr(rcx), Imm(123))),
        INSTRUCTION_TEST(62F1ED4073527F7B            , vpsrlq(zmm18, k0, zmmword_ptr(rdx, 8128), Imm(123))),
        INSTRUCTION_TEST(6291ED4073D1AB              , vpsrlq(zmm18, k0, zmm25, Imm(171))),
        INSTRUCTION_TEST(6291ED4073D17B              , vpsrlq(zmm18, k0, zmm25, Imm(123))),
        INSTRUCTION_TEST(62F1F54873541A0201          , vpsrlq(zmm1, k0, zmmword_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62B1ED407394F0230100007B    , vpsrlq(zmm18, k0, zmmword_ptr(rax, r14, 8, 291), Imm(123))),
        INSTRUCTION_TEST(62F25D4845527F              , vpsrlvd(zmm2, k0, zmm4, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62720D4845427F              , vpsrlvd(zmm8, k0, zmm14, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62720D484582C0DFFFFF        , vpsrlvd(zmm8, k0, zmm14, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62720D48454280              , vpsrlvd(zmm8, k0, zmm14, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62720D484501                , vpsrlvd(zmm8, k0, zmm14, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62320D484584F023010000      , vpsrlvd(zmm8, k0, zmm14, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62120D4845C4                , vpsrlvd(zmm8, k0, zmm14, zmm28)),
        INSTRUCTION_TEST(62F25D48459200200000        , vpsrlvd(zmm2, k0, zmm4, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62F25D484592C0DFFFFF        , vpsrlvd(zmm2, k0, zmm4, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62720D48458200200000        , vpsrlvd(zmm8, k0, zmm14, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62F25D484511                , vpsrlvd(zmm2, k0, zmm4, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62B25D484594F034120000      , vpsrlvd(zmm2, k0, zmm4, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62D25D4845D3                , vpsrlvd(zmm2, k0, zmm4, zmm11)),
        INSTRUCTION_TEST(62F26D48454C2B02            , vpsrlvd(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F26D4845CB                , vpsrlvd(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(C4E26D458C2B80000000        , vpsrlvd(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E26D45CB                  , vpsrlvd(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(C4E269458C2B80000000        , vpsrlvd(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E26945CB                  , vpsrlvd(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(62F25D48455280              , vpsrlvd(zmm2, k0, zmm4, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62E2C54845A200200000        , vpsrlvq(zmm20, k0, zmm7, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62529D4045C7                , vpsrlvq(zmm8, k0, zmm28, zmm15)),
        INSTRUCTION_TEST(62729D404501                , vpsrlvq(zmm8, k0, zmm28, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62729D40458200200000        , vpsrlvq(zmm8, k0, zmm28, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62729D404582C0DFFFFF        , vpsrlvq(zmm8, k0, zmm28, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62729D4045427F              , vpsrlvq(zmm8, k0, zmm28, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62E2C54845627F              , vpsrlvq(zmm20, k0, zmm7, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62729D40454280              , vpsrlvq(zmm8, k0, zmm28, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62E2C54845A2C0DFFFFF        , vpsrlvq(zmm20, k0, zmm7, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62329D404584F034120000      , vpsrlvq(zmm8, k0, zmm28, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62E2C5484521                , vpsrlvq(zmm20, k0, zmm7, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62A2C54845A4F023010000      , vpsrlvq(zmm20, k0, zmm7, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(6282C54845E2                , vpsrlvq(zmm20, k0, zmm7, zmm26)),
        INSTRUCTION_TEST(62F2ED48454C2B02            , vpsrlvq(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F2ED4845CB                , vpsrlvq(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(C4E2ED458C2B80000000        , vpsrlvq(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E2ED45CB                  , vpsrlvq(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(C4E2E9458C2B80000000        , vpsrlvq(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C4E2E945CB                  , vpsrlvq(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(62E2C548456280              , vpsrlvq(zmm20, k0, zmm7, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62F2ED48104C2B02            , vpsrlvw(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F2ED28104C2B04            , vpsrlvw(ymm1, k0, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F2ED4810CB                , vpsrlvw(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(62F2ED08104C2B08            , vpsrlvw(xmm1, k0, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F2ED0810CB                , vpsrlvw(xmm1, k0, xmm2, xmm3)),
        INSTRUCTION_TEST(62F2ED2810CB                , vpsrlvw(ymm1, k0, ymm2, ymm3)),
        INSTRUCTION_TEST(62F1754871541A0201          , vpsrlw(zmm1, k0, zmmword_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F16D48D14C2B08            , vpsrlw(zmm1, k0, zmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F16D48D1CB                , vpsrlw(zmm1, k0, zmm2, xmm3)),
        INSTRUCTION_TEST(62F1754871D201              , vpsrlw(zmm1, k0, zmm2, Imm(1))),
        INSTRUCTION_TEST(62F1752871541A0401          , vpsrlw(ymm1, k0, ymmword_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(C5EDD18C2B80000000          , vpsrlw(ymm1, ymm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C5EDD1CB                    , vpsrlw(ymm1, ymm2, xmm3)),
        INSTRUCTION_TEST(62F1750871541A0801          , vpsrlw(xmm1, k0, xmmword_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(C5E9D18C2B80000000          , vpsrlw(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C5E9D1CB                    , vpsrlw(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(C5F171D201                  , vpsrlw(xmm1, xmm2, Imm(1))),
        INSTRUCTION_TEST(C5F571D201                  , vpsrlw(ymm1, ymm2, Imm(1))),
        INSTRUCTION_TEST(62F16D48F84C2B02            , vpsubb(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F16D48F8CB                , vpsubb(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(C5EDF88C2B80000000          , vpsubb(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C5E9F88C2B80000000          , vpsubb(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C5E9F8CB                    , vpsubb(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(C5EDF8CB                    , vpsubb(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(62610D48FA9200200000        , vpsubd(zmm26, k0, zmm14, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62F11D40FABA00200000        , vpsubd(zmm7, k0, zmm28, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62F11D40FA7A7F              , vpsubd(zmm7, k0, zmm28, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62F11D40FABAC0DFFFFF        , vpsubd(zmm7, k0, zmm28, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62F11D40FA7A80              , vpsubd(zmm7, k0, zmm28, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62F11D40FA39                , vpsubd(zmm7, k0, zmm28, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62B11D40FABCF023010000      , vpsubd(zmm7, k0, zmm28, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62F11D40FAFF                , vpsubd(zmm7, k0, zmm28, zmm7)),
        INSTRUCTION_TEST(62610D48FA527F              , vpsubd(zmm26, k0, zmm14, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62610D48FA92C0DFFFFF        , vpsubd(zmm26, k0, zmm14, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62610D48FA5280              , vpsubd(zmm26, k0, zmm14, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62210D48FA94F034120000      , vpsubd(zmm26, k0, zmm14, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62210D48FAD7                , vpsubd(zmm26, k0, zmm14, zmm23)),
        INSTRUCTION_TEST(62F16D48FA4C2B02            , vpsubd(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F16D48FACB                , vpsubd(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(C5EDFA8C2B80000000          , vpsubd(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C5EDFACB                    , vpsubd(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(C5E9FA8C2B80000000          , vpsubd(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C5E9FACB                    , vpsubd(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(62610D48FA11                , vpsubd(zmm26, k0, zmm14, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62E1BD48FB8200200000        , vpsubq(zmm16, k0, zmm8, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62219D40FBE9                , vpsubq(zmm29, k0, zmm28, zmm17)),
        INSTRUCTION_TEST(62219D40FBACF023010000      , vpsubq(zmm29, k0, zmm28, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62619D40FB6A7F              , vpsubq(zmm29, k0, zmm28, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62619D40FBAAC0DFFFFF        , vpsubq(zmm29, k0, zmm28, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62619D40FBAA00200000        , vpsubq(zmm29, k0, zmm28, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62E1BD48FB427F              , vpsubq(zmm16, k0, zmm8, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62619D40FB6A80              , vpsubq(zmm29, k0, zmm28, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62E1BD48FB82C0DFFFFF        , vpsubq(zmm16, k0, zmm8, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62619D40FB29                , vpsubq(zmm29, k0, zmm28, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62E1BD48FB01                , vpsubq(zmm16, k0, zmm8, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(C5E9FBCB                    , vpsubq(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(C5EDFBCB                    , vpsubq(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(C5EDFB8C2B80000000          , vpsubq(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62E1BD48FB4280              , vpsubq(zmm16, k0, zmm8, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(C5E9FB8C2B80000000          , vpsubq(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F1ED48FB4C2B02            , vpsubq(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62A1BD48FBC7                , vpsubq(zmm16, k0, zmm8, zmm23)),
        INSTRUCTION_TEST(62A1BD48FB84F034120000      , vpsubq(zmm16, k0, zmm8, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62F1ED48FBCB                , vpsubq(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(C5E9E8CB                    , vpsubsb(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(C5E9E88C2B80000000          , vpsubsb(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C5EDE8CB                    , vpsubsb(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(C5EDE88C2B80000000          , vpsubsb(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F16D48E8CB                , vpsubsb(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(62F16D48E84C2B02            , vpsubsb(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F16D48E9CB                , vpsubsw(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(C5EDE98C2B80000000          , vpsubsw(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F16D48E94C2B02            , vpsubsw(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C5E9E98C2B80000000          , vpsubsw(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C5E9E9CB                    , vpsubsw(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(C5EDE9CB                    , vpsubsw(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(C5E9D8CB                    , vpsubusb(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(C5E9D88C2B80000000          , vpsubusb(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C5EDD8CB                    , vpsubusb(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(C5EDD88C2B80000000          , vpsubusb(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F16D48D8CB                , vpsubusb(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(62F16D48D84C2B02            , vpsubusb(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F16D48D94C2B02            , vpsubusw(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F16D48D9CB                , vpsubusw(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(C5EDD98C2B80000000          , vpsubusw(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C5E9D98C2B80000000          , vpsubusw(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C5E9D9CB                    , vpsubusw(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(C5EDD9CB                    , vpsubusw(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(C5E9F9CB                    , vpsubw(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(C5E9F98C2B80000000          , vpsubw(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C5EDF9CB                    , vpsubw(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(C5EDF98C2B80000000          , vpsubw(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F16D48F9CB                , vpsubw(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(62F16D48F94C2B02            , vpsubw(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62330D4825E4AB              , vpternlogd(zmm12, k0, zmm14, zmm20, Imm(171))),
        INSTRUCTION_TEST(627315482592002000007B      , vpternlogd(zmm10, k0, zmm13, zmmword_ptr(rdx, 8192), Imm(123))),
        INSTRUCTION_TEST(62330D4825E47B              , vpternlogd(zmm12, k0, zmm14, zmm20, Imm(123))),
        INSTRUCTION_TEST(62330D4825A4F0230100007B    , vpternlogd(zmm12, k0, zmm14, zmmword_ptr(rax, r14, 8, 291), Imm(123))),
        INSTRUCTION_TEST(6273154825527F7B            , vpternlogd(zmm10, k0, zmm13, zmmword_ptr(rdx, 8128), Imm(123))),
        INSTRUCTION_TEST(62730D482562807B            , vpternlogd(zmm12, k0, zmm14, zmmword_ptr(rdx, -8192), Imm(123))),
        INSTRUCTION_TEST(62730D4825A2C0DFFFFF7B      , vpternlogd(zmm12, k0, zmm14, zmmword_ptr(rdx, -8256), Imm(123))),
        INSTRUCTION_TEST(62730D4825627F7B            , vpternlogd(zmm12, k0, zmm14, zmmword_ptr(rdx, 8128), Imm(123))),
        INSTRUCTION_TEST(62730D4825A2002000007B      , vpternlogd(zmm12, k0, zmm14, zmmword_ptr(rdx, 8192), Imm(123))),
        INSTRUCTION_TEST(62730D4825217B              , vpternlogd(zmm12, k0, zmm14, zmmword_ptr(rcx), Imm(123))),
        INSTRUCTION_TEST(627315482592C0DFFFFF7B      , vpternlogd(zmm10, k0, zmm13, zmmword_ptr(rdx, -8256), Imm(123))),
        INSTRUCTION_TEST(62F36D08254C2B0801          , vpternlogd(xmm1, k0, xmm2, xmmword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(6273154825117B              , vpternlogd(zmm10, k0, zmm13, zmmword_ptr(rcx), Imm(123))),
        INSTRUCTION_TEST(627315482552807B            , vpternlogd(zmm10, k0, zmm13, zmmword_ptr(rdx, -8192), Imm(123))),
        INSTRUCTION_TEST(62F36D2825CB01              , vpternlogd(ymm1, k0, ymm2, ymm3, Imm(1))),
        INSTRUCTION_TEST(62F36D28254C2B0401          , vpternlogd(ymm1, k0, ymm2, ymmword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F36D4825CB01              , vpternlogd(zmm1, k0, zmm2, zmm3, Imm(1))),
        INSTRUCTION_TEST(62F36D0825CB01              , vpternlogd(xmm1, k0, xmm2, xmm3, Imm(1))),
        INSTRUCTION_TEST(6253154825D27B              , vpternlogd(zmm10, k0, zmm13, zmm10, Imm(123))),
        INSTRUCTION_TEST(6253154825D2AB              , vpternlogd(zmm10, k0, zmm13, zmm10, Imm(171))),
        INSTRUCTION_TEST(623315482594F0341200007B    , vpternlogd(zmm10, k0, zmm13, zmmword_ptr(rax, r14, 8, 4660), Imm(123))),
        INSTRUCTION_TEST(62F36D48254C2B0201          , vpternlogd(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(6273ED48257A7F7B            , vpternlogq(zmm15, k0, zmm2, zmmword_ptr(rdx, 8128), Imm(123))),
        INSTRUCTION_TEST(62E39D4825427F7B            , vpternlogq(zmm16, k0, zmm12, zmmword_ptr(rdx, 8128), Imm(123))),
        INSTRUCTION_TEST(62E39D482582C0DFFFFF7B      , vpternlogq(zmm16, k0, zmm12, zmmword_ptr(rdx, -8256), Imm(123))),
        INSTRUCTION_TEST(62E39D482542807B            , vpternlogq(zmm16, k0, zmm12, zmmword_ptr(rdx, -8192), Imm(123))),
        INSTRUCTION_TEST(62E39D4825017B              , vpternlogq(zmm16, k0, zmm12, zmmword_ptr(rcx), Imm(123))),
        INSTRUCTION_TEST(62A39D482584F0341200007B    , vpternlogq(zmm16, k0, zmm12, zmmword_ptr(rax, r14, 8, 4660), Imm(123))),
        INSTRUCTION_TEST(62C39D4825C5AB              , vpternlogq(zmm16, k0, zmm12, zmm13, Imm(171))),
        INSTRUCTION_TEST(62C39D4825C57B              , vpternlogq(zmm16, k0, zmm12, zmm13, Imm(123))),
        INSTRUCTION_TEST(6273ED4825BA002000007B      , vpternlogq(zmm15, k0, zmm2, zmmword_ptr(rdx, 8192), Imm(123))),
        INSTRUCTION_TEST(6273ED4825BAC0DFFFFF7B      , vpternlogq(zmm15, k0, zmm2, zmmword_ptr(rdx, -8256), Imm(123))),
        INSTRUCTION_TEST(6233ED4825FD7B              , vpternlogq(zmm15, k0, zmm2, zmm21, Imm(123))),
        INSTRUCTION_TEST(6273ED4825397B              , vpternlogq(zmm15, k0, zmm2, zmmword_ptr(rcx), Imm(123))),
        INSTRUCTION_TEST(6233ED4825BCF0230100007B    , vpternlogq(zmm15, k0, zmm2, zmmword_ptr(rax, r14, 8, 291), Imm(123))),
        INSTRUCTION_TEST(6233ED4825FDAB              , vpternlogq(zmm15, k0, zmm2, zmm21, Imm(171))),
        INSTRUCTION_TEST(62E39D482582002000007B      , vpternlogq(zmm16, k0, zmm12, zmmword_ptr(rdx, 8192), Imm(123))),
        INSTRUCTION_TEST(62F3ED48254C2B0201          , vpternlogq(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F3ED4825CB01              , vpternlogq(zmm1, k0, zmm2, zmm3, Imm(1))),
        INSTRUCTION_TEST(62F3ED28254C2B0401          , vpternlogq(ymm1, k0, ymm2, ymmword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F3ED2825CB01              , vpternlogq(ymm1, k0, ymm2, ymm3, Imm(1))),
        INSTRUCTION_TEST(62F3ED08254C2B0801          , vpternlogq(xmm1, k0, xmm2, xmmword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F3ED0825CB01              , vpternlogq(xmm1, k0, xmm2, xmm3, Imm(1))),
        INSTRUCTION_TEST(6273ED48257A807B            , vpternlogq(zmm15, k0, zmm2, zmmword_ptr(rdx, -8192), Imm(123))),
        INSTRUCTION_TEST(C4E27D17CA                  , vptest(ymm1, ymm2)),
        INSTRUCTION_TEST(C4E27D178C1A80000000        , vptest(ymm1, ymmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(C4E27917CA                  , vptest(xmm1, xmm2)),
        INSTRUCTION_TEST(C4E279178C1A80000000        , vptest(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F26D0826CB                , vptestmb(k1, k0, xmm2, xmm3)),
        INSTRUCTION_TEST(62F26D08264C2B08            , vptestmb(k1, k0, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F26D2826CB                , vptestmb(k1, k0, ymm2, ymm3)),
        INSTRUCTION_TEST(62F26D28264C2B04            , vptestmb(k1, k0, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F26D4826CB                , vptestmb(k1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(62F26D48264C2B02            , vptestmb(k1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F21D4827527F              , vptestmd(k2, k0, zmm12, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62F23D402792C0DFFFFF        , vptestmd(k2, k0, zmm24, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62F23D40275280              , vptestmd(k2, k0, zmm24, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62F23D402711                , vptestmd(k2, k0, zmm24, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62B23D402794F034120000      , vptestmd(k2, k0, zmm24, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62923D4027D0                , vptestmd(k2, k0, zmm24, zmm24)),
        INSTRUCTION_TEST(62F21D48279200200000        , vptestmd(k2, k0, zmm12, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62F23D40279200200000        , vptestmd(k2, k0, zmm24, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62F23D4027527F              , vptestmd(k2, k0, zmm24, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62F21D482792C0DFFFFF        , vptestmd(k2, k0, zmm12, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62F21D482711                , vptestmd(k2, k0, zmm12, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62B21D482794F023010000      , vptestmd(k2, k0, zmm12, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62B21D4827D0                , vptestmd(k2, k0, zmm12, zmm16)),
        INSTRUCTION_TEST(62F26D48274C2B02            , vptestmd(k1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F26D4827CB                , vptestmd(k1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(62F26D28274C2B04            , vptestmd(k1, k0, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F26D2827CB                , vptestmd(k1, k0, ymm2, ymm3)),
        INSTRUCTION_TEST(62F26D08274C2B08            , vptestmd(k1, k0, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F26D0827CB                , vptestmd(k1, k0, xmm2, xmm3)),
        INSTRUCTION_TEST(62F21D48275280              , vptestmd(k2, k0, zmm12, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62F2854827A200200000        , vptestmq(k4, k0, zmm15, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62B2CD4827ED                , vptestmq(k5, k0, zmm6, zmm21)),
        INSTRUCTION_TEST(62B2CD4827ACF023010000      , vptestmq(k5, k0, zmm6, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62F2CD4827AA00200000        , vptestmq(k5, k0, zmm6, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62F2CD48276A80              , vptestmq(k5, k0, zmm6, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62F2CD4827AAC0DFFFFF        , vptestmq(k5, k0, zmm6, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62F2CD48276A7F              , vptestmq(k5, k0, zmm6, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62F2CD482729                , vptestmq(k5, k0, zmm6, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62F2854827A2C0DFFFFF        , vptestmq(k4, k0, zmm15, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62F2854827627F              , vptestmq(k4, k0, zmm15, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62F285482721                , vptestmq(k4, k0, zmm15, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62F2ED0827CB                , vptestmq(k1, k0, xmm2, xmm3)),
        INSTRUCTION_TEST(62F2ED08274C2B08            , vptestmq(k1, k0, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F2ED2827CB                , vptestmq(k1, k0, ymm2, ymm3)),
        INSTRUCTION_TEST(62F2ED28274C2B04            , vptestmq(k1, k0, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F28548276280              , vptestmq(k4, k0, zmm15, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62F2ED48274C2B02            , vptestmq(k1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62B2854827E4                , vptestmq(k4, k0, zmm15, zmm20)),
        INSTRUCTION_TEST(62B2854827A4F034120000      , vptestmq(k4, k0, zmm15, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62F2ED4827CB                , vptestmq(k1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(62F2ED0826CB                , vptestmw(k1, k0, xmm2, xmm3)),
        INSTRUCTION_TEST(62F2ED08264C2B08            , vptestmw(k1, k0, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F2ED2826CB                , vptestmw(k1, k0, ymm2, ymm3)),
        INSTRUCTION_TEST(62F2ED28264C2B04            , vptestmw(k1, k0, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F2ED4826CB                , vptestmw(k1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(62F2ED48264C2B02            , vptestmw(k1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F26E48264C2B02            , vptestnmb(k1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F26E28264C2B04            , vptestnmb(k1, k0, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F26E4826CB                , vptestnmb(k1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(62F26E08264C2B08            , vptestnmb(k1, k0, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F26E0826CB                , vptestnmb(k1, k0, xmm2, xmm3)),
        INSTRUCTION_TEST(62F26E2826CB                , vptestnmb(k1, k0, ymm2, ymm3)),
        INSTRUCTION_TEST(62F26E4827CB                , vptestnmd(k1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(62F26E28274C2B04            , vptestnmd(k1, k0, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F26E48274C2B02            , vptestnmd(k1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F26E08274C2B08            , vptestnmd(k1, k0, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F26E0827CB                , vptestnmd(k1, k0, xmm2, xmm3)),
        INSTRUCTION_TEST(62F26E2827CB                , vptestnmd(k1, k0, ymm2, ymm3)),
        INSTRUCTION_TEST(62F2EE0827CB                , vptestnmq(k1, k0, xmm2, xmm3)),
        INSTRUCTION_TEST(62F2EE08274C2B08            , vptestnmq(k1, k0, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F2EE2827CB                , vptestnmq(k1, k0, ymm2, ymm3)),
        INSTRUCTION_TEST(62F2EE28274C2B04            , vptestnmq(k1, k0, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F2EE4827CB                , vptestnmq(k1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(62F2EE48274C2B02            , vptestnmq(k1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F2EE48264C2B02            , vptestnmw(k1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F2EE4826CB                , vptestnmw(k1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(62F2EE28264C2B04            , vptestnmw(k1, k0, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F2EE08264C2B08            , vptestnmw(k1, k0, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F2EE2826CB                , vptestnmw(k1, k0, ymm2, ymm3)),
        INSTRUCTION_TEST(62F2EE0826CB                , vptestnmw(k1, k0, xmm2, xmm3)),
        INSTRUCTION_TEST(C5E968CB                    , vpunpckhbw(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(C5E9688C2B80000000          , vpunpckhbw(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C5ED68CB                    , vpunpckhbw(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(C5ED688C2B80000000          , vpunpckhbw(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F16D4868CB                , vpunpckhbw(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(62F16D48684C2B02            , vpunpckhbw(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F15D486AB200200000        , vpunpckhdq(zmm6, k0, zmm4, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62714D486A5A7F              , vpunpckhdq(zmm11, k0, zmm6, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62714D486A9A00200000        , vpunpckhdq(zmm11, k0, zmm6, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62D15D486AF5                , vpunpckhdq(zmm6, k0, zmm4, zmm13)),
        INSTRUCTION_TEST(62F15D486AB2C0DFFFFF        , vpunpckhdq(zmm6, k0, zmm4, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62F15D486A31                , vpunpckhdq(zmm6, k0, zmm4, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62F15D486A7280              , vpunpckhdq(zmm6, k0, zmm4, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62714D486A9AC0DFFFFF        , vpunpckhdq(zmm11, k0, zmm6, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62B15D486AB4F023010000      , vpunpckhdq(zmm6, k0, zmm4, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62F15D486A727F              , vpunpckhdq(zmm6, k0, zmm4, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62714D486A5A80              , vpunpckhdq(zmm11, k0, zmm6, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62714D486ADF                , vpunpckhdq(zmm11, k0, zmm6, zmm7)),
        INSTRUCTION_TEST(62314D486A9CF034120000      , vpunpckhdq(zmm11, k0, zmm6, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62F16D486A4C2B02            , vpunpckhdq(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F16D486ACB                , vpunpckhdq(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(C5ED6A8C2B80000000          , vpunpckhdq(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C5ED6ACB                    , vpunpckhdq(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(C5E96A8C2B80000000          , vpunpckhdq(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C5E96ACB                    , vpunpckhdq(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(62714D486A19                , vpunpckhdq(zmm11, k0, zmm6, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(626185486D9A00200000        , vpunpckhqdq(zmm27, k0, zmm15, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(6211CD486DC2                , vpunpckhqdq(zmm8, k0, zmm6, zmm26)),
        INSTRUCTION_TEST(6231CD486D84F034120000      , vpunpckhqdq(zmm8, k0, zmm6, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(626185486D5A7F              , vpunpckhqdq(zmm27, k0, zmm15, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(6271CD486D4280              , vpunpckhqdq(zmm8, k0, zmm6, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(6271CD486D82C0DFFFFF        , vpunpckhqdq(zmm8, k0, zmm6, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(6271CD486D8200200000        , vpunpckhqdq(zmm8, k0, zmm6, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(6271CD486D01                , vpunpckhqdq(zmm8, k0, zmm6, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(626185486D9AC0DFFFFF        , vpunpckhqdq(zmm27, k0, zmm15, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(6271CD486D427F              , vpunpckhqdq(zmm8, k0, zmm6, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(626185486D19                , vpunpckhqdq(zmm27, k0, zmm15, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(622185486D9CF023010000      , vpunpckhqdq(zmm27, k0, zmm15, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(622185486DD8                , vpunpckhqdq(zmm27, k0, zmm15, zmm16)),
        INSTRUCTION_TEST(62F1ED486D4C2B02            , vpunpckhqdq(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F1ED486DCB                , vpunpckhqdq(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(C5ED6D8C2B80000000          , vpunpckhqdq(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C5ED6DCB                    , vpunpckhqdq(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(C5E96D8C2B80000000          , vpunpckhqdq(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(626185486D5A80              , vpunpckhqdq(zmm27, k0, zmm15, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(C5E96DCB                    , vpunpckhqdq(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(62F16D4869CB                , vpunpckhwd(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(C5ED698C2B80000000          , vpunpckhwd(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F16D48694C2B02            , vpunpckhwd(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C5E9698C2B80000000          , vpunpckhwd(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C5E969CB                    , vpunpckhwd(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(C5ED69CB                    , vpunpckhwd(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(62F16D4860CB                , vpunpcklbw(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(C5ED608C2B80000000          , vpunpcklbw(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F16D48604C2B02            , vpunpcklbw(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C5E9608C2B80000000          , vpunpcklbw(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C5E960CB                    , vpunpcklbw(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(C5ED60CB                    , vpunpcklbw(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(62E16D4062427F              , vpunpckldq(zmm16, k0, zmm18, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62616548628200200000        , vpunpckldq(zmm24, k0, zmm3, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(626165486282C0DFFFFF        , vpunpckldq(zmm24, k0, zmm3, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62616548624280              , vpunpckldq(zmm24, k0, zmm3, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(626165486201                , vpunpckldq(zmm24, k0, zmm3, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(622165486284F023010000      , vpunpckldq(zmm24, k0, zmm3, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(6221654862C1                , vpunpckldq(zmm24, k0, zmm3, zmm17)),
        INSTRUCTION_TEST(62E16D40628200200000        , vpunpckldq(zmm16, k0, zmm18, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62E16D406282C0DFFFFF        , vpunpckldq(zmm16, k0, zmm18, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(6261654862427F              , vpunpckldq(zmm24, k0, zmm3, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62E16D406201                , vpunpckldq(zmm16, k0, zmm18, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62E16D40624280              , vpunpckldq(zmm16, k0, zmm18, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(C5E9628C2B80000000          , vpunpckldq(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C5ED62CB                    , vpunpckldq(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(C5ED628C2B80000000          , vpunpckldq(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C5E962CB                    , vpunpckldq(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(62F16D48624C2B02            , vpunpckldq(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62C16D4062C5                , vpunpckldq(zmm16, k0, zmm18, zmm13)),
        INSTRUCTION_TEST(62A16D406284F034120000      , vpunpckldq(zmm16, k0, zmm18, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62F16D4862CB                , vpunpckldq(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(62F1DD486C9A00200000        , vpunpcklqdq(zmm3, k0, zmm4, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62E1FD406C4A7F              , vpunpcklqdq(zmm17, k0, zmm16, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62F1DD486C5A7F              , vpunpcklqdq(zmm3, k0, zmm4, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62F1DD486C9AC0DFFFFF        , vpunpcklqdq(zmm3, k0, zmm4, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62F1DD486C5A80              , vpunpcklqdq(zmm3, k0, zmm4, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62F1DD486C19                , vpunpcklqdq(zmm3, k0, zmm4, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62B1DD486CD9                , vpunpcklqdq(zmm3, k0, zmm4, zmm17)),
        INSTRUCTION_TEST(62E1FD406C8A00200000        , vpunpcklqdq(zmm17, k0, zmm16, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62E1FD406C8AC0DFFFFF        , vpunpcklqdq(zmm17, k0, zmm16, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62B1DD486C9CF023010000      , vpunpcklqdq(zmm3, k0, zmm4, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62E1FD406C09                , vpunpcklqdq(zmm17, k0, zmm16, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62A1FD406C8CF034120000      , vpunpcklqdq(zmm17, k0, zmm16, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62C1FD406CC9                , vpunpcklqdq(zmm17, k0, zmm16, zmm9)),
        INSTRUCTION_TEST(62F1ED486C4C2B02            , vpunpcklqdq(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F1ED486CCB                , vpunpcklqdq(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(C5ED6C8C2B80000000          , vpunpcklqdq(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C5ED6CCB                    , vpunpcklqdq(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(C5E96C8C2B80000000          , vpunpcklqdq(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C5E96CCB                    , vpunpcklqdq(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(62E1FD406C4A80              , vpunpcklqdq(zmm17, k0, zmm16, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62F16D48614C2B02            , vpunpcklwd(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C5ED618C2B80000000          , vpunpcklwd(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F16D4861CB                , vpunpcklwd(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(C5E9618C2B80000000          , vpunpcklwd(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C5E961CB                    , vpunpcklwd(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(C5ED61CB                    , vpunpcklwd(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(C5E9EFCB                    , vpxor(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(C5E9EF8C2B80000000          , vpxor(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C5EDEFCB                    , vpxor(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(C5EDEF8C2B80000000          , vpxor(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F16548EFAA00200000        , vpxord(zmm5, k0, zmm3, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62111D48EFC0                , vpxord(zmm8, k0, zmm12, zmm24)),
        INSTRUCTION_TEST(62311D48EF84F023010000      , vpxord(zmm8, k0, zmm12, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62711D48EF01                , vpxord(zmm8, k0, zmm12, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62711D48EF427F              , vpxord(zmm8, k0, zmm12, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62711D48EF82C0DFFFFF        , vpxord(zmm8, k0, zmm12, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62711D48EF8200200000        , vpxord(zmm8, k0, zmm12, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62F16548EF6A7F              , vpxord(zmm5, k0, zmm3, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62711D48EF4280              , vpxord(zmm8, k0, zmm12, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62F16548EFAAC0DFFFFF        , vpxord(zmm5, k0, zmm3, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62916548EFE9                , vpxord(zmm5, k0, zmm3, zmm25)),
        INSTRUCTION_TEST(62F16548EF29                , vpxord(zmm5, k0, zmm3, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62F16D08EFCB                , vpxord(xmm1, k0, xmm2, xmm3)),
        INSTRUCTION_TEST(62F16D08EF4C2B08            , vpxord(xmm1, k0, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F16D28EFCB                , vpxord(ymm1, k0, ymm2, ymm3)),
        INSTRUCTION_TEST(62F16D28EF4C2B04            , vpxord(ymm1, k0, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F16548EF6A80              , vpxord(zmm5, k0, zmm3, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62F16D48EF4C2B02            , vpxord(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62B16548EFACF034120000      , vpxord(zmm5, k0, zmm3, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62F16D48EFCB                , vpxord(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(62F1CD40EFBA00200000        , vpxorq(zmm7, k0, zmm22, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(6271DD48EF92C0DFFFFF        , vpxorq(zmm10, k0, zmm4, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62F1CD40EFBAC0DFFFFF        , vpxorq(zmm7, k0, zmm22, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62F1CD40EF7A80              , vpxorq(zmm7, k0, zmm22, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62F1CD40EF39                , vpxorq(zmm7, k0, zmm22, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62B1CD40EFBCF023010000      , vpxorq(zmm7, k0, zmm22, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62D1CD40EFFA                , vpxorq(zmm7, k0, zmm22, zmm10)),
        INSTRUCTION_TEST(6271DD48EF9200200000        , vpxorq(zmm10, k0, zmm4, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(6271DD48EF527F              , vpxorq(zmm10, k0, zmm4, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62F1CD40EF7A7F              , vpxorq(zmm7, k0, zmm22, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(6271DD48EF5280              , vpxorq(zmm10, k0, zmm4, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(6231DD48EF94F034120000      , vpxorq(zmm10, k0, zmm4, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(6231DD48EFD2                , vpxorq(zmm10, k0, zmm4, zmm18)),
        INSTRUCTION_TEST(62F1ED48EF4C2B02            , vpxorq(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F1ED48EFCB                , vpxorq(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(62F1ED28EF4C2B04            , vpxorq(ymm1, k0, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F1ED28EFCB                , vpxorq(ymm1, k0, ymm2, ymm3)),
        INSTRUCTION_TEST(62F1ED08EF4C2B08            , vpxorq(xmm1, k0, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F1ED08EFCB                , vpxorq(xmm1, k0, xmm2, xmm3)),
        INSTRUCTION_TEST(6271DD48EF11                , vpxorq(zmm10, k0, zmm4, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62F3ED0850CB01              , vrangepd(xmm1, k0, xmm2, xmm3, Imm(1))),
        INSTRUCTION_TEST(62F3ED08504C2B0801          , vrangepd(xmm1, k0, xmm2, xmmword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F3ED2850CB01              , vrangepd(ymm1, k0, ymm2, ymm3, Imm(1))),
        INSTRUCTION_TEST(62F3ED28504C2B0401          , vrangepd(ymm1, k0, ymm2, ymmword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F3ED4850CB01              , vrangepd(zmm1, k0, zmm2, zmm3, Imm(1))),
        INSTRUCTION_TEST(62F3ED48504C2B0201          , vrangepd(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F36D48504C2B0201          , vrangeps(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F36D4850CB01              , vrangeps(zmm1, k0, zmm2, zmm3, Imm(1))),
        INSTRUCTION_TEST(62F36D28504C2B0401          , vrangeps(ymm1, k0, ymm2, ymmword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F36D08504C2B0801          , vrangeps(xmm1, k0, xmm2, xmmword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F36D0850CB01              , vrangeps(xmm1, k0, xmm2, xmm3, Imm(1))),
        INSTRUCTION_TEST(62F36D2850CB01              , vrangeps(ymm1, k0, ymm2, ymm3, Imm(1))),
        INSTRUCTION_TEST(62F3ED08514C2B1001          , vrangesd(xmm1, k0, xmm2, qword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F3ED0851CB01              , vrangesd(xmm1, k0, xmm2, xmm3, Imm(1))),
        INSTRUCTION_TEST(62F36D08514C2B2001          , vrangess(xmm1, k0, xmm2, dword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F36D0851CB01              , vrangess(xmm1, k0, xmm2, xmm3, Imm(1))),
        INSTRUCTION_TEST(6262FD484C9200200000        , vrcp14pd(zmm26, k0, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(6262FD484C527F              , vrcp14pd(zmm26, k0, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(6262FD484C92C0DFFFFF        , vrcp14pd(zmm26, k0, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(6262FD484C5280              , vrcp14pd(zmm26, k0, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(6262FD484C11                , vrcp14pd(zmm26, k0, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(6222FD484C94F034120000      , vrcp14pd(zmm26, k0, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(6242FD484CD2                , vrcp14pd(zmm26, k0, zmm10)),
        INSTRUCTION_TEST(6272FD484CAA00200000        , vrcp14pd(zmm13, k0, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(6272FD484CAAC0DFFFFF        , vrcp14pd(zmm13, k0, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(6272FD484C6A7F              , vrcp14pd(zmm13, k0, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(6272FD484C29                , vrcp14pd(zmm13, k0, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(6272FD484CEC                , vrcp14pd(zmm13, k0, zmm4)),
        INSTRUCTION_TEST(6232FD484CACF023010000      , vrcp14pd(zmm13, k0, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62F2FD484C4C1A02            , vrcp14pd(zmm1, k0, zmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F2FD484CCA                , vrcp14pd(zmm1, k0, zmm2)),
        INSTRUCTION_TEST(62F2FD284C4C1A04            , vrcp14pd(ymm1, k0, ymmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F2FD284CCA                , vrcp14pd(ymm1, k0, ymm2)),
        INSTRUCTION_TEST(62F2FD084C4C1A08            , vrcp14pd(xmm1, k0, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F2FD084CCA                , vrcp14pd(xmm1, k0, xmm2)),
        INSTRUCTION_TEST(6272FD484C6A80              , vrcp14pd(zmm13, k0, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62727D484C9200200000        , vrcp14ps(zmm10, k0, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62927D484CDB                , vrcp14ps(zmm3, k0, zmm27)),
        INSTRUCTION_TEST(62B27D484C9CF034120000      , vrcp14ps(zmm3, k0, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62727D484C527F              , vrcp14ps(zmm10, k0, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62F27D484C5A80              , vrcp14ps(zmm3, k0, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62F27D484C9A00200000        , vrcp14ps(zmm3, k0, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62F27D484C5A7F              , vrcp14ps(zmm3, k0, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62F27D484C19                , vrcp14ps(zmm3, k0, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62727D484C92C0DFFFFF        , vrcp14ps(zmm10, k0, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62F27D484C9AC0DFFFFF        , vrcp14ps(zmm3, k0, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62727D484C11                , vrcp14ps(zmm10, k0, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62727D484C5280              , vrcp14ps(zmm10, k0, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62F27D084C4C1A08            , vrcp14ps(xmm1, k0, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F27D284CCA                , vrcp14ps(ymm1, k0, ymm2)),
        INSTRUCTION_TEST(62F27D284C4C1A04            , vrcp14ps(ymm1, k0, ymmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F27D084CCA                , vrcp14ps(xmm1, k0, xmm2)),
        INSTRUCTION_TEST(62F27D484C4C1A02            , vrcp14ps(zmm1, k0, zmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62127D484CD1                , vrcp14ps(zmm10, k0, zmm25)),
        INSTRUCTION_TEST(62327D484C94F023010000      , vrcp14ps(zmm10, k0, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62F27D484CCA                , vrcp14ps(zmm1, k0, zmm2)),
        INSTRUCTION_TEST(6232CD084DB4F034120000      , vrcp14sd(xmm14, k0, xmm6, qword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(6272CD084DF7                , vrcp14sd(xmm14, k0, xmm6, xmm7)),
        INSTRUCTION_TEST(6272CD084DB200040000        , vrcp14sd(xmm14, k0, xmm6, qword_ptr(rdx, 1024))),
        INSTRUCTION_TEST(6272CD084D727F              , vrcp14sd(xmm14, k0, xmm6, qword_ptr(rdx, 1016))),
        INSTRUCTION_TEST(6272CD084DB2F8FBFFFF        , vrcp14sd(xmm14, k0, xmm6, qword_ptr(rdx, -1032))),
        INSTRUCTION_TEST(6272CD084D7280              , vrcp14sd(xmm14, k0, xmm6, qword_ptr(rdx, -1024))),
        INSTRUCTION_TEST(6272CD084D31                , vrcp14sd(xmm14, k0, xmm6, qword_ptr(rcx))),
        INSTRUCTION_TEST(6272CD004DA200040000        , vrcp14sd(xmm12, k0, xmm22, qword_ptr(rdx, 1024))),
        INSTRUCTION_TEST(6272CD004DA2F8FBFFFF        , vrcp14sd(xmm12, k0, xmm22, qword_ptr(rdx, -1032))),
        INSTRUCTION_TEST(6272CD004D6280              , vrcp14sd(xmm12, k0, xmm22, qword_ptr(rdx, -1024))),
        INSTRUCTION_TEST(6272CD004D21                , vrcp14sd(xmm12, k0, xmm22, qword_ptr(rcx))),
        INSTRUCTION_TEST(6252CD004DE6                , vrcp14sd(xmm12, k0, xmm22, xmm14)),
        INSTRUCTION_TEST(6232CD004DA4F023010000      , vrcp14sd(xmm12, k0, xmm22, qword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62F2ED084DCB                , vrcp14sd(xmm1, k0, xmm2, xmm3)),
        INSTRUCTION_TEST(62F2ED084D4C2B10            , vrcp14sd(xmm1, k0, xmm2, qword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(6272CD004D627F              , vrcp14sd(xmm12, k0, xmm22, qword_ptr(rdx, 1016))),
        INSTRUCTION_TEST(62323D084D84F023010000      , vrcp14ss(xmm8, k0, xmm8, dword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62723D084DC3                , vrcp14ss(xmm8, k0, xmm8, xmm3)),
        INSTRUCTION_TEST(62723D084D427F              , vrcp14ss(xmm8, k0, xmm8, dword_ptr(rdx, 508))),
        INSTRUCTION_TEST(62723D084D82FCFDFFFF        , vrcp14ss(xmm8, k0, xmm8, dword_ptr(rdx, -516))),
        INSTRUCTION_TEST(62723D084D4280              , vrcp14ss(xmm8, k0, xmm8, dword_ptr(rdx, -512))),
        INSTRUCTION_TEST(62723D084D01                , vrcp14ss(xmm8, k0, xmm8, dword_ptr(rcx))),
        INSTRUCTION_TEST(626275084D8A00020000        , vrcp14ss(xmm25, k0, xmm1, dword_ptr(rdx, 512))),
        INSTRUCTION_TEST(62723D084D8200020000        , vrcp14ss(xmm8, k0, xmm8, dword_ptr(rdx, 512))),
        INSTRUCTION_TEST(626275084D8AFCFDFFFF        , vrcp14ss(xmm25, k0, xmm1, dword_ptr(rdx, -516))),
        INSTRUCTION_TEST(626275084D4A80              , vrcp14ss(xmm25, k0, xmm1, dword_ptr(rdx, -512))),
        INSTRUCTION_TEST(626275084D09                , vrcp14ss(xmm25, k0, xmm1, dword_ptr(rcx))),
        INSTRUCTION_TEST(622275084DCD                , vrcp14ss(xmm25, k0, xmm1, xmm21)),
        INSTRUCTION_TEST(622275084D8CF034120000      , vrcp14ss(xmm25, k0, xmm1, dword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62F26D084DCB                , vrcp14ss(xmm1, k0, xmm2, xmm3)),
        INSTRUCTION_TEST(62F26D084D4C2B20            , vrcp14ss(xmm1, k0, xmm2, dword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(626275084D4A7F              , vrcp14ss(xmm25, k0, xmm1, dword_ptr(rdx, 508))),
        INSTRUCTION_TEST(62F2FD48CACA                , vrcp28pd(zmm1, k0, zmm2)),
        INSTRUCTION_TEST(62F2FD48CA4C1A02            , vrcp28pd(zmm1, k0, zmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F27D48CACA                , vrcp28ps(zmm1, k0, zmm2)),
        INSTRUCTION_TEST(62F27D48CA4C1A02            , vrcp28ps(zmm1, k0, zmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F2ED08CBCB                , vrcp28sd(xmm1, k0, xmm2, xmm3)),
        INSTRUCTION_TEST(62F2ED08CB4C2B10            , vrcp28sd(xmm1, k0, xmm2, qword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F26D08CB4C2B20            , vrcp28ss(xmm1, k0, xmm2, dword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F26D08CBCB                , vrcp28ss(xmm1, k0, xmm2, xmm3)),
        INSTRUCTION_TEST(C5F853CA                    , vrcpps(xmm1, xmm2)),
        INSTRUCTION_TEST(C5F8538C1A80000000          , vrcpps(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(C5FC53CA                    , vrcpps(ymm1, ymm2)),
        INSTRUCTION_TEST(C5FC538C1A80000000          , vrcpps(ymm1, ymmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(C5EA538C2B80000000          , vrcpss(xmm1, xmm2, dword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C5EA53CB                    , vrcpss(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(62F3FD4856CA01              , vreducepd(zmm1, k0, zmm2, Imm(1))),
        INSTRUCTION_TEST(62F3FD48564C1A0201          , vreducepd(zmm1, k0, zmmword_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F3FD28564C1A0401          , vreducepd(ymm1, k0, ymmword_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F3FD08564C1A0801          , vreducepd(xmm1, k0, xmmword_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F3FD0856CA01              , vreducepd(xmm1, k0, xmm2, Imm(1))),
        INSTRUCTION_TEST(62F3FD2856CA01              , vreducepd(ymm1, k0, ymm2, Imm(1))),
        INSTRUCTION_TEST(62F37D0856CA01              , vreduceps(xmm1, k0, xmm2, Imm(1))),
        INSTRUCTION_TEST(62F37D08564C1A0801          , vreduceps(xmm1, k0, xmmword_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F37D2856CA01              , vreduceps(ymm1, k0, ymm2, Imm(1))),
        INSTRUCTION_TEST(62F37D28564C1A0401          , vreduceps(ymm1, k0, ymmword_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F37D4856CA01              , vreduceps(zmm1, k0, zmm2, Imm(1))),
        INSTRUCTION_TEST(62F37D48564C1A0201          , vreduceps(zmm1, k0, zmmword_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F3ED08574C2B1001          , vreducesd(xmm1, k0, xmm2, qword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F3ED0857CB01              , vreducesd(xmm1, k0, xmm2, xmm3, Imm(1))),
        INSTRUCTION_TEST(62F36D0857CB01              , vreducess(xmm1, k0, xmm2, xmm3, Imm(1))),
        INSTRUCTION_TEST(62F36D08574C2B2001          , vreducess(xmm1, k0, xmm2, dword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62E3FD4809727F7B            , vrndscalepd(zmm22, k0, zmmword_ptr(rdx, 8128), Imm(123))),
        INSTRUCTION_TEST(62E3FD4809B2C0DFFFFF7B      , vrndscalepd(zmm22, k0, zmmword_ptr(rdx, -8256), Imm(123))),
        INSTRUCTION_TEST(62E3FD480972807B            , vrndscalepd(zmm22, k0, zmmword_ptr(rdx, -8192), Imm(123))),
        INSTRUCTION_TEST(62E3FD4809317B              , vrndscalepd(zmm22, k0, zmmword_ptr(rcx), Imm(123))),
        INSTRUCTION_TEST(62A3FD4809B4F0230100007B    , vrndscalepd(zmm22, k0, zmmword_ptr(rax, r14, 8, 291), Imm(123))),
        INSTRUCTION_TEST(62E3FD4809F7AB              , vrndscalepd(zmm22, k0, zmm7, Imm(171))),
        INSTRUCTION_TEST(62E3FD4809B2002000007B      , vrndscalepd(zmm22, k0, zmmword_ptr(rdx, 8192), Imm(123))),
        INSTRUCTION_TEST(6273FD4809AA002000007B      , vrndscalepd(zmm13, k0, zmmword_ptr(rdx, 8192), Imm(123))),
        INSTRUCTION_TEST(6273FD48096A7F7B            , vrndscalepd(zmm13, k0, zmmword_ptr(rdx, 8128), Imm(123))),
        INSTRUCTION_TEST(6273FD4809AAC0DFFFFF7B      , vrndscalepd(zmm13, k0, zmmword_ptr(rdx, -8256), Imm(123))),
        INSTRUCTION_TEST(62E3FD4809F77B              , vrndscalepd(zmm22, k0, zmm7, Imm(123))),
        INSTRUCTION_TEST(6273FD4809297B              , vrndscalepd(zmm13, k0, zmmword_ptr(rcx), Imm(123))),
        INSTRUCTION_TEST(62F3FD0809CA01              , vrndscalepd(xmm1, k0, xmm2, Imm(1))),
        INSTRUCTION_TEST(62F3FD08094C1A0801          , vrndscalepd(xmm1, k0, xmmword_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F3FD2809CA01              , vrndscalepd(ymm1, k0, ymm2, Imm(1))),
        INSTRUCTION_TEST(62F3FD28094C1A0401          , vrndscalepd(ymm1, k0, ymmword_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(6273FD48096A807B            , vrndscalepd(zmm13, k0, zmmword_ptr(rdx, -8192), Imm(123))),
        INSTRUCTION_TEST(62F3FD48094C1A0201          , vrndscalepd(zmm1, k0, zmmword_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(6213FD4809EA7B              , vrndscalepd(zmm13, k0, zmm26, Imm(123))),
        INSTRUCTION_TEST(6213FD4809EAAB              , vrndscalepd(zmm13, k0, zmm26, Imm(171))),
        INSTRUCTION_TEST(6233FD4809ACF0341200007B    , vrndscalepd(zmm13, k0, zmmword_ptr(rax, r14, 8, 4660), Imm(123))),
        INSTRUCTION_TEST(62F3FD4809CA01              , vrndscalepd(zmm1, k0, zmm2, Imm(1))),
        INSTRUCTION_TEST(62737D48086A7F7B            , vrndscaleps(zmm13, k0, zmmword_ptr(rdx, 8128), Imm(123))),
        INSTRUCTION_TEST(62637D480882002000007B      , vrndscaleps(zmm24, k0, zmmword_ptr(rdx, 8192), Imm(123))),
        INSTRUCTION_TEST(62637D4808427F7B            , vrndscaleps(zmm24, k0, zmmword_ptr(rdx, 8128), Imm(123))),
        INSTRUCTION_TEST(62637D480882C0DFFFFF7B      , vrndscaleps(zmm24, k0, zmmword_ptr(rdx, -8256), Imm(123))),
        INSTRUCTION_TEST(62637D480842807B            , vrndscaleps(zmm24, k0, zmmword_ptr(rdx, -8192), Imm(123))),
        INSTRUCTION_TEST(62637D4808017B              , vrndscaleps(zmm24, k0, zmmword_ptr(rcx), Imm(123))),
        INSTRUCTION_TEST(62637D4808C7AB              , vrndscaleps(zmm24, k0, zmm7, Imm(171))),
        INSTRUCTION_TEST(62637D4808C77B              , vrndscaleps(zmm24, k0, zmm7, Imm(123))),
        INSTRUCTION_TEST(62237D480884F0341200007B    , vrndscaleps(zmm24, k0, zmmword_ptr(rax, r14, 8, 4660), Imm(123))),
        INSTRUCTION_TEST(62737D4808AA002000007B      , vrndscaleps(zmm13, k0, zmmword_ptr(rdx, 8192), Imm(123))),
        INSTRUCTION_TEST(62737D4808AAC0DFFFFF7B      , vrndscaleps(zmm13, k0, zmmword_ptr(rdx, -8256), Imm(123))),
        INSTRUCTION_TEST(62737D4808297B              , vrndscaleps(zmm13, k0, zmmword_ptr(rcx), Imm(123))),
        INSTRUCTION_TEST(62F37D08084C1A0801          , vrndscaleps(xmm1, k0, xmmword_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62737D4808EFAB              , vrndscaleps(zmm13, k0, zmm7, Imm(171))),
        INSTRUCTION_TEST(62737D4808EF7B              , vrndscaleps(zmm13, k0, zmm7, Imm(123))),
        INSTRUCTION_TEST(62337D4808ACF0230100007B    , vrndscaleps(zmm13, k0, zmmword_ptr(rax, r14, 8, 291), Imm(123))),
        INSTRUCTION_TEST(62F37D48084C1A0201          , vrndscaleps(zmm1, k0, zmmword_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F37D4808CA01              , vrndscaleps(zmm1, k0, zmm2, Imm(1))),
        INSTRUCTION_TEST(62F37D28084C1A0401          , vrndscaleps(ymm1, k0, ymmword_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F37D2808CA01              , vrndscaleps(ymm1, k0, ymm2, Imm(1))),
        INSTRUCTION_TEST(62737D48086A807B            , vrndscaleps(zmm13, k0, zmmword_ptr(rdx, -8192), Imm(123))),
        INSTRUCTION_TEST(62F37D0808CA01              , vrndscaleps(xmm1, k0, xmm2, Imm(1))),
        INSTRUCTION_TEST(6223C5080BD27B              , vrndscalesd(xmm26, k0, xmm7, xmm18, Imm(123))),
        INSTRUCTION_TEST(6263C5080B92000400007B      , vrndscalesd(xmm26, k0, xmm7, qword_ptr(rdx, 1024), Imm(123))),
        INSTRUCTION_TEST(6263C5080B117B              , vrndscalesd(xmm26, k0, xmm7, qword_ptr(rcx), Imm(123))),
        INSTRUCTION_TEST(6223C5080B94F0341200007B    , vrndscalesd(xmm26, k0, xmm7, qword_ptr(rax, r14, 8, 4660), Imm(123))),
        INSTRUCTION_TEST(6263C5080B92F8FBFFFF7B      , vrndscalesd(xmm26, k0, xmm7, qword_ptr(rdx, -1032), Imm(123))),
        INSTRUCTION_TEST(6263C5080B527F7B            , vrndscalesd(xmm26, k0, xmm7, qword_ptr(rdx, 1016), Imm(123))),
        INSTRUCTION_TEST(6263C5080B52807B            , vrndscalesd(xmm26, k0, xmm7, qword_ptr(rdx, -1024), Imm(123))),
        INSTRUCTION_TEST(62639D080B8A000400007B      , vrndscalesd(xmm25, k0, xmm12, qword_ptr(rdx, 1024), Imm(123))),
        INSTRUCTION_TEST(6223C5080BD2AB              , vrndscalesd(xmm26, k0, xmm7, xmm18, Imm(171))),
        INSTRUCTION_TEST(62639D080B8AF8FBFFFF7B      , vrndscalesd(xmm25, k0, xmm12, qword_ptr(rdx, -1032), Imm(123))),
        INSTRUCTION_TEST(62639D080B4A807B            , vrndscalesd(xmm25, k0, xmm12, qword_ptr(rdx, -1024), Imm(123))),
        INSTRUCTION_TEST(62639D080B097B              , vrndscalesd(xmm25, k0, xmm12, qword_ptr(rcx), Imm(123))),
        INSTRUCTION_TEST(62439D080BCFAB              , vrndscalesd(xmm25, k0, xmm12, xmm15, Imm(171))),
        INSTRUCTION_TEST(62439D080BCF7B              , vrndscalesd(xmm25, k0, xmm12, xmm15, Imm(123))),
        INSTRUCTION_TEST(62239D080B8CF0230100007B    , vrndscalesd(xmm25, k0, xmm12, qword_ptr(rax, r14, 8, 291), Imm(123))),
        INSTRUCTION_TEST(62F3ED080BCB01              , vrndscalesd(xmm1, k0, xmm2, xmm3, Imm(1))),
        INSTRUCTION_TEST(62F3ED080B4C2B1001          , vrndscalesd(xmm1, k0, xmm2, qword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62639D080B4A7F7B            , vrndscalesd(xmm25, k0, xmm12, qword_ptr(rdx, 1016), Imm(123))),
        INSTRUCTION_TEST(62F345000A397B              , vrndscaless(xmm7, k0, xmm23, dword_ptr(rcx), Imm(123))),
        INSTRUCTION_TEST(62F345000A7A807B            , vrndscaless(xmm7, k0, xmm23, dword_ptr(rdx, -512), Imm(123))),
        INSTRUCTION_TEST(62F345000ABAFCFDFFFF7B      , vrndscaless(xmm7, k0, xmm23, dword_ptr(rdx, -516), Imm(123))),
        INSTRUCTION_TEST(62F345000AFEAB              , vrndscaless(xmm7, k0, xmm23, xmm6, Imm(171))),
        INSTRUCTION_TEST(62F345000AFE7B              , vrndscaless(xmm7, k0, xmm23, xmm6, Imm(123))),
        INSTRUCTION_TEST(62B345000ABCF0341200007B    , vrndscaless(xmm7, k0, xmm23, dword_ptr(rax, r14, 8, 4660), Imm(123))),
        INSTRUCTION_TEST(62F345000A7A7F7B            , vrndscaless(xmm7, k0, xmm23, dword_ptr(rdx, 508), Imm(123))),
        INSTRUCTION_TEST(627325080A9A000200007B      , vrndscaless(xmm11, k0, xmm11, dword_ptr(rdx, 512), Imm(123))),
        INSTRUCTION_TEST(62F345000ABA000200007B      , vrndscaless(xmm7, k0, xmm23, dword_ptr(rdx, 512), Imm(123))),
        INSTRUCTION_TEST(627325080A9AFCFDFFFF7B      , vrndscaless(xmm11, k0, xmm11, dword_ptr(rdx, -516), Imm(123))),
        INSTRUCTION_TEST(627325080A5A807B            , vrndscaless(xmm11, k0, xmm11, dword_ptr(rdx, -512), Imm(123))),
        INSTRUCTION_TEST(627325080A197B              , vrndscaless(xmm11, k0, xmm11, dword_ptr(rcx), Imm(123))),
        INSTRUCTION_TEST(623325080AD9AB              , vrndscaless(xmm11, k0, xmm11, xmm17, Imm(171))),
        INSTRUCTION_TEST(623325080AD97B              , vrndscaless(xmm11, k0, xmm11, xmm17, Imm(123))),
        INSTRUCTION_TEST(623325080A9CF0230100007B    , vrndscaless(xmm11, k0, xmm11, dword_ptr(rax, r14, 8, 291), Imm(123))),
        INSTRUCTION_TEST(62F36D080ACB01              , vrndscaless(xmm1, k0, xmm2, xmm3, Imm(1))),
        INSTRUCTION_TEST(62F36D080A4C2B2001          , vrndscaless(xmm1, k0, xmm2, dword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(627325080A5A7F7B            , vrndscaless(xmm11, k0, xmm11, dword_ptr(rdx, 508), Imm(123))),
        INSTRUCTION_TEST(C4E37909CA01                , vroundpd(xmm1, xmm2, Imm(1))),
        INSTRUCTION_TEST(C4E379098C1A8000000001      , vroundpd(xmm1, xmmword_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(C4E37D09CA01                , vroundpd(ymm1, ymm2, Imm(1))),
        INSTRUCTION_TEST(C4E37D098C1A8000000001      , vroundpd(ymm1, ymmword_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(C4E37D088C1A8000000001      , vroundps(ymm1, ymmword_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(C4E37D08CA01                , vroundps(ymm1, ymm2, Imm(1))),
        INSTRUCTION_TEST(C4E37908CA01                , vroundps(xmm1, xmm2, Imm(1))),
        INSTRUCTION_TEST(C4E379088C1A8000000001      , vroundps(xmm1, xmmword_ptr(rdx, rbx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(C4E3690B8C2B8000000001      , vroundsd(xmm1, xmm2, qword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(C4E3690BCB01                , vroundsd(xmm1, xmm2, xmm3, Imm(1))),
        INSTRUCTION_TEST(C4E3690A8C2B8000000001      , vroundss(xmm1, xmm2, dword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(C4E3690ACB01                , vroundss(xmm1, xmm2, xmm3, Imm(1))),
        INSTRUCTION_TEST(6272FD484E727F              , vrsqrt14pd(zmm14, k0, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62E2FD484E5A7F              , vrsqrt14pd(zmm19, k0, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62E2FD484E9AC0DFFFFF        , vrsqrt14pd(zmm19, k0, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62E2FD484E9A00200000        , vrsqrt14pd(zmm19, k0, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62E2FD484E5A80              , vrsqrt14pd(zmm19, k0, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62E2FD484E19                , vrsqrt14pd(zmm19, k0, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62A2FD484E9CF023010000      , vrsqrt14pd(zmm19, k0, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62C2FD484EDE                , vrsqrt14pd(zmm19, k0, zmm14)),
        INSTRUCTION_TEST(6272FD484EB200200000        , vrsqrt14pd(zmm14, k0, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(6272FD484EB2C0DFFFFF        , vrsqrt14pd(zmm14, k0, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62F2FD284ECA                , vrsqrt14pd(ymm1, k0, ymm2)),
        INSTRUCTION_TEST(6272FD484E31                , vrsqrt14pd(zmm14, k0, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(6272FD484EF4                , vrsqrt14pd(zmm14, k0, zmm4)),
        INSTRUCTION_TEST(6232FD484EB4F034120000      , vrsqrt14pd(zmm14, k0, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62F2FD484E4C1A02            , vrsqrt14pd(zmm1, k0, zmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F2FD484ECA                , vrsqrt14pd(zmm1, k0, zmm2)),
        INSTRUCTION_TEST(62F2FD284E4C1A04            , vrsqrt14pd(ymm1, k0, ymmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F2FD084E4C1A08            , vrsqrt14pd(xmm1, k0, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F2FD084ECA                , vrsqrt14pd(xmm1, k0, xmm2)),
        INSTRUCTION_TEST(6272FD484E7280              , vrsqrt14pd(zmm14, k0, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62E27D484E8200200000        , vrsqrt14ps(zmm16, k0, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62927D484ED9                , vrsqrt14ps(zmm3, k0, zmm25)),
        INSTRUCTION_TEST(62B27D484E9CF034120000      , vrsqrt14ps(zmm3, k0, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62F27D484E19                , vrsqrt14ps(zmm3, k0, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62F27D484E9A00200000        , vrsqrt14ps(zmm3, k0, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62F27D484E9AC0DFFFFF        , vrsqrt14ps(zmm3, k0, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62F27D484E5A7F              , vrsqrt14ps(zmm3, k0, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62E27D484E427F              , vrsqrt14ps(zmm16, k0, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62F27D484E5A80              , vrsqrt14ps(zmm3, k0, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62E27D484E82C0DFFFFF        , vrsqrt14ps(zmm16, k0, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62E27D484E01                , vrsqrt14ps(zmm16, k0, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62A27D484E84F023010000      , vrsqrt14ps(zmm16, k0, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62C27D484EC1                , vrsqrt14ps(zmm16, k0, zmm9)),
        INSTRUCTION_TEST(62F27D484E4C1A02            , vrsqrt14ps(zmm1, k0, zmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F27D484ECA                , vrsqrt14ps(zmm1, k0, zmm2)),
        INSTRUCTION_TEST(62F27D284E4C1A04            , vrsqrt14ps(ymm1, k0, ymmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F27D284ECA                , vrsqrt14ps(ymm1, k0, ymm2)),
        INSTRUCTION_TEST(62F27D084E4C1A08            , vrsqrt14ps(xmm1, k0, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F27D084ECA                , vrsqrt14ps(xmm1, k0, xmm2)),
        INSTRUCTION_TEST(62E27D484E4280              , vrsqrt14ps(zmm16, k0, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(6222CD084F94F023010000      , vrsqrt14sd(xmm26, k0, xmm6, qword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(6262CD084F9200040000        , vrsqrt14sd(xmm26, k0, xmm6, qword_ptr(rdx, 1024))),
        INSTRUCTION_TEST(6262CD084F527F              , vrsqrt14sd(xmm26, k0, xmm6, qword_ptr(rdx, 1016))),
        INSTRUCTION_TEST(6262CD084F5280              , vrsqrt14sd(xmm26, k0, xmm6, qword_ptr(rdx, -1024))),
        INSTRUCTION_TEST(6262CD084F11                , vrsqrt14sd(xmm26, k0, xmm6, qword_ptr(rcx))),
        INSTRUCTION_TEST(6242CD084FD2                , vrsqrt14sd(xmm26, k0, xmm6, xmm10)),
        INSTRUCTION_TEST(62F2D5004FCB                , vrsqrt14sd(xmm1, k0, xmm21, xmm3)),
        INSTRUCTION_TEST(6262CD084F92F8FBFFFF        , vrsqrt14sd(xmm26, k0, xmm6, qword_ptr(rdx, -1032))),
        INSTRUCTION_TEST(62F2D5004F4A7F              , vrsqrt14sd(xmm1, k0, xmm21, qword_ptr(rdx, 1016))),
        INSTRUCTION_TEST(62F2D5004F8AF8FBFFFF        , vrsqrt14sd(xmm1, k0, xmm21, qword_ptr(rdx, -1032))),
        INSTRUCTION_TEST(62F2D5004F4A80              , vrsqrt14sd(xmm1, k0, xmm21, qword_ptr(rdx, -1024))),
        INSTRUCTION_TEST(62F2D5004F09                , vrsqrt14sd(xmm1, k0, xmm21, qword_ptr(rcx))),
        INSTRUCTION_TEST(62B2D5004F8CF034120000      , vrsqrt14sd(xmm1, k0, xmm21, qword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62F2D5004F8A00040000        , vrsqrt14sd(xmm1, k0, xmm21, qword_ptr(rdx, 1024))),
        INSTRUCTION_TEST(62F2ED084FCB                , vrsqrt14sd(xmm1, k0, xmm2, xmm3)),
        INSTRUCTION_TEST(62F2ED084F4C2B10            , vrsqrt14sd(xmm1, k0, xmm2, qword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62E22D004F9A00020000        , vrsqrt14ss(xmm19, k0, xmm26, dword_ptr(rdx, 512))),
        INSTRUCTION_TEST(62E22D004F5A7F              , vrsqrt14ss(xmm19, k0, xmm26, dword_ptr(rdx, 508))),
        INSTRUCTION_TEST(62E22D004F9AFCFDFFFF        , vrsqrt14ss(xmm19, k0, xmm26, dword_ptr(rdx, -516))),
        INSTRUCTION_TEST(62E22D004F5A80              , vrsqrt14ss(xmm19, k0, xmm26, dword_ptr(rdx, -512))),
        INSTRUCTION_TEST(62E22D004F19                , vrsqrt14ss(xmm19, k0, xmm26, dword_ptr(rcx))),
        INSTRUCTION_TEST(62A22D004F9CF034120000      , vrsqrt14ss(xmm19, k0, xmm26, dword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62720D084FB200020000        , vrsqrt14ss(xmm14, k0, xmm14, dword_ptr(rdx, 512))),
        INSTRUCTION_TEST(62A22D004FDA                , vrsqrt14ss(xmm19, k0, xmm26, xmm18)),
        INSTRUCTION_TEST(62720D084FB2FCFDFFFF        , vrsqrt14ss(xmm14, k0, xmm14, dword_ptr(rdx, -516))),
        INSTRUCTION_TEST(62720D084F7280              , vrsqrt14ss(xmm14, k0, xmm14, dword_ptr(rdx, -512))),
        INSTRUCTION_TEST(62720D084F31                , vrsqrt14ss(xmm14, k0, xmm14, dword_ptr(rcx))),
        INSTRUCTION_TEST(62520D084FF1                , vrsqrt14ss(xmm14, k0, xmm14, xmm9)),
        INSTRUCTION_TEST(62320D084FB4F023010000      , vrsqrt14ss(xmm14, k0, xmm14, dword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62F26D084FCB                , vrsqrt14ss(xmm1, k0, xmm2, xmm3)),
        INSTRUCTION_TEST(62F26D084F4C2B20            , vrsqrt14ss(xmm1, k0, xmm2, dword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62720D084F727F              , vrsqrt14ss(xmm14, k0, xmm14, dword_ptr(rdx, 508))),
        INSTRUCTION_TEST(62F2FD48CCCA                , vrsqrt28pd(zmm1, k0, zmm2)),
        INSTRUCTION_TEST(62F2FD48CC4C1A02            , vrsqrt28pd(zmm1, k0, zmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F27D48CCCA                , vrsqrt28ps(zmm1, k0, zmm2)),
        INSTRUCTION_TEST(62F27D48CC4C1A02            , vrsqrt28ps(zmm1, k0, zmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F2ED08CD4C2B10            , vrsqrt28sd(xmm1, k0, xmm2, qword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F2ED08CDCB                , vrsqrt28sd(xmm1, k0, xmm2, xmm3)),
        INSTRUCTION_TEST(62F26D08CDCB                , vrsqrt28ss(xmm1, k0, xmm2, xmm3)),
        INSTRUCTION_TEST(62F26D08CD4C2B20            , vrsqrt28ss(xmm1, k0, xmm2, dword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C5F852CA                    , vrsqrtps(xmm1, xmm2)),
        INSTRUCTION_TEST(C5F8528C1A80000000          , vrsqrtps(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(C5FC52CA                    , vrsqrtps(ymm1, ymm2)),
        INSTRUCTION_TEST(C5FC528C1A80000000          , vrsqrtps(ymm1, ymmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(C5EA528C2B80000000          , vrsqrtss(xmm1, xmm2, dword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C5EA52CB                    , vrsqrtss(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(6262AD402C527F              , vscalefpd(zmm26, k0, zmm26, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(6262AD402C9200200000        , vscalefpd(zmm26, k0, zmm26, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62B2DD482CF8                , vscalefpd(zmm7, k0, zmm4, zmm16)),
        INSTRUCTION_TEST(62B2DD482CBCF034120000      , vscalefpd(zmm7, k0, zmm4, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62F2DD482C7A7F              , vscalefpd(zmm7, k0, zmm4, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62F2DD482C7A80              , vscalefpd(zmm7, k0, zmm4, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62F2DD482CBAC0DFFFFF        , vscalefpd(zmm7, k0, zmm4, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62F2DD482CBA00200000        , vscalefpd(zmm7, k0, zmm4, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62F2DD482C39                , vscalefpd(zmm7, k0, zmm4, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(6262AD402C92C0DFFFFF        , vscalefpd(zmm26, k0, zmm26, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62F2ED282CCB                , vscalefpd(ymm1, k0, ymm2, ymm3)),
        INSTRUCTION_TEST(6262AD402C11                , vscalefpd(zmm26, k0, zmm26, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(6222AD402C94F023010000      , vscalefpd(zmm26, k0, zmm26, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(6202AD402CD4                , vscalefpd(zmm26, k0, zmm26, zmm28)),
        INSTRUCTION_TEST(62F2ED482C4C2B02            , vscalefpd(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F2ED482CCB                , vscalefpd(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(62F2ED282C4C2B04            , vscalefpd(ymm1, k0, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(6262AD402C5280              , vscalefpd(zmm26, k0, zmm26, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62F2ED082C4C2B08            , vscalefpd(xmm1, k0, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F2ED082CCB                , vscalefpd(xmm1, k0, xmm2, xmm3)),
        INSTRUCTION_TEST(62E24D482C9A00200000        , vscalefps(zmm19, k0, zmm6, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62B225482CE3                , vscalefps(zmm4, k0, zmm11, zmm19)),
        INSTRUCTION_TEST(62B225482CA4F034120000      , vscalefps(zmm4, k0, zmm11, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62F225482C21                , vscalefps(zmm4, k0, zmm11, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62F225482CA200200000        , vscalefps(zmm4, k0, zmm11, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62F225482CA2C0DFFFFF        , vscalefps(zmm4, k0, zmm11, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62F225482C627F              , vscalefps(zmm4, k0, zmm11, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62E24D482C5A7F              , vscalefps(zmm19, k0, zmm6, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62F225482C6280              , vscalefps(zmm4, k0, zmm11, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62E24D482C9AC0DFFFFF        , vscalefps(zmm19, k0, zmm6, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62E24D482C5A80              , vscalefps(zmm19, k0, zmm6, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62E24D482C19                , vscalefps(zmm19, k0, zmm6, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62F26D082C4C2B08            , vscalefps(xmm1, k0, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F26D282CCB                , vscalefps(ymm1, k0, ymm2, ymm3)),
        INSTRUCTION_TEST(62F26D282C4C2B04            , vscalefps(ymm1, k0, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F26D082CCB                , vscalefps(xmm1, k0, xmm2, xmm3)),
        INSTRUCTION_TEST(62F26D482C4C2B02            , vscalefps(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62A24D482CDA                , vscalefps(zmm19, k0, zmm6, zmm18)),
        INSTRUCTION_TEST(62A24D482C9CF023010000      , vscalefps(zmm19, k0, zmm6, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62F26D482CCB                , vscalefps(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(6222E5002DACF034120000      , vscalefsd(xmm29, k0, xmm19, qword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(6262E5002D6A7F              , vscalefsd(xmm29, k0, xmm19, qword_ptr(rdx, 1016))),
        INSTRUCTION_TEST(6262E5002DAAF8FBFFFF        , vscalefsd(xmm29, k0, xmm19, qword_ptr(rdx, -1032))),
        INSTRUCTION_TEST(6262E5002D6A80              , vscalefsd(xmm29, k0, xmm19, qword_ptr(rdx, -1024))),
        INSTRUCTION_TEST(6262E5002D29                , vscalefsd(xmm29, k0, xmm19, qword_ptr(rcx))),
        INSTRUCTION_TEST(6222E5002DEE                , vscalefsd(xmm29, k0, xmm19, xmm22)),
        INSTRUCTION_TEST(6262E5002DAA00040000        , vscalefsd(xmm29, k0, xmm19, qword_ptr(rdx, 1024))),
        INSTRUCTION_TEST(62A2CD002DED                , vscalefsd(xmm21, k0, xmm22, xmm21)),
        INSTRUCTION_TEST(62E2CD002DAAF8FBFFFF        , vscalefsd(xmm21, k0, xmm22, qword_ptr(rdx, -1032))),
        INSTRUCTION_TEST(62E2CD002D6A7F              , vscalefsd(xmm21, k0, xmm22, qword_ptr(rdx, 1016))),
        INSTRUCTION_TEST(62E2CD002D6A80              , vscalefsd(xmm21, k0, xmm22, qword_ptr(rdx, -1024))),
        INSTRUCTION_TEST(62E2CD002D29                , vscalefsd(xmm21, k0, xmm22, qword_ptr(rcx))),
        INSTRUCTION_TEST(62A2CD002DACF023010000      , vscalefsd(xmm21, k0, xmm22, qword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62F2ED082DCB                , vscalefsd(xmm1, k0, xmm2, xmm3)),
        INSTRUCTION_TEST(62F2ED082D4C2B10            , vscalefsd(xmm1, k0, xmm2, qword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62E2CD002DAA00040000        , vscalefsd(xmm21, k0, xmm22, qword_ptr(rdx, 1024))),
        INSTRUCTION_TEST(62E21D082DE6                , vscalefss(xmm20, k0, xmm12, xmm6)),
        INSTRUCTION_TEST(62E21D082DA200020000        , vscalefss(xmm20, k0, xmm12, dword_ptr(rdx, 512))),
        INSTRUCTION_TEST(62E21D082D627F              , vscalefss(xmm20, k0, xmm12, dword_ptr(rdx, 508))),
        INSTRUCTION_TEST(62E21D082DA2FCFDFFFF        , vscalefss(xmm20, k0, xmm12, dword_ptr(rdx, -516))),
        INSTRUCTION_TEST(62E21D082D6280              , vscalefss(xmm20, k0, xmm12, dword_ptr(rdx, -512))),
        INSTRUCTION_TEST(62E21D082D21                , vscalefss(xmm20, k0, xmm12, dword_ptr(rcx))),
        INSTRUCTION_TEST(62A21D082DA4F034120000      , vscalefss(xmm20, k0, xmm12, dword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(627205082DAA00020000        , vscalefss(xmm13, k0, xmm15, dword_ptr(rdx, 512))),
        INSTRUCTION_TEST(627205082DAAFCFDFFFF        , vscalefss(xmm13, k0, xmm15, dword_ptr(rdx, -516))),
        INSTRUCTION_TEST(627205082D6A80              , vscalefss(xmm13, k0, xmm15, dword_ptr(rdx, -512))),
        INSTRUCTION_TEST(627205082D29                , vscalefss(xmm13, k0, xmm15, dword_ptr(rcx))),
        INSTRUCTION_TEST(623205082DEF                , vscalefss(xmm13, k0, xmm15, xmm23)),
        INSTRUCTION_TEST(623205082DACF023010000      , vscalefss(xmm13, k0, xmm15, dword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62F26D082DCB                , vscalefss(xmm1, k0, xmm2, xmm3)),
        INSTRUCTION_TEST(62F26D082D4C2B20            , vscalefss(xmm1, k0, xmm2, dword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(627205082D6A7F              , vscalefss(xmm13, k0, xmm15, dword_ptr(rdx, 508))),
        INSTRUCTION_TEST(62F33D4023F3AB              , vshuff32x4(zmm6, k0, zmm24, zmm3, Imm(171))),
        INSTRUCTION_TEST(62B33D4023B4F0230100007B    , vshuff32x4(zmm6, k0, zmm24, zmmword_ptr(rax, r14, 8, 291), Imm(123))),
        INSTRUCTION_TEST(62F33D4023317B              , vshuff32x4(zmm6, k0, zmm24, zmmword_ptr(rcx), Imm(123))),
        INSTRUCTION_TEST(62F33D4023B2002000007B      , vshuff32x4(zmm6, k0, zmm24, zmmword_ptr(rdx, 8192), Imm(123))),
        INSTRUCTION_TEST(62F33D4023B2C0DFFFFF7B      , vshuff32x4(zmm6, k0, zmm24, zmmword_ptr(rdx, -8256), Imm(123))),
        INSTRUCTION_TEST(62F33D4023727F7B            , vshuff32x4(zmm6, k0, zmm24, zmmword_ptr(rdx, 8128), Imm(123))),
        INSTRUCTION_TEST(62E32548238A002000007B      , vshuff32x4(zmm17, k0, zmm11, zmmword_ptr(rdx, 8192), Imm(123))),
        INSTRUCTION_TEST(62F33D402372807B            , vshuff32x4(zmm6, k0, zmm24, zmmword_ptr(rdx, -8192), Imm(123))),
        INSTRUCTION_TEST(62E32548234A7F7B            , vshuff32x4(zmm17, k0, zmm11, zmmword_ptr(rdx, 8128), Imm(123))),
        INSTRUCTION_TEST(62F33D4023F37B              , vshuff32x4(zmm6, k0, zmm24, zmm3, Imm(123))),
        INSTRUCTION_TEST(62E32548234A807B            , vshuff32x4(zmm17, k0, zmm11, zmmword_ptr(rdx, -8192), Imm(123))),
        INSTRUCTION_TEST(62F36D2823CB01              , vshuff32x4(ymm1, k0, ymm2, ymm3, Imm(1))),
        INSTRUCTION_TEST(62F36D28234C2B0401          , vshuff32x4(ymm1, k0, ymm2, ymmword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F36D4823CB01              , vshuff32x4(zmm1, k0, zmm2, zmm3, Imm(1))),
        INSTRUCTION_TEST(62F36D48234C2B0201          , vshuff32x4(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62E32548238AC0DFFFFF7B      , vshuff32x4(zmm17, k0, zmm11, zmmword_ptr(rdx, -8256), Imm(123))),
        INSTRUCTION_TEST(62E3254823CCAB              , vshuff32x4(zmm17, k0, zmm11, zmm4, Imm(171))),
        INSTRUCTION_TEST(62A32548238CF0341200007B    , vshuff32x4(zmm17, k0, zmm11, zmmword_ptr(rax, r14, 8, 4660), Imm(123))),
        INSTRUCTION_TEST(62E3254823CC7B              , vshuff32x4(zmm17, k0, zmm11, zmm4, Imm(123))),
        INSTRUCTION_TEST(62E3254823097B              , vshuff32x4(zmm17, k0, zmm11, zmmword_ptr(rcx), Imm(123))),
        INSTRUCTION_TEST(62F39D4823B2002000007B      , vshuff64x2(zmm6, k0, zmm12, zmmword_ptr(rdx, 8192), Imm(123))),
        INSTRUCTION_TEST(62F39D4823727F7B            , vshuff64x2(zmm6, k0, zmm12, zmmword_ptr(rdx, 8128), Imm(123))),
        INSTRUCTION_TEST(62F39D4823B2C0DFFFFF7B      , vshuff64x2(zmm6, k0, zmm12, zmmword_ptr(rdx, -8256), Imm(123))),
        INSTRUCTION_TEST(62F39D482372807B            , vshuff64x2(zmm6, k0, zmm12, zmmword_ptr(rdx, -8192), Imm(123))),
        INSTRUCTION_TEST(62F39D4823317B              , vshuff64x2(zmm6, k0, zmm12, zmmword_ptr(rcx), Imm(123))),
        INSTRUCTION_TEST(62B39D4823B4F0341200007B    , vshuff64x2(zmm6, k0, zmm12, zmmword_ptr(rax, r14, 8, 4660), Imm(123))),
        INSTRUCTION_TEST(62939D4823F0AB              , vshuff64x2(zmm6, k0, zmm12, zmm24, Imm(171))),
        INSTRUCTION_TEST(6273B54023BA002000007B      , vshuff64x2(zmm15, k0, zmm25, zmmword_ptr(rdx, 8192), Imm(123))),
        INSTRUCTION_TEST(6273B540237A7F7B            , vshuff64x2(zmm15, k0, zmm25, zmmword_ptr(rdx, 8128), Imm(123))),
        INSTRUCTION_TEST(62939D4823F07B              , vshuff64x2(zmm6, k0, zmm12, zmm24, Imm(123))),
        INSTRUCTION_TEST(6273B540237A807B            , vshuff64x2(zmm15, k0, zmm25, zmmword_ptr(rdx, -8192), Imm(123))),
        INSTRUCTION_TEST(6273B54023BAC0DFFFFF7B      , vshuff64x2(zmm15, k0, zmm25, zmmword_ptr(rdx, -8256), Imm(123))),
        INSTRUCTION_TEST(62F3ED28234C2B0401          , vshuff64x2(ymm1, k0, ymm2, ymmword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F3ED4823CB01              , vshuff64x2(zmm1, k0, zmm2, zmm3, Imm(1))),
        INSTRUCTION_TEST(62F3ED48234C2B0201          , vshuff64x2(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F3ED2823CB01              , vshuff64x2(ymm1, k0, ymm2, ymm3, Imm(1))),
        INSTRUCTION_TEST(6253B54023FBAB              , vshuff64x2(zmm15, k0, zmm25, zmm11, Imm(171))),
        INSTRUCTION_TEST(6233B54023BCF0230100007B    , vshuff64x2(zmm15, k0, zmm25, zmmword_ptr(rax, r14, 8, 291), Imm(123))),
        INSTRUCTION_TEST(6273B54023397B              , vshuff64x2(zmm15, k0, zmm25, zmmword_ptr(rcx), Imm(123))),
        INSTRUCTION_TEST(6253B54023FB7B              , vshuff64x2(zmm15, k0, zmm25, zmm11, Imm(123))),
        INSTRUCTION_TEST(62F31D40438A002000007B      , vshufi32x4(zmm1, k0, zmm28, zmmword_ptr(rdx, 8192), Imm(123))),
        INSTRUCTION_TEST(62F33D40435A7F7B            , vshufi32x4(zmm3, k0, zmm24, zmmword_ptr(rdx, 8128), Imm(123))),
        INSTRUCTION_TEST(62F33D40439AC0DFFFFF7B      , vshufi32x4(zmm3, k0, zmm24, zmmword_ptr(rdx, -8256), Imm(123))),
        INSTRUCTION_TEST(62F33D40435A807B            , vshufi32x4(zmm3, k0, zmm24, zmmword_ptr(rdx, -8192), Imm(123))),
        INSTRUCTION_TEST(62F33D4043197B              , vshufi32x4(zmm3, k0, zmm24, zmmword_ptr(rcx), Imm(123))),
        INSTRUCTION_TEST(62B33D40439CF0341200007B    , vshufi32x4(zmm3, k0, zmm24, zmmword_ptr(rax, r14, 8, 4660), Imm(123))),
        INSTRUCTION_TEST(62F33D40439A002000007B      , vshufi32x4(zmm3, k0, zmm24, zmmword_ptr(rdx, 8192), Imm(123))),
        INSTRUCTION_TEST(62F33D4043DD7B              , vshufi32x4(zmm3, k0, zmm24, zmm5, Imm(123))),
        INSTRUCTION_TEST(62F31D40438AC0DFFFFF7B      , vshufi32x4(zmm1, k0, zmm28, zmmword_ptr(rdx, -8256), Imm(123))),
        INSTRUCTION_TEST(62F33D4043DDAB              , vshufi32x4(zmm3, k0, zmm24, zmm5, Imm(171))),
        INSTRUCTION_TEST(62F31D40434A7F7B            , vshufi32x4(zmm1, k0, zmm28, zmmword_ptr(rdx, 8128), Imm(123))),
        INSTRUCTION_TEST(62F36D2843CB01              , vshufi32x4(ymm1, k0, ymm2, ymm3, Imm(1))),
        INSTRUCTION_TEST(62F36D28434C2B0401          , vshufi32x4(ymm1, k0, ymm2, ymmword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F36D4843CB01              , vshufi32x4(zmm1, k0, zmm2, zmm3, Imm(1))),
        INSTRUCTION_TEST(62F36D48434C2B0201          , vshufi32x4(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F31D40434A807B            , vshufi32x4(zmm1, k0, zmm28, zmmword_ptr(rdx, -8192), Imm(123))),
        INSTRUCTION_TEST(62931D4043C9AB              , vshufi32x4(zmm1, k0, zmm28, zmm25, Imm(171))),
        INSTRUCTION_TEST(62B31D40438CF0230100007B    , vshufi32x4(zmm1, k0, zmm28, zmmword_ptr(rax, r14, 8, 291), Imm(123))),
        INSTRUCTION_TEST(62F31D4043097B              , vshufi32x4(zmm1, k0, zmm28, zmmword_ptr(rcx), Imm(123))),
        INSTRUCTION_TEST(62931D4043C97B              , vshufi32x4(zmm1, k0, zmm28, zmm25, Imm(123))),
        INSTRUCTION_TEST(62F3FD40439A002000007B      , vshufi64x2(zmm3, k0, zmm16, zmmword_ptr(rdx, 8192), Imm(123))),
        INSTRUCTION_TEST(62638D4843727F7B            , vshufi64x2(zmm30, k0, zmm14, zmmword_ptr(rdx, 8128), Imm(123))),
        INSTRUCTION_TEST(62638D4843B2C0DFFFFF7B      , vshufi64x2(zmm30, k0, zmm14, zmmword_ptr(rdx, -8256), Imm(123))),
        INSTRUCTION_TEST(62638D484372807B            , vshufi64x2(zmm30, k0, zmm14, zmmword_ptr(rdx, -8192), Imm(123))),
        INSTRUCTION_TEST(62638D4843317B              , vshufi64x2(zmm30, k0, zmm14, zmmword_ptr(rcx), Imm(123))),
        INSTRUCTION_TEST(62238D4843B4F0341200007B    , vshufi64x2(zmm30, k0, zmm14, zmmword_ptr(rax, r14, 8, 4660), Imm(123))),
        INSTRUCTION_TEST(62438D4843F1AB              , vshufi64x2(zmm30, k0, zmm14, zmm9, Imm(171))),
        INSTRUCTION_TEST(62438D4843F17B              , vshufi64x2(zmm30, k0, zmm14, zmm9, Imm(123))),
        INSTRUCTION_TEST(62638D4843B2002000007B      , vshufi64x2(zmm30, k0, zmm14, zmmword_ptr(rdx, 8192), Imm(123))),
        INSTRUCTION_TEST(62F3FD40435A7F7B            , vshufi64x2(zmm3, k0, zmm16, zmmword_ptr(rdx, 8128), Imm(123))),
        INSTRUCTION_TEST(62F3FD40435A807B            , vshufi64x2(zmm3, k0, zmm16, zmmword_ptr(rdx, -8192), Imm(123))),
        INSTRUCTION_TEST(62F3FD4043197B              , vshufi64x2(zmm3, k0, zmm16, zmmword_ptr(rcx), Imm(123))),
        INSTRUCTION_TEST(62B3FD40439CF0230100007B    , vshufi64x2(zmm3, k0, zmm16, zmmword_ptr(rax, r14, 8, 291), Imm(123))),
        INSTRUCTION_TEST(62B3FD4043DBAB              , vshufi64x2(zmm3, k0, zmm16, zmm19, Imm(171))),
        INSTRUCTION_TEST(62B3FD4043DB7B              , vshufi64x2(zmm3, k0, zmm16, zmm19, Imm(123))),
        INSTRUCTION_TEST(62F3ED48434C2B0201          , vshufi64x2(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F3ED4843CB01              , vshufi64x2(zmm1, k0, zmm2, zmm3, Imm(1))),
        INSTRUCTION_TEST(62F3ED28434C2B0401          , vshufi64x2(ymm1, k0, ymm2, ymmword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62F3ED2843CB01              , vshufi64x2(ymm1, k0, ymm2, ymm3, Imm(1))),
        INSTRUCTION_TEST(62F3FD40439AC0DFFFFF7B      , vshufi64x2(zmm3, k0, zmm16, zmmword_ptr(rdx, -8256), Imm(123))),
        INSTRUCTION_TEST(62F1ED48C6CB01              , vshufpd(zmm1, k0, zmm2, zmm3, Imm(1))),
        INSTRUCTION_TEST(62F1ED48C64C2B0201          , vshufpd(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(6221BD48C6E67B              , vshufpd(zmm28, k0, zmm8, zmm22, Imm(123))),
        INSTRUCTION_TEST(6221BD48C6E6AB              , vshufpd(zmm28, k0, zmm8, zmm22, Imm(171))),
        INSTRUCTION_TEST(6261BD48C6627F7B            , vshufpd(zmm28, k0, zmm8, zmmword_ptr(rdx, 8128), Imm(123))),
        INSTRUCTION_TEST(6261BD48C6217B              , vshufpd(zmm28, k0, zmm8, zmmword_ptr(rcx), Imm(123))),
        INSTRUCTION_TEST(6261BD48C662807B            , vshufpd(zmm28, k0, zmm8, zmmword_ptr(rdx, -8192), Imm(123))),
        INSTRUCTION_TEST(6261BD48C6A2C0DFFFFF7B      , vshufpd(zmm28, k0, zmm8, zmmword_ptr(rdx, -8256), Imm(123))),
        INSTRUCTION_TEST(6261BD48C6A2002000007B      , vshufpd(zmm28, k0, zmm8, zmmword_ptr(rdx, 8192), Imm(123))),
        INSTRUCTION_TEST(6221BD48C6A4F0230100007B    , vshufpd(zmm28, k0, zmm8, zmmword_ptr(rax, r14, 8, 291), Imm(123))),
        INSTRUCTION_TEST(62F1AD48C68A002000007B      , vshufpd(zmm1, k0, zmm10, zmmword_ptr(rdx, 8192), Imm(123))),
        INSTRUCTION_TEST(62F1AD48C64A7F7B            , vshufpd(zmm1, k0, zmm10, zmmword_ptr(rdx, 8128), Imm(123))),
        INSTRUCTION_TEST(62F1AD48C64A807B            , vshufpd(zmm1, k0, zmm10, zmmword_ptr(rdx, -8192), Imm(123))),
        INSTRUCTION_TEST(62F1AD48C6097B              , vshufpd(zmm1, k0, zmm10, zmmword_ptr(rcx), Imm(123))),
        INSTRUCTION_TEST(62B1AD48C68CF0341200007B    , vshufpd(zmm1, k0, zmm10, zmmword_ptr(rax, r14, 8, 4660), Imm(123))),
        INSTRUCTION_TEST(6291AD48C6C9AB              , vshufpd(zmm1, k0, zmm10, zmm25, Imm(171))),
        INSTRUCTION_TEST(6291AD48C6C97B              , vshufpd(zmm1, k0, zmm10, zmm25, Imm(123))),
        INSTRUCTION_TEST(C5EDC68C2B8000000001        , vshufpd(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(C5EDC6CB01                  , vshufpd(ymm1, ymm2, ymm3, Imm(1))),
        INSTRUCTION_TEST(C5E9C68C2B8000000001        , vshufpd(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(C5E9C6CB01                  , vshufpd(xmm1, xmm2, xmm3, Imm(1))),
        INSTRUCTION_TEST(62F1AD48C68AC0DFFFFF7B      , vshufpd(zmm1, k0, zmm10, zmmword_ptr(rdx, -8256), Imm(123))),
        INSTRUCTION_TEST(62F16C48C64C2B0201          , vshufps(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62D14C48C6E97B              , vshufps(zmm5, k0, zmm6, zmm9, Imm(123))),
        INSTRUCTION_TEST(62D14C48C6E9AB              , vshufps(zmm5, k0, zmm6, zmm9, Imm(171))),
        INSTRUCTION_TEST(62B14C48C6ACF0230100007B    , vshufps(zmm5, k0, zmm6, zmmword_ptr(rax, r14, 8, 291), Imm(123))),
        INSTRUCTION_TEST(62F14C48C6AA002000007B      , vshufps(zmm5, k0, zmm6, zmmword_ptr(rdx, 8192), Imm(123))),
        INSTRUCTION_TEST(62F14C48C66A807B            , vshufps(zmm5, k0, zmm6, zmmword_ptr(rdx, -8192), Imm(123))),
        INSTRUCTION_TEST(62F14C48C6AAC0DFFFFF7B      , vshufps(zmm5, k0, zmm6, zmmword_ptr(rdx, -8256), Imm(123))),
        INSTRUCTION_TEST(62F14C48C66A7F7B            , vshufps(zmm5, k0, zmm6, zmmword_ptr(rdx, 8128), Imm(123))),
        INSTRUCTION_TEST(62F16C48C6CB01              , vshufps(zmm1, k0, zmm2, zmm3, Imm(1))),
        INSTRUCTION_TEST(62F14C48C6297B              , vshufps(zmm5, k0, zmm6, zmmword_ptr(rcx), Imm(123))),
        INSTRUCTION_TEST(62F17448C68A002000007B      , vshufps(zmm1, k0, zmm1, zmmword_ptr(rdx, 8192), Imm(123))),
        INSTRUCTION_TEST(62F17448C64A7F7B            , vshufps(zmm1, k0, zmm1, zmmword_ptr(rdx, 8128), Imm(123))),
        INSTRUCTION_TEST(62F17448C68AC0DFFFFF7B      , vshufps(zmm1, k0, zmm1, zmmword_ptr(rdx, -8256), Imm(123))),
        INSTRUCTION_TEST(C5E8C68C2B8000000001        , vshufps(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(C5ECC6CB01                  , vshufps(ymm1, ymm2, ymm3, Imm(1))),
        INSTRUCTION_TEST(C5ECC68C2B8000000001        , vshufps(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128), Imm(1))),
        INSTRUCTION_TEST(62917448C6C97B              , vshufps(zmm1, k0, zmm1, zmm25, Imm(123))),
        INSTRUCTION_TEST(C5E8C6CB01                  , vshufps(xmm1, xmm2, xmm3, Imm(1))),
        INSTRUCTION_TEST(62B17448C68CF0341200007B    , vshufps(zmm1, k0, zmm1, zmmword_ptr(rax, r14, 8, 4660), Imm(123))),
        INSTRUCTION_TEST(62F17448C6097B              , vshufps(zmm1, k0, zmm1, zmmword_ptr(rcx), Imm(123))),
        INSTRUCTION_TEST(62F17448C64A807B            , vshufps(zmm1, k0, zmm1, zmmword_ptr(rdx, -8192), Imm(123))),
        INSTRUCTION_TEST(62917448C6C9AB              , vshufps(zmm1, k0, zmm1, zmm25, Imm(171))),
        INSTRUCTION_TEST(62E1FD48515A7F              , vsqrtpd(zmm19, k0, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(6271FD48514A7F              , vsqrtpd(zmm9, k0, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(6271FD48518AC0DFFFFF        , vsqrtpd(zmm9, k0, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(6271FD48514A80              , vsqrtpd(zmm9, k0, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(6271FD485109                , vsqrtpd(zmm9, k0, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(6271FD4851CD                , vsqrtpd(zmm9, k0, zmm5)),
        INSTRUCTION_TEST(6231FD48518CF034120000      , vsqrtpd(zmm9, k0, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62E1FD48519A00200000        , vsqrtpd(zmm19, k0, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(6271FD48518A00200000        , vsqrtpd(zmm9, k0, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62E1FD48519AC0DFFFFF        , vsqrtpd(zmm19, k0, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62A1FD48519CF023010000      , vsqrtpd(zmm19, k0, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62E1FD485119                , vsqrtpd(zmm19, k0, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(C5F951CA                    , vsqrtpd(xmm1, xmm2)),
        INSTRUCTION_TEST(C5F9518C1A80000000          , vsqrtpd(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(C5FD51CA                    , vsqrtpd(ymm1, ymm2)),
        INSTRUCTION_TEST(62E1FD48515A80              , vsqrtpd(zmm19, k0, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62F1FD4851CA                , vsqrtpd(zmm1, k0, zmm2)),
        INSTRUCTION_TEST(62F1FD48514C1A02            , vsqrtpd(zmm1, k0, zmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62A1FD4851DB                , vsqrtpd(zmm19, k0, zmm19)),
        INSTRUCTION_TEST(C5FD518C1A80000000          , vsqrtpd(ymm1, ymmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F17C4851A200200000        , vsqrtps(zmm4, k0, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62617C4851A2C0DFFFFF        , vsqrtps(zmm28, k0, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62F17C4851A2C0DFFFFF        , vsqrtps(zmm4, k0, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62F17C48516280              , vsqrtps(zmm4, k0, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62F17C485121                , vsqrtps(zmm4, k0, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62B17C4851A4F034120000      , vsqrtps(zmm4, k0, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62B17C4851E7                , vsqrtps(zmm4, k0, zmm23)),
        INSTRUCTION_TEST(62617C4851A200200000        , vsqrtps(zmm28, k0, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62617C4851627F              , vsqrtps(zmm28, k0, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62F17C4851627F              , vsqrtps(zmm4, k0, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62617C48516280              , vsqrtps(zmm28, k0, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62217C4851A4F023010000      , vsqrtps(zmm28, k0, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62017C4851E5                , vsqrtps(zmm28, k0, zmm29)),
        INSTRUCTION_TEST(62F17C48514C1A02            , vsqrtps(zmm1, k0, zmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62F17C4851CA                , vsqrtps(zmm1, k0, zmm2)),
        INSTRUCTION_TEST(C5FC518C1A80000000          , vsqrtps(ymm1, ymmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(C5FC51CA                    , vsqrtps(ymm1, ymm2)),
        INSTRUCTION_TEST(C5F8518C1A80000000          , vsqrtps(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(C5F851CA                    , vsqrtps(xmm1, xmm2)),
        INSTRUCTION_TEST(62617C485121                , vsqrtps(zmm28, k0, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(6271BF005101                , vsqrtsd(xmm8, k0, xmm24, qword_ptr(rcx))),
        INSTRUCTION_TEST(6271BF0051C5                , vsqrtsd(xmm8, k0, xmm24, xmm5)),
        INSTRUCTION_TEST(6271BF00514280              , vsqrtsd(xmm8, k0, xmm24, qword_ptr(rdx, -1024))),
        INSTRUCTION_TEST(6231BF005184F034120000      , vsqrtsd(xmm8, k0, xmm24, qword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(6271BF0051427F              , vsqrtsd(xmm8, k0, xmm24, qword_ptr(rdx, 1016))),
        INSTRUCTION_TEST(6271BF00518200040000        , vsqrtsd(xmm8, k0, xmm24, qword_ptr(rdx, 1024))),
        INSTRUCTION_TEST(6271BF005182F8FBFFFF        , vsqrtsd(xmm8, k0, xmm24, qword_ptr(rdx, -1032))),
        INSTRUCTION_TEST(C4C16B51F4                  , vsqrtsd(xmm6, xmm2, xmm12)),
        INSTRUCTION_TEST(C5EB51B200FCFFFF            , vsqrtsd(xmm6, xmm2, qword_ptr(rdx, -1024))),
        INSTRUCTION_TEST(C5EB51B2F8030000            , vsqrtsd(xmm6, xmm2, qword_ptr(rdx, 1016))),
        INSTRUCTION_TEST(C5EB51B2F8FBFFFF            , vsqrtsd(xmm6, xmm2, qword_ptr(rdx, -1032))),
        INSTRUCTION_TEST(C5EB5131                    , vsqrtsd(xmm6, xmm2, qword_ptr(rcx))),
        INSTRUCTION_TEST(C4A16B51B4F023010000        , vsqrtsd(xmm6, xmm2, qword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(C5EB51CB                    , vsqrtsd(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(C5EB518C2B80000000          , vsqrtsd(xmm1, xmm2, qword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C5EB51B200040000            , vsqrtsd(xmm6, xmm2, qword_ptr(rdx, 1024))),
        INSTRUCTION_TEST(62410E0851CB                , vsqrtss(xmm25, k0, xmm14, xmm11)),
        INSTRUCTION_TEST(62610E08518A00020000        , vsqrtss(xmm25, k0, xmm14, dword_ptr(rdx, 512))),
        INSTRUCTION_TEST(62610E08514A7F              , vsqrtss(xmm25, k0, xmm14, dword_ptr(rdx, 508))),
        INSTRUCTION_TEST(62610E08518AFCFDFFFF        , vsqrtss(xmm25, k0, xmm14, dword_ptr(rdx, -516))),
        INSTRUCTION_TEST(62610E08514A80              , vsqrtss(xmm25, k0, xmm14, dword_ptr(rdx, -512))),
        INSTRUCTION_TEST(62610E085109                , vsqrtss(xmm25, k0, xmm14, dword_ptr(rcx))),
        INSTRUCTION_TEST(62C1660051F0                , vsqrtss(xmm22, k0, xmm19, xmm8)),
        INSTRUCTION_TEST(62210E08518CF034120000      , vsqrtss(xmm25, k0, xmm14, dword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62E1660051727F              , vsqrtss(xmm22, k0, xmm19, dword_ptr(rdx, 508))),
        INSTRUCTION_TEST(62E1660051B200020000        , vsqrtss(xmm22, k0, xmm19, dword_ptr(rdx, 512))),
        INSTRUCTION_TEST(62E1660051B2FCFDFFFF        , vsqrtss(xmm22, k0, xmm19, dword_ptr(rdx, -516))),
        INSTRUCTION_TEST(62E16600517280              , vsqrtss(xmm22, k0, xmm19, dword_ptr(rdx, -512))),
        INSTRUCTION_TEST(62E166005131                , vsqrtss(xmm22, k0, xmm19, dword_ptr(rcx))),
        INSTRUCTION_TEST(62A1660051B4F023010000      , vsqrtss(xmm22, k0, xmm19, dword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(C5EA51CB                    , vsqrtss(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(C5EA518C2B80000000          , vsqrtss(xmm1, xmm2, dword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C5F8AE9C1180000000          , vstmxcsr(dword_ptr(rcx, rdx, 1, 128))),
        INSTRUCTION_TEST(62719D485C8A00200000        , vsubpd(zmm9, k0, zmm12, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62719D485C4A7F              , vsubpd(zmm9, k0, zmm12, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62719D485C8AC0DFFFFF        , vsubpd(zmm9, k0, zmm12, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62719D485C4A80              , vsubpd(zmm9, k0, zmm12, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62719D485C09                , vsubpd(zmm9, k0, zmm12, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62319D485C8CF023010000      , vsubpd(zmm9, k0, zmm12, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62519D485CC9                , vsubpd(zmm9, k0, zmm12, zmm9)),
        INSTRUCTION_TEST(62E19D405CAA00200000        , vsubpd(zmm21, k0, zmm28, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62E19D405CAAC0DFFFFF        , vsubpd(zmm21, k0, zmm28, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62E19D405C6A7F              , vsubpd(zmm21, k0, zmm28, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62E19D405C29                , vsubpd(zmm21, k0, zmm28, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62E19D405C6A80              , vsubpd(zmm21, k0, zmm28, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(C5E95C8C2B80000000          , vsubpd(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C5ED5CCB                    , vsubpd(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(C5ED5C8C2B80000000          , vsubpd(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C5E95CCB                    , vsubpd(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(62F1ED485C4C2B02            , vsubpd(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62C19D405CE8                , vsubpd(zmm21, k0, zmm28, zmm8)),
        INSTRUCTION_TEST(62A19D405CACF034120000      , vsubpd(zmm21, k0, zmm28, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62F1ED485CCB                , vsubpd(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(62E12C485C9A00200000        , vsubps(zmm19, k0, zmm10, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62E12C485C5A7F              , vsubps(zmm19, k0, zmm10, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62E12C485C9AC0DFFFFF        , vsubps(zmm19, k0, zmm10, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62E12C485C5A80              , vsubps(zmm19, k0, zmm10, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62E12C485C19                , vsubps(zmm19, k0, zmm10, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62A12C485C9CF034120000      , vsubps(zmm19, k0, zmm10, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62E12C485CDD                , vsubps(zmm19, k0, zmm10, zmm5)),
        INSTRUCTION_TEST(627124405CB200200000        , vsubps(zmm14, k0, zmm27, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(627124405C727F              , vsubps(zmm14, k0, zmm27, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(627124405CB2C0DFFFFF        , vsubps(zmm14, k0, zmm27, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(627124405C31                , vsubps(zmm14, k0, zmm27, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(623124405CB4F023010000      , vsubps(zmm14, k0, zmm27, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(623124405CF5                , vsubps(zmm14, k0, zmm27, zmm21)),
        INSTRUCTION_TEST(62F16C485C4C2B02            , vsubps(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F16C485CCB                , vsubps(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(C5EC5C8C2B80000000          , vsubps(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C5EC5CCB                    , vsubps(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(C5E85C8C2B80000000          , vsubps(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C5E85CCB                    , vsubps(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(627124405C7280              , vsubps(zmm14, k0, zmm27, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(6261C7005C01                , vsubsd(xmm24, k0, xmm23, qword_ptr(rcx))),
        INSTRUCTION_TEST(6241C7005CC5                , vsubsd(xmm24, k0, xmm23, xmm13)),
        INSTRUCTION_TEST(6261C7005C8200040000        , vsubsd(xmm24, k0, xmm23, qword_ptr(rdx, 1024))),
        INSTRUCTION_TEST(6261C7005C427F              , vsubsd(xmm24, k0, xmm23, qword_ptr(rdx, 1016))),
        INSTRUCTION_TEST(6261C7005C82F8FBFFFF        , vsubsd(xmm24, k0, xmm23, qword_ptr(rdx, -1032))),
        INSTRUCTION_TEST(6261C7005C4280              , vsubsd(xmm24, k0, xmm23, qword_ptr(rdx, -1024))),
        INSTRUCTION_TEST(6221C7005C84F034120000      , vsubsd(xmm24, k0, xmm23, qword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62C1A7005CE7                , vsubsd(xmm20, k0, xmm27, xmm15)),
        INSTRUCTION_TEST(62E1A7005C627F              , vsubsd(xmm20, k0, xmm27, qword_ptr(rdx, 1016))),
        INSTRUCTION_TEST(62E1A7005CA2F8FBFFFF        , vsubsd(xmm20, k0, xmm27, qword_ptr(rdx, -1032))),
        INSTRUCTION_TEST(62E1A7005C6280              , vsubsd(xmm20, k0, xmm27, qword_ptr(rdx, -1024))),
        INSTRUCTION_TEST(62E1A7005C21                , vsubsd(xmm20, k0, xmm27, qword_ptr(rcx))),
        INSTRUCTION_TEST(62A1A7005CA4F023010000      , vsubsd(xmm20, k0, xmm27, qword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(C5EB5CCB                    , vsubsd(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(C5EB5C8C2B80000000          , vsubsd(xmm1, xmm2, qword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62E1A7005CA200040000        , vsubsd(xmm20, k0, xmm27, qword_ptr(rdx, 1024))),
        INSTRUCTION_TEST(62F136005C6A80              , vsubss(xmm5, k0, xmm25, dword_ptr(rdx, -512))),
        INSTRUCTION_TEST(62D136005CE9                , vsubss(xmm5, k0, xmm25, xmm9)),
        INSTRUCTION_TEST(62F136005CAA00020000        , vsubss(xmm5, k0, xmm25, dword_ptr(rdx, 512))),
        INSTRUCTION_TEST(62F136005C6A7F              , vsubss(xmm5, k0, xmm25, dword_ptr(rdx, 508))),
        INSTRUCTION_TEST(62F136005CAAFCFDFFFF        , vsubss(xmm5, k0, xmm25, dword_ptr(rdx, -516))),
        INSTRUCTION_TEST(62F136005C29                , vsubss(xmm5, k0, xmm25, dword_ptr(rcx))),
        INSTRUCTION_TEST(62B136005CACF023010000      , vsubss(xmm5, k0, xmm25, dword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62215E005CD8                , vsubss(xmm27, k0, xmm20, xmm16)),
        INSTRUCTION_TEST(62615E005C5A7F              , vsubss(xmm27, k0, xmm20, dword_ptr(rdx, 508))),
        INSTRUCTION_TEST(62615E005C9AFCFDFFFF        , vsubss(xmm27, k0, xmm20, dword_ptr(rdx, -516))),
        INSTRUCTION_TEST(62615E005C5A80              , vsubss(xmm27, k0, xmm20, dword_ptr(rdx, -512))),
        INSTRUCTION_TEST(62615E005C19                , vsubss(xmm27, k0, xmm20, dword_ptr(rcx))),
        INSTRUCTION_TEST(62215E005C9CF034120000      , vsubss(xmm27, k0, xmm20, dword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(C5EA5CCB                    , vsubss(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(C5EA5C8C2B80000000          , vsubss(xmm1, xmm2, dword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62615E005C9A00020000        , vsubss(xmm27, k0, xmm20, dword_ptr(rdx, 512))),
        INSTRUCTION_TEST(C4E27D0F8C1A80000000        , vtestpd(ymm1, ymmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(C4E27D0FCA                  , vtestpd(ymm1, ymm2)),
        INSTRUCTION_TEST(C4E2790FCA                  , vtestpd(xmm1, xmm2)),
        INSTRUCTION_TEST(C4E2790F8C1A80000000        , vtestpd(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(C4E2790ECA                  , vtestps(xmm1, xmm2)),
        INSTRUCTION_TEST(C4E2790E8C1A80000000        , vtestps(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(C4E27D0ECA                  , vtestps(ymm1, ymm2)),
        INSTRUCTION_TEST(C4E27D0E8C1A80000000        , vtestps(ymm1, ymmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62E1FD082E6A80              , vucomisd(xmm21, qword_ptr(rdx, -1024))),
        INSTRUCTION_TEST(62E1FD082EAAF8FBFFFF        , vucomisd(xmm21, qword_ptr(rdx, -1032))),
        INSTRUCTION_TEST(C5792E9A00FCFFFF            , vucomisd(xmm11, qword_ptr(rdx, -1024))),
        INSTRUCTION_TEST(62E1FD082EAA00040000        , vucomisd(xmm21, qword_ptr(rdx, 1024))),
        INSTRUCTION_TEST(62A1FD082EEB                , vucomisd(xmm21, xmm19)),
        INSTRUCTION_TEST(62E1FD082E29                , vucomisd(xmm21, qword_ptr(rcx))),
        INSTRUCTION_TEST(62E1FD082E6A7F              , vucomisd(xmm21, qword_ptr(rdx, 1016))),
        INSTRUCTION_TEST(62A1FD082EACF034120000      , vucomisd(xmm21, qword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(C5F92ECA                    , vucomisd(xmm1, xmm2)),
        INSTRUCTION_TEST(C5792E9A00040000            , vucomisd(xmm11, qword_ptr(rdx, 1024))),
        INSTRUCTION_TEST(C5792E9AF8030000            , vucomisd(xmm11, qword_ptr(rdx, 1016))),
        INSTRUCTION_TEST(C5792E9AF8FBFFFF            , vucomisd(xmm11, qword_ptr(rdx, -1032))),
        INSTRUCTION_TEST(C5792E19                    , vucomisd(xmm11, qword_ptr(rcx))),
        INSTRUCTION_TEST(C421792E9CF023010000        , vucomisd(xmm11, qword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(C5F92E8C1A80000000          , vucomisd(xmm1, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(C441792EDA                  , vucomisd(xmm11, xmm10)),
        INSTRUCTION_TEST(62017C082EDD                , vucomiss(xmm27, xmm29)),
        INSTRUCTION_TEST(62617C082E9A00020000        , vucomiss(xmm27, dword_ptr(rdx, 512))),
        INSTRUCTION_TEST(62617C082E5A7F              , vucomiss(xmm27, dword_ptr(rdx, 508))),
        INSTRUCTION_TEST(62617C082E9AFCFDFFFF        , vucomiss(xmm27, dword_ptr(rdx, -516))),
        INSTRUCTION_TEST(62617C082E5A80              , vucomiss(xmm27, dword_ptr(rdx, -512))),
        INSTRUCTION_TEST(62617C082E19                , vucomiss(xmm27, dword_ptr(rcx))),
        INSTRUCTION_TEST(62217C082E9CF034120000      , vucomiss(xmm27, dword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62C17C082EF3                , vucomiss(xmm22, xmm11)),
        INSTRUCTION_TEST(62E17C082EB2FCFDFFFF        , vucomiss(xmm22, dword_ptr(rdx, -516))),
        INSTRUCTION_TEST(62E17C082E727F              , vucomiss(xmm22, dword_ptr(rdx, 508))),
        INSTRUCTION_TEST(62E17C082E7280              , vucomiss(xmm22, dword_ptr(rdx, -512))),
        INSTRUCTION_TEST(62E17C082E31                , vucomiss(xmm22, dword_ptr(rcx))),
        INSTRUCTION_TEST(62A17C082EB4F023010000      , vucomiss(xmm22, dword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(C5F82ECA                    , vucomiss(xmm1, xmm2)),
        INSTRUCTION_TEST(C5F82E8C1A80000000          , vucomiss(xmm1, dword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(62E17C082EB200020000        , vucomiss(xmm22, dword_ptr(rdx, 512))),
        INSTRUCTION_TEST(62D1C54015F4                , vunpckhpd(zmm6, k0, zmm23, zmm12)),
        INSTRUCTION_TEST(6261E540158A00200000        , vunpckhpd(zmm25, k0, zmm19, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62B1C54015B4F034120000      , vunpckhpd(zmm6, k0, zmm23, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(6261E540154A7F              , vunpckhpd(zmm25, k0, zmm19, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62F1C540157280              , vunpckhpd(zmm6, k0, zmm23, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62F1C54015B2C0DFFFFF        , vunpckhpd(zmm6, k0, zmm23, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62F1C54015727F              , vunpckhpd(zmm6, k0, zmm23, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62F1C5401531                , vunpckhpd(zmm6, k0, zmm23, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(6261E540158AC0DFFFFF        , vunpckhpd(zmm25, k0, zmm19, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62F1C54015B200200000        , vunpckhpd(zmm6, k0, zmm23, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(6261E5401509                , vunpckhpd(zmm25, k0, zmm19, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(C5E915CB                    , vunpckhpd(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(C5E9158C2B80000000          , vunpckhpd(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C5ED15CB                    , vunpckhpd(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(C5ED158C2B80000000          , vunpckhpd(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(6261E540154A80              , vunpckhpd(zmm25, k0, zmm19, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62F1ED48154C2B02            , vunpckhpd(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(6201E54015CA                , vunpckhpd(zmm25, k0, zmm19, zmm26)),
        INSTRUCTION_TEST(6221E540158CF023010000      , vunpckhpd(zmm25, k0, zmm19, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62F1ED4815CB                , vunpckhpd(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(62616C48156A7F              , vunpckhps(zmm29, k0, zmm2, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62F10C4815AA00200000        , vunpckhps(zmm5, k0, zmm14, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62F10C48156A7F              , vunpckhps(zmm5, k0, zmm14, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62F10C4815AAC0DFFFFF        , vunpckhps(zmm5, k0, zmm14, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62F10C48156A80              , vunpckhps(zmm5, k0, zmm14, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62F10C481529                , vunpckhps(zmm5, k0, zmm14, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62B10C4815ACF023010000      , vunpckhps(zmm5, k0, zmm14, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62B10C4815E8                , vunpckhps(zmm5, k0, zmm14, zmm16)),
        INSTRUCTION_TEST(62616C4815AA00200000        , vunpckhps(zmm29, k0, zmm2, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62616C4815AAC0DFFFFF        , vunpckhps(zmm29, k0, zmm2, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62616C48156A80              , vunpckhps(zmm29, k0, zmm2, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62616C481529                , vunpckhps(zmm29, k0, zmm2, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(C5E8158C2B80000000          , vunpckhps(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C5EC15CB                    , vunpckhps(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(C5EC158C2B80000000          , vunpckhps(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C5E815CB                    , vunpckhps(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(62F16C48154C2B02            , vunpckhps(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62616C4815EC                , vunpckhps(zmm29, k0, zmm2, zmm4)),
        INSTRUCTION_TEST(62216C4815ACF034120000      , vunpckhps(zmm29, k0, zmm2, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62F16C4815CB                , vunpckhps(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(62E19D40149200200000        , vunpcklpd(zmm18, k0, zmm28, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62E19540149200200000        , vunpcklpd(zmm18, k0, zmm29, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62E1954014527F              , vunpcklpd(zmm18, k0, zmm29, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62E195401492C0DFFFFF        , vunpcklpd(zmm18, k0, zmm29, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62E19540145280              , vunpcklpd(zmm18, k0, zmm29, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62E195401411                , vunpcklpd(zmm18, k0, zmm29, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62A195401494F023010000      , vunpcklpd(zmm18, k0, zmm29, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62A1954014D5                , vunpcklpd(zmm18, k0, zmm29, zmm21)),
        INSTRUCTION_TEST(62E19D4014527F              , vunpcklpd(zmm18, k0, zmm28, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62E19D401492C0DFFFFF        , vunpcklpd(zmm18, k0, zmm28, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(C5E914CB                    , vunpcklpd(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(C5E9148C2B80000000          , vunpcklpd(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C5ED14CB                    , vunpcklpd(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(C5ED148C2B80000000          , vunpcklpd(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F1ED4814CB                , vunpcklpd(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(62E19D40145280              , vunpcklpd(zmm18, k0, zmm28, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62C19D4014D0                , vunpcklpd(zmm18, k0, zmm28, zmm8)),
        INSTRUCTION_TEST(62A19D401494F034120000      , vunpcklpd(zmm18, k0, zmm28, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62E19D401411                , vunpcklpd(zmm18, k0, zmm28, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62F1ED48144C2B02            , vunpcklpd(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F16448144A7F              , vunpcklps(zmm1, k0, zmm3, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62610C48148A00200000        , vunpcklps(zmm25, k0, zmm14, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62610C48144A7F              , vunpcklps(zmm25, k0, zmm14, zmmword_ptr(rdx, 8128))),
        INSTRUCTION_TEST(62610C48148AC0DFFFFF        , vunpcklps(zmm25, k0, zmm14, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62610C48144A80              , vunpcklps(zmm25, k0, zmm14, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(62610C481409                , vunpcklps(zmm25, k0, zmm14, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(62210C48148CF034120000      , vunpcklps(zmm25, k0, zmm14, zmmword_ptr(rax, r14, 8, 4660))),
        INSTRUCTION_TEST(62610C4814CE                , vunpcklps(zmm25, k0, zmm14, zmm6)),
        INSTRUCTION_TEST(62F16448148A00200000        , vunpcklps(zmm1, k0, zmm3, zmmword_ptr(rdx, 8192))),
        INSTRUCTION_TEST(62F16448148AC0DFFFFF        , vunpcklps(zmm1, k0, zmm3, zmmword_ptr(rdx, -8256))),
        INSTRUCTION_TEST(62F164481409                , vunpcklps(zmm1, k0, zmm3, zmmword_ptr(rcx))),
        INSTRUCTION_TEST(C5E814CB                    , vunpcklps(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(62B16448148CF023010000      , vunpcklps(zmm1, k0, zmm3, zmmword_ptr(rax, r14, 8, 291))),
        INSTRUCTION_TEST(62F1644814CA                , vunpcklps(zmm1, k0, zmm3, zmm2)),
        INSTRUCTION_TEST(62F16C48144C2B02            , vunpcklps(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F16C4814CB                , vunpcklps(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(C5EC148C2B80000000          , vunpcklps(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C5EC14CB                    , vunpcklps(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(C5E8148C2B80000000          , vunpcklps(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F16448144A80              , vunpcklps(zmm1, k0, zmm3, zmmword_ptr(rdx, -8192))),
        INSTRUCTION_TEST(C5E957CB                    , vxorpd(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(C5E9578C2B80000000          , vxorpd(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C5ED57CB                    , vxorpd(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(C5ED578C2B80000000          , vxorpd(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F1ED4857CB                , vxorpd(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(62F1ED48574C2B02            , vxorpd(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C5EC578C2B80000000          , vxorps(ymm1, ymm2, ymmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C5EC57CB                    , vxorps(ymm1, ymm2, ymm3)),
        INSTRUCTION_TEST(62F16C4857CB                , vxorps(zmm1, k0, zmm2, zmm3)),
        INSTRUCTION_TEST(C5E857CB                    , vxorps(xmm1, xmm2, xmm3)),
        INSTRUCTION_TEST(C5E8578C2B80000000          , vxorps(xmm1, xmm2, xmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(62F16C48574C2B02            , vxorps(zmm1, k0, zmm2, zmmword_ptr(rbx, rbp, 1, 128))),
        INSTRUCTION_TEST(C5FC77                      , vzeroall()),
        INSTRUCTION_TEST(C5F877                      , vzeroupper()),
        INSTRUCTION_TEST(0F09                        , wbinvd()),
        INSTRUCTION_TEST(F30FAED1                    , wrfsbase(ecx)),
        INSTRUCTION_TEST(F3480FAED1                  , wrfsbase(rcx)),
        INSTRUCTION_TEST(F30FAED9                    , wrgsbase(ecx)),
        INSTRUCTION_TEST(F3480FAED9                  , wrgsbase(rcx)),
        INSTRUCTION_TEST(0F30                        , wrmsr()),
        INSTRUCTION_TEST(0F38F69C1180000000          , wrssd(dword_ptr(rcx, rdx, 1, 128), ebx)),
        INSTRUCTION_TEST(480F38F69C1180000000        , wrssq(qword_ptr(rcx, rdx, 1, 128), rbx)),
        INSTRUCTION_TEST(660F38F59C1180000000        , wrussd(dword_ptr(rcx, rdx, 1, 128), ebx)),
        INSTRUCTION_TEST(66480F38F59C1180000000      , wrussq(qword_ptr(rcx, rdx, 1, 128), rbx)),
        INSTRUCTION_TEST(C6F801                      , xabort(1)),
        INSTRUCTION_TEST(660FC19C1180000000          , xadd(word_ptr(rcx, rdx, 1, 128), bx)),
        INSTRUCTION_TEST(480FC1D1                    , xadd(rcx, rdx)),
        INSTRUCTION_TEST(480FC19C1180000000          , xadd(qword_ptr(rcx, rdx, 1, 128), rbx)),
        INSTRUCTION_TEST(0FC1D1                      , xadd(ecx, edx)),
        INSTRUCTION_TEST(0FC19C1180000000            , xadd(dword_ptr(rcx, rdx, 1, 128), ebx)),
        INSTRUCTION_TEST(660FC1D1                    , xadd(cx, dx)),
        INSTRUCTION_TEST(0FC0D5                      , xadd(ch, dl)),
        INSTRUCTION_TEST(0FC0F1                      , xadd(cl, dh)),
        INSTRUCTION_TEST(0FC0F5                      , xadd(ch, dh)),
        INSTRUCTION_TEST(0FC09C1180000000            , xadd(byte_ptr(rcx, rdx, 1, 128), bl)),
        INSTRUCTION_TEST(0FC0BC1180000000            , xadd(byte_ptr(rcx, rdx, 1, 128), bh)),
        INSTRUCTION_TEST(0FC0D1                      , xadd(cl, dl)),
        INSTRUCTION_TEST(66878C1A80000000            , xchg(cx, word_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(66879C1180000000            , xchg(word_ptr(rcx, rdx, 1, 128), bx)),
        INSTRUCTION_TEST(4887D1                      , xchg(rcx, rdx)),
        INSTRUCTION_TEST(48878C1A80000000            , xchg(rcx, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(48879C1180000000            , xchg(qword_ptr(rcx, rdx, 1, 128), rbx)),
        INSTRUCTION_TEST(878C1A80000000              , xchg(ecx, dword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(879C1180000000              , xchg(dword_ptr(rcx, rdx, 1, 128), ebx)),
        INSTRUCTION_TEST(6687D1                      , xchg(cx, dx)),
        INSTRUCTION_TEST(87D1                        , xchg(ecx, edx)),
        INSTRUCTION_TEST(86F1                        , xchg(cl, dh)),
        INSTRUCTION_TEST(86BC1180000000              , xchg(byte_ptr(rcx, rdx, 1, 128), bh)),
        INSTRUCTION_TEST(869C1180000000              , xchg(byte_ptr(rcx, rdx, 1, 128), bl)),
        INSTRUCTION_TEST(86AC1A80000000              , xchg(ch, byte_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(86F5                        , xchg(ch, dh)),
        INSTRUCTION_TEST(86D5                        , xchg(ch, dl)),
        INSTRUCTION_TEST(868C1A80000000              , xchg(cl, byte_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(86D1                        , xchg(cl, dl)),
        INSTRUCTION_TEST(0F01D5                      , xend()),
        INSTRUCTION_TEST(0F01D0                      , xgetbv()),
        INSTRUCTION_TEST(83F101                      , xor_(ecx, Imm(1))),
        INSTRUCTION_TEST(83B4118000000001            , xor_(dword_ptr(rcx, rdx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(338C1A80000000              , xor_(ecx, dword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(31D1                        , xor_(ecx, edx)),
        INSTRUCTION_TEST(4883B4118000000001          , xor_(qword_ptr(rcx, rdx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(66319C1180000000            , xor_(word_ptr(rcx, rdx, 1, 128), bx)),
        INSTRUCTION_TEST(4883F101                    , xor_(rcx, Imm(1))),
        INSTRUCTION_TEST(48338C1A80000000            , xor_(rcx, qword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(4831D1                      , xor_(rcx, rdx)),
        INSTRUCTION_TEST(319C1180000000              , xor_(dword_ptr(rcx, rdx, 1, 128), ebx)),
        INSTRUCTION_TEST(6683B4118000000001          , xor_(word_ptr(rcx, rdx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(48319C1180000000            , xor_(qword_ptr(rcx, rdx, 1, 128), rbx)),
        INSTRUCTION_TEST(66338C1A80000000            , xor_(cx, word_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(80F101                      , xor_(cl, Imm(1))),
        INSTRUCTION_TEST(6631D1                      , xor_(cx, dx)),
        INSTRUCTION_TEST(30BC1180000000              , xor_(byte_ptr(rcx, rdx, 1, 128), bh)),
        INSTRUCTION_TEST(6683F101                    , xor_(cx, Imm(1))),
        INSTRUCTION_TEST(80B4118000000001            , xor_(byte_ptr(rcx, rdx, 1, 128), Imm(1))),
        INSTRUCTION_TEST(32AC1A80000000              , xor_(ch, byte_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(30F5                        , xor_(ch, dh)),
        INSTRUCTION_TEST(309C1180000000              , xor_(byte_ptr(rcx, rdx, 1, 128), bl)),
        INSTRUCTION_TEST(80F501                      , xor_(ch, Imm(1))),
        INSTRUCTION_TEST(328C1A80000000              , xor_(cl, byte_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(30F1                        , xor_(cl, dh)),
        INSTRUCTION_TEST(30D1                        , xor_(cl, dl)),
        INSTRUCTION_TEST(30D5                        , xor_(ch, dl)),
        INSTRUCTION_TEST(660F57CA                    , xorpd(xmm1, xmm2)),
        INSTRUCTION_TEST(660F578C1A80000000          , xorpd(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(0F57CA                      , xorps(xmm1, xmm2)),
        INSTRUCTION_TEST(0F578C1A80000000            , xorps(xmm1, xmmword_ptr(rdx, rbx, 1, 128))),
        INSTRUCTION_TEST(F20F01E9                    , xresldtrk()),
        INSTRUCTION_TEST(0F01D1                      , xsetbv()),
        INSTRUCTION_TEST(F20F01E8                    , xsusldtrk()),
        INSTRUCTION_TEST(0F01D6                      , xtest()),
        INSTRUCTION_TEST(F3AB                        , rep().stosd()),
        INSTRUCTION_TEST(F3AD                        , rep().lodsd()),
    };
    // clang-format on

} // namespace zasm::tests::data
