## 1 概述

提供一个对象来顺序访问聚合对象中的一系列数据，而不暴露聚合对象的内部表示。

迭代器模式主要包含以下角色：

- 抽象聚合（Aggregate）角色：定义存储、添加、删除聚合元素以及创建迭代器对象的接口。
- 具体聚合（ConcreteAggregate）角色：实现抽象聚合类，返回一个具体迭代器的实例。
- 抽象迭代器（Iterator）角色：定义访问和遍历聚合元素的接口，通常包含 hasNext()、next() 等方法。
- 具体迭代器（Concretelterator）角色：实现抽象迭代器接口中所定义的方法，完成对聚合对象的遍历，记录遍历的当前位置。

## 2 例子

定义一个可以存储学生对象的容器对象，将遍历该容器的功能交由迭代器实现。

```java
public class Student {
    private String name;

    public Student(String name) {
        this.name = name;
    }

    public String getName() {
        return name;
    }

    public void setName(String name) {
        this.name = name;
    }

    @Override
    public String toString() {
        return "Student{" +
                "name='" + name + '\'' +
                '}';
    }
}

//迭代器接口
public interface StudentIterator {
    boolean hasNext();

    Student next();
}

//具体的迭代器类
public class StudentIteratorImpl implements StudentIterator {
    private List<Student> list;
    private int position = 0;

    public StudentIteratorImpl(List<Student> list) {
        this.list = list;
    }

    @Override
    public boolean hasNext() {
        return position < list.size();
    }

    @Override
    public Student next() {
        Student currentStudent = list.get(position);
        position++;
        return currentStudent;
    }
}

//定义抽象容器类，包含添加元素，删除元素，获取迭代器对象的方法
public interface StudentAggregate {
    void addStudent(Student student);

    void removeStudent(Student student);

    StudentIterator getStudentIterator();
}

//具体的容器类
public class StudentAggregateImpl implements StudentAggregate {
    private List<Student> list = new ArrayList<>();  // 学生列表

    @Override
    public void addStudent(Student student) {
        this.list.add(student);
    }

    @Override
    public void removeStudent(Student student) {
        this.list.remove(student);
    }

    @Override
    public StudentIterator getStudentIterator() {
        return new StudentIteratorImpl(list);
    }
}

public class TestClient {
    public static void main(String[] args) {
        final StudentAggregateImpl aggregate = new StudentAggregateImpl();
        aggregate.addStudent(new Student("AA"));
        aggregate.addStudent(new Student("BB"));

        final StudentIterator iterator = aggregate.getStudentIterator();
        while (iterator.hasNext()) {
            System.out.println(iterator.next().toString());
        }
    }
}
```

## 3 优缺点

优点：

* 它支持以不同的方式**遍历一个聚合对象**。 
* 迭代器简化了聚合类。 
* 在同一个聚合上可以有多个遍历。 
* 在迭代器模式中，增加新的聚合类和迭代器类都很方便，无须修改原有代码。

缺点：

* 由于迭代器模式将存储数据和遍历数据的职责分离，增加新的聚合类需要对应增加新的迭代器类，类的个数成对增加，这在一定程度上增加了系统的复杂性。