## 1 概述

将一个请求封装为一个对象，使发出请求的责任和执行请求的责任分割开。这样两者之间通过命令对象进行沟通，这样方便将命令对象进行存储、传递、调用、增加与管理。

命令模式包含以下主要角色：

- 抽象命令类（Command）角色： 定义命令的接口，声明执行的方法。
- 具体命令（Concrete Command）角色：具体的命令，实现命令接口；通常会持有接收者，并调用接收者的功能来完成命令要执行的操作。
- 实现者/接收者（Receiver）角色： 接收者，真正执行命令的对象。任何类都可能成为一个接收者，只要它能够实现命令要求实现的相应功能。
- 调用者/请求者（Invoker）角色： 要求命令对象执行请求，通常会持有命令对象，可以持有很多的命令对象。这个是客户端真正触发命令并要求命令执行相应操作的地方，也就是说相当于使用命令对象的入口。

## 2 例子

```java
//接收者：真正执行命令的对象
public class Receiver {
    public void action(){
        System.out.println("命令执行了.......");
    }
}

//抽象命令类：抽象的命令，可以根据不同类型的命令写出不同的实现
public interface ICommand {
    //调用命令
    void execute();
}

//具体命令类
class ConcreteCommand implements ICommand{
    private Receiver receiver;//持有真正执行命令对象的引用
    public ConcreteCommand(Receiver receiver) {
        super();
        this.receiver = receiver;
    }
    @Override
    public void execute() {
        //调用接收者执行命令的方法
        receiver.action();
    }
}

//请求者/调用者：发起执行命令请求的对象
public class Invoker {
    private ICommand command;//持有命令对象的引用

    public Invoker(ICommand command) {
        super();
        this.command = command;
    }

    public void call() {
        //请求者调用命令对象执行命令的那个execute方法
        command.execute();
    }
}

public class TestClient {
    public static void main(String[] args) {
        //通过请求者（invoker）调用命令对象（command），命令对象中调用了命令具体执行者（Receiver）
        ICommand command = new ConcreteCommand(new Receiver());
        Invoker invoker = new Invoker(command);
        invoker.call();
    }
}
```

## 3 优缺点

优点：

- 降低系统的耦合度。命令模式能将调用操作的对象与实现该操作的对象解耦。
- 增加或删除命令非常方便。采用命令模式增加与删除命令不会影响其他类，它满足“开闭原则”，对扩展比较灵活。
- 可以实现宏命令。命令模式可以与组合模式结合，将多个命令装配成一个组合命令，即宏命令。
- 方便实现 Undo 和 Redo 操作。命令模式可以与后面介绍的备忘录模式结合，实现命令的撤销与恢复。

缺点：

- 使用命令模式可能会导致某些系统有过多的具体命令类。
- 系统结构更加复杂。