import torch
import torch.utils.data


class TextAudioSpeakerCollate(object):
    """ Zero-pads model inputs and targets
    """

    def __init__(self, return_ids=False):
        self.return_ids = return_ids

    def __call__(self, batch):
        """Collate's training batch from normalized text, audio and speaker identities
        PARAMS
        ------
        batch: [text_normalized, spec_normalized, wav_normalized, sid]
        """
        # Right zero-pad all one-hot text sequences to max input length
        _, ids_sorted_decreasing = torch.sort(
            torch.tensor([x[1].size(1) for x in batch], dtype=torch.long),
            dim=0, descending=True)

        max_text_len = max([len(x[0]) for x in batch])
        max_spec_len = max([x[1].size(1) for x in batch])
        max_wav_len = max([x[2].size(1) for x in batch])

        text_lengths = torch.zeros(len(batch), dtype=torch.long)
        spec_lengths = torch.zeros(len(batch), dtype=torch.long)
        wav_lengths = torch.zeros(len(batch), dtype=torch.long)
        sid = torch.zeros(len(batch), dtype=torch.long)

        text_padded = torch.zeros([len(batch), max_text_len], dtype=torch.long)
        spec_padded = torch.zeros([len(batch), batch[0][1].size(0), max_spec_len], dtype=torch.float32)
        wav_padded = torch.zeros([len(batch), 1, max_wav_len], dtype=torch.float32)
        text_padded.zero_()
        spec_padded.zero_()
        wav_padded.zero_()
        for i in range(len(ids_sorted_decreasing)):
            row = batch[ids_sorted_decreasing[i]]

            text = row[0]
            text_padded[i, :text.size(0)] = text
            text_lengths[i] = text.size(0)

            spec = row[1]
            spec_padded[i, :, :spec.size(1)] = spec
            spec_lengths[i] = spec.size(1)

            wav = row[2]
            wav_padded[i, :, :wav.size(1)] = wav
            wav_lengths[i] = wav.size(1)

            sid[i] = row[3]

        if self.return_ids:
            return text_padded, text_lengths, spec_padded, spec_lengths, wav_padded, wav_lengths, sid, ids_sorted_decreasing
        return text_padded, text_lengths, spec_padded, spec_lengths, wav_padded, wav_lengths, sid
