#!/usr/bin/env python3

# Copyright (c) 2021 Alethea Katherine Flowers.
# Published under the standard MIT License.
# Full text available at: https://opensource.org/licenses/MIT

"""Generates the lookup table at ../src/generated/gem_ramp_table_data.c using the
reference calibration from ../../factory/libgemini."""

import argparse
import pathlib
import sys
import textwrap

factory_path = pathlib.Path(pathlib.Path(__file__).parent, "../../factory")
sys.path.insert(1, str(factory_path.resolve()))

from libgemini import oscillators, reference_calibration  # noqa


def main(output_file):
    with output_file.open("w") as fh:
        fh.write(
            textwrap.dedent(
                """\
        /* This file is generated by scripts/generate_ramp_table_data.py. Do not edit directly. */
        /* clang-format off */

        #include "gem_ramp_table.h"
        #include "fix16.h"

        struct GemRampTableEntry gem_ramp_table[] = {
        """
            )
        )

        for note in oscillators.calibration_note_range():
            voltage = oscillators.midi_note_to_voltage(note)
            frequency = oscillators.midi_note_to_frequency(note)
            period_reg = oscillators.frequency_to_timer_period(frequency)
            dac_code_castor = reference_calibration.castor[period_reg]
            dac_code_pollux = reference_calibration.pollux[period_reg]
            fh.write(
                f"  {{.pitch_cv = F16({voltage:0.4f}), .castor_ramp_cv = {dac_code_castor}, .pollux_ramp_cv = {dac_code_pollux} }},\n"
            )

        fh.write(
            textwrap.dedent(
                """\
        };

        size_t gem_ramp_table_len = sizeof(gem_ramp_table) / sizeof(struct GemRampTableEntry);

        /* clang-format on */
        """
            )
        )


if __name__ == "__main__":
    parser = argparse.ArgumentParser(
        formatter_class=argparse.ArgumentDefaultsHelpFormatter
    )
    parser.add_argument(
        "output_file",
        type=pathlib.Path,
    )

    args = parser.parse_args()

    main(args.output_file)
