// Copyright (c) 2013-2015 Vittorio Romeo
// License: Academic Free License ("AFL") v. 3.0
// AFL License page: https://opensource.org/licenses/AFL-3.0

/*
Copyright (c) 2010, Pierre KRIEGER
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:
    * Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    * Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    * Neither the name of the <organization> nor the
      names of its contributors may be used to endorse or promote products
      derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL <COPYRIGHT HOLDER> BE LIABLE FOR ANY
DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

// Check out Pierre Krieger's new version of luawrapper here:
// https://github.com/Tomaka17/luawrapper

#pragma once

#include "SSVOpenHexagon/Global/Assert.hpp"
#include "SSVOpenHexagon/Global/Macros.hpp"

#include "SSVOpenHexagon/Utils/UniquePtr.hpp"

#include <limits>
#include <map>
#include <memory>
#include <sstream>
#include <stdexcept>
#include <string>
#include <string_view>
#include <tuple>
#include <type_traits>
#include <vector>

#include <cstring>

#include <lua.hpp>

namespace Lua {

template <typename>
struct RemoveMemberPtr;

template <typename TReturn, typename TThis, typename... TArgs>
struct RemoveMemberPtr<TReturn (TThis::*)(TArgs...) const>
{
    using Type = TReturn(TArgs...);
};


template <typename FnType>
struct FnTupleWrapper
{};

template <typename... TArgs>
struct FnTupleWrapper<void(TArgs...)>
{
    enum
    {
        count = sizeof...(TArgs)
    };

    using ParamsType = std::tuple<TArgs...>;

    template <typename T>
    [[nodiscard, gnu::always_inline]] inline static constexpr std::tuple<> call(
        T&& fn, ParamsType&& mTpl)
    {
        std::apply(SSVOH_FWD(fn), SSVOH_MOVE(mTpl));
        return {};
    }
};

template <typename R, typename... TArgs>
struct FnTupleWrapper<R(TArgs...)>
{
    enum
    {
        count = sizeof...(TArgs)
    };

    using ParamsType = std::tuple<TArgs...>;

    template <typename T>
    [[nodiscard, gnu::always_inline]] inline static constexpr std::tuple<R>
    call(T&& fn, ParamsType&& mTpl)
    {
        return std::tuple<R>{std::apply(SSVOH_FWD(fn), SSVOH_MOVE(mTpl))};
    }
};

/**
 * @brief Defines a Lua context
 *
 * A Lua context is used to interpret Lua code. Since everything in Lua is a
 * variable (including functions),
 * we only provide few functions like readVariable and writeVariable. Note
 * that these functions can visit arrays,
 * ie. calling readVariable("a.b.c") will read variable "c" from array "b",
 * which is itself located in array "a".
 * You can also write variables with C++ functions so that they are callable
 * by Lua. Note however that you HAVE TO convert
 * your function to std::function (not directly std::bind or a lambda
 * function) so the class can detect which argument types
 * it wants. These arguments may only be of basic types (int, float, etc.)
 * or std::string.
 */
class LuaContext
{
public:
    explicit LuaContext(bool openDefaultLibs = true);

    LuaContext(const LuaContext&) = delete;
    LuaContext& operator=(const LuaContext&) = delete;

    LuaContext(LuaContext&& s) noexcept;
    LuaContext& operator=(LuaContext&& s) noexcept;

    ~LuaContext();


    /// \brief The table type can store any key and any value, and can be
    /// read or written by LuaContext
    class Table;

    /// \brief Thrown when an error happens during execution (like not
    /// enough parameters for a function)
    struct ExecutionErrorException : std::runtime_error
    {
        ExecutionErrorException(const std::string& msg);
    };

    /// \brief Generated by readVariable or isVariableArray when the asked
    /// variable doesn't exist/is nil
    struct VariableDoesntExistException : std::runtime_error
    {
        VariableDoesntExistException(const std::string& variable);
    };

    /// \brief Thrown when a syntax error happens in a lua script
    struct SyntaxErrorException : std::runtime_error
    {
        SyntaxErrorException(const std::string& msg);
    };

    /// \brief Thrown when trying to cast a lua variable to an invalid type
    struct WrongTypeException : std::runtime_error
    {
        WrongTypeException();
    };

    /// \brief Executes lua code from the stream \param code A stream that
    /// lua will read its code from
    [[gnu::always_inline]] inline void executeCode(std::istream& code)
    {
        _load(code);
        _call<std::tuple<>>(std::tuple<>());
    }

    /// \brief Executes lua code from the stream and returns a value \param
    /// code A stream that lua will read its code from
    template <typename T>
    [[nodiscard, gnu::always_inline]] inline T executeCode(std::istream& code)
    {
        _load(code);
        return _call<T>(std::tuple<>());
    }

    /// \brief Executes lua code given as parameter \param code A string
    /// containing code that will be executed by lua
    [[gnu::always_inline]] inline void executeCode(std::string_view code)
    {
        _load(code);
        _call<std::tuple<>>(std::tuple<>());
    }

    /// \brief Executes lua code from the stream and returns a value \param
    /// code A stream that lua will read its code from
    template <typename T>
    [[nodiscard, gnu::always_inline]] inline T executeCode(
        std::string_view code)
    {
        _load(code);
        return _call<T>(std::tuple<>());
    }

    /// \brief Tells that lua will be allowed to access an object's function
    template <typename T, typename R, typename... Args>
    [[gnu::always_inline]] inline void registerFunction(
        std::string_view name, R (T::*f)(Args...))
    {
        _registerFunction(name, [f](const std::shared_ptr<T>& ptr, Args... args)
            { return ((*ptr).*f)(args...); });
    }

    template <typename T, typename R, typename... Args>
    [[gnu::always_inline]] inline void registerFunction(
        std::string_view name, R (T::*f)(Args...) const)
    {
        _registerFunction(name, [f](const std::shared_ptr<T>& ptr, Args... args)
            { return ((*ptr).*f)(args...); });
    }

    template <typename T, typename R, typename... Args>
    [[gnu::always_inline]] inline void registerFunction(
        std::string_view name, R (T::*f)(Args...) volatile)
    {
        _registerFunction(name, [f](const std::shared_ptr<T>& ptr, Args... args)
            { return ((*ptr).*f)(args...); });
    }

    template <typename T, typename R, typename... Args>
    [[gnu::always_inline]] inline void registerFunction(
        std::string_view name, R (T::*f)(Args...) const volatile)
    {
        _registerFunction(name, [f](const std::shared_ptr<T>& ptr, Args... args)
            { return ((*ptr).*f)(args...); });
    }

    /// \brief Adds a custom function to a type determined using the
    /// function's first parameter
    /// \sa allowFunction
    /// \param fn Function which takes as first parameter a std::shared_ptr
    template <typename T>
    [[gnu::always_inline]] inline void registerFunction(
        std::string_view name, T&& fn, decltype(&T::operator())* = nullptr)
    {
        _registerFunction(name, SSVOH_FWD(fn));
    }

    /// \brief Inverse operation of registerFunction
    template <typename T>
    [[gnu::always_inline]] inline void unregisterFunction(std::string_view name)
    {
        _unregisterFunction<T>(name);
    }

    /// \brief Calls a function stored in a lua variable
    /// \details Template parameter of the function should be the expected
    /// return type (tuples and void are supported)
    /// \param mVarName Name of the variable containing the function to call
    /// \param ... Parameters to pass to the function
    template <typename R, typename... Args>
    [[nodiscard, gnu::always_inline]] inline R callLuaFunction(
        std::string_view mVarName, Args&&... args)
    {
        _getGlobal(mVarName);
        return _call<R>(std::make_tuple(SSVOH_FWD(args)...));
    }

    /// \brief Returns true if the value of the variable is an array \param
    /// mVarName Name of the variable to check
    [[nodiscard, gnu::always_inline]] inline bool isVariableArray(
        std::string_view mVarName) const
    {
        _getGlobal(mVarName);

        const bool answer = lua_istable(_state, -1);
        lua_pop(_state, 1);
        return answer;
    }

    /// \brief Writes an empty array into the given variable \note To write
    /// something in the array, use writeVariable. Example:
    /// writeArrayIntoVariable("myArr"); writeVariable("myArr.something",
    /// 5);
    [[gnu::always_inline]] inline void writeArrayIntoVariable(
        std::string_view mVarName)
    {
        lua_newtable(_state);
        _setGlobal(mVarName);
    }

    /// \brief Returns true if variable exists (ie. not nil)
    [[nodiscard, gnu::always_inline]] inline bool doesVariableExist(
        std::string_view mVarName) const
    {
        _getGlobal(mVarName);

        const bool answer = lua_isnil(_state, -1);
        lua_pop(_state, 1);
        return !answer;
    }

    /// \brief Destroys a variable \details Puts the nil value into it
    [[gnu::always_inline]] inline void clearVariable(std::string_view mVarName)
    {
        lua_pushnil(_state);
        _setGlobal(mVarName);
    }

    /// \brief Returns the content of a variable \throw
    /// VariableDoesntExistException if variable doesn't exist \note If you
    /// wrote a ObjectWrapper<T> into a variable, you can only read its
    /// value using a std::shared_ptr<T>
    template <typename T>
    [[nodiscard, gnu::always_inline]] inline T readVariable(
        std::string_view mVarName) const
    {
        _getGlobal(mVarName);
        return _readTopAndPop(1, (T*)nullptr);
    }

    /// \brief
    template <typename T>
    [[nodiscard, gnu::always_inline]] inline bool readVariableIfExists(
        std::string_view mVarName, T& out)
    {
        if(!doesVariableExist(mVarName))
        {
            return false;
        }

        out = readVariable<T>(mVarName);
        return true;
    }

    /// \brief Changes the content of a global lua variable
    /// \details Accepted values are: all base types (integers, floats),
    /// std::string, std::function or ObjectWrapper<...>. All objects are
    /// passed by copy and destroyed by the garbage collector.
    template <typename T>
    void writeVariable(std::string_view mVarName, T&& data)
    {
        static_assert(!std::is_same_v<std::tuple<T>, T>,
            "Error: you can't use LuaContext::writeVariable with a tuple");

        const int pushedElems = _push(SSVOH_FWD(data));

        try
        {
            _setGlobal(mVarName);
        }
        catch(...)
        {
            lua_pop(_state, pushedElems - 1);
            throw;
        }

        lua_pop(_state, pushedElems - 1);
    }

private:
    // the state is the most important variable in the class since it is our
    // interface with Lua
    // the mutex is here because the lua design is not thread safe (based on
    // a stack)
    //   eg. if multiple thread call "writeVariable" at the same time, we
    //   don't want them to be executed simultaneously
    // the mutex should be locked by all public functions that use the stack
    lua_State* _state;

    // all the user types in the _state must have the value of &typeid(T) in
    // their
    //   metatable at key "_typeid"

    // the getGlobal function is equivalent to lua_getglobal, except that it
    // can interpret
    //   any variable name in a table form (ie. names like table.value are
    //   supported)
    // see also https://www.lua.org/manual/5.1/manual.html#lua_getglobal
    // same for setGlobal <=> lua_setglobal
    // important: _setGlobal will pop the value even if it throws an
    // exception, while _getGlobal won't push the value if it throws an
    // exception
    void _getGlobal(std::string_view mVarName) const;
    void _setGlobal(std::string_view mVarName);

    // simple function that reads the top # elements of the stack, pops
    // them, and returns them
    // warning: first parameter is the number of parameters, not the
    // parameter index
    // if _read generates an exception, stack is popped anyway
    template <typename R>
    [[gnu::always_inline]] inline std::enable_if_t<!std::is_void_v<R>, R>
    _readTopAndPop(int nb, R* ptr = nullptr) const
    {
        try
        {
            R value = _read(-nb, ptr);
            lua_pop(_state, nb);
            return value;
        }
        catch(...)
        {
            lua_pop(_state, nb);
            throw;
        }
    }

    [[gnu::always_inline]] inline void _readTopAndPop(int nb, void*) const
    {
        lua_pop(_state, nb);
    }

    void _registerFunctionImpl(
        const char* nameCStr, const std::type_info& tiObjectType);

    /**************************************************/
    /*            FUNCTIONS REGISTRATION              */
    /**************************************************/
    // the "registerFunction" public functions call this one
    // this function writes in registry the list of functions for each
    // possible custom type (ie. T when pushing std::shared_ptr<T>)
    // to be clear, registry[&typeid(type)] contains an array where keys are
    // function names and values are functions
    //              (where type is the first parameter of the functor)
    template <typename T>
    void _registerFunction(std::string_view name, T&& function)
    {
        using FunctionType =
            typename RemoveMemberPtr<decltype(&T::operator())>::Type;

        using ObjectType = typename std::tuple_element_t<0,
            typename FnTupleWrapper<FunctionType>::ParamsType>::element_type;

        _registerFunctionImpl(name.data(), typeid(ObjectType));

        _push(SSVOH_FWD(function));
        lua_settable(_state, -3);
        lua_pop(_state, 1);
    }

    // inverse operation of _registerFunction
    template <typename T>
    void _unregisterFunction(std::string_view name)
    {
        // trying to get the existing functions list
        lua_pushlightuserdata(_state, const_cast<std::type_info*>(&typeid(T)));
        lua_gettable(_state, LUA_REGISTRYINDEX);

        if(!lua_istable(_state, -1))
        {
            lua_pop(_state, -1);
            return;
        }

        lua_pushstring(_state, name.data());
        lua_pushnil(_state);
        lua_settable(_state, -3);
        lua_pop(_state, 1);
    }

    /**************************************************/
    /*              LOADING AND CALLING               */
    /**************************************************/
    // this function loads data from the stream and pushes a function at the
    // top of the stack
    // it is defined in the .cpp
    void _load(std::istream& code);
    void _load(std::string_view code);

    // this function calls what is on the top of the stack and removes it
    // (just like lua_call)
    // if an exception is triggered, the top of the stack will be removed
    // anyway
    // In should be a tuple (at least until variadic templates are supported
    // everywhere), Out can be anything
    template <typename Out, typename In>
    Out _call(const In& in)
    {
        static_assert(std::tuple_size_v<In> >= 0,
            "Error: template parameter 'In' should be a tuple");

        int outArguments{0};
        int inArguments{0};

        try
        {
            // we push the parameters on the stack
            outArguments = std::tuple_size_v<std::tuple<Out>>;
            inArguments = _push(in);
        }
        catch(...)
        {
            lua_pop(_state, 1);
            throw;
        }

        // calling pcall automatically pops the parameters and pushes output
        const int pcallReturnValue =
            lua_pcall(_state, inArguments, outArguments, 0);

        // if pcall failed, analyzing the problem and throwing
        if(pcallReturnValue != 0)
        {
            // an error occurred during execution, an error message was
            // pushed on the stack
            const std::string errorMsg =
                _readTopAndPop(1, (std::string*)nullptr);

            if(pcallReturnValue == LUA_ERRMEM)
            {
                throw std::bad_alloc();
            }
            else if(pcallReturnValue == LUA_ERRRUN)
            {
                throw ExecutionErrorException(errorMsg);
            }
            else
            {
                throw std::runtime_error("UNKNOWN RC: " + errorMsg);
            }
        }

        // pcall succeeded, we pop the returned values and return them
        try
        {
            return _readTopAndPop(outArguments, (Out*)nullptr);
        }
        catch(...)
        {
            lua_pop(_state, outArguments);
            throw;
        }
    }


    /**************************************************/
    /*                 TABLE CLASS                    */
    /**************************************************/
public:
    class Table
    {
    public:
        Table() = default;
        Table(Table&& t) noexcept = default;
        Table& operator=(Table&& t) noexcept = default;

        template <typename... Args>
        explicit Table(Args&&... args)
        {
            insert(SSVOH_FWD(args)...);
        }

        friend void swap(Table& a, Table& b)
        {
            std::swap(a._elements, b._elements);
        }

        template <typename Key, typename Value, typename... Args>
        void insert(Key&& k, Value&& v, Args&&... args)
        {
            using RKey = typename ToPushableType<std::decay_t<Key>>::type;
            using RValue = typename ToPushableType<std::decay_t<Value>>::type;

            _elements.emplace_back(
                new Element<RKey, RValue>(SSVOH_FWD(k), SSVOH_FWD(v)));

            insert(SSVOH_FWD(args)...);
        }

        void insert()
        {}

        template <typename Value, typename Key>
        Value read(const Key& key)
        {
            using Key2 = typename ToPushableType<Key>::type;
            using Value2 = typename ToPushableType<Value>::type;

            for(int k = static_cast<int>(_elements.size()) - 1; k >= 0; --k)
            {
                auto element =
                    dynamic_cast<Element<Key2, Value2>*>(_elements[k].get());

                if(element != nullptr && element->key == key)
                {
                    return element->value;
                }
            }

            throw VariableDoesntExistException("<key in table>");
        }

    private:
        Table(const Table&);
        Table& operator=(const Table&);

        // this is the base structure
        // the push function should add the key/value pair to the table
        // currently at the top of the stack
        struct ElementBase
        {
            virtual ~ElementBase()
            {}

            virtual void push(LuaContext&) const = 0;
        };

        // derivate of ElementBase, real implementation
        template <typename Key, typename Value>
        struct Element : public ElementBase
        {
            Key key;
            Value value;

            Element(Key&& k, Value&& v) : key(SSVOH_FWD(k)), value(SSVOH_FWD(v))
            {}

            void push(LuaContext& ctxt) const override
            {
                SSVOH_ASSERT(lua_istable(ctxt._state, -1));

                ctxt._push(key);
                ctxt._push(value);
                lua_settable(ctxt._state, -3);
            }
        };

        // pushing the whole array
        friend class LuaContext;

        int _push(LuaContext& ctxt) const
        {
            lua_newtable(ctxt._state);

            try
            {
                for(auto& i : _elements)
                {
                    i->push(ctxt);
                }
            }
            catch(...)
            {
                lua_pop(ctxt._state, 1);
                throw;
            }

            return 1;
        }

        // elements storage
        std::vector<hg::Utils::UniquePtr<ElementBase>> _elements;
    };

private:
    /**************************************************/
    /*                PUSH FUNCTIONS                  */
    /**************************************************/
    // this structure converts an input type to a pushable output type
    template <typename Input, typename = void>
    struct ToPushableType;

    // first the basic ones: integer, number, boolean, string
    [[gnu::always_inline]] inline int _push()
    {
        return 0;
    }

    [[gnu::always_inline]] inline int _push(bool v)
    {
        lua_pushboolean(_state, v);
        return 1;
    }

    [[gnu::always_inline]] inline int _push(std::string_view s)
    {
        lua_pushstring(_state, s.data());
        return 1;
    }

    [[gnu::always_inline]] inline int _push(const char* s)
    {
        lua_pushstring(_state, s);
        return 1;
    }

    // pushing floating numbers
    template <typename T>
    [[gnu::always_inline]] inline std::enable_if_t<std::is_floating_point_v<T>,
        int>
    _push(T nb)
    {
        lua_pushnumber(_state, nb);
        return 1;
    }

    // pushing integers
    template <typename T>
    [[gnu::always_inline]] inline std::enable_if_t<std::is_integral_v<T>, int>
    _push(T nb)
    {
        lua_pushinteger(_state, nb);
        return 1;
    }

    // using variadic templates, you can push multiple values at once
    template <typename... Ts, typename = std::enable_if_t<(sizeof...(Ts) > 1)>>
    [[gnu::always_inline]] inline int _push(Ts&&... xs)
    {
        int p = 0;

        try
        {
            ((p += _push(SSVOH_FWD(xs))), ...);
        }
        catch(...)
        {
            lua_pop(_state, p);
            throw;
        }

        return p;
    }

    // pushing tables
    [[gnu::always_inline]] inline int _push(const Table& table)
    {
        return table._push(*this);
    }

    // pushing maps
    template <typename Key, typename Value>
    int _push(const std::map<Key, Value>& map)
    {
        lua_newtable(_state);

        for(const auto& [k, v] : map)
        {
            _push(k);
            _push(v);
            lua_settable(_state, -3);
        }

        return 1;
    }

    template <typename FunctionPushType>
    static auto callbackCall(lua_State* lua)
    {
        // this function is called when the lua script tries to call our
        // custom data type
        // what we do is we simply call the function
        SSVOH_ASSERT(lua_gettop(lua) >= 1);
        SSVOH_ASSERT(lua_isuserdata(lua, 1));

        auto function = (FunctionPushType*)lua_touserdata(lua, 1);

        SSVOH_ASSERT(function);
        return (*function)(lua);
    }

    template <typename FunctionPushType>
    static int callbackGarbage(lua_State* lua)
    {
        // this one is called when lua's garbage collector no longer
        // needs our custom data type
        // we call std::function<int (lua_State*)>'s destructor
        SSVOH_ASSERT(lua_gettop(lua) == 1);

        auto function = (FunctionPushType*)lua_touserdata(lua, 1);

        SSVOH_ASSERT(function);
        function->~FunctionPushType();

        return 0;
    }

    template <typename F, typename Op>
    struct FunctionToPush
    {
        LuaContext* _ctx;
        F _fn;

        auto operator()(lua_State* state) const
        {
            SSVOH_ASSERT(_ctx->_state == state);

            // FnTupleWrapper<FnType> is a specialized template
            // structure which defines
            // "ParamsType", "ReturnType" and "call"
            // the first two correspond to the params list and return
            // type as tuples
            //   and "call" is a static function which will call a
            //   function
            //   of this type using parameters passed as a tuple
            using FnType = typename RemoveMemberPtr<Op>::Type;
            using TupledFunction = FnTupleWrapper<FnType>;

            // checking if number of parameters is correct
            if(lua_gettop(state) < TupledFunction::count)
            {
                // if not, using lua_error to return an error
                luaL_where(state, 1);
                lua_pushstring(state, "this function requires at least ");
                lua_pushnumber(state, TupledFunction::count);
                lua_pushstring(state, " parameter(s)");
                lua_concat(state, 4);
                return lua_error(state); // lua_error throws an
                                         // exception when compiling as
                                         // C++
            }

            // pushing the result on the stack and returning number of
            // pushed elements
            return _ctx->_push(
                // calling the function, result should be a tuple
                TupledFunction::call(_fn,

                    // reading parameters from the stack
                    _ctx->_read(-TupledFunction::count,
                        static_cast<typename TupledFunction::ParamsType*>(
                            nullptr))));
        }
    };

    void _pushFnImpl(int (*callbackCall)(lua_State*),
        int (*callbackGarbage)(lua_State*), const std::type_info& tiObject);

    // when you call _push with a functor, this definition should be used
    // (thanks to SFINAE)
    // it will determine the function category using its () operator, then
    //   generate a callable user data and push it
    template <typename T, typename DecayT = std::decay_t<T>,
        typename Op = decltype(&DecayT::operator())>
    int _push(T&& fn, Op = nullptr)
    {
        // when the lua script calls the thing we will push on the stack, we
        // want "fn" to be executed
        // if we used lua's cfunctions system, we could not detect when the
        // function is no longer in use, which could cause problems
        // so we use userdata instead

        // typedefing the type of data we will push
        using FunctionPushType = FunctionToPush<DecayT, Op>;

        // creating the object
        // lua_newuserdata allocates memory in the internals of the lua
        // library and returns it so we can fill it
        //   and that's what we do with placement-new
        auto* const functionLocation = (FunctionPushType*)lua_newuserdata(
            _state, sizeof(FunctionPushType));

        new(functionLocation)
            FunctionPushType{._ctx = this, ._fn = SSVOH_FWD(fn)};

        _pushFnImpl(&callbackCall<FunctionPushType>,
            &callbackGarbage<FunctionPushType>, typeid(T));

        return 1;
    }

    // when pushing a unique_ptr, it is simply converted to a shared_ptr
    // this definition is necessary because unique_ptr has an implicit bool
    // conversion operator
    // with C++0x, this bool operator will certainly be declared explicit
    template <typename T>
    int _push(std::unique_ptr<T>&& mObj)
    {
        return _push(std::shared_ptr<T>(SSVOH_MOVE(mObj)));
    }

    void _pushSPtrImpl(int (*garbageCallback)(lua_State*),
        const std::type_info& tiSharedPtr, const std::type_info& tiObject);

    // when pushing a shared_ptr, we create a custom type
    // we store a copy of the shared_ptr inside lua's internals
    //   and add it a metatable: __gc for destruction and __index pointing
    //   to the corresponding
    //   table in the registry (see _registerFunction)
    template <typename T>
    int _push(std::shared_ptr<T>&& mObj)
    {
        // this is a structure providing static C-like functions that we can
        // feed to lua
        struct Callback
        {
            // this function is called when lua's garbage collector no
            // longer needs our shared_ptr
            // we simply call its destructor
            static int garbage(lua_State* lua)
            {
                SSVOH_ASSERT(lua_gettop(lua) == 1);

                auto* ptr = (std::shared_ptr<T>*)lua_touserdata(lua, 1);

                SSVOH_ASSERT(ptr && *ptr);
                ptr->~shared_ptr();

                return 0;
            }
        };

        // creating the object
        // lua_newuserdata allocates memory in the internals of the lua
        // library and returns it so we can fill it
        //   and that's what we do with placement-new
        const auto pointerLocation = static_cast<std::shared_ptr<T>*>(
            lua_newuserdata(_state, sizeof(std::shared_ptr<T>)));

        new(pointerLocation) std::shared_ptr<T>(SSVOH_MOVE(mObj));
        _pushSPtrImpl(
            &Callback::garbage, typeid(std::shared_ptr<T>), typeid(T));

        return 1;
    }

    // pushing tuples is also possible, though a bit complicated
    template <typename... Args>
    [[gnu::always_inline]] inline int _push(const std::tuple<Args...>& t)
    {
        return [this, &t]<int... Is>(std::integer_sequence<int, Is...>) {
            return (this->_push(std::get<Is>(t)) + ... + 0);
        }(std::make_integer_sequence<int, sizeof...(Args)>{});
    }

    /**************************************************/
    /*                READ FUNCTIONS                  */
    /**************************************************/
    // to use the _read function, pass as second parameter a null pointer
    // whose base type is the wanted return type
    // eg. if you want an int, pass "static_cast<int*>(nullptr)" as second
    // parameter

    // reading void
    [[gnu::always_inline]] inline void _read(int, void const* = nullptr) const
    {}

    // first the integer types
    template <typename T>
    [[gnu::always_inline]] inline std::enable_if_t<
        std::numeric_limits<T>::is_integer, T>
    _read(const int index, T const* = nullptr) const
    {
        if(lua_isuserdata(_state, index))
        {
            throw WrongTypeException{};
        }

        return T(lua_tointeger(_state, index));
    }

    // then the floating types
    template <typename T>
    [[gnu::always_inline]] inline std::enable_if_t<
        std::numeric_limits<T>::is_specialized &&
            !std::numeric_limits<T>::is_integer,
        T>
    _read(const int index, T const* = nullptr) const
    {
        if(lua_isuserdata(_state, index))
        {
            throw WrongTypeException{};
        }

        return T(lua_tonumber(_state, index));
    }

    // boolean
    [[gnu::always_inline]] inline bool _read(
        const int index, bool const* = nullptr) const
    {
        if(lua_isuserdata(_state, index))
        {
            throw WrongTypeException{};
        }

        return lua_toboolean(_state, index) != 0; // "!= 0" removes a
                                                  // warning because
                                                  // lua_toboolean returns
                                                  // an int
    }

    // string
    // lua_tostring returns a temporary pointer, but that's not a problem
    // since we copy
    //   the data in a std::string
    [[gnu::always_inline]] inline std::string _read(
        const int index, std::string const* = nullptr) const
    {
        if(lua_isuserdata(_state, index))
        {
            throw WrongTypeException{};
        }

        return lua_tostring(_state, index);
    }

    // maps
    template <typename Key, typename Value>
    std::map<Key, Value> _read(
        const int index, std::map<Key, Value> const* = nullptr) const
    {
        if(!lua_istable(_state, index))
        {
            throw WrongTypeException{};
        }

        std::map<Key, Value> retValue;

        // we traverse the table at the top of the stack
        lua_pushnil(_state); // first key
        while(lua_next(_state, index - 1) != 0)
        {
            // now a key and its value are pushed on the stack
            retValue.emplace(_read(-2, static_cast<Key*>(nullptr)),
                _read(-2, static_cast<Value*>(nullptr)));
            lua_pop(_state, 1); // we remove the value but keep the key for
                                // the next iteration
        }

        return retValue;
    }

    // reading array
    Table _read(int index, Table const* = nullptr) const
    {
        if(!lua_istable(_state, index))
        {
            throw WrongTypeException{};
        }

        throw std::logic_error{"Not implemented"};

        /*Table table;

// we traverse the table at the top of the stack
lua_pushnil(_state);            // first key
while(lua_next(_state, -2) != 0) {
        // now a key and its value are pushed on the stack
        auto keyType = lua_type(_state, -2);
        auto valueType = lua_type(_state, -1);

        switch (keyType) {
                case LUA_TNUMBER:                       break;
                case LUA_TBOOLEAN:                      break;
                case LUA_TSTRING:                       break;
                case LUA_TTABLE:                        break;
                case LUA_TFUNCTION:                     break;
                case LUA_TUSERDATA:                     break;
                case LUA_TLIGHTUSERDATA:        break;
                default:                throw(WrongTypeException());
        }

        lua_pop(_state, 1);             // we remove the value but keep the
key for the next iteration
}

return table;*/
    }

    // reading a shared_ptr
    // we check that type is correct by reading the metatable
    template <typename T>
    std::shared_ptr<T> _read(
        int mIdx, std::shared_ptr<T> const* = nullptr) const
    {
        if(!lua_isuserdata(_state, mIdx) || !lua_getmetatable(_state, mIdx))
        {
            throw WrongTypeException{};
        }

        // now we have our metatable on the top of the stack
        // retrieving its _typeid member
        lua_pushstring(_state, "_typeid");
        lua_gettable(_state, -2);

        // if wrong typeid, we throw
        if(lua_touserdata(_state, -1) !=
            const_cast<std::type_info*>(&typeid(std::shared_ptr<T>)))
        {
            lua_pop(_state, 2);
            throw WrongTypeException{};
        }

        lua_pop(_state, 2);

        // now we know that the type is correct, we retrieve the pointer
        const auto ptr =
            static_cast<std::shared_ptr<T>*>(lua_touserdata(_state, mIdx));

        SSVOH_ASSERT(ptr && *ptr);
        return *ptr; // returning a copy of the shared_ptr
    }

    template <typename... Ts>
    [[gnu::always_inline]] inline auto _read(
        const int index, std::tuple<Ts...> const* = nullptr)
    {
        return [this, index]<int... Is>(std::integer_sequence<int, Is...>)
        {
            return std::make_tuple(this->_read(
                index + Is, static_cast<std::decay_t<Ts>*>(nullptr))...);
        }(std::make_integer_sequence<int, sizeof...(Ts)>{});
    }

    [[gnu::always_inline]] inline constexpr std::tuple<> _read(
        int, std::tuple<> const* = nullptr) const noexcept
    {
        return {};
    }
};

template <typename T>
struct LuaContext::ToPushableType<T&>
{
    using type = typename ToPushableType<T>::type;
};

template <typename T>
struct LuaContext::ToPushableType<const T&>
{
    using type = typename ToPushableType<T>::type;
};

template <typename T>
struct LuaContext::ToPushableType<T, std::enable_if_t<std::is_integral_v<T>>>
{
    using type = lua_Integer;
};

template <typename T>
struct LuaContext::ToPushableType<T,
    std::enable_if_t<std::is_floating_point_v<T>>>
{
    using type = lua_Number;
};

template <>
struct LuaContext::ToPushableType<bool>
{
    using type = bool;
};

template <>
struct LuaContext::ToPushableType<const char*>
{
    using type = std::string;
};

template <int N>
struct LuaContext::ToPushableType<const char[N]>
{
    using type = std::string;
};

template <int N>
struct LuaContext::ToPushableType<char[N]>
{
    using type = std::string;
};

template <>
struct LuaContext::ToPushableType<std::string>
{
    using type = std::string;
};

template <typename T>
struct LuaContext::ToPushableType<std::unique_ptr<T>>
{
    using type = std::shared_ptr<T>;
};

template <typename T>
struct LuaContext::ToPushableType<std::shared_ptr<T>>
{
    using type = std::shared_ptr<T>;
};

template <>
struct LuaContext::ToPushableType<LuaContext::Table>
{
    using type = LuaContext::Table;
};

template <typename T>
struct LuaContext::ToPushableType<T, decltype(&T::operator(), void())>
{
    using type = T;
};

} // namespace Lua
