# This code was auto-generated using ArchitectureGenerator.pl on 2017-01-15T15:07:35
package RepositoryLayer;
use namespace::autoclean;
use Moose;
use MooseX::ClassAttribute;

use MooseX::StrictConstructor;
use Try::Tiny;

use BibSpace::Util::IUidProvider;
use BibSpace::DAO::MySQLDAOFactory;
use BibSpace::Util::EntityFactory;

# this is rewritten during backup-restore, thus rw
has 'e_factory' => (is => 'rw', isa => 'Maybe[EntityFactory]');

=item backendFactoryName
    Stores the name of the DAO Factory for this layer. E.g. SmartArrayDaoFactory
=cut

has 'backendFactoryName' => (is => 'ro', isa => 'Str', required => 1);
has 'name'               => (is => 'ro', isa => 'Str', required => 1);
has 'priority'           => (
  is            => 'ro',
  isa           => 'Int',
  default       => 99,
  documentation => q{
        Lower number = higher priority = will be saved before other layers with lower priority.
    },
);
has 'logger' => (is => 'ro', does => 'ILogger', required => 1);
has 'handle' => (
  is            => 'ro',
  isa           => 'Object',
  required      => 1,
  documentation => q{
        Holds a backend handle (e.g., DBI handle, ArrayRef, Redis Connection.
    },
);
has 'reset_data_callback' => (
  is            => 'ro',
  isa           => 'Maybe[CodeRef]',
  default       => undef,
  documentation => q{
        Holds a pointer to the function that resets data of this layer.
    },
);
has 'reset_data_callback_arguments' => (
  is            => 'ro',
  isa           => 'Maybe[ArrayRef]',
  default       => sub { [] },
  documentation => q{
        Holds an array of parameters for the function that resets data of this layer.
    },
);
has 'is_read'         => (is => 'ro', isa => 'Bool', default => undef);
has 'creates_on_read' => (
  is       => 'ro',
  isa      => 'Bool',
  required => 1,
  documentation =>
    q{Does this layer creates new objects (BibSpaceEntity) on reading them from backend?
        If yes, then moving data from this layer requires to reset all uid_providers.
        Example 1:
        0) MySQL do creates objects on read,
        1) Imagine moving data: MySQL -> overwrites -> SmartArray,
        2) We reset data (only data) in SmartArray - it will be anyway overwritten,
        3) MySQL created objects on read and registers new IDs in the id_provider,
        4) There is one instance of id_provider (for each type) per all layers,
        5) The id_provider MUST BE RESET, because the objects already residing in the SmartArray
           would be duplicated with new IDs generated by the id_provider.
        Example 2:
        0) SmartArray does not create objects on read - it stores references and does not offer persistence.
        1) Imagine moving data: SmartArray -> overwrites -> MySQL,
        2) We reset data (only data) in MySQL - it will be anyway overwritten,
        3) Nothing bad happens in MySQL, MySQL does not persist (directly) state of id_providers,
        4) The data remain in SmartArray, so the id_providers shall remain intact.
        Conclusion:
        If you move data FROM layer, which creates_on_read==true,
        then you need to reset id_providers.
    },
);

=item reset_data
    Hard reset removes all instances of repositories and resets all id providers.
    Use only for overwriting whole data set, e.g., during backup restore.
=cut

sub reset_data {
  my $self = shift;
  $self->logger->warn("HARD RESET of data in layer '" . $self->name . "'!");

  if (defined $self->reset_data_callback) {
    $self->logger->warn(
      "HARD RESET of data in layer '"
        . $self->name
        . "' using reset_data_callback",
      "" . __PACKAGE__ . "->reset_data"
    );

    my $reset_subroutine = \&{$self->reset_data_callback};
    if (@{$self->reset_data_callback_arguments}) {
      ($reset_subroutine)->(@{$self->reset_data_callback_arguments});
    }
    else {
      &$reset_subroutine;
    }

  }
  else {
    $self->handle->reset_data;
  }
}

sub daoDispatcher {
  my $self        = shift;
  my $factory     = shift;
  my $entity_type = shift;

  if ($entity_type eq 'TagType') {
    return $factory->getTagTypeDao();
  }
  if ($entity_type eq 'Team') {
    return $factory->getTeamDao();
  }
  if ($entity_type eq 'Author') {
    return $factory->getAuthorDao();
  }
  if ($entity_type eq 'Authorship') {
    return $factory->getAuthorshipDao();
  }
  if ($entity_type eq 'Membership') {
    return $factory->getMembershipDao();
  }
  if ($entity_type eq 'Entry') {
    return $factory->getEntryDao();
  }
  if ($entity_type eq 'Labeling') {
    return $factory->getLabelingDao();
  }
  if ($entity_type eq 'Tag') {
    return $factory->getTagDao();
  }
  if ($entity_type eq 'Exception') {
    return $factory->getExceptionDao();
  }
  if ($entity_type eq 'Type') {
    return $factory->getTypeDao();
  }
  if ($entity_type eq 'User') {
    return $factory->getUserDao();
  }
  $self->logger->error("Requested unknown entity_type: '$entity_type'");
  die "Requested unknown entity_type: '$entity_type'";
}

=item getDao
    Returns Data Access Object (DAO) for given entity_type and backend used by layer
=cut

sub getDao {
  my $self        = shift;
  my $entity_type = shift;
  my $daoAbstractFactory
    = DAOFactory->new(logger => $self->logger, e_factory => $self->e_factory);
  my $daoFactory = $daoAbstractFactory->getInstance($self->backendFactoryName,
    $self->handle);
  return $self->daoDispatcher($daoFactory, $entity_type);
}

=item get_summary_hash
    Provides a summary of a layer in form of a hash.
    Has is build like this: entity_name => number of stored objects
=cut

sub get_summary_hash {
  my $self = shift;
  my %hash = map { $_ => $self->count($_) } FlatRepository->get_models;
  return \%hash;
}

=item get_summary_table
    Prints nice summary table for all layers.
    Example:
    CNT_X = number of entities in layer X
    -------------------------------
    | entity          | CNT_mysql |
    -------------------------------
    | Author          |        74 |
    | Authorship      |       696 |
    | Entry           |       379 |
    | Exception       |         2 |
    | Labeling        |         6 |
    | Membership      |        27 |
    | Tag             |        50 |
    | TagType         |         4 |
    | Team            |         4 |
    | Type            |        24 |
    | User            |         1 |
    -------------------------------
=cut

sub get_summary_table {
  my $self = shift;
  my $str  = "\n";

  my %count_hash;    #layer_name => summary_hash
  my @layer_names;
  push @layer_names, "CNT_" . $self->name;
  $count_hash{"CNT_" . $self->name} = $self->get_summary_hash;

  my $tab_width = 31;

  for (1 .. $tab_width) { $str .= "_"; }
  $str .= "\n";
  $str .= sprintf "| %-15s |", 'entity';
  foreach my $ln (reverse sort @layer_names) {
    $str .= sprintf " %-9s |", $ln;
  }
  $str .= "\n";
  for (1 .. $tab_width) { $str .= "-"; }
  $str .= "\n";
  foreach my $entity (FlatRepository->get_entities) {
    $str .= sprintf "| %-15s |", $entity;
    foreach my $ln (reverse sort @layer_names) {
      $str .= sprintf " %9s |", $count_hash{$ln}->{$entity};
    }
    $str .= "\n";
  }
  for (1 .. $tab_width) { $str .= "-"; }
  $str .= "\n";
  foreach my $entity (sort FlatRepository->get_relations) {
    $str .= sprintf "| %-15s |", $entity;
    foreach my $ln (reverse sort @layer_names) {
      $str .= sprintf " %9s |", $count_hash{$ln}->{$entity};
    }
    $str .= "\n";
  }
  for (1 .. $tab_width) { $str .= "-"; }
  $str .= "\n";
  return $str;
}

=item all
    Returns all objects of a $type stored in this layer
=cut

sub all {
  my ($self, $type) = @_;
  return $self->getDao($type)->all;
}

sub count {
  my ($self, $type) = @_;
  return $self->getDao($type)->count;
}

sub empty {
  my ($self, $type) = @_;
  return $self->getDao($type)->empty;
}

sub exists {
  my ($self, $type, $obj) = @_;
  return $self->getDao($type)->exists($obj);
}

sub save {
  my ($self, $type, @objects) = @_;
  return $self->getDao($type)->save(@objects);
}

sub update {
  my ($self, $type, @objects) = @_;
  return $self->getDao($type)->update(@objects);
}

sub delete {
  my ($self, $type, @objects) = @_;
  return $self->getDao($type)->delete(@objects);
}

sub filter {
  my ($self, $type, $coderef) = @_;
  return $self->getDao($type)->filter($coderef);
}

sub find {
  my ($self, $type, $coderef) = @_;
  return $self->getDao($type)->find($coderef);
}

__PACKAGE__->meta->make_immutable;
no Moose;
1;
