import {
  ChatPromptTemplate,
  MessagesPlaceholder,
  PromptTemplate,
} from "langchain/prompts";

export const prefix = {
  investigation: `
  You are a smart AI assistant called Vespper. Your mission is to investigate incidents in Production
  and provide findings to the responders, with as much information as possible.
  Use the tools at your disposal to fetch information about the problem.

  Notes for the investigation:
  * Check logs/metrics/traces for clues
  * Check whether there were recent code changes
  * You can provide citations from the tools you use, but again, be concise.
  * Messages can sometimes contain image descriptions, generated by other models.
  
  IMPORTANT: Be concise with your answers. Don't write messages that are too long. Try to say more with less words.
  
  Begin!
  `,
  investigationLean: `
  You are an expert on-call engineer called Vespper. Your mission is to investigate incidents in Production and provide findings to the responders, with as much information as possible.

  Given the contextual information, produce a meaningful summarization about the data provided.
  Don't write a report, write in free text, 5 sentences max. Be very concise and produce a short answer in a human readable format.

  Begin! 

  Incident:
  {incident}

  Additional investigation information:
  {additionalInfo}
  
  Contextual information:
  {context}
  `,
  conversation: `
  You are a smart AI assistant called Vespper. Your mission is to help on-call developers and SREs investigate production incidents.
  
  IMPORTANT: Be concise with your answers. Don't write messages that are too long. Try to say more with less words.
  
  Begin!
  `,
  conversationIssues: `
  You are a smart AI assistant called Vespper, living inside Github. Your mission is to help developers find answers to their issues & questions.
  You can use the tools at your disposal to fetch information about the subject, if needed.

  Notes:
  - You have two types of tools: expert tools and general tools. When using expert tools, please propagate their results to the user.
  
  Begin!
  `,
  summarizeReadme: `
  Summarize this repository's README.md into a few words (10 words max). 
  Ignore technical stuff and focus on the core purpose of this repo.

  Repo name: {repo}

  Repo README.md:
  \`\`\`md
  {readme}
  \`\`\`
  `,
  captionImage: `
  Please describe what you see in this image
  `,
  generateQueries: `
  Hi! You are tasked with retrieving information about the following production incident:

  {incident}

  You have access to a semantic search engine (VectorDB), where you can fetch historical & related information about this issue. Please create {nQueries} different queries that you would want to search against this search engine.

  Your queries should be:
  * Related to the incident above
  * Short and simple. No long sentences.
  * Abstract and expressive, with no IP addresses & numbers.


  You should return your answer as JSON. It should contain 1 key called "queries", and it should be a list. For instance, here is an example response:
  \`\`\`json
  {{"queries": ["Service X issue", "500 error", "User could not pay issue"]}}
  \`\`\`


  IMPORTANT: Please respond only in JSON.

  Begin!
  `,
  verifyDocument: `
  Hi! You are tasked with verifying whether a document is actually relevant to a source information. The source information is a production incident information. You are tasked with judging whether a document is relevant. 

  For example, given the following incident:
  \`\`\`
  Title: Coralogix Alert: Service data-processor has high CPU usage
  Source: PagerDuty
  Time: 5 months ago
  Additional information: {{"Application":"demo-app","CompanyId":4014214}}
  \`\`\`
  
  And the following document:
  \`\`\`
  Does someone know where do we save our finance reports?
  \`\`\`

  You should return false. IMPORTANT: return only true or false.
  
  Begin!

  Incident:
  {incident}

  Document:
  {document}
  `,
  extractLogStructureKeys: `  
  Given some log records, return the key paths of the severity and message in a JSON format.
  Key paths are the paths to the severity and message fields in the log record.

  Examples:
  Input:
  [{{"message": "Successfully updated user 123", "timestamp": "some-time", "severityText": "INFO"}},
  {{"message": "Successfully updated user 456", "timestamp": "some-time", "severityText": "INFO"}}]

  Expected output:
  \`\`\`json
  {{
    "severityKey": "severityText",
    "messageKey": "message"
  }}
  \`\`\`

  Input:
  [{{"timestamp": "some-time", "severity": {{"severityNumber": 3, "severityText": "INFO"}}, "logRecord": {{"body": "Successfully updated user 123"}}}},
  {{"timestamp": "some-time2", "severity": {{"severityNumber": 3, "severityText": "INFO"}}, "logRecord": {{"body": "Successfully updated user 456"}}}}
  ]

  Expected output:
  \`\`\`json
  {{
    "severityKey": "severity.severityText",
    "messageKey": "logRecord.body"
  }}
  \`\`\`

  Return your answer as a valid JSON.

  Begin!
  
  Log records:
  {logRecords}
  `,
  filterHighCardinalityFields: `
  Given some log records and their fields, return the fields that don't have a high cardinality.
  Meaning, exclude fields which seem to have a lot of different values. This usually means fields
  such as user ids, session ids, messages, log body, timestamps, etc.

  Your output should be in JSON format which contains just one key called "fields" and its value should be the array of fields that don't have a high cardinality.

  For example, given the following log records:
  \`\`\`json
  [
    {{"userId": 123, "timestamp": "2021-01-01T00:00:00Z", "message": "User logged in", "service": "demo-app"}},
    {{"userId": 456, "timestamp": "2021-01-01T00:00:00Z", "message": "User logged in", "service": "demo-app"}},
    {{"userId": 789, "timestamp": "2021-01-01T00:00:00Z", "message": "User logged in", "service": "demo-app"}}
  ]
  \`\`\`

  The output should be:
  \`\`\`json
  {{"fields": ["service"]}}
  \`\`\`

  That's it! Begin!

  Log records:
  {logRecords}
  `,
};

export const investigationTemplate = ChatPromptTemplate.fromMessages([
  ["ai", prefix.investigation],
  new MessagesPlaceholder("history"),
  ["human", "{input}"],
  new MessagesPlaceholder("agent_scratchpad"),
]);

export const conversationTemplate = ChatPromptTemplate.fromMessages([
  ["ai", prefix.conversation],
  new MessagesPlaceholder("history"),
  ["human", "{input}"],
  new MessagesPlaceholder("agent_scratchpad"),
]);

export const conversationIssuesTemplate = ChatPromptTemplate.fromMessages([
  ["ai", prefix.conversationIssues],
  new MessagesPlaceholder("history"),
  ["human", "{input}"],
  new MessagesPlaceholder("agent_scratchpad"),
]);

export const summarizeReadmePrompt = PromptTemplate.fromTemplate(
  prefix.summarizeReadme,
);

export const generateQueriesPrompt = PromptTemplate.fromTemplate(
  prefix.generateQueries,
);

export const verifyDocumentPrompt = PromptTemplate.fromTemplate(
  prefix.verifyDocument,
);

export const investigationLeanTemplate = PromptTemplate.fromTemplate(
  prefix.investigationLean,
);

export const extractLogStructureKeysPrompt = PromptTemplate.fromTemplate(
  prefix.extractLogStructureKeys,
);

export const filterHighCardinalityFieldsPrompt = PromptTemplate.fromTemplate(
  prefix.filterHighCardinalityFields,
);
